/**
 * 2004
 * Example mini WMA player by McMCC <mcmcc@mail.ru>
 * Use ALSA 0.9 or high... (dmix support) 
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <alsa/asoundlib.h>

#ifdef HAVE_AV_CONFIG_H
#undef HAVE_AV_CONFIG_H
#endif

#include "avcodec.h"
#include "avformat.h"
#include "playlist.h"

#define ST_BUFF 2048

void audio_decode(const char *filename)
{
    AVCodec *codec;
    AVPacket pkt;
    AVCodecContext *c= NULL;
    AVFormatContext *ic= NULL;
    int out_size, size, len, err, i, s_rate;
    int tns, thh, tmm, tss, st_buff, frames;
    uint8_t *outbuf, *inbuf_ptr, *s_outbuf;
    snd_pcm_t *playback_handle;
    snd_pcm_hw_params_t *hw_params;
    FifoBuffer f;

    err = av_open_input_file(&ic, filename, NULL, 0, NULL);
    if (err < 0) {
        printf("Error file %s\n", filename);
	exit(1);
    }

    for(i = 0; i < ic->nb_streams; i++) {
        c = &ic->streams[i]->codec;
        if(c->codec_type == CODEC_TYPE_AUDIO)
            break;
    }

    av_find_stream_info(ic);

    codec = avcodec_find_decoder(c->codec_id);        
    if (!codec) {
        fprintf(stderr, "codec not found\n");
        exit(1);
    }
    /* open it */
    if (avcodec_open(c, codec) < 0) {
        fprintf(stderr, "could not open codec\n");
        exit(1);
    }
    
    st_buff = ST_BUFF;

    outbuf = malloc(AVCODEC_MAX_AUDIO_FRAME_SIZE);
    s_outbuf = malloc(st_buff);

        fprintf(stderr, "\n\n");
    dump_format(ic, 0, filename, 0);
    if (ic->title[0] != '\0')
        fprintf(stderr, "Title: %s\n", ic->title);
    if (ic->author[0] != '\0')
        fprintf(stderr, "Author: %s\n", ic->author);
    if (ic->album[0] != '\0')
        fprintf(stderr, "Album: %s\n", ic->album);
    if (ic->year != 0)
        fprintf(stderr, "Year: %d\n", ic->year);
    if (ic->track != 0)
        fprintf(stderr, "Track: %d\n", ic->track);
    if (ic->genre[0] != '\0')
        fprintf(stderr, "Genre: %s\n", ic->genre);
    if (ic->copyright[0] != '\0')
        fprintf(stderr, "Copyright: %s\n", ic->copyright);
    if (ic->comment[0] != '\0')
        fprintf(stderr, "Comments: %s\n", ic->comment);

    if (ic->duration != 0)
    {
	tns = ic->duration/1000000LL;
	thh = tns/3600;
	tmm = (tns%3600)/60;
	tss = (tns%60);
	fprintf(stderr, "Time: %2d:%02d:%02d\n", thh, tmm, tss);
    }
	tns = 0;
	thh = 0;
	tmm = 0;
	tss = 0;
    
    if ((err = snd_pcm_open (&playback_handle, "default", SND_PCM_STREAM_PLAYBACK, SND_PCM_NONBLOCK)) < 0) {
            fprintf (stderr, "cannot open audio device -default- (%s)\n", 
                     snd_strerror (err));
            exit (1);
    }
    
    snd_pcm_nonblock(playback_handle, 0);
                   
    if ((err = snd_pcm_hw_params_malloc (&hw_params)) < 0) {
            fprintf (stderr, "cannot allocate hardware parameter structure (%s)\n",
                     snd_strerror (err));
            exit (1);
    }
                                 
    if ((err = snd_pcm_hw_params_any (playback_handle, hw_params)) < 0) {
            fprintf (stderr, "cannot initialize hardware parameter structure (%s)\n",
                     snd_strerror (err));
            exit (1);
    }
        
    if ((err = snd_pcm_hw_params_set_access (playback_handle, hw_params, SND_PCM_ACCESS_RW_INTERLEAVED)) < 0) {
            fprintf (stderr, "cannot set access type (%s)\n",
                     snd_strerror (err));
            exit (1);
    }
        
    if ((err = snd_pcm_hw_params_set_format (playback_handle, hw_params, SND_PCM_FORMAT_S16_LE)) < 0) {
            fprintf (stderr, "cannot set sample format (%s)\n",
                     snd_strerror (err));
            exit (1);
    }

    s_rate = c->sample_rate;
    
    if ((err = snd_pcm_hw_params_set_channels_near(playback_handle, hw_params, 
	     &c->channels)) < 0) 
    {
            fprintf (stderr, "cannot set channel count (%s)\n",
                     snd_strerror (err));
            exit (1);
    }
    
    if ((err = snd_pcm_hw_params_set_rate_near(playback_handle, hw_params, &s_rate, 0)) < 0) {
            fprintf (stderr, "cannot set sample rate (%s)\n",
                     snd_strerror (err));
            exit (1);
    }

    int buffer_time = 500 * 1000; /* 500 ms */
    if ((err = snd_pcm_hw_params_set_buffer_time_near(playback_handle, hw_params,
                                                          &buffer_time, 0)) <0)
    {
            fprintf (stderr, "cannot set buffer time (%s)\n",
                     snd_strerror (err));
            exit (1);
    }

    int period_time = 50 * 1000; /* 50 ms */
    if ((err = snd_pcm_hw_params_set_period_time_near(playback_handle, hw_params,
                                                          &period_time, 0)) < 0)
    {
            fprintf (stderr, "cannot set period time (%s)\n",
                     snd_strerror (err));
            exit (1);
    }
    
    if ((err = snd_pcm_hw_params (playback_handle, hw_params)) < 0) {
            fprintf (stderr, "cannot set parameters (%s)\n",
                     snd_strerror (err));
            exit (1);
    }

    snd_pcm_hw_params_free (hw_params);
        
    if ((err = snd_pcm_prepare (playback_handle)) < 0) {
            fprintf (stderr, "cannot prepare audio interface for use (%s)\n",
                     snd_strerror (err));
            exit (1);
    }    
    fprintf(stderr, "\n");
    
    for(;;){

	if (av_read_frame(ic, &pkt) < 0)
	    break;

	size = pkt.size;
	inbuf_ptr = pkt.data;
	tns = pkt.pts/1000000LL;
	thh = tns/3600;
	tmm = (tns%3600)/60;
	tss = (tns%60);
	fprintf(stderr, "Play Time: %2d:%02d:%02d bitrate: %d kb/s\r", thh, tmm, 
		tss, c->bit_rate / 1000);

        if (size == 0)
            break;

        while (size > 0) {
            len = avcodec_decode_audio(c, (short *)outbuf, &out_size, 
                                       inbuf_ptr, size);
	    	
            if (len < 0) {
		break;
            }
	    
            if (out_size <= 0) {
		continue;
	    }
	    
            if (out_size > 0) {
		fifo_init(&f, out_size*2);
		fifo_write(&f, outbuf, out_size, &f.wptr);
		while(fifo_read(&f, s_outbuf, st_buff, &f.rptr) == 0)
		{
		    frames = snd_pcm_bytes_to_frames(playback_handle, st_buff);
            	    if((err = snd_pcm_writei (playback_handle, 
			                 (short *)s_outbuf, frames)) < 0) {
                	fprintf (stderr, "write to audio interface failed (%s)\n",
                             snd_strerror (err));
                	exit (1);
            	    }
		}
		fifo_free(&f);
            }
            size -= len;
            inbuf_ptr += len;
	        if (pkt.data)
        	    av_free_packet(&pkt);
        }

    }

    fprintf(stderr, "\n");
    snd_pcm_close (playback_handle);
    
    if(s_outbuf)
	free(s_outbuf);
    if(outbuf)
	free(outbuf);

    avcodec_close(c);
    av_close_input_file(ic);
}

int main(int argc, char *argv[])
{

    char **playlist_array;
    playlist_t *playlist;
    int items, i;
    struct stat stat_s;
    
    if(!argv[1])
    {
	fprintf(stderr,"\nUse %s file.wma\n\n", argv[0]);
	exit(0);
    }
    
    avcodec_init();

    avcodec_register_all();
    av_register_all();
    
    playlist = playlist_create();
    for (i = 1; i < argc; i++) {
    if (stat(argv[i], &stat_s) == 0) {
	if (S_ISDIR(stat_s.st_mode)) {
	    if (playlist_append_directory(playlist, argv[i]) == 0)
		fprintf(stderr, "Warning: Could not read directory %s.\n", argv[i]);
	} else {	
	    playlist_append_file(playlist, argv[i]);
	}
    } else 
	playlist_append_file(playlist, argv[i]);
    }

    if (playlist_length(playlist)) {
	playlist_array = playlist_to_array(playlist, &items);
	playlist_destroy(playlist);
	playlist = NULL;
    } else
	exit (1);
    
    i = 0;
    while ( i < items) {
	audio_decode(playlist_array[i]);
	i++;
    }

    playlist_array_destroy(playlist_array, items);
    exit (0);
}
