#ifndef __KONFORKA_RESPONSIBLE_WRAPPER_H
#define __KONFORKA_RESPONSIBLE_WRAPPER_H

/**
 * @file
 * @brief The konforka::responsible_wrapper class declaration.
 */

#include <konforka/basic_wrapper.h>

namespace konforka {

    /**
     * @brief The auto-cleanup wrapper class.
     *
     * The wrapper class that may feel responsible for releasing the resources
     * associated with the content attached.
     *
     */
    template<typename T>
	class responsible_wrapper : public basic_wrapper<T> {
	    public:
		/**
		 * The type of wrapped content.
		 */
		typedef typename basic_wrapper<T>::content_type content_type;
		/**
		 * Flag indicating whether the object feels responsible for
		 * releasing resources associated with the content.
		 */
		bool bresponsible;

		/**
		 * Default constructor creates the object with no content
		 * attached.
		 */
		responsible_wrapper()
		    : basic_wrapper<content_type>() { }
		/**
		 * Constructor, associating the content with the instance.
		 * @param o the content.
		 * @param br indicates whether resources associated with the
		 * content should be released.
		 */
		responsible_wrapper(content_type o,bool br=true)
		    : basic_wrapper<content_type>(o), bresponsible(br) { }
		/**
		 * Destructor releases resources associated with the content
		 * attached (if any), if the instance feels responsible for the
		 * content.
		 */
		virtual ~responsible_wrapper() { drop(); }

		/**
		 * Attaches the given content to the object.
		 * @param o the content.
		 * @param br indicates whether the object should feel
		 * responsible for releasing the content.
		 */
		void attach(content_type o,bool br=true) {
		    drop();
		    basic_wrapper<content_type>::attach(o);
		    bresponsible = true;
		}

		/**
		 * 'empties' object, releasing resources associated with the
		 * content if it feels responsible.
		 */
		virtual void drop() {
		    if(!this->is())
			return;
		    if(bresponsible)
			release();
		    this->bopkele = false;
		}

		/**
		 * Detaches the content from the object without releasing
		 * resources even if feels responsible for it.
		 * @return the content attached.
		 */
		virtual content_type detach() {
		    this->ensure();
		    this->bopkele = false;
		    return this->opkele;
		}

		/**
		 * Pure virtual provided for derived classes to override for
		 * doing whatever it takes to release resources associated with
		 * the content.
		 */
		virtual void release() = 0;
	};
}

#endif /* __KONFORKA_RESPONSIBLE_WRAPPER_H */
/* vim:set ft=cpp: */
