#ifndef __KONFORKA_POINTER_MAP_H
#define __KONFORKA_POINTER_MAP_H

#include <typeinfo>

/**
 * @file
 * @brief mapping of pointers.
 *
 * The support for global mapping of pointers. Useful when using third-party
 * libraries callbacks when the library does not provide mechanism for passing
 * along custom context-dependent data.
 */

namespace konforka {

    /**
     * @brief internally used actual implementation of mapping pointer.
     *
     * @param tf the typeid of the key pointer.
     * @param pf the key pointer.
     * @param tt the typeid of the value pointer.
     * @param pt the value pointer.
     */
    void _map_pointer(const type_info& tf,void *pf,const type_info& tt,void *pt);
    /**
     * @brief internally used actual implementation of destroying mapped
     * pointer.
     *
     * @param tf the typeid of the key pointer.
     * @param pf the key pointer.
     * @param tt the typeid of the value pointer.
     * @param pt the value pointer.
     */
    void _unmap_pointer(const type_info& tf,void *pf,const type_info& tt,void *pt);
    /**
     * @brief internally used actual implementation of retrieving mapped
     * pointer.
     *
     * @param tf the typeid of the key pointer.
     * @param pf the key pointer.
     * @param tt the typeid of the value pointer.
     * @return the value.
     */
    void *_mapped_pointer(const type_info& tf,void *pf,const type_info& tt);

    /**
     * @brief the object, maintaining mapped pointer.
     * 
     * @param from_t the key type.
     * @param to_t the value type.
     */
    template<typename from_t,typename to_t>
	class map_pointer {
	    public:
		/**
		 * stored key.
		 */
		from_t _from;
		/**
		 * stored value.
		 */
		to_t _to;
		/**
		 * flag, specifying that the object is currently mapped.
		 */
		bool bmapped;

		/**
		 * @brief constructs the object, mapping the key/value pair.
		 * 
		 * @param f the key.
		 * @param t the value.
		 */
		map_pointer(from_t f,to_t t)
		    : _from(f), _to(t), bmapped(false) {
			_map_pointer(typeid(from_t),_from,typeid(to_t),_to);
			bmapped=true;
		    }
		/**
		 * @brief destructor unmaps the key/value pair stored.
		 */
		~map_pointer() {
		    if(bmapped)
			_unmap_pointer(typeid(from_t),_from,typeid(to_t),_to);
		}
	};

    /**
     * The template function for pointer retrieval.
     *
     * @param from_t the key type.
     * @param to_t the value type.
     * @param f the key.
     * @return the value.
     */
    template<typename from_t,typename to_t>
	to_t mapped_pointer(from_t f) {
	    return (to_t)_mapped_pointer(typeid(from_t),f,typeid(to_t));
	}

}

#endif /* __KONFORKA_POINTER_MAP_H */
/* vim:set ft=cpp: */
