#ifndef __KONFORKA_EXCEPTION_H
#define __KONFORKA_EXCEPTION_H

#include <exception>
#include <string>
#include <list>

/**
 * @file
 * @brief exception-related stuff.
 *
 * Basic exception-related declarations and definitions.
 */

/**
 * @def CODEPOINT
 * the convenience definition of the parameters passed to the
 * konforka::code_point constructor.
 */
#define	CODEPOINT __FILE__,__PRETTY_FUNCTION__,__LINE__
/**
 * @def NOCODEPOINT
 * the convenience definition for the codepoint denoting no particular point in
 * code.
 */
#define NOCODEPOINT "no information"

/**
 * @brief The main konforka namespace.
 */
namespace konforka {
    using std::string;
    using std::list;

    /**
     * @brief Pinpoint the code context.
     *
     * Class, holding the point in code, for instance, where the exception
     * occured.
     */
    class code_point {
	public:
	    /**
	     * The string describing the point in code.
	     */
	    string where;
	    /**
	     * The file (as provided by __FILE__) if available.
	     */
	    string file;
	    /**
	     * The function name (as provided by __PRETTY_FUNCTION__) if
	     * available.
	     */
	    string function;
	    /**
	     * The line number (as provided by __LINE__) if available.
	     */
	    int line;

	    /**
	     * Constructs the object, using only textual description of the
	     * point in code (no file, function, line information).
	     * @param w the description of the point in code.
	     */
	    code_point(const string& w);
	    /**
	     * Constructs the object, specifying the exact position in code.
	     * @param fi source file name.
	     * @param fu function name.
	     * @param l the line number.
	     */
	    code_point(const string& fi,const string& fu,int l);

	    /**
	     * Extract the information on the point in code.
	     * @return the reference to the character string.
	     * @see where
	     */
	    const char *c_str() const throw();

	    /**
	     * Build the textual description from the broken down information on
	     * the point in code.
	     * @see where
	     * @see file
	     * @see function
	     * @see line
	     */
	    void make_where();
    };

    /**
     * @brief The basic exception class.
     */
    class exception : public std::exception {
	public:
	    /**
	     * The string, containing the description of exception.
	     */
	    string _what;
	    /**
	     * Reference to the point in code where the exception has occured.
	     */
	    code_point _where;
	    /**
	     * In case the exception has been rethrown a number of times, here
	     * we can trace where was it seen (a kind of backtrace).
	     */
	    list<code_point> _seen;

	    /**
	     * The simple constructor, only providing textual information on the
	     * exception nature and the point in code where the exception has
	     * occured.
	     * @param whe the point in code.
	     * @param wha the description of exception.
	     */
	    exception(const string& whe, const string& wha);
	    /**
	     * The constructor, storing elaborate information on where the
	     * exception has occured.
	     * @param fi source file name.
	     * @param fu function name.
	     * @param l line number.
	     * @param w the error message.
	     * @see #CODEPOINT
	     */
	    exception(const string& fi,const string& fu,int l,const string& w);
	    virtual ~exception() throw();

	    /**
	     * Extract the textual information on the point in code where
	     * exception has occured.
	     * @return the string describing point in code where exception has
	     * occured.
	     */
	    virtual const char* where() const throw();
	    /**
	     * Extract the textual information on the nature of the exception.
	     * @return the error message.
	     */
	    virtual const char* what() const throw();

	    /**
	     * Register the point in code (described by the string) in the
	     * 'backtrace' list.
	     * @param w the description of the point in code.
	     */
	    void see(const string& w);
	    /**
	     * Register the point in code in the 'backtrace' list.
	     * @param fi souce file name.
	     * @param fu function name.
	     * @param l line number.
	     * @see CODEPOINT
	     */
	    void see(const string& fi,const string& fu,int l);
    };

    /**
     * @brief errno-holding exception.
     *
     * The exception object storing the information provided by the errno
     * variable.
     */
    class system_error : public exception {
	public:
	    /**
	     * The value of errno.
	     */
	    int _errno;

	    /**
	     * Construct the exception object storing plain text information on
	     * the point in code.
	     * @param whe the description of point in code.
	     * @param wha the error message.
	     */
	    system_error(const string& whe,const string& wha);
	    /**
	     * Construct the exception object storing elaborate information on
	     * the point in code where it has occured.
	     * @param fi source file name.
	     * @param fu function name.
	     * @param l line number.
	     * @param w the error message.
	     * @see CODEPOINT
	     */
	    system_error(const string& fi,const string& fu,int l,const string& w);
	    virtual ~system_error() throw();

	    /**
	     * Return the information on the system error recorded.
	     * @return the string describing the error occured.
	     */
	    virtual const char* why() const throw();
    };

}

#endif /* __KONFORKA_EXCEPTION_H */
/* vim:set ft=cpp: */
