/*
 * GQradio
 * (C) 2004 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */

#include "gqradio.h"
#include "rcfile.h"

#include "io_radio.h"
#include "ui_fileops.h"

gchar *quoted_value(const gchar *text)
{
	const gchar *ptr;
	gint c = 0;
	gint l = strlen(text);

	if (l == 0) return NULL;

	while (c < l && text[c] !='"') c++;
	if (text[c] == '"')
		{
		gint e;
		c++;
		ptr = text + c;
		e = c;
		while (e < l && text[e] !='"') e++;
		if (text[e] == '"')
			{
			if (e - c > 0)
				{
				return g_strndup(ptr, e - c);
				}
			}
		}
	else
		/* for compatibility with older formats (<0.3.7)
		 * read a line without quotes too */
		{
		c = 0;
		while (c < l && text[c] !=' ' && text[c] !=8 && text[c] != '\n') c++;
		if (c != 0)
			{
			return g_strndup(text, c);
			}
		}

	return NULL;
}

void write_char_option(FILE *f, const gchar *label, const gchar *text)
{
	if (text)
		fprintf(f,"%s: \"%s\"\n", label, text);
	else
		fprintf(f,"%s: \n", label);
}

gchar *read_char_option(FILE *f, const gchar *option, const gchar *label, const gchar *value, gchar *text)
{
	if (!strcasecmp(option, label))
		{
		g_free(text);
		text = quoted_value(value);
		}
	return text;
}

void write_int_option(FILE *f, const gchar *label, gint n)
{
	fprintf(f,"%s: %d\n", label, n);
}

gint read_int_option(FILE *f, const gchar *option, const gchar *label, const gchar *value, gint n)
{
	if (!strcasecmp(option, label))
		{
		n = strtol(value,NULL,10);
		}
	return n;
}

void write_bool_option(FILE *f, const gchar *label, gint n)
{
	fprintf(f,"%s: ", label);
	if (n) fprintf(f,"true\n"); else fprintf(f,"false\n");
}

gint read_bool_option(FILE *f, const gchar *option, const gchar *label, const gchar *value, gint n)
{
	if (!strcasecmp(option, label))
		{
		if (!strcasecmp(value, "true"))
			n = TRUE;
		else
			n = FALSE;
		}
	return n;
}

void save_options(void)
{
	FILE *f;
	gchar *rc_path;
	gchar *rc_pathl;
	gchar *buf;

	rc_path = g_strconcat(homedir(), "/", GQRADIO_RC_DIR, "/", GQRADIO_RC, NULL);

	rc_pathl = path_from_utf8(rc_path);
	f = fopen(rc_pathl,"w");
	g_free(rc_pathl);
	if (!f)
		{
		printf(_("error saving config file: %s\n"), rc_path);
		g_free(rc_path);
		return;
		}

	fprintf(f,"######################################################################\n");
	fprintf(f,"#                     GQradio config file            version %7s #\n", VERSION);
	fprintf(f,"#                                                                    #\n");
	fprintf(f,"#  Everything in this file can be changed in the option dialogs.     #\n");
	fprintf(f,"#      (so there should be no need to edit this file by hand)        #\n");
	fprintf(f,"#                                                                    #\n");
	fprintf(f,"######################################################################\n");

	fprintf(f,"\n##### General Options #####\n\n");

	write_bool_option(f, "smart_window_placement", slik_smart_placement);
	write_bool_option(f, "remember_window_position", slik_remember_position);
	write_bool_option(f, "wm_decorations", wm_decorations);
	write_bool_option(f, "double_size", slik_double_size);
	fprintf(f,"\n");

	write_char_option(f, "skin_path", skin_default_path);
	write_char_option(f, "skin_mode", skin_mode_key);
	fprintf(f,"\n");

	write_char_option(f, "radio_device", radio_custom_device);
	write_int_option(f, "frequency_step", freq_step);
	fprintf(f,"\n");

	fprintf(f,"\n##### Transparency Options #####\n\n");

	write_bool_option(f, "colorshift_enable", slik_colorshift_on);
	write_int_option(f, "colorshift_red", slik_colorshift_r);
	write_int_option(f, "colorshift_green", slik_colorshift_g);
	write_int_option(f, "colorshift_blue", slik_colorshift_b);
	write_int_option(f, "colorshift_alpha", slik_colorshift_a);
	fprintf(f,"\n");
	write_bool_option(f, "transparency_force", slik_transparency_force);
	write_int_option(f, "transparency_force_alpha", slik_transparency_force_a);

	fprintf(f,"\n##### Misc Options #####\n\n");

	write_bool_option(f, "description_scrolls_always", description_scrolls_always);
	write_bool_option(f, "skinned_menus", skinned_menus_enable);
	fprintf(f,"\n");

	write_char_option(f, "mixer_command", mixer_command);
	write_int_option(f, "mixer_device", mixer_device_id);
	fprintf(f,"\n");

	write_int_option(f, "volume_boost", radio_volume_boost);
	fprintf(f,"\n");

	fprintf(f,"\n##### State #####\n\n");

	buf = radio_freq_to_path(frequency, NULL);
	write_char_option(f, "frequency", buf);
	g_free(buf);

	write_int_option(f, "mode", (gint)seek_mode);

	write_bool_option(f, "use_reported_frequency_limits", radio_limit_by_card_get());

	fprintf(f,"\n");
	fprintf(f,"######################################################################\n");
	fprintf(f,"#                     end of GQradio config file                     #\n");
	fprintf(f,"######################################################################\n");

	fclose(f);

	g_free(rc_path);
}

void load_options(void)
{
	FILE *f;
	gchar *rc_path;
	gchar *rc_pathl;
	gchar s_buf[1024];
	gchar *s_buf_ptr;
	gchar option[1024];
	gchar value[1024];
	gchar value_all[1024];
	gint c,l;
	gchar *buf;

	rc_path = g_strconcat(homedir(), "/", GQRADIO_RC_DIR, "/", GQRADIO_RC, NULL);

	rc_pathl = path_from_utf8(rc_path);
	f = fopen(rc_pathl,"r");
	g_free(rc_pathl);
	if (!f)
		{
		g_free(rc_path);
		return;
		}

	while (fgets(s_buf,1024,f))
		{
		if (s_buf[0]=='#') continue;
		if (s_buf[0]=='\n') continue;
		c = 0;
		l = strlen(s_buf);
		while (s_buf[c] != ':' && c < l) c++;
		if (c >= l) continue;
		s_buf[c] = '\0';
		c++;
		while (s_buf[c] == ' ' && c < l) c++;
		while (s_buf[c] == 8 && c < l) c++;
		while (s_buf[c] == ' ' && c < l) c++;
		s_buf_ptr = s_buf + c;
		strncpy(value_all, s_buf_ptr, sizeof(value_all));
		while (s_buf[c] != 8 && s_buf[c] != ' ' && s_buf[c] != '\n' && c < l) c++;
		s_buf[c] = '\0';
		strncpy(option, s_buf, sizeof(option));
		strncpy(value, s_buf_ptr, sizeof(value));

	/* general options */

		slik_smart_placement = read_bool_option(f, option,
			"smart_window_placement", value, slik_smart_placement);
		slik_remember_position = read_bool_option(f, option,
			"remember_window_position", value, slik_remember_position);
		wm_decorations = read_bool_option(f, option,
			"wm_decorations", value, wm_decorations);
		slik_double_size = read_bool_option(f, option,
			"double_size", value, slik_double_size);


		skin_default_path = read_char_option(f, option,
			"skin_path", value_all, skin_default_path);
		skin_mode_key = read_char_option(f, option,
			"skin_mode", value_all, skin_mode_key);

		radio_custom_device = read_char_option(f, option,
			"radio_device", value_all, radio_custom_device);

		freq_step = read_int_option(f, option,
			"frequency_step", value, freq_step);

	/* transparency */

		slik_colorshift_on = read_bool_option(f, option,
			"colorshift_enable", value, slik_colorshift_on);
		slik_colorshift_r = read_int_option(f, option,
			"colorshift_red", value, slik_colorshift_r);
		slik_colorshift_g = read_int_option(f, option,
			"colorshift_green", value, slik_colorshift_g);
		slik_colorshift_b = read_int_option(f, option,
			"colorshift_blue", value, slik_colorshift_b);
		slik_colorshift_a = read_int_option(f, option,
			"colorshift_alpha", value, slik_colorshift_a);

		slik_transparency_force = read_bool_option(f, option,
			"transparency_force", value, slik_transparency_force);
		slik_transparency_force_a = read_int_option(f, option,
			"transparency_force_alpha", value, slik_transparency_force_a);

	/* misc */

		description_scrolls_always = read_bool_option(f, option,
			"description_scrolls_always", value, description_scrolls_always);
		skinned_menus_enable = read_bool_option(f, option,
			"skinned_menus", value, skinned_menus_enable);

		mixer_command = read_char_option(f, option,
			"mixer_command", value_all, mixer_command);
		mixer_device_id = read_int_option(f, option,
			"mixer_device", value, mixer_device_id);

		radio_volume_boost = read_int_option(f, option,
			"volume_boost", value, radio_volume_boost);

	/* state */

		buf = NULL;
		buf = read_char_option(f, option,
			"frequency", value_all, buf);
		if (buf)
			{
			frequency = radio_path_to_freq(buf);
			g_free(buf);
			}

		seek_mode = (SeekMode)read_int_option(f, option,
			"mode", value, (gint)seek_mode);

		radio_limit_by_card_set(read_bool_option(f, option,
			"use_reported_frequency_limits", value, radio_limit_by_card_get()));
		}

	fclose(f);

	g_free(rc_path);
}

