/*
 *      AVL tree implementation for db.idx.
 *
 *      Copyright (c) 2005-2007 Naoaki Okazaki
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 *
 */

/* $Id: idx.c 331 2007-02-10 18:39:35Z nyaochi $ */

/*
Brief summary of db.idx structure:
- This file has 12-bytes header
- Indices consist of multiple binary search trees
- The offset addresses to the root nodes are described in db.dic

Implementation note:
- The byte order of values (such as offset address, key values, etc)
  in AVL trees depends on the CPU architecture on which this code runs.
  Since values in db.idx are written in big-endian, the functions
  from_be_avltree() and to_be_avltree() convert the byte order from/to
  big endian to the native byte-order on memory.
- The balance field of an AVL node stores values -1, 0, or 1 although
  the field stores the height of a node in db.idx. This conversion
  will take place in the function to_be_avltree().
*/

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif/*HAVE_CONFIG_H*/

#include <os.h>
#include <stdio.h>
#include <stdlib.h>
#include <memory.h>
#include <pmplib/ucs2char.h>

#include "serialize.h"
#include "util.h"
#include "ip3db.h"
#include "dic.h"
#include "dat.h"
#include "idx.h"

#define	PAGESIZE	0x00020000

#define	COMP(a, b)	((a)>(b))-((a)<(b))

struct tag_header_t {
	uint32_t	size;
	uint32_t	unknown1;
	uint32_t	unknown2;
};
typedef struct tag_header_t header_t;

struct tag_avlnode_t {
	uint32_t left;
	uint32_t right;
	int32_t balance;
	uint32_t tail;
};
typedef struct tag_avlnode_t avlnode_t;

struct tag_avltail_t {
	uint32_t data;
	uint32_t next;
};
typedef struct tag_avltail_t avltail_t;

struct tag_avl_t {
	uint8_t*	buffer;
	uint32_t	size;
	uint32_t	offset;
	uint8_t*	be_flag;
};

typedef int (*avl_comp_t)(avl_t* avl, uint32_t x, uint32_t y);

static avl_t* avl_new()
{
	return (avl_t*)calloc(1, sizeof(avl_t));
}

static void avl_finish(avl_t* avl)
{
	free(avl->buffer);
	free(avl->be_flag);
	free(avl);
}

static uint32_t avl_current(avl_t* avl)
{
	return avl->offset;
}

static uint32_t avl_allocate(avl_t* avl, uint32_t size)
{
	header_t* header = NULL;
	uint32_t base = 0, offset = 0;

	/* Expand the memory block if necessary. */
	if (avl->size < avl->offset + size) {
		uint32_t newsize = avl->size + PAGESIZE;
		avl->buffer = (uint8_t*)realloc(avl->buffer, sizeof(uint8_t) * newsize);
		memset(avl->buffer + avl->size, 0, PAGESIZE);
		avl->offset = avl->size + sizeof(header_t);
		avl->size = newsize;
	} else {
		int forward = 0;
		/* Avoid crossing a page boundary. */
		forward |= (((avl->offset + size - 1) / PAGESIZE) != (avl->offset / PAGESIZE));
		/* Avoid the region for a page header (caused by a rewind operation near a page boundary). */
		forward |= (avl->offset % PAGESIZE < sizeof(header_t));
		if (forward) {
			uint32_t page = (avl->offset + size) / PAGESIZE;
			avl->offset = PAGESIZE * page + sizeof(header_t);
		}
	}

	/* Allocate a memory block. */
	offset = avl->offset;
	avl->offset += size;

	/* Store the size of the current page. */
	base = avl->size - PAGESIZE;
	header = (header_t*)(avl->buffer + base);
	header->size = avl->offset - base;

	return offset;
}

static avlnode_t *avlnode(avl_t* avl, uint32_t offset)
{
	return (avlnode_t*)(avl->buffer + offset);
}

static void avl_rewind(avl_t* avl, uint32_t offset)
{
	memset(avlnode(avl, offset), 0, avl->offset - offset);
	//if (avl->offset / PAGESIZE == offset / PAGESIZE) {
		avl->offset = offset;
	/*} else {
		uint32_t page = avl->offset / PAGESIZE;
		avl->offset = PAGESIZE * page + sizeof(header_t);
	}*/
}

static uint32_t avlnode_new(avl_t* avl, size_t keysize)
{
	uint32_t offset = avl_allocate(avl, sizeof(avlnode_t) + keysize);
	avlnode_t* a = avlnode(avl, offset);
	a->left = 0;
	a->right = 0;
	a->balance = 0;
	a->tail = 0;
	return offset;
}

static uint32_t avltail_new(avl_t* avl)
{
	uint32_t offset = avl_allocate(avl, sizeof(avltail_t));
	avltail_t* a = (avltail_t*)avlnode(avl, offset);
	a->data = 0;
	a->next = 0;
	return offset;
}

static void avl_swl(avl_t* avl, uint32_t *offr)
{
	uint32_t offa = *offr;
	avlnode_t* a = avlnode(avl, offa);
	uint32_t offb = a->right;
	avlnode_t* b = avlnode(avl, offb);
	*offr = offb;
	a->right = b->left;
	b->left = offa;
}

static void avl_swr(avl_t* avl, uint32_t *offr)
{
	uint32_t offa = *offr;
	avlnode_t* a = avlnode(avl, offa);
	uint32_t offb = a->left;
	avlnode_t* b = avlnode(avl, offb);
	*offr = offb;
	a->left = b->right;
	b->right = offa;
}

static void avl_nasty(avl_t* avl, uint32_t offr)
{
	avlnode_t* root = avlnode(avl, offr);
	avlnode_t* left = avlnode(avl, root->left);
	avlnode_t* right = avlnode(avl, root->right);
	switch (root->balance) {
	case -1:
		left->balance = 0;
		right->balance = 1;
		break;
	case 0:
		left->balance = 0;
		right->balance = 0;
		break;
	case 1:
		left->balance = -1;
		right->balance = 0;
		break;
	}
	root->balance = 0;
}

static int avl_insert(avl_t* avl, uint32_t* offr, uint32_t* offa, avl_comp_t comp)
{
	avlnode_t* root = NULL;
	avlnode_t* left = NULL;
	avlnode_t* right = NULL;

	if (!*offr) {
		/* An empty tree: set the current node as the root node. */
		*offr = *offa;
		return 1;
	} else {
		uint32_t x = *offr + sizeof(avlnode_t);
		uint32_t y = *offa + sizeof(avlnode_t);
		int cmp = comp(avl, x, y);
		if (cmp > 0) {
			/* Insert the node to the left sub-tree. */
			root = avlnode(avl, *offr);
			if (root->left) {
				uint32_t off_left = root->left;
				int ret = avl_insert(avl, &off_left, offa, comp);
				if (ret > 0) {
					switch (root->balance--) {
					case 1: return 0;
					case 0: return 1;
					}
					left = avlnode(avl, root->left);
					if (left->balance < 0) {
						avl_swr(avl, offr);
						root = avlnode(avl, *offr);
						right = avlnode(avl, root->right);
						root->balance = 0;
						right->balance = 0;
					} else {
						avl_swl(avl, &root->left);
						avl_swr(avl, offr);
						avl_nasty(avl, *offr);
					}
				} else if (ret == 0) {
					root->left = off_left;
				} else {
					return -1;
				}
				return 0;
			} else {
				root->left = *offa;
				if (root->balance--) return 0;
				return 1;
			}
		} else if (cmp < 0) {
			/* Insert the node to the right sub-tree. */
			root = avlnode(avl, *offr);
			if (root->right) {
				uint32_t off_right = root->right;
				int ret = avl_insert(avl, &off_right, offa, comp);
				if (ret > 0) {
					switch (root->balance++) {
					case -1: return 0;
					case 0: return 1;
					}
					right = avlnode(avl, root->right);
					if (right->balance > 0) {
						avl_swl(avl, offr);
						root = avlnode(avl, *offr);
						left = avlnode(avl, root->left);
						root->balance = 0;
						left->balance = 0;
					} else {
						avl_swr(avl, &root->right);
						avl_swl(avl, offr);
						avl_nasty(avl, *offr);
					}
				} else if (ret == 0) {
					root->right = off_right;
				} else {
					return -1;
				}
				return 0;
			} else {
				root->right = *offa;
				if (root->balance++) return 0;
				return 1;
			}
		} else {
			*offa = *offr;
			return -1;
		}
	}
}

static int avl_comp_byte(avl_t* avl, uint32_t _x, uint32_t _y)
{
	uint8_t* x = avl->buffer + _x;
	uint8_t* y = avl->buffer + _y;
	return COMP(*x, *y);
}

static int avl_comp_word(avl_t* avl, uint32_t _x, uint32_t _y)
{
	uint16_t* x = (uint16_t*)(avl->buffer + _x);
	uint16_t* y = (uint16_t*)(avl->buffer + _y);
	return COMP(*x, *y);
}

static int avl_comp_dword(avl_t* avl, uint32_t _x, uint32_t _y)
{
	uint32_t* x = (uint32_t*)(avl->buffer + _x);
	uint32_t* y = (uint32_t*)(avl->buffer + _y);
	return COMP(*x, *y);
}

static const ucs2char_t* skip_prefix(const ucs2char_t* str)
{
	int i = 0;
	static const ucs2char_t ucs2cs_a[] = {'a',' ',0};
	static const ucs2char_t ucs2cs_an[] = {'a','n',' ',0};
	static const ucs2char_t ucs2cs_the[] = {'t','h','e',' ',0};
	static const ucs2char_t *prefix[] = {
		ucs2cs_a, ucs2cs_an, ucs2cs_the, NULL,
	};

	while (prefix[i]) {
		if (ucs2incmp(str, prefix[i], ucs2len(prefix[i])) == 0) {
			return str + ucs2len(prefix[i]);
		}
		++i;
	}
	return str;
}

static int avl_comp_ucs2string_nocase(avl_t* avl, uint32_t _x, uint32_t _y)
{
	ucs2char_t a, b;
	const ucs2char_t* x = (const ucs2char_t*)(avl->buffer + _x);
	const ucs2char_t* y = (const ucs2char_t*)(avl->buffer + _y);

	x = skip_prefix(x);
	y = skip_prefix(y);

	do {
		a = ucs2upper(*x);
		b = ucs2upper(*y);
		if (!*x || !*y) {
			break;
		}
		x++;
		y++;
	} while (a == b);
	return COMP(a, b);
}

static int avl_comp_ucs2string(avl_t* avl, uint32_t _x, uint32_t _y)
{
	/* Compare strings, ignoring case. */
	int ret = avl_comp_ucs2string_nocase(avl, _x, _y);
	if (ret == 0) {
		/* Compare strings with case. */
		ucs2char_t a, b;
		const ucs2char_t* x = (const ucs2char_t*)(avl->buffer + _x);
		const ucs2char_t* y = (const ucs2char_t*)(avl->buffer + _y);

		x = skip_prefix(x);
		y = skip_prefix(y);

		do {
			a = *x;
			b = *y;
			if (!*x || !*y) {
				break;
			}
			x++;
			y++;
		} while (a == b);
		return COMP(a, b);
	} else {
		return ret;
	}
}

static int avl_insert_key(avl_t* avl, const ip3db_variant_t* key, int type, uint32_t* offset, uint32_t* root)
{
	int ret = 0;
	static avl_comp_t comp[] = {NULL, avl_comp_ucs2string, avl_comp_byte, avl_comp_word, avl_comp_dword};
	uint32_t offset_prev = avl_current(avl);
	uint32_t offset_this = 0;
	uint32_t offset_key = 0;

	switch (type) {
	case IP3DBVT_BYTE:
		offset_this = avlnode_new(avl, sizeof(uint8_t));
		offset_key = offset_this + sizeof(avlnode_t);
		*((uint8_t*)avlnode(avl, offset_key)) = key->value.byte;
		break;
	case IP3DBVT_WORD:
		offset_this = avlnode_new(avl, sizeof(uint16_t));
		offset_key = offset_this + sizeof(avlnode_t);
		*((uint16_t*)avlnode(avl, offset_key)) = key->value.word;
		break;
	case IP3DBVT_DWORD:
		offset_this = avlnode_new(avl, sizeof(uint32_t));
		offset_key = offset_this + sizeof(avlnode_t);
		*((uint32_t*)avlnode(avl, offset_key)) = key->value.dword;
		break;
	case IP3DBVT_STRING:
		offset_this = avlnode_new(avl, sizeof(ucs2char_t) * (ucs2len(key->value.str) + 1));
		offset_key = offset_this + sizeof(avlnode_t);
		ucs2cpy((ucs2char_t*)avlnode(avl, offset_key), key->value.str);
		break;
	}

	ret = avl_insert(avl, root, &offset_this, comp[type]);
	if (ret != -1) {
		/* A new key. */
		*offset = offset_this;
		return 0;
	} else {
		/* The key exists in the AVL tree. */
		avl_rewind(avl, offset_prev);
		*offset = offset_this;
		return 1;
	}
}



static void from_uint16be(uint16_t* value)
{
	uint8_t* src = (uint8_t*)value;
	uint16_t tmp = (uint16_t)src[0] << 8 | (uint16_t)src[1];
	*value = tmp;
}

static void to_uint16be(uint16_t* value)
{
	uint16_t tmp = *value;
	uint8_t* dst = (uint8_t*)value;
	dst[0] = (uint8_t)(tmp >> 8);
	dst[1] = (uint8_t)(tmp & 0xFF);
}

static void from_uint32be(uint32_t* value)
{
	uint8_t* src = (uint8_t*)value;
	uint32_t tmp = (uint32_t)src[0] << 24 | (uint32_t)src[1] << 16 | (uint32_t)src[2] << 8 | (uint32_t)src[3];
	*value = tmp;
}

static void to_uint32be(uint32_t* value)
{
	uint32_t tmp = *value;
	uint8_t* dst = (uint8_t*)value;
	dst[0] = (uint8_t)(tmp >> 24);
	dst[1] = (uint8_t)(tmp >> 16);
	dst[2] = (uint8_t)(tmp >> 8);
	dst[3] = (uint8_t)(tmp & 0xFF);
}

static void from_ucs2be_string(ucs2char_t* value)
{
	while (*value) {
		from_uint16be((uint16_t*)value);
		value++;
	}
}

static void to_ucs2be_string(ucs2char_t* value)
{
	while (*value) {
		to_uint16be((uint16_t*)value);
		value++;
	}
}

static void from_be_avltree(avl_t* avl, uint32_t offset, dic_table_t* dic_table, int index, int level)
{
	avlnode_t* node = avlnode(avl, offset);
	int field = dic_table->indices[index].fields[level]; 
	int type = dic_table->fields[field].type;

	/* Skip this node if already converted to the native byte-order. */
	if (avl->be_flag[offset]) {
		return;
	}

	/* This node is converted to the native byte-order. */
	avl->be_flag[offset] = 1;

	/* Convert the current node. */
	from_uint32be(&node->left);
	from_uint32be(&node->right);
	from_uint32be(&node->balance);
	from_uint32be(&node->tail);

	/* Descend to the left node. */
	if (node->left) {
		from_be_avltree(avl, node->left, dic_table, index, level);
	}

	/* Convert the key value. */
	switch (type) {
	case IP3DBVT_BYTE:
		break;
	case IP3DBVT_WORD:
		from_uint16be((uint16_t*)(node+1));
		break;
	case IP3DBVT_DWORD:
		from_uint32be((uint32_t*)(node+1));
		break;
	case IP3DBVT_STRING:
		from_ucs2be_string((ucs2char_t*)(node+1));
		break;
	}

	/* Convert the sub AVL trees or tail. */
	if (node->tail) {
		if (level+1 < IP3DBIDX_MAX_KEYLEVEL && dic_table->indices[index].fields[level+1] != -1) {
			/* Convert the AVL tree in the next level. */
			from_be_avltree(avl, node->tail, dic_table, index, level+1);
		} else {
			/* Convert the tail. */
			uint32_t tail_offset = node->tail;
			while (!avl->be_flag[tail_offset]) {
				avltail_t* tail = (avltail_t*)avlnode(avl, tail_offset);
				from_uint32be(&tail->next);
				from_uint32be(&tail->data);
				avl->be_flag[tail_offset] = 1;	/* Mark this tail. */
				if (!tail->next) {
					break;
				}
				tail_offset = tail->next;
			}
		}
	}

	/* Descend to the right node. */
	if (node->right) {
		from_be_avltree(avl, node->right, dic_table, index, level);
	}
}

static uint32_t to_be_avltree(avl_t* avl, uint32_t offset, dic_table_t* dic_table, int index, int level)
{
	avlnode_t* node = avlnode(avl, offset);
	int field = dic_table->indices[index].fields[level]; 
	int type = dic_table->fields[field].type;
	uint32_t height = 1;

	/* Descend to the left node. */
	if (node->left) {
		uint32_t new_height = to_be_avltree(avl, node->left, dic_table, index, level) + 1;
		if (height < new_height) {
			height = new_height;
		}
	}

	/* Descend to the right node. */
	if (node->right) {
		uint32_t new_height = to_be_avltree(avl, node->right, dic_table, index, level) + 1;
		if (height < new_height) {
			height = new_height;
		}
	}

	/* Overwrite the balance field with the height of the current node. */
	node->balance = height;

	/* Convert the key value. */
	switch (type) {
	case IP3DBVT_BYTE:
		break;
	case IP3DBVT_WORD:
		to_uint16be((uint16_t*)(node+1));
		break;
	case IP3DBVT_DWORD:
		to_uint32be((uint32_t*)(node+1));
		break;
	case IP3DBVT_STRING:
		to_ucs2be_string((ucs2char_t*)(node+1));
		break;
	}

	/* Convert the sub AVL trees or tail. */
	if (node->tail) {
		if (level+1 < IP3DBIDX_MAX_KEYLEVEL && dic_table->indices[index].fields[level+1] != -1) {
			/* Convert the AVL tree in the next level. */
			to_be_avltree(avl, node->tail, dic_table, index, level+1);
		} else {
			/* Convert the tail. */
			avltail_t* tbe = (avltail_t*)avlnode(avl, node->tail);
			avltail_t tail = *tbe;
			for (;;) {
				from_uint32be(&tbe->next);
				from_uint32be(&tbe->data);
				if (!tail.next) {
					break;
				}
				tbe = (avltail_t*)avlnode(avl, tail.next);
				tail = *tbe;
			}
		}
	}

	/* Convert the current node. */
	to_uint32be(&node->left);
	to_uint32be(&node->right);
	to_uint32be(&node->balance);
	to_uint32be(&node->tail);
	return height;
}




static void fprinti(FILE *fp, int n)
{
	while (n--)	fputc(' ', fp);
}

static void avl_dump(avl_t* avl, uint32_t offset, dic_table_t* dic_table, int index, int level, FILE *fpo)
{
	avlnode_t* node = avlnode(avl, offset);
	int field = dic_table->indices[index].fields[level]; 
	int type = dic_table->fields[field].type;
	int indent = 2 * (level + 1);

	/* Descend to the left node. */
	if (node->left) {
		avl_dump(avl, node->left, dic_table, index, level, fpo);
	}

	/* Dump the current node. */
	fprinti(fpo, indent);
	fprintf(fpo, "NODE (0x%08X)\n", offset);
	fprinti(fpo, indent);
	fprintf(fpo, "  left:  0x%08X\n", node->left);
	fprinti(fpo, indent);
	fprintf(fpo, "  right: 0x%08X\n", node->right);
	fprinti(fpo, indent);
	fprintf(fpo, "  height:  0x%08X\n", node->balance);
	fprinti(fpo, indent);
	fprintf(fpo, "  tail:  0x%08X\n", node->tail);
	fprinti(fpo, indent);
	fprintf(fpo, "  key: ");
	switch (type) {
	case IP3DBVT_BYTE:
		fprintf(fpo, "0x%02X", *(uint8_t*)(node+1));
		break;
	case IP3DBVT_WORD:
		fprintf(fpo, "0x%04X", *(uint16_t*)(node+1));
		break;
	case IP3DBVT_DWORD:
		fprintf(fpo, "0x%08X", *(uint32_t*)(node+1));
		break;
	case IP3DBVT_STRING:
		fprints(fpo, "%s", (ucs2char_t*)(node+1));
		break;
	}
	fprintf(fpo, "\n");

	/* Convert the sub AVL trees or tail. */
	if (node->tail) {
		if (level+1 < IP3DBIDX_MAX_KEYLEVEL && dic_table->indices[index].fields[level+1] != -1) {
			/* Convert the AVL tree in the next level. */
			avl_dump(avl, node->tail, dic_table, index, level+1, fpo);
		} else {
			/* Convert the tail. */
			avltail_t* tail = (avltail_t*)avlnode(avl, node->tail);

			fprinti(fpo, indent);
			fprintf(fpo, "  TAIL: [\n");
			for (;;) {
				fprinti(fpo, indent);
				fprintf(fpo, "    0x%08X\n", tail->data);
				if (!tail->next) {
					break;
				}
				tail = (avltail_t*)avlnode(avl, tail->next);
			}
			fprinti(fpo, indent);
			fprintf(fpo, "  ]\n", node->left);
		}
	}

	/* Descend to the right node. */
	if (node->right) {
		avl_dump(avl, node->right, dic_table, index, level, fpo);
	}
}



typedef struct {
	uint32_t	size;
	uint32_t	unknown1;
	uint32_t	unknown2;
} idx_header_t;

idx_t *idx_new()
{
	idx_t* idx = (idx_t*)calloc(1, sizeof(idx_t));
	idx->avl = avl_new();
	return idx;
}

void idx_finish(idx_t* idx)
{
	avl_finish(idx->avl);
	free(idx);
}

int idx_read(idx_t* idx, dic_t* dic, FILE *fpi)
{
	int i;
	uint32_t offset = 0;
	long size = 0;

	/* Read the whole data at a time. */
	fread_all(fpi, &idx->avl->buffer, &size);
	idx->avl->size = size;

	/* Convert the byte order of the header from big endian to the native one. */
	while (offset < idx->avl->size) {
		idx_header_t* header = (idx_header_t*)(idx->avl->buffer + offset);
		from_uint32be(&header->size);
		from_uint32be(&header->unknown1);
		from_uint32be(&header->unknown2);
		offset += PAGESIZE;
	}

	/*
	 * Allocate memory block for indicating the status of byte-order conversion.
	 *	This is really ugly, but we had to prevent a node from being converted twice.
	 *	At first, I didn't think this treatment is necessary, but I found a node
	 *	referred by multiple indices in a database generated by my E10 player.
	 */
	idx->avl->be_flag = calloc(idx->avl->size, 1);

	/* Convert the byte order of values in AVL trees. */
	for (i = 0;i < dic->music.num_indices;++i) {
		uint32_t idx_root = dic_get_idxroot(dic, IP3DBIDX_MUSIC, i);
		if (idx_root) {
			from_be_avltree(idx->avl, idx_root, &dic->music, i, 0);
		}
	}
	for (i = 0;i < dic->references.num_indices;++i) {
		uint32_t idx_root = dic_get_idxroot(dic, IP3DBIDX_REFERENCES, i);
		if (idx_root) {
			from_be_avltree(idx->avl, idx_root, &dic->references, i, 0);
		}
	}
	for (i = 0;i < dic->objects.num_indices;++i) {
		uint32_t idx_root = dic_get_idxroot(dic, IP3DBIDX_OBJECTS, i);
		if (idx_root) {
			from_be_avltree(idx->avl, idx_root, &dic->objects, i, 0);
		}
	}

	/* Free the memory block for big-endian flags. */
	free(idx->avl->be_flag);
	idx->avl->be_flag = NULL;
	return 0;
}

result_t idx_write(idx_t* idx, dic_t* dic, FILE *fpo)
{
	int i;
	uint32_t offset = 0;

	/* Convert the byte order of the header to big endian from the native one. */
	while (offset < idx->avl->size) {
		idx_header_t* header = (idx_header_t*)(idx->avl->buffer + offset);
		to_uint32be(&header->size);
		to_uint32be(&header->unknown1);
		to_uint32be(&header->unknown2);
		offset += PAGESIZE;
	}

	/* Convert the byte order of values in AVL trees. */
	for (i = 0;i < dic->music.num_indices;++i) {
		uint32_t idx_root = dic_get_idxroot(dic, IP3DBIDX_MUSIC, i);
		if (idx_root) {
			to_be_avltree(idx->avl, idx_root, &dic->music, i, 0);
		}
	}
	for (i = 0;i < dic->references.num_indices;++i) {
		uint32_t idx_root = dic_get_idxroot(dic, IP3DBIDX_REFERENCES, i);
		if (idx_root) {
			to_be_avltree(idx->avl, idx_root, &dic->references, i, 0);
		}
	}
	for (i = 0;i < dic->objects.num_indices;++i) {
		uint32_t idx_root = dic_get_idxroot(dic, IP3DBIDX_OBJECTS, i);
		if (idx_root) {
			to_be_avltree(idx->avl, idx_root, &dic->objects, i, 0);
		}
	}

	/* Read the whole data at a time. */
	if (fwrite(idx->avl->buffer, 1, idx->avl->size, fpo) != idx->avl->size) {
		return 1;
	}

	dic->header.num_idx_pages = (idx->avl->size / PAGESIZE);
	return 0;
}

result_t idx_dump(idx_t* idx, dic_t* dic, FILE *fpo)
{
	int i;

	fprintf(fpo, "===== db.idx =====\n");

	/* Dump the binary search trees. */
	for (i = 0;i < dic->music.num_indices;++i) {
		uint32_t idx_root = dic_get_idxroot(dic, IP3DBIDX_MUSIC, i);
		fprintf(fpo, "[");
		dic_repr_index(dic, IP3DBIDX_MUSIC, i, fpo);
		fprintf(fpo, "]\n");
		if (idx_root) {
			avl_dump(idx->avl, idx_root, &dic->music, i, 0, fpo);
		}
	}
	for (i = 0;i < dic->references.num_indices;++i) {
		uint32_t idx_root = dic_get_idxroot(dic, IP3DBIDX_REFERENCES, i);
		fprintf(fpo, "[");
		dic_repr_index(dic, IP3DBIDX_REFERENCES, i, fpo);
		fprintf(fpo, "]\n");
		if (idx_root) {
			avl_dump(idx->avl, idx_root, &dic->references, i, 0, fpo);
		}
	}
	for (i = 0;i < dic->objects.num_indices;++i) {
		uint32_t idx_root = dic_get_idxroot(dic, IP3DBIDX_OBJECTS, i);
		fprintf(fpo, "[");
		dic_repr_index(dic, IP3DBIDX_OBJECTS, i, fpo);
		fprintf(fpo, "]\n");
		if (idx_root) {
			avl_dump(idx->avl, idx_root, &dic->objects, i, 0, fpo);
		}
	}

	fprintf(fpo, "\n");
	return 0;
}

static void idx_insert_dat(idx_t* idx, dic_table_t* dictbl, uint32_t* root, dat_entry_t* entry, int index, int level)
{
	uint32_t offset = 0;
	int field = dictbl->indices[index].fields[level]; 
	int type = dictbl->fields[field].type;
	int ret = avl_insert_key(idx->avl, &entry->fields[field], type, &offset, root);
	if (level+1 < IP3DBIDX_MAX_KEYLEVEL && dictbl->indices[index].fields[level+1] != -1) {
		/*	The following code does not work since we cannot guarantee that node->tail
		 *	is valid after idx_insert_dat() call, which may move the memory block to
		 *	expand it (realloc).
		 *		avlnode_t* node = avlnode(idx->avl, offset);
		 *		idx_insert_dat(idx, dictbl, &node->tail, entry, index, level+1);
		 *  Therefore, we need to update node->tail after idx_insert_dat() call.
		 */
		avlnode_t* node = avlnode(idx->avl, offset);
		uint32_t child = node->tail;
		idx_insert_dat(idx, dictbl, &child, entry, index, level+1);
		node = avlnode(idx->avl, offset);
		node->tail = child;
	} else {
		/*	The same here. We must obtain the pointer to the node after avltail_new() call.
		 */
		uint32_t offset_newelem = avltail_new(idx->avl);
		avlnode_t* node = avlnode(idx->avl, offset);
		avltail_t* newelem = (avltail_t*)avlnode(idx->avl, offset_newelem);
		if (node->tail) {
			/* Append to the end of the linked list. */
			avltail_t* elem = (avltail_t*)avlnode(idx->avl, node->tail);
			while (elem->next) {
				elem = (avltail_t*)avlnode(idx->avl, elem->next);
			}
			elem->next = offset_newelem;
			newelem->data = entry->offset;
			newelem->next = 0;
		} else {
			/* The first element in the linked list. */
			node->tail = offset_newelem;
			newelem->data = entry->offset;
			newelem->next = 0;
		}
	}
}

static void idx_construct_index(idx_t* idx, dic_table_t* dictbl, dat_list_t* list, uint32_t* root, int index)
{
	uint32_t i;

	for (i = 0;i < list->num_entries;++i) {
		dat_entry_t* entry = &list->entries[i];
		idx_insert_dat(idx, dictbl, root, entry, index, 0);
	}
}

result_t idx_construct(idx_t* idx, dic_t* dic, dat_t* dat)
{
	int index;

	avl_finish(idx->avl);
	idx->avl = avl_new();

	for (index = 0;index < dic->music.num_indices;++index) {
		uint32_t root = 0;
		idx_construct_index(idx, &dic->music, &dat->musics, &root, index);
		dic_set_idxroot(dic, IP3DBIDX_MUSIC, index, root);
	}

	for (index = 0;index < dic->references.num_indices;++index) {
		uint32_t root = 0;
		idx_construct_index(idx, &dic->references, &dat->references, &root, index);
		dic_set_idxroot(dic, IP3DBIDX_REFERENCES, index, root);
	}

	for (index = 0;index < dic->objects.num_indices;++index) {
		uint32_t root = 0;
		idx_construct_index(idx, &dic->objects, &dat->objects, &root, index);
		dic_set_idxroot(dic, IP3DBIDX_OBJECTS, index, root);
	}

	return 0;
}
