/*
 *      High level interface for idx (e.g., U10.idx).
 *
 *      Copyright (c) 2005-2007 Naoaki Okazaki
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 *
 */

/* $Id: ip2db_idx.c 328 2007-02-10 17:50:11Z nyaochi $ */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif/*HAVE_CONFIG_H*/
#ifdef	HAVE_STRING_H
#include <string.h>
#endif/*HAVE_STRING_H*/

#include <os.h>
#include <stdio.h>
#include <stdlib.h>
#include <pmplib/ucs2char.h>

#include "util.h"
#include "serialize.h"
#include "ip2db.h"

static result_t construct_nodes(ip2db_t* db, ip2db_record_t* records, uint32_t num_records)
{
	uint32_t type;
	sortitem_t* sortitems = calloc(num_records, sizeof(sortitem_t));
	idx_key_t* idxkeys = (idx_key_t*)calloc(num_records, sizeof(idx_key_t));
	static uint32_t nums_children[] = {
		0, 0, 0, 127, 42, 42, 42, 42, 42, 25, 25, 18, 25, 84, 84, 84, 84,
	};
	typedef int (*qsort_comp_t)(const void *, const void *);

	if (!sortitems) {
		free(idxkeys);
		free(sortitems);
		return PMPERR_INSUFFICIENTMEMORY;
	}
	if (!idxkeys) {
		free(idxkeys);
		free(sortitems);
		return PMPERR_INSUFFICIENTMEMORY;
	}

	for (type = IP2DBIDX_PAGE_NODE_ENTRYNUMBER;type <= IP2DBIDX_PAGE_NODE_FORMAT;++type) {
		uint8_t *p = NULL, *q = NULL;
		uint8_t height = 0;
		uint16_t num_children = 0;
		uint32_t i, j, n;
		uint32_t max_children = nums_children[type];
		uint32_t begin = 0, end = 0, next = 0, dup = 0;
		idx_export_t* idxexp = &g_idx_exports[type];

		/* Initialize idxkeys. */
		memset(idxkeys, 0, sizeof(idx_key_t) * num_records);

		/* Sort indices according to the node type. */
		for (i = 0;i < num_records;++i) {
			sortitems[i].records = records;
			sortitems[i].index = i;
		}
		qsort(
			sortitems,
			num_records,
			sizeof(sortitem_t),
			(qsort_comp_t)idxexp->comp
			);

		/* Convert record_t elements into idxkey elements. */
		for (i = 0;i < num_records;++i) {
			j = sortitems[i].index;
			idxexp->record_to_key(&idxkeys[i], &records[j]);
		}

		/* Check the duplicated keys and set dup fields. */
		for (i = 1;i < num_records;++i) {
			switch (type) {
			case IP2DBIDX_PAGE_NODE_ENTRYNUMBER:
			case IP2DBIDX_PAGE_NODE_RATING:
			case IP2DBIDX_PAGE_NODE_PLAYCOUNT:
			case IP2DBIDX_PAGE_NODE_RECENTPLAY:
			case IP2DBIDX_PAGE_NODE_FORMAT:
				if (idxkeys[i-1].data.dword == idxkeys[i].data.dword) {
					idxkeys[i].dup = idxkeys[i-1].dup + 1;
				}
				break;
			case IP2DBIDX_PAGE_NODE_FILENAME:
			case IP2DBIDX_PAGE_NODE_TITLE:
			case IP2DBIDX_PAGE_NODE_ARTIST:
			case IP2DBIDX_PAGE_NODE_ALBUM:
			case IP2DBIDX_PAGE_NODE_GENRE:
				if (ucs2cmp_fixed(idxkeys[i-1].data.str, idxkeys[i].data.str, 8) == 0) {
					idxkeys[i].dup = idxkeys[i-1].dup + 1;
				}
				break;
			case IP2DBIDX_PAGE_NODE_GENRE_ARTIST:
			case IP2DBIDX_PAGE_NODE_GENRE_ALBUM:
			case IP2DBIDX_PAGE_NODE_ARTIST_ALBUM:
				if (ucs2cmp_fixed(idxkeys[i-1].data.str, idxkeys[i].data.str, 16) == 0) {
					idxkeys[i].dup = idxkeys[i-1].dup + 1;
				}
				break;
			case IP2DBIDX_PAGE_NODE_GENRE_ARTIST_ALBUM:
				if (ucs2cmp_fixed(idxkeys[i-1].data.str, idxkeys[i].data.str, 24) == 0) {
					idxkeys[i].dup = idxkeys[i-1].dup + 1;
				}
				break;
			}
		}

		/*
		 * If all indices can be stored in one node (page), the procedure will be
		 * pretty easy: writing pointers to leaves in the root node at @type page.
		 */
		if (num_records < max_children) {
			p = PAGEBLOCK(db->idx_buffer, type);
			for (i = 0;i < num_records;++i) {
				j = i % max_children;
				next = sortitems[i].index + 1;

				// Write a child and increment the number of children.
				idxexp->setget_child(p, j, &idxkeys[i], &next, 1);
				idxexp->setget_num_childlen(p, &num_children, 0);
				++num_children;
				idxexp->setget_num_childlen(p, &num_children, 1);
			}
			// Skip the rest of the processing.
			continue;
		}

		/*
		 * If the number of indices is beyond the capacity of one node (page),
		 * arrange them using multiple pages. We store the beginning page (begin)
		 * and the ending page (end) for generating nodes whose height is higher
		 * than 0.
		 */
		begin = ip2db_get_num_pages(db) + 1;
		for (i = 0;i < num_records;++i) {
			j = i % max_children;
			next = sortitems[i].index + 1;

			// Create a new node page if necessary (i.e., every max_children).
			if (j == 0) {
				end = ip2dbidx_add_page(db, type);
				p = PAGEBLOCK(db->idx_buffer, end);
				idxnode_init(p);
				idxnode_setget_height(p, &height, 1);
			}

			// Write a child and increment the number of children.
			idxexp->setget_child(p, j, &idxkeys[i], &next, 1);
			idxexp->setget_num_childlen(p, &num_children, 0);
			++num_children;
			idxexp->setget_num_childlen(p, &num_children, 1);
		}

		/*
		 * If the condition (begin < end) is true, nodes with the same height exist
		 * across multiple pages. Generate (height+1) nodes by extracting the last
		 * child in each (height+0) nodes. Loop this process until (height+1) node
		 * can be stored in one page, in other words, the root node is generated.
		 * Note that the key of the last child over (height+0) must be treated as
		 * 0x7FFFFFFF (DWORD) or \uFFFF string. Make sure that the root node to be
		 * stored at @type page.
		 */
		while (begin < end) {
			idx_key_t idxkey;
			n = end - begin + 1;	// The overall number of children in (height+0) nodes.
			height++;				// Increment the height.

			if (n < max_children) {
				// We are going to write the root node.
				p = PAGEBLOCK(db->idx_buffer, type);
				idxnode_setget_height(p, &height, 1);

				for (i = begin;i <= end;++i) {
					j = (i-begin) % max_children;

					memset(&idxkey, 0, sizeof(idxkey));
					if (i < end) {
						// Retrieve the last child in node i.
						q = PAGEBLOCK(db->idx_buffer, i);
						idxexp->setget_num_childlen(q, &num_children, 0);
						idxexp->setget_child(q, num_children-1, &idxkey, &next, 0);
					} else {
						// Change the key of the very last child into a terminator.
						switch (type) {
						case IP2DBIDX_PAGE_NODE_ENTRYNUMBER:
						case IP2DBIDX_PAGE_NODE_RATING:
						case IP2DBIDX_PAGE_NODE_PLAYCOUNT:
						case IP2DBIDX_PAGE_NODE_RECENTPLAY:
						case IP2DBIDX_PAGE_NODE_FORMAT:
							idxkey.type = type;
							idxkey.dup = 0;
							idxkey.data.dword = 0x7FFFFFFF;
							break;
						default:
							idxkey.type = type;
							idxkey.dup = 0;
							memset(idxkey.data.str, 0xFF, 24*2);
							break;
						}
					}
					// Write a child (pointer to i) and increment the number of children.
					idxexp->setget_child(p, j, &idxkey, &i, 1);
					idxexp->setget_num_childlen(p, &num_children, 0);
					++num_children;
					idxexp->setget_num_childlen(p, &num_children, 1);
				}
				begin = end = type;
			} else {
				// We are going to write an intermediate node.
				uint32_t next_begin = ip2db_get_num_pages(db) + 1;
				uint32_t next_end = next_begin;

				for (i = begin;i <= end;++i) {
					j = (i-begin) % max_children;

					// Create a new node page if necessary (i.e., every max_children).
					if (j == 0) {
						next_end = ip2dbidx_add_page(db, type);
						p = PAGEBLOCK(db->idx_buffer, next_end);
						idxnode_init(p);
						idxnode_setget_height(p, &height, 1);
					}

					memset(&idxkey, 0, sizeof(idxkey));
					if (i < end) {
						// Retrieve the last child in node i.
						q = PAGEBLOCK(db->idx_buffer, i);
						idxexp->setget_num_childlen(q, &num_children, 0);
						idxexp->setget_child(q, num_children-1, &idxkey, &next, 0);
					} else {
						// Change the key of the very last child into a terminator.
						switch (type) {
						case IP2DBIDX_PAGE_NODE_ENTRYNUMBER:
						case IP2DBIDX_PAGE_NODE_RATING:
						case IP2DBIDX_PAGE_NODE_PLAYCOUNT:
						case IP2DBIDX_PAGE_NODE_RECENTPLAY:
						case IP2DBIDX_PAGE_NODE_FORMAT:
							idxkey.type = type;
							idxkey.dup = 0;
							idxkey.data.dword = 0x7FFFFFFF;
							break;
						default:
							idxkey.type = type;
							idxkey.dup = 0;
							memset(idxkey.data.str, 0xFF, 24*2);
							break;
						}
					}
					// Write a child (pointer to i) and increment the number of children.
					idxexp->setget_child(p, j, &idxkey, &i, 1);
					idxexp->setget_num_childlen(p, &num_children, 0);
					++num_children;
					idxexp->setget_num_childlen(p, &num_children, 1);
				}

				// Update the range of nodes with the height.
				begin = next_begin;
				end = next_end;
			}
		}

	}

	free(idxkeys);
	free(sortitems);
	return 0;
}

static result_t insert_leaf(ip2db_t* db, ip2db_record_t* record, dat_t* dat)
{
	uint8_t *p = NULL;
	int i;
	uint16_t record_size = 0, required_size = 0, min_size_remaining = 0xFFFF, size = 0;
	uint32_t leaftail[NUM_LEAF_CHAIN_LISTS], page = 0;
	ip2db_idxleaf_data_t data;
	offset_size_t offset_size;
	uint16_t n;

	data.pathname = record->pathname;
	data.filename = record->filename;
	data.title = record->title;
	data.artist = record->artist;
	data.album = record->album;
	data.genre = record->genre;

	idxleaf_set_field_access(&data, dat->idx_item_field_access);

	/* Calculate the necessary size for a leaf to store the record. */
	record_size  = dat->idx_item_field_access[IP2DBDAT_IDX_GENRE].offset;
	record_size += dat->idx_item_field_access[IP2DBDAT_IDX_GENRE].size;
	record_size += sizeof(ucs2char_t);					/* \u0000 of the last field. */
	required_size = record_size + sizeof(uint16_t) * 2;	/* space for offset and size. */

	/* Obtain tails of the leaves. */
	idxheader_setget_leaftail(db->idx_buffer, leaftail, 0);

	/* Obtain the most latter tail. */
	page = 0;
	for (i = 0;i < NUM_LEAF_CHAIN_LISTS;++i) {
		if (page < leaftail[i]) {
			page = leaftail[i];
		}
	}
	
	/* Check if the tail leaf has enough space to store the record. */
	if (page > 0) {
		uint16_t size_avail = 0;
		idxleaf_setget_size_avail(PAGEBLOCK(db->idx_buffer, page), &size_avail, 0);
		size_avail -= 0x17;
		if (size_avail < required_size) {
			page = 0;	/* No enough space in leaftail. */
		}
	}

	/* Insert a new leaf page if the tail leaf doesn't have enough space. */
	if (page == 0) {
		uint8_t leafline;

		page = ip2dbidx_add_page(db, IP2DBIDX_PAGE_LEAF);
		leafline = page % NUM_LEAF_CHAIN_LISTS;
		p = PAGEBLOCK(db->idx_buffer, page);

		idxleaf_init(p);
		idxleaf_setget_leaf_line(p, &leafline, 1);
		if (IP2DBIDX_PAGE_LEAF <= page - NUM_LEAF_CHAIN_LISTS) {
			uint32_t prev_page = page - NUM_LEAF_CHAIN_LISTS;
			uint8_t *q = PAGEBLOCK(db->idx_buffer, prev_page);
			idxleaf_setget_leaf_next(q, &page, 1);
			idxleaf_setget_leaf_prev(p, &prev_page, 1);
		}

		leaftail[leafline] = page;
		idxheader_setget_leaftail(db->idx_buffer, leaftail, 1);
	} else {
		p = PAGEBLOCK(db->idx_buffer, page);
	}

	/* */
	idxleaf_setget_num_data(p, &n, 0);
	if (n == 0) {
		offset_size.offset = 0x17;
		offset_size.size = record_size;
	} else {
		idxleaf_setget_data_offset_size(p, n-1, &offset_size, 0);
		offset_size.offset += offset_size.size;
		offset_size.size = record_size;
	}

	i = n++;
	idxleaf_setget_num_data(p, &n, 1);
	idxleaf_setget_data_offset_size(p, i, &offset_size, 1);
	idxleaf_setget_size_avail(p, &size, 0);
	size -= required_size;
	idxleaf_setget_size_avail(p, &size, 1);
	idxleaf_setget_size_used(p, &size, 0);
	size += record_size;
	idxleaf_setget_size_used(p, &size, 1);
	
	idxleaf_setget_data(p, i, dat->idx_item_field_access, &data, 1);

	/* Set fields in dat_t. */
	dat->idx_item_page = page;
	dat->idx_item_index = i;
	dat->idx_item_size = record_size;

	return 0;
}

static result_t construct_leaves(ip2db_t* db, ip2db_record_t* records, uint32_t num_records)
{
	uint32_t i;

	for (i = 0;i < num_records;++i) {
		insert_leaf(db, &records[i], &db->dat_array[i]);
	}

	return 0;
}

static result_t read_nodes(idxpage_t* pages, int type, uint8_t* buffer, uint32_t offset)
{
	uint8_t* p = &buffer[offset];
	uint32_t pn = offset / 0x400;
	uint8_t height = 0;
	uint16_t num_children = 0;

	pages[pn].offset = offset;
	pages[pn].type = type;

	height = g_idx_exports[type].get_height(p);
	g_idx_exports[type].setget_num_childlen(p, &num_children, 0);

	if (height > 0) {
		uint8_t i;
		for (i = 0;i < num_children;++i) {
			idx_key_t key;
			uint32_t child = 0;

			memset(&key, 0, sizeof(key));

			g_idx_exports[type].setget_child(p, i, &key, &child, 0);

			read_nodes(
				pages,
				type,
				buffer,
				(child-1) * 0x400
				);
		}
	}
	return 0;
}

result_t ip2dbidx_read(ip2db_t* db, FILE *fp)
{
	uint32_t i, num_pages;
	idxpage_t* pages = NULL;
	uint8_t *buffer = NULL;
	long filesize;

	/* Read idx file. */
	fread_all(fp, &buffer, &filesize);
	db->idx_buffer = buffer;
	db->idx_size = filesize;

	/* Read the number of pages. */
	idxheader_setget_num_pages(buffer, &num_pages, 0);

	/* Allocate pages. */
	pages = (idxpage_t*)malloc(sizeof(idxpage_t) * num_pages);
	memset(pages, 0, sizeof(idxpage_t) * num_pages);

	/* Read the header page (again just for simplicity). */
	pages[0].offset = 0x0000;
	pages[0].type = IP2DBIDX_PAGE_HEADER;

	/* Read the descriptor page (not implemented yet). */
	pages[1].offset = 0x0400;
	pages[1].type = IP2DBIDX_PAGE_DESCRIPTOR;

	/* Read various kinds of B-tree nodes from root nodes. */
	read_nodes(pages, IP2DBIDX_PAGE_NODE_ENTRYNUMBER, buffer, 0x0800);
	read_nodes(pages, IP2DBIDX_PAGE_NODE_FILENAME, buffer, 0x0C00);
	read_nodes(pages, IP2DBIDX_PAGE_NODE_TITLE, buffer, 0x1000);
	read_nodes(pages, IP2DBIDX_PAGE_NODE_ARTIST, buffer, 0x1400);
	read_nodes(pages, IP2DBIDX_PAGE_NODE_ALBUM, buffer, 0x1800);
	read_nodes(pages, IP2DBIDX_PAGE_NODE_GENRE, buffer, 0x1C00);
	read_nodes(pages, IP2DBIDX_PAGE_NODE_GENRE_ARTIST, buffer, 0x2000);
	read_nodes(pages, IP2DBIDX_PAGE_NODE_GENRE_ALBUM, buffer, 0x2400);
	read_nodes(pages, IP2DBIDX_PAGE_NODE_GENRE_ARTIST_ALBUM, buffer, 0x2800);
	read_nodes(pages, IP2DBIDX_PAGE_NODE_ARTIST_ALBUM, buffer, 0x2C00);
	read_nodes(pages, IP2DBIDX_PAGE_NODE_RATING, buffer, 0x3000);
	read_nodes(pages, IP2DBIDX_PAGE_NODE_PLAYCOUNT, buffer, 0x3400);
	read_nodes(pages, IP2DBIDX_PAGE_NODE_RECENTPLAY, buffer, 0x3800);
	read_nodes(pages, IP2DBIDX_PAGE_NODE_FORMAT, buffer, 0x3C00);

	for (i = 0;i < num_pages;++i) {
		/* Assume unread pages to be leaves. */
		if (pages[i].type == IP2DBIDX_PAGE_NONE) {
			pages[i].offset = 0x0400 * i;
			pages[i].type = IP2DBIDX_PAGE_LEAF;
		}
	}

	db->idx_pages = pages;
	return 0;
}

result_t ip2dbidx_write(ip2db_t* db, FILE *fp)
{
	/* Write idx. */
	fwrite(db->idx_buffer, PAGESIZE, ip2db_get_num_pages(db), fp);
	return 0;
}

uint32_t ip2dbidx_add_page(ip2db_t* db, int type)
{
	uint32_t num_pages;

	idxheader_setget_num_pages(db->idx_buffer, &num_pages, 0);

	db->idx_buffer = (uint8_t*)realloc(db->idx_buffer, PAGESIZE * (num_pages+1));
	db->idx_pages = (idxpage_t*)realloc(db->idx_pages, sizeof(idxpage_t) * (num_pages+1));
	db->idx_pages[num_pages].offset = PAGESIZE * num_pages;
	db->idx_pages[num_pages].type = type;

	++num_pages;
	memset(PAGEBLOCK(db->idx_buffer, num_pages), 0, PAGESIZE);
	idxheader_setget_num_pages(db->idx_buffer, &num_pages, 1);

	return num_pages;
}

result_t ip2dbidx_construct(ip2db_t* db, ip2db_record_t* records, uint32_t num_records)
{
	result_t ret = 0;

	// Construct leaves.
	ret = construct_leaves(db, records, num_records);
	if (ret) {
		return ret;
	}

	// Construct indices.
	ret = construct_nodes(db, records, num_records);
	if (ret) {
		return ret;
	}

	return ret;
}
