/*
 *      EasyPMP file enumeration routines.
 *
 *      Copyright (c) 2005-2007 Naoaki Okazaki
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA, or visit
 * http://www.gnu.org/copyleft/gpl.html .
 *
 */

/* $Id: enumerate.c 328 2007-02-10 17:50:11Z nyaochi $ */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif/*HAVE_CONFIG_H*/
#ifdef	HAVE_STRING_H
#include <string.h>
#endif/*HAVE_STRING_H*/

#include <os.h>
#include <stdio.h>
#include <stdlib.h>
#include <pmplib/ucs2char.h>
#include <pmplib/filepath.h>
#include <playlist.h>
#include <pmplib/pmp.h>

#include <easypmp.h>

typedef struct {
	const option_t* opt;
	easypmp_filelist_t* fl;
	pmp_t* pmp;
	easypmp_enumerate_progress_t proc;
	void *instance;
} enumerate_dat_t;

static int found_music_file(void *instance, const ucs2char_t* found_path, const ucs2char_t* found_file)
{
	uint32_t i;
	enumerate_dat_t* ed = (enumerate_dat_t*)instance;
	pmp_t* pmp = ed->pmp;
	pmp_music_t* pmp_music = ed->pmp->music;
	easypmp_filelist_t* fl = ed->fl;

	// Check if the file has an extension supported by the target player.
	for (i = 0;i < pmp->info.num_audio_extensions;++i) {
		if (filepath_hasext(found_file, pmp->info.audio_extensions[i])) {
			break;
		}
	}

	if (i != pmp->info.num_audio_extensions) {
		// Supported music file.
		easypmp_filename_t* new_filename = NULL;

		// Expand the target array.
		fl->elements = (easypmp_filename_t*)realloc(
			fl->elements,
			sizeof(easypmp_filename_t) * (fl->num_elements+1)
			);
		if (!fl->elements) {
			return -1;
		}

		// Set path and file.
		new_filename = &fl->elements[fl->num_elements++];
		ucs2cpy(new_filename->pathname, found_path);
		ucs2cpy(new_filename->filename, found_file);
	} else {
		// Exit if the filename does not have a supported extension.
		return 0;
	}

	// Report the progress.
	if (ed->proc) {
		ed->proc(ed->instance, found_path, found_file, fl->num_elements);
	}

	return 0;
}

int
easypmp_enumerate_music(
	easypmp_filelist_t* fl,
	pmp_t* pmp,
	const option_t* opt,
	easypmp_enumerate_progress_t proc,
	void *instance
	)
{
	enumerate_dat_t ed;
	ucs2char_t music_path[MAX_PATH];

	// Decode the music path prefix for system path separators
	filepath_combinepath(music_path, MAX_PATH, pmp->info.path_to_root, pmp->info.path_to_music);
	filepath_addslash(music_path);
	filepath_decode(music_path);

	fl->num_elements = 0;
	fl->elements = NULL;

	memset(&ed, 0, sizeof(ed));
	ed.opt = opt;
	ed.fl = fl;
	ed.pmp = pmp;
	ed.proc = proc;
	ed.instance = instance;

	return find_file(
		music_path,
		pmp->info.music_flag & PMPMF_RECURSIVE ? 1 : 0,
		found_music_file,
		&ed
		);
}

static int found_playlist_file(void *instance, const ucs2char_t* found_path, const ucs2char_t* found_file)
{
	enumerate_dat_t* ed = (enumerate_dat_t*)instance;
	pmp_t* pmp = ed->pmp;
	easypmp_filelist_t* fl = ed->fl;
	int flag = (ed->opt->verb & MODE_PLAYLIST_JSPL) ? PLAYLIST_JSPL : 0;

	if (playlist_is_supported(found_file, flag)) {
		// Supported music file.
		easypmp_filename_t* new_filename = NULL;

		// Expand the target array.
		fl->elements = (easypmp_filename_t*)realloc(
			fl->elements,
			sizeof(easypmp_filename_t) * (fl->num_elements+1)
			);
		if (!fl->elements) {
			return -1;
		}

		// Set path and file.
		new_filename = &fl->elements[fl->num_elements++];
		ucs2cpy(new_filename->pathname, found_path);
		ucs2cpy(new_filename->filename, found_file);
	} else {
		// Exit if the filename does not have a supported extension.
		return 0;
	}

	// Report the progress.
	if (ed->proc) {
		ed->proc(ed->instance, found_path, found_file, fl->num_elements);
	}

	return 0;
}

int
easypmp_enumerate_playlist(
	easypmp_filelist_t* fl,
	pmp_t* pmp,
	const option_t* opt,
	easypmp_enumerate_progress_t proc,
	void *instance
	)
{
	int ret = 0;
	enumerate_dat_t ed;
	ucs2char_t path[MAX_PATH];

	fl->num_elements = 0;
	fl->elements = NULL;

	memset(&ed, 0, sizeof(ed));
	ed.opt = opt;
	ed.fl = fl;
	ed.pmp = pmp;
	ed.proc = proc;
	ed.instance = instance;

	// Decode the playlist path prefix for system path separators
	if (opt->path_to_playlist_source) {
		ucs2cpy(path, opt->path_to_playlist_source);
	} else {
		filepath_combinepath(path, MAX_PATH, pmp->info.path_to_root, pmp->info.path_to_playlist);
	}
	filepath_addslash(path);
	filepath_decode(path);
	ret = find_file(
		path,
		opt->subdir_playlist_source,
		found_playlist_file,
		&ed
		);
	return ret;
}

void
easypmp_free_filelist(
	easypmp_filelist_t* fl
	)
{
	free(fl->elements);
	memset(fl, 0, sizeof(*fl));
}
