/*
 *      EasyPMP database construction.
 *
 *      Copyright (c) 2005-2007 Naoaki Okazaki
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA, or visit
 * http://www.gnu.org/copyleft/gpl.html .
 *
 */

/* $Id: database.c 328 2007-02-10 17:50:11Z nyaochi $ */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif/*HAVE_CONFIG_H*/
#ifdef	HAVE_STRING_H
#include <string.h>
#endif/*HAVE_STRING_H*/

#include <os.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <pmplib/ucs2char.h>
#include <pmplib/filepath.h>
#include <pmplib/pmp.h>
#include <gmi.h>
#include <easypmp.h>

#ifdef	_WIN32
#define COMP_STR(x, y)	lstrcmpiW(x, y)	/* FAT32 treats upper/lower letters as identical. */
#else
#define COMP_STR(x, y)	ucs2cmp(x, y)
#endif

static uint32_t get_current_time()
{
	return (uint32_t)time(NULL);
}

static int comp_filename(const void *_x, const void *_y)
{
	const pmp_music_record_t *x = (const pmp_music_record_t*)_x;
	const pmp_music_record_t *y = (const pmp_music_record_t*)_y;
	return COMP_STR(x->filename, y->filename);
}

static int find_mediafile(
	pmp_music_record_t* records,
	int num_records,
	const ucs2char_t* filename,
	int *begin,
	int *end
	)
{
	int low = 0, high = num_records-1;

	/* Binary search. */
	while (low <= high) {
		int middle = (low + high) / 2;
		int comp = COMP_STR(filename, records[middle].filename);
		if (comp == 0) {
			/* Found */
			*begin = *end = middle;
			while (*begin >= 0 && COMP_STR(filename, records[*begin].filename) == 0) {
				(*begin)--;
			}
			(*begin)++;
			while (*end < num_records && COMP_STR(filename, records[*end].filename) == 0) {
				(*end)++;
			}
			return 1;
		} else if (comp < 0) {
			high = middle - 1;
		} else {
			low = middle + 1;
		}
	}
	return 0;
}



int easypmp_database(
	pmp_t* pmp,
	const easypmp_filelist_t* fl,
	const option_t* opt,
	pmp_music_record_t** ptr_records,
	uint32_t* ptr_num_records,
	easypmp_progress_t progress,
	void *instance
	)
{
	int result = 0;
	uint32_t i;
	result_t res = 0;
	ucs2char_t music_path[MAX_PATH];
	pmp_music_t* pmpdb = pmp->music;
	pmp_music_record_t* records = NULL;
	pmp_music_record_t* old_records = NULL;
	uint32_t num_old_records = 0, num_obtained = 0;
	uint32_t ts_import = get_current_time();

	filepath_combinepath(music_path, MAX_PATH, pmp->info.path_to_root, pmp->info.path_to_music);

	/*
	 * Read the existing database for update processing mode.
	 */
	if (opt->verb & MODE_DATABASE_UPDATE) {
		// Obtain the number of records in the database.
		res = pmpdb->get_records(pmpdb, NULL, &num_old_records);
		if (res != 0) {
			result = MAKE_PMP_ERROR(res);
			goto error_exit;
		}

		// Allocate an array for the records.
		old_records = malloc(sizeof(pmp_music_record_t) * num_old_records);
		if (!old_records) {
			result = EASYPMPE_INSUFFICIENT_MEMORY;
			goto error_exit;
		}

		// Obtain the records from the database.
		res = pmpdb->get_records(pmpdb, old_records, &num_old_records);
		if (res != 0) {
			result = MAKE_PMP_ERROR(res);
			goto error_exit;
		}

		// Sort the records for binary search.
		qsort(old_records, num_old_records, sizeof(pmp_music_record_t), comp_filename);
	}

	/*
	 * Allocate an array for the new records.
	 */
	records = calloc(fl->num_elements, sizeof(pmp_music_record_t));
	if (!records) {
		result = EASYPMPE_INSUFFICIENT_MEMORY;
		goto error_exit;
	}

	/*
	 * Obtain music information.
	 */
	if (progress(instance, EASYPMPP_MUSIC_GMI | EASYPMPSP_START, fl->num_elements, 0, NULL) != 0) {
		result = EASYPMPE_CANCEL;
		goto error_exit;
	}

	for (i = 0;i < fl->num_elements;++i) {
		int is_skipping = 0;
		easypmp_filename_t* target = &fl->elements[i];
		pmp_music_record_t* record = &records[i];
		ucs2char_t filename[MAX_PATH];
		uint32_t timestamp = 0;
		
		filepath_combinepath(filename, MAX_PATH, target->pathname, target->filename);
		timestamp = (uint32_t)filepath_mtime(filename);
		
		// Report the progress.
		if (progress(instance, EASYPMPP_MUSIC_GMI | EASYPMPSP_PROGRESS, i, 0, filename) != 0) {
			result = EASYPMPE_CANCEL;
			goto error_exit;
		}

		// Initialize the record.
		pmplib_record_init(record);
		
		// Determine if we can skip obtaining a media information from this file.
		if (opt->verb & MODE_DATABASE_UPDATE) {
			// Find this file from the existing records.
			int begin = 0, end = 0;
			if (find_mediafile(old_records, num_old_records, filename, &begin, &end)) {
				if (begin + 1 == end) {
					// If the time stamps are identical, let us skip this file.
					if (old_records[begin].ts_update == timestamp) {
						pmplib_record_clone(record, &old_records[begin]);
						is_skipping = 1;
					}
				}
			}
		}

		// If we are not skipping this file, obtain the media information.
		if (!is_skipping) {
			if (gmi_get(
				record,
				filename,
				music_path,
				opt->media_info_source,
				opt->music_encoding,
				opt->strip_words,
				opt->num_strip_words
				) == 0) {
				++num_obtained;
			} else {
				// Report the error.
				if (progress(instance, EASYPMPP_MUSIC_GMI | EASYPMPSP_WARN_FAILURE, i, 0, filename) != 0) {
					result = EASYPMPE_CANCEL;
					goto error_exit;
				}
			}
			record->ts_update = timestamp;
			record->ts_import = ts_import;
		}

		// Warn empty artist or album name.
		if (!record->artist || !*record->artist || !record->album || !*record->album) {
			if (progress(instance, EASYPMPP_MUSIC_GMI | EASYPMPSP_WARN_EMPTY, i, 0, filename) != 0) {
				result = EASYPMPE_CANCEL;
				goto error_exit;
			}
		}
	}
	if (progress(instance, EASYPMPP_MUSIC_GMI | EASYPMPSP_END, num_obtained, 0, NULL) != 0) {
		result = EASYPMPE_CANCEL;
		goto error_exit;
	}

	res = pmpdb->set_records(pmpdb, records, fl->num_elements);
	if (res != 0) {
		result = MAKE_PMP_ERROR(res);
		goto error_exit;
	}
	if (progress(instance, EASYPMPP_MUSIC_UPDATE | EASYPMPSP_END, 0, 0, NULL) != 0) {
		result = EASYPMPE_CANCEL;
		goto error_exit;
	}

error_exit:
	/*
	 * Free allocated memory.
	 */
	if (old_records) {
		for (i = 0;i < num_old_records;++i) {
			pmplib_record_finish(&old_records[i]);
		}
		free(old_records);
	}
	if (records) {
		/* If ptr_records and ptr_num_records are specified, keep the record array. */
		if (ptr_records && ptr_num_records) {
			*ptr_records = records;
			*ptr_num_records = fl->num_elements;
		} else {
			/* Free the record array. */
			for (i = 0;i < fl->num_elements;++i) {
				pmplib_record_finish(&records[i]);
			}
			free(records);
		}
	}

	return result;
}



int database_dump(pmp_t* pmp, FILE *fp, int level)
{
	pmp_music_t* music = pmp->music;
	if (!music) {
		return 1;
	}
	music->dump(music, fp, level);
	return 0;
}



int easypmp_set_strip_words(option_t* opt, const ucs2char_t* str)
{
	size_t i;
	const ucs2char_t *p = 0, *q = 0;

	/*
	 * This function parses a list of strip words specified by comma separated
	 * values (CSV) and stores them into the option_t instance. Escape by quatation
	 * mark, eg., "word, " is not allowed at this moment.
	 */

	/* Free buffers allocated already. */
	for (i = 0;i < opt->num_strip_words;++i) {
		ucs2free(opt->strip_words[i]);
	}
	ucs2free(opt->strip_words);

	if (str && *str) {
		/* Count the number of strip words in str. */
		opt->num_strip_words = 1;
		for (p = str;*p;++p) {
			if (*p == ',') {
				opt->num_strip_words++;
			}
		}

		/* Allocate an array to store strip words. */
		opt->strip_words = (ucs2char_t**)ucs2malloc(sizeof(ucs2char_t*) * opt->num_strip_words);
		if (!opt->strip_words) {
			return -1;
		}

		/* Parse the list of strip words. */
		p = q = str;
		for (i = 0;i < opt->num_strip_words;++i) {
			opt->strip_words[i] = 0;
			for (;;) {
				if (*q == ',' || !*q) {
					size_t length = (size_t)(q - p);
					opt->strip_words[i] = (ucs2char_t*)ucs2malloc(sizeof(ucs2char_t) * (length+1));
					memset(opt->strip_words[i], 0, sizeof(ucs2char_t) * (length+1));
					ucs2ncpy(opt->strip_words[i], p, length);
					opt->strip_words[i][length] = 0;
					if (*q) {
						p = q+1;
					}
					break;
				}
				q++;
			}
		}
	}
	return 0;
}
