/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2010 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#define _POSIX_SOURCE /* struct sigaction */
#define _POSIX_C_SOURCE 200809L /* getline() */
#define _GNU_SOURCE /* getline() */

#include <stdio.h> /* getline() */
#include <signal.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h> /* access() */
#include <errno.h>

/* wait() */
#include <sys/types.h>
#include <sys/wait.h>

#include "cdw_sys.h"
#include "gettext.h"
#include "cdw_debug.h"
#include "cdw_string.h"
#include "cdw_cdio.h"

/**
   \file cdw/src/utilities/cdw_sys.c

   File with various functions interfacing operating system.
   Currently this file defines functions:
   \li accessing /proc file system,
   \li accessing /lib/modules/ directory,
   \li handling signals.
*/


/* data structures and functions related to handling signals */
static void cdw_sys_sigsegv_sa_handler(int signal_number);
static void cdw_sys_sigchild_sa_handler(__attribute__((unused)) int signal_number);
static struct sigaction cdw_sigsegv_action;
static struct sigaction cdw_sigchild_action;
static sig_atomic_t child_exit_status;

static cdw_rv_t cdw_sys_check_file_system_support_via_proc_fs(cdio_fs_t cdio_fs);
static cdw_rv_t cdw_sys_check_file_system_support_via_modules(cdio_fs_t cdio_fs);
static char *   cdw_sys_get_osrelease(void);
static char *   cdw_sys_get_kernel_modules_location(void);




/**
  \brief Remove zombie process created by run_command()

  \param signal_number - number of handled signal, unused
*/
void cdw_sys_sigchild_sa_handler(__attribute__((unused)) int signal_number)
{
	/* clean child process */
	int status;
	wait(&status);
	child_exit_status = status;

	return;
}





void cdw_sys_sigsegv_sa_handler(int signal_number)
{
	cdw_assert (signal_number == SIGSEGV, "ERROR: wrong signal to handle\n");
	/* 2TRANS: this is a message printed to console when cdw needs to
	   close unexpectedly. "signal" is a signal from operating system;
	   "SIGSEGV" is a signal's name; keep leading and ending "\n";
	   use "\n" in the middle of message as needed to keep line
	   lengths no more than 76 characters */
	fprintf(stderr, _("\ncdw received SIGSEGV signal and it needed to clean up after itself\nand leave. This means that there is a bug in cdw. Don't be mad...\n\n"));

	/* there was a SIGSEGV, so let's better leave now;
	   exit() triggers functions registered with atexit(); these
	   are <module_name>_clean() functions */
	exit(EXIT_FAILURE);
}





int cdw_sys_signal_handlers_init(void)
{
	/* set up code dealing with zombie processes created during run_command() call */
	cdw_vdm ("INFO: setting SIGCHILD handler\n");
	memset(&cdw_sigchild_action, 0, sizeof(cdw_sigchild_action));
	cdw_sigchild_action.sa_handler = &cdw_sys_sigchild_sa_handler;
	int rv = sigaction(SIGCHLD, &cdw_sigchild_action, (struct sigaction *) NULL);
	if (rv == -1) {
		cdw_vdm ("ERROR: failed call to sigaction(SIGCHILD, ...)\n");
		return -1;
	}

	/* set up code dealing with signals sent when program does invalid memory reference */
	cdw_vdm ("INFO: setting SIGSEGV handler\n");
	memset(&cdw_sigsegv_action, 0, sizeof(cdw_sigsegv_action));
	cdw_sigsegv_action.sa_handler = cdw_sys_sigsegv_sa_handler;
	rv = sigaction(SIGSEGV, &cdw_sigsegv_action, (struct sigaction *) NULL);
	if (rv == -1) {
		cdw_vdm ("ERROR: failed call to sigaction(SIGSEGV, ...)\n");
		return -1;
	}

	return 0;
}





cdw_rv_t cdw_sys_check_file_system_support(cdio_fs_t cdio_fs)
{
	cdw_assert (cdw_cdio_is_fs_iso(cdio_fs) || cdio_fs == CDIO_FS_UDF,
		    "ERROR: unrecognized fs: %d\n", cdio_fs);
	if (cdw_cdio_is_fs_iso(cdio_fs)) {
		/* there may be few subtypes of ISO9660 fs,
		   let's simplify this for code using cdio_fs;
		   CDIO_FS_ISO_UDF also counts as ISO9660 */
		cdio_fs = CDIO_FS_ISO_9660;
	}
#if 0
	/* force loading module for iso9660, so that it
	   becomes visible in /proc/filesystems */
	/* this works, but only if you are root */
	int rv = mount("/dev/scd0", "/media/cdrom0",
		       "iso9660", MS_MGC_VAL | MS_RDONLY | MS_NOSUID | MS_NOATIME, "");

#endif

	/* first let's try the polite way, searching /proc/filesystems;
	   if support for some file system is provided via kernel
	   module, and the module hasn't been loaded yet, it won't
	   be present in /proc/filesystems;
	   if it is compiled into kernel then it should be there */
	cdw_rv_t crv1 = cdw_sys_check_file_system_support_via_proc_fs(cdio_fs);
	if (crv1 == CDW_OK) {
		/* operating system supports given file system type */
		return CDW_OK;
	} else {
		/* either there is no information about given file system
		   in /proc/filesystems file (crv == CDW_NO), there is no
		   such file (crv == CDW_CANCEL), or ther was some error when
		   trying to access /proc/filesystems (crv == CDW_GEN_ERROR);
		   let's try different method */
		cdw_rv_t crv2 = cdw_sys_check_file_system_support_via_modules(cdio_fs);
		if (crv2 == CDW_OK) {
			/* operating system supports given file system
			   type (support is provided by kernel module) */
			return CDW_OK;
		} else if (crv2 == CDW_NO) {
			/* operating system does not support given file
			   system type */
			return CDW_NO;
		} else if (crv2 == CDW_CANCEL) {
			/* cdw can't tell if there is a support for given
			   file system type or not */
			return CDW_CANCEL;
		} else { /* CDW_GEN_ERROR */
			if (crv1 == CDW_CANCEL) {
				/* we got here because first method didn't
				   give definitive answer, so there is still
				   a chance that operating system supports
				   given file system type; "cancel" means
				   here "we don't know, but you may try" */
				return CDW_CANCEL;
			} else { /* crv1 == CDW_GEN_ERROR */
				/* two methods returned CDW_GEN_ERROR,
				   there is something very wrong;
				   assume that given file system is not
				   supported */
				return CDW_GEN_ERROR;
			}
		}
	}
}


cdw_rv_t cdw_sys_check_file_system_support_via_proc_fs(cdio_fs_t cdio_fs)
{
	FILE *fp = fopen("/proc/filesystems", "r");
	int e = errno;
	if (fp == (FILE *) NULL) {
		cdw_vdm ("ERROR: failed to open /proc/filesystems, error = \"%s\"\n", strerror(e));
		if (e == ENOENT) {
			/* incorrect assumption that all GNU/Linux
			   systems have similar proc file systems,
			   with /proc/filesystems file present */
			return CDW_CANCEL;
		} else {
			/* some other error */
			return CDW_GEN_ERROR;
		}
	}

	char *line = (char *) NULL;
	size_t len = 0;
	ssize_t r = 0;
	cdw_rv_t retval = CDW_NO; /* "no" = "support for file system not found (yet)" */
	while ((r = getline(&line, &len, fp)) != -1) {
		if (line[r - 1] == '\n') {
			line[r - 1] = '\0';
		}
		cdw_sdm ("INFO: line = \"%s\"\n", line);
		/* only basic file system types: iso9660 and udf */
		if (cdio_fs == CDIO_FS_ISO_9660 && strstr(line, "iso9660")) {
			cdw_vdm ("INFO: detected support for iso9660 in /proc/filesystems\n");
			retval = CDW_OK;
			break;

		} else if (cdio_fs == CDIO_FS_UDF && strstr(line, "udf")) {
			cdw_vdm ("INFO: detected support for udf in /proc/filesystems\n");
			retval = CDW_OK;
			break;
		} else {
			;
		}
#ifndef NDEBUG
		if (strstr(line, "ext3")) {
			cdw_vdm ("INFO: detected support for ext3 in /proc/filesystems\n");
		}
#endif
	}

	free(line);
	line = (char *) NULL;
	fclose(fp);
	fp = (FILE *) NULL;

	if (retval == CDW_OK) {
		/* support for given file system was found */
		return CDW_OK;
	} else if (retval == CDW_NO) {
		/* procedure of checking went without problems,
		   but no support for given file system was found */
		return CDW_NO;
	} else {
		/* some error? */
		return CDW_GEN_ERROR;
	}
}





cdw_rv_t cdw_sys_check_file_system_support_via_modules(cdio_fs_t cdio_fs)
{
	/* support for some file systems may be provided by kernel modules,
	   which (most of the time?) aren't loaded until they are needed;
	   they should be available in a specific place, let's hope that
	   it's the same place in all GNU/Linux distributions */

	/* let's check if our assumption about all modules'
	   location is correct */

	char *modules_location = cdw_sys_get_kernel_modules_location();
	if (modules_location == (char *) NULL) {
		cdw_vdm ("ERROR: can't get modules location path\n");
		return CDW_GEN_ERROR;
	}

	/* at this point we know that "/lib/modules/<os release>/kernel"
	   location is valid, let's check if there are modules implementing
	   support for specific file systems */
	char *path = (char *) NULL;
	if (cdio_fs == CDIO_FS_ISO_9660) {
		path = cdw_string_concat(modules_location, "/fs/isofs/isofs.ko", (char *) NULL);
	} else if (cdio_fs == CDIO_FS_UDF) {
		path = cdw_string_concat(modules_location, "/fs/udf/udf.ko", (char *) NULL);
	} else {
		;
	}
	free(modules_location);
	modules_location = (char *) NULL;

	if (path == (char *) NULL) {
		cdw_vdm ("ERROR: failed to concat path\n");
		return CDW_GEN_ERROR;
	}
	cdw_vdm ("INFO: inspecting path \"%s\"\n", path);
	int rv = access(path, F_OK);
	int e = errno;
	free(path);
	path = (char *) NULL;
	if (rv == 0) {
		cdw_vdm ("INFO: found \"%s\" module\n", cdio_fs == CDIO_FS_ISO_9660 ? "isofs" : "udf");
		return CDW_OK;
	} else {
		if (e == ENOENT) {
			/* no kernel module file, no support
			   for given file system */
			cdw_vdm ("INFO: can't find \"%s\" module, error = \"%s\"\n",
				 cdio_fs == CDIO_FS_ISO_9660 ? "isofs" : "udf",
				 strerror(e));
			return CDW_NO;
		} else {
			cdw_vdm ("ERROR: can't access \"%s\" module, error = \"%s\"\n",
				 cdio_fs == CDIO_FS_ISO_9660 ? "isofs" : "udf",
				 strerror(e));
			return CDW_GEN_ERROR;
		}
	}
}




char *cdw_sys_get_osrelease(void)
{
	/* osrelease file stores string that is the same as
	   result of "uname -r", e.g. "2.6.32F" */
	FILE *fp = fopen("/proc/sys/kernel/osrelease", "r");
	int e = errno;
	if (fp == (FILE *) NULL) {
		cdw_vdm ("ERROR: failed to open /proc/sys/kernel/osrelease, error = \"%s\"\n", strerror(e));
		if (e == ENOENT) {
			/* incorrect assumption that all GNU/Linux
			   systems have similar proc file systems,
			   with /proc/osrelease file present */
			return (char *) NULL;
		} else {
			/* some other error */
			return (char *) NULL;
		}
	}

	size_t len = 0;
	char *line = (char *) NULL;
	ssize_t r = getline(&line, &len, fp);
	e = errno;
	fclose(fp);
	if (r <= 0) {
		cdw_vdm ("ERROR: failed to get line, r = %zd, error = \"%s\"\n", r, strerror(e));
		return (char *) NULL;
	} else {
		if (line[r - 1] == '\n') {
			line[r - 1] = '\0';
		}
		cdw_vdm ("INFO: osrelease = \"%s\"\n", line);
		return line;
	}
}




char *cdw_sys_get_kernel_modules_location(void)
{
	char *osrelease = cdw_sys_get_osrelease();
	if (osrelease == (char *) NULL) {
		cdw_vdm ("ERROR: can't get osrelease\n");
		return (char *) NULL;
	}
	char *path = cdw_string_concat("/lib/modules/", osrelease, "/kernel/", (char *) NULL);
	free(osrelease);
	osrelease = (char *) NULL;

	if (path == (char *) NULL) {
		cdw_vdm ("ERROR: failed to concat modules release path\n");
		return (char *) NULL;
	} else {
		int rv = access(path, F_OK);
		int e = errno;
		if (rv != 0) {
			cdw_vdm ("ERROR: can't access path \"%s\", error = \"%s\"\n", path, strerror(e));
			free(path);
			path = (char *) NULL;
			/* even if the path exists, if we can't access it
			   using access(), it is useless */
			return (char *) NULL;
		} else {
			return path;
		}
	}
}
