/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2010 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* which method of reading IS09660 volume to use? iso9660_read()
   or regular read()? iso9660_read() may fail for large (>4GB) volumes */
#ifdef CDW_CDIO_USE_ISO9660_READ
#undef CDW_CDIO_USE_ISO9660_READ
#endif
//#define CDW_CDIO_USE_ISO9660_READ


#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <stdlib.h>

/* open() */
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>


#include <cdio/mmc.h>
#include <cdio/logging.h>

#include "cdw_cdio.h"
#include "cdw_cdio_drives.h"
#include "cdw_disc.h"
#include "cdw_utils.h"
#include "cdw_processwin.h"
#include "gettext.h"
#include "cdw_debug.h"
#include "cdw_logging.h"
#include "config.h"
#include "cdw_drive.h"
#include "cdw_sys.h"

/*
  \file cdw_cdio.c

  File defines some wrappers around cdio calls. cdw uses cdio library
  for two main purposes:
  \li first step of recognizing cd / dvd disc
  \li operations on cd discs (further investigation of cd, reading cd content)

  There are also first attempts at reading some dvd meta info, but it's hard
  to tell how far this will go.

  Parts of code in this file are heavily influenced by online documentation
  of cdio library: http://www.gnu.org/software/libcdio/libcdio.html

  Some technical details about CD sectors, CD types and other related
  information can be found in various online resources:

  \li http://www-is.informatik.uni-oldenburg.de/~dibo/teaching/mm98/buch/node113.html
  \li http://stian.lunafish.org/sample-cdrom.php
  \li http://www.digital-daydreams.com/tech/show_tech.php?id=42
  \li http://gcwebtw.com/Knowhow/AudioCD/AudioCD.htm
  \li http://club.cdfreaks.com/f52/sub-headers-207630/
  \li http://www.samsungodd.com/eng/Information/ODDTech/ODDTech.asp?FunctionValue=view&no=8&type_no=3
  \li http://fy.chalmers.se/~appro/CD-R.survey.html
  \li http://www.e-articles.info/e/a/title/Compact-Disc-information-and-CD-Drive-Formats/
  \li http://nets.rwth-aachen.de/content/teaching/lectures/sub/mms/mmsSS07/09_CD_4P.pdf
  \li http://www.icdia.co.uk/articles/filesystem.html
*/



/* maximal size of meta information area of CD disc sector;
   sync bytes = 12; header bytes = 4; subheader bytes = 8 */
#define CDW_CDIO_SECTOR_META_SIZE_MAX (12 + 4 + 8)


typedef enum cdw_track_format {
	CDW_TRACK_BLACK_BOOK_UNKNOWN = 0,     /* no information collected yet, or track has unsupported format */
	CDW_TRACK_RED_BOOK_AUDIO,             /* all 2352 bytes are useful data */
	CDW_TRACK_YELLOW_BOOK_MODE1,          /* sync + header + 2048 bytes of useful data + some error magic */
	CDW_TRACK_YELLOW_BOOK_MODE2_FORMLESS, /* sync + header + subheader + 2336 bytes of useful data + some error magic */
	CDW_TRACK_YELLOW_BOOK_MODE2_FORM1,    /* sync + header + subheader + 2048 bytes of useful data + some error magic */
	CDW_TRACK_YELLOW_BOOK_MODE2_FORM2,    /* sync + header + subheader + 2324 bytes of useful data + some error magic */
	CDW_TRACK_MIXED,                      /* "I'm puzzled, but ioctl() returns CDS_MIXED, which means data + audio" */
	CDW_TRACK_DVD                         /* DVD disc: all DVD tracks have the same size of useful data in sector (2048) */
} cdw_track_format_t;



/* index this table with values of cdw_track_format_t type */
struct {
	cdw_track_format_t cdw_track_format;
	const char        *cdw_track_format_label;
	uint16_t           payload_size;
} track_format_data[] = {{ CDW_TRACK_BLACK_BOOK_UNKNOWN,         "unknown",              0 },
			 { CDW_TRACK_RED_BOOK_AUDIO,             "audio",             2352 },
			 { CDW_TRACK_YELLOW_BOOK_MODE1,          "mode 1",            2048 },
			 { CDW_TRACK_YELLOW_BOOK_MODE2_FORMLESS, "mode 2 formless",   2336 },
			 { CDW_TRACK_YELLOW_BOOK_MODE2_FORM1,    "mode 2 form 1",     2048 },
			 { CDW_TRACK_YELLOW_BOOK_MODE2_FORM2,    "mode 2 form 2",     2324 },
			 { CDW_TRACK_MIXED,                      "mixed",                0 }, /* FIXME: it rather shouldn't be zero */
			 { CDW_TRACK_DVD,                        "dvd",               2048 }};

struct {
	cdio_fs_t   cdio_fs;
	const char *cdio_fs_label;
	const char *cdio_fs_short_label;
	const char *cdio_fs_full_label;
} track_fs_data[] = {
	/* CDIO_FS_* constants come from cdio/cd_types.h */
	{ 0,                              "none",                         "none",         "none"        },
	{ CDIO_FS_AUDIO,                  "CDIO_FS_AUDIO",                "AUDIO",        "CD-Audio"    },  /* CDIO_FS_AUDIO = 1 */
	{ CDIO_FS_HIGH_SIERRA,            "CDIO_FS_HIGH_SIERRA",          "HS",           "High Sierra" },
	{ CDIO_FS_ISO_9660,               "CDIO_FS_ISO_9660",             "ISO9660",      "ISO 9660"    },
	{ CDIO_FS_INTERACTIVE,            "CDIO_FS_INTERACTIVE",          "INT.",         "Interactive" },
	{ CDIO_FS_HFS,                    "CDIO_FS_HFS",                  "HFS",          "HFS (Hierarchical File System)" },  /* MacOS 6 through MacOS 9 */
	{ CDIO_FS_UFS,                    "CDIO_FS_UFS",                  "UFS",          "UFS (Unix File System" },  /* Generic Unix file system */
	{ CDIO_FS_EXT2,                   "CDIO_FS_EXT2",                 "EXT2",         "EXT2" },
	{ CDIO_FS_ISO_HFS,                "CDIO_FS_ISO_HFS",              "ISO+HFS",      "ISO 9660 + HFS (Hierarchical File System)"  },  /* both HFS & ISO-9660 file system */
	{ CDIO_FS_ISO_9660_INTERACTIVE,   "CDIO_FS_ISO_9660_INTERACTIVE", "ISO INT.",     "ISO 9660 Interactive" },  /* CD-RTOS and ISO filesystem */
	{ CDIO_FS_3DO,                    "CDIO_FS_3DO",                  "3DO",          "3DO"  },  /* 3DO Company */
	{ CDIO_FS_XISO,                   "CDIO_FS_XISO",                 "XISO",         "XISO" },  /* Microsoft X-BOX CD */
	{ CDIO_FS_UDFX,                   "CDIO_FS_UDFX",                 "UDFX",         "UDFX" },
	{ CDIO_FS_UDF,                    "CDIO_FS_UDF",                  "UDF",          "UDF (Universal Disk Format)" },
	{ CDIO_FS_ISO_UDF,                "CDIO_FS_ISO_UDF",              "ISO+UDF",      "ISO 9660 Interactive + UDF (Universal Disk Format"  },
	{ CDIO_FS_UNKNOWN,                "CDIO_FS_UNKNOWN",              "UNKNOWN",      "Unknown file system" },
	{ -1,                             (char *) NULL,                  (char *) NULL,  (char *) NULL }};


typedef struct cdw_track {
	track_format_t     cdio_track_format; /* AUDIO, DATA, XA, CDI, PSX, ERROR - this is track format provided by libcdio */
	cdw_track_format_t cdw_track_format;  /* this is track format defined in cdw, set by cdw code - it has more direct relation to size of useful data in sector */
	lsn_t    first_sector;
	lsn_t    last_sector;
	lsn_t    n_sectors;               /* (last_sector - first_sector + 1) this value may include padding zeros, which you may not want to read */
	lsn_t    n_sectors_to_read;       /* real number of sectors with real data, may be smaller than total number of sectors in track */
	uint16_t sector_size;             /* size of sectors in given track, using uint16_t type as in cdio library */
	struct {
		cdio_fs_t    cdio_fs;   /* cdio file system type */
		unsigned int size;      /* size in multiple of blocks (block size can be e.g. 2048 bytes) */
	} fs; /* file system */
} cdw_track_t;




typedef struct cdw_cdio_disc {
	CdIo_t *p_cdio;       /* main libcdio data structure */

	discmode_t mode;      /* disc mode as defined in cdio/disc.h */
	const char *device_fullpath;

	cdw_optical_fs_t fs;

	cdw_disc_simple_type_t simple_type; /* CD / DVD / UNKNOWN */
	char simple_type_label[9 + 1];

	bool open;            /* is the disc opened already */
	bool blank;           /* is disc in drive blank? */

	track_t first_track;  /* first track on disc, most probably always equals to 1 */
	track_t last_track;   /* last track on disc */
	track_t n_tracks;     /* number of all user tracks of all kinds */
	track_t unknown_tracks; /* total amount of tracks of unknown type */

	cdw_track_t tracks[100]; /* CD disc has no more than 99 tracks, track numbering by cdio is from 1 */
} cdw_cdio_disc_t;


/* there will be only one cdio disc open at a time, this is variable
   representing the disc; 'g' stands for "global in this file" */
static cdw_cdio_disc_t g_disc;

static void     cdw_cdio_disc_reset(cdw_cdio_disc_t *disc);
static cdw_rv_t cdw_cdio_get_disc_meta_info(cdw_cdio_disc_t *disc);
static void cdw_cdio_get_disc_meta_info_cd(cdw_cdio_disc_t *disc);
static void cdw_cdio_get_disc_meta_info_dvd(cdw_cdio_disc_t *disc);

static cdw_rv_t cdw_cdio_open_iso9660_fs(cdw_cdio_disc_t *disc);
static cdw_rv_t cdw_cdio_get_iso9660_fs_meta_information(cdw_cdio_disc_t *disc);


static cdw_track_format_t cdw_cdio_get_track_format_from_sector_header(cdw_cdio_disc_t *disc, lsn_t sector);
static const char *cdw_cdio_get_driver_error_label(driver_return_code_t error_num);

static cdw_rv_t cdw_cdio_recognize_track(cdw_cdio_disc_t *disc, track_t t);
static cdw_rv_t cdw_cdio_resolve_cd_track_cdw_format(cdw_cdio_disc_t *disc, track_t t);
static cdw_rv_t cdw_cdio_resolve_sector_size(cdw_cdio_disc_t *disc, track_t t);
static cdw_rv_t cdw_cdio_resolve_sectors_to_read(cdw_cdio_disc_t *disc, track_t t);

static int  cdw_cdio_get_cd_sector_meta_data(cdw_cdio_disc_t *disc, lsn_t sector, unsigned char *buffer);
static void cdw_cdio_update_processwin(lsn_t current_relative_sector, lsn_t n_sectors_in_track, int sector_size);
static int  cdw_cdio_copy_sectors_to_file_sub(cdw_cdio_disc_t *disc, track_t t, int output_file);

static inline long int cdw_cdio_read_sectors(cdw_cdio_disc_t *disc, track_t t, unsigned char *buffer, lsn_t sector, uint32_t n_sectors);
static void cdw_cdio_log_handler(cdio_log_level_t level, const char *message);



/**
   \brief Initialize cdio module
*/
void cdw_cdio_init(void)
{
	cdw_assert (CDIO_FS_AUDIO != 0, "ERROR: you should change \"init\" value of fs type\n");
	cdw_cdio_disc_reset(&g_disc);
#ifndef NDEBUG
	/* this is just a test call */
	cdw_sys_check_file_system_support(CDIO_FS_ISO_9660);
#endif
	return;
}





/**
   \brief Clean up cdio module
*/
void cdw_cdio_clean(void)
{
	if (g_disc.p_cdio != (CdIo_t *) NULL) {
		/* either cdw_cdio_clean() is called when cdw exits
		   in emergency mode, or there was a call to
		   cdw_cdio_open_disc(), but without matching call
		   to cdw_cdio_close_disc() */
		cdw_vdm ("ERROR: cleaning up cdio module, but you didn't close disc at some point\n");
	}
	cdw_cdio_close_disc();
	return;
}





/**
   \brief Open disc (in libcdio terms)

   Open optical disc using cdio_open(). Mark disc as open.
   Note that you will have to call cdw_cdio_get_disc_meta_info() to
   gather meta information about the disc.

   \param device_fullpath - full path to device file

   \return CDW_GEN_ERROR if function failed to open the disc
   \return CDW_OK on success
*/
cdw_rv_t cdw_cdio_open_disc(void)
{
	cdw_assert (!g_disc.open, "ERROR: disc is not reset correctly, or trying to open disc that is already open\n");
	cdw_assert (g_disc.p_cdio == (CdIo_t *) NULL, "ERROR: disc is not reset correctly, or trying to open disc that is already open\n");

	cdw_cdio_disc_reset(&g_disc);
	g_disc.device_fullpath = cdw_drive_get_drive_fullpath();
	cdw_assert (g_disc.device_fullpath != (char *) NULL, "ERROR: device fullpath is NULL\n");

	/* set to CDIO_LOG_ERROR in release builds! */
	cdio_loglevel_default = CDIO_LOG_ERROR;
	cdio_log_set_handler(cdw_cdio_log_handler);

	// cdio_init(); /* will be called by cdio_open() anyway */
	// g_disc.p_cdio = cdio_open_cd(g_disc.device_fullpath); /* alternative way of opening a CD-ROM */
	g_disc.p_cdio = cdio_open(g_disc.device_fullpath, DRIVER_LINUX);
	if (g_disc.p_cdio == (CdIo_t *) NULL) {
		cdw_vdm ("WARNING: failed to open cdio disc using DRIVER_LINUX\n");
		cdw_vdm ("WARNING: using DRIVER_UNKNOWN\n");

		g_disc.p_cdio = cdio_open(g_disc.device_fullpath, DRIVER_UNKNOWN);
		if (g_disc.p_cdio == (CdIo_t *) NULL) {
			cdw_vdm ("ERROR: failed to open cdio disc using DRIVER_UNKNOWN\n");
			return CDW_GEN_ERROR;
		}
	}

	cdw_rv_t crv = cdw_cdio_get_disc_meta_info(&g_disc);
	if (crv == CDW_NO) {
		/* mark disc as open to avoid opening it for a second time */
		g_disc.open = true;
		cdw_cdio_print_current_disc();
		return CDW_OK;
	} else if (crv == CDW_OK) {
		g_disc.fs.type = g_disc.tracks[g_disc.first_track].fs.cdio_fs;
		cdw_assert (g_disc.fs.type > 0 && g_disc.fs.type <= CDIO_FS_ISO_UDF,
			    "ERROR: disc file system type out of range: %d\n", g_disc.fs.type);
		if (cdw_cdio_is_fs_iso(g_disc.fs.type)) {
			crv = cdw_cdio_open_iso9660_fs(&g_disc);
			if (crv == CDW_OK) {
				crv = cdw_cdio_get_iso9660_fs_meta_information(&g_disc);
			}
			if (crv == CDW_GEN_ERROR) {
				cdw_vdm ("ERROR: failed to get iso9660 meta information\n");
				cdw_cdio_close_disc();
				return CDW_GEN_ERROR;
			}
		} else {
			; /* other, unsupported file system */
		}

		/* mark disc as open to avoid opening it for a second time */
		g_disc.open = true;
		cdw_cdio_print_current_disc();
		return CDW_OK;
	} else {
		cdw_vdm ("ERROR: failed to get disc meta information\n");
		cdw_cdio_close_disc();
		return CDW_GEN_ERROR;
	}
}





cdw_rv_t cdw_cdio_open_iso9660_fs(cdw_cdio_disc_t *disc)
{
	cdw_assert (cdw_cdio_is_fs_iso(disc->fs.type),
		    "ERROR: file system type is not ISO\n");
	disc->fs.fs.iso9660.p_iso = iso9660_open(disc->device_fullpath);
	if (disc->fs.fs.iso9660.p_iso == (iso9660_t *) NULL) {
		cdw_vdm ("ERROR: can't open ISO file system on a disc\n");
		return CDW_GEN_ERROR;
	}

	disc->fs.fs.iso9660.fd = open(disc->device_fullpath, O_RDONLY);
	if (disc->fs.fs.iso9660.fd == -1) {
		int e = errno;
		cdw_vdm ("ERROR: can't open ISO file system on a disc, errno = \"%s\"\n", strerror(e));
		iso9660_close(disc->fs.fs.iso9660.p_iso);
		disc->fs.fs.iso9660.p_iso = (iso9660_t *) NULL;

		return CDW_GEN_ERROR;
	}


	return CDW_OK;
}





cdw_rv_t cdw_cdio_get_iso9660_fs_meta_information(cdw_cdio_disc_t *disc)
{
	bool read_pvd = iso9660_fs_read_pvd(disc->p_cdio, &(disc->fs.fs.iso9660.pvd));
	if (!read_pvd) {
		cdw_vdm ("ERROR: failed to read pvd from a disc\n");

		return CDW_GEN_ERROR;
	}

	char *id = iso9660_get_volume_id(&(disc->fs.fs.iso9660.pvd));
	if (id == (char *) NULL) {
		cdw_vdm ("WARNING: can't get volume id from iso9660 file system\n");
		/* perhaps given file system simply has no volume id? */
		disc->fs.volume_id[0] = '\0';
	} else {
		strncpy(disc->fs.volume_id, id, ISO_MAX_VOLUME_ID);
		disc->fs.volume_id[ISO_MAX_VOLUME_ID] = '\0';
		free(id);
		id = (char *) NULL;
	}

	return CDW_OK;
}





/**
   \brief Read meta information from optical disc

   Read following information from disc:
    - disc mode (of type discmode_t, as defined in cdio/disc.h)
    - simple disc type (CD, DVD, NONE, UNKNOWN)
    - information if disc is blank
    - if disc is not blank: information about tracks number, sizes and types, and about sector sizes

   Read this information using libcdio calls only.

   \return CDW_OK if disc meta information was read without problems and disc is not blank
   \return CDW_NO if disc meta information was read without problems but disc is blank
   \return CDW_CANCEL if disc meta information was read only partially (perhaps unknown / unsupported disc type?)
   \return CDW_GEN_ERROR if function can't get any meta information
*/
cdw_rv_t cdw_cdio_get_disc_meta_info(cdw_cdio_disc_t *disc)
{
	cdw_assert (disc->p_cdio != (CdIo_t *) NULL, "ERROR: trying to get meta info from NULL disc\n");

	disc->mode = cdio_get_discmode(disc->p_cdio);

	if (cdio_is_discmode_cdrom(disc->mode)) {
		disc->simple_type = CDW_DISC_SIMPLE_TYPE_CD;
		strcpy(disc->simple_type_label, "CD");
	} else if (cdio_is_discmode_dvd(disc->mode)
		/* somehow cdio does not classify CDIO_DISC_MODE_DVD_OTHER as dvd */
		|| (disc->mode == CDIO_DISC_MODE_DVD_OTHER)) {
		disc->simple_type = CDW_DISC_SIMPLE_TYPE_DVD;
		strcpy(disc->simple_type_label, "DVD");
	} else {
		disc->simple_type = CDW_DISC_SIMPLE_TYPE_UNKNOWN;
		/* we won't work with unknown disc types at all */
		cdw_vdm ("ERROR: failed to get disc simple type, setting disc simple type to UNKNOWN\n");
		strcpy(disc->simple_type_label, "UNKNOWN");
		return CDW_GEN_ERROR;
	}

	disc->first_track = cdio_get_first_track_num(disc->p_cdio);
	disc->last_track = cdio_get_last_track_num(disc->p_cdio);
	disc->n_tracks = (track_t) (disc->last_track - disc->first_track + 1);

	if (disc->first_track == CDIO_INVALID_TRACK) {
		/* this is empirical result: disc is blank, don't try to
		   recognize tracks, because this will cause ioctl() errors; */
		cdw_vdm ("INFO: disc is blank\n");
		disc->blank = true;
		disc->fs.type = 0;

		disc->first_track = CDIO_INVALID_TRACK;
		disc->last_track = CDIO_INVALID_TRACK;
		disc->n_tracks = 0;
	} else {
		disc->unknown_tracks = 0;
		/* inspect every track on disc */
		for (track_t t = disc->first_track; t <= disc->last_track; t++) {
			cdw_rv_t crv = cdw_cdio_recognize_track(disc, t);
			if (crv == CDW_GEN_ERROR) {
				disc->unknown_tracks++;
			}
		}

		track_t t = disc->first_track;
		if (disc->tracks[t].fs.cdio_fs == CDIO_FS_UNKNOWN
		    || disc->tracks[t].fs.cdio_fs == 0) {
			cdw_vdm ("INFO: disc is blank\n");
			disc->blank = true;
		} else {
			cdw_vdm ("INFO: disc is NOT blank, it has \"%s\" file system\n",
				 track_fs_data[disc->tracks[t].fs.cdio_fs].cdio_fs_label);
			disc->blank = false;
		}
	}

	/* get some more, more disc-specific data; even if a disc
	   is blank, get its type (CD-RW, or DVD+R etc.) */
	if (disc->simple_type == CDW_DISC_SIMPLE_TYPE_CD) {
		cdw_cdio_get_disc_meta_info_cd(disc);
	} else {
		cdw_cdio_get_disc_meta_info_dvd(disc);
	}

	if (disc->blank) {
		return CDW_NO;
	} else {
		if (disc->unknown_tracks == disc->n_tracks) {
			cdw_vdm ("WARNING: all tracks are unknown\n");
			return CDW_GEN_ERROR;

		} else if (disc->unknown_tracks > 0
			   && disc->unknown_tracks < disc->n_tracks) {
			cdw_vdm ("WARNING: some tracks are unknown\n");
			return CDW_CANCEL;
		} else {
			cdw_vdm ("INFO: all tracks are known\n");
			return CDW_OK;
		}
	}
}





void cdw_cdio_get_disc_meta_info_cd(cdw_cdio_disc_t *disc)
{
	if (disc->mode == CDIO_DISC_MODE_CD_DATA) {
		/* libcdio 0.78 calls mmc command to get disc type, but the
		   call returns CD_DATA for mixed mode CD; let's do
		   some additional checks */
		cdw_rv_t crv = get_cds_mixed_with_ioctl();
		if (crv == CDW_OK) {
			/* Linux ioctl() says that this is MIXED mode cd,
			   so let's stick with what ioctl() says */
			disc->mode = CDIO_DISC_MODE_CD_MIXED;
		}
	}
	return;
}





void cdw_cdio_get_disc_meta_info_dvd(__attribute__((unused)) cdw_cdio_disc_t *disc)
{
#ifdef HAS_MMC_GET_DVD_STRUCT_PHYSICAL
	/* this info is not too useful at this moment, but it's a good idea
	   to play with new functions from libcdio */
	/*
	  typedef struct cdio_dvd_physical {
	          uint8_t type;
		  uint8_t layer_num;
		  cdio_dvd_layer_t layer[CDIO_DVD_MAX_LAYERS];
	  } cdio_dvd_physical_t;

	  typedef struct cdio_dvd_layer {
	          uint8_t book_version  : 4;
		  uint8_t book_type     : 4;
		  uint8_t min_rate      : 4;
		  uint8_t disc_size     : 4;
		  uint8_t layer_type    : 4;
		  uint8_t track_path    : 1;
		  uint8_t nlayers       : 2;
		  uint8_t track_density : 4;
		  uint8_t linear_density: 4;
		  uint8_t bca           : 1;
		  uint32_t start_sector;
		  uint32_t end_sector;
		  uint32_t end_sector_l0;
	  } cdio_dvd_layer_t;
	*/

	discmode_t m;
	cdio_dvd_struct_t s;
	s.physical.type = CDIO_DVD_STRUCT_PHYSICAL;
	s.physical.layer_num = 0;

	m = mmc_get_dvd_struct_physical(disc->p_cdio, &s);
	if (-m == EINVAL) {
		cdw_vdm ("dvd info from mmc call: following info is INVALID\n");
	}
	cdw_vdm ("dvd info from mmc call: nlayers = %d\n",
		 s.physical.layer[0].nlayers);
	cdw_vdm ("dvd info from mmc call: layer%d: book type = %d, book version = %d, \n",
		 s.physical.layer_num, s.physical.layer[0].book_type, s.physical.layer[0].book_version);
	cdw_vdm ("dvd info from mmc call: layer%d: disc size = %d, layer type = %d, \n",
		 s.physical.layer_num, s.physical.layer[0].disc_size, s.physical.layer[0].layer_type);
	cdw_vdm ("dvd info from mmc call: layer%d: start sector = %d, end sector = %d, \n",
		 s.physical.layer_num, s.physical.layer[0].start_sector, s.physical.layer[0].end_sector);

	if (s.physical.layer[0].nlayers == 1) {/* nlayers == 0 -> 1 layer, nlayers == 1 -> 2 layers? */

		s.physical.layer_num = 1;
		cdw_vdm ("dvd info from mmc call: layer%d: book type = %d, book version = %d, \n",
			 s.physical.layer_num, s.physical.layer[0].book_type, s.physical.layer[0].book_version);
		cdw_vdm ("dvd info from mmc call: layer%d: disc size = %d, layer type = %d, \n",
			 s.physical.layer_num, s.physical.layer[0].disc_size, s.physical.layer[0].layer_type);
		cdw_vdm ("dvd info from mmc call: layer%d: start sector = %d, end sector = %d, \n",
			 s.physical.layer_num, s.physical.layer[0].start_sector, s.physical.layer[0].end_sector);
	}

	if (s.physical.layer[0].nlayers == 0) {/* nlayers == 0 -> 1 layer, nlayers == 1 -> 2 layers? */
		disc->dvd_layers = 1;
	} else if (s.physical.layer[0].nlayers == 1) {/* nlayers == 0 -> 1 layer, nlayers == 1 -> 2 layers? */
		disc->dvd_layers = 2;
	} else {
		;
	}

#endif

	return;
}





/**
   \brief Close disc that was previously opened with cdw_cdio_disc_open()

   Calls cdio_destroy() for currently opened disc.
   Also calls cdw_cdio_disc_reset() to destroy any information collected
   and stored in local disc variable.
*/
void cdw_cdio_close_disc(void)
{
	if (g_disc.p_cdio != (CdIo_t *) NULL) {
		cdio_destroy(g_disc.p_cdio);
		g_disc.p_cdio = (CdIo_t *) NULL;
	}
	g_disc.open = false;

	if (g_disc.fs.fs.iso9660.p_iso != (iso9660_t *) NULL) {
		iso9660_close(g_disc.fs.fs.iso9660.p_iso);
		g_disc.fs.fs.iso9660.p_iso = (iso9660_t *) NULL;
	}

	if (g_disc.fs.fs.iso9660.fd != -1) {
		close(g_disc.fs.fs.iso9660.fd);
		g_disc.fs.fs.iso9660.fd = -1;
	}

	cdw_cdio_disc_reset(&g_disc);

	return;
}





/**
   \brief Reset all information collected by cdio function calls
*/
void cdw_cdio_disc_reset(cdw_cdio_disc_t *disc)
{
	disc->p_cdio = (CdIo_t *) NULL;
	disc->open = false;
	disc->blank = true;


	disc->fs.type = 0;
	disc->fs.type_label = (char *) NULL;
	disc->fs.volume_id[0] = '\0';
	disc->fs.n_sectors = -1;

	disc->fs.fs.iso9660.p_iso = (iso9660_t *) NULL;
	disc->fs.fs.iso9660.fd = -1;


	for (int i = 0; i < 99; i++) {
		disc->tracks[i].first_sector = CDIO_INVALID_LSN;
		disc->tracks[i].last_sector = CDIO_INVALID_LSN;
		disc->tracks[i].n_sectors = 0;
		disc->tracks[i].n_sectors_to_read = 0;

		disc->tracks[i].sector_size = 0;

		disc->tracks[i].cdw_track_format = CDW_TRACK_BLACK_BOOK_UNKNOWN;
		disc->tracks[i].cdio_track_format = TRACK_FORMAT_ERROR;
	}

	disc->mode = CDIO_DISC_MODE_NO_INFO;
	disc->simple_type = CDW_DISC_SIMPLE_TYPE_UNKNOWN;
	disc->simple_type_label[0] = '\0';

	disc->first_track = CDIO_INVALID_TRACK;
	disc->last_track = CDIO_INVALID_TRACK;
	disc->unknown_tracks = 0;
	disc->n_tracks = 0;

	return;
}





/**
   \brief Collect all available information about given track

   Function tries to collect all information about a track:
   \li start and end sector, number of sectors on track
   \li track formats: cdio track format and cdw track format
   \li size of sectors on the track
   \li number of sectors to read on given sector (I think that this value
       might be different that total number of sectors on track

   The function will fail if optical drive does not support MMC commands
   (if mmc_read_cd() fails).

   \param t - number of track to check

   \return CDW_OK if track is fully recognized
   \return CDW_GEN_ERROR if track is unknown or unsupported
*/
cdw_rv_t cdw_cdio_recognize_track(cdw_cdio_disc_t *disc, track_t t)
{
	cdw_assert (disc->p_cdio != (CdIo_t *) NULL, "ERROR: trying to get info from NULL disc\n");
	/* if disc type is unknown then we won't be able to get number
	   of tracks, and this function shouldn't be called at all */
	cdw_assert (disc->simple_type != CDW_DISC_SIMPLE_TYPE_UNKNOWN,
		    "ERROR: called the function for unknown disc type\n")

	/* 1: get range of sectors in given track, these values will be used
	   to check track format, and may be used later when reading sectors */
	disc->tracks[t].first_sector = cdio_get_track_lsn(disc->p_cdio, t);
	disc->tracks[t].last_sector = cdio_get_track_last_lsn(disc->p_cdio, t);
	disc->tracks[t].n_sectors = disc->tracks[t].last_sector - disc->tracks[t].first_sector + 1;

	cdw_vdm ("INFO: track #%d: sectors: first / last / total: %d / %d / %d\n", t,
		 disc->tracks[t].first_sector, disc->tracks[t].last_sector, disc->tracks[t].n_sectors);

	if (disc->tracks[t].first_sector == CDIO_INVALID_LSN
	    || disc->tracks[t].last_sector == CDIO_INVALID_LSN) {

		cdw_vdm ("ERROR: failed to fetch correct first_sector (%d) or last_sector (%d) of track #%d\n",
			 disc->tracks[t].first_sector, disc->tracks[t].last_sector, (int) t);
		return CDW_GEN_ERROR;
	}

	cdw_rv_t crv = CDW_OK;
	if (disc->simple_type == CDW_DISC_SIMPLE_TYPE_CD) {
		/* 2: get cdio format of track - for CD discs only! */
		disc->tracks[t].cdio_track_format = cdio_get_track_format(disc->p_cdio, t);
		if (disc->tracks[t].cdio_track_format == TRACK_FORMAT_ERROR) {
			cdw_vdm ("ERROR: failed to resolve cdio track format for track #%d\n", t);
			return CDW_GEN_ERROR;
		}

		/* 3: get cdw format of track (disc->tracks[t].cdw_format) */
		crv = cdw_cdio_resolve_cd_track_cdw_format(disc, t);
		if (crv != CDW_OK) {
			cdw_vdm ("ERROR: failed to resolve cdw track format for track #%d\n", t);
			return CDW_GEN_ERROR;
		}
	} else if (disc->simple_type == CDW_DISC_SIMPLE_TYPE_DVD) {
		disc->tracks[t].cdw_track_format = CDW_TRACK_DVD;
	} else { /* unknown disc type */
		disc->tracks[t].cdw_track_format = CDW_TRACK_BLACK_BOOK_UNKNOWN;
		cdw_vdm ("WARNING: track #%d has unknown format\n", t);
		return CDW_GEN_ERROR;
	}

	cdio_iso_analysis_t iso_analysis;
	memset(&iso_analysis, 0, sizeof(cdio_iso_analysis_t));
	cdio_fs_anal_t a = cdio_guess_cd_type(disc->p_cdio, disc->tracks[t].first_sector, t, &iso_analysis);
	disc->tracks[t].fs.cdio_fs = CDIO_FSTYPE(a);
	cdw_vdm ("INFO: fs.type = %d\n", disc->tracks[t].fs.cdio_fs);
	if (cdw_cdio_is_fs_iso(disc->tracks[t].fs.cdio_fs)) {
		disc->tracks[t].fs.size = iso_analysis.isofs_size;
	}

	/* 4: using cdio format and cdw format of track, set proper
	   value of disc->tracks[t].sector_size */
	crv = cdw_cdio_resolve_sector_size(disc, t);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to resolve sector size for track #%d\n", t);
		return CDW_GEN_ERROR;
	}

	/* 5: try to calculate value of disc->tracks[t].n_sectors_to_read,
	   using different methods, depending on track format */
	crv = cdw_cdio_resolve_sectors_to_read(disc, t);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to calculate number of sectors to read for track #%d\n", t);
		return CDW_GEN_ERROR;
	}

	return CDW_OK;
}





/**
   \brief Check cdw format of given track, save it in given disc data struct

   Function inspects given track to assign correct value to
   disc->tracks[t].cdw_format.

   If disc->tracks[t].cdio_track_format indicates Audio CD track, then this function
   simply assigns CDW_RED_BOOK_AUDIO to disc->tracks[t].cdw_format. In other
   case the function calls cdw_cdio_get_track_format_from_sector_header() to
   check cdw format of given track.

   If mmc call for given track fails or yields unknown results, the function
   returns CDW_GEN_ERROR.

   \param disc - disc data structure describing current disc
   \param t - number of track that you want to check

   \return CDW_OK if function sets cdw_format correctly
   \return CDW_GEN_ERROR on errors
*/
cdw_rv_t cdw_cdio_resolve_cd_track_cdw_format(cdw_cdio_disc_t *disc, track_t t)
{
	if (disc->tracks[t].cdio_track_format == TRACK_FORMAT_AUDIO) {
		/* cdw_cdio_get_track_format_from_sector_header() should not
		   be called for audio tracks, since audio track has no
		   header; we "resolve" cdw format manually */
		disc->tracks[t].cdw_track_format = CDW_TRACK_RED_BOOK_AUDIO;
	} else {
		/* 160 goes beyond possible 2 second silence gap (150) at
		   the beginning of track */
		lsn_t sector = disc->tracks[t].first_sector + 160;
		if (sector > disc->tracks[t].last_sector) { /* I'm not sure if this is possible */
			sector = disc->tracks[t].last_sector - 1;
		}

		disc->tracks[t].cdw_track_format = cdw_cdio_get_track_format_from_sector_header(disc, sector);
	}

	if (disc->tracks[t].cdw_track_format > CDW_TRACK_DVD) {
		disc->tracks[t].cdw_track_format = CDW_TRACK_BLACK_BOOK_UNKNOWN;
		cdw_assert (0, "ERROR: track format '%d' for track #%d is out of bounds\n",
			    disc->tracks[t].cdw_track_format, t);
	}

	cdw_vdm ("INFO: format of track #%d is \"%s\"\n",
		 t, track_format_data[disc->tracks[t].cdw_track_format].cdw_track_format_label);

	if (disc->tracks[t].cdw_track_format == CDW_TRACK_BLACK_BOOK_UNKNOWN) {
		cdw_vdm ("ERROR: failed to get cdw format for track #%d (is now CDW_TRACK_BLACK_BOOK_UNKNOWN)\n", t);
		return CDW_GEN_ERROR;
	} else {
		return CDW_OK;
	}
}





/**
   \brief Use MMC command to obtain cdw format of sector (and its parent track)

   Read full sector (all 2352 bytes) using mmc command and extract track
   mode data from its header (and subheader if one exists).

   Don't call this function for sectors from CD Audio track.

   \param sector - sector number of sector from track that we want to check

   \return CDW_BLACK_BOOK_UNKNOWN if function cannot get track format (perhaps mmc call failed)
   \return correct value of type cdw_track_format_t otherwise
*/
cdw_track_format_t cdw_cdio_get_track_format_from_sector_header(cdw_cdio_disc_t *disc, lsn_t sector)
{
	unsigned char buffer[CDW_CDIO_SECTOR_META_SIZE_MAX]; /* 12 + 4 + 8 */
	memset(buffer, 0x00, CDW_CDIO_SECTOR_META_SIZE_MAX);

	int rv = cdw_cdio_get_cd_sector_meta_data(disc, sector, buffer);
	if (rv <= 0) {
		return CDW_TRACK_BLACK_BOOK_UNKNOWN;
	} /* else buffer has some valid meta data from recognizable track */

	/* first 12 bytes are bytes of synchronization:
	   00 FF FF FF FF FF FF FF FF FF FF 00;
	   we collect them just for debug purposes - it doesn't cost
	   much in terms of computer resources */
	unsigned char sector_sync[12];
	for (int i = 0; i < 12; i++) {
		sector_sync[i] = buffer[i];
	}

	/* then 4-byte header - present in Data CD sectors of all formats
	   (CD-ROM Mode1/2-Formless, CD-ROM Mode2 Form1/2) */
	unsigned char sector_header[4];
	for (int i = 0; i < 4; i++) {
		sector_header[i] = buffer[12 + i];
	}

	/* then (possibly) 8-byte subheader - it exists only in XA (Yellow
	   Book eXtended Attributes, CD-ROM/XA Mode2 Form1/2) disc sectors */
	unsigned char sector_subheader[8];
	/* I don't check if (disc->mode == CDIO_DISC_MODE_CD_XA) because
	   this is what I want to learn myself, without relying on cdio
	   information */
	for (int i = 0; i < 8; i++) {
		sector_subheader[i] = buffer[12 + 4 + i];
	}

	/* debug code */
	/*
	cdw_vdm ("sector: %d, driver return value: %d\n", (int) sector, (int) drv);

	cdw_vdm ("sync: >>");
	for (i = 0; i < 12; i++) {
		cdw_vdm (" %x ", sector_sync[i]);
	}
	cdw_vdm ("<<\n");

	cdw_vdm ("header: >>");
	for (i = 0; i < 4; i++) {
		cdw_vdm (" %x ", sector_header[i]);
	}
	cdw_vdm ("<<\n");

	cdw_vdm ("subheader: >>");
	for (i = 0; i < 8; i++) {
		cdw_vdm (" %x ", sector_subheader[i]);
	}
	cdw_vdm ("<<\n");

	cdw_vdm ("\nAnalyzing header:\n");
	cdw_vdm ("header byte 0 = %u (minutes)\n", sector_header[0]);
	cdw_vdm ("header byte 1 = %u (seconds)\n", sector_header[1]);
	cdw_vdm ("header byte 2 = %u (sector number within one second (1-75))\n", sector_header[2]);
	cdw_vdm ("header byte 3 = %u (sector mode (0/1/2))\n", sector_header[3]);
	*/

	/* we have header and (possibly) subheader bytes, let's check them */

	/* CDW_RED_BOOK_AUDIO is not checked, because
	   cdw_cdio_get_track_format_from_sector_header() should not be
	   called for audio track; there are no headers for audio sector,
	   so this code would produce "garbage out"  */

	cdw_track_format_t track_format = CDW_TRACK_BLACK_BOOK_UNKNOWN;

	if (sector_header[3] == 2) {

		track_format = CDW_TRACK_YELLOW_BOOK_MODE2_FORMLESS;
		/* may be CD-ROM/XA Mode2 Form1/2 track as well,
		   this is to be checked (and perhaps modified) below */

	} else if (sector_header[3] == 1) {
		track_format = CDW_TRACK_YELLOW_BOOK_MODE1; /* no doubt */
	} else if (sector_header[3] == 0) {
		/* this is CDW_TRACK_YELLOW_BOOK_MODE0, but we don't support this */
		track_format = CDW_TRACK_BLACK_BOOK_UNKNOWN;
	} else {
		track_format = CDW_TRACK_BLACK_BOOK_UNKNOWN;
	}

	/* track_format may still be modified if the disc is XA */

	if (sector_header[3] == 2) {
		/* this is Mode 2 sector, so perhaps it has subheader */

		cdw_vdm ("\nChecking if subheader bytes are valid\n");

		if ( (sector_subheader[0] == sector_subheader[4])
		     && (sector_subheader[1] == sector_subheader[5])
		     && (sector_subheader[2] == sector_subheader[6])
		     && (sector_subheader[3] == sector_subheader[7])) {

			cdw_vdm ("there seems to be valid subheader\n");

			cdw_sdm ("subheader byte 0 = %u (?? (00 for not interleaved))\n", sector_subheader[0]);
			cdw_sdm ("subheader byte 1 = %u (channel number)\n",              sector_subheader[1]);
			cdw_sdm ("subheader byte 2 = %u (submode byte)\n",                sector_subheader[2]);
			cdw_sdm ("   subheader bit 2.0 = %u (EOR - End Of Record (?))\n",    (sector_subheader[2] & 0x01));
			cdw_sdm ("   subheader bit 2.1 = %u (Video)\n",                      (sector_subheader[2] & 0x02) >> 1);
			cdw_sdm ("   subheader bit 2.2 = %u (Audio)\n",                      (sector_subheader[2] & 0x04) >> 2);
			cdw_sdm ("   subheader bit 2.3 = %u (Data)\n",                       (sector_subheader[2] & 0x08) >> 3);
			cdw_sdm ("   subheader bit 2.4 = %u (?)\n",                          (sector_subheader[2] & 0x10) >> 4);
			cdw_sdm ("   subheader bit 2.5 = %u (Form 1/2)\n",                   (sector_subheader[2] & 0x20) >> 5);
			cdw_sdm ("   subheader bit 2.6 = %u (Real-Time-Sector (?))\n",       (sector_subheader[2] & 0x40) >> 6);
			cdw_sdm ("   subheader bit 2.7 = %u (EOF)\n",                        (sector_subheader[2] & 0x80) >> 7);
			cdw_sdm ("subheader byte 3 = %u (Audio/Video encoding (0 for Data))\n", sector_subheader[3]);
			cdw_sdm ("subheader byte 4 = %u (should be equal to byte 0)\n",    sector_subheader[4]);
			cdw_sdm ("subheader byte 5 = %u (should be equal to byte 1)\n",    sector_subheader[5]);
			cdw_sdm ("subheader byte 6 = %u (should be equal to byte 2)\n",    sector_subheader[6]);
			cdw_sdm ("subheader byte 7 = %u (should be equal to byte 3)\n",    sector_subheader[7]);


			/* check track format encoded in byte 2, bit 5 */
			if ( ((sector_subheader[2] & 0x20) >> 5) == 0x01) {
				track_format = CDW_TRACK_YELLOW_BOOK_MODE2_FORM2;
			} else {
				track_format = CDW_TRACK_YELLOW_BOOK_MODE2_FORM1;
			}
		} else {
			/* subheader is invalid so we have to assume that
			   this is _not_ a XA Form1 / Form2 track, we stick
			   with mode set previously, based only on value of
			   sector_header[3], that is
			   CDW_YELLOW_BOOK_MODE2_FORMLESS */

			/* "no sign of valid subheader" doesn't imply error */
			cdw_vdm ("INFO: no sign of valid subheader\n");
			if (disc->mode == CDIO_DISC_MODE_CD_XA) {
				cdw_vdm ("ERROR: (?) cdio reported CDIO_DISC_MODE_CD_XA, but this track does not have correct subheader\n\n");
			}
		}
	} /* if (sector_header[3] == 2); we don't have to do anything in case of Mode 1 track */

	return track_format; /* success, non-black cdw track format */
}





/**
   \brief Check cdw format of a track and assign correct value to sector_size of given track

   This simple function checks cdw_format field of a track data structure
   and assigns correct value to sector_size field of given track.

   Sector size for unknown (black-book) track is set to zero.

   \param disc - disc data structure describing current disc
   \param t - index of track, for which you want to check sector size

   \return CDW_OK if given track is supported by cdw and sector size was set as non-zero
   \return CDW_GEN_ERROR if given track is not supported by cdw and sector size was set as zero
*/
cdw_rv_t cdw_cdio_resolve_sector_size(cdw_cdio_disc_t *disc, track_t t)
{
	if (disc->tracks[t].cdw_track_format > CDW_TRACK_DVD) {
		disc->tracks[t].cdw_track_format = CDW_TRACK_BLACK_BOOK_UNKNOWN;
		cdw_assert (0, "ERROR: track format '%d' for track #%d is out of bounds\n",
			    disc->tracks[t].cdw_track_format, t);
	}

	disc->tracks[t].sector_size = track_format_data[disc->tracks[t].cdw_track_format].payload_size;
	cdw_vdm ("INFO: sector size for track #%d is %d\n", t, disc->tracks[t].sector_size);

	if (disc->tracks[t].sector_size == 0) {
		cdw_vdm ("ERROR: sector size for track #%d is 0\n", t);
		return CDW_GEN_ERROR;
	} else {
		return CDW_OK;
	}
}





/**
   \brief Check cdw format of a track and try to calculate correct value of "readable" sector in given track

   Each track has two parameters: n_sectors and n_sectors_to_read. It may
   happen that the two values are different (the second one smaller than
   the first one). This function calculates (probably) correct value of
   n_sectors_to_read using cdio_track_format field value or some cdio calls.

   Number of sectors to read for unknown (black-book) track is set to zero.

   \param disc - disc data structure describing current disc
   \param t - index of track, for which you want to calculate n_sectors_to_read

   \return CDW_OK if given track is supported by cdw and n_sectors_to_read was set as non-zero
   \return CDW_GEN_ERROR if given track is not supported by cdw and n_sectors_to_read was set as zero
*/
cdw_rv_t cdw_cdio_resolve_sectors_to_read(cdw_cdio_disc_t *disc, track_t t)
{
	if (disc->tracks[t].cdw_track_format == CDW_TRACK_RED_BOOK_AUDIO) {
		cdw_vdm ("INFO: calculating sectors to read for audio track #%d as \"n_sectors\"\n", t);
		disc->tracks[t].n_sectors_to_read = disc->tracks[t].n_sectors;

	} else if (disc->tracks[t].cdw_track_format == CDW_TRACK_BLACK_BOOK_UNKNOWN){
		cdw_vdm ("INFO: calculating sectors to read for black book track #%d as zero\n", t);
		disc->tracks[t].n_sectors_to_read = 0;

	} else {
		/* this will work for ISO file systems on both CDs and DVDs */
		cdw_vdm ("INFO: calculating sectors to read for track #%d with iso analysis...\n", t);

		cdio_fs_anal_t fs = disc->tracks[t].fs.cdio_fs;
		cdw_vdm ("INFO: track #%d, fs = \"%s\"\n",
			 t, track_fs_data[fs].cdio_fs_label);
		if (cdw_cdio_is_fs_iso(fs)) {

			/* WARNING: for DVDs size of ISO volume (counted
			   in sectors) may be very different than
			   "last_sector - first_sector + 1"; the difference
			   is *very* significant, and only one of the values
			   is correct. Correct value is isofs_size. Be careful
			   when reading data from such ISO volume on DVD, some
			   cdio read() functions will make sure that you don't
			   attempt to read past "last_sector". In such cases
			   use functions reading iso9660 volume instead of
			   functions reading from disc device. */

			cdw_vdm ("INFO: ISO 9660: track #%d, n sectors = %d, isofs size = %d\n",
				 t, disc->tracks[t].n_sectors, disc->tracks[t].fs.size);
			disc->tracks[t].n_sectors_to_read = (lsn_t) disc->tracks[t].fs.size;
		} else {
			/* using n_sectors in this situation is the
			   last possibility of getting some value of
			   n_sectors_to_read; may not be the best option,
			   but may be the only option; TODO: to be checked */
			disc->tracks[t].n_sectors_to_read = disc->tracks[t].n_sectors;
			cdw_vdm ("WARNING: no ISO FS detected, setting \"sectors to read\" = \"n_sectors\"\n");
		}
	}

	cdw_vdm ("INFO \"sectors to read\" for track #%d resolved as %d\n", t, disc->tracks[t].n_sectors_to_read);

	if (disc->tracks[t].n_sectors_to_read == 0) {
		cdw_vdm ("ERROR: failed to calculate sectors to read for track #%d\n", t);
		return CDW_GEN_ERROR;
	} else {
		return CDW_OK;
	}

}





/**
   \brief Read tracks from current disc and write them to given file descriptor

   More detailed description is following:
       cdw_cdio_copy_tracks_to_file()
   calls for every tracks
       cdw_cdio_copy_sectors_to_file()
   to read all tracks from disc.

   This is just a wrapper for one 'for' loop iterating on tracks, one
   function call inside loop, and some additional stuff.

   \p output_file must be file descriptior for file that is already
   open for writing.

   \param fd - file descriptor of output file

   \return CDW_OK when all tracks were copied successfully
   \return CDW_NO when function copied some tracks (but not all) correctly
   \return CDW_GEN_ERROR when function failed to read any track correctly
*/
cdw_rv_t cdw_cdio_copy_tracks_to_file(int fd)
{
	cdw_assert (g_disc.open, "ERROR: trying to read from closed disc\n");
	cdw_assert (fd != -1 && fd != 0, "ERROR: invalid output file: %d\n", fd);

	int failed_tracks = 0;

	cdw_vdm ("INFO: number of tracks on disc to read: %d\n", g_disc.n_tracks);

	for (track_t t = g_disc.first_track; t <= g_disc.last_track; t++) {
		if (g_disc.tracks[t].cdw_track_format == CDW_TRACK_BLACK_BOOK_UNKNOWN) {
			/* 2TRANS: this is message printed to log file;
			   "Copy" is a name of task of copying track from
			   CD to hard disc; %d is a track number */
			cdw_logging_write(_("Copy: skipping track #%d with unknown format\n"), t);

			failed_tracks++;
			continue;
		}

		int rv = cdw_cdio_copy_sectors_to_file_sub(&g_disc, t, fd);
		cdw_sdm ("INFO track #%d finished with rv = %d\n", t, rv);

		if (rv != 0) {
			failed_tracks++;
			cdw_vdm ("  ERROR: failed reading track #%d\n", t);
			cdw_cdio_copy_print_debug_on_error(rv);
		}
	}

	if (failed_tracks == g_disc.n_tracks) {
		cdw_vdm ("ERROR: number of tracks read correctly is zero\n")
		return CDW_GEN_ERROR;
	} else if (failed_tracks < g_disc.n_tracks && failed_tracks > 0) {
		cdw_vdm ("WARNING: failed to read correctly %d tracks\n", failed_tracks);
		return CDW_CANCEL;
	} else {
		return CDW_OK;
	}
}



/**
   \return 0 on success
   \return -1, -2 or -3 on function's internal errors
   \return 'errno - 5' on write() error, caller can interpret value of 'errno + 5' to check the error
*/
int cdw_cdio_copy_sectors_to_file(track_t t, int fd)
{
	return cdw_cdio_copy_sectors_to_file_sub(&g_disc, t, fd);
}





#define cdw_cdio_copy_sectors_to_file_debug_print_state(disc)		\
	cdw_vdm ("## INFO: ---------------------------------------- \n"); \
	cdw_vdm ("## INFO:      track number = %8d \n", disc->tracks[t].first_sector); \
	cdw_vdm ("## INFO:      first_sector = %8d \n", disc->tracks[t].first_sector); \
	cdw_vdm ("## INFO:       last_sector = %8d \n", disc->tracks[t].last_sector); \
	cdw_vdm ("## INFO:         n_sectors = %8d \n", disc->tracks[t].n_sectors); \
	cdw_vdm ("## INFO: n_sectors_to_read = %8d \n", disc->tracks[t].n_sectors_to_read);	\
	cdw_vdm ("## INFO:       sector_size = %8d \n", disc->tracks[t].sector_size); \
	cdw_vdm ("## INFO: ---------------------------------------- \n");





#define cdw_cdio_copy_sectors_to_file_debug_print_progress(disc, t, sector) \
	cdw_vdm ("INFO: track #%d: sector size=%d, sector=%d/%d (non-relative: %d/%d)\n", \
		 (int) t,						\
		 (int) disc->tracks[t].sector_size,			\
		 (int) sector - disc->tracks[t].first_sector + 1, /* +1 to display sector numbers from 1, not 0 */ \
		 (int) disc->tracks[t].n_sectors_to_read,		\
		 (int) sector + 1, /* +1 to display sector numbers from 1, not 0 */ \
		 (int) disc->tracks[t].first_sector + disc->tracks[t].n_sectors_to_read);





static inline long int cdw_cdio_read_sectors(cdw_cdio_disc_t *disc, track_t t, unsigned char *buffer, lsn_t sector, uint32_t n_sectors)
{
	const int n_tries = 2; /* two read attempts seem to be reasonable */
	long int rv = 0;
	for (int i = 0; i < n_tries; i++) {
		if (disc->tracks[t].cdw_track_format == CDW_TRACK_RED_BOOK_AUDIO) {
			rv = cdio_read_audio_sectors(disc->p_cdio, buffer, sector, n_sectors);
		} else if (disc->tracks[t].cdw_track_format == CDW_TRACK_DVD) {

#ifdef CDW_CDIO_USE_ISO9660_READ
			/* this special case for DVDs is important:
			   cdio_read_data_sectors() pays attention to
			   disc->tracks[t].last_sector, which may not
			   mark the real end of ISO volume on DVDs.
			   This is why it is better to use
			   iso9660_iso_seek_read() rather than
			   cdio_read_data_sectors() for DVDs */
			/* UNFORTUNATELY, iso9660_iso_seek_read() fails
			   for ISO volumes larger than 2^32 bytes :( */
			rv = iso9660_iso_seek_read(disc->iso_fs.p_iso, (void *) buffer,
						   sector, n_sectors);
#else
			rv = read(disc->fs.fs.iso9660.fd, (void *) buffer, disc->tracks[t].sector_size * n_sectors);
			if (rv == -1) {
				int e = errno;
				cdw_vdm ("ERROR: fd = %d, errno = \"%s\"\n", disc->fs.fs.iso9660.fd, strerror(e));
				break; /* no re-reads when using read(); */
			}
#endif

		} else { /* data CD */
			/* track-dependent sector size + universal read() function */
			rv = cdio_read_data_sectors(disc->p_cdio, buffer, sector,
						    disc->tracks[t].sector_size, n_sectors);
		}
		if (rv >= 0) {
			/* success */
			if (i > 0) {
				cdw_vdm ("INFO: success in attempt #%d\n", i);
			}
			break;
		} else {
			int e = errno;
			cdw_vdm ("ERROR: failed at read attempt #%d\n", i);
			cdw_vdm ("ERROR: sector #%d: %d / \"%s\" / %ld / \"%s\"\n",
				 sector, e, strerror(e), rv, cdw_cdio_get_driver_error_label(rv));
		}
	}

	return rv;
}





/**
   \brief Read sectors from given track of disc, write them to given file descriptor

   This function cannot be called for track with unsupported format.

   \p fd must be file descriptior for file that is already
   open for writing.

   \param disc - disc to read from
   \param t - track number of track to read sectors from
   \param fd - file descriptor to write the data to (already opened and with proper permissions)

   \return 0 on success
   \return -1, -2 on function's internal errors
   \return -3 if write() didn't write all data passed to it, which may mean not enough available space on hard disc
   \return 'errno - 5' on otehr write() errors, caller can interpret value of 'errno + 5' to check the error
*/
int cdw_cdio_copy_sectors_to_file_sub(cdw_cdio_disc_t *disc, track_t t, int fd)
{
#define CDW_CDIO_CALC_N_SECTORS(n_left) \
		(n_left >= CDW_CDIO_RW_N_SECTORS ? CDW_CDIO_RW_N_SECTORS : n_left % CDW_CDIO_RW_N_SECTORS);
	/* I'm repeating here these asserts as in cdw_cdio_copy_tracks_to_file()
	   because this function can be called either by
	   cdw_cdio_copy_tracks_to_file() or directly by other modules */
	cdw_assert (disc->p_cdio != (CdIo_t *) NULL, "ERROR: trying to get info from NULL disc\n");
	cdw_assert (disc->open, "ERROR: trying to read from closed disc\n");
	cdw_assert (fd != -1 && fd != 0, "ERROR: invalid output file: %d\n", fd);

	if (disc->tracks[t].cdw_track_format == CDW_TRACK_BLACK_BOOK_UNKNOWN) {
		cdw_vdm ("ERROR: called the function for black book (unknown) track\n");
		return -1;
	}

	/* we are re-using the same processwin for all tracks,
	   clean leftovers from reading previous track */
	cdw_processwin_reset_progress();

	cdw_cdio_copy_sectors_to_file_debug_print_state(disc);

	/* buffer for data read from disc, CDIO_CD_FRAMESIZE_RAWER
	   is maximal size of sector defined in cdio header files */
	unsigned char buffer[CDIO_CD_FRAMESIZE_RAWER * CDW_CDIO_RW_N_SECTORS];

	/* size of data left to read from ISO volume */
	lsn_t n_sectors_left = disc->tracks[t].n_sectors_to_read;
	/* first sector of ISO volume to start reading from */
	lsn_t sector = disc->tracks[t].first_sector;
	/* number of sectors to read/write at one time;
	   one sector has disc->tracks[t].sector_size bytes */
	lsn_t n_sectors = CDW_CDIO_CALC_N_SECTORS(n_sectors_left);

	cdw_rv_t retval = CDW_OK;
	while (n_sectors > 0) {
		cdw_cdio_copy_sectors_to_file_debug_print_progress(disc, t, sector);

		long int r = cdw_cdio_read_sectors(disc, t, buffer, sector, (uint32_t) n_sectors);
		if (r < 0) {
			cdw_vdm ("ERROR: failed read from track #%d\n", t);
			retval = -2;
			break;
		} else {
			ssize_t w = write(fd, buffer, disc->tracks[t].sector_size * (uint32_t) n_sectors);
			int e = errno;
			if (w != ((int) disc->tracks[t].sector_size * n_sectors)) {
				cdw_vdm ("ERROR: write error for track #%d (read %d, write %zd)\n",
					 t, disc->tracks[t].sector_size * n_sectors, w);
				cdw_vdm ("ERROR: errno = \"%s\"\n", strerror(e));

				if (w == -1) {
					retval = e - 5;
				} else {
					retval = -3;
				}
				break;
			}

			lsn_t current_relative_sector = sector - disc->tracks[t].first_sector;
			/* don't update processwin too often */
			if ((current_relative_sector % (40 * n_sectors)) == 0
			     || n_sectors < CDW_CDIO_RW_N_SECTORS) {

				lsn_t n_sectors_in_track = disc->tracks[t].n_sectors_to_read;
				int sector_size = disc->tracks[t].sector_size;
				cdw_cdio_update_processwin(current_relative_sector, n_sectors_in_track, sector_size);
			}
			sector += n_sectors;
			n_sectors_left -= n_sectors;
			n_sectors = CDW_CDIO_CALC_N_SECTORS(n_sectors_left);
		}
	}
	if (retval == 0) {
		/* because this function reads N sectors at once, it may happen
		   that it reads all data from disc, but last update of processwin
		   shows "99%" instead of "100%". Let's fake "100%" here */
		lsn_t current_relative_sector = disc->tracks[t].n_sectors_to_read - 1;
		lsn_t n_sectors_in_track = disc->tracks[t].n_sectors_to_read;
		int sector_size = disc->tracks[t].sector_size;
		cdw_cdio_update_processwin(current_relative_sector, n_sectors_in_track, sector_size);

		return 0;
	} else {
		return retval;
	}
}





void cdw_cdio_copy_print_debug_on_error(int error)
{
	if (error == -3) {
		cdw_vdm ("ERROR: error = -3, probably not enough space on disc\n");

		/* 2TRANS: this is string written to log file when
		   reading one track from CD disc is finished;
		   it is written after "Track %d: " string */
		cdw_logging_write(_("There is probably not enough space on hard disc for copied data\n"));
	} else if (error < -3) {
		/* cdw_cdio_copy_sectors_to_file() may
		   return 'errno - 5' for some errors */
		error = error + 5;
		cdw_vdm ("ERROR: errno = %s\n", strerror(error));

		/* 2TRANS: this is string written to log file when
		   reading one track from CD disc is finished with
		   problems; "%s" is system error message;
		   it is written after "Track %d: " string */
		cdw_logging_write(_("Maybe this error message will be helpful: \"%s\"\n"), strerror(error));
	} else {
		/* no more information to print */
	}

	return;
}





/**
   \brief Read sync bytes, header and subheader (if available) from given CD sector.

   No user data is read.
   \p buffer has to have size of (12 + 4 + 8) bytes.

   On success the function returns number of bytes placed in \p sector bytes:
   either this is 16 bytes (12 bytes sync + 4 bytes header) or 24 (12 bytes
   sync + 4 bytes header + 8 bytes subheader). This value can be used by
   caller to learn if this function fetched a subheader or not.

   On errors the function returns error code (negative number) returned by
   mmc_read_cd(), caller can cast the value to driver_return_code_t type and
   process further.

   \param disc - disc to read from
   \param sector - number of sector to read
   \param sector_buffer - place where to store sector bytes read from cd

   \return mmc error code (negative value) on mmc read error
   \return number of bytes read from cd and placed in \p sector_buffer
*/
int cdw_cdio_get_cd_sector_meta_data(cdw_cdio_disc_t *disc, lsn_t sector, unsigned char *buffer)
{
	/* NOTE - this will fail if drive does not have MMC capabilities or
	   this functionality is not supported, but what other portable
	   mechanism could we use?

	   This version of function does not try to guess track type
	   before calling mmc_read_cd(), it just tries to get
	   header + subheader, and if it fails then tries again to
	   get header alone. It is up to caller to decide what to do
	   with data read into sector_buffer[] and how to handle error
	   returned by mmc_read_cd(); */

	/* note that CDIO_DISC_MODE_CD_CDDA should be ruled out before
	   calling this function */
	cdw_assert (disc->mode != CDIO_DISC_MODE_CD_DA, "ERROR: you shouldn't call this function for Audio CD\n");
	cdw_assert (disc->p_cdio != (CdIo_t *) NULL, "ERROR: p_cdio is NULL\n");

	/* which sector header should be returned? (data header and subheader)
	    0: none, 1: 4-bit header, 2: 8-bit subheader, 3: both */
	uint8_t header_code = 0;
	uint16_t block_size = 0;
	driver_return_code_t drv;
	for (int i = 0; i < 2; i++) {
		if (i == 0) {
			/* first test most "aggressive" option */
			header_code = 3; /* get both header and subheader; (XA sector?) */
			/* 12: sync bytes; 4: header bytes; 8: subheader bytes */
			block_size = 12 + 4 + 8;
		} else {
			header_code = 1; /* get only header; (DATA sector?) */
			/* 12: sync bytes; 4: header bytes */
			block_size = 12 + 4;
		}
		cdw_sdm ("INFO: calling mmc_read_cd(), expecting to get %s from sector #%d\n",
			 (header_code == 3) ? "header + subheader" : "header", sector);

		drv = mmc_read_cd(disc->p_cdio,  /* object to read from */
				  buffer,        /* place to store data */
				  sector,        /* sector number to read */
				  0,             /* expected sector type; 0 - all sector types */
				  false,         /* b_digital_audio_play - audio related stuff, doesn't matter */
				  true,          /* b_sync_header - return synchronization header */
				  header_code,   /* uint8_t header_code - data header and subheader; 0: none, 1: 4-bit header, 2: 8-bit subheader. 3: both */
				  false,         /* bool b_user_data - return user data? */
				  false,         /* bool b_edc_ecc - error detection/correction data */
				  0,             /* uint8_t c2_error_information, yet another error-related field (?) */
				  0,             /* uint8_t subchannel_selection - subchannel selection bits (?)*/
				  block_size,    /* uint16_t i_blocksize - size of the block expected to be returned */
				  1);            /* uint32_t i_blocks - number of blocks expected to be returned */

		if (drv < 0) {
			cdw_vdm ("ERROR: mmc_read_cd() returned error value = %d\n", drv);
		} else {
			cdw_sdm ("INFO: mmc_read_cd() returned success value = %d\n", drv);
			return (int) block_size;
		}
	} /* for () */

	return (int) drv; /* this will be an error value */
}





void cdw_cdio_update_processwin(lsn_t current_relative_sector, lsn_t n_sectors_in_track, int sector_size)
{
	/* dividing by 4s to avoid overflows */
	long current = ((current_relative_sector / 4) * (sector_size / 4)) / (256 * 256);
	long total = ((n_sectors_in_track / 4) * (sector_size / 4)) / (256 * 256);
	char current_value_string[PROCESSWIN_MAX_RTEXT_LEN + 1];

	/* 2TRANS: this string will be displayed as message in progress window;
	   first %ld is amount of data already read from cd,
	   second %ld is total amount of data to be read */
	snprintf(current_value_string, PROCESSWIN_MAX_RTEXT_LEN + 1, _("%ld/%ld MB"), current, total);
	cdw_vdm ("INFO: current value string = \"%s\"\n", current_value_string);
	cdw_processwin_display_progress_conditional(current, total, current_value_string);

	return;
}





cdw_disc_simple_type_t cdw_cdio_get_simple_type(void)
{
	cdw_assert (g_disc.open, "ERROR: trying to get information about disc that is not open\n");
	cdw_assert (g_disc.simple_type != CDW_DISC_SIMPLE_TYPE_UNKNOWN,
		    "ERROR: trying to get simple disc type when it is set to CDW_DISC_SIMPLE_TYPE_UNKNOWN\n");

	return g_disc.simple_type;
}





discmode_t cdw_cdio_get_disc_mode(void)
{
	cdw_assert (g_disc.open, "ERROR: trying to get information about disc that is not open\n");
	return g_disc.mode;
}





track_t cdw_cdio_get_first_track_number(void)
{
	cdw_assert (g_disc.open, "ERROR: trying to get information about disc that is not open\n");
	return g_disc.first_track;
}





bool cdw_cdio_is_blank(void)
{
	cdw_assert (g_disc.open, "ERROR: trying to get information about disc that is not open\n");
	return g_disc.blank;
}





track_t cdw_cdio_get_first_track(void)
{
	cdw_assert (g_disc.open, "ERROR: trying to get information about disc that is not open\n");
	return g_disc.first_track;
}





track_t cdw_cdio_get_last_track(void)
{
	cdw_assert (g_disc.open, "ERROR: trying to get information about disc that is not open\n");
	return g_disc.last_track;
}





track_t cdw_cdio_get_number_of_tracks(void)
{
	cdw_assert (g_disc.open, "ERROR: trying to get information about disc that is not open\n");
	return g_disc.n_tracks;
}





bool cdw_cdio_is_audio_track(track_t t)
{
	cdw_assert (g_disc.open, "ERROR: trying to get information about disc that is not open\n");
	cdw_assert (t >= g_disc.first_track && t <= g_disc.last_track,
		    "ERROR: asking for track %d out of range %d-%d\n",
		    t, g_disc.first_track, g_disc.last_track);

	if (g_disc.tracks[t].cdw_track_format == CDW_TRACK_RED_BOOK_AUDIO) {
		return true;
	} else {
		return false;
	}
}





bool cdw_cdio_is_data_track(track_t t)
{
	cdw_assert (g_disc.open, "ERROR: trying to get information about disc that is not open\n");
	cdw_assert (t >= g_disc.first_track && t <= g_disc.last_track,
		    "ERROR: asking for track %d out of range %d-%d\n",
		    t, g_disc.first_track, g_disc.last_track);

	if (g_disc.tracks[t].cdw_track_format == CDW_TRACK_YELLOW_BOOK_MODE1
	    || g_disc.tracks[t].cdw_track_format == CDW_TRACK_YELLOW_BOOK_MODE2_FORMLESS
	    || g_disc.tracks[t].cdw_track_format == CDW_TRACK_YELLOW_BOOK_MODE2_FORM1
	    || g_disc.tracks[t].cdw_track_format == CDW_TRACK_YELLOW_BOOK_MODE2_FORM2
	    /* let's pretend that all DVDs are data DVDs */
	    || g_disc.tracks[t].cdw_track_format == CDW_TRACK_DVD) {

		cdw_vdm ("INFO: track #%d is a data track\n", t);
		return true;
	} else {
		cdw_vdm ("INFO: track #%d is not a data track\n", t);
		return false;
	}
}





void cdw_cdio_print_current_disc(void)
{
	cdw_assert (g_disc.open, "ERROR: trying to get information about disc that is not open\n");
	cdw_vdm ("INFO: disc simple type is %s\n", g_disc.simple_type_label);

	if (g_disc.simple_type == CDW_DISC_SIMPLE_TYPE_CD) {
		if (g_disc.mode == CDIO_DISC_MODE_CD_DATA) {
			cdw_vdm ("INFO: disc is CDIO_DISC_MODE_CD_DATA\n");
		} else if (g_disc.mode == CDIO_DISC_MODE_CD_XA) {
			cdw_vdm ("INFO: disc is CDIO_DISC_MODE_CD_XA\n");
		} else if (g_disc.mode == CDIO_DISC_MODE_CD_DA) {
			cdw_vdm ("INFO: disc is CDIO_DISC_MODE_CD_DA\n");
		} else {
			cdw_vdm ("INFO: disc is CD / UNKNOWN\n");
		}
	}

	cdw_vdm ("INFO: tracks: first / last / unknown / total: %d / %d / %d / %d\n",
		 g_disc.first_track, g_disc.last_track, g_disc.unknown_tracks, g_disc.n_tracks);

	if (g_disc.n_tracks > 0) {
		if (g_disc.unknown_tracks == g_disc.n_tracks) {
			cdw_vdm ("ERROR: there are %d / %d unknown tracks\n",
				 g_disc.unknown_tracks, g_disc.n_tracks);
		} else if (g_disc.unknown_tracks > 0
			   && g_disc.unknown_tracks < g_disc.n_tracks) {
			cdw_vdm ("WARNING: there are %d / %d unknown tracks\n",
				 g_disc.unknown_tracks, g_disc.n_tracks);

		} else {
			;
		}
	}

	for (int t = g_disc.first_track; t <= g_disc.n_tracks; t++) {
		cdw_vdm ("INFO: track #%d: format: \"%s\", size: %d, fs: \"%s\"\n",
			 t,
			 track_format_data[g_disc.tracks[t].cdw_track_format].cdw_track_format_label,
			 g_disc.tracks[t].n_sectors_to_read,
			 track_fs_data[g_disc.tracks[t].fs.cdio_fs].cdio_fs_label);
	}
	return;
}



cdw_id_label_t driver_errors[] = {
	{ DRIVER_OP_ERROR,          "DRIVER_OP_ERROR" },
	{ DRIVER_OP_UNSUPPORTED,    "DRIVER_OP_UNSUPPORTED" },
	{ DRIVER_OP_UNINIT,         "DRIVER_OP_UNINIT" },
	{ DRIVER_OP_NOT_PERMITTED,  "DRIVER_OP_NOT_PERMITTED" },
	{ DRIVER_OP_BAD_PARAMETER,  "DRIVER_OP_BAD_PARAMETER" },
	{ DRIVER_OP_BAD_POINTER,    "DRIVER_OP_BAD_POINTER" },
	{ DRIVER_OP_NO_DRIVER,      "DRIVER_OP_NO_DRIVER" },
	{ 0,                        (char *) NULL }};


/**
   \brief Debug function printing name of cdio driver error return value

   Some cdio functions return value of driver_return_code_t type. If this
   value is not DRIVER_OP_SUCCESS or greater than zero, then the value
   should be analyzed and (hopefully) acted upon. Here we print name of
   this error.
*/
const char *cdw_cdio_get_driver_error_label(driver_return_code_t error_num)
{
	return cdw_utils_id_label_table_get_label(driver_errors, (int) error_num);
}





static void cdw_cdio_log_handler(cdio_log_level_t level, const char *message)
{
	cdw_id_label_t handler_levels[] = {
		{ CDIO_LOG_DEBUG,  "debug"       },
		{ CDIO_LOG_INFO,   "info"        },
		{ CDIO_LOG_WARN,   "warning"     },
		{ CDIO_LOG_ERROR,  "error"       },
		{ CDIO_LOG_ASSERT, "assert"      },
		{ 0,               (char *) NULL }};

	cdw_vdm ("INFO: libcdio %s message: \"%s\"\n",
		 cdw_utils_id_label_table_get_label(handler_levels, level), message);

	return;
}





const char *cdw_cdio_get_fs_type_label(void)
{
	cdw_assert (g_disc.open, "ERROR: trying to get data from closed disc\n");
	return track_fs_data[g_disc.fs.type].cdio_fs_short_label;
}





cdio_fs_t cdw_cdio_get_fs_type(void)
{
	cdw_assert (g_disc.open, "ERROR: trying to get data from closed disc\n");
	return g_disc.fs.type;
}





bool cdw_cdio_is_fs_iso(cdio_fs_t fs)
{
	if (fs == CDIO_FS_ISO_9660
	    || fs == CDIO_FS_ISO_UDF
	    || fs == CDIO_FS_ISO_HFS
	    || fs == CDIO_FS_ISO_9660_INTERACTIVE) {

		return true;
	} else {
		cdw_vdm ("INFO: file system is not ISO9660\n");
		return false;
	}
}




const char *cdw_cdio_get_volume_id(void)
{
	return g_disc.fs.volume_id;
}
