/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2010 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


/**
   \file cdw_config.c
   \brief Functions operating on configuration file and configuration variable(s)

   This file defines functions that:
   \li initialize and destroy configuration module
   \li read and write configuration file (with error handling)
   \li set default configuration that can be used when program can't read configuration from file
   \li validate configuration
*/

#define _BSD_SOURCE /* strdup() */


#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <stdlib.h>
#include <errno.h>

#include "config.h"
#include "cdw_config.h"
#include "cdw_config_ui.h"
#include "cdw_config_ui_internals.h"
#include "gettext.h"
#include "cdw_string.h"
#include "cdw_fs.h"
#include "cdw_widgets.h"
#include "cdw_disc.h"
#include "cdw_debug.h"
#include "cdw_utils.h"
#include "cdw_ext_tools.h"
#include "cdw_erase_disc.h"
#include "cdw_drive.h"

extern const char *cdw_log_file_name;
extern cdw_id_label_t cdw_config_volume_size_items[];

/** \brief Flag letting you know if options module works in abnormal mode
    Set to true if something went wrong during module setup -
    cdw will work without reading/writing configuration to disk file */
bool failsafe_mode = false;

/** \brief Project-wide variable holding current configuration of cdw */
cdw_config_t global_config;



/** \brief Stores directory name where cdw config file is stored,
    also used as base path needed by some option values; if there are
    no errors during configuration of options module, this path is
    set to user home directory; since this path is initialized with
    value from cdw_fs.c module, ending slash is included */
static char *base_dir_fullpath = (char *) NULL;

/** \brief Full path to cdw configuration file */
static char *config_file_full_path = NULL;

/** \brief Hardwired name of cdw configuration file */
static const char *config_file_name = ".cdw.conf";


/* pair of low level functions that directly access configuration file;
   they are used by cdw_config_write_to_file() and
   cdw_config_read_from_file() respectively */
static void cdw_config_var_write_to_file(FILE *config_file, cdw_config_t *_config);
static cdw_rv_t cdw_config_var_read_from_file(FILE *config_file, cdw_config_t *_config);

static cdw_rv_t cdw_config_var_set_defaults(cdw_config_t *_config);
static cdw_rv_t cdw_config_module_set_paths(void);


struct cdw_config_vst_t {
	int id;
	const char *old_label;
	const char *new_label;
	long int size;
};

static struct cdw_config_vst_t cdw_config_volume_size_table[] = {
	/*         id;                      old_label             new_label;               size   */
	{ CDW_CONFIG_VOLUME_SIZE_CD74,        "650",                "cd74",                681984000  / (1024 * 1024)     },
	{ CDW_CONFIG_VOLUME_SIZE_CD80,        "703",                "cd80",                737280000  / (1024 * 1024)     },
	{ CDW_CONFIG_VOLUME_SIZE_DVD_GENERIC, "4482",               "dvd",                 4700372992 / (1024 * 1024)     },
	{ CDW_CONFIG_VOLUME_SIZE_DVD_R,       "4489",               "dvd-r",               4707319808 / (1024 * 1024)     },
	{ CDW_CONFIG_VOLUME_SIZE_DVD_RP,      "4482",               "dvd+r",               4700372992 / (1024 * 1024)     },
	{ CDW_CONFIG_VOLUME_SIZE_DVD_RW,      "4489",               "dvd-rw",              4707319808 / (1024 * 1024)     },
	{ CDW_CONFIG_VOLUME_SIZE_DVD_RWP,     "4482",               "dvd+rw",              4700372992 / (1024 * 1024)     },
	{ CDW_CONFIG_VOLUME_SIZE_DVD_R_DL,    "dvd-r dl",           "dvd-r dl",            8543666176 / (1024 * 1024)     },
	{ CDW_CONFIG_VOLUME_SIZE_DVD_RP_DL,   "dvd+r dl",           "dvd+r dl",            8547991552 / (1024 * 1024)     },
	{ CDW_CONFIG_VOLUME_SIZE_CUSTOM,      "custom",             "custom",              0      },
	{ CDW_CONFIG_VOLUME_SIZE_AUTO,        "auto",               "auto",                0      },
	{ -1,                                 (char *) NULL,        (char *) NULL,         0      } };


long int cdw_config_volume_size_by_id(int id)
{
	struct cdw_config_vst_t *table = cdw_config_volume_size_table;
	for (int i = 0; table[i].id != -1; i++) {
		if (table[i].id == id) {
			return table[i].size;
		}
	}
	cdw_vdm ("ERROR: id %d not found\n", id);
	return 0;
}





int cdw_config_volume_id_by_label(const char *label)
{
	struct cdw_config_vst_t *table = cdw_config_volume_size_table;
	for (int i = 0; table[i].id != -1; i++) {
		if (!strcasecmp(label, table[i].old_label) || !strcasecmp(label, table[i].new_label)) {
			return table[i].id;
		}
	}
	cdw_vdm ("WARNING: label \"%s\" not found\n", label);
	return -1;
}





/**
 * \brief Initialize configuration module
 *
 * Initialize configuration module - prepare some default option values and set paths
 * to base directory (should be user home directory) and to cdw configuration
 * file (will be in base directory).
 *
 * \return CDW_OK on success
 * \return CDW_GEN_ERROR on errors - the module wasn't configured at all
 * \return CDW_FAILSAFE_MODE if module cannot access cdw configuration file (module is working in failsafe mode)
 */
cdw_rv_t cdw_config_module_init(void)
{
	cdw_rv_t crv = cdw_config_module_set_paths();
	if (crv == CDW_NO || crv == CDW_OK) { /* paths are set correctly... */
		if (crv == CDW_NO) { /* ... but not to $HOME */
			/* 2TRANS: this is title of dialog window */
			cdw_buttons_dialog(_("Config file error"),
					   /* 2TRANS: this is message in dialog window */
					   _("Cannot find home directory and settings file. Will use configuration file in temporary location."),
					   CDW_BUTTONS_OK, CDW_COLORS_WARNING);
		}

		failsafe_mode = false;
	} else { /* error situation; we can check what happened, but not in this release */
		failsafe_mode = true;
	}

	crv = cdw_config_var_init_fields(&global_config);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to initialize fields of config var\n");
		return CDW_GEN_ERROR;
	}

	crv = cdw_config_var_set_defaults(&global_config);
	if (crv != CDW_OK) {
		cdw_config_var_free_fields(&global_config);
		cdw_vdm ("ERROR: failed to set defaults for config variable\n");
		return CDW_GEN_ERROR;
	}

	if (failsafe_mode) {
		/* 2TRANS: this is title of dialog window */
		cdw_buttons_dialog(_("Config file error"),
				   /* 2TRANS: this is message in dialog window */
				   _("Cannot read or write settings file. Will use temporary configuration."),
				   CDW_BUTTONS_OK, CDW_COLORS_ERROR);

		return CDW_FAILSAFE_MODE;
	} else {
		/* This overwrites default option values with values found
		   in config file, don't create config file if it doesn't
		   exist already */
		crv = cdw_config_read_from_file();

		/* TODO: move this to some new function that "resolves"
		   volume_size_value value (and perhaps values of other
		   cdw_config_t fields */
		if (global_config.volume_size_id == CDW_CONFIG_VOLUME_SIZE_CUSTOM) {
			global_config.volume_size_value = global_config.volume_size_custom_value;
		} else {
			global_config.volume_size_value = cdw_config_volume_size_by_id(global_config.volume_size_id);
		}

		if (crv == CDW_GEN_ERROR) {
			failsafe_mode = true;
			cdw_vdm ("ERROR: failed to read config from file, setting failsafe mode\n");
			return CDW_FAILSAFE_MODE;
		} else {
			return CDW_OK;
		}
	}
}





/**
 * \brief Write current configuration to disk file
 *
 * Write content of config variable to disk file.
 *
 * This function only opens (for writing) config file, (in future: does
 * basic error checking) calls function doing actual writing, and then
 * closes config file.
 *
 * If configuration module works in failsafe mode, this function silently
 * skips writing to file. Caller is responsible for informing user about
 * failsafe_mode being set to true (and about consequences of this fact).
 *
 * \return CDW_OK on success
 * \return CDW_GEN_ERROR if function failed to open file for writing
 * \return CDW_FAILSAFE_MODE if failsafe mode is in effect and writing was skipped
 */
cdw_rv_t cdw_config_write_to_file(void)
{
	if (failsafe_mode) {
		/* this might be redundant if caller checks for
		   failsafe_mode, but just to be sure: silently skip writing */

		/* emergency mode, don't work with filesystem */
		return CDW_FAILSAFE_MODE;
	}

	cdw_assert (config_file_full_path != (char *) NULL, "full path to config file is null\n");

	FILE *config_file = fopen(config_file_full_path, "w");
	if (config_file == (FILE *) NULL) {
		int e = errno;
		cdw_vdm ("ERROR: failed open config file \"%s\" for writing\n", config_file_full_path);

		if (e == EACCES) { /* conf file has no write permissions */
			/* 2TRANS: this is title of dialog window */
			cdw_buttons_dialog(_("Config file error"),
					   /* 2TRANS: this is message in dialog window */
					   _("An error occurred when saving configuration. Please check config file permissions. Any changes will be lost after closing cdw."),
					   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		} else {
			/* 2TRANS: this is title of dialog window */
			cdw_buttons_dialog(_("Config file error"),
					   /* 2TRANS: this is message in dialog window */
					   _("Unknown error occurred when saving configuration. Any changes will be lost after closing cdw."),
					   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
		}

		return CDW_GEN_ERROR;
	} else {
		cdw_config_var_write_to_file(config_file, &global_config);
		fclose(config_file);
		config_file = (FILE *) NULL;
		return CDW_OK;
	}
}





/**
 * \brief Read config file, put config values into config variable (high-level)
 *
 * Open config file, call function reading config file, close config
 * file. If config file does not exists, it is not created.
 *
 * If configuration module works in failsafe mode, this function silently skips
 * reading from file. Caller is responsible for informing user about
 * failsafe_mode being set to true (and about consequences of this fact).
 *
 * \return CDW_OK if configuration was read from file without problems
 * \return CDW_NO if there was no configuration file and now we are using default configuration
 * \return CDW_FAILSAFE_MODE if failsafe mode is in effect and reading was skipped (program is using default configuration now)
 * \return CDW_GEN_ERROR if function failed to open config file
 */
cdw_rv_t cdw_config_read_from_file(void)
{
	if (failsafe_mode) {
		cdw_vdm ("ERROR: failsafe mode is in effect, not reading config file\n");
		return CDW_FAILSAFE_MODE; /* emergency mode, don't work with filesystem */
	}

	cdw_assert (config_file_full_path != (char *) NULL, "full path to config file is null\n");

	FILE *config_file = fopen(config_file_full_path, "r");
	if (config_file == (FILE *) NULL) {
		int e = errno;
		cdw_vdm ("WARNING: failed open config file \"%s\" for reading (errno = \"%s\")\n",
			 config_file_full_path, strerror(e));

		if (e == ENOENT) { /* no such file */
			/* 2TRANS: this is title of dialog window */
			cdw_buttons_dialog(_("Config file problems"),
					   /* 2TRANS: this is message in dialog window */
					   _("Cannot open config file. Will use default configuration."),
					   CDW_BUTTONS_OK, CDW_COLORS_WARNING);
			return CDW_NO;
		} else if (e == EACCES) { /* file permission problems */
			/* 2TRANS: this is title of dialog window */
			cdw_buttons_dialog(_("Config file error"),
					   /* 2TRANS: this is message in dialog window */
					   _("Cannot open config file. Please check file permissions."),
					   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
			return CDW_GEN_ERROR;
		} else {
			/* 2TRANS: this is title of dialog window */
			cdw_buttons_dialog(_("Config file error"),
					   /* 2TRANS: this is message in dialog window */
					   _("Unknown error occurred when reading configuration. Default values will be used."),
					   CDW_BUTTONS_OK, CDW_COLORS_ERROR);
			return CDW_GEN_ERROR;
		}
	} else { /* config file already exists and it is now open to read */
		cdw_rv_t r = cdw_config_var_read_from_file(config_file, &global_config);
		fclose(config_file);
		config_file = (FILE *) NULL;
		if (r == CDW_OK) {
			return CDW_OK; /* configuration read from existing configuration file */
		} else { /* problems with reading config file */
			cdw_vdm ("ERROR: failed to read from config file\n");
			return CDW_GEN_ERROR;
		}
	}
}





/**
   \brief Set path to cdw config file and to "base directory"

   Function uses call to cdw fs module to get path to home dir, which is
   used to initialize two paths used by config module:
   Base directory full path (char *base_dir_fullpath)
   Full config file path (char *config_file_full_path)

   If path to user home directory can't be obtained, the function tries
   to get path to tmp directory. If this fails too, function returns
   CDW_GEN_ERROR and both paths are set to NULL.

   If function finishes successfully, dir paths in the two variables are
   the same, the difference is in file name in config_file_full_path.
   Base dir path is ended with slash.

   On success, if function used user home directory, CDW_OK is returned.
   If function used tmp directory, CDW_NO is returned.

   \return CDW_OK if base_dir_fullpath is set to $HOME
   \return CDW_NO if base_dir_fullpath is set to tmp dir
   \return CDW_GEN_ERROR function can't get any path because of errors
*/
cdw_rv_t cdw_config_module_set_paths(void)
{
	cdw_assert (base_dir_fullpath == (char *) NULL,
		    "ERROR: called this function when base dir is already initialized\n");
	cdw_assert (config_file_full_path == (char *) NULL,
		    "ERROR: called this function when base dir is already initialized\n");

	bool base_is_home = false;
	const char *path = cdw_fs_get_home_dir_fullpath();
	if (path == (char *) NULL) {
		path = cdw_fs_get_tmp_dir_fullpath();
		if (path == (char *) NULL) {
			cdw_vdm ("ERROR: failed to get any path from fs module\n");
			return CDW_GEN_ERROR;
		} else {
			/* there is some base dir, but it is not $HOME; */
			base_is_home = false;
		}
	} else {
		/* there is some base dir, and it is $HOME; */
		base_is_home = true;
	}

	base_dir_fullpath = strdup(path);
	if (base_dir_fullpath == (char *) NULL) {
		cdw_vdm ("ERROR: failed to create base dir fullpath\n");
		return CDW_GEN_ERROR;
	}
	cdw_vdm ("INFO: base dir path initialized as \"%s\"\n", base_dir_fullpath);
	cdw_vdm ("INFO: config file name = \"%s\"\n", config_file_name);

	/* at this point we have path to user home directory
	   or some tmp directory; now we set path to conf file */

	config_file_full_path = cdw_string_concat(base_dir_fullpath, config_file_name, (char *) NULL);
	if (config_file_full_path == (char *) NULL) {
		base_dir_fullpath = (char *) NULL;
		cdw_vdm ("ERROR: failed to create full path to config file\n");
		return CDW_GEN_ERROR;
	} else {
		cdw_vdm ("INFO: concatenated full path to conf file: \"%s\"\n", config_file_full_path);
		if (base_is_home) {
			return CDW_OK;
		} else {
			return CDW_NO;
		}
	}
}





/**
   \brief Deallocate 'configuration' module resources

   Deallocate all 'configuration' module resources that were allocated during
   program run and were not freed before.
*/
void cdw_config_module_clean(void)
{
	if (config_file_full_path != NULL) {
		/* some changes could be made outside of cdw configuration window,
		   this line ensures that the changes will be saved */
		cdw_config_write_to_file();
	} else {
		/* may happen if config module wasn't correctly
		   initialized, e.g. when there are some problems
		   at startup of application */
		cdw_vdm ("ERROR: path to config file not initialized\n");
	}

	if (config_file_full_path != (char *) NULL) {
		free(config_file_full_path);
		config_file_full_path = (char *) NULL;
	}

	if (base_dir_fullpath != (char *) NULL) {
		free(base_dir_fullpath);
		base_dir_fullpath = (char *) NULL;
	}

	cdw_config_var_free_fields(&global_config);

	return;
}





/**
   \brief Copy values of options from one configuration variable to other

   Function does not validate \p src nor \p dest

   \param dest - config variable to be used as destination of copied values
   \param src - config variable to be used as source of copied values

   \return CDW_GEN_ERROR if function failed to copy one of fields
   \return CDW_OK on success
*/
cdw_rv_t cdw_config_var_copy(cdw_config_t *dest, cdw_config_t *src)
{
	cdw_rv_t ss = CDW_MEM_ERROR;

	/* page A (writing) */
	strcpy(dest->dao, src->dao); /* obsolete, remove in future */
	dest->pad = src->pad;
	dest->pad_size = src->pad_size;
	dest->erase_mode = src->erase_mode;
	dest->eject = src->eject;
	dest->depreciated_speed = src->depreciated_speed; /* kept for compatibility */
	dest->speed_range = src->speed_range;
	dest->burnproof = src->burnproof;
	dest->dummy = src->dummy;
	ss = cdw_string_set(&(dest->other_cdrecord_options), src->other_cdrecord_options);
	if (ss != CDW_OK) {
		cdw_vdm ("ERROR: failed to set other_cdrecord_options from src = \"%s\"\n", src->other_cdrecord_options);
		return CDW_GEN_ERROR;
	}
	ss = cdw_string_set(&(dest->other_growisofs_options), src->other_growisofs_options);
	if (ss != CDW_OK) {
		cdw_vdm ("ERROR: failed to set other_growisofs_options from src = \"%s\"\n", src->other_growisofs_options);
		return CDW_GEN_ERROR;
	}

	/* page B (hardware) */
	strncpy(dest->custom_drive, src->custom_drive, OPTION_FIELD_LEN_MAX);
	dest->custom_drive[OPTION_FIELD_LEN_MAX] = '\0';
	strncpy(dest->selected_drive, src->selected_drive, OPTION_FIELD_LEN_MAX);
	dest->selected_drive[OPTION_FIELD_LEN_MAX] = '\0';
	strncpy(dest->scsi, src->scsi, OPTION_FIELD_LEN_MAX);
	dest->scsi[OPTION_FIELD_LEN_MAX] = '\0';
	strncpy(dest->mountpoint, src->mountpoint, OPTION_FIELD_LEN_MAX);
	dest->mountpoint[OPTION_FIELD_LEN_MAX] = '\0';
	strncpy(dest->cdrom, src->cdrom, OPTION_FIELD_LEN_MAX);
	dest->cdrom[OPTION_FIELD_LEN_MAX] = '\0';



	/* page C (audio) */
	ss = cdw_string_set(&(dest->audiodir), src->audiodir);
	if (ss != CDW_OK) {
		cdw_vdm ("ERROR: failed to set audiodir from src = \"%s\"\n", src->audiodir);
		return CDW_GEN_ERROR;
	}


	/* page D (iso filesystem) */
	strcpy(dest->volumeid, src->volumeid);
	dest->showvol = src->showvol;
	dest->iso_level = src->iso_level;
	dest->joliet = src->joliet;
	dest->rockridge = src->rockridge;
	dest->useful_rr = src->useful_rr;
	dest->joliet_long = src->joliet_long;
	dest->follow_symlinks = src->follow_symlinks;
	ss = cdw_string_set(&(dest->iso_image_full_path), src->iso_image_full_path);
	if (ss != CDW_OK) {
		cdw_vdm ("ERROR: failed to set iso_image_full path from src = \"%s\"\n", src->iso_image_full_path);
		return CDW_GEN_ERROR;
	}
	ss = cdw_string_set(&(dest->boot_disc_options), src->boot_disc_options);
	if (ss != CDW_OK) {
		cdw_vdm ("ERROR: failed to set boot_disc options = \"%s\"\n", src->boot_disc_options);
		return CDW_GEN_ERROR;
	}
	ss = cdw_string_set(&(dest->other_mkisofs_options), src->other_mkisofs_options);
	if (ss != CDW_OK) {
		cdw_vdm ("ERROR: failed to set other_mkisofs_options from src = \"%s\"\n", src->other_mkisofs_options);
		return CDW_GEN_ERROR;
	}


	/* page F (log and other) */
	ss = cdw_string_set(&(dest->log_full_path), src->log_full_path);
	if (ss != CDW_OK) {
		cdw_vdm ("ERROR: failed to set log_full_path from src = \"%s\"\n", src->log_full_path);
		return CDW_GEN_ERROR;
	}
	dest->showlog = src->showlog;

	dest->volume_size_id = src->volume_size_id;
	dest->volume_size_custom_value = src->volume_size_custom_value;
	dest->volume_size_value = src->volume_size_value;


	/* other, without field in config window */
	dest->support_dvd_rp_dl = src->support_dvd_rp_dl;
	dest->show_dvd_rw_support_warning = src->show_dvd_rw_support_warning;


	/* other, unused */
	strcpy(dest->autodic, src->autodic);
	strcpy(dest->bitsperchn, src->bitsperchn);
	strcpy(dest->stereo, src->stereo);
	strcpy(dest->echosound, src->echosound);
	strcpy(dest->encode, src->encode);
	strcpy(dest->lame, src->lame);
	strcpy(dest->highq, src->highq);
	strcpy(dest->bitrate, src->bitrate);
	strcpy(dest->cdbhost, src->cdbhost);
	strcpy(dest->cdbuser, src->cdbuser);
	strcpy(dest->sqlite_file, src->sqlite_file);

	return CDW_OK;
}





/**
   \brief Check if values in config variable are valid

   Check values stored in given configuration variable.
   Function checks only one specified subset of all config variable fields,
   or all verifiable config variable fields, depending on value of \p page.
   Use page symbolic name PAGE_{A,B,C,D,E}_INDEX to select one, specific
   subset corresponding to page in configuration window (see definitions
   in cdw_config_ui.h). Set \p page to -1 if you want to check all
   verifiable values from all pages. Page numbers correspond to
   configuration pages created in options_ui.c file, and fields of
   configuration variable are grouped just like fields in options_ui.c.

   If page, for which caller requested validation (or any page if \p page
   is set to -1) contains errors, then \p page will be set to number of
   page that has errors, and \p field will be set to number of invalid
   field in that page. CDW_NO is returned.

   If there are no invalid fields, value of \p field is unspecified, and
   CDW_OK is returned.

   Currently only part of field values are checked. E.g. 'pad' field is
   not checked.

   \param config - configuration variable to be checked
   \param page - number of page, from which parameters should be checked
   \param field - index of field with invalid value

   \return CDW_OK if all checked fields are valid
   \return CDW_NO if some field is invalid and the error must be fixed
*/
cdw_rv_t cdw_config_var_validate(cdw_config_t *config, int *page, int *field)
{
	/* indexes are defined in cdw_config_ui_internals.h: page_X_indexes */

	cdw_rv_t crv;
	if (*page == -1 || *page == PAGE_A_INDEX) { /* "writing" page */
		crv = cdw_string_security_parser(config->other_cdrecord_options, (char *) NULL);
		if (crv == CDW_NO) {
			*field = f_oco_i;
			*page = PAGE_A_INDEX;
			return CDW_NO;
		}
		crv = cdw_string_security_parser(config->other_growisofs_options, (char *) NULL);
		if (crv == CDW_NO) {
			*field = f_ogo_i;
			*page = PAGE_A_INDEX;
			return CDW_NO;
		}
		if (config->pad_size == -1) {
			*field = f_pad_size_i;
			*page = PAGE_A_INDEX;
			return CDW_NO;
		}
	}

	/* hardware */
	if (*page == -1 || *page == PAGE_B_INDEX) {
		int drive_id = cdw_drive_get_current_drive_id();
		if (drive_id == CDW_DRIVE_ID_CUSTOM) {
			/* user has selected "use custom drive path, and
			   the path cannot be empty; we only check if
			   the path is empty, we don't check if path is
			   in any way valid */

			if (!strlen(config->custom_drive)) {
				*field = f_custom_drive_i;
				*page = PAGE_B_INDEX;
				return CDW_NO;
			}
		}

		crv = cdw_string_security_parser(config->custom_drive, (char *) NULL);
		if (crv == CDW_NO) {
			*field = f_custom_drive_i;
			*page = PAGE_B_INDEX;
			return CDW_NO;
		}
		/* device path with ending slashes is not accepted by
		   other modules; info in config window asks to provide
		   device path without ending slashes, but let's
		   validate and fix path entered by user */
		size_t len = strlen(config->custom_drive);
		if (len > 1) {
			size_t i = 0;
			for (i = len - 1; i > 0; i--) {
				if (config->custom_drive[i] == '/') {
					config->custom_drive[i] = '\0';
				} else {
					break;
				}
			}
		}

		if (strlen(config->scsi)) {
			crv = cdw_string_security_parser(config->scsi, (char *) NULL);
			if (crv == CDW_NO) {
				*field = f_scsi_i;
				*page = PAGE_B_INDEX;
				return crv;
			}
		}

	}

	if (*page == -1 || *page == PAGE_C_INDEX) { /* "audio" page */
		crv = cdw_string_security_parser(config->audiodir, (char *) NULL);
		if (crv == CDW_NO) {
			*field = f_audiodir_i;
			*page = PAGE_C_INDEX;
			return CDW_NO;
		}
	}

	if (*page == -1 || *page == PAGE_D_INDEX) { /* "iso filesystem" page */
		crv = cdw_string_security_parser(config->volumeid, (char *) NULL);
		if (crv == CDW_NO) {
			*field = f_volumeid_i;
			*page = PAGE_D_INDEX;
			cdw_vdm ("ERROR: \"volume id\" string has insecure content\n");
			return CDW_NO;
		}
		crv = cdw_string_security_parser(config->iso_image_full_path, (char *) NULL);
		if (crv == CDW_NO) {
			*field = f_iso_image_full_path_i;
			*page = PAGE_D_INDEX;
			return CDW_NO;
		}
		crv = cdw_string_security_parser(config->boot_disc_options, (char *) NULL);
		if (crv == CDW_NO) {
			*field = f_boot_disc_options_i;
			*page = PAGE_D_INDEX;
			return CDW_NO;
		}
		crv = cdw_string_security_parser(config->other_mkisofs_options, (char *) NULL);
		if (crv == CDW_NO) {
			*field = f_omo_i;
			*page = PAGE_D_INDEX;
			return CDW_NO;
		}

		if (config->pad_size == -1) {
			*field = f_pad_size_i;
			*page = PAGE_D_INDEX;
			return CDW_NO;
		}

	}

	if (*page == -1 || *page == PAGE_E_INDEX) { /* "tools" page */
		; /* tool paths are read from "which" output, I won't validate this */
		; /* WARNING for the future: keep in mind that at some
		     occasions "tools" page is hidden, and it must not be
		     validated then, because in case of errors user will
		     be unable to visit the page to fix the errors */
	}

	if (*page == -1 || *page == PAGE_F_INDEX) { /* "log and other" page */
		if (!strlen(config->log_full_path)) {
			/* there has to be some log path specified */
			*field = f_log_fp_i;
			*page = PAGE_F_INDEX;
			return CDW_NO;
		}
		crv = cdw_string_security_parser(config->log_full_path, (char *) NULL);
		if (crv == CDW_NO) {
			*field = f_log_fp_i;
			*page = PAGE_F_INDEX;
			return CDW_NO;
		}
		if (config->volume_size_custom_value == -1) {
			*field = f_cust_volume_size_i;
			*page = PAGE_F_INDEX;
			return CDW_NO;
		}
	}

	return CDW_OK;
}





/**
   \brief Properly initialize fields of data structure of type cdw_config_t

   Set/initialize/allocate some fields of given variable of type
   cdw_config_t, so that it can be used safely by other C functions,
   i.e. there are no uninitialized pointers, and all non-NULL
   pointers point to some malloc()ed space.

   It is a good idea to call this function right after defining
   a cdw_config_t variable. You have to call this function before
   performing any operations on the variable.

   This function is different than cdw_config_var_set_defaults(),
   because cdw_config_var_set_defaults() is aware of cdw option values
   and sets higher-level values of fields that cdw_config_var_init_fields()
   only initialize with simple, low-level states.

   \p config must be valid, non-null pointer

   \param config - configuration variable with fields to initialize;

   \return CDW_OK on success
   \return CDW_MEM_ERROR on malloc() error
*/
cdw_rv_t cdw_config_var_init_fields(cdw_config_t *config)
{
	cdw_assert (config != (cdw_config_t *) NULL, "passing null config variable\n");

	/* page A (writing) */
	config->other_cdrecord_options = (char *) NULL;
	config->other_growisofs_options = (char *) NULL;
	config->log_full_path = (char *) NULL;


	/* page C (audio) */
	config->audiodir = (char *) NULL;


	/* page D (iso filesystem) */
	/* config->volumeid should be no longer than VOLUME_ID_LEN_MAX,
	   and since VOLUME_ID_LEN_MAX is rather small and constant, we
	   can call malloc(VOLUME_ID_LEN_MAX + 1) here and remember to
	   write no more than VOLUME_ID_LEN_MAX to config->volumeid */
	config->volumeid = (char *) malloc(VOLUME_ID_LEN_MAX + 1); /* 1 for ending '\0' */
	if (config->volumeid == NULL) {
		cdw_vdm ("ERROR: failed to allocate memory for volumeid\n");
		return CDW_MEM_ERROR;
	} else {
		*(config->volumeid) = '\0'; /* empty string */
	}
	config->iso_image_full_path = (char *) NULL;
	config->boot_disc_options = (char *) NULL;
	config->other_mkisofs_options = (char *) NULL;


	/* page E (external tools) */
	/* paths external tools are initialized by other module;
	   the paths aren't tightly related to options in this file */

	return CDW_OK;
}





/**
   \brief Deallocate all resources used by given cdw_config_t variable

   Free all memory that is referenced by fields (pointers) in given
   cdw_config_t variable.

   This function can recognize unallocated resources
   and can handle them properly.

   \param _config - config variable with fields to free

   \return CDW_OK otherwise
*/
cdw_rv_t cdw_config_var_free_fields(cdw_config_t *config)
{
	cdw_assert (config != (cdw_config_t *) NULL, "passing null config pointer\n");

	/* page A - writing */
	if (config->other_cdrecord_options != (char *) NULL) {
		free(config->other_cdrecord_options);
		config->other_cdrecord_options = (char *) NULL;
	}
	if (config->other_growisofs_options != (char *) NULL) {
		free(config->other_growisofs_options);
		config->other_growisofs_options = (char *) NULL;
	}


	/* page C - audio */
	if (config->audiodir != (char *) NULL) {
		free(config->audiodir);
		config->audiodir = (char *) NULL;
	}


	/* page D - iso filesystem */
	if (config->volumeid != (char *) NULL) {
		free(config->volumeid);
		config->volumeid = (char *) NULL;
	}
	if (config->iso_image_full_path != (char *) NULL) {
		free(config->iso_image_full_path);
		config->iso_image_full_path = (char *) NULL;
	}
	if (config->boot_disc_options != (char *) NULL) {
		free(config->boot_disc_options);
		config->boot_disc_options = (char *) NULL;
	}
	if (config->other_mkisofs_options != (char *) NULL) {
		free(config->other_mkisofs_options);
		config->other_mkisofs_options = (char *) NULL;
	}


	/* page F - log */
	if (config->log_full_path != (char *) NULL) {
		free(config->log_full_path);
		config->log_full_path = (char *) NULL;
	}

	return CDW_OK;
}





bool cdw_config_has_scsi_device(void)
{
	if (!strlen(global_config.scsi)) {
		return false;
	} else {
		return true;
	}
}




/* ***************************** */
/* ***** private functions ***** */
/* ***************************** */





/**
   \brief Set default values in given cdw configuration variable

   Set default values of fields in given configuration variable.
   This function can be called to make sure that some default values
   in configuration variable exist - just in case if config file does
   not exists or is broken or incomplete.

   This function depends on base_dir_fullpath to be set. If it is not set to some
   valid directory (which is indicated by failsafe_mode set to true), last
   attempt is made to set base dir to sane value: is is set to "/tmp".

   \param config - config variable in which to set default values

   \return CDW_MEM_ERROR on malloc() errors
   \return CDW_OK on success
 */
cdw_rv_t cdw_config_var_set_defaults(cdw_config_t *config)
{
	cdw_assert (base_dir_fullpath != (char *) NULL, "base_dir_fullpath must not be null\n");


	/* page A - writing */
	strcpy(config->dao, "0"); /* obsolete, remove in future */
	config->pad = true;
	/* the "150" value is selected after small tests with DVD; originally it was 63, which
	   worked just fine for CDs, but for DVDs it was insufficient;
	   TODO: the same value is used in code creating configuration forms,
	   so it should be a constant defined in some header */
	/* OLD COMMENT: value of 63 is taken from here:
	   http://www.troubleshooters.com/linux/coasterless.htm */
	config->pad_size = 150;
	config->erase_mode = CDW_ERASE_MODE_FAST;
	config->eject = false;
	config->depreciated_speed = 4;
	config->speed_range = SPEED_RANGE_AUTO;
	config->burnproof = true;
	config->dummy = false;

	/* can't set to (char *) NULL because these options are used as
	argument of concat() in some places */
	cdw_string_set(&(config->other_cdrecord_options), " ");
	if (config->other_cdrecord_options == (char *) NULL) {
		cdw_vdm ("ERROR: failed to set other_cdrecord_options\n");
		return CDW_MEM_ERROR;
	}

	memset(config->other, '\0', OPTION_FIELD_LEN_MAX + 1); /* kept for backward compatibility */

	cdw_string_set(&(config->other_growisofs_options), " ");
	if (config->other_growisofs_options == (char *) NULL) {
		cdw_vdm ("ERROR: failed to set other_growisofs_options\n");
		return CDW_MEM_ERROR;
	}


	/* page B - hardware */
	strcpy(config->custom_drive, ""); /* since we have drive discovery, default value for this field is not needed */
	strcpy(config->selected_drive, "default");
	strcpy(config->scsi,""); /* some (most?) users will prefer to leave it empty */
	strcpy(config->mountpoint,"/cdrom");
	strcpy(config->cdrom, "/dev/cdrom");


	/* page C - audio */
	cdw_rv_t crv = CDW_NO;
	if (failsafe_mode) {
		crv = cdw_string_set(&(config->audiodir), "/tmp/audio/");
	} else {
		char *tmp2 = cdw_string_concat(base_dir_fullpath, "audio/", (char *) NULL);
		if (tmp2 == (char *) NULL) {
			cdw_vdm ("ERROR: failed to concat string for audiodir\n");
			return CDW_MEM_ERROR;
		} else {
			crv = cdw_string_set(&(config->audiodir), tmp2);

			free(tmp2);
			tmp2 = (char *) NULL;
		}
	}
	if (config->audiodir == (char *) NULL || crv != CDW_OK) {
		cdw_vdm ("ERROR: failed to set audiodir\n");
		return CDW_MEM_ERROR;
	}



	/* page D - iso filesystem */
	cdw_assert (config->volumeid != (char *) NULL, "you forgot to init volumeid field\n");
	strcpy(config->volumeid, "cdrom");
	config->showvol = false;
	config->iso_level = 3;
	config->joliet = true;
	config->rockridge = true;
	config->useful_rr = true;
	config->joliet_long = false; /* breaks a standard, use with care */
	config->follow_symlinks = false;

	cdw_string_set(&(config->iso_image_full_path), "/tmp/image.iso");
	if (config->iso_image_full_path == (char *) NULL) {
		cdw_vdm ("ERROR: failed to set iso_image_full_path\n");
		return CDW_MEM_ERROR;
	}
	cdw_string_set(&(config->boot_disc_options), "");
	/* can't set to (char *) NULL because this option is used as
	   argument of concat() in some places */
	cdw_string_set(&(config->other_mkisofs_options), " ");


	/* page F - log and other */
#ifndef NDEBUG
	size_t len = strlen(base_dir_fullpath);
	cdw_assert (base_dir_fullpath[len - 1] == '/',
		    "ERROR: base_dir_fullpath is not ended with slash: \"%s\"\n",
		    base_dir_fullpath);
#endif
	char *tmp = cdw_string_concat(base_dir_fullpath, cdw_log_file_name, (char *) NULL);
	if (tmp == (char *) NULL) {
		cdw_vdm ("ERROR: failed to concat string for log_full_path\n");
		return CDW_MEM_ERROR;
	} else {
		cdw_string_set(&(config->log_full_path), tmp);
		free(tmp);
		tmp = (char *) NULL;

		if (config->log_full_path == (char *) NULL) {
			cdw_vdm ("ERROR: failed to set log_full_path to \"%s\"\n", tmp);
			return CDW_MEM_ERROR;
		}
	}

	config->showlog = true;

	config->volume_size_id = CDW_CONFIG_VOLUME_SIZE_CD74;
	config->volume_size_custom_value = cdw_config_volume_size_by_id(config->volume_size_id);
	config->volume_size_value = cdw_config_volume_size_by_id(config->volume_size_id);


	/* other - not saved in config file */
	config->support_dvd_rp_dl = false;
	/* show warning dialog; dialog code will display a message box,
	   and will set this flag to "false", to not to annoy user with
	   repeated messages */
	config->show_dvd_rw_support_warning = true;


	/* other - unused */
	strcpy(config->stereo,"1");
	strcpy(config->bitsperchn,"16");
	strcpy(config->echosound,"0");
	strcpy(config->encode,"0");
	strcpy(config->lame,"0");
	strcpy(config->highq,"1");
	strcpy(config->bitrate,"192");
	strcpy(config->cdbhost,"localhost");
	strcpy(config->cdbuser,"root");
	snprintf(config->sqlite_file, 255 + 1, "%s/cdw.db", failsafe_mode ? "/tmp" : base_dir_fullpath);
	sprintf(config->autodic, " ");



	return CDW_OK;
}





/**
   \brief Write values from given configuration variable to cdw config file

   Write values from given configuration variable config to hard disk
   configuration file config_file. This is low level function that does actual
   writing to file. The file should be opened for writing before calling this
   function. This function will not check for validity of file nor
   configuration variable.

   This is the only function that writes to configuration file. If you
   would like to change layout of config file - you can do it here.

   \param config_file - config file already opened for writing
   \param config - configuration variable with option values to be written to file
*/
void cdw_config_var_write_to_file(FILE *config_file, cdw_config_t *config)
{
	fprintf(config_file, "####################################\n");
	fprintf(config_file, "#   %s %s configuration file       \n", PACKAGE, VERSION);
	fprintf(config_file, "####################################\n");


	fprintf(config_file, "\n\n\n");
	fprintf(config_file, "########## \"Writing\" page #########\n");
	fprintf(config_file, "dao=%s\n", config->dao); /* obsolete, remove in future */
	fprintf(config_file, "pad=%d\n", config->pad ? 1 : 0);
	fprintf(config_file, "pad_size=%d\n", config->pad_size);

	fprintf(config_file, "\n# fast / all\n");
	fprintf(config_file, "blank=%s\n", config->erase_mode == CDW_ERASE_MODE_FAST ? "fast" : "all");

	fprintf(config_file, "eject=%d\n", config->eject ? 1 : 0);

	fprintf(config_file, "# \"speed\" field is depreciated\n");
	fprintf(config_file, "speed=%d\n",(int)  config->depreciated_speed); /* kept for compatibility - new versions of cdw (0.3.9X and up) don't read this from config file, but older do */
	fprintf(config_file, "\n# 0 - auto, 1 - lowest range, 2 - middle range, 3 - highest range\n"); /* small explanation for users who would like to edit configuration file by hand */
	fprintf(config_file, "speed_range=%d\n", (int) config->speed_range);

	fprintf(config_file, "burnproof=%d\n", config->burnproof ? 1 : 0);
	fprintf(config_file, "dummy=%d\n", config->dummy ? 1 : 0);

	fprintf(config_file, "other_cdrecord_options=%s\n", (config->other_cdrecord_options != (char *) NULL) ? config->other_cdrecord_options : "");
	fprintf(config_file, "other=%s\n", config->other); /* kept for backward compatibility */
	fprintf(config_file, "other_growisofs_options=%s\n", (config->other_growisofs_options != (char *) NULL) ? config->other_growisofs_options : "");



	fprintf(config_file, "\n\n\n");
	fprintf(config_file, "###### \"Hardware\" page #######\n");
	fprintf(config_file, "\n## depreciated entries\n");
	/* TODO: remove these fields in future */
	fprintf(config_file, "# 'scsi-dev' field is depreciated, use scsi\n");
	fprintf(config_file, "scsi-dev=%s\n", config->scsi);
	fprintf(config_file, "# 'cdrwdevice' field is depreciated, use custom_drive\n");
	fprintf(config_file, "cdrwdevice=%s\n", config->custom_drive);
	/* TODO: remove these fields in future */
	fprintf(config_file, "\n# unused entries\n");
	fprintf(config_file, "mountpoint=%s\n", config->mountpoint);
	fprintf(config_file, "cdrom=%s\n", config->cdrom);

	fprintf(config_file, "\n# string indicating drive to be used by cdw\n");
	fprintf(config_file, "selected_drive=%s\n", config->selected_drive);
	fprintf(config_file, "\n# manually entered path to a drive\n");
	fprintf(config_file, "custom_drive=%s\n", config->custom_drive);
	fprintf(config_file, "\n# SCSI device descriptor, used by cdrecord\n");
	fprintf(config_file, "scsi=%s\n", config->scsi);



	fprintf(config_file, "\n\n\n");
	fprintf(config_file, "########## \"Audio\" page ###########\n");
	fprintf(config_file, "audiodir=%s\n", (config->audiodir != (char *) NULL) ? config->audiodir : "");


	fprintf(config_file, "\n\n\n");
	fprintf(config_file, "########## \"ISO filesystem\" page ###########\n");
	fprintf(config_file, "volumeid=%s\n", config->volumeid);
	fprintf(config_file, "showvol=%d\n", config->showvol ? 1 : 0);

	fprintf(config_file, "\n# allowed values for iso_level are 1, 2, 3, 4\n");
	fprintf(config_file, "iso_level=%d\n", config->iso_level);

	fprintf(config_file, "joliet=%d\n", config->joliet ? 1 : 0);
	fprintf(config_file, "rockridge=%d\n", config->rockridge ? 1 : 0);
	fprintf(config_file, "\n## \"usefulRR\" is depreciated, use \"useful_rr\"\n");
	fprintf(config_file, "usefulRR=%d\n", config->useful_rr ? 1 : 0);
	fprintf(config_file, "useful_rr=%d\n", config->useful_rr ? 1 : 0);
	fprintf(config_file, "joliet_long=%d\n", config->joliet_long ? 1 : 0);
	fprintf(config_file, "follow_symlinks=%d\n", config->follow_symlinks ? 1 : 0);

	fprintf(config_file, "\n## \"tempdir\" is depreciated, use \"iso_image_full_path\"\n");
	fprintf(config_file, "tempdir=%s\n", (config->iso_image_full_path != (char *) NULL) ? config->iso_image_full_path : "");
	fprintf(config_file, "iso_image_full_path=%s\n", (config->iso_image_full_path != (char *) NULL) ? config->iso_image_full_path : "");
	fprintf(config_file, "\n## well, in fact this field stores all options related to boot disc\n");
	fprintf(config_file, "boot_image_path=%s\n", (config->boot_disc_options != (char *) NULL) ? config->boot_disc_options : "");
	fprintf(config_file, "other_mkisofs_options=%s\n", (config->other_mkisofs_options != (char *) NULL) ? config->other_mkisofs_options : "");

	fprintf(config_file, "\n\n\n");
	fprintf(config_file, "########## \"Log\" page ###########\n");

	fprintf(config_file, "showlog=%d\n", config->showlog ? 1 : 0);
	fprintf(config_file, "logfile=%s\n", (config->log_full_path != (char *) NULL) ? config->log_full_path : "");


	fprintf(config_file, "\n\n\n");
	fprintf(config_file, "cdsize=%s\n", cdw_config_volume_size_table[config->volume_size_id].new_label);
	fprintf(config_file, "cdsize=%s\n", cdw_config_volume_size_table[config->volume_size_id].old_label);
	fprintf(config_file, "user_cdsize=%ld\n", config->volume_size_custom_value);


	/* currently unused options follow */
	fprintf(config_file, "\n\n\n");
	fprintf(config_file, "autodic=%s\n", config->autodic);
	fprintf(config_file, "stereo=%s\n", config->stereo);
	fprintf(config_file, "bitsperchn=%s\n", config->bitsperchn);
	fprintf(config_file, "echosound=%s\n", config->echosound);
	fprintf(config_file, "encode=%s\n", config->encode);
	fprintf(config_file, "lame=%s\n", config->lame);
	fprintf(config_file, "highq=%s\n", config->highq);
	fprintf(config_file, "bitrate=%s\n", config->bitrate);


	fprintf(config_file, "\n\n\n");
	fprintf(config_file, "########### CDDB options ###########\n");
	fprintf(config_file, "cdbhost=%s\n", config->cdbhost);
	fprintf(config_file, "cdbuser=%s\n", config->cdbuser);
	fprintf(config_file, "sqlite_file=%s\n", config->sqlite_file);

	return;
}





/**
   \brief Read config file, store configuration in data structure (low-level code)

   Read config file line by line and store values found in that file into data
   structure. This is low level function that deals with disk file. It should
   be called by other function that will open file with correct permissions
   before passing it to this function.

   Data read from file is stored in \p config variable.

   This function calls cdw_string_security_parser() for every option
   value and additionally performs some basic validation of _some_ fields.
   When function finds some invalid value in config file, it discards it
   and doesn't modify existing value of field of \p config variable.

   \param config_file - config file opened for reading
   \param config - variable into which store values that were read

   \return CDW_OK on success
   \return CDW_MEM_ERROR on malloc() error
 */
cdw_rv_t cdw_config_var_read_from_file(FILE *config_file, cdw_config_t *config)
{
	cdw_rv_t ss = CDW_MEM_ERROR;

	char *line = (char *) NULL;
	while (1) {
		if (line != (char *) NULL) {
			free(line);
			line = (char *) NULL;
		}

		line = my_readline_10k(config_file);
		if (line == (char *) NULL) {
			break;
		}

		cdw_option_t option;
		/* these will be only pointers to places in existing string */
		option.name = (char *) NULL;
		option.value = (char *) NULL;

		option = cdw_config_split_options_line(line);
		if (option.name == (char *) NULL || option.value == (char *) NULL) {
			continue; /* empty or invalid line, or comment */
		}

		cdw_rv_t v = cdw_string_security_parser(option.value, (char *) NULL);
		if (v != CDW_OK) {
			cdw_vdm ("WARNING: option value \"%s=%s\" from config file rejected as insecure\n",
				 option.name, option.value);
			continue;
		}


		/* page A - writing */

		/* obsolete, remove in future */
		if (!strcasecmp(option.name, "dao")) {
			if (!strcmp(option.value, "0") || !strcmp(option.value, "1")) {
				strncpy(config->dao, option.value, 1);
				config->dao[1] = '\0';
			}
			continue;
		}

		if (!strcasecmp(option.name, "pad")) {
			cdw_string_get_bool_value(option.value, &(config->pad));
			continue;
		}

		if (!strcasecmp(option.name, "pad_size")) {
			config->pad_size = atoi(option.value);
			continue;
		}


		if (!strcasecmp(option.name, "blank")) {
			/* compare ignoring case, but store with correct case */
			if (!strcasecmp(option.value, "fast")) {
				config->erase_mode = CDW_ERASE_MODE_FAST;
			} else if (!strcasecmp(option.value, "all")) {
				config->erase_mode = CDW_ERASE_MODE_ALL;
			} else {
				; /* there should be some default value in option.name already */
			}
			continue;
		}

		if (!strcasecmp(option.name, "eject")) {
			cdw_string_get_bool_value(option.value, &(config->eject));
			continue;
		}

		if (!strcasecmp(option.name, "speed")) {
			int i = atoi(option.value);
			if (i < 0 || i > 50) {
				; /* don't change default value that
				     should already be assigned */
			} else {
				config->depreciated_speed = (size_t) i;
			}
			continue;
		}

		if (!strcasecmp(option.name, "speed_range")) {
			int i = atoi(option.value);
			if (i < 0 || i > 3) {
				; /* don't change default value that
				     should already be assigned */
			} else {
				config->speed_range = (size_t) i;
			}
			continue;
		}

		if (!strcasecmp(option.name, "burnproof")) {
			cdw_string_get_bool_value(option.value, &(config->burnproof));
			continue;
		}

		if (!strcasecmp(option.name, "dummy")) {
			cdw_string_get_bool_value(option.value, &(config->dummy));
			continue;
		}

		if (!strcasecmp(option.name, "other")) { /* depreciated */
			strncpy(config->other, option.value, OPTION_FIELD_LEN_MAX);
			config->other[OPTION_FIELD_LEN_MAX] = '\0';
			continue;
		}

		if (!strcasecmp(option.name, "other_cdrecord_options")) {
			ss = cdw_string_set(&(config->other_cdrecord_options), option.value);
			if (ss != CDW_OK) {
				cdw_vdm ("ERROR: failed to set other_cdrecord_options from option.value = \"%s\"\n", option.value);
				return CDW_MEM_ERROR;
			}
			continue;
		}

		if (!strcasecmp(option.name, "other_growisofs_options")) {
			ss = cdw_string_set(&(config->other_growisofs_options), option.value);
			if (ss != CDW_OK) {
				cdw_vdm ("ERROR: failed to set other_growisofs_options from option.value = \"%s\"\n", option.value);
				return CDW_MEM_ERROR;
			}

			continue;
		}


		/* page B - hardware */
		/* ** hardware ** */

		if (!strcasecmp(option.name, "cdrw_device")) {
			if (!strlen(option.value)) {
				/* don't erase value that may already be set
				   after reading cdrw_device field from config file */
				;
			} else {
				strncpy(config->custom_drive, option.value, OPTION_FIELD_LEN_MAX);
				config->custom_drive[OPTION_FIELD_LEN_MAX] = '\0';
			}
			continue;
		}

		if (!strcasecmp(option.name, "cdrwdevice")) { /* depreciated */
			if (!strlen(option.value)) {
				/* don't erase value that may already be set
				   after reading cdrw_device field from config file */
				;
			} else {
				strncpy(config->custom_drive, option.value, OPTION_FIELD_LEN_MAX);
				config->custom_drive[OPTION_FIELD_LEN_MAX] = '\0';
			}
			continue;
		}

		if (!strcasecmp(option.name, "selected_drive")) {
			strncpy(config->selected_drive, option.value, OPTION_FIELD_LEN_MAX);
			config->selected_drive[OPTION_FIELD_LEN_MAX] = '\0';
			continue;
		}

		if (!strcasecmp(option.name, "scsi-dev")) { /* depreciated */
			strncpy(config->scsi, option.value, OPTION_FIELD_LEN_MAX);
			config->scsi[OPTION_FIELD_LEN_MAX] = '\0';
			continue;
		}

		if (!strcasecmp(option.name, "scsi")) {
			strncpy(config->scsi, option.value, OPTION_FIELD_LEN_MAX);
			config->scsi[OPTION_FIELD_LEN_MAX] = '\0';
			continue;
		}

		if (!strcasecmp(option.name, "cdrom")) { /* unused */
			strncpy(config->cdrom, option.value, OPTION_FIELD_LEN_MAX);
			config->cdrom[OPTION_FIELD_LEN_MAX] = '\0';
			continue;
		}

		if (!strcasecmp(option.name, "mountpoint")) { /* unused */
			strncpy(config->mountpoint, option.value, OPTION_FIELD_LEN_MAX);
			config->mountpoint[OPTION_FIELD_LEN_MAX] = '\0';
			continue;
		}



		/* page C - audio */
		if (!strcasecmp(option.name, "audiodir")) {
			ss = cdw_string_set(&(config->audiodir), option.value);
			if (ss != CDW_OK) {
				cdw_vdm ("ERROR: failed to set audiodir from option.value = \"%s\"\n", option.value);
				return CDW_MEM_ERROR;
			}
			continue;
		}



		/* page D - iso filesystem */
		if (!strcasecmp(option.name, "volumeid")) {
			strncpy(config->volumeid, option.value, VOLUME_ID_LEN_MAX);
			config->volumeid[VOLUME_ID_LEN_MAX] = '\0';
			cdw_vdm ("INFO: \"volume id\" read as \"%s\"\n", config->volumeid);
			continue;
		}

		if (!strcasecmp(option.name, "showvol")) {
			cdw_string_get_bool_value(option.value, &(config->showvol));
			continue;
		}

		if (!strcasecmp(option.name, "iso_level")) {
			int i = atoi(option.value);
			if (i < 1 || i > 4) {
				; /* probably "trash" value, rejecting completely */
			} else {
				config->iso_level = (size_t) i;
			}
			continue;
		}

		if (!strcasecmp(option.name, "joliet")) {
			cdw_string_get_bool_value(option.value, &(config->joliet));
			continue;
		}

		if (!strcasecmp(option.name, "rockridge")) {
			cdw_string_get_bool_value(option.value, &(config->rockridge));
			continue;
		}

		/* this condition should be removed in future - "usefulRR" is depreciated */
		if (!strcasecmp(option.name, "usefulRR")) {
			cdw_string_get_bool_value(option.value, &(config->useful_rr));
			continue;
		}
		if (!strcasecmp(option.name, "useful_rr")) {
			cdw_string_get_bool_value(option.value, &(config->useful_rr));
			continue;
		}

		if (!strcasecmp(option.name, "joliet_long")) {
			cdw_string_get_bool_value(option.value, &(config->joliet_long));
			continue;
		}

		if (!strcasecmp(option.name, "follow_symlinks")) {
			cdw_string_get_bool_value(option.value, &(config->follow_symlinks));
			continue;
		}

		/* this condition should be removed in future - "tempdir" is depreciated */
		if (!strcasecmp(option.name, "tempdir")) {
			ss = cdw_string_set(&(config->iso_image_full_path), option.value);
			if (ss != CDW_OK) {
				cdw_vdm ("ERROR: failed to set iso_image_full_path from option.value = \"%s\"\n", option.value);
				return CDW_MEM_ERROR;
			}
			continue;
		}
		if (!strcasecmp(option.name, "iso_image_full_path")) {
			ss = cdw_string_set(&(config->iso_image_full_path), option.value);
			if (ss != CDW_OK) {
				cdw_vdm ("ERROR: failed to set iso_image_full_path from option.value = \"%s\"\n", option.value);
				return CDW_MEM_ERROR;
			}
			continue;
		}

		if (!strcasecmp(option.name, "boot_image_path")) {
			ss = cdw_string_set(&(config->boot_disc_options), option.value);
			if (ss != CDW_OK) {
				cdw_vdm ("ERROR: failed to set boot_disc_options from option.value = \"%s\"\n", option.value);
				return CDW_MEM_ERROR;
			}

			continue;
		}

		if (!strcasecmp(option.name, "other_mkisofs_options")) {
			ss = cdw_string_set(&(config->other_mkisofs_options), option.value);
			if (ss != CDW_OK) {
				cdw_vdm ("ERROR: failed to set other_mkisofs_options from option.value = \"%s\"\n", option.value);
				return CDW_MEM_ERROR;
			}
			continue;
		}


		/* page F - log and other */
		if (!strcasecmp(option.name, "logfile")) {
			ss = cdw_string_set(&(config->log_full_path), option.value);
			if (ss != CDW_OK) {
				cdw_vdm ("ERROR: failed to set log_full_path from option.value = \"%s\"\n", option.value);
				return CDW_MEM_ERROR;
			}
			continue;
		}

		if (!strcasecmp(option.name, "showlog")) {
			cdw_string_get_bool_value(option.value, &(config->showlog));
			continue;
		}

		if (!strcasecmp(option.name, "cdsize")) {
			int id = cdw_config_volume_id_by_label(option.value);
			if (id != -1) {
				config->volume_size_id = id;
			}
			continue;
		}

		if (!strcasecmp(option.name, "user_cdsize")) {
			config->volume_size_custom_value = strtol(option.value, (char **) NULL, 10);
			continue;
		}


		/* other - unused */

		if (!strcasecmp(option.name, "autodic")) {
			strncpy(config->autodic, option.value, 1);
			config->autodic[1] = '\0';
			continue;
		}

		if (!strcasecmp(option.name, "stereo")) {
			strncpy(config->stereo, option.value, 1);
			config->stereo[1] = '\0';
			continue;
		}

		if (!strcasecmp(option.name, "bitsperchn")) {
			strncpy(config->bitsperchn, option.value, 2);
			config->bitsperchn[2] = '\0';
			continue;
		}

		if (!strcasecmp(option.name, "echosound")) {
			strncpy(config->echosound, option.value, 1);
			config->echosound[1] = '\0';
			continue;
		}

		if (!strcasecmp(option.name, "encode")) {
			strncpy(config->encode, option.value, 1);
			config->encode[1] = '\0';
			continue;
		}

		if (!strcasecmp(option.name, "lame")) {
			strncpy(config->lame, option.value, 1);
			config->lame[1] = '\0';
			continue;
		}

		if (!strcasecmp(option.name, "highq")) {
			strncpy(config->highq, option.value, 1);
			config->highq[1] = '\0';
			continue;
		}

		if (!strcasecmp(option.name, "bitrate")) {
			strncpy(config->bitrate, option.value, 3);
			config->bitrate[3] = '\0';
			continue;
		}

		if (!strcasecmp(option.name, "cdbhost")) {
			strncpy(config->cdbhost, option.value, 254);
			config->cdbhost[254] = '\0';
			continue;
		}

		if (!strcasecmp(option.name, "cdbuser")) {
			strncpy(config->cdbuser, option.value, 19);
			config->cdbuser[19] = '\0';
			continue;
		}

		if (!strcasecmp(option.name, "sqlite_file")) {
			strncpy(config->sqlite_file, option.value, 255);
			config->sqlite_file[255] = '\0';
			continue;
		}
	} /* while () */
	if (line != (char *) NULL) {
		free(line);
		line = (char *) NULL;
	}


	/* resolving backward compatibility issues */
	size_t len_old = strlen(config->other);
	size_t len_new = strlen(config->other_cdrecord_options);
	if (len_old > len_new) {
		/* older version of variable contains more than
		   new version of variable */
		cdw_rv_t o = cdw_string_set(&(config->other_cdrecord_options), config->other);
		if (o == CDW_OK) {
			;
		} else {
			cdw_vdm ("ERROR: failed to set string when resolving old options\n");
			return CDW_MEM_ERROR;
		}
	}

	return CDW_OK;
}





/**
   \brief Split option containing '=' char into 'name' and 'value' part

   The function takes one 'char *' string, recognizes where is first
   '=' char in it. What is on the left side of the char is treated
   as option name, and what is on the right side of the char is treated
   as option value. Pointers to beginning of both name and value are placed
   in variable of type cdw_option_t. These pointers point to substrings of
   original string.

   Original string is modified so that strings representing name and value
   don't have any white chars at the beginning and end. '\0' chars are
   also inserted into original string (function's argument) to properly
   delimit the two substrings.

   The function heavily modifies its argument!

   The function recognizes '#' as comment char and erases everything from
   line starting from '#' char;

   line must be proper char * string ending with '\0'

   Both fields of returned value are set to NULL if:
   \li function's argument is NULL;
   \li function's argument is empty;
   \li function's argument does not contain '=' char;
   \li function's argument is a comment line.
   \li function's argument has '=' char but don't have any option name

   Otherwise the fields are set with valid pointers pointing to two
   substrings in function's argument. Again: these substrings are proper
   C strings and are ended with '\0';

   \param line - line that you want to extract option from

   \return value of type cdw_option_t
*/
cdw_option_t cdw_config_split_options_line(char *line)
{
	cdw_sdm ("input line = '%s'\n", line);

	cdw_option_t option;
	option.name = (char *) NULL;
	option.value = (char *) NULL;

	if (line == (char *) NULL) {
		return option;
	}

	/* this will make sure that comment, starting at any position
	   in line, will be erased from line */
	char *comment = strstr(line, "#"); /* beginning of in-line comment */
	if (comment != (char *) NULL) {
		size_t len = strlen(comment);
		size_t i = 0;
		for (i = 0; i < len; i++) {
			*(comment + i) = '\0';
		}
	}

	char *tline = cdw_string_ltrim(cdw_string_rtrim(line));
	char *eq = strstr(tline, "="); /* first occurrence of '=' in line */

	if (eq == (char *) NULL) {
		option.name = (char *) NULL;
		option.value = (char *) NULL;

		cdw_sdm ("  line is invalid\n");

		return option;
	} else {
		option.name = tline;
		*eq = '\0';
		option.name = cdw_string_rtrim(option.name);

		if (!strcmp(option.name, "")) {
			option.name = (char *) NULL;
			option.value = (char *) NULL;
			return option;
		}

		option.value = cdw_string_ltrim(eq + 1);
		/* value may be empty string too, but this is not an error
		   condition, since given option value can be empty
		   (no value set) - this is correct situation */

		cdw_sdm ("   option.name = '%s'\n", option.name);
		cdw_sdm ("   option.value = '%s'\n", option.value);

		return option;
	}
}





/* simple getter - some files use global config variable just to get this field */
const char *cdw_config_get_custom_drive(void)
{
	return global_config.custom_drive;
}





void cdw_config_debug_print_config(cdw_config_t *config)
{
	cdw_vdm ("\n\nINFO: configuration:\n\n");
#if 1
	cdw_vdm ("INFO: Page A, \"writing\" settings:\n");
	cdw_vdm ("INFO:              erase mode = \"%s\"\n", config->erase_mode == CDW_ERASE_MODE_FAST ? "fast" : "all");
	cdw_vdm ("INFO:               eject bit = \"%s\"\n", config->eject ? "true" : "false");
	cdw_vdm ("INFO:             speed range = \"%d\"\n", config->speed_range);
	cdw_vdm ("INFO:               dummy bit = \"%s\"\n", config->dummy ? "true" : "false");
	cdw_vdm ("INFO:                 pad bit = \"%s\"\n", config->pad ? "true" : "false");
	cdw_vdm ("INFO:                pad size = \"%d\"\n", config->pad_size);
	cdw_vdm ("INFO:           burnproof bit = \"%s\"\n", config->burnproof ? "true" : "false");
	cdw_vdm ("INFO:  other cdrecord options = \"%s\"\n", config->other_cdrecord_options);
	cdw_vdm ("INFO: other growisofs options = \"%s\"\n\n", config->other_growisofs_options);
#endif


#if 1
	cdw_vdm ("INFO: Page B, \"hardware\" settings:\n");
	cdw_vdm ("INFO:   custom drive = \"%s\"\n", config->custom_drive);
	cdw_vdm ("INFO: selected drive = \"%s\"\n", config->selected_drive);
	cdw_vdm ("INFO:           scsi = \"%s\"\n\n", config->scsi);
#if 0 /* unused, maybe will be enabled in future */
	cdw_vdm ("INFO:     mountpoint = \"%s\"\n", config->mountpoint);
	cdw_vdm ("INFO:          cdrom = \"%s\"\n", config->cdrom);
#endif
#endif


#if 1

	cdw_vdm ("INFO: Page C, \"audio\" settings:\n");
	cdw_vdm ("INFO: audiodir = \"%s\"\n\n", config->audiodir);
#endif


#if 1
	cdw_vdm ("INFO: Page D, \"ISO filesystem\" settings:\n");
	cdw_vdm ("INFO:             volume id = \"%s\"\n", config->volumeid);
	cdw_vdm ("INFO:     ask for volume id = \"%s\"\n", config->showvol ? "true" : "false");
	cdw_vdm ("INFO:             iso level = \"%d\"\n", config->iso_level);
	cdw_vdm ("INFO:                joliet = \"%s\"\n", config->joliet ? "true" : "false");
	cdw_vdm ("INFO:             rockridge = \"%s\"\n", config->rockridge ? "true" : "false");
	cdw_vdm ("INFO:             useful RR = \"%s\"\n", config->useful_rr ? "true" : "false");
	cdw_vdm ("INFO:           joliet long = \"%s\"\n", config->joliet_long ? "true" : "false");
	cdw_vdm ("INFO:       follow_symlinks = \"%s\"\n", config->follow_symlinks ? "true" : "false");
	cdw_vdm ("INFO:   iso image full path = \"%s\"\n", config->iso_image_full_path);
	cdw_vdm ("INFO:     boot disc options = \"%s\"\n", config->boot_disc_options);
	cdw_vdm ("INFO: other mkisofs options = \"%s\"\n\n", config->other_mkisofs_options);
#endif


#if 1
	cdw_vdm ("INFO: Page E, \"tools\" settings:\n");
	cdw_ext_tools_debug_print_config();
#endif


#if 1
	cdw_vdm ("INFO: Page F, \"logging and other\" settings:\n");
	cdw_vdm ("INFO:            log full path = \"%s\"\n", config->log_full_path);
	cdw_vdm ("INFO:                 show log = \"%s\"\n", config->showlog ? "true" : "false");
	cdw_vdm ("INFO:           volume size id = \"%s\" (%d)\n",
		 cdw_utils_id_label_table_get_label(cdw_config_volume_size_items, config->volume_size_id),
		 config->volume_size_id);
	cdw_vdm ("INFO:        volume size value = %ld\n", config->volume_size_value);
	cdw_vdm ("INFO: volume size custom value = %ld\n", config->volume_size_custom_value);

#endif

#if 1
	cdw_vdm ("INFO: Other setting:\n");
	cdw_vdm ("INFO:           support DVD+RP DL = \"%s\"\n", config->support_dvd_rp_dl ? "true" : "false");
	cdw_vdm ("INFO: show DVD-RW support warning = \"%s\"\n", config->show_dvd_rw_support_warning ? "true" : "false");

#endif

	return;
}


#ifdef CDW_UNIT_TEST_CODE


/* *********************** */
/* *** unit tests code *** */
/* *********************** */


static void test_cdw_config_split_options_line(void);


void cdw_config_run_tests(void)
{
	fprintf(stderr, "testing cdw_config.c\n");

	test_cdw_config_split_options_line();

	fprintf(stderr, "done\n\n");

	return;
}



void test_cdw_config_split_options_line(void)
{
	fprintf(stderr, "\ttesting cdw_config_split_options_line()... ");

	int d;

	/* remember that option fields are pointers to places inside
	   input strings, not pointers to newly allocated strings */
	cdw_option_t option;
	option.name = NULL;
	option.value = NULL;

	/* correct line 1 */
	char *o1 = strdup("option name=value string");
	cdw_assert_test (o1 != NULL, "failed creating test line\n");
	option = cdw_config_split_options_line(o1);
	cdw_assert_test (option.name != NULL && option.value != NULL, "failed to split correct line\n");
	d = strcmp(option.name, "option name");
	cdw_assert_test (d == 0, "failed at comparing option name\n");
	d = strcmp(option.value, "value string");
	cdw_assert_test (d == 0, "failed at comparing option value\n");
	free(o1);
	o1 = (char *) NULL;

	/* correct line 2 - with some white chars */
	char *o2 = strdup("\t option name \t\t = \t value string \t");
	cdw_assert_test (o2 != NULL, "failed creating test line\n");
	option = cdw_config_split_options_line(o2);
	cdw_assert_test (option.name != NULL && option.value != NULL, "failed to split correct line\n");
	d = strcmp(option.name, "option name");
	cdw_assert_test (d == 0, "failed at comparing option name\n");
	d = strcmp(option.value, "value string");
	cdw_assert_test (d == 0, "failed at comparing option value\n");
	free(o2);
	o2 = (char *) NULL;

	/* correct line, but no value */
	char *o3 = strdup("\t option name = \t \t");
	cdw_assert_test (o3 != NULL, "failed creating test line\n");
	option = cdw_config_split_options_line(o3);
	cdw_assert_test (option.name != NULL && option.value != NULL, "failed to split correct line\n");
	d = strcmp(option.name, "option name");
	cdw_assert_test (d == 0, "failed comparing option name\n");
	d = strcmp(option.value, "");
	cdw_assert_test (d == 0, "failed comparint option value\n");
	free(o3);
	o3 = (char *) NULL;

	/* correct line, but no value 2 */
	char *o4 = strdup("\t option name = #value string\t \t");
	cdw_assert_test (o4 != NULL, "failed creatint test line\n");
	option = cdw_config_split_options_line(o4);
	cdw_assert_test (option.name != NULL && option.value != NULL, "failed to split correct line\n");
	d = strcmp(option.name, "option name");
	cdw_assert_test (d == 0, "failed at comparing option name\n");
	d = strcmp(option.value, "");
	cdw_assert_test (d == 0, "failed at comparing option value\n");
	free(o4);
	o4 = (char *) NULL;

	/* incorrect line, no '=' char 1 */
	char *o5 = strdup("\t option name \t # = value string\t \t");
	cdw_assert_test (o5 != NULL, "failed creatint test line\n");
	option = cdw_config_split_options_line(o5);
	cdw_assert_test (option.name == NULL && option.value == NULL, "failed at recognizing incorrect line\n");
	free(o5);
	o5 = (char *) NULL;

	/* incorrect line, no '=' char 2 */
	char *o6 = strdup("\t option name \t  value string\t \t");
	cdw_assert_test (o6!= NULL, "failed creatint test line\n");
	option = cdw_config_split_options_line(o6);
	cdw_assert_test (option.name == NULL && option.value == NULL, "failed at recognizing incorrect line\n");
	free(o6);
	o6 = (char *) NULL;

	/* incorrect line, all content is comment */
	char *o7 = strdup(" \t #\t option name = value string\t \t");
	cdw_assert_test (o7 != NULL, "failed creatint test line\n");
	option = cdw_config_split_options_line(o7);
	cdw_assert_test (option.name == NULL && option.value == NULL, "failed at recognizing incorrect line\n");
	free(o7);
	o7 = (char *) NULL;

	/* incorrect line, empty line */
	char *o8 = strdup("");
	cdw_assert_test (o8 != NULL, "failed creatint test line\n");
	option = cdw_config_split_options_line(o8);
	cdw_assert_test (option.name == NULL && option.value == NULL, "failed at recognizing incorrect line\n");
	free(o8);
	o8 = (char *) NULL;

	/* incorrect line, empty line 2 */
	char *o9 = strdup(" \t \t    \t\t\t\t     \t               \t#    ");
	cdw_assert_test (o9 != NULL, "failed creatint test line\n");
	option = cdw_config_split_options_line(o9);
	cdw_assert_test (option.name == NULL && option.value == NULL, "failed at recognizing incorrect line\n");
	free(o9);
	o9 = (char *) NULL;

	/* incorrect line, line = NULL */
	char *o10 = (char *) NULL;
	option = cdw_config_split_options_line(o10);
	cdw_assert_test (option.name == NULL && option.value == NULL, "failed at recognizing incorrect line\n");

	fprintf(stderr, "OK\n");

	return;
}


#endif /* #ifdef CDW_UNIT_TEST_CODE */
