package org.mandas.docker.client.messages.swarm;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * Immutable implementation of {@link ExternalCa}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableExternalCa.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableExternalCa implements ExternalCa {
  private final String protocol;
  private final String url;
  private final Map<String, String> options;

  private ImmutableExternalCa(
      String protocol,
      String url,
      Map<String, String> options) {
    this.protocol = protocol;
    this.url = url;
    this.options = options;
  }

  /**
   * @return The value of the {@code protocol} attribute
   */
  @JsonProperty("Protocol")
  @Override
  public String protocol() {
    return protocol;
  }

  /**
   * @return The value of the {@code url} attribute
   */
  @JsonProperty("URL")
  @Override
  public String url() {
    return url;
  }

  /**
   * @return The value of the {@code options} attribute
   */
  @JsonProperty("Options")
  @Override
  public Map<String, String> options() {
    return options;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ExternalCa#protocol() protocol} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for protocol
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableExternalCa withProtocol(String value) {
    String newValue = Objects.requireNonNull(value, "protocol");
    if (this.protocol.equals(newValue)) return this;
    return new ImmutableExternalCa(newValue, this.url, this.options);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ExternalCa#url() url} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for url
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableExternalCa withUrl(String value) {
    String newValue = Objects.requireNonNull(value, "url");
    if (this.url.equals(newValue)) return this;
    return new ImmutableExternalCa(this.protocol, newValue, this.options);
  }

  /**
   * Copy the current immutable object by replacing the {@link ExternalCa#options() options} map with the specified map.
   * Nulls are not permitted as keys or values.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param entries The entries to replace the options map
   * @return A modified copy or {@code this} if not changed
   */
  public final ImmutableExternalCa withOptions(Map<String, ? extends String> entries) {
    if (this.options == entries) return this;
    Map<String, String> newValue = createUnmodifiableMap(true, false, entries);
    return new ImmutableExternalCa(this.protocol, this.url, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableExternalCa} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableExternalCa
        && equalsByValue((ImmutableExternalCa) another);
  }

  private boolean equalsByValue(ImmutableExternalCa another) {
    return protocol.equals(another.protocol)
        && url.equals(another.url)
        && options.equals(another.options);
  }

  /**
   * Computes a hash code from attributes: {@code protocol}, {@code url}, {@code options}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + protocol.hashCode();
    h += (h << 5) + url.hashCode();
    h += (h << 5) + options.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code ExternalCa} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "ExternalCa{"
        + "protocol=" + protocol
        + ", url=" + url
        + ", options=" + options
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link ExternalCa} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable ExternalCa instance
   */
  public static ImmutableExternalCa copyOf(ExternalCa instance) {
    if (instance instanceof ImmutableExternalCa) {
      return (ImmutableExternalCa) instance;
    }
    return ImmutableExternalCa.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableExternalCa ImmutableExternalCa}.
   * <pre>
   * ImmutableExternalCa.builder()
   *    .protocol(String) // required {@link ExternalCa#protocol() protocol}
   *    .url(String) // required {@link ExternalCa#url() url}
   *    .addOption|putAllOptions(String =&gt; String) // {@link ExternalCa#options() options} mappings
   *    .build();
   * </pre>
   * @return A new ImmutableExternalCa builder
   */
  public static ImmutableExternalCa.Builder builder() {
    return new ImmutableExternalCa.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableExternalCa ImmutableExternalCa}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder {
    private static final long INIT_BIT_PROTOCOL = 0x1L;
    private static final long INIT_BIT_URL = 0x2L;
    private long initBits = 0x3L;

    private String protocol;
    private String url;
    private Map<String, String> options = new LinkedHashMap<String, String>();

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code ExternalCa} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * Collection elements and entries will be added, not replaced.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(ExternalCa instance) {
      Objects.requireNonNull(instance, "instance");
      this.protocol(instance.protocol());
      this.url(instance.url());
      putAllOptions(instance.options());
      return this;
    }

    /**
     * Initializes the value for the {@link ExternalCa#protocol() protocol} attribute.
     * @param protocol The value for protocol 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Protocol")
    public final Builder protocol(String protocol) {
      this.protocol = Objects.requireNonNull(protocol, "protocol");
      initBits &= ~INIT_BIT_PROTOCOL;
      return this;
    }

    /**
     * Initializes the value for the {@link ExternalCa#url() url} attribute.
     * @param url The value for url 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("URL")
    public final Builder url(String url) {
      this.url = Objects.requireNonNull(url, "url");
      initBits &= ~INIT_BIT_URL;
      return this;
    }

    /**
     * Put one entry to the {@link ExternalCa#options() options} map.
     * @param key The key in the options map
     * @param value The associated value in the options map
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addOption(String key, String value) {
      this.options.put(
          Objects.requireNonNull(key, "options key"),
          Objects.requireNonNull(value, value == null ? "options value for key: " + key : null));
      return this;
    }

    /**
     * Put one entry to the {@link ExternalCa#options() options} map. Nulls are not permitted
     * @param entry The key and value entry
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addOption(Map.Entry<String, ? extends String> entry) {
      String k = entry.getKey();
      String v = entry.getValue();
      this.options.put(
          Objects.requireNonNull(k, "options key"),
          Objects.requireNonNull(v, v == null ? "options value for key: " + k : null));
      return this;
    }

    /**
     * Sets or replaces all mappings from the specified map as entries for the {@link ExternalCa#options() options} map. Nulls are not permitted
     * @param entries The entries that will be added to the options map
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Options")
    public final Builder options(Map<String, ? extends String> entries) {
      this.options.clear();
      return putAllOptions(entries);
    }

    /**
     * Put all mappings from the specified map as entries to {@link ExternalCa#options() options} map. Nulls are not permitted
     * @param entries The entries that will be added to the options map
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder putAllOptions(Map<String, ? extends String> entries) {
      for (Map.Entry<String, ? extends String> e : entries.entrySet()) {
        String k = e.getKey();
        String v = e.getValue();
        this.options.put(
            Objects.requireNonNull(k, "options key"),
            Objects.requireNonNull(v, v == null ? "options value for key: " + k : null));
      }
      return this;
    }

    /**
     * Builds a new {@link ImmutableExternalCa ImmutableExternalCa}.
     * @return An immutable instance of ExternalCa
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableExternalCa build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableExternalCa(protocol, url, createUnmodifiableMap(false, false, options));
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_PROTOCOL) != 0) attributes.add("protocol");
      if ((initBits & INIT_BIT_URL) != 0) attributes.add("url");
      return "Cannot build ExternalCa, some of required attributes are not set " + attributes;
    }
  }

  private static <K, V> Map<K, V> createUnmodifiableMap(boolean checkNulls, boolean skipNulls, Map<? extends K, ? extends V> map) {
    switch (map.size()) {
    case 0: return Collections.emptyMap();
    case 1: {
      Map.Entry<? extends K, ? extends V> e = map.entrySet().iterator().next();
      K k = e.getKey();
      V v = e.getValue();
      if (checkNulls) {
        Objects.requireNonNull(k, "key");
        Objects.requireNonNull(v, v == null ? "value for key: " + k : null);
      }
      if (skipNulls && (k == null || v == null)) {
        return Collections.emptyMap();
      }
      return Collections.singletonMap(k, v);
    }
    default: {
      Map<K, V> linkedMap = new LinkedHashMap<>(map.size() * 4 / 3 + 1);
      if (skipNulls || checkNulls) {
        for (Map.Entry<? extends K, ? extends V> e : map.entrySet()) {
          K k = e.getKey();
          V v = e.getValue();
          if (skipNulls) {
            if (k == null || v == null) continue;
          } else if (checkNulls) {
            Objects.requireNonNull(k, "key");
            Objects.requireNonNull(v, v == null ? "value for key: " + k : null);
          }
          linkedMap.put(k, v);
        }
      } else {
        linkedMap.putAll(map);
      }
      return Collections.unmodifiableMap(linkedMap);
    }
    }
  }
}
