/*-
 * Copyright 2009 Colin Percival
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * This file was originally written by Colin Percival as part of the Tarsnap
 * online backup system.
 */

#include "php.h"
#ifdef PHP_WIN32
#include "zend_config.w32.h"
#endif

#include <errno.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>

#include "sha256.h"
#include "sysendian.h"

#include "crypto_scrypt.h"

static void blkcpy(uint8_t *, uint8_t *, size_t);
static void blkxor(uint8_t *, uint8_t *, size_t);
static void salsa20_8(uint8_t[64]);
static void blockmix_salsa8(uint8_t *, uint8_t *, size_t);
static uint64_t integerify(uint8_t *, size_t);
static void smix(uint8_t *, size_t, uint64_t, uint8_t *, uint8_t *);

static void
blkcpy(uint8_t * dest, uint8_t * src, size_t len)
{
    size_t i;

    for (i = 0; i < len; i++)
        dest[i] = src[i];
}

static void
blkxor(uint8_t * dest, uint8_t * src, size_t len)
{
    size_t i;

    for (i = 0; i < len; i++)
        dest[i] ^= src[i];
}

/**
 * salsa20_8(B):
 * Apply the salsa20/8 core to the provided block.
 */
static void
salsa20_8(uint8_t B[64])
{
    uint32_t B32[16];
    uint32_t x[16];
    size_t i;

    /* Convert little-endian values in. */
    for (i = 0; i < 16; i++)
        B32[i] = le32dec(&B[i * 4]);

    /* Compute x = doubleround^4(B32). */
    for (i = 0; i < 16; i++)
        x[i] = B32[i];
    for (i = 0; i < 8; i += 2) {
#define R(a,b) (((a) << (b)) | ((a) >> (32 - (b))))
        /* Operate on columns. */
        x[ 4] ^= R(x[ 0]+x[12], 7);  x[ 8] ^= R(x[ 4]+x[ 0], 9);
        x[12] ^= R(x[ 8]+x[ 4],13);  x[ 0] ^= R(x[12]+x[ 8],18);

        x[ 9] ^= R(x[ 5]+x[ 1], 7);  x[13] ^= R(x[ 9]+x[ 5], 9);
        x[ 1] ^= R(x[13]+x[ 9],13);  x[ 5] ^= R(x[ 1]+x[13],18);

        x[14] ^= R(x[10]+x[ 6], 7);  x[ 2] ^= R(x[14]+x[10], 9);
        x[ 6] ^= R(x[ 2]+x[14],13);  x[10] ^= R(x[ 6]+x[ 2],18);

        x[ 3] ^= R(x[15]+x[11], 7);  x[ 7] ^= R(x[ 3]+x[15], 9);
        x[11] ^= R(x[ 7]+x[ 3],13);  x[15] ^= R(x[11]+x[ 7],18);

        /* Operate on rows. */
        x[ 1] ^= R(x[ 0]+x[ 3], 7);  x[ 2] ^= R(x[ 1]+x[ 0], 9);
        x[ 3] ^= R(x[ 2]+x[ 1],13);  x[ 0] ^= R(x[ 3]+x[ 2],18);

        x[ 6] ^= R(x[ 5]+x[ 4], 7);  x[ 7] ^= R(x[ 6]+x[ 5], 9);
        x[ 4] ^= R(x[ 7]+x[ 6],13);  x[ 5] ^= R(x[ 4]+x[ 7],18);

        x[11] ^= R(x[10]+x[ 9], 7);  x[ 8] ^= R(x[11]+x[10], 9);
        x[ 9] ^= R(x[ 8]+x[11],13);  x[10] ^= R(x[ 9]+x[ 8],18);

        x[12] ^= R(x[15]+x[14], 7);  x[13] ^= R(x[12]+x[15], 9);
        x[14] ^= R(x[13]+x[12],13);  x[15] ^= R(x[14]+x[13],18);
#undef R
    }

    /* Compute B32 = B32 + x. */
    for (i = 0; i < 16; i++)
        B32[i] += x[i];

    /* Convert little-endian values out. */
    for (i = 0; i < 16; i++)
        le32enc(&B[4 * i], B32[i]);
}

/**
 * blockmix_salsa8(B, Y, r):
 * Compute B = BlockMix_{salsa20/8, r}(B).  The input B must be 128r bytes in
 * length; the temporary space Y must also be the same size.
 */
static void
blockmix_salsa8(uint8_t * B, uint8_t * Y, size_t r)
{
    uint8_t X[64];
    size_t i;

    /* 1: X <-- B_{2r - 1} */
    blkcpy(X, &B[(2 * r - 1) * 64], 64);

    /* 2: for i = 0 to 2r - 1 do */
    for (i = 0; i < 2 * r; i++) {
        /* 3: X <-- H(X \xor B_i) */
        blkxor(X, &B[i * 64], 64);
        salsa20_8(X);

        /* 4: Y_i <-- X */
        blkcpy(&Y[i * 64], X, 64);
    }

    /* 6: B' <-- (Y_0, Y_2 ... Y_{2r-2}, Y_1, Y_3 ... Y_{2r-1}) */
    for (i = 0; i < r; i++)
        blkcpy(&B[i * 64], &Y[(i * 2) * 64], 64);
    for (i = 0; i < r; i++)
        blkcpy(&B[(i + r) * 64], &Y[(i * 2 + 1) * 64], 64);
}

/**
 * integerify(B, r):
 * Return the result of parsing B_{2r-1} as a little-endian integer.
 */
static uint64_t
integerify(uint8_t * B, size_t r)
{
    uint8_t * X = &B[(2 * r - 1) * 64];

    return (le64dec(X));
}

/**
 * smix(B, r, N, V, XY):
 * Compute B = SMix_r(B, N).  The input B must be 128r bytes in length; the
 * temporary storage V must be 128rN bytes in length; the temporary storage
 * XY must be 256r bytes in length.  The value N must be a power of 2.
 */
static void
smix(uint8_t * B, size_t r, uint64_t N, uint8_t * V, uint8_t * XY)
{
    uint8_t * X = XY;
    uint8_t * Y = &XY[128 * r];
    uint64_t i;
    uint64_t j;

    /* 1: X <-- B */
    blkcpy(X, B, 128 * r);

    /* 2: for i = 0 to N - 1 do */
    for (i = 0; i < N; i++) {
        /* 3: V_i <-- X */
        blkcpy(&V[i * (128 * r)], X, 128 * r);

        /* 4: X <-- H(X) */
        blockmix_salsa8(X, Y, r);
    }

    /* 6: for i = 0 to N - 1 do */
    for (i = 0; i < N; i++) {
        /* 7: j <-- Integerify(X) mod N */
        j = integerify(X, r) & (N - 1);

        /* 8: X <-- H(X \xor V_j) */
        blkxor(X, &V[j * (128 * r)], 128 * r);
        blockmix_salsa8(X, Y, r);
    }

    /* 10: B' <-- X */
    blkcpy(B, X, 128 * r);
}

/**
 * crypto_scrypt(passwd, passwdlen, salt, saltlen, N, r, p, buf, buflen):
 * Compute scrypt(passwd[0 .. passwdlen - 1], salt[0 .. saltlen - 1], N, r,
 * p, buflen) and write the result into buf.  The parameters r, p, and buflen
 * must satisfy r * p < 2^30 and buflen <= (2^32 - 1) * 32.  The parameter N
 * must be a power of 2.
 *
 * Return 0 on success; or -1 on error.
 */
int
crypto_scrypt(const uint8_t * passwd, size_t passwdlen,
    const uint8_t * salt, size_t saltlen, uint64_t N, uint32_t r, uint32_t p,
    uint8_t * buf, size_t buflen)
{
    uint8_t * B;
    uint8_t * V;
    uint8_t * XY;
    uint32_t i;

    TSRMLS_FETCH();

    /* Sanity-check parameters. */
#if SIZE_MAX > UINT32_MAX
    if (buflen > (((uint64_t)(1) << 32) - 1) * 32) {
        php_error_docref(NULL TSRMLS_CC, E_WARNING, "Invalid Parameters: $keyLength too big");
        errno = EFBIG;
        goto err0;
    }
#endif
    if ((uint64_t)(r) * (uint64_t)(p) >= (1 << 30)) {
        errno = EFBIG;
        php_error_docref(NULL TSRMLS_CC, E_WARNING, "Invalid Parameters; $r * $p is >= 2^30");
        goto err0;
    }
    if (((N & (N - 1)) != 0) || (N == 0)) {
        errno = EINVAL;
        php_error_docref(NULL TSRMLS_CC, E_WARNING, "Invalid Parameters; $N is not a power of two greater than 1");
        goto err0;
    }
    if ((r > SIZE_MAX / 128 / p) ||
#if SIZE_MAX / 256 <= UINT32_MAX
        (r > SIZE_MAX / 256) ||
#endif
        (N > SIZE_MAX / 128 / r)) {
        errno = ENOMEM;
        php_error_docref(NULL TSRMLS_CC, E_WARNING, "Invalid Parameters");
        goto err0;
    }

    /* Allocate memory. */
    if ((B = safe_emalloc(128, r * p, 0)) == NULL)
        goto err0;
    if ((XY = safe_emalloc(256, r, 0)) == NULL)
        goto err1;
    if ((V = safe_emalloc(128, r * N, 0)) == NULL)
        goto err2;

    /* 1: (B_0 ... B_{p-1}) <-- PBKDF2(P, S, 1, p * MFLen) */
    PBKDF2_SHA256_SCRYPT(passwd, passwdlen, salt, saltlen, 1, B, p * 128 * r);

    /* 2: for i = 0 to p - 1 do */
    for (i = 0; i < p; i++) {
        /* 3: B_i <-- MF(B_i, N) */
        smix(&B[i * 128 * r], r, N, V, XY);
    }

    /* 5: DK <-- PBKDF2(P, B, 1, dkLen) */
    PBKDF2_SHA256_SCRYPT(passwd, passwdlen, B, p * 128 * r, 1, buf, buflen);

    /* Free memory. */
    efree(V);
    efree(XY);
    efree(B);

    /* Success! */
    return (0);

err2:
    efree(XY);
err1:
    efree(B);
err0:
    /* Failure! */
    return (-1);
}
