/* PgSqlClient - ADO.NET Data Provider for PostgreSQL 7.4+
 * Copyright (c) 2003-2004 Carlos Guzman Alvarez
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

using System;
using System.Text;
using System.Data;
using System.Drawing;
using System.Collections;
using System.ComponentModel;
using System.Security.Cryptography;
using System.Security.Cryptography.X509Certificates;

using Mono.Security.Protocol.Tls;

using PostgreSql.Data.NPgClient;
using PostgreSql.Data.PgSqlClient.DbSchema;

namespace PostgreSql.Data.PgSqlClient
{	
	[ToolboxItem(true),
	ToolboxBitmap(typeof(PgConnection), "Resources.ToolBox.PgConnection.bmp"),
	DefaultEvent("InfoMessage")]
	public sealed class PgConnection : Component, IDbConnection, ICloneable
	{	
		#region Events

		public event StateChangeEventHandler		StateChange;
		public event PgInfoMessageEventHandler		InfoMessage;
		public event PgNotificationEventHandler		Notification;
		public event CertificateValidationCallback	ServerCertValidation;
		public event CertificateSelectionCallback	ClientCertSelection;
		public event PrivateKeySelectionCallback	PrivateKeySelection;

		#endregion

		#region Fields

		private PgDbConnection	dbConnection;
		private ConnectionState state;
		private bool			disposed;
		private string			connectionString;
		private PgDataReader	dataReader;
		private PgTransaction	activeTransaction;
		private ArrayList		activeCommands;

		private PgClientMessageEventHandler			infoMessageHandler;
		private PgClientNotificationEventHandler	notificationHandler;
		private CertificateValidationCallback		certificateValidationCallback;
		private CertificateSelectionCallback		certificateSelectionCallback;
		private PrivateKeySelectionCallback			privateKeySelectionCallback;

		#endregion
		
		#region Properties

		[Category("Data"), 
		RecommendedAsConfigurableAttribute(true),
		RefreshProperties(RefreshProperties.All),
		DefaultValue("")]
		// Editor(typeof(DesingTime.ConnectionStringUIEditor), typeof(System.Drawing.Design.UITypeEditor))]
		public string ConnectionString
		{
			get { return connectionString; }
			set
			{ 
				if (state == ConnectionState.Closed)
				{
					PgDbConnection tmpConn = new PgDbConnection(value);
					connectionString = value;
					tmpConn = null;
				}
			}
		}

		[DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
		public int ConnectionTimeout
		{
			get 
			{ 
				if (dbConnection != null)
				{
					return dbConnection.Settings.Timeout;
				}
				else
				{
					return 15; 
				}
			}
		}

		[DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
		public string Database
		{
			get 
			{ 
				if (dbConnection != null)
				{
					return dbConnection.Settings.Database;
				}
				else
				{
					return String.Empty; 
				}
			}
		}

		[DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
		public string DataSource
		{
			get 
			{ 
				if (dbConnection != null)
				{
					return dbConnection.Settings.ServerName;
				}
				else
				{
					return String.Empty; 
				}
			}
		}

		[DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
		public int PacketSize
		{
			get 
			{ 
				int packetSize = 8192;
				if (dbConnection != null)
				{
					packetSize = dbConnection.Settings.PacketSize;
				}

				return packetSize; 
			}
		}

		[Browsable(false),
		DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
		public string ServerVersion
		{
			get
			{
				if (dbConnection != null)
				{
					return (string)dbConnection.DB.ParameterStatus["server_version"];
				}
				else
				{
					return String.Empty; 
				}
			}
		}

		[Browsable(false),
		DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
		public ConnectionState State
		{
			get { return state; }
		}

		internal ArrayList ActiveCommands
		{
			get { return activeCommands; }
		}
				
		internal PgDataReader DataReader
		{
			get { return dataReader; }
			set { dataReader = value; }
		}

		internal PgDbConnection DbConnection
		{
			get { return dbConnection; }
			set { dbConnection = value; }
		}

		internal PgTransaction ActiveTransaction
		{
			get { return activeTransaction; }
			set { activeTransaction = value; }
		}

		#endregion		

		#region Constructors

		public PgConnection() : base()
		{			
			state				= ConnectionState.Closed;
			connectionString	= String.Empty;

			GC.SuppressFinalize(this);
		}
    		
		public PgConnection(string connString) : this()
		{			
			this.ConnectionString	= connString;
		}		

		#endregion

		#region IDisposable Methods

		protected override void Dispose(bool disposing)
		{
			if (!disposed)
			{
				try
				{	
					if (disposing)
					{
						// release any managed resources
						Close();

						dbConnection		= null;
						connectionString	= null;
					}

					// release any unmanaged resources
				}
				finally
				{
					base.Dispose(disposing);
				}

				disposed = true;
			}			
		}

		#endregion

		#region ICloneable MEthods

		object ICloneable.Clone()
		{
			return new PgConnection(ConnectionString);
		}

		#endregion

		#region Methods

		IDbTransaction IDbConnection.BeginTransaction()
		{
			return BeginTransaction();
		}

		IDbTransaction IDbConnection.BeginTransaction(IsolationLevel level)
		{
			return BeginTransaction(level);
		}

		public PgTransaction BeginTransaction()
		{
			if (state == ConnectionState.Closed)
			{
				throw new InvalidOperationException("BeginTransaction requires an open and available Connection.");
			}

			if (activeTransaction != null && !activeTransaction.IsUpdated)
			{
				throw new InvalidOperationException("A transaction is currently active. Parallel transactions are not supported.");
			}

			if (DataReader != null)
			{
				throw new InvalidOperationException("BeginTransaction requires an open and available Connection. The connection's current state is Open, Fetching.");
			}
			
			try
			{
				activeTransaction = new PgTransaction(this);
				activeTransaction.InternalBeginTransaction();
			}
			catch (PgClientException ex)
			{
				throw new PgException(ex.Message, ex);
			}

			return this.activeTransaction;
		}

		public PgTransaction BeginTransaction(IsolationLevel level)
		{
			if (state == ConnectionState.Closed)
			{
				throw new InvalidOperationException("BeginTransaction requires an open and available Connection.");
			}

			if (activeTransaction != null && !activeTransaction.IsUpdated)
			{
				throw new InvalidOperationException("A transaction is currently active. Parallel transactions are not supported.");
			}

			if (DataReader != null)
			{
				throw new InvalidOperationException("BeginTransaction requires an open and available Connection. The connection's current state is Open, Fetching.");
			}

			try
			{
				activeTransaction = new PgTransaction(this, level);
				activeTransaction.InternalBeginTransaction();
			}
			catch (PgClientException ex)
			{
				throw new PgException(ex.Message, ex);
			}

			return this.activeTransaction;			
		}

		public void ChangeDatabase(string db)
		{
			if (state == ConnectionState.Closed)
			{
				throw new InvalidOperationException("ChangeDatabase requires an open and available Connection.");
			}

			if (db == null || db.Trim().Length == 0)
			{
				throw new InvalidOperationException("Database name is not valid.");
			}

			if (this.DataReader != null)
			{
				throw new InvalidOperationException("ChangeDatabase requires an open and available Connection. The connection's current state is Open, Fetching.");
			}

			string oldDb = this.dbConnection.Settings.Database;

			try
			{
				/* Close current connection	*/
				this.Close();

				/* Set up the new Database	*/
				this.dbConnection.Settings.Database = db;

				/* Open new connection to new database	*/
				this.Open();
			}
			catch (PgException ex)
			{
				this.dbConnection.Settings.Database = oldDb;				
				throw ex;
			}
		}

		public void Open()
		{
			if (state != ConnectionState.Closed)
			{
				throw new InvalidOperationException("Connection already Open.");
			}

			try
			{
				state = ConnectionState.Connecting;

				dbConnection = new PgDbConnection(connectionString);

				// Add handler for Ssl connections
				dbConnection.DB.SslConnectionDelegate = new SslConnectionCallback(this.OnSslConnection);

				// Open connection
				if (dbConnection.Settings.Pooling)
				{
					dbConnection = PgConnectionPool.GetConnection(
						connectionString,
						dbConnection);
				}
				else
				{
					dbConnection.Pooled = false;
					dbConnection.Connect();				
				}
				
				// Set connection state to Open
				state = ConnectionState.Open;			
				if (StateChange != null)
				{
					StateChange(this, new StateChangeEventArgs(ConnectionState.Closed, state));
				}

				// Initialize active commands list
				activeCommands = new ArrayList();

				// Add Info message event handler
				infoMessageHandler = new PgClientMessageEventHandler(OnInfoMessage);
				dbConnection.DB.InfoMessage += infoMessageHandler;

				// Add notification event handler
				notificationHandler = new PgClientNotificationEventHandler(OnNotification);
				dbConnection.DB.Notification += notificationHandler;
			}
			catch (PgClientException ex)
			{
				state = ConnectionState.Closed;
				throw new PgException(ex.Message, ex);
			}
		}

		public void Close()
		{
			if (state == ConnectionState.Open)
			{
				try
				{		
					lock (dbConnection)
					{
						// Close DataReader
						if (dataReader != null &&
							!dataReader.IsClosed)
						{
							dataReader.Close();
						}

						// Dispose Active commands
						DisposeActiveCommands();

						// Rollback active transation
						if (activeTransaction != null)
						{
							activeTransaction.Dispose();
							activeTransaction = null;
						}

						// Remove info message event handler
						dbConnection.DB.InfoMessage -= infoMessageHandler;

						// Remove notification event handler
						dbConnection.DB.Notification -= notificationHandler;

						// Remove SSL handlers
						if (this.dbConnection.Settings.SSL)
						{
							ServerCertValidation	-= certificateValidationCallback;
							ClientCertSelection		-= certificateSelectionCallback;
							PrivateKeySelection		-= privateKeySelectionCallback;
						}

						// Close connection permanently or send it 
						// back to the pool
						if (dbConnection.Pooled)
						{
							PgConnectionPool.FreeConnection(dbConnection);
						}
						else
						{
							dbConnection.Disconnect();
						}
					}

					// Update state
					state = ConnectionState.Closed;

					// Raise StateChange event
					if (StateChange != null)
					{
						StateChange(this, new StateChangeEventArgs(ConnectionState.Open, state));
					}
				}
				catch (PgClientException ex)
				{
					throw new PgException(ex.Message, ex);
				}
			}
		}

		IDbCommand IDbConnection.CreateCommand()
		{			
			return CreateCommand();
		}

		public PgCommand CreateCommand()
		{		
			PgCommand command = new PgCommand();

			command.Connection = this;
	
			return command;
		}

		private void DisposeActiveCommands()
		{
			if (activeCommands != null)
			{
				if (activeCommands.Count > 0)
				{
					PgCommand[] commands = new PgCommand[activeCommands.Count];

					activeCommands.CopyTo(0, commands, 0, commands.Length);
					foreach (PgCommand command in commands)
					{
						command.Dispose();					
					}
					
					commands = null;
				}

				activeCommands.Clear();
				activeCommands = null;				
			}
		}

		public DataTable GetDbSchemaTable(PgDbSchemaType schema, object[] restrictions)
		{
			if (DataReader != null)
			{
				throw new InvalidOperationException("GetDbSchemaTable requires an open and available Connection. The connection's current state is Open, Fetching.");
			}

			IDbSchema dbSchema = PgDbSchemaFactory.GetSchema(schema);

			if (dbSchema == null)
			{
				throw new NotSupportedException("Specified schema type is not supported.");
			}
			if (restrictions != null)
			{
				if (restrictions.Length > dbSchema.RestrictionColumns.Count)
				{
					throw new InvalidOperationException("The number of specified restrictions is not valid.");	
				}
			}

			return dbSchema.GetDbSchemaTable(this, restrictions);
		}

		public void CreateDatabase(string database, string owner, string location, string template, string encoding)
		{
			if (DataReader != null)
			{
				throw new InvalidOperationException("CreateDatabase requires an open and available Connection. The connection's current state is Open, Fetching.");
			}

			if (database == null)
			{
				throw new InvalidOperationException("You need to specify the database name to create.");
			}

			try
			{
				lock (this)
				{
					if (owner == null)
					{
						owner = "DEFAULT";
					}
					if (location == null)
					{
						location = "DEFAULT";
					}
					if (template == null)
					{
						template = "template0";
					}
					if (encoding == null)
					{
						encoding = "SQL_ASCII";
					}
					
					// Build the command text
					StringBuilder commandText = new StringBuilder();
					commandText.AppendFormat(
						"CREATE DATABASE {0} WITH OWNER={1} LOCATION={2} TEMPLATE={3} ENCODING='{4}'",
						database,
						owner,
						location,
						template,
						encoding);
                    
					// Create database
					PgCommand command = new PgCommand(commandText.ToString(), this);
					command.ExecuteNonQuery();
					command.Dispose();
				}				
			}
			catch (PgClientException ex)
			{
				throw new PgException(ex.Message, ex);
			}
		}

		#endregion

		#region Event Handlers Methods

		private void OnInfoMessage(object sender, PgClientMessageEventArgs e)
		{
			if (InfoMessage != null)
			{
				InfoMessage(this, new PgInfoMessageEventArgs(e.Exception));
			}
		}

		private void OnNotification(object sender, PgClientNotificationEventArgs e)
		{
			if (Notification != null)
			{
				Notification(this, 
						new PgNotificationEventArgs(
								e.ProcessID,
								e.Condition,
								e.Aditional));
			}
		}

		private bool OnServerCertificateValidation(
			X509Certificate certificate, 
			int[]			certificateErrors)
		{
			if (this.ServerCertValidation != null)
			{
				return this.ServerCertValidation(certificate, certificateErrors);
			}

			return false;
		}

		private X509Certificate OnClientCertificateSelection(
			X509CertificateCollection	clientCertificates, 
			X509Certificate				serverCertificate, 
			string						targetHost, 
			X509CertificateCollection	serverRequestedCertificates)
		{
			if (this.ClientCertSelection != null)
			{
				return this.ClientCertSelection(
					clientCertificates,
					serverCertificate,
					targetHost,
					serverRequestedCertificates);
			}

			return null;
		}

		private AsymmetricAlgorithm OnPrivateKeySelection(
			X509Certificate	clientCertificate, 
			string			targetHost)
		{
			if (this.PrivateKeySelection != null)
			{
				return this.PrivateKeySelection(clientCertificate, targetHost);
			}

			return null;
		}

		private void OnSslConnection()
		{
			// Server certificate validation
			certificateValidationCallback = new CertificateValidationCallback(OnServerCertificateValidation);
			dbConnection.DB.SslClientStream.ServerCertValidationDelegate = certificateValidationCallback;

			// Client certificate selection
			certificateSelectionCallback = new CertificateSelectionCallback(OnClientCertificateSelection);
			dbConnection.DB.SslClientStream.ClientCertSelectionDelegate = certificateSelectionCallback;

			// Private key selection
			privateKeySelectionCallback = new PrivateKeySelectionCallback(OnPrivateKeySelection);
			dbConnection.DB.SslClientStream.PrivateKeyCertSelectionDelegate = privateKeySelectionCallback;
		}

		#endregion
	}
}
