/* PgSqlClient - ADO.NET Data Provider for PostgreSQL 7.4+
 * Copyright (c) 2003-2004 Carlos Guzman Alvarez
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

using System;
using System.Data;
using System.Text;

namespace PostgreSql.Data.PgSqlClient.DbSchema
{
	internal class PgColumnsSchema : PgAbstractDbSchema
	{
		#region Constructors

		public PgColumnsSchema() : base("Columns")
		{
		}

		#endregion

		#region Add Methods

		public override void AddTables()
		{
			AddTable("pg_attribute");
		}

		public override void AddRestrictionColumns()
		{
			AddRestrictionColumn("pg_namespace.nspname"	, "TABLE_SCHEMA", null);
			AddRestrictionColumn("pg_class.relname"		, "TABLE_NAME", null);
			AddRestrictionColumn("pg_attribute.attname"	, "COLUMN_NAME", null);
		}

		public override void AddDataColumns()
		{
			AddDataColumn("pg_attribute.atttypid"	, "DATA_TYPE");
			AddDataColumn("pg_attribute.attlen"		, "COLUMN_SIZE");
			AddDataColumn("pg_attribute.attndims"	, "COLUMN_DIMENSIONS");
			AddDataColumn("pg_attribute.attnum"		, "ORDINAL_POSITION");
			AddDataColumn("pg_attribute.atthasdef"	, "HAS_DEFAULT");
			AddDataColumn("pg_attrdef.adsrc"		, "COLUMN_DEFAULT");
			AddDataColumn("pg_attribute.attnotnull"	, "IS_NOT_NULL");
			AddDataColumn("(pg_depend.objid is not null)", "IS_AUTOINCREMENT");
			AddDataColumn(getStorageExpression("pg_attribute.attstorage"), "STORAGE");
			AddDataColumn("pg_description.description", "DESCRIPTION");
		}

		public override void AddJoins()
		{
			AddJoin("left join", "pg_class"		, "pg_attribute.attrelid = pg_class.oid");
			AddJoin("left join", "pg_namespace"	, "pg_class.relnamespace = pg_namespace.oid");
			AddJoin("left join", "pg_attrdef"	, "(pg_class.oid = pg_attrdef.adrelid AND pg_attribute.attnum = pg_attrdef.adnum)");
			AddJoin("left join", "pg_description", "(pg_attribute.attrelid = pg_description.objoid AND pg_attribute.attnum = pg_description.objsubid)");
			AddJoin("left join",
				"pg_depend", 
				"(pg_attribute.attrelid = pg_depend.refobjid AND pg_attribute.attnum = pg_depend.refobjsubid  AND pg_depend.deptype = 'i')");
		}

		public override void AddOrderByColumns()
		{
			AddOrderBy("pg_namespace.nspname");
			AddOrderBy("pg_class.relname");
			AddOrderBy("pg_attribute.attnum");
		}

		public override void AddWhereFilters()
		{
			// Do not get dropped columns
			AddWhereFilter("pg_attribute.attisdropped = false");
			// Get only columns with a number > 0
			AddWhereFilter("pg_attribute.attnum > 0");
		}

		#endregion

		#region Parse Methods

		public override object[] ParseRestrictions(object[] restrictions)
		{
			object[] parsed = restrictions;

			return parsed;
		}

		#endregion

		#region Private Methods

		private string getSerialExpression(string fieldName)
		{
			StringBuilder expression = new StringBuilder();

			expression.AppendFormat(" case {0} ", fieldName);
			expression.Append(" when 0 THEN true");
			expression.Append(" default THEN false");			
			expression.Append(" END ");

			return expression.ToString();
		}

		private string getStorageExpression(string fieldName)
		{
			StringBuilder expression = new StringBuilder();

			expression.AppendFormat(" case {0} ", fieldName);
			expression.Append(" when 'p' THEN 'PLAIN'");
			expression.Append(" when 'e' THEN 'EXTERNAL'");			
			expression.Append(" when 'm' THEN 'MAIN'");
			expression.Append(" when 'x' THEN 'EXTENDED'");
			expression.Append(" END ");

			return expression.ToString();
		}

		#endregion
	}
}
