<?php
// /* vim: set expandtab tabstop=4 shiftwidth=4: */
// +----------------------------------------------------------------------+
// | PHP Version 4                                                        |
// +----------------------------------------------------------------------+
// | Copyright (c) 2002 The PHP Group                                     |
// +----------------------------------------------------------------------+
// | This source file is subject to version 2.02 of the PHP license,      |
// | that is bundled with this package in the file LICENSE, and is        |
// | available at through the world-wide-web at                           |
// | http://www.php.net/license/2_02.txt.                                 |
// | If you did not receive a copy of the PHP license and are unable to   |
// | obtain it through the world-wide-web, please send a note to          |
// | license@php.net so we can mail you a copy immediately.               |
// +----------------------------------------------------------------------+
// | Author: Paul Querna <chip@force-elite.com>                             |
// |                                                                      |
// | Original Author: Tomas V.V.Cox <cox@idecnet.com>                     |
// |                                                                      |
// |   Some Code and ideas from Yapircl                                   |
// |   by Geir Torstein Kristiansen <gtk@techie.online.no>                |
// |                                                                      |
// +----------------------------------------------------------------------+
//

define("PEAR_NET_IRC_VERSION","0.0.7");

if (!(function_exists('stream_select'))) {
    echo "stream_select function is not available.\n";
    echo "Net_IRC Requires Stream Select!  Stream Select is included in PHP 4.3.0 or greater.\n";
}

/**
* Class for handling the client side of the IRC protocol (RFC 1459)
* @package Net_IRC
* @author Paul Querna <chip@force-elite.com>
*/
class Net_IRC
{

    /**
    * Running.
    * A value of 0 means that class is not running, and no connections have been made.
    * A value of 1 means connections have been made, and we are inside -Run()
    * @var int $running
    * @see Net_IRC::Run
    */
    var $running = 0;

    /**
    * Servers Specific Information Array
    *   Structure:
    *      [int server_id]
    *          array(
    *            channels - Store Channel Specific Information
    *            Socket  - Server Connection Object
    *            Options - Store General Server Specific Options
    * @var array $servers
    * @see Net_IRC::Add_Connection
    */
    var $servers;

    /**
    * Stores the Next Server Connection ID to be used.
    *
    * @var int $next_id
    * @see Net_IRC::Add_Connection
    */
    var $next_id = 0;

    /**
    * Callback list array
    *   Contains All the default Callbacks on Init. User added callbacks are also put inside this array
    *
    * @var array $callback_list
    * @see Net_IRC::callback
    */
    var $callback_list = array('ERROR' => array (array('Net_IRC_Event','event_error')),
                               'DISCONNECT' => array(array('Net_IRC_Event','event_error')),
                               'PRIVMSG' => array(array('Net_IRC_Event','event_privmsg')),
                               'NOTICE' => array(array('Net_IRC_Event','event_notice')),
                               'QUIT' => array(array('Net_IRC_Event','event_quit')),
                               'JOIN' => array(array('Net_IRC_Event','event_join')),
                               'PART' => array(array('Net_IRC_Event','event_part')),
                               'MODE' => array(array('Net_IRC_Event','event_mode')),
                               'PING' => array(array('Net_IRC_Event','event_ping')),
                               'RPL_WELCOME' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_YOURHOST' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_CREATED' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_MYINFO' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_ISUPPORT' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_NONE' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_USERHOST' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_ISON' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_AWAY' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_UNAWAY' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_NOWAWAY' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_WHOISUSER' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_WHOISSERVER' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_WHOISOPERATOR' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_WHOISIDLE' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_ENDOFWHOIS' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_WHOISCHANNELS' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_WHOWASUSER' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_ENDOFWHOWAS' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_LISTSTART' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_LIST' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_LISTEND' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_CHANNELMODEIS' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_NOTOPIC' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_TOPIC' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_INVITING' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_SUMMONING' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_VERSION' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_WHOREPLY' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_ENDOFWHO' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_NAMREPLY' => array(array('Net_IRC_Event','event_rpl_namreply')),
                               'RPL_ENDOFNAMES' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_LINKS' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_ENDOFLINKS' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_BANLIST' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_ENDOFBANLIST' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_INFO' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_ENDOFINFO' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_MOTDSTART' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_MOTD' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_ENDOFMOTD' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_YOUREOPER' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_REHASHING' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_TIME' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_USERSSTART' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_USERS' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_ENDOFUSERS' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_NOUSERS' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_TRACELINK' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_TRACECONNECTING' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_TRACEHANDSHAKE' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_TRACEUNKNOWN' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_TRACEOPERATOR' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_TRACEUSER' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_TRACESERVER' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_TRACENEWTYPE' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_TRACELOG' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_STATSLINKINFO' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_STATSCOMMANDS' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_STATSCLINE' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_STATSNLINE' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_STATSILINE' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_STATSKLINE' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_STATSYLINE' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_ENDOFSTATS' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_STATSLLINE' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_STATSUPTIME' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_STATSOLINE' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_STATSHLINE' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_UMODEIS' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_LUSERCLIENT' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_LUSEROP' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_LUSERUNKNOWN' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_LUSERCHANNELS' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_LUSERME' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_ADMINME' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_ADMINLOC1' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_ADMINLOC2' => array(array('Net_IRC_Event','default_handler')),
                               'RPL_ADMINEMAIL' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_NOSUCHNICK' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_NOSUCHSERVE' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_NOSUCHCHANNEL' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_CANNOTSENDTOCHAN' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_TOOMANYCHANNELS' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_WASNOSUCHNICK' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_TOOMANYTARGETS' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_NOORIGIN' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_NORECIPIENT' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_NOTEXTTOSEND' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_NOTOPLEVEL' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_WILDTOPLEVEL' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_UNKNOWNCOMMAND' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_NOMOTD' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_NOADMININFO' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_FILEERROR' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_NONICKNAMEGIVEN' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_ERRONEUSNICKNAME' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_NICKNAMEINUSE' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_NICKCOLLISION' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_USERNOTINCHANNEL' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_NOTONCHANNEL' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_USERONCHANNEL' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_NOLOGIN' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_SUMMONDISABLED' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_USERSDISABLED' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_NOTREGISTERED' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_NEEDMOREPARAMS' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_ALREADYREGISTRED' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_NOPERMFORHOST' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_PASSWDMISMATCH' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_YOUREBANNEDCREEP' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_KEYSET' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_CHANNELISFULL' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_UNKNOWNMODE' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_INVITEONLYCHAN' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_BANNEDFROMCHAN' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_BADCHANNELKEY' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_NOPRIVILEGES' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_CHANOPRIVSNEEDED' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_CANTKILLSERVER' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_NOOPERHOST' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_UMODEUNKNOWNFLAG' => array(array('Net_IRC_Event','default_handler')),
                               'ERR_USERSDONTMATCH' => array(array('Net_IRC_Event','default_handler'))
                           );

    /**
    * Array with statistics about each Connections Statistics
    *
    * @var array $stats
    */
    var $stats     = array();

    /** 
    *  Array with a complete list of the IRC RFC #1459 numerics
    *      Note: I know some servers have added on and changed this, 
    *      if there is a signifigant one missing, please contact me.
    * @var array $event_list
    */
    var $event_list = array(
               '001' => 'RPL_WELCOME',
               '002' => 'RPL_YOURHOST',
               '003' => 'RPL_CREATED',
               '004' => 'RPL_MYINFO',
               '005' => 'RPL_ISUPPORT',
               '200' =>  'RPL_TRACELINK',
               '201' =>  'RPL_TRACECONNECTING',
               '202' =>  'RPL_TRACEHANDSHAKE',
               '203' =>  'RPL_TRACEUNKNOWN',
               '204' =>  'RPL_TRACEOPERATOR',
               '205' =>  'RPL_TRACEUSER',
               '206' =>  'RPL_TRACESERVER',
               '208' =>  'RPL_TRACENEWTYPE',
               '261' =>  'RPL_TRACELOG',
               '211' =>  'RPL_STATSLINKINFO',
               '212' =>  'RPL_STATSCOMMANDS',
               '213' =>  'RPL_STATSCLINE',
               '214' =>  'RPL_STATSNLINE',
               '215' =>  'RPL_STATSILINE',
               '216' =>  'RPL_STATSKLINE',
               '218' =>  'RPL_STATSYLINE',
               '219' =>  'RPL_ENDOFSTATS',
               '241' =>  'RPL_STATSLLINE',
               '242' =>  'RPL_STATSUPTIME',
               '243' =>  'RPL_STATSOLINE',
               '244' =>  'RPL_STATSHLINE',
               '221' =>  'RPL_UMODEIS',
               '251' =>  'RPL_LUSERCLIENT',
               '252' =>  'RPL_LUSEROP',
               '253' =>  'RPL_LUSERUNKNOWN',
               '254' =>  'RPL_LUSERCHANNELS',
               '255' =>  'RPL_LUSERME',
               '256' =>  'RPL_ADMINME',
               '257' =>  'RPL_ADMINLOC1',
               '258' =>  'RPL_ADMINLOC2',
               '259' =>  'RPL_ADMINEMAIL',
               '300' =>  'RPL_NONE',
               '302' =>  'RPL_USERHOST',
               '303' =>  'RPL_ISON',
               '301' =>  'RPL_AWAY',
               '305' =>  'RPL_UNAWAY',
               '306' =>  'RPL_NOWAWAY',
               '311' =>  'RPL_WHOISUSER',
               '312' =>  'RPL_WHOISSERVER',
               '313' =>  'RPL_WHOISOPERATOR',
               '317' =>  'RPL_WHOISIDLE',
               '318' =>  'RPL_ENDOFWHOIS',
               '319' =>  'RPL_WHOISCHANNELS',
               '314' =>  'RPL_WHOWASUSER',
               '369' =>  'RPL_ENDOFWHOWAS',
               '321' =>  'RPL_LISTSTART',
               '322' =>  'RPL_LIST',
               '323' =>  'RPL_LISTEND',
               '324' =>  'RPL_CHANNELMODEIS',
               '331' =>  'RPL_NOTOPIC',
               '332' =>  'RPL_TOPIC',
               '341' =>  'RPL_INVITING',
               '342' =>  'RPL_SUMMONING',
               '351' =>  'RPL_VERSION',
               '352' =>  'RPL_WHOREPLY',
               '315' =>  'RPL_ENDOFWHO',
               '353' =>  'RPL_NAMREPLY',
               '366' =>  'RPL_ENDOFNAMES',
               '364' =>  'RPL_LINKS',
               '365' =>  'RPL_ENDOFLINKS',
               '367' =>  'RPL_BANLIST',
               '368' =>  'RPL_ENDOFBANLIST',
               '371' =>  'RPL_INFO',
               '374' =>  'RPL_ENDOFINFO',
               '375' =>  'RPL_MOTDSTART',
               '372' =>  'RPL_MOTD',
               '376' =>  'RPL_ENDOFMOTD',
               '381' =>  'RPL_YOUREOPER',
               '382' =>  'RPL_REHASHING',
               '391' =>  'RPL_TIME',
               '392' =>  'RPL_USERSSTART',
               '393' =>  'RPL_USERS',
               '394' =>  'RPL_ENDOFUSERS',
               '395' =>  'RPL_NOUSERS',
               '401' =>  'ERR_NOSUCHNICK',
               '402' =>  'ERR_NOSUCHSERVE',
               '403' =>  'ERR_NOSUCHCHANNEL',
               '404' =>  'ERR_CANNOTSENDTOCHAN',
               '405' =>  'ERR_TOOMANYCHANNELS',
               '406' =>  'ERR_WASNOSUCHNICK',
               '407' =>  'ERR_TOOMANYTARGETS',
               '409' =>  'ERR_NOORIGIN',
               '411' =>  'ERR_NORECIPIENT',
               '412' =>  'ERR_NOTEXTTOSEND',
               '413' =>  'ERR_NOTOPLEVEL',
               '414' =>  'ERR_WILDTOPLEVEL',
               '421' =>  'ERR_UNKNOWNCOMMAND',
               '422' =>  'ERR_NOMOTD',
               '423' =>  'ERR_NOADMININFO',
               '424' =>  'ERR_FILEERROR',
               '431' =>  'ERR_NONICKNAMEGIVEN',
               '432' =>  'ERR_ERRONEUSNICKNAME',
               '433' =>  'ERR_NICKNAMEINUSE',
               '436' =>  'ERR_NICKCOLLISION',
               '441' =>  'ERR_USERNOTINCHANNEL',
               '442' =>  'ERR_NOTONCHANNEL',
               '443' =>  'ERR_USERONCHANNEL',
               '444' =>  'ERR_NOLOGIN',
               '445' =>  'ERR_SUMMONDISABLED',
               '446' =>  'ERR_USERSDISABLED',
               '451' =>  'ERR_NOTREGISTERED',
               '461' =>  'ERR_NEEDMOREPARAMS',
               '462' =>  'ERR_ALREADYREGISTRED',
               '463' =>  'ERR_NOPERMFORHOST',
               '464' =>  'ERR_PASSWDMISMATCH',
               '465' =>  'ERR_YOUREBANNEDCREEP',
               '467' =>  'ERR_KEYSET',
               '471' =>  'ERR_CHANNELISFULL',
               '472' =>  'ERR_UNKNOWNMODE',
               '473' =>  'ERR_INVITEONLYCHAN',
               '474' =>  'ERR_BANNEDFROMCHAN',
               '475' =>  'ERR_BADCHANNELKEY',
               '481' =>  'ERR_NOPRIVILEGES',
               '482' =>  'ERR_CHANOPRIVSNEEDED',
               '483' =>  'ERR_CANTKILLSERVER',
               '491' =>  'ERR_NOOPERHOST',
               '501' =>  'ERR_UMODEUNKNOWNFLAG',
               '502' =>  'ERR_USERSDONTMATCH'
         );

    /**
    * Adds a new Connection to a IRC server. This will pre-set the settings, 
    * and create a new server_id.
    *
    * @param array $options The parameters of the connection
    * @see Net_IRC::options
    * @return int server_id
    */
    function add_connection($options)
    {
       $server_id = $this->next_id;
       $this->next_id++;
       $default_options = array(
           'server'    => 'irc.netgamers.org', 
           'port'      => 6667,
           'nick'      => 'PearNet_IRC', 
           'realname'  => 'PHP - /Pear/Net_IRC Bot',
           'identd'    => 'pear',
           'host'      => '10.0.0.8',
           'log_types' => array(0, 1, 2, 3));
        /* Load default Option Values.. */
        foreach( $default_options as $key => $value){
            if(!isset($options[$key])){
                $options[$key] = $value;
            }
        }
	$options['connected'] = 0;
        $this->servers[$server_id] = array('socket' => FALSE, 'options' => $options);
	if($this->running == 0) {
	        return $server_id;
	} else {
		return $this->_connect($server_id);
	}
    }

    /**
    * Really Creates the new connections
    *
    * @param int $server_id 
    * @see Net_IRC::run
    * @see Net_IRC::add_connection
    * @return int server_id
    */
    function _connect($server_id) 
     {
	        $this->log($server_id, 3, "connecting to " . $this->servers[$server_id]['options']['server'] . ":" . $this->servers[$server_id]['options']['port'] );
	        $this->servers[$server_id]['socket'] = fsockopen($this->servers[$server_id]['options']['server'],
							$this->servers[$server_id]['options']['port'],
				                        $errno, $errstr, 5);
	        if (!$this->servers[$server_id]['socket']) {
	            $this->log($server_id,0, "could not connect $errstr ($errno)");
	            return false;
	        }
	        $this->log($server_id, 3, "connected");
	        $this->init_stats($server_id);
	        $this->command($server_id, 'USER '.
                       $this->servers[$server_id]['options']['identd'] . ' '.
                       $this->servers[$server_id]['options']['host']   . ' '.
                       $this->servers[$server_id]['options']['server'] . ' '.
                       ':' . $this->servers[$server_id]['options']['realname']);
        	$this->command($server_id, 'NICK ' . $this->servers[$server_id]['options']['nick']);
        	socket_set_blocking($this->servers[$server_id]['socket'], false);
        	$this->callback($server_id,'CONNECT', false);
		$this->stats[$server_id]['events']['PING']['last'] = $this->get_microtime();
	return $server_id;
    }

    /**
    * Causes Net_IRC to enter the Running state.
    *
    * @param void
    * @see Net_IRC::_connect
    * @see Net_IRC::add_connection
    * @return void
    */
    function run () 
     {
	$this->running = 1;
        foreach(array_keys($this->servers) as $server_id) {
		$this->_connect($server_id);
        }
        while (true) {
			$this->read_event_all($server_id);
	}
    }    

    /**
    * Disconnects from a IRC by sending the QUIT command
    *
    * @param int $server_id
    * @param bool $autoreconnect
    * @return void 
    */
    function disconnect($server_id, $reconnect = FALSE, $quit_msg = "PEAR::Net_IRC")
    {
        $this->command($server_id,'QUIT: ' . $quit_msg);
	fclose($this->servers[$server_id]['socket']);
	$this->servers[$server_id]['connected'] = 0;	
        if($reconnect == TRUE) {
          $this->_connect($server_id);   
	} else {
            unset($this->servers[$server_id]);
	}
    }

    /**
    * Tests if a Connection to the IRC server is still alive by using feof();
    *
    * @param int $server_id
    * @see Net_IRC::_connect
    * @return bool $feof
    */
    function is_Connected($server_id)
    {
        return !feof($this->servers[$server_id]['socket']);
    }

    /**
    * Transform code messages returned by the server to text events
    * 
    * @param int    $code  Search the value for a code
    * @param string $handler Search the code for a value
    * @return string $text_code return the code, false on none found.
    */
    function get_event($code = null, $handler = null)
    {
        if ($code) {
            return isset($this->event_list[$code]) ? $this->event_list[$code] : $code;
        }
        $handler = strtoupper($handler);
        foreach ($events as $k => $e) {
            if ($handler == $e) {
                return $this->events[$k];
            }
        }
        return false;
    }
    
    /**
    * Adds a New Event
    * 
    * @param string $code Name of New Event
    * @param string $handler Default Handler
    * @return void
    */
    function add_event($code, $handler = array(array('Net_IRC_Event','default_handler')))
    {
        $this->event_list[$code] = $handler;
    }
    	
    /**
    * Send a IRC command to the server
    *
    * @param int $server_id The Server to Send it on
    * @param string $command The full command for sending
    * @return bool True or false depending on the write() response
    * @see Net_IRC::write()
    */
    function command($server_id,$command)
    {
        return $this->write($server_id,trim($command));
    }

    /**
    * Send a IRC Private Message to the server
    *
    * @param int $server_id The Server to Send it on
    * @param string $target The target of the PrivMsg
    * @param string $message The Message ( if Array, it will send each item as one message)
    * @return bool True or false depending on the send_msg() response
    * @see Net_IRC::send_msg()
    */
    function send_privmsg ($server_id, $target , $message) {
        return $this->send_msg($server_id,'PRIVMSG', $target, $message);
    }

    /**
    * Send a IRC Mode Message to the server
    *
    * @param int $server_id The Server to Send it on
    * @param string $target The target of the Mode Change
    * @param string $mode The Modes to set
    * @return bool True or false depending on the send_msg() response
    * @see Net_IRC::send_msg()
    */
    function send_mode ($server_id, $target,  $mode) {
        return $this->send_msg($server_id,'MODE', $target, $mode);
    }

    /**
    * Send a CTCP IRC Private Message to the server
    *
    * @param int $server_id The Server to Send it on
    * @param string $target The target of the CTCP
    * @param string $type The Type of CTCP Message (eg version, ping)
    * @return bool True or false depending on the send_msg() response
    * @see Net_IRC::send_msg()
    */
    function send_ctcp ( $server_id, $target , $type) {
        return $this->send_msg($server_id,'PRIVMSG', $target, "\001" . $type . "\001");
    }

    /**
    * Send a CTCP Reply Notice to the server
    *
    * @param int $server_id The Server to Send it on
    * @param string $target The target of the CTCP
    * @param string $type The Type of CTCP Message (eg version, ping)
    * @param string $value (optional) The Value
    * @return bool True or false depending on the send_msg() response
    * @see Net_IRC::send_notice()
    */
    function send_ctcpreply ( $server_id, $target , $type, $value = '') {
        return $this->send_notice($server_id, $target, "\001" . $type .' '. $value . "\001");
    }

    /**
    * Send a IRC NOTICE to the server
    *
    * @param int $server_id The Server to Send it on
    * @param string $target The target of the Notice
    * @param string $message The Message ( if Array, it will send each item as one Notice)
    * @return bool True or false depending on the send_msg() response
    * @see Net_IRC::send_msg()
    */
    function send_notice ( $server_id, $target , $message) {
        return $this->send_msg($server_id, 'NOTICE', $target, $message);
    }

    /**
    * Generic Send Message to Server Function 
    *   Auto splits all messages to 451 Byte maximum.
    *
    * @param int $server_id The Server to Send it on
    * @param string $type The type of Message (eg PRIVMSG, NOTICE ..etc)
    * @param string $target The target of the Message
    * @param string $message The Message ( if Array, it will send each item as one Message)
    * @return bool True or false depending on the command() response
    * @see Net_IRC::command()
    */
    function send_msg($server_id, $type, $target , $message) {
       $msg = $type .' ' . $target . ' :';
       $max_msg_len = 449 - strlen($msg); 
       if(is_array($message)){
          if(count($message) == 1){
              $msga =  array_shift($message);;
              unset($message);
          } else {
              $msga =  array_shift($message);
          }
       } else {
          $msga =  $message;
          unset($message);
       }
       if(strlen($msga) >= $max_msg_len){
           $new = substr ( $msga , $max_msg_len);
           $msga = substr ( $msga , 0, $max_msg_len);
           if(is_array($message)){
               array_unshift($message,$new);
           } else {
               if(isset($message)){
                  $message =  array($new , $message);
               } else {
                  $message = array($new);
               }   
           }
       }
       $msg .= $msga;
       echo "msg:  $msg --- $msga\n";
       if(isset($message)){
          $this->command($server_id,$msg);
          return $this->send_msg($server_id,$type, $target, $message);
       } else {
          return $this->command($server_id,$msg);
       }
    }

    /**
    * Writes a command to the openned socket
    *
    * @param int $server_id The Server to Send it on
    * @param string $command The full command for sending
    * @return bool True on success or False if the socket is not open
    */
    function write($server_id,$command)
    {
        if (feof($this->servers[$server_id]['socket'])) {
            $this->log($server_id,0, 'Write Disconnected');
            $this->callback($server_id,'DISCONNECT', false);
            return false;
        }

        if ($command && !fputs($this->servers[$server_id]['socket'], $command . "\r\n")) {
            $this->log($server_id,1, "could not write to socket");
            return false;
        } else {
            $this->log($server_id,4, "<- $command");
            $this->update_stats($server_id,'tx');
        }
        return true;
    }

    /**
    * Reads from the socket using stream_select() and fgets() for all connected servers
    *
    * @param int $sec Number of Seconds to Select() for
    * @param int $usec Number of Micro-Seconds to Select() for
    */
    function read_select($sec = NULL, $usec = NULL)
    {
	$set = array();
        /* build array */

	foreach(array_keys($this->servers) as $key){
		$set[$key] = $this->servers[$key]['socket'];
	}

	/* resource &read, resource &write, resource &except, int tv_sec [, int tv_usec] */

        $sec = 1;
	if(!($usec || $sec)) {
            $ret = stream_select($set, $b = NULL, $c = NULL,$sec);
	} else {
	    $ret = stream_select($set, $b = NULL, $c = NULL, $sec, $usec);
	}
	if($ret > 0) {
            foreach($set as $socket){
               foreach(array_keys($this->servers) as $key){
                   if($this->servers[$key]['socket'] == $socket){
                       $server_id = $key;
                       break;
                   }
               }
               if(!isset($server_id)){
                   die("Error: Could not Find Originating Socket!");
               }
               if (feof($this->servers[$server_id]['socket'])) {
                   $this->log($server_id,0, 'Read Disconnected');
                   $this->callback($server_id,'DISCONNECT', false);
                   continue;
               } else {
                   $receive = rtrim(fgets($this->servers[$server_id]['socket'], 1024));
                   if (!$receive) {
                       continue;
                   } else {
                       $this->update_stats($server_id,'rx');
                       $this->log($server_id,4, "-> $receive");
		       return array($server_id,$receive);
                  }
              }
         }
         } else {
             foreach(array_keys($this->servers) as $server_id){
                 if(($this->get_microtime() - $this->stats[$server_id]['events']['PING']['last']) > 300){
                      $this->log($server_id,1, "No Ping in 300 Seconds. Stoned Server. Disconnected. Reconnecting.");
                      $this->disconnect($server_id, TRUE, "Stoned Server. Reconnecting.");
                 }
             }

             return null;
        }
    }


    /**
    * Reads from the socket for all Servers and calls the event handler. It will automatically
    * return server PINGs
    *
    * @param bool $block When TRUE it will read until there is data
    * @return mixed  - FALSE on socket read errors
    *                - NULL on no data in the socket (only when $block=false)
    *                - STRING the event called
    */
    function read_event_all()
    {
        while (list($server_id, $response) = $this->read_select()) {
            $result = $this->parse_response($response);
            $event  = $result[0];
            if (is_numeric($event)) {
                $event = $this->get_event($event);
            }
            $this->update_event_stats($server_id,$event, $result[1]);
            $this->callback($server_id,$event, $result[1]);
            // Automatically answer server PINGs
            if ($event == 'PING') {
                continue;
            } else {
                break;
            }
        }
        // read() can return the response or null on no response. False means
        // error on socket read
        switch ($response) {
            case null:  return true;
            case false: return false;
            default:    return $event;
        }
    }

    /**
    * Parse a message comming from the IRC server. It will always return
    * this data structure:
    *
    * array($command, array($origin, $orighost, $target, $params))
    *
    *   $command  -> The command
    *   $origin   -> The nick which sents the command
    *   $orighost -> The full identd of the host which sent the command
    *   $target   -> The destination of the message
    *   $params   -> The rest of the IRC message
    *
    * The value of some of this params may be null depending on the response
    *
    * @param string $response The message returned sent from the server
    * @return array The parsed response
    */
    function parse_response($response)
    {
        /*
         <message>  ::= [':' <prefix> <SPACE> ] <command> <params> <crlf>
                          $origin!$orighost $command $target $params
        */
        $message = explode(' ', $response, 2);
        if ($message[0]{0} == ':') {
            // parse prefix
            $prefix = substr($message[0], 1);
            if (strpos($prefix, '!') !== false) {
                list($origin, $orighost) = explode('!', $prefix, 2);
            } else {
                $origin   = $prefix;
                $orighost = null;
            }
            list($command, $rest) = explode(' ', $message[1], 2);
            // foo :bar
            if (strpos($rest, ' :') !== false) {
                list($target, $params) = explode(' :', $rest, 2);
            // :bar
            } elseif ($rest{0} == ':') {
                $target = substr($rest, 1);
                $params = null;
            // foo
            } else {
                $target = $rest;
                $params = null;
            }
        } else {
            $origin   = null;
            $orighost = null;
            $command  = $message[0];
            $target   = null;
            $params   = substr($message[1], 1);
        }
        return array($command, array($origin, $orighost, $target, $params));
    }

    /**
    * Adds a new Callback function to be called.
    *  To call an Object with this, send func as an array(object, method)
    *  Example: ->add_callback("PRIVMSG", array("Net_IRC_bot","new_mesg"));
    *
    * @param string $command IRC Message to add callback to
    * @param string $func Name of Function to Call
    */
    function add_callback( $command, $func) 
    {
        array_push($this->callback_list[$command], $func);
    }
    /**
    * Deletes All Callbacks for an exv
    *
    * @param string $command IRC Message to delete callback for
    * @param string $func Name of Function to delete from the callbacks
    */
    function del_callback( $command, $func) 
    {
	foreach($this->callback_list[$command] as $k => $value)
	{
	    if($value == $func)
	    {
	      unset($this->callback_list[$command][$k]);
		break;
	    }
	}
    }

    
    /**
    * Calls the function that handles the given command (in the form:
    * "event_$command"). If the function is not present, will call
    * the "fallback" function, which has always to be present.
    *
    * @param int $server_id
    * @param string $command One word with the function to call
    * @param array  $params  The params for the event handler
    */
    function callback($server_id,$command, $params = array())
    {
        if(!is_array($params)){
            $params[] = $params;
        }
        array_unshift($params, &$this);
        array_unshift($params, $server_id);
	        if(is_array($this->callback_list[$command])){
            foreach($this->callback_list[$command] as $key => $value){
                if(is_array($value)) {
                    $this->log($server_id,5, "Calling callback {$value[0]}::{$value[1]}");
                } else {
                    $this->log($server_id,5, "Calling callback $value");
                }
                call_user_func_array($value, $params);
            }
        } else {
           $this->log($server_id,2, "No callback for $command found.");
        }
    }

    /**
    * Updates the internal stats
    *
    * @param int $server_id
    * @param string type 'rx' or 'tx'
    * @access private
    */
    function update_stats($server_id,$type = 'rx')
    {
        $this->stats[$server_id][$type . '_idle_since'] = $this->get_microtime();
    }

    /**
    * Updates the stats for a certain event
    * 
    * @param int $server_id
    * @param string $event Event Name
    * @param array $args 
    * @access private
    */
    function update_event_stats($server_id,$event, $args = array())
    {
        $this->log($server_id,5, "Updating event $event");
        if (!isset($this->stats[$server_id]['events'][$event])) {
            $this->stats[$server_id]['events'][$event] = array();
            $this->stats[$server_id]['events'][$event]['times']    = 1;
            $this->stats[$server_id]['events'][$event]['interval'] = 1;
            $this->stats[$server_id]['events'][$event]['last'] = $this->get_microtime();
		//var_dump($this->stats, $server_id, $event, $this->get_microtime());
		//die();
            return;
        }
        $event = &$this->stats[$server_id]['events'][$event];
        $event['times'] += 1;
        $int = 60;
        if ((time() - $event['last']) < $int) {
            $event['interval'] += 1;
        } else {
            $event['interval'] = 0;
        }
        $this->log($server_id,5, "event interval: " . $event['interval']);
        array_pop($args);
        foreach ($args as $k => $v) {
            if ($v) {
                $event[$k][$v] = isset($event[$k][$v]) ? $event[$k][$v] + 1 : 1;
                // To avoid stats flooding we only track the last 30 different ones
                if (count($event[$k]) > 30) {
                    $this->log($server_id,5, "Dropping key ($k) param");
                    array_shift($event[$k]);
                }
            }
        }
        $event['last'] = time();
    }

    /**
    * Initialize the stats
    * @param int $server_id 
    * @access private
    */
    function init_stats($server_id)
    {
        $this->stats[$server_id]['started'] = $this->get_microtime();
        $this->stats[$server_id]['rx_idle'] = 0;
        $this->stats[$server_id]['rx_idle_since'] = $this->get_microtime();
        $this->stats[$server_id]['tx_idle'] = 0;
        $this->stats[$server_id]['tx_idle_since'] = $this->get_microtime();
//        $this->stats[$server_id]['events']  = $this->get_microtime();
    }

    /**
    * Get the internal stats of the connection. Params accepted for $label:
    * - rx_idle:      The seconds passed since the last time we received a
    *                 message from the server
    * - rx_idle_since: The last date (timestamp) we received a message from the server
    * - tx_idle       The seconds since the last time we sent a message
    * - tx_idle_since: The last time we sent a message to the server
    * - started:      The date (timestamp) when the socket was openned
    * - running:      The amount of seconds since the start time
    *
    * @param int $server_id
    * @param mixed $label The string label to return or null for retuning
    *                     the full stats array
    * @return mixed Array or int depending on the $label parameter
    */
    function get_stats($server_id,$label = null)
    {
        $this->stats[$server_id]['rx_idle'] = time() - $this->stats[$server_id]['rx_idle_since'];
        $this->stats[$server_id]['tx_idle'] = time() - $this->stats[$server_id]['tx_idle_since'];
        $this->stats[$server_id]['running'] = time() - $this->stats[$server_id]['started'];
        if ($label) {
            return isset($this->stats[$server_id][$label]) ? $this->stats[$server_id][$label] : false;
        }
        return $this->stats[$server_id];
    }

    /**
    * Gets an option of this instance
    *
    * @param int $server_id
    * @param  string $options The option to retrieve
    * @return string The value of the option
    */
    function get_option($server_id,$option)
    {
        return isset($this->servers[$server_id]['option'][$option]) ? $this->servers[$server_id]['options'][$option] : null;
    }

    /**
    * Logs a message according to a conenctions log settings.
    *
    * @param int $server_id
    * @param  int $level
    * @param string $message
    * @return void
    */
    function log($server_id,$level, $message)
    {
        if (in_array($level, $this->servers[$server_id]['options']['log_types'])) {
           print $server_id . ':' . date('H:i:s') . " " . trim($message) . "\n"; flush();
        }
    }

    /**
    * Gets the currentime in microseconds
    *  Credit: PHP.net Manual Example
    *
    * @param void
    * @return float $microseconds
    */ 
    function get_microtime(){ 
        list($usec, $sec) = explode(" ",microtime()); 
        return ((float)$usec + (float)$sec); 
    }

    /**
    * Adds a nick to a per Server Channel list, updating Op and Voice counts
    *  Credit: Yapircl
    *
    * @param int $server_id
    * @param string $channel
    * @param string $nick
    * @return void
    */  
    function add_nick($server_id, $channel, $nick) 
    {
        // add nick to the nicklist with the modeflags
        $channel =& $this->servers[$server_id]['channels'][strtolower($channel)];
        switch ($nick[0]) {
        case '@':
            $channel['users'][substr($nick, 1)] = array('op' => true);
            $channel['ops']++;
            break;
        case '%':
            $channel['users'][substr($nick, 1)] = array('halfop' => true);
            $channel['halfops']++;
            break;
        case '+':
            $channel['users'][substr($nick, 1)] = array('voice' => true);
            $channel['voices']++;
            break;
        default:
            $channel['users'][$nick] = array();
        }

        $channel['total']++;
    }

    /**
    * Removes a nick to a per Server Channel list, updating Op and Voice counts
    *  Credit: Yapircl
    *
    * @param int $server_id
    * @param string $channel
    * @param string $nick
    * @see Net_IRC::del_nickall
    * @return void
    */  
    function del_nickone($server_id, $channel, $nick)
    {
        // delete a nick from the nicklist and update counters

        $mychan =& $this->servers[$server_id]['channels'][strtolower($channel)];

        if ($this->is_op($server_id,$channel, $nick)) {
            $mychan['ops']--;
        }
        if ($this->is_halfop($server_id,$channel, $nick)) {
            $mychan['halfops']--;
        }

        if ($this->is_voice($server_id,$channel, $nick)) {
            $mychan['voices']--;
        }

        $mychan['total']--;

        unset($mychan['users'][$nick]);
    }

    /**
    * Removes a nick from all channels on a Server, updating Op and Voice counts
    *  Credit: Yapircl
    *
    * @param int $server_id
    * @param string $nick
    * @see Net_IRC::del_nickone
    * @return void
    */  
    function del_nickall($server_id,$nick)
    {
        // delete a nick from the nicklist of all channels

        foreach($this->servers[$server_id]['channels'] as $key => $value) {
            $this->del_nickone($server_id, $key, $nick);
        }
    }
 
   /**
    * Tests if a Nick is an operator on a Channel
    *  Credit: Yapircl
    *
    * @param int $server_id 
    * @param string $channel 
    * @param string $nick 
    * @return bool
    */  
    function is_op($server_id,$channel, $nick)
    {
        // return true if a nick is operator

        if (isset($this->servers[$server_id]['channels'][$channel]['users'][$nick]['op'])) {
            return true;
        } else {
            return false;
        }
    }

   /**
    * Tests if a Nick is a half-operator on a Channel
    *  Credit: Yapircl
    *
    * @param int $server_id 
    * @param string $channel 
    * @param string $nick 
    * @return bool
    */  
    function is_halfop($server_id,$channel, $nick)
    {
        // return true if a nick is operator

        if (isset($this->servers[$server_id]['channels'][$channel]['users'][$nick]['halfop'])) {
            return true;
        } else {
            return false;
        }
    }

   /**
    * Tests if a Nick has a voice on a channel
    *  Credit: Yapircl
    *
    * @param int $server_id 
    * @param string $channel 
    * @param string $nick 
    * @return bool
    */  
    function is_voice($server_id, $channel, $nick)
    {
        // return true if a nick is operator

        if (isset($this->servers[$server_id]['channels'][$channel]['users'][$nick]['op'])) {
            return true;
        } else {
            return false;
        }
    }

   /**
    * Tests if a Nick is on a Channel
    *  Credit: Yapircl
    *
    * @param int $server_id 
    * @param string $channel 
    * @param string $nick 
    * @return bool
    */  
    function is_on($server_id, $channel, $nick)
    {
        // return true if a nick is operator

        if (isset($this->servers[$server_id]['channels'][$channel]['users'][$nick])) {
            return true;
        } else {
            return false;
        }
    }

   /**
    * Tests if we are on a channel
    *  Credit: Yapircl
    *
    * @param int $server_id
    * @param string $channel
    * @return bool
    */
    function on_channel($server_id,$channel)
    {
        // return true if it is a channel we are currently in

        if (isset($this->servers[$server_id]['channels'][$channel])) {
            return true;
        } else {
            return false;
        }
    }

   /**
    * Returns a list of channels we are currently on.
    *  Credit: Yapircl
    *
    * @param int $server_id
    * @return array $channel_list
    */
    function get_channels($server_id)
    {
        // return a sorted list of the channels we are currently joined too

        if (count($this->servers[$server_id]['channels']) < 0) {
            return false;
        } else {
            foreach ($this->servers[$server_id]['channels'] as $key => $value) {
                $channels[] = $key;
            }
        }
        natcasesort($channels);
        return $channels;
    }

   /**
    * Returns a list of all users in a Channel
    *  Credit: Yapircl
    *
    * @param int $server_id
    * @param string $channel
    * @param string $op_string
    * @param string $halfop_string
    * @param string $voice_string
    * @return array $nick_list
    */
    function get_nicklist($server_id, $channel, $ostr = '@', $hstr = '%', $vstr = '+')
    {
        // return an irc client like sorted nicklist, @nicks and +nicks first

        $nicklist =& $this->servers[$server_id]['channels'][strtolower($channel)]['users'];

        $oarr = array(); $harr = array(); $varr = array(); $rarr = array();

        foreach ($nicklist as $key => $value) {
            if (isset($value['op'])) {
                $oarr[] = $ostr . $key;
            } elseif (isset($value['halfop'])) {
                $harr[] = $vstr . $key;
            } elseif (isset($value['voice'])) {
                $varr[] = $vstr . $key;
            } else {
                $rarr[] = $key;
            }
        }

        natcasesort($oarr); natcasesort($harr); natcasesort($varr); natcasesort($rarr);
        return array_merge($oarr, $harr, $varr, $rarr);
    }




}

/**
* Basic class for handling the callbacks (events). 
* @package Net_IRC
*/
class Net_IRC_Event
{
    function event_error($server_id,&$irc, $origin = NULL, $orighost = NULL, $target = NULL, $params = NULL)
    {
        $irc->log( $server_id, 0, "Error ocurred ($origin, $orighost, $target, $params)");
    }

    function event_quit($server_id,&$irc, $origin, $orighost, $target, $params)
    {
	$irc->del_nickall( $server_id,$origin);
    }

    function event_mode($server_id,&$irc, $origin, $orighost, $target, $params)
    {
        $spaces = explode(' ', $params);
	if($irc->servers[$server_id]['options']['nick'] == $spaces[0] ) {
             /* usermode */

             /* user modes which takes a parameter
              * s = server notices (takes snomask on undernet)
              */
             $modeswparam = 's';
             $modesnparam = '';

	} else {
             /* channel mode */
             /* see also: http://freenode.info/using_the_network.shtml */ 
             /* channel modes which takes a parameter
              * l = user limit
              * k = key (password)
              */
             $modeswparam = 'kl';

             /* channel modes which takes a parameter, but dont add param to channels array
              * o = operator
              * v = voice
              * b = ban
              * e = ban excemption
              */
              $modesnparam = 'ovbeq';
	}

        $len = strlen($params);

        for ($i = 0; $i < $len; $i++) {

            $modechar = $this->params[$offset]{$i};

            switch ($modechar) {
            case '+':
                $type = true;
                break;
            case '-':
                $type = false;
                break;
            default:
                if (strstr($modeswparam, $modechar)) {
                    // modes with an argument that are handled here
                    if (isset($this->params[$argnr])) {
                        $this->modearg = $this->params[$argnr];
                    } else {
                        $this->modearg = '';
                    }

                    if ($type) {
                        $modearr[$modechar] = $this->modearg;
                    } else {
                        unset($modearr[$modechar]);
                    }

                    $argnr++;
                } elseif (strstr($modesnparam, $modechar)) {
                    // dont add to modearr array, handle in a mode callback
                    if (isset($this->params[$argnr])) {
                        $this->modearg = $this->params[$argnr];
                    } else {
                        $this->modearg = '';
                    }

                    $argnr++;
                } else {
                    // modes without an argument
                    $this->modearg = '';

                    if ($type) {
                        $modearr[$modechar] = $type;
                    } else {
                        unset($modearr[$modechar]);
                    }

                    $argnr++;
		}
//                $this->_tryModeCallbacks($str, $type, $modechar);
            }
	}

	$irc->log( $server_id, 3, "Mode ocurred ($origin, $orighost, $target, $params)");
    }
    function event_part($server_id,&$irc, $origin, $orighost, $target, $params)
    {
	$irc->del_nickone( $server_id, $target, $origin);
    }

    function event_join($server_id,&$irc, $origin, $orighost, $target, $params)
    {
	$irc->add_nick( $server_id, $target, $origin);
    }

    function event_err_nicknameinuse($server_id,&$irc,$origin, $orighost, $target, $params)
    {
        die("Could not connect: Nick already in use");
    }

    function event_rpl_namreply($server_id,&$irc,$origin, $orighost, $target, $params)
    {
        // add nicks to the nicklist and initialize counters
	$tmp  = explode(' ',$target);
        foreach (explode(' ', $params) as $nick) {
            $irc->add_nick( $server_id, $tmp[2], $nick);
        }
    }


    function event_ping($server_id,&$irc, $origin, $orighost, $target, $params)
    {
        $irc->command($server_id,"PONG :$params");
    }

    function event_notice($server_id,&$irc, $origin, $orighost, $target, $params)
    {
      // no op
    }

    function event_privmsg($server_id,&$irc, $origin, $orighost, $target, $params)
    {
	$spaces = preg_split("/\s+/",$params);
        if(preg_match("/^\001(\w|\s)+\001/", $params)){
            /* CTCP */
           $value = preg_split("/(\001|\s)+/", $params,2, PREG_SPLIT_NO_EMPTY);
           $value[0] =  strtolower($value[0]);
           $value[1] = str_replace("\001","",$value[1]);
          if($value[0] ==  "version") {
                $irc->send_ctcpreply($server_id,$origin,"VERSION",  "PEAR::Net_IRC - Version: ". PEAR_NET_IRC_VERSION." - PHP version: " .phpversion());
             } elseif($value[0] == "ping") {
		$irc->send_ctcpreply($server_id,$origin,"PING",$value[1]);
             }
        }
	$irc->log($server_id, 2,"$target : $origin($orighost)> $params");
    }

    function default_handler ($server_id,&$irc, $origin, $orighost, $target, $params) {
        echo "$server_id:DH: $origin : $orighost : $target : $params\n";
    }
}
