/*

Copyright (C) 2000 - 2010 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/

#include <nd.h>
#include <nd_gui.h>
#include <nd_protocol.h>
#include <protocols/snap/nd_snap.h>

#include <nd_fddi.h>
#include <nd_fddi_callbacks.h>

static ND_ProtoField fddi_fields[] = {
  { ND_VAL_FIELD, N_("Frame\nCtrl.\n(0x%.2x)"), N_("Frame Control"),    8,  nd_fddi_fc_cb    },
  { ND_VAL_FIELD, N_("Dst. addr.\n(%s)\n"),     N_("Destination host"), 48, nd_fddi_dhost_cb },
  { ND_VAL_FIELD, N_("Src. addr.\n(%s)\n"),     N_("Source host"),      48, nd_fddi_shost_cb },
  { 0, NULL, NULL, 0, NULL }
};

static LND_Protocol *fddi;
static ND_Protocol  *fddi_gui;

/* Plugin hook implementations: ---------------------------------------- */

const char *
name(void)
{
  return _("FDDI Plugin");
}


const char *
description(void)
{
  return _("A plugin providing support for the FDDI protocol.\n");
}


const char *
author(void)
{
  return ("Christian Kreibich, <christian@whoop.org>");
}


const char *
version(void)
{
  return VERSION_MAJOR;
}


LND_Protocol *
init(void)
{
  if (! (fddi = libnd_proto_registry_find(LND_PROTO_LAYER_LINK, DLT_FDDI)))
    return NULL;
		
  fddi_gui = nd_proto_new(fddi);
  
  fddi_gui->create_gui      = nd_fddi_create_gui;
  fddi_gui->set_gui         = nd_fddi_set_gui;

  /* We're using a button table to display the protocol content,
     so we need to hook them in here: */
  fddi_gui->fields       = fddi_fields;
  fddi_gui->header_width = 104;
	     
  return fddi;
}


/* Protocol method implementations: ------------------------------------ */

GtkWidget *
nd_fddi_create_gui(LND_Trace *trace, LND_ProtoInfo *pinf)
{
  GtkWidget *table;

  table = nd_gui_proto_table_create(trace, pinf);

  return table;
}


void       
nd_fddi_set_gui(const LND_Packet *packet, LND_ProtoInfo *pinf)
{
  struct fddi_header *fh;
  
  fh = (struct fddi_header*) libnd_packet_get_data(packet, fddi, pinf->inst.nesting);

  nd_fddi_set_gui_fc(pinf, fh);
  nd_fddi_set_gui_dhost(pinf, fh);
  nd_fddi_set_gui_shost(pinf, fh);
}


/* Misc helper stuff below --------------------------------------------- */

void
nd_fddi_set_gui_fc(LND_ProtoInfo *pinf, struct fddi_header *fh)
{
  nd_proto_field_set(pinf, &fddi_fields[0], DATA_TO_PTR(fh->fddi_fc));
}


void
nd_fddi_set_gui_dhost(LND_ProtoInfo *pinf, struct fddi_header *fh)
{
  char s[MAXPATHLEN];
  
  g_snprintf(s, MAXPATHLEN, "%.2x:%.2x:%.2x:%.2x:%.2x:%.2x",
	     fh->fddi_dhost[0], fh->fddi_dhost[1],
	     fh->fddi_dhost[2], fh->fddi_dhost[3],
	     fh->fddi_dhost[4], fh->fddi_dhost[5]);

  nd_proto_field_set(pinf, &fddi_fields[1], s);
}


void
nd_fddi_set_gui_shost(LND_ProtoInfo *pinf, struct fddi_header *fh)
{
  char s[MAXPATHLEN];

  g_snprintf(s, MAXPATHLEN, "%.2x:%.2x:%.2x:%.2x:%.2x:%.2x",
	     fh->fddi_shost[0], fh->fddi_shost[1],
	     fh->fddi_shost[2], fh->fddi_shost[3],
	     fh->fddi_shost[4], fh->fddi_shost[5]);

  nd_proto_field_set(pinf, &fddi_fields[2], s);
}


ND_Protocol *
nd_fddi_get_gui(void)
{
  return fddi_gui;
}


LND_Protocol *
nd_fddi_get(void)
{
  return fddi;
}

