/*
	LTOVRUDP.h

	Copyright (C) 2012 Michael Fort, Paul C. Pratt, Rob Mitchelmore

	You can redistribute this file and/or modify it under the terms
	of version 2 of the GNU General Public License as published by
	the Free Software Foundation.  You should have received a copy
	of the license along with this file; see the file COPYING.

	This file is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	license for more details.
*/

/*
	LocalTalk OVeR User Datagram Protocol
*/


#define UDP_dolog (dbglog_HAVE && 0)

/*
	Transmit buffer for localtalk data and its metadata
*/
static unsigned char tx_buffer[4 + LT_TxBfMxSz] =
	"pppp";
GLOBALVAR ui3p LT_TxBuffer = NULL;
GLOBALVAR ui4r LT_TxBuffSz = 0;


/*
	Receive buffer for LocalTalk data and its metadata
*/
GLOBALVAR ui3p LT_RxBuffer = NULL;
GLOBALVAR ui5r LT_RxBuffSz = 0;
static unsigned int rx_buffer_allocation = 1800;

LOCALVAR int sock_fd = 0;
LOCALVAR bool udp_ok = FALSE;

LOCALFUNC int start_udp(void)
{
	int one = 1;
	struct sockaddr_in addr;

	if ((sock_fd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
		/* todo: some error reporting here probably a good plan */
		return -1;
	}
	setsockopt(sock_fd, SOL_SOCKET, SO_REUSEADDR, &one, sizeof(one));
	setsockopt(sock_fd, SOL_SOCKET, SO_REUSEPORT, &one, sizeof(one));

	/* bind it to any address it fancies */
	memset((char*)&addr, 0, sizeof(addr));
	addr.sin_family = AF_INET;
	addr.sin_addr.s_addr = INADDR_ANY;
	addr.sin_port = htons(1954);

	/* bind it */
	errno = 0;
	if (bind(sock_fd, (struct sockaddr*)&addr, sizeof(addr)) < 0) {
#if UDP_dolog
		printf("bind: err %d (%s)", errno, strerror(errno));
#endif
		return -1;
	}

	/* whack it on a multicast group */
	struct ip_mreq mreq;
	mreq.imr_multiaddr.s_addr = inet_addr("239.192.76.84");
	mreq.imr_interface.s_addr = INADDR_ANY;

	if (setsockopt(sock_fd, IPPROTO_IP, IP_ADD_MEMBERSHIP,
		&mreq, sizeof(mreq)) < 0)
	{
#if UDP_dolog
		printf("IP_ADD_MEMBERSHIP: err %d (%s)\n",
			errno, strerror(errno));
#endif
	}

	/* non-blocking I/O is good for the soul */
	fcntl(sock_fd, F_SETFL, O_NONBLOCK);

	udp_ok = TRUE;
	return sock_fd;
}

LOCALVAR unsigned char *MyRxBuffer = NULL;
LOCALVAR struct sockaddr_in MyRxAddress;

/*
	External function needed at startup to initialize the LocalTalk
	functionality.
*/
LOCALFUNC int InitLocalTalk(void)
{
	/* Set up UDP socket */
	start_udp();

	LT_TxBuffer = (ui3p)&tx_buffer[4];

	MyRxBuffer = malloc(rx_buffer_allocation);
	if (NULL == MyRxBuffer) {
		return falseblnr;
	}

	/* Initialized properly */
	return trueblnr;
}

LOCALPROC embedMyPID(void)
{
	/*
		embeds my process ID in network byte order in the start of the
		Tx buffer we assume a pid is at most 32 bits.  As far as I know
		there's no actual implementation of POSIX with 64-bit PIDs so we
		should be ok.
	*/
	int32_t pid;
	pid = (int32_t)getpid();

	int i;
	for (i = 0; i < 4; i++) {
		tx_buffer[i] = (pid >> (3 - i)*8) & 0xff;
	}
}

LOCALPROC LT_TransmitPacket0(void)
{
	size_t bytes;
	/* Write the packet to UDP */
#if UDP_dolog
	printf("writing to udp\n");
#endif
	embedMyPID();
	if (udp_ok) {
		struct sockaddr_in dest;
		memset((char*)&dest, 0, sizeof(dest));
		dest.sin_family = AF_INET;
		dest.sin_addr.s_addr = inet_addr("239.192.76.84");
		dest.sin_port = htons(1954);

		bytes = sendto(sock_fd, tx_buffer, LT_TxBuffSz + 4, 0,
			(struct sockaddr*)&dest, sizeof(dest));
#if 0
		printf("sent %d bytes\n", bytes);
#endif
		(void) bytes; /* avoid warning about unused */
	}
}

/*
	pidInPacketIsMine returns 1 if the process ID embedded in the packet
	is the same as the process ID of the current process
*/
LOCALFUNC int pidInPacketIsMine(void)
{
	/* is the PID in the packet my own PID? */
	int32_t pid;
	pid = (int32_t)getpid();

	int i;
	for (i = 0; i < 4; i++) {
		if (MyRxBuffer[i] != ((pid >> (3 - i)*8) & 0xff)) {
			return 0;
		}
	}

	return 1;
}

/*
	ipInPacketIsMine returns 1 if the source IP for the just-received
	UDP packet is an IP address that is attached to an interface on this
	machine.
*/
LOCALFUNC int ipInPacketIsMine(void)
{
	if (MyRxAddress.sin_family != AF_INET) {
#if UDP_dolog
		printf(
			"baffling error: got a non-inet packet on an inet socket");
#endif
		return 1;
			/*
				because we should drop this garbled packet on the floor
			*/
	}
	int32_t raddr = MyRxAddress.sin_addr.s_addr;

	/*
		Now we need to iterate through all the interfaces on the machine
	*/
	struct ifaddrs *iflist, *ifptr;
	struct sockaddr_in *addr;

	int foundAddress = 0;

	getifaddrs(&iflist);
	for (ifptr = iflist; ifptr; ifptr = ifptr->ifa_next) {
		/* if there is no address in this slot, skip it and move on */
		if (! ifptr->ifa_addr) {
			continue;
		}

		/* if it's not an af_inet then we skip it */
		if (ifptr->ifa_addr->sa_family != AF_INET) {
			continue;
		}

		addr = (struct sockaddr_in*)ifptr->ifa_addr;

		if (addr->sin_addr.s_addr == raddr) {
			foundAddress = 1;
		}
	}
	freeifaddrs(iflist);

	return foundAddress;
}

/*
	packetIsOneISent returns 1 if this looks like a packet that this
	process sent and 0 if it looks like a packet that a different
	process sent.  This provides loopback protection so that we do not
	try to consume packets that we sent ourselves.  We do this by
	checking the process ID embedded in the packet and the IP address
	the packet was sent from.  It would be neater to just look at the
	LocalTalk node ID embedded in the LLAP packet, but this doesn't
	actually work, because during address acquisition it is entirely
	legitimate (and, in the case of collision, *required*) for another
	node to send a packet from what we think is our own node ID.
*/
LOCALFUNC int packetIsOneISent(void)
{
	/*
		do the PID comparison first because it's faster and most of the
		time will disambiguate for us
	*/
	if (pidInPacketIsMine()) {
		return ipInPacketIsMine();
	}
	return 0;
}

LOCALFUNC int GetNextPacket(void)
{
	unsigned char* device_buffer = MyRxBuffer;
	socklen_t addrlen = sizeof(MyRxAddress);

	errno = 0;
	int bytes = recvfrom(sock_fd, device_buffer,
		rx_buffer_allocation, 0,
		(struct sockaddr*)&MyRxAddress, &addrlen);
	if (bytes < 0) {
		if (errno != EAGAIN) {
#if UDP_dolog
			printf("ret %d, bufsize %d, errno = %s\n", bytes,
				rx_buffer_allocation, strerror(errno));
#endif
		}
	} else {
#if UDP_dolog
		printf("got %d, bufsize %d\n", bytes, rx_buffer_allocation);
#endif
	}
	return bytes;
}

LOCALPROC LT_ReceivePacket0(void)
{
	int bytes = GetNextPacket();
	if (bytes > 0) {
		if (! packetIsOneISent()) {
#if UDP_dolog
			printf("passing %d bytes to receiver\n", bytes - 4);
#endif
			LT_RxBuffer = MyRxBuffer + 4;
			LT_RxBuffSz = bytes - 4;
		}
	}
}

LOCALVAR blnr CTSpacketPending = falseblnr;
LOCALVAR ui3r CTSpacketRxDA;
LOCALVAR ui3r CTSpacketRxSA;

/*
	Function used when all the tx data is sent to the SCC as indicated
	by resetting the TX underrun/EOM latch.  If the transmit packet is
	a unicast RTS LAPD packet, we fake the corresponding CTS LAPD
	packet.  This is okay because it is only a collision avoidance
	mechanism and the Ethernet device itself and BPF automatically
	handle collision detection and retransmission.  Besides this is
	what a standard AppleTalk (LocalTalk to EtherTalk) bridge does.
*/
GLOBALOSGLUPROC LT_TransmitPacket(void)
{
	/* Check for LLAP RTS/CTS packets, which we won't send */
	if ((LT_TxBuffSz == 3)
		&& (LT_TxBuffer[2] != 0x81)
		&& (LT_TxBuffer[2] != 0x82))
	{
		/*
			We will automatically and immediately acknowledge
			any non-broadcast RTS packets
		*/
		if ((LT_TxBuffer[0] != 0xFF) && (LT_TxBuffer[2] == 0x84)) {
#if UDP_dolog
			dbglog_writeln("SCC LLAP packet in LT_TransmitPacket");
#endif
			if (CTSpacketPending) {
#if 0
				ReportAbnormalID(0x0701,
					"Already CTSpacketPending in LT_TransmitPacket");
#endif
#if UDP_dolog
				dbglog_writeln(
					"Already CTSpacketPending in LT_TransmitPacket");
#endif
			} else {
				CTSpacketRxDA = LT_TxBuffer[1]; /* rx da = tx sa */
				CTSpacketRxSA = LT_TxBuffer[0]; /* rx sa = tx da */
				CTSpacketPending = trueblnr;
			}
		}
	} else {
		LT_TransmitPacket0();
	}
}

LOCALVAR ui3b MyCTSBuffer[4];

LOCALPROC GetCTSpacket(void)
{
	/* Get a single buffer worth of packets at a time */
	ui3p device_buffer = MyCTSBuffer;

#if UDP_dolog
	dbglog_writeln("SCC receiving CTS packet");
#endif
	/* Create the fake response from the other node */
	device_buffer[0] = CTSpacketRxDA;
	device_buffer[1] = CTSpacketRxSA;
	device_buffer[2] = 0x85;          /* llap cts */

	/* Start the receiver */
	LT_RxBuffer = device_buffer;
	LT_RxBuffSz = 3;

	CTSpacketPending = falseblnr;
}

/* LLAP/SDLC address */
LOCALVAR ui3b my_node_address = 0;

LOCALVAR blnr LTAddrSrchMd = falseblnr;

LOCALPROC GetNextPacketForMe(void)
{
	ui3r dst;
	ui3r src;
	ui3r type;

label_retry:
	LT_ReceivePacket0();

	if (nullpr != LT_RxBuffer) {
		/* Is this packet destined for me? */
		dst = LT_RxBuffer[0];
		src = LT_RxBuffer[1];
		type = LT_RxBuffer[2];

#if UDP_dolog
		dbglog_writeln("SCC receiving packet from UDP");
		dbglog_writelnNum("LT_RxBuffSz", LT_RxBuffSz);
		dbglog_writelnNum("dst", dst);
		dbglog_writelnNum("src", src);
		dbglog_writelnNum("type", type);
#endif

		/*
			we should ignore packets "from" myself except ACK packets,
			which tell me that we've got an address collision,
			and ENQ packets which tell me I might be about to
		*/
#if 0
		/*
			checking for own packets isn't needed, because of
				packetIsOneISent check. if someone else is
				masquerading as our address, it probably is
				more accurate emulation to accept the packet.
		*/
		if ((src == my_node_address)
			&& (type != 0x82) && (type != 0x81))
		{
#if UDP_dolog
			dbglog_writeln("SCC ignore packet from myself");
#endif
			LT_RxBuffer = nullpr;
			goto label_retry;
		} else
#endif
		if ((dst == my_node_address)
			|| (dst == 0xFF)
			|| ! LTAddrSrchMd)
		{
			/* ok */
		} else {
#if UDP_dolog
			dbglog_writeln("SCC ignore packet not for me");
#endif
			LT_RxBuffer = nullpr;
			goto label_retry;
		}
	}
}

GLOBALOSGLUPROC LT_ReceivePacket(void)
{
	if (CTSpacketPending)  {
		GetCTSpacket();
	} else {
		GetNextPacketForMe();
	}
}

GLOBALOSGLUPROC LT_AddrSrchMdSet(blnr v)
{
	LTAddrSrchMd = v;
}

GLOBALOSGLUPROC LT_NodeAddressSet(ui3r v)
{
	if (0 != v) {
		my_node_address = v;
	}
}
