/***************************** LICENSE START ***********************************

 Copyright 2018 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/

#include "MvQEcLayerHelp.h"

#include <QtGlobal>
#include <QApplication>
#include <QComboBox>
#include <QGridLayout>
#include <QLabel>
#include <QLineEdit>
#include <QPainter>
#include <QSortFilterProxyModel>
#include <QToolButton>
#include <QTreeView>
#include <QVBoxLayout>

#include "MvQEcLayerDb.h"
#include "MvQEcLayerLine.h"

#include "MvMiscelaneous.h"

#include "HelpFactory.h"

//=====================================================
//
//  MvQEcLayerModel
//
//=====================================================

MvQEcLayerModel::MvQEcLayerModel(QObject *parent) :
    QAbstractItemModel(parent)
{
}

MvQEcLayerModel::~MvQEcLayerModel()
{
}

int MvQEcLayerModel::columnCount( const QModelIndex& /*parent */ ) const
{
     return 3;
}

int MvQEcLayerModel::rowCount( const QModelIndex& parent) const
{
    //Parent is the root:
    if(!parent.isValid())
    {
        return MvQEcLayerDb::instance()->items().count();
    }

    return 0;
}

QVariant MvQEcLayerModel::data( const QModelIndex& index, int role ) const
{
    int row=index.row();
    if(row < 0 || row >= MvQEcLayerDb::instance()->items().count())
        return QVariant();

    if(role == Qt::DisplayRole)
    {
        if(index.column() == 0)
            return row;
        else if(index.column() == 1)
            return MvQEcLayerDb::instance()->items()[row]->name();
        else if(index.column() == 2)
            return MvQEcLayerDb::instance()->items()[row]->description();
    }
    else if(role == Qt::UserRole)
    {
        return (isFiltered(MvQEcLayerDb::instance()->items()[row]) == true)?"1":"0";
    }
    else if(role == SortRole)
    {
        if(index.column() == 0)
            return row;
        else if(index.column() == 1)
            return MvQEcLayerDb::instance()->items()[row]->name();
    }

    return QVariant();
}

QVariant MvQEcLayerModel::headerData( const int section, const Qt::Orientation orient , const int role ) const
{
    if ( orient != Qt::Horizontal || (role != Qt::DisplayRole &&  role != Qt::ToolTipRole))
              return QAbstractItemModel::headerData( section, orient, role );

    if(role == Qt::DisplayRole)
    {
        switch ( section )
        {
        case 0: return "Preview";
        case 1: return "Name";
        case 2: return "Description";
        default: return QVariant();
        }
    }

    return QVariant();
}

QModelIndex MvQEcLayerModel::index( int row, int column, const QModelIndex & parent ) const
{
    if(row < 0 || column < 0)
    {
        return QModelIndex();
    }

    //When parent is the root this index refers to a node or server
    if(!parent.isValid())
    {
        return createIndex(row,column);
    }

    return QModelIndex();

}

QModelIndex MvQEcLayerModel::parent(const QModelIndex &child) const
{
    return QModelIndex();
}

void MvQEcLayerModel::setKeywordFilter(QString s)
{
    keywordFilter_=s;
}

void MvQEcLayerModel::setGroupFilter(QString s)
{
    groupFilter_=s;
}

void MvQEcLayerModel::setNameFilter(QString s)
{
    nameFilter_=s;
}

void MvQEcLayerModel::clearFilter()
{
    nameFilter_.clear();
    keywordFilter_.clear();
    groupFilter_.clear();
}

bool MvQEcLayerModel::isFiltered(MvQEcLayerDbItem *item) const
{
    if(!nameFilter_.isEmpty())
    {
        if(!item->name().contains(nameFilter_,Qt::CaseInsensitive))
            return false;
    }

    if(!keywordFilter_.isEmpty())
    {
        if(!item->keywords().contains(keywordFilter_))
            return false;
    }

    if(!groupFilter_.isEmpty())
    {
        if(!item->group().contains(groupFilter_))
            return false;
    }

    return true;
}

//==============================
//
// MvQEcLayerDelegate
//
//==============================

MvQEcLayerDelegate::MvQEcLayerDelegate(QWidget *parent) :
    QStyledItemDelegate(parent),
    borderCol_(QColor(210,210,210))
{
}

void MvQEcLayerDelegate::paint(QPainter *painter,const QStyleOptionViewItem &option,
                   const QModelIndex& index) const
{
    painter->save();

    if(index.column() == 0)
    {
        //Background
        QStyleOptionViewItem vopt(option);
        initStyleOption(&vopt, index);

        const QStyle *style = vopt.widget ? vopt.widget->style() : QApplication::style();
        const QWidget* widget = vopt.widget;

        //painter->fillRect(option.rect,QColor(238,238,238));

        //We render everything with the default method
        style->drawControl(QStyle::CE_ItemViewItem, &vopt, painter, widget);

        int row=index.data(Qt::DisplayRole).toInt();
        if(row < 0)
            return;

        QRect rect=option.rect.adjusted(2,2,-4,-2);

        MvQEcLayerDbItem* item=MvQEcLayerDb::instance()->items()[row];
        Q_ASSERT(item);
        if(item)
            item->paint(painter,rect);
    }
    else
    {
        QStyledItemDelegate::paint(painter,option,index);
    }

    //Render the horizontal border for rows. We only render the top border line.
    //With this technique we miss the bottom border line of the last row!!!
    //QRect fullRect=QRect(0,option.rect.y(),painter->device()->width(),option.rect.height());
    QRect bgRect=option.rect;
    painter->setPen(borderCol_);
    painter->drawLine(bgRect.topLeft(),bgRect.topRight());

    painter->restore();
}

QSize MvQEcLayerDelegate::sizeHint(const QStyleOptionViewItem &option, const QModelIndex &index) const
{
    //s.height()+2
    QSize s=QStyledItemDelegate::sizeHint(option,index);
    if(index.column() == 0)
        return QSize(180,50);

    return QSize(s.width(),50);
}



//=====================================================
//
// MvQEcLayerSelectionWidget
//
//=====================================================

MvQEcLayerSelectionWidget::MvQEcLayerSelectionWidget(QWidget* parent) :
    QWidget(parent),
    ignoreFilterChanged_(false)
{
    QVBoxLayout* vb=new QVBoxLayout(this);

    resetTb_=new QToolButton(this);
    resetTb_->setToolButtonStyle(Qt::ToolButtonTextBesideIcon);
    resetTb_->setText(tr("Clear all filters"));
    resetTb_->setIcon(QPixmap(":/desktop/undo.svg"));
    vb->addWidget(resetTb_);

    QGridLayout* grid=new QGridLayout();
    vb->addLayout(grid);

    int cbWidth=42; //number of characters!

    //name
    int row=0;
    grid->addWidget(new QLabel(tr("Name:"),this),row,0);
    grid->setContentsMargins(1,1,1,1);
    grid->setVerticalSpacing(2);

    nameLe_=new QLineEdit(this);
    nameLe_->setPlaceholderText("ANY");
//#if QT_VERSION >= QT_VERSION_CHECK(5, 2, 0)
//    nameLe_->setClearButtonEnabled(true);
//#endif
    grid->addWidget(nameLe_,row,1);

    nameResetTb_=new QToolButton(this);
    nameResetTb_->setIcon(QPixmap(":/desktop/undo.svg"));
    nameResetTb_->setAutoRaise(true);
    nameResetTb_->setToolTip(tr("Clear filter"));
    grid->addWidget(nameResetTb_,row,2);
    row++;

    //keyword
    grid->addWidget(new QLabel(tr("Keyword:"),this),row,0);
    grid->setContentsMargins(1,1,1,1);
    grid->setVerticalSpacing(2);

    keywordCb_=new QComboBox(this);
    keywordCb_->setSizeAdjustPolicy(QComboBox::AdjustToMinimumContentsLengthWithIcon);
    keywordCb_->setMinimumContentsLength(cbWidth);
    keywordCb_->addItem("ANY");
    keywordCb_->addItems(MvQEcLayerDb::instance()->keywords());
    grid->addWidget(keywordCb_,row,1);

    keywordResetTb_=new QToolButton(this);
    keywordResetTb_->setIcon(QPixmap(":/desktop/undo.svg"));
    keywordResetTb_->setAutoRaise(true);
    keywordResetTb_->setToolTip(tr("Clear filter"));
    grid->addWidget(keywordResetTb_,row,2);
    row++;

    //colour
    grid->addWidget(new QLabel(tr("Group:"),this),row,0);

    groupCb_=new QComboBox(this);
    groupCb_->setSizeAdjustPolicy(QComboBox::AdjustToMinimumContentsLengthWithIcon);
    groupCb_->setMinimumContentsLength(cbWidth);
    groupCb_->addItem("ANY");
    groupCb_->addItems(MvQEcLayerDb::instance()->groups());
    grid->addWidget(groupCb_,row,1);

    groupResetTb_=new QToolButton(this);
    groupResetTb_->setIcon(QPixmap(":/desktop/undo.svg"));
    groupResetTb_->setAutoRaise(true);
    groupResetTb_->setToolTip(tr("Clear filter"));
    grid->addWidget(groupResetTb_,row,2);
    row++;


    grid->setColumnStretch(1,1);

    //tree view and model

    tree_=new QTreeView(this);
    tree_->setRootIsDecorated(false);
    tree_->setUniformRowHeights(true);
    tree_->setMinimumHeight(200);
    vb->addWidget(tree_,1);

    model_=new MvQEcLayerModel(this);
    tree_->setItemDelegate(new MvQEcLayerDelegate(this));

    sortModel_=new QSortFilterProxyModel(this);
    sortModel_->setFilterRegExp("1");
    sortModel_->setFilterRole(Qt::UserRole);
    sortModel_->setSortRole(MvQEcLayerModel::SortRole);
    sortModel_->setSourceModel(model_);

    tree_->setSortingEnabled(true);
    tree_->sortByColumn(1,Qt::AscendingOrder);
    tree_->setModel(sortModel_);

    connect(resetTb_,SIGNAL(clicked()),
            this,SLOT(slotClearFilter()));

    connect(nameLe_,SIGNAL(textChanged(QString)),
            this,SLOT(slotNameFilter(QString)));

    connect(nameResetTb_,SIGNAL(clicked()),
            nameLe_,SLOT(clear()));

    connect(keywordCb_,SIGNAL(currentIndexChanged(int)),
            this,SLOT(slotKeywordFilter(int)));

    connect(keywordResetTb_,SIGNAL(clicked()),
            this,SLOT(slotResetKeywordFilter()));

    connect(groupCb_,SIGNAL(currentIndexChanged(int)),
            this,SLOT(slotGroupFilter(int)));

    connect(groupResetTb_,SIGNAL(clicked()),
            this,SLOT(slotResetGroupFilter()));

    connect(tree_,SIGNAL(clicked(QModelIndex)),
            this,SLOT(slotItemSelected(QModelIndex)));

    //Init the button states!!
    checkButtonState();
}

void MvQEcLayerSelectionWidget::slotKeywordFilter(int)
{
    if(ignoreFilterChanged_)
        return;

    QString s=keywordCb_->currentText();
    if(s ==  "ANY") s.clear();
    model_->setKeywordFilter(s);
    sortModel_->invalidate();
    checkButtonState();
}

void MvQEcLayerSelectionWidget::slotGroupFilter(int)
{
    if(ignoreFilterChanged_)
        return;

    QString s=groupCb_->currentText();
    if(s ==  "ANY") s.clear();
    model_->setGroupFilter(s);
    sortModel_->invalidate();
    checkButtonState();
}

void MvQEcLayerSelectionWidget::slotNameFilter(QString s)
{
    if(ignoreFilterChanged_)
        return;

    if(s == "ANY") s.clear();
    model_->setNameFilter(s);
    sortModel_->invalidate();
    checkButtonState();
}

void MvQEcLayerSelectionWidget::slotResetKeywordFilter()
{
    keywordCb_->setCurrentIndex(0);
}

void MvQEcLayerSelectionWidget::slotResetGroupFilter()
{
    groupCb_->setCurrentIndex(0);
}

void MvQEcLayerSelectionWidget::slotItemSelected(const QModelIndex& idx)
{
    QModelIndex idxSrc=sortModel_->mapToSource(idx);
    if(idxSrc.isValid())
    {
        emit itemSelected(idxSrc.row());
    }
}

void MvQEcLayerSelectionWidget::slotClearFilter()
{
    ignoreFilterChanged_=true;
    nameLe_->clear();
    keywordCb_->setCurrentIndex(0);
    groupCb_->setCurrentIndex(0);
    ignoreFilterChanged_=false;

    model_->clearFilter();
    sortModel_->invalidate();
    checkButtonState();
}

void MvQEcLayerSelectionWidget::checkButtonState()
{
    nameResetTb_->setEnabled(!nameLe_->text().isEmpty());
    keywordResetTb_->setEnabled(keywordCb_->currentText() != "ANY");
    groupResetTb_->setEnabled(groupCb_->currentText() != "ANY");

    resetTb_->setEnabled(keywordResetTb_->isEnabled() ||
                         nameResetTb_->isEnabled() || groupResetTb_->isEnabled());
}

void MvQEcLayerSelectionWidget::setCurrent(const std::string& name)
{
    QModelIndex idxSrc=sortModel_->mapToSource(tree_->currentIndex());
    if(idxSrc.isValid())
    {
        int row=idxSrc.row();
        if(row >=0 && row < MvQEcLayerDb::instance()->items().size() &&
           MvQEcLayerDb::instance()->items()[row]->name().toStdString() == name)
           return;
    }

    slotClearFilter();
    int row=MvQEcLayerDb::instance()->indexOf(name);
    if(row >= 0)
    {
        QModelIndex idxSrc=model_->index(row,0);
        if(idxSrc.isValid())
        {
            tree_->setCurrentIndex(sortModel_->mapFromSource(idxSrc));
        }
    }
}

//==================================================
//
//  MvQEcLayerHelp
//
//==================================================

MvQEcLayerHelp::MvQEcLayerHelp(RequestPanel& owner,const Parameter& param) :
           MvQRequestPanelHelp(owner,param)
{
    selector_=new MvQEcLayerSelectionWidget(parentWidget_);

    connect(selector_,SIGNAL(itemSelected(int)),
            this,SLOT(slotSelected(int)));

}

void MvQEcLayerHelp::slotSelected(int idx)
{
    if(idx >=0  && idx < MvQEcLayerDb::instance()->items().count())
    {
        if(MvQEcLayerDb::instance()->items()[idx]->name() != oriName_)
        {
            std::vector<string> vs;
            vs.push_back(metview::toString<int>(idx));
            emit edited(vs);
            oriName_.clear();
        }
    }
}

void MvQEcLayerHelp::refresh(const vector<string>& values)
{
    if(values.size() == 0)
        return;

    oriName_=QString::fromStdString(values[0]);
    selector_->setCurrent(values[0]);
}

static HelpMaker<MvQEcLayerHelp> maker2("help_eclayer");
