#***************************** LICENSE START ***********************************
#
# Copyright 2018 ECMWF and INPE. This software is distributed under the terms
# of the Apache License version 2.0. In applying this license, ECMWF does not
# waive the privileges and immunities granted to it by virtue of its status as
# an Intergovernmental Organization or submit itself to any jurisdiction.
#
# ***************************** LICENSE END *************************************

import json
import sys
import os
import datetime


def load_json_mars(mars_file):
    """ Reads json file and returns dictionary """
    with open(mars_file, 'r') as defs:
        dictionary_of_defs = json.load(defs)
    return dictionary_of_defs


def load_json_layer(layer_file):
    """ Reads json file and returns dictionary """

    # This json file contains a list of dictionaries each with one item only!
    with open(layer_file, 'r') as layers:
        dictionary_of_layers = {}
        lst = json.load(layers)
        # we turn the list into a dictionary
        for v in lst:
            for k in v.keys():
                dictionary_of_layers[k] = v[k]

    return dictionary_of_layers


def build_title_request(main_title, short_name):
    """ Generate Metview request for title """

    start_text = "<grib_info key=\\'base-date\\' format=\\'%d.%m.%Y. %H\\' where=\\'shortName=" \
                 + short_name + "\\' /> UTC"
    step_text = "<grib_info key=\\'step\\' where=\\'shortName=" + short_name + "\\' />h"
    valid_text = "<grib_info key=\\'valid-date\\' format=\\'%d.%m.%Y. %H\\' where=\\'shortName=" + short_name \
                 + "\\' /> UTC"

    title = "MTEXT,\n" \
        + " TEXT_LINE_1 = '" + main_title + "  START: " + start_text + "   STEP:  " + step_text \
        + "  VALID FOR:  " + valid_text + "',\n" \
        + " TEXT_COLOUR = CHARCOAL"

    return title


def build_title_macro(main_title, short_name):
    """ Generate macro code for title """

    start_text = "<grib_info key='base-date' format='%d.%m.%Y. %H' where='shortName=" \
                 + short_name + "' /> UTC"
    step_text = "<grib_info key='step' where='shortName=" + short_name + "' />h"
    valid_text = "<grib_info key='valid-date' format='%d.%m.%Y. %H' where='shortName=" + short_name \
                 + "' /> UTC"

    title = "mtext(\n" \
        + "    TEXT_LINE_1: \"" + main_title + "  START: " + start_text + "  STEP:  " + step_text \
        + "  VALID FOR:  " + valid_text + "\",\n" \
        + "    TEXT_COLOUR: \"CHARCOAL\")"

    return title


def build_visdef_request(style_name):
    """ Generate Metview request for contouring """

    visdef = "MCONT,\n" \
        + " CONTOUR_AUTOMATIC_SETTING = STYLE_NAME, \n" \
        + " CONTOUR_STYLE_NAME = " + style_name + ",\n" \
        + " LEGEND = ON"

    return visdef


def build_visdef_macro(style_name):
    """ Generate macro code for contouring """

    visdef = "mcont(\n" \
        + "    CONTOUR_AUTOMATIC_SETTING: \"STYLE_NAME\",\n" \
        + "    CONTOUR_STYLE_NAME:        \"" + style_name + "\",\n" \
        + "    LEGEND:                    \"ON\")"

    return visdef


def build_legend_request():
    """ Generate Metview request for legend """

    legend_req = "MLEGEND,\n" \
        + " LEGEND_TEXT_COLOUR = CHARCOAL"

    return legend_req


def build_legend_macro():
    """ Generate macro code for legend """

    legend = "mlegend(\n" \
        + "    LEGEND_TEXT_COLOUR: \"CHARCOAL\")"

    return legend


def build_mars_macro(mars):
    """ Generate macro code for MARS request """

    text_for_mars = 'retrieve(\n'
    for key in mars:
        v = mars[key]

        if type(v) == str:

            if not v.startswith('[') or not v.endswith(']'):
                if key == "date" or key == "time" or key == "step":
                    v = int(v)
                else:
                    v = "'" + v + "'"

        text_for_mars += '    ' + str(key) + ' : ' + str(v) + ',\n'
        
    text_for_mars = text_for_mars[:-2]    
    text_for_mars += ')'
    return text_for_mars

    
def main():
    
    # get arguments
    if len(sys.argv) < 7:
        print("Too few arguments = " + str(len(sys.argv)) + ". Minimum number of arguments is 7.")
        sys.exit(1)

    mode = sys.argv[1]
    if mode not in ("data", "macro"):
        print("Incorrect first argument: " + mode + " Allowed values: \"data\" and \"macro\"")
        sys.exit(1)

    layer_def_file = sys.argv[2]
    mars_def_file = sys.argv[3]
    macro_file = sys.argv[4]
    request_file = sys.argv[5]
    layer_name = sys.argv[6]
    style_name = sys.argv[7]

    # Read json defs
    layer_def = load_json_layer(layer_def_file)
    mars_def = load_json_mars(mars_def_file)

    # -----------------------------------
    # Build macro code for MARS retrieval
    # -----------------------------------

    extra_mars_options = {}

    # Read arguments for the mars request
    arguments = sys.argv
    for v in arguments[8:]:
        try:
            (key, val) = v.split(':')
        except ValueError:
            print("No key is defined in argument: ", v)
            raise

        extra_mars_options[key] = val

    # Get MARS request from layer definition
    mars = mars_def[layer_name]['mars']

    # Update it with the extra options
    mars.update(extra_mars_options)

    # Generate macro text for MARS request
    mars_text = build_mars_macro(mars)

    # print("name=", layer_name)
    # print("title=", type(layer_def))

    # -----------------------------------
    # Find out title and style
    # -----------------------------------

    # Get layer title
    layer_title = layer_def[layer_name]['title']

    # Get parameter shortName
    short_name = mars_def[layer_name]['shortName']

    # Get default style name if style is not defined
    if style_name == "":
        style_name = layer_def[layer_name]['style']

    # -----------------------------------
    # In data mode:
    # -generate macro to retrieve data
    # -define the title, contouring and legend as Metview requests
    # -----------------------------------

    if mode == "data":

        # -----------------------------------
        # Macro code to retrieve the data
        # -----------------------------------

        macro_text = "#Metview macro\n\n" \
                + "args=arguments()\n" \
                + "print(args)\n" \
                + "outputFile=args[1]\n" \
                + "data = " + mars_text + "\n\n" \
                + "write(outputFile,data)\n\n" \
                + "return(0)"

        # --------------------------------
        # Build title request
        # --------------------------------

        req_text = build_title_request(layer_title, short_name)

        # --------------------------------
        # Build visdef request
        # --------------------------------

        req_text += "\n\n" + build_visdef_request(style_name)

        # --------------------------------
        # Build legend request
        # --------------------------------

        req_text += "\n\n" + build_legend_request() + "\n"

        # --------------------------------
        # Save request into file
        # --------------------------------
        # print(req_text)

        with open(request_file, 'w') as fp:
            fp.write(req_text)

        # -----------------------------------
        # In macro mode generate macro to retrieve and plot the data
        # -----------------------------------

    else:

        macro_text = "# Metview Macro\n\n"

        # --------------------------------
        #  Add licence text
        # --------------------------------

        with open(os.environ["METVIEW_DIR_SHARE"] + "/etc/licence_for_macros.txt", 'r') as fp:
            licence_text = fp.read().replace("[YEAR]", str(datetime.datetime.now().year))
            for s in licence_text.split("\n"):
                macro_text += "# " + s + "\n"

        # --------------------------------
        # Build mars retrieval
        # --------------------------------

        macro_text += "\n#Retrieve data from MARS\ndata = " + mars_text + "\n\n"

        # --------------------------------
        # Build title
        # --------------------------------

        macro_text += "#Define title\ntitle = " + build_title_macro(layer_title, short_name) + "\n\n"

        # --------------------------------
        # Build visdef
        # --------------------------------

        macro_text += "#Define contouring\ncont = " + build_visdef_macro(style_name) + "\n\n"

        # --------------------------------
        # Add legend
        # --------------------------------

        macro_text += "#Define legend\nlegend = " + build_legend_macro() + "\n\n"

        # ---------------------------------
        # Plot command
        # ---------------------------------

        macro_text += "#Generate plot\nplot(data, title, cont, legend)\n\n"

    # print(macro_text)

    # -----------------------
    #  Save macro into file
    # -----------------------
    with open(macro_file, 'w') as fp:
        fp.write(macro_text)

    # print("macro=", macro_file)


if __name__ == "__main__":    
    main()      
