/* GIMP Plug-in for Macintosh misc file stuff
 * Copyright (C) 2004 Brion Vibber <brion@pobox.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#import <Cocoa/Cocoa.h>
#include <libgimp/gimp.h>

static void   query      (void);
static void   run        (const gchar      *name,
                          gint              nparams,
                          const GimpParam  *param,
                          gint             *nreturn_vals,
                          GimpParam       **return_vals);

static gboolean run_string_service    (gboolean  interactive,
                                       NSString *service,
                                       NSString *string);
static gboolean run_imagefile_service (gboolean  interactive,
                                       NSString *service,
                                       guint32   image_id);

GimpPlugInInfo PLUG_IN_INFO =
{
  NULL,  /* init_proc  */
  NULL,  /* quit_proc  */
  query, /* query_proc */
  run,   /* run_proc   */
};

MAIN ()

static void
query ()
{
  static GimpParamDef file_args[] =
  {
    { GIMP_PDB_INT32, "run_mode", "Interactive, non-interactive" },
    { GIMP_PDB_IMAGE, "image", "Image to show" },
    { GIMP_PDB_DRAWABLE, "drawable", "Drawable (ignored)" }
  };

  gimp_install_procedure ("plug_in_finder_reveal",
                          "show the current image file in Finder",
                          "Causes Finder to open the folder containing the current image file and select it.",
                          "Brion Vibber",
                          "Brion Vibber",
                          "2004",
                          "<Image>/File/Show in Finder",
                          "INDEXED*, GRAY*, RGB*",
                          GIMP_PLUGIN,
                          G_N_ELEMENTS (file_args), 0,
                          file_args, NULL);

}

static void
run (const gchar      *name,
     gint              nparams,
     const GimpParam  *param,
     gint             *nreturn_vals,
     GimpParam       **return_vals)
{
  static GimpParam values[1];
  GimpRunMode run_mode;
  
  run_mode = param[0].data.d_int32;
  gboolean interactive = (GIMP_RUN_INTERACTIVE==run_mode);

  NSAutoreleasePool *pool = [[NSAutoreleasePool alloc] init];

  *nreturn_vals = 1;
  *return_vals = values;
  values[0].type = GIMP_PDB_STATUS;
  values[0].data.d_status = GIMP_PDB_EXECUTION_ERROR;
  
  if (strcmp (name, "plug_in_finder_reveal") == 0)
    {
      if (run_imagefile_service (interactive,
                                 @"Finder/Reveal",
                                 param[1].data.d_int32))
        values[0].data.d_status = GIMP_PDB_SUCCESS;
    }
  else
    values[0].data.d_status = GIMP_PDB_CALLING_ERROR;

  [pool release];
}

static gboolean
run_string_service (gboolean  interactive,
                    NSString *service,
                    NSString *string)
{
  NSApplication *app = [NSApplication sharedApplication];

  NSArray *stringType = [NSArray arrayWithObject: NSStringPboardType];
  [app registerServicesMenuSendTypes: stringType returnTypes: nil];
  
  NSPasteboard *board = [NSPasteboard pasteboardWithUniqueName];
  [board declareTypes: stringType owner: nil];
  [board setString: string forType: NSStringPboardType];
  
  if (NSPerformService (service, board))
    return TRUE;

  if (interactive)
    g_message ("Couldn't run system service %s", [service cString]);
  return FALSE;
}

static gboolean
run_imagefile_service (gboolean  interactive,
                       NSString *service,
                       guint32   image_id)
{
  gchar *filename = gimp_image_get_filename (image_id);
  if (filename)
    return run_string_service (interactive, service,
                               [NSString stringWithUTF8String: filename]);
  if (interactive)
    g_message ("New image not saved on disk.");
  return FALSE;
}
