
/* Line transfer functions
src --> dest
		gray	graya	graya'	rgb		rgba	rgba'
index							x
indexa											x
mono	x
gray	x		x						x
graya					x				
graya'			x						x
rgb				x				x		x
rgba											x
rgba'			x						x
argb'			x						x
0rgb			x				x		x
*/

#define premultiply(value, alpha) ((unsigned char)((unsigned)(value) * alpha / 255))
#define  unmultiply(value, alpha) ((unsigned char)((unsigned)(value) * 255 / alpha))
#define gray_value(red, green, blue) ((unsigned char) \
			(0.30f * (float)(red) + \
			 0.59f * (float)(green) + \
			 0.11f * (float)(blue) ))

typedef void (*xfer_func_type)(unsigned char *,unsigned char*, int, unsigned char*);

/* index -> rgb */

static void xfer_index_rgb(unsigned char *bits, unsigned char *pLine, int width, unsigned char *cmap) {
	int x;
	unsigned index = 0;
	for(x = 0; x < width; x++) {
		index = 3*bits[x];
		pLine[x*3  ] = cmap[index  ];
		pLine[x*3+1] = cmap[index+1];
		pLine[x*3+2] = cmap[index+2];
	}
}

/* indexa -> rgba' */

static void xfer_indexa_rgbaP(unsigned char *bits, unsigned char *pLine, int width, unsigned char *cmap) {
	int x;
	unsigned index;
	for(x = 0; x < width; x++) {
		// Alpha is always on/off, no need to premultiply
		index = 3*bits[x*2];
		pLine[x*4  ] = cmap[index  ];
		pLine[x*4+1] = cmap[index+1];
		pLine[x*4+2] = cmap[index+2];
		pLine[x*4+3] = bits[x*2+1];
	}
}

/* black -> gray */
static void xfer_mono_gray(unsigned char *bits, unsigned char *pLine, int width, unsigned char *cmap) {
	int x;
	guchar *out;
	guchar thisByte;
	int bitNumber = 0;
	out = pLine;
	for (x = 0; x < width; x++) {
		if (bitNumber == 0)
		  {
		    bitNumber = 8;
		    thisByte = bits[x / 8];
		  }
		*out++ = (thisByte & 0x80) ? cmap[3] : cmap[0];
		thisByte <<= 1;
		bitNumber--;
	}
}

/* gray -> gray, graya, rgba */

static void xfer_gray_gray(unsigned char *bits, unsigned char *pLine, int width, unsigned char *cmap) {
	// Raw copy: pasting new images and copying out
	memcpy(pLine, bits, width);
}

static void xfer_gray_graya(unsigned char *bits, unsigned char *pLine, int width, unsigned char *cmap) {
	int x;
	for (x = 0; x < width; x++) {
		// Dummy alpha channel for pasting new layers
		pLine[x*2]   = bits[x];
		pLine[x*2+1] = 255;
	}
}

static void xfer_gray_rgba(unsigned char *bits, unsigned char *pLine, int width, unsigned char *cmap) {
	int x;
	for (x = 0; x < width; x++) {
		// Dummy alpha channel for pasting new layers
		pLine[x*4]   = bits[x];
		pLine[x*4+1] = bits[x];
		pLine[x*4+2] = bits[x];
		pLine[x*4+3] = 255;
	}
}

/* graya -> graya' */

static void xfer_graya_grayaP(unsigned char *bits, unsigned char *pLine, int width, unsigned char *cmap) {
	unsigned char al;
	int x;
	for (x = 0; x < width; x++) {
		al = bits[x*2+1];
		pLine[x*2  ] = premultiply(bits[x*2], al);
		pLine[x*2+1] = al;
	}
}

/* graya' -> graya, rgba */

static void xfer_grayaP_graya(unsigned char *bits, unsigned char *pLine, int width, unsigned char *cmap) {
	unsigned al;
	int x;
	for (x = 0; x < width; x++) {
		al = bits[x*2+1];
		if(al == 0 || al == 255) {
			pLine[x*2]   = bits[x*2];
		} else {
			pLine[x*2] = unmultiply(bits[x*2], al);
		}
		pLine[x*2+1] = (unsigned char)al;
	}
}

static void xfer_grayaP_rgba(unsigned char *bits, unsigned char *pLine, int width, unsigned char *cmap) {
	int x;
	for (x = 0; x < width; x++) {
		unsigned al = bits[x*2+1];
		unsigned char gray;
		if(al == 0 || al == 255) {
			gray = bits[x*2];
		} else {
			gray = unmultiply(bits[x*2], al);
		}
		pLine[x*4]   = gray;
		pLine[x*4+1] = gray;
		pLine[x*4+2] = gray;
		pLine[x*4+3] = (unsigned char)al;
	}
}


/* rgb -> graya, rgb, rgba */

static void xfer_rgb_graya(unsigned char *bits, unsigned char *pLine, int width, unsigned char *cmap) {
	int x;
	for (x = 0; x < width; x++) {
		// Dummy alpha for pasting new layers
		pLine[x*2  ] = gray_value(bits[x*3], bits[x*3+1], bits[x*3+2]);
		pLine[x*2+1] = 255;
	}
}

static void xfer_rgb_rgb(unsigned char *bits, unsigned char *pLine, int width, unsigned char *cmap) {
	memcpy(pLine, bits, width*3);
}

static void xfer_rgb_rgba(unsigned char *bits, unsigned char *pLine, int width, unsigned char *cmap) {
	int x;
	for (x = 0; x < width; x++) {
		pLine[x*4  ] = bits[x*3  ];
		pLine[x*4+1] = bits[x*3+1];
		pLine[x*4+2] = bits[x*3+2];
		pLine[x*4+3] = 255;
	}
}

/* rgba -> rgba' */

static void xfer_rgba_rgbaP(unsigned char *bits, unsigned char *pLine, int width, unsigned char *cmap) {
	int x;
	unsigned char al;
	for (x = 0; x < width; x++) {
		al = bits[x*4+3];
		pLine[x*4  ] = premultiply(bits[x*4  ], al);
		pLine[x*4+1] = premultiply(bits[x*4+1], al);
		pLine[x*4+2] = premultiply(bits[x*4+2], al);
		pLine[x*4+3] = al;
	}
}

/* rgba' -> graya, rgba */

static void xfer_rgbaP_graya(unsigned char *bits, unsigned char *pLine, int width, unsigned char *cmap) {
	int x;
	for (x = 0; x < width; x++) {
		unsigned al = bits[x*4+3];
		unsigned gray = gray_value(bits[x*4], bits[x*4+1], bits[x*4+2]);
		if(al == 0 || al == 255) {
			pLine[x*2] = (unsigned char)gray;
		} else {
			pLine[x*2] = unmultiply(gray, al);
		}
		pLine[x*2+1] = (unsigned char)al;
	}
}

static void xfer_rgbaP_rgba(unsigned char *bits, unsigned char *pLine, int width, unsigned char *cmap) {
	int x;
	for (x = 0; x < width; x++) {
		unsigned al = bits[x*4+3];
		if(al == 0 || al == 255) {
			pLine[x*4]   = bits[x*4  ];
			pLine[x*4+1] = bits[x*4+1];
			pLine[x*4+2] = bits[x*4+2];
		} else {
			pLine[x*4]   = unmultiply(bits[x*4  ], al);
			pLine[x*4+1] = unmultiply(bits[x*4+1], al);
			pLine[x*4+2] = unmultiply(bits[x*4+2], al);
		}
		pLine[x*4+3] = (unsigned char)al;
	}
}

/* argb' -> graya, rgba */

static void xfer_argbP_graya(unsigned char *bits, unsigned char *pLine, int width, unsigned char *cmap) {
	int x;
	for (x = 0; x < width; x++) {
		unsigned al = bits[x*4];
		unsigned gray = gray_value(bits[x*4+1], bits[x*4+2], bits[x*4+3]);
		if (al == 0 || al == 255) {
			pLine[x*2] = (unsigned char)gray;
		} else {
			// Un-premultiply alpha...
			pLine[x*2] = (unsigned char)(gray * 255 / al);
		}
		pLine[x*2+1] = (unsigned char)al;
	}
}


static void xfer_argbP_rgba(unsigned char *bits, unsigned char *pLine, int width, unsigned char *cmap) {
	int x;
	for (x = 0; x < width; x++) {
		unsigned al = bits[x*4];
		if(al == 0 || al == 255) {
			pLine[x*4]   = bits[x*4+1];
			pLine[x*4+1] = bits[x*4+2];
			pLine[x*4+2] = bits[x*4+3];
		} else {
			pLine[x*4]   = unmultiply(bits[x*4+1], al);
			pLine[x*4+1] = unmultiply(bits[x*4+2], al);
			pLine[x*4+2] = unmultiply(bits[x*4+3], al);
		}
		pLine[x*4+3] = (unsigned char)al;
	}
}


/* 0rgb -> graya, rgb, rgba */

static void xfer_0rgb_graya(unsigned char *bits, unsigned char *pLine, int width, unsigned char *cmap) {
	int x;
	for (x = 0; x < width; x++) {
		// Ignore alpha channel; create new dummy channel
		pLine[x*2] = gray_value(bits[x*4+1], bits[x*4+2], bits[x*4+3]);
		pLine[x*2+1] = 255;
	}
}

static void xfer_0rgb_rgb(unsigned char *bits, unsigned char *pLine, int width, unsigned char *cmap) {
	int x;
	for (x = 0; x < width; x++) {
		// Ignore alpha channel
		pLine[x*3]   = bits[x*4+1];
		pLine[x*3+1] = bits[x*4+2];
		pLine[x*3+2] = bits[x*4+3];
	}
}

static void xfer_0rgb_rgba(unsigned char *bits, unsigned char *pLine, int width, unsigned char *cmap) {
	int x;
	for (x = 0; x < width; x++) {
		// Ignore alpha channel; create new dummy channel
		pLine[x*4]   = bits[x*4+1];
		pLine[x*4+1] = bits[x*4+2];
		pLine[x*4+2] = bits[x*4+3];
		pLine[x*4+3] = 255;
	}
}
