function [SBMLfieldnames, nNumberFields] = getStructureFieldnames(varargin)
% [SBMLfieldnames, nNumberFields] = getStructureFieldnames(typecode, level, version, pkgversion(optional))
%
% Takes
%
% 1. typecode; a string representing the type of object being queried
% 2. level, an integer representing an SBML level
% 3. version, an integer representing an SBML version
% 4. pkgversion, an integer representing the SBML package version 
%       (defaults to 1)
%
% Returns
%
% 1. an array of fieldnames for an SBML structure of the given typecode, level and version
% 2. the number of fieldnames
%

%<!---------------------------------------------------------------------------
% This file is part of libSBML.  Please visit http://sbml.org for more
% information about SBML, and the latest version of libSBML.
%
% Copyright (C) 2013-2018 jointly by the following organizations:
%     1. California Institute of Technology, Pasadena, CA, USA
%     2. EMBL European Bioinformatics Institute (EMBL-EBI), Hinxton, UK
%     3. University of Heidelberg, Heidelberg, Germany
%
% Copyright (C) 2009-2013 jointly by the following organizations: 
%     1. California Institute of Technology, Pasadena, CA, USA
%     2. EMBL European Bioinformatics Institute (EMBL-EBI), Hinxton, UK
%  
% Copyright (C) 2006-2008 by the California Institute of Technology,
%     Pasadena, CA, USA 
%  
% Copyright (C) 2002-2005 jointly by the following organizations: 
%     1. California Institute of Technology, Pasadena, CA, USA
%     2. Japan Science and Technology Agency, Japan
% 
% This library is free software; you can redistribute it and/or modify it
% under the terms of the GNU Lesser General Public License as published by
% the Free Software Foundation.  A copy of the license agreement is provided
% in the file named "LICENSE.txt" included with this software distribution
% and also available online as http://sbml.org/software/libsbml/license.html
%----------------------------------------------------------------------- -->

typecode = varargin{1};
level = varargin{2};
version = varargin{3};

pkgCount = 0;

if (nargin > 3)
    if (nargin < 5)
        error('not enough arguments');
    end;
    pkgCount = length(varargin{4});
 %   packages = cell(1, pkgCount);
  %  pkgVersion = zeros(1, pkgCount);
    packages = varargin{4};
    pkgVersion = varargin{5};
end;


    SBMLfieldnames = [];
    nNumberFields = 0;
done = 1;


switch (typecode)
  case {'SBML_ALGEBRAIC_RULE', 'AlgebraicRule', 'algebraicRule', 'rule'}
    fhandle = str2func('getAlgebraicRuleFieldnames');
  case {'SBML_ASSIGNMENT_RULE', 'AssignmentRule', 'assignmentRule'}
    fhandle = str2func('getAssignmentRuleFieldnames');
  case {'SBML_COMPARTMENT', 'Compartment', 'compartment'}
    fhandle = str2func('getCompartmentFieldnames');
  case {'SBML_COMPARTMENT_TYPE', 'CompartmentType', 'compartmentType'}
    fhandle = str2func('getCompartmentTypeFieldnames');
  case {'SBML_COMPARTMENT_VOLUME_RULE', 'CompartmentVolumeRule', 'compartmentVolumeRule'}
    fhandle = str2func('getCompartmentVolumeRuleFieldnames');
  case {'SBML_CONSTRAINT', 'Constraint', 'constraint'}
    fhandle = str2func('getConstraintFieldnames');
  case {'SBML_DELAY', 'Delay', 'delay'}
    fhandle = str2func('getDelayFieldnames');
  case {'SBML_EVENT', 'Event', 'event'}
    fhandle = str2func('getEventFieldnames');
  case {'SBML_EVENT_ASSIGNMENT', 'EventAssignment', 'eventAssignment'}
    fhandle = str2func('getEventAssignmentFieldnames');
  case {'SBML_FUNCTION_DEFINITION', 'FunctionDefinition', 'functionDefinition'}
    fhandle = str2func('getFunctionDefinitionFieldnames');
  case {'SBML_INITIAL_ASSIGNMENT', 'InitialAssignment', 'initialAssignment'}
    fhandle = str2func('getInitialAssignmentFieldnames');
  case {'SBML_KINETIC_LAW', 'KineticLaw', 'kineticLaw'}
    fhandle = str2func('getKineticLawFieldnames');
  case {'SBML_LOCAL_PARAMETER', 'LocalParameter', 'localParameter'}
    fhandle = str2func('getLocalParameterFieldnames');
  case {'SBML_MODEL', 'Model', 'model'}
    fhandle = str2func('getModelFieldnames');
  case {'SBML_MODIFIER_SPECIES_REFERENCE', 'ModifierSpeciesReference', 'modifierSpeciesReference', 'modifier'}
    fhandle = str2func('getModifierSpeciesReferenceFieldnames');
  case {'SBML_PARAMETER', 'Parameter', 'parameter'}
    fhandle = str2func('getParameterFieldnames');
  case {'SBML_PARAMETER_RULE', 'ParameterRule', 'parameterRule'}
    fhandle = str2func('getParameterRuleFieldnames');
  case {'SBML_PRIORITY', 'Priority', 'priority'}
    fhandle = str2func('getPriorityFieldnames');
  case {'SBML_RATE_RULE', 'RateRule', 'ruleRule'}
    fhandle = str2func('getRateRuleFieldnames');
  case {'SBML_REACTION', 'Reaction', 'reaction'}
    fhandle = str2func('getReactionFieldnames');
  case {'SBML_SPECIES', 'Species', 'species'}
    fhandle = str2func('getSpeciesFieldnames');
  case {'SBML_SPECIES_CONCENTRATION_RULE', 'SpeciesConcentrationRule', 'speciesConcentrationRule'}
    fhandle = str2func('getSpeciesConcentrationRuleFieldnames');
  case {'SBML_SPECIES_REFERENCE', 'SpeciesReference', 'speciesReference', 'reactant', 'Reactant', 'product', 'Product'}
    fhandle = str2func('getSpeciesReferenceFieldnames');
  case {'SBML_SPECIES_TYPE', 'SpeciesType', 'speciesType'}
    fhandle = str2func('getSpeciesTypeFieldnames');
  case {'SBML_STOICHIOMETRY_MATH', 'StoichiometryMath', 'stoichiometryMath'}
    fhandle = str2func('getStoichiometryMathFieldnames');
  case {'SBML_TRIGGER', 'Trigger', 'trigger'}
    fhandle = str2func('getTriggerFieldnames');
  case {'SBML_UNIT', 'Unit', 'unit'}
    fhandle = str2func('getUnitFieldnames');
  case {'SBML_UNIT_DEFINITION', 'UnitDefinition', 'unitDefinition'}
    fhandle = str2func('getUnitDefinitionFieldnames');
  otherwise
    done = 0;  
end;

if done == 1
  [SBMLfieldnames, nNumberFields] = feval(fhandle, level, version);
else
    i = 1;
    found = 0;
    while (i < pkgCount+1)
        [found, fhandle] = getFields(typecode, packages{i}, 0);
        if (found == 1)
            break;
        end;
        i = i + 1;
    end;
    if (found == 1)
        [SBMLfieldnames, nNumberFields] = feval(fhandle, level, version, pkgVersion(i));
        done = 1;
    end;
end;
 
if (done == 1 && pkgCount > 0)
    % we may need to extend the fields 
    i = 1;
    while (i < pkgCount+1)
        [found, fhandle] = getFields(typecode, packages{i}, 1);
        if (found == 1)
            copySBML = SBMLfieldnames;
            [newSBMLfieldnames, newNumberFields] = feval(fhandle, level, version, pkgVersion(i));
            SBMLfieldnames = [copySBML, newSBMLfieldnames];
            nNumberFields = nNumberFields + newNumberFields;
        end;
        i = i + 1;
    end;
end;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [found, fhandle] = getFields(typecode, pkg, extension)
    fhandle = str2func('disp');
    found = 0;
    if (strcmp(pkg, 'fbc'))
        if (extension)
            if (isFBCExtension(typecode))
        [found, fhandle] = getFBCFieldnameFunction(typecode);
            end;
        else
        [found, fhandle] = getFBCFieldnameFunction(typecode);
        end;
    elseif strcmp(pkg, 'qual')
      if (extension)
        if (isQUALExtension(typecode))
          [found, fhandle] = getQUALFieldnameFunction(typecode);
        end;
      else
        [found, fhandle] = getQUALFieldnameFunction(typecode);
      end;
    elseif strcmp(pkg, 'groups')
      if (extension)
        if (isGROUPSExtension(typecode))
          [found, fhandle] = getGROUPSFieldnameFunction(typecode);
        end;
      else
        [found, fhandle] = getGROUPSFieldnameFunction(typecode);
      end;
    end;
%%%%% REMOVE END
%%%%% ADD ADDITIONAL


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%	
function extend = isFBCExtension(typecode)
   extend = 0;
   switch (typecode)
        case {'SBML_FBC_MODEL', 'FBCModel', 'SBML_MODEL', 'Model', 'model'}
             extend = 1;
       case {'SBML_FBC_SPECIES', 'FBCSpecies', 'SBML_SPECIES', 'Species', 'species'}
            extend = 1;
        case {'SBML_FBC_REACTION', 'FBCReaction','SBML_REACTION', 'Reaction', 'reaction'}
            extend = 1;
   end;
   
function extend = isQUALExtension(typecode)
  extend = 0;
  switch (typecode)
    case {'SBML_QUAL_MODEL', 'QUALModel', 'SBML_MODEL', 'Model', 'model'}
      extend = 1;
  end;

function extend = isGROUPSExtension(typecode)
  extend = 0;
  switch (typecode)
    case {'SBML_GROUPS_MODEL', 'GROUPSModel', 'SBML_MODEL', 'Model', 'model'}
      extend = 1;
  end;
%%%% ADD isExtension
%%%% ADD isExtension
%%%% END isExtension

function [found, fhandle] = getFBCFieldnameFunction(typecode)
    found = 1;
    switch (typecode)
        case {'SBML_FBC_FLUXBOUND', 'FluxBound', 'fluxBound', 'fbc_fluxBound'}
            fhandle = str2func('getFluxBoundFieldnames');
        case {'SBML_FBC_FLUXOBJECTIVE', 'FluxObjective', 'fluxObjective', 'fbc_fluxObjective'}
            fhandle = str2func('getFluxObjectiveFieldnames');
        case {'SBML_FBC_OBJECTIVE', 'Objective', 'objective', 'fbc_objective'}
           fhandle = str2func('getObjectiveFieldnames');
        case {'SBML_FBC_GENE_PRODUCT', 'GeneProduct', 'geneProduct', 'fbc_geneProduct'}
           fhandle = str2func('getGeneProductFieldnames');
        case {'SBML_FBC_GENE_PRODUCT_ASSOCIATION', 'GeneProductAssociation', 'geneProductAssociation', 'fbc_geneProductAssociation'}
           fhandle = str2func('getGeneProductAssociationFieldnames');
        case {'SBML_FBC_ASSOCIATION', 'Association', 'association', 'fbc_association'}
          fhandle = str2func('getAssociationFieldnames');
        case {'SBML_FBC_MODEL', 'FBCModel', 'SBML_MODEL', 'Model', 'model'}
           fhandle = str2func('getFBCModelFieldnames');    
        case {'SBML_FBC_SPECIES', 'FBCSpecies', 'SBML_SPECIES', 'Species', 'species'}
          fhandle = str2func('getFBCSpeciesFieldnames');
        case {'SBML_FBC_REACTION', 'FBCReaction','SBML_REACTION', 'Reaction', 'reaction'}
          fhandle = str2func('getFBCReactionFieldnames');
        otherwise
          fhandle = str2func('disp');
          found = 0;
    end;

function [found, fhandle] = getQUALFieldnameFunction(typecode)
  found = 1;
  switch (typecode)
    case {'SBML_QUAL_QUALITATIVE_SPECIES', 'QualitativeSpecies', 'qualitativeSpecies', 'qual_qualitativeSpecies'}
      fhandle = str2func('getQualitativeSpeciesFieldnames');
    case {'SBML_QUAL_TRANSITION', 'Transition', 'transition', 'qual_transition'}
      fhandle = str2func('getTransitionFieldnames');
    case {'SBML_QUAL_INPUT', 'Input', 'input', 'qual_input'}
      fhandle = str2func('getInputFieldnames');
    case {'SBML_QUAL_OUTPUT', 'Output', 'output', 'qual_output'}
      fhandle = str2func('getOutputFieldnames');
    case {'SBML_QUAL_DEFAULT_TERM', 'DefaultTerm', 'defaultTerm', 'qual_defaultTerm'}
      fhandle = str2func('getDefaultTermFieldnames');
    case {'SBML_QUAL_FUNCTION_TERM', 'FunctionTerm', 'functionTerm', 'qual_functionTerm'}
      fhandle = str2func('getFunctionTermFieldnames');
    case {'SBML_QUAL_MODEL', 'QUALModel', 'SBML_MODEL', 'Model', 'model'}
      fhandle = str2func('getQUALModelFieldnames');
    otherwise
      fhandle = str2func('disp');
      found = 0;
  end;

function [found, fhandle] = getGROUPSFieldnameFunction(typecode)
  found = 1;
  switch (typecode)
    case {'SBML_GROUPS_GROUP', 'Group', 'group', 'groups_group'}
      fhandle = str2func('getGroupFieldnames');
    case {'SBML_GROUPS_MEMBER', 'Member', 'member', 'groups_member'}
      fhandle = str2func('getMemberFieldnames');
    case {'SBML_GROUPS_MODEL', 'GROUPSModel', 'SBML_MODEL', 'Model', 'model'}
      fhandle = str2func('getGROUPSModelFieldnames');
    otherwise
      fhandle = str2func('disp');
      found = 0;
  end;
%%%% ADD getFieldname
%%%% ADD getFieldname
%%%% END getFieldname

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getAlgebraicRuleFieldnames(level, ...
                                                             version)










if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

if (level == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'notes', ...
		                   'annotation', ...
		                   'type', ...
		                   'formula', ...
		                   'variable', ...
		                   'species', ...
		                   'compartment', ...
		                   'name', ...
		                   'units', ...
		                 };
		nNumberFields = 10;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'formula', ...
		                   'variable', ...
		                   'species', ...
		                   'compartment', ...
		                   'name', ...
		                   'units', ...
		                 };
		nNumberFields = 10;
	elseif (version == 2)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'sboTerm', ...
		                   'formula', ...
		                   'variable', ...
		                   'species', ...
		                   'compartment', ...
		                   'name', ...
		                   'units', ...
		                 };
		nNumberFields = 11;
	elseif (version == 3)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'sboTerm', ...
		                   'formula', ...
		                   'variable', ...
		                   'species', ...
		                   'compartment', ...
		                   'name', ...
		                   'units', ...
		                 };
		nNumberFields = 11;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                           'cvterms', ...
		                   'sboTerm', ...
		                   'formula', ...
		                   'variable', ...
		                   'species', ...
		                   'compartment', ...
		                   'name', ...
		                   'units', ...
		                 };
		nNumberFields = 12;
	end;
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                           'cvterms', ...
		                   'sboTerm', ...
		                   'formula', ...
		                   'variable', ...
		                   'species', ...
		                   'compartment', ...
		                   'name', ...
		                   'units', ...
		                 };
		nNumberFields = 12;
        elseif (version == 2)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                           'cvterms', ...
		                   'sboTerm', ...
		                   'formula', ...
		                   'variable', ...
		                   'species', ...
		                   'compartment', ...
		                   'name', ...
		                   'units', ...
                                   'id', ...
                                   'name', ...
		                 };
                  nNumberFields = 14;
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getAssignmentRuleFieldnames(level, ...
                                                             version)











if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

if (level == 1)
		SBMLfieldnames = [];
		nNumberFields = 0;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'formula', ...
		                   'variable', ...
		                   'species', ...
		                   'compartment', ...
		                   'name', ...
		                   'units', ...
		                 };
		nNumberFields = 10;
	elseif (version == 2)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'sboTerm', ...
		                   'formula', ...
		                   'variable', ...
		                   'species', ...
		                   'compartment', ...
		                   'name', ...
		                   'units', ...
		                 };
		nNumberFields = 11;
	elseif (version == 3)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'sboTerm', ...
		                   'formula', ...
		                   'variable', ...
		                   'species', ...
		                   'compartment', ...
		                   'name', ...
		                   'units', ...
		                 };
		nNumberFields = 11;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                           'cvterms', ...
		                   'sboTerm', ...
		                   'formula', ...
		                   'variable', ...
		                   'species', ...
		                   'compartment', ...
		                   'name', ...
		                   'units', ...
		                 };
		nNumberFields = 12;
	end;
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                          'cvterms', ...
		                   'sboTerm', ...
		                   'formula', ...
		                   'variable', ...
		                   'species', ...
		                   'compartment', ...
		                   'name', ...
		                   'units', ...
		                 };
		nNumberFields = 12;
        elseif (version == 2)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                          'cvterms', ...
		                   'sboTerm', ...
		                   'formula', ...
		                   'variable', ...
		                   'species', ...
		                   'compartment', ...
		                   'name', ...
		                   'units', ...
                                   'id', ...
                                   'name', ...
		                 };
                  nNumberFields = 14;
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getCompartmentFieldnames(level, ...
                                                                    version)











if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

if (level == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'notes', ...
		                   'annotation', ...
		                   'name', ...
		                   'volume', ...
		                   'units', ...
		                   'outside', ...
		                   'isSetVolume', ...
		                 };
		nNumberFields = 8;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'name', ...
		                   'id', ...
		                   'spatialDimensions', ...
		                   'size', ...
		                   'units', ...
		                   'outside', ...
		                   'constant', ...
		                   'isSetSize', ...
		                   'isSetVolume', ...
		                 };
		nNumberFields = 13;
	elseif (version == 2)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'name', ...
		                   'id', ...
		                   'compartmentType', ...
		                   'spatialDimensions', ...
		                   'size', ...
		                   'units', ...
		                   'outside', ...
		                   'constant', ...
		                   'isSetSize', ...
		                   'isSetVolume', ...
		                 };
		nNumberFields = 14;
	elseif (version == 3)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'sboTerm', ...
		                   'name', ...
		                   'id', ...
		                   'compartmentType', ...
		                   'spatialDimensions', ...
		                   'size', ...
		                   'units', ...
		                   'outside', ...
		                   'constant', ...
		                   'isSetSize', ...
		                   'isSetVolume', ...
		                 };
		nNumberFields = 15;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                           'cvterms', ...
		                   'sboTerm', ...
		                   'name', ...
		                   'id', ...
		                   'compartmentType', ...
		                   'spatialDimensions', ...
		                   'size', ...
		                   'units', ...
		                   'outside', ...
		                   'constant', ...
		                   'isSetSize', ...
		                   'isSetVolume', ...
		                 };
		nNumberFields = 16;
	end;
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                          'cvterms', ...
		                   'sboTerm', ...
		                   'name', ...
		                   'id', ...
		                   'spatialDimensions', ...
		                   'size', ...
		                   'units', ...
		                   'constant', ...
		                   'isSetSize', ...
		                   'isSetSpatialDimensions', ...
		                 };
		nNumberFields = 14;
        elseif (version == 2)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                          'cvterms', ...
		                   'sboTerm', ...
		                   'name', ...
		                   'id', ...
		                   'spatialDimensions', ...
		                   'size', ...
		                   'units', ...
		                   'constant', ...
		                   'isSetSize', ...
		                   'isSetSpatialDimensions', ...
		                 };
		nNumberFields = 14;
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getCompartmentTypeFieldnames(level, ...
                                                                        version)











if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = [];
		nNumberFields = 0;
	elseif (version == 2)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'name', ...
		                   'id', ...
		                 };
		nNumberFields = 6;
	elseif (version == 3)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'sboTerm', ...
		                   'name', ...
		                   'id', ...
		                 };
		nNumberFields = 7;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                           'cvterms', ...
		                   'sboTerm', ...
		                   'name', ...
		                   'id', ...
		                 };
		nNumberFields = 8;
	end;
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = [];
		nNumberFields = 0;
        elseif (version == 2)
		SBMLfieldnames = [];
                  nNumberFields = 2;
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getCompartmentVolumeRuleFieldnames(level, ...
                                                             version)











if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

if (level == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'notes', ...
		                   'annotation', ...
		                   'type', ...
		                   'formula', ...
		                   'variable', ...
		                   'species', ...
		                   'compartment', ...
		                   'name', ...
		                   'units', ...
		                 };
		nNumberFields = 10;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = [];
		nNumberFields = 0;
	elseif (version == 2)
		SBMLfieldnames = [];
		nNumberFields = 0;
	elseif (version == 3)
		SBMLfieldnames = [];
		nNumberFields = 0;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = [];
		nNumberFields = 0;
	end;
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = [];
		nNumberFields = 0;
        elseif (version == 2)
		SBMLfieldnames = [];
                  nNumberFields = 2;
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getConstraintFieldnames(level, ...
                                                                   version)











if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = [];
		nNumberFields = 0;
	elseif (version == 2)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'sboTerm', ...
		                   'math', ...
		                   'message', ...
		                 };
		nNumberFields = 7;
	elseif (version == 3)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'sboTerm', ...
		                   'math', ...
		                   'message', ...
		                 };
		nNumberFields = 7;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                           'cvterms', ...
		                   'sboTerm', ...
		                   'math', ...
		                   'message', ...
		                 };
		nNumberFields = 8;
	end;
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                          'cvterms', ...
		                   'sboTerm', ...
		                   'math', ...
		                   'message', ...
		                 };
		nNumberFields = 8;
        elseif (version == 2)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                          'cvterms', ...
		                   'sboTerm', ...
		                   'math', ...
		                   'message', ...
                                   'id', ...
                                   'name', ...
		                 };
                  nNumberFields = 10;
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getDelayFieldnames(level, ...
                                                              version)











if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = [];
		nNumberFields = 0;
	elseif (version == 2)
		SBMLfieldnames = [];
		nNumberFields = 0;
	elseif (version == 3)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'sboTerm', ...
		                   'math', ...
		                 };
		nNumberFields = 6;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                           'cvterms', ...
		                   'sboTerm', ...
		                   'math', ...
		                 };
		nNumberFields = 7;
	end;
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                          'cvterms', ...
		                   'sboTerm', ...
		                   'math', ...
		                 };
		nNumberFields = 7;
        elseif (version == 2)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                          'cvterms', ...
		                   'sboTerm', ...
		                   'math', ...
                                   'id', ...
                                   'name', ...
		                 };
                  nNumberFields =9;
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getEventAssignmentFieldnames(level, ...
                                                                        version)











if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'variable', ...
		                   'math', ...
		                 };
		nNumberFields = 6;
	elseif (version == 2)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'variable', ...
		                   'sboTerm', ...
		                   'math', ...
		                 };
		nNumberFields = 7;
	elseif (version == 3)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'sboTerm', ...
		                   'variable', ...
		                   'math', ...
		                 };
		nNumberFields = 7;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                           'cvterms', ...
		                   'sboTerm', ...
		                   'variable', ...
		                   'math', ...
		                 };
		nNumberFields = 8;
	end;
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                          'cvterms', ...
		                   'sboTerm', ...
		                   'variable', ...
		                   'math', ...
		                 };
		nNumberFields = 8;
        elseif (version == 2)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                          'cvterms', ...
		                   'sboTerm', ...
		                   'variable', ...
		                   'math', ...
                                   'id', ...
                                   'name', ...
		                 };
                  nNumberFields = 10;
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getEventFieldnames(level, ...
                                                              version)











if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'name', ...
		                   'id', ...
		                   'trigger', ...
		                   'delay', ...
		                   'timeUnits', ...
		                   'eventAssignment', ...
		                 };
		nNumberFields = 10;
	elseif (version == 2)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'name', ...
		                   'id', ...
		                   'trigger', ...
		                   'delay', ...
		                   'timeUnits', ...
		                   'sboTerm', ...
		                   'eventAssignment', ...
		                 };
		nNumberFields = 11;
	elseif (version == 3)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'sboTerm', ...
		                   'name', ...
		                   'id', ...
		                   'trigger', ...
		                   'delay', ...
		                   'eventAssignment', ...
		                 };
		nNumberFields = 10;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                           'cvterms', ...
		                   'sboTerm', ...
		                   'name', ...
		                   'id', ...
		                   'useValuesFromTriggerTime', ...
		                   'trigger', ...
		                   'delay', ...
		                   'eventAssignment', ...
		                 };
		nNumberFields = 12;
	end;
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                          'cvterms', ...
		                   'sboTerm', ...
		                   'name', ...
		                   'id', ...
		                   'useValuesFromTriggerTime', ...
		                   'trigger', ...
		                   'delay', ...
		                   'priority', ...
		                   'eventAssignment', ...
		                 };
		nNumberFields = 13;
        elseif (version == 2)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                          'cvterms', ...
		                   'sboTerm', ...
		                   'name', ...
		                   'id', ...
		                   'useValuesFromTriggerTime', ...
		                   'trigger', ...
		                   'delay', ...
		                   'priority', ...
		                   'eventAssignment', ...
		                 };
		nNumberFields = 13;
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getFunctionDefinitionFieldnames(level, ...
                                                                           version)











if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'name', ...
		                   'id', ...
		                   'math', ...
		                 };
		nNumberFields = 7;
	elseif (version == 2)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'sboTerm', ...
		                   'name', ...
		                   'id', ...
		                   'math', ...
		                 };
		nNumberFields = 8;
	elseif (version == 3)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'sboTerm', ...
		                   'name', ...
		                   'id', ...
		                   'math', ...
		                 };
		nNumberFields = 8;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                           'cvterms', ...
		                   'sboTerm', ...
		                   'name', ...
		                   'id', ...
		                   'math', ...
		                 };
		nNumberFields = 9;
	end;
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                          'cvterms', ...
		                   'sboTerm', ...
		                   'name', ...
		                   'id', ...
		                   'math', ...
		                 };
		nNumberFields = 9;
        elseif (version == 2)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                          'cvterms', ...
		                   'sboTerm', ...
		                   'name', ...
		                   'id', ...
		                   'math', ...
		                 };
		nNumberFields = 9;
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getInitialAssignmentFieldnames(level, ...
                                                                          version)











if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = [];
		nNumberFields = 0;
	elseif (version == 2)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'sboTerm', ...
		                   'symbol', ...
		                   'math', ...
		                 };
		nNumberFields = 7;
	elseif (version == 3)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'sboTerm', ...
		                   'symbol', ...
		                   'math', ...
		                 };
		nNumberFields = 7;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                           'cvterms', ...
		                   'sboTerm', ...
		                   'symbol', ...
		                   'math', ...
		                 };
		nNumberFields = 8;
	end;
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                          'cvterms', ...
		                   'sboTerm', ...
		                   'symbol', ...
		                   'math', ...
		                 };
		nNumberFields = 8;
        elseif (version == 2)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                          'cvterms', ...
		                   'sboTerm', ...
		                   'symbol', ...
		                   'math', ...
                                   'id', ...
                                   'name', ...
		                 };
                  nNumberFields = 10;
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getKineticLawFieldnames(level, ...
                                                                   version)











if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

if (level == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'notes', ...
		                   'annotation', ...
		                   'formula', ...
		                   'parameter', ...
		                   'timeUnits', ...
		                   'substanceUnits', ...
		                 };
		nNumberFields = 7;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'formula', ...
		                   'math', ...
		                   'parameter', ...
		                   'timeUnits', ...
		                   'substanceUnits', ...
		                 };
		nNumberFields = 9;
	elseif (version == 2)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'formula', ...
		                   'math', ...
		                   'parameter', ...
		                   'sboTerm', ...
		                 };
		nNumberFields = 8;
	elseif (version == 3)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'sboTerm', ...
		                   'formula', ...
		                   'math', ...
		                   'parameter', ...
		                 };
		nNumberFields = 8;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                           'cvterms', ...
		                   'sboTerm', ...
		                   'formula', ...
		                   'math', ...
		                   'parameter', ...
		                 };
		nNumberFields = 9;
	end;
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                          'cvterms', ...
		                   'sboTerm', ...
		                   'math', ...
		                   'localParameter', ...
		                 };
		nNumberFields = 8;
        elseif (version == 2)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                          'cvterms', ...
		                   'sboTerm', ...
		                   'math', ...
		                   'localParameter', ...
                                   'id', ...
                                   'name', ...
		                 };
                  nNumberFields = 10;
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getLocalParameterFieldnames(level, ...
                                                                       version)











if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                          'cvterms', ...
		                   'sboTerm', ...
		                   'name', ...
		                   'id', ...
		                   'value', ...
		                   'units', ...
		                   'isSetValue', ...
		                 };
		nNumberFields = 11;
        elseif (version == 2)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                          'cvterms', ...
		                   'sboTerm', ...
		                   'name', ...
		                   'id', ...
		                   'value', ...
		                   'units', ...
		                   'isSetValue', ...
		                 };
		nNumberFields = 11;
  end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getModelFieldnames(level, ...
                                                              version)











if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

if (level == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'notes', ...
		                   'annotation', ...
		                   'SBML_level', ...
		                   'SBML_version', ...
		                   'name', ...
		                   'unitDefinition', ...
		                   'compartment', ...
		                   'species', ...
		                   'parameter', ...
		                   'rule', ...
		                   'reaction', ...
		                   'namespaces', ...
		                 };
		nNumberFields = 13;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'SBML_level', ...
		                   'SBML_version', ...
		                   'name', ...
		                   'id', ...
		                   'functionDefinition', ...
		                   'unitDefinition', ...
		                   'compartment', ...
		                   'species', ...
		                   'parameter', ...
		                   'rule', ...
		                   'reaction', ...
		                   'event', ...
		                   'time_symbol', ...
		                   'delay_symbol', ...
		                   'namespaces', ...
		                 };
		nNumberFields = 19;
	elseif (version == 2)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'SBML_level', ...
		                   'SBML_version', ...
		                   'name', ...
		                   'id', ...
		                   'sboTerm', ...
		                   'functionDefinition', ...
		                   'unitDefinition', ...
		                   'compartmentType', ...
		                   'speciesType', ...
		                   'compartment', ...
		                   'species', ...
		                   'parameter', ...
		                   'initialAssignment', ...
		                   'rule', ...
		                   'constraint', ...
		                   'reaction', ...
		                   'event', ...
		                   'time_symbol', ...
		                   'delay_symbol', ...
		                   'namespaces', ...
		                 };
		nNumberFields = 24;
	elseif (version == 3)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'SBML_level', ...
		                   'SBML_version', ...
		                   'name', ...
		                   'id', ...
		                   'sboTerm', ...
		                   'functionDefinition', ...
		                   'unitDefinition', ...
		                   'compartmentType', ...
		                   'speciesType', ...
		                   'compartment', ...
		                   'species', ...
		                   'parameter', ...
		                   'initialAssignment', ...
		                   'rule', ...
		                   'constraint', ...
		                   'reaction', ...
		                   'event', ...
		                   'time_symbol', ...
		                   'delay_symbol', ...
		                   'namespaces', ...
		                 };
		nNumberFields = 24;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                           'cvterms', ...
		                   'SBML_level', ...
		                   'SBML_version', ...
		                   'name', ...
		                   'id', ...
		                   'sboTerm', ...
		                   'functionDefinition', ...
		                   'unitDefinition', ...
		                   'compartmentType', ...
		                   'speciesType', ...
		                   'compartment', ...
		                   'species', ...
		                   'parameter', ...
		                   'initialAssignment', ...
		                   'rule', ...
		                   'constraint', ...
		                   'reaction', ...
		                   'event', ...
		                   'time_symbol', ...
		                   'delay_symbol', ...
		                   'namespaces', ...
		                 };
		nNumberFields = 25;
	end;
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                          'cvterms', ...
		                   'SBML_level', ...
		                   'SBML_version', ...
		                   'name', ...
		                   'id', ...
		                   'timeUnits', ...
		                   'substanceUnits', ...
		                   'volumeUnits', ...
		                   'areaUnits', ...
		                   'lengthUnits', ...
		                   'extentUnits', ...
		                   'conversionFactor', ...
		                   'sboTerm', ...
		                   'functionDefinition', ...
		                   'unitDefinition', ...
		                   'compartment', ...
		                   'species', ...
		                   'parameter', ...
		                   'initialAssignment', ...
		                   'rule', ...
		                   'constraint', ...
		                   'reaction', ...
		                   'event', ...
		                   'time_symbol', ...
		                   'delay_symbol', ...
		                   'avogadro_symbol', ...
		                   'namespaces', ...
		                 };
		nNumberFields = 31;
        elseif (version == 2)
		SBMLfieldnames = { 
                           'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                           'cvterms', ...
		                   'SBML_level', ...
		                   'SBML_version', ...
		                   'name', ...
		                   'id', ...
		                   'timeUnits', ...
		                   'substanceUnits', ...
		                   'volumeUnits', ...
		                   'areaUnits', ...
		                   'lengthUnits', ...
		                   'extentUnits', ...
		                   'conversionFactor', ...
		                   'sboTerm', ...
		                   'functionDefinition', ...
		                   'unitDefinition', ...
		                   'compartment', ...
		                   'species', ...
		                   'parameter', ...
		                   'initialAssignment', ...
		                   'rule', ...
		                   'constraint', ...
		                   'reaction', ...
		                   'event', ...
		                   'time_symbol', ...
		                   'delay_symbol', ...
		                   'avogadro_symbol', ...
		                   'rateOf_symbol', ...
		                   'namespaces', ...
		                 };
		nNumberFields = 32;
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getModifierSpeciesReferenceFieldnames(level, ...
                                                                                 version)











if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'species', ...
		                 };
		nNumberFields = 5;
	elseif (version == 2)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'species', ...
		                   'id', ...
		                   'name', ...
		                   'sboTerm', ...
		                 };
		nNumberFields = 8;
	elseif (version == 3)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'sboTerm', ...
		                   'species', ...
		                   'id', ...
		                   'name', ...
		                 };
		nNumberFields = 8;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                           'cvterms', ...
		                   'sboTerm', ...
		                   'species', ...
		                   'id', ...
		                   'name', ...
		                 };
		nNumberFields = 9;
	end;
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                          'cvterms', ...
		                   'sboTerm', ...
		                   'species', ...
		                   'id', ...
		                   'name', ...
		                 };
		nNumberFields = 9;
        elseif (version == 2)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                          'cvterms', ...
		                   'sboTerm', ...
		                   'species', ...
		                   'id', ...
		                   'name', ...
                                   'id', ...
                                   'name', ...
		                 };
                  nNumberFields = 11;
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getParameterFieldnames(level, ...
                                                                  version)











if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

if (level == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'notes', ...
		                   'annotation', ...
		                   'name', ...
		                   'value', ...
		                   'units', ...
		                   'isSetValue', ...
		                 };
		nNumberFields = 7;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'name', ...
		                   'id', ...
		                   'value', ...
		                   'units', ...
		                   'constant', ...
		                   'isSetValue', ...
		                 };
		nNumberFields = 10;
	elseif (version == 2)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'name', ...
		                   'id', ...
		                   'value', ...
		                   'units', ...
		                   'constant', ...
		                   'sboTerm', ...
		                   'isSetValue', ...
		                 };
		nNumberFields = 11;
	elseif (version == 3)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'sboTerm', ...
		                   'name', ...
		                   'id', ...
		                   'value', ...
		                   'units', ...
		                   'constant', ...
		                   'isSetValue', ...
		                 };
		nNumberFields = 11;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                           'cvterms', ...
		                   'sboTerm', ...
		                   'name', ...
		                   'id', ...
		                   'value', ...
		                   'units', ...
		                   'constant', ...
		                   'isSetValue', ...
		                 };
		nNumberFields = 12;
	end;
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                          'cvterms', ...
		                   'sboTerm', ...
		                   'name', ...
		                   'id', ...
		                   'value', ...
		                   'units', ...
		                   'constant', ...
		                   'isSetValue', ...
		                 };
		nNumberFields = 12;
        elseif (version == 2)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                          'cvterms', ...
		                   'sboTerm', ...
		                   'name', ...
		                   'id', ...
		                   'value', ...
		                   'units', ...
		                   'constant', ...
		                   'isSetValue', ...
		                 };
		nNumberFields = 12;
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getParameterRuleFieldnames(level, ...
                                                             version)











if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

if (level == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'notes', ...
		                   'annotation', ...
		                   'type', ...
		                   'formula', ...
		                   'variable', ...
		                   'species', ...
		                   'compartment', ...
		                   'name', ...
		                   'units', ...
		                 };
		nNumberFields = 10;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = [];
		nNumberFields = 0;
	elseif (version == 2)
		SBMLfieldnames = [];
		nNumberFields = 0;
	elseif (version == 3)
		SBMLfieldnames = [];
		nNumberFields = 0;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = [];
		nNumberFields = 0;
	end;
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = [];
		nNumberFields = 0;
        elseif (version == 2)
		SBMLfieldnames = [];
                  nNumberFields = 2;
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getPriorityFieldnames(level, ...
                                                                 version)











if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                          'cvterms', ...
		                   'sboTerm', ...
		                   'math', ...
		                 };
		nNumberFields = 7;
        elseif (version == 2)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                          'cvterms', ...
		                   'sboTerm', ...
		                   'math', ...
                                   'id', ...
                                   'name', ...
		                 };
                  nNumberFields = 9;
  end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getRateRuleFieldnames(level, ...
                                                             version)











if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

if (level == 1)
		SBMLfieldnames = [];
		nNumberFields = 0;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'formula', ...
		                   'variable', ...
		                   'species', ...
		                   'compartment', ...
		                   'name', ...
		                   'units', ...
		                 };
		nNumberFields = 10;
	elseif (version == 2)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'sboTerm', ...
		                   'formula', ...
		                   'variable', ...
		                   'species', ...
		                   'compartment', ...
		                   'name', ...
		                   'units', ...
		                 };
		nNumberFields = 11;
	elseif (version == 3)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'sboTerm', ...
		                   'formula', ...
		                   'variable', ...
		                   'species', ...
		                   'compartment', ...
		                   'name', ...
		                   'units', ...
		                 };
		nNumberFields = 11;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                           'cvterms', ...
		                   'sboTerm', ...
		                   'formula', ...
		                   'variable', ...
		                   'species', ...
		                   'compartment', ...
		                   'name', ...
		                   'units', ...
		                 };
		nNumberFields = 12;
	end;
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                          'cvterms', ...
		                   'sboTerm', ...
		                   'formula', ...
		                   'variable', ...
		                   'species', ...
		                   'compartment', ...
		                   'name', ...
		                   'units', ...
		                 };
		nNumberFields = 12;
        elseif (version == 2)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                          'cvterms', ...
		                   'sboTerm', ...
		                   'formula', ...
		                   'variable', ...
		                   'species', ...
		                   'compartment', ...
		                   'name', ...
		                   'units', ...
                                   'id', ...
                                   'name', ...
		                 };
                  nNumberFields = 14;
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getReactionFieldnames(level, ...
                                                                 version)











if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

if (level == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'notes', ...
		                   'annotation', ...
		                   'name', ...
		                   'reactant', ...
		                   'product', ...
		                   'kineticLaw', ...
		                   'reversible', ...
		                   'fast', ...
		                 };
		nNumberFields = 9;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'name', ...
		                   'id', ...
		                   'reactant', ...
		                   'product', ...
		                   'modifier', ...
		                   'kineticLaw', ...
		                   'reversible', ...
		                   'fast', ...
		                   'isSetFast', ...
		                 };
		nNumberFields = 13;
	elseif (version == 2)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'name', ...
		                   'id', ...
		                   'reactant', ...
		                   'product', ...
		                   'modifier', ...
		                   'kineticLaw', ...
		                   'reversible', ...
		                   'fast', ...
		                   'sboTerm', ...
		                   'isSetFast', ...
		                 };
		nNumberFields = 14;
	elseif (version == 3)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'sboTerm', ...
		                   'name', ...
		                   'id', ...
		                   'reactant', ...
		                   'product', ...
		                   'modifier', ...
		                   'kineticLaw', ...
		                   'reversible', ...
		                   'fast', ...
		                   'isSetFast', ...
		                 };
		nNumberFields = 14;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                           'cvterms', ...
		                   'sboTerm', ...
		                   'name', ...
		                   'id', ...
		                   'reactant', ...
		                   'product', ...
		                   'modifier', ...
		                   'kineticLaw', ...
		                   'reversible', ...
		                   'fast', ...
		                   'isSetFast', ...
		                 };
		nNumberFields = 15;
	end;
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                          'cvterms', ...
		                   'sboTerm', ...
		                   'name', ...
		                   'id', ...
		                   'reactant', ...
		                   'product', ...
		                   'modifier', ...
		                   'kineticLaw', ...
		                   'reversible', ...
		                   'fast', ...
		                   'isSetFast', ...
		                   'compartment', ...
		                 };
		nNumberFields = 16;
        elseif (version == 2)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                          'cvterms', ...
		                   'sboTerm', ...
		                   'name', ...
		                   'id', ...
		                   'reactant', ...
		                   'product', ...
		                   'modifier', ...
		                   'kineticLaw', ...
		                   'reversible', ...
		                   'compartment', ...
		                 };
		nNumberFields = 14;
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getSpeciesConcentrationRuleFieldnames(level, ...
                                                             version)











if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

if (level == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'notes', ...
		                   'annotation', ...
		                   'type', ...
		                   'formula', ...
		                   'variable', ...
		                   'species', ...
		                   'compartment', ...
		                   'name', ...
		                   'units', ...
		                 };
		nNumberFields = 10;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = [];
		nNumberFields = 0;
	elseif (version == 2)
		SBMLfieldnames = [];
		nNumberFields = 0;
	elseif (version == 3)
		SBMLfieldnames = [];
		nNumberFields = 0;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = [];
		nNumberFields = 0;
	end;
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = [];
		nNumberFields = 0;
        elseif (version == 2)
		SBMLfieldnames = [];
                  nNumberFields = 2;
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getSpeciesFieldnames(level, ...
                                                                version)











if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

if (level == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'notes', ...
		                   'annotation', ...
		                   'name', ...
		                   'compartment', ...
		                   'initialAmount', ...
		                   'units', ...
		                   'boundaryCondition', ...
		                   'charge', ...
		                   'isSetInitialAmount', ...
		                   'isSetCharge', ...
		                 };
		nNumberFields = 11;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'name', ...
		                   'id', ...
		                   'compartment', ...
		                   'initialAmount', ...
		                   'initialConcentration', ...
		                   'substanceUnits', ...
		                   'spatialSizeUnits', ...
		                   'hasOnlySubstanceUnits', ...
		                   'boundaryCondition', ...
		                   'charge', ...
		                   'constant', ...
		                   'isSetInitialAmount', ...
		                   'isSetInitialConcentration', ...
		                   'isSetCharge', ...
		                 };
		nNumberFields = 18;
	elseif (version == 2)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'name', ...
		                   'id', ...
		                   'speciesType', ...
		                   'compartment', ...
		                   'initialAmount', ...
		                   'initialConcentration', ...
		                   'substanceUnits', ...
		                   'spatialSizeUnits', ...
		                   'hasOnlySubstanceUnits', ...
		                   'boundaryCondition', ...
		                   'charge', ...
		                   'constant', ...
		                   'isSetInitialAmount', ...
		                   'isSetInitialConcentration', ...
		                   'isSetCharge', ...
		                 };
		nNumberFields = 19;
	elseif (version == 3)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'sboTerm', ...
		                   'name', ...
		                   'id', ...
		                   'speciesType', ...
		                   'compartment', ...
		                   'initialAmount', ...
		                   'initialConcentration', ...
		                   'substanceUnits', ...
		                   'hasOnlySubstanceUnits', ...
		                   'boundaryCondition', ...
		                   'charge', ...
		                   'constant', ...
		                   'isSetInitialAmount', ...
		                   'isSetInitialConcentration', ...
		                   'isSetCharge', ...
		                 };
		nNumberFields = 19;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                           'cvterms', ...
		                   'sboTerm', ...
		                   'name', ...
		                   'id', ...
		                   'speciesType', ...
		                   'compartment', ...
		                   'initialAmount', ...
		                   'initialConcentration', ...
		                   'substanceUnits', ...
		                   'hasOnlySubstanceUnits', ...
		                   'boundaryCondition', ...
		                   'charge', ...
		                   'constant', ...
		                   'isSetInitialAmount', ...
		                   'isSetInitialConcentration', ...
		                   'isSetCharge', ...
		                 };
		nNumberFields = 20;
	end;
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                          'cvterms', ...
		                   'sboTerm', ...
		                   'name', ...
		                   'id', ...
		                   'compartment', ...
		                   'initialAmount', ...
		                   'initialConcentration', ...
		                   'substanceUnits', ...
		                   'hasOnlySubstanceUnits', ...
		                   'boundaryCondition', ...
		                   'constant', ...
		                   'isSetInitialAmount', ...
		                   'isSetInitialConcentration', ...
		                   'conversionFactor', ...
		                 };
		nNumberFields = 18;
        elseif (version == 2)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                          'cvterms', ...
		                   'sboTerm', ...
		                   'name', ...
		                   'id', ...
		                   'compartment', ...
		                   'initialAmount', ...
		                   'initialConcentration', ...
		                   'substanceUnits', ...
		                   'hasOnlySubstanceUnits', ...
		                   'boundaryCondition', ...
		                   'constant', ...
		                   'isSetInitialAmount', ...
		                   'isSetInitialConcentration', ...
		                   'conversionFactor', ...
		                 };
		nNumberFields = 18;
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getSpeciesReferenceFieldnames(level, ...
                                                                         version)











if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

if (level == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'notes', ...
		                   'annotation', ...
		                   'species', ...
		                   'stoichiometry', ...
		                   'denominator', ...
		                 };
		nNumberFields = 6;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'species', ...
		                   'stoichiometry', ...
		                   'denominator', ...
		                   'stoichiometryMath', ...
		                 };
		nNumberFields = 8;
	elseif (version == 2)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'species', ...
		                   'id', ...
		                   'name', ...
		                   'sboTerm', ...
		                   'stoichiometry', ...
		                   'stoichiometryMath', ...
		                 };
		nNumberFields = 10;
	elseif (version == 3)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'sboTerm', ...
		                   'species', ...
		                   'id', ...
		                   'name', ...
		                   'stoichiometry', ...
		                   'stoichiometryMath', ...
		                 };
		nNumberFields = 10;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                           'cvterms', ...
		                   'sboTerm', ...
		                   'species', ...
		                   'id', ...
		                   'name', ...
		                   'stoichiometry', ...
		                   'stoichiometryMath', ...
		                 };
		nNumberFields = 11;
	end;
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                          'cvterms', ...
		                   'sboTerm', ...
		                   'species', ...
		                   'id', ...
		                   'name', ...
		                   'stoichiometry', ...
		                   'constant', ...
		                   'isSetStoichiometry', ...
		                 };
		nNumberFields = 12;
        elseif (version == 2)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                          'cvterms', ...
                          'sboTerm', ...
		                   'species', ...
		                   'id', ...
		                   'name', ...
		                   'stoichiometry', ...
		                   'constant', ...
		                   'isSetStoichiometry', ...
		                 };
		nNumberFields = 12;
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getSpeciesTypeFieldnames(level, ...
                                                                    version)











if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = [];
		nNumberFields = 0;
	elseif (version == 2)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'name', ...
		                   'id', ...
		                 };
		nNumberFields = 6;
	elseif (version == 3)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'sboTerm', ...
		                   'name', ...
		                   'id', ...
		                 };
		nNumberFields = 7;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                           'cvterms', ...
		                   'sboTerm', ...
		                   'name', ...
		                   'id', ...
		                 };
		nNumberFields = 8;
	end;
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = [];
		nNumberFields = 0;
        elseif (version == 2)
		SBMLfieldnames = [];
                  nNumberFields = 2;
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getStoichiometryMathFieldnames(level, ...
                                                                          version)











if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = [];
		nNumberFields = 0;
	elseif (version == 2)
		SBMLfieldnames = [];
		nNumberFields = 0;
	elseif (version == 3)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'sboTerm', ...
		                   'math', ...
		                 };
		nNumberFields = 6;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                           'cvterms', ...
		                   'sboTerm', ...
		                   'math', ...
		                 };
		nNumberFields = 7;
	end;
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = [];
		nNumberFields = 0;
        elseif (version == 2)
		SBMLfieldnames = [];
                  nNumberFields = 2;
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getTriggerFieldnames(level, ...
                                                                version)











if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = [];
		nNumberFields = 0;
	elseif (version == 2)
		SBMLfieldnames = [];
		nNumberFields = 0;
	elseif (version == 3)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'sboTerm', ...
		                   'math', ...
		                 };
		nNumberFields = 6;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                           'cvterms', ...
		                   'sboTerm', ...
		                   'math', ...
		                 };
		nNumberFields = 7;
	end;
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                          'cvterms', ...
		                   'sboTerm', ...
		                   'persistent', ...
		                   'initialValue', ...
		                   'math', ...
		                 };
		nNumberFields = 9;
        elseif (version == 2)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                          'cvterms', ...
		                   'sboTerm', ...
		                   'persistent', ...
		                   'initialValue', ...
		                   'math', ...
                                   'id', ...
                                   'name', ...
		                 };
                  nNumberFields = 11;
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getUnitDefinitionFieldnames(level, ...
                                                                       version)











if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

if (level == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'notes', ...
		                   'annotation', ...
		                   'name', ...
		                   'unit', ...
		                 };
		nNumberFields = 5;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'name', ...
		                   'id', ...
		                   'unit', ...
		                 };
		nNumberFields = 7;
	elseif (version == 2)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'name', ...
		                   'id', ...
		                   'unit', ...
		                 };
		nNumberFields = 7;
	elseif (version == 3)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'sboTerm', ...
		                   'name', ...
		                   'id', ...
		                   'unit', ...
		                 };
		nNumberFields = 8;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                           'cvterms', ...
		                   'sboTerm', ...
		                   'name', ...
		                   'id', ...
		                   'unit', ...
		                 };
		nNumberFields = 9;
	end;
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                          'cvterms', ...
		                   'sboTerm', ...
		                   'name', ...
		                   'id', ...
		                   'unit', ...
		                 };
		nNumberFields = 9;
        elseif (version == 2)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                          'cvterms', ...
		                   'sboTerm', ...
		                   'name', ...
		                   'id', ...
		                   'unit', ...
		                 };
		nNumberFields = 9;
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getUnitFieldnames(level, ...
                                                             version)











if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

if (level == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'notes', ...
		                   'annotation', ...
		                   'kind', ...
		                   'exponent', ...
		                   'scale', ...
		                 };
		nNumberFields = 6;
elseif (level == 2)
	if (version == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'kind', ...
		                   'exponent', ...
		                   'scale', ...
		                   'multiplier', ...
		                   'offset', ...
		                 };
		nNumberFields = 9;
	elseif (version == 2)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'kind', ...
		                   'exponent', ...
		                   'scale', ...
		                   'multiplier', ...
		                 };
		nNumberFields = 8;
	elseif (version == 3)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
		                   'sboTerm', ...
		                   'kind', ...
		                   'exponent', ...
		                   'scale', ...
		                   'multiplier', ...
		                 };
		nNumberFields = 9;
    elseif (version == 4 || version == 5)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                           'cvterms', ...
		                   'sboTerm', ...
		                   'kind', ...
		                   'exponent', ...
		                   'scale', ...
		                   'multiplier', ...
		                 };
		nNumberFields = 10;
	end;
elseif (level == 3)
	if (version == 1)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                          'cvterms', ...
		                   'sboTerm', ...
		                   'kind', ...
		                   'exponent', ...
		                   'scale', ...
		                   'multiplier', ...
		                 };
		nNumberFields = 10;
        elseif (version == 2)
		SBMLfieldnames = { 'typecode', ...
		                   'metaid', ...
		                   'notes', ...
		                   'annotation', ...
                          'cvterms', ...
		                   'sboTerm', ...
		                   'kind', ...
		                   'exponent', ...
		                   'scale', ...
		                   'multiplier', ...
                                   'id', ...
                                   'name', ...
		                 };
                  nNumberFields = 12;
	end;
end;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getFBCModelFieldnames(level, ...
                                                         version, pkgVersion)
if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

% need a check on package version

if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 3)
	if (version < 3)
    if (pkgVersion == 1)
          SBMLfieldnames = { 'fbc_version', ...
                             'fbc_fluxBound', ...
                             'fbc_objective', ...
                             'fbc_activeObjective', ...
                           };
          nNumberFields = 4;
    elseif(pkgVersion ==2)
          SBMLfieldnames = { 'fbc_version', ...
                             'fbc_geneProduct', ...
                             'fbc_objective', ...
                             'fbc_activeObjective', ...
                             'fbc_strict'
                           };
          nNumberFields = 5;
    end;
	end;
end;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getFBCSpeciesFieldnames(level, ...
                                                         version, pkgVersion)

if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

% need a check on package version

if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 3)
	if (version < 3)
    if (pkgVersion < 3)
      SBMLfieldnames = { 'fbc_charge', ...
                         'fbc_chemicalFormula', ...
                         'isSetfbc_charge', ...
                         'level', ...
                         'version', ...
                         'fbc_version', ...
                       };
      nNumberFields = 6;
    end;
	end;
end;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getFluxBoundFieldnames(level, ...
                                                         version, pkgVersion)

if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

% need a check on package version

if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 3)
	if (version <3)
    if (pkgVersion == 1)
      SBMLfieldnames = { 'typecode', ...
                         'metaid', ...
                         'notes', ...
                         'annotation', ...
                           'cvterms', ...
                        'sboTerm', ...
                         'fbc_id', ...
                         'fbc_reaction', ...
                         'fbc_operation', ...
                         'fbc_value', ...
                         'isSetfbc_value', ...
                         'level', ...
                         'version', ...
                         'fbc_version', ...
                       };
      nNumberFields = 14;
    else
        SBMLfieldnames = [];
        nNumberFields = 0;
    end;
	end;
end;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getFluxObjectiveFieldnames(level, ...
                                                         version, pkgVersion)

if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

% need a check on package version

if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 3)
	if (version < 3)
    if (pkgVersion < 3)
      SBMLfieldnames = { 'typecode', ...
                         'metaid', ...
                         'notes', ...
                         'annotation', ...
                           'cvterms', ...
                        'sboTerm', ...
                         'fbc_reaction', ...
                         'fbc_coefficient', ...
                         'isSetfbc_coefficient', ...
                         'level', ...
                         'version', ...
                         'fbc_version', ...
                       };
      nNumberFields = 12;
    end;
	end;
end;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getObjectiveFieldnames(level, ...
                                                         version, pkgVersion)

if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

% need a check on package version

if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 3)
	if (version < 3)
    if (pkgVersion < 3)
      SBMLfieldnames = { 'typecode', ...
                         'metaid', ...
                         'notes', ...
                         'annotation', ...
                          'cvterms', ...
                         'sboTerm', ...
                         'fbc_id', ...
                         'fbc_type', ...
                         'fbc_fluxObjective', ...
                         'level', ...
                         'version', ...
                         'fbc_version', ...
                       };
      nNumberFields = 12;
    end;
	end;
end;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getGeneProductFieldnames(level, ...
                                                         version, pkgVersion)

if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

% need a check on package version

if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 3)
	if (version < 3)
    if (pkgVersion  == 2)
      SBMLfieldnames = { 'typecode', ...
                         'metaid', ...
                         'notes', ...
                         'annotation', ...
                           'cvterms', ...
                        'sboTerm', ...
                         'fbc_id', ...
                         'fbc_name', ...
                         'fbc_label', ...
                         'fbc_associatedSpecies', ...
                         'level', ...
                         'version', ...
                         'fbc_version', ...
                       };
      nNumberFields = 13;
    else
        SBMLfieldnames = [];
        nNumberFields = 0;
    end;
	end;
end;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getFBCReactionFieldnames(level, ...
                                                         version, pkgVersion)

if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

% need a check on package version

if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 3)
	if (version < 3)
    if (pkgVersion == 2)
      SBMLfieldnames = { 'fbc_lowerFluxBound', ...
                         'fbc_upperFluxBound', ...
                         'fbc_geneProductAssociation', ...
                         'level', ...
                         'version', ...
                         'fbc_version', ...
                       };
      nNumberFields = 6;
    else
       SBMLfieldnames = [];
	   nNumberFields = 0;
    end;
	end;
end;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getGeneProductAssociationFieldnames(level, ...
                                                         version, pkgVersion)

if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

% need a check on package version

if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 3)
	if (version < 3)
    if (pkgVersion == 2)

      SBMLfieldnames = { 'typecode', ...
                         'metaid', ...
                         'notes', ...
                         'annotation', ...
                          'cvterms', ...
                         'sboTerm', ...
                         'fbc_id', ...
                         'fbc_name', ...
                         'fbc_association', ...
                         'level', ...
                         'version', ...
                         'fbc_version', ...
                       };
                   nNumberFields = 12;
    else
       SBMLfieldnames = [];
	   nNumberFields = 0;
    end;
	end;
end;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [SBMLfieldnames, nNumberFields] = getAssociationFieldnames(level, ...
                                                         version, pkgVersion)

if (~isValidLevelVersionCombination(level, version))
  error ('invalid level/version combination');
end;

% need a check on package version

if (level == 1)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 2)
	SBMLfieldnames = [];
	nNumberFields = 0;
elseif (level == 3)
	if (version < 3)
    if (pkgVersion == 2)

      SBMLfieldnames = { 'typecode', ...
                         'metaid', ...
                         'notes', ...
                         'annotation', ...
                          'cvterms', ...
                         'sboTerm', ...
                         'fbc_association', ...
                         'level', ...
                         'version', ...
                         'fbc_version', ...
                       };
                   nNumberFields = 10;
    else
       SBMLfieldnames = [];
	   nNumberFields = 0;
    end;
	end;
end;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getQUALModelFieldnames(level, ...
version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end;

  SBMLfieldnames = [];
  nNumberFields = 0;

  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
        SBMLfieldnames = {
        'qual_version', ...
        'qual_qualitativeSpecies', ...
        'qual_transition', ...
        };
        nNumberFields = 3;
      end;
    end;
  end;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getQualitativeSpeciesFieldnames(level, ...
version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end;

  SBMLfieldnames = [];
  nNumberFields = 0;

  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
        SBMLfieldnames = {
        'typecode', ...
        'metaid', ...
        'notes', ...
        'annotation', ...
        'cvterms', ...
        'sboTerm', ...
        'qual_id', ...
        'qual_name', ...
        'qual_compartment', ...
        'qual_constant', ...
        'qual_initialLevel', ...
        'qual_maxLevel', ...
        'level', ...
        'version', ...
        'qual_version', ...
        };
        nNumberFields = 15;
      end;
    end;
  end;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getTransitionFieldnames(level, ...
version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end;

  SBMLfieldnames = [];
  nNumberFields = 0;

  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
        SBMLfieldnames = {
        'typecode', ...
        'metaid', ...
        'notes', ...
        'annotation', ...
        'cvterms', ...
        'sboTerm', ...
        'qual_id', ...
        'qual_name', ...
        'qual_input', ...
        'qual_output', ...
        'qual_functionTerm', ...
        'qual_defaultTerm', ...
        'level', ...
        'version', ...
        'qual_version', ...
        };
        nNumberFields = 14;
      end;
    end;
  end;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getInputFieldnames(level, ...
version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end;

  SBMLfieldnames = [];
  nNumberFields = 0;

  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
        SBMLfieldnames = {
        'typecode', ...
        'metaid', ...
        'notes', ...
        'annotation', ...
        'cvterms', ...
        'sboTerm', ...
        'qual_id', ...
        'qual_name', ...
        'qual_sign', ...
        'qual_qualitativeSpecies', ...
        'qual_transitionEffect', ...
        'qual_thresholdLevel', ...
        'level', ...
        'version', ...
        'qual_version', ...
        };
        nNumberFields = 15;
      end;
    end;
  end;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getOutputFieldnames(level, ...
version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end;

  SBMLfieldnames = [];
  nNumberFields = 0;

  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
        SBMLfieldnames = {
        'typecode', ...
        'metaid', ...
        'notes', ...
        'annotation', ...
        'cvterms', ...
        'sboTerm', ...
        'qual_id', ...
        'qual_qualitativeSpecies', ...
        'qual_transitionEffect', ...
        'qual_name', ...
        'qual_outputLevel', ...
        'level', ...
        'version', ...
        'qual_version', ...
        };
        nNumberFields = 14;
      end;
    end;
  end;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getDefaultTermFieldnames(level, ...
version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end;

  SBMLfieldnames = [];
  nNumberFields = 0;

  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
        SBMLfieldnames = {
        'typecode', ...
        'metaid', ...
        'notes', ...
        'annotation', ...
        'cvterms', ...
        'sboTerm', ...
        'qual_resultLevel', ...
        'level', ...
        'version', ...
        'qual_version', ...
        };
        nNumberFields = 10;
      end;
    end;
  end;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getFunctionTermFieldnames(level, ...
version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end;

  SBMLfieldnames = [];
  nNumberFields = 0;

  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
        SBMLfieldnames = {
        'typecode', ...
        'metaid', ...
        'notes', ...
        'annotation', ...
        'cvterms', ...
        'sboTerm', ...
        'qual_resultLevel', ...
        'math', ...
        'level', ...
        'version', ...
        'qual_version', ...
        };
        nNumberFields = 11;
      end;
    end;
  end;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getGROUPSModelFieldnames(level, ...
version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end;

  SBMLfieldnames = [];
  nNumberFields = 0;

  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
        SBMLfieldnames = {
        'groups_version', ...
        'groups_group', ...
        };
        nNumberFields = 2;
      end;
    end;
  end;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getGroupFieldnames(level, ...
version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end;

  SBMLfieldnames = [];
  nNumberFields = 0;

  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
        SBMLfieldnames = {
        'typecode', ...
        'metaid', ...
        'notes', ...
        'annotation', ...
        'cvterms', ...
        'sboTerm', ...
        'groups_id', ...
        'groups_name', ...
        'groups_kind', ...
        'groups_member', ...
        'level', ...
        'version', ...
        'groups_version', ...
        };
        nNumberFields = 13;
      end;
    end;
  end;


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [SBMLfieldnames, nNumberFields] = getMemberFieldnames(level, ...
version, pkgVersion)
  if (~isValidLevelVersionCombination(level, version))
    error ('invalid level/version combination');
  end;

  SBMLfieldnames = [];
  nNumberFields = 0;

  if (level == 3)
    if (version < 3)
      if (pkgVersion == 1)
        SBMLfieldnames = {
        'typecode', ...
        'metaid', ...
        'notes', ...
        'annotation', ...
        'cvterms', ...
        'sboTerm', ...
        'groups_id', ...
        'groups_name', ...
        'groups_idRef', ...
        'groups_metaIdRef', ...
        'level', ...
        'version', ...
        'groups_version', ...
        };
        nNumberFields = 13;
      end;
    end;
  end;

%%%% ADD functions
%%%% ADD functions

%%%% END functions



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function valid = isValidLevelVersionCombination(level, version)
valid = 1;

if ~isIntegralNumber(level)
	error('level must be an integer');
elseif ~isIntegralNumber(version)
	error('version must be an integer');
end;

if (level < 1 || level > 3)
	error('current SBML levels are 1, 2 or 3');
end;

if (level == 1)
	if (version < 1 || version > 2)
		error('SBMLToolbox supports versions 1-2 of SBML Level 1');
	end;

elseif (level == 2)
	if (version < 1 || version > 5)
		error('SBMLToolbox supports versions 1-5 of SBML Level 2');
	end;

elseif (level == 3)
	if (version < 1 || version > 2)
		error('SBMLToolbox supports version 1-2 of SBML Level 3');
	end;

end;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function value = isIntegralNumber(number)
value = 0;

integerClasses = {'int8', 'uint8', 'int16', 'uint16', 'int32', 'uint32', 'int64', 'uint64'};

% since the function isinteger does not exist in MATLAB Rel 13
% this is not used
%if (isinteger(number))
if (ismember(class(number), integerClasses))
    value = 1;
elseif (isnumeric(number))
    % if it is an integer 
    if (number == fix(number))
        value = 1;
    end;
end;



