/* -*- pftp-c -*- */
#ifdef HAVE_CONFIG_H
# include "config.h"
#endif
#if HAVE_STDLIB_H
# include <stdlib.h>
#endif
#if HAVE_INTTYPES_H
# include <inttypes.h>
#endif
#if HAVE_STRING_H
# include <string.h>
#endif
#if HAVE_NETINET_IN_H
# include <netinet/in.h>
#endif

#if HAVE_NTOHLL
#else
# ifdef WORDS_BIGENDIAN
#  define ntohll(num) (num)
# else
#  define ntohll(num) ((((num >> 56) & 0xff) <<  0) |  \
                       (((num >> 48) & 0xff) <<  8) |  \
                       (((num >> 40) & 0xff) << 16) |  \
                       (((num >> 32) & 0xff) << 24) |  \
                       (((num >> 24) & 0xff) << 32) |  \
                       (((num >> 16) & 0xff) << 40) |  \
                       (((num >>  8) & 0xff) << 48) |  \
                       (((num >>  0) & 0xff) << 56))
# endif
#endif
#if HAVE_HTONLL
#else
# define htonll ntohll
#endif

#ifndef NO_SSL

#include <openssl/bn.h>

#include "pftp_ssh_buf.h"

#ifdef DEBUG
# include <assert.h>
#else
# define assert(x) // x
#endif

#ifdef WITH_DMALLOC
# include <dmalloc.h>
#endif

#ifndef BN_is_negative
# define BN_is_negative(__bn) ((__bn)->neg)
#endif

struct pftp_ssh_buf_s
{
    char *data;
    size_t pos, size;
    int readonly;
};

pftp_ssh_buf_t pftp_ssh_create_buf(void)
{
    pftp_ssh_buf_t ret;
    ret = malloc(sizeof(struct pftp_ssh_buf_s));
    memset(ret, 0, sizeof(struct pftp_ssh_buf_s));

    return ret;
}

pftp_ssh_buf_t pftp_ssh_attach_buf(const char *data, size_t size)
{
    pftp_ssh_buf_t ret;
    ret = malloc(sizeof(struct pftp_ssh_buf_s));
    memset(ret, 0, sizeof(struct pftp_ssh_buf_s));
    memcpy(&ret->data, &data, sizeof(char *));
    ret->size = size;
    ret->readonly = 1;

    return ret;
}

void pftp_ssh_buf_free(pftp_ssh_buf_t buf)
{
    if (buf) {
	if (buf->readonly) {
	    free(buf);
	} else {
	    if (buf->data)
		free(buf->data);
	    free(buf);
	}
    }
}

/* Writing */

void pftp_ssh_buf_put_string(pftp_ssh_buf_t buf, const char *str)
{
    assert(!buf->readonly);
    pftp_ssh_buf_put_string2(buf, str, strlen(str));
}

void pftp_ssh_buf_put_string2(pftp_ssh_buf_t buf, const char *str, size_t len)
{
    uint32_t tmp;
    assert(!buf->readonly);
    buf->data = realloc(buf->data, buf->size + 4 + len);
    tmp = htonl((u_long) len);
    memcpy(buf->data + buf->size, &tmp, 4);
    buf->size += 4;
    memcpy(buf->data + buf->size, str, len);
    buf->size += len;
}

void pftp_ssh_buf_put_bignum(pftp_ssh_buf_t buf, const BIGNUM *num)
{
    uint32_t size = 0;
    unsigned char *numbuf = NULL;
    unsigned char *start = NULL;	

    assert(!buf->readonly);

    if (BN_is_negative(num)) {
	size = BN_num_bytes(num) + 1;
	numbuf = malloc(size);
	numbuf[0] = 0xFF;
	BN_bn2bin(num, numbuf + 1);
	if ((numbuf[1] & 0x80)) {
	    start = numbuf + 1;
	    size--;
	} else {
	    start = numbuf;
	}
    } else {
	size = BN_num_bytes(num) + 1;
	numbuf = malloc(size);
	numbuf[0] = 0x00;
	BN_bn2bin(num, numbuf + 1);
	if ((numbuf[1] & 0x80)) {
	    start = numbuf;
	} else {
	    start = numbuf + 1;
	    size--;
	}
    }

    buf->data = realloc(buf->data, buf->size + 4 + size);
    size = htonl(size);
    memcpy(buf->data + buf->size, &size, 4);
    buf->size += 4;
    size = ntohl(size);
    memcpy(buf->data + buf->size, start, size);
    buf->size += size;

    free(numbuf);
}

void pftp_ssh_buf_put_raw(pftp_ssh_buf_t buf, const void *data, size_t size)
{
    assert(!buf->readonly);
    buf->data = realloc(buf->data, buf->size + size);
    memcpy(buf->data + buf->size, data, size);
    buf->size += size;
}

void pftp_ssh_buf_put_uint16(pftp_ssh_buf_t buf, uint16_t num)
{
    uint16_t tmp;
    assert(!buf->readonly);
    buf->data = realloc(buf->data, buf->size + 2);
    tmp = htons(num);
    memcpy(buf->data + buf->size, &tmp, 2);
    buf->size += 2;
}

void pftp_ssh_buf_put_uint32(pftp_ssh_buf_t buf, uint32_t num)
{
    uint32_t tmp;
    assert(!buf->readonly);
    buf->data = realloc(buf->data, buf->size + 4);
    tmp = htonl(num);
    memcpy(buf->data + buf->size, &tmp, 4);
    buf->size += 4;
}

void pftp_ssh_buf_put_uint64(pftp_ssh_buf_t buf, uint64_t num)
{
    uint64_t tmp;
    assert(!buf->readonly);
    buf->data = realloc(buf->data, buf->size + 8);
    tmp = htonll(num);
    memcpy(buf->data + buf->size, &tmp, 8);
    buf->size += 8;
}

void pftp_ssh_buf_put_int64(pftp_ssh_buf_t buf, int64_t num)
{
    uint64_t tmp;
    assert(!buf->readonly);
    buf->data = realloc(buf->data, buf->size + 8);
    if (num < 0) {
	tmp = htonll((uint64_t)-num) | 0x8000000000000000LL;
    } else {
	tmp = htonll((uint64_t)num);
    }
    memcpy(buf->data + buf->size, &tmp, 8);
    buf->size += 8;
}

void pftp_ssh_buf_put_char(pftp_ssh_buf_t buf, char c)
{
    assert(!buf->readonly);
    buf->data = realloc(buf->data, buf->size + 1);
    buf->data[buf->size++] = c;
}

/* Reading */

char *pftp_ssh_buf_get_string(pftp_ssh_buf_t buf)
{
    size_t len;
    return pftp_ssh_buf_get_string2(buf, &len);
}

char *pftp_ssh_buf_get_string2(pftp_ssh_buf_t buf, size_t *len)
{
    if (buf->pos + 4 <= buf->size) {
	uint32_t slen;
	memcpy(&slen, buf->data + buf->pos, 4);
	buf->pos += 4;
	slen = ntohl(slen);
	*len = slen;
	if (buf->pos + slen <= buf->size) {
	    char *ret;
	    ret = malloc(slen + 1);
	    if (ret) {
		memcpy(ret, buf->data + buf->pos, slen);
		ret[slen] = '\0';
		buf->pos += slen;
		return ret;
	    }
	}
			
	buf->pos -= 4;
    }

    return NULL;
}

BIGNUM *pftp_ssh_buf_get_bignum(pftp_ssh_buf_t buf)
{
    if (buf->pos + 4 <= buf->size) {
	uint32_t size;
	memcpy(&size, buf->data + buf->pos, 4);
	buf->pos += 4;
	size = ntohl(size);
	if (buf->pos + size <= buf->size) {
	    BIGNUM *ret;
	    ret = BN_bin2bn((unsigned char *) buf->data + buf->pos, size, NULL);
	    if (ret) {
		buf->pos += size;
		return ret;
	    }
	}

	buf->pos -= 4;
    }

    return NULL;
}

const char *pftp_ssh_buf_get_raw(pftp_ssh_buf_t buf, size_t len)
{
    if (buf->pos + len <= buf->size) {
	const char *ret;
	ret = buf->data + buf->pos;
	buf->pos += len;
	return ret;
    }

    return NULL;
}

char pftp_ssh_buf_get_char(pftp_ssh_buf_t buf, int *valid)
{
    if (buf->pos + 1 <= buf->size) {
	char ret;
	ret = buf->data[buf->pos++];
	*valid = 1;
	return ret;
    }

    *valid = 0;

    return '\0';
}

uint16_t pftp_ssh_buf_get_uint16(pftp_ssh_buf_t buf, int *valid)
{
    if (buf->pos + 2 <= buf->size) {
	uint16_t ret;
	memcpy(&ret, buf->data + buf->pos, 2);
	*valid = 1;
	buf->pos += 2;
	ret = ntohs(ret);
	return ret;
    }

    *valid = 0;
    return 0;
}

uint32_t pftp_ssh_buf_get_uint32(pftp_ssh_buf_t buf, int *valid)
{
    if (buf->pos + 4 <= buf->size) {
	uint32_t ret;
	memcpy(&ret, buf->data + buf->pos, 4);
	*valid = 1;
	buf->pos += 4;
	ret = ntohl(ret);
	return ret;
    }

    *valid = 0;
    return 0;
}

uint64_t pftp_ssh_buf_get_uint64(pftp_ssh_buf_t buf, int *valid)
{
    if (buf->pos + 8 <= buf->size) {
	uint64_t ret;
	memcpy(&ret, buf->data + buf->pos, 8);
	*valid = 1;
	buf->pos += 8;
	ret = ntohll(ret);
	return ret;
    }

    *valid = 0;
    return 0;
}

int64_t pftp_ssh_buf_get_int64(pftp_ssh_buf_t buf, int *valid)
{
    if (buf->pos + 8 <= buf->size) {
	uint64_t tmp;
	int64_t ret;
	memcpy(&tmp, buf->data + buf->pos, 8);
	*valid = 1;
	buf->pos += 8;
	if (tmp & 0x8000000000000000LL) {
	    ret = ntohll(tmp & 0x7FFFFFFFFFFFFFFFLL);
	    ret = -ret;
	} else {
	    ret = ntohll(tmp);
	}
	return ret;
    }

    *valid = 0;
    return 0;
}

size_t pftp_ssh_buf_size(pftp_ssh_buf_t buf)
{
    return buf->size;
}

void pftp_ssh_buf_reset(pftp_ssh_buf_t buf)
{
    buf->pos = 0;
}

const char *pftp_ssh_buf_raw(pftp_ssh_buf_t buf)
{
    return buf->data;
}

size_t pftp_ssh_buf_data_left(pftp_ssh_buf_t buf)
{
    return (buf->size - buf->pos);
}

void pftp_ssh_buf_seek(pftp_ssh_buf_t buf, size_t seek)
{
    buf->pos = seek;
}

size_t pftp_ssh_buf_tell(pftp_ssh_buf_t buf)
{
    return buf->pos;
}

#endif /* NO_SSL */
