/* -*- pftp-c -*- */
#ifdef HAVE_CONFIG_H
# include "config.h"
#endif
#if HAVE_STDLIB_H
# include <stdlib.h>
#endif
#if HAVE_STDIO_H
# include <stdio.h>
#endif
#if HAVE_INTTYPES_H
# include <inttypes.h>
#endif
#if HAVE_TIME_H
# include <time.h>
#endif
#if HAVE_STRING_H
# include <string.h>
#endif
#if HAVE_SYS_STAT_H
# include <sys/stat.h>
#endif
#if HAVE_ERRNO_H
# include <errno.h>
#endif

#ifndef NO_SSL

#include <openssl/bn.h>

#include "pftp_directory.h"
#include "pftp_ssh_buf.h"
#include "pftp_sftp_attrib.h"

#if WIN32
#define S_ISDIR(mode) ((mode) & _S_IFDIR)
#define S_ISREG(mode) ((mode) & _S_IFREG)
#define S_ISCHR(mode) ((mode) & _S_IFCHR)
#define S_ISBLK(mode) (0)
#define S_ISFIFO(mode) (0)
#define S_ISLNK(mode) (0)
#define S_ISSOCK(mode) (0)

#endif

/* valid_attribute_flags */
#define SSH_FILEXFER_ATTR_SIZE              0x00000001
#define SSH_FILEXFER_ATTR_UIDGID            0x00000002    // v3
#define SSH_FILEXFER_ATTR_PERMISSIONS       0x00000004
#define SSH_FILEXFER_ATTR_ACCESSTIME        0x00000008
#define SSH_FILEXFER_ATTR_ACMODTIME         0x00000008    // v3
#define SSH_FILEXFER_ATTR_CREATETIME        0x00000010
#define SSH_FILEXFER_ATTR_MODIFYTIME        0x00000020
#define SSH_FILEXFER_ATTR_ACL               0x00000040
#define SSH_FILEXFER_ATTR_OWNERGROUP        0x00000080
#define SSH_FILEXFER_ATTR_SUBSECOND_TIMES   0x00000100
#define SSH_FILEXFER_ATTR_BITS              0x00000200
#define SSH_FILEXFER_ATTR_ALLOCATION_SIZE   0x00000400
#define SSH_FILEXFER_ATTR_TEXT_HINT         0x00000800
#define SSH_FILEXFER_ATTR_MIME_TYPE         0x00001000
#define SSH_FILEXFER_ATTR_LINK_COUNT        0x00002000
#define SSH_FILEXFER_ATTR_UNTRANSLATED_NAME 0x00004000
#define SSH_FILEXFER_ATTR_CTIME             0x00008000
#define SSH_FILEXFER_ATTR_EXTENDED          0x80000000

/* type */
#define SSH_FILEXFER_TYPE_REGULAR          1
#define SSH_FILEXFER_TYPE_DIRECTORY        2
#define SSH_FILEXFER_TYPE_SYMLINK          3
#define SSH_FILEXFER_TYPE_SPECIAL          4
#define SSH_FILEXFER_TYPE_UNKNOWN          5
#define SSH_FILEXFER_TYPE_SOCKET           6
#define SSH_FILEXFER_TYPE_CHAR_DEVICE      7
#define SSH_FILEXFER_TYPE_BLOCK_DEVICE     8
#define SSH_FILEXFER_TYPE_FIFO             9

/* permissions */
#define SSH_FILEXFER_PERM_IRUSR  0000400 
#define SSH_FILEXFER_PERM_IWUSR  0000200
#define SSH_FILEXFER_PERM_IXUSR  0000100
#define SSH_FILEXFER_PERM_IRGRP  0000040
#define SSH_FILEXFER_PERM_IWGRP  0000020
#define SSH_FILEXFER_PERM_IXGRP  0000010
#define SSH_FILEXFER_PERM_IROTH  0000004
#define SSH_FILEXFER_PERM_IWOTH  0000002
#define SSH_FILEXFER_PERM_IXOTH  0000001
#define SSH_FILEXFER_PERM_ISUID  0004000
#define SSH_FILEXFER_PERM_ISGID  0002000
#define SSH_FILEXFER_PERM_ISVTX  0001000

/* attrib_bits & attrib_bits_valid */
#define SSH_FILEXFER_ATTR_FLAGS_READONLY         0x00000001
#define SSH_FILEXFER_ATTR_FLAGS_SYSTEM           0x00000002
#define SSH_FILEXFER_ATTR_FLAGS_HIDDEN           0x00000004
#define SSH_FILEXFER_ATTR_FLAGS_CASE_INSENSITIVE 0x00000008
#define SSH_FILEXFER_ATTR_FLAGS_ARCHIVE          0x00000010
#define SSH_FILEXFER_ATTR_FLAGS_ENCRYPTED        0x00000020
#define SSH_FILEXFER_ATTR_FLAGS_COMPRESSED       0x00000040
#define SSH_FILEXFER_ATTR_FLAGS_SPARSE           0x00000080
#define SSH_FILEXFER_ATTR_FLAGS_APPEND_ONLY      0x00000100
#define SSH_FILEXFER_ATTR_FLAGS_IMMUTABLE        0x00000200
#define SSH_FILEXFER_ATTR_FLAGS_SYNC             0x00000400
#define SSH_FILEXFER_ATTR_FLAGS_TRANSLATION_ERR  0x00000800

/* text_hint */
#define SSH_FILEXFER_ATTR_KNOWN_TEXT        0x00
#define SSH_FILEXFER_ATTR_GUESSED_TEXT      0x01
#define SSH_FILEXFER_ATTR_KNOWN_BINARY      0x02
#define SSH_FILEXFER_ATTR_GUESSED_BINARY    0x03

struct pftp_sftp_attr_s {
    uint32_t flags;
    uint8_t type;
    uint64_t size;
    uint64_t allocation_size;
    char *owner;
    char *group;
    uint32_t permissions;
    int64_t atime;
    uint32_t atime_nseconds;
    int64_t createtime;
    uint32_t createtime_nseconds;
    int64_t mtime;
    uint32_t mtime_nseconds;
    int64_t ctime;
    uint32_t ctime_nseconds;
    char *acl;  
    size_t acl_len;
    uint32_t attrib_bits;
    uint32_t attrib_bits_valid;
    uint8_t text_hint;
    char *mime_type;
    uint32_t link_count;
    char *untranslated_name;
};

static __inline int is_digit(char c)
{
    return (c >= '0' && c <= '9');
}

static void trim(char *str)
{
    char *s, *e;
    s = str;
    e = s + strlen(s);
    while (*s == ' ')
	s++;
    while (e > s && e[-1] == ' ')
	e--;
    *e = '\0';
    memmove(str, s, (e - s) + 1);
}

pftp_sftp_attr_t pftp_sftp_parse_attr(uint32_t version, const char *server_str,
				      pftp_ssh_buf_t buf, const char *longinfo)
{
    pftp_sftp_attr_t ret;
    int valid;
    ret = malloc(sizeof(struct pftp_sftp_attr_s));
    memset(ret, 0, sizeof(struct pftp_sftp_attr_s));

    ret->flags = pftp_ssh_buf_get_uint32(buf, &valid);
    if (!valid) goto invalid_attr;

    if (version > 3) {
	ret->type = pftp_ssh_buf_get_char(buf, &valid);
    } else {
	ret->type = SSH_FILEXFER_TYPE_UNKNOWN;
    }
    if (!valid) goto invalid_attr;
    if (ret->flags & SSH_FILEXFER_ATTR_SIZE) {
	ret->size = pftp_ssh_buf_get_uint64(buf, &valid);
	if (!valid) goto invalid_attr;
    }
    if (ret->flags & SSH_FILEXFER_ATTR_ALLOCATION_SIZE) {
	ret->allocation_size = pftp_ssh_buf_get_uint64(buf, &valid);
	if (!valid) goto invalid_attr;
    }
    if (ret->flags & SSH_FILEXFER_ATTR_UIDGID) {
	/* v3 */
	uint32_t uid, gid;
	uid = pftp_ssh_buf_get_uint32(buf, &valid);
	if (!valid) goto invalid_attr;
	gid = pftp_ssh_buf_get_uint32(buf, &valid);
	if (!valid) goto invalid_attr;
	ret->owner = malloc(20);
	ret->group = malloc(20);
	snprintf(ret->owner, 20, "%lu", (unsigned long) uid);
	snprintf(ret->group, 20, "%lu", (unsigned long) gid);

	ret->flags |= SSH_FILEXFER_ATTR_OWNERGROUP;
    } else if (ret->flags & SSH_FILEXFER_ATTR_OWNERGROUP) {
	/* > 3 */
	ret->owner = pftp_ssh_buf_get_string(buf);
	ret->group = pftp_ssh_buf_get_string(buf);
	if (!ret->owner || !ret->group) goto invalid_attr;
	ret->flags |= SSH_FILEXFER_ATTR_UIDGID;
    }
    if (ret->flags & SSH_FILEXFER_ATTR_PERMISSIONS) {
	ret->permissions = pftp_ssh_buf_get_uint32(buf, &valid);
	if (!valid) goto invalid_attr;
	if (version <= 3) {
	    if (S_ISDIR(ret->permissions)) {
		ret->type = SSH_FILEXFER_TYPE_DIRECTORY;
	    } else if (S_ISREG(ret->permissions)) {
		ret->type = SSH_FILEXFER_TYPE_REGULAR;
	    } else if (S_ISCHR(ret->permissions)) {
		ret->type = SSH_FILEXFER_TYPE_CHAR_DEVICE;
	    } else if (S_ISBLK(ret->permissions)) {
                ret->type = SSH_FILEXFER_TYPE_BLOCK_DEVICE;
	    } else if (S_ISFIFO(ret->permissions)) {
		ret->type = SSH_FILEXFER_TYPE_FIFO;
	    } else if (S_ISLNK(ret->permissions)) {
		ret->type = SSH_FILEXFER_TYPE_SYMLINK;
	    } else if (S_ISSOCK(ret->permissions)) {
		ret->type = SSH_FILEXFER_TYPE_SOCKET;
	    }

	    ret->permissions &= ~S_IFMT;
	}
    }
    if (version <= 3) {
	if (ret->flags & SSH_FILEXFER_ATTR_ACMODTIME) {
	    ret->atime = pftp_ssh_buf_get_uint32(buf, &valid);
	    if (!valid) goto invalid_attr;
	    ret->mtime = pftp_ssh_buf_get_uint32(buf, &valid);
	    if (!valid) goto invalid_attr;

	    ret->flags |= SSH_FILEXFER_ATTR_ACCESSTIME;
	    ret->flags |= SSH_FILEXFER_ATTR_MODIFYTIME;
	}
    } else {
	if (ret->flags & SSH_FILEXFER_ATTR_ACCESSTIME) {
	    ret->atime = pftp_ssh_buf_get_int64(buf, &valid);
	    if (!valid) goto invalid_attr;
	    if (ret->flags & SSH_FILEXFER_ATTR_SUBSECOND_TIMES) {
		ret->atime_nseconds = pftp_ssh_buf_get_uint32(buf, &valid);
		if (!valid) goto invalid_attr;
	    }
	}
	if (ret->flags & SSH_FILEXFER_ATTR_CREATETIME) {
	    ret->createtime = pftp_ssh_buf_get_int64(buf, &valid);
	    if (!valid) goto invalid_attr;
	    if (ret->flags & SSH_FILEXFER_ATTR_SUBSECOND_TIMES) {
		ret->createtime_nseconds = pftp_ssh_buf_get_uint32(buf, &valid);
		if (!valid) goto invalid_attr;
	    }
	}
	if (ret->flags & SSH_FILEXFER_ATTR_MODIFYTIME) {
	    ret->mtime = pftp_ssh_buf_get_int64(buf, &valid);
	    if (!valid) goto invalid_attr;
	    if (ret->flags & SSH_FILEXFER_ATTR_SUBSECOND_TIMES) {
		ret->mtime_nseconds = pftp_ssh_buf_get_uint32(buf, &valid);
		if (!valid) goto invalid_attr;
	    }
	}
	if (ret->flags & SSH_FILEXFER_ATTR_CTIME) {
	    ret->ctime = pftp_ssh_buf_get_int64(buf, &valid);
	    if (!valid) goto invalid_attr;
	    if (ret->flags & SSH_FILEXFER_ATTR_SUBSECOND_TIMES) {
		ret->ctime_nseconds = pftp_ssh_buf_get_uint32(buf, &valid);
		if (!valid) goto invalid_attr;
	    }
	}
    }
    if (ret->flags & SSH_FILEXFER_ATTR_ACL) {
	ret->acl = pftp_ssh_buf_get_string2(buf, &ret->acl_len);
	if (!ret->acl) goto invalid_attr;
    }
    if (ret->flags & SSH_FILEXFER_ATTR_BITS) {
	ret->attrib_bits = pftp_ssh_buf_get_uint32(buf, &valid);
	if (!valid) goto invalid_attr;
	if (version > 5) {
	    ret->attrib_bits_valid = pftp_ssh_buf_get_uint32(buf, &valid);
	    if (!valid) goto invalid_attr;
	} else {
	    ret->attrib_bits_valid = 0xFFFFFFFF;
	}
    }
    if (ret->flags & SSH_FILEXFER_ATTR_TEXT_HINT) {
	ret->text_hint = pftp_ssh_buf_get_char(buf, &valid);
	if (!valid) goto invalid_attr;
    }
    if (ret->flags & SSH_FILEXFER_ATTR_MIME_TYPE) {
	ret->mime_type = pftp_ssh_buf_get_string(buf);
	if (!ret->mime_type) goto invalid_attr;
    }
    if (ret->flags & SSH_FILEXFER_ATTR_LINK_COUNT) {
	ret->link_count = pftp_ssh_buf_get_uint32(buf, &valid);
	if (!valid) goto invalid_attr;
    }
    if (ret->flags & SSH_FILEXFER_ATTR_UNTRANSLATED_NAME) {
	ret->untranslated_name = pftp_ssh_buf_get_string(buf);
	if (!ret->untranslated_name) goto invalid_attr;
    }
    if (ret->flags & SSH_FILEXFER_ATTR_EXTENDED) {
	uint32_t extended_count, e;
	extended_count = pftp_ssh_buf_get_uint32(buf, &valid);
	if (!valid) goto invalid_attr;
	for (e = 0; e < extended_count; e++) {
	    char *tmp;
	    /* extention type */
	    tmp = pftp_ssh_buf_get_string(buf);
	    if (!tmp) goto invalid_attr;
	    free(tmp);
	    /* extention data */
	    tmp = pftp_ssh_buf_get_string(buf);
	    if (!tmp) goto invalid_attr;
	    free(tmp);
	}
    }

    if (version <= 3 && longinfo) {
	const char *p, *q;
	char *end;

	/* Do exactly what the draft for v3 tell you not to,
	 * parse longfilename string. */

	/* Link count */
	if (*(p = (longinfo + 10)) == ' ') {
	    p++;
	    while (*p == ' ') p++;
	    if (is_digit(*p)) {
		unsigned long num;
		end = NULL;
		errno = 0;
		num = strtoul(p, &end, 10);
		if (errno == 0 && end && end[0] == ' ') {
		    ret->link_count = num;
		}
	    }
	}

	/* Owner & Group */
	if (*(p = (longinfo + 15)) == ' ') {
	    p++;
	    q = p + 8;
	    if (*q == ' ') {
		if (ret->owner)
		    free(ret->owner);
		ret->owner = strndup(p, q - p);
		trim(ret->owner);
	    }
	    p = q + 1;
	    q = p + 8;
	    if (*end == ' ') {
		if (ret->group)
		    free(ret->group);
		ret->group = strndup(p, q - p);
		trim(ret->group);
	    }
	}
    }

    return ret;

invalid_attr:
    pftp_sftp_free_attr(ret);
    return NULL;
}

pftp_sftp_attr_t pftp_sftp_create_attr(void)
{
    pftp_sftp_attr_t ret;
    ret = malloc(sizeof(struct pftp_sftp_attr_s));
    memset(ret, 0, sizeof(struct pftp_sftp_attr_s));

    return ret;
}

int pftp_sftp_export_attr(uint32_t version, pftp_ssh_buf_t buf,
			  pftp_sftp_attr_t attr)
{
    uint32_t flags;

    flags = 0;

    if (version <= 3) {
	flags = (attr->flags & (SSH_FILEXFER_ATTR_SIZE |
				SSH_FILEXFER_ATTR_PERMISSIONS));
		 
	if (attr->flags & SSH_FILEXFER_ATTR_OWNERGROUP)
	    flags |= SSH_FILEXFER_ATTR_UIDGID;	
	if (attr->flags & (SSH_FILEXFER_ATTR_ACCESSTIME |
			   SSH_FILEXFER_ATTR_MODIFYTIME)) {
	    flags |= SSH_FILEXFER_ATTR_ACMODTIME;
	}
    } else {
	if (version == 4) {
	    flags = (attr->flags & (SSH_FILEXFER_ATTR_SIZE |
				    SSH_FILEXFER_ATTR_OWNERGROUP |
				    SSH_FILEXFER_ATTR_PERMISSIONS |
				    SSH_FILEXFER_ATTR_ACCESSTIME |
				    SSH_FILEXFER_ATTR_CREATETIME |
				    SSH_FILEXFER_ATTR_MODIFYTIME |
				    SSH_FILEXFER_ATTR_ACL));
	} else if (version == 5) {
	    flags = (attr->flags & (SSH_FILEXFER_ATTR_SIZE |
				    SSH_FILEXFER_ATTR_OWNERGROUP |
				    SSH_FILEXFER_ATTR_PERMISSIONS |
				    SSH_FILEXFER_ATTR_ACCESSTIME |
				    SSH_FILEXFER_ATTR_CREATETIME |
				    SSH_FILEXFER_ATTR_MODIFYTIME |
				    SSH_FILEXFER_ATTR_ACL |
				    SSH_FILEXFER_ATTR_BITS));
	} else if (version >= 6) {
	    flags = attr->flags;
	}
    }

    pftp_ssh_buf_put_uint32(buf, flags);

    if (version > 3) {
	pftp_ssh_buf_put_char(buf, attr->type);
    }
    if (flags & SSH_FILEXFER_ATTR_SIZE) {
	pftp_ssh_buf_put_uint64(buf, attr->size);
    }
    if (flags & SSH_FILEXFER_ATTR_ALLOCATION_SIZE) {
	pftp_ssh_buf_put_uint64(buf, attr->allocation_size);
    }
    if (flags & SSH_FILEXFER_ATTR_UIDGID) {
	/* v3 */
	uint32_t uid, gid;
	uid = strtoul(attr->owner, NULL, 10);
	gid = strtoul(attr->group, NULL, 10);
	pftp_ssh_buf_put_uint32(buf, uid);
	pftp_ssh_buf_put_uint32(buf, gid);
    } else if (flags & SSH_FILEXFER_ATTR_OWNERGROUP) {
	/* > 3 */
	pftp_ssh_buf_put_string(buf, attr->owner);
	pftp_ssh_buf_put_string(buf, attr->group);
    }
    if (flags & SSH_FILEXFER_ATTR_PERMISSIONS) {
	if (version <= 3) {
	    uint32_t perm;
	    perm = attr->permissions;
	    
	    switch (attr->type) {
	    case SSH_FILEXFER_TYPE_DIRECTORY:
		perm |= S_IFDIR;
		break;
	    case SSH_FILEXFER_TYPE_REGULAR:
		perm |= S_IFREG;
		break;
	    case SSH_FILEXFER_TYPE_CHAR_DEVICE:
		perm |= S_IFCHR;
		break;
#ifndef WIN32
	    case SSH_FILEXFER_TYPE_BLOCK_DEVICE:
		perm |= S_IFBLK;
		break;
	    case SSH_FILEXFER_TYPE_FIFO:
		perm |= S_IFIFO;
		break;
	    case SSH_FILEXFER_TYPE_SYMLINK:
		perm |= S_IFLNK;
		break;
	    case SSH_FILEXFER_TYPE_SOCKET:
		perm |= S_IFSOCK;
		break;
#endif
	    }

	    pftp_ssh_buf_put_uint32(buf, perm);
	} else {
	    pftp_ssh_buf_put_uint32(buf, attr->permissions);
	}
    }
    if (version <= 3) {
	if (flags & SSH_FILEXFER_ATTR_ACMODTIME) {
	    pftp_ssh_buf_put_uint32(buf, (uint32_t) attr->atime);
	    pftp_ssh_buf_put_uint32(buf, (uint32_t) attr->mtime);
	}
    } else {
	if (flags & SSH_FILEXFER_ATTR_ACCESSTIME) {
	    pftp_ssh_buf_put_int64(buf, attr->atime);
	    if (flags & SSH_FILEXFER_ATTR_SUBSECOND_TIMES) {
		pftp_ssh_buf_put_uint32(buf, attr->atime_nseconds);
	    }
	}
	if (flags & SSH_FILEXFER_ATTR_CREATETIME) {
	    pftp_ssh_buf_put_int64(buf, attr->createtime);
	    if (flags & SSH_FILEXFER_ATTR_SUBSECOND_TIMES) {
		pftp_ssh_buf_put_uint32(buf, attr->createtime_nseconds);
	    }
	}
	if (flags & SSH_FILEXFER_ATTR_MODIFYTIME) {
	    pftp_ssh_buf_put_int64(buf, attr->mtime);
	    if (flags & SSH_FILEXFER_ATTR_SUBSECOND_TIMES) {
		pftp_ssh_buf_put_uint32(buf, attr->mtime_nseconds);
	    }
	}
	if (flags & SSH_FILEXFER_ATTR_CTIME) {
	    pftp_ssh_buf_put_int64(buf, attr->ctime);
	    if (flags & SSH_FILEXFER_ATTR_SUBSECOND_TIMES) {
		pftp_ssh_buf_put_uint32(buf, attr->ctime_nseconds);
	    }
	}
    }
    if (flags & SSH_FILEXFER_ATTR_ACL) {
	pftp_ssh_buf_put_string2(buf, attr->acl, attr->acl_len);
    }
    if (flags & SSH_FILEXFER_ATTR_BITS) {
	pftp_ssh_buf_put_uint32(buf, attr->attrib_bits);
	if (version > 5) {
	    pftp_ssh_buf_put_uint32(buf, attr->attrib_bits_valid);
	}
    }
    if (flags & SSH_FILEXFER_ATTR_TEXT_HINT) {
	pftp_ssh_buf_put_char(buf, attr->text_hint);
    }
    if (flags & SSH_FILEXFER_ATTR_MIME_TYPE) {
	pftp_ssh_buf_put_string(buf, attr->mime_type);
    }
    if (flags & SSH_FILEXFER_ATTR_LINK_COUNT) {
	pftp_ssh_buf_put_uint32(buf, attr->link_count);
    }
    if (flags & SSH_FILEXFER_ATTR_UNTRANSLATED_NAME) {
	pftp_ssh_buf_put_string(buf, attr->untranslated_name);
    }

    return 0;
}

void pftp_sftp_free_attr(pftp_sftp_attr_t attr)
{
    if (attr) {
	if (attr->owner)
	    free(attr->owner);
	if (attr->group)
	    free(attr->group);
	if (attr->acl)
	    free(attr->acl);
	if (attr->mime_type)
	    free(attr->mime_type);
	if (attr->untranslated_name)
	    free(attr->untranslated_name);
	free(attr);
    }
}

int pftp_sftp_is_dir(pftp_sftp_attr_t attr)
{
    return (attr->type == SSH_FILEXFER_TYPE_DIRECTORY);
}

pftp_file_type_t pftp_sftp_convert_attr_type(pftp_sftp_attr_t attr)
{
    switch (attr->type) {
    case SSH_FILEXFER_TYPE_DIRECTORY:
	return pft_directory;
    case SSH_FILEXFER_TYPE_BLOCK_DEVICE:
	return pft_block;
    case SSH_FILEXFER_TYPE_CHAR_DEVICE:
	return pft_character;
    case SSH_FILEXFER_TYPE_FIFO:
	return pft_fifo;
    case SSH_FILEXFER_TYPE_SOCKET:
	return pft_s_socket;
    case SSH_FILEXFER_TYPE_REGULAR:
	return pft_file;
    default:
	return pft_unknown;
    }
}

void pftp_sftp_convert_attr(pftp_file_t *file, pftp_sftp_attr_t attr)
{
    file->type = pftp_sftp_convert_attr_type(attr);

    file->perm = attr->permissions;
    file->links = attr->link_count;
    file->user = attr->owner ? strdup(attr->owner) : strdup("unknown");
    file->group = attr->group ? strdup(attr->group) : strdup("unknown");
    file->link = (attr->type == SSH_FILEXFER_TYPE_SYMLINK);
    file->size = attr->size;
    if (attr->mtime >= 0) {
	time_t sec;
	struct tm date, *ret;
	sec = (time_t) attr->mtime;
	file->changed = malloc(15);
#if HAVE_GMTIME_R
	ret = gmtime_r(&sec, &date);
#else
	ret = gmtime(&sec);
	if (ret)
	    memcpy(&date, ret, sizeof(struct tm));
#endif
	pftp_set_date(file->changed, &date);
    } else {
	file->changed = strdup("");
    }
}

void pftp_sftp_attr_set_permissions(pftp_sftp_attr_t attr, 
				    unsigned short access)
{
    attr->flags |= SSH_FILEXFER_ATTR_PERMISSIONS;
    attr->permissions = (access & 0777);
}

void pftp_sftp_attr_set_modificationtime(pftp_sftp_attr_t attr, 
					 const struct tm *date)
{
    struct tm _date;
    memcpy(&_date, date, sizeof(struct tm));
    attr->flags |= SSH_FILEXFER_ATTR_MODIFYTIME;
    attr->mtime = mktime(&_date);
}

#endif /* NO_SSL */
