/* -*- pftp-c -*- */
#ifdef HAVE_CONFIG_H
# include "config.h"
#endif
#if HAVE_STDLIB_H
# include <stdlib.h>
#endif
#if HAVE_STDIO_H
# include <stdio.h>
#endif
#if HAVE_INTTYPES_H
# include <inttypes.h>
#endif
#if HAVE_TIME_H
# include <time.h>
#endif
#if HAVE_STRING_H
# include <string.h>
#endif
#if HAVE_LIMITS_H
# include <limits.h>
#endif
#if HAVE_NETINET_IN_H
# include <netinet/in.h>
#endif
#if HAVE_ERRNO_H
# include <errno.h>
#endif

#ifndef NO_SSL

#include <openssl/ssl.h>

#ifdef WIN32
typedef SOCKET socket_t;
#else
typedef int socket_t; 
#endif

#ifdef DEBUG
# include <assert.h>
#else
# define assert(x) // x
#endif

#include "pftp_sfvcheck.h"
#include "pftp_default.h"
#include "pftp_settings.h"
#include "pftp.h"
#include "pftp_sftp.h"
#include "pftp_speed.h"
#include "pftp_internal.h"
#include "pftp_ssh_packet.h"
#include "pftp_ssh.h"
#include "pftp_ssh_userauth.h"
#include "pftp_ssh_channel.h"
#include "pftp_ssh_buf.h"
#include "pftp_sftp_msg.h"
#include "pftp_utf8.h"
#include "pftp_sftp_attrib.h"

#ifdef WITH_DMALLOC
# include <dmalloc.h>
#endif

typedef struct {
    unsigned char type;
    uint32_t req_id;
    char *data;
    size_t len;
} sftp_packet_s, *sftp_packet_t;

struct pftp_sftp_s {
    pftp_server_t ftp;
    pftp_ssh_t ssh;
    pftp_ssh_channel_master_t channel_master;
    char *cwd, *home;
    pftp_ssh_channel_t main;
    uint32_t version;
    uint32_t req_id;

    size_t in_packets;
    sftp_packet_t *in_packet;
    
    char *newline;
};

#define CLIENT_SFTP_VERSION 6
#define CLIENT_SFTP_MIN_VERSION 0

#define SSH_FXP_REALPATH_NO_CHECK 1

#define SSH_FX_OK                            0
#define SSH_FX_EOF                           1
#define SSH_FX_NO_SUCH_FILE                  2
#define SSH_FX_PERMISSION_DENIED             3
#define SSH_FX_FAILURE                       4
#define SSH_FX_BAD_MESSAGE                   5
#define SSH_FX_NO_CONNECTION                 6
#define SSH_FX_CONNECTION_LOST               7
#define SSH_FX_OP_UNSUPPORTED                8
#define SSH_FX_INVALID_HANDLE                9
#define SSH_FX_NO_SUCH_PATH                  10
#define SSH_FX_FILE_ALREADY_EXISTS	     11
#define SSH_FX_WRITE_PROTECT		     12
#define SSH_FX_NO_MEDIA                      13
#define SSH_FX_NO_SPACE_ON_FILESYSTEM        14
#define SSH_FX_QUOTA_EXCEEDED                15
#define SSH_FX_UNKNOWN_PRINCIPAL             16
#define SSH_FX_LOCK_CONFLICT                 17
#define SSH_FX_DIR_NOT_EMPTY                 18
#define SSH_FX_NOT_A_DIRECTORY               19
#define SSH_FX_INVALID_FILENAME              20
#define SSH_FX_LINK_LOOP                     21
#define SSH_FX_CANNOT_DELETE                 22
#define SSH_FX_INVALID_PARAMETER             23
#define SSH_FX_FILE_IS_A_DIRECTORY           24
#define SSH_FX_BYTE_RANGE_LOCK_CONFLICT      25
#define SSH_FX_BYTE_RANGE_LOCK_REFUSED       26
#define SSH_FX_DELETE_PENDING                27
#define SSH_FX_FILE_CORRUPT                  28

#define ACE4_READ_DATA         0x00000001
#define ACE4_LIST_DIRECTORY    0x00000001
#define ACE4_WRITE_DATA        0x00000002
#define ACE4_ADD_FILE          0x00000002
#define ACE4_APPEND_DATA       0x00000004
#define ACE4_ADD_SUBDIRECTORY  0x00000004
#define ACE4_READ_NAMED_ATTRS  0x00000008
#define ACE4_WRITE_NAMED_ATTRS 0x00000010
#define ACE4_EXECUTE           0x00000020
#define ACE4_DELETE_CHILD      0x00000040
#define ACE4_READ_ATTRIBUTES   0x00000080
#define ACE4_WRITE_ATTRIBUTES  0x00000100
#define ACE4_DELETE            0x00010000
#define ACE4_READ_ACL          0x00020000
#define ACE4_WRITE_ACL         0x00040000
#define ACE4_WRITE_OWNER       0x00080000
#define ACE4_SYNCHRONIZE       0x00100000

#define SSH_FXF_ACCESS_DISPOSITION     0x00000007
#define SSH_FXF_CREATE_NEW             0x00000000
#define SSH_FXF_CREATE_TRUNCATE        0x00000001
#define SSH_FXF_OPEN_EXISTING          0x00000002
#define SSH_FXF_OPEN_OR_CREATE         0x00000003
#define SSH_FXF_TRUNCATE_EXISTING      0x00000004
#define SSH_FXF_ACCESS_APPEND_DATA         0x00000008
#define SSH_FXF_ACCESS_APPEND_DATA_ATOMIC  0x00000010
#define SSH_FXF_ACCESS_TEXT_MODE           0x00000020
#define SSH_FXF_ACCESS_BLOCK_READ          0x00000040
#define SSH_FXF_ACCESS_BLOCK_WRITE         0x00000080
#define SSH_FXF_ACCESS_BLOCK_DELETE        0x00000100
#define SSH_FXF_ACCESS_BLOCK_ADVISORY      0x00000200
#define SSH_FXF_ACCESS_NOFOLLOW            0x00000400
#define SSH_FXF_ACCESS_DELETE_ON_CLOSE     0x00000800

#define SSH_FXF_READ            0x00000001
#define SSH_FXF_WRITE           0x00000002
#define SSH_FXF_APPEND          0x00000004
#define SSH_FXF_CREAT           0x00000008
#define SSH_FXF_TRUNC           0x00000010
#define SSH_FXF_EXCL            0x00000020
#define SSH_FXF_TEXT            0x00000040

static void _free_sftp_packet(sftp_packet_t pkg);
static int _send_fxp_init(pftp_sftp_t sftp);
static int _get_fxp_version(pftp_sftp_t sftp);
static int _send_packet(pftp_sftp_t sftp, uint32_t *req_id, unsigned char type, 
			pftp_ssh_buf_t buf, speed_data_t speed);
static int _get_packet(pftp_sftp_t sftp, uint32_t req_id, unsigned char *type, 
		       pftp_ssh_buf_t buf, speed_data_t speed);
static int _get_pwd(pftp_sftp_t sftp);
static int _fxp_status(pftp_sftp_t sftp, pftp_ssh_buf_t buf, const char *def);
static void _close_fxp_handle(pftp_sftp_t sftp, char *handle, 
			      size_t handle_len);
static void _fill_pftp_file_t(pftp_sftp_t sftp, pftp_file_t *file, 
			      const char *filename, pftp_sftp_attr_t attr);
static char *_make_fullpath(pftp_sftp_t sftp, const char *filename);
static void _sort_ftp_directory(pftp_directory_t *dir);
static char *_fxp_open_file(pftp_sftp_t sftp, const char *filename, 
			    uint32_t flags, char ascii,
			    size_t *handle_len, pftp_sftp_attr_t attr);
static void freesplit(char **part, size_t parts);
static size_t splitchar(const char *str, char needle, char ***part);

/* Login, make the connection to the ftp. Takes a settings structure pointer
   that must exist until logout is called. 
   Returns NULL if error. Any error messages are sent to status */
pftp_sftp_t pftp_sftp_login(pftp_server_t ftp)
{
    pftp_sftp_t ret;
    int none_method;
    ret = malloc(sizeof(struct pftp_sftp_s));
    memset(ret, 0, sizeof(struct pftp_sftp_s));

    none_method = 1;
    ret->ftp = ftp;
    ret->cwd = strdup("");
    ret->home = strdup("");

    ret->ssh = pftp_ssh_connect(ret->ftp);
    if (!ret->ssh) {
	pftp_sftp_logout(&ret);
	return NULL;
    }

    if (pftp_ssh_userauth(ret->ftp, ret->ssh)) {
	pftp_sftp_logout(&ret);
	return NULL;
    } 

    ret->channel_master = pftp_ssh_init_channel_master(ret->ssh, ret->ftp);
    if (!ret->channel_master) {
	pftp_sftp_logout(&ret);
	return NULL;
    }

    ret->main = pftp_ssh_open_channel_session(ret->channel_master,
					      32768);
    if (!ret->main) {
	pftp_sftp_logout(&ret);
	return NULL;
    }

    if (pftp_ssh_channel_request_subsystem(ret->main, 1, "sftp")) {
	pftp_sftp_logout(&ret);
	return NULL;
    }

    if (_send_fxp_init(ret) || _get_fxp_version(ret)) {
	pftp_sftp_logout(&ret);
	return NULL;
    }

    if (ret->version < CLIENT_SFTP_MIN_VERSION) {
	pftp_status_message(ret->ftp, 
			    "SFTP: Server version (%lu) is a little to old for "
			    "me..", (unsigned long) ret->version);
	pftp_sftp_logout(&ret);
	return NULL;
    }

    pftp_status_message(ret->ftp, "SFTP: Using SFTP version %lu.",
			(unsigned long) ret->version);

    if (_get_pwd(ret)) {
	pftp_status_message(ret->ftp, "SFTP: Unable to get start path.");
    } else {
	free(ret->home);
	ret->home = strdup(ret->cwd);
    }

    return ret;
}

/* Logout, disconnect from a sftp and free memory of pointer */
void pftp_sftp_logout(pftp_sftp_t *sftp)
{
    if (*sftp) {
	size_t p;
	for (p = 0; p < (*sftp)->in_packets; p++) 	
	    _free_sftp_packet((*sftp)->in_packet[p]);
	if ((*sftp)->in_packet)
	    free((*sftp)->in_packet);
	if ((*sftp)->main)
	    pftp_ssh_channel_close((*sftp)->main);
	if ((*sftp)->channel_master)
	    pftp_ssh_free_channel_master((*sftp)->channel_master);
	if ((*sftp)->ssh)
	    pftp_ssh_close(&(*sftp)->ssh);		
	if ((*sftp)->newline)
	    free((*sftp)->newline);
	free((*sftp)->cwd);
	free((*sftp)->home);
	free(*sftp);
	*sftp = NULL;
    }
}

/* Change current directory, 
   If newdir is not null:
   The point newdir points at must be NULL.
   newdir is set to what the current path is then.
   returns 0 on success, -1 on error */
int pftp_sftp_cd(pftp_sftp_t sftp, const char *dir, char **newdir)
{
    pftp_ssh_buf_t buf;
    uint32_t id, cnt;
    int valid_cnt;
    char *path = NULL;
    unsigned char type;

    if (sftp->version == 6) {
	char *_tmp;
	buf = pftp_ssh_create_buf();
	_tmp = strdup(sftp->cwd);
	pftp_make_unicode_utf8(&_tmp);
	pftp_ssh_buf_put_string(buf, _tmp);
	free(_tmp);
	/* Will use older version check instead. */
	pftp_ssh_buf_put_char(buf, SSH_FXP_REALPATH_NO_CHECK);
	_tmp = strdup(dir);
	pftp_make_unicode_utf8(&_tmp);
	pftp_ssh_buf_put_string(buf, _tmp);
	free(_tmp);
	
	if (_send_packet(sftp, &id, SSH_FXP_REALPATH, buf, NULL)) {
	    pftp_ssh_buf_free(buf);
	    return -1;
	}
	
	pftp_ssh_buf_free(buf);
    } else {
	char *tmp;

	if (dir[0] == '/') {
	    /* Absolute (as far as I know) */
	    tmp = strdup(dir);
	} else {
	    /* Relative */
	    tmp = _make_fullpath(sftp, dir);
	}       

	buf = pftp_ssh_create_buf();

	if (sftp->version > 3) {
	    pftp_make_unicode_utf8(&tmp);
	}

	pftp_ssh_buf_put_string(buf, tmp);
	free(tmp);
	
	if (_send_packet(sftp, &id, SSH_FXP_REALPATH, buf, NULL)) {
	    pftp_ssh_buf_free(buf);
	    return -1;
	}
	
	pftp_ssh_buf_free(buf);
    }

    buf = pftp_ssh_create_buf();
	
    if (_get_packet(sftp, id, &type, buf, NULL)) {
	pftp_ssh_buf_free(buf);
	return -1;
    }

    if (type == SSH_FXP_STATUS) {
	_fxp_status(sftp, buf, "Invalid path");
	pftp_ssh_buf_free(buf);
	return -1;
    } else if (type != SSH_FXP_NAME) {
	pftp_status_message(sftp->ftp,
			    "SFTP: Unexpected response to "
			    "SSH_FXP_REALPATH (%u).", type);
	pftp_ssh_buf_free(buf);
	return -1;
    }

    cnt = pftp_ssh_buf_get_uint32(buf, &valid_cnt);
    path = pftp_ssh_buf_get_string(buf);
	
    if (cnt == 0 || !valid_cnt || !path) {
	pftp_status_message(sftp->ftp,
			    "SFTP: Invalid SSH_FXP_NAME packet.");
	if (path) free(path);
	pftp_ssh_buf_free(buf);
	return -1;
    }
	
    pftp_ssh_buf_free(buf);

    /* Now check if path exists. */
    buf = pftp_ssh_create_buf();
    pftp_ssh_buf_put_string(buf, path);
    if (sftp->version > 3) {
	/* !!! Better flags ? */
	pftp_ssh_buf_put_uint32(buf, 0);
    }
    
    if (_send_packet(sftp, &id, SSH_FXP_STAT, buf, NULL)) {
	pftp_ssh_buf_free(buf);
	free(path);
	return -1;
    }
	
    pftp_ssh_buf_free(buf);
    buf = pftp_ssh_create_buf();

    if (_get_packet(sftp, id, &type, buf, NULL)) {
	pftp_ssh_buf_free(buf);
	free(path);
	return -1;
    }

    if (type == SSH_FXP_ATTRS) {
	pftp_sftp_attr_t attr;
	attr = pftp_sftp_parse_attr(sftp->version, 
				    pftp_ssh_get_server_str(sftp->ssh),
				    buf, NULL);
	if (!attr) {
	    free(path);
	    pftp_ssh_buf_free(buf);
	    return -1;
	}
	if (!pftp_sftp_is_dir(attr)) {
	    pftp_status_message(sftp->ftp, "599 Not a directory.");
	    free(path);
	    pftp_sftp_free_attr(attr);
	    pftp_ssh_buf_free(buf);
	    return -1;
	}

	pftp_sftp_free_attr(attr);
	pftp_ssh_buf_free(buf);
    } else if (type == SSH_FXP_STATUS) {
	_fxp_status(sftp, buf, "Invalid path");
	pftp_ssh_buf_free(buf);
	free(path);
	return -1;
    } else {
	pftp_status_message(sftp->ftp,
			    "SFTP: Unexpected response to "
			    "SSH_FXP_STAT (%u).", type);
	pftp_ssh_buf_free(buf);
	free(path);
	return -1;
    }

    if (sftp->version > 3) {
	pftp_parse_unicode_utf8(path);
    }       

    if (sftp->cwd)
	free(sftp->cwd);
    sftp->cwd = strdup(path);
    free(path);
    
    if (newdir) {
	assert(*newdir == NULL);
	*newdir = strdup(sftp->cwd);
    }
    return 0;
}

/* Remove directory, returns 0 on success, -1 on error */
int pftp_sftp_rmdir(pftp_sftp_t sftp, const char *dir)
{
    pftp_ssh_buf_t buf;
    char *path;
    uint32_t id;
    unsigned char type;

    path = _make_fullpath(sftp, dir);

    if (sftp->version > 3) {
	pftp_make_unicode_utf8(&path);
    }

    buf = pftp_ssh_create_buf();
    pftp_ssh_buf_put_string(buf, path);
    free(path);

    if (_send_packet(sftp, &id, SSH_FXP_RMDIR, buf, NULL)) {
	pftp_ssh_buf_free(buf);
	return -1;
    }
    pftp_ssh_buf_free(buf);
    buf = pftp_ssh_create_buf();

    if (_get_packet(sftp, id, &type, buf, NULL)) {
	pftp_ssh_buf_free(buf);
	return -1;
    }

    if (type == SSH_FXP_STATUS) {
	if (_fxp_status(sftp, buf, "Unable to remove directory")) {
	    pftp_ssh_buf_free(buf);
	    return -1;
	} else {
	    pftp_ssh_buf_free(buf);
	    return 0;
	}
    } else {
	pftp_status_message(sftp->ftp,
			    "SFTP: Unexpected response to "
			    "SSH_FXP_RMDIR (%u).", type);
	pftp_ssh_buf_free(buf);       
	return -1;
    }
}

/* Create directory, returns 0 on success, -1 on error */
int pftp_sftp_mkdir(pftp_sftp_t sftp, const char *dir)
{
    pftp_sftp_attr_t attr;
    pftp_ssh_buf_t buf;
    char *path;
    uint32_t id;
    unsigned char type;

    path = _make_fullpath(sftp, dir);

    if (sftp->version > 3) {
	pftp_make_unicode_utf8(&path);
    }

    buf = pftp_ssh_create_buf();
    pftp_ssh_buf_put_string(buf, path);
    attr = pftp_sftp_create_attr();
    /* !!! Enter some good defaults here ? */
    pftp_sftp_export_attr(sftp->version, buf, attr);
    pftp_sftp_free_attr(attr);
    free(path);

    if (_send_packet(sftp, &id, SSH_FXP_MKDIR, buf, NULL)) {
	pftp_ssh_buf_free(buf);
	return -1;
    }
    pftp_ssh_buf_free(buf);
    buf = pftp_ssh_create_buf();

    if (_get_packet(sftp, id, &type, buf, NULL)) {
	pftp_ssh_buf_free(buf);
	return -1;
    }

    if (type == SSH_FXP_STATUS) {
	if (_fxp_status(sftp, buf, "Unable to create directory")) {
	    pftp_ssh_buf_free(buf);
	    return -1;
	} else {
	    pftp_ssh_buf_free(buf);
	    return 0;
	}
    } else {
	pftp_status_message(sftp->ftp,
			    "SFTP: Unexpected response to "
			    "SSH_FXP_MKDIR (%u).", type);
	pftp_ssh_buf_free(buf);       
	return -1;
    }
}


/* Get current directory, The pointer currentDirectory must point to NULL and
   will be left to NULL if any errors accur. Otherwise it will be set to 
   current directory. User must free the string themself.  */
void pftp_sftp_curdir(pftp_sftp_t sftp, char **currentDirectory)
{
    assert(*currentDirectory == NULL);
    *currentDirectory = strdup(sftp->cwd);
}

/* List current directory. Will fill the structure dir with info. 
   Returns 0 on no errors, -1 on errors (and then dir is not touched) */
int pftp_sftp_ls(pftp_sftp_t sftp, pftp_directory_t *dir)
{
    pftp_ssh_buf_t buf;
    uint32_t id;
    unsigned char type;
    char *handle;
    size_t handle_len;
    speed_data_t speed;
    
    buf = pftp_ssh_create_buf();

    if (sftp->version > 3) {
	char *tmp;
	tmp = strdup(sftp->cwd);
	pftp_make_unicode_utf8(&tmp);
	pftp_ssh_buf_put_string(buf, tmp);
	free(tmp);
    } else {
	pftp_ssh_buf_put_string(buf, sftp->cwd);
    }

    if (_send_packet(sftp, &id, SSH_FXP_OPENDIR, buf, NULL)) {
	pftp_ssh_buf_free(buf);
	return -1;
    }

    pftp_ssh_buf_free(buf);
    buf = pftp_ssh_create_buf();

    if (_get_packet(sftp, id, &type, buf, NULL)) {
	pftp_ssh_buf_free(buf);
	return -1;
    }

    if (type == SSH_FXP_HANDLE) {
	handle = pftp_ssh_buf_get_string2(buf, &handle_len);
	pftp_ssh_buf_free(buf);
	if (!handle) {
	    pftp_status_message(sftp->ftp, 
				"SFTP: Invalid handle packet.");
	    return -1;
	}
    } else if (type == SSH_FXP_STATUS) {
	_fxp_status(sftp, buf, "Unable to open dir");
	pftp_ssh_buf_free(buf);
	return -1;
    } else {
	pftp_status_message(sftp->ftp, 
			    "SFTP: Unexpected response to OPENDIR (%u).",
			    type);
	pftp_ssh_buf_free(buf);
	return -1;
    }

    dir->length = 0;
    dir->files = NULL;

    speed = pftp_init_speed(0, 0, sftp->ftp->lastgoodbuffersize);
    pftp_start_wait(sftp->ftp, DOWNLOAD, 1);
	
    for (;;) {
	buf = pftp_ssh_create_buf();
	pftp_ssh_buf_put_string2(buf, handle, handle_len);
	
	if (_send_packet(sftp, &id, SSH_FXP_READDIR, buf, NULL)) {
	    pftp_free_speed(&speed, &sftp->ftp->lastgoodbuffersize);
	    pftp_end_wait(sftp->ftp, DOWNLOAD, 1);
	    pftp_ssh_buf_free(buf);
	    return -1;
	}
	
	pftp_ssh_buf_free(buf);
	buf = pftp_ssh_create_buf();

	if (_get_packet(sftp, id, &type, buf, speed)) {
	    pftp_free_speed(&speed, &sftp->ftp->lastgoodbuffersize);
	    pftp_end_wait(sftp->ftp, DOWNLOAD, 1);
	    pftp_ssh_buf_free(buf);
	    return -1;
	}	

	if (type == SSH_FXP_STATUS) {
	    uint32_t err_code;
	    int err_valid;
	    char *err_msg;
	    
	    err_code = pftp_ssh_buf_get_uint32(buf, &err_valid);
	    err_msg = pftp_ssh_buf_get_string(buf);
    
	    if (err_valid) {
		if (err_code == SSH_FX_EOF) {
		    /* All names read */
		    if (err_msg) free(err_msg);
		    break;
		}
		
		if (err_msg) {		    
		    pftp_parse_unicode_utf8(err_msg);
		    pftp_status_message(sftp->ftp, "599 %s (%lu)",
					err_msg, (unsigned long)err_code);
		    free(err_msg);
		} else {
		    pftp_status_message(sftp->ftp, "599 %s (%lu)",
					"Error reading dir", 
					(unsigned long)err_code);
		}
	    } else {
		pftp_status_message(sftp->ftp, "599 Invalid status packet.");
	    }
	    pftp_ssh_buf_free(buf);
	    _close_fxp_handle(sftp, handle, handle_len);
	    pftp_free_speed(&speed, &sftp->ftp->lastgoodbuffersize);
	    pftp_end_wait(sftp->ftp, DOWNLOAD, 1);
	    return -1;
	} else if (type == SSH_FXP_NAME) {
	    uint32_t cnt, c;
	    int valid;

	    cnt = pftp_ssh_buf_get_uint32(buf, &valid);
	    if (!valid) {
		pftp_status_message(sftp->ftp, "SFTP: Invalid name packet.");
		pftp_ssh_buf_free(buf);
		_close_fxp_handle(sftp, handle, handle_len);
		pftp_free_speed(&speed, &sftp->ftp->lastgoodbuffersize);
		pftp_end_wait(sftp->ftp, DOWNLOAD, 1);
		return -1;
	    }

	    dir->files = realloc(dir->files,
				 (dir->length + cnt) * sizeof(pftp_file_t));

	    for (c = (uint32_t) dir->length; 
		 c < (uint32_t) (dir->length + cnt); c++) {
		char *file, *longinfo;
		pftp_sftp_attr_t attr;

		file = pftp_ssh_buf_get_string(buf);
		if (sftp->version <= 3) {
		    longinfo = pftp_ssh_buf_get_string(buf);
		} else {
		    longinfo = NULL;
		}

		attr = pftp_sftp_parse_attr(sftp->version, 
					    pftp_ssh_get_server_str(sftp->ssh),
					    buf, longinfo);

		if (!file || !attr) {
		    pftp_status_message(sftp->ftp, 
					"SFTP: Invalid name packet.");
		    if (longinfo) free(longinfo);
		    pftp_ssh_buf_free(buf);		    
		    _close_fxp_handle(sftp, handle, handle_len);
		    dir->length = (c - 1);
		    pftp_free_fdt(*dir);
		    dir->files = NULL;		    
		    dir->length = 0;
		    pftp_free_speed(&speed, &sftp->ftp->lastgoodbuffersize);
		    pftp_end_wait(sftp->ftp, DOWNLOAD, 1);
		    return -1;
		} else {
		    if (longinfo) 
			free(longinfo);
		}

		_fill_pftp_file_t(sftp, dir->files + c, file, attr);
		free(file);
		pftp_sftp_free_attr(attr);
	    }

	    dir->length += cnt;
	    
	    pftp_ssh_buf_free(buf);
	} else {
	    pftp_status_message(sftp->ftp, 
				"SFTP: Unexpected response to READDIR (%u).",
				type);
	    pftp_ssh_buf_free(buf);
	    _close_fxp_handle(sftp, handle, handle_len);
	    pftp_free_speed(&speed, &sftp->ftp->lastgoodbuffersize);
	    pftp_end_wait(sftp->ftp, DOWNLOAD, 1);
	    return -1;
	} 
    }

    pftp_free_speed(&speed, &sftp->ftp->lastgoodbuffersize);
    pftp_end_wait(sftp->ftp, DOWNLOAD, 0);

    _sort_ftp_directory(dir);

    _close_fxp_handle(sftp, handle, handle_len);
    return 0;
}

/* Download a file from current direcory.
   filename - is the file's name.
   ascii - if set to 'B' then download is in binary mode otherwise ascii mode.
   fh - must point to a writeable stream.
   resume_from - Tells from where we should start reading from sftp 
   (if supported by server). If it's supported, get will continue. But if
   resume_from is > 0 and it's not supported, error will be returned.
   total_size - Information about total filesize, will be used when sending
   info to update function.
   crc - If not NULL, after file download this will be the crc32 code of 
   downloaded data. (So, dont rely on this when using resume).

   Returns 0 on OK and -1 on error.
*/
int pftp_sftp_get(pftp_sftp_t sftp, const char *filename, char ascii, 
		  FILE *fh, uint64_t resume_from, uint64_t total_size, 
		  uint32_t *crc)
{
    char *handle = NULL;
    pftp_ssh_buf_t buf;
    size_t handle_len, ret_size;
    uint64_t offset;
    speed_data_t speed;
    uint32_t id;
    unsigned char type;
    pftp_sftp_attr_t attr;

    if (ascii == 'A' && resume_from > 0) {
	pftp_status_message(sftp->ftp, 
			    "SFTP: Can\'t resume files in TEXT mode.");
	return -1;
    }

    attr = pftp_sftp_create_attr();
    
    handle = _fxp_open_file(sftp, filename, SSH_FXF_READ, ascii, &handle_len, 
			    attr);
    pftp_sftp_free_attr(attr);
    if (!handle)
	return -1;    

    offset = resume_from;

    pftp_status_filetransfer(sftp->ftp, resume_from, total_size);
    speed = pftp_init_speed(total_size, resume_from, 
			    sftp->ftp->lastgoodbuffersize);
    pftp_start_wait(sftp->ftp, DOWNLOAD, 1);

    if (crc)
	pftp_initCRC32(&sftp->ftp->cur_crc);    

    for (;;) {
	buf = pftp_ssh_create_buf();
	pftp_ssh_buf_put_string2(buf, handle, handle_len);
	pftp_ssh_buf_put_uint64(buf, offset);
	pftp_ssh_buf_put_uint32(buf, (uint32_t) speed->buffersize);

	if (_send_packet(sftp, &id, SSH_FXP_READ, buf, NULL)) {
	    pftp_ssh_buf_free(buf);
	    pftp_free_speed(&speed, &sftp->ftp->lastgoodbuffersize);
	    pftp_end_wait(sftp->ftp, DOWNLOAD, 1);
	    _close_fxp_handle(sftp, handle, handle_len);
	    return -1;
	}

	pftp_ssh_buf_free(buf);
	buf = pftp_ssh_create_buf();

	if (_get_packet(sftp, id, &type, buf, speed)) {
	    pftp_ssh_buf_free(buf);
	    pftp_free_speed(&speed, &sftp->ftp->lastgoodbuffersize);
	    pftp_end_wait(sftp->ftp, DOWNLOAD, 1);
	    _close_fxp_handle(sftp, handle, handle_len);
	    return -1;
	}

	if (type == SSH_FXP_DATA) {
	    char *buffer;
	    buffer = pftp_ssh_buf_get_string2(buf, &ret_size);
	    pftp_ssh_buf_free(buf);

	    if (!buffer) {
		pftp_status_message(sftp->ftp, "599 Invalid data packet.");	
		pftp_free_speed(&speed, &sftp->ftp->lastgoodbuffersize);
		pftp_end_wait(sftp->ftp, DOWNLOAD, 1);
		_close_fxp_handle(sftp, handle, handle_len);
		return -1;	    
	    }

	    if (fwrite(buffer, 1, ret_size, fh) != ret_size) {
		pftp_status_message(sftp->ftp, 
				    "ERR: Unable to write to file: %s", 
				    strerror(errno));
		pftp_free_speed(&speed, &sftp->ftp->lastgoodbuffersize);
		pftp_end_wait(sftp->ftp, DOWNLOAD, 1);
		_close_fxp_handle(sftp, handle, handle_len);
		free(buffer);
		return -1;
	    }

	    if (crc)
		pftp_addCRC32(&sftp->ftp->cur_crc, buffer, ret_size);

	    offset += ret_size;

	    free(buffer);
	} else if (type == SSH_FXP_STATUS) {
	    uint32_t err_code;
	    int err_valid;
	    char *err_msg;
	    
	    err_code = pftp_ssh_buf_get_uint32(buf, &err_valid);
	    err_msg = pftp_ssh_buf_get_string(buf);
	    
	    if (err_valid) {
		if (err_code == SSH_FX_EOF) {
		    /* All data read */
		    if (err_msg) free(err_msg);
		    break;
		}
		
		if (err_msg) {		    
		    pftp_parse_unicode_utf8(err_msg);
		    pftp_status_message(sftp->ftp, "599 %s (%lu)",
					err_msg, (unsigned long)err_code);
		    free(err_msg);
		} else {
		    pftp_status_message(sftp->ftp, "599 %s (%lu)",
					"Error reading file", 
					(unsigned long)err_code);
		}
	    } else {
		pftp_status_message(sftp->ftp, "599 Invalid status packet.");
	    }
	    pftp_ssh_buf_free(buf);
	    pftp_free_speed(&speed, &sftp->ftp->lastgoodbuffersize);
	    pftp_end_wait(sftp->ftp, DOWNLOAD, 1);
	    _close_fxp_handle(sftp, handle, handle_len);
	    return -1;	    
	} else {
	    pftp_status_message(sftp->ftp,
				"SFTP: Unexpected response to "
				"SSH_FXP_READ (%u).", type);	    
	    pftp_ssh_buf_free(buf);
	    pftp_free_speed(&speed, &sftp->ftp->lastgoodbuffersize);
	    pftp_end_wait(sftp->ftp, DOWNLOAD, 1);
	    _close_fxp_handle(sftp, handle, handle_len);
	    return -1;
	}

	pftp_update_speed(speed, ret_size);

	if (pftp_do_wait(sftp->ftp, speed, 1, "DOWNLOAD-DATA") == -1) {
	    pftp_free_speed(&speed, &sftp->ftp->lastgoodbuffersize);
	    pftp_end_wait(sftp->ftp, DOWNLOAD, 1);
	    _close_fxp_handle(sftp, handle, handle_len);
	    return -1;
	}
    }

    pftp_free_speed(&speed, &sftp->ftp->lastgoodbuffersize);
    pftp_end_wait(sftp->ftp, DOWNLOAD, 0);

    _close_fxp_handle(sftp, handle, handle_len);

    if (crc) {
	pftp_finishCRC32(&sftp->ftp->cur_crc);
	(*crc) = sftp->ftp->cur_crc;
    }

    return 0;
}

/* Upload a file to the current direcory.
   filename - is the file's name.
   ascii - if set to 'B' then upload is in binary mode otherwise ascii mode.
   fh - must point to a readable stream.
   resume_from - Tells from where we should start writing to the file on the
   ftp (if supported by server). If it's supported, put will continue. But if
   resume_from is > 0 and it's not supported, error will be returned.
   total_size - Information about total filesize, will be used when sending
   info to update function.

   Returns 0 on OK and -1 on error.
*/
int pftp_sftp_put(pftp_sftp_t sftp, const char *filename, char ascii, 
		  FILE *fh, uint64_t append_to, uint64_t total_size)
{
    char *handle = NULL;
    pftp_ssh_buf_t buf;
    size_t handle_len, ret_size;
    uint64_t offset;
    speed_data_t speed;
    uint32_t id;
    unsigned char type;
    pftp_sftp_attr_t attr;
    char *buffer = NULL;
    size_t buf_len = 0;

    if (ascii == 'A' && append_to > 0) {
	pftp_status_message(sftp->ftp, 
			    "SFTP: Can\'t append files in TEXT mode.");
	return -1;
    }

    attr = pftp_sftp_create_attr();

    if (append_to == 0) {
        handle = _fxp_open_file(sftp, filename,  SSH_FXF_CREAT | SSH_FXF_WRITE 
				| SSH_FXF_CREATE_TRUNCATE, 
				ascii, &handle_len, attr);
    } else {
        handle = _fxp_open_file(sftp, filename,  SSH_FXF_CREAT | SSH_FXF_WRITE, 
				ascii, &handle_len, attr);
    }
    pftp_sftp_free_attr(attr);
    if (!handle)
	return -1;    

    offset = append_to;

    pftp_status_filetransfer(sftp->ftp, append_to, total_size);
    speed = pftp_init_speed(total_size, append_to,
			    sftp->ftp->lastgoodbuffersize);
    pftp_start_wait(sftp->ftp, UPLOAD, 1);

    for (;;) {
	if (buf_len != speed->buffersize) {
	    buffer = realloc(buffer, buf_len = speed->buffersize);
	}
	
	if ((ret_size = fread(buffer, 1, speed->buffersize, fh)) < 0) {
	    pftp_status_message(sftp->ftp, 
				"ERR: Unable to read from file: %s", 
				strerror(errno));
	    pftp_free_speed(&speed, &sftp->ftp->lastgoodbuffersize);
	    pftp_end_wait(sftp->ftp, UPLOAD, 1);
	    _close_fxp_handle(sftp, handle, handle_len);
	    free(buffer);
	    return -1;
	}

	if (ret_size == 0) {
	    /* Done reading */
	    break;
	}

	buf = pftp_ssh_create_buf();
	pftp_ssh_buf_put_string2(buf, handle, handle_len);
	pftp_ssh_buf_put_uint64(buf, offset);
	pftp_ssh_buf_put_string2(buf, buffer, ret_size);

	if (_send_packet(sftp, &id, SSH_FXP_WRITE, buf, NULL)) {
	    pftp_ssh_buf_free(buf);
	    pftp_free_speed(&speed, &sftp->ftp->lastgoodbuffersize);
	    pftp_end_wait(sftp->ftp, UPLOAD, 1);
	    _close_fxp_handle(sftp, handle, handle_len);
	    free(buffer);
	    return -1;
	}

	pftp_ssh_buf_free(buf);
	buf = pftp_ssh_create_buf();

	if (_get_packet(sftp, id, &type, buf, speed)) {
	    pftp_ssh_buf_free(buf);
	    pftp_free_speed(&speed, &sftp->ftp->lastgoodbuffersize);
	    pftp_end_wait(sftp->ftp, UPLOAD, 1);
	    _close_fxp_handle(sftp, handle, handle_len);
	    free(buffer);
	    return -1;
	}

	if (type == SSH_FXP_STATUS) {
	    uint32_t err_code;
	    int err_valid;
	    char *err_msg;
	    
	    err_code = pftp_ssh_buf_get_uint32(buf, &err_valid);
	    err_msg = pftp_ssh_buf_get_string(buf);
	    
	    if (err_valid) {
		if (err_code == SSH_FX_OK) {
		    /* Everything is peachy */
		    if (err_msg) free(err_msg);		    
		} else {
		    if (err_msg) {		    
			pftp_parse_unicode_utf8(err_msg);
			pftp_status_message(sftp->ftp, "599 %s (%lu)",
					    err_msg, (unsigned long)err_code);
			free(err_msg);
			pftp_ssh_buf_free(buf);
			pftp_free_speed(&speed, &sftp->ftp->lastgoodbuffersize);
			pftp_end_wait(sftp->ftp, UPLOAD, 1);
			_close_fxp_handle(sftp, handle, handle_len);
			free(buffer);
			return -1;	    
		    } else {
			pftp_status_message(sftp->ftp, "599 %s (%lu)",
					    "Error writing file", 
					    (unsigned long)err_code);
			pftp_ssh_buf_free(buf);
			pftp_free_speed(&speed, &sftp->ftp->lastgoodbuffersize);
			pftp_end_wait(sftp->ftp, UPLOAD, 1);
			_close_fxp_handle(sftp, handle, handle_len);
			free(buffer);
			return -1;	    
		    }
		}
	    } else {
		pftp_status_message(sftp->ftp, "599 Invalid status packet.");
		pftp_ssh_buf_free(buf);
		pftp_free_speed(&speed, &sftp->ftp->lastgoodbuffersize);
		pftp_end_wait(sftp->ftp, UPLOAD, 1);
		_close_fxp_handle(sftp, handle, handle_len);
		free(buffer);
		return -1;	    
	    }	    
	} else {
	    pftp_status_message(sftp->ftp,
				"SFTP: Unexpected response to "
				"SSH_FXP_WRITE (%u).", type);	    
	    pftp_ssh_buf_free(buf);
	    pftp_free_speed(&speed, &sftp->ftp->lastgoodbuffersize);
	    pftp_end_wait(sftp->ftp, UPLOAD, 1);
	    _close_fxp_handle(sftp, handle, handle_len);
	    free(buffer);
	    return -1;
	}

	pftp_update_speed(speed, ret_size);

	if (pftp_do_wait(sftp->ftp, speed, 1, "UPLOAD-DATA") == -1) {
	    pftp_free_speed(&speed, &sftp->ftp->lastgoodbuffersize);
	    pftp_end_wait(sftp->ftp, UPLOAD, 1);
	    _close_fxp_handle(sftp, handle, handle_len);
	    free(buffer);
	    return -1;
	}
    }

    if (buffer)
	free(buffer);

    pftp_free_speed(&speed, &sftp->ftp->lastgoodbuffersize);
    pftp_end_wait(sftp->ftp, UPLOAD, 0);

    _close_fxp_handle(sftp, handle, handle_len);

    return 0;
}

/* Remove a file in current directory 
   Return 0 on OK and -1 on error. */
int pftp_sftp_rm(pftp_sftp_t sftp, const char *filename)
{
    pftp_ssh_buf_t buf;
    char *path;
    uint32_t id;
    unsigned char type;

    path = _make_fullpath(sftp, filename);

    if (sftp->version > 3) {
	pftp_make_unicode_utf8(&path);
    }

    buf = pftp_ssh_create_buf();
    pftp_ssh_buf_put_string(buf, path);
    free(path);

    if (_send_packet(sftp, &id, SSH_FXP_REMOVE, buf, NULL)) {
	pftp_ssh_buf_free(buf);
	return -1;
    }
    pftp_ssh_buf_free(buf);
    buf = pftp_ssh_create_buf();

    if (_get_packet(sftp, id, &type, buf, NULL)) {
	pftp_ssh_buf_free(buf);
	return -1;
    }

    if (type == SSH_FXP_STATUS) {
	if (_fxp_status(sftp, buf, "Unable to remove file")) {
	    pftp_ssh_buf_free(buf);
	    return -1;
	} else {
	    pftp_ssh_buf_free(buf);
	    return 0;
	}
    } else {
	pftp_status_message(sftp->ftp,
			    "SFTP: Unexpected response to "
			    "SSH_FXP_REMOVE (%u).", type);
	pftp_ssh_buf_free(buf);       
	return -1;
    }
}

/* Rename a file from filename to new in current directory
   Return 0 on OK and -1 on error. */
int pftp_sftp_rename(pftp_sftp_t sftp, const char *filename, const char *new)
{
    pftp_ssh_buf_t buf;
    char *oldpath, *newpath;
    uint32_t id;
    unsigned char type;

    if (sftp->version < 2) {
	pftp_status_message(sftp->ftp, 
			    "SFTP: Rename is not supported in "
			    "versions below 2.");
	return -1;
    }

    oldpath = _make_fullpath(sftp, filename);
    newpath = _make_fullpath(sftp, new);

    if (sftp->version > 3) {
	pftp_make_unicode_utf8(&oldpath);
	pftp_make_unicode_utf8(&newpath);
    }

    buf = pftp_ssh_create_buf();
    pftp_ssh_buf_put_string(buf, oldpath);
    pftp_ssh_buf_put_string(buf, newpath);
    if (sftp->version > 4) {
	/* Flags */
	pftp_ssh_buf_put_uint32(buf, 0);
    }
    free(oldpath);
    free(newpath);

    if (_send_packet(sftp, &id, SSH_FXP_RENAME, buf, NULL)) {
	pftp_ssh_buf_free(buf);
	return -1;
    }
    pftp_ssh_buf_free(buf);
    buf = pftp_ssh_create_buf();

    if (_get_packet(sftp, id, &type, buf, NULL)) {
	pftp_ssh_buf_free(buf);
	return -1;
    }

    if (type == SSH_FXP_STATUS) {
	if (_fxp_status(sftp, buf, "Unable to rename file/directory")) {
	    pftp_ssh_buf_free(buf);
	    return -1;
	} else {
	    pftp_ssh_buf_free(buf);
	    return 0;
	}
    } else {
	pftp_status_message(sftp->ftp,
			    "SFTP: Unexpected response to "
			    "SSH_FXP_RENAME (%u).", type);
	pftp_ssh_buf_free(buf);       
	return -1;
    }
}

/* Set access to a file on SFTP server. 
   Access should be between 000 and 777 in octal.
   Returns -1 on error and 0 on OK. */
int pftp_sftp_chmod(pftp_sftp_t sftp, const char *file, unsigned short access)
{
    pftp_sftp_attr_t attr;
    pftp_ssh_buf_t buf;
    char *path;
    uint32_t id;
    unsigned char type;

    path = _make_fullpath(sftp, file);

    if (sftp->version > 3) {
	pftp_make_unicode_utf8(&path);
    }

    buf = pftp_ssh_create_buf();
    pftp_ssh_buf_put_string(buf, path);
    attr = pftp_sftp_create_attr();

    pftp_sftp_attr_set_permissions(attr, access);

    pftp_sftp_export_attr(sftp->version, buf, attr);
    pftp_sftp_free_attr(attr);
    free(path);

    if (_send_packet(sftp, &id, SSH_FXP_SETSTAT, buf, NULL)) {
	pftp_ssh_buf_free(buf);
	return -1;
    }
    pftp_ssh_buf_free(buf);
    buf = pftp_ssh_create_buf();

    if (_get_packet(sftp, id, &type, buf, NULL)) {
	pftp_ssh_buf_free(buf);
	return -1;
    }

    if (type == SSH_FXP_STATUS) {
	if (_fxp_status(sftp, buf, "Unable to set file/dir attributes")) {
	    pftp_ssh_buf_free(buf);
	    return -1;
	} else {
	    pftp_ssh_buf_free(buf);
	    return 0;
	}
    } else {
	pftp_status_message(sftp->ftp,
			    "SFTP: Unexpected response to "
			    "SSH_FXP_SETSTAT (%u).", type);
	pftp_ssh_buf_free(buf);       
	return -1;
    }
}

/* Set file modification date on SFTP server.
   Returns -1 on error and 0 on OK. */
int pftp_sftp_mdtm(pftp_sftp_t sftp, const char *file, const struct tm *date)
{
    pftp_sftp_attr_t attr;
    pftp_ssh_buf_t buf;
    char *path;
    uint32_t id;
    unsigned char type;

    path = _make_fullpath(sftp, file);

    if (sftp->version > 3) {
	pftp_make_unicode_utf8(&path);
    }

    buf = pftp_ssh_create_buf();
    pftp_ssh_buf_put_string(buf, path);
    attr = pftp_sftp_create_attr();

    pftp_sftp_attr_set_modificationtime(attr, date);

    pftp_sftp_export_attr(sftp->version, buf, attr);
    pftp_sftp_free_attr(attr);
    free(path);

    if (_send_packet(sftp, &id, SSH_FXP_SETSTAT, buf, NULL)) {
	pftp_ssh_buf_free(buf);
	return -1;
    }
    pftp_ssh_buf_free(buf);
    buf = pftp_ssh_create_buf();

    if (_get_packet(sftp, id, &type, buf, NULL)) {
	pftp_ssh_buf_free(buf);
	return -1;
    }

    if (type == SSH_FXP_STATUS) {
	if (_fxp_status(sftp, buf, "Unable to set file/dir attributes")) {
	    pftp_ssh_buf_free(buf);
	    return -1;
	} else {
	    pftp_ssh_buf_free(buf);
	    return 0;
	}
    } else {
	pftp_status_message(sftp->ftp,
			    "SFTP: Unexpected response to "
			    "SSH_FXP_SETSTAT (%u).", type);
	pftp_ssh_buf_free(buf);       
	return -1;
    }
}


void _free_sftp_packet(sftp_packet_t pkg)
{
    if (pkg) {
	if (pkg->data)
	    free(pkg->data);
	free(pkg);
    }
}

int _send_fxp_init(pftp_sftp_t sftp)
{
    pftp_ssh_buf_t buf;
    int ret;
    buf = pftp_ssh_create_buf();
    /*
     * On second thougth, not very informative in v3 either.
     * Only change is modification time..., which only is missing
     * for directories in v4.
     *

    if (strncmp(pftp_ssh_get_server_str(sftp->ssh), 
		"Foxit-WAC-Server-", 17) == 0) {
	double ver;
	ver = strtod(pftp_ssh_get_server_str(sftp->ssh) + 17, NULL);
	if (ver <= 2.0) {
	    pftp_status_message(sftp->ftp, 
				"Forcing version 3 as Foxit WAC version 4"
				" isn't very informative.");
	    pftp_ssh_buf_put_uint32(buf, 3);
	}
    } else {
    */
    pftp_ssh_buf_put_uint32(buf, CLIENT_SFTP_VERSION);
    ret = _send_packet(sftp, NULL, SSH_FXP_INIT, buf, NULL);
    pftp_ssh_buf_free(buf);
    return ret;
}

int _get_fxp_version(pftp_sftp_t sftp)
{
    unsigned char type;
    pftp_ssh_buf_t buf;
    int valid, request_version = 0;
    uint32_t request_version_target = 0;

    if (sftp->newline) {
	free(sftp->newline);
	sftp->newline = NULL;
    }	

    buf = pftp_ssh_create_buf();
    if (_get_packet(sftp, -1, &type, buf, NULL)) {
	pftp_ssh_buf_free(buf);
	return -1;
    }
    if (type != SSH_FXP_VERSION) {
	pftp_status_message(sftp->ftp, "SFTP: VERSION not first packet (%d).",
			    type);
	pftp_ssh_buf_free(buf);
	return -1;
    }
    sftp->version = pftp_ssh_buf_get_uint32(buf, &valid);
    if (!valid) {
	pftp_status_message(sftp->ftp, "SFTP: Invalid VERSION packet (1).");
	pftp_ssh_buf_free(buf);
	return -1;
    }
    while (pftp_ssh_buf_data_left(buf) > 0) {
	char *name, *data;
	size_t data_len;
	name = pftp_ssh_buf_get_string(buf);
	data = pftp_ssh_buf_get_string2(buf, &data_len);
	if (!name || !data) {
	    pftp_status_message(sftp->ftp, "SFTP: Invalid VERSION packet (2).");
	    if (name) free(name);
	    if (data) free(data);
	    pftp_ssh_buf_free(buf);
	    return -1;
	}

	if (strcmp(name, "newline") == 0) {
	    if (sftp->newline) {
		free(sftp->newline);
	    }
	    sftp->newline = strndup(data, data_len);
	} else if (strcmp(name, "newline@vandyke.com") == 0) {
	    if (!sftp->newline) {
		pftp_ssh_buf_t buf;
		char *data2;
		buf = pftp_ssh_attach_buf(data, data_len);
		data2 = pftp_ssh_buf_get_string(buf);
		pftp_ssh_buf_free(buf);
		if (data2) {
		    sftp->newline = strdup(data2);
		    free(data2);
		}
	    }
	} else if (strcmp(name, "vendor-id") == 0) {
	    pftp_ssh_buf_t buf;
	    char *vendor_name, *product_name, *product_version;
	    uint64_t product_build;
	    int valid;

	    buf = pftp_ssh_attach_buf(data, data_len);
	    
	    vendor_name = pftp_ssh_buf_get_string(buf);
	    product_name = pftp_ssh_buf_get_string(buf);
	    product_version = pftp_ssh_buf_get_string(buf);
	    product_build = pftp_ssh_buf_get_uint64(buf, &valid);

	    pftp_ssh_buf_free(buf);

	    if (!vendor_name || !product_name || !product_version || !valid) {
		pftp_status_message(sftp->ftp, 
				    "SFTP: Invalid vendor-id structure.");
		free(name);
		free(data);
		return -1;
	    }

	    pftp_status_message(sftp->ftp, "SFTP: %s %s %s (%llu)",
				vendor_name, product_name, product_version,
				product_build);
	    free(vendor_name);
	    free(product_name);
	    free(product_version);
	} else if (strcmp(name, "supported2") == 0) {
	    pftp_ssh_buf_t buf;
	    uint32_t supported_attribute_mask;
	    uint32_t supported_attribute_bits;
	    uint32_t supported_open_flags;
	    uint32_t supported_access_mask;
	    uint32_t max_read_size;
	    uint16_t supported_open_block_masks;
	    uint16_t supported_block_masks;
	    uint32_t attrib_extension_count = 0;
	    char **attrib_extension_name = NULL;
	    uint32_t extension_count = 0;
	    char **extension_name = NULL;
	    uint32_t c;
	    int valid;

	    buf = pftp_ssh_attach_buf(data, data_len);

	    supported_attribute_mask = pftp_ssh_buf_get_uint32(buf, &valid);
	    if (!valid) goto invalid_support2;
	    supported_attribute_bits = pftp_ssh_buf_get_uint32(buf, &valid);
	    if (!valid) goto invalid_support2;
	    supported_open_flags = pftp_ssh_buf_get_uint32(buf, &valid);
	    if (!valid) goto invalid_support2;
	    supported_access_mask = pftp_ssh_buf_get_uint32(buf, &valid);
	    if (!valid) goto invalid_support2;
	    max_read_size = pftp_ssh_buf_get_uint32(buf, &valid);
	    if (!valid) goto invalid_support2;
	    supported_open_block_masks = pftp_ssh_buf_get_uint16(buf, &valid);
	    if (!valid) goto invalid_support2;
	    supported_block_masks = pftp_ssh_buf_get_uint16(buf, &valid);
	    if (!valid) goto invalid_support2;
	    attrib_extension_count = pftp_ssh_buf_get_uint32(buf, &valid);
	    if (!valid) goto invalid_support2;
	    attrib_extension_name = malloc(attrib_extension_count 
					   * sizeof(char *));
	    memset(attrib_extension_name, 0, attrib_extension_count 
		   * sizeof(char *));
	    for (c = 0; c < attrib_extension_count; c++) {
		attrib_extension_name[c] = pftp_ssh_buf_get_string(buf);
		if (!attrib_extension_name[c]) 
		    goto invalid_support2;
	    }
	    extension_count = pftp_ssh_buf_get_uint32(buf, &valid);
	    if (!valid) goto invalid_support2;
	    extension_name = malloc(extension_count * sizeof(char *));
	    memset(extension_name, 0, extension_count * sizeof(char *));
	    for (c = 0; c < extension_count; c++) {
		extension_name[c] = pftp_ssh_buf_get_string(buf);
		if (!extension_name[c]) 
		    goto invalid_support2;
	    }

	    goto free_support2;

	    /* !!! Should probably do something with all this info? */
	invalid_support2:
	    pftp_status_message(sftp->ftp, "SFTP: Invalid support2 data."
				" Not fatal tho.");

	free_support2:
	    for (c = 0; c < extension_count; c++)
		if (extension_name[c]) 
		    free(extension_name[c]);
	    if (extension_name) free(extension_name);
	    for (c = 0; c < attrib_extension_count; c++)
		if (attrib_extension_name[c]) 
		    free(attrib_extension_name[c]);
	    if (attrib_extension_name) free(attrib_extension_name);

	    pftp_ssh_buf_free(buf);
	} else if (strcmp(name, "supported") == 0) {
	    pftp_ssh_buf_t buf;
	    uint32_t supported_attribute_mask;
	    uint32_t supported_attribute_bits;
	    uint32_t supported_open_flags;
	    uint32_t supported_access_mask;
	    uint32_t max_read_size;
	    uint32_t extension_count = 0;
	    char **extension_name = NULL;
	    uint32_t c;
	    int valid;

	    buf = pftp_ssh_attach_buf(data, data_len);

	    supported_attribute_mask = pftp_ssh_buf_get_uint32(buf, &valid);
	    if (!valid) goto invalid_support;
	    supported_attribute_bits = pftp_ssh_buf_get_uint32(buf, &valid);
	    if (!valid) goto invalid_support;
	    supported_open_flags = pftp_ssh_buf_get_uint32(buf, &valid);
	    if (!valid) goto invalid_support;
	    supported_access_mask = pftp_ssh_buf_get_uint32(buf, &valid);
	    if (!valid) goto invalid_support;
	    max_read_size = pftp_ssh_buf_get_uint32(buf, &valid);
	    if (!valid) goto invalid_support;

	    while (1) {
		if (pftp_ssh_buf_data_left(buf) == 0)
		    break;
		extension_name = realloc(extension_name, 
					 (extension_count + 1) 
					 * sizeof(char *));
		extension_name[extension_count] = pftp_ssh_buf_get_string(buf);
		if (!extension_name[extension_count]) 
		    goto invalid_support;
		extension_count++;
	    }

	    goto free_support;

	    /* !!! Should probably do something with all this info? */

	invalid_support:
	    pftp_status_message(sftp->ftp, "SFTP: Invalid support data."
				" Not fatal tho.");

	free_support:
	    for (c = 0; c < extension_count; c++)
		if (extension_name[c]) 
		    free(extension_name[c]);
	    if (extension_name) free(extension_name);

	    pftp_ssh_buf_free(buf);
	} else if (strcmp(name, "versions") == 0) {
	    char **version = NULL;
	    size_t versions = 0;

	    if (data_len > 0) {
		if (data[0] == '\0') {
		    /* As far as I can read the specs, this *SHOULD* be a
		     * comma-seperated string. Not a string in a string but
		     * atleast VanDyke does it like this. */
		    pftp_ssh_buf_t buf;
		    char *data2;
		    buf = pftp_ssh_attach_buf(data, data_len);
		    data2 = pftp_ssh_buf_get_string(buf);
		    pftp_ssh_buf_free(buf);
		    if (data2) {
			versions = splitchar(data2, ',', &version);
			free(data2);
		    }		    
		} else {
		    versions = splitchar(data, ',', &version);
		}
	    }

	    if (versions > 0) {
		size_t v;
		for (v = 0; v < versions; v++) {
		    unsigned long ver;
		    char *end = NULL;
		    errno = 0;
		    ver = strtoul(version[v], &end, 10);
		    if (errno == 0 && end && !end[0]) {
			if (ver > sftp->version && ver <= CLIENT_SFTP_VERSION) {
			    request_version = 1;
			    request_version_target = ver;
			}
		    }
		}

		freesplit(version, versions);
	    }	    
	} else if (strcmp(name, "filename-charset") == 0) {
	    pftp_status_message(sftp->ftp, 
				"SFTP: Server is using charset %s for"
				" filenames.", data);
	} else if (strcmp(name, "fs-roots@vandyke.com") == 0) {
	    /* Is string after string telling us what root drives there are on
	     * a win32 machine. Maybe do something about it some day?
	     * Ending with a empty string btw.
	     */
	} else if (strcmp(name, "default-fs-attribs@vandyke.com") == 0) {
	    /*
	     * This is what I've parsed. Nothing really usefull here?
	     *

	      00 . 00 . 00 . 01 . Strings to follow
	      00   00 . 00 . 09 . " \/:*?"<>|"         // disallowed chars?
	      00 . 00 . 00 . 17   Strings to follow
	      00 . 00 . 00 . 04 . COM1 
	      ...
	      00 . 00 . 00 . 04 . COM9
	      00 . 00 . 00 . 04 . LPT1
	      ...
	      00 . 00 . 00 . 04 . LPT9
	      00 . 00 . 00 . 03 . PRN
	      00 . 00 . 00 . 03 . CON
	      00 . 00 . 00 . 03 . AUX 
	      00 . 00 . 00 . 03 . NUL
	      00 . 00 . 00 . 06 . CLOCK$ 
	    */
	} else {
#ifdef DEBUG
	    fprintf(stderr, "SFTP: Unknown extension `%s': `%s' (%lu bytes).\n",
		    name, data, (unsigned long)data_len);
	    {
		size_t b;
		for (b = 0; b < data_len; b++) {
		    fprintf(stderr, "%02x %c ", ((unsigned char *)data)[b],
			    data[b] < ' ' ? '.' : data[b]);

		    if (b > 0 && b % 16 == 0)
			fprintf(stderr, "\n");
		}
		fprintf(stderr, "\n");
	    }
#endif
	}

	free(name);
	free(data);
    }

    if (!sftp->newline) {
	sftp->newline = strdup("\r\n");
    }

    pftp_ssh_buf_free(buf);

    if (request_version) {
	/**
	 * The ONLY server I've seen doing this is VanDyke 2.5.0 BETA (b 160)
	 * And that same server doesn't allow version 6 (as it states).	 
	 */

	/*

	pftp_ssh_buf_t buf;
	uint32_t id;
	unsigned char type;
	char tmp[20];
#ifdef DEBUG
	fprintf(stderr, "SFTP: Strange server, setting version %lu"
		" but supporting version %lu.\n",
		(unsigned long)sftp->version, 
		(unsigned long)request_version_target);
#endif

	snprintf(tmp, 20, "%lu", (unsigned long)request_version_target);
	buf = pftp_ssh_create_buf();
	pftp_ssh_buf_put_string(buf, "version-select");
	pftp_ssh_buf_put_string(buf, tmp);

	if (_send_packet(sftp, &id, SSH_FXP_EXTENDED, buf, NULL)) {
	    pftp_ssh_buf_free(buf);
	    return -1;
	}

	pftp_ssh_buf_free(buf);
	buf = pftp_ssh_create_buf();

	if (_get_packet(sftp, id, &type, buf, NULL)) {
	    pftp_ssh_buf_free(buf);
	    return -1;
	}

	if (type == SSH_FXP_STATUS) {
	    if (_fxp_status(sftp, buf, "Error requesting version")) {
		pftp_ssh_buf_free(buf);
		return -1;
	    }
	    
	    sftp->version = request_version_target;
	    pftp_ssh_buf_free(buf);

	    pftp_status_message(sftp->ftp,
				"SFTP: Upgraded to version %lu.",
				(unsigned long)sftp->version);
	} else {
	    pftp_status_message(sftp->ftp,
				"SFTP: Unexpected response to "
				"SSH_FXP_EXTENDED (version-select) (%u).", 
				type);
	    pftp_ssh_buf_free(buf);
	    return -1;	    
	}

	*/
    }

    return 0;
}

int _send_packet(pftp_sftp_t sftp, uint32_t *req_id, unsigned char type, 
		 pftp_ssh_buf_t buf, speed_data_t speed)
{
    pftp_ssh_buf_t newbuf;
    size_t buf_size;
    int ret;

    buf_size = buf ? pftp_ssh_buf_size(buf) : 0;
    
    newbuf = pftp_ssh_create_buf();

    if (type != SSH_FXP_INIT) {
	pftp_ssh_buf_put_uint32(newbuf, (uint32_t) (1 + 4 + buf_size));
    } else {
	pftp_ssh_buf_put_uint32(newbuf, (uint32_t) (1 + buf_size));
    }
    
    pftp_ssh_buf_put_char(newbuf, type);
    
    if (type != SSH_FXP_INIT) {
	*req_id = sftp->req_id++;
	pftp_ssh_buf_put_uint32(newbuf, *req_id);
    }

    if (buf_size)
	pftp_ssh_buf_put_raw(newbuf, pftp_ssh_buf_raw(buf), buf_size);

    ret = pftp_ssh_channel_write(sftp->main, pftp_ssh_buf_raw(newbuf),
				 pftp_ssh_buf_size(newbuf), speed);

    pftp_ssh_buf_free(newbuf);

    return ret;
}

int _get_packet(pftp_sftp_t sftp, uint32_t req_id, unsigned char *type, 
		pftp_ssh_buf_t buf, speed_data_t speed)
{
    size_t p;
    /* Check in packet buffer first */
    for (p = 0; p < sftp->in_packets; p++) {
	if (sftp->in_packet[p]->req_id == req_id) {
	    *type = sftp->in_packet[p]->type;
	    if (sftp->in_packet[p]->len > 0)
		pftp_ssh_buf_put_raw(buf, sftp->in_packet[p]->data, 
				     sftp->in_packet[p]->len);
	    _free_sftp_packet(sftp->in_packet[p]);
	    sftp->in_packets--;
	    memmove(sftp->in_packet + p, sftp->in_packet + p + 1,
		    (sftp->in_packets - p) * sizeof(sftp_packet_t));
	    return 0;
	}
    }

    for (;;) {
	unsigned char _type;
	char *_data;
	uint32_t _len, _req_id;
	
	if (pftp_ssh_channel_read(sftp->main, &_len, 4, speed))
	    return -1;
	_len = ntohl(_len);
	_data = malloc(_len);	

	if (pftp_ssh_channel_read(sftp->main, _data, _len, speed)) {
	    free(_data);
	    return -1;
	}

	_type = _data[0];
	if (_type != SSH_FXP_VERSION) {
	    memcpy(&_req_id, _data + 1, 4);
	    _req_id = ntohl(_req_id);
	    if (_req_id != req_id) {
		sftp_packet_t _packet;

		memmove(_data, _data + 5, _len - 5);
		_data = realloc(_data, _len - 5);

		_packet = malloc(sizeof(sftp_packet_s));
		_packet->type = _type;
		_packet->req_id = _req_id;
		_packet->data = _data;
		_packet->len = _len - 5;

		sftp->in_packet = realloc(sftp->in_packet, 
					  (sftp->in_packets + 1) 
					  * sizeof(sftp_packet_t));
		sftp->in_packet[sftp->in_packets++] = _packet;

		continue;
	    }
	}

	*type = _type;
	if (_type != SSH_FXP_VERSION) {
	    if (_len > 5)
		pftp_ssh_buf_put_raw(buf, _data + 5, _len - 5);
	} else {
	    if (_len > 1)
		pftp_ssh_buf_put_raw(buf, _data + 1, _len - 1);
	}
	free(_data);
	return 0;
    }
}

int _get_pwd(pftp_sftp_t sftp)
{
    pftp_ssh_buf_t buf;
    uint32_t id;
    unsigned char type;
    buf = pftp_ssh_create_buf();
    pftp_ssh_buf_put_string(buf, ".");

    if (_send_packet(sftp, &id, SSH_FXP_REALPATH, buf, NULL)) {
	pftp_ssh_buf_free(buf);
	return -1;
    }

    pftp_ssh_buf_free(buf);
    buf = pftp_ssh_create_buf();

    if (_get_packet(sftp, id, &type, buf, NULL)) {
	pftp_ssh_buf_free(buf);
	return -1;
    }
	
    if (type == SSH_FXP_NAME) {
	uint32_t cnt;
	int valid;
	char *path;
	cnt = pftp_ssh_buf_get_uint32(buf, &valid);
	path = pftp_ssh_buf_get_string(buf);

	if (cnt == 0 || !valid || !path) {
	    pftp_status_message(sftp->ftp,
				"SFTP: Invalid SSH_FXP_NAME packet.");
	    if (path) free(path);
	    pftp_ssh_buf_free(buf);
	    return -1;
	}

	pftp_parse_unicode_utf8(path);

	if (sftp->cwd)
	    free(sftp->cwd);
	sftp->cwd = strdup(path);
	free(path);
	pftp_ssh_buf_free(buf);
	return 0;
    } else {
	pftp_status_message(sftp->ftp,
			    "SFTP: Unexpected response to "
			    "SSH_FXP_REALPATH (%u).", type);
	pftp_ssh_buf_free(buf);
	return -1;
    }
}

int _fxp_status(pftp_sftp_t sftp, pftp_ssh_buf_t buf, const char *def)
{
    uint32_t err_code;
    int err_valid;
    char *err_msg;
    
    err_code = pftp_ssh_buf_get_uint32(buf, &err_valid);
    err_msg = pftp_ssh_buf_get_string(buf);
    
    if (err_valid) {
	if (err_code == SSH_FX_OK)
	    return 0;

	if (err_msg) {
	    pftp_parse_unicode_utf8(err_msg);
	    pftp_status_message(sftp->ftp, "599 %s (%lu)",
				err_msg, (unsigned long)err_code);
	    free(err_msg);
	} else {
	    pftp_status_message(sftp->ftp, "599 %s (%lu)",
				def ? def : "", (unsigned long)err_code);
	}
    } else {
	pftp_status_message(sftp->ftp, "599 Invalid status packet.");
    }

    return -1;
}

void _close_fxp_handle(pftp_sftp_t sftp, char *handle, size_t handle_len)
{
    uint32_t id;
    pftp_ssh_buf_t buf;
    unsigned char type;
    buf = pftp_ssh_create_buf();
    pftp_ssh_buf_put_string2(buf, handle, handle_len);
    if (_send_packet(sftp, &id, SSH_FXP_CLOSE, buf, NULL)) {
	pftp_ssh_buf_free(buf);
	free(handle);
	return;
    }
    pftp_ssh_buf_free(buf);
    buf = pftp_ssh_create_buf();
    if (_get_packet(sftp, id, &type, buf, NULL)) {
	pftp_ssh_buf_free(buf);
	free(handle);
	return;
    }

    /* Ignore response */

    pftp_ssh_buf_free(buf);
    free(handle);
}

char *_make_fullpath(pftp_sftp_t sftp, const char *filename)
{
    char *ret;
    if (sftp->cwd) {
	size_t f_len, c_len;
	c_len = strlen(sftp->cwd);
	f_len = strlen(filename);
	ret = malloc(c_len + 1 + f_len + 1);
	memcpy(ret, sftp->cwd, c_len);
	ret[c_len] = '/';
	memcpy(ret + c_len + 1, filename, f_len + 1);
    } else {
	assert(0);
	ret = strdup(filename);
    }

    return ret;
}

void _fill_pftp_file_t(pftp_sftp_t sftp, pftp_file_t *file, 
		       const char *filename, pftp_sftp_attr_t attr)
{
    memset(file, 0, sizeof(pftp_file_t));
    file->name = strdup(filename);
    pftp_sftp_convert_attr(file, attr);
    if (file->link && file->type == pft_unknown) {
	pftp_ssh_buf_t buf;
	char *path;
	unsigned char type;
	uint32_t id;
	path = _make_fullpath(sftp, filename);
	if (sftp->version > 3) {
	    pftp_make_unicode_utf8(&path);
	}
	buf = pftp_ssh_create_buf();
	pftp_ssh_buf_put_string(buf, path);
	if (sftp->version > 3) {
	    /* !!! Better flags ? */
	    pftp_ssh_buf_put_uint32(buf, 0);
	}

	free(path);
    
	if (_send_packet(sftp, &id, SSH_FXP_STAT, buf, NULL)) {
	    pftp_ssh_buf_free(buf);
	    return;
	}
	
	pftp_ssh_buf_free(buf);
	buf = pftp_ssh_create_buf();

	if (_get_packet(sftp, id, &type, buf, NULL)) {
	    pftp_ssh_buf_free(buf);
	    return;
	}

	if (type == SSH_FXP_ATTRS) {
	    pftp_sftp_attr_t attr;
	    attr = pftp_sftp_parse_attr(sftp->version, 
					pftp_ssh_get_server_str(sftp->ssh),
					buf, NULL);
	    if (!attr) {
		pftp_ssh_buf_free(buf);
		return;
	    }

	    file->type = pftp_sftp_convert_attr_type(attr);

	    pftp_sftp_free_attr(attr);
	    pftp_ssh_buf_free(buf);
	} else if (type == SSH_FXP_STATUS) {
	    pftp_ssh_buf_free(buf);
	    return;
	} else {
	    pftp_ssh_buf_free(buf);
	    return;
	}	
    }
}

static int _file_cmp(const void *f1, const void *f2)
{
    return strcmp(((const pftp_file_t *)f1)->name, 
		  ((const pftp_file_t *)f2)->name);
}

void _sort_ftp_directory(pftp_directory_t *dir)
{
    /* sort by name, case sensitive. */

    qsort(dir->files, dir->length, sizeof(pftp_file_t), _file_cmp);
}

char *_fxp_open_file(pftp_sftp_t sftp, const char *filename, uint32_t flags, 
		     char ascii, size_t *handle_len, pftp_sftp_attr_t attr)
{
    pftp_ssh_buf_t buf;
    char *path, *ret = NULL;
    uint32_t id;
    unsigned char type;

    if (sftp->version <= 3) {
	if (ascii != 'B') {
	    pftp_status_message(sftp->ftp, 
				"SFTP: Text mode transfer not supported by "
				"server version.");
	    return NULL;
	}
    }

    buf = pftp_ssh_create_buf();

    path = _make_fullpath(sftp, filename);

    if (sftp->version > 3) {
	pftp_make_unicode_utf8(&path);
    }

    pftp_ssh_buf_put_string(buf, path);
    free(path);

    if (sftp->version <= 5) {
	if (ascii == 'A')
	    pftp_ssh_buf_put_uint32(buf, flags | SSH_FXF_TEXT);
	else
	    pftp_ssh_buf_put_uint32(buf, flags);
    } else {
	uint32_t new_flags = 0;
	uint32_t access = 0;

	if (flags & SSH_FXF_READ) {
	    access |= ACE4_READ_DATA | ACE4_READ_ATTRIBUTES;
	}
	if (flags & SSH_FXF_WRITE) {
	    access |= ACE4_WRITE_DATA | ACE4_WRITE_ATTRIBUTES;

	    if (flags & SSH_FXF_APPEND) {
		access |= ACE4_APPEND_DATA;
	    }
	}

	if (flags & SSH_FXF_CREAT) {
	    if (flags & SSH_FXF_EXCL) {
		new_flags |= SSH_FXF_CREATE_NEW;
	    } else {
		if (flags & SSH_FXF_TRUNC) {
		    new_flags |= SSH_FXF_CREATE_TRUNCATE;
		} else {
		    new_flags |= SSH_FXF_OPEN_OR_CREATE;
		}
	    }
	} else {
	    new_flags |= SSH_FXF_OPEN_EXISTING;
	}

	if (flags & SSH_FXF_APPEND) {
	    new_flags |= SSH_FXF_ACCESS_APPEND_DATA;
	}

	if (ascii == 'A') {
	    new_flags |= SSH_FXF_ACCESS_TEXT_MODE;
	}

	pftp_ssh_buf_put_uint32(buf, access);
	pftp_ssh_buf_put_uint32(buf, new_flags);
    }

    pftp_sftp_export_attr(sftp->version, buf, attr);

    if (_send_packet(sftp, &id, SSH_FXP_OPEN, buf, NULL)) {
	pftp_ssh_buf_free(buf);
	return NULL;
    }

    pftp_ssh_buf_free(buf);
    buf = pftp_ssh_create_buf();
    
    if (_get_packet(sftp, id, &type, buf, NULL)) {
	pftp_ssh_buf_free(buf);
	return NULL;
    }

    if (type == SSH_FXP_HANDLE) {
	ret = pftp_ssh_buf_get_string2(buf, handle_len);
	pftp_ssh_buf_free(buf);
	if (!ret) {
	    pftp_status_message(sftp->ftp, 
				"SFTP: Invalid handle packet.");
	    return NULL;
	}

	return ret;
    } else if (type == SSH_FXP_STATUS) {
	_fxp_status(sftp, buf, "Unable to open file");
	pftp_ssh_buf_free(buf);
	return NULL;
    } else {
	pftp_status_message(sftp->ftp, 
			    "SFTP: Unexpected response to OPEN (%u).",
			    type);
	pftp_ssh_buf_free(buf);
	return NULL;
    }    
}

void freesplit(char **part, size_t parts)
{
    size_t c;
    
    if (part != NULL) {
	for (c = 0; c < parts; c++)
	    if (part[c])
		free(part[c]);
	
	free(part);
    }
}

size_t splitchar(const char *str, char needle, char ***part)
{
    size_t parts = 0;
    char *pos;
    const char *start = str;

    assert(str && part);

    while ((pos = strchr(start, needle))) {
	*part = realloc(*part, (parts + 1) * sizeof(char *));	
	(*part)[parts++] = strndup(start, pos - start);
	start = pos;
	while (*(++start) == needle);
    }

    if (*start) {
	*part = realloc(*part, (parts + 1) * sizeof(char *));
	(*part)[parts++] = strdup(start);
    }

    return parts;
}


#endif /* NO_SSL */



