/* -*- pftp-c -*- */
/**************************************************************
 * Parses a directory listing into an internal representation.*
 * Packaged with libftp and released under GPL, see LICENSE.  *
 **************************************************************/
#ifdef HAVE_CONFIG_H
# include "config.h"
#endif
#if HAVE_STDLIB_H
# include <stdlib.h>
#endif
#if HAVE_STDIO_H
# include <stdio.h>
#endif
#if HAVE_UNISTD_H
# include <unistd.h>
#endif
#if HAVE_STRING_H
# include <string.h>
#endif
#if TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif
#if HAVE_INTTYPES_H
# include <inttypes.h>
#endif

#ifdef DEBUG
# include <assert.h>
#else
# define assert(x) // x
#endif

#ifdef WITH_DMALLOC
# include <dmalloc.h>
#endif

#include "pftp_directory.h"

/* The three first powers of 8. */
static const int EXP8[3] = {1, 8, 64};

/* Global used by get_line_type so that I doesn\'t have to download the whole
   site hopefully */
static int pftp_in_get_line_type = 0;

/* Counts occurrences of a character in a string. */
static int chrcnt(const char *str, char c)
{
    int num = 0;
    char *str2;
    memcpy(&str2, &str, sizeof(char *));
    while ((str2 = strchr(str2, c)) != NULL) {
	str2++;
	num++;
    }
    return num;
}

static int total_row(const char *str)
{
    if (strncmp(str, "total", 5) == 0)
	return 1;
    return 0;
}


/* Moves along pos until there is no longer whitespace there. */
static void mvwhite(const char *output, int *pos)
{
    if (output[*pos] == '\0')
	return;
    while (output[*pos] == ' ')
	(*pos)++;
}

/* Move past a newline */
static void mvnl(const char *output, int *pos)
{
    if (output[*pos] == '\0')
	return;
    while (output[*pos] == '\n' || output[*pos] == '\r')
	(*pos)++;
}

/* Skip row */
static void mvrow(const char *str, int *pos)
{
    while(str[*pos] != '\n' && str[*pos] != '\r')
	(*pos)++;
    mvnl(str, pos);
}

/* The length of non-whitespace followed by pos. */
static unsigned int nwhitelen(const char *output, int pos)
{
    int length = 0;
    if (output[pos] == '\0')
	return 0;
    while (output[pos] != ' ' && output[pos] != '\0') {
	pos++;
	length++;
    }
    return length;
}

/* The length of non-end-of-line followed by pos. */
static unsigned int neollen(const char *output, int pos)
{
    int length = 0;
    while (output[pos] != '\n' && output[pos] != '\r' &&
	   output[pos] != '\0') {
	pos++;
	length++;
    }
    return length;
}

/* Returns what type a certain file is (eg. directory, link) */
static pftp_file_type_t file_type(const char *output, int *pos, int *link)
{
    pftp_file_type_t ret;
    (*link) = 0;
    if (output[*pos] == '\0')
	return pft_unknown;
    switch (output[*pos]) {
    case 'd':
	ret = pft_directory;
	break;
    case 'D':
	ret = pft_door;
	break;
    case 'l': {
	ret = pft_unknown;
	(*link) = 1;
    }; break;
    case 'b':
	ret = pft_block;
	break;
    case 'c':
	ret = pft_character;
	break;
    case 'p':
	ret = pft_fifo;
	break;
    case 's':
	ret = pft_s_socket;
	break;
    case '-':
	ret = pft_file;
	break;
    default:
	ret = pft_unknown;
	break;
    }
    (*pos)++;
    return ret;
}

/* Returns the permissions of a file in octal format. */
static unsigned short file_perm(const char *output, int *pos)
{
    unsigned short perm = 0;
    int i;
    if (output[*pos] == '\0')
	return perm;
    for (i = 0; i < 3; i++) {
	if (output[(*pos) + i * 3] == 'r')
	    perm += 4*EXP8[2-i];
	if (output[(*pos) + (i * 3) + 1] == 'w')
	    perm += 2*EXP8[2-i];
	if (output[(*pos) + (i * 3) + 2] == 'x')
	    perm += EXP8[2-i];
    }
    (*pos) += 9;
    return perm;
}

/* Returns the user or group, depending on pos. */
static char *file_ug(const char *output, int *pos)
{
    int length;
    char *ret;
    if (output[*pos] == '\0') {
	ret = malloc(2);
	strcpy(ret, "");
	return ret;
    }
    mvwhite(output, pos);
    length = nwhitelen(output, *pos);
    ret = malloc(length + 1);
    strncpy(ret, output + *pos, length);
    ret[length] = '\0';
    (*pos) += length;
    return ret;
}

/* Returns the size of a file. */
static uint64_t file_size(const char *output, int *pos)
{
    int length;
    char *tmp;
    uint64_t ret;
    if (output[*pos] == '\0')
	return 0;
    mvwhite(output, pos);
    length = nwhitelen(output, *pos);
    tmp = malloc(length + 1);
    strncpy(tmp, output + *pos, length);
    tmp[length] = '\0';
    ret = strtoull(tmp, NULL, 10);
    free(tmp);
    (*pos) += length;
    return ret;
}

static uint64_t dosfile_size(const char *output, int *pos, 
			     pftp_file_type_t *type)
{
    int length;
    char *tmp;
    uint64_t ret;
    if (output[*pos] == '\0')
	return 0;
    mvwhite(output, pos);
    length = nwhitelen(output, (*pos));
    tmp = malloc(length + 1);
    strncpy(tmp, output + (*pos), length);
    tmp[length] = '\0';

    if (strcmp(tmp, "<DIR>") == 0) {
	(*type) = pft_directory;
	ret = 0;
	length += 9;
    } else {
	(*type) = pft_file;
	ret = strtoull(tmp, NULL, 10);
    }
    free(tmp);
    (*pos) += length;
    return ret;    
}

/* Returns when a file has been changed. */
static char *file_changed(const char *output, int *pos)
{
    char *ret = malloc(13);
    if (output[*pos] == '\0') {
	strcpy(ret, "");
	return ret;
    }
    mvwhite(output, pos);
    strncpy(ret, output + *pos, 12);
    ret[12] = '\0';
    (*pos) += 12;
    return ret;
}

/* Parse date string into a date */

void pftp_get_fft_date(const pftp_file_t *fft, struct tm *date)
{
    pftp_get_date(fft->changed, date);
}

void pftp_get_date(const char *changed, struct tm *date)
{
    const char *ret;
    struct tm today, *rettm;
    time_t now;
    now = time(NULL);
#if HAVE_GMTIME_R
    rettm = gmtime_r(&now, &today);
#else
    rettm = gmtime(&now);
    if (rettm)
	memcpy(&today, rettm, sizeof(struct tm));
#endif
    memset(date, 0, sizeof(struct tm));
    date->tm_year = today.tm_year;
#ifdef WIN32
    ret = strptime(changed, "%b %d %H:%M", date);
#else
    ret = strptime(changed, "%b %e %H:%M", date);
#endif
    if (ret && *ret == '\0')
	return;
    memset(date, 0, sizeof(struct tm));
#ifdef WIN32
    ret = strptime(changed, "%b %d  %Y", date);
#else
    ret = strptime(changed, "%b %e  %Y", date);
#endif
    if (ret && *ret == '\0')
	return;
    memset(date, 0, sizeof(struct tm));
#if HAVE_GETDATE_R
    if (getdate_r(changed, date) == 0)
	return;
#else
# if HAVE_GETDATE
    {
	struct tm *tmp;
	tmp = getdate(changed);
	if (tmp) {
	    memcpy(date, tmp, sizeof(struct tm));
	    return;
	}
    }
# endif
#endif

    assert(0);
}

static void parse_dosdatefmt(char **str)
{
    char *s1, *s2;
    unsigned long tmp;
    struct tm data;
    char *old = malloc(strlen((*str)) + 1);
    strcpy(old, (*str));
    free((*str));
    (*str) = NULL;

    tmp = strtoul(old, &s1, 10);
    if (!s1 || s1[0] != '-') {
	(*str) = old; 
	return;
    }

    if (tmp >= 100) {
	if (tmp >= 1900) {
	    data.tm_year = tmp - 1899;
	} else {
	    (*str) = old; 
	    return;
	}
    } else {
	if (tmp < 50) {
	    data.tm_year = 101 + tmp;
	} else {
	    data.tm_year = tmp + 1;
	}
    }

    s1++;
    tmp = strtoul(s1, &s2, 10);
    if (!s2 || s2[0] != '-' || tmp < 1 || tmp > 31) {
	(*str) = old;
	return;
    }
    data.tm_mday = tmp;

    s1 = s2 + 1;
    tmp = strtoul(s1, &s2, 10);
    if (!s2 || s2[0] != ' ' || tmp < 1 || tmp > 12) {
	(*str) = old;
	return;
    }
    data.tm_mon = tmp - 2;
    
    s1 = s2 + 1;
    while (s1[0] == ' ') s1++;
    tmp = strtoul(s1, &s2, 10);
    if (!s2 || s2[0] != ':' || tmp > 24) {
	(*str) = old;
	return;
    }
    data.tm_hour = tmp % 24;

    s1 = s2 + 1;
    tmp = strtoul(s1, &s2, 10);
    if (!s2 || tmp > 59) {
	(*str) = old;
	return;
    }
    data.tm_min = tmp;

    if (s2[0] == 'P') {
	data.tm_hour += 12;
    }

    (*str) = malloc(13);
    pftp_set_date((*str), &data);
}

void pftp_set_date(char *changed, const struct tm *date)
{
    time_t now;
    struct tm today, *ret;
    now = time(NULL);
#if HAVE_GMTIME_R
    ret = gmtime_r(&now, &today);
#else
    ret = gmtime(&now);
    if (ret)
	memcpy(&today, ret, sizeof(struct tm));
#endif
    if ((date->tm_year == today.tm_year && (today.tm_mon - date->tm_mon) < 6)
	|| (date->tm_year == (today.tm_year - 1) 
	    && ((today.tm_mon + 12) - date->tm_mon) < 6)) {
	/* Recent */
#ifdef WIN32
	strftime(changed, 13, "%b %d %H:%M", date);
#else
	strftime(changed, 13, "%b %e %H:%M", date);
#endif
    } else {
	/* Old */
#ifdef WIN32
        strftime(changed, 13, "%b %d  %Y", date);
#else
        strftime(changed, 13, "%b %e  %Y", date);
#endif
    }
}

static char *dosfile_changed(const char *output, int *pos)
{
    char *ret;
    size_t length;
    if (output[(*pos)] == '\0') {
	ret = malloc(2);
	strcpy(ret, "");
	return ret;
    }
    length = 15 + nwhitelen(output, (*pos) + 15);
    ret = malloc(length + 1);
    strncpy(ret, output + (*pos), length);
    ret[length] = '\0';
    (*pos) += (int) length;

    /* now, make it unix changed format */
    parse_dosdatefmt(&ret);

    return ret;
}

/* Returns the number of links to a file. */
static unsigned long file_links(char *output, int *pos)
{
    char *tmp;
    int length;
    unsigned long ret;
    if (output[*pos] == '\0')
	return 0;
    mvwhite(output, pos);
    length = nwhitelen(output, *pos);
    tmp = malloc(length + 1);
    strncpy(tmp, output + *pos, length);
    tmp[length] = '\0';
    ret = strtoul(tmp, NULL, 10);
    free(tmp);
    (*pos) += length;
    return ret;
}

/* Returns the filename. */
static char *file_name(char *output, int *pos)
{
    int length;
    char *ret;
    if (output[*pos] == '\0') {
	ret = malloc(2);
	strcpy(ret, "");
	return ret;
    }
    (*pos)++;
    length = neollen(output, *pos);
    ret = malloc(length + 1);
    strncpy(ret, output + *pos, length);
    ret[length] = '\0';
    (*pos) += length;
    return ret;
}

static char *dosfile_name(char *output, int *pos)
{
    return file_name(output, pos);
}

typedef struct {
    size_t count;
    char **path;
} clear_cache_t;

static void add_to_clear_cache(clear_cache_t *clr, const char *cur_dir)
{
    size_t c;

    for (c = 0; c < clr->count; c++)
	if (strcmp(clr->path[c], cur_dir) == 0)
	    return;

    clr->path = realloc(clr->path, (clr->count + 1) * sizeof(char *));
    clr->path[clr->count] = strdup(cur_dir);
    clr->count++;
}

static void clear_and_free_cache(pftp_server_t ftp, clear_cache_t *clr)
{
    size_t i;

    for (i = 0; i < clr->count; i++) {
	pftp_clear_dir_cache(ftp, clr->path[i]);
	free(clr->path[i]);
    }

    if (clr->path)
	free(clr->path);

    memset(clr, 0, sizeof(clear_cache_t));
}

/*
 * Compares two filenames. f1 can be name -> target but not f2.
 */

static int same_file(const char *f1, const char *f2)
{
    size_t f2_len = strlen(f2);
    if (strncmp(f1, f2, f2_len) == 0) {
	if (f1[f2_len] == '\0')
	    return 1;
	if (strncmp(f1 + f2_len, " -> ", 4) == 0)
	    return 1;
    }

    return 0;
}

static int get_link_type(pftp_server_t ftp, pftp_directory_t *me, 
			 pftp_file_t *file, const char *cur_dir, 
			 clear_cache_t *clr)
{
    char *target = strstr(file->name, " -> ");
    int err = 0, same_dir = 0, restore_dir = 0;
    char *name = NULL, *path = NULL;
    size_t f;

    if (pftp_in_get_line_type)
	return 0;

    pftp_in_get_line_type = 1;

    if (target && strlen(target)) {
	target += 4;

	if ((name = strrchr(target, '/'))) {
	    if (name[1] == '\0' && name > target) {
		name[0] = '\0';
		name--;
		while (name > target && name[0] != '/')
		    name--;
		if (name == target)
		    name = NULL;
	    }
	}

	if (name) {
	    name++;
	    path = malloc(name - target);
	    memcpy(path, target, name - (target + 1));
	    path[name - (target + 1)] = '\0';
	    if (strcmp(path, cur_dir)) {
		char *tmp = NULL;
		restore_dir = 1;
		if (pftp_cd(ftp, path, &tmp)) {
		    /* Unable to cd dir */
		    err = 1;
		    restore_dir = 0;
		} else {
		    if (!tmp)
			pftp_curdir(ftp, &tmp, 1);

		    if (tmp) {
			same_dir = (strcmp(cur_dir, tmp) == 0);
			restore_dir = !same_dir;
			free(tmp);
		    } else {
			/* Unable to get cur dir */
			err = 1;
		    }
		}
	    } else {
		same_dir = 1;
	    }
	} else {
	    name = target;
	    same_dir = 1;
	}
    } else {
	/* Unable to get link target */
	err = 1;
    }

    if (!err && !same_dir) {    
	pftp_directory_t dir;
	int cached; 
	char *cur_dir2 = NULL;
	memset(&dir, 0, sizeof(pftp_directory_t));

	pftp_curdir(ftp, &cur_dir2, 0);

	if (!(err = pftp_ls(ftp, &dir, 0, &cached))) {
	    err = -1; /* Not found */
	    for (f = 0; f < dir.length; f++)
		if (same_file(dir.files[f].name, name)) {
		    if (dir.files[f].link 
			&& dir.files[f].type == pft_unknown) {
			pftp_in_get_line_type = 0;
			err = get_link_type(ftp, &dir, dir.files + f, cur_dir2,
					    clr);
			pftp_in_get_line_type = 1;
		    } else {
			err = 0;
		    }		    

		    file->type = dir.files[f].type;
		    break;
		}

	    pftp_free_fdt(dir);
	    if (!cached)
		add_to_clear_cache(clr, cur_dir2);
	}

	if (cur_dir2)
	    free(cur_dir2);
    } else if (!err) {
	err = -1; /* Not found */

	for (f = 0; f < me->length; f++) {
	    if (same_file(me->files[f].name, name)) {
		if (me->files[f].link && me->files[f].type == pft_unknown) {
		    pftp_in_get_line_type = 0;
		    err = get_link_type(ftp, me, me->files + f, cur_dir, clr);
		    pftp_in_get_line_type = 1;
		} else {
		    err = 0;
		}

		file->type = me->files[f].type;
		break;
	    }
	}
    }

    if (restore_dir)
	pftp_cd(ftp, cur_dir, NULL);    
    if (path)
	free(path);

    pftp_in_get_line_type = 0;

    return err;
}

/* Parses a row. */
static pftp_file_t row_parse(char *output, int *pos, pftp_server_t ftp, 
			     int sys_unix)
{
    pftp_file_t file;
    if (output == NULL)
	exit(1);
    if (sys_unix) {
	file.type = file_type(output, pos, &file.link);
	file.perm = file_perm(output, pos);
	file.links = file_links(output, pos);
	file.user = file_ug(output, pos);
	file.group = file_ug(output, pos);
	file.size = file_size(output, pos);
	file.changed = file_changed(output, pos);
	file.name = file_name(output, pos);
    } else {
	file.perm = 666;
	file.links = 0;
	file.link = 0;
	file.user = malloc(8);
	strcpy(file.user, "unknown");
	file.group = malloc(8);
	strcpy(file.group, "unknown");

	file.changed = dosfile_changed(output, pos);
	file.size = dosfile_size(output, pos, &file.type);
	file.name = dosfile_name(output, pos);	
    }

    mvnl(output, pos);
    return file;
}

/* Parses an entire listing. */
pftp_directory_t pftp_dir_parse(const char *ls_output, pftp_server_t ftp, 
				int sys_unix)
{
    pftp_directory_t dir_info;
    pftp_file_t *files;
    char *ls = NULL, *cur_dir = NULL;
    int rows, i, pos = 0;
    size_t f;
    clear_cache_t clr;

    memset(&dir_info, 0, sizeof(pftp_directory_t));

    pftp_curdir(ftp, &cur_dir, 0);
    if (!cur_dir)
	return dir_info;

    ls = malloc(strlen(ls_output) + 1);
    strcpy(ls, ls_output);
    rows = chrcnt(ls_output, '\n');
    if (rows == 0)
	rows = chrcnt(ls_output, '\r');
    if (total_row(ls_output) == 1) {
	mvrow(ls_output, &pos);
	rows--;
    }

    files = malloc(rows * sizeof(pftp_file_t));

    for (i = 0; i < rows; i++) {
	files[i] = row_parse(ls, &pos, ftp, sys_unix);
    }
    dir_info.length = rows;
    dir_info.files = files;

    memset(&clr, 0, sizeof(clear_cache_t));

    for (f = 0; f < dir_info.length; f++) {
	if (dir_info.files[f].link && dir_info.files[f].type == pft_unknown)
	    get_link_type(ftp, &dir_info, dir_info.files + f, cur_dir, &clr);
    }

    clear_and_free_cache(ftp, &clr);

    free(ls);
    free(cur_dir);
    return dir_info;
}

void pftp_free_fft(pftp_file_t fft)
{
    free(fft.user);
    free(fft.group);
    free(fft.changed);
    free(fft.name);
}

void pftp_free_fdt(pftp_directory_t fdt)
{
    size_t i;
    if (fdt.files) {
	for (i = 0; i < fdt.length; i++)
	    pftp_free_fft(fdt.files[i]);
	free(fdt.files);
    }
}

#define safe_str(x) (x ? x : "(null)")

static void cpy_file(pftp_file_t *dest, const pftp_file_t *src)
{
    memcpy(dest, src, sizeof(pftp_file_t));

    if (src->user) {
	dest->user = malloc(strlen(src->user) + 1);
	strcpy(dest->user, src->user);
    }
    if (src->group) {
	dest->group = malloc(strlen(src->group) + 1);
	strcpy(dest->group, src->group);
    }
    if (src->changed) {
	dest->changed = malloc(strlen(src->changed) + 1);
	strcpy(dest->changed, src->changed);
    }
    if (src->name) {
	dest->name = malloc(strlen(src->name) + 1);
	strcpy(dest->name, src->name);
    }
}

void pftp_cpy_dir(pftp_directory_t *dest, const pftp_directory_t *src)
{
    size_t c;

    memcpy(dest, src, sizeof(pftp_directory_t));

    if (dest->length) {
	dest->files = malloc(dest->length * sizeof(pftp_file_t));

	for (c = 0; c < dest->length; c++)
	    cpy_file(&(dest->files[c]), &(src->files[c]));
    } else {
	dest->files = NULL;
    }
}
