/* -*- pftp-c -*- */
/*
 * Copyright (c) 2000 Niels Provos.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/* Most of this file is from dh.c in OpenSSH. */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif
#if HAVE_STDLIB_H
# include <stdlib.h>
#endif
#if HAVE_LIMITS_H
# include <limits.h>
#endif

#ifndef NO_SSL

#include <openssl/bn.h>
#include <openssl/dh.h>
#include <openssl/evp.h>

#include "pftp_dh.h"

#ifdef DEBUG
# include <assert.h>
#else
# define assert(x) // x
#endif

#ifdef WITH_DMALLOC
# include <dmalloc.h>
#endif

int pftp_dh_pub_is_valid(DH *dh, BIGNUM *dh_pub)
{
    int i;
    int n = BN_num_bits(dh_pub);
    int bits_set = 0;
    
    if (dh_pub->neg) {
	fprintf(stderr, "invalid public DH value: negativ");
	return 0;
    }
    for (i = 0; i <= n; i++)
	if (BN_is_bit_set(dh_pub, i))
	    bits_set++;
    
    /* if g==2 and bits_set==1 then computing log_g(dh_pub) is trivial */
    if (bits_set > 1 && (BN_cmp(dh_pub, dh->p) == -1))
	return 1;
    
    fprintf(stderr, "invalid public DH value (%d/%d)", 
	    bits_set, BN_num_bits(dh->p));
    return 0;
}

void pftp_dh_gen_key(DH *dh, int need)
{
    int i, bits_set, tries = 0, ret;
    
    assert(dh->p != NULL);
    assert(need <= INT_MAX / 2 && 2 * need < BN_num_bits(dh->p));
    
    do {
	if (dh->priv_key != NULL)
	    BN_clear_free(dh->priv_key);
	dh->priv_key = BN_new();
	assert(dh->priv_key);
	/* generate a 2*need bits random private exponent */
	ret = BN_rand(dh->priv_key, 2*need, 0, 0);
	assert(ret);
	ret = DH_generate_key(dh);
	assert(ret);
	for (i = 0, bits_set = 0; i <= BN_num_bits(dh->priv_key); i++)
	    if (BN_is_bit_set(dh->priv_key, i))
		bits_set++;
	tries++;
	assert(tries < 10);
    } while (!pftp_dh_pub_is_valid(dh, dh->pub_key));
}

static DH *_dh_new_group_asc(const char *gen, const char *modulus)
{
    DH *dh;
    int ret;
    
    dh = DH_new();
    assert(dh);
    
    ret = BN_hex2bn(&dh->p, modulus);
    assert(ret);
    ret = BN_hex2bn(&dh->g, gen);
    assert(ret);
    
    return (dh);
}

DH *pftp_dh_new_group1(void)
{
    static char *gen = "2", *group1 =
	"FFFFFFFF" "FFFFFFFF" "C90FDAA2" "2168C234" "C4C6628B" "80DC1CD1"
	"29024E08" "8A67CC74" "020BBEA6" "3B139B22" "514A0879" "8E3404DD"
	"EF9519B3" "CD3A431B" "302B0A6D" "F25F1437" "4FE1356D" "6D51C245"
	"E485B576" "625E7EC6" "F44C42E9" "A637ED6B" "0BFF5CB6" "F406B7ED"
	"EE386BFB" "5A899FA5" "AE9F2411" "7C4B1FE6" "49286651" "ECE65381"
	"FFFFFFFF" "FFFFFFFF";
    
    return (_dh_new_group_asc(gen, group1));
}

DH *pftp_dh_new_group14(void)
{
    static char *gen = "2", *group14 =
	"FFFFFFFF" "FFFFFFFF" "C90FDAA2" "2168C234" "C4C6628B" "80DC1CD1"
	"29024E08" "8A67CC74" "020BBEA6" "3B139B22" "514A0879" "8E3404DD"
	"EF9519B3" "CD3A431B" "302B0A6D" "F25F1437" "4FE1356D" "6D51C245"
	"E485B576" "625E7EC6" "F44C42E9" "A637ED6B" "0BFF5CB6" "F406B7ED"
	"EE386BFB" "5A899FA5" "AE9F2411" "7C4B1FE6" "49286651" "ECE45B3D"
	"C2007CB8" "A163BF05" "98DA4836" "1C55D39A" "69163FA8" "FD24CF5F"
	"83655D23" "DCA3AD96" "1C62F356" "208552BB" "9ED52907" "7096966D"
	"670C354E" "4ABC9804" "F1746C08" "CA18217C" "32905E46" "2E36CE3B"
	"E39E772C" "180E8603" "9B2783A2" "EC07A28F" "B5C55DF0" "6F4C52C9"
	"DE2BCBF6" "95581718" "3995497C" "EA956AE5" "15D22618" "98FA0510"
	"15728E5A" "8AACAA68" "FFFFFFFF" "FFFFFFFF";
    
    return (_dh_new_group_asc(gen, group14));
}

#endif /* NO_SSL */
