/* -*- pftp-c -*- */
#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#if HAVE_STDLIB_H
# include <stdlib.h>
#endif
#if HAVE_STDIO_H
# include <stdio.h>
#endif
#if HAVE_UNISTD_H
# include <unistd.h>
#endif
#if HAVE_INTTYPES_H
# include <inttypes.h>
#endif
#if HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif
#if HAVE_SYS_SOCKET_H
# include <sys/socket.h>
#endif
#if HAVE_FCNTL_H
# include <fcntl.h>
#endif
#if HAVE_ERRNO_H
# include <errno.h>
#endif
#if HAVE_NETINET_IN_H
# include <netinet/in.h>
#endif
#if HAVE_ARPA_INET_H
# include <arpa/inet.h>
#endif
#if HAVE_STDARG_H
# include <stdarg.h>
#endif
#if HAVE_NETDB_H
# include <netdb.h>
#endif
#if HAVE_STRING_H
# include <string.h>
#endif
#if HAVE_LIMITS_H
# include <limits.h>
#endif
#if TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif

#ifdef WIN32
typedef SOCKET socket_t;
#else
typedef int socket_t; 
#endif

#ifndef NO_SSL
# include <openssl/ssl.h>
# include <openssl/rand.h>
# include <openssl/err.h>
#endif /* NO_SSL */

#include "pftp.h"
#include "pftp_sfvcheck.h"
#include "pftp_speed.h"
#ifndef NO_SSL
# include "pftp_sftp.h"
# include "pftp_ssl.h"
#endif
#include "pftp_internal.h"

#ifdef WITH_DMALLOC
# include <dmalloc.h>
#endif

#ifndef SCNu64
# define SCNu64 "llu"
#endif 

#ifdef DEBUG
# include <assert.h>
#else
# define assert(x) //
#endif

/* 
 * Define this to 1 only, and I mean only, if you happen to be working against
 * a copy of a glftpd 2.0 release candidate.
 * (glftpd < 2.0 works and glftpd 2.0 stable does to).
 * glftpd-2.0-rcX sent the response to an ABOR as a multiline, without
 * sending the rest of the multiline message. I.e. pftp stands and waits 
 * forever (or until 421 comes around).
 * As the fix breaks RFC959 when handling ABOR responses it's now disabled
 * by default.
 */

#define FIX_GLFTPD_ABOR_RESPONSE 0

/* Some nice macros */

#ifndef min
# define min(__x, __y) ((__x) > (__y) ? (__y) : (__x))
#endif /* min */

#define getValue(__f, __v)						\
    (__f->settings->__v == -1 ? __f->settings->global->__v : __f->settings->__v)
#define getStrValue(__f, __v)						\
    (__f->settings->__v == NULL ? __f->settings->global->__v:__f->settings->__v)

/* Some internal data types and default values */

#ifdef WIN32
typedef int socklen_t;
#else
# define closesocket close
#endif

/* Timeouts in sec */

static const int TIMEOUT[] = { 10, /* DOWNLOAD */
			       10, /* UPLOAD */
			       20, /* PORT_CONN */
			       20, /* CONNECT */
			       20  /* SSL_HANDSHAKE */}; 

/* Internal structure used by parseCmd & c.o. */

typedef struct {
    unsigned short *code;
    char **data;
    size_t datalen;
} parsecmd_data_s, *parsecmd_data_t;

/* Static lib vars */
#ifndef NO_SSL
# ifdef WIN32
static int libSSLEnabled = 0;
# endif
#endif
static int libInited = 0;

/* Internal functions */

/* All functions (almost) returns 0 on OK and -1 on error (or -2 even) */

/* Wait for a cmd on ftp control connection and return that */
#if FIX_GLFTPD_ABOR_RESPONSE
# define receiveCmd(ftp, data) _receiveCmd(ftp, data, 0)
static short _receiveCmd(pftp_server_t ftp, char **data, int abort);
#else
static short receiveCmd(pftp_server_t ftp, char **data);
#endif
/* Wait for a cmd on two ftp control connections and return those... */
static int receiveCmd2(pftp_server_t ftp1, pftp_server_t ftp2, 
		       unsigned short *ret1, unsigned short *ret2, 
		       char **data1, char **data2);
/* Send a cmd to a ftp */
static int sendCmd(pftp_server_t ftp, const char *cmd ,const char *buffer);
/* Send data over data connection to ftp */
static int sendData(pftp_server_t ftp, const void *buffer, size_t buffersize,
		    size_t *done, speed_data_t speed);
/* Receive data from data connection to ftp */
static int receiveData(pftp_server_t ftp, void **buffer, size_t *buffersize, 
		       speed_data_t speed);
/* Connect the control connection for ftp */
static int connectCmd(pftp_server_t ftp);
/* Get a data connection to ftp server for given cmds */
static int getDataConnection(pftp_server_t ftp, const char *precmd, 
			     const char *predata, const char *cmd, 
			     const char *data);
/* When sent cmds for data connection, wait for some response */
static int continueDataConnection(pftp_server_t ftp, const char *precmd, 
				  const char *predata, const char *cmd, 
				  const char *data);
/* Setup a passive data connection */
static int getPassiveConnection(pftp_server_t ftp, const char *cmd, 
				const char *data);
/* Parser help function for getPassiveConnection */
static int parsePasvResponse(pftp_server_t ftp, const char *data, 
			     struct sockaddr_in *addr);
/* Send a PRE warning before PASV */
static int setupPretPasv(pftp_server_t ftp, const char *cmd, const char *data);
/* Generator help function for getDirectConnection */
static int generatePortData(pftp_server_t ftp, const struct sockaddr_in *addr, 
			    char **tmpstr, int dont_use_myaddr);
/* Setup a port data connection */
static int getDirectConnection(pftp_server_t ftp);
/* Wait for incoming data connection, when using port */
static int acceptDirectConnection(pftp_server_t ftp);
/* Compare two command responses, ie 221 == 225 */
static int cmdcmp(unsigned short code1, unsigned short code2);
/* Send account data if that's needed */
static int sendAccount(pftp_server_t ftp);
/* Abort current data transfer */
static int abortData(pftp_server_t ftp);
/* Send command via comm_central */
static int send_comm(pftp_server_t ftp, pftp_msg_t msg, 
		     pftp_param_t param1, pftp_param_t param2);
/* "Parse" SYST response */
static system_t parseSYST(const char *data);
/* Helper function for turning a hostname or IP into a sockaddr_in */
static int parseHostname(const char *hostname, unsigned short port, 
			 struct sockaddr_in *addr);
/* Connect a socket and wait for connection */
static int connectSocket(pftp_server_t ftp, socket_t *sock, 
			 struct sockaddr_in *addr);
/* Creates a nonblocking socket */
static socket_t createNonblockingSocket(pftp_server_t ftp);
/* Set sockopt that the sockets address is reusable */
static int makeSocketReusable(pftp_server_t ftp, socket_t s);
/* "Parse" FEAT response */
static int handleFeat(pftp_server_t ftp, const char *feat_data);
/* Init secure command connection */
static int initAuth(pftp_server_t ftp);
/* Init secure data connection */
static int initDataAuth(pftp_server_t ftp, int list);
/* Start secure data connection */
static int startDataAuth(pftp_server_t ftp);
/* End secure data connection */
static void stopDataAuth(pftp_server_t ftp);
/* When secure data connection was interrupted, restore things */
static void restoreAuthData(pftp_server_t ftp, int list);
/* Internal recv and send function, used for different cases */
static ssize_t intRecv(pftp_server_t ftp, socket_t socket, void *buffer, 
		       size_t len);
static ssize_t intCRCRecv(pftp_server_t ftp, socket_t socket, void *buffer, 
			  size_t len);
static ssize_t intSend(pftp_server_t ftp, socket_t socket, const void *buffer, 
		       size_t len);
#ifndef NO_SSL
static ssize_t intSSLRecv(pftp_server_t ftp, socket_t socket, void *buffer, 
			  size_t len);
static ssize_t intCRCSSLRecv(pftp_server_t ftp, socket_t socket, void *buffer, 
			     size_t len);
static ssize_t intSSLSend(pftp_server_t ftp, socket_t socket, const void *buffer, 
			  size_t len);
#endif /* NO_SSL */

void pftp_initlib(void)
{
    if (libInited) return;
    
#ifndef NO_SSL

# ifdef WIN32
    {
        HMODULE ssl_dll, lib_dll;
    
        ssl_dll = LoadLibrary("ssleay32.dll");
        lib_dll = LoadLibrary("libeay32.dll");

        if (ssl_dll && lib_dll) {
            if (initSSLWrapper(ssl_dll, lib_dll)) {
                libSSLEnabled = 1;
            }
        }
	
        if (!libSSLEnabled) {
            fprintf(stdout, "PFTP: Unable to load ssleay32.dll and libeay32.dll, SSL support disabled.\n");
        }
    }
    
    if (libSSLEnabled) {
# endif

	pftp_init_libSSL();

# ifdef WIN32
    } /* if libSSLEnabled */
# endif

#endif /* NO_SSL */

#ifdef WIN32
    {
        WORD wVersionRequested;
        WSADATA wsaData;
        int err;
	
        wVersionRequested = MAKEWORD( 2, 2 );
	
        err = WSAStartup(wVersionRequested, &wsaData);
        if (err != 0) {
            fprintf(stderr, "Unable to find a usable Winsock DLL.\n");
            return;
        }
	
        if (LOBYTE(wsaData.wVersion) != 2 ||
            HIBYTE(wsaData.wVersion) != 2 ) {
            fprintf(stderr, "Unable to find a usable Winsock DLL.\n");
            WSACleanup();
            return; 
        }
	
        fprintf(stdout, "PFTP: WinSock loaded (atleast version %u.%u)\n",
                LOBYTE(wsaData.wVersion), HIBYTE(wsaData.wVersion));
    }
#endif
    
    libInited = 1;
}

void pftp_donelib(void)
{
    if (!libInited) return;
    
#ifndef NO_SSL
# ifdef WIN32
    if (libSSLEnabled) {
# endif
	pftp_free_libSSL();
# ifdef WIN32
    }
# endif
#endif /* NO_SSL */

#ifdef WIN32
    WSACleanup();
#endif
    
    libInited = 0;
}

static int _int_login(pftp_server_t ftp)
{
    short retdata;

    /* Connect to ftp server */
    if (connectCmd(ftp) == -1) {
	return -1;
    }

#ifndef NO_SSL
# ifdef WIN32
    if (libSSLEnabled) {
# endif
	if (getValue(ftp, implicid_ssl)) {
	    if (pftp_init_SSL(ftp, &ftp->cmdssl, ftp->cmdstream) != 0) {
		return -1;
	    }

	    ftp->ssl_private = 1;
	    ftp->cmdRecv = intSSLRecv;
	    ftp->cmdSend = intSSLSend;
	}
# ifdef WIN32
    }
# endif
#endif
    
    /* Wait for welcome message */
    do {
	retdata = receiveCmd(ftp, NULL);
    } while (cmdcmp(retdata, 120) == 0);
    
    if (retdata != 220) {   
	return -1;
    }

    return 0;
}

pftp_server_t pftp_login(pftp_settings_t settings, 
			 pftp_comm_centralFunc_t comm_central)
{
    pftp_server_t ret;
    int retdata, feat = 0;
    char *data = NULL;
    
    if (!libInited)
	return NULL;
    
    if (!settings)
	return NULL;
    
    ret = malloc(sizeof(struct pftp_server_s));
    memset(ret, 0, sizeof(struct pftp_server_s));
    
    ret->comm_central = comm_central;
    ret->settings = settings;
    ret->cmdstream = -1;
    ret->datastream = -1;
    ret->port_socket = -1;
    ret->inbuffer = NULL;
    ret->inbuffer_len = 0;
    ret->lastgoodbuffersize = 32767;
    
    ret->cmdRecv = intRecv;
    ret->cmdSend = intSend;

    ret->pasv = getValue(ret, pasv);

#ifndef NO_SSL    
    if (getValue(ret, sftp)) {
	if (connectCmd(ret) == -1) {
	    free(ret);
	    return NULL;
	}	

	ret->sftp = pftp_sftp_login(ret);

	if (ret->sftp) {
	    return ret;
	} else {
	    free(ret);
	    return NULL;
	}
    }
#endif
    
    /* Capabilities are set to -5, as to note "unknown".
       This is not a very good idea as -5 is "true". But well... */
    ret->resume = -5;
    ret->binary = -5;
    ret->abort = -5;
    ret->help = -5;
    ret->site = -5;
    ret->tls = -5;
    ret->cdup = -5;
    ret->mdtm = -5;
    ret->chmod = -5;
    /* Handle this as not supported until proven otherwise */
    ret->pret = 0; 

    if (_int_login(ret)) {
	free(ret);
	return NULL;
    }
    
    if (sendCmd(ret, "FEAT", NULL) == -1) {
	free(ret);
	return NULL;
    }

    pftp_next_status_not_fatal(ret);
    
    if ((retdata = receiveCmd(ret, &data)) == 211) {
	handleFeat(ret, data);
	free(data);
	feat = 1;
    } else {
	if (data) free(data);
	if (retdata == -1 && _int_login(ret)) {
	    free(ret);
	    return NULL;
	}
    }
    
    data = NULL;
    
    if (!getValue(ret, implicid_ssl) && 
	(ret->tls == 1 || getValue(ret, use_secure))) {
	if (initAuth(ret) == -2) {
	    shutdown(ret->cmdstream, 2);
	    closesocket(ret->cmdstream);

	    if (_int_login(ret)) {
		free(ret);
		return NULL;
	    }
	}
    }

    if (!ret->settings->username || !ret->settings->username[0]) {
	if (!send_comm(ret, PFTP_NEEDPASS, 
		       (pftp_param_t) &ret->settings->username, NULL)) {
	    free(ret);
	    return NULL;
	}
    }
    
    /* Send username */
    if (sendCmd(ret, "USER", ret->settings->username) == -1) {
	free(ret);
	return NULL;
    }
    
    /* Wait for logged in */
    retdata = receiveCmd(ret, NULL);
    
    if (retdata == 331) {
        char *clean_pass, *cur;
        
	/* Send password */
	if (!ret->settings->password || !ret->settings->password[0]) {
	    if (!send_comm(ret, PFTP_NEEDPASS, NULL, 
			   (pftp_param_t) &ret->settings->password)) {
		free(ret);
		return NULL;
	    }
	}

        clean_pass = strdup(ret->settings->password);
        cur = clean_pass;
        while (*cur != '\0') {
            if (*cur == '\n' || *cur == '\r') {
                *cur = '\0';
                break;
            }
            cur++;
        }        

	if (sendCmd(ret, "PASS", clean_pass) == -1) {
	    free(clean_pass);
	    free(ret);
	    return NULL;
	}

	free(clean_pass);
	
	/* Wait for logged in */
	retdata = receiveCmd(ret, NULL);
	
	if (cmdcmp(retdata, 230) && cmdcmp(retdata, 200)) {
	    if (send_comm(ret, PFTP_NEEDPASS, 
			  (pftp_param_t) &ret->settings->username, 
			  (pftp_param_t) &ret->settings->password)) {
		free(ret);
		return pftp_login(settings, comm_central);
	    } else {
		free(ret);
		return NULL;
	    }
	}
    } else if (cmdcmp(retdata, 230)) {
	if (send_comm(ret, PFTP_NEEDPASS,
		      (pftp_param_t)&ret->settings->username, 
		      (pftp_param_t)&ret->settings->password)) {
	    free(ret);
	    return pftp_login(settings, comm_central);
	} else {
	    free(ret);
	    return NULL;
	}
    }
    
    /* Ask for some info */
    if (sendCmd(ret, "SYST", NULL) == -1) {
	free(ret);
	return NULL;
    }

    pftp_next_status_not_fatal(ret);
    
    if (receiveCmd(ret, &data) == 215) {
	ret->system = parseSYST(data);
	free(data);
	data = NULL;
    } else {
	ret->system = UNIX; /* Guessing */
	if (data) free(data);
	data = NULL;
    }
    
    if (!feat) {
	if (sendCmd(ret, "FEAT", NULL) == -1) {
	    free(ret);
	    return NULL;
	}

	pftp_next_status_not_fatal(ret);
	
	if (receiveCmd(ret, &data) == 211) {
	    handleFeat(ret, data);
	    free(data);	    
	} else {
	    ret->tls = 0;
	    
	    if (data)
		free(data);
	}
	
	data = NULL;
    }

    /* Check for help & site support */
    if (sendCmd(ret, "HELP", "SITE") == -1) {
	free(ret);
	return NULL;
    }

    pftp_next_status_not_fatal(ret);
    
    if (cmdcmp(receiveCmd(ret, NULL), 200) == 0) {
	ret->help = 1;
	ret->site = 1; // ?
    } else {
	ret->help = 0; // ?
	ret->site = 0; // ?
    }

    /* Check for resume support */
    if (ret->resume == -5) {
	if (sendCmd(ret, "REST", "1024") == -1) {
	    free(ret);
	    return NULL;
	}

	pftp_next_status_not_fatal(ret);
	
	retdata = receiveCmd(ret, NULL);
	
	if (retdata != 350) {
	    ret->resume = 0;
	} else {
	    ret->resume = 1;
	}

	/* For most ftp servers, the abort below will fix this, but not all
	   proftpd for one. */
	if (sendCmd(ret, "REST", "0") == -1) {
	    free(ret);
	    return NULL;
	}

	pftp_next_status_not_fatal(ret);
	
	retdata = receiveCmd(ret, NULL);
    }
    
    if (ret->abort == -5) {
	if (sendCmd(ret, "ABOR", NULL) == -1) {
	    free(ret);
	    return NULL;
	}

	pftp_next_status_not_fatal(ret);
	
	retdata = receiveCmd(ret, NULL);
	
	if (cmdcmp(retdata, 220) == 0) {
	    ret->abort = 1;
	} else {
	    ret->abort = 0;
	}
    }

    if (ret->cdup) { // -5 or 1
	char *s1, *s2;
	/* The reason we do this even if we know CDUP is supported
	   is because some ftp servers send welcome text with first CWD or
	   CDUP issued */
	if (sendCmd(ret, "PWD", NULL) == -1) {
	    free(ret);
	    return NULL;
	}

	pftp_next_status_not_fatal(ret);

	if (receiveCmd(ret, &data) != 257) {
	    free(ret);
	    return NULL;
	}

	if ((s1 = strchr(data, '\"')) && (s2 = strchr(s1 + 1, '\"'))) {
	    s1++;
	    memmove(data, s1, s2 - s1);
	    data[s2 - s1] = '\0';
	}

	if (sendCmd(ret, "CDUP", NULL) == -1) {
	    free(ret);
	    free(data);
	    return NULL;
	}

	pftp_next_status_not_fatal(ret);
	
	retdata = receiveCmd(ret, NULL);
	
	if (retdata != 250 && retdata != 550) {
	    // 550 = Some serverns whine when you do CDUP in home directory
	    ret->cdup = 0;
	} else {
	    ret->cdup = 1;
	}

	if (sendCmd(ret, "CWD", data) == -1) {
	    free(ret);
	    free(data);
	    return NULL;
	}

	pftp_next_status_not_fatal(ret);

	free(data);	    
	data = NULL;

	if (receiveCmd(ret, NULL) != 250) {
	    free(ret);
	    return NULL;
	}
    }
    
    if (sendCmd(ret, "TYPE", "I") == -1) {
	free(ret);
	return NULL;
    }

    if (receiveCmd(ret, NULL) != 200) {
	ret->binary = 0;
	ret->cur_binary = 0;
    } else {
	ret->binary = 1;
	
	if (sendCmd(ret, "TYPE", "A") == -1) {
	    free(ret);
	    return NULL;
	}

	if (receiveCmd(ret, NULL) != 200) {
	    // Must be supported by RFC 959
	    free(ret);
	    return NULL;
	}
	
	ret->cur_binary = 0;
    }

    {
	pftp_directory_t tmp;
	memset(&tmp, 0, sizeof(pftp_directory_t));
	ret->hidden_ls = 1;
	if (pftp_ls(ret, &tmp, 0, NULL)) {
	    ret->hidden_ls = 0;	    
	}
	pftp_free_fdt(tmp);
    }
    
    pftp_curdir(ret, &data, 0);    
    if (data) free(data);
    
    return ret;
}

int parseHostname(const char *hostname, unsigned short port, 
		  struct sockaddr_in *addr)
{
    //    host = gethostbyname2(hostname, AF_INET);
    struct hostent *host = gethostbyname(hostname);
    
    if (!host) {
	if (inet_aton(hostname, &addr->sin_addr)) {
	    addr->sin_family = AF_INET;
	} else {
	    return -1;
	}
    } else {
	addr->sin_family = host->h_addrtype;
	memcpy(&addr->sin_addr, host->h_addr, host->h_length);
    }
    
    addr->sin_port = htons(port);
    
    return 0;
}

int makeSocketReusable(pftp_server_t ftp, socket_t s)
{
#ifdef WIN32
    BOOL val;
    int vallen;
    vallen = sizeof(val);
    val = TRUE;
    
    if (setsockopt(s, SOL_SOCKET, SO_REUSEADDR, (const char *)&val, vallen)) {
	pftp_status_message(ftp, "WARN: Unable to make socket reusable: %s.",
			    strerror(errno));
	return -1;
    }
    
    return 0;
#else
    int i;
    i = 1;

    if (setsockopt(s, SOL_SOCKET, SO_REUSEADDR, &i, sizeof(int))) {
	pftp_status_message(ftp, "WARN: Unable to make socket reusable: %s.",
			    strerror(errno));
	return -1;
    }
    
    return 0;
#endif
}

int makeSocketNonblocking(pftp_server_t ftp, socket_t s)
{
    int oldflags;
    oldflags = fcntl(s, F_GETFL, 0);
    
    if (oldflags == -1) {
	pftp_status_message(ftp, 
			    "ERR: Unable to get socket stream status flags: %s",
			    strerror(errno));
	return -1;
    } 
    
    if (fcntl(s, F_SETFL, oldflags | O_NONBLOCK) == -1) {
	pftp_status_message(ftp, 
			    "ERR: Unable to set socket stream status flags: %s",
			    strerror(errno));
	return -1;
    }

    return 0;
}

socket_t createNonblockingSocket(pftp_server_t ftp)
{
    socket_t ret = socket(PF_INET, SOCK_STREAM, 0); 
    
    if (ret == -1) {
	pftp_status_message(ftp, "ERR: Unable to create a socket: %s", 
			    strerror(errno));
	return -1;
    }
    
    if (makeSocketNonblocking(ftp, ret)) {
	closesocket(ret);
	return -1;
    }
	
    return ret;
}

/* Special return status:
   -2 for not a change
   -1 for unable to connect
   0 for ok */
int connectSocket(pftp_server_t ftp, socket_t *sock, struct sockaddr_in *addr)
{
    int ret;
    struct timeval timeout;
    fd_set write_set;
    
    (*sock) = createNonblockingSocket(ftp);
    
    if ((*sock) == -1) {
	return -2;
    }

    // Always returns -1 as nonblocking
    ret = connect((*sock), (struct sockaddr *)addr, 
		  sizeof(struct sockaddr_in));

#ifdef WIN32
    if (WSAGetLastError() != WSAEALREADY 
	&& WSAGetLastError() != WSAEINPROGRESS 
	&& WSAGetLastError() != WSAEWOULDBLOCK) {
	pftp_status_message(ftp, "ERR: Unable to connect: %lu", 
			    WSAGetLastError());
	closesocket((*sock));
	(*sock) = -1;
	return -1;
    }
#else
    if (errno != EINPROGRESS && errno != EALREADY) {
	pftp_status_message(ftp, "ERR: Unable to connect: %s", 
			    strerror(errno));
	closesocket((*sock));
	(*sock) = -1;
	return -1;
    }
#endif
    
    /* Connected, now wait for the real connection */
    FD_ZERO(&write_set);
    pftp_start_wait(ftp, CONNECT, 0);
    
    for (;;) {
	timeout.tv_sec = 0;
	timeout.tv_usec = DEFAULT_TIMEOUT;
	FD_SET((*sock), &write_set);

	for (;;) {
	    ret = select(FD_SETSIZE, NULL, &write_set, NULL, &timeout);
	    if (ret == -1L && errno == EINTR)
		continue;
	    break;
	}
	
	if (ret == -1) {
	    pftp_status_message(ftp, "ERR: Select returns error: %s", 
				strerror(errno));
	    pftp_end_wait(ftp, CONNECT, 1);
	    shutdown((*sock), 2);
	    closesocket((*sock));
	    (*sock) = -1;
	    return -1;
	} else if (ret == 0) {
	    if (pftp_do_wait(ftp, NULL, 0, "CONNECT") == -1) {
		pftp_end_wait(ftp, CONNECT, 1);
		shutdown((*sock), 2);
		closesocket((*sock));
		(*sock) = -1;
		return -1;
	    }
	} else {
	    break;
	}
    }

    /* OK, Connect is done, now check if were really connected */
    {
	struct sockaddr_in addr;
	socklen_t len = sizeof(addr);
	memset(&addr, 0, len);
	if (getpeername((*sock), (struct sockaddr *)&addr, &len)) {
	    pftp_status_message(ftp, "ERR: Connect failed: %s", 
				strerror(errno));
	    pftp_end_wait(ftp, CONNECT, 1);
	    shutdown((*sock), 2);
	    closesocket((*sock));
	    (*sock) = -1;
	    return -1;
	}
    }

    pftp_end_wait(ftp, CONNECT, 0);
    
    return 0;
}

int connectCmd(pftp_server_t ftp)
{
    size_t h;
    int connected = 0;
    struct sockaddr_in addr;
    const char *hostname;
    unsigned short port;
    
    for (h = 0; h < ftp->settings->hosts && !connected; h++) {
	hostname = ftp->settings->hostname[h];
	port = ftp->settings->port[h];
	
	if (parseHostname(hostname, port, &addr) == -1) {
	    pftp_status_message(ftp, "ERR: Unable to find host %s", hostname);
	    continue;
	}

	pftp_status_message(ftp, "Trying %s:%u", hostname, port);
	
	switch (connectSocket(ftp, &ftp->cmdstream, &addr)) {
	case 0:
	    connected = 1;
	    break;
	case -1:
	    /* Try next */
	    break;
	default:
	case -2:
	    /* Give up */
	    h = ftp->settings->hosts;
	    break;
	}
    }
    
    if (!connected) {
	pftp_status_message(ftp, "ERR: Unable to connect to ftp server");
	closesocket(ftp->cmdstream);
	ftp->cmdstream = -1;
	return -1;   
    } else {
	pftp_status_message(ftp, "Connected.");
    }
    
    return 0;
}

void pftp_logout(pftp_server_t *ftp) 
{
    if (ftp && (*ftp)) {
#ifndef NO_SSL
	if ((*ftp)->sftp) {
	    pftp_sftp_logout(&(*ftp)->sftp);
	}
#endif
		
	if ((*ftp)->datastream != -1) {
	    abortData((*ftp));
	    stopDataAuth((*ftp));
	}
		
	if ((*ftp)->cmdstream != -1) {
	    sendCmd((*ftp), "QUIT", NULL);
	}
		
	if ((*ftp)->cmdstream != -1) {
	    receiveCmd((*ftp), NULL);
			
#ifndef NO_SSL
	    if ((*ftp) && (*ftp)->cmdssl) SSL_shutdown((*ftp)->cmdssl);	    
#endif /* NO_SSL */
			
	    shutdown((*ftp)->cmdstream, 2);
	    closesocket((*ftp)->cmdstream);
	}    
		
	if ((*ftp)->inbuffer) {
	    free((*ftp)->inbuffer);
	    (*ftp)->inbuffer = NULL;
	}
		
#ifndef NO_SSL
	if ((*ftp)->cmdssl) SSL_free((*ftp)->cmdssl);
	if ((*ftp)->datassl) SSL_free((*ftp)->datassl);
#endif /* NO_SSL */
		
	if ((*ftp)->cur_path) free((*ftp)->cur_path);
	if ((*ftp)->cache_dir) {
	    size_t d;
	    for (d = 0; d < (*ftp)->cached; d++)
		pftp_free_fdt((*ftp)->cache_dir[d]);	       
	    free((*ftp)->cache_dir);
	} 
	if ((*ftp)->cache_path) {
	    size_t d;
	    for (d = 0; d < (*ftp)->cached; d++)
		free((*ftp)->cache_path[d]);	       
	    free((*ftp)->cache_path);
	}
	
	free((*ftp));
	(*ftp) = NULL;
    }
}

int pftp_compactPath(char **path)
{
    size_t len = strlen((*path));
    char *s1, *s2, *cpy = malloc(len + 2);
    strcpy(cpy, (*path));

    if (len > 1 && cpy[len - 1] != '/') {
	strcpy(cpy + len, "/");
	len++;
    }

    while ((s1 = strstr(cpy, "//"))) {
	memmove(s1, s1 + 1, (len - (s1 - cpy)) + 1);
	len--;
    }

    while ((s1 = strstr(cpy, "/./"))) {
	memmove(s1, s1 + 2, (len - ((s1 + 2) - cpy)) + 1);
	len -= 2;
    }

    while ((s1 = strstr(cpy, "/../"))) {
	if (s1 == cpy) {
	    memmove(cpy, cpy + 3, len + 1);
	    len -= 3;
	} else {
	    s1[0] = '\0';
	    s2 = strrchr(cpy, '/');
	    if (s2) {
		memmove(s2 + 1, s1 + 4, (len - ((s1 + 4) - cpy)) + 1);
		len -= (s1 + 4) - (s2 + 1);
	    } else {
		free(cpy);
		return 0;
	    }
	}
    }

    (*path) = realloc((*path), len + 1);
    memcpy((*path), cpy, len);

    if (len > 1 && (*path)[len-1] == '/')
	(*path)[len-1] = '\0';
    else
	(*path)[len] = '\0';

    free(cpy);

    return 1;
}

static void updateCurpath(pftp_server_t ftp, const char *dir)
{
    if (ftp->cur_path) {
	if (dir[0] == '/') {
	    ftp->cur_path = realloc(ftp->cur_path, strlen(dir) + 1);
	    strcpy(ftp->cur_path, dir);	    
	} else {
	    ftp->cur_path = realloc(ftp->cur_path, strlen(ftp->cur_path) + 
				    strlen(dir) + 2);
		
	    if (ftp->cur_path[strlen(ftp->cur_path) - 1] != '/')
		strcat(ftp->cur_path, "/");
		
	    strcat(ftp->cur_path, dir);	    
	}

	if (!pftp_compactPath(&ftp->cur_path)) {
	    char *data = NULL;
	    free(ftp->cur_path);
	    ftp->cur_path = NULL;
	    pftp_curdir(ftp, &data, 0);
	    if (data) free(data);
	}
    } else {
	char *data = NULL;
	pftp_curdir(ftp, &data, 0);
	if (data) free(data);
    }
}

int pftp_cd(pftp_server_t ftp, const char *dir, char **newdir)
{
    char *data = NULL, *pos;

#ifndef NO_SSL
    if (ftp->sftp) {
	return pftp_sftp_cd(ftp->sftp, dir, newdir);
    }
#endif

    if ((ftp->cur_path && strcmp(ftp->cur_path, dir)) == 0 ||
	strcmp(dir, ".") == 0) {
	if (newdir) {
	    size_t len = strlen(ftp->cur_path) + 1;
	    (*newdir) = malloc(len);
	    memcpy((*newdir), ftp->cur_path, len);
	}
	return 0;
    }

    if (strcmp(dir, "..") == 0 && ftp->cdup) {
	if (sendCmd(ftp, "CDUP", NULL) == -1) {
	    return -1;
	}
    } else {
	if (sendCmd(ftp, "CWD", dir) == -1) {
	    return -1;
	}
    }
    
    if (receiveCmd(ftp, &data) != 250) {
	if (data) free(data);

	if (strcmp(dir, "..") == 0 && ftp->cdup) {
	    
	    ftp->cdup = 0; // Never make the same mistake twice
	    return pftp_cd(ftp, dir, newdir);
	}
	
	return -1;
    }

    if ((pos = strstr(data, "Current directory is "))) {
	size_t len;
	pos += 21;
	len = strlen(pos) + 1;

	if (len > 0 && pos[len-1] == '/') 
	    pos[--len] = '\0';

	if (newdir) {
	    (*newdir) = malloc(len);
	    memcpy((*newdir), pos, len);
	}

	updateCurpath(ftp, pos);
    } else {
	updateCurpath(ftp, dir);
    }

    if (data) free(data);
    
    return 0;
}

void pftp_curdir(pftp_server_t ftp, char **currentDirectory, int force)
{
    char *s1, *s2;
    
    if ((*currentDirectory)) {
	free((*currentDirectory));
	(*currentDirectory) = NULL;
    }

#ifndef NO_SSL
    if (ftp->sftp) {
	pftp_sftp_curdir(ftp->sftp, currentDirectory);
	return;
    }
#endif

    if (ftp->cur_path && !force) {
	(*currentDirectory) = strdup(ftp->cur_path);
	return;
    }

    if (sendCmd(ftp, "PWD", NULL) == -1) {
	return;
    }

    if (receiveCmd(ftp, currentDirectory) != 257) {
	if ((*currentDirectory)) {
	    free((*currentDirectory));
	    (*currentDirectory) = NULL;
	}

	return;
    }
    
    if ((s1 = strchr((*currentDirectory), '\"')) && 
	(s2 = strchr(s1 + 1, '\"'))) {
	s1++;
	memmove((*currentDirectory), s1, s2 - s1);
	(*currentDirectory)[s2 - s1] = '\0';
    }

    if (ftp->cur_path) free(ftp->cur_path);
    ftp->cur_path = malloc(strlen((*currentDirectory)) + 1);
    strcpy(ftp->cur_path, (*currentDirectory));
}

static int getMode(pftp_server_t ftp, int binary)
{
    if (binary) {
	if (ftp->binary && !ftp->cur_binary) {
	    if (sendCmd(ftp, "TYPE", "I") == -1) {
		return -1;
	    }
	    
	    if (receiveCmd(ftp, NULL) != 200) {
		return -1;
	    }
	    
	    ftp->cur_binary = 1;
	}
    } else {
	if (ftp->cur_binary) {
	    if (sendCmd(ftp, "TYPE", "A") == -1) {
		return -1;
	    }
	    
	    if (receiveCmd(ftp, NULL) != 200) {
		return -1;
	    }
	    
	    ftp->cur_binary = 0;
	}
    }
    
    return 0;
}

int pftp_ls(pftp_server_t ftp, pftp_directory_t *dir, int force, int *cache)
{
    void *buffer = NULL;
    size_t buffersize = 0;
    int ret = 0;
    char *cur_dir = NULL;
    speed_data_t speed;
    if (cache) (*cache) = 0;

    /* Check cache */
    if (!force) {
	pftp_curdir(ftp, &cur_dir, 0);
	if (cur_dir) {
	    size_t c;
	    for (c = 0; c < ftp->cached; c++) {
		if (strcmp(ftp->cache_path[c], cur_dir) == 0) {
		    pftp_cpy_dir(dir, &ftp->cache_dir[c]);		    
		    free(cur_dir);
		    if (cache) (*cache) = 1;
		    return 0;
		}
	    }   
	}
    }

#ifndef NO_SSL
    if (ftp->sftp) {
	if (pftp_sftp_ls(ftp->sftp, dir))
	    return -1;
    } else
#endif
    {
        if (getMode(ftp, 0) == -1) {
	    if (cur_dir) free(cur_dir);
	    return -1;
	}
    
	if (getDataConnection(ftp, NULL, NULL, 
			      "LIST", ftp->hidden_ls ? "-a" : NULL) == -1) {
	    if (cur_dir) free(cur_dir);
	    return -1;
	}
	
	speed = pftp_init_speed(0, 0, ftp->lastgoodbuffersize);
	pftp_start_wait(ftp, DOWNLOAD, 3);
	
	for (;;) {
	    ret = receiveData(ftp, &buffer, &buffersize, speed);
	    if (ret == -1) {
		if (cur_dir) free(cur_dir);
		pftp_free_speed(&speed, &ftp->lastgoodbuffersize);
		pftp_end_wait(ftp, DOWNLOAD, 1);
		abortData(ftp);
		stopDataAuth(ftp);
		return -1;
	    }
	    if (ret == 0) {
		break;
	    }
	}
	
	((char *)buffer)[buffersize] = '\0';
    
	pftp_free_speed(&speed, &ftp->lastgoodbuffersize);
	pftp_end_wait(ftp, DOWNLOAD, 0);

	if (receiveCmd(ftp, NULL) != 226) {
	    if (cur_dir) free(cur_dir);
	    free(buffer);
	    return -1;
	}
	
	(*dir) = pftp_dir_parse((char *)buffer, ftp, ftp->system == UNIX);
	free(buffer);
    }
    
    if (!cur_dir)
	pftp_curdir(ftp, &cur_dir, 0);

    if (cur_dir) {
	size_t c;
	for (c = 0; c < ftp->cached; c++)
	    if (strcmp(cur_dir, ftp->cache_path[c]) == 0) {
		pftp_cpy_dir(&ftp->cache_dir[c], dir);
		free(cur_dir);
		cur_dir = NULL;
		break;
	    }

	if (cur_dir) {
	    ftp->cache_path = realloc(ftp->cache_path, (ftp->cached + 1) *
				      sizeof(char *));
	    ftp->cache_path[ftp->cached] = malloc(strlen(cur_dir) + 1);
	    strcpy(ftp->cache_path[ftp->cached], cur_dir);
	    ftp->cache_dir = realloc(ftp->cache_dir, (ftp->cached + 1) * 
				     sizeof(pftp_directory_t));
	    pftp_cpy_dir(&ftp->cache_dir[ftp->cached], dir);
	    ftp->cached++;
	    free(cur_dir);
	}
    }
    
    return 0;
}

int pftp_rmdir(pftp_server_t ftp, const char *dir)
{
#ifndef NO_SSL
    if (ftp->sftp) {
	return pftp_sftp_rmdir(ftp->sftp, dir);
    }
#endif

    if (sendCmd(ftp, "RMD", dir) == -1) {
	return -1;
    }
    
    if (receiveCmd(ftp, NULL) != 250) {
	return -1;
    }
    
    return 0;
}

int pftp_mkdir(pftp_server_t ftp, const char *dir)
{
#ifndef NO_SSL
    if (ftp->sftp) {
	return pftp_sftp_mkdir(ftp->sftp, dir);
    }
#endif

    if (sendCmd(ftp, "MKD", dir) == -1) {
	return -1;
    }
    
    if (receiveCmd(ftp, NULL) != 257) {
	return -1;
    }
    
    return 0;
}

int pftp_rm(pftp_server_t ftp, const char *filename)
{
#ifndef NO_SSL
    if (ftp->sftp) {
	return pftp_sftp_rm(ftp->sftp, filename);
    }
#endif

    if (sendCmd(ftp, "DELE", filename) == -1) {
	return -1;
    }
    
    if (receiveCmd(ftp, NULL) != 250) {
	return -1;
    }
    
    return 0; 
}

int pftp_rename(pftp_server_t ftp, const char *filename, const char *new)
{
#ifndef NO_SSL
    if (ftp->sftp) {
	return pftp_sftp_rename(ftp->sftp, filename, new);
    }
#endif

    if (sendCmd(ftp, "RNFR", filename) == -1) {
	return -1;
    }
    
    if (receiveCmd(ftp, NULL) != 350) {
	return -1;
    }
    
    if (sendCmd(ftp, "RNTO", new) == -1) {
	return -1;
    }
    
    if (receiveCmd(ftp, NULL) != 250) {
	return -1;
    }
    
    return 0; 
}

int pftp_get(pftp_server_t ftp, const char *filename, char ascii, FILE *fh, 
	     uint64_t resume_from, uint64_t total_size, uint32_t *crc)
{
    void *buffer = NULL;
    size_t buffersize = 0;
    int ret;
    speed_data_t speed;

#ifndef NO_SSL
    if (ftp->sftp) {
	return pftp_sftp_get(ftp->sftp, filename, ascii, fh, resume_from,
			     total_size, crc);
    }
#endif
    
    if (getMode(ftp, ascii == 'A' ? 0 : 1) == -1) {
	return -1;
    }
    
    if (resume_from > 0) {
	char *tmpstr;

	if (!ftp->resume)
	    return -1;

	tmpstr = malloc(50);
	snprintf(tmpstr, 50, "%" SCNu64, resume_from);
	
	if (getDataConnection(ftp, "REST", tmpstr, "RETR", filename) == -1) {
	    free(tmpstr);
	    return -1;
	}
	
	free(tmpstr);
    } else {
	if (getDataConnection(ftp, NULL, NULL, "RETR", filename) == -1) {
	    return -1;
	}
    }

    send_comm(ftp, PFTP_FILETRANSFER, (pftp_param_t) &resume_from, 
	      (pftp_param_t) &total_size);
    speed = pftp_init_speed(total_size, resume_from, ftp->lastgoodbuffersize);
    pftp_start_wait(ftp, DOWNLOAD, 1);
    if (crc) {
	pftp_initCRC32(&ftp->cur_crc);
#ifndef NO_SSL
	if (ftp->ssl_private)
	    ftp->dataRecv = intCRCSSLRecv;
	else
#endif
	    ftp->dataRecv = intCRCRecv;       
    }
    
    for (;;) {
	ret = receiveData(ftp, &buffer, &buffersize, speed);

	if (ret == -1) {
	    pftp_end_wait(ftp, DOWNLOAD, 1);
	    abortData(ftp);	    
	    stopDataAuth(ftp);
	    pftp_free_speed(&speed, &ftp->lastgoodbuffersize);
	    if (buffer) free(buffer);
	    return -1;
	}

	if (buffersize > 0) {
	    if (fwrite(buffer, 1, buffersize, fh) != buffersize) {
		pftp_status_message(ftp, "ERR: Unable to write to file: %s", 
				    strerror(errno));
		pftp_end_wait(ftp, DOWNLOAD, 1);
		abortData(ftp);	    
		stopDataAuth(ftp);
		pftp_free_speed(&speed, &ftp->lastgoodbuffersize);
		if (buffer) free(buffer);
		return -1;
	    }
	}
	
	buffersize = 0;
	
	if (ret == 0) 
	    break;
    }
    
    pftp_free_speed(&speed, &ftp->lastgoodbuffersize);
    pftp_end_wait(ftp, DOWNLOAD, 0);
    if (buffer) free(buffer);
    
    if (receiveCmd(ftp, NULL) != 226) {
	if (ftp->datastream != -1) {
	    stopDataAuth(ftp);
	    shutdown(ftp->datastream, 2);
	    closesocket(ftp->datastream);
	    ftp->datastream = -1;
	}     
	return -1;
    }
   
    if (crc) {
	pftp_finishCRC32(&ftp->cur_crc);
	(*crc) = ftp->cur_crc;
    }
    
    return 0;
}

int pftp_put(pftp_server_t ftp, const char *filename, char ascii, FILE *fh, 
	     uint64_t resume_from, uint64_t total_size)
{
    void *buffer = NULL;
    size_t buffersize = 0, done, lastbuffer = 0;
    int ret;
    speed_data_t speed;

#ifndef NO_SSL
    if (ftp->sftp) {
	return pftp_sftp_put(ftp->sftp, filename, ascii, fh, resume_from, 
			     total_size);
    }
#endif
    
    if (getMode(ftp, ascii == 'A' ? 0 : 1) == -1) {
	return -1;
    }

    if (resume_from > 0) {
	char *tmpstr;

	if (!ftp->resume)
	    return -1;

	tmpstr = malloc(50);
	snprintf(tmpstr, 50, "%" SCNu64, resume_from);
	
	if (getDataConnection(ftp, "REST", tmpstr, "STOR", filename) == -1) {
	    free(tmpstr);
	    return -1;
	}
	
	free(tmpstr);
    } else {
	if (getDataConnection(ftp, NULL, NULL, "STOR", filename) == -1) {
	    return -1;
	}  
    }
    
    send_comm(ftp, PFTP_FILETRANSFER, (pftp_param_t)&resume_from, 
	      (pftp_param_t)&total_size);
    speed = pftp_init_speed(total_size, resume_from, ftp->lastgoodbuffersize);
    lastbuffer = speed->buffersize;
    buffer = malloc(speed->buffersize + 1);

    pftp_start_wait(ftp, UPLOAD, 2);
    
    for (;;) {
	if (lastbuffer != speed->buffersize) {
	    buffer = realloc(buffer, speed->buffersize + 1);
	}
	ret = fread((char *)buffer + buffersize, 1, 
		    (int) speed->buffersize-buffersize, fh);
	
	if (ferror(fh)) {
	    pftp_status_message(ftp, "ERR: Unable to read from file: %s", 
				strerror(errno));
	    pftp_end_wait(ftp, UPLOAD, 1);
	    abortData(ftp);
	    stopDataAuth(ftp);
	    if (buffer) free(buffer);
	    pftp_free_speed(&speed, &ftp->lastgoodbuffersize);
	    return -1;
	}
	
	buffersize += ret;
	done = 0;
	
	while ((buffersize - done) >= speed->buffersize 
	       || (feof(fh) && (buffersize - done) > 0)) {
	    size_t newdone;
	    ret = sendData(ftp, (char *)buffer + done, buffersize - done, 
			   &newdone, speed);
	    
	    if (ret == -1) {
		pftp_end_wait(ftp, UPLOAD, 1);
		abortData(ftp);
		stopDataAuth(ftp);
		pftp_free_speed(&speed, &ftp->lastgoodbuffersize);
		if (buffer) free(buffer);
		return -1;
	    }

	    done += newdone;
	}

	if (done < buffersize) {
	    buffersize -= done;
	    memmove(buffer, (char *)buffer + done, buffersize);
	} else {
	    buffersize = 0;
	}
	
	if (feof(fh))
	    break;
    }
    
    if (buffer) free(buffer);
    pftp_free_speed(&speed, &ftp->lastgoodbuffersize);
    pftp_end_wait(ftp, UPLOAD, 0);
    
    stopDataAuth(ftp);
    shutdown(ftp->datastream, 2);
    closesocket(ftp->datastream);
    ftp->datastream = -1;
    
    if (receiveCmd(ftp, NULL) != 226) {
	return -1;
    }
    
    return 0;
}

/* Helpers for internal datastructure for parseCmd below. */

static parsecmd_data_t initParseCmd(unsigned short *code, char **data)
{
    parsecmd_data_t ret;

    (*code) = 0;
    if (data && (*data)) {
	free((*data));
	(*data) = NULL;
    }   

    ret = malloc(sizeof(parsecmd_data_s));
    memset(ret, 0, sizeof(parsecmd_data_s));
    
    ret->code = code;
    ret->data = data;
    ret->datalen = 0;

    return ret;
}

static void freeParseCmd(parsecmd_data_t data)
{
    if (data) {
	free(data);
    }
}

/*
 * Helper for parseCmd.
 * Just removes the line from inbuffer. 
 * Handles both \n and \r\n \r and \n\r (just to be sertain) with some
 * help from parseCmd.
 */

static void eat_line_inbuffer(pftp_server_t ftp, size_t len)
{
    assert(ftp && ftp->inbuffer);
    assert(len < ftp->inbuffer_len);

    /* ftp->inbuffer[len + 1] is safe as inbuffer always must have space
     * for the ending '\0'. */

    if (ftp->inbuffer[len] == '\r') {
	if (ftp->inbuffer[len + 1] == '\n')
	    len++;
    } else { /* ftp->inbuffer[len] == '\n' */
	if (ftp->inbuffer[len + 1] == '\r')
	    len++;
    }

    len++;

    ftp->inbuffer_len -= len;
    memmove(ftp->inbuffer, ftp->inbuffer + len, ftp->inbuffer_len + 1);
}

/* 
 * Parsing FTP server response. 
 * Returns 0 until whole response has come in.
 * Clears pftp_server_t incoming buffer by itself.
 */

#if FIX_GLFTPD_ABOR_RESPONSE		
static int parseCmd(pftp_server_t ftp, parsecmd_data_t data, int abort)
#else
    static int parseCmd(pftp_server_t ftp, parsecmd_data_t data)
#endif    
{
    char *pos, *end;
    unsigned long code;

    assert(ftp && ftp->inbuffer);
    assert(data);

    for (;;) {
	assert(ftp->inbuffer[ftp->inbuffer_len] == '\0');

	if (!(pos = strchr(ftp->inbuffer, '\n')) &&
	    !(pos = strchr(ftp->inbuffer, '\r'))) {
	    /* Need more data - i.e. not done */
	    return 0;	
	}

	if (pos == ftp->inbuffer) {
	    eat_line_inbuffer(ftp, 0);
	    /* Silently ignore empty lines as these can be a result of
	     * \r\n where only the \r is available the first round. 
	     * Also, next statement doesn't handle pos == ftp->inbuffer very 
	     * well... */
	    continue;
	}

	if (pos[-1] == '\r')
	    pos--;	

	*pos = '\0';

	if ((*data->code) == 0) {
	    /* First line (and in most cases, last) */
	    int multiline;

	    errno = 0;
	    code = strtoul(ftp->inbuffer, &end, 10);
    
	    if (errno || code < 100 || code >= 600 || 
		end == NULL || (end[0] != ' ' && end[0] != '-')) {    
		pftp_status_message(ftp, "Invalid line from server: %s", 
				    ftp->inbuffer);
		eat_line_inbuffer(ftp, pos - ftp->inbuffer);
		/* Still need some correct data */
		continue;
	    }

#if FIX_GLFTPD_ABOR_RESPONSE		
	    if (end[0] != '-' || (abort && code == 226)) {
		multiline = 0;
	    } else {
		multiline = 1;
	    }
#else
	    if (end[0] != '-') {
		multiline = 0;
	    } else {
		multiline = 1;
	    }
#endif

	    *data->code = (unsigned short)(code & USHRT_MAX);
	    if (data->data) {
		assert((*data->data) == NULL);
		data->datalen = pos - (end + 1);
		*data->data = strndup(end + 1, data->datalen);
	    }
	    pftp_status_message(ftp, "<- %s", ftp->inbuffer);
	    eat_line_inbuffer(ftp, pos - ftp->inbuffer);	    
	    
	    if (!multiline) {
		/* Not a multiline response - so done. */
		return 1;
	    } else {
		/* Multiline response - Need more lines */
		continue;
	    }
	} else {
	    /* Not first line. Part pf multiline response then (we hope). */
	    size_t newdata;
	    char *start;
	    int last_line;

	    errno = 0;
	    code = strtoul(ftp->inbuffer, &end, 10);
    
	    if (errno || code < 100 || code >= 600 || 
		end == NULL || (end[0] != ' ' && end[0] != '-')) {
		/* Middle of multiline response. */
		last_line = 0;
		start = ftp->inbuffer;		
	    } else {		
		if (code == (*data->code)) {
		    /* Last line of multiline response. */		  
		    if (end[0] == ' ') {
			last_line = 1;
		    } else {
			last_line = 0;
		    }
		    start = end + 1;
		} else {
		    /* Not good. Multiline response not terminated correctly. */
		    pftp_status_message(ftp, 
					"Warning, the server did not end "
					"multiline response correctly. "
					"(got %lu, expected %u)", 
					code, (*data->code));
		    /* Do the best we can, so let's pretend we got the end
		     * and save the line with the new responsecode for later.*/
		    last_line = 1;		    
		}
	    }

	    /* Multiline data to take care of */

	    newdata = pos - start;

	    if (data->data) {
		*data->data = realloc(*data->data, 
				      data->datalen + 2 + newdata);
		(*data->data)[data->datalen++] = '\n';
		memcpy((*data->data) + data->datalen, start, newdata);
		data->datalen += newdata;
		(*data->data)[data->datalen] = '\0';
	    }
	    pftp_status_message(ftp, "<- %s", ftp->inbuffer);
	    eat_line_inbuffer(ftp, pos - ftp->inbuffer);

	    if (last_line) {
		/* Data collected - lets finish */
		return 1;
	    } else {
		/* Continue as last line hasn't turned up yet */
		continue;
	    }
	}
    }
}

#if FIX_GLFTPD_ABOR_RESPONSE
short _receiveCmd(pftp_server_t ftp, char **data, int abort)
#else
    short receiveCmd(pftp_server_t ftp, char **data)
#endif
{
    fd_set read_set;
    struct timeval timeout;
    int ret, need_more = 0;
    unsigned short code;
    parsecmd_data_t parse_data;
	    
    FD_ZERO(&read_set);
    pftp_start_wait(ftp, DOWNLOAD, 0);

    parse_data = initParseCmd(&code, data);
    
    for (;;) {
	if (ftp->inbuffer_len == 0 || need_more) {
	    need_more = 0;
	    timeout.tv_sec = 0;
	    timeout.tv_usec = DEFAULT_TIMEOUT;
	    FD_SET(ftp->cmdstream, &read_set);

	    for (;;) {
		ret = select(FD_SETSIZE, &read_set, NULL, NULL, &timeout);
		if (ret == -1 && errno == EINTR)
		    continue;
		break;
	    }
    
	    if (ret == -1) {
		pftp_status_message(ftp, "ERR: Select returns error: %s", 
				    strerror(errno));
		freeParseCmd(parse_data);
		pftp_end_wait(ftp, DOWNLOAD, 1);	       
		return -1;
	    } else if (ret == 0) {
		if (pftp_do_wait(ftp, NULL, 0, "DOWNLOAD-CMD1") == -1) {
		    freeParseCmd(parse_data);
		    pftp_end_wait(ftp, DOWNLOAD, 1);
		    return -1;
		}
	    }
	} else {
	    ret = 1;
	}

	if (ret > 0) {
	    for (;;) {
		ftp->inbuffer = realloc(ftp->inbuffer, 
					ftp->inbuffer_len + 1024 + 1);
		ret = ftp->cmdRecv(ftp, ftp->cmdstream, 
				   ftp->inbuffer + ftp->inbuffer_len, 1024);

		if (ret == -1) {
#ifdef WIN32
		    if (WSAGetLastError() != WSAEWOULDBLOCK) {
			pftp_status_message(ftp, 
					    "ERR: Recv returns error: %lu", 
					    WSAGetLastError());
			freeParseCmd(parse_data);
			pftp_end_wait(ftp, DOWNLOAD, 1);
			return -1;
		    }
#else
		    if (errno != EWOULDBLOCK) {
			pftp_status_message(ftp, "ERR: Recv returns error: %s", 
					    strerror(errno));
			freeParseCmd(parse_data);
			pftp_end_wait(ftp, DOWNLOAD, 1);
			return -1;
		    }
#endif
		    
		    ret = 0;
		} else if (ret == 0) {
		    pftp_status_message(ftp, "ERR: Connection closed (1)");
		    sleep(5);
		    freeParseCmd(parse_data);
		    pftp_end_wait(ftp, DOWNLOAD, 1);
		    send_comm(ftp, PFTP_DISCONNECTED, NULL, NULL);
		    return -1;
		}
		
		if (pftp_do_wait(ftp, NULL, 1, "DOWNLOAD-CMD2") == -1) {
		    freeParseCmd(parse_data);
		    pftp_end_wait(ftp, DOWNLOAD, 1);
		    return -1;
		}
		
		ftp->inbuffer_len += ret;
		ftp->inbuffer[ftp->inbuffer_len] = '\0';
		
#if FIX_GLFTPD_ABOR_RESPONSE
		if (parseCmd(ftp, parse_data, abort) == 0) {
		    need_more = 1;
		    break;
		}
#else
		if (parseCmd(ftp, parse_data) == 0) {
		    need_more = 1;
		    break;
		}
#endif
		
		freeParseCmd(parse_data);
		
		if (code == 332 || code == 532) {
		    if (sendAccount(ftp) == -1) {
			pftp_end_wait(ftp, DOWNLOAD, 1);
			return -1;	  
		    }		    
		    break;
		} else {
		    pftp_end_wait(ftp, DOWNLOAD, 0);
		    if (code == 421) {			
			send_comm(ftp, PFTP_DISCONNECTED, NULL, NULL);
		    }
		    return code;
		}

		/* Do it all over again. */

		parse_data = initParseCmd(&code, data);
	    }
	}	
    }
}

int receiveCmd2(pftp_server_t ftp1, pftp_server_t ftp2, unsigned short *ret1, 
		unsigned short *ret2, char **data1, char **data2)
{
    fd_set read_set;
    struct timeval timeout;
    int ret;
    parsecmd_data_t parse_data1, parse_data2;
    
    FD_ZERO(&read_set);
    pftp_start_wait(ftp1, DOWNLOAD, 0);
    pftp_start_wait(ftp2, DOWNLOAD, 0);

    parse_data1 = initParseCmd(ret1, data1);
    parse_data2 = initParseCmd(ret2, data2);

    while (parse_data1 || parse_data2) {
	timeout.tv_sec = 0;
	timeout.tv_usec = DEFAULT_TIMEOUT;

	FD_ZERO(&read_set);
	if (parse_data1) FD_SET(ftp1->cmdstream, &read_set);
	if (parse_data2) FD_SET(ftp2->cmdstream, &read_set);

	for (;;) {
	    ret = select(FD_SETSIZE, &read_set, NULL, NULL, &timeout);
	    if (ret == -1 && errno == EINTR)
		continue;
	    break;
	}
	
	if (ret == -1) {
	    pftp_status_message(ftp1, "ERR: Select returns error: %s", 
				strerror(errno));
	    pftp_status_message(ftp2, "ERR: Select returns error: %s", 
				strerror(errno));
	    freeParseCmd(parse_data1);
	    freeParseCmd(parse_data2);
	    pftp_end_wait(ftp1, DOWNLOAD, 1);
	    pftp_end_wait(ftp2, DOWNLOAD, 1);
	    return -1;
	} else if (ret == 0) {
	    if (pftp_do_wait(ftp1, NULL, 1, "DOWNLOAD-CMD3") == -1 || 
		pftp_do_wait(ftp2, NULL, 1, "DOWNLOAD-CMD4") == -1) {
		freeParseCmd(parse_data1);
		freeParseCmd(parse_data2);
		pftp_end_wait(ftp1, DOWNLOAD, 1);
		pftp_end_wait(ftp2, DOWNLOAD, 1);
		return -1;
	    }
	} else {
	    if (FD_ISSET(ftp1->cmdstream, &read_set)) {
		for (;;) {
		    ftp1->inbuffer = realloc(ftp1->inbuffer, 
					     ftp1->inbuffer_len + 1024 + 1);
		    ret = ftp1->cmdRecv(ftp1, ftp1->cmdstream, 
					ftp1->inbuffer + ftp1->inbuffer_len, 
					1024);

		    if (ret == -1) {
#ifdef WIN32
			if (WSAGetLastError() != WSAEWOULDBLOCK) {
			    pftp_status_message(ftp1,
						"ERR: Recv returns error: %lu", 
						WSAGetLastError());
			    freeParseCmd(parse_data1);
			    freeParseCmd(parse_data2);
			    pftp_end_wait(ftp1, DOWNLOAD, 1);
			    pftp_end_wait(ftp2, DOWNLOAD, 1);
			    return -1;
			}
#else
			if (errno != EWOULDBLOCK) {
			    pftp_status_message(ftp1,
						"ERR: Recv returns error: %s", 
						strerror(errno));
			    freeParseCmd(parse_data1);
			    freeParseCmd(parse_data2);
			    pftp_end_wait(ftp1, DOWNLOAD, 1);
			    pftp_end_wait(ftp2, DOWNLOAD, 1);
			    return -1;
			}
#endif
		    
			ret = 0;
		    } else if (ret == 0) {
			pftp_status_message(ftp1, "ERR: Connection closed (2)");
			freeParseCmd(parse_data1);
			freeParseCmd(parse_data2);
			pftp_end_wait(ftp1, DOWNLOAD, 1);
			pftp_end_wait(ftp2, DOWNLOAD, 1);
			send_comm(ftp1, PFTP_DISCONNECTED, NULL, NULL);
			return -1;
		    }		    
		    if (pftp_do_wait(ftp1, NULL, 1, "DOWNLOAD-CMD5") == -1) {
			freeParseCmd(parse_data1);
			freeParseCmd(parse_data2);
			pftp_end_wait(ftp1, DOWNLOAD, 1);
			pftp_end_wait(ftp2, DOWNLOAD, 1);
			return -1;
		    }
		
		    ftp1->inbuffer_len += ret;
		    ftp1->inbuffer[ftp1->inbuffer_len] = '\0';

#if FIX_GLFTPD_ABOR_RESPONSE
		    if (parseCmd(ftp1, parse_data1, 0) == 0) {
			/* More data needed */
			break;
		    }
#else
		    if (parseCmd(ftp1, parse_data1) == 0) {
			/* More data needed */
			break;
		    }
#endif

		    freeParseCmd(parse_data1);
		    parse_data1 = NULL;
		
		    if ((*ret1) == 332 || (*ret1) == 532) {		
			if (sendAccount(ftp1) == -1) {	    
			    freeParseCmd(parse_data2);
			    pftp_end_wait(ftp1, DOWNLOAD, 1);
			    pftp_end_wait(ftp2, DOWNLOAD, 1);
			    return -1;
			}

			/* All over again */

			parse_data1 = initParseCmd(ret1, data1);
		    }
		    
		    break;
		}
	    } else if (FD_ISSET(ftp2->cmdstream, &read_set)) {
		for (;;) {
		    ftp2->inbuffer = realloc(ftp2->inbuffer, 
					     ftp2->inbuffer_len + 1024 + 1);
		    ret = ftp2->cmdRecv(ftp2, ftp2->cmdstream, 
					ftp2->inbuffer + ftp2->inbuffer_len, 
					1024);

		    if (ret == -1) {
#ifdef WIN32
			if (WSAGetLastError() != WSAEWOULDBLOCK) {
			    pftp_status_message(ftp2,
						"ERR: Recv returns error: %lu", 
						WSAGetLastError());
			    freeParseCmd(parse_data1);
			    freeParseCmd(parse_data2);
			    pftp_end_wait(ftp1, DOWNLOAD, 1);
			    pftp_end_wait(ftp2, DOWNLOAD, 1);
			    return -1;
			}
#else
			if (errno != EWOULDBLOCK) {
			    pftp_status_message(ftp2,
						"ERR: Recv returns error: %s", 
						strerror(errno));
			    freeParseCmd(parse_data1);
			    freeParseCmd(parse_data2);
			    pftp_end_wait(ftp1, DOWNLOAD, 1);
			    pftp_end_wait(ftp2, DOWNLOAD, 1);
			    return -1;
			}
#endif
			
			ret = 0;
		    } else if (ret == 0) {
			pftp_status_message(ftp2, "ERR: Connection closed (3)");
			freeParseCmd(parse_data1);
			freeParseCmd(parse_data2);
			pftp_end_wait(ftp1, DOWNLOAD, 1);
			pftp_end_wait(ftp2, DOWNLOAD, 1);
			send_comm(ftp2, PFTP_DISCONNECTED, NULL, NULL);
			return -1;
		    }		    
		    if (pftp_do_wait(ftp2, NULL, 1, "DOWNLOAD-CMD6") == -1) {
			freeParseCmd(parse_data1);
			freeParseCmd(parse_data2);
			pftp_end_wait(ftp1, DOWNLOAD, 1);
			pftp_end_wait(ftp2, DOWNLOAD, 1);
			return -1;
		    }
		    
		    ftp2->inbuffer_len += ret;
		    ftp2->inbuffer[ftp2->inbuffer_len] = '\0';

#if FIX_GLFTPD_ABOR_RESPONSE
		    if (parseCmd(ftp2, parse_data1, 0) == 0) {
			/* More data needed */
			break;
		    }
#else
		    if (parseCmd(ftp2, parse_data2) == 0) {
			/* More data needed */
			break;
		    }
#endif

		    freeParseCmd(parse_data2);
		    parse_data2 = NULL;
		
		    if ((*ret2) == 332 || (*ret2) == 532) {		
			if (sendAccount(ftp2) == -1) {
			    pftp_end_wait(ftp1, DOWNLOAD, 1);
			    pftp_end_wait(ftp2, DOWNLOAD, 1);
			    return -1;
			}

			/* All over again */
			parse_data2 = initParseCmd(ret2, data2);
		    }
		    
		    break;
		}
	    }
	}	
    }

    pftp_end_wait(ftp1, DOWNLOAD, 0);
    pftp_end_wait(ftp2, DOWNLOAD, 0);

    return 0;
}

int pftp_rawcmd(pftp_server_t ftp, const char *cmd, char **data)
{
#ifndef NO_SSL
    if (ftp->sftp) {
	/* Not supported. */
	return -1;
    }
#endif

    if (sendCmd(ftp, cmd, NULL) == -1) {
	return -1;
    }

    return receiveCmd(ftp, data); 
}

int sendCmd(pftp_server_t ftp, const char *cmd, const char *buffer)
{
    char *data;
    int ret;
    size_t pos = 0, datalen = 0;
    
    if (buffer) {
	data = malloc(strlen(cmd) + strlen(buffer) + 5);
	strcpy(data, cmd);
	strcat(data, " ");
	strcat(data, buffer);
    } else {
	data = malloc(strlen(cmd) + 5);
	strcpy(data, cmd);
    }
    
    if (strcmp(cmd, "PASS")){
	pftp_status_message(ftp, "-> %s", data);
    } else {
	pftp_status_message(ftp, "-> PASS xxx");
    }
    
    strcat(data, "\r\n");
    datalen = strlen(data);
    
    for (;;) {
	ret = ftp->cmdSend(ftp, ftp->cmdstream, data + pos, datalen - pos);

	if (ret == datalen - pos)
	    break;
	
	if (ret < 1) {
#ifdef WIN32
	    if (ret == -1 && WSAGetLastError() == WSAEWOULDBLOCK)
		continue;
#else
	    if (ret == -1 && errno == EWOULDBLOCK)
		continue;
#endif
	    shutdown(ftp->datastream, 2);
	    closesocket(ftp->datastream);
	    ftp->datastream = -1;
#ifndef NO_SSL
	    if (ftp->cmdssl) SSL_shutdown(ftp->cmdssl);	    
#endif /* NO_SSL */
	    shutdown(ftp->cmdstream, 2);
	    closesocket(ftp->cmdstream);
	    ftp->cmdstream = -1;
	    free(data);
	    send_comm(ftp, PFTP_DISCONNECTED, NULL, NULL);
	    return -1;
	} 
	
	pos += ret;
    }
    
    free(data);
    
    return 0;
}

int sendData(pftp_server_t ftp, const void *buffer, size_t buffersize, 
	     size_t *done, speed_data_t speed)
{
    int ret;
    size_t start = 0;
    fd_set write_set;
    struct timeval timeout;
    
    FD_ZERO(&write_set);
    
    if (ftp->datastream == -1) {
	pftp_status_message(ftp, "ERR: No data connection");
	return -1;  
    }

    for (;;) {
	ret = ftp->dataSend(ftp, ftp->datastream, (char *)buffer + start, 
			    min(speed->buffersize, buffersize - start)); 
	
	if (ret == -1) {
#ifdef WIN32
	    if (WSAGetLastError() != WSAEWOULDBLOCK) {
		pftp_status_message(ftp, "ERR: Send returns error: %lu", 
				    WSAGetLastError());
		return -1;
	    }
#else
	    if (errno != EWOULDBLOCK) {
		pftp_status_message(ftp, "ERR: Send returns error: %s", 
				    strerror(errno));
		return -1;
	    }
#endif
	    
	    for (;;) {
		timeout.tv_sec = 0;
		timeout.tv_usec = DEFAULT_TIMEOUT;
		FD_SET(ftp->datastream, &write_set);
		
		for (;;) {
		    ret = select(FD_SETSIZE, NULL, &write_set, NULL, &timeout);
		    if (ret == -1 && errno == EINTR)
			continue;
		    break;
		}
		
		if (ret == -1) {
		    pftp_status_message(ftp, "ERR: Select returns error: %s", 
					strerror(errno));
		    return -1;
		} else if (ret > 0) {
		    break;
		}
		
		if (pftp_do_wait(ftp, speed, 0, "UPLOAD") == -1) {
		    return -1;
		}
	    }
	} else if (ret == 0) {
	    pftp_status_message(ftp, "ERR: Connection closed (4)");
	    return -1;
	} else {    
	    start += ret;
	    pftp_update_speed(speed, ret);
	    
	    if (pftp_do_wait(ftp, speed, 1, "UPLOAD") == -1) {
		return -1;
	    }
	    
	    if (start + speed->buffersize > buffersize) {
		(*done) = start;
		return 0;
	    }
	}
    }
}

static unsigned long parseUL(const char *str, int *valid)
{
    char *end;
    unsigned long ret;
    (*valid) = 0;
    if (!str) return 0;
    ret = strtoul(str, &end, 10);
    if (ret == ULONG_MAX || !end || end[0] != '\0')
	return 0;
    (*valid) = 1;
    return ret;
}

static size_t csplit(const char *str, char c, char ***part)
{
    size_t parts = 0;
    const char *last = str;
    char *s;
    
    while ((s = strchr(last, c))) {
	(*part) = realloc((*part), sizeof(char *) * (parts + 1));
	(*part)[parts] = malloc((s - last) + 1);
	strncpy((*part)[parts], last, (s - last));
	(*part)[parts][s - last] = '\0';
	last = s + 1;
	parts++;
    }
    
    if (last[0] != '\0') {
	(*part) = realloc((*part), sizeof(char *) * (parts + 1));
	(*part)[parts] = malloc(strlen(last) + 1);
	strcpy((*part)[parts], last);
	parts++;
    }
    
    return parts;
}

int parsePasvResponse(pftp_server_t ftp, const char *data, 
		      struct sockaddr_in *addr)
{
    char *s1, *s2, *tmpstr, **part = NULL;
    size_t parts;
    int valid;
    uint32_t ipaddr = 0;
    uint16_t port = 0;
    unsigned long tmp;
    
    if (!(s1 = strrchr(data, '(')) || !(s2 = strrchr(data, ')')) || s1 > s2) {
	pftp_status_message(ftp, "ERR: Unable to parse PASV response");
	return -1;
    }
    
    s1++;
    
    tmpstr = malloc((s2 - s1) + 1);
    strncpy(tmpstr, s1, (s2 - s1));
    tmpstr[s2 - s1] = '\0';
    
    /* Parse the PASV address */
    parts = csplit(tmpstr, ',', &part);
    free(tmpstr);
    
    if (parts == 6) {
	size_t p;    
	
	for (p = 0; p < 4; p++) {
	    tmp = parseUL(part[p], &valid);
	    if (!valid || tmp >= 256) { 
		ipaddr = 0; 
		break;
	    }
	    ipaddr *= 256;
	    ipaddr += tmp;
	}
	
	for (p = 4; p < 6; p++) {
	    tmp = parseUL(part[p], &valid);
	    if (!valid || tmp >= 256) { 
		port = 0; 
		break;
	    }
	    port *= 256;
	    port += (uint16_t) tmp;
	}
    }
    
    while (parts-- > 0) free(part[parts]);
    if (part) free(part);
    
    if (port == 0 && ipaddr == 0) {
	pftp_status_message(ftp, "ERR: Unable to parse PASV response");
	return -1;
    }

    addr->sin_family = AF_INET;
    addr->sin_addr.s_addr = htonl(ipaddr);
    addr->sin_port = htons(port);
    
    return 0;
}

int setupPretPasv(pftp_server_t ftp, const char *cmd, const char *data)
{
    /* PRE Transfer */
    char *arg;
    size_t c_len, d_len;
    int ret;

    c_len = strlen(cmd);
    d_len = data ? (strlen(data) + 1) : 0;
    arg = malloc(c_len + d_len + 1);
    memcpy(arg, cmd, c_len);
    if (data) {
	arg[c_len] = ' ';
	memcpy(arg + c_len + 1, data, d_len - 1);
    } 
    arg[c_len + d_len] = '\0';

    if (sendCmd(ftp, "PRET", arg) == -1) {
	free(arg);
	return -1;
    }

    free(arg);

    ret = receiveCmd(ftp, NULL);

    if (cmdcmp(ret, 200) == 0) {
	/* OK */
    } else if (ret == 500 || ret == 502) {
	/* FEAT lied, but still OK presumably */
	ftp->pret = 0;
    } else {
	/* Error "ignored". Let PASV or <cmd> fail. */
    }

    return 0;
}

int getPassiveConnection(pftp_server_t ftp, const char *cmd, const char *data)
{
    char *pasvdata = NULL;
    struct sockaddr_in addr;

    if (ftp->pret) {
	if (setupPretPasv(ftp, cmd, data) == -1)
	    return -1;
    }
    
    if (sendCmd(ftp, "PASV", NULL) == -1) {
	return -1;
    }
    
    if (receiveCmd(ftp, &pasvdata) != 227) {
	if (data) free(pasvdata);
	return -1;
    }
    
    if (parsePasvResponse(ftp, pasvdata, &addr) != 0) {
	free(pasvdata);
	return -1;
    }
    
    free(pasvdata);
    
    pftp_status_message(ftp, "Connecting to %s:%u", 
			inet_ntoa(addr.sin_addr), ntohs(addr.sin_port));
    
    if (connectSocket(ftp, &ftp->datastream, &addr) == 0)
	return 0;
    
    return -1;
}

int generatePortData(pftp_server_t ftp, const struct sockaddr_in *addr, 
		     char **tmpstr, int dont_use_myaddr)
{
    uint32_t ipaddr = ntohl(addr->sin_addr.s_addr);
    uint16_t port = ntohs(addr->sin_port);
    
    if (!dont_use_myaddr
	&& getStrValue(ftp, myaddr) && getStrValue(ftp, myaddr)[0]) {
	struct sockaddr_in tmpaddr;
	
	if (parseHostname(getStrValue(ftp, myaddr), port, &tmpaddr) == -1) {
	    pftp_status_message(ftp, "ERR: Unable to use %s as my address.", 
				getStrValue(ftp, myaddr));
	    return -1; 
	}
	
	ipaddr = ntohl(tmpaddr.sin_addr.s_addr);
    }
    
    if (ipaddr == INADDR_ANY) {
	struct sockaddr_in tmpaddr;
	socklen_t len = sizeof(struct sockaddr_in);
	
	if (getsockname(ftp->cmdstream, 
			(struct sockaddr *)&tmpaddr, &len) == -1 
	    || len != sizeof(struct sockaddr_in)) {
	    pftp_status_message(ftp, "ERR: Unable to get my own address: %s", 
				strerror(errno));
	    return -1;
	}
	
	ipaddr = ntohl(tmpaddr.sin_addr.s_addr);
    }
    
    (*tmpstr) = realloc((*tmpstr), 26);
    
    snprintf((*tmpstr), 26, "%u,%u,%u,%u,%u,%u", 
	     (uint16_t)((ipaddr & 0xFF000000) >> 24),
	     (uint16_t)((ipaddr & 0x00FF0000) >> 16),
	     (uint16_t)((ipaddr & 0x0000FF00) >>  8),
	     (uint16_t)((ipaddr & 0x000000FF) >>  0),
	     (uint16_t)((port & 0xFF00) >>  8),
	     (uint16_t)((port & 0x00FF) >>  0));
    
    return 0;
}

int getDirectConnection(pftp_server_t ftp)
{
    char *tmpstr = NULL;
    struct sockaddr_in addr;
    uint16_t port;

    send_comm(ftp, PFTP_NEEDPORT, (pftp_param_t)&port, 
	      (pftp_param_t)ftp->settings);
    
    addr.sin_family = AF_INET;
    addr.sin_port = htons(port);
    
    if (getStrValue(ftp, bind_to) && getStrValue(ftp, bind_to)[0]) {
	if (parseHostname(getStrValue(ftp, bind_to), ntohs(addr.sin_port), 
			  &addr) == -1) {
	    pftp_status_message(ftp, "ERR: Unable to bind socket to: %s", 
				getStrValue(ftp, bind_to));
	    return -1;
	}
    } else {
	addr.sin_addr.s_addr = htonl(INADDR_ANY);
    }
    
    if ((ftp->port_socket = createNonblockingSocket(ftp)) == -1) {
	return -1;
    }
    
    if (bind(ftp->port_socket, (struct sockaddr *)&addr, 
	     sizeof(struct sockaddr_in)) == -1 ||
	listen(ftp->port_socket, 1) == -1) {
	if (getStrValue(ftp, bind_to) && getStrValue(ftp, bind_to)[0] &&
	    port) {
	    pftp_status_message(ftp, 
				"ERR: Unable to bind socket to: %s:%u (%s)", 
				getStrValue(ftp, bind_to), 
				ntohs(addr.sin_port), 
				strerror(errno));
	} else if (port != 0) {
	    pftp_status_message(ftp, 
				"ERR: Unable to bind socket to: <any>:%u (%s)", 
				ntohs(addr.sin_port), strerror(errno));
	} else if (getStrValue(ftp, bind_to) && getStrValue(ftp, bind_to)[0]) {
	    pftp_status_message(ftp, 
				"ERR: Unable to bind socket to: %s:<any> (%s)",
				getStrValue(ftp, bind_to), strerror(errno));
	} else {
	    pftp_status_message(ftp, "ERR: Unable to bind socket to: "
				"<any>:<any> (%s)", strerror(errno));
	}
	
	closesocket(ftp->port_socket);
	ftp->port_socket = -1;
	return -1;
    }

    /* Try to make next PORT work aswell */ 
    makeSocketReusable(ftp, ftp->port_socket);
    
    /* Get the port we bound to, if needed */
    if (!port) {
	struct sockaddr_in tmp_addr;
	socklen_t tmp_addrlen = sizeof(struct sockaddr_in);
	if (getsockname(ftp->port_socket, (struct sockaddr *)&tmp_addr,
			&tmp_addrlen) == -1) {
	    pftp_status_message(ftp, "ERR: Unable to get port number. (%s)",
				strerror(errno));
	    closesocket(ftp->port_socket);
	    ftp->port_socket = -1;
	    return -1;
	} else {
	    addr.sin_port = tmp_addr.sin_port;
	}
    }
    
    if (generatePortData(ftp, &addr, &tmpstr, 0) == -1) {
	free(tmpstr);
	shutdown(ftp->port_socket, 2);
	closesocket(ftp->port_socket);
	ftp->port_socket = -1;
	return -1;
    }
    
    if (sendCmd(ftp, "PORT", tmpstr) == -1) {
	free(tmpstr);
	shutdown(ftp->port_socket, 2);
	closesocket(ftp->port_socket);
	ftp->port_socket = -1;
	return -1;
    }
    
    free(tmpstr);
    
    if (receiveCmd(ftp, NULL) != 200) {
	shutdown(ftp->port_socket, 2);
	closesocket(ftp->port_socket);
	ftp->port_socket = -1;
	return -1;
    }
    
    return 0;
}

int acceptDirectConnection(pftp_server_t ftp)
{
    fd_set read_set;
    struct timeval timeout;
    int ret;
    
    if (ftp->port_socket == -1)
	return -1;
    
    FD_ZERO(&read_set);
    pftp_start_wait(ftp, PORT_CONN, 3);
    
    for (;;) {
	FD_SET(ftp->port_socket, &read_set);
	timeout.tv_sec = 0;
	timeout.tv_usec = DEFAULT_TIMEOUT;

	for (;;) {
	    ret = select(FD_SETSIZE, &read_set, NULL, NULL, &timeout);
	    if (ret == -1 && errno == EINTR)
		continue;
	    break;
	}
	
	if (ret == -1) {
	    pftp_status_message(ftp, "ERR: Select returned error: %s", 
				strerror(errno));
	    shutdown(ftp->port_socket, 2);
	    closesocket(ftp->port_socket);
	    ftp->port_socket = -1;
	    pftp_end_wait(ftp, PORT_CONN, 1);
	    return -1;
	} else if (ret == 0) {
	    if (pftp_do_wait(ftp, NULL, 0, "PORT_CONNECT") == -1) {
		shutdown(ftp->port_socket, 2);
		closesocket(ftp->port_socket);
		ftp->port_socket = -1;
		pftp_end_wait(ftp, PORT_CONN, 1);
		return -1;
	    }
	} else {
	    struct sockaddr_in addr;
	    socklen_t len = sizeof(struct sockaddr_in);
	    ftp->datastream = accept(ftp->port_socket, 
				     (struct sockaddr *)&addr, 
				     &len);
	    
	    if (ftp->datastream == -1) {
		pftp_status_message(ftp, "ERR: accept returned error: %s", 
				    strerror(errno));
		shutdown(ftp->port_socket, 2);
		closesocket(ftp->port_socket);
		ftp->port_socket = -1;
		pftp_end_wait(ftp, PORT_CONN, 1);
		return -1;
	    }      
	    break;
	}
    }
    
    shutdown(ftp->port_socket, 2);
    closesocket(ftp->port_socket);
    ftp->port_socket = -1;
    pftp_end_wait(ftp, PORT_CONN, 0);
    
    return 0;
}

int getDataConnection(pftp_server_t ftp, const char *precmd, 
		      const char *predata, const char *cmd, const char *data)
{
    int connected = 0, list = (strcmp(cmd, "LIST") == 0);
    
    ftp->dataRecv = intRecv;
    ftp->dataSend = intSend;

    if (!ftp->pasv) {
	if (initDataAuth(ftp, list) != 0)
	    return -1;

	if (getDirectConnection(ftp) == 0)
	    connected = 1;
    }
    
    if (!connected) {
	if (initDataAuth(ftp, list) != 0)
	    return -1;

	if (getPassiveConnection(ftp, cmd, data) == 0) {
	    ftp->pasv = 1;
	    connected = 1;
	}
    }
    
    if (!connected) {
	restoreAuthData(ftp, list);
	return -1;   
    }
    
    if (precmd) {
	if (sendCmd(ftp, precmd, predata) == -1) {      
	    if (ftp->datastream != -1) {
		shutdown(ftp->datastream, 2);
		closesocket(ftp->datastream);
		ftp->datastream = -1;
	    }     
	    if (ftp->port_socket != -1) {
		shutdown(ftp->port_socket, 2);
		closesocket(ftp->port_socket);
		ftp->port_socket = -1;
	    }
	    restoreAuthData(ftp, list);
	    return -1;
	}
	
	if (receiveCmd(ftp, NULL) != 350) {
	    if (ftp->datastream != -1) {
		shutdown(ftp->datastream, 2);
		closesocket(ftp->datastream);
		ftp->datastream = -1;
	    }     
	    if (ftp->port_socket != -1) {
		shutdown(ftp->port_socket, 2);
		closesocket(ftp->port_socket);
		ftp->port_socket = -1;
	    }
	    restoreAuthData(ftp, list);
	    return -1;
	}
    }

    if (sendCmd(ftp, cmd, data) == -1) {
	if (ftp->datastream != -1) {
	    shutdown(ftp->datastream, 2);
	    closesocket(ftp->datastream);
	    ftp->datastream = -1;
	}     
	if (ftp->port_socket != -1) {
	    shutdown(ftp->port_socket, 2);
	    closesocket(ftp->port_socket);
	    ftp->port_socket = -1;
	}
	restoreAuthData(ftp, list);
	return -1;
    }

    if (continueDataConnection(ftp, precmd, predata, cmd, data) == -1) {
	return -1;
    }
    
    return 0;
}

int continueDataConnection(pftp_server_t ftp, const char *precmd, 
			   const char *predata, const char *cmd, 
			   const char *data)
{
    int retdata, list = (strcmp(cmd, "LIST") == 0);
    
    retdata = receiveCmd(ftp, NULL);

    if (retdata != 150 && retdata != 125) {
	if (retdata == 425) {
	    if (!ftp->pasv) {
		ftp->pasv = 1;
		
		if (getDataConnection(ftp, precmd, predata, cmd, data) == 0) {
		    return 0;
		} else {
		    ftp->pasv = 0;
		}
	    }
	}     
	
	if (ftp->datastream != -1) {
	    shutdown(ftp->datastream, 2);
	    closesocket(ftp->datastream);
	    ftp->datastream = -1;
	}     
	if (ftp->port_socket != -1) {
	    shutdown(ftp->port_socket, 2);
	    closesocket(ftp->port_socket);
	    ftp->port_socket = -1;
	}
	restoreAuthData(ftp, list);
	return -1;
    }

    if (!ftp->pasv) {
	if (acceptDirectConnection(ftp) == -1) {
	    receiveCmd(ftp, NULL);

	    ftp->pasv = 1;
	    
	    if (getDataConnection(ftp, precmd, predata, cmd, data) == 0) {
		return 0;
	    } else {
		ftp->pasv = 0;
		
		if (ftp->datastream != -1) {
		    shutdown(ftp->datastream, 2);
		    closesocket(ftp->datastream);
		    ftp->datastream = -1;
		}     
		if (ftp->port_socket != -1) {
		    shutdown(ftp->port_socket, 2);
		    closesocket(ftp->port_socket);
		    ftp->port_socket = -1;
		}
		restoreAuthData(ftp, list);
		return -1;	
	    }
	}
    }

    if (startDataAuth(ftp) == -1) {
	abortData(ftp);
	restoreAuthData(ftp, list);
	return -1;
    }

    return 0;
}

int receiveData(pftp_server_t ftp, void **buffer, size_t *buffersize, 
		speed_data_t speed)
{
    fd_set read_set;
    struct timeval timeout;
    int ret;

    ret = 0;
    
    if (ftp->datastream == -1) {
	pftp_status_message(ftp, "ERR: No data connection");
	if ((*buffer)) free((*buffer));
	(*buffer) = NULL;
	(*buffersize) = 0;
	return -1;  
    }
    
    FD_ZERO(&read_set);
    
    for (;;) {
	timeout.tv_sec = 0;
	timeout.tv_usec = DEFAULT_TIMEOUT;
	FD_SET(ftp->datastream, &read_set);

	for (;;) {
	    ret = select(FD_SETSIZE, &read_set, NULL, NULL, &timeout);
	    if (ret == -1 && errno == EINTR)
		continue;
	    break;
	}

	if (ret == -1) {
	    pftp_status_message(ftp, "ERR: Select returns error: %s", 
				strerror(errno));
	    if ((*buffer)) free((*buffer));
	    (*buffer) = NULL;
	    (*buffersize) = 0;
	    return -1;
	} else if (ret == 0) {
	    if (pftp_do_wait(ftp, speed, 0, "DOWNLOAD-DATA1") == -1) {
		if ((*buffer)) free((*buffer));
		(*buffer) = NULL;
		(*buffersize) = 0;
		return -1;
	    }

	    return 1;
	} else {
	    (*buffer) = realloc((*buffer), 
				(*buffersize) + speed->buffersize + 1);
	    ret = ftp->dataRecv(ftp, ftp->datastream, 
				(char *)(*buffer) + (*buffersize), 
				speed->buffersize); 
	    
	    if (ret == -1) {
#ifdef WIN32
		if (WSAGetLastError() != WSAEWOULDBLOCK) {
		    pftp_status_message(ftp, "ERR: Recv returns error: %lu", 
					WSAGetLastError());
		    if ((*buffer)) free((*buffer));
		    (*buffer) = NULL;
		    (*buffersize) = 0;
		    return -1;	
		}
#else
		if (errno != EWOULDBLOCK) {
		    pftp_status_message(ftp, "ERR: Recv returns error: %s", 
					strerror(errno));
		    if ((*buffer)) free((*buffer));
		    (*buffer) = NULL;
		    (*buffersize) = 0;
		    return -1;	
		}
#endif
		
		if (pftp_do_wait(ftp, speed, 0, "DOWNLOAD-DATA2") == -1) {
		    if ((*buffer)) free((*buffer));
		    (*buffer) = NULL;
		    (*buffersize) = 0;
		    return -1;
		}
	    } else if (ret == 0) {		
		/* Done */
		stopDataAuth(ftp);
		shutdown(ftp->datastream, 2);
		closesocket(ftp->datastream);
		ftp->datastream = -1;
		return 0;
	    } else {
		pftp_update_speed(speed, ret);
		(*buffersize) += ret;
		((char *)(*buffer))[(*buffersize)] = '\0';
		
		if (pftp_do_wait(ftp, speed, 1, "DOWNLOAD-DATA3") == -1) {
		    if ((*buffer)) free((*buffer));
		    (*buffer) = NULL;
		    (*buffersize) = 0;
		    return -1;
		}

		return 1;
	    }      
	}
    }  
    
    return -1;
}

static size_t calcFormatLength(const char *format, va_list list)
{
    size_t ret = strlen(format) + 10;
    char *s = 0;
    const char *last = format;
    
    while ((s = strchr(last, '%'))) {
	switch (s[1]) {
	case '\0':
	    last = s + 1;
	    break;
	case '%':
	    last = s + 2;
	    break;
	case 's': {
	    const char *str = va_arg(list, const char *);
	    ret += strlen(str);
	    last = s + 1;
	}; break;
	default:
	    va_arg(list, void *);
	    last = s + 1;
	}
    }

    return ret;
}

void pftp_status_message(pftp_server_t ftp, const char *format, ...)
{
    va_list list, save;
    char *dump;
    size_t len;

    va_start(list, format);
    
#ifdef __va_copy
    __va_copy(save, list);
#else
    save = list;
#endif /* __va_copy */
    
    len = calcFormatLength(format, list) + 512;
    dump = malloc(len);
    
    vsnprintf(dump, len, format, save);
    send_comm(ftp, PFTP_STATUS, ftp->settings->name, dump);
    free(dump);
    va_end(list);
}

void pftp_next_status_not_fatal(pftp_server_t ftp)
{
    send_comm(ftp, PFTP_NEXTSTATUS_IS_NOT_FATAL, NULL, NULL);
}

int cmdcmp(unsigned short code1, unsigned short code2)
{
    unsigned short c1 = code1 / 10, c2 = code2 / 10;
    
    if (c1 < c2) return -1;
    if (c1 > c2) return 1;
    
    return 0;
}

int sendAccount(pftp_server_t ftp)
{
    static int imrunning = 0;
    
    if (imrunning) {
	pftp_status_message(ftp, "ERR: FTP server was naughty");
	return -1;
    }
    
    if (ftp->settings->account) {
	if (sendCmd(ftp, "ACCT", ftp->settings->account) == -1) {
	    return -1;
	}
	
	imrunning = 1;
	
	if (receiveCmd(ftp, NULL) == 230) {
	    imrunning = 0;
	    return 1;
	}
	
	imrunning = 0;
    }
    
    return -1;
}

int abortData(pftp_server_t ftp)
{
    if (ftp->abort) {
	unsigned short ret;
	if (sendCmd(ftp, "ABOR", NULL) == -1) {
	    shutdown(ftp->datastream, 2);
	    closesocket(ftp->datastream);
	    ftp->datastream = -1;
	    return -1;	
	}

	shutdown(ftp->datastream, 2);
	closesocket(ftp->datastream);
	ftp->datastream = -1;

	do {
#if FIX_GLFTPD_ABOR_RESPONSE
	    ret = _receiveCmd(ftp, NULL, 1);
#else
	    ret = receiveCmd(ftp, NULL);
#endif
	    if (cmdcmp(ret, 220) == 0)
		return 0;
	    if (ret == (unsigned short)-1) {
		/* Timeout. Close control connection. */
#ifndef NO_SSL
		if (ftp->cmdssl) SSL_shutdown(ftp->cmdssl);	    
#endif /* NO_SSL */
		shutdown(ftp->cmdstream, 2);
		closesocket(ftp->cmdstream);
		ftp->cmdstream = -1;
		send_comm(ftp, PFTP_DISCONNECTED, NULL, NULL);
		return -1;
	    }
	} while (cmdcmp(ret, 500));
    } else {
	shutdown(ftp->datastream, 2);
	closesocket(ftp->datastream);
	ftp->datastream = -1;
    }
    
    return -1;
}

void pftp_start_wait(pftp_server_t ftp, wait_mode_t mode, intptr_t trans)
{
    ftp->lastdata = time(NULL);
    ftp->waitmode = mode;
    send_comm(ftp, PFTP_INITTRANSFER, (pftp_param_t)trans, NULL);
}

int pftp_do_wait(pftp_server_t ftp, speed_data_t speed, int gotdata, 
		 const char *op)
{
    time_t now = time(NULL);
    int cancel = 0;
    
    if (gotdata) {
	ftp->lastdata = now;
    } else {
	if (now - ftp->lastdata > TIMEOUT[(int)ftp->waitmode]) {
	    pftp_status_message(ftp, "ERR: Operation (%s) timed out.", op);
	    return -1;
	}
    }
    
    if (speed) { 
	if (speed->unknown)
	    cancel = send_comm(ftp, PFTP_TRANSFER, 
			       (pftp_param_t) &speed->speed, NULL);
	else
	    cancel = send_comm(ftp, PFTP_TRANSFER, 
			       (pftp_param_t) &speed->speed, 
			       (pftp_param_t) &speed->percent);
    } else
	cancel = send_comm(ftp, PFTP_TRANSFER, NULL, NULL);
    
    if (cancel == 0) {
	pftp_status_message(ftp, "ERR: User aborted (%s)", op);
	return -1;
    }
    
    return 0;
}

void pftp_end_wait(pftp_server_t ftp, wait_mode_t mode, intptr_t error)
{
    ftp->lastdata = 0;
    if (mode != ftp->waitmode)
	pftp_status_message(ftp, "WARN: Drunk programmer!");
    send_comm(ftp, PFTP_DONETRANSFER, (pftp_param_t) error, NULL);
}

ssize_t intRecv(pftp_server_t ftp, socket_t socket, void *buffer, size_t len)
{
    return recv(socket, buffer, (int) len, 0);
}

ssize_t intCRCRecv(pftp_server_t ftp, socket_t socket, void *buffer, size_t len)
{
    int ret = recv(socket, buffer, (int) len, 0);
    if (ret > 0) 
	pftp_addCRC32(&ftp->cur_crc, buffer, ret);
    return ret;
}

ssize_t intSend(pftp_server_t ftp, socket_t socket, const void *buffer, size_t len)
{
    return send(socket, buffer, (int) len, 0);
}

#ifndef NO_SSL
ssize_t intSSLRecv(pftp_server_t ftp, socket_t socket, void *buffer, size_t len)
{
    SSL *ssl;
    
    if (socket == ftp->cmdstream)
	ssl = ftp->cmdssl;
    else
	ssl = ftp->datassl;
    
    return SSL_read(ssl, buffer, (int)(len & INT_MAX));
}

ssize_t intCRCSSLRecv(pftp_server_t ftp, socket_t socket, void *buffer, size_t len)
{
    SSL *ssl;
    int ret;
    
    if (socket == ftp->cmdstream)
	ssl = ftp->cmdssl;
    else
	ssl = ftp->datassl;
    
    ret = SSL_read(ssl, buffer, (int)(len & INT_MAX));
    if (ret > 0)
	pftp_addCRC32(&ftp->cur_crc, buffer, ret);
    return ret;
}

ssize_t intSSLSend(pftp_server_t ftp, socket_t socket, const void *buffer, 
		   size_t len)
{
    SSL *ssl;
    
    if (socket == ftp->cmdstream)
	ssl = ftp->cmdssl;
    else
	ssl = ftp->datassl;
    
    return SSL_write(ssl, buffer, (int)(len & INT_MAX));
}

#endif /* NO_SSL */

/* Returns specifically, -2 = Error settings up TLS
   -1 = No protection used (allowed)
   0 = Protected */

int initAuth(pftp_server_t ftp)
{
#ifdef NO_SSL
    return -1; /* somewhat funny but better, take it like "couldn't setup tls"
		  but no error */
#else
    int retdata;

#ifdef WIN32
    if (!libSSLEnabled) {
        /* Same behaivor as NO_SSL above */
        return -1;
    }
#endif
    
    if (sendCmd(ftp, "AUTH", "TLS") == -1) {
	return -2;
    }

    pftp_next_status_not_fatal(ftp);
    
    retdata = receiveCmd(ftp, NULL);
    
    if (retdata != 234) {
	return -1;
    }
    
    if (!pftp_ok_libSSL())
	return -2;
    
    /* Init TLS connection */
    retdata = pftp_init_SSL(ftp, &ftp->cmdssl, ftp->cmdstream);
    
    if (retdata != 0)
	return retdata;
    
    ftp->cmdRecv = intSSLRecv;
    ftp->cmdSend = intSSLSend;
    
    if (sendCmd(ftp, "PBSZ", "0") == -1) {
	pftp_status_message(ftp, "ERR: Unable to send data using SSL");
	return -2;
    }
    
    if (receiveCmd(ftp, NULL) != 200) {
	return -1;
    }
    
    if (sendCmd(ftp, "PROT", "P") == -1) {
	return -2;
    }
    
    if ((retdata = receiveCmd(ftp, NULL)) != 200) {
	if (retdata == 530) { 
	    /* Not logged in, drftpd for one */
	    ftp->ssl_private = 0;
	    return 0;
	}

	if (sendCmd(ftp, "PROT", "C") == -1) {
	    return -2;
	}
	if (receiveCmd(ftp, NULL) != 200) {
	    return -2;
	}    
	
	ftp->ssl_private = 0;
	
	return -1;
    }
    
    ftp->ssl_private = 1;
    
    return 0;
#endif  /* NO_SSL */
}

int handleFeat(pftp_server_t ftp, const char *feat_data)
{
    /* AUTH TLS && PBSZ && PROT ==> Handles auth tls*/
    char *s, *last = strchr(feat_data, '\n');
    
    ftp->tls = 0;
  
    if (!last)
	return -1;
    
    for (;;) {
	last++;
	if (last[0] == '\r') last++;
	while (last[0] == ' ') last++;
	
	s = strchr(last, '\n');
	
	if (!s)
	    break;
	
	if (strncasecmp(last, "AUTH TLS", s - last) == 0 || 
	    strncasecmp(last, "PBSZ", s - last) == 0 ||
	    strncasecmp(last, "PROT", s - last) == 0) {
	    ftp->tls++;
	} else if (strncasecmp(last, "REST", s - last) == 0) {
	    ftp->resume = 1;
	} else if (strncasecmp(last, "CDUP", s - last) == 0) {
	    ftp->cdup = 1;
	} else if (strncasecmp(last, "PRET", s - last) == 0) {
	    ftp->pret = 1;
	} else if (strncasecmp(last, "MDTM", s - last) == 0 
		   || strncasecmp(last, "MDTM ", 5) == 0) {
	    ftp->mdtm = 1;
	} else if (strncasecmp(last, "SITE CHMOD", s - last) == 0) {
	    ftp->chmod = 1;
	}
	
	last = s;
    }
    
    if (ftp->tls == 3) {
	ftp->tls = 1;
    } else {
	ftp->tls = 0;
    }
    
    return 0;
}

int initDataAuth(pftp_server_t ftp, int list)
{
#ifdef NO_SSL
    return 0;
#else
    if (ftp->cmdssl) {
	if ((list && !getValue(ftp, secure_list)) ||
	    (!list && !getValue(ftp, secure_data))) {
	    if (ftp->ssl_private) {
		/* Clear protections */
		if (sendCmd(ftp, "PROT", "C") == -1) {
		    return -1;
		}
		
		if (receiveCmd(ftp, NULL) != 200) {
		    ftp->ssl_private = 1;
		} else {
		    ftp->ssl_private = 0;
		}
	    }
	} else {
	    /* Private protections */
	    if (!ftp->ssl_private) {
		if (sendCmd(ftp, "PROT", "P") == -1) {
		    return -1;
		}
		
		if (receiveCmd(ftp, NULL) != 200) {
		    ftp->ssl_private = 0;
		} else {
		    ftp->ssl_private = 1;
		}
	    }
	}
    }
    
    return 0;
#endif /* NO_SSL */
}

int startDataAuth(pftp_server_t ftp)
{
#ifdef NO_SSL
    return 0;
#else
    if (ftp->cmdssl && ftp->ssl_private) {
	int retdata = pftp_init_SSL(ftp, &ftp->datassl, ftp->datastream);
	
	if (retdata != 0)
	    return retdata;
	
	ftp->dataRecv = intSSLRecv;
	ftp->dataSend = intSSLSend;
    }
    
    return 0;
#endif /* NO_SSL */
}

void stopDataAuth(pftp_server_t ftp)
{
#ifndef NO_SSL
    if (ftp->cmdssl && ftp->ssl_private && ftp->datassl) {
	SSL_shutdown(ftp->datassl);
	SSL_free(ftp->datassl);
	ftp->datassl = NULL;
    }
#endif /* NO_SSL */
}

void restoreAuthData(pftp_server_t ftp, int list)
{
#ifndef NO_SSL
    if (ftp->cmdssl) {
	if (!ftp->ssl_private) {
	    /* Private protections */
	    if (sendCmd(ftp, "PROT", "P") == -1)
		return;      
	    
	    if (receiveCmd(ftp, NULL) != 200) {
		ftp->ssl_private = 0;
	    } else {
		ftp->ssl_private = 1;
	    }
	}
    }
#endif /* NO_SSL */
}

int pftp_fxp(pftp_server_t srcftp, const char *srcfile, pftp_server_t destftp, 
	     const char *destfile, char ascii, uint64_t resume_from, 
	     uint64_t append_to, uint64_t total_size)
{
    char *data = NULL;
    struct sockaddr_in addr;
    int need_abort, i, retdata;
    unsigned short srcret = 0, destret = 0;
    pftp_server_t pasvftp, portftp;

#ifndef NO_SSL
    if (srcftp->sftp || destftp->sftp) {
	/* Not supported. */
	return -1;
    }
#endif
    
#ifndef NO_SSL
    if (srcftp->ssl_private) {
	if (sendCmd(srcftp, "PROT", "C") == -1 || 
	    receiveCmd(srcftp, NULL) != 200) {
	    return -1;
	}
	srcftp->ssl_private = 0;
    }

    if (destftp->ssl_private) {
	if (sendCmd(destftp, "PROT", "C") == -1 || 
	    receiveCmd(destftp, NULL) != 200) {
	    restoreAuthData(srcftp, 0);
	    return -1;
	}
	destftp->ssl_private = 0;
    }
#endif

    need_abort = 0;

    /* These are switched because for switch them first thing */
    if (srcftp->pasv) {
	pasvftp = destftp;
	portftp = srcftp;
    } else {
	pasvftp = srcftp;
	portftp = destftp;
    }

    if (getMode(srcftp, (ascii == 'B')) == -1) {
	if (getMode(srcftp, 0) == -1 || getMode(destftp, 0) == -1)
	    return -1;
    } else {
	if (getMode(destftp, (ascii == 'B')) == -1) {
	    if (getMode(srcftp, 0) == -1 || getMode(destftp, 0) == -1)
		return -1;
	}
    }
  
    for (i = 0; i < 2; i++) {	
	if (need_abort) {
	    abortData(srcftp);
	    abortData(destftp);
	    need_abort = 0;
	}	
	if (pasvftp == srcftp) {
	    pasvftp = destftp;
	    portftp = srcftp;
	} else {
	    pasvftp = srcftp;
	    portftp = destftp;
	}

	if (pasvftp->pret) {
	    if (pasvftp == srcftp) {
		if (setupPretPasv(pasvftp, "RETR", srcfile) == -1)
		    continue;
	    } else { // pasvftp == destftp
		if (setupPretPasv(pasvftp, "STOR", destfile) == -1)
		    continue;
	    }
	}

	if (sendCmd(pasvftp, "PASV", NULL) == -1) continue;
	if (receiveCmd(pasvftp, &data) != 227) {
	    if (data) free(data);
	    data = NULL;
	    continue;
	}   
	if (parsePasvResponse(pasvftp, data, &addr) != 0) {
	    free(data);
	    data = NULL;
	    continue;
	}
	if (data) free(data);   
	data = NULL;
	
	if (generatePortData(portftp, &addr, &data, 1) == -1) {
	    if (data) free(data);
	    data = NULL;
	    continue;
	}    
	if (sendCmd(portftp, "PORT", data) == -1) {
	    free(data);
	    data = NULL;
	    continue;
	}
	if (data) free(data);   
	data = NULL;
	if (receiveCmd(portftp, NULL) != 200) continue;
	need_abort = 1;

	if (resume_from > 0) {
	    char *tmpstr = malloc(50);
	    snprintf(tmpstr, 50, "%" SCNu64, resume_from);

	    if (sendCmd(srcftp, "REST", tmpstr) == -1) {
		free(tmpstr);
		continue;
	    }
	    retdata = receiveCmd(srcftp, NULL);
	    if (retdata != 150 && retdata != 125) {
		free(tmpstr);
		continue;
	    }
	    free(tmpstr);
	}
	if (append_to > 0) {
	    char *tmpstr = malloc(50);
	    snprintf(tmpstr, 50, "%" SCNu64, append_to);
	    
	    if (sendCmd(destftp, "REST", tmpstr) == -1) {
		free(tmpstr);
		continue;
	    }
	    retdata = receiveCmd(destftp, NULL);
	    if (retdata != 150 && retdata != 125) {
		free(tmpstr);
		continue;
	    }
	    free(tmpstr);
	}

	if (sendCmd(srcftp, "RETR", srcfile) == -1) continue;
	if (sendCmd(destftp, "STOR", destfile) == -1) continue;
	retdata = receiveCmd(srcftp, NULL);
	if (retdata != 150 && retdata != 125) continue;
	retdata = receiveCmd(destftp, NULL);
	if (retdata != 150 && retdata != 125) continue;

	break;
    }

    if (i == 2) {
	if (need_abort) {
	    abortData(srcftp);
	    abortData(destftp);
	}
	restoreAuthData(srcftp, 0);
	restoreAuthData(destftp, 0);	
	return -1;
    }

/*
  send_comm(srcftp, PFTP_FILETRANSFER, (pftp_param_t)&resume_from,
  (pftp_param_t)&total_size);
  send_comm(srcftp, PFTP_INITTRANSFER, 1, 0);
  send_comm(destftp, PFTP_INITTRANSFER, 2, 0);
*/
    if (receiveCmd2(srcftp, destftp, &srcret, &destret, NULL, NULL) == -1) {
	abortData(srcftp);
	abortData(destftp);
	restoreAuthData(srcftp, 0);
	restoreAuthData(destftp, 0);	
/*
  send_comm(srcftp, PFTP_DONETRANSFER, 1, 0);
  send_comm(destftp, PFTP_DONETRANSFER, 1, 0);
*/
	return -1;
    }

    restoreAuthData(srcftp, 0);
    restoreAuthData(destftp, 0);	

    if (srcret != 226 || destret != 226) {
/*
  send_comm(srcftp, PFTP_DONETRANSFER, 1, 0);
  send_comm(destftp, PFTP_DONETRANSFER, 1, 0);
*/
	return -1;	
    } else {
/*	
  send_comm(srcftp, PFTP_DONETRANSFER, 0, 0);
  send_comm(destftp, PFTP_DONETRANSFER, 0, 0);
*/
	return 0;
    }
}

system_t parseSYST(const char *data)
{
    if (strncmp(data, "UNIX", 4) == 0) {
	return UNIX;
    }
    if (strncasecmp(data, "Windows", 7) == 0) {
	return DOS;
    }

    return UNIX; /* Default */
}

void pftp_clear_dir_cache(pftp_server_t ftp, const char *directory)
{
    if (directory) {
	size_t len = strlen(directory), c;
	char *real_path = NULL;
	if (len > 1 && directory[len - 1] == '/') len--;

	pftp_curdir(ftp, &real_path, 0);
	
	if (real_path) {
	    if (directory[0] == '/') {
		real_path = realloc(real_path, len + 1);
		strncpy(real_path, directory, len);
		real_path[len] = '\0';
	    } else {
		c = strlen(real_path);
		real_path = realloc(real_path, c + len + 2);
		
		if (real_path[c - 1] != '/') {
		    strcat(real_path, "/");
		    c++;
		}
				
		strncat(real_path, directory, len);
		len += c;
		real_path[len] = '\0';
	    }
	    
	    if (!pftp_compactPath(&real_path)) {
		pftp_curdir(ftp, &real_path, 1);
	    }

	    len = (real_path ? strlen(real_path) : 0);
	} else {
	    real_path = malloc(len + 1);
	    strncpy(real_path, directory, len);
	    real_path[len] = '\0';
	}

	if (len > 1 && real_path[len - 1] == '/') len--;

	if (real_path) {
	    for (c = 0; c < ftp->cached; c++) {
		if (strncmp(real_path, ftp->cache_path[c], len) == 0) {
		    pftp_free_fdt(ftp->cache_dir[c]);
		    free(ftp->cache_path[c]);
		    ftp->cached--;
		    memmove(ftp->cache_dir + c, ftp->cache_dir + c + 1, 
			    (ftp->cached - c) * sizeof(pftp_directory_t));
		    memmove(ftp->cache_path + c, ftp->cache_path + c + 1, 
			    (ftp->cached - c) * sizeof(char *));
		    break;
		}
	    }
	    free(real_path);
	}
    } else {
	size_t c;
	for (c = 0; c < ftp->cached; c++) {
	    pftp_free_fdt(ftp->cache_dir[c]);
	    free(ftp->cache_path[c]);
	}
	if (ftp->cache_dir) free(ftp->cache_dir);
	if (ftp->cache_path) free(ftp->cache_path);
	ftp->cache_dir = NULL;
	ftp->cache_path = NULL;
	ftp->cached = 0;
    }
}

int pftp_userauth(pftp_server_t ftp, char **username, char **password)
{
    return send_comm(ftp, PFTP_NEEDPASS, (pftp_param_t) username,
		     (pftp_param_t) password);
}

void pftp_status_filetransfer(pftp_server_t ftp, uint64_t resume_from,
			     uint64_t total_size)
{
    send_comm(ftp, PFTP_FILETRANSFER, (pftp_param_t) &resume_from, 
	      (pftp_param_t) &total_size);
}

int send_comm(pftp_server_t ftp, pftp_msg_t msg, 
	      pftp_param_t param1, pftp_param_t param2)
{
    return ftp->comm_central(msg, ftp, ftp->settings->userdata, 
			     param1, param2);
}

int pftp_chmod(pftp_server_t ftp, const char *file, unsigned short access)
{
    int ret;
    char *tmp;
    size_t len;

#ifndef NO_SSL
    if (ftp->sftp) {
	return pftp_sftp_chmod(ftp->sftp, file, access);
    }
#endif

    if (!ftp->chmod) {
	/* No supported. */
	pftp_status_message(ftp, "WARN: Changing file permissions isn\'t supported.");
	return -1;
    }

    len = strlen(file);

    tmp = malloc(len + 11);
    memcpy(tmp, "CHMOD XXX ", 10);
    memcpy(tmp + 10, file, len + 1);

    tmp[6] = '0' + ((access / (8 * 8)) % 8);
    tmp[7] = '0' + ((access % (8 * 8)) / 8);
    tmp[8] = '0' + (access % 8);
    
    if (sendCmd(ftp, "SITE", tmp) == -1) {
	free(tmp);
	return -1;
    }
    
    free(tmp);
    
    ret = receiveCmd(ftp, NULL);

    if (ret < 500) {
	ftp->chmod = 1;
	if (ret >= 300)
	    return -1;
    } else {
	ftp->chmod = 0;
	return -1;
    }
    
    return 0;
}

/* Set file modification date on FTP server.
   Returns -1 on error and 0 on OK. */
int pftp_mdtm(pftp_server_t ftp, const char *file, const struct tm *date)
{
    int ret;
    char *tmp;
    size_t len;

#ifndef NO_SSL
    if (ftp->sftp) {
	return pftp_sftp_mdtm(ftp->sftp, file, date);
    }
#endif

    if (!ftp->mdtm) {
	/* No supported. */
	pftp_status_message(ftp, 
			    "WARN: Setting file modification time isn\'t supported.");
	return -1;
    }

    len = strlen(file);

    tmp = malloc(len + 16);
    snprintf(tmp, len + 16, "%04u%02u%02u%02u%02u%02u;%s", date->tm_year + 1900,
	     date->tm_mon + 1, date->tm_mday, date->tm_hour, date->tm_min, 
	     date->tm_sec, file);

    if (sendCmd(ftp, "MDTM", tmp) == -1) {
	free(tmp);
	return -1;
    }
    
    free(tmp);
    
    ret = receiveCmd(ftp, NULL);

    if (ret < 500) {
	ftp->mdtm = 1;
	if (ret >= 300)
	    return -1;
    } else {
	ftp->mdtm = 0;
	return -1;
    }
    
    return 0;
}
