/* -*- pftp-c -*- */
#ifndef PFTP_SFTP_H
#define PFTP_SFTP_H

typedef struct pftp_sftp_s *pftp_sftp_t;

/* Login, make the connection to the ftp. Takes a settings structure pointer
   that must exist until logout is called. 
   Returns NULL if error. Any error messages are sent to status */
pftp_sftp_t pftp_sftp_login(pftp_server_t ftp); 

/* Logout, disconnect from a sftp and free memory of pointer */
void pftp_sftp_logout(pftp_sftp_t *ftp);

/* Change current directory, 
   If newdir is not null:
     The point newdir points at must be NULL.
     newdir is set to what the current path is then.
   returns 0 on success, -1 on error */
int pftp_sftp_cd(pftp_sftp_t sftp, const char *dir, char **newdir);

/* Remove directory, returns 0 on success, -1 on error */
int pftp_sftp_rmdir(pftp_sftp_t sftp, const char *dir);

/* Create directory, returns 0 on success, -1 on error */
int pftp_sftp_mkdir(pftp_sftp_t sftp, const char *dir);

/* Get current directory, The pointer currentDirectory must point to NULL and
   will be left to NULL if any errors accur. Otherwise it will be set to 
   current directory. User must free the string themself.  */
void pftp_sftp_curdir(pftp_sftp_t sftp, char **currentDirectory);

/* List current directory. Will fill the structure dir with info. 
   Returns 0 on no errors, -1 on errors (and then dir is not touched) */
int pftp_sftp_ls(pftp_sftp_t sftp, pftp_directory_t *dir);

/* Download a file from current direcory.
   filename - is the file's name.
   ascii - if set to 'B' then download is in binary mode otherwise ascii mode.
   fh - must point to a writeable stream.
   resume_from - Tells from where we should start reading from sftp 
   (if supported by server). If it's supported, get will continue. But if
   resume_from is > 0 and it's not supported, error will be returned.
   total_size - Information about total filesize, will be used when sending
   info to update function.
   crc - If not NULL, after file download this will be the crc32 code of 
   downloaded data. (So, dont rely on this when using resume).

   Returns 0 on OK and -1 on error.
 */
int pftp_sftp_get(pftp_sftp_t sftp, const char *filename, char ascii, 
		  FILE *fh, uint64_t resume_from, uint64_t total_size, 
		  uint32_t *crc);

/* Upload a file to the current direcory.
   filename - is the file's name.
   ascii - if set to 'B' then upload is in binary mode otherwise ascii mode.
   fh - must point to a readable stream.
   resume_from - Tells from where we should start writing to the file on the
   ftp (if supported by server). If it's supported, put will continue. But if
   resume_from is > 0 and it's not supported, error will be returned.
   total_size - Information about total filesize, will be used when sending
   info to update function.

   Returns 0 on OK and -1 on error.
 */
int pftp_sftp_put(pftp_sftp_t sftp, const char *filename, char ascii, 
		  FILE *fh, uint64_t resume_from, uint64_t total_size);

/* Remove a file in current directory 
   Return 0 on OK and -1 on error. */
int pftp_sftp_rm(pftp_sftp_t sftp, const char *filename);

/* Rename a file from filename to new in current directory
   Return 0 on OK and -1 on error. */
int pftp_sftp_rename(pftp_sftp_t sftp, const char *filename, const char *new);

/* Set access to a file on SFTP server. 
   Access should be between 000 and 777 in octal.
   Returns -1 on error and 0 on OK. */
int pftp_sftp_chmod(pftp_sftp_t sftp, const char *file, unsigned short access);

/* Set file modification date on SFTP server.
   Returns -1 on error and 0 on OK. */
int pftp_sftp_mdtm(pftp_sftp_t sftp, const char *file, const struct tm *date);

#endif /* PFTP_SFTP_H */
