/* -*- pftp-c -*- */
#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <inttypes.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <fcntl.h>
#include <errno.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <stdarg.h>
#include <netdb.h>
#include <string.h>
#include <limits.h>
#if TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif

#ifndef SCNu64
# define SCNu64 "llu"
#endif 

#ifdef WITH_DMALLOC
# include <dmalloc.h>
#endif

#ifndef NO_SSL
# include <openssl/ssl.h>
# include <openssl/rand.h>
# include <openssl/err.h>
#endif /* NO_SSL */

#include "pftp.h"
#include "pftp_sfvcheck.h"

/* Some nice macros */

#ifndef min
# define min(__x, __y) ((__x) > (__y) ? (__y) : (__x))
#endif /* min */

#ifndef TEMP_FAILURE_RETRY
# define TEMP_FAILURE_RETRY(expression) \
   (__extension__							      \
     ({ long int __result;						      \
        do __result = (long int) (expression);				      \
        while (__result == -1L && errno == EINTR);			      \
        __result; }))
#endif

#define getValue(__f, __v) \
  (__f->settings->__v == -1 ? __f->settings->global->__v : __f->settings->__v)
#define getStrValue(__f, __v) \
  (__f->settings->__v == NULL ? __f->settings->global->__v:__f->settings->__v)

/* Some internal data types and default values */

typedef int socket_t; 

#define DEFAULT_TIMEOUT (100000L)

typedef enum { DOWNLOAD = 0, UPLOAD, 
	       PORT_CONN, CONNECT, SSL_HANDSHAKE } wait_mode_t;
typedef enum { UNIX = 0, DOS } system_t;

static const int TIMEOUT[] = { 15, 15, 15, 15, 15 }; // All values are in sec

typedef ssize_t (*RecvFunc_t)(pftp_server_t ftp, int socket, void *buffer, 
			      size_t size);
typedef ssize_t (*SendFunc_t)(pftp_server_t ftp, int socket, 
			      const void *buffer, size_t size);

struct pftp_server_s {
    // sockets
    socket_t cmdstream, datastream, port_socket;
    // buffer
    char *inbuffer;
    size_t inbuffer_len;
    // capabilities
    int resume, cur_binary, binary, abort, tls, site, help, cdup, hidden_ls;
    int pasv;
    // used with speed (or will be used with speed)
    size_t lastgoodbuffersize;
    // data for waiting
    wait_mode_t waitmode;
    time_t lastdata;
    
#ifndef NO_SSL
    SSL *cmdssl, *datassl;
    int ssl_private;
#endif /* NO_SSL */
    
    // used to beable to switch between secure and insecure transfers
    RecvFunc_t cmdRecv, dataRecv;
    SendFunc_t cmdSend, dataSend;

    // buffer for crc calc
    uint32_t cur_crc;
    
    // is it a unix or?
    system_t system;

    // directory cache
    char *cur_path;
    pftp_directory_t *cache_dir;
    char **cache_path;
    size_t cached;

    pftp_comm_centralFunc_t comm_central;
    pftp_settings_t settings;
};

struct speed_data_s {
    uint64_t total, done;
    size_t buffersize;
    int unknown;
    // int lastchange;
    double speed; //, lastspeed;
    float percent;
    
    uint64_t lastdata;
    struct timeval lasttime;
};

typedef struct speed_data_s *speed_data_t;

/* Static lib vars */
#ifndef NO_SSL
static SSL_CTX *ssl_ctx = NULL;
static X509_STORE *x509_store = NULL;
static char ssl_rand_file[256];
static pftp_server_t ssl_cur_ftp = NULL;
#endif /* NO_SSL */
static int libInited = 0;

/* Internal functions */

/* All functions (almost) returns 0 on OK and -1 on error (or -2 even) */

/* Wait for a cmd on ftp control connection and return that */
#define receiveCmd(ftp, data) _receiveCmd(ftp, data, 0)
static short _receiveCmd(pftp_server_t ftp, char **data, int abort);
/* Wait for a cmd on two ftp control connections and return those... */
static int receiveCmd2(pftp_server_t ftp1, pftp_server_t ftp2, short *ret1, 
		       short *ret2, char **data1, char **data2);
/* Send a cmd to a ftp */
static int sendCmd(pftp_server_t ftp, const char *cmd ,const char *buffer);
/* Send data over data connection to ftp */
static int sendData(pftp_server_t ftp, const void *buffer, size_t buffersize,
		    size_t *done, speed_data_t speed);
/* Receive data from data connection to ftp */
static int receiveData(pftp_server_t ftp, void **buffer, size_t *buffersize, 
		       speed_data_t speed);
/* Connect the control connection for ftp */
static int connectCmd(pftp_server_t ftp);
/* Format a message sent to client using statusMsgFunc */
static void statusMessage(pftp_server_t ftp, const char *format, ...);
/* Get a data connection to ftp server for given cmds */
static int getDataConnection(pftp_server_t ftp, const char *precmd, 
			     const char *predata, const char *cmd, 
			     const char *data);
/* When sent cmds for data connection, wait for some response */
static int continueDataConnection(pftp_server_t ftp, const char *precmd, 
				  const char *predata, const char *cmd, 
				  const char *data);
/* Setup a passive data connection */
static int getPassiveConnection(pftp_server_t ftp);
/* Parser help function for getPassiveConnection */
static int parsePasvResponse(pftp_server_t ftp, const char *data, 
			     struct sockaddr_in *addr);
/* Generator help function for getDirectConnection */
static int generatePortData(pftp_server_t ftp, const struct sockaddr_in *addr, 
			    char **tmpstr, int dont_use_myaddr);
/* Setup a port data connection */
static int getDirectConnection(pftp_server_t ftp);
/* Wait for incoming data connection, when using port */
static int acceptDirectConnection(pftp_server_t ftp);
/* Compare two command responses, ie 221 == 225 */
static int cmdcmp(unsigned short code1, unsigned short code2);
/* Send account data if that's needed */
static int sendAccount(pftp_server_t ftp);
/* Abort current data transfer */
static int abortData(pftp_server_t ftp);
/* Send command via comm_central */
static int send_comm(pftp_server_t ftp, pftp_msg_t msg, 
		     pftp_param_t param1, pftp_param_t param2);
/* "Parse" SYST response */
static system_t parseSYST(const char *data);
/* Helper function for speed calculation during data transfers */
static speed_data_t initSpeed(pftp_server_t ftp, uint64_t total, 
			      uint64_t start);
/* Helper function for speed calculation during data transfers */
static void updateSpeed(speed_data_t speed, size_t data);
/* Helper function for speed calculation during data transfers */
static void freeSpeed(pftp_server_t ftp, speed_data_t *speed);
/* Helper function for turning a hostname or IP into a sockaddr_in */
static int parseHostname(const char *hostname, unsigned short port, 
			 struct sockaddr_in *addr);
/* Connect a socket and wait for connection */
static int connectSocket(pftp_server_t ftp, socket_t *sock, 
			 struct sockaddr_in *addr);
/* Creates a nonblocking socket */
static socket_t createNonblockingSocket(pftp_server_t ftp);
/* Helper function for data transfers (makes timeout and cancel work) */
static void startWait(pftp_server_t ftp, wait_mode_t mode, intptr_t stats);
/* Helper function for data transfers (makes timeout and cancel work) */
static int doWait(pftp_server_t ftp, speed_data_t speed, int gotdata, 
		  const char *op);
/* Helper function for data transfers (makes timeout and cancel work) */
static void endWait(pftp_server_t ftp, wait_mode_t mode, intptr_t error);
/* "Parse" FEAT response */
static int handleFeat(pftp_server_t ftp, const char *feat_data);
/* Init secure command connection */
static int initAuth(pftp_server_t ftp);
/* Init secure data connection */
static int initDataAuth(pftp_server_t ftp, int list);
/* Start secure data connection */
static int startDataAuth(pftp_server_t ftp);
/* End secure data connection */
static void stopDataAuth(pftp_server_t ftp);
/* When secure data connection was interrupted, restore things */
static void restoreAuthData(pftp_server_t ftp, int list);
/* Internal recv and send function, used for different cases */
static ssize_t intRecv(pftp_server_t ftp, int socket, void *buffer, 
		       size_t len);
static ssize_t intCRCRecv(pftp_server_t ftp, int socket, void *buffer, 
			  size_t len);
static ssize_t intSend(pftp_server_t ftp, int socket, const void *buffer, 
		       size_t len);
#ifndef NO_SSL
static ssize_t intSSLRecv(pftp_server_t ftp, int socket, void *buffer, 
			  size_t len);
static ssize_t intCRCSSLRecv(pftp_server_t ftp, int socket, void *buffer, 
			     size_t len);
static ssize_t intSSLSend(pftp_server_t ftp, int socket, const void *buffer, 
			  size_t len);
/* Init SSL part of secure connection */
static int initSSL(pftp_server_t ftp, SSL **ssl, socket_t sock);
/* "Verify" a SSL Certificate */
static int sslVerify(int ok, X509_STORE_CTX *ctx);
#endif /* NO_SSL */

void pftp_initlib(void)
{
    if (libInited) return;
    
#ifndef NO_SSL
    
    SSL_load_error_strings();                /* readable error messages */
    SSL_library_init();                      /* initialize library */
    
    RAND_file_name(ssl_rand_file, sizeof(ssl_rand_file));
    
    if (RAND_egd(ssl_rand_file) < 0) {
	if (!RAND_load_file(ssl_rand_file, -1) || !RAND_status())
	    strcpy(ssl_rand_file, "");
    }
    
#if SSLEAY_VERSION_NUMBER < 0x0800
    ssl_ctx = SSL_CTX_new();
    
    if (ssl_ctx) {
	X509_set_default_verify_paths(ssl_ctx->cert);
    }
#else
    //  SSLeay_add_ssl_algorithmns();
    ssl_ctx = SSL_CTX_new(SSLv23_client_method());
    
    if (ssl_ctx) {
	SSL_CTX_set_options(ssl_ctx, SSL_OP_ALL);
	SSL_CTX_set_verify(ssl_ctx, SSL_VERIFY_PEER, sslVerify);
	SSL_CTX_set_default_verify_paths(ssl_ctx);    
    }
#endif /* SSLEAY_VERSION_NUMBER */ 
#endif /* NO_SSL */
    
    libInited = 1;
}

void pftp_donelib(void)
{
    if (!libInited) return;
    
#ifndef NO_SSL
    if (ssl_ctx) {
	SSL_CTX_free(ssl_ctx);
	ssl_ctx = NULL;
    }
    
    if (strlen(ssl_rand_file) > 0)
	RAND_write_file(ssl_rand_file);
#endif /* NO_SSL */
    
    libInited = 0;
}

static int _int_login(pftp_server_t ftp)
{
    short retdata;

    /* Connect to ftp server */
    if (connectCmd(ftp) == -1) {
	return -1;
    }

#ifndef NO_SSL
    if (getValue(ftp, implicid_ssl)) {
	if (initSSL(ftp, &ftp->cmdssl, ftp->cmdstream) != 0) {
	    return -1;
	}

	ftp->ssl_private = 1;
	ftp->cmdRecv = intSSLRecv;
	ftp->cmdSend = intSSLSend;
    }
#endif
    
    /* Wait for welcome message */
    do {
	retdata = receiveCmd(ftp, NULL);
    } while (cmdcmp(retdata, 120) == 0);
    
    if (retdata != 220) {   
	return -1;
    }

    return 0;
}

pftp_server_t pftp_login(pftp_settings_t settings, 
			 pftp_comm_centralFunc_t comm_central)
{
    pftp_server_t ret;
    int retdata, feat = 0;
    char *data = NULL;
    
    if (!libInited)
	return NULL;
    
    if (!settings)
	return NULL;
    
    ret = malloc(sizeof(struct pftp_server_s));
    memset(ret, 0, sizeof(struct pftp_server_s));
    
    ret->comm_central = comm_central;
    ret->settings = settings;
    ret->cmdstream = -1;
    ret->datastream = -1;
    ret->port_socket = -1;
    ret->inbuffer = NULL;
    ret->inbuffer_len = 0;
    ret->lastgoodbuffersize = 32767;
    
    ret->cmdRecv = intRecv;
    ret->cmdSend = intSend;

    ret->pasv = getValue(ret, pasv);
    
    /* Capabilities are set to -5, as to note "unknown".
       This is not a very good idea as -5 is "true". But well... */
    ret->resume = -5;
    ret->binary = -5;
    ret->abort = -5;
    ret->help = -5;
    ret->site = -5;
    ret->tls = -5;
    ret->cdup = -5;

    if (_int_login(ret)) {
	free(ret);
	return NULL;
    }
    
    if (sendCmd(ret, "FEAT", NULL) == -1) {
	free(ret);
	return NULL;
    }
    
    if ((retdata = receiveCmd(ret, &data)) == 211) {
	handleFeat(ret, data);
	free(data);
	feat = 1;
    } else {
	if (data) free(data);
	if (retdata == -1 && _int_login(ret)) {
	    free(ret);
	    return NULL;
	}
    }
    
    data = NULL;
    
    if (!getValue(ret, implicid_ssl) && 
	(ret->tls == 1 || getValue(ret, use_secure))) {
	if (initAuth(ret) == -2 && _int_login(ret)) {
	    free(ret);
	    return NULL;
	}
    }

    if (!ret->settings->username || !strlen(ret->settings->username)) {
	if (!send_comm(ret, PFTP_NEEDPASS, 
		       (pftp_param_t) &ret->settings->username, NULL)) {
	    free(ret);
	    return NULL;
	}
    }
    
    /* Send username */
    if (sendCmd(ret, "USER", ret->settings->username) == -1) {
	free(ret);
	return NULL;
    }
    
    /* Wait for logged in */
    retdata = receiveCmd(ret, NULL);
    
    if (retdata == 331) {
	/* Send password */
	if (!ret->settings->password || !strlen(ret->settings->password)) {
	    if (!send_comm(ret, PFTP_NEEDPASS, NULL, 
			   (pftp_param_t) &ret->settings->password)) {
		free(ret);
		return NULL;
	    }
	}

	if (sendCmd(ret, "PASS", ret->settings->password) == -1) {
	    free(ret);
	    return NULL;
	}
	
	/* Wait for logged in */
	retdata = receiveCmd(ret, NULL);
	
	if (cmdcmp(retdata, 230) && cmdcmp(retdata, 200)) {
	    if (send_comm(ret, PFTP_NEEDPASS, 
			  (pftp_param_t) &ret->settings->username, 
			  (pftp_param_t) &ret->settings->password)) {
		free(ret);
		return pftp_login(settings, comm_central);
	    } else {
		free(ret);
		return NULL;
	    }
	}
    } else if (cmdcmp(retdata, 230)) {
	if (send_comm(ret, PFTP_NEEDPASS,
		      (pftp_param_t)&ret->settings->username, 
		      (pftp_param_t)&ret->settings->password)) {
	    free(ret);
	    return pftp_login(settings, comm_central);
	} else {
	    free(ret);
	    return NULL;
	}
    }
    
    /* Ask for some info */
    if (sendCmd(ret, "SYST", NULL) == -1) {
	free(ret);
	return NULL;
    }
    
    if (receiveCmd(ret, &data) == 215) {
	ret->system = parseSYST(data);
	free(data);
	data = NULL;
    } else {
	ret->system = UNIX; /* Guessing */
	if (data) free(data);
	data = NULL;
    }
    
    if (!feat) {
	if (sendCmd(ret, "FEAT", NULL) == -1) {
	    free(ret);
	    return NULL;
	}
	
	if (receiveCmd(ret, &data) == 211) {
	    handleFeat(ret, data);
	    free(data);	    
	} else {
	    ret->tls = 0;
	    
	    if (data)
		free(data);
	}
	
	data = NULL;
    }

    /* Check for help & site support */
    if (sendCmd(ret, "HELP", "SITE") == -1) {
	free(ret);
	return NULL;
    }
    
    if (cmdcmp(receiveCmd(ret, NULL), 200) == 0) {
	ret->help = 1;
	ret->site = 1; // ?
    } else {
	ret->help = 0; // ?
	ret->site = 0; // ?
    }

    /* Check for resume support */
    if (ret->resume == -5) {
	if (sendCmd(ret, "REST", "1024") == -1) {
	    free(ret);
	    return NULL;
	}
	
	retdata = receiveCmd(ret, NULL);
	
	if (retdata != 350) {
	    ret->resume = 0;
	} else {
	    ret->resume = 1;
	}

	/* For most ftp servers, the abort below will fix this, but not all
	   proftpd for one. */
	if (sendCmd(ret, "REST", "0") == -1) {
	    free(ret);
	    return NULL;
	}
	
	retdata = receiveCmd(ret, NULL);
    }
    
    if (ret->abort == -5) {
	if (sendCmd(ret, "ABOR", NULL) == -1) {
	    free(ret);
	    return NULL;
	}
	
	retdata = receiveCmd(ret, NULL);
	
	if (cmdcmp(retdata, 220) == 0) {
	    ret->abort = 1;
	} else {
	    ret->abort = 0;
	}
    }

    if (ret->cdup) { // -5 or 1
	char *s1, *s2;
	/* The reason we do this even if we know CDUP is supported
	   is because some ftp servers send welcome text with first CWD or
	   CDUP issued */
	if (sendCmd(ret, "PWD", NULL) == -1) {
	    free(ret);
	    return NULL;
	}

	if (receiveCmd(ret, &data) != 257) {
	    free(ret);
	    return NULL;
	}

	if ((s1 = strchr(data, '\"')) && (s2 = strchr(s1 + 1, '\"'))) {
	    s1++;
	    memmove(data, s1, s2 - s1);
	    data[s2 - s1] = '\0';
	}

	if (sendCmd(ret, "CDUP", NULL) == -1) {
	    free(ret);
	    free(data);
	    return NULL;
	}
	
	retdata = receiveCmd(ret, NULL);
	
	if (retdata != 250 && retdata != 550) {
	    // 550 = Some serverns whine when you do CDUP in home directory
	    ret->cdup = 0;
	} else {
	    ret->cdup = 1;
	}

	if (sendCmd(ret, "CWD", data) == -1) {
	    free(ret);
	    free(data);
	    return NULL;
	}

	free(data);	    
	data = NULL;

	if (receiveCmd(ret, NULL) != 250) {
	    free(ret);
	    return NULL;
	}
    }
    
    if (sendCmd(ret, "TYPE", "I") == -1) {
	free(ret);
	return NULL;
    }
    
    if (receiveCmd(ret, NULL) != 200) {
	ret->binary = 0;
	ret->cur_binary = 0;
    } else {
	ret->binary = 1;
	
	if (sendCmd(ret, "TYPE", "A") == -1) {
	    free(ret);
	    return NULL;
	}
	
	if (receiveCmd(ret, NULL) != 200) {
	    // Must be supported by RFC 959
	    free(ret);
	    return NULL;
	}
	
	ret->cur_binary = 0;
    }

    {
	pftp_directory_t tmp;
	memset(&tmp, 0, sizeof(pftp_directory_t));
	ret->hidden_ls = 1;
	if (pftp_ls(ret, &tmp, 0, NULL)) {
	    ret->hidden_ls = 0;	    
	}
	pftp_free_fdt(tmp);
    }
    
    pftp_curdir(ret, &data, 0);    
    if (data) free(data);
    
    return ret;
}

int parseHostname(const char *hostname, unsigned short port, 
		  struct sockaddr_in *addr)
{
    //    host = gethostbyname2(hostname, AF_INET);
    struct hostent *host = gethostbyname(hostname);
    
    if (!host) {
	if (inet_aton(hostname, &addr->sin_addr)) {
	    addr->sin_family = AF_INET;
	} else {
	    return -1;
	}
    } else {
	addr->sin_family = host->h_addrtype;
	memcpy(&addr->sin_addr, host->h_addr, host->h_length);
    }
    
    addr->sin_port = htons(port);
    
    return 0;
}

socket_t createNonblockingSocket(pftp_server_t ftp)
{
    int oldflags;
    socket_t ret = socket(PF_INET, SOCK_STREAM, 0);
    
    if (ret == -1) {
	statusMessage(ftp, "ERR: Unable to create a socket: %s", 
		      strerror(errno));
	return -1;
    }
    
    oldflags = fcntl(ret, F_GETFL, 0);
    
    if (oldflags == -1) {
	statusMessage(ftp, "ERR: Unable to get socket stream status flags: %s",
		      strerror(errno));
	close(ret);
	return -1;
    } 
    
    if (fcntl(ret, F_SETFL, oldflags | O_NONBLOCK) == -1) {
	statusMessage(ftp, "ERR: Unable to set socket stream status flags: %s",
		      strerror(errno));
	close(ret);
	return -1;
    }
    
    return ret;
}

/* Special return status:
   -2 for not a change
   -1 for unable to connect
   0 for ok */
int connectSocket(pftp_server_t ftp, socket_t *sock, struct sockaddr_in *addr)
{
    int ret;
    struct timeval timeout;
    fd_set write_set;
    
    (*sock) = createNonblockingSocket(ftp);
    
    if ((*sock) == -1) {
	return -2;
    }

    // Always returns -1 as nonblocking
    ret = connect((*sock), (struct sockaddr *)addr, 
		  sizeof(struct sockaddr_in));

    if (errno != EINPROGRESS && errno != EALREADY) {
	statusMessage(ftp, "ERR: Unable to connect: %s", strerror(errno));
	close((*sock));
	(*sock) = -1;
	return -1;
    }
    
    /* Connected, now wait for the real connection */
    FD_ZERO(&write_set);
    startWait(ftp, CONNECT, 0);
    
    for (;;) {
	timeout.tv_sec = 0;
	timeout.tv_usec = DEFAULT_TIMEOUT;
	FD_SET((*sock), &write_set);
	
	ret = TEMP_FAILURE_RETRY(select(FD_SETSIZE, NULL, &write_set, NULL, 
					&timeout));
	
	if (ret == -1) {
	    statusMessage(ftp, "ERR: Select returns error: %s", 
			  strerror(errno));
	    endWait(ftp, CONNECT, 1);
	    shutdown((*sock), 2);
	    close((*sock));
	    (*sock) = -1;
	    return -1;
	} else if (ret == 0) {
	    if (doWait(ftp, NULL, 0, "CONNECT") == -1) {
		endWait(ftp, CONNECT, 1);
		shutdown((*sock), 2);
		close((*sock));
		(*sock) = -1;
		return -1;
	    }
	} else {
	    break;
	}
    }

    /* OK, Connect is done, now check if were really connected */
    {
	struct sockaddr_in addr;
	socklen_t len = sizeof(addr);
	memset(&addr, 0, len);
	if (getpeername((*sock), (struct sockaddr *)&addr, &len)) {
	    statusMessage(ftp, "ERR: Connect failed: %s", strerror(errno));
	    endWait(ftp, CONNECT, 1);
	    shutdown((*sock), 2);
	    close((*sock));
	    (*sock) = -1;
	    return -1;
	}
    }

    endWait(ftp, CONNECT, 0);
    
    return 0;
}

int connectCmd(pftp_server_t ftp)
{
    size_t h;
    int connected = 0;
    struct sockaddr_in addr;
    const char *hostname;
    unsigned short port;
    
    for (h = 0; h < ftp->settings->hosts && !connected; h++) {
	hostname = ftp->settings->hostname[h];
	port = ftp->settings->port[h];
	
	if (parseHostname(hostname, port, &addr) == -1) {
	    statusMessage(ftp, "ERR: Unable to find host %s", hostname);
	    continue;
	}

	statusMessage(ftp, "Trying %s:%u", hostname, port);
	
	switch (connectSocket(ftp, &ftp->cmdstream, &addr)) {
	case 0:
	    connected = 1;
	    break;
	case -1:
	    /* Try next */
	    break;
	default:
	case -2:
	    /* Give up */
	    h = ftp->settings->hosts;
	    break;
	}
    }
    
    if (!connected) {
	statusMessage(ftp, "ERR: Unable to connect to ftp server");
	close(ftp->cmdstream);
	ftp->cmdstream = -1;
	return -1;   
    } else {
	statusMessage(ftp, "Connected.");
    }
    
    return 0;
}

void pftp_logout(pftp_server_t *ftp) 
{
    if (ftp && (*ftp)) {
	if ((*ftp)->datastream != -1) {
	    abortData((*ftp));
	    stopDataAuth((*ftp));
	}
	
	if ((*ftp)->cmdstream != -1) {
	    sendCmd((*ftp), "QUIT", NULL);
	}
	
	if ((*ftp)->cmdstream != -1) {
	    receiveCmd((*ftp), NULL);
	    
#ifndef NO_SSL
	    if ((*ftp) && (*ftp)->cmdssl) SSL_shutdown((*ftp)->cmdssl);	    
#endif /* NO_SSL */
	    
	    shutdown((*ftp)->cmdstream, 2);
	    close((*ftp)->cmdstream);
	}    
	
	if ((*ftp)->inbuffer) free((*ftp)->inbuffer);
	
#ifndef NO_SSL
	if ((*ftp)->cmdssl) SSL_free((*ftp)->cmdssl);
	if ((*ftp)->datassl) SSL_free((*ftp)->datassl);
#endif /* NO_SSL */

	if ((*ftp)->cur_path) free((*ftp)->cur_path);
	if ((*ftp)->cache_dir) {
	    size_t d;
	    for (d = 0; d < (*ftp)->cached; d++)
		pftp_free_fdt((*ftp)->cache_dir[d]);	       
	    free((*ftp)->cache_dir);
	} 
	if ((*ftp)->cache_path) {
	    size_t d;
	    for (d = 0; d < (*ftp)->cached; d++)
		free((*ftp)->cache_path[d]);	       
	    free((*ftp)->cache_path);
	}
	
	free((*ftp));
	(*ftp) = NULL;
    }
}

int pftp_compactPath(char **path)
{
    size_t len = strlen((*path));
    char *s1, *s2, *cpy = malloc(len + 2);
    strcpy(cpy, (*path));

    if (len > 1 && cpy[len - 1] != '/') {
	strcpy(cpy + len, "/");
	len++;
    }

    while ((s1 = strstr(cpy, "//"))) {
	memmove(s1, s1 + 1, (len - (s1 - cpy)) + 1);
	len--;
    }

    while ((s1 = strstr(cpy, "/../"))) {
	if (s1 == cpy) {
	    memmove(cpy, cpy + 3, len + 1);
	    len -= 3;
	} else {
	    s1[0] = '\0';
	    s2 = strrchr(cpy, '/');
	    if (s2) {
		memmove(s2 + 1, s1 + 4, (len - ((s1 + 4) - cpy)) + 1);
		len -= (s1 + 4) - (s2 + 1);
	    } else {
		free(cpy);
		return 0;
	    }
	}
    }

    while ((s1 = strstr(cpy, "/./"))) {
	memmove(s1, s1 + 2, (len - ((s1 + 2) - cpy)) + 1);
	len -= 2;
    }

    (*path) = realloc((*path), len + 1);
    memcpy((*path), cpy, len - 1);

    if (len > 1 && (*path)[len-1] == '/')
	(*path)[len-1] = '\0';
    else
	(*path)[len] = '\0';

    free(cpy);

    return 1;
}

static void updateCurpath(pftp_server_t ftp, const char *dir)
{
    if (ftp->cur_path) {
	if (dir[0] == '/') {
	    ftp->cur_path = realloc(ftp->cur_path, strlen(dir) + 1);
	    strcpy(ftp->cur_path, dir);	    
	} else {
	    ftp->cur_path = realloc(ftp->cur_path, strlen(ftp->cur_path) + 
				    strlen(dir) + 2);
		
	    if (ftp->cur_path[strlen(ftp->cur_path) - 1] != '/')
		strcat(ftp->cur_path, "/");
		
	    strcat(ftp->cur_path, dir);	    
	}

	if (!pftp_compactPath(&ftp->cur_path)) {
	    char *data = NULL;
	    free(ftp->cur_path);
	    ftp->cur_path = NULL;
	    pftp_curdir(ftp, &data, 0);
	    if (data) free(data);
	}
    } else {
	char *data = NULL;
	pftp_curdir(ftp, &data, 0);
	if (data) free(data);
    }
}

int pftp_cd(pftp_server_t ftp, const char *dir, char **newdir)
{
    char *data = NULL, *pos;

    if ((ftp->cur_path && strcmp(ftp->cur_path, dir)) == 0 ||
	strcmp(dir, ".") == 0) {
	if (newdir) {
	    size_t len = strlen(ftp->cur_path) + 1;
	    (*newdir) = malloc(len);
	    memcpy((*newdir), ftp->cur_path, len);
	}
	return 0;
    }

    if (strcmp(dir, "..") == 0 && ftp->cdup) {
	if (sendCmd(ftp, "CDUP", NULL) == -1) {
	    return -1;
	}
    } else {
	if (sendCmd(ftp, "CWD", dir) == -1) {
	    return -1;
	}
    }
    
    if (receiveCmd(ftp, &data) != 250) {
	if (data) free(data);

	if (strcmp(dir, "..") == 0 && ftp->cdup) {
	    
	    ftp->cdup = 0; // Never make the same mistake twice
	    return pftp_cd(ftp, dir, newdir);
	}
	
	return -1;
    }

    if ((pos = strstr(data, "Current directory is "))) {
	size_t len;
	pos += 21;
	len = strlen(pos) + 1;

	if (len > 0 && pos[len-1] == '/') 
	    pos[--len] = '\0';

	if (newdir) {
	    (*newdir) = malloc(len);
	    memcpy((*newdir), pos, len);
	}

	updateCurpath(ftp, pos);
    } else {
	updateCurpath(ftp, dir);
    }

    if (data) free(data);
    
    return 0;
}

void pftp_curdir(pftp_server_t ftp, char **currentDirectory, int force)
{
    char *s1, *s2;
    
    if ((*currentDirectory)) {
	free((*currentDirectory));
	(*currentDirectory) = NULL;
    }

    if (ftp->cur_path && !force) {
	(*currentDirectory) = malloc(strlen(ftp->cur_path) + 1);
	strcpy((*currentDirectory), ftp->cur_path);
	return;
    }

    if (sendCmd(ftp, "PWD", NULL) == -1) {
	return;
    }

    if (receiveCmd(ftp, currentDirectory) != 257) {
	if ((*currentDirectory)) {
	    free((*currentDirectory));
	    (*currentDirectory) = NULL;
	}

	return;
    }
    
    if ((s1 = strchr((*currentDirectory), '\"')) && 
	(s2 = strchr(s1 + 1, '\"'))) {
	s1++;
	memmove((*currentDirectory), s1, s2 - s1);
	(*currentDirectory)[s2 - s1] = '\0';
    }

    if (ftp->cur_path) free(ftp->cur_path);
    ftp->cur_path = malloc(strlen((*currentDirectory)) + 1);
    strcpy(ftp->cur_path, (*currentDirectory));
}

static int getMode(pftp_server_t ftp, int binary)
{
    if (binary) {
	if (ftp->binary && !ftp->cur_binary) {
	    if (sendCmd(ftp, "TYPE", "I") == -1) {
		return -1;
	    }
	    
	    if (receiveCmd(ftp, NULL) != 200) {
		return -1;
	    }
	    
	    ftp->cur_binary = 1;
	}
    } else {
	if (ftp->cur_binary) {
	    if (sendCmd(ftp, "TYPE", "A") == -1) {
		return -1;
	    }
	    
	    if (receiveCmd(ftp, NULL) != 200) {
		return -1;
	    }
	    
	    ftp->cur_binary = 0;
	}
    }
    
    return 0;
}

int pftp_ls(pftp_server_t ftp, pftp_directory_t *dir, int force, int *cache)
{
    void *buffer = NULL;
    size_t buffersize = 0;
    int ret;
    char *cur_dir = NULL;
    speed_data_t speed;
    if (cache) (*cache) = 0;

    /* Check cache */
    if (!force) {
	pftp_curdir(ftp, &cur_dir, 0);
	if (cur_dir) {
	    size_t c;
	    for (c = 0; c < ftp->cached; c++) {
		if (strcmp(ftp->cache_path[c], cur_dir) == 0) {
		    pftp_cpy_dir(dir, &ftp->cache_dir[c]);		    
		    free(cur_dir);
		    if (cache) (*cache) = 1;
		    return 0;
		}
	    }   
	}
    }
    
    if (getMode(ftp, 0) == -1) {
	if (cur_dir) free(cur_dir);
	return -1;
    }
    
    if (getDataConnection(ftp, NULL, NULL, 
			  "LIST", ftp->hidden_ls ? "-a" : NULL) == -1) {
	if (cur_dir) free(cur_dir);
	return -1;
    }

#ifdef _DEBUG
    statusMessage(ftp, "got data connection...");
#endif

    speed = initSpeed(ftp, 0, 0);
    startWait(ftp, DOWNLOAD, 3);
    
    for (;;) {
	ret = receiveData(ftp, &buffer, &buffersize, speed);
	if (ret == -1) {
	    if (cur_dir) free(cur_dir);
	    endWait(ftp, DOWNLOAD, 1);
	    abortData(ftp);
	    stopDataAuth(ftp);
	    return -1;
	}
	if (ret == 0) {
	    break;
	}
    }

    ((char *)buffer)[buffersize] = '\0';
    
    freeSpeed(ftp, &speed);
    endWait(ftp, DOWNLOAD, 0);

    if (receiveCmd(ftp, NULL) != 226) {
	if (cur_dir) free(cur_dir);
	free(buffer);
	return -1;
    }

    (*dir) = pftp_dir_parse((char *)buffer, ftp, ftp->system == UNIX);
    free(buffer);
    
    if (!cur_dir)
	pftp_curdir(ftp, &cur_dir, 0);

    if (cur_dir) {
	size_t c;
	for (c = 0; c < ftp->cached; c++)
	    if (strcmp(cur_dir, ftp->cache_path[c]) == 0) {
		pftp_cpy_dir(&ftp->cache_dir[c], dir);
		free(cur_dir);
		cur_dir = NULL;
		break;
	    }

	if (cur_dir) {
	    ftp->cache_path = realloc(ftp->cache_path, (ftp->cached + 1) *
				      sizeof(char *));
	    ftp->cache_path[ftp->cached] = malloc(strlen(cur_dir) + 1);
	    strcpy(ftp->cache_path[ftp->cached], cur_dir);
	    ftp->cache_dir = realloc(ftp->cache_dir, (ftp->cached + 1) * 
				     sizeof(pftp_directory_t));
	    pftp_cpy_dir(&ftp->cache_dir[ftp->cached], dir);
	    ftp->cached++;
	    free(cur_dir);
	}
    }
    
    return 0;
}

int pftp_rmdir(pftp_server_t ftp, const char *dir)
{
    if (sendCmd(ftp, "RMD", dir) == -1) {
	return -1;
    }
    
    if (receiveCmd(ftp, NULL) != 250) {
	return -1;
    }
    
    return 0;
}

int pftp_mkdir(pftp_server_t ftp, const char *dir)
{
    if (sendCmd(ftp, "MKD", dir) == -1) {
	return -1;
    }
    
    if (receiveCmd(ftp, NULL) != 257) {
	return -1;
    }
    
    return 0;
}

int pftp_rm(pftp_server_t ftp, const char *filename)
{
    if (sendCmd(ftp, "DELE", filename) == -1) {
	return -1;
    }
    
    if (receiveCmd(ftp, NULL) != 250) {
	return -1;
    }
    
    return 0; 
}

int pftp_rename(pftp_server_t ftp, const char *filename, const char *new)
{
    if (sendCmd(ftp, "RNFR", filename) == -1) {
	return -1;
    }
    
    if (receiveCmd(ftp, NULL) != 350) {
	return -1;
    }
    
    if (sendCmd(ftp, "RNTO", new) == -1) {
	return -1;
    }
    
    if (receiveCmd(ftp, NULL) != 250) {
	return -1;
    }
    
    return 0; 
}

int pftp_get(pftp_server_t ftp, const char *filename, char ascii, FILE *fh, 
	     uint64_t resume_from, uint64_t total_size, uint32_t *crc)
{
    void *buffer = NULL;
    size_t buffersize = 0;
    int ret;
    speed_data_t speed;
    
    if (getMode(ftp, ascii == 'A' ? 0 : 1) == -1) {
	return -1;
    }
    
    if (ftp->resume && resume_from > 0) {
	char *tmpstr = malloc(50);
	snprintf(tmpstr, 50, "%" SCNu64, resume_from);
	
	if (getDataConnection(ftp, "REST", tmpstr, "RETR", filename) == -1) {
	    free(tmpstr);
	    return -1;
	}
	
	free(tmpstr);
    } else {
	if (getDataConnection(ftp, NULL, NULL, "RETR", filename) == -1) {
	    return -1;
	}
    }

    send_comm(ftp, PFTP_FILETRANSFER, (pftp_param_t) &resume_from, 
	      (pftp_param_t) &total_size);
    speed = initSpeed(ftp, total_size, resume_from);
    startWait(ftp, DOWNLOAD, 1);
    if (crc) {
	pftp_initCRC32(&ftp->cur_crc);
#ifndef NO_SSL
	if (ftp->ssl_private)
	    ftp->dataRecv = intCRCSSLRecv;
	else
#endif
	    ftp->dataRecv = intCRCRecv;       
    }
    
    for (;;) {
	ret = receiveData(ftp, &buffer, &buffersize, speed);

	if (ret == -1) {
	    endWait(ftp, DOWNLOAD, 1);
	    abortData(ftp);	    
	    stopDataAuth(ftp);
	    freeSpeed(ftp, &speed);
	    return -1;
	}

	if (buffersize > 0) {
	    if (fwrite(buffer, 1, buffersize, fh) != buffersize) {
		statusMessage(ftp, "ERR: Unable to write to file: %s", 
			      strerror(errno));
		endWait(ftp, DOWNLOAD, 1);
		abortData(ftp);	    
		stopDataAuth(ftp);
		freeSpeed(ftp, &speed);
		return -1;
	    }
	}
	
	buffersize = 0;
	
	if (ret == 0) 
	    break;
    }
    
    freeSpeed(ftp, &speed);
    endWait(ftp, DOWNLOAD, 0);
    
    if (receiveCmd(ftp, NULL) != 226) {
	if (ftp->datastream != -1) {
	    stopDataAuth(ftp);
	    shutdown(ftp->datastream, 2);
	    close(ftp->datastream);
	    ftp->datastream = -1;
	}     
	return -1;
    }
   
    pftp_finishCRC32(&ftp->cur_crc);

    if (crc)
	(*crc) = ftp->cur_crc;
    
    return 0;
}

int pftp_put(pftp_server_t ftp, const char *filename, char ascii, FILE *fh, 
	     uint64_t resume_from, uint64_t total_size)
{
    void *buffer = NULL;
    size_t buffersize = 0, done, lastbuffer = 0;
    int ret;
    speed_data_t speed;
    
    if (getMode(ftp, ascii == 'A' ? 0 : 1) == -1) {
	return -1;
    }

    if (ftp->resume && resume_from > 0) {
	char *tmpstr = malloc(50);
	snprintf(tmpstr, 50, "%" SCNu64, resume_from);
	
	if (getDataConnection(ftp, "REST", tmpstr, "STOR", filename) == -1) {
	    free(tmpstr);
	    return -1;
	}
	
	free(tmpstr);
    } else {
	if (getDataConnection(ftp, NULL, NULL, "STOR", filename) == -1) {
	    return -1;
	}  
    }
    
    send_comm(ftp, PFTP_FILETRANSFER, (pftp_param_t)&resume_from, 
	      (pftp_param_t)&total_size);
    speed = initSpeed(ftp, total_size, resume_from);
    lastbuffer = speed->buffersize;
    buffer = malloc(speed->buffersize + 1);

    startWait(ftp, UPLOAD, 2);
    
    for (;;) {
	if (lastbuffer != speed->buffersize) {
	    buffer = realloc(buffer, speed->buffersize + 1);
	}
	ret = fread(buffer + buffersize, 1, speed->buffersize-buffersize, fh);
	
	if (ferror(fh)) {
	    statusMessage(ftp, "ERR: Unable to read from file: %s", 
			  strerror(errno));
	    endWait(ftp, UPLOAD, 1);
	    abortData(ftp);
	    stopDataAuth(ftp);
	    free(buffer);
	    freeSpeed(ftp, &speed);
	    return -1;
	}
	
	buffersize += ret;
	done = 0;
	
	while ((buffersize - done) >= speed->buffersize 
	       || (feof(fh) && (buffersize - done) > 0)) {
	    size_t newdone;
	    ret = sendData(ftp, buffer + done, buffersize - done, &newdone, 
			   speed);
	    
	    if (ret == -1) {
		endWait(ftp, UPLOAD, 1);
		abortData(ftp);
		stopDataAuth(ftp);
		free(buffer);
		freeSpeed(ftp, &speed);
		return -1;
	    }

	    done += newdone;
	}

	if (done < buffersize) {
	    buffersize -= done;
	    memmove(buffer, buffer + done, buffersize);
	} else {
	    buffersize = 0;
	}
	
	if (feof(fh))
	    break;
    }
    
    free(buffer);
    freeSpeed(ftp, &speed);
    endWait(ftp, UPLOAD, 0);
    
    stopDataAuth(ftp);
    shutdown(ftp->datastream, 2);
    close(ftp->datastream);
    ftp->datastream = -1;
    
    if (receiveCmd(ftp, NULL) != 226) {
	return -1;
    }
    
    return 0;
}

static int parseCmd(pftp_server_t ftp, unsigned short *code, char **data, 
		    int abort)
{
    char *s1, *s2, *end = NULL;
    unsigned long ret;

    (*code) = 0;
    if (data && (*data)) {
	free((*data));
	(*data) = NULL;
    }   
    
    if (!(s1 = strchr(ftp->inbuffer, '\n')))
	return 0;
    
    s2 = s1;
    
    s1++;
    if (s1[0] == '\r') s1++;
    if (s2 > ftp->inbuffer && s2[-1] == '\r') s2--;
    
    errno = 0;
    ret = strtoul(ftp->inbuffer, &end, 10);
    
    if (errno == ERANGE || ret < 100 || ret >= 600 || end == NULL) {    
	char *tmpstr = malloc((s2 - ftp->inbuffer) + 5);
	strncpy(tmpstr, ftp->inbuffer, (s2 - ftp->inbuffer));
	tmpstr[(s2 - ftp->inbuffer)] = '\0';
	statusMessage(ftp, "Invalid response from server: %s", tmpstr);
	free(tmpstr);
	return s1 - ftp->inbuffer;
    }
    
    if (end[0] == ' ' || (end[0] == '-' && ret == 226 && abort)) {
	(*code) = ret;
	if (data) {
	    (*data) = malloc((s2 - (ftp->inbuffer + 4)) + 1);
	    strncpy((*data), ftp->inbuffer + 4, (s2 - (ftp->inbuffer + 4)));
	    (*data)[(s2 - (ftp->inbuffer + 4))] = '\0';
	    statusMessage(ftp, "<- %u %s", (*code), (*data));
	} else {
	    char *tmpstr = malloc((s2 - ftp->inbuffer) + 5);
	    strncpy(tmpstr, ftp->inbuffer, (s2 - ftp->inbuffer));
	    tmpstr[(s2 - ftp->inbuffer)] = '\0';
	    statusMessage(ftp, "<- %s", tmpstr);
	    free(tmpstr);
	}
	return s1 - ftp->inbuffer;
    } else if (end[0] == '-') {
	char *loc, *tmpstr = malloc(7);
	
	snprintf(tmpstr, 7, "%c%lu ", s1[-1], ret);
	loc = strstr(ftp->inbuffer, tmpstr);
	free(tmpstr);

	if (loc) {
	    char *start, *tmpstr2 = malloc(5);
	    size_t pos;
	    
	    loc++;
	    
	    snprintf(tmpstr2, 5, "%lu-", ret);
	    (*code) = ret;
	    tmpstr = malloc((s2 - (ftp->inbuffer + 4)) + 7);
	    snprintf(tmpstr, 7, " %lu ", ret);
	    strncat(tmpstr, ftp->inbuffer + 4, (s2 - (ftp->inbuffer + 4)));
	    tmpstr[5 + (s2 - (ftp->inbuffer + 4))] = '\0';
	    
	    for (;;) {
		start = s1;
		if (!strncmp(start, tmpstr2, 4))
		    start += 4;
		s2 = s1 = strchr(start, '\n');
		
		if (!s1) {
		    (*code) = 0;
		    free(tmpstr2);
		    free(tmpstr);
		    return 0;
		}  
		
		s1++;
		if (s1[0] == '\r') s1++;
		if (s2[-1] == '\r') s2--;	
		
		pos = strlen(tmpstr);
		tmpstr = realloc(tmpstr, pos + (s2 - start) + 2);
		strcat(tmpstr, "\n");
		pos++;
		strncat(tmpstr, start, s2 - start);
		tmpstr[pos + (s2 - start)] = '\0';

		if (s2 > loc)
		    break;       
	    }
	    
	    free(tmpstr2);
	    statusMessage(ftp, "<-%s", tmpstr);
	    
	    if (data) {
		(*data) = malloc(strlen(tmpstr) + 1);
		strcpy((*data), tmpstr);
	    }
	    
	    free(tmpstr);
	    
	    return s1 - ftp->inbuffer;
	}         
    } else {
	char *tmpstr = malloc((s2 - ftp->inbuffer) + 5);
	strncpy(tmpstr, ftp->inbuffer, (s2 - ftp->inbuffer));
	tmpstr[(s2 - ftp->inbuffer)] = '\0';
	statusMessage(ftp, "Invalid response from server: %s", tmpstr);
	free(tmpstr);
	return s1 - ftp->inbuffer;
    }
    
    return 0;
}

short _receiveCmd(pftp_server_t ftp, char **data, int abort)
{
    fd_set read_set;
    struct timeval timeout;
    int ret, need_more = 0;

    FD_ZERO(&read_set);
    startWait(ftp, DOWNLOAD, 0);
    
    for (;;) {
	if (ftp->inbuffer_len == 0 || need_more) {
	    need_more = 0;
	    timeout.tv_sec = 0;
	    timeout.tv_usec = DEFAULT_TIMEOUT;
	    FD_SET(ftp->cmdstream, &read_set);

	    ret = TEMP_FAILURE_RETRY(select(FD_SETSIZE, &read_set, NULL, NULL, 
					&timeout));
    
	    if (ret == -1) {
		statusMessage(ftp, "ERR: Select returns error: %s", 
			      strerror(errno));
		endWait(ftp, DOWNLOAD, 1);
		return -1;
	    } else if (ret == 0) {
		if (doWait(ftp, NULL, 0, "DOWNLOAD-CMD1") == -1) {
		    endWait(ftp, DOWNLOAD, 1);
		    return -1;
		}
	    }
	} else {
	    ret = 1;
	}

	if (ret > 0) {
	    unsigned short code;
	    
	    for (;;) {
		ftp->inbuffer = realloc(ftp->inbuffer, 
					ftp->inbuffer_len + 1024 + 1);
		ret = ftp->cmdRecv(ftp, ftp->cmdstream, 
				   ftp->inbuffer + ftp->inbuffer_len, 1024);

		if (ret == -1) {
		    if (errno != EWOULDBLOCK) {
			statusMessage(ftp, "ERR: Recv returns error: %s", 
				      strerror(errno));
			endWait(ftp, DOWNLOAD, 1);
			return -1;
		    }
		    
		    ret = 0;
		} else if (ret == 0) {
		    statusMessage(ftp, "ERR: Connection closed (1)");
		    sleep(5);
		    endWait(ftp, DOWNLOAD, 1);
		    send_comm(ftp, PFTP_DISCONNECTED, NULL, NULL);
		    return -1;
		}
		
		if (doWait(ftp, NULL, 1, "DOWNLOAD-CMD2") == -1) {
		    endWait(ftp, DOWNLOAD, 1);
		    return -1;
		}
		
		ftp->inbuffer_len += ret;
		ftp->inbuffer[ftp->inbuffer_len] = '\0';
		
		ret = parseCmd(ftp, &code, data, abort);
		
		if (ret > 0) {
		    ftp->inbuffer_len -= ret;
		    memmove(ftp->inbuffer, ftp->inbuffer + ret, 
			    ftp->inbuffer_len + 1);
		} else {
		    need_more = 1;
		    break;
		}
		
		if (code == 332 || code == 532) {
		    if (sendAccount(ftp) == -1) {
			endWait(ftp, DOWNLOAD, 1);
			return -1;	  
		    }		    
		    break;
		} else {
		    endWait(ftp, DOWNLOAD, 0);
		    if (code == 421) {			
			send_comm(ftp, PFTP_DISCONNECTED, NULL, NULL);
		    }
		    return code;
		}
	    }
	}	
    }
}

int receiveCmd2(pftp_server_t ftp1, pftp_server_t ftp2, short *ret1, 
		short *ret2, char **data1, char **data2)
{
    fd_set read_set;
    struct timeval timeout;
    int ret;
    
    FD_ZERO(&read_set);
    startWait(ftp1, DOWNLOAD, 0);
    startWait(ftp2, DOWNLOAD, 0);

    (*ret1) = 0;
    (*ret2) = 0;
    
    while ((*ret1) == 0 || (*ret2) == 0) {
	timeout.tv_sec = 0;
	timeout.tv_usec = DEFAULT_TIMEOUT;

	FD_ZERO(&read_set);
	if ((*ret1) == 0) FD_SET(ftp1->cmdstream, &read_set);
	if ((*ret2) == 0) FD_SET(ftp2->cmdstream, &read_set);
	
	ret = TEMP_FAILURE_RETRY(select(FD_SETSIZE, &read_set, NULL, NULL, 
					&timeout));
	
	if (ret == -1) {
	    statusMessage(ftp1, "ERR: Select returns error: %s", 
			  strerror(errno));
	    statusMessage(ftp2, "ERR: Select returns error: %s", 
			  strerror(errno));
	    endWait(ftp1, DOWNLOAD, 1);
	    endWait(ftp2, DOWNLOAD, 1);
	    return -1;
	} else if (ret == 0) {
	    if (doWait(ftp1, NULL, 1, "DOWNLOAD-CMD3") == -1 || 
		doWait(ftp2, NULL, 1, "DOWNLOAD-CMD4") == -1) {
		endWait(ftp1, DOWNLOAD, 1);
		endWait(ftp2, DOWNLOAD, 1);
		return -1;
	    }
	} else {
	    if (FD_ISSET(ftp1->cmdstream, &read_set)) {
		for (;;) {
		    ftp1->inbuffer = realloc(ftp1->inbuffer, 
					     ftp1->inbuffer_len + 1024 + 1);
		    ret = ftp1->cmdRecv(ftp1, ftp1->cmdstream, 
					ftp1->inbuffer + ftp1->inbuffer_len, 
					1024);

		    if (ret == -1) {
			if (errno != EWOULDBLOCK) {
			    statusMessage(ftp1,"ERR: Recv returns error: %s", 
					  strerror(errno));
			    endWait(ftp1, DOWNLOAD, 1);
			    endWait(ftp2, DOWNLOAD, 1);
			    return -1;
			}
		    
			ret = 0;
		    } else if (ret == 0) {
			statusMessage(ftp1, "ERR: Connection closed (2)");
			endWait(ftp1, DOWNLOAD, 1);
			endWait(ftp2, DOWNLOAD, 1);
			send_comm(ftp1, PFTP_DISCONNECTED, NULL, NULL);
			return -1;
		    }		    
		    if (doWait(ftp1, NULL, 1, "DOWNLOAD-CMD5") == -1) {
			endWait(ftp1, DOWNLOAD, 1);
			endWait(ftp2, DOWNLOAD, 1);
			return -1;
		    }
		
		    ftp1->inbuffer_len += ret;
		    ftp1->inbuffer[ftp1->inbuffer_len] = '\0';

		    ret = parseCmd(ftp1, ret1, data1, 0);
		
		    if (ret > 0) {
			ftp1->inbuffer_len -= ret;
			memmove(ftp1->inbuffer, ftp1->inbuffer + ret, 
				ftp1->inbuffer_len + 1);
		    } else {
			break;
		    }
		
		    if ((*ret1) == 332 || (*ret1) == 532) {
			(*ret1) = 0;
			if (sendAccount(ftp1) == -1) {
			    endWait(ftp1, DOWNLOAD, 1);
			    endWait(ftp2, DOWNLOAD, 1);
			    return -1;
			}
		    }
		    
		    break;
		}
	    } else if (FD_ISSET(ftp2->cmdstream, &read_set)) {
		for (;;) {
		    ftp2->inbuffer = realloc(ftp2->inbuffer, 
					     ftp2->inbuffer_len + 1024 + 1);
		    ret = ftp2->cmdRecv(ftp2, ftp2->cmdstream, 
					ftp2->inbuffer + ftp2->inbuffer_len, 
					1024);

		    if (ret == -1) {
			if (errno != EWOULDBLOCK) {
			    statusMessage(ftp2,"ERR: Recv returns error: %s", 
					  strerror(errno));
			    endWait(ftp1, DOWNLOAD, 1);
			    endWait(ftp2, DOWNLOAD, 1);
			    return -1;
			}
			
			ret = 0;
		    } else if (ret == 0) {
			statusMessage(ftp2, "ERR: Connection closed (3)");
			endWait(ftp1, DOWNLOAD, 1);
			endWait(ftp2, DOWNLOAD, 1);
			send_comm(ftp2, PFTP_DISCONNECTED, NULL, NULL);
			return -1;
		    }		    
		    if (doWait(ftp2, NULL, 1, "DOWNLOAD-CMD6") == -1) {
			endWait(ftp1, DOWNLOAD, 1);
			endWait(ftp2, DOWNLOAD, 1);
			return -1;
		    }
		    
		    ftp2->inbuffer_len += ret;
		    ftp2->inbuffer[ftp2->inbuffer_len] = '\0';

		    ret = parseCmd(ftp2, ret2, data2, 0);
		
		    if (ret > 0) {
			ftp2->inbuffer_len -= ret;
			memmove(ftp2->inbuffer, ftp2->inbuffer + ret, 
				ftp2->inbuffer_len + 1);
		    } else {			
			break;
		    }
		
		    if ((*ret2) == 332 || (*ret2) == 532) {
			(*ret2) = 0;
			if (sendAccount(ftp2) == -1) {
			    endWait(ftp1, DOWNLOAD, 1);
			    endWait(ftp2, DOWNLOAD, 1);
			    return -1;
			}
		    }
		    
		    break;
		}
	    }
	}	
    }

    endWait(ftp1, DOWNLOAD, 0);
    endWait(ftp2, DOWNLOAD, 0);

    return 0;
}

int pftp_rawcmd(pftp_server_t ftp, const char *cmd, char **data)
{
    if (sendCmd(ftp, cmd, NULL) == -1) {
	return -1;
    }
    
    return receiveCmd(ftp, data); 
}

int sendCmd(pftp_server_t ftp, const char *cmd, const char *buffer)
{
    char *data;
    int ret;
    size_t pos = 0, datalen = 0;
    
    if (buffer) {
	data = malloc(strlen(cmd) + strlen(buffer) + 5);
	strcpy(data, cmd);
	strcat(data, " ");
	strcat(data, buffer);
    } else {
	data = malloc(strlen(cmd) + 5);
	strcpy(data, cmd);
    }
    
    if (strcmp(cmd, "PASS")){
	statusMessage(ftp, "-> %s", data);
    } else {
	statusMessage(ftp, "-> PASS xxx");
    }
    
    strcat(data, "\r\n");
    datalen = strlen(data);
    
    for (;;) {
	ret = ftp->cmdSend(ftp, ftp->cmdstream, data + pos, datalen - pos);

	if (ret == datalen - pos)
	    break;
	
	if (ret < 1) {
	    if (ret == -1 && errno == EWOULDBLOCK)
		continue;
	    shutdown(ftp->datastream, 2);
	    close(ftp->datastream);
	    ftp->datastream = -1;
#ifndef NO_SSL
	    if (ftp->cmdssl) SSL_shutdown(ftp->cmdssl);	    
#endif /* NO_SSL */
	    shutdown(ftp->cmdstream, 2);
	    close(ftp->cmdstream);
	    ftp->cmdstream = -1;
	    free(data);
	    send_comm(ftp, PFTP_DISCONNECTED, NULL, NULL);
	    return -1;
	} 
	
	pos += ret;
    }
    
    free(data);
    
    return 0;
}

int sendData(pftp_server_t ftp, const void *buffer, size_t buffersize, 
	     size_t *done, speed_data_t speed)
{
    int ret;
    size_t start = 0;
    fd_set write_set;
    struct timeval timeout;
    
    FD_ZERO(&write_set);
    
    if (ftp->datastream == -1) {
	statusMessage(ftp, "ERR: No data connection");
	return -1;  
    }

    for (;;) {
	ret = ftp->dataSend(ftp, ftp->datastream, buffer + start, 
			    min(speed->buffersize, buffersize - start)); 
	
	if (ret == -1) {
	    if (errno != EWOULDBLOCK) {
		statusMessage(ftp, "ERR: Send returns error: %s", 
			      strerror(errno));
		return -1;
	    }
	    
	    for (;;) {
		timeout.tv_sec = 0;
		timeout.tv_usec = DEFAULT_TIMEOUT;
		FD_SET(ftp->datastream, &write_set);
		
		ret = TEMP_FAILURE_RETRY(select(FD_SETSIZE, NULL, &write_set, 
						NULL, &timeout));
		
		if (ret == -1) {
		    statusMessage(ftp, "ERR: Select returns error: %s", 
				  strerror(errno));
		    return -1;
		} else if (ret > 0) {
		    break;
		}
		
		if (doWait(ftp, speed, 0, "UPLOAD") == -1) {
		    return -1;
		}
	    }
	} else if (ret == 0) {
	    statusMessage(ftp, "ERR: Connection closed (4)");
	    return -1;
	} else {    
	    start += ret;
	    updateSpeed(speed, ret);
	    
	    if (doWait(ftp, speed, 1, "UPLOAD") == -1) {
		return -1;
	    }
	    
	    if (start + speed->buffersize > buffersize) {
		(*done) = start;
		return 0;
	    }
	}
    }
}

static unsigned long parseUL(const char *str, int *valid)
{
    char *end;
    unsigned long ret;
    (*valid) = 0;
    if (!str) return 0;
    ret = strtoul(str, &end, 10);
    if (ret == ULONG_MAX || !end || end[0] != '\0')
	return 0;
    (*valid) = 1;
    return ret;
}

static size_t csplit(const char *str, char c, char ***part)
{
    size_t parts = 0;
    const char *last = str;
    char *s;
    
    while ((s = strchr(last, c))) {
	(*part) = realloc((*part), sizeof(char *) * (parts + 1));
	(*part)[parts] = malloc((s - last) + 1);
	strncpy((*part)[parts], last, (s - last));
	(*part)[parts][s - last] = '\0';
	last = s + 1;
	parts++;
    }
    
    if (last[0] != '\0') {
	(*part) = realloc((*part), sizeof(char *) * (parts + 1));
	(*part)[parts] = malloc(strlen(last) + 1);
	strcpy((*part)[parts], last);
	parts++;
    }
    
    return parts;
}

int parsePasvResponse(pftp_server_t ftp, const char *data, 
		      struct sockaddr_in *addr)
{
    char *s1, *s2, *tmpstr, **part = NULL;
    size_t parts;
    int valid;
    uint32_t ipaddr = 0;
    uint16_t port = 0;
    unsigned long tmp;
    
    if (!(s1 = strrchr(data, '(')) || !(s2 = strrchr(data, ')')) || s1 > s2) {
	statusMessage(ftp, "ERR: Unable to parse PASV response");
	return -1;
    }
    
    s1++;
    
    tmpstr = malloc((s2 - s1) + 1);
    strncpy(tmpstr, s1, (s2 - s1));
    tmpstr[s2 - s1] = '\0';
    
    /* Parse the PASV address */
    parts = csplit(tmpstr, ',', &part);
    free(tmpstr);
    
    if (parts == 6) {
	size_t p;    
	
	for (p = 0; p < 4; p++) {
	    tmp = parseUL(part[p], &valid);
	    if (!valid || tmp >= 256) { 
		ipaddr = 0; 
		break;
	    }
	    ipaddr *= 256;
	    ipaddr += tmp;
	}
	
	for (p = 4; p < 6; p++) {
	    tmp = parseUL(part[p], &valid);
	    if (!valid || tmp >= 256) { 
		port = 0; 
		break;
	    }
	    port *= 256;
	    port += tmp;
	}
    }
    
    while (parts-- > 0) free(part[parts]);
    if (part) free(part);
    
    if (port == 0 && ipaddr == 0) {
	statusMessage(ftp, "ERR: Unable to parse PASV response");
	return -1;
    }

    addr->sin_family = AF_INET;
    addr->sin_addr.s_addr = htonl(ipaddr);
    addr->sin_port = htons(port);
    
    return 0;
}

int getPassiveConnection(pftp_server_t ftp)
{
    char *data = NULL;
    struct sockaddr_in addr;
    
    if (sendCmd(ftp, "PASV", NULL) == -1) {
	return -1;
    }
    
    if (receiveCmd(ftp, &data) != 227) {
	if (data) free(data);
	return -1;
    }
    
    if (parsePasvResponse(ftp, data, &addr) != 0) {
	free(data);
	return -1;
    }
    
    free(data);
    
    statusMessage(ftp, "Connecting to %s:%u", inet_ntoa(addr.sin_addr), ntohs(addr.sin_port));
    
    if (connectSocket(ftp, &ftp->datastream, &addr) == 0)
	return 0;
    
    return -1;
}

int generatePortData(pftp_server_t ftp, const struct sockaddr_in *addr, 
		     char **tmpstr, int dont_use_myaddr)
{
    uint32_t ipaddr = ntohl(addr->sin_addr.s_addr);
    uint16_t port = ntohs(addr->sin_port);
    
    if (!dont_use_myaddr
	&& getStrValue(ftp, myaddr) && getStrValue(ftp, myaddr)[0]) {
	struct sockaddr_in tmpaddr;
	
	if (parseHostname(getStrValue(ftp, myaddr), port, &tmpaddr) == -1) {
	    statusMessage(ftp, "ERR: Unable to use %s as my address.", 
			  getStrValue(ftp, myaddr));
	    return -1; 
	}
	
	ipaddr = ntohl(tmpaddr.sin_addr.s_addr);
    }
    
    if (ipaddr == INADDR_ANY) {
	struct sockaddr_in tmpaddr;
	socklen_t len = sizeof(struct sockaddr_in);
	
	if (getsockname(ftp->cmdstream, 
			(struct sockaddr *)&tmpaddr, &len) == -1 
	    || len != sizeof(struct sockaddr_in)) {
	    statusMessage(ftp, "ERR: Unable to get my own address: %s", 
			  strerror(errno));
	    return -1;
	}
	
	ipaddr = ntohl(tmpaddr.sin_addr.s_addr);
    }
    
    (*tmpstr) = realloc((*tmpstr), 26);
    
    snprintf((*tmpstr), 26, "%u,%u,%u,%u,%u,%u", 
	     (uint16_t)((ipaddr & 0xFF000000) >> 24),
	     (uint16_t)((ipaddr & 0x00FF0000) >> 16),
	     (uint16_t)((ipaddr & 0x0000FF00) >>  8),
	     (uint16_t)((ipaddr & 0x000000FF) >>  0),
	     (uint16_t)((port & 0xFF00) >>  8),
	     (uint16_t)((port & 0x00FF) >>  0));
    
    return 0;
}

int getDirectConnection(pftp_server_t ftp)
{
    char *tmpstr = NULL;
    struct sockaddr_in addr;
    uint16_t port;

    send_comm(ftp, PFTP_NEEDPORT, (pftp_param_t)&port, 
	      (pftp_param_t)ftp->settings);
    
    addr.sin_family = AF_INET;
    addr.sin_port = htons(port);
    
    if (getStrValue(ftp, bind_to) && getStrValue(ftp, bind_to)[0]) {
	if (parseHostname(getStrValue(ftp, bind_to), ntohs(addr.sin_port), 
			  &addr) == -1) {
	    statusMessage(ftp, "ERR: Unable to bind socket to: %s", 
			  getStrValue(ftp, bind_to));
	    return -1;
	}
    } else {
	addr.sin_addr.s_addr = htonl(INADDR_ANY);
    }
    
    if ((ftp->port_socket = createNonblockingSocket(ftp)) == -1) {
	return -1;
    }
    
    if (bind(ftp->port_socket, (struct sockaddr *)&addr, 
	     sizeof(struct sockaddr_in)) == -1 ||
	listen(ftp->port_socket, 1) == -1) {
	if (getStrValue(ftp, bind_to) && getStrValue(ftp, bind_to)[0] &&
	    port) {
	    statusMessage(ftp, "ERR: Unable to bind socket to: %s:%u (%s)", 
			  getStrValue(ftp, bind_to), ntohs(addr.sin_port), 
			  strerror(errno));
	} else if (port != 0) {
	    statusMessage(ftp, "ERR: Unable to bind socket to: <any>:%u (%s)", 
			  ntohs(addr.sin_port), strerror(errno));
	} else if (getStrValue(ftp, bind_to) && getStrValue(ftp, bind_to)[0]) {
	    statusMessage(ftp, "ERR: Unable to bind socket to: %s:<any> (%s)",
			  getStrValue(ftp, bind_to), strerror(errno));
	} else {
	    statusMessage(ftp, 
			  "ERR: Unable to bind socket to: <any>:<any> (%s)",
			  strerror(errno));
	}
	
	close(ftp->port_socket);
	ftp->port_socket = -1;
	return -1;
    }
    
    /* Get the port we bound to, if needed */
    if (!port) {
	struct sockaddr_in tmp_addr;
	socklen_t tmp_addrlen = sizeof(struct sockaddr_in);
	if (getsockname(ftp->port_socket, (struct sockaddr *)&tmp_addr,
			&tmp_addrlen) == -1) {
	    statusMessage(ftp, "ERR: Unable to get port number. (%s)",
			  strerror(errno));
	    close(ftp->port_socket);
	    ftp->port_socket = -1;
	    return -1;
	} else {
	    addr.sin_port = tmp_addr.sin_port;
	}
    }
    
    if (generatePortData(ftp, &addr, &tmpstr, 0) == -1) {
	free(tmpstr);
	shutdown(ftp->port_socket, 2);
	close(ftp->port_socket);
	ftp->port_socket = -1;
	return -1;
    }
    
    if (sendCmd(ftp, "PORT", tmpstr) == -1) {
	free(tmpstr);
	shutdown(ftp->port_socket, 2);
	close(ftp->port_socket);
	ftp->port_socket = -1;
	return -1;
    }
    
    free(tmpstr);
    
    if (receiveCmd(ftp, NULL) != 200) {
	shutdown(ftp->port_socket, 2);
	close(ftp->port_socket);
	ftp->port_socket = -1;
	return -1;
    }
    
    return 0;
}

int acceptDirectConnection(pftp_server_t ftp)
{
    fd_set read_set;
    struct timeval timeout;
    int ret;
    
    if (ftp->port_socket == -1)
	return -1;
    
    FD_ZERO(&read_set);
    startWait(ftp, PORT_CONN, 3);
    
    for (;;) {
	FD_SET(ftp->port_socket, &read_set);
	timeout.tv_sec = 0;
	timeout.tv_usec = DEFAULT_TIMEOUT;
	
	ret = TEMP_FAILURE_RETRY(select(FD_SETSIZE, &read_set, NULL, NULL, 
					&timeout));
	
	if (ret == -1) {
	    statusMessage(ftp, "ERR: Select returned error: %s", 
			  strerror(errno));
	    shutdown(ftp->port_socket, 2);
	    close(ftp->port_socket);
	    ftp->port_socket = -1;
	    endWait(ftp, PORT_CONN, 1);
	    return -1;
	} else if (ret == 0) {
	    if (doWait(ftp, NULL, 0, "PORT_CONNECT") == -1) {
		shutdown(ftp->port_socket, 2);
		close(ftp->port_socket);
		ftp->port_socket = -1;
		endWait(ftp, PORT_CONN, 1);
		return -1;
	    }
	} else {
	    struct sockaddr_in addr;
	    socklen_t len = sizeof(struct sockaddr_in);
	    ftp->datastream = accept(ftp->port_socket, 
				     (struct sockaddr *)&addr, 
				     &len);
	    
	    if (ftp->datastream == -1) {
		statusMessage(ftp, "ERR: accept returned error: %s", 
			      strerror(errno));
		shutdown(ftp->port_socket, 2);
		close(ftp->port_socket);
		ftp->port_socket = -1;
		endWait(ftp, PORT_CONN, 1);
		return -1;
	    }      
	    break;
	}
    }
    
    shutdown(ftp->port_socket, 2);
    close(ftp->port_socket);
    ftp->port_socket = -1;
    endWait(ftp, PORT_CONN, 0);
    
    return 0;
}

int getDataConnection(pftp_server_t ftp, const char *precmd, 
		      const char *predata, const char *cmd, const char *data)
{
    int connected = 0, list = (strcmp(cmd, "LIST") == 0);
    
    ftp->dataRecv = intRecv;
    ftp->dataSend = intSend;

    if (!ftp->pasv) {
	if (initDataAuth(ftp, list) != 0)
	    return -1;

	if (getDirectConnection(ftp) == 0)
	    connected = 1;
    }
    
    if (!connected) {
	if (initDataAuth(ftp, list) != 0)
	    return -1;

	if (getPassiveConnection(ftp) == 0) {
	    ftp->pasv = 1;
	    connected = 1;
	}
    }
    
    if (!connected) {
	restoreAuthData(ftp, list);
	return -1;   
    }
    
    if (precmd) {
	if (sendCmd(ftp, precmd, predata) == -1) {      
	    if (ftp->datastream != -1) {
		shutdown(ftp->datastream, 2);
		close(ftp->datastream);
		ftp->datastream = -1;
	    }     
	    if (ftp->port_socket != -1) {
		shutdown(ftp->port_socket, 2);
		close(ftp->port_socket);
		ftp->port_socket = -1;
	    }
	    restoreAuthData(ftp, list);
	    return -1;
	}
	
	if (receiveCmd(ftp, NULL) != 350) {
	    if (ftp->datastream != -1) {
		shutdown(ftp->datastream, 2);
		close(ftp->datastream);
		ftp->datastream = -1;
	    }     
	    if (ftp->port_socket != -1) {
		shutdown(ftp->port_socket, 2);
		close(ftp->port_socket);
		ftp->port_socket = -1;
	    }
	    restoreAuthData(ftp, list);
	    return -1;
	}
    }

    if (sendCmd(ftp, cmd, data) == -1) {
	if (ftp->datastream != -1) {
	    shutdown(ftp->datastream, 2);
	    close(ftp->datastream);
	    ftp->datastream = -1;
	}     
	if (ftp->port_socket != -1) {
	    shutdown(ftp->port_socket, 2);
	    close(ftp->port_socket);
	    ftp->port_socket = -1;
	}
	restoreAuthData(ftp, list);
	return -1;
    }

    if (continueDataConnection(ftp, precmd, predata, cmd, data) == -1) {
	return -1;
    }
    
    return 0;
}

int continueDataConnection(pftp_server_t ftp, const char *precmd, 
			   const char *predata, const char *cmd, 
			   const char *data)
{
    int retdata, list = (strcmp(cmd, "LIST") == 0);
    
    retdata = receiveCmd(ftp, NULL);

    if (retdata != 150 && retdata != 125) {
	if (retdata == 425) {
	    if (!ftp->pasv) {
		ftp->pasv = 1;
		
		if (getDataConnection(ftp, precmd, predata, cmd, data) == 0) {
		    return 0;
		} else {
		    ftp->pasv = 0;
		}
	    }
	}     
	
	if (ftp->datastream != -1) {
	    shutdown(ftp->datastream, 2);
	    close(ftp->datastream);
	    ftp->datastream = -1;
	}     
	if (ftp->port_socket != -1) {
	    shutdown(ftp->port_socket, 2);
	    close(ftp->port_socket);
	    ftp->port_socket = -1;
	}
	restoreAuthData(ftp, list);
	return -1;
    }

    if (!ftp->pasv) {
	if (acceptDirectConnection(ftp) == -1) {
	    receiveCmd(ftp, NULL);

	    ftp->pasv = 1;
	    
	    if (getDataConnection(ftp, precmd, predata, cmd, data) == 0) {
		return 0;
	    } else {
		ftp->pasv = 0;
		
		if (ftp->datastream != -1) {
		    shutdown(ftp->datastream, 2);
		    close(ftp->datastream);
		    ftp->datastream = -1;
		}     
		if (ftp->port_socket != -1) {
		    shutdown(ftp->port_socket, 2);
		    close(ftp->port_socket);
		    ftp->port_socket = -1;
		}
		restoreAuthData(ftp, list);
		return -1;	
	    }
	}
    }

    if (startDataAuth(ftp) == -1) {
	abortData(ftp);
	restoreAuthData(ftp, list);
	return -1;
    }

#ifdef _DEBUG
    statusMessage(ftp, "dataconnection continued");
#endif 

    return 0;
}

int receiveData(pftp_server_t ftp, void **buffer, size_t *buffersize, 
		speed_data_t speed)
{
    fd_set read_set;
    struct timeval timeout;
    int ret;
    
    if (ftp->datastream == -1) {
	statusMessage(ftp, "ERR: No data connection");
	if ((*buffer)) free((*buffer));
	(*buffer) = NULL;
	(*buffersize) = 0;
	return -1;  
    }
    
    FD_ZERO(&read_set);
    
    for (;;) {
	timeout.tv_sec = 0;
	timeout.tv_usec = DEFAULT_TIMEOUT;
	FD_SET(ftp->datastream, &read_set);
	
	ret = TEMP_FAILURE_RETRY(select(FD_SETSIZE, &read_set, NULL, NULL, 
					&timeout));

#ifdef _DEBUG
	statusMessage(ftp, "select in receiveData returned: %d", ret);
#endif
	
	if (ret == -1) {
	    statusMessage(ftp, "ERR: Select returns error: %s", 
			  strerror(errno));
	    if ((*buffer)) free((*buffer));
	    (*buffer) = NULL;
	    (*buffersize) = 0;
	    return -1;
	} else if (ret == 0) {
	    if (doWait(ftp, speed, 0, "DOWNLOAD-DATA1") == -1) {
		if ((*buffer)) free((*buffer));
		(*buffer) = NULL;
		(*buffersize) = 0;
		return -1;
	    }

	    return 1;
	} else {
	    (*buffer) = realloc((*buffer), 
				(*buffersize) + speed->buffersize + 1);
	    ret = ftp->dataRecv(ftp, ftp->datastream, 
				(*buffer) + (*buffersize), 
				speed->buffersize); 

#ifdef _DEBUG
	    statusMessage(ftp, "dataRecv in receiveData returned: %d", ret);
#endif

	    if (ret == -1) {
		if (errno != EWOULDBLOCK) {
		    statusMessage(ftp, "ERR: Recv returns error: %s", 
				  strerror(errno));
		    if ((*buffer)) free((*buffer));
		    (*buffer) = NULL;
		    (*buffersize) = 0;
		    return -1;	
		}
		
		if (doWait(ftp, speed, 0, "DOWNLOAD-DATA2") == -1) {
		    if ((*buffer)) free((*buffer));
		    (*buffer) = NULL;
		    (*buffersize) = 0;
		    return -1;
		}
	    } else if (ret == 0) {		
		/* Done */
		stopDataAuth(ftp);
		shutdown(ftp->datastream, 2);
		close(ftp->datastream);
		ftp->datastream = -1;
		return 0;
	    } else {
		updateSpeed(speed, ret);
		(*buffersize) += ret;
		((char *)(*buffer))[(*buffersize)] = '\0';
		
		if (doWait(ftp, speed, 1, "DOWNLOAD-DATA3") == -1) {
		    if ((*buffer)) free((*buffer));
		    (*buffer) = NULL;
		    (*buffersize) = 0;
		    return -1;
		}

		return 1;
	    }      
	}
    }  
    
    return -1;
}

static size_t calcFormatLength(const char *format, va_list list)
{
    size_t ret = strlen(format) + 10;
    char *s = 0;
    const char *last = format;
    
    while ((s = strchr(last, '%'))) {
	switch (s[1]) {
	case '\0':
	    last = s + 1;
	    break;
	case '%':
	    last = s + 2;
	    break;
	case 's': {
	    const char *str = va_arg(list, const char *);
	    ret += strlen(str);
	    last = s + 1;
	}; break;
	default:
	    va_arg(list, void *);
	    last = s + 1;
	}
    }

    return ret;
}

void statusMessage(pftp_server_t ftp, const char *format, ...)
{
    va_list list, save;
    char *dump;
    size_t len;

    va_start(list, format);
    
#ifdef __va_copy
    __va_copy(save, list);
#else
    save = list;
#endif /* __va_copy */
    
    len = calcFormatLength(format, list) + 512;
    dump = malloc(len);
    
    vsnprintf(dump, len, format, save);
    send_comm(ftp, PFTP_STATUS, ftp->settings->name, dump);
    free(dump);
    va_end(list);
}

int cmdcmp(unsigned short code1, unsigned short code2)
{
    unsigned short c1 = code1 / 10, c2 = code2 / 10;
    
    if (c1 < c2) return -1;
    if (c1 > c2) return 1;
    
    return 0;
}

speed_data_t initSpeed(pftp_server_t ftp, uint64_t total, uint64_t start)
{
    speed_data_t ret = malloc(sizeof(struct speed_data_s));
    
    memset(ret, 0, sizeof(struct speed_data_s));
    
    ret->total = total;
    ret->done = start;
    ret->unknown = (ret->total == 0);
    // ret->lastspeed = 0.0;
    // ret->lastchange = 0;
    ret->speed = 0.0;
    ret->percent = 0.0f;
    ret->buffersize = ftp->lastgoodbuffersize;
    ret->lastdata = 0;
    gettimeofday(&ret->lasttime, NULL);
    
    return ret;
}

void freeSpeed(pftp_server_t ftp, speed_data_t *speed)
{
    if (speed && (*speed)) {
	ftp->lastgoodbuffersize = (*speed)->buffersize;
	free((*speed));
	(*speed) = NULL;
    }
}

void updateSpeed(speed_data_t speed, size_t data)
{
    struct timeval now;
    long diffsec;
    
    gettimeofday(&now, NULL);
    diffsec = now.tv_sec - speed->lasttime.tv_sec;
    
    speed->done += data;
    speed->lastdata += data;
    
    if (diffsec > 0) {
	speed->speed = (double)speed->lastdata / (double)diffsec;    
	
	/*  if (speed->speed < speed->lastspeed) {
	    printf("speed < lastspeed (%d)\n", speed->lastchange);
	    speed->lastchange = !speed->lastchange;
	    }
	    
	    if (!speed->lastchange) {
	    speed->buffersize *= 2;
	    } else {
	    if (speed->buffersize > 256)
	    speed->buffersize /= 2;
	    }
	    
	    printf("%lu\n", speed->buffersize);
	    
	    speed->lastspeed = speed->speed; */
	
	speed->lasttime.tv_sec = now.tv_sec;
	speed->lasttime.tv_usec = 0;
	
	speed->lastdata = 0;
    }
    
    if (speed->total > 0) {
	speed->percent = ((float)speed->done * 100.0f) / (float)speed->total;
    } else {
	speed->percent = 0.0f;
    }
}

int sendAccount(pftp_server_t ftp)
{
    static int imrunning = 0;
    
    if (imrunning) {
	statusMessage(ftp, "ERR: FTP server was naughty");
	return -1;
    }
    
    if (ftp->settings->account) {
	if (sendCmd(ftp, "ACCT", ftp->settings->account) == -1) {
	    return -1;
	}
	
	imrunning = 1;
	
	if (receiveCmd(ftp, NULL) == 230) {
	    imrunning = 0;
	    return 1;
	}
	
	imrunning = 0;
    }
    
    return -1;
}

int abortData(pftp_server_t ftp)
{
    if (ftp->abort) {
	unsigned short ret;
	if (sendCmd(ftp, "ABOR", NULL) == -1) {
	    shutdown(ftp->datastream, 2);
	    close(ftp->datastream);
	    ftp->datastream = -1;
	    return -1;	
	}

	shutdown(ftp->datastream, 2);
	close(ftp->datastream);
	ftp->datastream = -1;

	do {
	    ret = _receiveCmd(ftp, NULL, 1);
	    if (cmdcmp(ret, 220) == 0)
		return 0;
	    if (ret == (unsigned short)-1) {
		/* Timeout. Close control connection. */
#ifndef NO_SSL
		if (ftp->cmdssl) SSL_shutdown(ftp->cmdssl);	    
#endif /* NO_SSL */
		shutdown(ftp->cmdstream, 2);
		close(ftp->cmdstream);
		ftp->cmdstream = -1;
		send_comm(ftp, PFTP_DISCONNECTED, NULL, NULL);
		return -1;
	    }
	} while (cmdcmp(ret, 500));
    } else {
	shutdown(ftp->datastream, 2);
	close(ftp->datastream);
	ftp->datastream = -1;
    }
    
    return -1;
}

void startWait(pftp_server_t ftp, wait_mode_t mode, intptr_t trans)
{
    ftp->lastdata = time(NULL);
    ftp->waitmode = mode;
    send_comm(ftp, PFTP_INITTRANSFER, (pftp_param_t)trans, NULL);
}

int doWait(pftp_server_t ftp, speed_data_t speed, int gotdata, const char *op)
{
    time_t now = time(NULL);
    int cancel = 0;
    
    if (gotdata) {
	ftp->lastdata = now;
    } else {
	if (now - ftp->lastdata > TIMEOUT[(int)ftp->waitmode]) {
	    statusMessage(ftp, "ERR: Operation (%s) timed out.", op);
	    return -1;
	}
    }
    
    if (speed) { 
	if (speed->unknown)
	    cancel = send_comm(ftp, PFTP_TRANSFER, 
			       (pftp_param_t) &speed->speed, NULL);
	else
	    cancel = send_comm(ftp, PFTP_TRANSFER, 
			       (pftp_param_t) &speed->speed, 
			       (pftp_param_t) &speed->percent);
    } else
	cancel = send_comm(ftp, PFTP_TRANSFER, NULL, NULL);
    
    if (cancel == 0) {
	statusMessage(ftp, "ERR: User aborted (%s)", op);
	return -1;
    }
    
    return 0;
}

void endWait(pftp_server_t ftp, wait_mode_t mode, intptr_t error)
{
    ftp->lastdata = 0;
    if (mode != ftp->waitmode)
	statusMessage(ftp, "WARN: Drunk programmer!");
    send_comm(ftp, PFTP_DONETRANSFER, (pftp_param_t) error, NULL);
}

ssize_t intRecv(pftp_server_t ftp, int socket, void *buffer, size_t len)
{
    return recv(socket, buffer, len, 0);
}

ssize_t intCRCRecv(pftp_server_t ftp, int socket, void *buffer, size_t len)
{
    int ret = recv(socket, buffer, len, 0);
    if (ret > 0) 
	pftp_addCRC32(&ftp->cur_crc, buffer, ret);
    return ret;
}

ssize_t intSend(pftp_server_t ftp, int socket, const void *buffer, size_t len)
{
    return send(socket, buffer, len, 0);
}

#ifndef NO_SSL
ssize_t intSSLRecv(pftp_server_t ftp, int socket, void *buffer, size_t len)
{
    SSL *ssl;
    
    if (socket == ftp->cmdstream)
	ssl = ftp->cmdssl;
    else
	ssl = ftp->datassl;
    
    return SSL_read(ssl, buffer, (int)(len & INT_MAX));
}

ssize_t intCRCSSLRecv(pftp_server_t ftp, int socket, void *buffer, size_t len)
{
    SSL *ssl;
    int ret;
    
    if (socket == ftp->cmdstream)
	ssl = ftp->cmdssl;
    else
	ssl = ftp->datassl;
    
    ret = SSL_read(ssl, buffer, (int)(len & INT_MAX));
    if (ret > 0)
	pftp_addCRC32(&ftp->cur_crc, buffer, ret);
    return ret;
}

ssize_t intSSLSend(pftp_server_t ftp, int socket, const void *buffer, 
		   size_t len)
{
    SSL *ssl;
    
    if (socket == ftp->cmdstream)
	ssl = ftp->cmdssl;
    else
	ssl = ftp->datassl;
    
    return SSL_write(ssl, buffer, (int)(len & INT_MAX));
}

int initSSL(pftp_server_t ftp, SSL **ssl, socket_t sock)
{
    int retdata;
    
    statusMessage(ftp, "SSL: Init secure connection");
    ssl_cur_ftp = ftp;
    
    (*ssl) = SSL_new(ssl_ctx);
    
    if (!(*ssl)) {
	statusMessage(ftp, "ERR: Unable to init SSL");
	return -2;
    }
    if (!SSL_set_fd((*ssl), sock)) {
	statusMessage(ftp, "ERR: Unable to add socket to SSL");
	SSL_free((*ssl));
	(*ssl) = NULL;
	return -2;
    }
    
    SSL_ctrl((*ssl), SSL_CTRL_MODE, SSL_MODE_ACCEPT_MOVING_WRITE_BUFFER, 0);
    
    if (ftp->cmdssl && (*ssl) != ftp->cmdssl)
	SSL_copy_session_id((*ssl), ftp->cmdssl);
    
    startWait(ftp, SSL_HANDSHAKE, 0);
    
    for (;;) {
	retdata = SSL_connect((*ssl));
	
	if (retdata == 1) {
	    break;
	} else if (retdata == 0) {
	    ERR_print_errors_fp(stderr);
	    statusMessage(ftp, "WARN: TLS handshake was shutdown (%s)", 
			  ERR_error_string(SSL_get_error((*ssl), retdata), 
					   NULL));
	    SSL_free((*ssl));
	    (*ssl) = NULL;
	    endWait(ftp, SSL_HANDSHAKE, 1);
	    return -1;
	} else {
	    fd_set set;
	    struct timeval timeout;
	    
	    timeout.tv_sec = 0;
	    timeout.tv_usec = DEFAULT_TIMEOUT;
	    FD_ZERO(&set); 
	    FD_SET(sock, &set);
	    
	    if (BIO_sock_should_retry(retdata)) {
		if (SSL_want_read((*ssl))) {
		    retdata = TEMP_FAILURE_RETRY(select(FD_SETSIZE, &set, 
							NULL, NULL, &timeout));
		} else if (SSL_want_write((*ssl))) {
		    retdata = TEMP_FAILURE_RETRY(select(FD_SETSIZE, NULL,
							&set, NULL, &timeout));
		} else {
		    if (SSL_want_x509_lookup((*ssl))) {
			puts("wants x509 lookup?");
		    } else {
			puts("What?!");
		    }	  
		}
	    } else {
		ERR_print_errors_fp(stderr);
		retdata = SSL_get_error((*ssl), retdata);
		statusMessage(ftp, "ERR: SSL returned connect error (%d)", 
			      retdata);
		SSL_free((*ssl));
		(*ssl) = NULL;
		endWait(ftp, SSL_HANDSHAKE, 1);
		return -2;
	    }
	    
	    if (retdata == -1) {
		statusMessage(ftp, "ERR: Select returned error: %s", 
			      strerror(errno));
		SSL_free((*ssl));
		(*ssl) = NULL;
		endWait(ftp, SSL_HANDSHAKE, 1);
		return -2;
	    } else if (retdata == 0) {
		if (doWait(ftp, NULL, 0, "SSL_HANDSHAKE") == -1) {
		    SSL_free((*ssl));
		    (*ssl) = NULL;
		    endWait(ftp, SSL_HANDSHAKE, 1);
		    return -1;
		}
	    } else {
		if (doWait(ftp, NULL, 1, "SSL_HANDSHAKE") == -1) {
		    SSL_free((*ssl));
		    (*ssl) = NULL;
		    endWait(ftp, SSL_HANDSHAKE, 1);
		    return -1;
		}
	    }      
	}
    }

    endWait(ftp, SSL_HANDSHAKE, 0);
    
    return 0;
}

/* Most of this code is based on source from lftp */
static int sslVerifyCrl(X509_STORE_CTX *ctx)
{
    X509_OBJECT obj;
    X509_NAME *subject;
    X509_NAME *issuer;
    X509 *xs;
    X509_CRL *crl;
    X509_REVOKED *revoked;
    X509_STORE_CTX store_ctx;
    long serial;
    int i, n, rc;
    char *cp;
    
    statusMessage(ssl_cur_ftp, "SSL: Verifying CERT...");
    
    if (!x509_store) {
	statusMessage(ssl_cur_ftp, "OK (Nothing to check agains)");
	return 1;
    }
    
    xs      = X509_STORE_CTX_get_current_cert(ctx);
    subject = X509_get_subject_name(xs);
    issuer  = X509_get_issuer_name(xs);
    
    memset((char *)&obj, 0, sizeof(obj));
    X509_STORE_CTX_init(&store_ctx, x509_store, NULL, NULL);
    rc = X509_STORE_get_by_subject(&store_ctx, X509_LU_CRL, subject, &obj);
    X509_STORE_CTX_cleanup(&store_ctx);
    crl = obj.data.crl;
    
    if (rc > 0 && crl != NULL) {
	if (X509_CRL_verify(crl, X509_get_pubkey(xs)) <= 0) {
	    statusMessage(ssl_cur_ftp, "Invalid signature on CRL!");
	    X509_STORE_CTX_set_error(ctx, X509_V_ERR_CRL_SIGNATURE_FAILURE);
	    X509_OBJECT_free_contents(&obj);
	    return 0;
	}
	
	i = X509_cmp_current_time(X509_CRL_get_nextUpdate(crl));
	if (i == 0) {
	    statusMessage(ssl_cur_ftp, 
			  "Found CRL has invalid nextUpdate field!");
	    X509_STORE_CTX_set_error(ctx, X509_V_ERR_ERROR_IN_CRL_NEXT_UPDATE_FIELD);
	    X509_OBJECT_free_contents(&obj);
	    return 0;
	}
	if (i < 0) {
	    statusMessage(ssl_cur_ftp, "Found CRL is expired!");
	    X509_STORE_CTX_set_error(ctx, X509_V_ERR_CRL_HAS_EXPIRED);
	    X509_OBJECT_free_contents(&obj);
	    return 0;
	}
	X509_OBJECT_free_contents(&obj);
    }
    
    memset((char *)&obj, 0, sizeof(obj));
    X509_STORE_CTX_init(&store_ctx, x509_store, NULL, NULL);
    rc = X509_STORE_get_by_subject(&store_ctx, X509_LU_CRL, issuer, &obj);
    X509_STORE_CTX_cleanup(&store_ctx);
    crl = obj.data.crl;
    
    if (rc > 0 && crl != NULL) {
	n = sk_X509_REVOKED_num(X509_CRL_get_REVOKED(crl));
	for (i = 0; i < n; i++) {
	    revoked = sk_X509_REVOKED_value(X509_CRL_get_REVOKED(crl), i);
	    if (ASN1_INTEGER_cmp(revoked->serialNumber, 
				 X509_get_serialNumber(xs)) == 0) {
		serial = ASN1_INTEGER_get(revoked->serialNumber);
		cp = X509_NAME_oneline(issuer, NULL, 0);
		statusMessage(ssl_cur_ftp, "Certificate with serial %ld (0x%lX) revoked per CRL from issuer %s!", serial, serial, cp ? cp : "(ERROR)");
		free(cp);
		X509_STORE_CTX_set_error(ctx, X509_V_ERR_CERT_REVOKED);
		X509_OBJECT_free_contents(&obj);
		return 0;
	    }
	}
	X509_OBJECT_free_contents(&obj);
    }
    
    statusMessage(ssl_cur_ftp, "OK");
    
    return 1;
}

int sslVerify(int ok, X509_STORE_CTX *ctx)
{
    statusMessage(ssl_cur_ftp, "SSL: Incoming CERT");
    
    if (ok) {
	if (!sslVerifyCrl(ctx))
	    ok = 0;
    } else {
	statusMessage(ssl_cur_ftp, "SSL: CERT Accepted (ha-ha)");
	ok = 1;
    }
    
    return ok;
}

#endif /* NO_SSL */

/* Returns specifically, -2 = Error settings up TLS
   -1 = No protection used (allowed)
   0 = Protected */

int initAuth(pftp_server_t ftp)
{
#ifdef NO_SSL
    return -1; /* somewhat funny but better, take it like "couldn't setup tls"
		  but no error */
#else
    int retdata;
    
    if (sendCmd(ftp, "AUTH", "TLS") == -1) {
	return -2;
    }
    
    retdata = receiveCmd(ftp, NULL);
    
    if (retdata != 234) {
	return -1;
    }
    
    if (!ssl_ctx)
	return -2;
    
    /* Init TLS connection */
    retdata = initSSL(ftp, &ftp->cmdssl, ftp->cmdstream);
    
    if (retdata != 0)
	return retdata;
    
    ftp->cmdRecv = intSSLRecv;
    ftp->cmdSend = intSSLSend;
    
    if (sendCmd(ftp, "PBSZ", "0") == -1) {
	statusMessage(ftp, "ERR: Unable to send data using SSL");
	return -2;
    }
    
    if (receiveCmd(ftp, NULL) != 200) {
	return -1;
    }
    
    if (sendCmd(ftp, "PROT", "P") == -1) {
	return -2;
    }
    
    if (receiveCmd(ftp, NULL) != 200) {
	if (sendCmd(ftp, "PROT", "C") == -1) {
	    return -2;
	}
	if (receiveCmd(ftp, NULL) != 200) {
	    return -2;
	}    
	
	ftp->ssl_private = 0;
	
	return -1;
    }
    
    ftp->ssl_private = 1;
    
    return 0;
#endif  /* NO_SSL */
}

int handleFeat(pftp_server_t ftp, const char *feat_data)
{
    /* AUTH TLS && PBSZ && PROT ==> Handles auth tls*/
    char *s, *last = strchr(feat_data, '\n');
    
    ftp->tls = 0;
  
    if (!last)
	return -1;
    
    for (;;) {
	last++;
	if (last[0] == '\r') last++;
	while (last[0] == ' ') last++;
	
	s = strchr(last, '\n');
	
	if (!s)
	    break;
	
	if (strncmp(last, "AUTH TLS", s - last) == 0 || 
	    strncmp(last, "PBSZ", s - last) == 0 ||
	    strncmp(last, "PROT", s - last) == 0) {
	    ftp->tls++;
	} else if (strncmp(last, "REST", s - last) == 0) {
	    ftp->resume = 1;
	} else if (strncmp(last, "CDUP", s - last) == 0) {
	    ftp->cdup = 1;
	}
	
	last = s;
    }
    
    if (ftp->tls == 3) {
	ftp->tls = 1;
    } else {
	ftp->tls = 0;
    }
    
    return 0;
}

int initDataAuth(pftp_server_t ftp, int list)
{
#ifdef NO_SSL
    return 0;
#else
    if (ftp->cmdssl) {
	if ((list && !getValue(ftp, secure_list)) ||
	    (!list && !getValue(ftp, secure_data))) {
	    if (ftp->ssl_private) {
		/* Clear protections */
		if (sendCmd(ftp, "PROT", "C") == -1) {
		    return -1;
		}
		
		if (receiveCmd(ftp, NULL) != 200) {
		    ftp->ssl_private = 1;
		} else {
		    ftp->ssl_private = 0;
		}
	    }
	} else {
	    /* Private protections */
	    if (!ftp->ssl_private) {
		if (sendCmd(ftp, "PROT", "P") == -1) {
		    return -1;
		}
		
		if (receiveCmd(ftp, NULL) != 200) {
		    ftp->ssl_private = 0;
		} else {
		    ftp->ssl_private = 1;
		}
	    }
	}
    }
    
    return 0;
#endif /* NO_SSL */
}

int startDataAuth(pftp_server_t ftp)
{
#ifdef NO_SSL
    return 0;
#else
    if (ftp->cmdssl && ftp->ssl_private) {
	int retdata = initSSL(ftp, &ftp->datassl, ftp->datastream);
	
	if (retdata != 0)
	    return retdata;
	
	ftp->dataRecv = intSSLRecv;
	ftp->dataSend = intSSLSend;
    }
    
    return 0;
#endif /* NO_SSL */
}

void stopDataAuth(pftp_server_t ftp)
{
#ifndef NO_SSL
    if (ftp->cmdssl && ftp->ssl_private && ftp->datassl) {
	SSL_shutdown(ftp->datassl);
	SSL_free(ftp->datassl);
	ftp->datassl = NULL;
    }
#endif /* NO_SSL */
}

void restoreAuthData(pftp_server_t ftp, int list)
{
#ifndef NO_SSL
    if (ftp->cmdssl) {
	if (!ftp->ssl_private) {
	    /* Private protections */
	    if (sendCmd(ftp, "PROT", "P") == -1)
		return;      
	    
	    if (receiveCmd(ftp, NULL) != 200) {
		ftp->ssl_private = 0;
	    } else {
		ftp->ssl_private = 1;
	    }
	}
    }
#endif /* NO_SSL */
}

int pftp_fxp(pftp_server_t srcftp, const char *srcfile, pftp_server_t destftp, 
	     const char *destfile, char ascii, uint64_t resume_from, 
	     uint64_t append_to, uint64_t total_size)
{
    char *data = NULL;
    struct sockaddr_in addr;
    int need_abort, i, retdata;
    short srcret = 0, destret = 0;
    pftp_server_t pasvftp, portftp;
    
#ifndef NO_SSL
    if (srcftp->ssl_private) {
	if (sendCmd(srcftp, "PROT", "C") == -1 || 
	    receiveCmd(srcftp, NULL) != 200) {
	    return -1;
	}
	srcftp->ssl_private = 0;
    }

    if (destftp->ssl_private) {
	if (sendCmd(destftp, "PROT", "C") == -1 || 
	    receiveCmd(destftp, NULL) != 200) {
	    restoreAuthData(srcftp, 0);
	    return -1;
	}
	destftp->ssl_private = 0;
    }
#endif

    need_abort = 0;

    /* These are switched because for switch them first thing */
    if (srcftp->pasv) {
	pasvftp = destftp;
	portftp = srcftp;
    } else {
	pasvftp = srcftp;
	portftp = destftp;
    }

    if (getMode(srcftp, (ascii == 'B')) == -1) {
	if (getMode(srcftp, 0) == -1 || getMode(destftp, 0) == -1)
	    return -1;
    } else {
	if (getMode(destftp, (ascii == 'B')) == -1) {
	    if (getMode(srcftp, 0) == -1 || getMode(destftp, 0) == -1)
		return -1;
	}
    }
  
    for (i = 0; i < 2; i++) {	
	if (need_abort) {
	    abortData(srcftp);
	    abortData(destftp);
	    need_abort = 0;
	}	
	if (pasvftp == srcftp) {
	    pasvftp = destftp;
	    portftp = srcftp;
	} else {
	    pasvftp = srcftp;
	    portftp = destftp;
	}

	if (sendCmd(pasvftp, "PASV", NULL) == -1) continue;
	if (receiveCmd(pasvftp, &data) != 227) {
	    if (data) free(data);
	    data = NULL;
	    continue;
	}   
	if (parsePasvResponse(pasvftp, data, &addr) != 0) {
	    free(data);
	    data = NULL;
	    continue;
	}
	if (data) free(data);   
	data = NULL;
	
	if (generatePortData(portftp, &addr, &data, 1) == -1) {
	    if (data) free(data);
	    data = NULL;
	    continue;
	}    
	if (sendCmd(portftp, "PORT", data) == -1) {
	    free(data);
	    data = NULL;
	    continue;
	}
	if (data) free(data);   
	data = NULL;
	if (receiveCmd(portftp, NULL) != 200) continue;
	need_abort = 1;

	if (resume_from > 0) {
	    char *tmpstr = malloc(50);
	    snprintf(tmpstr, 50, "%" SCNu64, resume_from);

	    if (sendCmd(srcftp, "REST", tmpstr) == -1) {
		free(tmpstr);
		continue;
	    }
	    retdata = receiveCmd(srcftp, NULL);
	    if (retdata != 150 && retdata != 125) {
		free(tmpstr);
		continue;
	    }
	    free(tmpstr);
	}
	if (append_to > 0) {
	    char *tmpstr = malloc(50);
	    snprintf(tmpstr, 50, "%" SCNu64, append_to);
	    
	    if (sendCmd(destftp, "REST", tmpstr) == -1) {
		free(tmpstr);
		continue;
	    }
	    retdata = receiveCmd(destftp, NULL);
	    if (retdata != 150 && retdata != 125) {
		free(tmpstr);
		continue;
	    }
	    free(tmpstr);
	}

	if (sendCmd(srcftp, "RETR", srcfile) == -1) continue;
	if (sendCmd(destftp, "STOR", destfile) == -1) continue;
	retdata = receiveCmd(srcftp, NULL);
	if (retdata != 150 && retdata != 125) continue;
	retdata = receiveCmd(destftp, NULL);
	if (retdata != 150 && retdata != 125) continue;

	break;
    }

    if (i == 2) {
	if (need_abort) {
	    abortData(srcftp);
	    abortData(destftp);
	}
	restoreAuthData(srcftp, 0);
	restoreAuthData(destftp, 0);	
	return -1;
    }

/*
  send_comm(srcftp, PFTP_FILETRANSFER, (pftp_param_t)&resume_from,
                       (pftp_param_t)&total_size);
  send_comm(srcftp, PFTP_INITTRANSFER, 1, 0);
  send_comm(destftp, PFTP_INITTRANSFER, 2, 0);
*/
    if (receiveCmd2(srcftp, destftp, &srcret, &destret, NULL, NULL) == -1) {
	abortData(srcftp);
	abortData(destftp);
	restoreAuthData(srcftp, 0);
	restoreAuthData(destftp, 0);	
/*
  send_comm(srcftp, PFTP_DONETRANSFER, 1, 0);
  send_comm(destftp, PFTP_DONETRANSFER, 1, 0);
*/
	return -1;
    }

    restoreAuthData(srcftp, 0);
    restoreAuthData(destftp, 0);	

    if (srcret != 226 || destret != 226) {
/*
  send_comm(srcftp, PFTP_DONETRANSFER, 1, 0);
  send_comm(destftp, PFTP_DONETRANSFER, 1, 0);
*/
	return -1;	
    } else {
/*	
  send_comm(srcftp, PFTP_DONETRANSFER, 0, 0);
  send_comm(destftp, PFTP_DONETRANSFER, 0, 0);
*/
	return 0;
    }
}

system_t parseSYST(const char *data)
{
    if (strncmp(data, "UNIX", 4) == 0) {
	return UNIX;
    }
    if (strncasecmp(data, "Windows", 7) == 0) {
	return DOS;
    }

    return UNIX; /* Default */
}

void pftp_clear_dir_cache(pftp_server_t ftp, const char *directory)
{
    if (directory) {
	size_t len = strlen(directory), c;
	char *real_path = NULL;
	if (len > 1 && directory[len - 1] == '/') len--;

	pftp_curdir(ftp, &real_path, 0);
	
	if (real_path) {
	    if (directory[0] == '/') {
		real_path = realloc(real_path, len + 1);
		strncpy(real_path, directory, len);
		real_path[len] = '\0';
	    } else {
		c = strlen(real_path);
		real_path = realloc(real_path, c + len + 2);
		
		if (real_path[c - 1] != '/') {
		    strcat(real_path, "/");
		    c++;
		}
				
		strncat(real_path, directory, len);
		len += c;
		real_path[len] = '\0';
	    }
	    
	    if (!pftp_compactPath(&real_path)) {
		pftp_curdir(ftp, &real_path, 1);
	    }

	    len = (real_path ? strlen(real_path) : 0);
	} else {
	    real_path = malloc(len + 1);
	    strncpy(real_path, directory, len);
	    real_path[len] = '\0';
	}

	if (len > 1 && real_path[len - 1] == '/') len--;

	if (real_path) {
	    for (c = 0; c < ftp->cached; c++) {
		if (strncmp(real_path, ftp->cache_path[c], len) == 0) {
		    pftp_free_fdt(ftp->cache_dir[c]);
		    free(ftp->cache_path[c]);
		    ftp->cached--;
		    memmove(ftp->cache_dir + c, ftp->cache_dir + c + 1, 
			    (ftp->cached - c) * sizeof(pftp_directory_t));
		    memmove(ftp->cache_path + c, ftp->cache_path + c + 1, 
			    (ftp->cached - c) * sizeof(char *));
		    break;
		}
	    }
	    free(real_path);
	}
    } else {
	size_t c;
	for (c = 0; c < ftp->cached; c++) {
	    pftp_free_fdt(ftp->cache_dir[c]);
	    free(ftp->cache_path[c]);
	}
	if (ftp->cache_dir) free(ftp->cache_dir);
	if (ftp->cache_path) free(ftp->cache_path);
	ftp->cache_dir = NULL;
	ftp->cache_path = NULL;
	ftp->cached = 0;
    }
}

int send_comm(pftp_server_t ftp, pftp_msg_t msg, 
	      pftp_param_t param1, pftp_param_t param2)
{
    return ftp->comm_central(msg, ftp, ftp->settings->userdata, 
			     param1, param2);
}
