#include <diskcache_config.h>

#include <string>

#include "genericAPI/Stat.hh"

#include "diskcacheAPI/Common/Variables.hh"

#include "diskcacheAPI/Streams/ASCII.hh"
#include "diskcacheAPI/Streams/Binary.hh"

#include "diskcacheAPI/Cache/HotDirectory.hh"
#include "diskcacheAPI/Cache/SDGTxStreamASCII.hh"
#include "diskcacheAPI/Cache/SDGTxStreamBinary.hh"

#include "diskcache.hh"
#include "diskcachecmd.hh"
#include "diskcacheAPI.hh"
#include "MetaCommands.hh"
#include "MountPointScanner.hh"

static void variables_init( );

using LDASTools::AL::MutexLock;

namespace
{
  // Directory Variables
  void directory_timeout_reader( std::ostringstream& Value );
  void directory_timeout_writer( const std::string& Value );

  // Hot Directory Variables
  void hdas_reader( std::ostringstream& Value );
  void hdas_writer( const std::string& Value );
  void hdsi_reader( std::ostringstream& Value );
  void hdsi_writer( const std::string& Value );
  // MountPointScanner
  void mps_concurrency_reader( std::ostringstream& Value );
  void mps_concurrency_writer( const std::string& Value );

  void oav_reader( std::ostringstream& Value );
  void oav_writer( const std::string& Value );

  void rwlock_interval_ms_reader( std::ostringstream& Value );
  void rwlock_interval_ms_writer( const std::string& Value );
  void rwlock_timeout_ms_reader( std::ostringstream& Value );
  void rwlock_timeout_ms_writer( const std::string& Value );

  void st_reader( std::ostringstream& Value );
  void st_writer( const std::string& Value );
}

namespace diskCache
{
  //-----------------------------------------------------------------------------
  // Number of micro seconds to allow for lock acquisition
  // int RWLOCK_TIMEOUT = 10000;
  //-----------------------------------------------------------------------------
  int RWLOCK_TIMEOUT = 0;

  //-----------------------------------------------------------------------------
  /// This variable specifies the number of milliseconds to delay between
  /// successive scans of the list of mount points.
  ///
  /// The default value is 500 ( 0.5 sec. ).
  //-----------------------------------------------------------------------------
  INT_4U ScanInterval = 500;

  //-----------------------------------------------------------------------------
  /// The runtime initialization of the core ensures that many of the callback
  /// tables for various classes are properly initialized
  //-----------------------------------------------------------------------------
  void
  Initialize( )
  {
    typedef diskCache::Common::Registry::AlreadyRegisteredException
      AlreadyRegisteredException;

    variables_init( );
    diskCache::Cache::RegistrySingleton::id_type
      sdgtx_id( diskCache::Cache::RegistrySingleton::KEY_NULL );

    try
    {
      sdgtx_id = diskCache::Cache::SDGTx::RegisterSearchEngine( );
    }
    catch( const AlreadyRegisteredException& Exception )
    {
      sdgtx_id = Exception.Id( );
    }
    //---------------------------------------------------------------------
    // Initialize ascii stream callbacks
    //---------------------------------------------------------------------
    Streams::OASCII::Writer( sdgtx_id,
			     Cache::SDGTxDirectoryInfoToOASCII );
    //---------------------------------------------------------------------
    // Initialize binary stream callbacks
    //---------------------------------------------------------------------
    Streams::IBinary::Reader( sdgtx_id,
			      Cache::SDGTxDirectoryInfoFromIBinary );
    Streams::OBinary::Writer( sdgtx_id,
			      Cache::SDGTxDirectoryInfoToOBinary );
    //---------------------------------------------------------------------
    // Initialize indexing scheme
    //---------------------------------------------------------------------

  }

  static MutexLock::baton_type  p_scanned_files_baton;
  static total_file_count_type	p_scanned_files = 0;

  total_file_count_type
  ScannedFiles( )
  {
    MutexLock      lock( p_scanned_files_baton,
			 __FILE__, __LINE__ );

    return p_scanned_files;
  }

  void
  ScannedFiles( total_file_count_type Value )
  {
    MutexLock      lock( p_scanned_files_baton,
			 __FILE__, __LINE__ );

    if ( Value == SCANNED_FILES_RESET )
    {
      p_scanned_files = 0;
    }
    else
    {
      p_scanned_files += Value;
    }
  }
 }


void
variables_init( )
{
  std::ostringstream dflt;

  dflt.str( "" );
  directory_timeout_reader( dflt );
  diskCache::Common::Variables::Init( diskCache::MetaCommand::Daemon::VAR_NAME_DIRECTORY_TIMEOUT,
				      directory_timeout_reader,
				      directory_timeout_writer,
				      dflt.str( ) );

  dflt.str( "" );
  mps_concurrency_reader( dflt );
  diskCache::Common::Variables::Init( diskCache::MetaCommand::Daemon::VAR_NAME_CONCURRENCY,
				      mps_concurrency_reader,
				      mps_concurrency_writer,
				      dflt.str( ) );

  dflt.str( "" );
  hdas_reader( dflt );
  diskCache::Common::Variables::Init( diskCache::MetaCommand::Daemon::VAR_NAME_HOT_DIRECTORY_AGE,
				      hdas_reader,
				      hdas_writer,
				      dflt.str( ) );

  dflt.str( "" );
  hdsi_reader( dflt );
  diskCache::Common::Variables::Init( diskCache::MetaCommand::Daemon::VAR_NAME_HOT_DIRECTORY_SCAN_INTERVAL,
				      hdsi_reader,
				      hdsi_writer,
				      dflt.str( ) );

  dflt.str( "" );
  oav_reader( dflt );
  diskCache::Common::Variables::Init( "OUTPUT_ASCII_VERSION",
				      oav_reader,
				      oav_writer,
				      dflt.str( ) );

  dflt.str( "" );
  st_reader( dflt );
  diskCache::Common::Variables::Init( diskCache::MetaCommand::Daemon::VAR_NAME_RWLOCK_INTERVAL,
				      rwlock_interval_ms_reader,
				      rwlock_interval_ms_writer,
				      dflt.str( ) );

  dflt.str( "" );
  st_reader( dflt );
  diskCache::Common::Variables::Init( diskCache::MetaCommand::Daemon::VAR_NAME_RWLOCK_TIMEOUT,
				      rwlock_timeout_ms_reader,
				      rwlock_timeout_ms_writer,
				      dflt.str( ) );

  dflt.str( "" );
  st_reader( dflt );
  diskCache::Common::Variables::Init( diskCache::MetaCommand::Daemon::VAR_NAME_STAT_TIMEOUT,
				      st_reader,
				      st_writer,
				      dflt.str( ) );
}

namespace
{
  using namespace diskCache;
  using Cache::HotDirectory;

  void
  hdas_reader( std::ostringstream& Value )
  {
    Value << HotDirectory::HotLowerBound( );
  }

  void
  hdas_writer( const std::string& Value )
  {
    std::istringstream			vs( Value );
    HotDirectory::timestamp_type	v;

    vs >> v;

    HotDirectory::HotLowerBound( v );
  }

  void
  hdsi_reader( std::ostringstream& Value )
  {
    Value << HotDirectory::ScanInterval( );
  }

  void
  hdsi_writer( const std::string& Value )
  {
    std::istringstream			vs( Value );
    HotDirectory::timestamp_type	v;

    vs >> v;

    HotDirectory::ScanInterval( v );
  }

  //---------------------------------------------------------------------

  void
  mps_concurrency_reader( std::ostringstream& Value )
  {
    Value << MountPointScanner::Concurrency( );
  }

  void
  mps_concurrency_writer( const std::string& Value )
  {
    INT_4U		v;

    std::istringstream	s( Value );

    s >> v;
    ScanConcurrency( v );
  }

  //---------------------------------------------------------------------

  void
  oav_reader( std::ostringstream& Value )
  {
    Value << diskCache::DumpCacheDaemon::ASCIIVersion( );
  }

  void
  oav_writer( const std::string& Value )
  {
    diskCache::DumpCacheDaemon::version_type	v;

    if ( Value.substr( 0, 2 ).compare( "0x" ) == 0 )
    {
      std::istringstream	s( Value.substr( 2 ) );
      s >> std::hex >> v;
    }
    else if ( Value[0] == '0' )
    {
      std::istringstream	s( Value.substr( 1 ) );
      s >> std::oct >> v;
    }
    else
    {
      std::istringstream	s( Value );
      s >> std::hex >> v;
    }
    diskCache::DumpCacheDaemon::ASCIIVersion( v );
  }

  //---------------------------------------------------------------------

  void
  rwlock_interval_ms_reader( std::ostringstream& Value )
  {
    Value << LDASTools::AL::ReadWriteLock::Interval( -1 );
  }

  void
  rwlock_interval_ms_writer( const std::string& Value )
  {
    int			v;

    std::istringstream	s( Value );

    s >> v;
    if ( v >= 0 )
    {
      (void)LDASTools::AL::ReadWriteLock::Interval( v );
    }
  }

  //---------------------------------------------------------------------

  void
  rwlock_timeout_ms_reader( std::ostringstream& Value )
  {
    Value << LDASTools::AL::ReadWriteLock::Timeout( -1 );
  }

  void
  rwlock_timeout_ms_writer( const std::string& Value )
  {
    int			v;

    std::istringstream	s( Value );

    s >> v;
    if ( v >= 0 )
    {
      (void)LDASTools::AL::ReadWriteLock::Timeout( v );
    }
  }

  //---------------------------------------------------------------------

  void
  st_reader( std::ostringstream& Value )
  {
    Value << GenericAPI::lstat_timeout( );
  }

  void
  st_writer( const std::string& Value )
  {
    INT_4U		v;

    std::istringstream	s( Value );

    s >> v;
    if ( v > 0 )
    {
      GenericAPI::SetStatTimeout( v );
    }
  }

  //---------------------------------------------------------------------

  void
  directory_timeout_reader( std::ostringstream& Value )
  {
    Value << LDASTools::AL::Directory::Timeout( );
  }

  void
  directory_timeout_writer( const std::string& Value )
  {
    INT_4U		v;

    std::istringstream	s( Value );

    s >> v;
    if ( v > 0 )
    {
      LDASTools::AL::Directory::Timeout( v );
    }
  }

}
