package testvectors

const ChainTests = `
{
  "tests": {
    "test_ralph_sig_chain": {
      "_comment": "Ralph is a test user I created by hand on my local server. I fetched his sigs and keys from the API, and then massaged them into our input format. This test is mainly to make sure that the generated chains we're using in other tests bear some relationship to reality.  - Jack",
      "input": "ralph_chain.json",
      "len": 5,
      "sibkeys": 3,
      "subkeys": 2,
      "eldest": "second_eldest"
    },

    "test_max_sig_chain": {
      "_comment": "This is Max Krohn's real live sigchain, as of 10 Aug 2017. The immediate reason for including it is the following interesting case: Max has revoked his very first link, which is both 1) a Twitter proof but also 2) implicitly delegates his eldest key. We should not treat this revocation as revoking the eldest key, and if we do, validating his chain will break.",
      "input": "max_chain.json",
      "len": 313,
      "sibkeys": 12,
      "subkeys": 10
    },

    "test_simple_chain": {
      "_comment": "Test a simple chain, just one link.",
      "input": "simple_chain.json",
      "len": 1,
      "sibkeys": 1,
      "subkeys": 0
    },

    "test_error_unknown_key": {
      "_comment": "Check the case where a signing kid is simply missing from the list of available keys (as opposed to invalid for some other reason, like having been revoked).",
      "input": "missing_kid_chain.json",
      "eldest": "e",
      "err_type": "NONEXISTENT_KID"
    },

    "test_error_unknown_reverse_sig_key": {
      "_comment": "As above, but for a reverse sig.",
      "input": "missing_reverse_kid_chain.json",
      "eldest": "e",
      "err_type": "NONEXISTENT_KID"
    },

    "test_error_bad_signature": {
      "_comment": "Change some bytes from the valid signature, and confirm it gets rejected.",
      "input": "bad_signature_chain.json",
      "err_type": "VERIFY_FAILED"
    },

    "test_error_bad_reverse_signature": {
      "_comment": "Change some bytes from the valid reverse signature, and confirm it gets rejected.",
      "input": "bad_reverse_signature_chain.json",
      "err_type": "REVERSE_SIG_VERIFY_FAILED"
    },

    "test_error_mismatched_kid": {
      "_comment": "We need to use the server-provided KID to unbox a signature. We always need to check back after unboxing to make sure the internal KID matches the one we actually used. This test exercises that check. NOTE: I generated this chain by hacking some code into kbpgp to modify the payload right before it was signed.",
      "input": "mismatched_kid_chain.json",
      "err_type": "KID_MISMATCH"
    },

    "test_error_mismatched_fingerprint": {
      "_comment": "We don't use fingerprints in unboxing, but nonetheless we want to make sure that if a chain link claims to have been signed by a given fingerprint, that does in fact correspond to the KID of the PGP key that signed it. NOTE: I generated this chain by hacking some code into kbpgp to modify the payload right before it was signed.",
      "input": "mismatched_fingerprint_chain.json",
      "err_type": "FINGERPRINT_MISMATCH"
    },

    "test_revokes": {
      "_comment": "The chain is unrevoked-length 17, but 4 sigs are revoked, so the apparent length should be 13. 5 sibkeys get delegated (plus the eldest), but 4 get revoked, for a final total of 2. Likewise 3 subkeys get delegated, but 2 get revoked, for a final total of 1.",
      "input": "example_revokes_chain.json",
      "len": 13,
      "sibkeys": 2,
      "subkeys": 1
    },

    "test_error_revoked_key": {
      "_comment": "Try signing a link with a key that was previously revoked.",
      "input": "signed_with_revoked_key_chain.json",
      "err_type": "INVALID_SIBKEY"
    },

    "test_error_expired_key": {
      "_comment": "Try signing a link with a key that has expired.",
      "input": "expired_key_chain.json",
      "err_type": "EXPIRED_SIBKEY"
    },

    "test_error_bad_uid": {
      "input": "bad_uid_chain.json",
      "err_type": "WRONG_UID"
    },

    "test_error_bad_username": {
      "input": "bad_username_chain.json",
      "err_type": "WRONG_USERNAME"
    },

    "test_error_bad_prev": {
      "input": "bad_prev_chain.json",
      "err_type": "WRONG_PREV"
    },

    "test_error_bad_seqno": {
      "input": "bad_seqno_chain.json",
      "err_type": "WRONG_SEQNO"
    },

    "test_empty_sigchain": {
      "input": "empty_chain.json",
      "len": 0,
      "sibkeys": 1,
      "subkeys": 0,
      "eldest": "ralph"
    },

    "test_error_empty_sigchain_bad_key_userid": {
      "input": "empty_chain.json",
      "eldest": "steve",
      "err_type": "KEY_OWNERSHIP"
    },

    "test_error_empty_sigchain_nonexistent_pgp": {
      "input": "empty_chain.json",
      "eldest": "nonexistent",
      "err_type": "NONEXISTENT_KID"
    },

    "test_error_empty_sigchain_eldest_key_is_nacl": {
      "_comment": "NaCl keys don't have any internal userid info, so they can't stand alone as an eldest key without having signed some links.",
      "input": "empty_chain.json",
      "eldest": "naclkey",
      "err_type": "KEY_OWNERSHIP"
    },

    "test_jack_chain": {
      "_comment": "Just a sanity check with my own sigchain",
      "input": "jack_chain.json",
      "len": 29,
      "sibkeys": 1,
      "subkeys": 0
    },

    "test_error_missing_first_link": {
      "_comment": "We need to make sure the server can't drop links off the front. This would be a risk if we assumed that, for example, they belonged to a previous eldest key.",
      "input": "jack_chain_missing_first_link.json",
      "err_type": "WRONG_SEQNO"
    },

    "test_error_substitute_first_link": {
      "_comment": "Here the first link has been substituted for one signed by a different eldest key. We should notice the payload hash mismatch.",
      "input": "jack_chain_substitute_first_link.json",
      "eldest": "real_eldest",
      "err_type": "WRONG_PREV"
    },

    "test_ralph_with_earlier_eldest_key": {
      "_comment": "Eldest keys can be reused, which means there can be subchains other than the latest one that correspond to the current eldest key. Those must be ignored. Previously, before we supported repeating eldest keys at all, we used to require that this was an error. Note that if the latest links don't correspond to the current eldest key, as we're testing here, that's an implicit reset, and we return no links.",
      "input": "ralph_chain.json",
      "eldest": "first_eldest",
      "len": 0,
      "sibkeys": 1,
      "subkeys": 0
    },

    "test_ralph_with_new_eldest_key": {
      "input": "ralph_chain.json",
      "eldest": "new_eldest",
      "len": 0,
      "sibkeys": 1,
      "subkeys": 0
    },

    "test_melvin_with_expiring_pgp_key": {
      "_comment": "Right now this just exercises some of the code that recognizes the expiration time. Eventually we should add tests that make sure the etime is really enforced.",
      "input": "melvin_chain.json",
      "len": 1,
      "sibkeys": 1,
      "subkeys": 0
    },

    "test_error_ralph_with_missing_subkey": {
      "input": "nonexistent_subkey_chain.json",
      "err_type": "NONEXISTENT_KID"
    },

    "test_helen_chain": {
      "_comment": "A copy of t_helen's sigchain (also available as a permanent test user). Tests PGP keys with changing subkeys. She signed up with a PGP key, added a bitcoin address, and then changed that address, for a total of 3 sigchain links (but the first bitcoin link is revoked, so the final len is 2). In between each one she deleted the signing subkey that was used before and added a new one. So the only way to validate this sigchain is to do proper PGP key merging on all the bundles that the server returns. Note that the order of the key bundles given is intentionally rearranged, to make it less likely that the Go client will spuriously pass this test by only checking the last link against the last key.",
      "input": "t_helen_chain.json",
      "len": 2,
      "sibkeys": 1,
      "subkeys": 0
    },
    "test_pgp_subkey_after_revoke": {
      "_comment": "The eldest link is signed with a PGP key with an extra signing subkey, then a pgp_update link is used to update the key to a version where that subkey has been revoked. Then, another pgp_update link appears which tries to revert to the original version of the key, but is signed by the revoked subkey.",
      "input": "old_pgp_key.json",
      "err_type": "VERIFY_FAILED"
    },
    "test_pgp_bad_full_hash": {
      "_comment": "The eldest link is signed with a PGP key and includes a full_hash which doesn't match the key",
      "input": "bad_full_hash.json",
      "err_type": "NO_KEY_WITH_THIS_HASH"
    },
    "test_pnelsen_sigchain": {
      "_comment": "pnelsen was one of several users affected by the 15 Sep 2015 white-space stripping bug on the server",
      "input": "pnelsen_chain.json",
      "len": 21,
      "sibkeys": 1,
      "subkeys": 0
    },
    "test_akalin_sigchain": {
      "_comment": "Fred has some unverifiable links in his chain from old bugs of ours. These need to be special cased.",
      "input": "akalin_chain.json",
      "len": 31,
      "sibkeys": 1,
      "subkeys": 0
    },
    "test_repeat_eldest": {
      "_comment": "We now define 'eldest' links to imply a sigchain reset. This is a basic test case with two links, both of type eldest, delegating the same eldest key. Old clients would've parsed this as a 2-link chain (where the second link was a no-op), but conforming clients should respect the reset adn return a 1-link chain.",
      "input": "repeat_eldest.json",
      "len": 1,
      "sibkeys": 1,
      "subkeys": 0
    },
    "test_hardcoded_reset": {
      "_comment": "Several users (6) managed to do a sigchain reset and reuse their eldest key without an 'eldest' link, before we prohibited that on the server. This test uses the chain of one of those users, as a canary to make sure they're covered.",
      "input": "hardcoded_reset.json",
      "len": 1,
      "sibkeys": 1,
      "subkeys": 0
    },
    "test_v1_v2_stubbed_mix" : {
      "_comment" : "A chain some V1, some V2 links, and some are stubbed",
      "input" : "v1_v2_stubbed_mix.json",
      "len" : 4,
      "sibkeys" : 2,
      "subkeys" : 2
    },
    "test_v2_stubbed" : {
      "_comment" : "A chain with all V2 links, and some are stubbed",
      "input" : "v2_stubbed.json",
      "len" : 4,
      "sibkeys" : 2,
      "subkeys" : 2
    },
    "test_v2_simple" : {
      "_comment" : "A chain with all V2 links, but none are stubbed",
      "input" : "v2_simple.json",
      "len" : 4,
      "sibkeys" : 2,
      "subkeys" : 2
    },
    "test_bad_v2_delegate_stubbed" : {
      "_comment" : "A chain that has a key delegate stubbed (a no-no)",
      "input" : "bad_v2_delegate_stubbed.json",
      "err_type" : "SIGCHAIN_V2_STUBBED_SIGNATURE_NEEDED"
    },
    "test_bad_v2_first_stubbed" : {
      "_comment" : "A chain that has an eldest link stubbed (a no-no)",
      "input" : "bad_v2_first_stubbed.json",
      "err_type" : "SIGCHAIN_V2_STUBBED_FIRST_LINK"
    },
    "test_bad_v2_hash_mismatch" : {
      "_comment" : "A chain that has a link=2 wrong hash to body",
      "input" : "bad_v2_hash_mismatch.json",
      "err_type" : "SIGCHAIN_V2_MISMATCHED_FIELD"
    },
    "test_bad_v2_prev_mismatch" : {
      "_comment" : "A chain that has a link=2 wrong prev",
      "input" : "bad_v2_prev_mismatch.json",
      "err_type" : "SIGCHAIN_V2_MISMATCHED_FIELD"
    },
    "test_bad_v2_seqno_mismatch" : {
      "_comment" : "A chain that has a link=2 wrong seqno",
      "input" : "bad_v2_seqno_mismatch.json",
      "err_type" : "SIGCHAIN_V2_MISMATCHED_FIELD"
    },
    "test_bad_v2_type_mismatch" : {
      "_comment" : "A chain that has a link=2 type subkey v sibkey mismatch",
      "input" : "bad_v2_type_mismatch.json",
      "err_type" : "SIGCHAIN_V2_MISMATCHED_FIELD"
    },
    "test_v2_with_revokes" : {
      "_comment" : "A chain with all V2 links, and some revokes, 1 stubbed link",
      "input" : "v2_with_revokes.json",
      "len" : 5,
      "sibkeys" : 2
    },
    "test_bad_v2_stubbed_revoke_sig" : {
      "_comment" : "A chain with all V2 links, with an errant stubbed revoke sig",
      "input" : "bad_v2_stubbed_revoke_sig.json",
      "err_type" : "SIGCHAIN_V2_STUBBED_SIGNATURE_NEEDED"
    },
    "test_bad_v2_stubbed_revoke_sigs" : {
      "_comment" : "A chain with all V2 links, with an errant stubbed revoke sig",
      "input" : "bad_v2_stubbed_revoke_sigs.json",
      "err_type" : "SIGCHAIN_V2_STUBBED_SIGNATURE_NEEDED"
    },
    "test_bad_v2_revoke_type_mismatch" : {
      "_comment" : "A chain with all V2 links, with an errant stubbed revoke sig",
      "input" : "bad_v2_revoke_type_mismatch.json",
      "err_type" : "SIGCHAIN_V2_MISMATCHED_FIELD"
    },
    "test_bad_v2_wrong_payload_hash" : {
      "_comment" : "A chain with a payload_hash that doesn't match the has of the outer",
      "input" : "bad_v2_wrong_payload_hash.json",
      "err_type" : "SIGCHAIN_V2_MISMATCHED_HASH"
    },
    "00dani": {
      "_comment": "00dani's chain got broken after pgp key update with invalid usage flags.",
      "input": "00dani.json",
      "len": 55,
      "sibkeys": 4,
      "subkeys": 3
    },
    "goozbach": {
      "_comment": "goozbach's chain got broken after revoking and adding pgp key, wrong bundle was used to verify reverse_sig.",
      "input": "goozbach.json",
      "len": 14,
      "sibkeys": 3,
      "subkeys": 2
    },
    "reset_2x" : {
      "_comment": "a sigchain that was reset 2x and has 3 eldest links therefore",
      "input": "reset_2x.json",
      "len": 4,
      "sibkeys": 2,
      "subkeys": 2,
      "eldest" : "g",
      "eldest_seqno" : 11,
      "previous_subchains" : [
        {
          "eldest_seqno" : 1,
          "sibkeys" : 1,
          "subkeys" : 1
        },
        {
          "eldest_seqno" : 5,
          "sibkeys" : 3,
          "subkeys" : 3
        }
      ]
    },
    "test_per_user_key": {
      "_comment": "Test a simple per user key link.",
      "input": "per_user_key.json",
      "len": 2,
      "sibkeys": 1,
      "subkeys": 0
    },
    "test_bad_per_user_key_reverse_sig" : {
      "_comment": "a sigchain per user key addition with an corrupted reverse sig",
      "input": "bad_per_user_key_reverse_sig.json",
      "err_type": "WRONG_PER_USER_KEY_REVERSE_SIG"
    }
  }
}
`

var ChainTestInputs = map[string]string{
	"00dani.json": `{
  "uid": "fe1e0c96654603c5880b3f571c862919",
  "username": "00dani",
  "chain": [
    {
      "seqno": 1,
      "payload_hash": "df57a2df0975c592f83fbddaff7f5c819e91dc61de3c6984571677b1e0a01689",
      "sig_id": "f64a866c17aa2fb086947eba14239f9a83544806b7668da0223abe9bf2ea22300f",
      "sig_id_short": "9kqGbBeqL7CGlH66FCOfmoNUSAa3Zo2gIjq-",
      "kid": "0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a",
      "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgvO0RhtCQPaHyjQfACTCE+4HxBwKLJbRpFcPYRaUM15MKp3BheWxvYWTFAtF7ImJvZHkiOnsiZGV2aWNlIjp7ImlkIjoiNWNjNjc4YTcyMjI1NzFjZDk4NTBiNTQ3MTkyOTcyMTgiLCJraWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwibmFtZSI6IkRhbmkncyBNYWNib29rIEFpciIsInN0YXR1cyI6MSwidHlwZSI6ImRlc2t0b3AifSwia2V5Ijp7Imhvc3QiOiJrZXliYXNlLmlvIiwia2lkIjoiMDEyMGJjZWQxMTg2ZDA5MDNkYTFmMjhkMDdjMDA5MzA4NGZiODFmMTA3MDI4YjI1YjQ2OTE1YzNkODQ1YTUwY2Q3OTMwYSIsInVpZCI6ImZlMWUwYzk2NjU0NjAzYzU4ODBiM2Y1NzFjODYyOTE5IiwidXNlcm5hbWUiOiIwMGRhbmkifSwidHlwZSI6ImVsZGVzdCIsInZlcnNpb24iOjF9LCJjbGllbnQiOnsibmFtZSI6ImtleWJhc2UuaW8gZ28gY2xpZW50IiwidmVyc2lvbiI6IjEuMC4xNiJ9LCJjdGltZSI6MTQ2NzA3ODY4NSwiZXhwaXJlX2luIjo1MDQ1NzYwMDAsIm1lcmtsZV9yb290Ijp7ImN0aW1lIjoxNDY3MDc4MTM5LCJoYXNoIjoiMWQ1N2M3NzUxZmFiMDU2NWY2NzJmMDU5ODg4YTg1ZDg0ZTNlMmNkY2Y2ZTI2OTcwYzdmNWNkMzRjY2UyMTU3MDBjOGFlMTMwNDUwOTQ5ZTM1OThlOGU4Nzk5NjlmN2Y5NGE5NTM0ZjUxY2MwZDk0OWZmNmMxZmYxMWFlYmNhOWMiLCJzZXFubyI6NTAxNTkxfSwicHJldiI6bnVsbCwic2Vxbm8iOjEsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RAp26U8Q9X9+lkE1z0egZDqBs3iDjUg9F1cZWfSkwBgbMgiZ2ri9fONk9UEN7+Tiwac/YC1tkfyC9zv/WAMh+5A6hzaWdfdHlwZSCjdGFnzQICp3ZlcnNpb24B",
      "payload_json": "{\"body\":{\"device\":{\"id\":\"5cc678a7222571cd9850b54719297218\",\"kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"name\":\"Dani's Macbook Air\",\"status\":1,\"type\":\"desktop\"},\"key\":{\"host\":\"keybase.io\",\"kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"type\":\"eldest\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.16\"},\"ctime\":1467078685,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1467078139,\"hash\":\"1d57c7751fab0565f672f059888a85d84e3e2cdcf6e26970c7f5cd34cce215700c8ae130450949e3598e8e879969f7f94a9534f51cc0d949ff6c1ff11aebca9c\",\"seqno\":501591},\"prev\":null,\"seqno\":1,\"tag\":\"signature\"}",
      "sig_type": 1,
      "sig_version": 1,
      "ctime": 1467082285,
      "etime": 1971658285,
      "rtime": null,
      "sig_status": 0,
      "prev": null,
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 2,
      "payload_hash": "8ec6674537ccdb7fd22c8b2b555e79807e7dd07718ddad9b77eaa959fe74b7b9",
      "sig_id": "bd2ba14da159c9bdb1c9eece386ec1f8aba37fc9f2848c79052ee52055c6ddcd0f",
      "sig_id_short": "vSuhTaFZyb2xye7OOG7B-Kujf8nyhIx5BS7l",
      "kid": "0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a",
      "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgvO0RhtCQPaHyjQfACTCE+4HxBwKLJbRpFcPYRaUM15MKp3BheWxvYWTFA+h7ImJvZHkiOnsiZGV2aWNlIjp7ImlkIjoiNWNjNjc4YTcyMjI1NzFjZDk4NTBiNTQ3MTkyOTcyMTgiLCJraWQiOiIwMTIxNDc4NDMwNzFhMGE1OGU3MzhjMjYzZjc2ZmY4ZDVkNzhkNjVmYzdiNzE4YTg2ZTdiN2FkZmVmNmU1Njc4Y2QyZjBhIiwic3RhdHVzIjoxfSwia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwidWlkIjoiZmUxZTBjOTY2NTQ2MDNjNTg4MGIzZjU3MWM4NjI5MTkiLCJ1c2VybmFtZSI6IjAwZGFuaSJ9LCJzdWJrZXkiOnsia2lkIjoiMDEyMTQ3ODQzMDcxYTBhNThlNzM4YzI2M2Y3NmZmOGQ1ZDc4ZDY1ZmM3YjcxOGE4NmU3YjdhZGZlZjZlNTY3OGNkMmYwYSIsInBhcmVudF9raWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIn0sInR5cGUiOiJzdWJrZXkiLCJ2ZXJzaW9uIjoxfSwiY2xpZW50Ijp7Im5hbWUiOiJrZXliYXNlLmlvIGdvIGNsaWVudCIsInZlcnNpb24iOiIxLjAuMTYifSwiY3RpbWUiOjE0NjcwNzg2ODUsImV4cGlyZV9pbiI6NTA0NTc2MDAwLCJtZXJrbGVfcm9vdCI6eyJjdGltZSI6MTQ2NzA3ODEzOSwiaGFzaCI6IjFkNTdjNzc1MWZhYjA1NjVmNjcyZjA1OTg4OGE4NWQ4NGUzZTJjZGNmNmUyNjk3MGM3ZjVjZDM0Y2NlMjE1NzAwYzhhZTEzMDQ1MDk0OWUzNTk4ZThlODc5OTY5ZjdmOTRhOTUzNGY1MWNjMGQ5NDlmZjZjMWZmMTFhZWJjYTljIiwic2Vxbm8iOjUwMTU5MX0sInByZXYiOiJkZjU3YTJkZjA5NzVjNTkyZjgzZmJkZGFmZjdmNWM4MTllOTFkYzYxZGUzYzY5ODQ1NzE2NzdiMWUwYTAxNjg5Iiwic2Vxbm8iOjIsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RAahEKAwz9Ls+ahooouPFMHNjCgkAYD3j9cieBizsM9u378f5rxl52yWKrGb78VUWOE3/kHf/Tt1AO9TOUyNH+D6hzaWdfdHlwZSCjdGFnzQICp3ZlcnNpb24B",
      "payload_json": "{\"body\":{\"device\":{\"id\":\"5cc678a7222571cd9850b54719297218\",\"kid\":\"012147843071a0a58e738c263f76ff8d5d78d65fc7b718a86e7b7adfef6e5678cd2f0a\",\"status\":1},\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"subkey\":{\"kid\":\"012147843071a0a58e738c263f76ff8d5d78d65fc7b718a86e7b7adfef6e5678cd2f0a\",\"parent_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\"},\"type\":\"subkey\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.16\"},\"ctime\":1467078685,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1467078139,\"hash\":\"1d57c7751fab0565f672f059888a85d84e3e2cdcf6e26970c7f5cd34cce215700c8ae130450949e3598e8e879969f7f94a9534f51cc0d949ff6c1ff11aebca9c\",\"seqno\":501591},\"prev\":\"df57a2df0975c592f83fbddaff7f5c819e91dc61de3c6984571677b1e0a01689\",\"seqno\":2,\"tag\":\"signature\"}",
      "sig_type": 1,
      "sig_version": 1,
      "ctime": 1467082285,
      "etime": 1971658285,
      "rtime": null,
      "sig_status": 0,
      "prev": "df57a2df0975c592f83fbddaff7f5c819e91dc61de3c6984571677b1e0a01689",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 3,
      "payload_hash": "3d33e53a4e8ae524858ff57d60d19e513ad37f8d2ab075d731ce71f72e7923d4",
      "sig_id": "673be7b3ad4081de016cd94b733480dad3f2110d82343965378db46d32980f3b0f",
      "sig_id_short": "Zzvns61Agd4BbNlLczSA2tPyEQ2CNDllN420",
      "kid": "0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a",
      "sig": "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",
      "payload_json": "{\"body\":{\"device\":{\"id\":\"3db39ec911e292ee4701535dcdb67218\",\"kid\":\"01202f8f0998d6e9daddf0bd5c1295c0fa005f086b054b13be7b8a0668bf68d6f6cf0a\",\"name\":\"muscle neither\",\"status\":1,\"type\":\"backup\"},\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"sibkey\":{\"kid\":\"01202f8f0998d6e9daddf0bd5c1295c0fa005f086b054b13be7b8a0668bf68d6f6cf0a\",\"reverse_sig\":\"g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgL48JmNbp2t3wvVwSlcD6AF8IawVLE757igZov2jW9s8Kp3BheWxvYWTFA817ImJvZHkiOnsiZGV2aWNlIjp7ImlkIjoiM2RiMzllYzkxMWUyOTJlZTQ3MDE1MzVkY2RiNjcyMTgiLCJraWQiOiIwMTIwMmY4ZjA5OThkNmU5ZGFkZGYwYmQ1YzEyOTVjMGZhMDA1ZjA4NmIwNTRiMTNiZTdiOGEwNjY4YmY2OGQ2ZjZjZjBhIiwibmFtZSI6Im11c2NsZSBuZWl0aGVyIiwic3RhdHVzIjoxLCJ0eXBlIjoiYmFja3VwIn0sImtleSI6eyJlbGRlc3Rfa2lkIjoiMDEyMGJjZWQxMTg2ZDA5MDNkYTFmMjhkMDdjMDA5MzA4NGZiODFmMTA3MDI4YjI1YjQ2OTE1YzNkODQ1YTUwY2Q3OTMwYSIsImhvc3QiOiJrZXliYXNlLmlvIiwia2lkIjoiMDEyMGJjZWQxMTg2ZDA5MDNkYTFmMjhkMDdjMDA5MzA4NGZiODFmMTA3MDI4YjI1YjQ2OTE1YzNkODQ1YTUwY2Q3OTMwYSIsInVpZCI6ImZlMWUwYzk2NjU0NjAzYzU4ODBiM2Y1NzFjODYyOTE5IiwidXNlcm5hbWUiOiIwMGRhbmkifSwic2lia2V5Ijp7ImtpZCI6IjAxMjAyZjhmMDk5OGQ2ZTlkYWRkZjBiZDVjMTI5NWMwZmEwMDVmMDg2YjA1NGIxM2JlN2I4YTA2NjhiZjY4ZDZmNmNmMGEiLCJyZXZlcnNlX3NpZyI6bnVsbH0sInR5cGUiOiJzaWJrZXkiLCJ2ZXJzaW9uIjoxfSwiY2xpZW50Ijp7Im5hbWUiOiJrZXliYXNlLmlvIGdvIGNsaWVudCIsInZlcnNpb24iOiIxLjAuMTYifSwiY3RpbWUiOjE0NjcwNzg2ODYsImV4cGlyZV9pbiI6NTA0NTc2MDAwLCJtZXJrbGVfcm9vdCI6eyJjdGltZSI6MTQ2NzA3ODEzOSwiaGFzaCI6IjFkNTdjNzc1MWZhYjA1NjVmNjcyZjA1OTg4OGE4NWQ4NGUzZTJjZGNmNmUyNjk3MGM3ZjVjZDM0Y2NlMjE1NzAwYzhhZTEzMDQ1MDk0OWUzNTk4ZThlODc5OTY5ZjdmOTRhOTUzNGY1MWNjMGQ5NDlmZjZjMWZmMTFhZWJjYTljIiwic2Vxbm8iOjUwMTU5MX0sInByZXYiOiI4ZWM2Njc0NTM3Y2NkYjdmZDIyYzhiMmI1NTVlNzk4MDdlN2RkMDc3MThkZGFkOWI3N2VhYTk1OWZlNzRiN2I5Iiwic2Vxbm8iOjMsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RAErg3fGVetJDKQ/qrfdxkU/JCgrxMUHmo4loOA6zEwgeGlbR5JPquTlDleGHgFSQpOTZSKefERVY5p6vS/Ua2A6hzaWdfdHlwZSCjdGFnzQICp3ZlcnNpb24B\"},\"type\":\"sibkey\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.16\"},\"ctime\":1467078686,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1467078139,\"hash\":\"1d57c7751fab0565f672f059888a85d84e3e2cdcf6e26970c7f5cd34cce215700c8ae130450949e3598e8e879969f7f94a9534f51cc0d949ff6c1ff11aebca9c\",\"seqno\":501591},\"prev\":\"8ec6674537ccdb7fd22c8b2b555e79807e7dd07718ddad9b77eaa959fe74b7b9\",\"seqno\":3,\"tag\":\"signature\"}",
      "sig_type": 1,
      "sig_version": 1,
      "ctime": 1467082286,
      "etime": 1971658286,
      "rtime": null,
      "sig_status": 0,
      "prev": "8ec6674537ccdb7fd22c8b2b555e79807e7dd07718ddad9b77eaa959fe74b7b9",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 4,
      "payload_hash": "eb405b8008b60640c13435e877d982d5c88b2300de7be5504aef2ac6695921e8",
      "sig_id": "1009bf4aac80d81e7cd734392a158475fd97b75fac78bf11c840cc876ab051b10f",
      "sig_id_short": "EAm_SqyA2B581zQ5KhWEdf2Xt1-seL8RyEDM",
      "kid": "01202f8f0998d6e9daddf0bd5c1295c0fa005f086b054b13be7b8a0668bf68d6f6cf0a",
      "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgL48JmNbp2t3wvVwSlcD6AF8IawVLE757igZov2jW9s8Kp3BheWxvYWTFA+h7ImJvZHkiOnsiZGV2aWNlIjp7ImlkIjoiM2RiMzllYzkxMWUyOTJlZTQ3MDE1MzVkY2RiNjcyMTgiLCJraWQiOiIwMTIxMzgxNTFkNDViYmZkODA3ZDgxYTk0ZjkxOGI0OTgwZDlhZDJkMWU1YmRkNDRlZmY5ODAzZjJlMjZlMzc4ODM0MjBhIiwic3RhdHVzIjoxfSwia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwMmY4ZjA5OThkNmU5ZGFkZGYwYmQ1YzEyOTVjMGZhMDA1ZjA4NmIwNTRiMTNiZTdiOGEwNjY4YmY2OGQ2ZjZjZjBhIiwidWlkIjoiZmUxZTBjOTY2NTQ2MDNjNTg4MGIzZjU3MWM4NjI5MTkiLCJ1c2VybmFtZSI6IjAwZGFuaSJ9LCJzdWJrZXkiOnsia2lkIjoiMDEyMTM4MTUxZDQ1YmJmZDgwN2Q4MWE5NGY5MThiNDk4MGQ5YWQyZDFlNWJkZDQ0ZWZmOTgwM2YyZTI2ZTM3ODgzNDIwYSIsInBhcmVudF9raWQiOiIwMTIwMmY4ZjA5OThkNmU5ZGFkZGYwYmQ1YzEyOTVjMGZhMDA1ZjA4NmIwNTRiMTNiZTdiOGEwNjY4YmY2OGQ2ZjZjZjBhIn0sInR5cGUiOiJzdWJrZXkiLCJ2ZXJzaW9uIjoxfSwiY2xpZW50Ijp7Im5hbWUiOiJrZXliYXNlLmlvIGdvIGNsaWVudCIsInZlcnNpb24iOiIxLjAuMTYifSwiY3RpbWUiOjE0NjcwNzg2ODYsImV4cGlyZV9pbiI6NTA0NTc2MDAwLCJtZXJrbGVfcm9vdCI6eyJjdGltZSI6MTQ2NzA3ODEzOSwiaGFzaCI6IjFkNTdjNzc1MWZhYjA1NjVmNjcyZjA1OTg4OGE4NWQ4NGUzZTJjZGNmNmUyNjk3MGM3ZjVjZDM0Y2NlMjE1NzAwYzhhZTEzMDQ1MDk0OWUzNTk4ZThlODc5OTY5ZjdmOTRhOTUzNGY1MWNjMGQ5NDlmZjZjMWZmMTFhZWJjYTljIiwic2Vxbm8iOjUwMTU5MX0sInByZXYiOiIzZDMzZTUzYTRlOGFlNTI0ODU4ZmY1N2Q2MGQxOWU1MTNhZDM3ZjhkMmFiMDc1ZDczMWNlNzFmNzJlNzkyM2Q0Iiwic2Vxbm8iOjQsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RAwsjrKy5PsaV+QzoMVIzIZo83dqhO4V3mbFKaMbXAfSzDbhYo5AxRq7FFG1QujSHWiwwKjS8JeE8eDnsSJjwkDahzaWdfdHlwZSCjdGFnzQICp3ZlcnNpb24B",
      "payload_json": "{\"body\":{\"device\":{\"id\":\"3db39ec911e292ee4701535dcdb67218\",\"kid\":\"012138151d45bbfd807d81a94f918b4980d9ad2d1e5bdd44eff9803f2e26e37883420a\",\"status\":1},\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"01202f8f0998d6e9daddf0bd5c1295c0fa005f086b054b13be7b8a0668bf68d6f6cf0a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"subkey\":{\"kid\":\"012138151d45bbfd807d81a94f918b4980d9ad2d1e5bdd44eff9803f2e26e37883420a\",\"parent_kid\":\"01202f8f0998d6e9daddf0bd5c1295c0fa005f086b054b13be7b8a0668bf68d6f6cf0a\"},\"type\":\"subkey\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.16\"},\"ctime\":1467078686,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1467078139,\"hash\":\"1d57c7751fab0565f672f059888a85d84e3e2cdcf6e26970c7f5cd34cce215700c8ae130450949e3598e8e879969f7f94a9534f51cc0d949ff6c1ff11aebca9c\",\"seqno\":501591},\"prev\":\"3d33e53a4e8ae524858ff57d60d19e513ad37f8d2ab075d731ce71f72e7923d4\",\"seqno\":4,\"tag\":\"signature\"}",
      "sig_type": 1,
      "sig_version": 1,
      "ctime": 1467082286,
      "etime": 1971658286,
      "rtime": null,
      "sig_status": 0,
      "prev": "3d33e53a4e8ae524858ff57d60d19e513ad37f8d2ab075d731ce71f72e7923d4",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 5,
      "payload_hash": "7d57d318d6dd31f9cc43bb7345fd19091a455365dc214a2d4564d13b26681d1f",
      "sig_id": "d43b06a2b12764d8db81baedb0313fec533d9a5d31b5a2fcf840a604bae21dbe0f",
      "sig_id_short": "1DsGorEnZNjbgbrtsDE_7FM9ml0xtaL8-ECm",
      "kid": "0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a",
      "sig": "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",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"sibkey\":{\"fingerprint\":\"7fd74ecea6ac7c073e7f602938c5ddc6bda4fae3\",\"full_hash\":\"e3a17021226805004271ff1f9cdc23dffff592ca71ab1bf662cbde20293f462f\",\"key_id\":\"38C5DDC6BDA4FAE3\",\"kid\":\"0101482ecf3dce65743c1f48f1bb6bd1868b90e8e44c8b39752f1569d4dff150f4040a\",\"reverse_sig\":\"-----BEGIN PGP MESSAGE-----\\nComment: https://keybase.io/download\\nVersion: Keybase Go 1.0.16 (darwin)\\n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\\nRERIwHR0HKAtDXDk/bjM8wi4CjtzOK2alS7qQJXQMAHEuKDV8FPq7vdfvA==\\n=zM//\\n-----END PGP MESSAGE-----\\n\"},\"type\":\"sibkey\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.16\"},\"ctime\":1467078744,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1467078687,\"hash\":\"46fc707a3a862d063a3442377745dd61aaca08231c9c0a544d59fcdcaf458a8ffe5cca028774cd6b1c18fb89f8c08ce890c8a0d8f6200ca624a387f77c206a1f\",\"seqno\":501594},\"prev\":\"eb405b8008b60640c13435e877d982d5c88b2300de7be5504aef2ac6695921e8\",\"seqno\":5,\"tag\":\"signature\"}",
      "sig_type": 1,
      "sig_version": 1,
      "ctime": 1467082344,
      "etime": 1971658344,
      "rtime": null,
      "sig_status": 0,
      "prev": "eb405b8008b60640c13435e877d982d5c88b2300de7be5504aef2ac6695921e8",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 6,
      "payload_hash": "3faab1a0caa3035be6f7c4248f62c7be627f471ca093f411965f763116f9a54c",
      "sig_id": "f323ddde615e72024f89bcf21897033a6bb1da9983b942e13da5cca3054bf9660f",
      "sig_id_short": "8yPd3mFecgJPibzyGJcDOmux2pmDuULhPaXM",
      "kid": "0101482ecf3dce65743c1f48f1bb6bd1868b90e8e44c8b39752f1569d4dff150f4040a",
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.53\nComment: https://keybase.io/crypto\n\nyMNdAnicbVJtUFRVGN4VtNyGlQSD5GPsQhIEzDn3+25Docj0haGNhCa53HvPuXgD\ndpe7H4grU6g5EoSbDA7CZMSYwTTDhzJBEjIEZshkOGQypQXplJrMmjNCGGh3Gf1V\n5897znue53mf9513KCTIYDLypy9/03f07hXjyEC125BbMt7lJSQ7KiMsXqIQLwRc\nhLDTZS1UEWEhACSBJGMEIc8iIAAKiVAheQQ4GQCBAjytSDxUIOAAyUskI9GsABmZ\nQjzNiAyQEaeDRCKZUFRbAdYcmmpz6bKcgjgay1hkRVlX4ijMKSwgBYqXGYRkVkIi\nrYiY0onb7c4AQzcniU6cqtr1nP6wLtj7H/xD3wDSPIllhUIyZhmOpmSo0LwCJUlH\n693wkgAwj2la5iVK4BhSgQwrIBop+gUoNKAXfLsX5BQMMZAFlmVoFlAyw/NAohSG\ngzLPkgIUAkAn1mxiMQ4UB0i0qUR5MqHnPKqMA2N98OcqVV0urP2X4LEHCK4yRyBR\niiXrA65VUm1IH57O8GDNqdpthAXqSNmlBriQZjnA8TzFJhN4h0PVsFUNIBiO5YF+\nkgmHhj2BkSOGQxTkEYv0oAiyTFOSxFE0oyAoAAGKNMNQLINkEtIiiWiGpRGkJJJl\neYigQgS6KbHZCYteySUW6JJOtcAmutwaJsq/HsgLNhhNhiWLFwWWymBaGvpw1dpv\nL7ufXbm+Za2QfjzeuNWZ0BH925kbGdeMYK3xx5awb203AXur+LDFvyotLX9tUl/0\nntb48zssKUF/9Kdfdr90lFgRNfNE49Ca49PU5zF/+8ndyYc6Q33m7OEP+Kau1IHO\n5d+tGOuefUPLHU+NMte5L360zdsZMjdZfHfv4Nbnv2gI7n46x3j21ORY3fzrjxP+\nquZLE13LPMNnv7p/xtLY7xuov2JO+PiV+kcYyzvn2lgxxnigbXBJz6bKyYZ1nu4v\nN27Jac2qCJ/ogHdSIo6dCnsqaVXTrh/STY8Nzbb0Omu1VyNyExztz96e2RTnVuxF\ns5y5qHe7t8a17pnRcLc/osjbgIVf33K5ExBpCi1efm8kvX39iXN7F6Xxb685uS/r\nZmRHWGj52GdRU32V18mVBrGnwj9z5Pf0kKT6DY9WSd9nQkdkVe1Bn0GxeLKonk+k\nDc/l/xWX0tF2eKR6W0Fs3L3o3g9La1pHjXssP1X6Irf0ZA5MJIZ3Zo6fTBFfztH2\nX5jKrrYZ92/2YaR5hg9lrhTtjVPMNev0iaDF1WVVc/Ghb65Ou3Xj6oRybKPvH7i7\nsNV/4IgpCczP79y1ObGped/S8ZE7QsfOdzO8sc2lBit8jbd+mt8bW1HbNZk3Pbh6\n7lLq+f73X7zwQk198JPxUX+azVOjGb/kHax7by5uxHn955KgmMSLp68W/AvpFbYP\n=k+aC\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"fingerprint\":\"7fd74ecea6ac7c073e7f602938c5ddc6bda4fae3\",\"host\":\"keybase.io\",\"key_id\":\"38c5ddc6bda4fae3\",\"kid\":\"0101482ecf3dce65743c1f48f1bb6bd1868b90e8e44c8b39752f1569d4dff150f4040a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"service\":{\"name\":\"twitter\",\"username\":\"00davo\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1467078836,\"expire_in\":157680000,\"prev\":\"7d57d318d6dd31f9cc43bb7345fd19091a455365dc214a2d4564d13b26681d1f\",\"seqno\":6,\"tag\":\"signature\"}",
      "sig_type": 2,
      "sig_version": 1,
      "ctime": 1467082436,
      "etime": 1624762436,
      "rtime": 1470490104,
      "sig_status": 1,
      "prev": "7d57d318d6dd31f9cc43bb7345fd19091a455365dc214a2d4564d13b26681d1f",
      "proof_id": "95db2a8237d9a3b3022a7510",
      "proof_type": 2,
      "proof_text_check": "Verifying myself: I am 00dani on Keybase.io. 8yPd3mFecgJPibzyGJcDOmux2pmDuULhPaXM /",
      "proof_text_full": "Verifying myself: I am 00dani on Keybase.io. 8yPd3mFecgJPibzyGJcDOmux2pmDuULhPaXM / https://keybase.io/00dani/sigs/8yPd3mFecgJPibzyGJcDOmux2pmDuULhPaXM\n",
      "check_data_json": "{\"name\":\"twitter\",\"username\":\"00davo\"}",
      "remote_id": "747608994923962369",
      "api_url": "https://twitter.com/00davo/status/747608994923962369",
      "human_url": "https://twitter.com/00davo/status/747608994923962369",
      "proof_state": 7,
      "proof_status": 203,
      "retry_count": 1,
      "hard_fail_count": 40,
      "last_check": 1470489772,
      "last_success": 1470406137,
      "version": 3,
      "fingerprint": "7fd74ecea6ac7c073e7f602938c5ddc6bda4fae3"
    },
    {
      "seqno": 7,
      "payload_hash": "920ebda08248c6d7ef65629f6a606f7b1a5ab26a35fd18630dd5c90a43b89672",
      "sig_id": "8ae3aeaf98654cd6ffa751a8b7fa7675950238f71251f97086d54b50e6ab09720f",
      "sig_id_short": "iuOur5hlTNb_p1Got_p2dZUCOPcSUflwhtVL",
      "kid": "0101482ecf3dce65743c1f48f1bb6bd1868b90e8e44c8b39752f1569d4dff150f4040a",
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.53\nComment: https://keybase.io/crypto\n\nyMNeAnicdVJ7UFRVGN91XYunZTxGdHK95TQxROfsvffce5cZYBAQZ4gISaZkXO/j\n3PWOsrvsA+QhSWxMMzhUwJLAtMBoMFETgdBEiUTxCrJUDK0sBMcHOqPNRBQi0l1G\n/6vzzznfd36/7/t9v3OGQnSaQC07/NtI/4l7V7UTg1VuTU7+pckSQrBJRYSphDiA\nVzd8UMJOl/mAIhEmAkAjEEQsQcgiCXCAlHgoG1kJMCIAHAlYShZYKEPAACMrGGmB\nQhykRVJiKZqngSgxKognYghZsVqww+5QrC61LCNLDIVFzCNeVCsxJGZkBIwcyYq0\nJIlIkHhK5jGpEvfbnH6GKk7gnThWsak5NTCvyvsP/CPdAFKsEYsyKYkY0QxFilCm\nWBkKgopWp2EFDmAWU5TICiTH0EYZ0oiTKElWD0CmALWq271aTsYQA5FDiKYQIEWa\nZYFAyjQDRRYZOcj5gU7ssPJ52N8cSLxVIQ7HEGquQBGx39aHdxbFtd8t/B/eVWT3\nJwqxYH5INQuKVVK9UxkF2OFUbFbCBFWk6FL8XEghBjAsi5gYAh+yKw5sVvwImkEs\nUFcMYXfgAr9TMs8LkAciz5OApAWMZEakjKojyCgyamhkZEqdh1dfVaYg5BAtM4iE\nEMkcT1Mi4R8m32ojTGonF29RSzoVi5V3uR2YOPzNYO5ajTZQs06/xv+nNIEBTzz6\naSb7+gdbTS2l+jO34umblzOLwmeZ5PCeg8fqR7Kvf350rNhn/j7j5mTiXNjgQjeu\n+0JbEt9qntp115UWFe5t0NFbkqpqR+OuFPqeWekti6tabh2O8zjNiyMvoOjdeecz\ngtKZDbG2am1OQeLZ0PiYYzMz164Rc2GO2/qg02faXQlTa7b8RX3dU7E2ob0jq2oe\nv2Go2Qk9eYvu8Pr7nuPfTbQV5ra+k6fPWMKViWLU1N4jgSEtQyHTbTee3PtZ6pW/\nPc9xDyLfzHc3fhow/OFgwLdL/cd1y5Hj2rGBc0e53vSzqOJtgq/r+j2h6W5lWMtb\ngZdoT1985/zEa6fmUl93OotDvYbZPzuNup7R+acvbnh8IH025I9fek8YTtfNVI/t\nuKDZ/NJI2p3LG4N/Cq72+pKlRKmBTIvLj+bHn41EUbqnBtj2sOVaYNrT8djCnmD3\n+YCdwiI7b1HmdEmafR/UNL8KC64a2lZ03tL2CG9xfAoaSM6znGw+lfRlR82Lm28w\nY3fKst/X7PZ8TI7PHCnry537Z7ur25fZrfnx+o7s0Jye5JqvDA1ZP2yt7KvqGlze\n5C2NaNTdurjpZTz6vE9f5+vqHypfaq67rddXltPr719ozNz+yWTnQsV8aGpbU+3P\nH917pT73UL4hKGtbxHJR7DrHdMdS+fQuffTGlBXLuU77r++9m7JvdFvTyX8BSTu8\nWg==\n=T7gj\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"fingerprint\":\"7fd74ecea6ac7c073e7f602938c5ddc6bda4fae3\",\"host\":\"keybase.io\",\"key_id\":\"38c5ddc6bda4fae3\",\"kid\":\"0101482ecf3dce65743c1f48f1bb6bd1868b90e8e44c8b39752f1569d4dff150f4040a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"service\":{\"name\":\"github\",\"username\":\"00dani\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1467078867,\"expire_in\":157680000,\"prev\":\"3faab1a0caa3035be6f7c4248f62c7be627f471ca093f411965f763116f9a54c\",\"seqno\":7,\"tag\":\"signature\"}",
      "sig_type": 2,
      "sig_version": 1,
      "ctime": 1467082467,
      "etime": 1624762467,
      "rtime": null,
      "sig_status": 0,
      "prev": "3faab1a0caa3035be6f7c4248f62c7be627f471ca093f411965f763116f9a54c",
      "proof_id": "5f182e52082ad0fbe6146d10",
      "proof_type": 3,
      "proof_text_check": "\n\nyMNeAnicdVJ7UFRVGN91XYunZTxGdHK95TQxROfsvffce5cZYBAQZ4gISaZkXO/j\n3PWOsrvsA+QhSWxMMzhUwJLAtMBoMFETgdBEiUTxCrJUDK0sBMcHOqPNRBQi0l1G\n/6vzzznfd36/7/t9v3OGQnSaQC07/NtI/4l7V7UTg1VuTU7+pckSQrBJRYSphDiA\nVzd8UMJOl/mAIhEmAkAjEEQsQcgiCXCAlHgoG1kJMCIAHAlYShZYKEPAACMrGGmB\nQhykRVJiKZqngSgxKognYghZsVqww+5QrC61LCNLDIVFzCNeVCsxJGZkBIwcyYq0\nJIlIkHhK5jGpEvfbnH6GKk7gnThWsak5NTCvyvsP/CPdAFKsEYsyKYkY0QxFilCm\nWBkKgopWp2EFDmAWU5TICiTH0EYZ0oiTKElWD0CmALWq271aTsYQA5FDiKYQIEWa\nZYFAyjQDRRYZOcj5gU7ssPJ52N8cSLxVIQ7HEGquQBGx39aHdxbFtd8t/B/eVWT3\nJwqxYH5INQuKVVK9UxkF2OFUbFbCBFWk6FL8XEghBjAsi5gYAh+yKw5sVvwImkEs\nUFcMYXfgAr9TMs8LkAciz5OApAWMZEakjKojyCgyamhkZEqdh1dfVaYg5BAtM4iE\nEMkcT1Mi4R8m32ojTGonF29RSzoVi5V3uR2YOPzNYO5ajTZQs06/xv+nNIEBTzz6\naSb7+gdbTS2l+jO34umblzOLwmeZ5PCeg8fqR7Kvf350rNhn/j7j5mTiXNjgQjeu\n+0JbEt9qntp115UWFe5t0NFbkqpqR+OuFPqeWekti6tabh2O8zjNiyMvoOjdeecz\ngtKZDbG2am1OQeLZ0PiYYzMz164Rc2GO2/qg02faXQlTa7b8RX3dU7E2ob0jq2oe\nv2Go2Qk9eYvu8Pr7nuPfTbQV5ra+k6fPWMKViWLU1N4jgSEtQyHTbTee3PtZ6pW/\nPc9xDyLfzHc3fhow/OFgwLdL/cd1y5Hj2rGBc0e53vSzqOJtgq/r+j2h6W5lWMtb\ngZdoT1985/zEa6fmUl93OotDvYbZPzuNup7R+acvbnh8IH025I9fek8YTtfNVI/t\nuKDZ/NJI2p3LG4N/Cq72+pKlRKmBTIvLj+bHn41EUbqnBtj2sOVaYNrT8djCnmD3\n+YCdwiI7b1HmdEmafR/UNL8KC64a2lZ03tL2CG9xfAoaSM6znGw+lfRlR82Lm28w\nY3fKst/X7PZ8TI7PHCnry537Z7ur25fZrfnx+o7s0Jye5JqvDA1ZP2yt7KvqGlze\n5C2NaNTdurjpZTz6vE9f5+vqHypfaq67rddXltPr719ozNz+yWTnQsV8aGpbU+3P\nH917pT73UL4hKGtbxHJR7DrHdMdS+fQuffTGlBXLuU77r++9m7JvdFvTyX8BSTu8\nWg==\n",
      "proof_text_full": "### Keybase proof\n\nI hereby claim:\n\n  * I am 00dani on github.\n  * I am 00dani (https://keybase.io/00dani) on keybase.\n  * I have a public key whose fingerprint is 7FD7 4ECE A6AC 7C07 3E7F  6029 38C5 DDC6 BDA4 FAE3\n\nTo claim this, I am signing this object:\n\n`+"`"+``+"`"+``+"`"+`json\n{\n    \"body\": {\n        \"key\": {\n            \"eldest_kid\": \"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\n            \"fingerprint\": \"7fd74ecea6ac7c073e7f602938c5ddc6bda4fae3\",\n            \"host\": \"keybase.io\",\n            \"key_id\": \"38c5ddc6bda4fae3\",\n            \"kid\": \"0101482ecf3dce65743c1f48f1bb6bd1868b90e8e44c8b39752f1569d4dff150f4040a\",\n            \"uid\": \"fe1e0c96654603c5880b3f571c862919\",\n            \"username\": \"00dani\"\n        },\n        \"service\": {\n            \"name\": \"github\",\n            \"username\": \"00dani\"\n        },\n        \"type\": \"web_service_binding\",\n        \"version\": 1\n    },\n    \"ctime\": 1467078867,\n    \"expire_in\": 157680000,\n    \"prev\": \"3faab1a0caa3035be6f7c4248f62c7be627f471ca093f411965f763116f9a54c\",\n    \"seqno\": 7,\n    \"tag\": \"signature\"\n}\n`+"`"+``+"`"+``+"`"+`\n\nwith the key [7FD7 4ECE A6AC 7C07 3E7F  6029 38C5 DDC6 BDA4 FAE3](https://keybase.io/00dani), yielding the signature:\n\n`+"`"+``+"`"+``+"`"+`\n-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.53\nComment: https://keybase.io/crypto\n\nyMNeAnicdVJ7UFRVGN91XYunZTxGdHK95TQxROfsvffce5cZYBAQZ4gISaZkXO/j\n3PWOsrvsA+QhSWxMMzhUwJLAtMBoMFETgdBEiUTxCrJUDK0sBMcHOqPNRBQi0l1G\n/6vzzznfd36/7/t9v3OGQnSaQC07/NtI/4l7V7UTg1VuTU7+pckSQrBJRYSphDiA\nVzd8UMJOl/mAIhEmAkAjEEQsQcgiCXCAlHgoG1kJMCIAHAlYShZYKEPAACMrGGmB\nQhykRVJiKZqngSgxKognYghZsVqww+5QrC61LCNLDIVFzCNeVCsxJGZkBIwcyYq0\nJIlIkHhK5jGpEvfbnH6GKk7gnThWsak5NTCvyvsP/CPdAFKsEYsyKYkY0QxFilCm\nWBkKgopWp2EFDmAWU5TICiTH0EYZ0oiTKElWD0CmALWq271aTsYQA5FDiKYQIEWa\nZYFAyjQDRRYZOcj5gU7ssPJ52N8cSLxVIQ7HEGquQBGx39aHdxbFtd8t/B/eVWT3\nJwqxYH5INQuKVVK9UxkF2OFUbFbCBFWk6FL8XEghBjAsi5gYAh+yKw5sVvwImkEs\nUFcMYXfgAr9TMs8LkAciz5OApAWMZEakjKojyCgyamhkZEqdh1dfVaYg5BAtM4iE\nEMkcT1Mi4R8m32ojTGonF29RSzoVi5V3uR2YOPzNYO5ajTZQs06/xv+nNIEBTzz6\naSb7+gdbTS2l+jO34umblzOLwmeZ5PCeg8fqR7Kvf350rNhn/j7j5mTiXNjgQjeu\n+0JbEt9qntp115UWFe5t0NFbkqpqR+OuFPqeWekti6tabh2O8zjNiyMvoOjdeecz\ngtKZDbG2am1OQeLZ0PiYYzMz164Rc2GO2/qg02faXQlTa7b8RX3dU7E2ob0jq2oe\nv2Go2Qk9eYvu8Pr7nuPfTbQV5ra+k6fPWMKViWLU1N4jgSEtQyHTbTee3PtZ6pW/\nPc9xDyLfzHc3fhow/OFgwLdL/cd1y5Hj2rGBc0e53vSzqOJtgq/r+j2h6W5lWMtb\ngZdoT1985/zEa6fmUl93OotDvYbZPzuNup7R+acvbnh8IH025I9fek8YTtfNVI/t\nuKDZ/NJI2p3LG4N/Cq72+pKlRKmBTIvLj+bHn41EUbqnBtj2sOVaYNrT8djCnmD3\n+YCdwiI7b1HmdEmafR/UNL8KC64a2lZ03tL2CG9xfAoaSM6znGw+lfRlR82Lm28w\nY3fKst/X7PZ8TI7PHCnry537Z7ur25fZrfnx+o7s0Jye5JqvDA1ZP2yt7KvqGlze\n5C2NaNTdurjpZTz6vE9f5+vqHypfaq67rddXltPr719ozNz+yWTnQsV8aGpbU+3P\nH917pT73UL4hKGtbxHJR7DrHdMdS+fQuffTGlBXLuU77r++9m7JvdFvTyX8BSTu8\nWg==\n=T7gj\n-----END PGP MESSAGE-----\n\n`+"`"+``+"`"+``+"`"+`\n\nAnd finally, I am proving ownership of the github account by posting this as a gist.\n\n### My publicly-auditable identity:\n\nhttps://keybase.io/00dani\n\n### From the command line:\n\nConsider the [keybase command line program](https://keybase.io/download).\n\n`+"`"+``+"`"+``+"`"+`bash\n# look me up\nkeybase id 00dani\n`+"`"+``+"`"+``+"`"+`\n",
      "check_data_json": "{\"name\":\"github\",\"username\":\"00dani\"}",
      "remote_id": "243631fa5e7e23a363ef",
      "api_url": "https://gist.githubusercontent.com/00dani/243631fa5e7e23a363ef/raw/79862b9b4a5e39fad25f57bc7a0e2fbdbf96054c/keybase.md",
      "human_url": "https://gist.github.com/243631fa5e7e23a363ef",
      "proof_state": 1,
      "proof_status": 1,
      "retry_count": 236,
      "hard_fail_count": 0,
      "last_check": 1491707053,
      "last_success": 1491707053,
      "version": 1,
      "fingerprint": "7fd74ecea6ac7c073e7f602938c5ddc6bda4fae3"
    },
    {
      "seqno": 8,
      "payload_hash": "860fa0d6f2a37e0c8563b6f3e88872bfbe585e65c690edc3cf7c2c2a0afe350b",
      "sig_id": "a5d8748767595da0258cf483e1c573a1f8210f27ffd17e941f68883b3aef95c60f",
      "sig_id_short": "pdh0h2dZXaAljPSD4cVzofghDyf_0X6UH2iI",
      "kid": "0101482ecf3dce65743c1f48f1bb6bd1868b90e8e44c8b39752f1569d4dff150f4040a",
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.53\nComment: https://keybase.io/crypto\n\nyMNcAnicbVJ7UFVFHIaL0MQMGGVMjmXM4aagN9zz2rOHJlNGkkfC0DQV2HQ55+ye\nyxG7Fy4XEBjMgIyHDQiEMCoiEBpJD50GejkDCnKJgkgDSkYJRh4NJS8ZaQY6l9G/\nav/Z3d9+37ff75vfVR8PN2931DnS9X3j8pj7D+0l6W5vpA775lCyDWdRoTlUMlnb\nyEFM0hzmZA1ToRSgGSArBNM0ghiIgMUSrTIIA0EBQGQB4lQZ0SoNBMAgmeFlDoo0\nr7AYcbzEAwULOkiiTJSqWS3EnmLXrA5dVlCxwBGFSFBSdCWBJYIKASOySOExVqCM\nJU6VCKsTk2xpLoZuTpbSSIhm02v6xbxm73/wD30DmkMMUVQWKwTyAscqtMohlZZl\nHa13g2QREEQ4TkEyKwo8o9I8FDGHVf0AVA5wa77T1+RUQhOgiBDyHASswiMEZFbl\nBVpBkBFp0QVMI3ar9A5xfQ6wZNWoXBOl1zI0hbhiffBmJxhrjv/iM2wuvCMrxVXI\nJLL5AdUsa1asZ6czMog9TbNZqVBaRyoOzcWlOSgAAYkMNFHkUIpmJ2bNheAFiIC+\nTFSKnWTokiIDiJ4SQAyHFIgFokJet65CCQKoCjIt8ZLMQInlVVc6LMCYV0QgcayM\nRCgwlKuZVKuNCkW6TcmiS6ZpFqvkSLcTKrej/a11bu7ebl6eBtdMuXk/+tjDSSP5\n61ejH68v6fzy5M7sWZ9O+MvErpk9d6pGttcNrWzZHHHC6YkNC80pTXftie8tDL9y\n48VRQ+MFhtqfd/GKIb7EUOhoKD413/rCzrm8Mx3LW43zz+QNFP9uMoy3hW7MDEpY\namK8YOHtNq5uMQCH/ZU/VLp9728VFTPTm06TrqdGt0X3na+tEaqjrn9QsOF2/CPT\nR5/z8PjuZouhX3Z2DxUe9zdYz0ydTQwYSZ982j+Mu/QVNN/q6zr58Y9/lH8oXZqQ\nvQ7NdaPcj57I6uuNKdtdGnklKTUzOtHXWRvWUpw9NnC2v+q68fymi4bemGHLcnTS\nGLmTm9hclRrXWnT/6EvnGmIuG7XjA7Ev/xmx4txYvTsh+4tPnrUaP206eDrqWGRR\nIK7LLrwfWEm1HjkX7hl++NUp0W3h86prvKmjZf21Gee9Xcb6SmO4he/9ORAM1j7/\nPsli/D7zGa+Zr3kzvmk28eaFmOTY0o6+DdH7AqvHR/bXv1u07VRZxdebp9JLA9rm\ngovzkYcxhDJpPYuOwe6JwrnVsP6FmbgDV5fGgsqH2MhvV2dju/29Isdet93qH/rJ\nw1mdFW/ZGlx7ArdHNUz/M7h6eRTlBFvKCvzSe2L2spANuSfcmH7SWfra5GTFFrrl\ncNy+niNDSxPNEzXmrF+DVt5ebNthLWj0XQf87jr/PpCgln9TGRSRv2PPv8nFyUU=\n=LIHW\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"fingerprint\":\"7fd74ecea6ac7c073e7f602938c5ddc6bda4fae3\",\"host\":\"keybase.io\",\"key_id\":\"38c5ddc6bda4fae3\",\"kid\":\"0101482ecf3dce65743c1f48f1bb6bd1868b90e8e44c8b39752f1569d4dff150f4040a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"service\":{\"name\":\"reddit\",\"username\":\"00davo\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1467078926,\"expire_in\":157680000,\"prev\":\"920ebda08248c6d7ef65629f6a606f7b1a5ab26a35fd18630dd5c90a43b89672\",\"seqno\":8,\"tag\":\"signature\"}",
      "sig_type": 2,
      "sig_version": 1,
      "ctime": 1467082526,
      "etime": 1624762526,
      "rtime": null,
      "sig_status": 0,
      "prev": "920ebda08248c6d7ef65629f6a606f7b1a5ab26a35fd18630dd5c90a43b89672",
      "proof_id": "ff6a713d7a9d3b29b16ca510",
      "proof_type": 4,
      "proof_text_check": "\n\nyMNcAnicbVJ7UFVFHIaL0MQMGGVMjmXM4aagN9zz2rOHJlNGkkfC0DQV2HQ55+ye\nyxG7Fy4XEBjMgIyHDQiEMCoiEBpJD50GejkDCnKJgkgDSkYJRh4NJS8ZaQY6l9G/\nav/Z3d9+37ff75vfVR8PN2931DnS9X3j8pj7D+0l6W5vpA775lCyDWdRoTlUMlnb\nyEFM0hzmZA1ToRSgGSArBNM0ghiIgMUSrTIIA0EBQGQB4lQZ0SoNBMAgmeFlDoo0\nr7AYcbzEAwULOkiiTJSqWS3EnmLXrA5dVlCxwBGFSFBSdCWBJYIKASOySOExVqCM\nJU6VCKsTk2xpLoZuTpbSSIhm02v6xbxm73/wD30DmkMMUVQWKwTyAscqtMohlZZl\nHa13g2QREEQ4TkEyKwo8o9I8FDGHVf0AVA5wa77T1+RUQhOgiBDyHASswiMEZFbl\nBVpBkBFp0QVMI3ar9A5xfQ6wZNWoXBOl1zI0hbhiffBmJxhrjv/iM2wuvCMrxVXI\nJLL5AdUsa1asZ6czMog9TbNZqVBaRyoOzcWlOSgAAYkMNFHkUIpmJ2bNheAFiIC+\nTFSKnWTokiIDiJ4SQAyHFIgFokJet65CCQKoCjIt8ZLMQInlVVc6LMCYV0QgcayM\nRCgwlKuZVKuNCkW6TcmiS6ZpFqvkSLcTKrej/a11bu7ebl6eBtdMuXk/+tjDSSP5\n61ejH68v6fzy5M7sWZ9O+MvErpk9d6pGttcNrWzZHHHC6YkNC80pTXftie8tDL9y\n48VRQ+MFhtqfd/GKIb7EUOhoKD413/rCzrm8Mx3LW43zz+QNFP9uMoy3hW7MDEpY\namK8YOHtNq5uMQCH/ZU/VLp9728VFTPTm06TrqdGt0X3na+tEaqjrn9QsOF2/CPT\nR5/z8PjuZouhX3Z2DxUe9zdYz0ydTQwYSZ982j+Mu/QVNN/q6zr58Y9/lH8oXZqQ\nvQ7NdaPcj57I6uuNKdtdGnklKTUzOtHXWRvWUpw9NnC2v+q68fymi4bemGHLcnTS\nGLmTm9hclRrXWnT/6EvnGmIuG7XjA7Ev/xmx4txYvTsh+4tPnrUaP206eDrqWGRR\nIK7LLrwfWEm1HjkX7hl++NUp0W3h86prvKmjZf21Gee9Xcb6SmO4he/9ORAM1j7/\nPsli/D7zGa+Zr3kzvmk28eaFmOTY0o6+DdH7AqvHR/bXv1u07VRZxdebp9JLA9rm\ngovzkYcxhDJpPYuOwe6JwrnVsP6FmbgDV5fGgsqH2MhvV2dju/29Isdet93qH/rJ\nw1mdFW/ZGlx7ArdHNUz/M7h6eRTlBFvKCvzSe2L2spANuSfcmH7SWfra5GTFFrrl\ncNy+niNDSxPNEzXmrF+DVt5ebNthLWj0XQf87jr/PpCgln9TGRSRv2PPv8nFyUU=\n",
      "proof_text_full": "### Keybase proof\n\nI hereby claim:\n\n  * I am [00davo](https://www.reddit.com/user/00davo) on reddit.\n  * I am [00dani](https://keybase.io/00dani) on keybase.\n  * I have a public key whose fingerprint is 7FD7 4ECE A6AC 7C07 3E7F  6029 38C5 DDC6 BDA4 FAE3\n\nTo claim this, I am signing this object:\n\n\n    {\n        \"body\": {\n            \"key\": {\n                \"eldest_kid\": \"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\n                \"fingerprint\": \"7fd74ecea6ac7c073e7f602938c5ddc6bda4fae3\",\n                \"host\": \"keybase.io\",\n                \"key_id\": \"38c5ddc6bda4fae3\",\n                \"kid\": \"0101482ecf3dce65743c1f48f1bb6bd1868b90e8e44c8b39752f1569d4dff150f4040a\",\n                \"uid\": \"fe1e0c96654603c5880b3f571c862919\",\n                \"username\": \"00dani\"\n            },\n            \"service\": {\n                \"name\": \"reddit\",\n                \"username\": \"00davo\"\n            },\n            \"type\": \"web_service_binding\",\n            \"version\": 1\n        },\n        \"ctime\": 1467078926,\n        \"expire_in\": 157680000,\n        \"prev\": \"920ebda08248c6d7ef65629f6a606f7b1a5ab26a35fd18630dd5c90a43b89672\",\n        \"seqno\": 8,\n        \"tag\": \"signature\"\n    }\n\nwith the key from above, yielding:\n\n\n    -----BEGIN PGP MESSAGE-----\n    Version: Keybase OpenPGP v2.0.53\n    Comment: https://keybase.io/crypto\n    \n    yMNcAnicbVJ7UFVFHIaL0MQMGGVMjmXM4aagN9zz2rOHJlNGkkfC0DQV2HQ55+ye\n    yxG7Fy4XEBjMgIyHDQiEMCoiEBpJD50GejkDCnKJgkgDSkYJRh4NJS8ZaQY6l9G/\n    av/Z3d9+37ff75vfVR8PN2931DnS9X3j8pj7D+0l6W5vpA775lCyDWdRoTlUMlnb\n    yEFM0hzmZA1ToRSgGSArBNM0ghiIgMUSrTIIA0EBQGQB4lQZ0SoNBMAgmeFlDoo0\n    r7AYcbzEAwULOkiiTJSqWS3EnmLXrA5dVlCxwBGFSFBSdCWBJYIKASOySOExVqCM\n    JU6VCKsTk2xpLoZuTpbSSIhm02v6xbxm73/wD30DmkMMUVQWKwTyAscqtMohlZZl\n    Ha13g2QREEQ4TkEyKwo8o9I8FDGHVf0AVA5wa77T1+RUQhOgiBDyHASswiMEZFbl\n    BVpBkBFp0QVMI3ar9A5xfQ6wZNWoXBOl1zI0hbhiffBmJxhrjv/iM2wuvCMrxVXI\n    JLL5AdUsa1asZ6czMog9TbNZqVBaRyoOzcWlOSgAAYkMNFHkUIpmJ2bNheAFiIC+\n    TFSKnWTokiIDiJ4SQAyHFIgFokJet65CCQKoCjIt8ZLMQInlVVc6LMCYV0QgcayM\n    RCgwlKuZVKuNCkW6TcmiS6ZpFqvkSLcTKrej/a11bu7ebl6eBtdMuXk/+tjDSSP5\n    61ejH68v6fzy5M7sWZ9O+MvErpk9d6pGttcNrWzZHHHC6YkNC80pTXftie8tDL9y\n    48VRQ+MFhtqfd/GKIb7EUOhoKD413/rCzrm8Mx3LW43zz+QNFP9uMoy3hW7MDEpY\n    amK8YOHtNq5uMQCH/ZU/VLp9728VFTPTm06TrqdGt0X3na+tEaqjrn9QsOF2/CPT\n    R5/z8PjuZouhX3Z2DxUe9zdYz0ydTQwYSZ982j+Mu/QVNN/q6zr58Y9/lH8oXZqQ\n    vQ7NdaPcj57I6uuNKdtdGnklKTUzOtHXWRvWUpw9NnC2v+q68fymi4bemGHLcnTS\n    GLmTm9hclRrXWnT/6EvnGmIuG7XjA7Ev/xmx4txYvTsh+4tPnrUaP206eDrqWGRR\n    IK7LLrwfWEm1HjkX7hl++NUp0W3h86prvKmjZf21Gee9Xcb6SmO4he/9ORAM1j7/\n    Psli/D7zGa+Zr3kzvmk28eaFmOTY0o6+DdH7AqvHR/bXv1u07VRZxdebp9JLA9rm\n    govzkYcxhDJpPYuOwe6JwrnVsP6FmbgDV5fGgsqH2MhvV2dju/29Isdet93qH/rJ\n    w1mdFW/ZGlx7ArdHNUz/M7h6eRTlBFvKCvzSe2L2spANuSfcmH7SWfra5GTFFrrl\n    cNy+niNDSxPNEzXmrF+DVt5ebNthLWj0XQf87jr/PpCgln9TGRSRv2PPv8nFyUU=\n    =LIHW\n    -----END PGP MESSAGE-----\n\nFinally, I am proving my reddit account by posting it in /r/KeybaseProofs\n",
      "check_data_json": "{\"name\":\"reddit\",\"username\":\"00davo\"}",
      "remote_id": "t3_4q6zzs",
      "api_url": "https://www.reddit.com/r/KeybaseProofs/comments/4q6zzs/my_keybase_proof_reddit00davo_keybase00dani/.json",
      "human_url": "https://www.reddit.com/r/KeybaseProofs/comments/4q6zzs/my_keybase_proof_reddit00davo_keybase00dani/",
      "proof_state": 5,
      "proof_status": 1,
      "retry_count": 169,
      "hard_fail_count": 0,
      "last_check": 1490220935,
      "last_success": 1490220935,
      "version": 7,
      "fingerprint": "7fd74ecea6ac7c073e7f602938c5ddc6bda4fae3"
    },
    {
      "seqno": 9,
      "payload_hash": "efcff6422032de5b92f0d0025308b309b681d03c92911ed982d061252ede578b",
      "sig_id": "454a0469a7c6086673c320a77208e899b72f90f7fb4fc21a7367c06951fc38960f",
      "sig_id_short": "RUoEaafGCGZzwyCncgjombcvkPf7T8Iac2fA",
      "kid": "0101482ecf3dce65743c1f48f1bb6bd1868b90e8e44c8b39752f1569d4dff150f4040a",
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.53\nComment: https://keybase.io/crypto\n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\n=oqey\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"f10c6c4525ffbcb321eed85c16e0c874a157a0e7\"},\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"fingerprint\":\"7fd74ecea6ac7c073e7f602938c5ddc6bda4fae3\",\"host\":\"keybase.io\",\"key_id\":\"38c5ddc6bda4fae3\",\"kid\":\"0101482ecf3dce65743c1f48f1bb6bd1868b90e8e44c8b39752f1569d4dff150f4040a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"track\":{\"basics\":{\"id_version\":15,\"last_id_change\":1442256800,\"username\":\"benjbrent\"},\"id\":\"89a66ba1faf6bbda845b84d48129a500\",\"key\":{\"key_fingerprint\":\"2827eab057a6789be7d6a4444025bad742f9af6f\",\"kid\":\"0101ffe822cfd1de4871891194812756e5f7535bce4cbf8f88e6b627a89c8a3dd4640a\"},\"pgp_keys\":[{\"key_fingerprint\":\"2827eab057a6789be7d6a4444025bad742f9af6f\",\"kid\":\"0101ffe822cfd1de4871891194812756e5f7535bce4cbf8f88e6b627a89c8a3dd4640a\"}],\"remote_proofs\":[{\"ctime\":1415763104,\"curr\":\"4860bba658be4cf9f7ccf64c13983ea7879edfd8b95702ab47c4a13d2df178a3\",\"etime\":1573443104,\"prev\":\"dbf02861cabe2b962526ffcbe36f5b03993fbd36465430993e97adbc4fc48f10\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"benjbrent\"},\"proof_type\":2,\"state\":1},\"seqno\":3,\"sig_id\":\"d3310b1d7b82fce5902754d27638a8148494404204912dc89e5fa09144c031330f\",\"sig_type\":2},{\"ctime\":1415763180,\"curr\":\"9837b128efe58a6b2a64f385c9c73f02f3fbb02f882dc1272b15f8c6f3c98448\",\"etime\":1573443180,\"prev\":\"4860bba658be4cf9f7ccf64c13983ea7879edfd8b95702ab47c4a13d2df178a3\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"benjaminbrent\"},\"proof_type\":3,\"state\":1},\"seqno\":4,\"sig_id\":\"5ac4825925afabfc8e7c7f9b4922c97b0f69b3768f379cf927947bcf7c6f74aa0f\",\"sig_type\":2},{\"ctime\":1415763308,\"curr\":\"693c841d22eb71a628d11354e20162fbe7bf55f536f45bb4c52ff747b0995d9e\",\"etime\":1573443308,\"prev\":\"9837b128efe58a6b2a64f385c9c73f02f3fbb02f882dc1272b15f8c6f3c98448\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"benjaminbrent.com\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":5,\"sig_id\":\"ba3843a206a79022ea7408ca5e7fd8f1a1ccef5052fee303a633e760e4c8ae920f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"693c841d22eb71a628d11354e20162fbe7bf55f536f45bb4c52ff747b0995d9e\",\"seqno\":5,\"sig_id\":\"ba3843a206a79022ea7408ca5e7fd8f1a1ccef5052fee303a633e760e4c8ae920f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1467079242,\"expire_in\":157680000,\"prev\":\"860fa0d6f2a37e0c8563b6f3e88872bfbe585e65c690edc3cf7c2c2a0afe350b\",\"seqno\":9,\"tag\":\"signature\"}",
      "sig_type": 3,
      "sig_version": 1,
      "ctime": 1467082842,
      "etime": 1624762842,
      "rtime": null,
      "sig_status": 0,
      "prev": "860fa0d6f2a37e0c8563b6f3e88872bfbe585e65c690edc3cf7c2c2a0afe350b",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": "7fd74ecea6ac7c073e7f602938c5ddc6bda4fae3"
    },
    {
      "seqno": 10,
      "payload_hash": "a575a71f5ad00475a10328803827f519f513d55d5484fb6c857d0a45cce9f425",
      "sig_id": "5e76c100fd4a299930e86dfb130bdcb4a6662896156b2f74cee9e211be5b88ad0f",
      "sig_id_short": "XnbBAP1KKZkw6G37EwvctKZmKJYVay90zuni",
      "kid": "0101482ecf3dce65743c1f48f1bb6bd1868b90e8e44c8b39752f1569d4dff150f4040a",
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.53\nComment: https://keybase.io/crypto\n\nyMNuAnicbZJtUBVVHMYvIBlMIkRShsa4kxYN0Nn3XVIihgFCoxIUHIvbvpyFFdh7\n2bu8BSSMoijTheKSaQhY5NBETKVlEC9XQ4YCq4s2IIWlRIhWWh8AY7DOMvmhmc6X\n8/Y8v/N/zjn9K3ws/l7c2YmBnta/Jr2GTr9VYEnPn88vxUSbXIJFl2I5cKmDuTJ0\nGNYcVcaiMYATQJSgjOMcIwMekLKAKwQnA1YCgCcBRykihys4YAHBiQQtUgyP0xIp\ncxQt0ECSWSQSsAhMUbUsqNt1VTMQllVkloISFBhBQiSWhKzCAIInOYmWZYkRZYFS\nBEgiY7bNYTpQcaLggFGqDa2hiXWpvP/R36kb4BRHQEkhZQkyNEuREq5QnIKLIlKj\nNJzIA8hBipI4keRZmlBwmuFlSlbQACgUoJbqLljCKRCHQOIZhqYYQEo0xwGRVGgW\nlziG4HHeFDqgrgl50DwcyIKmYuURGForVCVoXquZ4z/7UUZBnpirR0m2PGS36zbD\nJtly0Xa2Ydgd0abdKLGb+iIoWv8lWUVVk9FVIkch1B2qTcOicaSUDNVE4xTDAo7h\nGTwCg8V2VYdW1VTQLMMB1MxzYCFCQkVSFIYiCEASMqRFnlCADABBoycVScCLDIfL\nKCmP0uFQ5jlCBgxO0AREapYTMTNbvmZDbAQ1hCzEdKhZmmAU6BArP3P6hWUWL3/L\nXb7e5h+z+PsF3vl5bfes/Fs7/nBPRl/RK3pi37kjoHJ3pm/itfSeq8CTk/JHxcgv\n22O3pnQeuMoPJZ0/FJLS6+ue6Xk3jTF2Hr95LPujJ0Kp1eMTMW1vv5E41XOtfWHc\n1Ur7tSw+MufdBCIrvru4qn7z9YNB50Lv3XDBuztpX5qnopG5Lsb7OVev3WJ9fbKG\niRtJzXjgz7Nj28bh/URo6pMvfm6N8UzVlvqfqPOae7BibcijZd5VnZtc8SVbmlpa\nTunddfdNV1YljH5VGpyQeiSybU9getBv3MAHzVtriy0+IfY38bpNqb+Gz9wYCS78\nft1Cx2u3Y2c7Ooc3Bu8P25g8IGW4Gwc/+Xoo+fzhSb1pp9/B2fU/rpxteGZwW9gP\nXgMrRm8FfpFQ5rw4ZdVivv30SuHh+N5lt0++VyIy7zONcZeV7a7BuhS7a7R+b0PC\nqeIb+XNJca3DG5x8vVH9MtW3GLDceYjamz/lXUS4+l+NPVG2a/FS1eYaop2Yn3UH\npBh5lzqm13y5u/dC0TvFesTvx1q+8ZQ/+7R2pduNj/mvCnfuSlenI/cPr3vqcknX\nS5lDzesnndWZC4PtasHMQ8ljSWcag0a6Fj0T4V1H3Y37Hpu86fS0+aypDBNb6/fE\n/lSzvHrHZ49rOdNjBnYSl7t39BLuZr+yhrLa+aNp8f0/YzH04hTmagmRbn04Tt79\n3PPenrSP5wMqqoR/ANluyno=\n=Rysj\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"fingerprint\":\"7fd74ecea6ac7c073e7f602938c5ddc6bda4fae3\",\"host\":\"keybase.io\",\"key_id\":\"38c5ddc6bda4fae3\",\"kid\":\"0101482ecf3dce65743c1f48f1bb6bd1868b90e8e44c8b39752f1569d4dff150f4040a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"service\":{\"hostname\":\"00dani.tumblr.com\",\"protocol\":\"https:\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1467086961,\"expire_in\":157680000,\"prev\":\"efcff6422032de5b92f0d0025308b309b681d03c92911ed982d061252ede578b\",\"seqno\":10,\"tag\":\"signature\"}",
      "sig_type": 2,
      "sig_version": 1,
      "ctime": 1467090561,
      "etime": 1624770561,
      "rtime": null,
      "sig_status": 0,
      "prev": "efcff6422032de5b92f0d0025308b309b681d03c92911ed982d061252ede578b",
      "proof_id": "19d79b1f12cec06f65dab910",
      "proof_type": 1000,
      "proof_text_check": "\n\nyMNuAnicbZJtUBVVHMYvIBlMIkRShsa4kxYN0Nn3XVIihgFCoxIUHIvbvpyFFdh7\n2bu8BSSMoijTheKSaQhY5NBETKVlEC9XQ4YCq4s2IIWlRIhWWh8AY7DOMvmhmc6X\n8/Y8v/N/zjn9K3ws/l7c2YmBnta/Jr2GTr9VYEnPn88vxUSbXIJFl2I5cKmDuTJ0\nGNYcVcaiMYATQJSgjOMcIwMekLKAKwQnA1YCgCcBRykihys4YAHBiQQtUgyP0xIp\ncxQt0ECSWSQSsAhMUbUsqNt1VTMQllVkloISFBhBQiSWhKzCAIInOYmWZYkRZYFS\nBEgiY7bNYTpQcaLggFGqDa2hiXWpvP/R36kb4BRHQEkhZQkyNEuREq5QnIKLIlKj\nNJzIA8hBipI4keRZmlBwmuFlSlbQACgUoJbqLljCKRCHQOIZhqYYQEo0xwGRVGgW\nlziG4HHeFDqgrgl50DwcyIKmYuURGForVCVoXquZ4z/7UUZBnpirR0m2PGS36zbD\nJtly0Xa2Ydgd0abdKLGb+iIoWv8lWUVVk9FVIkch1B2qTcOicaSUDNVE4xTDAo7h\nGTwCg8V2VYdW1VTQLMMB1MxzYCFCQkVSFIYiCEASMqRFnlCADABBoycVScCLDIfL\nKCmP0uFQ5jlCBgxO0AREapYTMTNbvmZDbAQ1hCzEdKhZmmAU6BArP3P6hWUWL3/L\nXb7e5h+z+PsF3vl5bfes/Fs7/nBPRl/RK3pi37kjoHJ3pm/itfSeq8CTk/JHxcgv\n22O3pnQeuMoPJZ0/FJLS6+ue6Xk3jTF2Hr95LPujJ0Kp1eMTMW1vv5E41XOtfWHc\n1Ur7tSw+MufdBCIrvru4qn7z9YNB50Lv3XDBuztpX5qnopG5Lsb7OVev3WJ9fbKG\niRtJzXjgz7Nj28bh/URo6pMvfm6N8UzVlvqfqPOae7BibcijZd5VnZtc8SVbmlpa\nTunddfdNV1YljH5VGpyQeiSybU9getBv3MAHzVtriy0+IfY38bpNqb+Gz9wYCS78\nft1Cx2u3Y2c7Ooc3Bu8P25g8IGW4Gwc/+Xoo+fzhSb1pp9/B2fU/rpxteGZwW9gP\nXgMrRm8FfpFQ5rw4ZdVivv30SuHh+N5lt0++VyIy7zONcZeV7a7BuhS7a7R+b0PC\nqeIb+XNJca3DG5x8vVH9MtW3GLDceYjamz/lXUS4+l+NPVG2a/FS1eYaop2Yn3UH\npBh5lzqm13y5u/dC0TvFesTvx1q+8ZQ/+7R2pduNj/mvCnfuSlenI/cPr3vqcknX\nS5lDzesnndWZC4PtasHMQ8ljSWcag0a6Fj0T4V1H3Y37Hpu86fS0+aypDBNb6/fE\n/lSzvHrHZ49rOdNjBnYSl7t39BLuZr+yhrLa+aNp8f0/YzH04hTmagmRbn04Tt79\n3PPenrSP5wMqqoR/ANluyno=\n",
      "proof_text_full": "==================================================================\nhttps://keybase.io/00dani\n--------------------------------------------------------------------\n\nI hereby claim:\n\n  * I am an admin of https://00dani.tumblr.com\n  * I am 00dani (https://keybase.io/00dani) on keybase.\n  * I have a public key with fingerprint 7FD7 4ECE A6AC 7C07 3E7F  6029 38C5 DDC6 BDA4 FAE3\n\nTo do so, I am signing this object:\n\n{\n    \"body\": {\n        \"key\": {\n            \"eldest_kid\": \"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\n            \"fingerprint\": \"7fd74ecea6ac7c073e7f602938c5ddc6bda4fae3\",\n            \"host\": \"keybase.io\",\n            \"key_id\": \"38c5ddc6bda4fae3\",\n            \"kid\": \"0101482ecf3dce65743c1f48f1bb6bd1868b90e8e44c8b39752f1569d4dff150f4040a\",\n            \"uid\": \"fe1e0c96654603c5880b3f571c862919\",\n            \"username\": \"00dani\"\n        },\n        \"service\": {\n            \"hostname\": \"00dani.tumblr.com\",\n            \"protocol\": \"https:\"\n        },\n        \"type\": \"web_service_binding\",\n        \"version\": 1\n    },\n    \"ctime\": 1467086961,\n    \"expire_in\": 157680000,\n    \"prev\": \"efcff6422032de5b92f0d0025308b309b681d03c92911ed982d061252ede578b\",\n    \"seqno\": 10,\n    \"tag\": \"signature\"\n}\n\nwhich yields the signature:\n\n-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.53\nComment: https://keybase.io/crypto\n\nyMNuAnicbZJtUBVVHMYvIBlMIkRShsa4kxYN0Nn3XVIihgFCoxIUHIvbvpyFFdh7\n2bu8BSSMoijTheKSaQhY5NBETKVlEC9XQ4YCq4s2IIWlRIhWWh8AY7DOMvmhmc6X\n8/Y8v/N/zjn9K3ws/l7c2YmBnta/Jr2GTr9VYEnPn88vxUSbXIJFl2I5cKmDuTJ0\nGNYcVcaiMYATQJSgjOMcIwMekLKAKwQnA1YCgCcBRykihys4YAHBiQQtUgyP0xIp\ncxQt0ECSWSQSsAhMUbUsqNt1VTMQllVkloISFBhBQiSWhKzCAIInOYmWZYkRZYFS\nBEgiY7bNYTpQcaLggFGqDa2hiXWpvP/R36kb4BRHQEkhZQkyNEuREq5QnIKLIlKj\nNJzIA8hBipI4keRZmlBwmuFlSlbQACgUoJbqLljCKRCHQOIZhqYYQEo0xwGRVGgW\nlziG4HHeFDqgrgl50DwcyIKmYuURGForVCVoXquZ4z/7UUZBnpirR0m2PGS36zbD\nJtly0Xa2Ydgd0abdKLGb+iIoWv8lWUVVk9FVIkch1B2qTcOicaSUDNVE4xTDAo7h\nGTwCg8V2VYdW1VTQLMMB1MxzYCFCQkVSFIYiCEASMqRFnlCADABBoycVScCLDIfL\nKCmP0uFQ5jlCBgxO0AREapYTMTNbvmZDbAQ1hCzEdKhZmmAU6BArP3P6hWUWL3/L\nXb7e5h+z+PsF3vl5bfes/Fs7/nBPRl/RK3pi37kjoHJ3pm/itfSeq8CTk/JHxcgv\n22O3pnQeuMoPJZ0/FJLS6+ue6Xk3jTF2Hr95LPujJ0Kp1eMTMW1vv5E41XOtfWHc\n1Ur7tSw+MufdBCIrvru4qn7z9YNB50Lv3XDBuztpX5qnopG5Lsb7OVev3WJ9fbKG\niRtJzXjgz7Nj28bh/URo6pMvfm6N8UzVlvqfqPOae7BibcijZd5VnZtc8SVbmlpa\nTunddfdNV1YljH5VGpyQeiSybU9getBv3MAHzVtriy0+IfY38bpNqb+Gz9wYCS78\nft1Cx2u3Y2c7Ooc3Bu8P25g8IGW4Gwc/+Xoo+fzhSb1pp9/B2fU/rpxteGZwW9gP\nXgMrRm8FfpFQ5rw4ZdVivv30SuHh+N5lt0++VyIy7zONcZeV7a7BuhS7a7R+b0PC\nqeIb+XNJca3DG5x8vVH9MtW3GLDceYjamz/lXUS4+l+NPVG2a/FS1eYaop2Yn3UH\npBh5lzqm13y5u/dC0TvFesTvx1q+8ZQ/+7R2pduNj/mvCnfuSlenI/cPr3vqcknX\nS5lDzesnndWZC4PtasHMQ8ljSWcag0a6Fj0T4V1H3Y37Hpu86fS0+aypDBNb6/fE\n/lSzvHrHZ49rOdNjBnYSl7t39BLuZr+yhrLa+aNp8f0/YzH04hTmagmRbn04Tt79\n3PPenrSP5wMqqoR/ANluyno=\n=Rysj\n-----END PGP MESSAGE-----\n\nAnd finally, I am proving ownership of this host by posting or\nappending to this document.\n\nView my publicly-auditable identity here: https://keybase.io/00dani\n\n==================================================================\n",
      "check_data_json": "{\"hostname\":\"00dani.tumblr.com\",\"protocol\":\"https:\"}",
      "remote_id": "https://00dani.tumblr.com/keybase.txt",
      "api_url": "https://00dani.tumblr.com/keybase.txt",
      "human_url": "https://00dani.tumblr.com/keybase.txt",
      "proof_state": 1,
      "proof_status": 1,
      "retry_count": 831,
      "hard_fail_count": 0,
      "last_check": 1491817466,
      "last_success": 1491817466,
      "version": 1,
      "fingerprint": "7fd74ecea6ac7c073e7f602938c5ddc6bda4fae3"
    },
    {
      "seqno": 11,
      "payload_hash": "9fdd66d9bffc819012a1b5889958337e9c098256d7b860a80e00350c4eee0270",
      "sig_id": "c24e23b86bf57da9c9db5d7f52c1c37678cabbffe4809d9e14561b5d4ed6d0810f",
      "sig_id_short": "wk4juGv1fanJ211_UsHDdnjKu__kgJ2eFFYb",
      "kid": "0101482ecf3dce65743c1f48f1bb6bd1868b90e8e44c8b39752f1569d4dff150f4040a",
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.53\nComment: https://keybase.io/crypto\n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\nnukr9r+9f00+1Kx4YNumsp6Fzxzf+NDNs179J7GcZ28=\n=Fe29\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"f10c6c4525ffbcb321eed85c16e0c874a157a0e7\"},\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"fingerprint\":\"7fd74ecea6ac7c073e7f602938c5ddc6bda4fae3\",\"host\":\"keybase.io\",\"key_id\":\"38c5ddc6bda4fae3\",\"kid\":\"0101482ecf3dce65743c1f48f1bb6bd1868b90e8e44c8b39752f1569d4dff150f4040a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"track\":{\"basics\":{\"id_version\":10,\"last_id_change\":1467089135,\"username\":\"stralytic\"},\"id\":\"a4a66a37a121fbed669d0aac245a0419\",\"key\":{\"key_fingerprint\":\"359f0c0233052ff75b3a927b02d1d23ba37d54ea\",\"kid\":\"010157447252c7c124af441d59896fa9e20df6b0c05de9972bade2364f537d25d0360a\"},\"pgp_keys\":[{\"key_fingerprint\":\"359f0c0233052ff75b3a927b02d1d23ba37d54ea\",\"kid\":\"010157447252c7c124af441d59896fa9e20df6b0c05de9972bade2364f537d25d0360a\"}],\"remote_proofs\":[{\"ctime\":1467088611,\"curr\":\"63d0f8c012cb3504c2e5a680f03dd0897e24090fc0613fd7948dbe3b3330e3fb\",\"etime\":1624768611,\"prev\":\"1fcdd80c8727d4d163000cc025a4fe3f7ebe464a5fd31b05185d32dcfaaa8128\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"stralytic\"},\"proof_type\":2,\"state\":1},\"seqno\":2,\"sig_id\":\"0e3387db3a61fd9eb0a8ce150fb91b321bb86d06bbf7a81b95ed0e04014052a80f\",\"sig_type\":2},{\"ctime\":1467088658,\"curr\":\"4f87a09db9b8f4ff8fb0f026725b41f420214b2cbc135ce0c2c39bbddbb05304\",\"etime\":1624768658,\"prev\":\"63d0f8c012cb3504c2e5a680f03dd0897e24090fc0613fd7948dbe3b3330e3fb\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"stralytic\"},\"proof_type\":3,\"state\":1},\"seqno\":3,\"sig_id\":\"8310d5a0dd8b81108ce104c651dc80098b32929a7ad4d33f0770089bbd8f87a00f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"189a38243becbc983f7f09b1cbb5be93118c5b65aac5533a72005272e99266c8\",\"seqno\":8,\"sig_id\":\"da6bb6e6d5f9146b68a1cfdec346e6c84140f92685350285b561e71e802165800f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1467089334,\"expire_in\":157680000,\"prev\":\"a575a71f5ad00475a10328803827f519f513d55d5484fb6c857d0a45cce9f425\",\"seqno\":11,\"tag\":\"signature\"}",
      "sig_type": 3,
      "sig_version": 1,
      "ctime": 1467092934,
      "etime": 1624772934,
      "rtime": null,
      "sig_status": 0,
      "prev": "a575a71f5ad00475a10328803827f519f513d55d5484fb6c857d0a45cce9f425",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": "7fd74ecea6ac7c073e7f602938c5ddc6bda4fae3"
    },
    {
      "seqno": 12,
      "payload_hash": "f5b786139a6173c3be5a48a7f54b1d9f1d2f434b42033a7b5b40e2fc70962e4b",
      "sig_id": "7e3867692058126963e36d8441c6d8c507fc26b76636120c37d4e60cdcc43b030f",
      "sig_id_short": "fjhnaSBYEmlj422EQcbYxQf8JrdmNhIMN9Tm",
      "kid": "0101482ecf3dce65743c1f48f1bb6bd1868b90e8e44c8b39752f1569d4dff150f4040a",
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.53\nComment: https://keybase.io/crypto\n\nyMNaAnicdVJ7UFRVHAZEBQIkpXgMMnINjFiYc/a+ziXcDKMhSnYweYwD7tx7z7lw\nY9tddxdcHosMElamMEOFjpnjKjIxWjijBggEBTWCMRJJTYQ6s2mROU1JDBJO3WX0\nvzr/nHN+5/u+8/0ew2Er/EL80cjMl/2nFr3+Y0OHKvwK7Sv/rKEkK66i0muocrK8\nETMmDqepXMVUOgWgHkgywRAiDgMB0FiEih5hwMsACDRAjCIhqEDAAz2S9KzEcAJk\nZRojhhVZIGNeA4mUjlJUSymx2+yqxanJ8grmGSITkRNlTYmnCa9wQC/QSGYxljkJ\ni4wiElojllkdPoZmThIdJE21ajHtYlq29x/4R74BZJCeyAqNZcKxPEPLUGGQAiVJ\nQ2vZIEkABBGGkZFECzyrVyDLCZjBinYACgOYZd8Vy3IKgQTIAsexDAdomUUISLTC\n8lBGnF6Agg/oIHaL+DrxfQ6waFEpt47SYpWqTHxlffgmW1WLL5H/YzirbL7AHiKZ\nHpJNkmrBWvU0RiWxO1SrhUqHGlJ2qj4uZDheawwDkY4iLptqJybVh2B5DgFt6Sib\nnVRqkoKCMcdhQVIUGUFB66wIJS0TQWARTfNEkIGA9CyHeQlxQESAAEBrLWQIIUDP\nA8qXzm6LVdPWaz7FUk3ToZZaRGeFnVDuz4eKA/38Q/xWrQzwjZVfSHDEo2FrUtY8\nWJc7/0Nm/knvlaWSvy5nxg+UXGYXo2Lf+z0/gjzbPNd4PKpg7+GAsnNXKrbfOXR7\n0D4lpjpnn2uPCk0ZFbZEY/uWkwmvtpieNt9q2BszFu+Obtj60jeF8wXlkcagEyOe\nXue0szrj/I1Z482eZuj58Ls8+9HubeZLG7Ja8WTu9bO/5tXsct389HDQB6Wxl9ry\nkl6bCFO3xxyELUlnYiZii5T+hav7LTY3FTc8bdmWweVvHnizX328dKEkc4PbDKvd\nHfdgzt3m8brcqZ3Q5XJ/kfGtHz+Y4zBuXjU90OO6c/vjhB3394kX0xez7xrG5hjb\n8aW+FcMdkdwnJOWq4xnDE4kzi91dxoyYIt3bKW+dSrAZNgbNyatrm1FWZ0znntCu\n6tO78Hxi1v3C6Ch33+55b3ZA6Np97//YNto10tTdYp41BJvrjoZfe+zv6P1Tgcbv\nebrGc6S92Lqupbfo2LV34rPp5H+iDH8EJ6fkuGsvzHhO76CGxz2bOs9EuB540rnI\nWx1rCp+PCxunWn/+aLL1xeuutPpJ79mC2fKvgpIrXb9xR5Z0tWudjScysgLXF1+o\nT1w6qJsQIfdZ+ZPH/I3g3fDZntHexd6mxvN1pq2/bGh4ISXp5Smxx3uveCltcP25\nvvqcnQd+qmkPXx1YZcts475uubjRU0anpm4aMhyIm3rjxoL3qVdS/wU8/rPu\n=z6s8\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"fingerprint\":\"7fd74ecea6ac7c073e7f602938c5ddc6bda4fae3\",\"host\":\"keybase.io\",\"key_id\":\"38c5ddc6bda4fae3\",\"kid\":\"0101482ecf3dce65743c1f48f1bb6bd1868b90e8e44c8b39752f1569d4dff150f4040a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"service\":{\"name\":\"coinbase\",\"username\":\"00dani\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1467090418,\"expire_in\":157680000,\"prev\":\"9fdd66d9bffc819012a1b5889958337e9c098256d7b860a80e00350c4eee0270\",\"seqno\":12,\"tag\":\"signature\"}",
      "sig_type": 2,
      "sig_version": 1,
      "ctime": 1467094018,
      "etime": 1624774018,
      "rtime": null,
      "sig_status": 0,
      "prev": "9fdd66d9bffc819012a1b5889958337e9c098256d7b860a80e00350c4eee0270",
      "proof_id": "7e5d2340cd060e91131f9410",
      "proof_type": 5,
      "proof_text_check": "\n\nyMNaAnicdVJ7UFRVHAZEBQIkpXgMMnINjFiYc/a+ziXcDKMhSnYweYwD7tx7z7lw\nY9tddxdcHosMElamMEOFjpnjKjIxWjijBggEBTWCMRJJTYQ6s2mROU1JDBJO3WX0\nvzr/nHN+5/u+8/0ew2Er/EL80cjMl/2nFr3+Y0OHKvwK7Sv/rKEkK66i0muocrK8\nETMmDqepXMVUOgWgHkgywRAiDgMB0FiEih5hwMsACDRAjCIhqEDAAz2S9KzEcAJk\nZRojhhVZIGNeA4mUjlJUSymx2+yqxanJ8grmGSITkRNlTYmnCa9wQC/QSGYxljkJ\ni4wiElojllkdPoZmThIdJE21ajHtYlq29x/4R74BZJCeyAqNZcKxPEPLUGGQAiVJ\nQ2vZIEkABBGGkZFECzyrVyDLCZjBinYACgOYZd8Vy3IKgQTIAsexDAdomUUISLTC\n8lBGnF6Agg/oIHaL+DrxfQ6waFEpt47SYpWqTHxlffgmW1WLL5H/YzirbL7AHiKZ\nHpJNkmrBWvU0RiWxO1SrhUqHGlJ2qj4uZDheawwDkY4iLptqJybVh2B5DgFt6Sib\nnVRqkoKCMcdhQVIUGUFB66wIJS0TQWARTfNEkIGA9CyHeQlxQESAAEBrLWQIIUDP\nA8qXzm6LVdPWaz7FUk3ToZZaRGeFnVDuz4eKA/38Q/xWrQzwjZVfSHDEo2FrUtY8\nWJc7/0Nm/knvlaWSvy5nxg+UXGYXo2Lf+z0/gjzbPNd4PKpg7+GAsnNXKrbfOXR7\n0D4lpjpnn2uPCk0ZFbZEY/uWkwmvtpieNt9q2BszFu+Obtj60jeF8wXlkcagEyOe\nXue0szrj/I1Z482eZuj58Ls8+9HubeZLG7Ja8WTu9bO/5tXsct389HDQB6Wxl9ry\nkl6bCFO3xxyELUlnYiZii5T+hav7LTY3FTc8bdmWweVvHnizX328dKEkc4PbDKvd\nHfdgzt3m8brcqZ3Q5XJ/kfGtHz+Y4zBuXjU90OO6c/vjhB3394kX0xez7xrG5hjb\n8aW+FcMdkdwnJOWq4xnDE4kzi91dxoyYIt3bKW+dSrAZNgbNyatrm1FWZ0znntCu\n6tO78Hxi1v3C6Ch33+55b3ZA6Np97//YNto10tTdYp41BJvrjoZfe+zv6P1Tgcbv\nebrGc6S92Lqupbfo2LV34rPp5H+iDH8EJ6fkuGsvzHhO76CGxz2bOs9EuB540rnI\nWx1rCp+PCxunWn/+aLL1xeuutPpJ79mC2fKvgpIrXb9xR5Z0tWudjScysgLXF1+o\nT1w6qJsQIfdZ+ZPH/I3g3fDZntHexd6mxvN1pq2/bGh4ISXp5Smxx3uveCltcP25\nvvqcnQd+qmkPXx1YZcts475uubjRU0anpm4aMhyIm3rjxoL3qVdS/wU8/rPu\n",
      "proof_text_full": "### Keybase proof\n\nI hereby claim:\n\n  * I am 00dani on coinbase.\n  * I am 00dani on keybase.\n  * I have a public key whose fingerprint is 7FD7 4ECE A6AC 7C07 3E7F  6029 38C5 DDC6 BDA4 FAE3\n\nTo claim this, I am signing this object:\n\n`+"`"+``+"`"+``+"`"+`json\n{\n    \"body\": {\n        \"key\": {\n            \"eldest_kid\": \"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\n            \"fingerprint\": \"7fd74ecea6ac7c073e7f602938c5ddc6bda4fae3\",\n            \"host\": \"keybase.io\",\n            \"key_id\": \"38c5ddc6bda4fae3\",\n            \"kid\": \"0101482ecf3dce65743c1f48f1bb6bd1868b90e8e44c8b39752f1569d4dff150f4040a\",\n            \"uid\": \"fe1e0c96654603c5880b3f571c862919\",\n            \"username\": \"00dani\"\n        },\n        \"service\": {\n            \"name\": \"coinbase\",\n            \"username\": \"00dani\"\n        },\n        \"type\": \"web_service_binding\",\n        \"version\": 1\n    },\n    \"ctime\": 1467090418,\n    \"expire_in\": 157680000,\n    \"prev\": \"9fdd66d9bffc819012a1b5889958337e9c098256d7b860a80e00350c4eee0270\",\n    \"seqno\": 12,\n    \"tag\": \"signature\"\n}\n`+"`"+``+"`"+``+"`"+`\n\nwith the key 7FD7 4ECE A6AC 7C07 3E7F  6029 38C5 DDC6 BDA4 FAE3, yielding the signature:\n\n`+"`"+``+"`"+``+"`"+`\n-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.53\nComment: https://keybase.io/crypto\n\nyMNaAnicdVJ7UFRVHAZEBQIkpXgMMnINjFiYc/a+ziXcDKMhSnYweYwD7tx7z7lw\nY9tddxdcHosMElamMEOFjpnjKjIxWjijBggEBTWCMRJJTYQ6s2mROU1JDBJO3WX0\nvzr/nHN+5/u+8/0ew2Er/EL80cjMl/2nFr3+Y0OHKvwK7Sv/rKEkK66i0muocrK8\nETMmDqepXMVUOgWgHkgywRAiDgMB0FiEih5hwMsACDRAjCIhqEDAAz2S9KzEcAJk\nZRojhhVZIGNeA4mUjlJUSymx2+yqxanJ8grmGSITkRNlTYmnCa9wQC/QSGYxljkJ\ni4wiElojllkdPoZmThIdJE21ajHtYlq29x/4R74BZJCeyAqNZcKxPEPLUGGQAiVJ\nQ2vZIEkABBGGkZFECzyrVyDLCZjBinYACgOYZd8Vy3IKgQTIAsexDAdomUUISLTC\n8lBGnF6Agg/oIHaL+DrxfQ6waFEpt47SYpWqTHxlffgmW1WLL5H/YzirbL7AHiKZ\nHpJNkmrBWvU0RiWxO1SrhUqHGlJ2qj4uZDheawwDkY4iLptqJybVh2B5DgFt6Sib\nnVRqkoKCMcdhQVIUGUFB66wIJS0TQWARTfNEkIGA9CyHeQlxQESAAEBrLWQIIUDP\nA8qXzm6LVdPWaz7FUk3ToZZaRGeFnVDuz4eKA/38Q/xWrQzwjZVfSHDEo2FrUtY8\nWJc7/0Nm/knvlaWSvy5nxg+UXGYXo2Lf+z0/gjzbPNd4PKpg7+GAsnNXKrbfOXR7\n0D4lpjpnn2uPCk0ZFbZEY/uWkwmvtpieNt9q2BszFu+Obtj60jeF8wXlkcagEyOe\nXue0szrj/I1Z482eZuj58Ls8+9HubeZLG7Ja8WTu9bO/5tXsct389HDQB6Wxl9ry\nkl6bCFO3xxyELUlnYiZii5T+hav7LTY3FTc8bdmWweVvHnizX328dKEkc4PbDKvd\nHfdgzt3m8brcqZ3Q5XJ/kfGtHz+Y4zBuXjU90OO6c/vjhB3394kX0xez7xrG5hjb\n8aW+FcMdkdwnJOWq4xnDE4kzi91dxoyYIt3bKW+dSrAZNgbNyatrm1FWZ0znntCu\n6tO78Hxi1v3C6Ch33+55b3ZA6Np97//YNto10tTdYp41BJvrjoZfe+zv6P1Tgcbv\nebrGc6S92Lqupbfo2LV34rPp5H+iDH8EJ6fkuGsvzHhO76CGxz2bOs9EuB540rnI\nWx1rCp+PCxunWn/+aLL1xeuutPpJ79mC2fKvgpIrXb9xR5Z0tWudjScysgLXF1+o\nT1w6qJsQIfdZ+ZPH/I3g3fDZntHexd6mxvN1pq2/bGh4ISXp5Smxx3uveCltcP25\nvvqcnQd+qmkPXx1YZcts475uubjRU0anpm4aMhyIm3rjxoL3qVdS/wU8/rPu\n=z6s8\n-----END PGP MESSAGE-----\n\n`+"`"+``+"`"+``+"`"+`\n\nAnd finally, I am proving ownership of the coinbase account by posting this on coinbase.com\n\n### My publicly-auditable identity:\n\nhttps://keybase.io/00dani\n\n`+"`"+``+"`"+``+"`"+`\n",
      "check_data_json": "{\"name\":\"coinbase\",\"username\":\"00dani\"}",
      "remote_id": "00dani",
      "api_url": "https://coinbase.com/00dani/public-key",
      "human_url": "https://coinbase.com/00dani/public-key",
      "proof_state": 1,
      "proof_status": 1,
      "retry_count": 141,
      "hard_fail_count": 0,
      "last_check": 1489817369,
      "last_success": 1489817369,
      "version": 8,
      "fingerprint": "7fd74ecea6ac7c073e7f602938c5ddc6bda4fae3"
    },
    {
      "seqno": 13,
      "payload_hash": "4df3bbf919b0719b0e6661d688896cdba690dd763a201ce268ac688dcc6203c5",
      "sig_id": "89bb37e4e993e011c5d4166d54158ed3853e40540777aee7739f9cb79036cb3d0f",
      "sig_id_short": "ibs35OmT4BHF1BZtVBWO04U-QFQHd67nc5-c",
      "kid": "0101482ecf3dce65743c1f48f1bb6bd1868b90e8e44c8b39752f1569d4dff150f4040a",
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.55\nComment: https://keybase.io/crypto\n\nyMOiAnicbZJ9UBVVGMbvxa+RptTRsTINZw38Ajln9+zXJadhmsnURsCJtNRoz55z\nrjvAvZf7BWqEDJMlmiaJMoANiYZlZlqOFKLgFwQl2jhBJFcEE03xk1QcrM5l9K/a\nf3bPmed53t/77nv8ySG2SLt2ouNk7Y4H3fbm+isB26KdrT2rBOwmKwTHKiGdDr5o\nBqE+f1q6RQSHAKAIsEkJhJpCgA4kYkAmagSoJgC6BDTEsAYZBCoQNSzKGCk6lE2J\naEg2ZGASlYsMIVZglstJvR6v5fLzWJURFVGTGoph8iRVoipTgKhLmikTYiqYGIgZ\nVOLG5W5f2MHhsOGjsyw3v+OHtEG8/9E/5gYQaSI1mURMqsgqkkzIkMYgxlzNu9Gw\nDqhGETI1LOmqLDIoKzpBhPEPwBBAg9yBwThGIQWmrigyUoBkypoGsMRkFZqaIupQ\nDwt91OsyMmm4OCCGyxJyYwUvDbrTaXiqPsvJkX2CY4nAJFEihFAFylQVgYiYpmOT\niVDTVSBJhoIxJIauaxLWkUghn7lsmoYEZIQZZwBMWMazeb2gZQ6GP6rrz7b8fur9\nDwxywTCMf4UnfJFNcdojbxq2XIT/GO4IUq/PcrsEB+RK02+FvRCpAGmKDKRYgeZ4\nLC9Ns8IKWVU0wJ9YwcMbDI9HxqqmQEk3FKhKpoSpbCDNUBknhkRnkIgMSQgjMdyf\nivmaACoyUwW6IlKEhXA3WS43z+al/IaTZ/KBuQx/wEuF3KP1S4fa7JG24cMiwhtr\nixw5+vEeJ2WOtlXka8OX3jox/95rVW+cVjwv7g3NOdxQ/zAiMeFUU2HDuhHRmw6X\nH60+njnGnns1cOvpQ207ruW3xSZ4S/OlK2xae60+oa/qYMWsTd1K74q9NzYHGvsi\nBNtEX8rzA2e7PnxodcXkDXml7Euh8Ns9Y8+4G7+o2zIl4gjtGDi54MIzXTN7EteO\nKj73/qIW18iVjs87Q5trZ93Hi8/WjImm1QUvzNuXXW8PbvjEyi+UZkZNqPuodvzF\n8hHtvpmfFWXmlW+f+17n/rvtl7M6m+6Oe3ZI3ORXbwTrp/5ZsevmoRt99uCbO+6s\nTpw//uql30pe/n510bI71zcfcV7ck55a4SnPKBXkxmmhyUKlOPV689aq+I+3TVzZ\nEYx5avfBgi3xtk9X/nTh7Yb+lvNX0gvb4NiNoQ+22X9Fp9RK21sL3MU1lQXgu4i4\nEd/MGHj3TAp2TKpMrbGPx4WpP8SPcTxwJic/ofzdbiW0lBX/XHAtku07079123Pr\n1vyypi0mxfNPzsXQtO3xTTk/LsxZVhUddzqpOMvq3ti7U97fNb2IRP1eMn32zf66\nw/PWvr7w7pxRq75uHZoIOuHtSalTUvrrSG7J8ttLyi4fqN3Vpt9+SeiuGRj2l/NY\ndV5S7YPzrfd6d+4+1xD1R3NyILB+0oYu40BEbnZ+/KXeGdHjMqJnry+tdB99x7n4\nXtz9tP6e0LHepV/RUPHIfwHkYOnw\n=M5So\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"fingerprint\":\"7fd74ecea6ac7c073e7f602938c5ddc6bda4fae3\",\"host\":\"keybase.io\",\"key_id\":\"38c5ddc6bda4fae3\",\"kid\":\"0101482ecf3dce65743c1f48f1bb6bd1868b90e8e44c8b39752f1569d4dff150f4040a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"revoke\":{\"sig_ids\":[\"f323ddde615e72024f89bcf21897033a6bb1da9983b942e13da5cca3054bf9660f\"]},\"service\":{\"name\":\"twitter\",\"username\":\"00d4n1\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1470486503,\"expire_in\":157680000,\"prev\":\"f5b786139a6173c3be5a48a7f54b1d9f1d2f434b42033a7b5b40e2fc70962e4b\",\"seqno\":13,\"tag\":\"signature\"}",
      "sig_type": 2,
      "sig_version": 1,
      "ctime": 1470490103,
      "etime": 1628170103,
      "rtime": null,
      "sig_status": 0,
      "prev": "f5b786139a6173c3be5a48a7f54b1d9f1d2f434b42033a7b5b40e2fc70962e4b",
      "proof_id": "a845a69ad3c3767d7eb42110",
      "proof_type": 2,
      "proof_text_check": "Verifying myself: I am 00dani on Keybase.io. ibs35OmT4BHF1BZtVBWO04U-QFQHd67nc5-c /",
      "proof_text_full": "Verifying myself: I am 00dani on Keybase.io. ibs35OmT4BHF1BZtVBWO04U-QFQHd67nc5-c / https://keybase.io/00dani/sigs/ibs35OmT4BHF1BZtVBWO04U-QFQHd67nc5-c\n",
      "check_data_json": "{\"name\":\"twitter\",\"username\":\"00d4n1\"}",
      "remote_id": "761901778623279104",
      "api_url": "https://twitter.com/00d4n1/status/761901778623279104",
      "human_url": "https://twitter.com/00d4n1/status/761901778623279104",
      "proof_state": 1,
      "proof_status": 1,
      "retry_count": 214,
      "hard_fail_count": 0,
      "last_check": 1491754904,
      "last_success": 1491754904,
      "version": 1,
      "fingerprint": "7fd74ecea6ac7c073e7f602938c5ddc6bda4fae3"
    },
    {
      "seqno": 14,
      "payload_hash": "66426e1fb11e39c8677c484b46cd94cd99d0fdcf69ea2804293845793745226e",
      "sig_id": "908e20c44fc76ccdbde4b4ee51d2fe6403ff3079c8ce952058142848413ba53a0f",
      "sig_id_short": "kI4gxE_HbM295LTuUdL-ZAP_MHnIzpUgWBQo",
      "kid": "0101482ecf3dce65743c1f48f1bb6bd1868b90e8e44c8b39752f1569d4dff150f4040a",
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.58\nComment: https://keybase.io/crypto\n\nyMNrAnicbVJ7UFRVHF5k0UBXyanM1IjrE4X13MfeB4IGRoRl0+A0YyQs955zLntb\n2MV98BKSgYQRDURSGa2lKJyKhglLwFQcEBGGUUmmMSTYCrEmy1GkXSKi7mXyj2Y6\nf5zX7/u+3/edOV2GQF1IAH9puPv8R1OjAX0dR926nU80zOwhJDvKJ6L3EFY8u+BM\nhJ0us1VBRDQBSApIECOS5FkEBEAjkZQpHgEOAiDQgGdkiSdlEnCA4iXKJDGsQJog\njXjGJJoARJwKEolIQlZsGdiR7VBsLlWWkxHHYIhFVoSqEkdjTmYBJdA8NCEEWQmJ\njCxiWiVa7E6NoZqTRCc2Knb1Tj2YZ+39D/6hb0AyPIWhTCOIWRPH0JCUGV4mJUlF\nq2l4SQCYxwwDeYkWOBMlkyZWQAyS1Q2QGcDM+nbPysmYxAAKLGtiWEBDE88DiZZN\nHAl5lhJIQQM6scMmZmGtOUCiTSGKIgn1LkeBWHtWLcd/6kYFGUW3ysx22F12aM9U\nKxaXK9sZrTFd+dkaNBdL5n9FzJJiQ+orqowc7HAqdhsRTapI6FI0VZLhOJ4TKIGJ\nJHBetuLAZkVDmDiWB+rQ+uAcVVJNSEuSrLqWAKdNmGVZErE8zwssRJLICgAhjqVF\nCpAQUywvQrWIIGQpLTuhxdpts2stVZ9ihqrpVDJsosvtwERRZ8cuvS4gRDc3aI72\nvXQhwaEPP13/4UV/b1Me8Y7NmNO/Mz77Y8MH9xobmoK3Tk6th+GX6cub+ybHp3x6\nXdfdyJc6uRdK64b8voThlsqSmvCemoHAH9r1udWvt1UpluX9fZOxLQvCXGBx/cjK\n8qxr8U9WtX0Wu6PL6z1YcIDoIcIGu8vjzlk+TR9Yu6Z1VeORhLTGpQsfhDx3u6b3\nRpL/QtumGXz/mLcw+OQ8IvdS41fby1aECL6KisXg6SAqwmhtSq+LGTg/k9Kji/hl\nb5nn2560sz/nHTBEDtX9WVCyvHBwdDhZzEwSzXIns+XF0qtif9bEocr9qbXt+KxU\nNWQt2BZVuiQ9sffmfhD1h+w6GJxpWOt5bfXncTuNt0pPP7qwMNE7+MzmI+FmT0S5\n6CumvzhX6fk1N3VTRIpn7xI+ZuTMtHD445yUpez86WXt9Ik38t6zVH/Ss9VqWK87\nHZe561Spu/TC9RUBTpif6um9+HjJseuh/kXJI/tGc5vk8d/91978a6OhZSjs7p0N\n+z6M2pHmu623FBbFXkyKf1v/my8msPZE2c3J+LArQdSqidY7xYf0/sCk1fFjWQ/C\nkuduLzZsDOgmvpkTcyu57qnEir6rp5pHi96Pq254LCjVvfKkf0v8cGd3wvMTVffs\ndu+62tYz85oXvLy7YdmaV4+z9e9Ox16ZfufG9+OvdIT+ZF0XfP/oGPdl7Nfzi457\n6zOa69/CMRv+ARJsxEQ=\n=O3sd\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"fingerprint\":\"7fd74ecea6ac7c073e7f602938c5ddc6bda4fae3\",\"host\":\"keybase.io\",\"key_id\":\"38c5ddc6bda4fae3\",\"kid\":\"0101482ecf3dce65743c1f48f1bb6bd1868b90e8e44c8b39752f1569d4dff150f4040a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"service\":{\"hostname\":\"00dani.id.au\",\"protocol\":\"https:\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1477879294,\"expire_in\":157680000,\"prev\":\"4df3bbf919b0719b0e6661d688896cdba690dd763a201ce268ac688dcc6203c5\",\"seqno\":14,\"tag\":\"signature\"}",
      "sig_type": 2,
      "sig_version": 1,
      "ctime": 1477882894,
      "etime": 1635562894,
      "rtime": null,
      "sig_status": 0,
      "prev": "4df3bbf919b0719b0e6661d688896cdba690dd763a201ce268ac688dcc6203c5",
      "proof_id": "803a6c9a527c3bbb72ef4b10",
      "proof_type": 1000,
      "proof_text_check": "\n\nyMNrAnicbVJ7UFRVHF5k0UBXyanM1IjrE4X13MfeB4IGRoRl0+A0YyQs955zLntb\n2MV98BKSgYQRDURSGa2lKJyKhglLwFQcEBGGUUmmMSTYCrEmy1GkXSKi7mXyj2Y6\nf5zX7/u+3/edOV2GQF1IAH9puPv8R1OjAX0dR926nU80zOwhJDvKJ6L3EFY8u+BM\nhJ0us1VBRDQBSApIECOS5FkEBEAjkZQpHgEOAiDQgGdkiSdlEnCA4iXKJDGsQJog\njXjGJJoARJwKEolIQlZsGdiR7VBsLlWWkxHHYIhFVoSqEkdjTmYBJdA8NCEEWQmJ\njCxiWiVa7E6NoZqTRCc2Knb1Tj2YZ+39D/6hb0AyPIWhTCOIWRPH0JCUGV4mJUlF\nq2l4SQCYxwwDeYkWOBMlkyZWQAyS1Q2QGcDM+nbPysmYxAAKLGtiWEBDE88DiZZN\nHAl5lhJIQQM6scMmZmGtOUCiTSGKIgn1LkeBWHtWLcd/6kYFGUW3ysx22F12aM9U\nKxaXK9sZrTFd+dkaNBdL5n9FzJJiQ+orqowc7HAqdhsRTapI6FI0VZLhOJ4TKIGJ\nJHBetuLAZkVDmDiWB+rQ+uAcVVJNSEuSrLqWAKdNmGVZErE8zwssRJLICgAhjqVF\nCpAQUywvQrWIIGQpLTuhxdpts2stVZ9ihqrpVDJsosvtwERRZ8cuvS4gRDc3aI72\nvXQhwaEPP13/4UV/b1Me8Y7NmNO/Mz77Y8MH9xobmoK3Tk6th+GX6cub+ybHp3x6\nXdfdyJc6uRdK64b8voThlsqSmvCemoHAH9r1udWvt1UpluX9fZOxLQvCXGBx/cjK\n8qxr8U9WtX0Wu6PL6z1YcIDoIcIGu8vjzlk+TR9Yu6Z1VeORhLTGpQsfhDx3u6b3\nRpL/QtumGXz/mLcw+OQ8IvdS41fby1aECL6KisXg6SAqwmhtSq+LGTg/k9Kji/hl\nb5nn2560sz/nHTBEDtX9WVCyvHBwdDhZzEwSzXIns+XF0qtif9bEocr9qbXt+KxU\nNWQt2BZVuiQ9sffmfhD1h+w6GJxpWOt5bfXncTuNt0pPP7qwMNE7+MzmI+FmT0S5\n6CumvzhX6fk1N3VTRIpn7xI+ZuTMtHD445yUpez86WXt9Ik38t6zVH/Ss9VqWK87\nHZe561Spu/TC9RUBTpif6um9+HjJseuh/kXJI/tGc5vk8d/91978a6OhZSjs7p0N\n+z6M2pHmu623FBbFXkyKf1v/my8msPZE2c3J+LArQdSqidY7xYf0/sCk1fFjWQ/C\nkuduLzZsDOgmvpkTcyu57qnEir6rp5pHi96Pq254LCjVvfKkf0v8cGd3wvMTVffs\ndu+62tYz85oXvLy7YdmaV4+z9e9Ox16ZfufG9+OvdIT+ZF0XfP/oGPdl7Nfzi457\n6zOa69/CMRv+ARJsxEQ=\n",
      "proof_text_full": "==================================================================\nhttps://keybase.io/00dani\n--------------------------------------------------------------------\n\nI hereby claim:\n\n  * I am an admin of https://00dani.id.au\n  * I am 00dani (https://keybase.io/00dani) on keybase.\n  * I have a public key with fingerprint 7FD7 4ECE A6AC 7C07 3E7F  6029 38C5 DDC6 BDA4 FAE3\n\nTo do so, I am signing this object:\n\n{\n    \"body\": {\n        \"key\": {\n            \"eldest_kid\": \"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\n            \"fingerprint\": \"7fd74ecea6ac7c073e7f602938c5ddc6bda4fae3\",\n            \"host\": \"keybase.io\",\n            \"key_id\": \"38c5ddc6bda4fae3\",\n            \"kid\": \"0101482ecf3dce65743c1f48f1bb6bd1868b90e8e44c8b39752f1569d4dff150f4040a\",\n            \"uid\": \"fe1e0c96654603c5880b3f571c862919\",\n            \"username\": \"00dani\"\n        },\n        \"service\": {\n            \"hostname\": \"00dani.id.au\",\n            \"protocol\": \"https:\"\n        },\n        \"type\": \"web_service_binding\",\n        \"version\": 1\n    },\n    \"ctime\": 1477879294,\n    \"expire_in\": 157680000,\n    \"prev\": \"4df3bbf919b0719b0e6661d688896cdba690dd763a201ce268ac688dcc6203c5\",\n    \"seqno\": 14,\n    \"tag\": \"signature\"\n}\n\nwhich yields the signature:\n\n-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.58\nComment: https://keybase.io/crypto\n\nyMNrAnicbVJ7UFRVHF5k0UBXyanM1IjrE4X13MfeB4IGRoRl0+A0YyQs955zLntb\n2MV98BKSgYQRDURSGa2lKJyKhglLwFQcEBGGUUmmMSTYCrEmy1GkXSKi7mXyj2Y6\nf5zX7/u+3/edOV2GQF1IAH9puPv8R1OjAX0dR926nU80zOwhJDvKJ6L3EFY8u+BM\nhJ0us1VBRDQBSApIECOS5FkEBEAjkZQpHgEOAiDQgGdkiSdlEnCA4iXKJDGsQJog\njXjGJJoARJwKEolIQlZsGdiR7VBsLlWWkxHHYIhFVoSqEkdjTmYBJdA8NCEEWQmJ\njCxiWiVa7E6NoZqTRCc2Knb1Tj2YZ+39D/6hb0AyPIWhTCOIWRPH0JCUGV4mJUlF\nq2l4SQCYxwwDeYkWOBMlkyZWQAyS1Q2QGcDM+nbPysmYxAAKLGtiWEBDE88DiZZN\nHAl5lhJIQQM6scMmZmGtOUCiTSGKIgn1LkeBWHtWLcd/6kYFGUW3ysx22F12aM9U\nKxaXK9sZrTFd+dkaNBdL5n9FzJJiQ+orqowc7HAqdhsRTapI6FI0VZLhOJ4TKIGJ\nJHBetuLAZkVDmDiWB+rQ+uAcVVJNSEuSrLqWAKdNmGVZErE8zwssRJLICgAhjqVF\nCpAQUywvQrWIIGQpLTuhxdpts2stVZ9ihqrpVDJsosvtwERRZ8cuvS4gRDc3aI72\nvXQhwaEPP13/4UV/b1Me8Y7NmNO/Mz77Y8MH9xobmoK3Tk6th+GX6cub+ybHp3x6\nXdfdyJc6uRdK64b8voThlsqSmvCemoHAH9r1udWvt1UpluX9fZOxLQvCXGBx/cjK\n8qxr8U9WtX0Wu6PL6z1YcIDoIcIGu8vjzlk+TR9Yu6Z1VeORhLTGpQsfhDx3u6b3\nRpL/QtumGXz/mLcw+OQ8IvdS41fby1aECL6KisXg6SAqwmhtSq+LGTg/k9Kji/hl\nb5nn2560sz/nHTBEDtX9WVCyvHBwdDhZzEwSzXIns+XF0qtif9bEocr9qbXt+KxU\nNWQt2BZVuiQ9sffmfhD1h+w6GJxpWOt5bfXncTuNt0pPP7qwMNE7+MzmI+FmT0S5\n6CumvzhX6fk1N3VTRIpn7xI+ZuTMtHD445yUpez86WXt9Ik38t6zVH/Ss9VqWK87\nHZe561Spu/TC9RUBTpif6um9+HjJseuh/kXJI/tGc5vk8d/91978a6OhZSjs7p0N\n+z6M2pHmu623FBbFXkyKf1v/my8msPZE2c3J+LArQdSqidY7xYf0/sCk1fFjWQ/C\nkuduLzZsDOgmvpkTcyu57qnEir6rp5pHi96Pq254LCjVvfKkf0v8cGd3wvMTVffs\ndu+62tYz85oXvLy7YdmaV4+z9e9Ox16ZfufG9+OvdIT+ZF0XfP/oGPdl7Nfzi457\n6zOa69/CMRv+ARJsxEQ=\n=O3sd\n-----END PGP MESSAGE-----\n\nAnd finally, I am proving ownership of this host by posting or\nappending to this document.\n\nView my publicly-auditable identity here: https://keybase.io/00dani\n\n==================================================================\n",
      "check_data_json": "{\"hostname\":\"00dani.id.au\",\"protocol\":\"https:\"}",
      "remote_id": "https://00dani.id.au/keybase.txt",
      "api_url": "https://00dani.id.au/keybase.txt",
      "human_url": "https://00dani.id.au/keybase.txt",
      "proof_state": 1,
      "proof_status": 1,
      "retry_count": 412,
      "hard_fail_count": 0,
      "last_check": 1491820595,
      "last_success": 1491820595,
      "version": 1,
      "fingerprint": "7fd74ecea6ac7c073e7f602938c5ddc6bda4fae3"
    },
    {
      "seqno": 15,
      "payload_hash": "c96060d7ac314314ad9c76edc83713546693f80adf9e87abe8026b9008918645",
      "sig_id": "134508ea128b103894024d3faee420a7513b1dd5618a1fc15c2f48f3cf3636860f",
      "sig_id_short": "E0UI6hKLEDiUAk0_ruQgp1E7HdVhih_BXC9I",
      "kid": "0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a",
      "sig": "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",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"sibkey\":{\"fingerprint\":\"ee4a204fce849118445e2a62cc91589719027e94\",\"full_hash\":\"d09ec9dc0a027fddc87478a14886d2364b4227a7938ae3c85a0aed60cc3d3e51\",\"key_id\":\"CC91589719027E94\",\"kid\":\"01016b832486fd78ab28a4f52ac81518bd36fec6e06d8942a6325f743c5c629f9f3b0a\",\"reverse_sig\":\"-----BEGIN PGP MESSAGE-----\\nComment: https://keybase.io/download\\nVersion: Keybase Go 1.0.17 (darwin)\\n\\nxA0DAAoBzJFYlxkCfpQBy+F0AOIAAAAA6XsiYm9keSI6eyJrZXkiOnsiZWxkZXN0\\nX2tpZCI6IjAxMjBiY2VkMTE4NmQwOTAzZGExZjI4ZDA3YzAwOTMwODRmYjgxZjEw\\nNzAyOGIyNWI0NjkxNWMzZDg0NWE1MGNkNzkzMGEiLCJob3N0Ijoia2V5YmFzZS5p\\nbyIsImtpZCI6IjAxMjBiY2VkMTE4NmQwOTAzZGExZjI4ZDA3YzAwOTMwODRmYjgx\\nZjEwNzAyOGIyNWI0NjkxNWMzZDg0NWE1MGNkNzkzMGEiLCJ1aWQiOiJmZTFlMGM5\\nNjY1NDYwM2M1ODgwYjNmNTcxYzg2MjkxOSIsInVzZXJuYW1lIjoiMDBkYW5pIn0s\\nInNpYmtleSI6eyJmaW5nZXJwcmludCI6ImVlNGEyMDRmY2U4NDkxMTg0NDVlMmE2\\nMmNjOTE1ODk3MTkwMjdlOTQiLCJmdWxsX2hhc2giOiJkMDllYzlkYzBhMDI3ZmRk\\nYzg3NDc4YTE0ODg2ZDIzNjRiNDIyN2E3OTM4YWUzYzg1YTBhZWQ2MGNjM2QzZTUx\\nIiwia2V5X2lkIjoiQ0M5MTU4OTcxOTAyN0U5NCIsImtpZCI6IjAxMDE2YjgzMjQ4\\nNmZkNzhhYjI4YTRmNTJhYzgxNTE4YmQzNmZlYzZlMDZkODk0MmE2MzI1Zjc0M2M1\\nYzYyOWY5ZjNi6DBhIiwicmV2ZXJzZV9zaWciOm51bGx9LCJ0eXBlIjoic2lia2V5\\nIiwidmVyc2lvbiI6MX0sImNsaWVudCI6eyJuYW1lIjoia2V5YmFzZS5pbyBnbyBj\\nbGllbnQiLCJ2ZXJzaW9uIjoiMS4wLjE3In0sImN0aW1lIjoxNDc3ODkzMDcyLCJl\\neHBpcmVfaW4iOjUwNDU3NjAwMCwibWVya2xlX3Jvb3QiOnsiY3RpbWUiOjE0Nzc4\\nOTIzNTksImhhc2giOiJkZTA3NzliNWUyOGY5N2U4ZTQ1NjEyOGM3M2JiNTdlYWI0\\nNjNmZGVkZDA4NjNmNTE5YjE5ZjczYzY0NjfnZGJmY2U5OGQ3ZTVjODI0OWMzMGZl\\nNmQxODA3MGU3YjRjM2U0YjE4OTYyMzgxZDA1ODk5ODAzMjRkNjBmNTQ5MGNmNjQi\\nLCJzZXFubyI6Njk0MTM5fSwicHJldiI6IjY2NDI2ZTFmYjExZTM5Yzg2NzdjNDg0\\nYjQ2Y2Q5NGNkOTnlZDBmZGNmNjllYTI4MDQyOTM4NDU3OTM3NDUyMjZlIizkInNl\\ncW5vIjoxNSwidGFnIuM6InNpZ25hdOJ1cmUi4H0AwsFcBAABCgAQBQJYFtvQCRDM\\nkViXGQJ+lAAAoioQAAPk5Sz6GPGdnRDCNxYgi+E30gSgdtvIKsTI6UcedjHhDGPv\\nD8il5qJvwAVI4oYOKEfP+PkXbEpn9nFF6q1jHvH8KUTJnN8XVCW+KNvDDGGl+cml\\niix2x1R429+l4Hx2Gbkli0zLlM19u7Ne8hI3mASsLzUjqfkjEoyvb/pfjzy/bGif\\nklJ+FBJDw0jx/B7o+nDtVb/J6DS2quMv1zNFYt0agB9ydbWJntQchIrzVYWdpkWv\\nl1LBdRunr9UqSM89gCyX8wwS7caqsboUllnzFI+xl6zwWGdLsYhQ9bVcjm6GjC5P\\nVO/0aZKlRbfChcN5aiuPaHyY7u93+bIE2hmqfjxyc5DRNoLSdyCs2K9XEr2lVGC2\\nfuM0gMdC014OpBS085zHMZHgzxd97JZdpSCil/0xafrrcSWgrxsCVceutTP0Gr87\\npadwsSV2rgqqQa3yxCpoRVgHw2/H+pvcy0sf/NvfnEnzmz38r6dVUn+WmStwfb9e\\nyWsaYnwur/Uyc1FTiC+oRE4qPi7eNe/EZ2fIXQDsy0E6E8ImcPsj40wn+EvlLMCL\\nvZxkQziZ4p/ATLEa+h2bHE7L0uF+gf/4bRhCgKRrzj8ZoJN0q69umBpwHITbaFoO\\n3YCx6Pjza8HCSFBT+IBVuEWajWz4rfb3cObqA0NfhHu4oZH2esV4LMpX3DQU\\n=k3Dl\\n-----END PGP MESSAGE-----\\n\"},\"type\":\"sibkey\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.17\"},\"ctime\":1477893072,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1477892359,\"hash\":\"de0779b5e28f97e8e456128c73bb57eab463fdedd0863f519b19f73c6467dbfce98d7e5c8249c30fe6d18070e7b4c3e4b18962381d0589980324d60f5490cf64\",\"seqno\":694139},\"prev\":\"66426e1fb11e39c8677c484b46cd94cd99d0fdcf69ea2804293845793745226e\",\"seqno\":15,\"tag\":\"signature\"}",
      "sig_type": 1,
      "sig_version": 1,
      "ctime": 1477896672,
      "etime": 1982472672,
      "rtime": null,
      "sig_status": 0,
      "prev": "66426e1fb11e39c8677c484b46cd94cd99d0fdcf69ea2804293845793745226e",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 16,
      "payload_hash": "c7c0c853c81b5e3b277892d13da7f34cc736f379c65405394f2e49e6d06d6380",
      "sig_id": "e69de0ee4aa4e9c65eb11a6a88f4842d21036429d119b8d06188eb6c68b80a0c0f",
      "sig_id_short": "5p3g7kqk6cZesRpqiPSELSEDZCnRGbjQYYjr",
      "kid": "0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a",
      "sig": "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",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"track\":{\"basics\":{\"id_version\":5,\"last_id_change\":1477896864,\"username\":\"ralim\"},\"id\":\"f65c02a382724e8892e8960d89cc0c19\",\"key\":{\"key_fingerprint\":\"69e9e9699f80e2b3d95299b5e245388447456f8f\",\"kid\":\"0101a1b12c46e0bc12e3d66134c642949de424550febc1ac820ad07b77efc3e335a90a\"},\"pgp_keys\":[{\"key_fingerprint\":\"69e9e9699f80e2b3d95299b5e245388447456f8f\",\"kid\":\"0101a1b12c46e0bc12e3d66134c642949de424550febc1ac820ad07b77efc3e335a90a\"}],\"remote_proofs\":[{\"ctime\":1477896342,\"curr\":\"179895c8ab6a1323982adb430504237f41f3890a754f46445352d32e82197bc4\",\"etime\":1635576342,\"prev\":\"1d07d619788448b235689bf9b3d338f1fed15625cdc8d9752a27c135c4aca338\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"Ralimtek\"},\"proof_type\":2,\"state\":1},\"sig_id\":\"c3b0ca151a8caca39f3d88188de80b2920be846a6102efb4fe14c41153f55e330f\",\"sig_type\":2},{\"ctime\":1477896791,\"curr\":\"06039dbc837a67762a1e53fc821e04e035dd374f7e12c11cb43d9e62d8b91fb1\",\"etime\":1635576791,\"prev\":\"179895c8ab6a1323982adb430504237f41f3890a754f46445352d32e82197bc4\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"Ralim\"},\"proof_type\":3,\"state\":1},\"sig_id\":\"aa592723095b662488f762ebf67ab0bd3e93290d5ff8d1e11ec0a15eccdb3e5a0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"06039dbc837a67762a1e53fc821e04e035dd374f7e12c11cb43d9e62d8b91fb1\",\"seqno\":3,\"sig_id\":\"aa592723095b662488f762ebf67ab0bd3e93290d5ff8d1e11ec0a15eccdb3e5a0f\"}},\"type\":\"track\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.17\"},\"ctime\":1477897252,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1477896960,\"hash\":\"f56c511245592d97ddc6e04fd549f3044a5b4731c2c987719b71bf44f6deaa374084e2798b9bfdf0fef83ea7463bf585c6b4c9fcdd73fef65e7a877f6517a70e\",\"seqno\":694164},\"prev\":\"c96060d7ac314314ad9c76edc83713546693f80adf9e87abe8026b9008918645\",\"seqno\":16,\"tag\":\"signature\"}",
      "sig_type": 3,
      "sig_version": 1,
      "ctime": 1477900852,
      "etime": 1982476852,
      "rtime": null,
      "sig_status": 0,
      "prev": "c96060d7ac314314ad9c76edc83713546693f80adf9e87abe8026b9008918645",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 17,
      "payload_hash": "fcf63866f7fb4aeb0020d4ce087a6c6b81c6a1ded4e0431adda7bfd8b07825d0",
      "sig_id": "0bf4c89f4bff724d27a9d33698db170c71cf0d02ac43ebdd39e3135180e8d20c0f",
      "sig_id_short": "C_TIn0v_ck0nqdM2mNsXDHHPDQKsQ-vdOeMT",
      "kid": "0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a",
      "sig": "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",
      "payload_json": "{\"body\":{\"device\":{\"id\":\"cf340099040938fe14e47833fd437518\",\"kid\":\"0120d3e4960119263b3d09382906e21022e79a7e92057654c8ccb8e6fdc7a593a01a0a\",\"name\":\"reaper\",\"status\":1,\"type\":\"desktop\"},\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"sibkey\":{\"kid\":\"0120d3e4960119263b3d09382906e21022e79a7e92057654c8ccb8e6fdc7a593a01a0a\",\"reverse_sig\":\"hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEg0+SWARkmOz0JOCkG4hAi55p+kgV2VMjMuOb9x6WToBoKp3BheWxvYWTFA8d7ImJvZHkiOnsiZGV2aWNlIjp7ImlkIjoiY2YzNDAwOTkwNDA5MzhmZTE0ZTQ3ODMzZmQ0Mzc1MTgiLCJraWQiOiIwMTIwZDNlNDk2MDExOTI2M2IzZDA5MzgyOTA2ZTIxMDIyZTc5YTdlOTIwNTc2NTRjOGNjYjhlNmZkYzdhNTkzYTAxYTBhIiwibmFtZSI6InJlYXBlciIsInN0YXR1cyI6MSwidHlwZSI6ImRlc2t0b3AifSwia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwidWlkIjoiZmUxZTBjOTY2NTQ2MDNjNTg4MGIzZjU3MWM4NjI5MTkiLCJ1c2VybmFtZSI6IjAwZGFuaSJ9LCJzaWJrZXkiOnsia2lkIjoiMDEyMGQzZTQ5NjAxMTkyNjNiM2QwOTM4MjkwNmUyMTAyMmU3OWE3ZTkyMDU3NjU0YzhjY2I4ZTZmZGM3YTU5M2EwMWEwYSIsInJldmVyc2Vfc2lnIjpudWxsfSwidHlwZSI6InNpYmtleSIsInZlcnNpb24iOjF9LCJjbGllbnQiOnsibmFtZSI6ImtleWJhc2UuaW8gZ28gY2xpZW50IiwidmVyc2lvbiI6IjEuMC4xNyJ9LCJjdGltZSI6MTQ3NzkwODA3OSwiZXhwaXJlX2luIjo1MDQ1NzYwMDAsIm1lcmtsZV9yb290Ijp7ImN0aW1lIjoxNDc3OTA4MDQzLCJoYXNoIjoiZWU2NTY0ZTNjYmZlNWQ5OGU1MDM2YWVmNjJmMThlYWM0MzBhNjk2MzI0ZTIzMGViMmY2OWViMDBmNWI2Mzc5YmZiODkwN2MxMWQ5ZWZmZDYwOTBmMzUxZTZhYjE1YTVkMzExZWZkMDQ4NzllY2ViYzU3ZTU1ZmNiMzFhM2IxNzMiLCJzZXFubyI6Njk0Mjk1fSwicHJldiI6ImM3YzBjODUzYzgxYjVlM2IyNzc4OTJkMTNkYTdmMzRjYzczNmYzNzljNjU0MDUzOTRmMmU0OWU2ZDA2ZDYzODAiLCJzZXFubyI6MTcsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RAHff/HwkJb3r+GuQaHV5Dhp2wPLahU31w0tV9FpVhWKfSIuBuYD3DILzvjwaTIXAgTJJYp6WJVaw38hC+92AFAKhzaWdfdHlwZSCkaGFzaIKkdHlwZQildmFsdWXEIBPIEjyAwInz8AOm2D6w2xrXFs1+6+LQUOLdpRB0bbUBo3RhZ80CAqd2ZXJzaW9uAQ==\"},\"type\":\"sibkey\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.17\"},\"ctime\":1477908079,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1477908043,\"hash\":\"ee6564e3cbfe5d98e5036aef62f18eac430a696324e230eb2f69eb00f5b6379bfb8907c11d9effd6090f351e6ab15a5d311efd04879ecebc57e55fcb31a3b173\",\"seqno\":694295},\"prev\":\"c7c0c853c81b5e3b277892d13da7f34cc736f379c65405394f2e49e6d06d6380\",\"seqno\":17,\"tag\":\"signature\"}",
      "sig_type": 1,
      "sig_version": 1,
      "ctime": 1477911679,
      "etime": 1982487679,
      "rtime": null,
      "sig_status": 0,
      "prev": "c7c0c853c81b5e3b277892d13da7f34cc736f379c65405394f2e49e6d06d6380",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 18,
      "payload_hash": "b26f3ab4a3e49156d4859d3a1081b78d0a1f7c7fec726bcd2f428e193ce1f73a",
      "sig_id": "db60f557f16f4f12f50b9271d751f1b36e83e6981be58e8bc98f4c0cc0fbfda60f",
      "sig_id_short": "22D1V_FvTxL1C5Jx11Hxs26D5pgb5Y6LyY9M",
      "kid": "0120d3e4960119263b3d09382906e21022e79a7e92057654c8ccb8e6fdc7a593a01a0a",
      "sig": "hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEg0+SWARkmOz0JOCkG4hAi55p+kgV2VMjMuOb9x6WToBoKp3BheWxvYWTFA+l7ImJvZHkiOnsiZGV2aWNlIjp7ImlkIjoiY2YzNDAwOTkwNDA5MzhmZTE0ZTQ3ODMzZmQ0Mzc1MTgiLCJraWQiOiIwMTIxMTkwMDM0YTYxOTQ3NzUyNWE0YzRlYzI0YTlhYTM2OWE4YmZiNmEyNmI5ZTI5Y2JlYTQ2ZmNiMjY0NTU2OTQ3MDBhIiwic3RhdHVzIjoxfSwia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwZDNlNDk2MDExOTI2M2IzZDA5MzgyOTA2ZTIxMDIyZTc5YTdlOTIwNTc2NTRjOGNjYjhlNmZkYzdhNTkzYTAxYTBhIiwidWlkIjoiZmUxZTBjOTY2NTQ2MDNjNTg4MGIzZjU3MWM4NjI5MTkiLCJ1c2VybmFtZSI6IjAwZGFuaSJ9LCJzdWJrZXkiOnsia2lkIjoiMDEyMTE5MDAzNGE2MTk0Nzc1MjVhNGM0ZWMyNGE5YWEzNjlhOGJmYjZhMjZiOWUyOWNiZWE0NmZjYjI2NDU1Njk0NzAwYSIsInBhcmVudF9raWQiOiIwMTIwZDNlNDk2MDExOTI2M2IzZDA5MzgyOTA2ZTIxMDIyZTc5YTdlOTIwNTc2NTRjOGNjYjhlNmZkYzdhNTkzYTAxYTBhIn0sInR5cGUiOiJzdWJrZXkiLCJ2ZXJzaW9uIjoxfSwiY2xpZW50Ijp7Im5hbWUiOiJrZXliYXNlLmlvIGdvIGNsaWVudCIsInZlcnNpb24iOiIxLjAuMTcifSwiY3RpbWUiOjE0Nzc5MDgxMDYsImV4cGlyZV9pbiI6NTA0NTc2MDAwLCJtZXJrbGVfcm9vdCI6eyJjdGltZSI6MTQ3NzkwODA0MywiaGFzaCI6ImVlNjU2NGUzY2JmZTVkOThlNTAzNmFlZjYyZjE4ZWFjNDMwYTY5NjMyNGUyMzBlYjJmNjllYjAwZjViNjM3OWJmYjg5MDdjMTFkOWVmZmQ2MDkwZjM1MWU2YWIxNWE1ZDMxMWVmZDA0ODc5ZWNlYmM1N2U1NWZjYjMxYTNiMTczIiwic2Vxbm8iOjY5NDI5NX0sInByZXYiOiJmY2Y2Mzg2NmY3ZmI0YWViMDAyMGQ0Y2UwODdhNmM2YjgxYzZhMWRlZDRlMDQzMWFkZGE3YmZkOGIwNzgyNWQwIiwic2Vxbm8iOjE4LCJ0YWciOiJzaWduYXR1cmUifaNzaWfEQPexmJQHZ718fbihBooe416TpFneCUGVNxCeCTsbAcnEjppP/JTkdSdscQCugCZxLVHgHUZfw+IExU36PZK27QGoc2lnX3R5cGUgpGhhc2iCpHR5cGUIpXZhbHVlxCBC6XPuK4yPgCCqCOglfJjIHJGoSER9vxk1t6l/JBHM46N0YWfNAgKndmVyc2lvbgE=",
      "payload_json": "{\"body\":{\"device\":{\"id\":\"cf340099040938fe14e47833fd437518\",\"kid\":\"0121190034a619477525a4c4ec24a9aa369a8bfb6a26b9e29cbea46fcb26455694700a\",\"status\":1},\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"0120d3e4960119263b3d09382906e21022e79a7e92057654c8ccb8e6fdc7a593a01a0a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"subkey\":{\"kid\":\"0121190034a619477525a4c4ec24a9aa369a8bfb6a26b9e29cbea46fcb26455694700a\",\"parent_kid\":\"0120d3e4960119263b3d09382906e21022e79a7e92057654c8ccb8e6fdc7a593a01a0a\"},\"type\":\"subkey\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.17\"},\"ctime\":1477908106,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1477908043,\"hash\":\"ee6564e3cbfe5d98e5036aef62f18eac430a696324e230eb2f69eb00f5b6379bfb8907c11d9effd6090f351e6ab15a5d311efd04879ecebc57e55fcb31a3b173\",\"seqno\":694295},\"prev\":\"fcf63866f7fb4aeb0020d4ce087a6c6b81c6a1ded4e0431adda7bfd8b07825d0\",\"seqno\":18,\"tag\":\"signature\"}",
      "sig_type": 1,
      "sig_version": 1,
      "ctime": 1477911706,
      "etime": 1982487706,
      "rtime": null,
      "sig_status": 0,
      "prev": "fcf63866f7fb4aeb0020d4ce087a6c6b81c6a1ded4e0431adda7bfd8b07825d0",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 19,
      "payload_hash": "3139597da320b5c479e4045339f8ce4541ed24bff5c34a7a1c30b6bf1e47c03a",
      "sig_id": "886ed97656e461dadcff5891f26957fb87e2e922d6857276d3940275133256480f",
      "sig_id_short": "iG7ZdlbkYdrc_1iR8mlX-4fi6SLWhXJ205QC",
      "kid": "0101482ecf3dce65743c1f48f1bb6bd1868b90e8e44c8b39752f1569d4dff150f4040a",
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.58\nComment: https://keybase.io/crypto\n\nyMaPAniczVZrjFXVFZ6BQmSKiI5UWx7iAU3UUfb7gZQ2JMSA6IR2YiVgr/s5c5lh\n7njvGaYjjIINYGWslY4UaaqlQkOhDxrwUQOGhwIKnfKwSWuEttEUCVoKGBVbade5\njHQGadom/ui9f07O3mftb33ft9baL17cv6KqUr10aOeW1R++Ubln98utFXdeU/Hw\nvMQWfHsyfl7imvKhOc2ems2ckIxPGkO7NaVwU74wui3YpCaZG4qlfKEZliiSUWNr\nrfYiIOWdZyFIKjmJxHJrLKUR40CSjposTBY0NPlQSnONeQ/fI0yQdcFjrIRHGlFv\ncCTKI+kQ0hQpFq3CESOJiLKEWyY05o56xbjhyHkJmwxAivnm+lBsKeYz5ImMXrLg\nghHGQSRJg4wCEU2V4947Yb1h0QQKHzYUSmmfHJMy0lwZ3gX2f4wbYaZIcJF6FwSX\njDocmYrABeyGbJTVKKjAmFOW6owQzIX2zEd4QJEhVsbdWg4XAw7IaSE4E4g6rhSy\nNHKJnRJEY51tLIVijx4IedOczyhNi8Y1ZqQC9rwrZU95nzsnD8a6JmkywDa8dQ0G\nKIKXTCqF4a/6BE2L7c35tD2LWoYkLTM8MGONkNJahDXTATPiQsTGI5ScEzRjqy/9\nRDJgwZgAaTFBGfMSqKCCGqBGBoFQpJoQ2YdOCk6KigcTUMBgG6qCkBFYxdQYgQlE\nCERH7AlQqYNgVnEukXNCI6OBN5Nhb6lvyQEgoGLm/xWyu2qSYphTSEOupVgoxLP4\nXJqfc04RqTWvSVxrsVg+FqyNMffAuCVgKiipwDRxiAIMariCFAQ2Ds6SYE1Mg4eE\nCHNCsMgV4A89wbWiVNNy8JZimJu5WiIZuIDt0UMwFhXBDCokCG2ICSoaj52Brxim\nWHCMOBE0s6S1REma2bYnl4zgcj7lvtEQXGPOm9TkZpcy+51rIGlbPk1Dsa+Lw9x8\nU71phyXXUFYui5NL21tgkdQkpdSkGXxYKYV7mgvJeE7hMV9/tjKV4lJwaiSoRpAL\n1mjoJDxGkEoLjLRUgYNMkjKnbciyAN8yRnBU3mqJYnI2Ws+BHTXnyaEQQefk4NEi\nZZRXVgihqCcoZG8UZpZm+hLPDMqkB428VILBURjEUJo6yxQ6Xw5WDt4jx6eg9f8m\nR30+bWi1yb8r/95C0AsKwXoJ4algwSLJmJDOakE4MdB6haDAfRTGBysUpAil4BEU\nkUMwFZSDZiKQI8L+ZyFwLyEE84YahYO00Eq5NMRCaSLiaUTAUySMAk+cIW0dyEIo\nfK0449gBrZqHTwiBewnxKaj8XwqRDZ0e4ufki6Gp6aZCsT4pU58WXKEJ3jekaUtp\n/Pl6YITQBSXhvSQhknsuEEAnTscoIoJRyQNwAcNYQ21TaXngVsEogsIXklLvtbWG\nMOu00p+Q5K7yObnU5Jsy9C2mvalgfK7BlBqyQpYI4WAZF9zJrBnKiLDK5jKXWnit\no8XRMJhvWNHoIoGeYm201DnvnCHJv7IQvY3lpaWGSuyh/RKwf9CcUygRbqB6NZS8\n41wL6gL8dNaXIDcZMYtMMhIiZNGRjchyDj2TstfVJSOvj8+0Vhjc8Y0WECSXz3ZA\nh1GoTHiPQSwRESYFMxQcAOPcM8W1h4ORwlbCrQUuL9JBIwUehHWeREZUwGCPAAvl\nxtmTaTaYU1MPMSHfZpO2FkPSsX3brM9UVFZVDBzQL7uRVVQNGvrxPW3TpKEVXYUf\n/+bD15dfO3vkm/uufqi1bdNsKg69dMOCS55d/+WV3e+t3fnU4LXFqvc7j78eS/K+\nNwfNuPKrk+qrrjg47PF1FTc8+btXjnUMHTzjO9O21U50B37xy9Ubdt9bdfRX37yx\ne1/dkAU3b4jTjuG6X285daSy/rdb8l0PvHP9rMsqN6/f9cxbX0Q//ewHu1ZNWTBz\n35Ht6USZnvzDpLFn9uwYc8uW6ksnV/Y/ff3p+tqBV00YtvnUrvWrqye/NmTPhtU7\nW+9Y98DPBrVfPnPSVauqw5B371w2cv7ypYdOLMNr1v/91PfJ/Yu+suKdsT84Mqy2\n821V8+r+iaNWtF5TsaPu7uPxwOHu2Wjbc6P24K2PLHz6iuqTa5YM/vOE6U//fm91\noVJf9srBAdPrJuxY+wY7eeiRm49OHLBv5cV/WT7zuan8sRH99i/Yc//G73XY2VP3\nfXvc2Oruv3a+9URbx/ER8zs/sv/gSwcd3Xvrxhu/NWiv23178diqnU9NXLhu8qN/\nHHPpF5aemTFy3Lqvz31w6sJte4eMPrHo8Ptd2w527Tgx8uebjs07+NgLb1/Z3X/W\n5VvdohdqT169+LY1UzaeedY8/9HQ4R3z/1Z17YmtY3/U79XFnd89cOsT70X5Afva\nLfhzSx4aPeLutnDvTyY0Drjkuum3Ty7etv34yrFdLw4+PLB1y+gH/zTtvsX7+Q9f\nW7WiafiA4RfVTXn8TLL+ji/d8/kxa0cteXnMqHcfve6i2nFPPrNsx+nnN2/+J556\n3VA=\n=mbKT\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"307f91bbb9d6e08dcd4ee73752f2b5bab33f11e2\"},\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"fingerprint\":\"7fd74ecea6ac7c073e7f602938c5ddc6bda4fae3\",\"host\":\"keybase.io\",\"key_id\":\"38c5ddc6bda4fae3\",\"kid\":\"0101482ecf3dce65743c1f48f1bb6bd1868b90e8e44c8b39752f1569d4dff150f4040a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"track\":{\"basics\":{\"id_version\":119,\"last_id_change\":1478818188,\"username\":\"trynity\"},\"id\":\"7b4a5e4aba677bb01949e142cef1ad00\",\"key\":{\"key_fingerprint\":\"274b6baaee1e46344d7e8e363abd17e600f39227\",\"kid\":\"010137f9f85eae0e133f38e67fc1f13aa612e8ee29f1d22f19e64b85570cc690a9460a\"},\"pgp_keys\":[{\"key_fingerprint\":\"274b6baaee1e46344d7e8e363abd17e600f39227\",\"kid\":\"010137f9f85eae0e133f38e67fc1f13aa612e8ee29f1d22f19e64b85570cc690a9460a\"}],\"remote_proofs\":[{\"ctime\":1478817995,\"curr\":\"01cd7115d142b21bb375e492c03eae3a58aee61ac4b874fa13ede4624c664f58\",\"etime\":1983393995,\"prev\":\"3707e56462fdae34f8214e7fe69a2ae8fad1ca339413165105263880bbb2873a\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"evilgaywitch\"},\"proof_type\":2,\"state\":1},\"seqno\":53,\"sig_id\":\"8857653a717e20ceba920b5fff9f9610978e5e67734c9be6510f1a4421f8db970f\",\"sig_type\":2},{\"ctime\":1478818020,\"curr\":\"5fb08a8d8b66683d20efb08814b3460a2d4a019e6e49d78641a4124c893cb480\",\"etime\":1983394020,\"prev\":\"01cd7115d142b21bb375e492c03eae3a58aee61ac4b874fa13ede4624c664f58\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"trynity\"},\"proof_type\":3,\"state\":1},\"seqno\":54,\"sig_id\":\"d364eb074467cb96252a4ec663773f6adeb6801c90ad085ec0bab8c49e60c26b0f\",\"sig_type\":2},{\"ctime\":1478818120,\"curr\":\"64da3a81e7bd4d57a2b9f802d3f058af243ac45409bc0a22381285451cbb395e\",\"etime\":1983394120,\"prev\":\"5fb08a8d8b66683d20efb08814b3460a2d4a019e6e49d78641a4124c893cb480\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"mirell.org\",\"protocol\":\"https:\"},\"proof_type\":1000,\"state\":1},\"seqno\":55,\"sig_id\":\"275d560b082c9ff6f0a505e3813f19d1c37b5e5b8ff114e6733dd9bba24bc9890f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"e7001eb4565c72e8e7f018930a5796d99fb1fa40c9183fcf2165bbfb3ccdcca2\",\"seqno\":56,\"sig_id\":\"dd7b3a371d7f921ace955342b5a70f9cebc55963ceeee9873a19d7f14f4742ef0f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1478819981,\"expire_in\":157680000,\"prev\":\"b26f3ab4a3e49156d4859d3a1081b78d0a1f7c7fec726bcd2f428e193ce1f73a\",\"seqno\":19,\"tag\":\"signature\"}",
      "sig_type": 3,
      "sig_version": 1,
      "ctime": 1478819981,
      "etime": 1636499981,
      "rtime": null,
      "sig_status": 0,
      "prev": "b26f3ab4a3e49156d4859d3a1081b78d0a1f7c7fec726bcd2f428e193ce1f73a",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": "7fd74ecea6ac7c073e7f602938c5ddc6bda4fae3"
    },
    {
      "seqno": 20,
      "payload_hash": "3093656c8b433b363a2ed21e71ffae84c800d1975af0821107fcaf52fac5560f",
      "sig_id": "6978bd92e7af03365f21ed02e430dee0ea5bcbcb338bbfee6ef8e035169e1b2b0f",
      "sig_id_short": "aXi9kuevAzZfIe0C5DDe4Opby8szi7_ubvjg",
      "kid": "0101482ecf3dce65743c1f48f1bb6bd1868b90e8e44c8b39752f1569d4dff150f4040a",
      "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.58\nComment: https://keybase.io/crypto\n\nyMNiAnicbVJrUFVVFL7IQy+KQU4PsGI6IKaDtM97n3udELWCG5XNOEQMBuexD5yQ\nc+E+UDQatGCEq/iIahBK6iJSlDFOIClgkDECWhkQFAoBab6CcagQKGhfRn/V/rP3\nXvv7vr3Wt9bZAG+Dvxf85nJ7c9X0qFdXa4nTkCgkrdxJSFYllzDtJDLQ/Ia2Ksju\nSMnQFMJEAJICkowUkoScAgRAKyKpUlABvAyAQAPIqBIkVRLwgIISxUoMJ5CsTCuQ\nYUUWyAqPQSIRSaianoZsWTZNd2BZXlV4BslI5EQZK/E04lUOUAINZVZRZE5SREYV\nEY2J6Va7h4GTk0Q7itKsOIYvKfPp/Q/+Xt6AZCCFZJVWZMSxPEPLpMpAlZQkjMbV\nQEkACCKGkaFECzxLqSTLCQqjqPgAVAYw83k75+VURCIgCxzHMhygZRZCINEqy5My\n5CiBFDxAO7LpYibyfA4UUdeIvEgCx3I0GXlsvfumijKSrNaM/zAYnfQwHLlZnsA2\nJKXcJadImq5g9zAjB9nsmlUnTCRGyg7Nw8VlApKlOVqIJND2LM2GUjQPguU5CPCK\nJLJsKMfjFUkLrMArIo1bysoMLyBcJEvTggplxLAMiRSKkVQVt48ReZGUaYC9UknE\n4BbRHjPsKFu3EiYKizrENKxp19J00eG0ISKvrTXZx+Dlb/DzXeAZK4O/MfDesD2y\nKNBQURz+jIm61Keaz5qdAfEfPzby2cFlM7rxgfQoc3/Hed8G94GHv9r4JBxwDY0Y\nQw0d3qdtKyyr3RFBqee4hDV/9w4Hl++v2xgDB1ddGe9vaWvZNxBv7Ok1byGNBYP9\nY0V9xMn904+WPOvXUyYF9ub9ZNk9nl+ekG25c+Pb4fVxfnB4MPz5401T5l4o+RYq\nE+/94x9bMRXyUdOHl0rr474PuFb27r6W0M2BJXNsdUFdY3hSjQ8IK2u+teFQRE67\nVT9lrj+TvXLLG2t+Hi/4406oV9322R+zfqhaH9fxUnf3pqqAy5kDN9c6b6cuPJZd\nnOSKqy4t/H16du+6B8c7Q35LXzLpXtuyI6GwePTkQ5bFqMI/4cqe6JKlf5X36e+/\nDP90ni9d3PRWrpI6AOVzIbuKQF5Q+6kjwS+2XUh//dWL7uQzu/hfKluPvp3x+Ctd\nq2+/2X5hcnnTzNxTsUm/rjhaGJNfNHHjA+is8G6Md8Ejm2/OmI2dPobnti0rGbw6\neniusjNkQenXByK629KrM5a6lz99vGPPou8ak2umCjOvT+y4NeZquf9a3cDUxImt\n4V+GBd3nau4hM22fMol7Az95oa3SiyoTD5+ONuVLS16zsFd9umoXOt6pvR48Nus2\nVxXEjA9NdokXRxLjYl3uJ8KOxXzB1Z4wadFlqxobWqM27Z6wWOjPa0KHzDnrGur/\nBeB6xFA=\n=PFW6\n-----END PGP MESSAGE-----",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"fingerprint\":\"7fd74ecea6ac7c073e7f602938c5ddc6bda4fae3\",\"host\":\"keybase.io\",\"key_id\":\"38c5ddc6bda4fae3\",\"kid\":\"0101482ecf3dce65743c1f48f1bb6bd1868b90e8e44c8b39752f1569d4dff150f4040a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"service\":{\"name\":\"facebook\",\"username\":\"00d4n1\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1480153639,\"expire_in\":157680000,\"prev\":\"3139597da320b5c479e4045339f8ce4541ed24bff5c34a7a1c30b6bf1e47c03a\",\"seqno\":20,\"tag\":\"signature\"}",
      "sig_type": 2,
      "sig_version": 1,
      "ctime": 1480153639,
      "etime": 1637833639,
      "rtime": null,
      "sig_status": 0,
      "prev": "3139597da320b5c479e4045339f8ce4541ed24bff5c34a7a1c30b6bf1e47c03a",
      "proof_id": "bf547c9775fe0ec431fc7d10",
      "proof_type": 8,
      "proof_text_check": "Verifying myself: I am 00dani on Keybase.io. aXi9kuevAzZfIe0C5DDe4Opby8szi7_ubvjgNRaeGys",
      "proof_text_full": "https://facebook.com/dialog/feed?app_id=270998186616745&link=https%3A%2F%2Fkeybase.io%2F00dani%2Fsigs%2FaXi9kuevAzZfIe0C5DDe4Opby8szi7_ubvjgNRaeGys&picture=https%3A%2F%2Fkeybase.io%2Fimages%2Ficons%2Fandroid-icon-192x192.png&name=Verifying%20myself%3A%20I%20am%2000dani%20on%20Keybase.io.%20aXi9kuevAzZfIe0C5DDe4Opby8szi7_ubvjgNRaeGys&redirect_uri=https%3A%2F%2Fkeybase.io%2F_%2Ffacebook_callback%3Fsig_id_med%3DaXi9kuevAzZfIe0C5DDe4Opby8szi7_ubvjgNRaeGys%26token%3DlgHZIGZlMWUwYzk2NjU0NjAzYzU4ODBiM2Y1NzFjODYyOTE5zlg5WinOAAP0gMDEILeIgcrx5lXwZvQCRXaxeWBDVOuhM1z1XMT9pDLMWnzU\n",
      "check_data_json": "{\"name\":\"facebook\",\"username\":\"00d4n1\"}",
      "remote_id": "",
      "api_url": "https://m.facebook.com/00d4n1/posts/10211524921262682",
      "human_url": "https://facebook.com/10211524920782670_10211524921262682",
      "proof_state": 1,
      "proof_status": 1,
      "retry_count": 541,
      "hard_fail_count": 0,
      "last_check": 1491821467,
      "last_success": 1491821467,
      "version": 1,
      "fingerprint": "7fd74ecea6ac7c073e7f602938c5ddc6bda4fae3"
    },
    {
      "seqno": 21,
      "payload_hash": "cacd72c7a4e8b4050d837206787c2ae6d988af7e15f117313acc42cd4ae3982e",
      "sig_id": "a0390abb88bd4f1df857aa28daac615822c2feb6d7ccf6e8ca913871293e8eb10f",
      "sig_id_short": "oDkKu4i9Tx34V6oo2qxhWCLC_rbXzPboypE4",
      "kid": "0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a",
      "sig": "hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgvO0RhtCQPaHyjQfACTCE+4HxBwKLJbRpFcPYRaUM15MKp3BheWxvYWTFA6F7ImJvZHkiOnsiZGV2aWNlIjp7ImlkIjoiY2YzNDAwOTkwNDA5MzhmZTE0ZTQ3ODMzZmQ0Mzc1MTgiLCJzdGF0dXMiOjIsInR5cGUiOiJkZXNrdG9wIn0sImtleSI6eyJlbGRlc3Rfa2lkIjoiMDEyMGJjZWQxMTg2ZDA5MDNkYTFmMjhkMDdjMDA5MzA4NGZiODFmMTA3MDI4YjI1YjQ2OTE1YzNkODQ1YTUwY2Q3OTMwYSIsImhvc3QiOiJrZXliYXNlLmlvIiwia2lkIjoiMDEyMGJjZWQxMTg2ZDA5MDNkYTFmMjhkMDdjMDA5MzA4NGZiODFmMTA3MDI4YjI1YjQ2OTE1YzNkODQ1YTUwY2Q3OTMwYSIsInVpZCI6ImZlMWUwYzk2NjU0NjAzYzU4ODBiM2Y1NzFjODYyOTE5IiwidXNlcm5hbWUiOiIwMGRhbmkifSwicmV2b2tlIjp7ImtpZHMiOlsiMDEyMGQzZTQ5NjAxMTkyNjNiM2QwOTM4MjkwNmUyMTAyMmU3OWE3ZTkyMDU3NjU0YzhjY2I4ZTZmZGM3YTU5M2EwMWEwYSIsIjAxMjExOTAwMzRhNjE5NDc3NTI1YTRjNGVjMjRhOWFhMzY5YThiZmI2YTI2YjllMjljYmVhNDZmY2IyNjQ1NTY5NDcwMGEiXX0sInR5cGUiOiJyZXZva2UiLCJ2ZXJzaW9uIjoxfSwiY2xpZW50Ijp7Im5hbWUiOiJrZXliYXNlLmlvIGdvIGNsaWVudCIsInZlcnNpb24iOiIxLjAuMTgifSwiY3RpbWUiOjE0ODMwMTc2MDUsImV4cGlyZV9pbiI6NTA0NTc2MDAwLCJtZXJrbGVfcm9vdCI6eyJjdGltZSI6MTQ4MzAxNzYwMSwiaGFzaCI6IjczMmYyOWUwMDE1MjViNjY0NmY1YWIwNjIzNzZjMjJjYjViYzQxZTNiNDY5ZDQ2MDJhY2I0OGY2NGUyYTVmNWI4MDhiYTAzMDEzNjMyOTQ3ZDg3NGU3ZDUwNGYzNDAwZTNjYmI5MWJkMDRlNWM3NTM4YWI2YzUzZjc3YTU0YjI4Iiwic2Vxbm8iOjc3NzkyMX0sInByZXYiOiIzMDkzNjU2YzhiNDMzYjM2M2EyZWQyMWU3MWZmYWU4NGM4MDBkMTk3NWFmMDgyMTEwN2ZjYWY1MmZhYzU1NjBmIiwic2Vxbm8iOjIxLCJ0YWciOiJzaWduYXR1cmUifaNzaWfEQJZ8GYRBM3tlYa50OeeupmwJYP+tGQt0pAmu4RuD549DgFzHD+Xq6eProU6LJCfe2aRx1WCSdS5SDA5+TLDQcQmoc2lnX3R5cGUgpGhhc2iCpHR5cGUIpXZhbHVlxCANZYrU0qp30ylZcJaVkL5FvUILqe8I8MCN7s1MgnoSs6N0YWfNAgKndmVyc2lvbgE=",
      "payload_json": "{\"body\":{\"device\":{\"id\":\"cf340099040938fe14e47833fd437518\",\"status\":2,\"type\":\"desktop\"},\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"revoke\":{\"kids\":[\"0120d3e4960119263b3d09382906e21022e79a7e92057654c8ccb8e6fdc7a593a01a0a\",\"0121190034a619477525a4c4ec24a9aa369a8bfb6a26b9e29cbea46fcb26455694700a\"]},\"type\":\"revoke\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.18\"},\"ctime\":1483017605,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1483017601,\"hash\":\"732f29e001525b6646f5ab062376c22cb5bc41e3b469d4602acb48f64e2a5f5b808ba03013632947d874e7d504f3400e3cbb91bd04e5c7538ab6c53f77a54b28\",\"seqno\":777921},\"prev\":\"3093656c8b433b363a2ed21e71ffae84c800d1975af0821107fcaf52fac5560f\",\"seqno\":21,\"tag\":\"signature\"}",
      "sig_type": 5,
      "sig_version": 1,
      "ctime": 1483017605,
      "etime": 1987593605,
      "rtime": null,
      "sig_status": 0,
      "prev": "3093656c8b433b363a2ed21e71ffae84c800d1975af0821107fcaf52fac5560f",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 22,
      "payload_hash": "00d20ae5e99a1c034735ea6cd32d6a4c590ac1ef2205a74cb21e3e2c7566ad8f",
      "sig_id": "b6b5f4bc0eb2b74c50c624fbf5b53ec7af68aa42ce81c49a7f927e3bbcc1970c0f",
      "sig_id_short": "trX0vA6yt0xQxiT79bU-x69oqkLOgcSaf5J-",
      "kid": "0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a",
      "sig": "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",
      "payload_json": "{\"body\":{\"device\":{\"id\":\"21505378859b87ae4107dcba5b4e2f18\",\"kid\":\"012014c8419a2ee47e8fa115edb6f1b3b6d8d5b7b739bd62cafe2803bda2356a27450a\",\"name\":\"reaper-redux\",\"status\":1,\"type\":\"desktop\"},\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"sibkey\":{\"kid\":\"012014c8419a2ee47e8fa115edb6f1b3b6d8d5b7b739bd62cafe2803bda2356a27450a\",\"reverse_sig\":\"hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgFMhBmi7kfo+hFe228bO22NW3tzm9Ysr+KAO9ojVqJ0UKp3BheWxvYWTFA817ImJvZHkiOnsiZGV2aWNlIjp7ImlkIjoiMjE1MDUzNzg4NTliODdhZTQxMDdkY2JhNWI0ZTJmMTgiLCJraWQiOiIwMTIwMTRjODQxOWEyZWU0N2U4ZmExMTVlZGI2ZjFiM2I2ZDhkNWI3YjczOWJkNjJjYWZlMjgwM2JkYTIzNTZhMjc0NTBhIiwibmFtZSI6InJlYXBlci1yZWR1eCIsInN0YXR1cyI6MSwidHlwZSI6ImRlc2t0b3AifSwia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwidWlkIjoiZmUxZTBjOTY2NTQ2MDNjNTg4MGIzZjU3MWM4NjI5MTkiLCJ1c2VybmFtZSI6IjAwZGFuaSJ9LCJzaWJrZXkiOnsia2lkIjoiMDEyMDE0Yzg0MTlhMmVlNDdlOGZhMTE1ZWRiNmYxYjNiNmQ4ZDViN2I3MzliZDYyY2FmZTI4MDNiZGEyMzU2YTI3NDUwYSIsInJldmVyc2Vfc2lnIjpudWxsfSwidHlwZSI6InNpYmtleSIsInZlcnNpb24iOjF9LCJjbGllbnQiOnsibmFtZSI6ImtleWJhc2UuaW8gZ28gY2xpZW50IiwidmVyc2lvbiI6IjEuMC4xOCJ9LCJjdGltZSI6MTQ4MzAxODYyNCwiZXhwaXJlX2luIjo1MDQ1NzYwMDAsIm1lcmtsZV9yb290Ijp7ImN0aW1lIjoxNDgzMDE4NTk0LCJoYXNoIjoiODFlYWMzNmM2YTMzNDVkNzkzNjI4OThkYTAzY2M0MTMwY2E2MDQ3NjE4ZDJjY2EyZWMwMGFmNTAyNDE1OTAxNWI2ZDliMmQzMDUwMzllODQ2ZDk4YWVkNDI3MjkwYmJkMmFkOTQ4MmRjMTRmNDlmYmE1ODg5Y2IzNTNkMmI3ZjQiLCJzZXFubyI6Nzc3OTUxfSwicHJldiI6ImNhY2Q3MmM3YTRlOGI0MDUwZDgzNzIwNjc4N2MyYWU2ZDk4OGFmN2UxNWYxMTczMTNhY2M0MmNkNGFlMzk4MmUiLCJzZXFubyI6MjIsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RAHpor2se5Lfbls9rBdiiw84MYi40KKoqLeBzk1Ps0xasx+lBUJELeDWNnuAsuqKOeBRlVfQtvGWJp2q4fBhTsB6hzaWdfdHlwZSCkaGFzaIKkdHlwZQildmFsdWXEIJSRg7WWJCagZkZJIDJezMpoMuX4Lb6+6y+POEwCc4Z5o3RhZ80CAqd2ZXJzaW9uAQ==\"},\"type\":\"sibkey\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.18\"},\"ctime\":1483018624,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1483018594,\"hash\":\"81eac36c6a3345d79362898da03cc4130ca6047618d2cca2ec00af5024159015b6d9b2d305039e846d98aed427290bbd2ad9482dc14f49fba5889cb353d2b7f4\",\"seqno\":777951},\"prev\":\"cacd72c7a4e8b4050d837206787c2ae6d988af7e15f117313acc42cd4ae3982e\",\"seqno\":22,\"tag\":\"signature\"}",
      "sig_type": 1,
      "sig_version": 1,
      "ctime": 1483018624,
      "etime": 1987594624,
      "rtime": null,
      "sig_status": 0,
      "prev": "cacd72c7a4e8b4050d837206787c2ae6d988af7e15f117313acc42cd4ae3982e",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 23,
      "payload_hash": "7dca0290cb828c3c17c689af9a639595a9fc052a1eb3d7c4b687ed74f58594f1",
      "sig_id": "11d481c59453fafc5d66d24270d50b2531e7debc273573853b34764517aedd490f",
      "sig_id_short": "EdSBxZRT-vxdZtJCcNULJTHn3rwnNXOFOzR2",
      "kid": "012014c8419a2ee47e8fa115edb6f1b3b6d8d5b7b739bd62cafe2803bda2356a27450a",
      "sig": "hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgFMhBmi7kfo+hFe228bO22NW3tzm9Ysr+KAO9ojVqJ0UKp3BheWxvYWTFA+l7ImJvZHkiOnsiZGV2aWNlIjp7ImlkIjoiMjE1MDUzNzg4NTliODdhZTQxMDdkY2JhNWI0ZTJmMTgiLCJraWQiOiIwMTIxMmYxNDE1ODlkNDliNDA3YzhmOWE0M2I1Mzk3NDI4MGQyYTU1M2Y1YTg1ZGNjYTU0NmE3OTM3NmZhNzUwMjk2MjBhIiwic3RhdHVzIjoxfSwia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwMTRjODQxOWEyZWU0N2U4ZmExMTVlZGI2ZjFiM2I2ZDhkNWI3YjczOWJkNjJjYWZlMjgwM2JkYTIzNTZhMjc0NTBhIiwidWlkIjoiZmUxZTBjOTY2NTQ2MDNjNTg4MGIzZjU3MWM4NjI5MTkiLCJ1c2VybmFtZSI6IjAwZGFuaSJ9LCJzdWJrZXkiOnsia2lkIjoiMDEyMTJmMTQxNTg5ZDQ5YjQwN2M4ZjlhNDNiNTM5NzQyODBkMmE1NTNmNWE4NWRjY2E1NDZhNzkzNzZmYTc1MDI5NjIwYSIsInBhcmVudF9raWQiOiIwMTIwMTRjODQxOWEyZWU0N2U4ZmExMTVlZGI2ZjFiM2I2ZDhkNWI3YjczOWJkNjJjYWZlMjgwM2JkYTIzNTZhMjc0NTBhIn0sInR5cGUiOiJzdWJrZXkiLCJ2ZXJzaW9uIjoxfSwiY2xpZW50Ijp7Im5hbWUiOiJrZXliYXNlLmlvIGdvIGNsaWVudCIsInZlcnNpb24iOiIxLjAuMTgifSwiY3RpbWUiOjE0ODMwMTg2MzEsImV4cGlyZV9pbiI6NTA0NTc2MDAwLCJtZXJrbGVfcm9vdCI6eyJjdGltZSI6MTQ4MzAxODU5NCwiaGFzaCI6IjgxZWFjMzZjNmEzMzQ1ZDc5MzYyODk4ZGEwM2NjNDEzMGNhNjA0NzYxOGQyY2NhMmVjMDBhZjUwMjQxNTkwMTViNmQ5YjJkMzA1MDM5ZTg0NmQ5OGFlZDQyNzI5MGJiZDJhZDk0ODJkYzE0ZjQ5ZmJhNTg4OWNiMzUzZDJiN2Y0Iiwic2Vxbm8iOjc3Nzk1MX0sInByZXYiOiIwMGQyMGFlNWU5OWExYzAzNDczNWVhNmNkMzJkNmE0YzU5MGFjMWVmMjIwNWE3NGNiMjFlM2UyYzc1NjZhZDhmIiwic2Vxbm8iOjIzLCJ0YWciOiJzaWduYXR1cmUifaNzaWfEQBTHyPR9cr6uSTLPlw0aLLhGS480URsOOkbbtyP+JEQAZvPJZqZ/7xaJhzA54FzguHAAx7V0rnmvwlHOU9WbWAmoc2lnX3R5cGUgpGhhc2iCpHR5cGUIpXZhbHVlxCBAE4p+msq9QpCtXwUmU4m78/RyILK6EU1dd6HBdu6Rx6N0YWfNAgKndmVyc2lvbgE=",
      "payload_json": "{\"body\":{\"device\":{\"id\":\"21505378859b87ae4107dcba5b4e2f18\",\"kid\":\"01212f141589d49b407c8f9a43b53974280d2a553f5a85dcca546a79376fa75029620a\",\"status\":1},\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"012014c8419a2ee47e8fa115edb6f1b3b6d8d5b7b739bd62cafe2803bda2356a27450a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"subkey\":{\"kid\":\"01212f141589d49b407c8f9a43b53974280d2a553f5a85dcca546a79376fa75029620a\",\"parent_kid\":\"012014c8419a2ee47e8fa115edb6f1b3b6d8d5b7b739bd62cafe2803bda2356a27450a\"},\"type\":\"subkey\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.18\"},\"ctime\":1483018631,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1483018594,\"hash\":\"81eac36c6a3345d79362898da03cc4130ca6047618d2cca2ec00af5024159015b6d9b2d305039e846d98aed427290bbd2ad9482dc14f49fba5889cb353d2b7f4\",\"seqno\":777951},\"prev\":\"00d20ae5e99a1c034735ea6cd32d6a4c590ac1ef2205a74cb21e3e2c7566ad8f\",\"seqno\":23,\"tag\":\"signature\"}",
      "sig_type": 1,
      "sig_version": 1,
      "ctime": 1483018631,
      "etime": 1987594631,
      "rtime": null,
      "sig_status": 0,
      "prev": "00d20ae5e99a1c034735ea6cd32d6a4c590ac1ef2205a74cb21e3e2c7566ad8f",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 24,
      "payload_hash": "2a1ef9a9779044e942883375c10a00f97d528ceed9d9c7319884408adf47e5f7",
      "sig_id": "209cbe430d5b9e599bf58649b197105c1e09aa7fba5a220e1d412115b8f4c40a0f",
      "sig_id_short": "IJy-Qw1bnlmb9YZJsZcQXB4Jqn-6WiIOHUEh",
      "kid": "0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a",
      "sig": "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",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"track\":{\"basics\":{\"id_version\":11,\"last_id_change\":1480593334,\"username\":\"lissm\"},\"id\":\"ab1cc2bf81d47e9393666b76c563da19\",\"key\":{\"key_fingerprint\":\"c8196dade970a46a36e4ffff924b32adc4979458\",\"kid\":\"01016adaf6abc2bf6f52109d9f59dfd0760f163886322d22eab009b4acaf894b4ff80a\"},\"pgp_keys\":[{\"key_fingerprint\":\"c8196dade970a46a36e4ffff924b32adc4979458\",\"kid\":\"01016adaf6abc2bf6f52109d9f59dfd0760f163886322d22eab009b4acaf894b4ff80a\"}],\"remote_proofs\":[{\"ctime\":1427434512,\"curr\":\"01e748cac3a2faea5cec60d4505b0e8c0d22026aa48a96406ceedbcac448afa1\",\"etime\":1585114512,\"prev\":\"055b51e3754bc237b528df4ac3233f2c8fe3bba2ccd804f47889879e7e7f885a\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"lissmccabe\"},\"proof_type\":2,\"state\":1},\"sig_id\":\"606824d9c7cc823362d23ff5451a939bf7810e3a3c151647127c7403d0c194a70f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"01e748cac3a2faea5cec60d4505b0e8c0d22026aa48a96406ceedbcac448afa1\",\"seqno\":2,\"sig_id\":\"606824d9c7cc823362d23ff5451a939bf7810e3a3c151647127c7403d0c194a70f\"}},\"type\":\"track\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.18\"},\"ctime\":1483939592,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1483939591,\"hash\":\"6161ab1696f4e6549ccf0dbf341b8422f4b9df969380e0cc5f335c670c4efb9747bfe577d4d9ea7f1a6398437446a58fdaa12cdda8006f7714498ea0335a3821\",\"seqno\":791711},\"prev\":\"7dca0290cb828c3c17c689af9a639595a9fc052a1eb3d7c4b687ed74f58594f1\",\"seqno\":24,\"tag\":\"signature\"}",
      "sig_type": 3,
      "sig_version": 1,
      "ctime": 1483939592,
      "etime": 1988515592,
      "rtime": null,
      "sig_status": 0,
      "prev": "7dca0290cb828c3c17c689af9a639595a9fc052a1eb3d7c4b687ed74f58594f1",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 25,
      "payload_hash": "3d4eb39733c7fa77d3e1e23c9669b341c600d4f1df68caee5caa8c6672825641",
      "sig_id": "ab8abe0ff451d744861106fde36fb20084512b5f6f9163601b0464a02c6d182c0f",
      "sig_id_short": "q4q-D_RR10SGEQb942-yAIRRK19vkWNgGwRk",
      "kid": "0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a",
      "sig": "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",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"track\":{\"basics\":{\"id_version\":5,\"last_id_change\":1483961005,\"username\":\"jezgra\"},\"id\":\"74951cce7cd398eb1dec2c3aa8728a19\",\"key\":{\"key_fingerprint\":\"8438a7e08be8e5b6e6f1e1b8b6bc602afef0da80\",\"kid\":\"0101a02e9bde97429647b4aafda6913da589648a5522161811f53f57bc73f3642a1d0a\"},\"pgp_keys\":[{\"key_fingerprint\":\"8438a7e08be8e5b6e6f1e1b8b6bc602afef0da80\",\"kid\":\"0101a02e9bde97429647b4aafda6913da589648a5522161811f53f57bc73f3642a1d0a\"}],\"remote_proofs\":[{\"ctime\":1483958546,\"curr\":\"16fa217800526b36e93bcd2234924970d54a00fc79539a3768f3dfe77b6db4b7\",\"etime\":1641638546,\"prev\":\"ece0b7cd7a70c12d6a879c3bf952c6e685bc5d5ceea287779ee39dd2b87e2f3e\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"reddit\",\"username\":\"tehlivingpencil\"},\"proof_type\":4,\"state\":0},\"sig_id\":\"9f619eae4b5b6c51ff133a4198ec86e3dcc9e733e74d199bfcdf79322399a0680f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"16fa217800526b36e93bcd2234924970d54a00fc79539a3768f3dfe77b6db4b7\",\"seqno\":2,\"sig_id\":\"9f619eae4b5b6c51ff133a4198ec86e3dcc9e733e74d199bfcdf79322399a0680f\"}},\"type\":\"track\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.18\"},\"ctime\":1484144394,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1484144371,\"hash\":\"c0de492d5af004b2f5f1cf427db2f8f361a2e75463553d4f23e05f0979aa7566fc2db445a81e51f1e00ea2ee3491ceb634baaf4bd6f2ec44451385bfd690be72\",\"seqno\":795186},\"prev\":\"2a1ef9a9779044e942883375c10a00f97d528ceed9d9c7319884408adf47e5f7\",\"seqno\":25,\"tag\":\"signature\"}",
      "sig_type": 3,
      "sig_version": 1,
      "ctime": 1484144394,
      "etime": 1988720394,
      "rtime": null,
      "sig_status": 0,
      "prev": "2a1ef9a9779044e942883375c10a00f97d528ceed9d9c7319884408adf47e5f7",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 26,
      "payload_hash": "cf4e5f04bdf478f0af389105274afb81406a84376c80bc18949fceebb56203a0",
      "sig_id": "fad29f6d5234d5349f34dc66cc3513399b7fba663b09c4a25a4806b07b11c7040f",
      "sig_id_short": "-tKfbVI01TSfNNxmzDUTOZt_umY7CcSiWkgG",
      "kid": "012014c8419a2ee47e8fa115edb6f1b3b6d8d5b7b739bd62cafe2803bda2356a27450a",
      "sig": "hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgFMhBmi7kfo+hFe228bO22NW3tzm9Ysr+KAO9ojVqJ0UKp3BheWxvYWTFAwh7ImJvZHkiOnsiY3J5cHRvY3VycmVuY3kiOnsiYWRkcmVzcyI6InQxWHNqMmJoVjY3Q2hHNHhQWkw0aXBkMmhYZFlYcjkzZm1tIiwidHlwZSI6InpjYXNoLnQifSwia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwMTRjODQxOWEyZWU0N2U4ZmExMTVlZGI2ZjFiM2I2ZDhkNWI3YjczOWJkNjJjYWZlMjgwM2JkYTIzNTZhMjc0NTBhIiwidWlkIjoiZmUxZTBjOTY2NTQ2MDNjNTg4MGIzZjU3MWM4NjI5MTkiLCJ1c2VybmFtZSI6IjAwZGFuaSJ9LCJ0eXBlIjoiY3J5cHRvY3VycmVuY3kiLCJ2ZXJzaW9uIjoxfSwiY2xpZW50Ijp7Im5hbWUiOiJrZXliYXNlLmlvIGdvIGNsaWVudCIsInZlcnNpb24iOiIxLjAuMTgifSwiY3RpbWUiOjE0ODQzNzg1MjksImV4cGlyZV9pbiI6NTA0NTc2MDAwLCJtZXJrbGVfcm9vdCI6eyJjdGltZSI6MTQ4NDM3NzcxMCwiaGFzaCI6ImU4MmU2Yzc2Y2I4ZDYzODA5OWM4YzU3ZGNkOTFhMjYzNWZjZDFmYmYxYjdhNzE5MTg1ZjQ4MDFhZDExZTBlZjVkOTYwMDVhYWFiZjIwN2U5NzJiNzcyNjE2Nzk1NmQ5YzNlOTU1ZjYyMzI0OTg3ZGQwMTNjMTZlYmZlNzRiMjg2Iiwic2Vxbm8iOjc5OTIyOX0sInByZXYiOiIzZDRlYjM5NzMzYzdmYTc3ZDNlMWUyM2M5NjY5YjM0MWM2MDBkNGYxZGY2OGNhZWU1Y2FhOGM2NjcyODI1NjQxIiwic2Vxbm8iOjI2LCJ0YWciOiJzaWduYXR1cmUifaNzaWfEQHMe9K5lAUT/NXqI6QoNyVdAXHCgtmCYwpK+HLcYK8ec4++JgCqNptwuJhY9J0U5E6OJow/HGoDHAt2utV77Fgmoc2lnX3R5cGUgpGhhc2iCpHR5cGUIpXZhbHVlxCCSq2vYAD1MTFEkGiK45iljFFeiYeGu2fkZ4AV3+HqexKN0YWfNAgKndmVyc2lvbgE=",
      "payload_json": "{\"body\":{\"cryptocurrency\":{\"address\":\"t1Xsj2bhV67ChG4xPZL4ipd2hXdYXr93fmm\",\"type\":\"zcash.t\"},\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"012014c8419a2ee47e8fa115edb6f1b3b6d8d5b7b739bd62cafe2803bda2356a27450a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"type\":\"cryptocurrency\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.18\"},\"ctime\":1484378529,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1484377710,\"hash\":\"e82e6c76cb8d638099c8c57dcd91a2635fcd1fbf1b7a719185f4801ad11e0ef5d96005aaabf207e972b7726167956d9c3e955f62324987dd013c16ebfe74b286\",\"seqno\":799229},\"prev\":\"3d4eb39733c7fa77d3e1e23c9669b341c600d4f1df68caee5caa8c6672825641\",\"seqno\":26,\"tag\":\"signature\"}",
      "sig_type": 6,
      "sig_version": 1,
      "ctime": 1484378529,
      "etime": 1988954529,
      "rtime": 1487803070,
      "sig_status": 1,
      "prev": "3d4eb39733c7fa77d3e1e23c9669b341c600d4f1df68caee5caa8c6672825641",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 27,
      "payload_hash": "553a05af05b5c23dfe622f1c5a73e0ae76346ebf8c8b4da324e8f524c7e4be34",
      "sig_id": "737bfb5c163bfbdc5c6acb679d5f6d3acdfa6eb5a17679dba0f4e9fe8e8e9cd40f",
      "sig_id_short": "c3v7XBY7-9xcastnnV9tOs36brWhdnnboPTp",
      "kid": "012014c8419a2ee47e8fa115edb6f1b3b6d8d5b7b739bd62cafe2803bda2356a27450a",
      "sig": "hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgFMhBmi7kfo+hFe228bO22NW3tzm9Ysr+KAO9ojVqJ0UKp3BheWxvYWTFAwd7ImJvZHkiOnsiY3J5cHRvY3VycmVuY3kiOnsiYWRkcmVzcyI6IjFKVmZNWTN0cFJ2THZDeFdpVnpZZlBQQ3Q0dVFKSlVSZVUiLCJ0eXBlIjoiYml0Y29pbiJ9LCJrZXkiOnsiZWxkZXN0X2tpZCI6IjAxMjBiY2VkMTE4NmQwOTAzZGExZjI4ZDA3YzAwOTMwODRmYjgxZjEwNzAyOGIyNWI0NjkxNWMzZDg0NWE1MGNkNzkzMGEiLCJob3N0Ijoia2V5YmFzZS5pbyIsImtpZCI6IjAxMjAxNGM4NDE5YTJlZTQ3ZThmYTExNWVkYjZmMWIzYjZkOGQ1YjdiNzM5YmQ2MmNhZmUyODAzYmRhMjM1NmEyNzQ1MGEiLCJ1aWQiOiJmZTFlMGM5NjY1NDYwM2M1ODgwYjNmNTcxYzg2MjkxOSIsInVzZXJuYW1lIjoiMDBkYW5pIn0sInR5cGUiOiJjcnlwdG9jdXJyZW5jeSIsInZlcnNpb24iOjF9LCJjbGllbnQiOnsibmFtZSI6ImtleWJhc2UuaW8gZ28gY2xpZW50IiwidmVyc2lvbiI6IjEuMC4xOCJ9LCJjdGltZSI6MTQ4NDQ2NTMyNywiZXhwaXJlX2luIjo1MDQ1NzYwMDAsIm1lcmtsZV9yb290Ijp7ImN0aW1lIjoxNDg0NDY1MjkzLCJoYXNoIjoiZDJmYjIzOTdmNGJhMDA5ZDY1OGZkNzkxMTllNjM3YzcxODg0NTZkYmU0ZGUxZGYwMmE5YWZlODQ4MDY3NWFlMTgwNWY2NWU5N2QxMGUyMWI2NzdjNGQ5MTliMTVhMWE2MzRkMDQzMWMzZDY0M2M5NTcwMmI2YWY4NWQ3NjNjZGUiLCJzZXFubyI6ODAwNDM3fSwicHJldiI6ImNmNGU1ZjA0YmRmNDc4ZjBhZjM4OTEwNTI3NGFmYjgxNDA2YTg0Mzc2YzgwYmMxODk0OWZjZWViYjU2MjAzYTAiLCJzZXFubyI6MjcsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RAKmx4r5oUakcV9axEEqcVihwlVNY0DtscRpunOkGu/BzVS1H4yql3YGgU4svOar/BLJjFxbnh/KgOCb0XgrhDCqhzaWdfdHlwZSCkaGFzaIKkdHlwZQildmFsdWXEIPcWwDXJmejA7bSWIfSRckgLgdEPgbmyGGpYOpM6gWsxo3RhZ80CAqd2ZXJzaW9uAQ==",
      "payload_json": "{\"body\":{\"cryptocurrency\":{\"address\":\"1JVfMY3tpRvLvCxWiVzYfPPCt4uQJJUReU\",\"type\":\"bitcoin\"},\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"012014c8419a2ee47e8fa115edb6f1b3b6d8d5b7b739bd62cafe2803bda2356a27450a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"type\":\"cryptocurrency\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.18\"},\"ctime\":1484465327,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1484465293,\"hash\":\"d2fb2397f4ba009d658fd79119e637c7188456dbe4de1df02a9afe8480675ae1805f65e97d10e21b677c4d919b15a1a634d0431c3d643c95702b6af85d763cde\",\"seqno\":800437},\"prev\":\"cf4e5f04bdf478f0af389105274afb81406a84376c80bc18949fceebb56203a0\",\"seqno\":27,\"tag\":\"signature\"}",
      "sig_type": 6,
      "sig_version": 1,
      "ctime": 1484465327,
      "etime": 1989041327,
      "rtime": null,
      "sig_status": 0,
      "prev": "cf4e5f04bdf478f0af389105274afb81406a84376c80bc18949fceebb56203a0",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 28,
      "payload_hash": "e50c346693b3fdecd31326d226328dbb7eb42bc121d3607d045543534a0c0b18",
      "sig_id": "cd513767f7402cc5c2e2a6c0a024750d96c92509f0d0b11d6a0855c15c4bb0fd0f",
      "sig_id_short": "zVE3Z_dALMXC4qbAoCR1DZbJJQnw0LEdaghV",
      "kid": "012014c8419a2ee47e8fa115edb6f1b3b6d8d5b7b739bd62cafe2803bda2356a27450a",
      "sig": "hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgFMhBmi7kfo+hFe228bO22NW3tzm9Ysr+KAO9ojVqJ0UKp3BheWxvYWTFCyZ7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwMTRjODQxOWEyZWU0N2U4ZmExMTVlZGI2ZjFiM2I2ZDhkNWI3YjczOWJkNjJjYWZlMjgwM2JkYTIzNTZhMjc0NTBhIiwidWlkIjoiZmUxZTBjOTY2NTQ2MDNjNTg4MGIzZjU3MWM4NjI5MTkiLCJ1c2VybmFtZSI6IjAwZGFuaSJ9LCJzaWJrZXkiOnsiZmluZ2VycHJpbnQiOiIzMTBiYzY4OGQ1Y2YzNDhhYTczZGU4Y2M5MTMxNDI3OTQ4NzhiMDZlIiwiZnVsbF9oYXNoIjoiMTQ1M2M0ODQ4ZTQ1Njk3MTljZjBiZGVkY2FjOGEwOTcyZGNmMmU2ZTViMDdmMzNhMDAwNjcyNTVkMTJkZjgyNSIsImtleV9pZCI6IjkxMzE0Mjc5NDg3OEIwNkUiLCJraWQiOiIwMTAxMWQyM2JjZmMwYzQyYjRkNTRhZWM4MTJhYTkzODVmYjk2NTZhZThlZDBhNTVlZDNhNmRiNzBlN2M3YThmY2EwZTBhIiwicmV2ZXJzZV9zaWciOiItLS0tLUJFR0lOIFBHUCBNRVNTQUdFLS0tLS1cblZlcnNpb246IEtleWJhc2UgR28gMS4wLjE4ICh3aW5kb3dzKVxuQ29tbWVudDogaHR0cHM6Ly9rZXliYXNlLmlvL2Rvd25sb2FkXG5cbnhBMERBQW9Ca1RGQ2VVaDRzRzRCeStGMEFPSUFBQUFBNlhzaVltOWtlU0k2ZXlKclpYa2lPbnNpWld4a1pYTjBcblgydHBaQ0k2SWpBeE1qQmlZMlZrTVRFNE5tUXdPVEF6WkdFeFpqSTRaREEzWXpBd09UTXdPRFJtWWpneFpqRXdcbk56QXlPR0l5TldJME5qa3hOV016WkRnME5XRTFNR05rTnprek1HRWlMQ0pvYjNOMElqb2lhMlY1WW1GelpTNXBcbmJ5SXNJbXRwWkNJNklqQXhNakF4TkdNNE5ERTVZVEpsWlRRM1pUaG1ZVEV4TldWa1lqWm1NV0l6WWpaa09HUTFcbllqZGlOek01WW1RMk1tTmhabVV5T0RBelltUmhNak0xTm1FeU56UTFNR0VpTENKMWFXUWlPaUptWlRGbE1HTTVcbk5qWTFORFl3TTJNMU9EZ3dZak5tTlRjeFl6ZzJNamt4T1NJc0luVnpaWEp1WVcxbElqb2lNREJrWVc1cEluMHNcbkluTnBZbXRsZVNJNmV5Sm1hVzVuWlhKd2NtbHVkQ0k2SWpNeE1HSmpOamc0WkRWalpqTTBPR0ZoTnpOa1pUaGpcbll6a3hNekUwTWpjNU5EZzNPR0l3Tm1VaUxDSm1kV3hzWDJoaGMyZ2lPaUl4TkRVell6UTRORGhsTkRVMk9UY3hcbk9XTm1NR0prWldSallXTTRZVEE1TnpKa1kyWXlaVFpsTldJd04yWXpNMkV3TURBMk56STFOV1F4TW1SbU9ESTFcbklpd2lhMlY1WDJsa0lqb2lPVEV6TVRReU56azBPRGM0UWpBMlJTSXNJbXRwWkNJNklqQXhNREV4WkRJelltTm1cbll6QmpOREppTkdRMU5HRmxZemd4TW1GaE9UTTROV1ppT1RZMU5tRmxPR1ZrTUdFMU5XVmtNMkUyWkdJM01HVTNcbll6ZGhPR1pqWVRCbDZEQmhJaXdpY21WMlpYSnpaVjl6YVdjaU9tNTFiR3g5TENKMGVYQmxJam9pYzJsaWEyVjVcbklpd2lkbVZ5YzJsdmJpSTZNWDBzSW1Oc2FXVnVkQ0k2ZXlKdVlXMWxJam9pYTJWNVltRnpaUzVwYnlCbmJ5QmpcbmJHbGxiblFpTENKMlpYSnphVzl1SWpvaU1TNHdMakU0SW4wc0ltTjBhVzFsSWpveE5EZzBORFkyTVRReExDSmxcbmVIQnBjbVZmYVc0aU9qVXdORFUzTmpBd01Dd2liV1Z5YTJ4bFgzSnZiM1FpT25zaVkzUnBiV1VpT2pFME9EUTBcbk5qWXdORElzSW1oaGMyZ2lPaUppT0RSaU16ZzRPREUxWlRSaE9XVmxOREpoTURRMVpUTXpaak13TURrMlpHSTFcbk5qYzNabVEyTnpoak5UZ3lZak5oTjJNMk1qZ3haakpsTkdMbk56WTJNV00zTldNeVpHWmhOelk1TVdJNU56Smhcbk5EWmlZMkV4TXpGbVlqRmxaRFF3TkRjMk9URmlObVJtWXpCbU5UYzRZamd3WVRoallXSTRNamxpTnpJMk9UY2lcbkxDSnpaWEZ1YnlJNk9EQXdORFE1ZlN3aWNISmxkaUk2SWpVMU0yRXdOV0ZtTURWaU5XTXlNMlJtWlRZeU1tWXhcbll6VmhOek5sTUdGbE56YmxNelEyWldKbU9HTTRZalJrWVRNeU5HVTRaalV5TkdNM1pUUmlaVE0wSWl6a0luTmxcbmNXNXZJam95T0N3aWRHRm5JdU02SW5OcFoyNWhkT0oxY21VaTRIMEF3c0JjQkFBQkNnQVFCUUpZZXlmZENSQ1Jcbk1VSjVTSGl3YmdBQVEzQUlBQXNrbUJ3aDRPazRVWkI5YVFqM3ZBZFFzbDU0S0tidlEybmg2M3N4cnB6VkRSRjhcbkgvazNjUndQQlg4a2tQTDRNVkJjQmVwbUJtZGJqNjJlWWJZWG1XaDdPajB4bXpCY0E1SDNIb0ZtbkV0Z1N1aUdcbkVadXpWeTVaZU02UnFZbG0xNEZVSWEwMmcvaFVZZjBxYU9mVTQ1NHhvYUNiTGF0UUdHeVBwZnBXRXVaS29Bc01cbk8wVWxYdWxyN0Y3V3psUlN1S2F0NlF3SUIwZCt5RW4xKzFoK0VnTXhSMktIemxVU0pvUEkyLzRmUjhTVFhqYXNcblZvTTdBdVIrTEl5djNFMGwrQmJlSFFaMndOMFoyekdYN2tEYTNFUnhjYlNPTXRrblRWL2pWeVdBUVdxY0o4b1ZcbkpKb3VJOTVqRFp4TFZwemd2aDRFMWJoaDBkSXcvbysxdTlobDRuaz1cbj1TajdHXG4tLS0tLUVORCBQR1AgTUVTU0FHRS0tLS0tXG4ifSwidHlwZSI6InNpYmtleSIsInZlcnNpb24iOjF9LCJjbGllbnQiOnsibmFtZSI6ImtleWJhc2UuaW8gZ28gY2xpZW50IiwidmVyc2lvbiI6IjEuMC4xOCJ9LCJjdGltZSI6MTQ4NDQ2NjE0MSwiZXhwaXJlX2luIjo1MDQ1NzYwMDAsIm1lcmtsZV9yb290Ijp7ImN0aW1lIjoxNDg0NDY2MDQyLCJoYXNoIjoiYjg0YjM4ODgxNWU0YTllZTQyYTA0NWUzM2YzMDA5NmRiNTY3N2ZkNjc4YzU4MmIzYTdjNjI4MWYyZTRiNzY2MWM3NWMyZGZhNzY5MWI5NzJhNDZiY2ExMzFmYjFlZDQwNDc2OTFiNmRmYzBmNTc4YjgwYThjYWI4MjliNzI2OTciLCJzZXFubyI6ODAwNDQ5fSwicHJldiI6IjU1M2EwNWFmMDViNWMyM2RmZTYyMmYxYzVhNzNlMGFlNzYzNDZlYmY4YzhiNGRhMzI0ZThmNTI0YzdlNGJlMzQiLCJzZXFubyI6MjgsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RAfjXycFHkfLG2A1l5VuGTKgnFphRAd3LXKLLiewDtkxeCWq+G6Y3/eV8SiFVIoq38d36NOhC9u4Sdk3I4hfS4DKhzaWdfdHlwZSCkaGFzaIKkdHlwZQildmFsdWXEIHz5XXLzd2850efCyxxpF9jVpWSkbMqmQak8yJ8gngCMo3RhZ80CAqd2ZXJzaW9uAQ==",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"012014c8419a2ee47e8fa115edb6f1b3b6d8d5b7b739bd62cafe2803bda2356a27450a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"sibkey\":{\"fingerprint\":\"310bc688d5cf348aa73de8cc913142794878b06e\",\"full_hash\":\"1453c4848e4569719cf0bdedcac8a0972dcf2e6e5b07f33a00067255d12df825\",\"key_id\":\"913142794878B06E\",\"kid\":\"01011d23bcfc0c42b4d54aec812aa9385fb9656ae8ed0a55ed3a6db70e7c7a8fca0e0a\",\"reverse_sig\":\"-----BEGIN PGP MESSAGE-----\\nVersion: Keybase Go 1.0.18 (windows)\\nComment: https://keybase.io/download\\n\\nxA0DAAoBkTFCeUh4sG4By+F0AOIAAAAA6XsiYm9keSI6eyJrZXkiOnsiZWxkZXN0\\nX2tpZCI6IjAxMjBiY2VkMTE4NmQwOTAzZGExZjI4ZDA3YzAwOTMwODRmYjgxZjEw\\nNzAyOGIyNWI0NjkxNWMzZDg0NWE1MGNkNzkzMGEiLCJob3N0Ijoia2V5YmFzZS5p\\nbyIsImtpZCI6IjAxMjAxNGM4NDE5YTJlZTQ3ZThmYTExNWVkYjZmMWIzYjZkOGQ1\\nYjdiNzM5YmQ2MmNhZmUyODAzYmRhMjM1NmEyNzQ1MGEiLCJ1aWQiOiJmZTFlMGM5\\nNjY1NDYwM2M1ODgwYjNmNTcxYzg2MjkxOSIsInVzZXJuYW1lIjoiMDBkYW5pIn0s\\nInNpYmtleSI6eyJmaW5nZXJwcmludCI6IjMxMGJjNjg4ZDVjZjM0OGFhNzNkZThj\\nYzkxMzE0Mjc5NDg3OGIwNmUiLCJmdWxsX2hhc2giOiIxNDUzYzQ4NDhlNDU2OTcx\\nOWNmMGJkZWRjYWM4YTA5NzJkY2YyZTZlNWIwN2YzM2EwMDA2NzI1NWQxMmRmODI1\\nIiwia2V5X2lkIjoiOTEzMTQyNzk0ODc4QjA2RSIsImtpZCI6IjAxMDExZDIzYmNm\\nYzBjNDJiNGQ1NGFlYzgxMmFhOTM4NWZiOTY1NmFlOGVkMGE1NWVkM2E2ZGI3MGU3\\nYzdhOGZjYTBl6DBhIiwicmV2ZXJzZV9zaWciOm51bGx9LCJ0eXBlIjoic2lia2V5\\nIiwidmVyc2lvbiI6MX0sImNsaWVudCI6eyJuYW1lIjoia2V5YmFzZS5pbyBnbyBj\\nbGllbnQiLCJ2ZXJzaW9uIjoiMS4wLjE4In0sImN0aW1lIjoxNDg0NDY2MTQxLCJl\\neHBpcmVfaW4iOjUwNDU3NjAwMCwibWVya2xlX3Jvb3QiOnsiY3RpbWUiOjE0ODQ0\\nNjYwNDIsImhhc2giOiJiODRiMzg4ODE1ZTRhOWVlNDJhMDQ1ZTMzZjMwMDk2ZGI1\\nNjc3ZmQ2NzhjNTgyYjNhN2M2MjgxZjJlNGLnNzY2MWM3NWMyZGZhNzY5MWI5NzJh\\nNDZiY2ExMzFmYjFlZDQwNDc2OTFiNmRmYzBmNTc4YjgwYThjYWI4MjliNzI2OTci\\nLCJzZXFubyI6ODAwNDQ5fSwicHJldiI6IjU1M2EwNWFmMDViNWMyM2RmZTYyMmYx\\nYzVhNzNlMGFlNzblMzQ2ZWJmOGM4YjRkYTMyNGU4ZjUyNGM3ZTRiZTM0IizkInNl\\ncW5vIjoyOCwidGFnIuM6InNpZ25hdOJ1cmUi4H0AwsBcBAABCgAQBQJYeyfdCRCR\\nMUJ5SHiwbgAAQ3AIAAskmBwh4Ok4UZB9aQj3vAdQsl54KKbvQ2nh63sxrpzVDRF8\\nH/k3cRwPBX8kkPL4MVBcBepmBmdbj62eYbYXmWh7Oj0xmzBcA5H3HoFmnEtgSuiG\\nEZuzVy5ZeM6RqYlm14FUIa02g/hUYf0qaOfU454xoaCbLatQGGyPpfpWEuZKoAsM\\nO0UlXulr7F7WzlRSuKat6QwIB0d+yEn1+1h+EgMxR2KHzlUSJoPI2/4fR8STXjas\\nVoM7AuR+LIyv3E0l+BbeHQZ2wN0Z2zGX7kDa3ERxcbSOMtknTV/jVyWAQWqcJ8oV\\nJJouI95jDZxLVpzgvh4E1bhh0dIw/o+1u9hl4nk=\\n=Sj7G\\n-----END PGP MESSAGE-----\\n\"},\"type\":\"sibkey\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.18\"},\"ctime\":1484466141,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1484466042,\"hash\":\"b84b388815e4a9ee42a045e33f30096db5677fd678c582b3a7c6281f2e4b7661c75c2dfa7691b972a46bca131fb1ed4047691b6dfc0f578b80a8cab829b72697\",\"seqno\":800449},\"prev\":\"553a05af05b5c23dfe622f1c5a73e0ae76346ebf8c8b4da324e8f524c7e4be34\",\"seqno\":28,\"tag\":\"signature\"}",
      "sig_type": 1,
      "sig_version": 1,
      "ctime": 1484466141,
      "etime": 1989042141,
      "rtime": null,
      "sig_status": 0,
      "prev": "553a05af05b5c23dfe622f1c5a73e0ae76346ebf8c8b4da324e8f524c7e4be34",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 29,
      "payload_hash": "2003b4d3c5eeb3ccf3eb852afcf72dd719a8aa078c32ea612fa734fbb39ce06c",
      "sig_id": "386d246c7aea8ddfd34c05c8d72f9b81ea924841b4343d5b6e75a0f227bb19ed0f",
      "sig_id_short": "OG0kbHrqjd_TTAXI1y-bgeqSSEG0ND1bbnWg",
      "kid": "012014c8419a2ee47e8fa115edb6f1b3b6d8d5b7b739bd62cafe2803bda2356a27450a",
      "sig": "hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgFMhBmi7kfo+hFe228bO22NW3tzm9Ysr+KAO9ojVqJ0UKp3BheWxvYWTFAu17ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwMTRjODQxOWEyZWU0N2U4ZmExMTVlZGI2ZjFiM2I2ZDhkNWI3YjczOWJkNjJjYWZlMjgwM2JkYTIzNTZhMjc0NTBhIiwidWlkIjoiZmUxZTBjOTY2NTQ2MDNjNTg4MGIzZjU3MWM4NjI5MTkiLCJ1c2VybmFtZSI6IjAwZGFuaSJ9LCJzZXJ2aWNlIjp7Im5hbWUiOiJoYWNrZXJuZXdzIiwidXNlcm5hbWUiOiIwMGRhbmkifSwidHlwZSI6IndlYl9zZXJ2aWNlX2JpbmRpbmciLCJ2ZXJzaW9uIjoxfSwiY2xpZW50Ijp7Im5hbWUiOiJrZXliYXNlLmlvIGdvIGNsaWVudCIsInZlcnNpb24iOiIxLjAuMTgifSwiY3RpbWUiOjE0ODQ2MTA0NzUsImV4cGlyZV9pbiI6NTA0NTc2MDAwLCJtZXJrbGVfcm9vdCI6eyJjdGltZSI6MTQ4NDYxMDQzNSwiaGFzaCI6IjQ2MTUzMDQ0YzhiYWRmZGVjOTE5NmU3ZDk0N2M5NGJlZjk1YzIwZGUxNzM3OTdiNGRjMzVhNmYwNjBkNTM1NmU4Y2E0NmUwODQ5ZTc2MjRjMmFlODMwMTBkMzA1OTJiODhhMjdlZDM5MzVjMmViNDM5OTJkMzk1MmQyY2ZiYjJiIiwic2Vxbm8iOjgwMjY4NX0sInByZXYiOiJlNTBjMzQ2NjkzYjNmZGVjZDMxMzI2ZDIyNjMyOGRiYjdlYjQyYmMxMjFkMzYwN2QwNDU1NDM1MzRhMGMwYjE4Iiwic2Vxbm8iOjI5LCJ0YWciOiJzaWduYXR1cmUifaNzaWfEQE73zTeF2OeK+KpsTJGK/SgEMck1tUFkNdW15c1MTQbnPVtZA5HF9KRkMSKGw+avNUrkVPXlPmpg7o5oeVLRygyoc2lnX3R5cGUgpGhhc2iCpHR5cGUIpXZhbHVlxCCq5p2R8sFJBK0w2cjEtvZUWUVnPV139yKt9W8h62f1uqN0YWfNAgKndmVyc2lvbgE=",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"012014c8419a2ee47e8fa115edb6f1b3b6d8d5b7b739bd62cafe2803bda2356a27450a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"service\":{\"name\":\"hackernews\",\"username\":\"00dani\"},\"type\":\"web_service_binding\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.18\"},\"ctime\":1484610475,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1484610435,\"hash\":\"46153044c8badfdec9196e7d947c94bef95c20de173797b4dc35a6f060d5356e8ca46e0849e7624c2ae83010d30592b88a27ed3935c2eb43992d3952d2cfbb2b\",\"seqno\":802685},\"prev\":\"e50c346693b3fdecd31326d226328dbb7eb42bc121d3607d045543534a0c0b18\",\"seqno\":29,\"tag\":\"signature\"}",
      "sig_type": 2,
      "sig_version": 1,
      "ctime": 1484610475,
      "etime": 1989186475,
      "rtime": null,
      "sig_status": 0,
      "prev": "e50c346693b3fdecd31326d226328dbb7eb42bc121d3607d045543534a0c0b18",
      "proof_id": "42d44f540b5411c7cc786910",
      "proof_type": 6,
      "proof_text_check": "OG0kbHrqjd_TTAXI1y-bgeqSSEG0ND1bbnWg8ie7Ge0",
      "proof_text_full": "[ my public key: https://keybase.io/00dani; my proof: https://keybase.io/00dani/sigs/OG0kbHrqjd_TTAXI1y-bgeqSSEG0ND1bbnWg8ie7Ge0 ]\n",
      "check_data_json": "{\"name\":\"hackernews\",\"username\":\"00dani\"}",
      "remote_id": "00dani",
      "api_url": "https://hacker-news.firebaseio.com/v0/user/00dani/about.json",
      "human_url": "https://news.ycombinator.com/user?id=00dani",
      "proof_state": 1,
      "proof_status": 1,
      "retry_count": 335,
      "hard_fail_count": 0,
      "last_check": 1491826111,
      "last_success": 1491826111,
      "version": 2,
      "fingerprint": ""
    },
    {
      "seqno": 30,
      "payload_hash": "79f37fa2b1013082252146157957d8cb3816571d3dbb3f58d322eab9743f8ce9",
      "sig_id": "f9b662f07a30d6b692d415dc783d26115b53fba19ecf591d82f0db9a3386432c0f",
      "sig_id_short": "-bZi8How1raS1BXceD0mEVtT-6Gez1kdgvDb",
      "kid": "0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a",
      "sig": "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",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"track\":{\"basics\":{\"id_version\":20,\"last_id_change\":1481118592,\"username\":\"ecbfus\"},\"id\":\"15040491a4926c7b4fea1d36d1426e19\",\"key\":{\"kid\":\"01203f6abe7d13e58c9459cb29097a79f1d8dab76fd1fd43e3534ebb257fdbb5b5620a\"},\"pgp_keys\":[{\"key_fingerprint\":\"ceefad11314e9366307d623ac218f6a73a30fd96\",\"kid\":\"0101c01fd87152361461bba0b6c504570f2ea01ae7322a7ff939a56964f7a2cb33910a\"}],\"remote_proofs\":[{\"ctime\":1481115252,\"curr\":\"b274c5d52c5651ab5b78cc31cf9bb5615ddcbc78380e4e21bffa89af62982815\",\"etime\":1985691252,\"prev\":\"1a588ba022662d816c22c3c31af3f8f584470fbed30769ae23714e4d11f1d9a4\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"facebook\",\"username\":\"carey.ecbf.us\"},\"proof_type\":8,\"state\":1},\"sig_id\":\"328f0eb55b9fbdd23803627d1fa9b5f80ebda24239a6ba33bad4e2fe6877b7f70f\",\"sig_type\":2},{\"ctime\":1481115632,\"curr\":\"0f15292c88c2959efbec50a03b691be58cd28acdcb77debb58e37ce16d638f2e\",\"etime\":1985691632,\"prev\":\"b274c5d52c5651ab5b78cc31cf9bb5615ddcbc78380e4e21bffa89af62982815\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"jcspence.us\",\"protocol\":\"https:\"},\"proof_type\":1000,\"state\":1},\"sig_id\":\"a68cc0d232babf692bcfa3037cc5f4ff614a0c1e00ea4a1ab10423ebbf1824730f\",\"sig_type\":2},{\"ctime\":1481116117,\"curr\":\"0dc11264dd99acbbd1cc270a7c060198f65f996ff795c08c7eff893c06c0db38\",\"etime\":1985692117,\"prev\":\"0f15292c88c2959efbec50a03b691be58cd28acdcb77debb58e37ce16d638f2e\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"jcspence\"},\"proof_type\":3,\"state\":1},\"sig_id\":\"c8c96d149e292f1d779ff811add869d82a1bcc7e5c0a699ce688d1142ce67b890f\",\"sig_type\":2},{\"ctime\":1481116350,\"curr\":\"970990662387cc5cda3fa147ee15639f217606ba52f13ca7014a1e7cdd2289bc\",\"etime\":1985692350,\"prev\":\"0dc11264dd99acbbd1cc270a7c060198f65f996ff795c08c7eff893c06c0db38\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"ecbf.us\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"sig_id\":\"53400b6fcda7dcb67afc050fdccda0615b3a5b713a00d532db2120a6d80b5aec0f\",\"sig_type\":2},{\"ctime\":1481115229,\"curr\":\"1a588ba022662d816c22c3c31af3f8f584470fbed30769ae23714e4d11f1d9a4\",\"etime\":1985691229,\"prev\":\"6de3a64694c2aa82c9ef784d0665d7cf9011cfebfe2c5687b48813a067de0b06\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"ecbfus\"},\"proof_type\":2,\"state\":10},\"sig_id\":\"228d3e0d0bd66f054102e507257849d880462c24304800fe581a7445c5ab5fb90f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"b8ded301bb6ebea9bc1d843aed2e38d03120d2aa0bdcf9ed09d28b4aff5ab20a\",\"seqno\":15,\"sig_id\":\"01d85f248ac1cae8534132ccbe6c3dd81899b88e94d7977851a6d374f3300a8c0f\"}},\"type\":\"track\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.18\"},\"ctime\":1486097904,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1486097515,\"hash\":\"55c814c785e46400cd62413652934bb0ce82909d8fed0e0ed530b8310d381b6ad60cc11f137e937e89fcd9fb9829f86ea4c5b227bc583a27ae76e521e34f29a4\",\"seqno\":832909},\"prev\":\"2003b4d3c5eeb3ccf3eb852afcf72dd719a8aa078c32ea612fa734fbb39ce06c\",\"seqno\":30,\"tag\":\"signature\"}",
      "sig_type": 3,
      "sig_version": 1,
      "ctime": 1486097904,
      "etime": 1990673904,
      "rtime": null,
      "sig_status": 0,
      "prev": "2003b4d3c5eeb3ccf3eb852afcf72dd719a8aa078c32ea612fa734fbb39ce06c",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 31,
      "payload_hash": "91f468d7230c2d15ff5776864cd61501565a7bff68c03a7d2ab9a765b6e7d918",
      "sig_id": "6f148a3047e06402a5f8136f2dc1e693d818a608b2d9052fd05c914e1531e95b0f",
      "sig_id_short": "bxSKMEfgZAKl-BNvLcHmk9gYpgiy2QUv0FyR",
      "kid": "012014c8419a2ee47e8fa115edb6f1b3b6d8d5b7b739bd62cafe2803bda2356a27450a",
      "sig": "hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgFMhBmi7kfo+hFe228bO22NW3tzm9Ysr+KAO9ojVqJ0UKp3BheWxvYWTFA7J7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwMTRjODQxOWEyZWU0N2U4ZmExMTVlZGI2ZjFiM2I2ZDhkNWI3YjczOWJkNjJjYWZlMjgwM2JkYTIzNTZhMjc0NTBhIiwidWlkIjoiZmUxZTBjOTY2NTQ2MDNjNTg4MGIzZjU3MWM4NjI5MTkiLCJ1c2VybmFtZSI6IjAwZGFuaSJ9LCJwZ3BfdXBkYXRlIjp7ImZpbmdlcnByaW50IjoiN2ZkNzRlY2VhNmFjN2MwNzNlN2Y2MDI5MzhjNWRkYzZiZGE0ZmFlMyIsImZ1bGxfaGFzaCI6IjBlZmZhZTdhNWFkOTRhZjFkMzk3YTI0NmUwZTg0MGIyMGU1ZDc2MGZiNTFlZDk0YzM0ZGFiNjYwMzAwYjExYmEiLCJrZXlfaWQiOiIzOEM1RERDNkJEQTRGQUUzIiwia2lkIjoiMDEwMTQ4MmVjZjNkY2U2NTc0M2MxZjQ4ZjFiYjZiZDE4NjhiOTBlOGU0NGM4YjM5NzUyZjE1NjlkNGRmZjE1MGY0MDQwYSJ9LCJ0eXBlIjoicGdwX3VwZGF0ZSIsInZlcnNpb24iOjF9LCJjbGllbnQiOnsibmFtZSI6ImtleWJhc2UuaW8gZ28gY2xpZW50IiwidmVyc2lvbiI6IjEuMC4xOCJ9LCJjdGltZSI6MTQ4NjIwNTU0MCwiZXhwaXJlX2luIjo1MDQ1NzYwMDAsIm1lcmtsZV9yb290Ijp7ImN0aW1lIjoxNDg2MjA1MzI2LCJoYXNoIjoiNTJmMzJhOWNhOTk1ZTM2MTY4Y2I0NWYxODMwNmJjYTJkMjA0Zjc0N2M2MzUzMjk4ODlhMDQ3Mjg3MzE4NGE3NDNlZWNmZTg1NGJhN2NjZWIzYTYzNjllNjEzMDk2YmI4YjA2NjBmOWJmYzI2OThjYWY3ZTY0NjI2ZWU1ZWEzNmEiLCJzZXFubyI6ODM0OTM0fSwicHJldiI6Ijc5ZjM3ZmEyYjEwMTMwODIyNTIxNDYxNTc5NTdkOGNiMzgxNjU3MWQzZGJiM2Y1OGQzMjJlYWI5NzQzZjhjZTkiLCJzZXFubyI6MzEsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RArkjKaHNiyMDK3v9BB7lSKOhB7eOOOs13yHDj6hRBS5kmcpWCuy0j2QhU/VLAXqzUSFaern4Akd9468kO2LKlCqhzaWdfdHlwZSCkaGFzaIKkdHlwZQildmFsdWXEIFtAPbqoCMgUyC/ELg1fhnTIi8lWUX0laUADMiq87GT3o3RhZ80CAqd2ZXJzaW9uAQ==",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"012014c8419a2ee47e8fa115edb6f1b3b6d8d5b7b739bd62cafe2803bda2356a27450a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"pgp_update\":{\"fingerprint\":\"7fd74ecea6ac7c073e7f602938c5ddc6bda4fae3\",\"full_hash\":\"0effae7a5ad94af1d397a246e0e840b20e5d760fb51ed94c34dab660300b11ba\",\"key_id\":\"38C5DDC6BDA4FAE3\",\"kid\":\"0101482ecf3dce65743c1f48f1bb6bd1868b90e8e44c8b39752f1569d4dff150f4040a\"},\"type\":\"pgp_update\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.18\"},\"ctime\":1486205540,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1486205326,\"hash\":\"52f32a9ca995e36168cb45f18306bca2d204f747c635329889a0472873184a743eecfe854ba7cceb3a6369e613096bb8b0660f9bfc2698caf7e64626ee5ea36a\",\"seqno\":834934},\"prev\":\"79f37fa2b1013082252146157957d8cb3816571d3dbb3f58d322eab9743f8ce9\",\"seqno\":31,\"tag\":\"signature\"}",
      "sig_type": 1,
      "sig_version": 1,
      "ctime": 1486205540,
      "etime": 1990781540,
      "rtime": null,
      "sig_status": 0,
      "prev": "79f37fa2b1013082252146157957d8cb3816571d3dbb3f58d322eab9743f8ce9",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 32,
      "payload_hash": "247226586c012bd704ba62b97e97dc2113604b4e84eb9784ad886887c9ca6e7e",
      "sig_id": "582d8558233683a3239e60df20823bffff6b29f622f9dc087c83ec18a64b0abe0f",
      "sig_id_short": "WC2FWCM2g6MjnmDfIII7__9rKfYi-dwIfIPs",
      "kid": "0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a",
      "sig": "hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgvO0RhtCQPaHyjQfACTCE+4HxBwKLJbRpFcPYRaUM15MKp3BheWxvYWTFAvB7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwidWlkIjoiZmUxZTBjOTY2NTQ2MDNjNTg4MGIzZjU3MWM4NjI5MTkiLCJ1c2VybmFtZSI6IjAwZGFuaSJ9LCJzZXJ2aWNlIjp7Imhvc3RuYW1lIjoiMDBkYW5pLm1lIiwicHJvdG9jb2wiOiJodHRwczoifSwidHlwZSI6IndlYl9zZXJ2aWNlX2JpbmRpbmciLCJ2ZXJzaW9uIjoxfSwiY2xpZW50Ijp7Im5hbWUiOiJrZXliYXNlLmlvIGdvIGNsaWVudCIsInZlcnNpb24iOiIxLjAuMTgifSwiY3RpbWUiOjE0ODY1MDIzODgsImV4cGlyZV9pbiI6NTA0NTc2MDAwLCJtZXJrbGVfcm9vdCI6eyJjdGltZSI6MTQ4NjUwMjM3NywiaGFzaCI6IjYzOGE1MjQxNTIyN2IwYzZiOGY4Mzk3M2ZhZTlkMjU2YzUyMDI1ODJjZDU5YTZhNmZiZDEyZTAzOWVlZDg2ZjJhY2JjZTdkYWY4YjM2MWE4Y2I1N2VhOGMzMzJiNzY2Zjk1MDg3OTY5MzExYTgyNDBkYjA3YjE4OWI5MjRiNTdmIiwic2Vxbm8iOjg0MDg1Mn0sInByZXYiOiI5MWY0NjhkNzIzMGMyZDE1ZmY1Nzc2ODY0Y2Q2MTUwMTU2NWE3YmZmNjhjMDNhN2QyYWI5YTc2NWI2ZTdkOTE4Iiwic2Vxbm8iOjMyLCJ0YWciOiJzaWduYXR1cmUifaNzaWfEQKov/TgFskcRzXzb6LXS9KjfGuT3OTZfK/j2XpJGfsBHegX2rV/MaFV/+uadpTK+CqXhgJrGwZ5MmDWhwC/a4A+oc2lnX3R5cGUgpGhhc2iCpHR5cGUIpXZhbHVlxCAPixSzmYh/lsU+EUCNL74J4e/cfAsx0SjwPp8WKRoxwqN0YWfNAgKndmVyc2lvbgE=",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"service\":{\"hostname\":\"00dani.me\",\"protocol\":\"https:\"},\"type\":\"web_service_binding\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.18\"},\"ctime\":1486502388,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1486502377,\"hash\":\"638a52415227b0c6b8f83973fae9d256c5202582cd59a6a6fbd12e039eed86f2acbce7daf8b361a8cb57ea8c332b766f95087969311a8240db07b189b924b57f\",\"seqno\":840852},\"prev\":\"91f468d7230c2d15ff5776864cd61501565a7bff68c03a7d2ab9a765b6e7d918\",\"seqno\":32,\"tag\":\"signature\"}",
      "sig_type": 2,
      "sig_version": 1,
      "ctime": 1486502388,
      "etime": 1991078388,
      "rtime": null,
      "sig_status": 0,
      "prev": "91f468d7230c2d15ff5776864cd61501565a7bff68c03a7d2ab9a765b6e7d918",
      "proof_id": "4da3b052c0df557e90c77910",
      "proof_type": 1000,
      "proof_text_check": "hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgvO0RhtCQPaHyjQfACTCE+4HxBwKLJbRpFcPYRaUM15MKp3BheWxvYWTFAvB7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwidWlkIjoiZmUxZTBjOTY2NTQ2MDNjNTg4MGIzZjU3MWM4NjI5MTkiLCJ1c2VybmFtZSI6IjAwZGFuaSJ9LCJzZXJ2aWNlIjp7Imhvc3RuYW1lIjoiMDBkYW5pLm1lIiwicHJvdG9jb2wiOiJodHRwczoifSwidHlwZSI6IndlYl9zZXJ2aWNlX2JpbmRpbmciLCJ2ZXJzaW9uIjoxfSwiY2xpZW50Ijp7Im5hbWUiOiJrZXliYXNlLmlvIGdvIGNsaWVudCIsInZlcnNpb24iOiIxLjAuMTgifSwiY3RpbWUiOjE0ODY1MDIzODgsImV4cGlyZV9pbiI6NTA0NTc2MDAwLCJtZXJrbGVfcm9vdCI6eyJjdGltZSI6MTQ4NjUwMjM3NywiaGFzaCI6IjYzOGE1MjQxNTIyN2IwYzZiOGY4Mzk3M2ZhZTlkMjU2YzUyMDI1ODJjZDU5YTZhNmZiZDEyZTAzOWVlZDg2ZjJhY2JjZTdkYWY4YjM2MWE4Y2I1N2VhOGMzMzJiNzY2Zjk1MDg3OTY5MzExYTgyNDBkYjA3YjE4OWI5MjRiNTdmIiwic2Vxbm8iOjg0MDg1Mn0sInByZXYiOiI5MWY0NjhkNzIzMGMyZDE1ZmY1Nzc2ODY0Y2Q2MTUwMTU2NWE3YmZmNjhjMDNhN2QyYWI5YTc2NWI2ZTdkOTE4Iiwic2Vxbm8iOjMyLCJ0YWciOiJzaWduYXR1cmUifaNzaWfEQKov/TgFskcRzXzb6LXS9KjfGuT3OTZfK/j2XpJGfsBHegX2rV/MaFV/+uadpTK+CqXhgJrGwZ5MmDWhwC/a4A+oc2lnX3R5cGUgpGhhc2iCpHR5cGUIpXZhbHVlxCAPixSzmYh/lsU+EUCNL74J4e/cfAsx0SjwPp8WKRoxwqN0YWfNAgKndmVyc2lvbgE=",
      "proof_text_full": "==================================================================\nhttps://keybase.io/00dani\n--------------------------------------------------------------------\n\nI hereby claim:\n\n  * I am an admin of https://00dani.me\n  * I am 00dani (https://keybase.io/00dani) on keybase.\n  * I have a public key ASC87RGG0JA9ofKNB8AJMIT7gfEHAosltGkVw9hFpQzXkwo\n\nTo do so, I am signing this object:\n\n{\n    \"body\": {\n        \"key\": {\n            \"eldest_kid\": \"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\n            \"host\": \"keybase.io\",\n            \"kid\": \"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\n            \"uid\": \"fe1e0c96654603c5880b3f571c862919\",\n            \"username\": \"00dani\"\n        },\n        \"service\": {\n            \"hostname\": \"00dani.me\",\n            \"protocol\": \"https:\"\n        },\n        \"type\": \"web_service_binding\",\n        \"version\": 1\n    },\n    \"client\": {\n        \"name\": \"keybase.io go client\",\n        \"version\": \"1.0.18\"\n    },\n    \"ctime\": 1486502388,\n    \"expire_in\": 504576000,\n    \"merkle_root\": {\n        \"ctime\": 1486502377,\n        \"hash\": \"638a52415227b0c6b8f83973fae9d256c5202582cd59a6a6fbd12e039eed86f2acbce7daf8b361a8cb57ea8c332b766f95087969311a8240db07b189b924b57f\",\n        \"seqno\": 840852\n    },\n    \"prev\": \"91f468d7230c2d15ff5776864cd61501565a7bff68c03a7d2ab9a765b6e7d918\",\n    \"seqno\": 32,\n    \"tag\": \"signature\"\n}\n\nwhich yields the signature:\n\nhKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgvO0RhtCQPaHyjQfACTCE+4HxBwKLJbRpFcPYRaUM15MKp3BheWxvYWTFAvB7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwidWlkIjoiZmUxZTBjOTY2NTQ2MDNjNTg4MGIzZjU3MWM4NjI5MTkiLCJ1c2VybmFtZSI6IjAwZGFuaSJ9LCJzZXJ2aWNlIjp7Imhvc3RuYW1lIjoiMDBkYW5pLm1lIiwicHJvdG9jb2wiOiJodHRwczoifSwidHlwZSI6IndlYl9zZXJ2aWNlX2JpbmRpbmciLCJ2ZXJzaW9uIjoxfSwiY2xpZW50Ijp7Im5hbWUiOiJrZXliYXNlLmlvIGdvIGNsaWVudCIsInZlcnNpb24iOiIxLjAuMTgifSwiY3RpbWUiOjE0ODY1MDIzODgsImV4cGlyZV9pbiI6NTA0NTc2MDAwLCJtZXJrbGVfcm9vdCI6eyJjdGltZSI6MTQ4NjUwMjM3NywiaGFzaCI6IjYzOGE1MjQxNTIyN2IwYzZiOGY4Mzk3M2ZhZTlkMjU2YzUyMDI1ODJjZDU5YTZhNmZiZDEyZTAzOWVlZDg2ZjJhY2JjZTdkYWY4YjM2MWE4Y2I1N2VhOGMzMzJiNzY2Zjk1MDg3OTY5MzExYTgyNDBkYjA3YjE4OWI5MjRiNTdmIiwic2Vxbm8iOjg0MDg1Mn0sInByZXYiOiI5MWY0NjhkNzIzMGMyZDE1ZmY1Nzc2ODY0Y2Q2MTUwMTU2NWE3YmZmNjhjMDNhN2QyYWI5YTc2NWI2ZTdkOTE4Iiwic2Vxbm8iOjMyLCJ0YWciOiJzaWduYXR1cmUifaNzaWfEQKov/TgFskcRzXzb6LXS9KjfGuT3OTZfK/j2XpJGfsBHegX2rV/MaFV/+uadpTK+CqXhgJrGwZ5MmDWhwC/a4A+oc2lnX3R5cGUgpGhhc2iCpHR5cGUIpXZhbHVlxCAPixSzmYh/lsU+EUCNL74J4e/cfAsx0SjwPp8WKRoxwqN0YWfNAgKndmVyc2lvbgE=\n\nAnd finally, I am proving ownership of this host by posting or\nappending to this document.\n\nView my publicly-auditable identity here: https://keybase.io/00dani\n\n==================================================================\n",
      "check_data_json": "{\"hostname\":\"00dani.me\",\"protocol\":\"https:\"}",
      "remote_id": "",
      "api_url": null,
      "human_url": null,
      "proof_state": 5,
      "proof_status": 0,
      "retry_count": 0,
      "hard_fail_count": 0,
      "last_check": 1486502388,
      "last_success": null,
      "version": 0,
      "fingerprint": ""
    },
    {
      "seqno": 33,
      "payload_hash": "27f635031339cb5d7fb2db3e7f4f14e2e2fa007864cd6bb36d284ca78650c6bf",
      "sig_id": "90443f437e44b8c38fd67d5413859e009ff20200a604202b7a992b43de737a590f",
      "sig_id_short": "kEQ_Q35EuMOP1n1UE4WeAJ_yAgCmBCArepkr",
      "kid": "0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a",
      "sig": "hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgvO0RhtCQPaHyjQfACTCE+4HxBwKLJbRpFcPYRaUM15MKp3BheWxvYWTFAvB7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwidWlkIjoiZmUxZTBjOTY2NTQ2MDNjNTg4MGIzZjU3MWM4NjI5MTkiLCJ1c2VybmFtZSI6IjAwZGFuaSJ9LCJzZXJ2aWNlIjp7Imhvc3RuYW1lIjoiMDBkYW5pLm1lIiwicHJvdG9jb2wiOiJodHRwczoifSwidHlwZSI6IndlYl9zZXJ2aWNlX2JpbmRpbmciLCJ2ZXJzaW9uIjoxfSwiY2xpZW50Ijp7Im5hbWUiOiJrZXliYXNlLmlvIGdvIGNsaWVudCIsInZlcnNpb24iOiIxLjAuMTgifSwiY3RpbWUiOjE0ODY1MDI0NzEsImV4cGlyZV9pbiI6NTA0NTc2MDAwLCJtZXJrbGVfcm9vdCI6eyJjdGltZSI6MTQ4NjUwMjM4OCwiaGFzaCI6IjdmYmM4NmJlNjViNWUwOWM0MWY5MGEyYTk4NmMyMmM2Y2JhMGJlYzBhZGM3M2IzY2RlMzhhN2Y0ZjUzY2ViNDcxOWZiYmFkOTY4ZTY1Y2Q3N2FlZGQ3NDU5OGZjOWY3NWNlYzUyZTVkYzMzN2ZlYjM1YjMzNTlmM2VhNmE5ZjBkIiwic2Vxbm8iOjg0MDg1M30sInByZXYiOiIyNDcyMjY1ODZjMDEyYmQ3MDRiYTYyYjk3ZTk3ZGMyMTEzNjA0YjRlODRlYjk3ODRhZDg4Njg4N2M5Y2E2ZTdlIiwic2Vxbm8iOjMzLCJ0YWciOiJzaWduYXR1cmUifaNzaWfEQLHcXK1/BmBNEcwKl8sw2RIShjZkL15+rqNbHd4RXb6OjAAvoug0pCZWFU2NoG8chttm+U6WW3mADOwHGy5HqwCoc2lnX3R5cGUgpGhhc2iCpHR5cGUIpXZhbHVlxCAovpeXiCcQcquE0p+VhDOjdJpxwKlVWElcn4oSMliIDKN0YWfNAgKndmVyc2lvbgE=",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"service\":{\"hostname\":\"00dani.me\",\"protocol\":\"https:\"},\"type\":\"web_service_binding\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.18\"},\"ctime\":1486502471,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1486502388,\"hash\":\"7fbc86be65b5e09c41f90a2a986c22c6cba0bec0adc73b3cde38a7f4f53ceb4719fbbad968e65cd77aedd74598fc9f75cec52e5dc337feb35b3359f3ea6a9f0d\",\"seqno\":840853},\"prev\":\"247226586c012bd704ba62b97e97dc2113604b4e84eb9784ad886887c9ca6e7e\",\"seqno\":33,\"tag\":\"signature\"}",
      "sig_type": 2,
      "sig_version": 1,
      "ctime": 1486502471,
      "etime": 1991078471,
      "rtime": null,
      "sig_status": 0,
      "prev": "247226586c012bd704ba62b97e97dc2113604b4e84eb9784ad886887c9ca6e7e",
      "proof_id": "6407c36c58e28c58290a6e10",
      "proof_type": 1000,
      "proof_text_check": "hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgvO0RhtCQPaHyjQfACTCE+4HxBwKLJbRpFcPYRaUM15MKp3BheWxvYWTFAvB7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwidWlkIjoiZmUxZTBjOTY2NTQ2MDNjNTg4MGIzZjU3MWM4NjI5MTkiLCJ1c2VybmFtZSI6IjAwZGFuaSJ9LCJzZXJ2aWNlIjp7Imhvc3RuYW1lIjoiMDBkYW5pLm1lIiwicHJvdG9jb2wiOiJodHRwczoifSwidHlwZSI6IndlYl9zZXJ2aWNlX2JpbmRpbmciLCJ2ZXJzaW9uIjoxfSwiY2xpZW50Ijp7Im5hbWUiOiJrZXliYXNlLmlvIGdvIGNsaWVudCIsInZlcnNpb24iOiIxLjAuMTgifSwiY3RpbWUiOjE0ODY1MDI0NzEsImV4cGlyZV9pbiI6NTA0NTc2MDAwLCJtZXJrbGVfcm9vdCI6eyJjdGltZSI6MTQ4NjUwMjM4OCwiaGFzaCI6IjdmYmM4NmJlNjViNWUwOWM0MWY5MGEyYTk4NmMyMmM2Y2JhMGJlYzBhZGM3M2IzY2RlMzhhN2Y0ZjUzY2ViNDcxOWZiYmFkOTY4ZTY1Y2Q3N2FlZGQ3NDU5OGZjOWY3NWNlYzUyZTVkYzMzN2ZlYjM1YjMzNTlmM2VhNmE5ZjBkIiwic2Vxbm8iOjg0MDg1M30sInByZXYiOiIyNDcyMjY1ODZjMDEyYmQ3MDRiYTYyYjk3ZTk3ZGMyMTEzNjA0YjRlODRlYjk3ODRhZDg4Njg4N2M5Y2E2ZTdlIiwic2Vxbm8iOjMzLCJ0YWciOiJzaWduYXR1cmUifaNzaWfEQLHcXK1/BmBNEcwKl8sw2RIShjZkL15+rqNbHd4RXb6OjAAvoug0pCZWFU2NoG8chttm+U6WW3mADOwHGy5HqwCoc2lnX3R5cGUgpGhhc2iCpHR5cGUIpXZhbHVlxCAovpeXiCcQcquE0p+VhDOjdJpxwKlVWElcn4oSMliIDKN0YWfNAgKndmVyc2lvbgE=",
      "proof_text_full": "==================================================================\nhttps://keybase.io/00dani\n--------------------------------------------------------------------\n\nI hereby claim:\n\n  * I am an admin of https://00dani.me\n  * I am 00dani (https://keybase.io/00dani) on keybase.\n  * I have a public key ASC87RGG0JA9ofKNB8AJMIT7gfEHAosltGkVw9hFpQzXkwo\n\nTo do so, I am signing this object:\n\n{\n    \"body\": {\n        \"key\": {\n            \"eldest_kid\": \"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\n            \"host\": \"keybase.io\",\n            \"kid\": \"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\n            \"uid\": \"fe1e0c96654603c5880b3f571c862919\",\n            \"username\": \"00dani\"\n        },\n        \"service\": {\n            \"hostname\": \"00dani.me\",\n            \"protocol\": \"https:\"\n        },\n        \"type\": \"web_service_binding\",\n        \"version\": 1\n    },\n    \"client\": {\n        \"name\": \"keybase.io go client\",\n        \"version\": \"1.0.18\"\n    },\n    \"ctime\": 1486502471,\n    \"expire_in\": 504576000,\n    \"merkle_root\": {\n        \"ctime\": 1486502388,\n        \"hash\": \"7fbc86be65b5e09c41f90a2a986c22c6cba0bec0adc73b3cde38a7f4f53ceb4719fbbad968e65cd77aedd74598fc9f75cec52e5dc337feb35b3359f3ea6a9f0d\",\n        \"seqno\": 840853\n    },\n    \"prev\": \"247226586c012bd704ba62b97e97dc2113604b4e84eb9784ad886887c9ca6e7e\",\n    \"seqno\": 33,\n    \"tag\": \"signature\"\n}\n\nwhich yields the signature:\n\nhKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgvO0RhtCQPaHyjQfACTCE+4HxBwKLJbRpFcPYRaUM15MKp3BheWxvYWTFAvB7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwidWlkIjoiZmUxZTBjOTY2NTQ2MDNjNTg4MGIzZjU3MWM4NjI5MTkiLCJ1c2VybmFtZSI6IjAwZGFuaSJ9LCJzZXJ2aWNlIjp7Imhvc3RuYW1lIjoiMDBkYW5pLm1lIiwicHJvdG9jb2wiOiJodHRwczoifSwidHlwZSI6IndlYl9zZXJ2aWNlX2JpbmRpbmciLCJ2ZXJzaW9uIjoxfSwiY2xpZW50Ijp7Im5hbWUiOiJrZXliYXNlLmlvIGdvIGNsaWVudCIsInZlcnNpb24iOiIxLjAuMTgifSwiY3RpbWUiOjE0ODY1MDI0NzEsImV4cGlyZV9pbiI6NTA0NTc2MDAwLCJtZXJrbGVfcm9vdCI6eyJjdGltZSI6MTQ4NjUwMjM4OCwiaGFzaCI6IjdmYmM4NmJlNjViNWUwOWM0MWY5MGEyYTk4NmMyMmM2Y2JhMGJlYzBhZGM3M2IzY2RlMzhhN2Y0ZjUzY2ViNDcxOWZiYmFkOTY4ZTY1Y2Q3N2FlZGQ3NDU5OGZjOWY3NWNlYzUyZTVkYzMzN2ZlYjM1YjMzNTlmM2VhNmE5ZjBkIiwic2Vxbm8iOjg0MDg1M30sInByZXYiOiIyNDcyMjY1ODZjMDEyYmQ3MDRiYTYyYjk3ZTk3ZGMyMTEzNjA0YjRlODRlYjk3ODRhZDg4Njg4N2M5Y2E2ZTdlIiwic2Vxbm8iOjMzLCJ0YWciOiJzaWduYXR1cmUifaNzaWfEQLHcXK1/BmBNEcwKl8sw2RIShjZkL15+rqNbHd4RXb6OjAAvoug0pCZWFU2NoG8chttm+U6WW3mADOwHGy5HqwCoc2lnX3R5cGUgpGhhc2iCpHR5cGUIpXZhbHVlxCAovpeXiCcQcquE0p+VhDOjdJpxwKlVWElcn4oSMliIDKN0YWfNAgKndmVyc2lvbgE=\n\nAnd finally, I am proving ownership of this host by posting or\nappending to this document.\n\nView my publicly-auditable identity here: https://keybase.io/00dani\n\n==================================================================\n",
      "check_data_json": "{\"hostname\":\"00dani.me\",\"protocol\":\"https:\"}",
      "remote_id": "https://00dani.me/keybase.txt",
      "api_url": "https://00dani.me/keybase.txt",
      "human_url": "https://00dani.me/keybase.txt",
      "proof_state": 1,
      "proof_status": 1,
      "retry_count": 8,
      "hard_fail_count": 0,
      "last_check": 1491805641,
      "last_success": 1491805641,
      "version": 5,
      "fingerprint": ""
    },
    {
      "seqno": 34,
      "payload_hash": "585c07359220211bf1f85bb68d92d89d303027883c2e801c52ddfff404a1a4fb",
      "sig_id": "5b1da0d98811e9646d57531f410ba9343a92dbf14218cb1233f5b5709656de8a0f",
      "sig_id_short": "Wx2g2YgR6WRtV1MfQQupNDqS2_FCGMsSM_W1",
      "kid": "0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a",
      "sig": "hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgvO0RhtCQPaHyjQfACTCE+4HxBwKLJbRpFcPYRaUM15MKp3BheWxvYWTFCB57ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwidWlkIjoiZmUxZTBjOTY2NTQ2MDNjNTg4MGIzZjU3MWM4NjI5MTkiLCJ1c2VybmFtZSI6IjAwZGFuaSJ9LCJ0cmFjayI6eyJiYXNpY3MiOnsiaWRfdmVyc2lvbiI6MTEsImxhc3RfaWRfY2hhbmdlIjoxNDIzNzU3MzI3LCJ1c2VybmFtZSI6ImF1dGh5In0sImlkIjoiNjFkMTNmMmE0YWZhYjZiODQ0ZTE5YjgzYzIxMmE0MDAiLCJrZXkiOnsia2V5X2ZpbmdlcnByaW50IjoiOGI3NjRlY2FkZWU1MTAwYzJhZDJmOGI5ZWJjZGYxYmFkZjAyMDYxOCIsImtpZCI6IjAxMDEwNjQ3Zjc4ZDEwNTRkOTk2NTMyZDk5MjZkZTIwZjNiZTRhZjUyMzM0N2IyZGM4MThkYzhhYTFhY2U5YzhjNDAwMGEifSwicGdwX2tleXMiOlt7ImtleV9maW5nZXJwcmludCI6IjhiNzY0ZWNhZGVlNTEwMGMyYWQyZjhiOWViY2RmMWJhZGYwMjA2MTgiLCJraWQiOiIwMTAxMDY0N2Y3OGQxMDU0ZDk5NjUzMmQ5OTI2ZGUyMGYzYmU0YWY1MjMzNDdiMmRjODE4ZGM4YWExYWNlOWM4YzQwMDBhIn1dLCJyZW1vdGVfcHJvb2ZzIjpbeyJjdGltZSI6MTQyMzc1Njk3NSwiY3VyciI6ImI5ZWIyMzY5NGNkZGJmMTI1NzQ0MTQ5MDc5YjdlYzgxMjE0ZjczNTcxYWJmMjRhNzAwOTdlZGI3MGFkZTNkNjUiLCJldGltZSI6MTU4MTQzNjk3NSwicHJldiI6IjljMGUwYWE2MWNkYTRmNGFlZTBkNWYyNzM2ODI3NzBlMzhjYjI0YzcwODJjNDMwOThlNzYzNmFkODhkNTMxZmYiLCJyZW1vdGVfa2V5X3Byb29mIjp7ImNoZWNrX2RhdGFfanNvbiI6eyJuYW1lIjoidHdpdHRlciIsInVzZXJuYW1lIjoiYXV0aHkifSwicHJvb2ZfdHlwZSI6Miwic3RhdGUiOjF9LCJzaWdfaWQiOiIzYzliZjEyYmY1NTBjOGE2ZjU1ZjI0N2RhMmM3ZWE0M2JkZWY5OGJkMGY4MDJiM2E4YWEyOTE1YmViNGQzOTdkMGYiLCJzaWdfdHlwZSI6Mn0seyJjdGltZSI6MTQyMzc1NzE4MCwiY3VyciI6IjZkOWM5M2NmYWExZTRlZjUwYjFjYWVhNGRkMzZhYzFhMDI1Mzc3NzBmMTEyZjAxNTUzZDBmZTJlMzQwNGQzZmIiLCJldGltZSI6MTU4MTQzNzE4MCwicHJldiI6Ijg5ZTIzMWRmM2EyMmY5OTgyZDBkNjU5OWY0ZGFjNmYyNzhlOWY1ZTczZmY5NWZhM2ZiMTJkZjI0YjlhYjQ5NWQiLCJyZW1vdGVfa2V5X3Byb29mIjp7ImNoZWNrX2RhdGFfanNvbiI6eyJkb21haW4iOiJhdXRoeS5jb20iLCJwcm90b2NvbCI6ImRucyJ9LCJwcm9vZl90eXBlIjoxMDAxLCJzdGF0ZSI6MX0sInNpZ19pZCI6ImIxNjg5MjczYmVkOTRhNGU4ZDA3ZTc4NzNjMzM3ZWU4MTYwZmJiYjVkZGVkY2MwYWRjNzY3NDY5NGEyMTVhZGMwZiIsInNpZ190eXBlIjoyfV0sInNlcV90YWlsIjp7InBheWxvYWRfaGFzaCI6IjIwMTE1MjQyOGUwY2M4NTBiZWE2YTYyNTM0ZWQwYjg2YWE5M2FlMGVhZTYxMWE2MzY4Y2UzY2FmZDNhOWMyNmMiLCJzZXFubyI6Niwic2lnX2lkIjoiN2EzMTRjZDAyZTA4ZDBhNjNlNjFmODZjZmJmZGUxOThhZDI4ZTRmOTE1MTBjMjdlM2NiMGMwZWE0NWFkYmJiNDBmIn19LCJ0eXBlIjoidHJhY2siLCJ2ZXJzaW9uIjoxfSwiY2xpZW50Ijp7Im5hbWUiOiJrZXliYXNlLmlvIGdvIGNsaWVudCIsInZlcnNpb24iOiIxLjAuMTgifSwiY3RpbWUiOjE0ODY2MTg1MzgsImV4cGlyZV9pbiI6NTA0NTc2MDAwLCJtZXJrbGVfcm9vdCI6eyJjdGltZSI6MTQ4NjYxODUzMywiaGFzaCI6ImE0NzU2OWFiOGM5NjVhOTZiZTBjOTY3YTE3OTEzMzYyM2I0NTNiNDI3MjEzZTIxYjRmYzYxNjY4NWI3MWQyYTFhOGUzM2YxOWZiNzYyNWRhZDUyN2JjZjUyYzM2Nzk4YmQxMzk4MzFiYzZhMDNjMTIwYTU1MmYxZWNlNWQ1YjYzIiwic2Vxbm8iOjg1MzI4OH0sInByZXYiOiIyN2Y2MzUwMzEzMzljYjVkN2ZiMmRiM2U3ZjRmMTRlMmUyZmEwMDc4NjRjZDZiYjM2ZDI4NGNhNzg2NTBjNmJmIiwic2Vxbm8iOjM0LCJ0YWciOiJzaWduYXR1cmUifaNzaWfEQPJzsJ3cGKVxhz4x/9I4JNu7+ChnhmAwr0YfGgsypM+e5jImcOHdGIEXv+3YcmkZl0b1TiE35Vo7A4JLkxdXcgWoc2lnX3R5cGUgpGhhc2iCpHR5cGUIpXZhbHVlxCCcPanwO/GbUoGu47q7OBKgWFbrFJ0amMdXDG4758GHoKN0YWfNAgKndmVyc2lvbgE=",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"track\":{\"basics\":{\"id_version\":11,\"last_id_change\":1423757327,\"username\":\"authy\"},\"id\":\"61d13f2a4afab6b844e19b83c212a400\",\"key\":{\"key_fingerprint\":\"8b764ecadee5100c2ad2f8b9ebcdf1badf020618\",\"kid\":\"01010647f78d1054d996532d9926de20f3be4af523347b2dc818dc8aa1ace9c8c4000a\"},\"pgp_keys\":[{\"key_fingerprint\":\"8b764ecadee5100c2ad2f8b9ebcdf1badf020618\",\"kid\":\"01010647f78d1054d996532d9926de20f3be4af523347b2dc818dc8aa1ace9c8c4000a\"}],\"remote_proofs\":[{\"ctime\":1423756975,\"curr\":\"b9eb23694cddbf125744149079b7ec81214f73571abf24a70097edb70ade3d65\",\"etime\":1581436975,\"prev\":\"9c0e0aa61cda4f4aee0d5f273682770e38cb24c7082c43098e7636ad88d531ff\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"authy\"},\"proof_type\":2,\"state\":1},\"sig_id\":\"3c9bf12bf550c8a6f55f247da2c7ea43bdef98bd0f802b3a8aa2915beb4d397d0f\",\"sig_type\":2},{\"ctime\":1423757180,\"curr\":\"6d9c93cfaa1e4ef50b1caea4dd36ac1a02537770f112f01553d0fe2e3404d3fb\",\"etime\":1581437180,\"prev\":\"89e231df3a22f9982d0d6599f4dac6f278e9f5e73ff95fa3fb12df24b9ab495d\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"authy.com\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"sig_id\":\"b1689273bed94a4e8d07e7873c337ee8160fbbb5ddedcc0adc7674694a215adc0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"201152428e0cc850bea6a62534ed0b86aa93ae0eae611a6368ce3cafd3a9c26c\",\"seqno\":6,\"sig_id\":\"7a314cd02e08d0a63e61f86cfbfde198ad28e4f91510c27e3cb0c0ea45adbbb40f\"}},\"type\":\"track\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.18\"},\"ctime\":1486618538,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1486618533,\"hash\":\"a47569ab8c965a96be0c967a179133623b453b427213e21b4fc616685b71d2a1a8e33f19fb7625dad527bcf52c36798bd139831bc6a03c120a552f1ece5d5b63\",\"seqno\":853288},\"prev\":\"27f635031339cb5d7fb2db3e7f4f14e2e2fa007864cd6bb36d284ca78650c6bf\",\"seqno\":34,\"tag\":\"signature\"}",
      "sig_type": 3,
      "sig_version": 1,
      "ctime": 1486618538,
      "etime": 1991194538,
      "rtime": null,
      "sig_status": 0,
      "prev": "27f635031339cb5d7fb2db3e7f4f14e2e2fa007864cd6bb36d284ca78650c6bf",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 35,
      "payload_hash": "60ea9451ad8ebb441842bf41df762a7afd9804da03932aa20e380697ac45ce1c",
      "sig_id": "8c0070af96a4f64cf6a31afa75fd00e38ecfa50df3601d1f7b9d5c43dd87c9d30f",
      "sig_id_short": "jABwr5ak9kz2oxr6df0A447PpQ3zYB0fe51c",
      "kid": "0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a",
      "sig": "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",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"track\":{\"basics\":{\"id_version\":52,\"last_id_change\":1481914988,\"username\":\"coinbase\"},\"id\":\"4ed9f7d6a9d41779d46bd876c3d8f700\",\"key\":{\"key_fingerprint\":\"ad4833210ab05db1e7d4a8c4c73f1c17f994ab30\",\"kid\":\"01013709ed2a1ec64dec43d88c02040ba45b6ecb15f1635d6e4b48d8a66d7ed6a2520a\"},\"pgp_keys\":[{\"key_fingerprint\":\"ad4833210ab05db1e7d4a8c4c73f1c17f994ab30\",\"kid\":\"01013709ed2a1ec64dec43d88c02040ba45b6ecb15f1635d6e4b48d8a66d7ed6a2520a\"}],\"remote_proofs\":[{\"ctime\":1423077827,\"curr\":\"9751bc85e01de9c5903b56d508e3aa1901b1d132bf8e939c46f31e19836857aa\",\"etime\":1580757827,\"prev\":\"ca3ff703799b79de04eff23137ce25bf0753ab3e5454cfdff78fa6502cf43533\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"coinbase\"},\"proof_type\":2,\"state\":1},\"sig_id\":\"f083f03eedc9ed1cc2971a4b605e2ef950028bcf14b79d570605fb404925b1db0f\",\"sig_type\":2},{\"ctime\":1422237602,\"curr\":\"38f83deac183aebae0d800321c6e170a8d048869aa0bee7e2e7a9a481fc150f0\",\"etime\":1579917602,\"prev\":null,\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"coinbase.com\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"sig_id\":\"4588b044b3f4fe26a1e8c981ea0d57f3d8cb198de90b01504987d5a0cd0bcf6c0f\",\"sig_type\":2},{\"ctime\":1422237643,\"curr\":\"48883e35c169696f0aac19f34329dc62b64602bcb6bdb30305c601f3331142cf\",\"etime\":1579917643,\"prev\":\"38f83deac183aebae0d800321c6e170a8d048869aa0bee7e2e7a9a481fc150f0\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"reddit\",\"username\":\"CoinbaseOfficial\"},\"proof_type\":4,\"state\":1},\"sig_id\":\"59d93766d847b88303d345fd3d0d92d0baa1208d39040f184932b4e7c71d73680f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"387287152ea60ed5c3014decbb1ce2a36288b1ba3f4042816cdc1f9c4f86e721\",\"seqno\":39,\"sig_id\":\"0f544f3cad36389b0dc1a546fa01c589f73c215d96eeffde1ae8555df71a14340f\"}},\"type\":\"track\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.18\"},\"ctime\":1486618556,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1486618551,\"hash\":\"f2aa96a1d28a89164d01b13ee2fd5f112669b3e8ef3b401eb0b40a730f69c8994a650f206afc1c8eaed0cf413c7f68fab9f4bc06f623fed0eb47dfe01e668f20\",\"seqno\":853292},\"prev\":\"585c07359220211bf1f85bb68d92d89d303027883c2e801c52ddfff404a1a4fb\",\"seqno\":35,\"tag\":\"signature\"}",
      "sig_type": 3,
      "sig_version": 1,
      "ctime": 1486618556,
      "etime": 1991194556,
      "rtime": null,
      "sig_status": 0,
      "prev": "585c07359220211bf1f85bb68d92d89d303027883c2e801c52ddfff404a1a4fb",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 36,
      "payload_hash": "a895d224e7a68a97a26b4d7988009818ffcbed62423f0039b43afde904e1693f",
      "sig_id": "8d12bc97c6b17480f30af9724c54f521b947c0b59cf52d736bba9c67613fbf9c0f",
      "sig_id_short": "jRK8l8axdIDzCvlyTFT1IblHwLWc9S1za7qc",
      "kid": "0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a",
      "sig": "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",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"track\":{\"basics\":{\"id_version\":11,\"last_id_change\":1486625993,\"username\":\"catlover1672\"},\"id\":\"2f1681c6069a9a4edd9cda1c6c2ab419\",\"key\":{\"kid\":\"0120fd992780608063e7a98756a83a382b19aa96c8fdd2d4ea0eb9f04eff5e87240f0a\"},\"remote_proofs\":[{\"ctime\":1486625696,\"curr\":\"b29f5da9df5e32daf245fef753daa996c1be581eb039a0f3806a4fa4157313a2\",\"etime\":1991201696,\"prev\":\"7c4cc73e7779669213e4ce48516fa1a0e1c8b345734bcb19b758a935a4c0ac4d\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"fuzzywhatzitinvasion.tumblr.com\",\"protocol\":\"https:\"},\"proof_type\":1000,\"state\":1},\"sig_id\":\"9e2e4bbfa9c10301cf386f004796ab7afbc16152cf4b7349e734bd43d59389350f\",\"sig_type\":2},{\"ctime\":1486624717,\"curr\":\"694b30bc5afca81f48de0d77be7ee76883ed7d81d8c1bff653c83a2cc9422622\",\"etime\":1991200717,\"prev\":\"b04e384851778dd5a5a90cd7f33176622187d2d30366dad3fd29b4fc4d07e6aa\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"queenwhatzit\"},\"proof_type\":2,\"state\":1},\"sig_id\":\"898a02ed409c7238613b6fb73e6eea77ec8792a7dae95b9fc9d07378f39645d80f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"e1d968fe2aa782f487bb489c3a8ae11ca6e2d20d6ed9c2d6501d36c0e3b991ef\",\"seqno\":9,\"sig_id\":\"422fdc9944a055188a4ac26d9784a0a0f8d0cb78aa8cda5e4972b1f11af21cea0f\"}},\"type\":\"track\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.18\"},\"ctime\":1486629075,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1486629072,\"hash\":\"f153ab1380ad0ec43836a9d80bc29c19b26b636c56fa12bfc15155f4e61810bae90fa47a04146c0fa65ccfa5679186d562d2fd8bf674b9332fb543073d6cdba4\",\"seqno\":854583},\"prev\":\"60ea9451ad8ebb441842bf41df762a7afd9804da03932aa20e380697ac45ce1c\",\"seqno\":36,\"tag\":\"signature\"}",
      "sig_type": 3,
      "sig_version": 1,
      "ctime": 1486629075,
      "etime": 1991205075,
      "rtime": null,
      "sig_status": 0,
      "prev": "60ea9451ad8ebb441842bf41df762a7afd9804da03932aa20e380697ac45ce1c",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 37,
      "payload_hash": "3d96e01bb39075393b1c757272a8d1846bad296abe22584b6891a95e4f1d948f",
      "sig_id": "daf78ec9d9b47be3c033f315c772ab5d41fdc1b91c21aaba2d26a08a37f0bb110f",
      "sig_id_short": "2veOydm0e-PAM_MVx3KrXUH9wbkcIaq6LSag",
      "kid": "0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a",
      "sig": "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",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"track\":{\"basics\":{\"id_version\":59,\"last_id_change\":1483724867,\"username\":\"steevc\"},\"id\":\"eb31007e4b8e19f95b0a081c4bb08900\",\"key\":{\"key_fingerprint\":\"a5d284d0059ebcb4f8072a994175dcf03f9a241a\",\"kid\":\"01018fe98257f91c8d8c59b25893eabada7d8a68ae2191afcff968d7ff18920c20fe0a\"},\"pgp_keys\":[{\"key_fingerprint\":\"a5d284d0059ebcb4f8072a994175dcf03f9a241a\",\"kid\":\"01018fe98257f91c8d8c59b25893eabada7d8a68ae2191afcff968d7ff18920c20fe0a\"}],\"remote_proofs\":[{\"ctime\":1419259635,\"curr\":\"ae6ddce726806ce413fcc7bc05c03b6bb68f00f6b23464378644b3eb20692bfa\",\"etime\":1576939635,\"prev\":\"5cacbe7c08001cdc5b079895ff906ecddfd037efdb4f7a8bc853acbc9b6606bb\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"bagofspoons.net\",\"protocol\":\"http:\"},\"proof_type\":1000,\"state\":1},\"sig_id\":\"9c3a1f596986974bf7ba477bf0d94415e7e6125779179f35071c8d5e3ac149eb0f\",\"sig_type\":2},{\"ctime\":1471786992,\"curr\":\"e9c89337f35fd94bcaeaa41346e581b333ee1f5b853f98c20aa52f974c13bf4e\",\"etime\":1976362992,\"prev\":\"2eb698e7aa740e287fc43fdc3fb3ecba4d2a8c6828856754816a5dfb22227e39\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"reddit\",\"username\":\"steevc\"},\"proof_type\":4,\"state\":1},\"sig_id\":\"1624737807ac8550c2f73dbf44df44b50412217fb2bc2f1cecf76486e4f7e3760f\",\"sig_type\":2},{\"ctime\":1479375112,\"curr\":\"c59ea84ebadc24c425a2c728537e0725ae4d14ae801fcff749e6923a0b01f930\",\"etime\":1983951112,\"prev\":\"949f4cfe8f3f8a5dcbf96db23fb6059ef04a07b86a9618f2e0ca3edf1584bba4\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"facebook\",\"username\":\"steevc\"},\"proof_type\":8,\"state\":1},\"sig_id\":\"9d60a10125853250ce103639aafb4c8b014a899dfdac52b64298032df873c4b80f\",\"sig_type\":2},{\"ctime\":1412942510,\"curr\":\"e5d0d5bb376e438a5cf416d7d85cc8d69f1e31f74199e508a19247124e4c88dc\",\"etime\":1570622510,\"prev\":\"3855de0435a632448e37e825191a2685072bef4bdf28b9757cc44963b0e5e895\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"steevc\"},\"proof_type\":2,\"state\":1},\"sig_id\":\"4340b722b48ec82caf5030c83025af940dcd8e17201f1b3ea932ef7300ab3fb00f\",\"sig_type\":2},{\"ctime\":1412945273,\"curr\":\"1bd8ee99f43b35fda842b109d327cee83e406c9b2afbf3c5f22def6e3c58a9ce\",\"etime\":1570625273,\"prev\":\"f3ff93c965c528bb038fcc2f8218461d4e451737928b200bf378f2235ccaca3f\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"steevc\"},\"proof_type\":3,\"state\":1},\"sig_id\":\"2c983429b561f75f4dababa699be9a46a59ad209f8a33b8ff2f7344848c0a5a80f\",\"sig_type\":2},{\"ctime\":1419259094,\"curr\":\"5cacbe7c08001cdc5b079895ff906ecddfd037efdb4f7a8bc853acbc9b6606bb\",\"etime\":1576939094,\"prev\":\"32addd32f0d3daa9a45d6c4f5126ad0be53c562c278f9a345a58aff9ff9410df\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"coinbase\",\"username\":\"steevc\"},\"proof_type\":5,\"state\":1},\"sig_id\":\"cf8737993849e1bc5a26d5200794dfc1a9aa9e38300684e11b55d29cec60adbd0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"38a0458ac1848a266b745cab20406958561b14eb106f1f6dce271775b365a752\",\"seqno\":27,\"sig_id\":\"2271e342aaaa5cc7ee94c91c4a63b2a7502d55fab7bd4df8bbe1338eaef9794d0f\"}},\"type\":\"track\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.18\"},\"ctime\":1486909147,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1486909143,\"hash\":\"7bdbc3f7466ef756968970ad6719155fceaa25bc03d930822c6dbc053899a27a03139f247d990fe882f2df4d51512162f0abfc87bd0ed8f6b790bf3a44a462dc\",\"seqno\":880373},\"prev\":\"a895d224e7a68a97a26b4d7988009818ffcbed62423f0039b43afde904e1693f\",\"seqno\":37,\"tag\":\"signature\"}",
      "sig_type": 3,
      "sig_version": 1,
      "ctime": 1486909147,
      "etime": 1991485147,
      "rtime": null,
      "sig_status": 0,
      "prev": "a895d224e7a68a97a26b4d7988009818ffcbed62423f0039b43afde904e1693f",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 38,
      "payload_hash": "234d9f4fdf31b59f4dd791e08ff7652c8f4560931e771b643f1e45e2e2de2a88",
      "sig_id": "4538ce8cd9536faeb6695138a7f645751ccab0f3f14ab741346da25c857527cf0f",
      "sig_id_short": "RTjOjNlTb662aVE4p_ZFdRzKsPPxSrdBNG2i",
      "kid": "012014c8419a2ee47e8fa115edb6f1b3b6d8d5b7b739bd62cafe2803bda2356a27450a",
      "sig": "hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgFMhBmi7kfo+hFe228bO22NW3tzm9Ysr+KAO9ojVqJ0UKp3BheWxvYWTFA7J7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwMTRjODQxOWEyZWU0N2U4ZmExMTVlZGI2ZjFiM2I2ZDhkNWI3YjczOWJkNjJjYWZlMjgwM2JkYTIzNTZhMjc0NTBhIiwidWlkIjoiZmUxZTBjOTY2NTQ2MDNjNTg4MGIzZjU3MWM4NjI5MTkiLCJ1c2VybmFtZSI6IjAwZGFuaSJ9LCJwZ3BfdXBkYXRlIjp7ImZpbmdlcnByaW50IjoiMzEwYmM2ODhkNWNmMzQ4YWE3M2RlOGNjOTEzMTQyNzk0ODc4YjA2ZSIsImZ1bGxfaGFzaCI6ImNhNDg0MTA1ZTJjYTdmMDg5YjE2NzlkMzk2OGEyMDEwZmQ3MzQ2YTljMzEwYjYxMzZlMDBiMDIwZjQ2MmYxYjEiLCJrZXlfaWQiOiI5MTMxNDI3OTQ4NzhCMDZFIiwia2lkIjoiMDEwMTFkMjNiY2ZjMGM0MmI0ZDU0YWVjODEyYWE5Mzg1ZmI5NjU2YWU4ZWQwYTU1ZWQzYTZkYjcwZTdjN2E4ZmNhMGUwYSJ9LCJ0eXBlIjoicGdwX3VwZGF0ZSIsInZlcnNpb24iOjF9LCJjbGllbnQiOnsibmFtZSI6ImtleWJhc2UuaW8gZ28gY2xpZW50IiwidmVyc2lvbiI6IjEuMC4xOCJ9LCJjdGltZSI6MTQ4NzY3NzgwNSwiZXhwaXJlX2luIjo1MDQ1NzYwMDAsIm1lcmtsZV9yb290Ijp7ImN0aW1lIjoxNDg3Njc3NzkzLCJoYXNoIjoiNDJmNWY0NjI0ZWE2MGUwZWZiYmU1NGNiYWViNDYwMDExNTQ2MDViYTIwZjg4OWIxNzM1M2VlNmEzY2M4ZGJiZjFhZDJhZDQ0NGQ5MjkyNWI5MzIzZWU3NTBmZmQwNTgzMmE4ZDQzY2NhMTI2YWE4MzIzMDk4ZmUyMTM4YTFhMTIiLCJzZXFubyI6OTEwMjMzfSwicHJldiI6IjNkOTZlMDFiYjM5MDc1MzkzYjFjNzU3MjcyYThkMTg0NmJhZDI5NmFiZTIyNTg0YjY4OTFhOTVlNGYxZDk0OGYiLCJzZXFubyI6MzgsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RALtQ3XyvvogEtYMmT8qRFJW04L1sU9kAiufHowZVOuHMDhS4UvMD9YFHS0Tbass2+kUjqG1IRfPe9RaVr7ef0DKhzaWdfdHlwZSCkaGFzaIKkdHlwZQildmFsdWXEIFDsldDLe4FEzFw7PDwA+Ja2CfQLuP0uZg9vvvGL3abMo3RhZ80CAqd2ZXJzaW9uAQ==",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"012014c8419a2ee47e8fa115edb6f1b3b6d8d5b7b739bd62cafe2803bda2356a27450a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"pgp_update\":{\"fingerprint\":\"310bc688d5cf348aa73de8cc913142794878b06e\",\"full_hash\":\"ca484105e2ca7f089b1679d3968a2010fd7346a9c310b6136e00b020f462f1b1\",\"key_id\":\"913142794878B06E\",\"kid\":\"01011d23bcfc0c42b4d54aec812aa9385fb9656ae8ed0a55ed3a6db70e7c7a8fca0e0a\"},\"type\":\"pgp_update\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.18\"},\"ctime\":1487677805,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1487677793,\"hash\":\"42f5f4624ea60e0efbbe54cbaeb46001154605ba20f889b17353ee6a3cc8dbbf1ad2ad444d92925b9323ee750ffd05832a8d43cca126aa8323098fe2138a1a12\",\"seqno\":910233},\"prev\":\"3d96e01bb39075393b1c757272a8d1846bad296abe22584b6891a95e4f1d948f\",\"seqno\":38,\"tag\":\"signature\"}",
      "sig_type": 1,
      "sig_version": 1,
      "ctime": 1487677805,
      "etime": 1992253805,
      "rtime": null,
      "sig_status": 0,
      "prev": "3d96e01bb39075393b1c757272a8d1846bad296abe22584b6891a95e4f1d948f",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 39,
      "payload_hash": "b263c355d5dafbcd3f0a885272f0c654a2e53b96c363f8502c1e97a9a92c378b",
      "sig_id": "8c782ad3fd3fc926f8cbf0eaa8d8486d79f9cfc77e89509f4241f6fc985641250f",
      "sig_id_short": "jHgq0_0_ySb4y_DqqNhIbXn5z8d-iVCfQkH2",
      "kid": "0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a",
      "sig": "hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgvO0RhtCQPaHyjQfACTCE+4HxBwKLJbRpFcPYRaUM15MKp3BheWxvYWTFA7J7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwidWlkIjoiZmUxZTBjOTY2NTQ2MDNjNTg4MGIzZjU3MWM4NjI5MTkiLCJ1c2VybmFtZSI6IjAwZGFuaSJ9LCJwZ3BfdXBkYXRlIjp7ImZpbmdlcnByaW50IjoiMzEwYmM2ODhkNWNmMzQ4YWE3M2RlOGNjOTEzMTQyNzk0ODc4YjA2ZSIsImZ1bGxfaGFzaCI6ImM5MDJlMWIxYTIxNTk4ZGNkMTFlM2RiNjcxMjY4ZDZmMDA3NTZhOTIzNzZlMGFlNzdhOTNkYWEzYzEwYzk1ZWMiLCJrZXlfaWQiOiI5MTMxNDI3OTQ4NzhCMDZFIiwia2lkIjoiMDEwMTFkMjNiY2ZjMGM0MmI0ZDU0YWVjODEyYWE5Mzg1ZmI5NjU2YWU4ZWQwYTU1ZWQzYTZkYjcwZTdjN2E4ZmNhMGUwYSJ9LCJ0eXBlIjoicGdwX3VwZGF0ZSIsInZlcnNpb24iOjF9LCJjbGllbnQiOnsibmFtZSI6ImtleWJhc2UuaW8gZ28gY2xpZW50IiwidmVyc2lvbiI6IjEuMC4xOCJ9LCJjdGltZSI6MTQ4NzcyMjg1NSwiZXhwaXJlX2luIjo1MDQ1NzYwMDAsIm1lcmtsZV9yb290Ijp7ImN0aW1lIjoxNDg3NzIyODU0LCJoYXNoIjoiYTk4ZTJkYmRhYTUxM2E2MGQ5OTFkM2IyM2JlMjVlODlhOTNiZjgzYjI0MmM4NTA0MjIyMGJjOTlmZDM4ZmJiZTVmMWM3ZGQ3ZTUzOTkxZGZlZTdjZGE1YWE2OWMxMmM1MzE0ZGY1YzkwMjg0MjMwNjEyZDhmYmYyYjg0MzgxZTciLCJzZXFubyI6OTEyMjg0fSwicHJldiI6IjIzNGQ5ZjRmZGYzMWI1OWY0ZGQ3OTFlMDhmZjc2NTJjOGY0NTYwOTMxZTc3MWI2NDNmMWU0NWUyZTJkZTJhODgiLCJzZXFubyI6MzksInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RAjc/alm8Y6IHLTGTLcd3aQ/4M8JvJwiVch6rrVLV5UUwo6VdcB9/tVkAMBq2awZymUU12LtSADfouVUYBrScIC6hzaWdfdHlwZSCkaGFzaIKkdHlwZQildmFsdWXEILbFd5ZklyEq9HQlGiuR9Rj7x0HK8d1L03p33LsqWg3Ao3RhZ80CAqd2ZXJzaW9uAQ==",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"pgp_update\":{\"fingerprint\":\"310bc688d5cf348aa73de8cc913142794878b06e\",\"full_hash\":\"c902e1b1a21598dcd11e3db671268d6f00756a92376e0ae77a93daa3c10c95ec\",\"key_id\":\"913142794878B06E\",\"kid\":\"01011d23bcfc0c42b4d54aec812aa9385fb9656ae8ed0a55ed3a6db70e7c7a8fca0e0a\"},\"type\":\"pgp_update\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.18\"},\"ctime\":1487722855,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1487722854,\"hash\":\"a98e2dbdaa513a60d991d3b23be25e89a93bf83b242c85042220bc99fd38fbbe5f1c7dd7e53991dfee7cda5aa69c12c5314df5c90284230612d8fbf2b84381e7\",\"seqno\":912284},\"prev\":\"234d9f4fdf31b59f4dd791e08ff7652c8f4560931e771b643f1e45e2e2de2a88\",\"seqno\":39,\"tag\":\"signature\"}",
      "sig_type": 1,
      "sig_version": 1,
      "ctime": 1487722855,
      "etime": 1992298855,
      "rtime": null,
      "sig_status": 0,
      "prev": "234d9f4fdf31b59f4dd791e08ff7652c8f4560931e771b643f1e45e2e2de2a88",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 40,
      "payload_hash": "2ef2edc2a98ad1f70f2843b5850f728892dbd37713becfae7d013729ce2bdbdd",
      "sig_id": "ede4582da68eaac896786a14e5eaa9013ce9a5532f2bf90e3bed1e9d5cb042930f",
      "sig_id_short": "7eRYLaaOqsiWeGoU5eqpATzppVMvK_kOO-0e",
      "kid": "012014c8419a2ee47e8fa115edb6f1b3b6d8d5b7b739bd62cafe2803bda2356a27450a",
      "sig": "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",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"012014c8419a2ee47e8fa115edb6f1b3b6d8d5b7b739bd62cafe2803bda2356a27450a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"track\":{\"basics\":{\"id_version\":19,\"last_id_change\":1486906568,\"username\":\"benjbrent\"},\"id\":\"89a66ba1faf6bbda845b84d48129a500\",\"key\":{\"key_fingerprint\":\"2827eab057a6789be7d6a4444025bad742f9af6f\",\"kid\":\"0101ffe822cfd1de4871891194812756e5f7535bce4cbf8f88e6b627a89c8a3dd4640a\"},\"pgp_keys\":[{\"key_fingerprint\":\"2827eab057a6789be7d6a4444025bad742f9af6f\",\"kid\":\"0101ffe822cfd1de4871891194812756e5f7535bce4cbf8f88e6b627a89c8a3dd4640a\"}],\"remote_proofs\":[{\"ctime\":1415763104,\"curr\":\"4860bba658be4cf9f7ccf64c13983ea7879edfd8b95702ab47c4a13d2df178a3\",\"etime\":1573443104,\"prev\":\"dbf02861cabe2b962526ffcbe36f5b03993fbd36465430993e97adbc4fc48f10\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"benjbrent\"},\"proof_type\":2,\"state\":2},\"sig_id\":\"d3310b1d7b82fce5902754d27638a8148494404204912dc89e5fa09144c031330f\",\"sig_type\":2},{\"ctime\":1415763180,\"curr\":\"9837b128efe58a6b2a64f385c9c73f02f3fbb02f882dc1272b15f8c6f3c98448\",\"etime\":1573443180,\"prev\":\"4860bba658be4cf9f7ccf64c13983ea7879edfd8b95702ab47c4a13d2df178a3\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"benjaminbrent\"},\"proof_type\":3,\"state\":2},\"sig_id\":\"5ac4825925afabfc8e7c7f9b4922c97b0f69b3768f379cf927947bcf7c6f74aa0f\",\"sig_type\":2},{\"ctime\":1415763308,\"curr\":\"693c841d22eb71a628d11354e20162fbe7bf55f536f45bb4c52ff747b0995d9e\",\"etime\":1573443308,\"prev\":\"9837b128efe58a6b2a64f385c9c73f02f3fbb02f882dc1272b15f8c6f3c98448\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"benjaminbrent.com\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"sig_id\":\"ba3843a206a79022ea7408ca5e7fd8f1a1ccef5052fee303a633e760e4c8ae920f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"693c841d22eb71a628d11354e20162fbe7bf55f536f45bb4c52ff747b0995d9e\",\"seqno\":5,\"sig_id\":\"ba3843a206a79022ea7408ca5e7fd8f1a1ccef5052fee303a633e760e4c8ae920f\"}},\"type\":\"track\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.18\"},\"ctime\":1487757962,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1487757942,\"hash\":\"53318dfd18d5c687ffede68cbe18ad5a147d656838bc656a31b2c042fc3c750f4e6053c605b3b60cf686ea0bf2b512256ce78ceca5ec05d84418a750c9601ded\",\"seqno\":913539},\"prev\":\"b263c355d5dafbcd3f0a885272f0c654a2e53b96c363f8502c1e97a9a92c378b\",\"seqno\":40,\"tag\":\"signature\"}",
      "sig_type": 3,
      "sig_version": 1,
      "ctime": 1487757962,
      "etime": 1992333962,
      "rtime": null,
      "sig_status": 0,
      "prev": "b263c355d5dafbcd3f0a885272f0c654a2e53b96c363f8502c1e97a9a92c378b",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 41,
      "payload_hash": "776eefbb8056eb49e3a8ab29f8c46021ab3bf8c7cd327d6d0fa68d785ed8af51",
      "sig_id": "d1244a6f66b0146d2967be2845dffbfc397f96e7fa237cf54c3fc0809f107fad0f",
      "sig_id_short": "0SRKb2awFG0pZ74oRd_7_Dl_luf6I3z1TD_A",
      "kid": "0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a",
      "sig": "hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgvO0RhtCQPaHyjQfACTCE+4HxBwKLJbRpFcPYRaUM15MKp3BheWxvYWTFA517ImJvZHkiOnsiY3J5cHRvY3VycmVuY3kiOnsiYWRkcmVzcyI6InpjY3VwN2p0OFlqS3RNV2VyWUdFYnhudldpQmtjZ2lmTUNkWDFqY3ZBemkxRVRvZTJKUXc2NGZXTUFlc0x1WllReHpZNFZaTFI4bXlWWjdLQm53RnE3N282bXJmTGRtIiwidHlwZSI6InpjYXNoLnoifSwia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwidWlkIjoiZmUxZTBjOTY2NTQ2MDNjNTg4MGIzZjU3MWM4NjI5MTkiLCJ1c2VybmFtZSI6IjAwZGFuaSJ9LCJyZXZva2UiOnsic2lnX2lkIjoiZmFkMjlmNmQ1MjM0ZDUzNDlmMzRkYzY2Y2MzNTEzMzk5YjdmYmE2NjNiMDljNGEyNWE0ODA2YjA3YjExYzcwNDBmIn0sInR5cGUiOiJjcnlwdG9jdXJyZW5jeSIsInZlcnNpb24iOjF9LCJjbGllbnQiOnsibmFtZSI6ImtleWJhc2UuaW8gZ28gY2xpZW50IiwidmVyc2lvbiI6IjEuMC4xOCJ9LCJjdGltZSI6MTQ4NzgwMzA2OSwiZXhwaXJlX2luIjo1MDQ1NzYwMDAsIm1lcmtsZV9yb290Ijp7ImN0aW1lIjoxNDg3ODAzMDY1LCJoYXNoIjoiNTA4Zjc2MThmOTE2NzE1OTI5YTA0OTQ4NDgxMjIyZThjZTJlNjcxMWQ4ZjRiZjUwODA4NzI2NWUwODVlMDQwNTE0NWRkNWVjNTM3ZTg5Y2ExY2JiOWY0ODhmYTVhOGFlNDFkNDcxNmUyNzJhYTdiZDdjZmViY2YwNDhmNzkyZDciLCJzZXFubyI6OTE1NjAxfSwicHJldiI6IjJlZjJlZGMyYTk4YWQxZjcwZjI4NDNiNTg1MGY3Mjg4OTJkYmQzNzcxM2JlY2ZhZTdkMDEzNzI5Y2UyYmRiZGQiLCJzZXFubyI6NDEsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RA8Jg3JviH6HNlgo0wn6V+8KNfG3lvTtKdz7ulvR2Wey4nC1Szok8OFdQbj5pH6ejmR4m/Rcyhp5TE9sxEG/q0AKhzaWdfdHlwZSCkaGFzaIKkdHlwZQildmFsdWXEIKIwRTajPE4Bjv+KqyBxVTNiwkPZZ+tZb7eB8xY0IjxIo3RhZ80CAqd2ZXJzaW9uAQ==",
      "payload_json": "{\"body\":{\"cryptocurrency\":{\"address\":\"zccup7jt8YjKtMWerYGEbxnvWiBkcgifMCdX1jcvAzi1EToe2JQw64fWMAesLuZYQxzY4VZLR8myVZ7KBnwFq77o6mrfLdm\",\"type\":\"zcash.z\"},\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"revoke\":{\"sig_id\":\"fad29f6d5234d5349f34dc66cc3513399b7fba663b09c4a25a4806b07b11c7040f\"},\"type\":\"cryptocurrency\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.18\"},\"ctime\":1487803069,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1487803065,\"hash\":\"508f7618f916715929a04948481222e8ce2e6711d8f4bf508087265e085e0405145dd5ec537e89ca1cbb9f488fa5a8ae41d4716e272aa7bd7cfebcf048f792d7\",\"seqno\":915601},\"prev\":\"2ef2edc2a98ad1f70f2843b5850f728892dbd37713becfae7d013729ce2bdbdd\",\"seqno\":41,\"tag\":\"signature\"}",
      "sig_type": 6,
      "sig_version": 1,
      "ctime": 1487803069,
      "etime": 1992379069,
      "rtime": null,
      "sig_status": 0,
      "prev": "2ef2edc2a98ad1f70f2843b5850f728892dbd37713becfae7d013729ce2bdbdd",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 42,
      "payload_hash": "1a7d1e29418f1bf4121ef9a709b0ddfbbeca0c0bf7eb83ac52bd710b7bdd11a7",
      "sig_id": "38947bb203842a3d8819d98208d51f53a14cb6169a19575a497cf50ea2247acd0f",
      "sig_id_short": "OJR7sgOEKj2IGdmCCNUfU6FMthaaGVdaSXz1",
      "kid": "0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a",
      "sig": "hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgvO0RhtCQPaHyjQfACTCE+4HxBwKLJbRpFcPYRaUM15MKp3BheWxvYWTFDjd7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwidWlkIjoiZmUxZTBjOTY2NTQ2MDNjNTg4MGIzZjU3MWM4NjI5MTkiLCJ1c2VybmFtZSI6IjAwZGFuaSJ9LCJ0cmFjayI6eyJiYXNpY3MiOnsiaWRfdmVyc2lvbiI6MTI4LCJsYXN0X2lkX2NoYW5nZSI6MTQ4NzkyNTAwMCwidXNlcm5hbWUiOiJ0cnluaXR5In0sImlkIjoiN2I0YTVlNGFiYTY3N2JiMDE5NDllMTQyY2VmMWFkMDAiLCJrZXkiOnsia2V5X2ZpbmdlcnByaW50IjoiMjc0YjZiYWFlZTFlNDYzNDRkN2U4ZTM2M2FiZDE3ZTYwMGYzOTIyNyIsImtpZCI6IjAxMDEzN2Y5Zjg1ZWFlMGUxMzNmMzhlNjdmYzFmMTNhYTYxMmU4ZWUyOWYxZDIyZjE5ZTY0Yjg1NTcwY2M2OTBhOTQ2MGEifSwicGdwX2tleXMiOlt7ImtleV9maW5nZXJwcmludCI6ImVmNDQxY2EwZTAyMmVlMDQyNDk3ZjBlNzRiNjY2Yzk4YmU1MTY2YjQiLCJraWQiOiIwMTAxYjFhOGMzOTgxMjU4OWZhNTI5OWE5MmZhZDM5NTU3ZDI1YmU2MTk0ODY4YzNiNzU4OWJiZTBmYTU2MmVlY2FjMzBhIn1dLCJyZW1vdGVfcHJvb2ZzIjpbeyJjdGltZSI6MTQ3ODgxODAyMCwiY3VyciI6IjVmYjA4YThkOGI2NjY4M2QyMGVmYjA4ODE0YjM0NjBhMmQ0YTAxOWU2ZTQ5ZDc4NjQxYTQxMjRjODkzY2I0ODAiLCJldGltZSI6MTk4MzM5NDAyMCwicHJldiI6IjAxY2Q3MTE1ZDE0MmIyMWJiMzc1ZTQ5MmMwM2VhZTNhNThhZWU2MWFjNGI4NzRmYTEzZWRlNDYyNGM2NjRmNTgiLCJyZW1vdGVfa2V5X3Byb29mIjp7ImNoZWNrX2RhdGFfanNvbiI6eyJuYW1lIjoiZ2l0aHViIiwidXNlcm5hbWUiOiJ0cnluaXR5In0sInByb29mX3R5cGUiOjMsInN0YXRlIjoxfSwic2lnX2lkIjoiZDM2NGViMDc0NDY3Y2I5NjI1MmE0ZWM2NjM3NzNmNmFkZWI2ODAxYzkwYWQwODVlYzBiYWI4YzQ5ZTYwYzI2YjBmIiwic2lnX3R5cGUiOjJ9LHsiY3RpbWUiOjE0Nzg4MTgxMjAsImN1cnIiOiI2NGRhM2E4MWU3YmQ0ZDU3YTJiOWY4MDJkM2YwNThhZjI0M2FjNDU0MDliYzBhMjIzODEyODU0NTFjYmIzOTVlIiwiZXRpbWUiOjE5ODMzOTQxMjAsInByZXYiOiI1ZmIwOGE4ZDhiNjY2ODNkMjBlZmIwODgxNGIzNDYwYTJkNGEwMTllNmU0OWQ3ODY0MWE0MTI0Yzg5M2NiNDgwIiwicmVtb3RlX2tleV9wcm9vZiI6eyJjaGVja19kYXRhX2pzb24iOnsiaG9zdG5hbWUiOiJtaXJlbGwub3JnIiwicHJvdG9jb2wiOiJodHRwczoifSwicHJvb2ZfdHlwZSI6MTAwMCwic3RhdGUiOjF9LCJzaWdfaWQiOiIyNzVkNTYwYjA4MmM5ZmY2ZjBhNTA1ZTM4MTNmMTlkMWMzN2I1ZTViOGZmMTE0ZTY3MzNkZDliYmEyNGJjOTg5MGYiLCJzaWdfdHlwZSI6Mn0seyJjdGltZSI6MTQ4NzkyNDcwNywiY3VyciI6IjRhMTczNTZkMGNjZTQ1MmQxYzgxM2UyZjAzNGUwZTMwZWMzYzgzNDYyNDIyZGYwMjY4OGUzMjgyMGIyMjgwNTAiLCJldGltZSI6MTk5MjUwMDcwNywicHJldiI6ImIwYmY2MTc0ZmQ5OWI2OWI1OWY0NmEzM2VmZTU1NThjMmZiZjAzZjQ2YzBmODE0OTI4MjY2MjU2NDFiYWYxNWQiLCJyZW1vdGVfa2V5X3Byb29mIjp7ImNoZWNrX2RhdGFfanNvbiI6eyJuYW1lIjoicmVkZGl0IiwidXNlcm5hbWUiOiJldmlsZ2F5d2l0Y2gifSwicHJvb2ZfdHlwZSI6NCwic3RhdGUiOjF9LCJzaWdfaWQiOiJmNWM1MzQ3NzViNGIwMWU4MTUyYWE0Y2UzN2YxMWFlZGY5Zjk3NmJlYmZmOGNkYmMyNzIzOGZkMWQ3ZjdkMDVlMGYiLCJzaWdfdHlwZSI6Mn0seyJjdGltZSI6MTQ4NzkyNDc1MiwiY3VyciI6IjEwZTY0MmZkMGE3NWU2NmVhODE2ODNlZDYzMDk4ZjkyOWM0NzNiZWNiMjU1YWU1NmQyM2QxZDcwOTZiYzU4YTAiLCJldGltZSI6MTk5MjUwMDc1MiwicHJldiI6IjRhMTczNTZkMGNjZTQ1MmQxYzgxM2UyZjAzNGUwZTMwZWMzYzgzNDYyNDIyZGYwMjY4OGUzMjgyMGIyMjgwNTAiLCJyZW1vdGVfa2V5X3Byb29mIjp7ImNoZWNrX2RhdGFfanNvbiI6eyJuYW1lIjoiY29pbmJhc2UiLCJ1c2VybmFtZSI6InRyeW5pdHkifSwicHJvb2ZfdHlwZSI6NSwic3RhdGUiOjF9LCJzaWdfaWQiOiI4YTA2NGY2NmQyMDg5ODEzYTEwNjUxODcwZTFhZTJiMGM0YTU4YWRhNGI3ZTAwNjZjYzI0NGVmNzY1NDUxNWFiMGYiLCJzaWdfdHlwZSI6Mn0seyJjdGltZSI6MTQ4NzkyNDk4NCwiY3VyciI6IjU1ZjI3ZWYzOGIyMDlkODg2MGNjOTM0M2U2ODY1N2Y0NTE4MTkwMGU5ODI4ZGFmZjM0Y2MwYWI0YmFiYzMwNGMiLCJldGltZSI6MTk5MjUwMDk4NCwicHJldiI6IjEwZTY0MmZkMGE3NWU2NmVhODE2ODNlZDYzMDk4ZjkyOWM0NzNiZWNiMjU1YWU1NmQyM2QxZDcwOTZiYzU4YTAiLCJyZW1vdGVfa2V5X3Byb29mIjp7ImNoZWNrX2RhdGFfanNvbiI6eyJuYW1lIjoiZmFjZWJvb2siLCJ1c2VybmFtZSI6InRyeW5pdHkubWlyZWxsIn0sInByb29mX3R5cGUiOjgsInN0YXRlIjoxfSwic2lnX2lkIjoiN2YyNjFmNjY0NGZkNWFhMmE2NWJlMTE4YzU2Yjg2OWNlNzc3YjYwMjM0MWI0NjJkMzY4ZmI5NjVmZjA1YmNkMjBmIiwic2lnX3R5cGUiOjJ9LHsiY3RpbWUiOjE0Nzg4MTc5OTUsImN1cnIiOiIwMWNkNzExNWQxNDJiMjFiYjM3NWU0OTJjMDNlYWUzYTU4YWVlNjFhYzRiODc0ZmExM2VkZTQ2MjRjNjY0ZjU4IiwiZXRpbWUiOjE5ODMzOTM5OTUsInByZXYiOiIzNzA3ZTU2NDYyZmRhZTM0ZjgyMTRlN2ZlNjlhMmFlOGZhZDFjYTMzOTQxMzE2NTEwNTI2Mzg4MGJiYjI4NzNhIiwicmVtb3RlX2tleV9wcm9vZiI6eyJjaGVja19kYXRhX2pzb24iOnsibmFtZSI6InR3aXR0ZXIiLCJ1c2VybmFtZSI6ImV2aWxnYXl3aXRjaCJ9LCJwcm9vZl90eXBlIjoyLCJzdGF0ZSI6MX0sInNpZ19pZCI6Ijg4NTc2NTNhNzE3ZTIwY2ViYTkyMGI1ZmZmOWY5NjEwOTc4ZTVlNjc3MzRjOWJlNjUxMGYxYTQ0MjFmOGRiOTcwZiIsInNpZ190eXBlIjoyfV0sInNlcV90YWlsIjp7InBheWxvYWRfaGFzaCI6IjU1ZjI3ZWYzOGIyMDlkODg2MGNjOTM0M2U2ODY1N2Y0NTE4MTkwMGU5ODI4ZGFmZjM0Y2MwYWI0YmFiYzMwNGMiLCJzZXFubyI6NjQsInNpZ19pZCI6IjdmMjYxZjY2NDRmZDVhYTJhNjViZTExOGM1NmI4NjljZTc3N2I2MDIzNDFiNDYyZDM2OGZiOTY1ZmYwNWJjZDIwZiJ9fSwidHlwZSI6InRyYWNrIiwidmVyc2lvbiI6MX0sImNsaWVudCI6eyJuYW1lIjoia2V5YmFzZS5pbyBnbyBjbGllbnQiLCJ2ZXJzaW9uIjoiMS4wLjE4In0sImN0aW1lIjoxNDg4MDExNjczLCJleHBpcmVfaW4iOjUwNDU3NjAwMCwibWVya2xlX3Jvb3QiOnsiY3RpbWUiOjE0ODgwMTE2NjYsImhhc2giOiI5YmQ1NzRkMDNhMDk3ZTRiMTI1MWM0MTc2Y2E4Nzc4NGEyY2JjMjBjYzQyMjExNWViNTZjNGJmYzdiMGM3YjJmMGMwODEwMTM4NTU4ZWM4NzE2NDRiMDlmYTI3ZDYxOTI4OGI0YmI1ZDg0ODdjMDVmM2M4ZDQ0OTJmZGExZjA2YyIsInNlcW5vIjo5MjI4ODZ9LCJwcmV2IjoiNzc2ZWVmYmI4MDU2ZWI0OWUzYThhYjI5ZjhjNDYwMjFhYjNiZjhjN2NkMzI3ZDZkMGZhNjhkNzg1ZWQ4YWY1MSIsInNlcW5vIjo0MiwidGFnIjoic2lnbmF0dXJlIn2jc2lnxEB6Eg9swkpLLGp48frM2nz/7yEyNTwuc6K3wgwzCd/nVU24I1EVJieefF/9h6zokHEqI/D5FyfFSPq0XVYYwx4HqHNpZ190eXBlIKRoYXNogqR0eXBlCKV2YWx1ZcQgS/f/cSW7BIQ8tFHwGrZoA8DoPV+lKa4MUCuRhJOllzGjdGFnzQICp3ZlcnNpb24B",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"track\":{\"basics\":{\"id_version\":128,\"last_id_change\":1487925000,\"username\":\"trynity\"},\"id\":\"7b4a5e4aba677bb01949e142cef1ad00\",\"key\":{\"key_fingerprint\":\"274b6baaee1e46344d7e8e363abd17e600f39227\",\"kid\":\"010137f9f85eae0e133f38e67fc1f13aa612e8ee29f1d22f19e64b85570cc690a9460a\"},\"pgp_keys\":[{\"key_fingerprint\":\"ef441ca0e022ee042497f0e74b666c98be5166b4\",\"kid\":\"0101b1a8c39812589fa5299a92fad39557d25be6194868c3b7589bbe0fa562eecac30a\"}],\"remote_proofs\":[{\"ctime\":1478818020,\"curr\":\"5fb08a8d8b66683d20efb08814b3460a2d4a019e6e49d78641a4124c893cb480\",\"etime\":1983394020,\"prev\":\"01cd7115d142b21bb375e492c03eae3a58aee61ac4b874fa13ede4624c664f58\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"trynity\"},\"proof_type\":3,\"state\":1},\"sig_id\":\"d364eb074467cb96252a4ec663773f6adeb6801c90ad085ec0bab8c49e60c26b0f\",\"sig_type\":2},{\"ctime\":1478818120,\"curr\":\"64da3a81e7bd4d57a2b9f802d3f058af243ac45409bc0a22381285451cbb395e\",\"etime\":1983394120,\"prev\":\"5fb08a8d8b66683d20efb08814b3460a2d4a019e6e49d78641a4124c893cb480\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"mirell.org\",\"protocol\":\"https:\"},\"proof_type\":1000,\"state\":1},\"sig_id\":\"275d560b082c9ff6f0a505e3813f19d1c37b5e5b8ff114e6733dd9bba24bc9890f\",\"sig_type\":2},{\"ctime\":1487924707,\"curr\":\"4a17356d0cce452d1c813e2f034e0e30ec3c83462422df02688e32820b228050\",\"etime\":1992500707,\"prev\":\"b0bf6174fd99b69b59f46a33efe5558c2fbf03f46c0f81492826625641baf15d\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"reddit\",\"username\":\"evilgaywitch\"},\"proof_type\":4,\"state\":1},\"sig_id\":\"f5c534775b4b01e8152aa4ce37f11aedf9f976bebff8cdbc27238fd1d7f7d05e0f\",\"sig_type\":2},{\"ctime\":1487924752,\"curr\":\"10e642fd0a75e66ea81683ed63098f929c473becb255ae56d23d1d7096bc58a0\",\"etime\":1992500752,\"prev\":\"4a17356d0cce452d1c813e2f034e0e30ec3c83462422df02688e32820b228050\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"coinbase\",\"username\":\"trynity\"},\"proof_type\":5,\"state\":1},\"sig_id\":\"8a064f66d2089813a10651870e1ae2b0c4a58ada4b7e0066cc244ef7654515ab0f\",\"sig_type\":2},{\"ctime\":1487924984,\"curr\":\"55f27ef38b209d8860cc9343e68657f45181900e9828daff34cc0ab4babc304c\",\"etime\":1992500984,\"prev\":\"10e642fd0a75e66ea81683ed63098f929c473becb255ae56d23d1d7096bc58a0\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"facebook\",\"username\":\"trynity.mirell\"},\"proof_type\":8,\"state\":1},\"sig_id\":\"7f261f6644fd5aa2a65be118c56b869ce777b602341b462d368fb965ff05bcd20f\",\"sig_type\":2},{\"ctime\":1478817995,\"curr\":\"01cd7115d142b21bb375e492c03eae3a58aee61ac4b874fa13ede4624c664f58\",\"etime\":1983393995,\"prev\":\"3707e56462fdae34f8214e7fe69a2ae8fad1ca339413165105263880bbb2873a\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"evilgaywitch\"},\"proof_type\":2,\"state\":1},\"sig_id\":\"8857653a717e20ceba920b5fff9f9610978e5e67734c9be6510f1a4421f8db970f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"55f27ef38b209d8860cc9343e68657f45181900e9828daff34cc0ab4babc304c\",\"seqno\":64,\"sig_id\":\"7f261f6644fd5aa2a65be118c56b869ce777b602341b462d368fb965ff05bcd20f\"}},\"type\":\"track\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.18\"},\"ctime\":1488011673,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1488011666,\"hash\":\"9bd574d03a097e4b1251c4176ca87784a2cbc20cc422115eb56c4bfc7b0c7b2f0c0810138558ec871644b09fa27d619288b4bb5d8487c05f3c8d4492fda1f06c\",\"seqno\":922886},\"prev\":\"776eefbb8056eb49e3a8ab29f8c46021ab3bf8c7cd327d6d0fa68d785ed8af51\",\"seqno\":42,\"tag\":\"signature\"}",
      "sig_type": 3,
      "sig_version": 1,
      "ctime": 1488011673,
      "etime": 1992587673,
      "rtime": null,
      "sig_status": 0,
      "prev": "776eefbb8056eb49e3a8ab29f8c46021ab3bf8c7cd327d6d0fa68d785ed8af51",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 43,
      "payload_hash": "0205a9d2f6f844f688152ebb102b7d7cd8fa8a076c6c2b5013da981f720d5155",
      "sig_id": "6328d47c12c85230e8eeae117271038a9b5f09d3a82a87d8f8d42e88e90292060f",
      "sig_id_short": "YyjUfBLIUjDo7q4RcnEDiptfCdOoKofY-NQu",
      "kid": "0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a",
      "sig": "hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgvO0RhtCQPaHyjQfACTCE+4HxBwKLJbRpFcPYRaUM15MKp3BheWxvYWTFA7J7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwidWlkIjoiZmUxZTBjOTY2NTQ2MDNjNTg4MGIzZjU3MWM4NjI5MTkiLCJ1c2VybmFtZSI6IjAwZGFuaSJ9LCJwZ3BfdXBkYXRlIjp7ImZpbmdlcnByaW50IjoiN2ZkNzRlY2VhNmFjN2MwNzNlN2Y2MDI5MzhjNWRkYzZiZGE0ZmFlMyIsImZ1bGxfaGFzaCI6IjcxZTRhNjA0MGIxNTdkYTMyNDQ2YzY1Nzg1ZDg3ZjNjM2FjNzZkYzQ3ZWY0YThiN2RiNzY3MGQyMjE0NTliNzUiLCJrZXlfaWQiOiIzOEM1RERDNkJEQTRGQUUzIiwia2lkIjoiMDEwMTQ4MmVjZjNkY2U2NTc0M2MxZjQ4ZjFiYjZiZDE4NjhiOTBlOGU0NGM4YjM5NzUyZjE1NjlkNGRmZjE1MGY0MDQwYSJ9LCJ0eXBlIjoicGdwX3VwZGF0ZSIsInZlcnNpb24iOjF9LCJjbGllbnQiOnsibmFtZSI6ImtleWJhc2UuaW8gZ28gY2xpZW50IiwidmVyc2lvbiI6IjEuMC4xOCJ9LCJjdGltZSI6MTQ4ODMyMTE0NiwiZXhwaXJlX2luIjo1MDQ1NzYwMDAsIm1lcmtsZV9yb290Ijp7ImN0aW1lIjoxNDg4MzIxMDgxLCJoYXNoIjoiOWRiNTExOWExNmQxNjQ4ODMxZjk2NTNkZWU1NmI5ODhhZDNmZDkwNjg4YzUxMGJkNjkwNTZhOGNkNWNjNjZiOWU2NmYzNzc4ZGQyYmMwYzE4NDYwNTgzNGYyYzAwMjlkYzljYmRhMTc3ZjQxMzNlNzIxODY3MGFiMWY0MGZlOTciLCJzZXFubyI6OTMwODI3fSwicHJldiI6IjFhN2QxZTI5NDE4ZjFiZjQxMjFlZjlhNzA5YjBkZGZiYmVjYTBjMGJmN2ViODNhYzUyYmQ3MTBiN2JkZDExYTciLCJzZXFubyI6NDMsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RASZKpD2QeFRMii4KOnYsxUmlEYXCalddLroRxGvChNpep7a0DMvWSmtxN83YWlYLbPSVSbaoWKXvvfYHRwyuADKhzaWdfdHlwZSCkaGFzaIKkdHlwZQildmFsdWXEIBYeVa3glZykJTRB1TNaU/vRqvgkzYotmJ5IXdQB9Vp9o3RhZ80CAqd2ZXJzaW9uAQ==",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"pgp_update\":{\"fingerprint\":\"7fd74ecea6ac7c073e7f602938c5ddc6bda4fae3\",\"full_hash\":\"71e4a6040b157da32446c65785d87f3c3ac76dc47ef4a8b7db7670d221459b75\",\"key_id\":\"38C5DDC6BDA4FAE3\",\"kid\":\"0101482ecf3dce65743c1f48f1bb6bd1868b90e8e44c8b39752f1569d4dff150f4040a\"},\"type\":\"pgp_update\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.18\"},\"ctime\":1488321146,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1488321081,\"hash\":\"9db5119a16d1648831f9653dee56b988ad3fd90688c510bd69056a8cd5cc66b9e66f3778dd2bc0c184605834f2c0029dc9cbda177f4133e7218670ab1f40fe97\",\"seqno\":930827},\"prev\":\"1a7d1e29418f1bf4121ef9a709b0ddfbbeca0c0bf7eb83ac52bd710b7bdd11a7\",\"seqno\":43,\"tag\":\"signature\"}",
      "sig_type": 1,
      "sig_version": 1,
      "ctime": 1488321146,
      "etime": 1992897146,
      "rtime": null,
      "sig_status": 0,
      "prev": "1a7d1e29418f1bf4121ef9a709b0ddfbbeca0c0bf7eb83ac52bd710b7bdd11a7",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 44,
      "payload_hash": "b93d3183d7edca19a391d8a449400a1471b1eafef356ed1743de8cf3b571b266",
      "sig_id": "18c8c2e74798ec92b8008e7137712bf9f8ef981aaffda43426a273ad6cd64ec40f",
      "sig_id_short": "GMjC50eY7JK4AI5xN3Er-fjvmBqv_aQ0JqJz",
      "kid": "0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a",
      "sig": "hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgvO0RhtCQPaHyjQfACTCE+4HxBwKLJbRpFcPYRaUM15MKp3BheWxvYWTFA7J7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwidWlkIjoiZmUxZTBjOTY2NTQ2MDNjNTg4MGIzZjU3MWM4NjI5MTkiLCJ1c2VybmFtZSI6IjAwZGFuaSJ9LCJwZ3BfdXBkYXRlIjp7ImZpbmdlcnByaW50IjoiN2ZkNzRlY2VhNmFjN2MwNzNlN2Y2MDI5MzhjNWRkYzZiZGE0ZmFlMyIsImZ1bGxfaGFzaCI6Ijc4YWNjMWM0OGZkMDg4ZDIwODViNGNmMGYxZmUyMDljNTE2ZTM2MjlmMWRkN2ZkMDk1N2ExZjk0ZjU5ZWIzYjAiLCJrZXlfaWQiOiIzOEM1RERDNkJEQTRGQUUzIiwia2lkIjoiMDEwMTQ4MmVjZjNkY2U2NTc0M2MxZjQ4ZjFiYjZiZDE4NjhiOTBlOGU0NGM4YjM5NzUyZjE1NjlkNGRmZjE1MGY0MDQwYSJ9LCJ0eXBlIjoicGdwX3VwZGF0ZSIsInZlcnNpb24iOjF9LCJjbGllbnQiOnsibmFtZSI6ImtleWJhc2UuaW8gZ28gY2xpZW50IiwidmVyc2lvbiI6IjEuMC4xOCJ9LCJjdGltZSI6MTQ4ODMyMTcyNCwiZXhwaXJlX2luIjo1MDQ1NzYwMDAsIm1lcmtsZV9yb290Ijp7ImN0aW1lIjoxNDg4MzIxNjczLCJoYXNoIjoiMDM2ZWJkMzRhNGU4NTFmZWExMTgzNTlmYjg4ZmZmNTRmY2I5YjFkYWUzYWI2MTA0ZGEyYjgyMTI0MDQ5YzA4OGY1ZWExNDc4MGI3NmRhODgxYzQ1ZWE4NThlYjM4OGYwNTNlZTMyZGFmOGJjMWY1ZjEyMDE3ZTRmYjE3ZGFhNmUiLCJzZXFubyI6OTMwODM5fSwicHJldiI6IjAyMDVhOWQyZjZmODQ0ZjY4ODE1MmViYjEwMmI3ZDdjZDhmYThhMDc2YzZjMmI1MDEzZGE5ODFmNzIwZDUxNTUiLCJzZXFubyI6NDQsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RACvD+te9cF4Rv0HX8/0wPLw/lEiOy3Wzt+mSXTlpsfVnvuJL7UW3PiYsZ0xFvJSQW2rKCRqQZbi+z2D57CreTC6hzaWdfdHlwZSCkaGFzaIKkdHlwZQildmFsdWXEIIqBCJfDD35mLZ+KqmnJM42XCczAQXzqXl7d30JdzeEko3RhZ80CAqd2ZXJzaW9uAQ==",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"pgp_update\":{\"fingerprint\":\"7fd74ecea6ac7c073e7f602938c5ddc6bda4fae3\",\"full_hash\":\"78acc1c48fd088d2085b4cf0f1fe209c516e3629f1dd7fd0957a1f94f59eb3b0\",\"key_id\":\"38C5DDC6BDA4FAE3\",\"kid\":\"0101482ecf3dce65743c1f48f1bb6bd1868b90e8e44c8b39752f1569d4dff150f4040a\"},\"type\":\"pgp_update\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.18\"},\"ctime\":1488321724,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1488321673,\"hash\":\"036ebd34a4e851fea118359fb88fff54fcb9b1dae3ab6104da2b82124049c088f5ea14780b76da881c45ea858eb388f053ee32daf8bc1f5f12017e4fb17daa6e\",\"seqno\":930839},\"prev\":\"0205a9d2f6f844f688152ebb102b7d7cd8fa8a076c6c2b5013da981f720d5155\",\"seqno\":44,\"tag\":\"signature\"}",
      "sig_type": 1,
      "sig_version": 1,
      "ctime": 1488321724,
      "etime": 1992897724,
      "rtime": null,
      "sig_status": 0,
      "prev": "0205a9d2f6f844f688152ebb102b7d7cd8fa8a076c6c2b5013da981f720d5155",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 45,
      "payload_hash": "f0a740f8a713a3296dd6eae91108b76eb91bf54d7f1a3f24438ec31a12e93fe7",
      "sig_id": "d608af508feee103c28d28f9425758a8a24d8ad172fb751970de54f2ad87bb450f",
      "sig_id_short": "1givUI_u4QPCjSj5QldYqKJNitFy-3UZcN5U",
      "kid": "0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a",
      "sig": "hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgvO0RhtCQPaHyjQfACTCE+4HxBwKLJbRpFcPYRaUM15MKp3BheWxvYWTFA7J7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwidWlkIjoiZmUxZTBjOTY2NTQ2MDNjNTg4MGIzZjU3MWM4NjI5MTkiLCJ1c2VybmFtZSI6IjAwZGFuaSJ9LCJwZ3BfdXBkYXRlIjp7ImZpbmdlcnByaW50IjoiZWU0YTIwNGZjZTg0OTExODQ0NWUyYTYyY2M5MTU4OTcxOTAyN2U5NCIsImZ1bGxfaGFzaCI6IjdkNDViNjY0ZDc3YmI4ODhkMjZiMzY3YjMyNTgxMTRhYTVjMGU5N2UwYWI2MTkyZmZhZTFhZTI4NjY5MTNjNmIiLCJrZXlfaWQiOiJDQzkxNTg5NzE5MDI3RTk0Iiwia2lkIjoiMDEwMTZiODMyNDg2ZmQ3OGFiMjhhNGY1MmFjODE1MThiZDM2ZmVjNmUwNmQ4OTQyYTYzMjVmNzQzYzVjNjI5ZjlmM2IwYSJ9LCJ0eXBlIjoicGdwX3VwZGF0ZSIsInZlcnNpb24iOjF9LCJjbGllbnQiOnsibmFtZSI6ImtleWJhc2UuaW8gZ28gY2xpZW50IiwidmVyc2lvbiI6IjEuMC4xOCJ9LCJjdGltZSI6MTQ4ODMyMjY2MCwiZXhwaXJlX2luIjo1MDQ1NzYwMDAsIm1lcmtsZV9yb290Ijp7ImN0aW1lIjoxNDg4MzIyNTk4LCJoYXNoIjoiMjkzNjRjNjc2YzZmZDA3Njc1NWUxNzcwMmEwMTJkYmExYTA0YjczYzY2YjU2OTRkMzI3OTk2MjBiNTBjNTAzNWQxNzM2NzczMTJiNGY2N2NiYzA2NWQ2MzYxMTNjMjYxODkwNzVjM2U5NWQ4YTNkM2RiNDQwZjkwMTllMjY2YjgiLCJzZXFubyI6OTMwODg1fSwicHJldiI6ImI5M2QzMTgzZDdlZGNhMTlhMzkxZDhhNDQ5NDAwYTE0NzFiMWVhZmVmMzU2ZWQxNzQzZGU4Y2YzYjU3MWIyNjYiLCJzZXFubyI6NDUsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RAJ7OJn4I2QVfyfQNX/giQWs8JHzPOfdBiGwy5cK87mfu49N/88po8RmmpwQjVgHedjoWSBnzRwRxnqf4U0QizBqhzaWdfdHlwZSCkaGFzaIKkdHlwZQildmFsdWXEIAucxzbsstx9Or+oBfo9/4Mixyjjmcbo7rADe0gghgeSo3RhZ80CAqd2ZXJzaW9uAQ==",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"pgp_update\":{\"fingerprint\":\"ee4a204fce849118445e2a62cc91589719027e94\",\"full_hash\":\"7d45b664d77bb888d26b367b3258114aa5c0e97e0ab6192ffae1ae2866913c6b\",\"key_id\":\"CC91589719027E94\",\"kid\":\"01016b832486fd78ab28a4f52ac81518bd36fec6e06d8942a6325f743c5c629f9f3b0a\"},\"type\":\"pgp_update\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.18\"},\"ctime\":1488322660,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1488322598,\"hash\":\"29364c676c6fd076755e17702a012dba1a04b73c66b5694d32799620b50c5035d173677312b4f67cbc065d636113c26189075c3e95d8a3d3db440f9019e266b8\",\"seqno\":930885},\"prev\":\"b93d3183d7edca19a391d8a449400a1471b1eafef356ed1743de8cf3b571b266\",\"seqno\":45,\"tag\":\"signature\"}",
      "sig_type": 1,
      "sig_version": 1,
      "ctime": 1488322660,
      "etime": 1992898660,
      "rtime": null,
      "sig_status": 0,
      "prev": "b93d3183d7edca19a391d8a449400a1471b1eafef356ed1743de8cf3b571b266",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 46,
      "payload_hash": "a533e0102bb4cd9379727b4f0ed91948004376ce34633f401abf9bc1bb72f286",
      "sig_id": "e75fc14ab7c1cabb780e85da827f77b553e6391b8406f1c163788b54e08235fa0f",
      "sig_id_short": "51_BSrfByrt4DoXagn93tVPmORuEBvHBY3iL",
      "kid": "0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a",
      "sig": "hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgvO0RhtCQPaHyjQfACTCE+4HxBwKLJbRpFcPYRaUM15MKp3BheWxvYWTFAwl7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwidWlkIjoiZmUxZTBjOTY2NTQ2MDNjNTg4MGIzZjU3MWM4NjI5MTkiLCJ1c2VybmFtZSI6IjAwZGFuaSJ9LCJyZXZva2UiOnsia2lkcyI6WyIwMTAxNmI4MzI0ODZmZDc4YWIyOGE0ZjUyYWM4MTUxOGJkMzZmZWM2ZTA2ZDg5NDJhNjMyNWY3NDNjNWM2MjlmOWYzYjBhIl19LCJ0eXBlIjoicmV2b2tlIiwidmVyc2lvbiI6MX0sImNsaWVudCI6eyJuYW1lIjoia2V5YmFzZS5pbyBnbyBjbGllbnQiLCJ2ZXJzaW9uIjoiMS4wLjE4In0sImN0aW1lIjoxNDg4MzIyODk5LCJleHBpcmVfaW4iOjUwNDU3NjAwMCwibWVya2xlX3Jvb3QiOnsiY3RpbWUiOjE0ODgzMjI4OTIsImhhc2giOiIzYWI3NDljZDVlMmM2YzJmNTM4YTMxZjA5ZjdjNWU3MDhiMjdjNGIxM2FhMzg1M2UwYTgxY2EwZWMyMGM5ODRmNzk0ZjNiNWIyODcwYWY0ZmNhNTkzOGViZjMyNGU1MGRkZGUwYjQ3OGU4ODQwNTcwYTE3ZTNiNjJhN2I0MTg1NyIsInNlcW5vIjo5MzA4OTB9LCJwcmV2IjoiZjBhNzQwZjhhNzEzYTMyOTZkZDZlYWU5MTEwOGI3NmViOTFiZjU0ZDdmMWEzZjI0NDM4ZWMzMWExMmU5M2ZlNyIsInNlcW5vIjo0NiwidGFnIjoic2lnbmF0dXJlIn2jc2lnxEBNWR0yQt7hmF3G4r4MNbwC5uzNP21nanjNeoHO2if57jzi99KumFhn5iYtiewc8xe0iF2pOqEa/Rg7nvGegMQLqHNpZ190eXBlIKRoYXNogqR0eXBlCKV2YWx1ZcQgiiV098ZvGD6PO1RLsPAus6LI1uzFmOHHc0eMIB1fz5KjdGFnzQICp3ZlcnNpb24B",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"revoke\":{\"kids\":[\"01016b832486fd78ab28a4f52ac81518bd36fec6e06d8942a6325f743c5c629f9f3b0a\"]},\"type\":\"revoke\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.18\"},\"ctime\":1488322899,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1488322892,\"hash\":\"3ab749cd5e2c6c2f538a31f09f7c5e708b27c4b13aa3853e0a81ca0ec20c984f794f3b5b2870af4fca5938ebf324e50ddde0b478e8840570a17e3b62a7b41857\",\"seqno\":930890},\"prev\":\"f0a740f8a713a3296dd6eae91108b76eb91bf54d7f1a3f24438ec31a12e93fe7\",\"seqno\":46,\"tag\":\"signature\"}",
      "sig_type": 5,
      "sig_version": 1,
      "ctime": 1488322899,
      "etime": 1992898899,
      "rtime": null,
      "sig_status": 0,
      "prev": "f0a740f8a713a3296dd6eae91108b76eb91bf54d7f1a3f24438ec31a12e93fe7",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 47,
      "payload_hash": "6ecfd0df5da9498cc44cd2b058718f6d1684ee97e603c4ea6a61122b65c40f83",
      "sig_id": "396733cb73ef4071f10065d8afb627175b1229756a8a033744d718832da1470b0f",
      "sig_id_short": "OWczy3PvQHHxAGXYr7YnF1sSKXVqigM3RNcY",
      "kid": "0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a",
      "sig": "hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgvO0RhtCQPaHyjQfACTCE+4HxBwKLJbRpFcPYRaUM15MKp3BheWxvYWTFAwl7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwidWlkIjoiZmUxZTBjOTY2NTQ2MDNjNTg4MGIzZjU3MWM4NjI5MTkiLCJ1c2VybmFtZSI6IjAwZGFuaSJ9LCJyZXZva2UiOnsia2lkcyI6WyIwMTAxMWQyM2JjZmMwYzQyYjRkNTRhZWM4MTJhYTkzODVmYjk2NTZhZThlZDBhNTVlZDNhNmRiNzBlN2M3YThmY2EwZTBhIl19LCJ0eXBlIjoicmV2b2tlIiwidmVyc2lvbiI6MX0sImNsaWVudCI6eyJuYW1lIjoia2V5YmFzZS5pbyBnbyBjbGllbnQiLCJ2ZXJzaW9uIjoiMS4wLjE4In0sImN0aW1lIjoxNDg4MzIzOTEyLCJleHBpcmVfaW4iOjUwNDU3NjAwMCwibWVya2xlX3Jvb3QiOnsiY3RpbWUiOjE0ODgzMjM5MDEsImhhc2giOiJjMzU2MmEyYmNjMWIwMThjMjJiNmE0MDU1ZWFmYjdiZDIzZTM3MTEwMGQzMTRjNzY4YTE4YzFhMWQ0ZWIwNTc0YzAyMDljOWUwMGI4NjRjOWY5ZGQ4NDE0ODlmZTY1YzQ1ZDI3NGVhZWRjM2Q0ZTQ2NWRjYzgxZjhkZDQwODE4ZCIsInNlcW5vIjo5MzA5MTR9LCJwcmV2IjoiYTUzM2UwMTAyYmI0Y2Q5Mzc5NzI3YjRmMGVkOTE5NDgwMDQzNzZjZTM0NjMzZjQwMWFiZjliYzFiYjcyZjI4NiIsInNlcW5vIjo0NywidGFnIjoic2lnbmF0dXJlIn2jc2lnxEAgZnxMuv/TBi4cWoN5QUHcBXmIbKYKugKRXFnTxI2QkzV04MrWyqkt446BeTX2q+cG4sJIgyHWa04aA/R+RCcLqHNpZ190eXBlIKRoYXNogqR0eXBlCKV2YWx1ZcQguFkfGjIkVpmQK3GZx2PSm0sKOR8hPN3V1S9AYsOPvlWjdGFnzQICp3ZlcnNpb24B",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"revoke\":{\"kids\":[\"01011d23bcfc0c42b4d54aec812aa9385fb9656ae8ed0a55ed3a6db70e7c7a8fca0e0a\"]},\"type\":\"revoke\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.18\"},\"ctime\":1488323912,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1488323901,\"hash\":\"c3562a2bcc1b018c22b6a4055eafb7bd23e371100d314c768a18c1a1d4eb0574c0209c9e00b864c9f9dd841489fe65c45d274eaedc3d4e465dcc81f8dd40818d\",\"seqno\":930914},\"prev\":\"a533e0102bb4cd9379727b4f0ed91948004376ce34633f401abf9bc1bb72f286\",\"seqno\":47,\"tag\":\"signature\"}",
      "sig_type": 5,
      "sig_version": 1,
      "ctime": 1488323912,
      "etime": 1992899912,
      "rtime": null,
      "sig_status": 0,
      "prev": "a533e0102bb4cd9379727b4f0ed91948004376ce34633f401abf9bc1bb72f286",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 48,
      "payload_hash": "571cba79301ddcf7fc35279427ea73b746427eea4d3312a03faddb035308a9a0",
      "sig_id": "cd827fc2e3eebc681f9d6690ae7d581599aacf93dd32f9f341046a55323e74bf0f",
      "sig_id_short": "zYJ_wuPuvGgfnWaQrn1YFZmqz5PdMvnzQQRq",
      "kid": "012014c8419a2ee47e8fa115edb6f1b3b6d8d5b7b739bd62cafe2803bda2356a27450a",
      "sig": "hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgFMhBmi7kfo+hFe228bO22NW3tzm9Ysr+KAO9ojVqJ0UKp3BheWxvYWTFA7J7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwMTRjODQxOWEyZWU0N2U4ZmExMTVlZGI2ZjFiM2I2ZDhkNWI3YjczOWJkNjJjYWZlMjgwM2JkYTIzNTZhMjc0NTBhIiwidWlkIjoiZmUxZTBjOTY2NTQ2MDNjNTg4MGIzZjU3MWM4NjI5MTkiLCJ1c2VybmFtZSI6IjAwZGFuaSJ9LCJwZ3BfdXBkYXRlIjp7ImZpbmdlcnByaW50IjoiN2ZkNzRlY2VhNmFjN2MwNzNlN2Y2MDI5MzhjNWRkYzZiZGE0ZmFlMyIsImZ1bGxfaGFzaCI6IjFhNzkyOWFmZTUzZTQ3YjdkOWFmNzU1MjQ3YmJmOWFmMDNhMzllOWY2MWY1MGIyYzI1NGFjMzY2NGVkNzFlMDkiLCJrZXlfaWQiOiIzOEM1RERDNkJEQTRGQUUzIiwia2lkIjoiMDEwMTQ4MmVjZjNkY2U2NTc0M2MxZjQ4ZjFiYjZiZDE4NjhiOTBlOGU0NGM4YjM5NzUyZjE1NjlkNGRmZjE1MGY0MDQwYSJ9LCJ0eXBlIjoicGdwX3VwZGF0ZSIsInZlcnNpb24iOjF9LCJjbGllbnQiOnsibmFtZSI6ImtleWJhc2UuaW8gZ28gY2xpZW50IiwidmVyc2lvbiI6IjEuMC4xOCJ9LCJjdGltZSI6MTQ4ODM2NDI2NiwiZXhwaXJlX2luIjo1MDQ1NzYwMDAsIm1lcmtsZV9yb290Ijp7ImN0aW1lIjoxNDg4MzY0MjM1LCJoYXNoIjoiYmEzYTU4Mzg5NDRkOTVkMzI0NjZjNjFlODcyODk5NDE2NWYyODYyODIzOGUzZjY2Y2RlOTQ1ZGYwYmU2Y2JjYWFiYWUzODAzNWUzZjE1NmMyMzQ0ZDIzZGM5ODA4ZTM2MjI3YjRlMGM2NTc0NGZmNzZhY2I4ODRmYzNhMTNhM2QiLCJzZXFubyI6OTMxNzk5fSwicHJldiI6IjZlY2ZkMGRmNWRhOTQ5OGNjNDRjZDJiMDU4NzE4ZjZkMTY4NGVlOTdlNjAzYzRlYTZhNjExMjJiNjVjNDBmODMiLCJzZXFubyI6NDgsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RAh8ajYWz1vryyZ2VuGLSEsZDJ/bEeIH2TJGCSEi3hFJ1akvau8tcT4euegM7E2gsoTmq9XQj6sgHH/x64Er03DKhzaWdfdHlwZSCkaGFzaIKkdHlwZQildmFsdWXEICgowmkXA6h86Bnu3XGDcEBEsHIpECYe7rlfV3h29ptuo3RhZ80CAqd2ZXJzaW9uAQ==",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"012014c8419a2ee47e8fa115edb6f1b3b6d8d5b7b739bd62cafe2803bda2356a27450a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"pgp_update\":{\"fingerprint\":\"7fd74ecea6ac7c073e7f602938c5ddc6bda4fae3\",\"full_hash\":\"1a7929afe53e47b7d9af755247bbf9af03a39e9f61f50b2c254ac3664ed71e09\",\"key_id\":\"38C5DDC6BDA4FAE3\",\"kid\":\"0101482ecf3dce65743c1f48f1bb6bd1868b90e8e44c8b39752f1569d4dff150f4040a\"},\"type\":\"pgp_update\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.18\"},\"ctime\":1488364266,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1488364235,\"hash\":\"ba3a5838944d95d32466c61e8728994165f28628238e3f66cde945df0be6cbcaabae38035e3f156c2344d23dc9808e36227b4e0c65744ff76acb884fc3a13a3d\",\"seqno\":931799},\"prev\":\"6ecfd0df5da9498cc44cd2b058718f6d1684ee97e603c4ea6a61122b65c40f83\",\"seqno\":48,\"tag\":\"signature\"}",
      "sig_type": 1,
      "sig_version": 1,
      "ctime": 1488364266,
      "etime": 1992940266,
      "rtime": null,
      "sig_status": 0,
      "prev": "6ecfd0df5da9498cc44cd2b058718f6d1684ee97e603c4ea6a61122b65c40f83",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 49,
      "payload_hash": "4038c0d4a0fbdd4c5b95266db5da7d378745eabe03504b7d5f9c98798d35e5a6",
      "sig_id": "c3bade08c7590a2aa75488b1ae6cac43130cdb00b23527d48c76a873bcf1ecda0f",
      "sig_id_short": "w7reCMdZCiqnVIixrmysQxMM2wCyNSfUjHao",
      "kid": "0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a",
      "sig": "hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgvO0RhtCQPaHyjQfACTCE+4HxBwKLJbRpFcPYRaUM15MKp3BheWxvYWTFA7J7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwidWlkIjoiZmUxZTBjOTY2NTQ2MDNjNTg4MGIzZjU3MWM4NjI5MTkiLCJ1c2VybmFtZSI6IjAwZGFuaSJ9LCJwZ3BfdXBkYXRlIjp7ImZpbmdlcnByaW50IjoiN2ZkNzRlY2VhNmFjN2MwNzNlN2Y2MDI5MzhjNWRkYzZiZGE0ZmFlMyIsImZ1bGxfaGFzaCI6ImUyM2U4Nzk4NGUwYzBlY2ZhODQwZmY5ZWU1ODAxZTZiMTg4NGNkZTI2ODg5OGEzZDg5MzViZWI2OWU5MzZlMGQiLCJrZXlfaWQiOiIzOEM1RERDNkJEQTRGQUUzIiwia2lkIjoiMDEwMTQ4MmVjZjNkY2U2NTc0M2MxZjQ4ZjFiYjZiZDE4NjhiOTBlOGU0NGM4YjM5NzUyZjE1NjlkNGRmZjE1MGY0MDQwYSJ9LCJ0eXBlIjoicGdwX3VwZGF0ZSIsInZlcnNpb24iOjF9LCJjbGllbnQiOnsibmFtZSI6ImtleWJhc2UuaW8gZ28gY2xpZW50IiwidmVyc2lvbiI6IjEuMC4xOCJ9LCJjdGltZSI6MTQ4ODc4MjQ2MywiZXhwaXJlX2luIjo1MDQ1NzYwMDAsIm1lcmtsZV9yb290Ijp7ImN0aW1lIjoxNDg4NzgyMzk3LCJoYXNoIjoiN2JmNTMzZTM1ZTJjYThmMmMwNmJkNGI4ZjdkNzIzNWEyOWUwZmYzZDI4NzU0NDA3MzU1NjNmYTUwNzY1YWU4M2ZiMjkzYzhiMWQwODIwOGVjZjUyMWU1NjdmMTIzMmE2ZmQ4ODg5ZGY1MmQxMjY4MDMwMjY1ZjVkNWM4NDBlNzUiLCJzZXFubyI6OTQwOTk5fSwicHJldiI6IjU3MWNiYTc5MzAxZGRjZjdmYzM1Mjc5NDI3ZWE3M2I3NDY0MjdlZWE0ZDMzMTJhMDNmYWRkYjAzNTMwOGE5YTAiLCJzZXFubyI6NDksInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RAZGHENiGZE3Qf35HUY4YjgRnS55/5NOP7b5VdfLKMsnrMxO6BLTk6EYG7tYdtqkNcQ9vnEUSKdIermhZp31ySAqhzaWdfdHlwZSCkaGFzaIKkdHlwZQildmFsdWXEINqaEOWIadMkqqYXWNPp1ugxmWjS+EnXLDJFU/R/VF7wo3RhZ80CAqd2ZXJzaW9uAQ==",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"pgp_update\":{\"fingerprint\":\"7fd74ecea6ac7c073e7f602938c5ddc6bda4fae3\",\"full_hash\":\"e23e87984e0c0ecfa840ff9ee5801e6b1884cde268898a3d8935beb69e936e0d\",\"key_id\":\"38C5DDC6BDA4FAE3\",\"kid\":\"0101482ecf3dce65743c1f48f1bb6bd1868b90e8e44c8b39752f1569d4dff150f4040a\"},\"type\":\"pgp_update\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.18\"},\"ctime\":1488782463,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1488782397,\"hash\":\"7bf533e35e2ca8f2c06bd4b8f7d7235a29e0ff3d2875440735563fa50765ae83fb293c8b1d08208ecf521e567f1232a6fd8889df52d1268030265f5d5c840e75\",\"seqno\":940999},\"prev\":\"571cba79301ddcf7fc35279427ea73b746427eea4d3312a03faddb035308a9a0\",\"seqno\":49,\"tag\":\"signature\"}",
      "sig_type": 1,
      "sig_version": 1,
      "ctime": 1488782463,
      "etime": 1993358463,
      "rtime": null,
      "sig_status": 0,
      "prev": "571cba79301ddcf7fc35279427ea73b746427eea4d3312a03faddb035308a9a0",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 50,
      "payload_hash": "fceded7e0ef807f65e3fc05fe7ae09077623469fdc0f371e5e30640863a0502f",
      "sig_id": "5d1d5912b72472350abff10569773ae03a59cc971865cb9216967c337910627d0f",
      "sig_id_short": "XR1ZErckcjUKv_EFaXc64DpZzJcYZcuSFpZ8",
      "kid": "0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a",
      "sig": "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",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"track\":{\"basics\":{\"id_version\":10,\"last_id_change\":1489060197,\"username\":\"flowblok\"},\"id\":\"ba0bc3b2ccc2b128412821ac14e57719\",\"key\":{\"key_fingerprint\":\"410d28f01dfb0dc40e0cd663e060b7cd3906c4af\",\"kid\":\"0101d75b4bb2eb3ea0d70ff3914897bcbec0855abb7962b1a99c8702d5afdf169a370a\"},\"pgp_keys\":[{\"key_fingerprint\":\"410d28f01dfb0dc40e0cd663e060b7cd3906c4af\",\"kid\":\"0101d75b4bb2eb3ea0d70ff3914897bcbec0855abb7962b1a99c8702d5afdf169a370a\"}],\"remote_proofs\":[{\"ctime\":1489057545,\"curr\":\"5d26d179a59c7ad952cedf16d7b14da4b88e2bcb68e5c556a8d7898f382372c1\",\"etime\":1646737545,\"prev\":\"d746342b4353808eded12a873be36a42d038a86d649985ab13a94510cc5b0856\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"flowblok\"},\"proof_type\":2,\"state\":1},\"sig_id\":\"e847d6295eef684506f616007bbfc2de822325f63c2e85592272f409bfc78c220f\",\"sig_type\":2},{\"ctime\":1489057579,\"curr\":\"a8ac9607a34103795443da3dd82672ce96d5776b81b83fcd19b156cdb3361aa7\",\"etime\":1646737579,\"prev\":\"5d26d179a59c7ad952cedf16d7b14da4b88e2bcb68e5c556a8d7898f382372c1\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"facebook\",\"username\":\"flowblok\"},\"proof_type\":8,\"state\":10},\"sig_id\":\"8e3c326067e03f0dfc57292a8e8df6322e1a7534b619434c4befc4e1b1fed8e70f\",\"sig_type\":2},{\"ctime\":1489057613,\"curr\":\"11bbaa86a13ccc33eccdabe5d79f2be28edfe565b33333940dc9fd796fec5f64\",\"etime\":1646737613,\"prev\":\"a8ac9607a34103795443da3dd82672ce96d5776b81b83fcd19b156cdb3361aa7\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"flowblok.id.au\",\"protocol\":\"http:\"},\"proof_type\":1000,\"state\":10},\"sig_id\":\"07acbdeb34a8be19c71816d5c3c4113c151d28d86aeca54cd5be6b2074f6f5e60f\",\"sig_type\":2},{\"ctime\":1489057684,\"curr\":\"c7eb564c8663f03039524544885fcad2283827dfb3431b41664a23258c019d8e\",\"etime\":1646737684,\"prev\":\"11bbaa86a13ccc33eccdabe5d79f2be28edfe565b33333940dc9fd796fec5f64\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"flowblok.id.au\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"sig_id\":\"0e1e75a2947957cd0abcddfbdc8b33a887b4194b27e79ea71e662e1535774d1a0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"c7eb564c8663f03039524544885fcad2283827dfb3431b41664a23258c019d8e\",\"seqno\":5,\"sig_id\":\"0e1e75a2947957cd0abcddfbdc8b33a887b4194b27e79ea71e662e1535774d1a0f\"}},\"type\":\"track\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.18\"},\"ctime\":1489061177,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1489061166,\"hash\":\"a1ebe9326320e2a496303fe84fb2ace9f9fe201adcecb9680f74b28a6ddfcf25a3a0f6cfbe0eb022c4d2149673b1bb8bfa427df890ee6f8e9a24cab80f43f779\",\"seqno\":948238},\"prev\":\"4038c0d4a0fbdd4c5b95266db5da7d378745eabe03504b7d5f9c98798d35e5a6\",\"seqno\":50,\"tag\":\"signature\"}",
      "sig_type": 3,
      "sig_version": 1,
      "ctime": 1489061177,
      "etime": 1993637177,
      "rtime": null,
      "sig_status": 0,
      "prev": "4038c0d4a0fbdd4c5b95266db5da7d378745eabe03504b7d5f9c98798d35e5a6",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 51,
      "payload_hash": "f26b3207e9823780c365849d0d79b182e186fa623773ae6f516bb5e6001d63eb",
      "sig_id": "cfbbbe0c2fef7e5cab9add04fd4a8c53af1942a779120e1fcfadc92e1f45d1600f",
      "sig_id_short": "z7u-DC_vflyrmt0E_UqMU68ZQqd5Eg4fz63J",
      "kid": "0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a",
      "sig": "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",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"track\":{\"basics\":{\"id_version\":18,\"last_id_change\":1483915483,\"username\":\"deli73\"},\"id\":\"45633eb852e46b3475009c6dd8159219\",\"key\":{\"key_fingerprint\":\"50385f63b7d4d26e4ff2b4f3f31670e06ff32d60\",\"kid\":\"0101d899366a8a2651d448547c3dcc67830f7bd9a8566ea1b73de5f050ed68de46c50a\"},\"pgp_keys\":[{\"key_fingerprint\":\"50385f63b7d4d26e4ff2b4f3f31670e06ff32d60\",\"kid\":\"0101d899366a8a2651d448547c3dcc67830f7bd9a8566ea1b73de5f050ed68de46c50a\"}],\"remote_proofs\":[{\"ctime\":1452133318,\"curr\":\"5a059c4a8aad7841cb0790facd71e875e618dc05bc86c8a65ad86c8f64aa73bb\",\"etime\":1609813318,\"prev\":\"25f370e677bbbb932af2062d2162c1385313cc87b9053b8bc1c8171e13b823c7\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"deli731234\"},\"proof_type\":2,\"state\":1},\"sig_id\":\"e174f7a352e268b054444ad7f5d50699bfdbf46ca519fb64094ee8ccb85d2ffb0f\",\"sig_type\":2},{\"ctime\":1452133596,\"curr\":\"69c5c774d21471f0d8b665493f82c233550ddbf1e171bddbee7a7ab3c042e7a6\",\"etime\":1609813596,\"prev\":\"5a059c4a8aad7841cb0790facd71e875e618dc05bc86c8a65ad86c8f64aa73bb\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"www.deli73.com\",\"protocol\":\"http:\"},\"proof_type\":1000,\"state\":2},\"sig_id\":\"4224e915713559917196855a19965511f9b4c61bf1331abb71ebc6ee3d6e83e60f\",\"sig_type\":2},{\"ctime\":1452136924,\"curr\":\"3d24411e2e34f2da8d80ac8db1e99b28f4fa5de30b5e9fd98f8f7c22dbfe5dcb\",\"etime\":1609816924,\"prev\":\"69c5c774d21471f0d8b665493f82c233550ddbf1e171bddbee7a7ab3c042e7a6\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"deli73\"},\"proof_type\":3,\"state\":1},\"sig_id\":\"9ab6235ea53afbc3f99a6d56d37b5be0aedae974c2d59df01775ca70d911a31f0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"0ce7f0b530693d7106f448d010ab3f46d8d331112a4d632ab5dcb3690f2e1085\",\"seqno\":6,\"sig_id\":\"15ab4e2a5e82eea55d1f4398fef7bd402e76cfe341cb8806ead2a75b7668e5860f\"}},\"type\":\"track\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.20\"},\"ctime\":1489097363,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1489097358,\"hash\":\"5db96210cdf808210652ab342a3ffdf9b7fd65f3dc1717b2069e6c0a8115d1477d55ccfcfaa6bf80be6566bf303c2b500952692a6bc2397130455c5a9f4d7053\",\"seqno\":949424},\"prev\":\"fceded7e0ef807f65e3fc05fe7ae09077623469fdc0f371e5e30640863a0502f\",\"seqno\":51,\"tag\":\"signature\"}",
      "sig_type": 3,
      "sig_version": 1,
      "ctime": 1489097363,
      "etime": 1993673363,
      "rtime": null,
      "sig_status": 0,
      "prev": "fceded7e0ef807f65e3fc05fe7ae09077623469fdc0f371e5e30640863a0502f",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 52,
      "payload_hash": "0be27f8d5e48c7daf4cb6a863af1054315af245530aebd4c57771833f560abe9",
      "sig_id": "80c02514ee5122858505599a76a23c084f5272780d7713dc561554f5b08402070f",
      "sig_id_short": "gMAlFO5RIoWFBVmadqI8CE9ScngNdxPcVhVU",
      "kid": "0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a",
      "sig": "hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgvO0RhtCQPaHyjQfACTCE+4HxBwKLJbRpFcPYRaUM15MKp3BheWxvYWTFA7J7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwidWlkIjoiZmUxZTBjOTY2NTQ2MDNjNTg4MGIzZjU3MWM4NjI5MTkiLCJ1c2VybmFtZSI6IjAwZGFuaSJ9LCJwZ3BfdXBkYXRlIjp7ImZpbmdlcnByaW50IjoiN2ZkNzRlY2VhNmFjN2MwNzNlN2Y2MDI5MzhjNWRkYzZiZGE0ZmFlMyIsImZ1bGxfaGFzaCI6IjhmZDJiN2Y4ZmVmNDA3YTg2ZWFhZjY0ZTljYjI1NTI0ZDMwYzQwZGZiNmIwZjFmYTQ0MjgwYzFhYjI0MzYwYTciLCJrZXlfaWQiOiIzOEM1RERDNkJEQTRGQUUzIiwia2lkIjoiMDEwMTQ4MmVjZjNkY2U2NTc0M2MxZjQ4ZjFiYjZiZDE4NjhiOTBlOGU0NGM4YjM5NzUyZjE1NjlkNGRmZjE1MGY0MDQwYSJ9LCJ0eXBlIjoicGdwX3VwZGF0ZSIsInZlcnNpb24iOjF9LCJjbGllbnQiOnsibmFtZSI6ImtleWJhc2UuaW8gZ28gY2xpZW50IiwidmVyc2lvbiI6IjEuMC4yMCJ9LCJjdGltZSI6MTQ4OTYxOTI3OSwiZXhwaXJlX2luIjo1MDQ1NzYwMDAsIm1lcmtsZV9yb290Ijp7ImN0aW1lIjoxNDg5NjE5MTY0LCJoYXNoIjoiYzA3YmE5MWViMjQzYjE0NDMwMGViZjQ0YzMxN2JiMTRhMzdjZTZhNGMwY2MyZjdiODY3NTY0Y2Q1NjNmNGY0OTVhMjFkOWNiZDMzZTYzNmFkMTQzN2U3OGY5ZDY1ODcxMTMyNzIxMTIwZjlkZDAwYjQ1YzRjYTcxZjQyYjQ1ZjkiLCJzZXFubyI6OTYxNDM4fSwicHJldiI6ImYyNmIzMjA3ZTk4MjM3ODBjMzY1ODQ5ZDBkNzliMTgyZTE4NmZhNjIzNzczYWU2ZjUxNmJiNWU2MDAxZDYzZWIiLCJzZXFubyI6NTIsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RAcCdctVygt7W10tV1bAD67oysc5QGiYfkgilu7ZRCGUlAXVhIUuB9HF5FMaJtq5xrOT/b/TgeCJLB1Kex5QkMAqhzaWdfdHlwZSCkaGFzaIKkdHlwZQildmFsdWXEIPJBs85UVlSxW+PBn0fGa2pa0qN+bJVjWV1SiJrPduySo3RhZ80CAqd2ZXJzaW9uAQ==",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"pgp_update\":{\"fingerprint\":\"7fd74ecea6ac7c073e7f602938c5ddc6bda4fae3\",\"full_hash\":\"8fd2b7f8fef407a86eaaf64e9cb25524d30c40dfb6b0f1fa44280c1ab24360a7\",\"key_id\":\"38C5DDC6BDA4FAE3\",\"kid\":\"0101482ecf3dce65743c1f48f1bb6bd1868b90e8e44c8b39752f1569d4dff150f4040a\"},\"type\":\"pgp_update\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.20\"},\"ctime\":1489619279,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1489619164,\"hash\":\"c07ba91eb243b144300ebf44c317bb14a37ce6a4c0cc2f7b867564cd563f4f495a21d9cbd33e636ad1437e78f9d65871132721120f9dd00b45c4ca71f42b45f9\",\"seqno\":961438},\"prev\":\"f26b3207e9823780c365849d0d79b182e186fa623773ae6f516bb5e6001d63eb\",\"seqno\":52,\"tag\":\"signature\"}",
      "sig_type": 1,
      "sig_version": 1,
      "ctime": 1489619279,
      "etime": 1994195279,
      "rtime": null,
      "sig_status": 0,
      "prev": "f26b3207e9823780c365849d0d79b182e186fa623773ae6f516bb5e6001d63eb",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 53,
      "payload_hash": "d71b8133da3153094fb0011fb4cdf180575c928684482a83b2688c04438f2641",
      "sig_id": "acb031a40b67adb40dac0a8dbeaa522702385801edcf4c923d0bb3661106e8170f",
      "sig_id_short": "rLAxpAtnrbQNrAqNvqpSJwI4WAHtz0ySPQuz",
      "kid": "0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a",
      "sig": "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",
      "payload_json": "{\"body\":{\"device\":{\"id\":\"885093956ef5eb10d7d35c80d92c8b18\",\"kid\":\"0120b1bf850fde5bebdbd59bb3d42496685473111f6404086b2049733d0a5cb2c7800a\",\"name\":\"reaper-3\",\"status\":1,\"type\":\"desktop\"},\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"sibkey\":{\"kid\":\"0120b1bf850fde5bebdbd59bb3d42496685473111f6404086b2049733d0a5cb2c7800a\",\"reverse_sig\":\"hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgsb+FD95b69vVm7PUJJZoVHMRH2QECGsgSXM9Clyyx4AKp3BheWxvYWTFA8l7ImJvZHkiOnsiZGV2aWNlIjp7ImlkIjoiODg1MDkzOTU2ZWY1ZWIxMGQ3ZDM1YzgwZDkyYzhiMTgiLCJraWQiOiIwMTIwYjFiZjg1MGZkZTViZWJkYmQ1OWJiM2Q0MjQ5NjY4NTQ3MzExMWY2NDA0MDg2YjIwNDk3MzNkMGE1Y2IyYzc4MDBhIiwibmFtZSI6InJlYXBlci0zIiwic3RhdHVzIjoxLCJ0eXBlIjoiZGVza3RvcCJ9LCJrZXkiOnsiZWxkZXN0X2tpZCI6IjAxMjBiY2VkMTE4NmQwOTAzZGExZjI4ZDA3YzAwOTMwODRmYjgxZjEwNzAyOGIyNWI0NjkxNWMzZDg0NWE1MGNkNzkzMGEiLCJob3N0Ijoia2V5YmFzZS5pbyIsImtpZCI6IjAxMjBiY2VkMTE4NmQwOTAzZGExZjI4ZDA3YzAwOTMwODRmYjgxZjEwNzAyOGIyNWI0NjkxNWMzZDg0NWE1MGNkNzkzMGEiLCJ1aWQiOiJmZTFlMGM5NjY1NDYwM2M1ODgwYjNmNTcxYzg2MjkxOSIsInVzZXJuYW1lIjoiMDBkYW5pIn0sInNpYmtleSI6eyJraWQiOiIwMTIwYjFiZjg1MGZkZTViZWJkYmQ1OWJiM2Q0MjQ5NjY4NTQ3MzExMWY2NDA0MDg2YjIwNDk3MzNkMGE1Y2IyYzc4MDBhIiwicmV2ZXJzZV9zaWciOm51bGx9LCJ0eXBlIjoic2lia2V5IiwidmVyc2lvbiI6MX0sImNsaWVudCI6eyJuYW1lIjoia2V5YmFzZS5pbyBnbyBjbGllbnQiLCJ2ZXJzaW9uIjoiMS4wLjIwIn0sImN0aW1lIjoxNDkwMTg0Mzc1LCJleHBpcmVfaW4iOjUwNDU3NjAwMCwibWVya2xlX3Jvb3QiOnsiY3RpbWUiOjE0OTAxODQyNTEsImhhc2giOiIzMDI2NTVhNzFhM2RmOGI0ZjFkYWM1ZTVhZWRhYzdmN2U0Mzk4YzZiZTg2M2E3Zjk2YmY5ZmM3M2Y3NGVmNzk3YjdmNzQzMTA3MGI2ZmNjMTlhNTk3ZWEzYWM4MjE3NTIxNWZjZjQyNmVlODMyOGE4YzJmZWM0OTliYmM2ZTYyNiIsInNlcW5vIjo5NzI2NTV9LCJwcmV2IjoiMGJlMjdmOGQ1ZTQ4YzdkYWY0Y2I2YTg2M2FmMTA1NDMxNWFmMjQ1NTMwYWViZDRjNTc3NzE4MzNmNTYwYWJlOSIsInNlcW5vIjo1MywidGFnIjoic2lnbmF0dXJlIn2jc2lnxECQ59sz7kT/WB/Sey/k5LebfNJCmTi4ZGfaCPyJNv0aegFHWcBTat16zyB3pw0OnGn11AulSH5Xq6goQJyrkO4HqHNpZ190eXBlIKRoYXNogqR0eXBlCKV2YWx1ZcQgrhMvV9TpPLndKdsqZTL3ElhF4barO4t0Hl+JIpvJD4+jdGFnzQICp3ZlcnNpb24B\"},\"type\":\"sibkey\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.20\"},\"ctime\":1490184375,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1490184251,\"hash\":\"302655a71a3df8b4f1dac5e5aedac7f7e4398c6be863a7f96bf9fc73f74ef797b7f7431070b6fcc19a597ea3ac82175215fcf426ee8328a8c2fec499bbc6e626\",\"seqno\":972655},\"prev\":\"0be27f8d5e48c7daf4cb6a863af1054315af245530aebd4c57771833f560abe9\",\"seqno\":53,\"tag\":\"signature\"}",
      "sig_type": 1,
      "sig_version": 1,
      "ctime": 1490184375,
      "etime": 1994760375,
      "rtime": null,
      "sig_status": 0,
      "prev": "0be27f8d5e48c7daf4cb6a863af1054315af245530aebd4c57771833f560abe9",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 54,
      "payload_hash": "f53365f377dc45f76ab83decc543ba225a9a4ecc01092fb304594354f225508f",
      "sig_id": "8b24e979df708d9c332399d16a085aee1ef094e208e4e61e5c8f123ca24e27270f",
      "sig_id_short": "iyTped9wjZwzI5nRagha7h7wlOII5OYeXI8S",
      "kid": "0120b1bf850fde5bebdbd59bb3d42496685473111f6404086b2049733d0a5cb2c7800a",
      "sig": "hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgsb+FD95b69vVm7PUJJZoVHMRH2QECGsgSXM9Clyyx4AKp3BheWxvYWTFA+l7ImJvZHkiOnsiZGV2aWNlIjp7ImlkIjoiODg1MDkzOTU2ZWY1ZWIxMGQ3ZDM1YzgwZDkyYzhiMTgiLCJraWQiOiIwMTIxNTFkYTM3YWFiYThmZTYxZDI4ZDRmNjIwYTY4ZDFiNzMyNjM4NDA1NjRjMWEyZTJmNzI3YTBhZTk1M2M3ODU1MjBhIiwic3RhdHVzIjoxfSwia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwYjFiZjg1MGZkZTViZWJkYmQ1OWJiM2Q0MjQ5NjY4NTQ3MzExMWY2NDA0MDg2YjIwNDk3MzNkMGE1Y2IyYzc4MDBhIiwidWlkIjoiZmUxZTBjOTY2NTQ2MDNjNTg4MGIzZjU3MWM4NjI5MTkiLCJ1c2VybmFtZSI6IjAwZGFuaSJ9LCJzdWJrZXkiOnsia2lkIjoiMDEyMTUxZGEzN2FhYmE4ZmU2MWQyOGQ0ZjYyMGE2OGQxYjczMjYzODQwNTY0YzFhMmUyZjcyN2EwYWU5NTNjNzg1NTIwYSIsInBhcmVudF9raWQiOiIwMTIwYjFiZjg1MGZkZTViZWJkYmQ1OWJiM2Q0MjQ5NjY4NTQ3MzExMWY2NDA0MDg2YjIwNDk3MzNkMGE1Y2IyYzc4MDBhIn0sInR5cGUiOiJzdWJrZXkiLCJ2ZXJzaW9uIjoxfSwiY2xpZW50Ijp7Im5hbWUiOiJrZXliYXNlLmlvIGdvIGNsaWVudCIsInZlcnNpb24iOiIxLjAuMjAifSwiY3RpbWUiOjE0OTAxODQzODcsImV4cGlyZV9pbiI6NTA0NTc2MDAwLCJtZXJrbGVfcm9vdCI6eyJjdGltZSI6MTQ5MDE4NDM4MiwiaGFzaCI6IjQ1MmI4MGE2N2ZiNzFhOTlkMWI1YjM2MDFhZmExY2NjMzQwYWQyZDYxMDIzNDI4MGFmNzg0N2FiYjBjMjQ2NWVkOGE0N2VlNjczM2RlN2NmYjk4ZDZhZjVmNmFlMzc5ZGE5NGY4N2M1MWI0MmRlMGNkNzM1NjZjMjQyOTQwYTQ4Iiwic2Vxbm8iOjk3MjY1OH0sInByZXYiOiJkNzFiODEzM2RhMzE1MzA5NGZiMDAxMWZiNGNkZjE4MDU3NWM5Mjg2ODQ0ODJhODNiMjY4OGMwNDQzOGYyNjQxIiwic2Vxbm8iOjU0LCJ0YWciOiJzaWduYXR1cmUifaNzaWfEQKCPJfhUcIS8k09FnSvCOxXMc83jDSBUZ3EG7US8quD1j71GcToKgKlDoOhtJxFVBtlVXx8Isf5+RaWvswQHtgqoc2lnX3R5cGUgpGhhc2iCpHR5cGUIpXZhbHVlxCDIEhav/PUYWI833mz7Pd7A7/xAMlmN8Zv6c9a225n8d6N0YWfNAgKndmVyc2lvbgE=",
      "payload_json": "{\"body\":{\"device\":{\"id\":\"885093956ef5eb10d7d35c80d92c8b18\",\"kid\":\"012151da37aaba8fe61d28d4f620a68d1b73263840564c1a2e2f727a0ae953c785520a\",\"status\":1},\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"0120b1bf850fde5bebdbd59bb3d42496685473111f6404086b2049733d0a5cb2c7800a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"subkey\":{\"kid\":\"012151da37aaba8fe61d28d4f620a68d1b73263840564c1a2e2f727a0ae953c785520a\",\"parent_kid\":\"0120b1bf850fde5bebdbd59bb3d42496685473111f6404086b2049733d0a5cb2c7800a\"},\"type\":\"subkey\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.20\"},\"ctime\":1490184387,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1490184382,\"hash\":\"452b80a67fb71a99d1b5b3601afa1ccc340ad2d610234280af7847abb0c2465ed8a47ee6733de7cfb98d6af5f6ae379da94f87c51b42de0cd73566c242940a48\",\"seqno\":972658},\"prev\":\"d71b8133da3153094fb0011fb4cdf180575c928684482a83b2688c04438f2641\",\"seqno\":54,\"tag\":\"signature\"}",
      "sig_type": 1,
      "sig_version": 1,
      "ctime": 1490184387,
      "etime": 1994760387,
      "rtime": null,
      "sig_status": 0,
      "prev": "d71b8133da3153094fb0011fb4cdf180575c928684482a83b2688c04438f2641",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 55,
      "payload_hash": "92d120d97d1e523a98016a032b768efd697a9dee8ce07620fd6bd8b8711db454",
      "sig_id": "3c3155e62cd4aadde31b68c8d9a8b2163cfed82ad35cf6d6d513429a5b02a8d40f",
      "sig_id_short": "PDFV5izUqt3jG2jI2aiyFjz-2CrTXPbW1RNC",
      "kid": "0120b1bf850fde5bebdbd59bb3d42496685473111f6404086b2049733d0a5cb2c7800a",
      "sig": "hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgsb+FD95b69vVm7PUJJZoVHMRH2QECGsgSXM9Clyyx4AKp3BheWxvYWTFA6F7ImJvZHkiOnsiZGV2aWNlIjp7ImlkIjoiMjE1MDUzNzg4NTliODdhZTQxMDdkY2JhNWI0ZTJmMTgiLCJzdGF0dXMiOjIsInR5cGUiOiJkZXNrdG9wIn0sImtleSI6eyJlbGRlc3Rfa2lkIjoiMDEyMGJjZWQxMTg2ZDA5MDNkYTFmMjhkMDdjMDA5MzA4NGZiODFmMTA3MDI4YjI1YjQ2OTE1YzNkODQ1YTUwY2Q3OTMwYSIsImhvc3QiOiJrZXliYXNlLmlvIiwia2lkIjoiMDEyMGIxYmY4NTBmZGU1YmViZGJkNTliYjNkNDI0OTY2ODU0NzMxMTFmNjQwNDA4NmIyMDQ5NzMzZDBhNWNiMmM3ODAwYSIsInVpZCI6ImZlMWUwYzk2NjU0NjAzYzU4ODBiM2Y1NzFjODYyOTE5IiwidXNlcm5hbWUiOiIwMGRhbmkifSwicmV2b2tlIjp7ImtpZHMiOlsiMDEyMDE0Yzg0MTlhMmVlNDdlOGZhMTE1ZWRiNmYxYjNiNmQ4ZDViN2I3MzliZDYyY2FmZTI4MDNiZGEyMzU2YTI3NDUwYSIsIjAxMjEyZjE0MTU4OWQ0OWI0MDdjOGY5YTQzYjUzOTc0MjgwZDJhNTUzZjVhODVkY2NhNTQ2YTc5Mzc2ZmE3NTAyOTYyMGEiXX0sInR5cGUiOiJyZXZva2UiLCJ2ZXJzaW9uIjoxfSwiY2xpZW50Ijp7Im5hbWUiOiJrZXliYXNlLmlvIGdvIGNsaWVudCIsInZlcnNpb24iOiIxLjAuMjAifSwiY3RpbWUiOjE0OTAxODQ0MzgsImV4cGlyZV9pbiI6NTA0NTc2MDAwLCJtZXJrbGVfcm9vdCI6eyJjdGltZSI6MTQ5MDE4NDQzNSwiaGFzaCI6ImY5NGJkMTNjODlmNzcwNmI5NmNhYmEyODhmMDEyMzAxM2QwNDRkZDc1YjdmOTZkMDFjMjY5NDU4NjEyNTJiY2UwYzFkZDY0YjNiYWQzMDU5ZTQwMGY4YTVjMDc0OGU1ZTEwNjEwNDc1ODFlYTY0ODM4Mzg4NjRjZDYyYjY2OTJlIiwic2Vxbm8iOjk3MjY2MH0sInByZXYiOiJmNTMzNjVmMzc3ZGM0NWY3NmFiODNkZWNjNTQzYmEyMjVhOWE0ZWNjMDEwOTJmYjMwNDU5NDM1NGYyMjU1MDhmIiwic2Vxbm8iOjU1LCJ0YWciOiJzaWduYXR1cmUifaNzaWfEQOYADH+8jtpyB29OHcMLA0CMFHfVIY2pXJevIFrjWIh3QQn3ouMCbQLo7jLvlhHQ+M51KeXGQ6elCyvQokpjcAqoc2lnX3R5cGUgpGhhc2iCpHR5cGUIpXZhbHVlxCClyM7IdB9QfWcvDgOpgq+gVskK6vcbdeu16WRq/6CD5KN0YWfNAgKndmVyc2lvbgE=",
      "payload_json": "{\"body\":{\"device\":{\"id\":\"21505378859b87ae4107dcba5b4e2f18\",\"status\":2,\"type\":\"desktop\"},\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"0120b1bf850fde5bebdbd59bb3d42496685473111f6404086b2049733d0a5cb2c7800a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"revoke\":{\"kids\":[\"012014c8419a2ee47e8fa115edb6f1b3b6d8d5b7b739bd62cafe2803bda2356a27450a\",\"01212f141589d49b407c8f9a43b53974280d2a553f5a85dcca546a79376fa75029620a\"]},\"type\":\"revoke\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.20\"},\"ctime\":1490184438,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1490184435,\"hash\":\"f94bd13c89f7706b96caba288f0123013d044dd75b7f96d01c26945861252bce0c1dd64b3bad3059e400f8a5c0748e5e1061047581ea6483838864cd62b6692e\",\"seqno\":972660},\"prev\":\"f53365f377dc45f76ab83decc543ba225a9a4ecc01092fb304594354f225508f\",\"seqno\":55,\"tag\":\"signature\"}",
      "sig_type": 5,
      "sig_version": 1,
      "ctime": 1490184438,
      "etime": 1994760438,
      "rtime": null,
      "sig_status": 0,
      "prev": "f53365f377dc45f76ab83decc543ba225a9a4ecc01092fb304594354f225508f",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 56,
      "payload_hash": "898225644975026efc0e05275a60957870e83e4cc5c70219393ad471c54ba4cb",
      "sig_id": "77c59362c2875339ba61b2cae17170cf4f4675170c7c25dd62b621e1b726b3750f",
      "sig_id_short": "d8WTYsKHUzm6YbLK4XFwz09GdRcMfCXdYrYh",
      "kid": "0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a",
      "sig": "hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgvO0RhtCQPaHyjQfACTCE+4HxBwKLJbRpFcPYRaUM15MKp3BheWxvYWTFAul7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwidWlkIjoiZmUxZTBjOTY2NTQ2MDNjNTg4MGIzZjU3MWM4NjI5MTkiLCJ1c2VybmFtZSI6IjAwZGFuaSJ9LCJzZXJ2aWNlIjp7Im5hbWUiOiJyZWRkaXQiLCJ1c2VybmFtZSI6IjAwZGFuaSJ9LCJ0eXBlIjoid2ViX3NlcnZpY2VfYmluZGluZyIsInZlcnNpb24iOjF9LCJjbGllbnQiOnsibmFtZSI6ImtleWJhc2UuaW8gZ28gY2xpZW50IiwidmVyc2lvbiI6IjEuMC4yMCJ9LCJjdGltZSI6MTQ5MDIyNzcyNCwiZXhwaXJlX2luIjo1MDQ1NzYwMDAsIm1lcmtsZV9yb290Ijp7ImN0aW1lIjoxNDkwMjI3NDk1LCJoYXNoIjoiZWIzMDJiYjYyNzJkMjQwMzFiNWQzN2Q2ZGMyMWIwZjcyYjI1NjkwZWE1YTAwNzVlMzAyYTcyN2EyYjBiYjA3NDNhYThmMmJkNDNiYzRkODg3NzliZWNmYjUzMzFhMTQyYTEyYmZiZTFkYjgzNTdhYWQ0MjU3NWQxZjdmOWM4NmEiLCJzZXFubyI6OTczOTgzfSwicHJldiI6IjkyZDEyMGQ5N2QxZTUyM2E5ODAxNmEwMzJiNzY4ZWZkNjk3YTlkZWU4Y2UwNzYyMGZkNmJkOGI4NzExZGI0NTQiLCJzZXFubyI6NTYsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RA/05MqiMz1NSVYv0E+YmqsvW0G+O6JpfnTaoFJMlnVpf/wsp/t99zShgJ5z7Du3VLMGwhB2H5rxMgvWTa98GhAahzaWdfdHlwZSCkaGFzaIKkdHlwZQildmFsdWXEIFdMXvIiopcCA1JXQdALpiJlfuVeUww1JSFeA3+qcLsOo3RhZ80CAqd2ZXJzaW9uAQ==",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"service\":{\"name\":\"reddit\",\"username\":\"00dani\"},\"type\":\"web_service_binding\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.20\"},\"ctime\":1490227724,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1490227495,\"hash\":\"eb302bb6272d24031b5d37d6dc21b0f72b25690ea5a0075e302a727a2b0bb0743aa8f2bd43bc4d88779becfb5331a142a12bfbe1db8357aad42575d1f7f9c86a\",\"seqno\":973983},\"prev\":\"92d120d97d1e523a98016a032b768efd697a9dee8ce07620fd6bd8b8711db454\",\"seqno\":56,\"tag\":\"signature\"}",
      "sig_type": 2,
      "sig_version": 1,
      "ctime": 1490227724,
      "etime": 1994803724,
      "rtime": null,
      "sig_status": 0,
      "prev": "92d120d97d1e523a98016a032b768efd697a9dee8ce07620fd6bd8b8711db454",
      "proof_id": "2e05f82a3c03ffd4c598ed10",
      "proof_type": 4,
      "proof_text_check": "hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgvO0RhtCQPaHyjQfACTCE+4HxBwKLJbRpFcPYRaUM15MKp3BheWxvYWTFAul7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwidWlkIjoiZmUxZTBjOTY2NTQ2MDNjNTg4MGIzZjU3MWM4NjI5MTkiLCJ1c2VybmFtZSI6IjAwZGFuaSJ9LCJzZXJ2aWNlIjp7Im5hbWUiOiJyZWRkaXQiLCJ1c2VybmFtZSI6IjAwZGFuaSJ9LCJ0eXBlIjoid2ViX3NlcnZpY2VfYmluZGluZyIsInZlcnNpb24iOjF9LCJjbGllbnQiOnsibmFtZSI6ImtleWJhc2UuaW8gZ28gY2xpZW50IiwidmVyc2lvbiI6IjEuMC4yMCJ9LCJjdGltZSI6MTQ5MDIyNzcyNCwiZXhwaXJlX2luIjo1MDQ1NzYwMDAsIm1lcmtsZV9yb290Ijp7ImN0aW1lIjoxNDkwMjI3NDk1LCJoYXNoIjoiZWIzMDJiYjYyNzJkMjQwMzFiNWQzN2Q2ZGMyMWIwZjcyYjI1NjkwZWE1YTAwNzVlMzAyYTcyN2EyYjBiYjA3NDNhYThmMmJkNDNiYzRkODg3NzliZWNmYjUzMzFhMTQyYTEyYmZiZTFkYjgzNTdhYWQ0MjU3NWQxZjdmOWM4NmEiLCJzZXFubyI6OTczOTgzfSwicHJldiI6IjkyZDEyMGQ5N2QxZTUyM2E5ODAxNmEwMzJiNzY4ZWZkNjk3YTlkZWU4Y2UwNzYyMGZkNmJkOGI4NzExZGI0NTQiLCJzZXFubyI6NTYsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RA/05MqiMz1NSVYv0E+YmqsvW0G+O6JpfnTaoFJMlnVpf/wsp/t99zShgJ5z7Du3VLMGwhB2H5rxMgvWTa98GhAahzaWdfdHlwZSCkaGFzaIKkdHlwZQildmFsdWXEIFdMXvIiopcCA1JXQdALpiJlfuVeUww1JSFeA3+qcLsOo3RhZ80CAqd2ZXJzaW9uAQ==",
      "proof_text_full": "### Keybase proof\n\nI am:\n\n  * [00dani](https://www.reddit.com/user/00dani) on reddit.\n  * [00dani](https://keybase.io/00dani) on keybase.\n\nProof:\n\n    hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgvO0RhtCQPaHyjQfACTCE+4HxBwKLJbRpFcPYRaUM15MKp3BheWxvYWTFAul7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwidWlkIjoiZmUxZTBjOTY2NTQ2MDNjNTg4MGIzZjU3MWM4NjI5MTkiLCJ1c2VybmFtZSI6IjAwZGFuaSJ9LCJzZXJ2aWNlIjp7Im5hbWUiOiJyZWRkaXQiLCJ1c2VybmFtZSI6IjAwZGFuaSJ9LCJ0eXBlIjoid2ViX3NlcnZpY2VfYmluZGluZyIsInZlcnNpb24iOjF9LCJjbGllbnQiOnsibmFtZSI6ImtleWJhc2UuaW8gZ28gY2xpZW50IiwidmVyc2lvbiI6IjEuMC4yMCJ9LCJjdGltZSI6MTQ5MDIyNzcyNCwiZXhwaXJlX2luIjo1MDQ1NzYwMDAsIm1lcmtsZV9yb290Ijp7ImN0aW1lIjoxNDkwMjI3NDk1LCJoYXNoIjoiZWIzMDJiYjYyNzJkMjQwMzFiNWQzN2Q2ZGMyMWIwZjcyYjI1NjkwZWE1YTAwNzVlMzAyYTcyN2EyYjBiYjA3NDNhYThmMmJkNDNiYzRkODg3NzliZWNmYjUzMzFhMTQyYTEyYmZiZTFkYjgzNTdhYWQ0MjU3NWQxZjdmOWM4NmEiLCJzZXFubyI6OTczOTgzfSwicHJldiI6IjkyZDEyMGQ5N2QxZTUyM2E5ODAxNmEwMzJiNzY4ZWZkNjk3YTlkZWU4Y2UwNzYyMGZkNmJkOGI4NzExZGI0NTQiLCJzZXFubyI6NTYsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RA/05MqiMz1NSVYv0E+YmqsvW0G+O6JpfnTaoFJMlnVpf/wsp/t99zShgJ5z7Du3VLMGwhB2H5rxMgvWTa98GhAahzaWdfdHlwZSCkaGFzaIKkdHlwZQildmFsdWXEIFdMXvIiopcCA1JXQdALpiJlfuVeUww1JSFeA3+qcLsOo3RhZ80CAqd2ZXJzaW9uAQ==\n",
      "check_data_json": "{\"name\":\"reddit\",\"username\":\"00dani\"}",
      "remote_id": "t3_60yifs",
      "api_url": "https://www.reddit.com/r/KeybaseProofs/comments/60yifs/my_keybase_proof_reddit00dani_keybase00dani/.json",
      "human_url": "https://www.reddit.com/r/KeybaseProofs/comments/60yifs/my_keybase_proof_reddit00dani_keybase00dani/",
      "proof_state": 1,
      "proof_status": 1,
      "retry_count": 38,
      "hard_fail_count": 0,
      "last_check": 1491809262,
      "last_success": 1491809262,
      "version": 1,
      "fingerprint": ""
    },
    {
      "seqno": 57,
      "payload_hash": "1779c49965ebef6d238c581ff982553884394866fee8ccaf33d739f19238ece0",
      "sig_id": "d3dcd5adb05283916109f908d09c84ea1abb8f2e7b47958ec022db9b939c31700f",
      "sig_id_short": "09zVrbBSg5FhCfkI0JyE6hq7jy57R5WOwCLb",
      "kid": "0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a",
      "sig": "hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgvO0RhtCQPaHyjQfACTCE+4HxBwKLJbRpFcPYRaUM15MKp3BheWxvYWTFA7J7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwYmNlZDExODZkMDkwM2RhMWYyOGQwN2MwMDkzMDg0ZmI4MWYxMDcwMjhiMjViNDY5MTVjM2Q4NDVhNTBjZDc5MzBhIiwidWlkIjoiZmUxZTBjOTY2NTQ2MDNjNTg4MGIzZjU3MWM4NjI5MTkiLCJ1c2VybmFtZSI6IjAwZGFuaSJ9LCJwZ3BfdXBkYXRlIjp7ImZpbmdlcnByaW50IjoiN2ZkNzRlY2VhNmFjN2MwNzNlN2Y2MDI5MzhjNWRkYzZiZGE0ZmFlMyIsImZ1bGxfaGFzaCI6IjkzZmU1YzA2YjUyNzYzNmM5YTliZWJmZDYwM2ViYzc1M2I4MjNkMzNiNjFkNTA3NDhmNDJjOTcwOTI3MjQyODEiLCJrZXlfaWQiOiIzOEM1RERDNkJEQTRGQUUzIiwia2lkIjoiMDEwMTQ4MmVjZjNkY2U2NTc0M2MxZjQ4ZjFiYjZiZDE4NjhiOTBlOGU0NGM4YjM5NzUyZjE1NjlkNGRmZjE1MGY0MDQwYSJ9LCJ0eXBlIjoicGdwX3VwZGF0ZSIsInZlcnNpb24iOjF9LCJjbGllbnQiOnsibmFtZSI6ImtleWJhc2UuaW8gZ28gY2xpZW50IiwidmVyc2lvbiI6IjEuMC4yMCJ9LCJjdGltZSI6MTQ5MDU2NTE5OCwiZXhwaXJlX2luIjo1MDQ1NzYwMDAsIm1lcmtsZV9yb290Ijp7ImN0aW1lIjoxNDkwNTY1MTc4LCJoYXNoIjoiZGM5MDM4ZTA2MzY3YjY0OGFmMzhiOWNkNmQ3YjIxNjdkZTBjZGZhNDBhMjNmY2M2YzJhYWM2ZjFiYzVkOWY0ZTEwYmUzMDAyODBlN2MxYTYzZjdlNDAyZWQ0Y2Y2ZTllMzEwZmY2MDIxYmY1OTEwMGU5NDk0ZWNjZmMxMjUxZDMiLCJzZXFubyI6OTc5MzIwfSwicHJldiI6Ijg5ODIyNTY0NDk3NTAyNmVmYzBlMDUyNzVhNjA5NTc4NzBlODNlNGNjNWM3MDIxOTM5M2FkNDcxYzU0YmE0Y2IiLCJzZXFubyI6NTcsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RA+qyZWjSe6TmB7h56KLQPmJP/gj6PJQpX+FApRsNXKX2KKWn6F7YuZxq0cxDnP6AJPfu/aSIzYvHyIQBzrLCmBKhzaWdfdHlwZSCkaGFzaIKkdHlwZQildmFsdWXEIEzMW0A3LGA2eejrrr281LBNGIKKMzgMN7i2S4jwIXx5o3RhZ80CAqd2ZXJzaW9uAQ==",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"host\":\"keybase.io\",\"kid\":\"0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a\",\"uid\":\"fe1e0c96654603c5880b3f571c862919\",\"username\":\"00dani\"},\"pgp_update\":{\"fingerprint\":\"7fd74ecea6ac7c073e7f602938c5ddc6bda4fae3\",\"full_hash\":\"93fe5c06b527636c9a9bebfd603ebc753b823d33b61d50748f42c97092724281\",\"key_id\":\"38C5DDC6BDA4FAE3\",\"kid\":\"0101482ecf3dce65743c1f48f1bb6bd1868b90e8e44c8b39752f1569d4dff150f4040a\"},\"type\":\"pgp_update\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.20\"},\"ctime\":1490565198,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1490565178,\"hash\":\"dc9038e06367b648af38b9cd6d7b2167de0cdfa40a23fcc6c2aac6f1bc5d9f4e10be300280e7c1a63f7e402ed4cf6e9e310ff6021bf59100e9494eccfc1251d3\",\"seqno\":979320},\"prev\":\"898225644975026efc0e05275a60957870e83e4cc5c70219393ad471c54ba4cb\",\"seqno\":57,\"tag\":\"signature\"}",
      "sig_type": 1,
      "sig_version": 1,
      "ctime": 1490565198,
      "etime": 1995141198,
      "rtime": null,
      "sig_status": 0,
      "prev": "898225644975026efc0e05275a60957870e83e4cc5c70219393ad471c54ba4cb",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    }
  ],
  "keys": [
    "0120bced1186d0903da1f28d07c0093084fb81f107028b25b46915c3d845a50cd7930a",
    "012147843071a0a58e738c263f76ff8d5d78d65fc7b718a86e7b7adfef6e5678cd2f0a",
    "012138151d45bbfd807d81a94f918b4980d9ad2d1e5bdd44eff9803f2e26e37883420a",
    "01202f8f0998d6e9daddf0bd5c1295c0fa005f086b054b13be7b8a0668bf68d6f6cf0a",
    "-----BEGIN PGP PUBLIC KEY BLOCK-----\n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\nUFuuPUiMTC0=\n=plqs\n-----END PGP PUBLIC KEY BLOCK-----",
    "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: GnuPG v2\n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\nc0dWcJRvOFQ=\n=sGhe\n-----END PGP PUBLIC KEY BLOCK-----",
    "0120d3e4960119263b3d09382906e21022e79a7e92057654c8ccb8e6fdc7a593a01a0a",
    "0121190034a619477525a4c4ec24a9aa369a8bfb6a26b9e29cbea46fcb26455694700a",
    "01212f141589d49b407c8f9a43b53974280d2a553f5a85dcca546a79376fa75029620a",
    "012014c8419a2ee47e8fa115edb6f1b3b6d8d5b7b739bd62cafe2803bda2356a27450a",
    "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: GnuPG v2\n\nmQENBFh7JwUBCACsA/Tk1etTvvajcC6fjlzkyfwmkBSnjuaQt7gi+OPOZw9Pn74q\nFSzAV/PCGKY5vvdvZP8sSp8vP56OwSzZr6S2ASRLHwHcm7uNq/yulPEVotxjIKcP\ndvEXHTvLg84fhvHTHSaVBXQ6S8XV4l5LOamZxMDu1tbOh9aI4C7I8N0cg2/zxYZY\nqeuxgmMNnsvYounlKWber68HcDxCnxGR/45CgMzU5mzlPfmQ1KbF7WGrlAi4jUUg\nGuG8g+iATZr9fsgNCgRTTYsKQNORM9X2bpAxRGBNfSLki8yXgeMNeIR/HPVqUUhz\nNVGRBMY0Je2t0+8VzwHM/18DzGXqlH/OH8I5ABEBAAG0KURhbmllbGxlIE1jTGVh\nbiA8Z29wc3ljaG9uYXV0c0BnbWFpbC5jb20+iQE5BBMBCAAjBQJYeycFAhsDBwsJ\nCAcDAgEGFQgCCQoLBBYCAwECHgECF4AACgkQkTFCeUh4sG6wGgf9E7l81IqJviSm\nNqKLkkpXllnvcZa11/qtsU3T4WIeCqrfK3WyLZcVkJkP3/8DyttD4Ye4+FZg5u1E\nzmv7Sr1a+FR8p1vNWu/Pe1JGy9kTk/khOZxNQleJIIKWcCznC+nmwa0JyaQjtViN\nkAMnxgvmKiAFyOG6fgyIF51xuadPCraNfbPM8e1GrOBSK+LY65MAnczVPGycBfdM\nj8gPOVwtYRhhg8Dgf4UtwW5lSNaMfo+D9SVuwhODro5mZU9B+TlCpEOXoAPfhXDo\nUlRzC77e1oihm1/0j2++MhTWPZ3aILYRILEvNd3sxQTczjOVNYSf/93KQjhOdI12\nWxhsXluHJbkBDQRYeycFAQgAqrU1tEvUvSigRo7sID6AixauPv0tDP+l89VInPBF\n3va5tCDeDVGmacVXmpUT3MF4p1uNELKcwKpkOoe6S6shNv0DDNITJ3KiHqVHg+QM\nSDccHF+I7mFLNVwaJcAYB985K+fclnG1zbVRrNzIFfLMy1shvtBKw1hUMgjehxPB\nvpfYlKd3aehTJGxvxf5ao4OkoxlSDPziactYHCO5USyq7m8OffCSFlfHO64BxqbN\nRAyZOmgDZnXSvQ43CGZy2dCOoFxOHwJBmZIbpe6UUwokfGMzwvEkFS+IjJd8u3xa\nIy7QR/rXqHpJdNqGMEgamfLqp+hD1U+EdVGKUneJsHqokwARAQABiQEfBBgBCAAJ\nBQJYeycFAhsMAAoJEJExQnlIeLBupVoH/0cYIVRMKldqgqQqGrMlv3mrUZO6IvH8\nTPwrrjL3MI6+4OHevJLDjbya9+NSuP+ckD9kMeHpbZohNmQ+GuDfYhw1xG64bMxd\nrC0GggR1KqUBLvrWW1kZCAhQSKdW+Y7aLDzEwNnF5ZwU6xQbYxRUSL0pjWWbZ2Ss\nzc0hou8rgpvK8nRxBNr3G/My1IEObIGjvoeYivZq0RUaYUdCbUyz251X1CtZZGAx\neSP3BUub6c4ZEaKio5UvZtmf3ZcjnBZfRCstDM3G3upx3XUJmEyzXWRYXPi0Qc3B\nUYrkP6eRZWsJQXsBThdMp3a3IaA9LSFERsdwVWqBEhjLd2mDm2J0PH0=\n=1vEu\n-----END PGP PUBLIC KEY BLOCK-----",
    "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: GnuPG v2\n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\nUFuuPUiMTC0=\n=plqs\n-----END PGP PUBLIC KEY BLOCK-----",
    "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: GnuPG v2\n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\n4QM=\n=ZhJf\n-----END PGP PUBLIC KEY BLOCK-----",
    "-----BEGIN PGP PUBLIC KEY BLOCK-----\n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\nKL/sJkHqugB2JwncRAMV1AEZRuXnrP3WLV/lu+ED\n=SPjn\n-----END PGP PUBLIC KEY BLOCK-----",
    "-----BEGIN PGP PUBLIC KEY BLOCK-----\n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\nlqVi+OSwudh8qOOyYiR4fZRa9bEajxGyG38N9DCNwEQ6fxs=\n=4Wi0\n-----END PGP PUBLIC KEY BLOCK-----",
    "-----BEGIN PGP PUBLIC KEY BLOCK-----\n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\nlqVi+OSwudh8qOOyYiR4fZRa9bEajxGyG38N9DCNwEQ6fxs=\n=NlV7\n-----END PGP PUBLIC KEY BLOCK-----",
    "-----BEGIN PGP PUBLIC KEY BLOCK-----\n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\n0mFptNxipaW/kHGj+QCygip5c0dWcJRvOFQ=\n=qp9Q\n-----END PGP PUBLIC KEY BLOCK-----",
    "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: GnuPG v2\n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\n7qq8HEi2jgullqVi+OSwudh8qOOyYiR4fZRa9bEajxGyG38N9DCNwEQ6fxs=\n=LKv0\n-----END PGP PUBLIC KEY BLOCK-----",
    "-----BEGIN PGP PUBLIC KEY BLOCK-----\n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\nsht/DfQwjcBEOn8b\n=d4Cu\n-----END PGP PUBLIC KEY BLOCK-----",
    "-----BEGIN PGP PUBLIC KEY BLOCK-----\n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\npIHXag3WsjZd\n=nvq1\n-----END PGP PUBLIC KEY BLOCK-----",
    "0120b1bf850fde5bebdbd59bb3d42496685473111f6404086b2049733d0a5cb2c7800a",
    "012151da37aaba8fe61d28d4f620a68d1b73263840564c1a2e2f727a0ae953c785520a",
    "-----BEGIN PGP PUBLIC KEY BLOCK-----\n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\nbGRIsadXRaSB12oN1rI2XQ==\n=ZH//\n-----END PGP PUBLIC KEY BLOCK-----"
  ]
}
`,
	"akalin_chain.json": `{
    "username": "akalin",
    "uid": "ebbe1d99410ab70123262cf8dfc87900",
    "chain": [
        {
            "seqno": 1,
            "payload_hash": "738bebce2b4ed88c9319075bfed58779734efd03f3ee5386b15b38b61d2024f5",
            "sig_id": "5520adf71d91197d52cdb69d45fb31250d633bf30c5c07c4c4644bac263953860f",
            "sig_id_short": "VSCt9x2RGX1SzbadRfsxJQ1jO_MMXAfExGRL",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowGbwMvMwMW4vyE6ITA1uJDx9IHKJIYQpocXqpWS8lMqlayqlbJTwVRaZl56alFB\nUWZeiZKVkoF5UpqhuXGySWKikYGZSZKFhYWRRSKQkWZhYJpkZmBqaGZqamxuqKSj\nlJFfDNIBNCYpsThVLzMfKAbkxGemAEWxqC8FS6QmJaUaplhamhgaJCaZGxgaGRuZ\nGSWnWaSkJVuYWxoYgBQWpxblJeamAlUnZifmZOYp1eoolVQWgATKU5PigdJlmcmp\n8UmZeSlAxwN1lKUWFWfm5ylZGQJVJpdkgvQamhhYmpibGhgY6iilVhRkFqXGZ4JU\nmJqbWRgAgY5SQVFqmZJVXmlOjo5ScWphXj5QFmhRYjrQnuLM9LzEktKiVKVark5G\nGRYGRi4GNlYmUAgycHEKwMJ16XwOho0zP+3Jav7+8uPF2J0Hjui4929Ze+rz35fx\n6+5kG8S0ijiKFTJZ/nE9ZqZbMDthy+GkreJLb8+vfDWvZvujxXl50f9rSn7eEfm1\nK16L4/dxvxyz+gviXNd6H7Nv2nLz/zIdiQ4Z+TKOH2sqnG0ztxwvFZufyrXsq4ES\na/jah+1y6/0zLz1+q5LSPz+ttIstxS/GgbvWm+HjG5GCP7PtG/u3TS2/L/tIfkb7\ne5kv7R7VAlvqd+nsKLT/18b76eZ0v6+HsyZM/nRcbdPJxplvH067sP3hteOLjtTG\nM5To1/AxbuZPODZHWka4vVZn14JuQXd2s415K7etE61eUbbkziMtNpZudl+lsD6R\nm/pbRAE=\n=M4q7\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"bf805b6051655371\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1409475001,\"expire_in\":157680000,\"prev\":null,\"seqno\":1,\"tag\":\"signature\"}\n",
            "sig_type": 1,
            "ctime": 1409475001,
            "etime": 1567155001,
            "rtime": null,
            "sig_status": 0,
            "prev": null,
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 2,
            "payload_hash": "d8cd14f0d6da7dad0b0227b024ecb99c920e6110ef9486262be08df3b6431e00",
            "sig_id": "1a67183170ef64b00e805c74d39c5286a6a9b50939b3604b15f0722aae27500b0f",
            "sig_id_short": "GmcYMXDvZLAOgFx005xShqaptQk5s2BLFfBy",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowFtkVtIFFEYx2c1L60amUqYlnisINzqzH1mkR6iQNAX00hU2ObsnNkmdXadnfWC\nWCsKVkYaET50AS3IlITyApbtmj4oKpYvkmCCRVKQgmHRQ1IzYkTQeTnn/L//78/3\n8bUnRBJ2WzhYer4AF1bZpkc/I6IoYnmtHiCvXAec9aAcb12Kqnmw7tNVzQBOAHmk\nkDztZiSJghyDBEGgBMl8KAJkEQdZkmNZmieBA1zw+i3CjEGSHx9TvaZmflyqbKr/\n8Qe2ChghTMqiyJBQQjwkKZriKLciyIpb4EUILaMf65pUiU23VC5VqBpocABTq1bd\n2Op4u2bUqIaB9X8B5S9h1PkspQYj1zbsQqomm+OaSDXW/apXA07SdLoN1YJJBooM\nz0KGdwBc61N17FItB8tzAjSPA/h0XG1G8rSAMHJjCjFYFgS3SJMi5FmkYJkVeF7k\naQYrMqQVGmOWFjhEsshEOFKmIMUoLLDGqdK8wEmZbUoeM9KvejTJCOgYNNiv2tJ3\nEDY7ER0VYW2MsO/c/WeP52ZiieCNRaGzvyzubk1ld9JqV2nYNSgFUxxOR8LZguTa\n2cBgRWk4/smePP5STurM2MjYtfnRlbGObykv+ovyN+WN4+OpE7nqUPSvzfX7R+bf\nvTfmZHyireXiQHE8jHkcUEf6Jk6eTu972zT6ZnL6yvL+jAetMaEfzd/XSzpXpm7d\nHkg6nLb2LPdmo5IFuyKDiwuPpk7t6m5JyQwtaaGfZU83szu4ucZXxa9XssJdzw/k\n9fa2z8U9LPzYFrqux36Pa/Keac4GU9EHiXHi5cbC0dbikp7ZRE/6XnuLlPgh6jJO\n/JrZU9/6KaPq3vA+fvKOkbw+9GWYS14yDhWm+bnVnPzZ3w==\n=qYLb\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"bf805b6051655371\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"service\":{\"name\":\"twitter\",\"username\":\"fakalin\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1409475047,\"expire_in\":157680000,\"prev\":\"738bebce2b4ed88c9319075bfed58779734efd03f3ee5386b15b38b61d2024f5\",\"seqno\":2,\"tag\":\"signature\"}\n",
            "sig_type": 2,
            "ctime": 1409475047,
            "etime": 1567155047,
            "rtime": null,
            "sig_status": 0,
            "prev": "738bebce2b4ed88c9319075bfed58779734efd03f3ee5386b15b38b61d2024f5",
            "proof_id": "a511b030f44e30d1aa00c410",
            "proof_type": 2,
            "proof_text_check": "Verifying myself: I am akalin on Keybase.io. GmcYMXDvZLAOgFx005xShqaptQk5s2BLFfBy /",
            "proof_text_full": "Verifying myself: I am akalin on Keybase.io. GmcYMXDvZLAOgFx005xShqaptQk5s2BLFfBy / https://keybase.io/akalin/sigs/GmcYMXDvZLAOgFx005xShqaptQk5s2BLFfBy\n",
            "check_data_json": "{\"name\":\"twitter\",\"username\":\"fakalin\"}",
            "remote_id": "506001226417180673",
            "api_url": "https://twitter.com/fakalin/status/506001226417180673",
            "human_url": "https://twitter.com/fakalin/status/506001226417180673",
            "proof_state": 1,
            "proof_status": 1,
            "retry_count": 246,
            "hard_fail_count": 0,
            "last_check": 1443796918,
            "last_success": 1443796918,
            "version": 2,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 3,
            "payload_hash": "99456cb27d7c6c8e6978be4c0928c19a07428e236fa9f7ed4d823d926f054701",
            "sig_id": "68a547eed57a11b62bfe2511fe128cc10cfb70f199453aa9cb964b97999f5d800f",
            "sig_id_short": "aKVH7tV6EbYr_iUR_hKMwQz7cPGZRTqpy5ZL",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowGbwMvMwMW4vyE6ITA1uJDx9IHnSQwhTI9kq5WS8lMqlayqlbJTwVRaZl56alFB\nUWZeiZKVkoF5UpqhuXGySWKikYGZSZKFhYWRRSKQkWZhYJpkZmBqaGZqamxuqKSj\nlJFfDNIBNCYpsThVLzMfKAbkxGemAEWxqC8FS6QmJaUaplhamhgaJCaZGxgaGRuZ\nGSWnWaSkJVuYWxoYgBQWpxblJeamAlUnZifmZOYp1eooAcXKMpNTQS6GyqVnlmSU\nJuFSX1JZABIoT02Kh2qNT8rMSwF6FqijLLWoODM/T8nKEKgyuSQTpNfQxMDSxNzU\nwNJcRym1oiCzKDU+E6TC1NzMwgAIdJQKilLLgEamWCSnGJqkGaSYpSSapySmGCQZ\nGBmZAwmT1OQkS8tkSyODVDNDQ4PUNEsTCzOg75JSDYDeM04yMzE2TAX7sDi1MC9f\nycoY6MzEdKCRxZnpeYklpUWpSrVcnYwyLAyMXAxsrEyg+GLg4hSAxaLKHfb/qWGc\n7rI9Gy7yeK52VJgqyZpfkMVde5tJ4m/EkvInerq6bAfVnCtTPILfbl1zxMfgT4la\nWyTfG4feRzYpizTuPn1Sra3/pftvyveKpUoGMwv7jt869XbtaX7/lm0/BcJMD0Yb\nPl7/av3nrVcaq5jv8vMWOq5Rlrg9/8Xt31dT2gWr4w4yPc1uqJrEeHn926Df91Y9\nLDtrws3Mva7gzWk/lfkaX8t4yrVebD3swRa2+1GHRRY307UL2e3MK1ZYXGHtbYmK\nrXpt+TDGRTK4dbIT5+bj5w+lvokr+hJQ7Teta9esFjGe7kOSJyTVtt5a/NuJ70D6\nl2L2k8alve6HguZdLFmrpKPx1CB2yd/f6VMB\n=/lLk\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"bf805b6051655371\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"service\":{\"name\":\"github\",\"username\":\"akalin\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1409475097,\"expire_in\":157680000,\"prev\":\"d8cd14f0d6da7dad0b0227b024ecb99c920e6110ef9486262be08df3b6431e00\",\"seqno\":3,\"tag\":\"signature\"}\n",
            "sig_type": 2,
            "ctime": 1409475097,
            "etime": 1567155097,
            "rtime": null,
            "sig_status": 0,
            "prev": "d8cd14f0d6da7dad0b0227b024ecb99c920e6110ef9486262be08df3b6431e00",
            "proof_id": "ef1f23491d18d10b404c9c10",
            "proof_type": 3,
            "proof_text_check": "\n\nowGbwMvMwMW4vyE6ITA1uJDx9IHnSQwhTI9kq5WS8lMqlayqlbJTwVRaZl56alFB\nUWZeiZKVkoF5UpqhuXGySWKikYGZSZKFhYWRRSKQkWZhYJpkZmBqaGZqamxuqKSj\nlJFfDNIBNCYpsThVLzMfKAbkxGemAEWxqC8FS6QmJaUaplhamhgaJCaZGxgaGRuZ\nGSWnWaSkJVuYWxoYgBQWpxblJeamAlUnZifmZOYp1eooAcXKMpNTQS6GyqVnlmSU\nJuFSX1JZABIoT02Kh2qNT8rMSwF6FqijLLWoODM/T8nKEKgyuSQTpNfQxMDSxNzU\nwNJcRym1oiCzKDU+E6TC1NzMwgAIdJQKilLLgEamWCSnGJqkGaSYpSSapySmGCQZ\nGBmZAwmT1OQkS8tkSyODVDNDQ4PUNEsTCzOg75JSDYDeM04yMzE2TAX7sDi1MC9f\nycoY6MzEdKCRxZnpeYklpUWpSrVcnYwyLAyMXAxsrEyg+GLg4hSAxaLKHfb/qWGc\n7rI9Gy7yeK52VJgqyZpfkMVde5tJ4m/EkvInerq6bAfVnCtTPILfbl1zxMfgT4la\nWyTfG4feRzYpizTuPn1Sra3/pftvyveKpUoGMwv7jt869XbtaX7/lm0/BcJMD0Yb\nPl7/av3nrVcaq5jv8vMWOq5Rlrg9/8Xt31dT2gWr4w4yPc1uqJrEeHn926Df91Y9\nLDtrws3Mva7gzWk/lfkaX8t4yrVebD3swRa2+1GHRRY307UL2e3MK1ZYXGHtbYmK\nrXpt+TDGRTK4dbIT5+bj5w+lvokr+hJQ7Teta9esFjGe7kOSJyTVtt5a/NuJ70D6\nl2L2k8alve6HguZdLFmrpKPx1CB2yd/f6VMB\n",
            "proof_text_full": "### Keybase proof\n\nI hereby claim:\n\n  * I am akalin on github.\n  * I am akalin (https://keybase.io/akalin) on keybase.\n  * I have a public key whose fingerprint is 07BF 173C 4AA2 064B 8882  8A64 BF80 5B60 5165 5371\n\nTo claim this, I am signing this object:\n\n`+"`"+``+"`"+``+"`"+`json\n{\n    \"body\": {\n        \"key\": {\n            \"fingerprint\": \"07bf173c4aa2064b88828a64bf805b6051655371\",\n            \"host\": \"keybase.io\",\n            \"key_id\": \"bf805b6051655371\",\n            \"uid\": \"ebbe1d99410ab70123262cf8dfc87900\",\n            \"username\": \"akalin\"\n        },\n        \"service\": {\n            \"name\": \"github\",\n            \"username\": \"akalin\"\n        },\n        \"type\": \"web_service_binding\",\n        \"version\": 1\n    },\n    \"ctime\": 1409475097,\n    \"expire_in\": 157680000,\n    \"prev\": \"d8cd14f0d6da7dad0b0227b024ecb99c920e6110ef9486262be08df3b6431e00\",\n    \"seqno\": 3,\n    \"tag\": \"signature\"\n}\n`+"`"+``+"`"+``+"`"+`\n\nwith the PGP key whose fingerprint is\n[07BF 173C 4AA2 064B 8882  8A64 BF80 5B60 5165 5371](https://keybase.io/akalin)\n(captured above as `+"`"+`body.key.fingerprint`+"`"+`), yielding the PGP signature:\n\n`+"`"+``+"`"+``+"`"+`\n-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowGbwMvMwMW4vyE6ITA1uJDx9IHnSQwhTI9kq5WS8lMqlayqlbJTwVRaZl56alFB\nUWZeiZKVkoF5UpqhuXGySWKikYGZSZKFhYWRRSKQkWZhYJpkZmBqaGZqamxuqKSj\nlJFfDNIBNCYpsThVLzMfKAbkxGemAEWxqC8FS6QmJaUaplhamhgaJCaZGxgaGRuZ\nGSWnWaSkJVuYWxoYgBQWpxblJeamAlUnZifmZOYp1eooAcXKMpNTQS6GyqVnlmSU\nJuFSX1JZABIoT02Kh2qNT8rMSwF6FqijLLWoODM/T8nKEKgyuSQTpNfQxMDSxNzU\nwNJcRym1oiCzKDU+E6TC1NzMwgAIdJQKilLLgEamWCSnGJqkGaSYpSSapySmGCQZ\nGBmZAwmT1OQkS8tkSyODVDNDQ4PUNEsTCzOg75JSDYDeM04yMzE2TAX7sDi1MC9f\nycoY6MzEdKCRxZnpeYklpUWpSrVcnYwyLAyMXAxsrEyg+GLg4hSAxaLKHfb/qWGc\n7rI9Gy7yeK52VJgqyZpfkMVde5tJ4m/EkvInerq6bAfVnCtTPILfbl1zxMfgT4la\nWyTfG4feRzYpizTuPn1Sra3/pftvyveKpUoGMwv7jt869XbtaX7/lm0/BcJMD0Yb\nPl7/av3nrVcaq5jv8vMWOq5Rlrg9/8Xt31dT2gWr4w4yPc1uqJrEeHn926Df91Y9\nLDtrws3Mva7gzWk/lfkaX8t4yrVebD3swRa2+1GHRRY307UL2e3MK1ZYXGHtbYmK\nrXpt+TDGRTK4dbIT5+bj5w+lvokr+hJQ7Teta9esFjGe7kOSJyTVtt5a/NuJ70D6\nl2L2k8alve6HguZdLFmrpKPx1CB2yd/f6VMB\n=/lLk\n-----END PGP MESSAGE-----\n\n`+"`"+``+"`"+``+"`"+`\n\nAnd finally, I am proving ownership of the github account by posting this as a gist.\n\n### My publicly-auditable identity:\n\nhttps://keybase.io/akalin\n\n### From the command line:\n\nConsider the [keybase command line program](https://keybase.io/docs/command_line).\n\n`+"`"+``+"`"+``+"`"+`bash\n# look me up\nkeybase id akalin\n\n# encrypt a message to me\nkeybase encrypt akalin -m 'a secret message...'\n\n# ...and more...\n`+"`"+``+"`"+``+"`"+`\n",
            "check_data_json": "{\"name\":\"github\",\"username\":\"akalin\"}",
            "remote_id": "cd238f392d451ea22da7",
            "api_url": "https://gist.githubusercontent.com/akalin/cd238f392d451ea22da7/raw/5adddb8173ed66e035a67cd2bce8555a7c798034/keybase.md",
            "human_url": "https://gist.github.com/cd238f392d451ea22da7",
            "proof_state": 1,
            "proof_status": 1,
            "retry_count": 807,
            "hard_fail_count": 0,
            "last_check": 1443780855,
            "last_success": 1443780855,
            "version": 1,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 4,
            "payload_hash": "a2a877a4cda04a32aa02de40137d22b2a0a04f35f7f677bdcae15fbf91f4ec04",
            "sig_id": "dece17e125e06d6b44438218d29ab55a7785beaeb7bf372d8f98eb228009635c0f",
            "sig_id_short": "3s4X4SXgbWtEQ4IY0pq1WneFvq63vzctj5jr",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowF1UUtIVFEYnrEUZ3qg9KIcWpzQHox57p37nDYFtbBFEQZhNU73ce54UO+Md0Zz\nskGtREQSF5KoEUEZVgpqZUXRTGCKiUEmUTFKCmItXIT2GCrtXLFdnc055/u/7+P7\n+JvWrbLYrdHqU2eOobxS60hsVrYcT5rOrwSyXw0DdyUoQsuXhnUfMgIG1kPADSAv\naxTvUhhJoiHHyIIg0IJEHpoAWZmDLMWxrIungBMU+oOmgtjIUhDtxX6CkY8XqwT9\nB79seYBkGVGqKDIUlGQeUrSL5mhFE1RNEXgRQpMYRIYulSDCloqkYqyDiBMQrBwr\nyEy8MjOQquLQ//ihcMAEziLZuyL1ylhXSVmiKEdGEPt14KYIUwlhU0sxUGR4lmI5\nJ0AVAWwgLzYZLM8JkBwnCBionFiS7CynyDSv8gqnCIgTeUFGjAJFWlAoUYI8QwuI\ndnGaJGo8UhlVoF2qSHMaZBnSGJhlSnU/cDMkpuQjlkHs06VQmYFAxF5vday2WO2W\nlOQkc18Wuy3t7xZzalItfbY22/XtT+Ij9S/66x5/Vye5o73r38c8P5Y8sXjH+fyU\n0eZzvxrXFnfVJbVPeXrG98/mBsYHWr5cobunN1Tfe7ax4KkWmZpy1kZ21Q4NuBur\nEu+GPTVrhtoy0gOvotH2QLTl5cLXsh46O9Y/aWMeZR7RC7f8bswani55k+isT0/2\n3Rnc3emQLt3IvtsafjBxoC/c13Cy+dDtbXhnZvqOmydej+7D3qK0xMG0hapE9/zP\neJ2jcCbj4Vg4q2ux/+PWOXu8puBDMZt3nz98sSO4OJb2HHKnl77lXm2dzXHMwNTN\n8723dHZizl/LDpZcnqu40LBp7Jrr01s29rlpT3XWHw==\n=GIZO\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"bf805b6051655371\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"service\":{\"name\":\"reddit\",\"username\":\"akalin\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1409475156,\"expire_in\":157680000,\"prev\":\"99456cb27d7c6c8e6978be4c0928c19a07428e236fa9f7ed4d823d926f054701\",\"seqno\":4,\"tag\":\"signature\"}\n",
            "sig_type": 2,
            "ctime": 1409475156,
            "etime": 1567155156,
            "rtime": null,
            "sig_status": 0,
            "prev": "99456cb27d7c6c8e6978be4c0928c19a07428e236fa9f7ed4d823d926f054701",
            "proof_id": "deea58754d6d77d60e621810",
            "proof_type": 4,
            "proof_text_check": "\n\nowF1UUtIVFEYnrEUZ3qg9KIcWpzQHox57p37nDYFtbBFEQZhNU73ce54UO+Md0Zz\nskGtREQSF5KoEUEZVgpqZUXRTGCKiUEmUTFKCmItXIT2GCrtXLFdnc055/u/7+P7\n+JvWrbLYrdHqU2eOobxS60hsVrYcT5rOrwSyXw0DdyUoQsuXhnUfMgIG1kPADSAv\naxTvUhhJoiHHyIIg0IJEHpoAWZmDLMWxrIungBMU+oOmgtjIUhDtxX6CkY8XqwT9\nB79seYBkGVGqKDIUlGQeUrSL5mhFE1RNEXgRQpMYRIYulSDCloqkYqyDiBMQrBwr\nyEy8MjOQquLQ//ihcMAEziLZuyL1ylhXSVmiKEdGEPt14KYIUwlhU0sxUGR4lmI5\nJ0AVAWwgLzYZLM8JkBwnCBionFiS7CynyDSv8gqnCIgTeUFGjAJFWlAoUYI8QwuI\ndnGaJGo8UhlVoF2qSHMaZBnSGJhlSnU/cDMkpuQjlkHs06VQmYFAxF5vday2WO2W\nlOQkc18Wuy3t7xZzalItfbY22/XtT+Ij9S/66x5/Vye5o73r38c8P5Y8sXjH+fyU\n0eZzvxrXFnfVJbVPeXrG98/mBsYHWr5cobunN1Tfe7ax4KkWmZpy1kZ21Q4NuBur\nEu+GPTVrhtoy0gOvotH2QLTl5cLXsh46O9Y/aWMeZR7RC7f8bswani55k+isT0/2\n3Rnc3emQLt3IvtsafjBxoC/c13Cy+dDtbXhnZvqOmydej+7D3qK0xMG0hapE9/zP\neJ2jcCbj4Vg4q2ux/+PWOXu8puBDMZt3nz98sSO4OJb2HHKnl77lXm2dzXHMwNTN\n8723dHZizl/LDpZcnqu40LBp7Jrr01s29rlpT3XWHw==\n",
            "proof_text_full": "### Keybase proof\n\nI hereby claim:\n\n  * I am [akalin](https://www.reddit.com/user/akalin) on reddit.\n  * I am [akalin](https://keybase.io/akalin) on keybase.\n  * I have a public key whose fingerprint is 07BF 173C 4AA2 064B 8882  8A64 BF80 5B60 5165 5371\n\nTo claim this, I am signing this object:\n\n\n    {\n        \"body\": {\n            \"key\": {\n                \"fingerprint\": \"07bf173c4aa2064b88828a64bf805b6051655371\",\n                \"host\": \"keybase.io\",\n                \"key_id\": \"bf805b6051655371\",\n                \"uid\": \"ebbe1d99410ab70123262cf8dfc87900\",\n                \"username\": \"akalin\"\n            },\n            \"service\": {\n                \"name\": \"reddit\",\n                \"username\": \"akalin\"\n            },\n            \"type\": \"web_service_binding\",\n            \"version\": 1\n        },\n        \"ctime\": 1409475156,\n        \"expire_in\": 157680000,\n        \"prev\": \"99456cb27d7c6c8e6978be4c0928c19a07428e236fa9f7ed4d823d926f054701\",\n        \"seqno\": 4,\n        \"tag\": \"signature\"\n    }\n\nwith the PGP key whose fingerprint is\n[07BF 173C 4AA2 064B 8882  8A64 BF80 5B60 5165 5371](https://keybase.io/akalin)\n(captured above as `+"`"+`body.key.fingerprint`+"`"+`), yielding the PGP signature:\n\n\n    -----BEGIN PGP MESSAGE-----\n    Version: GnuPG/MacGPG2 v2.0.22 (Darwin)\n    Comment: GPGTools - https://gpgtools.org\n    \n    owF1UUtIVFEYnrEUZ3qg9KIcWpzQHox57p37nDYFtbBFEQZhNU73ce54UO+Md0Zz\n    skGtREQSF5KoEUEZVgpqZUXRTGCKiUEmUTFKCmItXIT2GCrtXLFdnc055/u/7+P7\n    +JvWrbLYrdHqU2eOobxS60hsVrYcT5rOrwSyXw0DdyUoQsuXhnUfMgIG1kPADSAv\n    axTvUhhJoiHHyIIg0IJEHpoAWZmDLMWxrIungBMU+oOmgtjIUhDtxX6CkY8XqwT9\n    B79seYBkGVGqKDIUlGQeUrSL5mhFE1RNEXgRQpMYRIYulSDCloqkYqyDiBMQrBwr\n    yEy8MjOQquLQ//ihcMAEziLZuyL1ylhXSVmiKEdGEPt14KYIUwlhU0sxUGR4lmI5\n    J0AVAWwgLzYZLM8JkBwnCBionFiS7CynyDSv8gqnCIgTeUFGjAJFWlAoUYI8QwuI\n    dnGaJGo8UhlVoF2qSHMaZBnSGJhlSnU/cDMkpuQjlkHs06VQmYFAxF5vday2WO2W\n    lOQkc18Wuy3t7xZzalItfbY22/XtT+Ij9S/66x5/Vye5o73r38c8P5Y8sXjH+fyU\n    0eZzvxrXFnfVJbVPeXrG98/mBsYHWr5cobunN1Tfe7ax4KkWmZpy1kZ21Q4NuBur\n    Eu+GPTVrhtoy0gOvotH2QLTl5cLXsh46O9Y/aWMeZR7RC7f8bswani55k+isT0/2\n    3Rnc3emQLt3IvtsafjBxoC/c13Cy+dDtbXhnZvqOmydej+7D3qK0xMG0hapE9/zP\n    eJ2jcCbj4Vg4q2ux/+PWOXu8puBDMZt3nz98sSO4OJb2HHKnl77lXm2dzXHMwNTN\n    8723dHZizl/LDpZcnqu40LBp7Jrr01s29rlpT3XWHw==\n    =GIZO\n    -----END PGP MESSAGE-----\n\nAnd finally, I am proving ownership of the reddit account by posting this on the subreddit [KeybaseProofs](https://www.reddit.com/r/KeybaseProofs).\n\n### My publicly-auditable identity:\n\nhttps://keybase.io/akalin\n\n### To join me:\n\nAfter a day of posting this and completing the proof, I'll be granted invitations to Keybase. Let me know\nif you would like access to the alpha.\n",
            "check_data_json": "{\"name\":\"reddit\",\"username\":\"akalin\"}",
            "remote_id": "t3_2f2l3y",
            "api_url": "https://www.reddit.com/r/KeybaseProofs/comments/2f2l3y/my_keybase_proof_redditakalin_keybaseakalin/.json",
            "human_url": "https://www.reddit.com/r/KeybaseProofs/comments/2f2l3y/my_keybase_proof_redditakalin_keybaseakalin/",
            "proof_state": 1,
            "proof_status": 1,
            "retry_count": 33,
            "hard_fail_count": 0,
            "last_check": 1443790133,
            "last_success": 1443790133,
            "version": 24,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 5,
            "payload_hash": "297ac58d197e52956fae5d628b4db81de68e8cebf1bfd17fbb4404cee7edba6f",
            "sig_id": "fc16a961ab69ae81910f3a76ca000fca19699dec8461fc7ec0e8fcb78764069b0f",
            "sig_id_short": "_BapYatproGRDzp2ygAPyhlpneyEYfx-wOj8",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowGbwMvMwMW4vyE6ITA1uJDx9IHXSQwhTI92VCsl5adUKllVK2Wngqm0zLz01KKC\nosy8EiUrJQPzpDRDc+Nkk8REIwMzkyQLCwsji0QgI83CwDTJzMDU0MzU1NjcUElH\nKSO/GKQDaExSYnGqXmY+UAzIic9MAYpiUV8KlkhNSko1TLG0NDE0SEwyNzA0MjYy\nM0pOs0hJS7YwtzQwACksTi3KS8xNBapOzE7MycxTqtVRAoqVZSanglwMlctITM4G\nKkwtL8alp6SyACRQnpoUD9Uen5SZlwL0MFBHWWpRcWZ+npKVIVBlckkmSK+hiYGl\nibmpkamRjlJqRUFmUWp8JkiFqbmZhQEQ6CgVFKWWgewwSrQwN080SU5JNDBJNDZK\nTDQwSkk1MTA0Nk8xMkoySjQAiqcZm6aZp5mZmyelJCemGpqmJaVZGqaZpCYbmCiB\nPFSYl69kZQp0ZmI60MjizPS8xJLSolSlWq5ORhkWBkYuBjZWJlCcMXBxCsBiMlCB\n/X++0Ivzb27KOOkcX77oYEHUY8mvdza5T9IwZsmcKerHve7BMqfyafvXSx09HFXv\nbsWpMO2qtmtd2YLvX/xn7lfV0hZ3NW1ZcTDpdNYXERl+jpv/L/AYpdppV0y70fF8\nkspFbd7jLh8Tf2vcKmfkiT9wU+qB7eLkkz7KWxliNFfN6l5t/YR/y6QlMnoSHEWW\ne3TSJugLrlA+8GShEc9bk/zo1brtJ6Y7SD9QPDoprbliR12/JG+N2nv9lanPX/9L\nSvTi43Hin6gXv4tRuYrnsdnCabsPHhbbEvfx9z6HD1s3tlnxx5aVzj+qvrCy6dOj\nJK9Tus4anyrMcjSrM+L3ZvxQnn34i4z5gaX2kv0SlgA=\n=RqrD\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"bf805b6051655371\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"service\":{\"name\":\"hackernews\",\"username\":\"akalin\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1409475252,\"expire_in\":157680000,\"prev\":\"a2a877a4cda04a32aa02de40137d22b2a0a04f35f7f677bdcae15fbf91f4ec04\",\"seqno\":5,\"tag\":\"signature\"}\n",
            "sig_type": 2,
            "ctime": 1409475252,
            "etime": 1567155252,
            "rtime": null,
            "sig_status": 0,
            "prev": "a2a877a4cda04a32aa02de40137d22b2a0a04f35f7f677bdcae15fbf91f4ec04",
            "proof_id": "1e78f7bc1e740b086b64ef10",
            "proof_type": 6,
            "proof_text_check": "_BapYatproGRDzp2ygAPyhlpneyEYfx-wOj8t4dkBps",
            "proof_text_full": "[ my public key: https://keybase.io/akalin; my proof: https://keybase.io/akalin/sigs/_BapYatproGRDzp2ygAPyhlpneyEYfx-wOj8t4dkBps ]\n",
            "check_data_json": "{\"name\":\"hackernews\",\"username\":\"akalin\"}",
            "remote_id": "akalin",
            "api_url": "https://hacker-news.firebaseio.com/v0/user/akalin/about.json",
            "human_url": "https://news.ycombinator.com/user?id=akalin",
            "proof_state": 1,
            "proof_status": 1,
            "retry_count": 172,
            "hard_fail_count": 0,
            "last_check": 1443810056,
            "last_success": 1443810056,
            "version": 22,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 6,
            "payload_hash": "04e5fba11ba32227cacb8b8345cec9bee563bfd4c25b77aa716b3763c3857735",
            "sig_id": "ac377cb7d4df7bc9db7c50b3bc06146e05e29fd0eb715b478ea461074a8770550f",
            "sig_id_short": "rDd8t9Tfe8nbfFCzvAYUbgXin9DrcVtHjqRh",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowFtUVtIFFEYnjXXciPYJBDSHhrDlhKdmZ3rUtRDFBZY4o0usp4zc2adVmfW2XW9\nJWw3osCMraA0RHMV8iHSSskStZsPWr0EQi4E9qBbD1JiYgZ1RvKt83LO//3f9/3/\nf/4bWzYQDtto5HRFISqqsU2OJSBRnPSFayKhoTSQnibSj9YuVdN9yAyYmh4iPSQl\nQJUW3DILAEPxLBRFkREBfqgixUGe4mie49wCTeaQlUbQUmAbCIIoVzMwhgOvpmD0\nP/zatQSCENGKJLE0BaBA0Yyb4RlZFRVVFgWJoixiEJk6qEaYDfygStPJ5hwSY2FN\nRlbHilENMLiezZXrsShgGiFDNqowrOhBSxFqCFgWdQh6/4m9UNMVPC6mh5EZ1Axs\nQmOmHNKsajRLSayAu2VzSFQf0EzktcrQnMCLFD5WERTGlowkAJkTFVoSEMdIHK8C\nxCk8I0JWgSKtIF5EoozwT0JVoQUVQpalWBkhASkQ8CppjVOjG6SHx20CH7YMaj4d\nhGpNRDY7rtoykwmbg0ixJ1kbIxypzvU95hEb//jbylrmj0y1vD/oW9w+7Mh8WVBx\nXJ7ImGZ25HculT9alj8VVZ4/2p1uTzt3K1p6J7GvpXy+qytt+G5b/IO0/Pzai7TL\nsZsX98R+j77bvfp4PjGQAGS90w8bGzp/fH/Wk1cCxg8/eBM5k5L9tTQ8qM2uXhox\n29u3tvqnJ4gr/U+mFiNsxq6ToqNuhpvJTl0pdrjMlWjcTOr72Xc9NnIv43VvR1XF\nCfsCd6ij0xntjxT8upD8Od7o2nS7tNG3kCVlndLm+P0fdw6m2xfPHhg7NuZpNebS\nl0bvd0z29hDjJc4hV7R7c9lsrPjtq2Dkm3tb/tPl7DqXgMShgcL4w71/AQ==\n=FZj7\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"bf805b6051655371\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"service\":{\"domain\":\"akalin.cx\",\"protocol\":\"dns\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1409475374,\"expire_in\":157680000,\"prev\":\"297ac58d197e52956fae5d628b4db81de68e8cebf1bfd17fbb4404cee7edba6f\",\"seqno\":6,\"tag\":\"signature\"}\n",
            "sig_type": 2,
            "ctime": 1409475374,
            "etime": 1567155374,
            "rtime": null,
            "sig_status": 0,
            "prev": "297ac58d197e52956fae5d628b4db81de68e8cebf1bfd17fbb4404cee7edba6f",
            "proof_id": "c8968b3d21de18ae33d30210",
            "proof_type": 1001,
            "proof_text_check": "keybase-site-verification=rDd8t9Tfe8nbfFCzvAYUbgXin9DrcVtHjqRhB0qHcFU",
            "proof_text_full": "keybase-site-verification=rDd8t9Tfe8nbfFCzvAYUbgXin9DrcVtHjqRhB0qHcFU",
            "check_data_json": "{\"domain\":\"akalin.cx\",\"protocol\":\"dns\"}",
            "remote_id": "dns://akalin.cx",
            "api_url": "dns://akalin.cx",
            "human_url": "dns://akalin.cx",
            "proof_state": 1,
            "proof_status": 1,
            "retry_count": 2518,
            "hard_fail_count": 0,
            "last_check": 1443826575,
            "last_success": 1443826575,
            "version": 2,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 7,
            "payload_hash": "bc78b22a430b8c6fd1b7f13b4d518db0b59a6ffa9356a501d087078fde529f5b",
            "sig_id": "043dabef311e0061afad845c64f4a9bd2098353bff93d8267e70c9d90d353c1f0f",
            "sig_id_short": "BD2r7zEeAGGvrYRcZPSpvSCYNTv_k9gmfnDJ",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowG1VFuIVVUYnuPYjE4K81AqNUFuMQqmXPfLRD70UEKIDZqWlse11l5rzp4zc/Zx\nn33UaRobCIx5CCRQ6CGMSCJFcYiUUBm7IWSh0ZQP1kxPXWQetKykUFvrOE4JIfTQ\ny97/3mvxff/3/Zed85ub2gpjwxs2ddvVmwunzuumNbOmzg5GOo0Hoq7ByPQltpKH\nqKL6bdQVle2AVjX7UJLeu9XqqDPaYrNaklb8kUHIQeCcdUZj7iiKqcKIcy6Z5JoC\nACwxAJNoqDPABFCXVHpsVs2SwBEBrh3k2BClEGBECyGQUD5wAlDNAIWMUsyhZy2l\ntfymbKIGZjGJ/d9/uV9vHFitLYylJBAozQFEGDFknIidEVwCEC7WbDYtVZVVX1IJ\n2eaZMuWQrydLTC1ESVycUQ5pZ9SnarlnL5qS8pL8PywZEQxgfhNmnvb3qnJdp1tr\n5SRAX8/LcQa9TZpCq4xT/u0EwVRiBrVt5DVtWNB4s2lGCetkjAzXsdPEEaktYZzF\nGjsFAeGaECpFMKHcIAMQQAqkw1z5AjlidRxLyxylEiKDDKOMIySsYloQf8VHMQNE\nSUMoUIhSYyRQIffM9qe5LVazNHXelA2+XfKkf0Y8lVJ0RqaeZYEVKK6hMDERmnKN\noBXEYkhjYxBBiCLLFYuhhtR/K2OoY0xL4qADGEnnfPp2GpxS4gvbAK9mdkvwABDt\nqx17C5E12AjnKysERIJiwbyXxlF/wQMzxYVvAWGtktIoyAxzMHgzrSX429DTaP6S\nNeVirHJV7K2FQs9MQU+Sl+o6unVpGzjFfKDqT3FnVMtVHtL3JzW7uZJGXdJHSc/1\nnhWGW+JtMVYoY5UyjCnhtH8AHpqDC86h8F5owbRhvl1FjJkfDuSMZVyD4FBAu86H\nhp5r0BRzlfSFxKtqoC9VcbGkaqVblONE9T8U5P+RMRQGriFieu7+sWOCeTd6jABJ\nPB7yA2a3VZPMFpPGLHIm/KoBM80BiKVOKwi130YIcaOMFlpgQo01flgsZVi7mBhE\nNedKccj8+mLYYEE5x/RvoZ4pVz0e0sutqLye2WiobaTQMbup0NbUctussDqb2ua2\n39inP/3Qem1w14VV2w7OG5m89MiKjjte6u9a504/4AaOFc6eO08m9u7e2Xx6weGN\nR0ZZyexp/w2f6V7+x7uLlq99/GF2dGzFyovdFw5exW+fudhy7IUDYwvoPeOz7zbN\n8xdWWt6Y/OXOg082jV4+crl+6K6lVx4kmx79mJ3bY0D164tPfLB4Ysmn30+OyVeu\npGv3L31qJbj9aHe+6J2NO19vXfzs8NxLB+ZNrdnzzfAzBXioY86H63/eseS7184+\n/dZj5sev3n/v5ODYSXJo1viy3qXjI2tevnpi6rOJ5TtWdXwkeveN7r7v/muVF79Y\nn33L9x3evvfL39uj53e1Hj+1LNv7ycKF2/88/uvnq1816f6Tb87Ztynv/As=\n=BdWf\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"c22f10ffefcb37f52d5a327779697b5000e4c034\"},\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"bf805b6051655371\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"track\":{\"basics\":{\"id_version\":15,\"last_id_change\":1396486037,\"username\":\"tomjakubowski\"},\"id\":\"ef761796b51eacfab51f84359361be00\",\"key\":{\"key_fingerprint\":\"ca8ef9d2c7bdfb4f49be4676db3fa1047b445981\",\"kid\":\"0101509f37ad5af4ebdd9e6f55912c2c6567228ea6b84ad58ead604a9c450a255cc90a\"},\"remote_proofs\":[{\"ctime\":1396485998,\"curr\":\"00a7b18cd48b57b21e84e315dcc242252e7a6d1b15c24acc5f66b94f1f0329ff\",\"etime\":1554165998,\"prev\":\"c04b051d7962ec3c8f2328812853861f8cf5b0524a6a78e1d8eea99ca16c6f11\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"tomjakubowski\"},\"proof_type\":3,\"state\":1},\"seqno\":9,\"sig_id\":\"8c7e4b21ce8aceaac66a8fb6a8077617787718032b86bc68798d36a642fce67b0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"00a7b18cd48b57b21e84e315dcc242252e7a6d1b15c24acc5f66b94f1f0329ff\",\"seqno\":9,\"sig_id\":\"8c7e4b21ce8aceaac66a8fb6a8077617787718032b86bc68798d36a642fce67b0f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1409477827,\"expire_in\":157680000,\"prev\":\"04e5fba11ba32227cacb8b8345cec9bee563bfd4c25b77aa716b3763c3857735\",\"seqno\":7,\"tag\":\"signature\"}\n",
            "sig_type": 3,
            "ctime": 1409477827,
            "etime": 1567157827,
            "rtime": null,
            "sig_status": 0,
            "prev": "04e5fba11ba32227cacb8b8345cec9bee563bfd4c25b77aa716b3763c3857735",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 8,
            "payload_hash": "509a15dbf19107aa39362160bc2dd5a02175eac38b15315af8d845dabc2836b3",
            "sig_id": "2d0f5396223f80fb142bc5fc577c332533da015b4b7c0820b2959c267e6782040f",
            "sig_id_short": "LQ9TliI_gPsUK8X8V3wzJTPaAVtLfAggspWc",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowGtVG1ollUYfrf1oQNNghqUlD3+EGwb5/tjCElBSK7CDBJEX8/n3meb7/vueZ5p\nUwZiBE5i9CMHs0hBS/pADQtNpDRHIbmCCCLGKAkWSFAZZRFp57wu04iI6Pl185zD\ndV/XdV/3eXZeS6m16d1tazescqsHmj48r0uPt0wt2Jromh1KurYmpj911SJWVbXR\nJV1JnxvSKnedaW3RZqeT9mSTy/K0Vg1HBiEPgffOG425p8hShRHnXDLJNQUAOGIA\nJslwe4SJoD6t9risnqWxRwK49pBjQ5RCgBEthEBChcILQDUDFDJKMYeha6WWF9ex\nSRqY5dSGv39zf7Bx4LR20EpJIFCaA4gwYsh4Yb0RXAIQL+Yum5Wq+lR/Wo1si0yZ\nvsg3NEtNHqvUlq8qp+1Jv8qL0LxsKiooSrogAZRhgqG4DrJ3MC/Sam4qg5WI2yAF\nnYZIYiMsslJooAj1ghhEmLMGNEjNuhUF/sUxJD1o6BFeY2SED/5rQ6EPDiIikWAA\ncewjRqMZgAAqZiDW1hACDGKKSusZxAxBBQgXyEKlJATMAq0VMQ5w5JgNnwTYWoUM\nUJF75jbWCleuZ7WaD46sDVkp0o1XpGMuMWOyPTGDWRa6ShlMZ0wQqa313FBmEOQO\nWwm9p0G1MhQgBwmX3lGsCTOGQUQ0CW2VijFzs+CUhVtXwGcZRFcaLBp5rTjTV7aq\nUOXePM7manCLzWlRuCz5p3k0YMrFUD2cofYkL1QRe4aT3A1Ua6EMVdpzJWWGCymA\nJR4hGMy3WEIjoBDUaG1dVAWDx1zFKYQ0hlAaLiklDISRQGxAnEpEm+03vK7Rplyo\ntD/yrquh/pqy5YrKK/+Ph/9CxMnkP8gYjivSEDG7Kde8CtG8P4MhKRZcBALuyXqa\nuXIab1DORMg5iP67TXF/AyuNkCIYaGGYt1BzHyJLLIXCaqCpVMx7JTEN8QXQAsEB\nFz7QDftAr5Eatq9QPQEyCK6qYjBzyXDrSNPCG0pNraWbbmyOj12pde6CP17AI7tu\nvryq6dSSxV89M3X/B8UbC16aefFnUYyufN3f+sil3345P1Kfs3xm/05T7Ridev+t\n4qlP9xw9NL3l+cXq5BOfFROnDt03fXTdmn29B7/Nxvpfnhlc/APeOwAfvrRjYGC9\nHdveJtq+x8vGhiZP7B9H88+27K59ueKOvd3vXJhHx++dPDZ9J7pwYs3Biz+tGDl7\n2y6oDu9/oPuu20vfDcxdf/Ch5yYvP71nfu2jjuadby85snx8UXHxzNjUo8eXvdn2\n8TcLV54bWVo6PWEefO3A0qXNW7Pz43dveI/U89Wf2K7jZzofG57b/euBWsemr+U9\nO0Z73b4ft7xKPz+37/SxOc0vuFcubB/5YtvuWzrHZ+wE+h0=\n=LHXi\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"c22f10ffefcb37f52d5a327779697b5000e4c034\"},\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"bf805b6051655371\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"track\":{\"basics\":{\"id_version\":5,\"last_id_change\":1405634318,\"username\":\"justinschuh\"},\"id\":\"1eb1293c8d2d98b0a45f84c246edc000\",\"key\":{\"key_fingerprint\":\"029f0994108fb32c8f0ffbc51fa202492860273f\",\"kid\":\"0101a6c13bdc440c26a59df613621a04782d1aa9106d0bba4ce072e6dddd903dda2c0a\"},\"remote_proofs\":[{\"ctime\":1403793669,\"curr\":\"99d9966849bddf7c56c217e3d91ff5d98ac502e1479fe53b46cc6124b4dddaab\",\"etime\":1561473669,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"justinschuh\"},\"proof_type\":2,\"state\":1},\"seqno\":1,\"sig_id\":\"c78980d4f221fb3d391c81885cbbde5d981dc47a49285b6bf8c795546002713c0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"99d9966849bddf7c56c217e3d91ff5d98ac502e1479fe53b46cc6124b4dddaab\",\"seqno\":1,\"sig_id\":\"c78980d4f221fb3d391c81885cbbde5d981dc47a49285b6bf8c795546002713c0f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1409538781,\"expire_in\":157680000,\"prev\":\"bc78b22a430b8c6fd1b7f13b4d518db0b59a6ffa9356a501d087078fde529f5b\",\"seqno\":8,\"tag\":\"signature\"}\n",
            "sig_type": 3,
            "ctime": 1409538781,
            "etime": 1567218781,
            "rtime": null,
            "sig_status": 0,
            "prev": "bc78b22a430b8c6fd1b7f13b4d518db0b59a6ffa9356a501d087078fde529f5b",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 9,
            "payload_hash": "9ca79719d489099193dfefd2f76f9780cf73762c19305d4e377c1b4903a75278",
            "sig_id": "5bf1e508a73c3f37aa737731c2474d64446786f4615bfd7e04e407aa2278549e0f",
            "sig_id_short": "W_HlCKc8Pzeqc3cxwkdNZERnhvRhW_1-BOQH",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n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\nfnT2VVu+W/gz3vnle+f9/umjI9//Aw==\n=F1Eg\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"c22f10ffefcb37f52d5a327779697b5000e4c034\"},\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"bf805b6051655371\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"track\":{\"basics\":{\"id_version\":10,\"last_id_change\":1411673314,\"username\":\"gobiner\"},\"id\":\"e77a4f4103429648f544ca88ae9ac400\",\"key\":{\"key_fingerprint\":\"c29b5d863ea7e71d94f521b3ffa425daa3785f5f\",\"kid\":\"0101941c6e733b0a7a039b1eebe6bf4ebf8ae494fb6b500bf36078d27cae184ea8500a\"},\"remote_proofs\":[{\"ctime\":1411671108,\"curr\":\"7fc0d1dcc8e9de68d1352d8e67312e798eed2d75e12f6490e135ab39057bd2fd\",\"etime\":1569351108,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"Gobiner\"},\"proof_type\":2,\"state\":1},\"seqno\":1,\"sig_id\":\"a138e56f51d3d857b25576e12599d029873f2a19f7189a5e7813fc9c28c6809c0f\",\"sig_type\":2},{\"ctime\":1411671207,\"curr\":\"95d2e9a018ad49b9ff12f9830b943862e75b7e00027cecde9fc86cf4af7a8ee9\",\"etime\":1569351207,\"prev\":\"7fc0d1dcc8e9de68d1352d8e67312e798eed2d75e12f6490e135ab39057bd2fd\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"Gobiner\"},\"proof_type\":3,\"state\":1},\"seqno\":2,\"sig_id\":\"68fd1b7ac6a4ce123e92292b68a4c2c627cc6cd741cb3cd69dac702aa19363ab0f\",\"sig_type\":2},{\"ctime\":1411672221,\"curr\":\"01f8ca1c36d94eda2bd5165dd77a7a361b3b218c6ce886714975c3e6aeef3d98\",\"etime\":1569352221,\"prev\":\"23c40df880d6b972d48de4469ebd5e7a7120ae0d506f70bf31d5108ef69bf3e6\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"hackernews\",\"username\":\"Gobiner\"},\"proof_type\":6,\"state\":1},\"seqno\":5,\"sig_id\":\"004ddea191ea4ef4b04632a8aca00c686c3657e15661c027c82b8f7c68907bf00f\",\"sig_type\":2},{\"ctime\":1411673282,\"curr\":\"e38c8530776b8c9f2b87545a7406695ba1f122ac4a1d5a17fdf4b6da8413bcf3\",\"etime\":1569353282,\"prev\":\"01f8ca1c36d94eda2bd5165dd77a7a361b3b218c6ce886714975c3e6aeef3d98\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"reddit\",\"username\":\"Gobiner\"},\"proof_type\":4,\"state\":1},\"seqno\":6,\"sig_id\":\"852c8d8bc5438ada66ab7e533f8185eef244bcbcc1cf8b1d230c3ad9855315550f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"165f7421dfcaba511f5c47bcd82100b93f7a81ce1008078f3fd6183d0edc1ffd\",\"seqno\":13,\"sig_id\":\"1d298f57b048c2d5407fa1636754ff8d73128880261ff9f95885c4d9399685140f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1411694605,\"expire_in\":157680000,\"prev\":\"509a15dbf19107aa39362160bc2dd5a02175eac38b15315af8d845dabc2836b3\",\"seqno\":9,\"tag\":\"signature\"}\n",
            "sig_type": 3,
            "ctime": 1411694605,
            "etime": 1569374605,
            "rtime": null,
            "sig_status": 0,
            "prev": "509a15dbf19107aa39362160bc2dd5a02175eac38b15315af8d845dabc2836b3",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 10,
            "payload_hash": "4cb503967897079bcbd6f37ad4a0446a0b4cd4b08ab7b271a1bfe98694384972",
            "sig_id": "769dbf1042afea9698d97bc36aad578f968f6a32893a641fdfcbf19727e8a21c0f",
            "sig_id_short": "dp2_EEKv6paY2XvDaq1Xj5aPajKJOmQf38vx",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowGtVmmIVWUYnlFLnIwisEgl5Kg/glG+fTEUcYFCBoqZH4HL7Vvnnlnunbn3zEyD\njmlFIYEYJC2mBGKk9KPFJWyZSgS1RbACIUrJJLKiIqSsMXvPdRpHR0Khfx/nnPu8\n77N873s33Ty+rqF+cN3yhx4Izd31H52xdS2z3u9Zndiy70/mrU5cRxpKWX4qmc6Q\nzEvaQ7811TA3Lc/oCzZpTHpDpZqWS/DKERIxijFEZ6mMnHhuKJFSaqGl5QihwByi\nLBlozGFy0JiWWkOlq5LmNRIkbcSSOmYMQYJZpRRRBg5RIW4F4lhwTiWGqsVyNbus\nm6SGWUg9PL3K9z21F8HagL3WDCNjJcKEEkFcVD46JTVC+YfVUBmmatpNR1rKu80q\nxrXn/UKx1FXzU+oLI8xlY9JhqhkUL7iiAUbJPMwwFlJiTC6D7OqLOV6tGaYV8OQk\nMNDGBRsQ4dhHa4mGZ7VmhlXKiV2ulKFcMIIZRY5ySqkg0bNAqSEaBy+YN/AAW5Zj\n1IohjLDmmDGjuSeaYvh9UFRbjaMAoSnj3AkkrGEKySiNkoZJFXnAxMtgXPQBmbz3\nSugsZ6HQVSmXIyixHDKSpZ0jlIXiqjFxPZVKbgTWjDElFYrMByqCwxAQF6jSSllK\nUbBGYgAWXCIRtfOagDUE5FAM2wjth2FwLjRwroEPd5CrUuuiltNicO0FbzJTaKvm\nnowENutLsyxUkit9iDUnagCFrL8LnoJV1cxkeTV4Uw3dpTIc4ZS2XsyVJloRa4CF\nZ4YTYZShCiOlNXJIemSMJp4oFQQ2XBMKCQNfMcUawpgLmxPK0YbrDTReKZ7GfEQ8\ngqIHUVSMMgbKILOBGGeZt54jz3AMkBzoRtLASRRBKMKog9B7boOjdIx4NfCuSuj9\nD2e+vw5rrs+H1jQr9thrsIFezQYyygZisFUwUQz1CBTADmwRBkvtMfKeMxEIkpII\n7pTyyEetKNeMMyK04qDxNdig0YgNEcaWijZ45BxDEVTGHC4TQxZQY65roDHqQJim\nyjO4K9RJJHkUNBhJ+VgbcvBhG/4Hj6/PhSIMMpA/9FXHOHHRhw9mXnJCXM0JOsoJ\nRRD3kXOYFsTCHGJSMOS5DM4QC+tAGRUksjCQFEwZxBQMt2CZIYIamDRkjBMra2UK\nmUk78qa7TH9H2fhC0VSLUM5rQBIQTyYwIXDDIkOQXOE4pFOIgDEVDKTyxgZJnAmW\nKKQgwiEfHYgkIyTEKBJRUQg/TM68YxRQ4BLUV0p4DTdZRoi8ExwCxhiQicBBCRG9\nwFw7giBzoFy+I2oUhlfFqLWYSzc6XRoYwMIID3ellVBI8y8glAq246VYaGekllh7\npjTSGmvqYat6EiXcRLizLkoqYXPBCzAABr+UDqa9RtRITqS6RBQDaGZaARP4lkzW\nUwnJQMOG+ukT6uob6m68YVy+7usaJt3673+A5z+deGHtqj1y6PHmjw9P/mXt20sf\nfP18eq77jbdWrZ29YOfeFX81tfWen/z3thPP3ffqyZVTjxbv2Dm4a8WCl7fM7zy0\nfc69O1sOHqifNRWtPrfjlSWHPlk8NLB5WemxU0faTrBF4/Zt+3B/+c3Zu9IJC9V4\nsuie5n3LBh+R+4/sn3ns9+bBw0OnX1q35r1pv3319FP3P7n90c6zP3/z4jGR7iTv\nLH5h68Znw207bunfg7ubzv7pv14zMDB3+9De2P9H76Q5myZu3dp0amFlyozJ64+3\nodsP3rmBbJy7tDDtps2/vjt995LXLjzxeemLZ/ae+XLLt+uzQ013zz989PjiCy1r\nJhwgyzsP3DXp9Mkfpu6eUjj602ctP6rv/gE=\n=Oz7R\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"c22f10ffefcb37f52d5a327779697b5000e4c034\"},\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"bf805b6051655371\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"track\":{\"basics\":{\"id_version\":7,\"last_id_change\":1411677112,\"username\":\"pwf\"},\"id\":\"498aa252e4b50cebe0251dfbb292e400\",\"key\":{\"key_fingerprint\":\"a356421430c3533362fd4e33a291ed64da2fd1b4\",\"kid\":\"010195144a95d2931564e839b91f6b883455c606ba4807f7a87a478f5e12d7eacfde0a\"},\"remote_proofs\":[{\"ctime\":1411676858,\"curr\":\"b194448780f4de36ec110fce38988b330eba71e0a65706f9cd9210a20ce841bf\",\"etime\":1569356858,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"pwfff\"},\"proof_type\":2,\"state\":1},\"seqno\":1,\"sig_id\":\"92982baec1d4a526a8a38108990c07d0aa92d288e61a5923701ceb1319b60a480f\",\"sig_type\":2},{\"ctime\":1411676915,\"curr\":\"20fd06f8ff7fe34410e2acb4dbd50d41fee4bbae73e52f6e68243c051d5bec33\",\"etime\":1569356915,\"prev\":\"b194448780f4de36ec110fce38988b330eba71e0a65706f9cd9210a20ce841bf\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"pwfff\"},\"proof_type\":3,\"state\":1},\"seqno\":2,\"sig_id\":\"2a1b897ba3d04bb1c2986a179d10dd546e2077265c88d0df98359454269857690f\",\"sig_type\":2},{\"ctime\":1411676990,\"curr\":\"f00e8fbed0cc40f6821510140b359f9356e3ff9e24938d47ea3c7075f63ea735\",\"etime\":1569356990,\"prev\":\"20fd06f8ff7fe34410e2acb4dbd50d41fee4bbae73e52f6e68243c051d5bec33\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"hackernews\",\"username\":\"pwf\"},\"proof_type\":6,\"state\":1},\"seqno\":3,\"sig_id\":\"8205df55a872b53347640d57eca2bc228a8e70bd648f6b048bb2eb4a263aba420f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"d9d64688b46122592f406ec6c5cd966e1136473edabe72caeb28080a6e356802\",\"seqno\":6,\"sig_id\":\"f83c11a29bc220e0e57bd5886d98e67f20cc656a144e70f8bb866fd6159c20d10f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1411694687,\"expire_in\":157680000,\"prev\":\"9ca79719d489099193dfefd2f76f9780cf73762c19305d4e377c1b4903a75278\",\"seqno\":10,\"tag\":\"signature\"}\n",
            "sig_type": 3,
            "ctime": 1411694687,
            "etime": 1569374687,
            "rtime": null,
            "sig_status": 0,
            "prev": "9ca79719d489099193dfefd2f76f9780cf73762c19305d4e377c1b4903a75278",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 11,
            "payload_hash": "a8d6f791591901bd574cc1f9ec97a713f772a59ffceb1de6ef07296b0be7cc8d",
            "sig_id": "7d2d653312c4644daf5b6089cb1c3651d3bb8a8f57b7055810ef804fa5c871310f",
            "sig_id_short": "fS1lMxLEZE2vW2CJyxw2UdO7io9XtwVYEO-A",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowG1VFuIVVUYnmNm02RZEaONGbYRRuo4rfvaa5QuU0JKBVIv4cR2Xefs5szZx332\nTA7ToYLAlG4zBkUGZhciBCvSqHlJBdMYM6QkejG1KWuwm0iUJLX2cRoVQnrp6Sz2\nWuf7v+/7v/9/7vKLmloKHz22ctUKe9/qwtiEarp/wY51Q4FKzGDQORTocmwrWX6q\nyD4bdAa9dlDJmu2Ik/kPWxUUgwGb1uKk4q80Qg4C56zTCnNHkaESI865YIIrCgCw\nRANMgnoxh8lBXVzpsWk1jfMaAeDKQY41kRIBRlQYhiiU/uBCQBUDFDJKMYe+aimp\nZeexCRqYUWz8139539+4sEpZaIQgEEjFAUQYMaRdaJwOuQAgf1iz6aRU2SvLcSVn\nm6VS9+Z8fbFY1/JTbKIp5bgYlGUt88UjXZJeUdAJCYQsJBTS8yAHkjSpJHE1yWSW\n5MgNWtoi5ek6aJRhyCAKuckNgIYpRhq0Jv3KJZ7vmQLcCoqIQ0Jarw4hQTnGyihF\nvXMEcQQ1oTbHaBQDEECpsQydERJjY6kJESMMUyEQkowy3zGDAbGOQmcVZxpqw5GU\nTulQQ6ERBTLnntq+JLNRNU0S5z1Z6dOSxX1nxRMuioHuT9OcpoWQ+j9Laoi3X0qN\nOBNSh9RQGiopOCFGUI9PBKTC/xjEFAkJsABpJjx9OwlOmcDsDHg1tQMeHAHEMEQh\nQta3G0PMKVDQaWcwlyBUyHHHhPXOMAt9MDSxDAAuufEP/LNgSkvub0NPI/slq3sj\nIzMZPVTL+zw1BD1xVupXwYVb28CJssGqbSSk5r/n9P1Nza6uJEGnj0Yt7jkTWent\n8K0OhQKepMUOU0cw5aH32mDhx0YoK7S3CBsDsOWSYO6nSno3qDEc5CJytDP1UP3B\nRpkok3E5J16Vg+VEmqgka6ULtWNH9b835P+RUc/nrSFicuzOWTG5eedmTBCOkE/G\nmmqc2ijOX1DOQr9pwFQ4iParBwvGQ8EBF0r7GXM+F4ZIQAiTQBFtiAKh3wgKcSih\nclaEHhuHRHB0ViiEnpns8Zheb0Vm/akN6i3rCtdNbyq0NM24eFq+OptaLr3yn316\n02hz05bi3LnNn4ebdy7p2Ds2btGzv5xYvz0aveW3jrJ7D/341bvF16Yt2PXm8LaN\nY+v/WnXwtnTiwMJXn9rbt3T2no1bb3jj8LIlbZ+MTT88/vZQ26n2d37eeslVL7cO\n/3T3S918eF97+eDSjiWbj+7/4fjvX5Se2d8VzVh+xzVdb83bNf3PFx/47MvZ0etX\nvP/xp8XThW7X3jZavOuF1jldx++5vVq+9pWeRUdawe5bb1zb/MFss2HZsa+ZmXls\nfP7IvCfvfaK17frTp+5cM7d522Xdi9KZszoWl3c/PvHdHzfvW3vgm/ZfBx/9fr45\n0bPnyMyTJ+3iqzd1P3J0LAm2HKrX0aHnl6/ZWf3w6e0L53w7a6SwYWTF3w==\n=qunj\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"c22f10ffefcb37f52d5a327779697b5000e4c034\"},\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"bf805b6051655371\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"track\":{\"basics\":{\"id_version\":3,\"last_id_change\":1411684515,\"username\":\"voronoipotato\"},\"id\":\"ce2bb60f1dbd62d2517d20641d6b6400\",\"key\":{\"key_fingerprint\":\"b07e9524f29aebe12295733bdbb564b42721c45e\",\"kid\":\"0101ac3a8fd9a33de5d82646359922a656777d304ef51feb76c1cd72aafbc8c19c250a\"},\"remote_proofs\":[{\"ctime\":1411684479,\"curr\":\"be115c19a5d4941aac2769ac85d558ba9744d958c149159c14d26b4840e02c69\",\"etime\":1569364479,\"prev\":\"2026312822e537313750b1fcfd37a08b2f7f69e2956e15b6c4e6007a7dd37fcf\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"voronoipotato\"},\"proof_type\":3,\"state\":1},\"seqno\":5,\"sig_id\":\"a27664089b056ee3f35f4357850ad39c039be9cd953dd03e7a43797ba5695dd70f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"be115c19a5d4941aac2769ac85d558ba9744d958c149159c14d26b4840e02c69\",\"seqno\":5,\"sig_id\":\"a27664089b056ee3f35f4357850ad39c039be9cd953dd03e7a43797ba5695dd70f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1411694722,\"expire_in\":157680000,\"prev\":\"4cb503967897079bcbd6f37ad4a0446a0b4cd4b08ab7b271a1bfe98694384972\",\"seqno\":11,\"tag\":\"signature\"}\n",
            "sig_type": 3,
            "ctime": 1411694722,
            "etime": 1569374722,
            "rtime": null,
            "sig_status": 0,
            "prev": "4cb503967897079bcbd6f37ad4a0446a0b4cd4b08ab7b271a1bfe98694384972",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 12,
            "payload_hash": "f7fd588cbcc7708776343be0ea13152ff7fd2f49ec6f27c4c0334dd7feb2c192",
            "sig_id": "73f473aa10f68bb05b817ce34fcfc22a94a44d092100d92b03deea274c3054db0f",
            "sig_id_short": "c_RzqhD2i7BbgXzjT8_CKpSkTQkhANkrA97q",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowGVVWuIVGUY3jU12zI2rKxMsqPUj6b87hdbitXSbkYXy9Bk+q67p5mdWWfO6m7r\nWpSBUaGg1LIZEUT+KVNyKYq0MsQy/xSEiCBFIEmBVrAKgb1ndt1UDGoYho/vHJ73\nfS7vOxunXtTU0rz7ueVPPRweXdm8/5htWjLn8y39ia36vmRef+LKaahk+aliukIy\nLymFPmvq4ba0Omt1sEkhWRVq9bRagUeOkIhRjCE6S2XkxHNDiZRSCy0tRwgF5hBl\nyUAhh8lBY1rpCLXuWprXSJC0EUvqmDEECWaVUkQZOESFuBWIY8E5lRiqdlbr2Tnd\nJA3MYurh9gLv9zQeBGsD9lozjIyVCBNKBHFR+eiU1AjlL9ZDbYyqKZlyWsm7zWrG\nlfJ+oVjq6vkp9cVx5pgWkrKpZ1C96DoNUMrvoIrQCqNzMF3V9HaFzJRz2EZPgtAY\nOVdMcWjDOG6MdwzTgCM1rtHTmFg5v3MFo9wajKgjhiGlKTGCIgWUggwMMK10Ab5K\n5BiNYggjzA2zwltEqKWKU2IpvEWkUc5LTRzRnjusoDLlEk5M0hhQNFyovJIMyOS9\n10JXNQvF7lq1GkGQ5RCVLO0aJ66ElIXE9dRqUFVbRpRTnHDGjfWBO6+k9DYSqoMm\nPEYqrAvgisBReB2gM4Yitgxp6WXefhgD5xyDUg3w7lpYBeDESck1Dkz4wIikyNHA\nNTEqIFAfEJQNwDJYai3lIJHGTDMjAgbVuAjJOJdc3wafRvA7gysVvclM8el6bvL4\nBGSr0ywLteTffW2AFLO+bnhCCkk9M1neOzyph5WV6uhl2jEaVpgRprE3MjJDInIi\nCKRAcXATWUZBTYaMkIpRhYPG0jPuWdAeOa6pVgTFZBRtrN5A4TwrtCBq3ApJmdCR\nEi4wR1QyUEcya5WJBqZN8V8c8wruAyVI43yclYWyXAeELZfGnO/FKPqYF5Bcj40m\niDkjQWKYZux0IMgZY+BHMSCltA+aOcqshqwZsMQZHSMKjv5vLzrSrLPH/kcr6IWs\nEGdZIVwMTGurHXMCCec8g0EyVEvoVsoQBaVeUmrhEkRUwEV47JAW0ROfi3WeFSsa\nZYqZSct5092mr1w1vthp6p15OSEBnDMpILcRa0a1p9TgGCX3CgQM0qroYaZ9CJ4E\nxZVFFBnwBDIRcyvGSOCzWThYebDRPFeUcqcMYFkIPQ0Cw3bQBuYM1rAG8WE5YIED\nZtZZTiKsEUSGaU5jIF96DRJju++sPZ+LdyZgDEO+mGQQ6NDbndZCMc3f4BKWBXz+\nCYbyIkqNYVA1BMlzyZzDUQenpZGYRimJ4ZACFyz2AZKDJNHCIgs9OuXPogqlMtMB\nmEC4YrKeWkgGWl5qvn5iU3NL0+RJE/L/r6aWS1rP/Knd8s2Upi8H35x1eKSv/YnT\nWx+65/iNP7d9cvCDQ9sPvLr1SXeycsfJ9p2/D/UeODZp88iat/9aP3Tf5pnTyjuP\nHmrbsuyH8isvf3XdrHWffT+9c2nv1aXXFp6Y/Mz9fHDSyIF1i+9+7OYXtv+5YeqC\n03HbxrlrLp+/ZNeHRsyePfHd9sX9p9cODjUNL5v+Or5y8J1Fpb179l31x+OHW4YO\nXXHDjAml9oGTh2fs/viNPVML29ZuQHun3fvswW27pv248KOvN61d8OCyKY8M72o7\neOmeFwdaT8y5eMWnk/s7NsWl7920763vRk7d/kVpR8uCy9agia2tp46sXrdo/613\nPRCvbS+0td/ZPcyePz7/yIlvr/npaFxf/rV37sz3f9vxNw==\n=G+4d\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"c22f10ffefcb37f52d5a327779697b5000e4c034\"},\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"bf805b6051655371\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"track\":{\"basics\":{\"id_version\":13,\"last_id_change\":1394169810,\"username\":\"coaxmetal\"},\"id\":\"623ff558485dfcac5aadc413e1f3ac00\",\"key\":{\"key_fingerprint\":\"35ba103c2a408932a6308262e7e4f55b7ce7ce86\",\"kid\":\"01015a4b6db023b38532b3ce727a8cd792c29d5c18c003575c1473fe0fa56803c27e0a\"},\"remote_proofs\":[{\"ctime\":1394168677,\"curr\":\"9b428c852545abde5cd877dbf239e925ff36bce41061f6d9e3b340f1b4097d76\",\"etime\":1551848677,\"prev\":\"2c77591e46de42730c3e592a8e09814098be532eb3bb352a691494a6e12a456e\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"coaxmetal\"},\"proof_type\":2,\"state\":1},\"seqno\":2,\"sig_id\":\"5a3491da7f4a2f0c6e608fe032a0b4339440a6784381e917d45d4e9d0c5939820f\",\"sig_type\":2},{\"ctime\":1394169628,\"curr\":\"73469f325615037498b74bb8afa88885c4d8374e32091f10f8b1e959e01b57aa\",\"etime\":1551849628,\"prev\":\"adcd1a9204ca7a6ef171c9e20caaa0ca842a089de94c34b929da52aca9ff0ec3\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"coaxmetal\"},\"proof_type\":3,\"state\":1},\"seqno\":6,\"sig_id\":\"6cfe499b9c4c606ccd47e4a397c3477ef633d733be4a9f38a086d1c096fd2df10f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"6674a35476427f19439d33a1ff75d8a7ae7b8fd630deed2e858b030a0f8440fa\",\"seqno\":16,\"sig_id\":\"c5378dfd58335c8a5d8b6913e613089ae414c09929e7c161e14bcb52fc2a02310f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1411694742,\"expire_in\":157680000,\"prev\":\"a8d6f791591901bd574cc1f9ec97a713f772a59ffceb1de6ef07296b0be7cc8d\",\"seqno\":12,\"tag\":\"signature\"}\n",
            "sig_type": 3,
            "ctime": 1411694742,
            "etime": 1569374742,
            "rtime": null,
            "sig_status": 0,
            "prev": "a8d6f791591901bd574cc1f9ec97a713f772a59ffceb1de6ef07296b0be7cc8d",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 13,
            "payload_hash": "0521cd6cb3239aee69eeecb70866c8d88f0079883b2d3db473e16362e182cdf2",
            "sig_id": "96505ee653081f7c2e742283fbf559e13b3e330d943e964d3fc487c3438d07520f",
            "sig_id_short": "llBe5lMIH3wudCKD-_VZ4Ts-Mw2UPpZNP8SH",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowG1VWmIVWUYnilbHLQJNPVHWp5CEEb79mXsR1BIWhAypoGN12+dOc1yZ+496ow2\nIIJLGygotCgpiD9CKotissgxC7QMFcM2aJHSNJsMygUles+daXRKgqDun3u437nP\n+7zP87zvt270tVU11btXLFg0JzR0Vn90ylbN7Vc7lie26LuT+uWJa01De5Y/tZu2\nkNQnLaHbmnKYnhZvXxpsUpcsCaVyWmyHI021ENzY4KR2kmslNA+UGcW11ohboqNX\nxomkpy6HyUFj2t4USh2lNK+RIGkjltQxYwgSzCqliDLwEBX8XSCOBedUYqjaXCxn\nw9gkFcxC6uHXq7zfUjlAGGGvPXXGEketN8zmHIM3gWAVaSSaK8ulkTYIqp1HlBgh\nsUDUyaA94EZkAG5xBS5YGwBPM4yMlQgTSgRxUfnolNQI5S+WQ2lQOdNiWtP2vPms\nZFxL3j5wT105f0p9YUhIwuqSVlPOoJmCazagUFKPGRYEManlcMzW0FXOTFuxnONW\nSEUcKOVWcsO5jpwHpXxAijALNEmF1KD4uV7DDVDcY2qI8T7wgLTg1BNDiYTiFNoD\nC2NkhPlhggovpYmOYapYQNgRSq2MynEqhAzUO4+x0CJoaySj3gMfgjRxoKa0RgsQ\nFLiXQlsxC4WOUrEYQZEFEL0sbat0jihnSiDo3C0ulaAqd4HwAMWk88ozg4MOnpNg\nLJNEIIWkcojGqLiTXkmKsQxCBCYMp1hV8hAGwbnAhAyAd5TCEgCnGjCNCBbwRAyQ\nCOEDsI7WMu8hDBwr+NKeeweaIIqtNQREYIREHWQy1Euub6WfyiA1B9dS8CYzhcfK\nuctDE5UtTbMslJJ/MLaCUsi6O+CI1CVwkOXk4aQcOtuLST0kppw2DaRfSSZilNg6\nxKgB1xh05JDTjgSYLIEYqGS4kC4YRoWiIKEgFDLskDiplEExGYAbLNhTd6UZCsNI\naT5kBrY0aMawD1J5CJ0jOmhNggrOGa9o8JQx65EBFoIHrSAwMHggIg/SUjXMDK7o\nAPigGTFGgzgMsYLxtBqChA0hCEY10GgZxx5xpJlHKFrIPWdYWq0gDcxZwZT712Y0\nw2SCCWFpebgf5eByS/5qhriaGRhf6YYjkVjYOA4SjxysQodguLR0MCiYwn6BDcki\nIZKDuDBynKioJXgGliFn/2ZGY6VOITNpa867w3S3Fo0vNJty83/hRt+s/62Pnnz1\nVboY3IBXXB65fEMZI0zlxBQko6sjLYVCmr/BpVAIPpfDIaPnSjnrnJRISSkoozag\nYDDFHFYVnJPIdHAiEukY7AQKIywjzLbDmiSXO6XAzDTlPqdN7SZbXApJT80T1beO\nqKquqbr+umvyS7GqZuTNf96UMtxwccTRcSvWfrN7zOpzuL9h0vbrzzaU1087NnLu\n+Pt6+5a9OyG7aUz//BMzSd+MlhfjuY5H1516bsfm+Z+c3zp23enZv6/t227HnZm4\n//xdM4oXto3fMOpS7bjPjs3ou3f/5hMbvtwzobFr9YKtu6esPfTh9GmPT73w1Ovz\nXlk5deKuZ4qryk+uaVwyf0p89cAHta9l6cU1k+JPl+74dNPz22rJWxtvnHzgyPdn\nR/+oFu18+oVNt3V9XtV5fO+WMw98sXL93Q8dn3xy0S+H73z26843D07r3/PrnHf2\n/PzywYaNe82W2Qsbe6cfefi7o72Hd95Se2Dm/WMObRv13rwH3++/Z9+sR6rHzv7t\nq5U/vLRs1fjTb3/8bc3CN/bt6v0D\n=/VzW\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"939665abec79c7598695e34a8599905b29fd8ac6\"},\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"bf805b6051655371\",\"kid\":\"0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"track\":{\"basics\":{\"id_version\":24,\"last_id_change\":1416204797,\"username\":\"alexstamos\"},\"id\":\"f1e335b75a559f55e88de0824b410200\",\"key\":{\"key_fingerprint\":\"85d13a2adde5e09653d2a32741630125b2ff424d\",\"kid\":\"01016d77afc41384e01c233b7f8c53667e3dcd11696e9ba743dd88d2092cdf87ba960a\"},\"remote_proofs\":[{\"ctime\":1403548607,\"curr\":\"5ce25e4e07cd8d4a1e9ed52eab472608078c03ff85c7d873117e66e46a531871\",\"etime\":1561228607,\"prev\":\"39d8da6eb2ea6fe2956dedd8fbb4dd671518d679d5dc012031bba26d7422f9e7\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"alexstamos\"},\"proof_type\":2,\"state\":1},\"seqno\":4,\"sig_id\":\"8746ff71bc043a12549d8c0c9c2e4aa604472a567cea436838d462310ac0688a0f\",\"sig_type\":2},{\"ctime\":1408153795,\"curr\":\"1b3e9441de78d5a5c29e992e8eccad83ed344bd0a9d865e9816dda418d5e7b38\",\"etime\":1565833795,\"prev\":\"fffa05553818fb91161a2208f3e3fb451d05094d00fb1e35417b989ed4cb648c\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"hackernews\",\"username\":\"secalex\"},\"proof_type\":6,\"state\":1},\"seqno\":11,\"sig_id\":\"8c2f2b3cac2cd0c599c0e5e97cc231332a34a4f2275140965528f9749dc0c0cb0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"1b3e9441de78d5a5c29e992e8eccad83ed344bd0a9d865e9816dda418d5e7b38\",\"seqno\":11,\"sig_id\":\"8c2f2b3cac2cd0c599c0e5e97cc231332a34a4f2275140965528f9749dc0c0cb0f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1424898168,\"expire_in\":157680000,\"prev\":\"f7fd588cbcc7708776343be0ea13152ff7fd2f49ec6f27c4c0334dd7feb2c192\",\"seqno\":13,\"tag\":\"signature\"}\n",
            "sig_type": 3,
            "ctime": 1424898168,
            "etime": 1582578168,
            "rtime": null,
            "sig_status": 0,
            "prev": "f7fd588cbcc7708776343be0ea13152ff7fd2f49ec6f27c4c0334dd7feb2c192",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 14,
            "payload_hash": "c9c48bfe71bb8255b3b58a0fb093b9feadffa5679793ccdd1b87ac519a234c98",
            "sig_id": "0f2649140b094cd58f5e7fc9a01e7a417fbe0b41ea73cd82a33f25447801bd600f",
            "sig_id_short": "DyZJFAsJTNWPXn_JoB56QX--C0Hqc82Coz8l",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowGtVVuIVVUYnjEtnRKcKKiQlO1bTbnuF8uiRNOXLLPBmOy4rjPbM845nrNndJhG\nRUEJAufFKBGRLhCCUIQ6JGovXlBBuhAUEWk+mEpDpRBJ1L+O4+SUQYEHNmzWWudb\n3+X//z009bamlubDGzpWPB9eWN188oJtWvqjOjaQ2Yrvz+YMZK47Dz1Feusxq0I2\nJyuHfmvq4dG8MnNNsFlb1hdq9bzSA1uaaiG4scFJ7STXSmgeKDOKa60Rt0RHr4wT\n2WBbgkmgMe/pDLVqLU93ZEjaiCV1zBiCBLNKKaIMvEQFfxeIY8E5lRhu7arUi3Fs\nsgZmKfewepPz5cYGwgh77akzljhqvWE2cQzeBIJVpJForiyXRtogqHYeUWKExAJR\nJ4P2gBuRAbjeBlywNgCeZhgZKxEmlAjiovLRKakRSgfroTbqnCmb7rwniS9qxpWT\nfOCeu3p6y31pzEhM27JuUy9ATMl1GXAI1hgWBDFF9DhMXy96YywSaIMRUQKpEAXD\nhGlFCPI2cmyslpxG2WA06nwya7z70gkrHfc8ECsoF0YTERGLIlBnXdCUSuJdNOPc\nhMiEjtYJ4zQ3VPhoonCWEjAKspaGc81gH0M9MCYiEAskGK0SRwtlYRL3WlhVKUKp\nWqtUItjRAXVX5EkgpmAuI0SDJa63VkvhWuwhGx8kjw5Zp4nCnFmtIINEBQXtDPNO\nBTikNIHMrKUWGUiTeSaBfhgF5xyDRw3wai30JT8jppgxT6K2yiDAdSRahpRyWEjF\nJY7ewwVUSew0FRE7BoQEdpzFEGk2piX529DT6KKu4MolbwpTWllPEY+1U7EmL4pQ\ny/4t1QZEqeivwjppy+qFKRJz2KmH1T2Va4t557W691RjC9VMmIJqBbYeWQ5xBhWU\n5yZS7gMNXFHhCHAOTFmnOIFg4ajQDsXsh2twoxcOtv09Ccr1WBLQnIj6EMAIeIKS\nFiEhOcFRcYSg73xqMm4Yp5Q7LoJIjYQd8jQyKJJ/JNEAH03iFsT8/5LozIuuXvuf\ngqA3C4LeEASoD5hyE6jQMEh4xIhocEkZHGXg1BKYG9xhg6E3EZGeQEYYIcKs0TB0\nIIjxOSxvXFMqTN6dKFdNf3fF+FKXqXfdmiDGRHwy95bKGEzjriFidOrd8MFI5l2v\nLgY1CwOLMaiJtdW8Fkp5OsGlUAh+Y2WBQJXzjQlDtQkBiIFuGL5KCKe8UhEhqZUC\nbp56yyQoEFQAUUWcj+QvqRiuKkwnYILeHlP01kI22PJa8/SJTc0tTbdPmpA+hE0t\nU6Zd/zoePTe56dBdjDy5vNTsw+V7du15ecPDez98ZviBKyNDa1uOfVrefvTA8LG+\nb2duah+a/+XpnUemnKr8bPfvRN+3nj65a8ubZ8nHnTMufXXh7s9WLurcxi8d2HF2\n+6Zdcy9deWjhnTtP3Dtnwdahvt3P3v/FbzM33ldtP3L1l2XllnnZT3vWP7Vw2/ER\n/O78FQc3VJcv2X9odwe6+uCmBWc268dfKi1FZ9ZN3Xf27dl7j4989EHH1lfWbVk/\nccK0Na2VgZFtT1w8987+V636ZsrXG2d/Pnz5YGXwZO+J4WVLnn7k/PDrz3Us3jxp\n8ns7vjv/1uFFvH3W4cfa/5B7FkyY8cbeX6evGNAvnmq9+H7r7/O2zFpcvmPfkjNZ\n+58=\n=64ao\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"939665abec79c7598695e34a8599905b29fd8ac6\"},\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"bf805b6051655371\",\"kid\":\"0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"track\":{\"basics\":{\"id_version\":13,\"last_id_change\":1416204829,\"username\":\"dstufft\"},\"id\":\"28608ef6412498220dbf51ab9753f700\",\"key\":{\"key_fingerprint\":\"7c6b7c5d5e2b6356a926f04f6e3cbce93372dcfa\",\"kid\":\"010107b69fbc6ac95a36dfaf6cb320f08ac7a5594fbc1759446f412e2ea98f641b990a\"},\"remote_proofs\":[{\"ctime\":1394142293,\"curr\":\"bb1d639de75fc0bc928154b980ab69fb0e9ca4dc8e39d892f80bb3b0acd04d47\",\"etime\":1551822293,\"prev\":\"df13144d2f9b8a054bc2fb4088c1678571fdde9c3871c936f1c4bb161c54fef3\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"dstufft\"},\"proof_type\":2,\"state\":1},\"seqno\":2,\"sig_id\":\"d391beed2487e9857d0b55e2e8e8d5af35de3e5836c2161e48bc85233757d69c0f\",\"sig_type\":2},{\"ctime\":1394142359,\"curr\":\"82803deec93ec9e87b0067521f8500371d2c3b5a45335c56e616031c0d3f46cb\",\"etime\":1551822359,\"prev\":\"bb1d639de75fc0bc928154b980ab69fb0e9ca4dc8e39d892f80bb3b0acd04d47\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"dstufft\"},\"proof_type\":3,\"state\":1},\"seqno\":3,\"sig_id\":\"003e135ae369e635f102993e8a1f7e53b2e1d5c1a11ab027d22e810024ba9d030f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"82803deec93ec9e87b0067521f8500371d2c3b5a45335c56e616031c0d3f46cb\",\"seqno\":3,\"sig_id\":\"003e135ae369e635f102993e8a1f7e53b2e1d5c1a11ab027d22e810024ba9d030f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1424898244,\"expire_in\":157680000,\"prev\":\"0521cd6cb3239aee69eeecb70866c8d88f0079883b2d3db473e16362e182cdf2\",\"seqno\":14,\"tag\":\"signature\"}\n",
            "sig_type": 3,
            "ctime": 1424898244,
            "etime": 1582578244,
            "rtime": null,
            "sig_status": 0,
            "prev": "0521cd6cb3239aee69eeecb70866c8d88f0079883b2d3db473e16362e182cdf2",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 15,
            "payload_hash": "d17b636b8e47993aa99a850e3140bb3c80f11f80c6517d8db5e225c747489d62",
            "sig_id": "20447ba9569d3e9cfb833be429b18afbac5561cc3916b4d35c43df77b7b1126a0f",
            "sig_id_short": "IER7qVadPpz7gzvkKbGK-6xVYcw5FrTTXEPf",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowF9VFuIVVUYnvESOaiURldJ2YGITbbulykSmiJLLUIfBJMz6+psZ+ac8Zw92iBD\nYb4olUJREVGJBGZKl4cke9DKQTEfmiDTyWq0G5V4QUl0XvrXmXFwINoPey/22vv7\nv+/7v39tmzaxoalx//Mr254Oy9Y2fv2XbVh+Vh3fkNmK781aNmSuMw/lIq3Kpitk\nLVlH6LWmFhbklTnrg82as3WhWssrZdjSVAvBjQ1Oaie5VkLzQJlRXGuNuCU6emWc\nyPqaE0wCjXl5dah2V/NUI0PSRiypY8YQJJhVShFlYBEV/C4Qx4JzKjFUba/UinFs\nsjpmKffw9j++76hvIIyw1546Y4mj1htmE8fgTSBYRRqJ5spyaaQNgmrnESVGSCwQ\ndTJoD7gRGYDrqcMFawPgaYaRsRJhQokgLiofnZIaofRhLVRHnTMdpjMvJ/FF1biO\nJB+4566WVrkvjRmJcXPWaWoFiCm5dgMOwTuqFUMCYT4O0+drOmoJrqfTJOQRWpII\n66yKKDIO3BChmhlECMLOiTqtUfuTY+NboDRjPriAsXQaae85QYopwixG1joDHmAn\nmRxnKQ3M6qCER9wzI8Ai7R1FDoozhBgPkTokebockInCYmirEdRGsJ17VOdeDV2V\nIpS6q5VKBE9WQviKvGtEOyNYYAm+uJ5qNUkkAVhRj7nEWtvAsLYxKuipgwbg6DHm\n0VKNCbfK40SOR66Rgn6SaDjQD6PgnGOlR8C7q2Fd8iBg43VUQXCCLfzPISNGOuUp\nCfCAFPtILYbKCG6BC0wkjUxGFyNBCXxUS/K3rqc+Su3BdZS8KUxpTS31eWymivV5\nUYRq9r+treOUit5u2CTNWa0wRaIPO7WwtlzJWiis8tWl0WQy75gWSgN3KaLmKCpG\nCaMKHHOSGOMUVYFRGoPm1mCpCRHIHDigUMAOQ3ayEbzRin2r6oVKhck7E/Vu09tZ\nMb7UbmrtUJCBvSJ46zVx3hqkDRJaCkuIAkdCVDQqrOBJrBIGcgTtl9Jb6iEuHuls\nTEZK/5gOrwRTyDMhoBkMxpVAGQMwBmPovgPvg8Oew8Cl+BFOGadKYikYc0gnFX1p\n4OoaRufuuiMruXctZIwwpRURUD48251XQylPX4B7CsE1lg6nHVM2BpmiQTi31HJl\nULRIU6tjMD5CwITUUlPnvMdWSeM41oZQ5rS6TinMcmFWAyboLZuipxqyvqbNjbMm\nNTQ2NdwweUI6ihuaptx07XzuX3Bjw74tL6uPBnqmf3lLdefeO2ee2H5m0zsvFa/M\nOvLJ61/9cn7rradmDF+8dOHKxXO3HZ7/8NQfp33/4Xt237Ydrf70QPj50saDT36x\ns+u7E0OfHz/wwbsLt29+e+Vw23NzTx566I4jZ97sf6NtRe+itnL/g0/sPnV464Th\n4Y1X588cbPz7rt8Glz8+7+g3i3+aMVUe2nFwqOne/XNaj3169+UXHh3o/PbsDze3\nlzc/1v3xPasGiv5Th1oXnivP/fOPrsVLe47u3nTs18kbBrbNPnl6+arWt+bO27Jg\nyqsLf182OPDa1aGjK6Z8dmFvw65HLmxcOnfSLnJ54H5537Tpsncwdix5YKJds6gY\natly+/k9z7z4/p4ri5+aveSffwE=\n=SdW+\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"939665abec79c7598695e34a8599905b29fd8ac6\"},\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"bf805b6051655371\",\"kid\":\"0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"track\":{\"basics\":{\"id_version\":11,\"last_id_change\":1398406015,\"username\":\"dijkstracula\"},\"id\":\"e726bcb8f0f4541002394a02201cc600\",\"key\":{\"key_fingerprint\":\"8944dece117c909dd52084824b10bbca9df1c747\",\"kid\":\"01013e4b9e86d05d4a6f0a9dc30c54140045ef3c0755555c023f6b18a6a63bf2185d0a\"},\"remote_proofs\":[{\"ctime\":1394216171,\"curr\":\"e2e8483d157199be419bff8c3bc2cf1fd115fb39125b8d19e865f590832a2fa5\",\"etime\":1551896171,\"prev\":\"8e1ad9f8e6521bb8d595ea7c8d32e7c8fd8df3b11990119e561273f47fcff205\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"dijkstracula\"},\"proof_type\":2,\"state\":1},\"seqno\":3,\"sig_id\":\"eb4dc4968959576f950f8432438571c72aac838e433fe95ba1792260a80e1c1f0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"45f56edbd92cdba09a06976b228205ef83f818ef82b86aca96a677db3d6f0d09\",\"seqno\":11,\"sig_id\":\"d86480d466e654a4b2f56aef8a113d1ce56ec1d5fc886d02534538717644c09f0f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1424898261,\"expire_in\":157680000,\"prev\":\"c9c48bfe71bb8255b3b58a0fb093b9feadffa5679793ccdd1b87ac519a234c98\",\"seqno\":15,\"tag\":\"signature\"}\n",
            "sig_type": 3,
            "ctime": 1424898261,
            "etime": 1582578261,
            "rtime": null,
            "sig_status": 0,
            "prev": "c9c48bfe71bb8255b3b58a0fb093b9feadffa5679793ccdd1b87ac519a234c98",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 16,
            "payload_hash": "89a241dc68e2a9a74aeb144c704585282d7ff2f1cadfbe4838d269f6b738df34",
            "sig_id": "5e647529aa812916a15a2bbfd26d9e99af30db81a9884ef44bb8099369a4309c0f",
            "sig_id_short": "XmR1KaqBKRahWiu_0m2ema8w24GpiE70S7gJ",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowGtVVuIVVUYnslSnG7SBbUp0V0PpoOz7pcRDCwfskIjM8TstK4zu5k553jOPtpk\noyX1IBKK1VOhiC+RGSVeICmdCiaKgvHFrlSOmkrYg6bRxfr3mXFy0KCg87Q4a63v\n//7v+9a/N147pqGpcf8zSx9/MDy0vPGTk7Zh0Sk1uCqxJd+TtK1KXFcailm+Kpru\nkLQlnaHHmmqYlZamrgw2aUlWhEo1LRVhS1MtBDc2OKmd5FoJzQNlRnGtNeKW6OiV\ncSLpbclhctCYFttDpVxJ8xoJkjZiSR0zhiDBrFKKKAOLqOC6QBwLzqnEULWjVM1G\nsUnqmIXUw7+XOd9Z30AYYa89dcYSR603zOYcgzeBYBVpJJory6WRNgiqnUeUGCGx\nQNTJoD3gRmQArlaHC9YGwNMMI2MlwoQSQVxUPjolNUL5wWqoDCtnOk1XWsybzyrG\ndebtA/fUVfNV6gsjQvKWpMtUM+il4DoMCJS0Yao5KIq4GAVZKRV9Cvdy0DojDwZY\n5h0jXoYgAkPa26BtzsoNMRpWPhdrtPrUKxBEgSw6gJMER2+dZpogb6RWznFknaF8\nlJoxBmFBN4o1DVZqyRXFElkZLBYCe4RY5FxHzIQl3qKANGitmAZmPFgEagL3Sugu\nZaFQrpRKEeRYCrnL0u6RvpXGoImrVSpQVTAFVXzgVDNquDeI8mhMVCQ65gIKVjgS\nopXKBMytIQIzyq0WgIOdcUA/DINDOBgdAh9mkKtSZ1HPfkdwnQVvMlN4opobM/II\nspVploHs/2hGHaOQ9ZRhg7Qk1cxkeUHYqYblxRIsYZW2D8U1YMJo1FZT7CySwhsu\nrbGIaY+ZseApE+ChIkJKMNFRCKXhDsIQGQEZUEyG0Ibr9bZcop8kI/opHjiDi44h\nBUYpFzS24DdCQchwAgnrcdDwcKTHmAsUwDWEVYBnLLD2RF0qYI5eroQV/487/82K\n9jTrqNl/5wS9nBPkIieIdohoFaKB5x8jtQzirX1wggoVWPDRc0tpCAHLGBzoBtMO\n3o30CiaBkZc4saxeppCZtCvnXDY9XSXjCx2m2gHlmLKEI86IsdY5ywKLSEZMIhhu\nsXMmCqUsBMNKJwlTnMBsijCPlJWMydyJ4SboRU0YZVj0lMM9bCFW0junLafeURQU\nIUh561HEkVOCIXTeKo8Vg64wjWI3JtBFbz6l6j0MD6uL5nyu3YV4MWClFZFQPzxZ\nTiuhkOYnuBQKwW8kFh5LCxpaEFFqTY3RGvKEAsUMWUsdzFWMYbo6wTGI6S0PhHAn\nmQR8L8jfnWKYgJlpB0xouGiyWiUkvU3rGm+9sqGxqWHsVVfk36+GpvETLnzUBheM\n+9MmCz5Yv0Rs31t57ba1h358t3nDPbV5026eff+i9mObvn3qpVe+mZw++vyM6TuO\nzVy8rGnObLf56rvXbv/18MGJY97qHHh/9Z1q7FT18/ls44Q5Lw8+vW7f7ob5B/ua\n2e7bJ33xR+u+7Y2DC7/etebZZdcdvW/cuJ5JMzdM3hYWfzjYv5P/dO+88uEbbzjU\n9RV7AZ04M/2WuZvWvNmK7trzwMmz3285/duX036/ae+pcw8np2t95daxS/o/LU5p\n/ug98Wp/47nG1jeOnEUvfn70sx/6dh0fWHhs7s5Zq2z/tvVba82rx5z5+LmJc8+2\nHR/Ysmf8kZV3vLN38cCUGefb13w358COX5ofeX38jL7z86+/5rED+9/u3vwX\n=1fCS\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"939665abec79c7598695e34a8599905b29fd8ac6\"},\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"bf805b6051655371\",\"kid\":\"0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"track\":{\"basics\":{\"id_version\":5,\"last_id_change\":1395759056,\"username\":\"rondiver\"},\"id\":\"d665b4dc42d7ee6e409dbe9b2cf8c900\",\"key\":{\"key_fingerprint\":\"3d8da483bd9e39621fdbc94920da798cc50bca35\",\"kid\":\"0101ffe6b3f23193eb797583170b7eb1661d004f559f146b2db0e09ae28494095eb00a\"},\"remote_proofs\":[{\"ctime\":1395758915,\"curr\":\"648975de53943a5da035faaf82fc4ce0eb6c2efb78ae15ba261435b967581cac\",\"etime\":1553438915,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"rondiver\"},\"proof_type\":2,\"state\":1},\"seqno\":1,\"sig_id\":\"e1243f9b931cb076da57bab049d14abd6646e9b82677c90c332aa5c056f425390f\",\"sig_type\":2},{\"ctime\":1395758972,\"curr\":\"85e54f42c40870b8ce91b39600e67e06bd1e96557d11560edb0018ea85619d28\",\"etime\":1553438972,\"prev\":\"648975de53943a5da035faaf82fc4ce0eb6c2efb78ae15ba261435b967581cac\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"rondiver\"},\"proof_type\":3,\"state\":1},\"seqno\":2,\"sig_id\":\"29c0298efae63ff3b4fe69dec6368e4edfd5b33eee17fec1b3abe3d87d8be1a70f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"48b250542abbccb4e4f07f12f646b1ccaf688b9b9b7c724852b57f6718b74478\",\"seqno\":3,\"sig_id\":\"a8a4fd359b91bf9b7dcc9b53dc30e82208dbd0f1f5321b93db8d1843ee13f6120f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1424898273,\"expire_in\":157680000,\"prev\":\"d17b636b8e47993aa99a850e3140bb3c80f11f80c6517d8db5e225c747489d62\",\"seqno\":16,\"tag\":\"signature\"}\n",
            "sig_type": 3,
            "ctime": 1424898273,
            "etime": 1582578273,
            "rtime": null,
            "sig_status": 0,
            "prev": "d17b636b8e47993aa99a850e3140bb3c80f11f80c6517d8db5e225c747489d62",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 17,
            "payload_hash": "e463e786a6fe37c0ae5108279292ddb5327fe9e2a76e851a618e223e883d0d93",
            "sig_id": "99ef2973d3f58d8f99bcc761357f34c566296169404f41ff461699f3cd74f18d0f",
            "sig_id_short": "me8pc9P1jY-ZvMdhNX80xWYpYWlAT0H_RhaZ",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowGtVV2IVVUYnbGUaczwxYfAhzzli0y5/38MhiLC9KGUpESR6/6dOc2de6/3nhkd\ndKqpCPtlCpQIShwT8SEIszQMDa2MCTQ0KxJDy4ykYmiotMzpO9dxmEmDgu7Tvvvs\ns771rW/tdfqnXdPQ3Ljv0eUrF4f7VzV+fNY2LPlJ/bg2sWXfk8xbm7hiGkpZviqZ\nzpDMSzpCjzW1cFtavml1sElL0h2qtbRcgkeaaiG4scFJ7STXSmgeKDOKa60Rt0RH\nr4wTSW9LDpODxrTUFqqVaprXSJC0EUvqmDEECWaVUkQZWEQFrwvEseCcSgxV28u1\nbAKbpI5ZSD3sXuV8R/0Bwgh77akzljhqvWE25xi8CQSrSCPRXFkujbRBUO08osQI\niQWiTgbtATciA3BddbhgbQA8zTAyViJMKBHEReWjU1IjlB+sheqocqbDFNNS3nxW\nNa4jbx+4p66Wr1JfGBMS45akaGoZNFNw7QYUgj2GNNKcCj0Bs7OrmKVtxRz0UudM\nOMqF1QYpx7AJllulsURSKyLrjEaVz8WaqD6TVpIoQtROE66EcBxzLaApT6wH4bCV\nMjA5QU3hIyExIPhFR1gINFfcexEEjJor7hUXSEpmcYgeY0Mxx1qTiBBFQXhQE7hX\nQ2c5C4VKtVyOIMdy8F2W5g1iqhlhQgrekriuahWqCuuggMBRGMY0CO4jchF6NYyb\nwAVhXjBNmeTYczBfMNRjA4cIQ9SGAPTDKDgHKuQS+CiDXJU6i7r324PrKHiTmcJD\ntXwwY5cgW51mWagm/zSLOkQh66nAPmlJapnJ8nrwpBZWlcqwhFXaNupWLSIYkHFE\nYFIKGxYwbFgDWglLKTg5KAtXQ0UkFOYErGg4oeA4A3NmKCaX0Ebr9bb8XT5F1Zh8\nmHLGOEWKaCKNklJQg62B28CJRv575DFS1mpGnRIWSeekAltwD38VJ9FeoV8dvVIN\n3f/PcP7bJNrSrL3L/qtB0KsNgowbhAZVLWaYSUxZEFwZiyHKokORMO0tjjqClRV1\niBrjMex7B/s0amYwVlcMYkW9TCEzaTGnXDE9xbLxhXZTa8/LYSSVUxjCEjOnIHgM\n83BfDFMC7p0jGFMTiUcSWQuDpDESwVCAqJLcy7zpy25C47oQEphyCdcrGsJijBZ7\nxaiCUKWBEIOhIQgxSaivI0KqwMAookJqHNxOuMhJbx5S9SZGs2pczOfiXbYXmEtB\nsChQMayppNVQSPMTXAqVJ8KYLZQGKtg7oQIBz0oGwYQZcxIxiAh430ugErEDK9jA\nwFGeCB2FlbCKlI1rVQIz0waY0HDJZF3VkPQ2P9U489qGxuaGKZMn5Z+vhubrpl/+\npm1+qalh/8L+DSuX3nXxkerA9mfndO7qPrDlxo2zPvzk+oH03CuHa9+d2vHR8PGm\n6cWB1hmzT/z61vC2NV3fvDry3paTxzf+smzVlAWd9wwsefKFkW23Dp7aOXRs/5sr\nTh5qfXvqujMz37lw+1D57l0Ne97t29TH5t7bGt2+rdNOZ3P/mLz57GDp88c3ncIv\nH9n6Yuu54ae33vHw8/3nZvcvvPmxSZUTi35+f9bM+U3d+0/3Dd83tGfHwjliyZdN\nR2754tPPlu34/bW2qeuXH+3tazmDD8rzT6i2B18fGFm3+NsLa4YGZ6WLigcu7j58\nrG/pjG1v/LB+Q9z+3N7f9OAHD3w1svvrXQsOng8Hnxk0R09U5sy/Ye+dfw4NHfoL\n=Co9J\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"939665abec79c7598695e34a8599905b29fd8ac6\"},\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"bf805b6051655371\",\"kid\":\"0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"track\":{\"basics\":{\"id_version\":11,\"last_id_change\":1409095369,\"username\":\"multigl\"},\"id\":\"b46c356b9a08c41aeb5b891707982700\",\"key\":{\"key_fingerprint\":\"47b72f6ef9c925866c51596232d2bd01d1b77e47\",\"kid\":\"01016df22fe0000fc24ee35537dd6e68ac585d8560774b1efd11a3151992f0030e6d0a\"},\"remote_proofs\":[{\"ctime\":1394246765,\"curr\":\"6bc37d61f6a449262df0cf41aa45ae5624d64934751d54a8ea3d1adf02403bee\",\"etime\":1551926765,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"multigl\"},\"proof_type\":2,\"state\":1},\"seqno\":1,\"sig_id\":\"b96fdae450291781a4e1fdabaa316b335b6e8b7bf8f068152c7ea523b70a9a040f\",\"sig_type\":2},{\"ctime\":1394246838,\"curr\":\"1354453082927a87763a1bac3b5290d0d108bb943c86b07cc7872f5dc86852fb\",\"etime\":1551926838,\"prev\":\"6bc37d61f6a449262df0cf41aa45ae5624d64934751d54a8ea3d1adf02403bee\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"multigl\"},\"proof_type\":3,\"state\":1},\"seqno\":2,\"sig_id\":\"9a52b14147134e658ab1c79fc0f249db1f9f07783c03aad1c0fdc9db3f94a1180f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"91078c8179c14c858ba4d5d8a486596c2113af2d070bb3943ff2640ee2175d7b\",\"seqno\":10,\"sig_id\":\"67db157030fa24fffb1d84384aa3e22a19fcf0a723d3ff26095d613036791ec00f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1424898282,\"expire_in\":157680000,\"prev\":\"89a241dc68e2a9a74aeb144c704585282d7ff2f1cadfbe4838d269f6b738df34\",\"seqno\":17,\"tag\":\"signature\"}\n",
            "sig_type": 3,
            "ctime": 1424898282,
            "etime": 1582578282,
            "rtime": null,
            "sig_status": 0,
            "prev": "89a241dc68e2a9a74aeb144c704585282d7ff2f1cadfbe4838d269f6b738df34",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 18,
            "payload_hash": "63a81ade57cb9d522845f6b4ecbdf0a70ded314cf4868eaca28b5d1f88397262",
            "sig_id": "c593d6ee7b05a1290d16fa2b4a8a581bf4f2ae286829eefda632707c63db4b7d0f",
            "sig_id_short": "xZPW7nsFoSkNFvorSopYG_TyrihoKe79pjJw",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowGtVlloXFUYTmJTa8AqioJajVwwDxLTsy95UFGRui8VXEodz/KfzJjJzHTmJmmU\nQIWCLWgREaSIVmweVLT6pKiVPlhxBaWCuODSKFSJGoRalfrgfydjTBdQwYdhDufc\n+53/+75/uQ+dfEJXX/eeTevuugnWbuh+b8533fKT+eXezNfjVDZ8bxaqFajlxarm\nxiAbzkZhyrsWDFXq50+CzwazCWi2KvUaHllulZLOQ9A2aGmNshK4cEZaa4n0zKZo\nXFDZ9GABU4CmSm0Emo1mpbgjI9onqnkQzjGihDfGMONwkQy+roikSkquKd5arrfy\nI6LJ2pilSsTd4zw/2j4glNBoIw/Os8B9dMIXMUJ0wKhJPDErjZfaaQ+K2xAJZ05p\nqggPGmxE3EQcwo234cB7QDwrKHFeE8o4UywkE1Mw2hJSPNiCZkc5N+qqlVpBPm+6\nMFrQx9groVWsKrG0KCQlg1nVtXIkUwplhwrhnqCSc0aoPQJzrFbPC8R2OMrwGLww\nCVA17gse0VJGKY1Ei3Y4HdkLpY6SPnJjEgVURaAYFiQYRzXoCFIGQSRIJCWPlNLi\noxCIIsZzzRUwFI37IBPjzvEYvWROUC0VlSwwh3oG8BYRvbYR40MpMfYmjNVzKDWa\n9XpCLdZh0uWVsUXOVBPkHMabTbxVCgLMJalTAJeScsEGCSL6gDxFNNQngsFT72Oy\nLEmZZEpWkyCj8xgghg8dcKk5oQvgjSZMILhGETgS55ErjYQsF1wEAjFEBpQ6RZTC\nv0App8pQzRgiU5qYI1wlbbNFLoW+bT7tEipDGC1Fl7vS3a3C38VayicreQ7N7LiW\ntt8v5VMN3GSDWSt3eRE2nrRgQ62+sFkZWch4SSO1OnhUK4jIbXSEKJp0VCG6FG2C\nSLUF602y6I3Dn1BcM2UhgNaOpGwBrXPf9ODRLhjCFl1QoIB4ZWSQNkoU13MmeNI6\nOSEgRa9UMjolSRNarkliNLlEQlKGgDf8GBfa4B0X/geL/5sLI5W8PO7/2QR+PBP4\nEhN4EtjkmMdMoQ4Mw1IwkmlCsKepGEGhFAK7jmGUm+RIsN5xLXVkLnLm7T+bgJ1m\n0QSnuBKYrYivse5wwbnWihKDfddawwhIzSTFtaaMaGpsUZmEOSxEHt0xJrTBOyb8\nDw7/SxOKRt4RfXJycqgQfqgGedbWPq+HehVPynneaA0fbQglhBzPE7G0MIzXMWJg\nniQOhntrA3Ea7aGOsKK7R2zsIRirOOYXjgUSUVZGkAnOoXSMJ+vb15RyV6kW4Tfc\nVLXuYqnsWuXCFAuFEpRZxkHhaKBEahMsJ9hCSFDOy0CMZkKhc5ILHZig4KhzYB2O\no+zv6rZLWBhhkxUiBMaj1yYab3A6RtRd4XhNXnLkWaQZGmkEJh+J6FQwOEajZhqn\nVjZdzJ02h874WTK5C+06ibaVCiYMZo9FaWFjo9KEUqXWzhG0ti14J0UAGwhoo5xK\nwDWKChJzj2mL3IvWz5lOgHI4rQAHB9aEAYaqGJxSJFr+N1VqMDQ3gphIuOby8SZk\n031bu1ct6+ru61re21N8knT1nXTqX98pYvOKrh1nnD20+6zXdh8uz577vb/yy5vv\n+6HW89W+ZR9MzuzadPIN608Z+nTLrlWHvnvwp8uu3snXv/nI5HXzTy975qwHLn7p\nrQs///H1y4fv2b8xDvw+v/2UP27UGwfihm/ePu+Q71lzfz7bn9S1/ees3vbB3BMX\nX7Vl30z/Zf3XtK5f+fLcmjPnX9mzcvb5g5fMQONg6fS56sHe8+t3Dfz6xRUT+z57\nccUbVz/Xc+Cib/d/0v8aefzAVwPbNves2hN2jO+484T9vffdvn17vdq3bfaR5R/N\nPTr6xc4TV3/upyY25+tXvPrzrjsu6Pnxt5mn1Gk3HX6s94kDz5I/5vfyCwbeeeHj\nJ3/e23vru8uzF9ZeeN2l5ZkPv374tvd3bvoT\n=4ya7\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"939665abec79c7598695e34a8599905b29fd8ac6\"},\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"bf805b6051655371\",\"kid\":\"0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"track\":{\"basics\":{\"id_version\":10,\"last_id_change\":1415332019,\"username\":\"mnot\"},\"id\":\"683dcb48fe28a3bb57ad912111d07400\",\"key\":{\"key_fingerprint\":\"0d388f1ec3b4ae29e5e8a17e7de55c405e5fc851\",\"kid\":\"010194aeec0608b3736e2e633bc5f23aa3ddb52a41756152c2a9cdceb9de5b79dbb50a\"},\"remote_proofs\":[{\"ctime\":1415331709,\"curr\":\"540e2af57fceaff6ac9c5e4dbc1214d81bf029e1bbdf92f55f5ff970c5dab6e2\",\"etime\":1573011709,\"prev\":\"70d33e5e3d36737393434c0edcd2e11a606611ac113168172297011f2a036f79\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"mnot\"},\"proof_type\":2,\"state\":1},\"seqno\":2,\"sig_id\":\"51d197cb317c4d39da0061f7d6cdafd9fed179e9b8f93aaa3aa4637269ece77a0f\",\"sig_type\":2},{\"ctime\":1415331802,\"curr\":\"6e6e0b685c59d50c5b3243f77fa44efdb66f87ff51fcdc70f21faf0cf680eb83\",\"etime\":1573011802,\"prev\":\"540e2af57fceaff6ac9c5e4dbc1214d81bf029e1bbdf92f55f5ff970c5dab6e2\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"mnot\"},\"proof_type\":3,\"state\":1},\"seqno\":3,\"sig_id\":\"3f45b22b1a61ae823bc8527001736dde6c7043bd82138fa0c9ba3757d2ad32b90f\",\"sig_type\":2},{\"ctime\":1415331941,\"curr\":\"a6364e5e7007e635e73377610886999820e5725199971207189c5f202aa9c3da\",\"etime\":1573011941,\"prev\":\"6e6e0b685c59d50c5b3243f77fa44efdb66f87ff51fcdc70f21faf0cf680eb83\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"www.mnot.net\",\"protocol\":\"https:\"},\"proof_type\":1000,\"state\":1},\"seqno\":4,\"sig_id\":\"58b7dd0cfb0f3e83b99c0a7ae81a0232a6d7e9cc8963121b2c0d64e20f68c4af0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"a9e21fa12923e6d9910578c930d2e0c6ab5c087246e5e5347c241ea1aae9ada4\",\"seqno\":29,\"sig_id\":\"849f944cc23db78d8b84b8d70f64a8fb5358b0173730841ae0d7fac8a20d72780f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1424898290,\"expire_in\":157680000,\"prev\":\"e463e786a6fe37c0ae5108279292ddb5327fe9e2a76e851a618e223e883d0d93\",\"seqno\":18,\"tag\":\"signature\"}\n",
            "sig_type": 3,
            "ctime": 1424898290,
            "etime": 1582578290,
            "rtime": null,
            "sig_status": 0,
            "prev": "e463e786a6fe37c0ae5108279292ddb5327fe9e2a76e851a618e223e883d0d93",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 19,
            "payload_hash": "c17a8f34177819e167e8aa727d8e391dbaadb33d2c58d8bdfe3a244ba4340828",
            "sig_id": "da45313ac0c2eaa0bc3669e3829b2b24b6f446e415d5652a31cfb054be1030760f",
            "sig_id_short": "2kUxOsDC6qC8NmnjgpsrJLb0RuQV1WUqMc-w",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowGVVWuIVVUUnjEzm1BMMRRM8tCk1jTt98MKC1FDjMiRTJy87ued04z3Xs89o11s\nYEhUykIjCCWISBQCpShMMUvqRy8pE4oMo18WZUyvHyVI0Tp3xmGm7EfcP4u99/3W\nWt/3rXX2TLmqpa315OC6DQ+Grk2tpy7YltU/qUtbE1v1jWTR1sT1paGSF1HFbAzJ\noqQ3NKyph860etOWYJOOZHPI6mm1AleaaiG4scFJ7STXSmgeKDOKa60Rt0RHr4wT\nyUBHAVOAxrRSDlktS4scCZI2YkkdM4YgwaxSiigDQVTwd4E4FpxTiSFrT7Wej6sm\naWKWUg+nV3jf27xAGGGvPXXGEketN8wWNQZvAsEq0kg0V5ZLI20QVDuPKDFCYoGo\nk0F7wI3IAFx/Ey5YGwBPM4yMlQgTSgRxUfnolNQIFQ/rIRthzvSavrRSNJ9nxvUW\n7UPtqasXUepLo0Ri0ZH0mXoOzZRcjwGG4IwhRiWVEo3DtC5r1PICs1mQ8pIIQzh2\nQaOgVeQuascZpU4g1SxohPiCq/HkWx+k50hRaykP3kkjhPVaUqVcIKCdcVK5KMeR\naSK8c9LrSATTOFrLonMU8wjHwTPNACFyEa0GahmEOsiItCSURm0LMqH2LGys5qFU\ny6rVCGysA9vladEfBkcRJhjjHYnrzzLISrhhLiLiqbQMyGbIkSgVcoqyaJCkIK+V\nnjIbEHPeUWGEDz4KQbzEhEH5YQQcvKHJMPhIBQUrzSqa1u8JrrfkTW5Kj9YLXUZn\nIN+S5nnIkv+QoolQyhs1OCYdST03eZEObuphU6UKIURpedirWipWDIq0nipJhYxM\nYEwsKzpxDkQBq3KPHNJCRUyjQxTr4scodgoMmQyjjeQb6BjHHqhHMB1lj4ZgotdG\nK4lkCAhZDjIq4AzI0ZzzHxgmILgx2issDQo+aCywpioQcFAYRx9DYhi9loXNBQXg\nfMWs58SawLTDmnigt5CLwmKAiYMbAtLAgDPvqfAYFbaSmsMACZX8XyHKad7Tb8fr\n4FOYLh/6clP/pxj0imKIMWpEHQwJjmgXjRZOwHRbo4FsDYtI08gtjho7jSjMVwxK\nUcQJtVg465U3+l9qPNLMU8pN2lfUXTONvqrxpR5T74F0jPqIQF+kiHHMccOD8Z4J\nCfPmpYoBecQdEB88ETCZII7DXojokdTRFYvtchdyTBfKcfAMaI/gD7BJORUYdIF6\njdMGdJbGROWUtTgEHSjIFRWPRXv8qIZFDW0MFHuq2cTIuhqz6Qv2LnuMEaa0IhpY\nDI/V0iyU0uIFlwL2DUKj3hDUKGx84NJZDQ4hivEoLAvAHKwBiWBIKYbRhmlQwThD\nYBF7HIFi2BaCjGlVQ2WmDJjQcMXk/VlIBtqebJ0zsaW1rWXS1ROKL1hL27XTLn/W\njh2e3LL/xuWrfr/tpdLTc7+btmb52a5N761c9tT69beuvG/b0PMLyFDb7NVfPGG+\nH1xbP/nxhHlLLk7qap9A3743+/OX9z8YWnjhzVfcA8eX9t/V9+K+x+nnP849fEP5\noRk/751ZPt+z7+tzz7RPPtYol05suLvzuamnPppTujh/+tEds89UJi+duHPviTvn\n/frpNr17w5kdK47lk7oPLpt9/1+btw/OH1qwfdYfB3YNdk999/SZtYOd7edvP3kN\nWdl/yXXes3bV7qnfnrvluheW1GYsO3xozZJWObDimw9PDzXeOb7mq7d2Hliw7csj\nv8k7zu46+EnXyw8vvnlOubv7tZ3XV80e9Nn+6QtfbW88e2nmlMX5G0cOzZr/+sS/\nAQ==\n=8nRc\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"939665abec79c7598695e34a8599905b29fd8ac6\"},\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"bf805b6051655371\",\"kid\":\"0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"track\":{\"basics\":{\"id_version\":16,\"last_id_change\":1404373770,\"username\":\"bcrypt\"},\"id\":\"8d726a251ce90e98f5cf9c5433c60800\",\"key\":{\"key_fingerprint\":\"bde7d5083bb35edc7a66bd97388ce229fac78cf7\",\"kid\":\"0101afc7ac7d9f26491fbb4fcc315f7aced494d97f56fb929547f59e7f097233f9bf0a\"},\"remote_proofs\":[{\"ctime\":1396246445,\"curr\":\"25a4cf02d37b42cf40c2f780c834fa0739d3b7d34be04cdc36a6dedf662d7124\",\"etime\":1553926445,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"bcrypt\"},\"proof_type\":2,\"state\":1},\"seqno\":1,\"sig_id\":\"978486957bd387367f46112b4739dccde7b2c5d0c0968f13fc03191919431c880f\",\"sig_type\":2},{\"ctime\":1396388213,\"curr\":\"3eeafd9a98707ee00b501089d3d71955541288caa9d817a0ede9161938e2080e\",\"etime\":1554068213,\"prev\":\"b8df84bd52bae49c192d445cf02365ac3b52b234b4aa4dd36d10fac77950ab68\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"diracdeltas\"},\"proof_type\":3,\"state\":1},\"seqno\":16,\"sig_id\":\"f9ea2ec29cfa96c61d9ba9919928a93f5b1f91c90390efe8830523b16cbd8da90f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"43df0b2c082ac4c5a5eadd467229d78fe0d05c208ed2683b412c1d66fd079fc1\",\"seqno\":17,\"sig_id\":\"8c52c56380412206536184bfe8ac9a1957aaf8c8bb1ee9e32baf85f3f5b595b20f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1424898296,\"expire_in\":157680000,\"prev\":\"63a81ade57cb9d522845f6b4ecbdf0a70ded314cf4868eaca28b5d1f88397262\",\"seqno\":19,\"tag\":\"signature\"}\n",
            "sig_type": 3,
            "ctime": 1424898296,
            "etime": 1582578296,
            "rtime": null,
            "sig_status": 0,
            "prev": "63a81ade57cb9d522845f6b4ecbdf0a70ded314cf4868eaca28b5d1f88397262",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 20,
            "payload_hash": "11df754f5792a2e1fca7358791f7f3ad2c5ef2adb2696af84f3743d174c27b34",
            "sig_id": "1f2c30069e6632bf697a4729aa2bb0538e475444d9f6eec15a1a395e131468630f",
            "sig_id_short": "HywwBp5mMr9pekcpqiuwU45HVETZ9u7BWho5",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowGtVVuIVVUYnmk0aaKSSFHIrC10c9B1v2gPapFYEEhRkOTxX7c5x5k5+8w5e7TR\nkYqiK0RaEphFRIJdSAfDMIh6ySQxwgdNitCHKIUwkEilqH+fOdXYBXrobe+z9vnW\nd1v/eu6ynq7e7g8fXrVmZbx7uPvgKdd1z+l6vjFzeRjNFm3M/GAt1ovyqQ5DMVuU\nDcRRB624oJZfuz66rC9bF5utWl7HJcutUhJc9Np6La1RVkYuwEhrLZGO2RQMeJVt\n6ithStBUq/fHZqNZK/fIiHaJau4FACNKOGMMM4APyeDfFZFUSck1xV2reau4gE3W\nxqzUAv76D98PtBcIJTTYwD045rkLIFzJMQaIjJrEE7PSOKlBu6i49YFwBkpTRbjX\n0QbETQQQbqQNF52LiGcFJeA0oYwzxXwyIXmjLSHlh63Y7DgHAzBYq5fiiyb4gVI+\ncq/5VvlUC5U/jKS8LxuEVoFiKr4K6BD+JpjQUlBGL8Dsb/QXeT7YKlHblJAHt5Fp\nrUKIJOlkU1I0Bsq1iK5NqWN96daF9htMyyRliVBOCE+jt8lhJoilTSKSkECY8mKS\nnZTaCFKpZEySQhj8GDly6a2KQhJtg0jKKJOEY4rS4MCFaLlyRsugGCRAO5F7Mw7l\nRaw0mnme0I9VWLyiNjShm1AhCEHdfqTZLNNVXEYffFAE0QFJ0sQ1AaToqKBAdcSS\ncUut8FEzrFOgyN5FQCbWlJ2NHXCptWET4I1mXIfgwgQdubZOJKkVtUxbjjW02num\nRXKo0oZADSeMQsR+EWBGg2SOW9BQVq2jpfS3rad9jKrRD1QCFFBZ2yoz3thu8F9S\nXJA3+7OSSl7kPh/ElWpRNFqLSofaUJVitFHyJoT0Za0CivIFF1txuJ5nixg+1fon\nzkDCdkdjvAlCoDEsGQeAcYJmAbDM1DOaPLFKmgQYFcqjStGTREoB0XlHUjYBN7En\n29T311CENX+EwikPmgYRZCIaj7kH4ssm8WRNwMzRzmANni/FmaTce6osWOa0EcE5\n6/8WShu8E8r/kPh/DKUTSLG+VhSxmf3rWZucBvunKPikKMpTxJxLjoMTgIQZvhIu\nAWkKcM6LiO3CqeGk9wSCDkQrxQQEghFR+FsSD7S3qRRQGyxJN2B0MIdQqUKritvR\nANIJRoU3KQpnOYKBN+gOVZgcEwlth+B0YMClJQEL7jElyhxz6aM7oRwSHRlykgzs\nSOQpWEwvMOaBWU9dxEAwK4oDInnbHv2inNeS4fkOeMg5ScxLjamjjE3l8GuL6MzA\nSddHad6f/RJWa2HQxfhgo9aMlVq93QxlSLv7nWJ4qrEEXFCsDE4iqnQ0gA3XwUSs\nA44b1Ml5QAYmGBdS5IDVciC4IHi3/CmUIWgB/YiJeutQjDRjtqn3qe6rp3R193Zd\nPPWi8lrs6r1k+u935TyY9uvynvl0/OYzPSvzqcsunzPnlXnbzp768YMTb751pu/b\nHz/Y9cbYV9NWNF7YfeindcOfv/Ss1+nxmfN675g965NlXwxfx3fvOLJ34fbN5298\nZfXpl589vLAY7g93PTM2Zc+M1w79dP7k+K1vfXjT02MLFt+w49J7lzx2+6drNtP3\nZ10z6+PZM9/eNX3fxiXVnQde3LCUH3vv3LI5x969aUPr7srCzduvOnr6u/X7j9+y\ndH99xTdb5j5ktqz+8rP7eqeMnyK37Zk6Y96xrxu/nG8dH3l97pktO6G679ZH5m94\nvu/7J8fWHnnsiTd++Hl1z81dh/ceeO3Vs0cWb1v+6L3nrhi60r5z17bs6G0Pbds6\nuvb++viNc4vtB7eeOHr9bw==\n=JlfJ\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"939665abec79c7598695e34a8599905b29fd8ac6\"},\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"bf805b6051655371\",\"kid\":\"0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"track\":{\"basics\":{\"id_version\":13,\"last_id_change\":1424754121,\"username\":\"gpgtools\"},\"id\":\"ab739e2776dde0f7f9ff61ed1374eb00\",\"key\":{\"key_fingerprint\":\"85e38f69046b44c1ec9fb07b76d78f0500d026c4\",\"kid\":\"01119ea566f88f54487b741235c96e45079d4f6868f4b2611dbabde936b875d62afa0a\"},\"remote_proofs\":[{\"ctime\":1420144001,\"curr\":\"b635ecdcd60d4fac9f1f370a8f0b141a17eac639194ce723c4d1f05bea68698b\",\"etime\":1577824001,\"prev\":\"48d7e379b4f57619279388897cc274fbf549dd183021ae1650a287a52b39a7a1\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"gpgtools.org\",\"protocol\":\"https:\"},\"proof_type\":1000,\"state\":1},\"seqno\":2,\"sig_id\":\"fbdae88c8d44f0b2f8baafb0a72dac7e1c21fc09658fa450c2716610554aebcb0f\",\"sig_type\":2},{\"ctime\":1420144498,\"curr\":\"313d71d4d5f07fd8ca0c69043f98dbde4f5d98f29632513cc169a92b784dbb9c\",\"etime\":1577824498,\"prev\":\"b635ecdcd60d4fac9f1f370a8f0b141a17eac639194ce723c4d1f05bea68698b\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"gpgtools\"},\"proof_type\":2,\"state\":1},\"seqno\":3,\"sig_id\":\"9fb02bbfb3ab4a141202b035a4d14abbc4ecc28dfb5cc0ad7d076624ad01fc1a0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"1da5b4214c8fe4b93766ac89191614224f2b7adb7d2a3590d388c98d12b2bfa0\",\"seqno\":5,\"sig_id\":\"50ce3fd913cd22ca29c1bed4f449178ffc9bec7944bf8525d6d75d30f2c573250f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1424977483,\"expire_in\":157680000,\"prev\":\"c17a8f34177819e167e8aa727d8e391dbaadb33d2c58d8bdfe3a244ba4340828\",\"seqno\":20,\"tag\":\"signature\"}\n",
            "sig_type": 3,
            "ctime": 1424977483,
            "etime": 1582657483,
            "rtime": null,
            "sig_status": 0,
            "prev": "c17a8f34177819e167e8aa727d8e391dbaadb33d2c58d8bdfe3a244ba4340828",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 21,
            "payload_hash": "daad146752067612cd4a10091d6ffb1246c1029b92a3e71e7fc86468e150b48b",
            "sig_id": "4b5a10b7a32536fe976be26eb7b42641ec1eab36a06327f1dd6f836ab7d910480f",
            "sig_id_short": "S1oQt6MlNv6Xa-Jut7QmQeweqzagYyfx3W-D",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowGtVXuIFVUYv2uWtaSISlQYxVhgeJXznnOE1JTKPwoTIxKN63neHXf3znpnVt1s\nK4qSpDCSrIjFFIS2oDIhDfGFaGRJRRmiIEp/iNJiBD0os74Zt8VViYq4XDhzzszv\n+z2+c85Lo6+qtLbsfmrRkvl+wbKWT8+YykNnGytXRSZ1PdH0VZHtSHwjL0YN3emj\n6VG77zE689OS9LYV3kTVaLlvZknagCVFlRBcG29jZWOupFDcU6YlV0ohbogKTmor\not5qAVOAhqRR982uZlLUiFBsAo6pZVoTJJiRUhKpYRAkfC4Qx4JzGmOo2pZm+TA2\nUYlZSxzMXuH99nIBYYSdctRqQyw1TjNTcPROe4JloIEoLg2PdWy8oMo6RIkWMRaI\n2tgrB7gBaYDrLuG8MR7wFMNImxhhQokgNkgXrIwVQsWLmW8OOqfbdUfSKMTnTW3b\nC/nAPbFZMUpcbchITKpRh85yEFOzbRocgjkGBCUVnAzDzHVPR9rMViQhL4BLViGw\nOIAEi4wFqYhR71EIVCHMAi5ZDbpfGDY8ASaZI4RyCtZJxRFTghFuYyaE9VgYZrRE\njvJhjgoI3DIhNTLaKysMJohrbxhVhBFLKHzivPZCCEk4w545SwN3oAiYSgGOAvem\n70xzX+tqpmkASxZB7+VJZykdY8akFKIa2e5ms+y02FEHASnvsXKW2WA5zBjCsVSW\nEh40KdrOEaQF/CEXr50nYIYSTgB9PwjOhcLkAvggg8KVkkXZ/23ettecznVtaVaE\nM7QR8hVJnvvm8IzLJNJG5oFokvcUukqoWt7TBesQXpbrvKgLK5lf1khhCKOkfqFz\nOXMuYI84gSBAt8PEMR+4xggmuQUPpUGQLdPYceg0Cs1KkfESC3DJoBBdQBus11u9\n1EZF1JCNIfaSBgO2OUYNdYTh0x4anEnhVbCYOhEklbCjoQo8GSXiICy0e2CMCqsu\n87FE72r65f9PSP8wEZd26qRxuf3TbNoZlQnkqU074AXXyC7NBHYEvlIs5KJYMDWS\nY/jRWCtcJAQnEzJUOS5hA0AUHMExYhihzgdCjLBacCoUnIESIXdZLI+WZWq5TjoK\nAV3FNtau1qaztivn8m9j+c8i9oz+Wxm9xfFVihg8xS66AArzhpqNMBXHnMC28iu7\nkqavFQFhHhdACA01CZQNMWeBx4po4nGwOqYcTk8c4kC1I5YDEw1WCCV0kCzQmFGH\nY2ZJbCi7SCqkmOt60QVJvaHz7qaPelufb5k4stLSWrnm6hHFxVZpvW7sX7dd97lR\nv9967Mv9jS37+vuP1irzduZL6zMOjeyfemDniflrJ0y4e/2OQ5t/7ct7Zn/zxA9j\nts86f6p6bM9MO7mv88O39n9WyUaMe3vx60feO37ftlvmLJ23OXnhYP3avmz1qPVz\nBvbufpFmJ3bs2sfGvnlXtmEM/nj9xtkj7v1pxuPv1+pfdH9w/mRXPHXLJLNxYMkr\nfzw7YfzW754+vWzNwoF6Y6/97ebXvv4qXfPyw71P5u8efnXdt5P2b8OTf94gtj84\nrj7lo3Tx/Vtn3XQ92fRG0nJu7mN3njrKdh09oA+Pz3cer/x4/pG+G8imKevWzjx4\n/JOwfO7BZ9g9az6f//3qM0duFL/sWthvp0+84+xz1ZNt7wzcHj3wJw==\n=bbK7\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"939665abec79c7598695e34a8599905b29fd8ac6\"},\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"bf805b6051655371\",\"kid\":\"0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"track\":{\"basics\":{\"id_version\":12,\"last_id_change\":1421883652,\"username\":\"taylorswift\"},\"id\":\"ff47f7a7c0bcd3c043ee0ff39014f100\",\"key\":{\"key_fingerprint\":\"484d223535168950496425c7466ce16b4ba80d35\",\"kid\":\"01016ec7c468a0bae9c6b1205aeb439242c2380ddeae66682541e4dc3f5d21847f860a\"},\"remote_proofs\":[{\"ctime\":1411448866,\"curr\":\"937d3d39c9ee19dc4cfc5d3db25189c325fa28599d20a620a262eade204396d6\",\"etime\":1569128866,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"swiftonsecurity\"},\"proof_type\":2,\"state\":1},\"seqno\":1,\"sig_id\":\"54ddf1e05284d411d12d4ef5a10e055ce668b0a7c4a1d5f8d357a30be816114b0f\",\"sig_type\":2},{\"ctime\":1411448929,\"curr\":\"f7e83fbdb2d43b3d241ef0a486e9fc13d6f838abe57a13db967f6c410f4436c9\",\"etime\":1569128929,\"prev\":\"937d3d39c9ee19dc4cfc5d3db25189c325fa28599d20a620a262eade204396d6\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"swiftonsecurity.com\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":2,\"sig_id\":\"13b85151537a91df1e8820b39d581011d5505eeb423def22b6ca65369986800d0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"f7e83fbdb2d43b3d241ef0a486e9fc13d6f838abe57a13db967f6c410f4436c9\",\"seqno\":2,\"sig_id\":\"13b85151537a91df1e8820b39d581011d5505eeb423def22b6ca65369986800d0f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1424977526,\"expire_in\":157680000,\"prev\":\"11df754f5792a2e1fca7358791f7f3ad2c5ef2adb2696af84f3743d174c27b34\",\"seqno\":21,\"tag\":\"signature\"}\n",
            "sig_type": 3,
            "ctime": 1424977526,
            "etime": 1582657526,
            "rtime": null,
            "sig_status": 0,
            "prev": "11df754f5792a2e1fca7358791f7f3ad2c5ef2adb2696af84f3743d174c27b34",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 22,
            "payload_hash": "694ed7166cee72449964e97bcd4be58243877718425c4dc655d2d80832bd5cdf",
            "sig_id": "2a0da9730f049133ce728ba30de8c91b6658b7a375e82c4b3528d7ddb1a21f7a0f",
            "sig_id_short": "Kg2pcw8EkTPOcoujDejJG2ZYt6N16CxLNSjX",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase Go CLI 0.1.4 (darwin)\n\nxA0DAAoBv4BbYFFlU3EBy+F0AOIAAAAA6nsiYm9keSI6eyJkZXZpY2UiOnsiaWQi\nOiJhZDZiNzk3MmU4NGFjMTZkNTJiNTI4M2I0MmRmYjcxOCIsInR5cGUiOiJkZXNr\ndG9wIiwia2lkIjoiMDEyMGJhZDYzZDRhOWI3ODUyYjFjMDkyZGVhMDg4YmVmMzAw\nMGVmYTFlMTRjOGQ3MmJmMDc3MDI1ZjdhZDA1NTBiMDUwYSIsImRlc2NyaXB0aW9u\nIjoick1CUCIsInN0YXR1cyI6MX0sImtleSI6eyJlbGRlc3Rfa2lkIjoiMDEwMWQ5\nZDNjYWIyYzNiZGE0YjY5NWVlZGFlMjE4ZjNmMjk1OGI1N2E3YmU2MzljZDAzMmE2\nNzE2MDNjN2U5ZGY4MGYwYSIsImZpbmdlcnByaW50IjoiMDdiZjE3M2M0YWEyMDY0\nYjg4ODI4YTY0YmY4MDViNjA1MTY1NTM3MSIsImhvc3QiOiJrZXliYXNlLmlvIiwi\na2V5X2lkIjoiQkY4MDVCNjA1MTY1NTM3MSIsImtpZCI6IjAxMDFkOWQzY2FiMmMz\nYmRhNGI2OTVlZWRhZTIxOGYzZjI5NThiNTdhN2JlNjM5Y2QwMzJhNjcxNjAzYzdl\nOWRmODBmMGEiLCJ1aWQiOiJlYmJlMWQ5OTQxMGFiNzAxMjMyNjJjZjhkZmM4Nzkw\nMCIsInVzZXJuYW1lIjoiYWthbGluIn0sInNpYmtleSI6eyJraWQiOiIwMTIwYmFk\nNjNkNGE5Yjc4NTJiMWMwOTJkZWEwODhiZWYzMDAwZWZhMWUxNGM4ZDcyYmYwNzcw\nMjVmN2FkMDU1MGIwNTBhIiwicmV2ZXJzZV9zaWciOnsidHlwZSI6ImtiIiwic2ln\nIjoiZzZSaWIyUjVocWhrWlhSaFkyaGxaTU9wYUdGemFGOTBlWEJsQ3FOclpYbkVJ\nd0VndXRZOVNwdDRVckhBa3Q2Z2lMN3pBQTc2SGhUSTF5dndkd0pmZXRCVkN3VUtw\nM0JoZVd4dllXVEVxWHNpWTNScGJXVWlPakUwTWpjeE5EQTVNamNzSW1SbGJHVm5Z\nWFJsWkY5aWVTSTZJakF4TURGa09XUXpZMkZpTW1NelltUmhOR0kyT1RWbFpXUmha\nVEl4T0dZelpqSTVOVGhpTlRkaE4ySmxOak01WTJRd016SmhOamN4TmpBell6ZGxP\nV1JtT0RCbU1HRWlMQ0oxYVdRaU9pSmxZbUpsTVdRNU9UUXhNR0ZpTnpBeE1qTXlO\nakpqWmpoa1ptTTROemt3TUNJc0luVnpaWEp1WVcxbElqb2lZV3RoYkdsdUluMmpj\nMmxueEVBVmJ0OG95dWx1a3dqK0JuQi8ySVkxN1dlaDhXTVluU3l6WUXoK1VlNXBw\nQTAvWmVZbWpOWG5Bdkx2Y1BwdHQ1Zi8wMXlSVjB3YzJYMis3d1pBZG5GUUVxSE5w\nWjE5MGVYQmxJS04wWVdmTkFnS25kbVZ5YzJsdmJnRT0ifX0sInR5cGUiOiJzaWJr\nZXkiLCJ2ZXJzaW9uIjoxfSwiY2xpZW50Ijp7Im5hbWUiOiJrZXliYXNlLmlvIGdv\nIGNsaWVudCIsInZlcnNpb24iOiIwLjEuNCJ9LCJjdGltZSI6MTQyNzE0MDkyNywi\nZXhwaXJlX2luIjo5NDYwODAwMCwibWVya2xlX3Jvb3QiOnsiY3RpbWUiOjE0Mjcx\nNDA4ODcsImhhc+doIjoiOTljYzk5YjUzN2E3MjRlN2FlYjgzNjc0MjVlNjkyYmIw\nY2EwZjBjNmI1NjM0YmQzYjQ3NzUzNWE1MzEzMDZmZjBjYjNkOTQxZTA5MDliNjY2\nNzE3YWU0MDNiZjZkYjBlYTUwYTI4MTQ2NTY3MGE0ZTA0NTg3M2VjOTkxMeYyZWI4\nIiwic2Vxbm8iOjE4MjU5Mn0sInByZXYiOiJkYWFkMTQ2NzUyMDY3NjEyY2Q0YTEw\nMDkxZDZmZmIxMjQ25WMxMDI5YjkyYTNlNzFlN2ZjODY0NjhlMTUwYjQ4YiIs5CJz\nZXFubyI6MjIsInRhZyLjOiJzaWduYXTidXJlIuB9AMLAXAQAAQoAEAUCVRBxPwkQ\nv4BbYFFlU3EAAHezCAAxK+ZfGYAS4rm3oRWa7pQHW0eQvXBNN9TzlqPRkkcA1Qxc\n+pc7PiE7Zz2dFBhe3IFNsCTc0Ry9wVIFYTlKWLSou33SH9viXHUXwswj4Cno+Ung\nKb7Ssnq+qID8DGTTkhIPb6KlTpzj68xxaPuUljmemhCNflD52V1nui9o6imcx1PL\nh/V+9TAmWSjxYgZ3R+6CHBTFkI3N5+0sabGMaW1K/9s4zJQqTVhi4fV4IKUTWHIF\n11p7i/4Mll8ykaErL9oFSol+AE+F/7ofzIjaseqevdyRg1KxgiHKu8DVzH0OFE4u\nlV3oFKu987EdVQilS37pKNHJ61z+OfbOhoWDNUWR\n=GlPj\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"device\":{\"id\":\"ad6b7972e84ac16d52b5283b42dfb718\",\"type\":\"desktop\",\"kid\":\"0120bad63d4a9b7852b1c092dea088bef3000efa1e14c8d72bf077025f7ad0550b050a\",\"description\":\"rMBP\",\"status\":1},\"key\":{\"eldest_kid\":\"0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a\",\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"BF805B6051655371\",\"kid\":\"0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"sibkey\":{\"kid\":\"0120bad63d4a9b7852b1c092dea088bef3000efa1e14c8d72bf077025f7ad0550b050a\",\"reverse_sig\":{\"type\":\"kb\",\"sig\":\"g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgutY9Spt4UrHAkt6giL7zAA76HhTI1yvwdwJfetBVCwUKp3BheWxvYWTEqXsiY3RpbWUiOjE0MjcxNDA5MjcsImRlbGVnYXRlZF9ieSI6IjAxMDFkOWQzY2FiMmMzYmRhNGI2OTVlZWRhZTIxOGYzZjI5NThiNTdhN2JlNjM5Y2QwMzJhNjcxNjAzYzdlOWRmODBmMGEiLCJ1aWQiOiJlYmJlMWQ5OTQxMGFiNzAxMjMyNjJjZjhkZmM4NzkwMCIsInVzZXJuYW1lIjoiYWthbGluIn2jc2lnxEAVbt8oyulukwj+BnB/2IY17Weh8WMYnSyzYE+Ue5ppA0/ZeYmjNXnAvLvcPptt5f/01yRV0wc2X2+7wZAdnFQEqHNpZ190eXBlIKN0YWfNAgKndmVyc2lvbgE=\"}},\"type\":\"sibkey\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"0.1.4\"},\"ctime\":1427140927,\"expire_in\":94608000,\"merkle_root\":{\"ctime\":1427140887,\"hash\":\"99cc99b537a724e7aeb8367425e692bb0ca0f0c6b5634bd3b477535a531306ff0cb3d941e0909b666717ae403bf6db0ea50a281465670a4e045873ec99112eb8\",\"seqno\":182592},\"prev\":\"daad146752067612cd4a10091d6ffb1246c1029b92a3e71e7fc86468e150b48b\",\"seqno\":22,\"tag\":\"signature\"}",
            "sig_type": 1,
            "ctime": 1427140927,
            "etime": 1521748927,
            "rtime": null,
            "sig_status": 0,
            "prev": "daad146752067612cd4a10091d6ffb1246c1029b92a3e71e7fc86468e150b48b",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 23,
            "payload_hash": "27bc88059a768a82b1a21dcc1c46f7fc61c2d2b80c445eb2d18fed3a5bb42e49",
            "sig_id": "eb5c7e7d3cf8370bed8ab55c0d8833ce9d74fd2c614cf2cd2d4c30feca4518fa0f",
            "sig_id_short": "61x-fTz4NwvtirVcDYgzzp10_SxhTPLNLUww",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase Go CLI 0.1.4 (darwin)\n\nxA0DAAoBv4BbYFFlU3EBy+F0AOIAAAAA6nsiYm9keSI6eyJkZXZpY2UiOnsiaWQi\nOiJhZDZiNzk3MmU4NGFjMTZkNTJiNTI4M2I0MmRmYjcxOCIsInR5cGUiOiJkZXNr\ndG9wIiwia2lkIjoiMDEyMTE4NzgxZWRlYWFlOWM2MTNkNTMwMjZhNGVmOTIwYzZj\nODM4MmExYzRmZDQwYzM5OTJhOWM3OTcwYTcwMTU4MzEwYSIsImRlc2NyaXB0aW9u\nIjoick1CUCIsInN0YXR1cyI6MX0sImtleSI6eyJlbGRlc3Rfa2lkIjoiMDEwMWQ5\nZDNjYWIyYzNiZGE0YjY5NWVlZGFlMjE4ZjNmMjk1OGI1N2E3YmU2MzljZDAzMmE2\nNzE2MDNjN2U5ZGY4MGYwYSIsImZpbmdlcnByaW50IjoiMDdiZjE3M2M0YWEyMDY0\nYjg4ODI4YTY0YmY4MDViNjA1MTY1NTM3MSIsImhvc3QiOiJrZXliYXNlLmlvIiwi\na2V5X2lkIjoiQkY4MDVCNjA1MTY1NTM3MSIsImtpZCI6IjAxMDFkOWQzY2FiMmMz\nYmRhNGI2OTVlZWRhZTIxOGYzZjI5NThiNTdhN2JlNjM5Y2QwMzJhNjcxNjAzYzdl\nOWRmODBmMGEiLCJ1aWQiOiJlYmJlMWQ5OTQxMGFiNzAxMjMyNjJjZjhkZmM4Nzkw\nMCIsInVzZXJuYW1lIjoiYWthbGluIn0sInN1YmtleSI6eyJraWQiOiIwMTIxMTg3\nODFlZGVhYWU5YzYxM2Q1MzAyNmE0ZWY5MjBjNmM4MzgyYTFjNGZkNDBjMzk5MmE5\nYzc5NzBhNzAxNTgzMTBhIiwicGFyZW50X2tpZCI6IjAxMDFkOWQzY2FiMmMzYmRh\nNGI2OTVlZWRhZTIxOGYzZjI5NThiNTdhN2JlNjM5Y2QwMzJhNjcxNjAzYzdlOWRm\nODBmMGEifSwidHlwZSI6InN1YmtleSIsInZlcnNpb24iOjF9LCJjbGllbnQiOnsi\nbmFtZSI6ImtleWJhc2UuaW8gZ28gY2xpZW50IiwidmVyc2lvbiI6IjAuMS40In0s\nImN0aW1lIjoxNDI3MTQwOTI4LCJleHBpcmVfaW4iOjk0NjA4MDAwLCJtZXJrbGVf\ncm9vdCI6eyJjdGltZSI6MTQyNzE0MDg4NywiaGFzaCI6Ijk5Y2M5OWI1MzdhNzI0\nZTdhZWI4MzY3NDI1ZTY5MmJiMGNhMGYwYzZiNTYzNGJkM2I0Nzc1MzVhNTMxMzA2\nZmYwY2IzZDk0MWUwOTA5YjY2NjcxN2FlNDAzYmY2ZGIwZWE1MGEyODE0NjU2NzBh\nNGUwNDU4NzNlYzk5MTEyZWI4Iiwic2Vxbm8iOjE4MjU5Mn0sInByZXbmIjoiNjk0\nZWQ3MTY2Y2VlNzI0NDk5NjRlOTdiY2Q0YmU1ODI0Mzg3NzcxODQyNWM0ZGM2NTVk\nMmQ4MDgzMmJkNeVjZGYiLCJzZXFubyI6MjMsInRhZyI6InNpZ25hdHVyZeEifQDC\nwFwEAAEKABAFAlUQcUAJEL+AW2BRZVNxAACIaggAdOXNBG5V3ZyH/jqezi0gbOhp\nd88r4Fugil9n5jvY2Rp/yGWL+/Axh7NaReyw9Em3j1f9HBMwJLBO38j05KNDM1ES\nRsHBQz8YDU7QrmK2ZCbXJijfHbd6KgAJOUDMATVQxHcZX1ty3cgACTzbpGYsWNds\nC6lunXiGWlG7U5PFBupFWnI2BXXnA7PwL0ZuWPMfxFWT2xeIWYOgoe0XaRfP2hHH\nGtY5Bfv9rG4NLj9OuK0tK8gE35ikYw0aJY5BV1nbyiSNQNBrv7En+0luBGFcD3Ks\nRgGlu/LnJcttuk8aHLHZEr320LkA2tOlSnZMcWHnNoPMHfjIoOhM1AJk1QZLVw==\n=tm6V\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"device\":{\"id\":\"ad6b7972e84ac16d52b5283b42dfb718\",\"type\":\"desktop\",\"kid\":\"012118781edeaae9c613d53026a4ef920c6c8382a1c4fd40c3992a9c7970a70158310a\",\"description\":\"rMBP\",\"status\":1},\"key\":{\"eldest_kid\":\"0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a\",\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"BF805B6051655371\",\"kid\":\"0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"subkey\":{\"kid\":\"012118781edeaae9c613d53026a4ef920c6c8382a1c4fd40c3992a9c7970a70158310a\",\"parent_kid\":\"0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a\"},\"type\":\"subkey\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"0.1.4\"},\"ctime\":1427140928,\"expire_in\":94608000,\"merkle_root\":{\"ctime\":1427140887,\"hash\":\"99cc99b537a724e7aeb8367425e692bb0ca0f0c6b5634bd3b477535a531306ff0cb3d941e0909b666717ae403bf6db0ea50a281465670a4e045873ec99112eb8\",\"seqno\":182592},\"prev\":\"694ed7166cee72449964e97bcd4be58243877718425c4dc655d2d80832bd5cdf\",\"seqno\":23,\"tag\":\"signature\"}",
            "sig_type": 1,
            "ctime": 1427140928,
            "etime": 1521748928,
            "rtime": null,
            "sig_status": 0,
            "prev": "694ed7166cee72449964e97bcd4be58243877718425c4dc655d2d80832bd5cdf",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 24,
            "payload_hash": "12b594e44d9289349283f8b14a6f83ad144a17a3025a758e17d4eca70fbdc923",
            "sig_id": "0f175ef0d3b57a9991db5deb30f2432a85bc05922bbe727016f3fb660863a1890f",
            "sig_id_short": "Dxde8NO1epmR213rMPJDKoW8BZIrvnJwFvP7",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase Go CLI 0.1.4 (darwin)\n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\nkqI=\n=YWnj\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"device\":{\"id\":\"4c37e8fb6cb1c6ffcd7a7b8d9f533e18\",\"type\":\"web\",\"kid\":\"0120aa39aa0458019ae603412421f7136806ce8f1efc13ba1d0ccec5121df647df490a\",\"status\":1},\"key\":{\"eldest_kid\":\"0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a\",\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"BF805B6051655371\",\"kid\":\"0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"sibkey\":{\"kid\":\"0120aa39aa0458019ae603412421f7136806ce8f1efc13ba1d0ccec5121df647df490a\",\"reverse_sig\":{\"type\":\"kb\",\"sig\":\"g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgqjmqBFgBmuYDQSQh9xNoBs6PHvwTuh0MzsUSHfZH30kKp3BheWxvYWTEqXsiY3RpbWUiOjE0MjcxNDA5MjgsImRlbGVnYXRlZF9ieSI6IjAxMDFkOWQzY2FiMmMzYmRhNGI2OTVlZWRhZTIxOGYzZjI5NThiNTdhN2JlNjM5Y2QwMzJhNjcxNjAzYzdlOWRmODBmMGEiLCJ1aWQiOiJlYmJlMWQ5OTQxMGFiNzAxMjMyNjJjZjhkZmM4NzkwMCIsInVzZXJuYW1lIjoiYWthbGluIn2jc2lnxEAdq9XpWaaPSfEeWeZwgj5kETElNTEsVgxpwD/ou4cMuvMCGIYItoDLPdoMhcr47n5DJbi9qPEXVEKToC4drqYLqHNpZ190eXBlIKN0YWfNAgKndmVyc2lvbgE=\"}},\"type\":\"sibkey\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"0.1.4\"},\"ctime\":1427140928,\"expire_in\":94608000,\"merkle_root\":{\"ctime\":1427140887,\"hash\":\"99cc99b537a724e7aeb8367425e692bb0ca0f0c6b5634bd3b477535a531306ff0cb3d941e0909b666717ae403bf6db0ea50a281465670a4e045873ec99112eb8\",\"seqno\":182592},\"prev\":\"27bc88059a768a82b1a21dcc1c46f7fc61c2d2b80c445eb2d18fed3a5bb42e49\",\"seqno\":24,\"tag\":\"signature\"}",
            "sig_type": 1,
            "ctime": 1427140928,
            "etime": 1521748928,
            "rtime": null,
            "sig_status": 0,
            "prev": "27bc88059a768a82b1a21dcc1c46f7fc61c2d2b80c445eb2d18fed3a5bb42e49",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 25,
            "payload_hash": "ce162011e380c954de15f30db28f8b7b358866d2721143d9d0d4424166ce5ed8",
            "sig_id": "48267f0e3484b2f97859829503e20c2f598529b42c1d840a8fc1eceda71458400f",
            "sig_id_short": "SCZ_DjSEsvl4WYKVA-IML1mFKbQsHYQKj8Hs",
            "kid": "0120aa39aa0458019ae603412421f7136806ce8f1efc13ba1d0ccec5121df647df490a",
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgqjmqBFgBmuYDQSQh9xNoBs6PHvwTuh0MzsUSHfZH30kKp3BheWxvYWTFA/R7ImJvZHkiOnsiZGV2aWNlIjp7ImlkIjoiYjhjOGEzMGU0NTMwZjc2OGZhNjhjYzA5NWQ5ZGQ0MTgiLCJ0eXBlIjoid2ViIiwia2lkIjoiMDEyMTRkNzNiN2Q1YjRlMjAwNGRiMzk1ZmRmNjdiODgxOGYzZmY0MzI0ZGZmMWJjMzAxNjg5ZjgwNjE1N2I4ZWRhNWEwYSIsInN0YXR1cyI6MX0sImtleSI6eyJlbGRlc3Rfa2lkIjoiMDEwMWQ5ZDNjYWIyYzNiZGE0YjY5NWVlZGFlMjE4ZjNmMjk1OGI1N2E3YmU2MzljZDAzMmE2NzE2MDNjN2U5ZGY4MGYwYSIsImhvc3QiOiJrZXliYXNlLmlvIiwia2lkIjoiMDEyMGFhMzlhYTA0NTgwMTlhZTYwMzQxMjQyMWY3MTM2ODA2Y2U4ZjFlZmMxM2JhMWQwY2NlYzUxMjFkZjY0N2RmNDkwYSIsInVpZCI6ImViYmUxZDk5NDEwYWI3MDEyMzI2MmNmOGRmYzg3OTAwIiwidXNlcm5hbWUiOiJha2FsaW4ifSwic3Via2V5Ijp7ImtpZCI6IjAxMjE0ZDczYjdkNWI0ZTIwMDRkYjM5NWZkZjY3Yjg4MThmM2ZmNDMyNGRmZjFiYzMwMTY4OWY4MDYxNTdiOGVkYTVhMGEiLCJwYXJlbnRfa2lkIjoiMDEyMGFhMzlhYTA0NTgwMTlhZTYwMzQxMjQyMWY3MTM2ODA2Y2U4ZjFlZmMxM2JhMWQwY2NlYzUxMjFkZjY0N2RmNDkwYSJ9LCJ0eXBlIjoic3Via2V5IiwidmVyc2lvbiI6MX0sImNsaWVudCI6eyJuYW1lIjoia2V5YmFzZS5pbyBnbyBjbGllbnQiLCJ2ZXJzaW9uIjoiMC4xLjQifSwiY3RpbWUiOjE0MjcxNDA5MjgsImV4cGlyZV9pbiI6OTQ2MDgwMDAsIm1lcmtsZV9yb290Ijp7ImN0aW1lIjoxNDI3MTQwODg3LCJoYXNoIjoiOTljYzk5YjUzN2E3MjRlN2FlYjgzNjc0MjVlNjkyYmIwY2EwZjBjNmI1NjM0YmQzYjQ3NzUzNWE1MzEzMDZmZjBjYjNkOTQxZTA5MDliNjY2NzE3YWU0MDNiZjZkYjBlYTUwYTI4MTQ2NTY3MGE0ZTA0NTg3M2VjOTkxMTJlYjgiLCJzZXFubyI6MTgyNTkyfSwicHJldiI6IjEyYjU5NGU0NGQ5Mjg5MzQ5MjgzZjhiMTRhNmY4M2FkMTQ0YTE3YTMwMjVhNzU4ZTE3ZDRlY2E3MGZiZGM5MjMiLCJzZXFubyI6MjUsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RAlN+pEeGja9mb7MqufdZQrb5scAuaTn/ECcNviA29FBgyeYXEEm164XDDXogDBalKMxizqP14/0Zmw8WkhGi8DqhzaWdfdHlwZSCjdGFnzQICp3ZlcnNpb24B",
            "payload_json": "{\"body\":{\"device\":{\"id\":\"b8c8a30e4530f768fa68cc095d9dd418\",\"type\":\"web\",\"kid\":\"01214d73b7d5b4e2004db395fdf67b8818f3ff4324dff1bc301689f806157b8eda5a0a\",\"status\":1},\"key\":{\"eldest_kid\":\"0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a\",\"host\":\"keybase.io\",\"kid\":\"0120aa39aa0458019ae603412421f7136806ce8f1efc13ba1d0ccec5121df647df490a\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"subkey\":{\"kid\":\"01214d73b7d5b4e2004db395fdf67b8818f3ff4324dff1bc301689f806157b8eda5a0a\",\"parent_kid\":\"0120aa39aa0458019ae603412421f7136806ce8f1efc13ba1d0ccec5121df647df490a\"},\"type\":\"subkey\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"0.1.4\"},\"ctime\":1427140928,\"expire_in\":94608000,\"merkle_root\":{\"ctime\":1427140887,\"hash\":\"99cc99b537a724e7aeb8367425e692bb0ca0f0c6b5634bd3b477535a531306ff0cb3d941e0909b666717ae403bf6db0ea50a281465670a4e045873ec99112eb8\",\"seqno\":182592},\"prev\":\"12b594e44d9289349283f8b14a6f83ad144a17a3025a758e17d4eca70fbdc923\",\"seqno\":25,\"tag\":\"signature\"}",
            "sig_type": 1,
            "ctime": 1427140928,
            "etime": 1521748928,
            "rtime": null,
            "sig_status": 0,
            "prev": "12b594e44d9289349283f8b14a6f83ad144a17a3025a758e17d4eca70fbdc923",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": ""
        },
        {
            "seqno": 26,
            "payload_hash": "55b1d9f7b6ac608ab5a1b361bd941c83cbd600f62e687071669904ad3500e109",
            "sig_id": "543be0b42d22ba4ee8f2cea0b8d58b8c9d0294bacc9e9884a0a6315a9943279d0f",
            "sig_id_short": "VDvgtC0iuk7o8s6guNWLjJ0ClLrMnpiEoKYx",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowGbwMvMwMW4vyE6ITA1uJDx9IHtSQyhsneKq5WS8lMqlayqlbJTwVRaZl56alFB\nUWZeiZKVkoF5UpqhuXGySWKikYGZSZKFhYWRRSKQkWZhYJpkZmBqaGZqamxuqKSj\nlJFfDNIBNCYpsThVLzMfKAbkxGemAEWd3IDqnVDUl4IlUpOSUg1TLC1NDA0Sk8wN\nDI2MjcyMktMsUtKSLcwtDQxACotTi/ISc1OBqhOzE3My85RqdZRKKgtAAuWpSfFA\n6bLM5NT4pMy8FKDjgTrKUouKM/PzlKwMgSqTSzJBeg1NjCwMjAyMDQ11lFIrCjKL\nUuMzQSpMzc0sDIBAR6mgKLUMaGRyqqGZkYGhYaqxhUGypalJSqqhaZqxQUqSkUWa\nRZJ5krGphYWZWYqRuZGhoYlximWKQYqJiZGJoZlZcqppaooF0P7i1MK8fCUrIzOg\nOxPTgWYWZ6bnJZaUFqUq1XYyyrAwMHIxsLEygcKfgYtTABYrqys4GE5IZOgYFhr2\nNkxO2X/zysmXYUqLy9s3mT+8lSFzUOrvXi+DdfZ35kvZnO+dufrF+0LHWp9mzU3z\nZc7/2vCa+5TyvsfO9esVT4UdenVhj7yewf6NWZMk7m47rLXonoRK0Mq6CcuZV+Wz\n9f+QLQub5xFu4zF5ypZttYKBT42+nnc4uk5Nladr21ve2x0VK3daN1c/tlgmU8g/\nf+mc5vehLyUPVyQ8OONjonhfoaih5N7itnnWFUaGVpxPhL2i59S2pz3kVgiIS13h\neTXEZe7Fj93hkoFd050WvFQKe5I3ncX9UnO7kfSD3YItSbbWLHceq7I8nr+b/8X5\neb1H/3f/+RGWEVr7Ibj8kCSPySWjZAA=\n=foi1\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"BF805B6051655371\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1428020311,\"expire_in\":157680000,\"prev\":\"ce162011e380c954de15f30db28f8b7b358866d2721143d9d0d4424166ce5ed8\",\"seqno\":26,\"tag\":\"signature\"}",
            "sig_type": 1,
            "ctime": 1428020311,
            "etime": 1585700311,
            "rtime": null,
            "sig_status": 0,
            "prev": "ce162011e380c954de15f30db28f8b7b358866d2721143d9d0d4424166ce5ed8",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 27,
            "payload_hash": "60cc8edaa70c70a32591d35bb1e2f5f18a862b6b178b1aec27730ceeb166d9ea",
            "sig_id": "34a370138e5a3efe49946d049b52fd63fd825e702f76a0f560f550e7de43b0250f",
            "sig_id_short": "NKNwE45aPv5JlG0Em1L9Y_2CXnAvdqD1YPVQ",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n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\n6pMnPvHs3Qf2qDB28n0X/ws=\n=0H6x\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.7.7\"},\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"BF805B6051655371\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"merkle_root\":{\"ctime\":1428026414,\"hash\":\"396523208ff0f485144d71755b26b654f25b5487307efa845973863ffbee6fdc95262959d587f35e0a8457fc54a2b4bcaddf12500c8ed5c7b158ba153fa16204\",\"seqno\":190010},\"track\":{\"basics\":{\"id_version\":50,\"last_id_change\":1426792504,\"username\":\"gabrielh\"},\"ctime\":1428026581,\"id\":\"b7c2eaddcced7727bcb229751d91e800\",\"key\":{\"key_fingerprint\":\"3f6e504945ae44e6a53d49845431890fe1eebcde\",\"kid\":\"0101230aa5a191471449cc05ee95ce3637ebb16e409bb3b92ac0a8cc28eef92c0b500a\"},\"remote_proofs\":[{\"ctime\":1413405722,\"curr\":\"55d1e2013c91f4ef410538cb03cadfcd0b40e86dfa8a0594a42036eedfd61fff\",\"etime\":1571085722,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"gabrlh\"},\"proof_type\":2,\"state\":1},\"seqno\":19,\"sig_id\":\"dbbb07105a43c18ab1a9d01b564c591a6d4d57e35dd4ba3f1a0ed8e33f8517eb0f\",\"sig_type\":2},{\"ctime\":1413405744,\"curr\":\"5381de2b714095bcf65f41c1b235c11ab6fb87cea8b2c9a3bcb95095ab26e191\",\"etime\":1571085744,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"gabriel\"},\"proof_type\":3,\"state\":1},\"seqno\":20,\"sig_id\":\"916bccc5e1fd7e784340cca2795d1ae352922bd498ec196bbff6c00b7a304d2d0f\",\"sig_type\":2},{\"ctime\":1413406032,\"curr\":\"eba1b91ed0048983d7e32efeee7fecce16547998ff3e05b67045a751ef99be1c\",\"etime\":1571086032,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"coinbase\",\"username\":\"gabrielh\"},\"proof_type\":5,\"state\":1},\"seqno\":21,\"sig_id\":\"2c9e345b5acedbcfe3d5d9b2d6115b59e9d6ad31ad262bb07266ccc6db0b95940f\",\"sig_type\":2},{\"ctime\":1421805307,\"curr\":\"a0a74b05db02a6c66c9cf2ffd3754449ff680cd4fb61f80cba2038a23e7c0bfb\",\"etime\":1579485307,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"hackernews\",\"username\":\"gabrielhn\"},\"proof_type\":6,\"state\":1},\"seqno\":28,\"sig_id\":\"2787406a0b57a2eb7de822c5ecea6bdcd28a3ed96191a27d46459d747a6b292c0f\",\"sig_type\":2},{\"ctime\":1421558613,\"curr\":\"fdd474158530c9816ca9707c3cda809aaa24a0f69b5bcc06f5776a54cea54fff\",\"etime\":1579238613,\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"rel.me\",\"protocol\":\"https:\"},\"proof_type\":1000,\"state\":1},\"seqno\":25,\"sig_id\":\"fdee6f84bdd854d055fe51be696b175e5f2d2d6b143bc09a52f7bad57dfe05160f\",\"sig_type\":2},{\"ctime\":1421558741,\"curr\":\"158668859ea086ccd02ce16724f2b35a6a71834f98867429cd1210a075346443\",\"etime\":1579238741,\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"rel.me\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":26,\"sig_id\":\"9c8edb92c68188a43266c7037e1148c9320a969b248b97059aeca65ddfc39e100f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"d8f8fbfc4262d411a8fb26182238cbbdf4047ad3ffe4fdd3491a0a805ccb5420\",\"seqno\":30,\"sig_id\":\"b641e8b6ee2fbf7419d48e70aae0ae7072dccfeeb47656e49f9b359b0a4cb7900f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1428026581,\"expire_in\":157680000,\"prev\":\"55b1d9f7b6ac608ab5a1b361bd941c83cbd600f62e687071669904ad3500e109\",\"seqno\":27,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1428026581,
            "etime": 1585706581,
            "rtime": null,
            "sig_status": 0,
            "prev": "55b1d9f7b6ac608ab5a1b361bd941c83cbd600f62e687071669904ad3500e109",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 28,
            "payload_hash": "c40f853ecc8678a37fe6158b988a71929a088240ffef484d944c37917b8796c9",
            "sig_id": "239c0c304862c19c2caf2a07fb80f18c06459a96c89ae6aa9efb30498b86afb10f",
            "sig_id_short": "I5wMMEhiwZwsryoH-4DxjAZFmpbImuaqnvsw",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n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\nX7tNXi965chtZ/7wzQcPPfwf\n=TvdI\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.7.7\"},\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"BF805B6051655371\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"merkle_root\":{\"ctime\":1428026605,\"hash\":\"36441d7625f087c67c465c51f88d8ec455040e22de716d1170f6c671fb19bde2879a5e039a8f44109ac3220fa33639bfb12fe9597f5ba86af20f780027bb9b50\",\"seqno\":190012},\"track\":{\"basics\":{\"id_version\":171,\"last_id_change\":1427576513,\"username\":\"chris\"},\"ctime\":1428026633,\"id\":\"23260c2ce19420f97b58d7d95b68ca00\",\"key\":{\"key_fingerprint\":\"94aa3a5bdbd40ea549cabaf9fbc07d6a97016cb3\",\"kid\":\"0101d49265e97c636dbffa62070f29e07bdee44aca90b0eaa1645cc29fe5320f8a6f0a\"},\"remote_proofs\":[{\"ctime\":1392240892,\"curr\":\"4396ec2c10ba4c4e1d931af42ed4d65ecea9f57a858fd44e86f2950071ef871f\",\"etime\":1549920892,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"malgorithms\"},\"proof_type\":2,\"state\":1},\"seqno\":7,\"sig_id\":\"0d9f6b71c043f2ef80b6dea44338471edc3ef76e2cee236dd397412ad9631aa90f\",\"sig_type\":2},{\"ctime\":1391738324,\"curr\":\"461262ff766cf5da5eb40e5db29128a453ffb939840a23c0ffcb95073c888302\",\"etime\":1549418324,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"malgorithms\"},\"proof_type\":3,\"state\":1},\"seqno\":4,\"sig_id\":\"3e1e20a1a11f4f0ee4ea4470b694fc115f928ce0b1f9aa2951a65081da8340820f\",\"sig_type\":2},{\"ctime\":1414608934,\"curr\":\"2f7e338d5d1915975611c4d99f225986f7978c2b0a89f7efed973cabc92b6e0b\",\"etime\":1572288934,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"reddit\",\"username\":\"malgorithms\"},\"proof_type\":4,\"state\":1},\"seqno\":255,\"sig_id\":\"1a8fa75aa365dd96c95fd229722668dfb0026973d89e489b256d87ee06a4bd670f\",\"sig_type\":2},{\"ctime\":1407170371,\"curr\":\"9fe1cf15276a8c34ebe03eae24e90f4ea44e6c6afd69a96bd6a85bf1ec025780\",\"etime\":1564850371,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"coinbase\",\"username\":\"bitcoyne\"},\"proof_type\":5,\"state\":1},\"seqno\":202,\"sig_id\":\"c6d2c827fd7292ae36a359fef3ad04736ff373901b3db037af773ce13db7fab90f\",\"sig_type\":2},{\"ctime\":1407767238,\"curr\":\"510fb8ebf6ea77aa220f0a4cd7d0be212f9b294161f0cb2b43b6be024aa9ec67\",\"etime\":1565447238,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"hackernews\",\"username\":\"malgorithms\"},\"proof_type\":6,\"state\":1},\"seqno\":209,\"sig_id\":\"c60cae4ab2b165fd15960433b0af48c66d37f3940e56a8d3b4955991947cb3f00f\",\"sig_type\":2},{\"ctime\":1395868048,\"curr\":\"45c6893a87e18f27502266c08f0d0118f5ecd3fd0bd52432b18cddbd3f6520a8\",\"etime\":1553548048,\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"keybase.io\",\"protocol\":\"https:\"},\"proof_type\":1000,\"state\":1},\"seqno\":82,\"sig_id\":\"58ff6251889edccbb54495ea3f7d33f57c2d18e114dda887aeab6e06552aed780f\",\"sig_type\":2},{\"ctime\":1408376008,\"curr\":\"ba697b9daa2617a80f53d94a8204a6db5aaf2215faf1feb4a43d2a09a95c6a7c\",\"etime\":1566056008,\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"chriscoyne.com\",\"protocol\":\"https:\"},\"proof_type\":1000,\"state\":1},\"seqno\":222,\"sig_id\":\"7dbc723fc7c0abfe761af6729fb27603ef9be58718153329ed2d4ddd8d6ddb910f\",\"sig_type\":2},{\"ctime\":1408375965,\"curr\":\"57a78d3f45c749eab6a6b2796db103c6887ccbd7827a27e3f33cf3f0950cad8a\",\"etime\":1566055965,\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"chriscoyne.com\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":221,\"sig_id\":\"dbf530c689d2f3af60c5b11341d5eb2ad229995d6ef39dfd1661962d088a75ec0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"65a8942f6e853227bc825cc8a543dfd0289413f1143bef83d4269724030cb504\",\"seqno\":359,\"sig_id\":\"df3db2edb6a6d34872d88f03313af2ca23127bc4aa153caf5e3086c4035525580f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1428026633,\"expire_in\":157680000,\"prev\":\"60cc8edaa70c70a32591d35bb1e2f5f18a862b6b178b1aec27730ceeb166d9ea\",\"seqno\":28,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1428026633,
            "etime": 1585706633,
            "rtime": null,
            "sig_status": 0,
            "prev": "60cc8edaa70c70a32591d35bb1e2f5f18a862b6b178b1aec27730ceeb166d9ea",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 29,
            "payload_hash": "fb0f0572e190507cfd6185aacc0c0778cbc6dd86dfcdbb1238774e8a788a1109",
            "sig_id": "1d78177fbb9a05eae2e67dabdfe96bd5129f286f4b4c7ae8d7b27c3a7651f95e0f",
            "sig_id_short": "HXgXf7uaBeri5n2r3-lr1RKfKG9LTHro17J8",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n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\n7xHvLp/98fsfeai65unxH/7h/pt/vf3Oaz/43avq9O1nf/Tb7x/+Pw==\n=BRPx\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.7.7\"},\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"BF805B6051655371\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"merkle_root\":{\"ctime\":1428026634,\"hash\":\"6bd6f8e31b6f1aacd6dc13f1e61c72fbb9e3b528b09ffe5503b016b0120bdeda6f28499548b12d726121b1a6598fe8f240b7b96cc8b71921d717765188ddcc43\",\"seqno\":190013},\"track\":{\"basics\":{\"id_version\":22,\"last_id_change\":1426940266,\"username\":\"oconnor663\"},\"ctime\":1428026683,\"id\":\"41b1f75fb55046d370608425a3208100\",\"key\":{\"key_fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\"},\"remote_proofs\":[{\"ctime\":1419205029,\"curr\":\"b421b35b23fc5da8a69fab3978ff34689fcaaf0048fc8af822aa32b5121c8193\",\"etime\":1576885029,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"oconnor663\"},\"proof_type\":2,\"state\":1},\"seqno\":1,\"sig_id\":\"e43b8971ff4bdc1beb57fecc15c1b3455997d1c6ea5596e14f25bdf4d5ad47580f\",\"sig_type\":2},{\"ctime\":1419205131,\"curr\":\"382e1d68b4b81c6643483a74715ac0964e291fb2b52c4e9fa6c73d304ab9108f\",\"etime\":1576885131,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"oconnor663\"},\"proof_type\":3,\"state\":1},\"seqno\":2,\"sig_id\":\"9d5b60c0d486f34089a922bcc5e77fb643b9781cbcc894af7e123fad50645d6a0f\",\"sig_type\":2},{\"ctime\":1419205270,\"curr\":\"44a567581a31b596f31745c4633feced0806f427bac69e720aecfda02279b9bb\",\"etime\":1576885270,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"reddit\",\"username\":\"oconnor663\"},\"proof_type\":4,\"state\":1},\"seqno\":3,\"sig_id\":\"a5039b256eb381b395e2fe9018e931c7dbe052dc1f1cf213b9e8e930ef0feabf0f\",\"sig_type\":2},{\"ctime\":1419205360,\"curr\":\"00279b0704b01ac6171beef1a5718a8de52d5f0fd0ab63dc947dd6f111bb673f\",\"etime\":1576885360,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"coinbase\",\"username\":\"oconnor663\"},\"proof_type\":5,\"state\":1},\"seqno\":4,\"sig_id\":\"89e70d05df9c659d1ad8b0e2ead5790cc500d29345979af2e5807edefab5666f0f\",\"sig_type\":2},{\"ctime\":1419205501,\"curr\":\"d64c8d89811ef70c83ab58cf6350d4e847f1750cff3c468d7e8556c65073e09a\",\"etime\":1576885501,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"hackernews\",\"username\":\"oconnor663\"},\"proof_type\":6,\"state\":1},\"seqno\":5,\"sig_id\":\"8b4bc6a736a2a651763300384d54e72daa0ee3f9464ae98fd2df4f06f93d59f60f\",\"sig_type\":2},{\"ctime\":1422256755,\"curr\":\"c49c8c4bef4862d13da267f19416f3edc650159b18f25abfe8a0f3c5417159e2\",\"etime\":1579936755,\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"jacko.io\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":17,\"sig_id\":\"8afe41e09f1bb4b99765f8fc97ad11aad3d2de26c110d5b858628db443decf600f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"5cae5e8dc5d0cd6b233f12f6aeb03f45ca4c12b3fc2c337ed31112f0ebaba2a7\",\"seqno\":24,\"sig_id\":\"c930a19b6e372177c3ba9e840544cb27fad487f3b72cbafd5da863b97eed66cc0f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1428026683,\"expire_in\":157680000,\"prev\":\"c40f853ecc8678a37fe6158b988a71929a088240ffef484d944c37917b8796c9\",\"seqno\":29,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1428026683,
            "etime": 1585706683,
            "rtime": null,
            "sig_status": 0,
            "prev": "c40f853ecc8678a37fe6158b988a71929a088240ffef484d944c37917b8796c9",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 30,
            "payload_hash": "07dd7b38ba8f44b4dde84a959177cee50e5969eab92e4d962939389556fbf9cb",
            "sig_id": "bff44a113fa565c2b5cbfaa10467183938067cac147d16d176a6dc2318bd71af0f",
            "sig_id_short": "v_RKET-lZcK1y_qhBGcYOTgGfKwUfRbRdqbc",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowG1lluIVWUUx2eaTLTCLNLKJDsaXhqG737xoUzCC1kUkYhix+86s50z54z77NEm\nmdAUNMuwCOlCiA85hRSUD2JiWoSmUj1ICvVgFqagUGHmlWztM9M0Y0MU1Nt3zl57\nXX7rv9a3N9zYUDe8fveKBYseC48vqT94ytY9MfbsouUFW/GdhanLC66UhHKWn8qm\nLRSmFlpDpzXV0JRUxpUrPjQtro7rtWksLA1pNamUwQo1ySZZ6GrMzfOXY1JuDml7\nmuS+CkjaiCV1zBiCBLNKKaIMHKJC3ArEseCcSgwuWyrVbEDUQs1nMfHw7/QZYD99\ngH1H7UGwNmCvNcPIWIkwoUQQF5WPTkmNUG5YDWlvSabVlJJynm1bSFtLoZhWKrWS\nXZbkBpgRhYgQikI+ptoCr0SqLUEaEakddoEQKIER71CEkNgzr210XFGveYQ6pbeM\nBxQ0DcR5iYRWmHmisDLWcR6VVZAfjl5ZJjSNkmIduZbCIOsoZxAJY6c9Qh4QMeeg\ngGpYUq5AclAOZpB7lhrXmmcNoBJXzU+JL/a1hIjGQslUMyBXdC0G2lErTCqMuVID\neFSzNLE5joH1S4IaCz14IwlMa4GD98ZIiTTL6/OCchddDW9v3/NWDey9jMExxQAa\n9oIwJ6F3NlpjqNBMQC+dJcE7k/uoBUMAlCsrpdA6SksN0dwzFqzS1EJAT3jUShIA\nxxQNAmoKmBuwYIYj6IyGjpu8nDS0VbJQbIf2RuCzoF+DkRJMCo6h5o40hajKaB08\nZ8oox7z3lBOuXKBCSUqd8FYqQj0JEWvrEeXSGe+dNkp7ESPK0w+9zrkQlPQ4780g\np1LLoiayluBai95kpri4mneqb9Sak6ylwxYG703t/WLW2Q7/gi6rmcnyYF19woB+\nV5PmnkHBkTlMuHWgI+dUjEFJz0IginsfMMllLEjEAjkVkFCgR4atkJxLxAQTKBZ6\nvPXEI12N/YcDyBEg0MfOYk8jRwEj7TVR3gluiBecRsci89B+HKQ1MUrGhZdUGAcc\nEffUaQ0T0Z+dIoz1OP937FKQZpINZFcOvi1pDVfTY4PRw7QfPgcyxzYQjr10JIIo\nDNY8GBhsGGMuA1IcBxsjs0QT4y2MKMgb6OXCRO5v8cG7gsMU9OHTInoZoieE+Yg8\nkpTJaDhxyjjLYSMEDT9Y5IJpOFpBvYd1omAYrMc8DpQeYT3O/yG+fN/24qqUvKuk\noclV2go1YlnFVUrwoCXL2qdeTREjhAYDyftxhESMgmUhqMoFQQ3oQmKHjc1FhwSo\nBVBibQTWoAQHu5t7J4Ox3kcY+L9wXFgLU8xMUsqTbzedpYrxxd4l/R/o8H9SQ1e+\nr2s19K7tfncnHnzxhqfakzQUk9yCS6BV492ehqX5dWRRRFySALcBR9JFL7DixjgH\n4aRUzsLK8krA/eethetQScmCMlIpgwHPn5XeSsFrZprzVZM0l03WkcLAPFd/57V1\n9cPrrhtyTf55UDd82E1/fDMc3T/0yuy6K+s7Fjx9tKtjdPeZkSM3vfzhwXffGz5N\nvXPy2xOrTk64bdgDz084fnLJrPNPLpzx6z3z953dM3Tvq90NP53f9swHeMuaH6/f\nOfvsN29Pen3t4bmTT8+ZdOaR8tjxp8yhETPvnL6uYcT6ZfOm/Ny9+IW5k3+5Y+Ku\nCeeWn2vaWn6WPnhJfzz++8+P3bdjwt1f75zSfaxzaMdvZt7pdP/2G27pWv1aaf6b\nXz06Z+0r/uEDD+nLdvLFDYfemDhqzOFlDXTqpy9Ne2vF+ztG7b63oXHfsMsbx667\n/aOZX25ee/MQvX3Xyo0HJm3edKG05rsLF1fO/KR1xepLF5NwYuuRWXfNP/jZ6L1H\nzm/y9/9waRvasufFVV+MWX189+8=\n=shI5\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.7.7\"},\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"BF805B6051655371\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"merkle_root\":{\"ctime\":1428026683,\"hash\":\"f39b2090279c1ce2288842dc0f4101d4d9bfc583d95f3c47db45e0e93e2cd7069814d2818abc55f8b862c1fd8b4693f7319f5976a0bc35409011c9d00df804cc\",\"seqno\":190014},\"track\":{\"basics\":{\"id_version\":26,\"last_id_change\":1427811588,\"username\":\"strib\"},\"ctime\":1428026720,\"id\":\"ef2e49961eddaa77094b45ed635cfc00\",\"key\":{\"key_fingerprint\":\"7fec4841ce1d624c7165bfbaa36946605cb2edca\",\"kid\":\"010158b77699f7b3a295d44eb893b35cd25f9872469483e6427e15a44e4a502289790a\"},\"remote_proofs\":[{\"ctime\":1408647651,\"curr\":\"8a99ed548a8c4ddd35258ce368733c6db7823d2ef19bd0357caddc9a89d6ff0a\",\"etime\":1566327651,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"strib\"},\"proof_type\":3,\"state\":1},\"seqno\":6,\"sig_id\":\"1f4c125bc901cc8ffe87d4ee285dde12279c62f160c8e0688b841b67557046460f\",\"sig_type\":2},{\"ctime\":1424762357,\"curr\":\"b1d3f50e109d928dc65a2d653fc4f4dc481e7baff7456d736acef105d3c998ab\",\"etime\":1582442357,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"reddit\",\"username\":\"nedmike\"},\"proof_type\":4,\"state\":1},\"seqno\":13,\"sig_id\":\"cdda1be251d7c2f48aa195eac4714d57e0851ebff4b292adb04ce1d55798720c0f\",\"sig_type\":2},{\"ctime\":1408565716,\"curr\":\"96fd7efd224df0d07347fa52c8acb5a0be952c4f5649e95b63dd8148b35bd15f\",\"etime\":1566245716,\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"oldcore.com\",\"protocol\":\"http:\"},\"proof_type\":1000,\"state\":1},\"seqno\":5,\"sig_id\":\"716a8094638a2d63a5a271c1abe06806d92e0819a619998cab75dc7eabddfeb80f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"b1d3f50e109d928dc65a2d653fc4f4dc481e7baff7456d736acef105d3c998ab\",\"seqno\":13,\"sig_id\":\"cdda1be251d7c2f48aa195eac4714d57e0851ebff4b292adb04ce1d55798720c0f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1428026720,\"expire_in\":157680000,\"prev\":\"fb0f0572e190507cfd6185aacc0c0778cbc6dd86dfcdbb1238774e8a788a1109\",\"seqno\":30,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1428026720,
            "etime": 1585706720,
            "rtime": null,
            "sig_status": 0,
            "prev": "fb0f0572e190507cfd6185aacc0c0778cbc6dd86dfcdbb1238774e8a788a1109",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 31,
            "payload_hash": "85aed10aa65e11763a402b7cadcff428144eb4e9f23fb1b361cf68dcbbec687e",
            "sig_id": "6469223cfcfc6fc7b9a5ea156cab74a0d6450d26a051702300fcdb39efa234580f",
            "sig_id_short": "ZGkiPPz8b8e5peoVbKt0oNZFDSagUXAjAPzb",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowGNVWuIlGUU3i0t3DC8RGZ42aYosnV87xdhSRREbKNUDMlsem/f7ufOzszOfGsu\nOtYPIYsMKwoCyaIEwcoCya4ouZRdoMIi0WUtNMiQBE0KKut8307rrvkj5s/LfOc9\n53mec87zbpt4ZVNL8/5HVz+0LKzobf78Z9u0cub56oacLfv+3PwNOVeMQylJTyXT\nE3Lzc92h35payMfl1lLZh/zaWmsjpi23LlRrcbkEUSgv8zJXb0vD08tRXOoM1Uo1\nTnPlkLQRltQxYwgSzCqliDJwiBTiViCOBedUYkjZVa4lY6rmspyF2MO/CxdD/MIx\n8X3Zh2BtwF5rhpGxEmFCiSAuUj5ySmqE0sBaqDYomW5TjEsp2p5Q7S6GQrVczii7\nJE4DMCMKESEJBjym1gVXIkmVdcYqjXCktEfeS4YoQUpZrS3CQnrNufQCh0ADUwbp\ngLlTXijmZcThHxUAd+BOEs+cxMwoynlEMNHGKx4YhyuQyxOJpMfESK4iqhHhKOJA\noBZ6S2UAB3SwAOxJ1bjuFDUIFbtaeop9YaQlGMAXTS0B5Qquy0A7MmJYg/ZEjtFj\nbVzqh4BUkEsUoKotlwmsgWzwAXGNhNWMRF5SgSOGjJCaZwI3Op82a2z3rRPBYaOZ\nZ0JxhyiNQBlCvDWWEkaklwYpzdIcWTGEEbYSY6RMGqgwkZZpLJWioC6Rxii4qiS3\n2kjmbCQ9tMF5LEBirTQMgUQmpVMNPeUkFCrQ4AgUWj26xUgyhRUCzn3VKlR1MFaM\nWOR4RBGKmIYRDUI6wZBALFBBDUBVhBlGrQbclDkacY5psBpbB/BDIzmXCgTMkjcQ\npKpkKLIx6wquu+BNYgpra2mvRpatM066+mzuct0pximhLEch6a/AFwozkZgkLVgf\nGQ8Cp7hzeF0IcRqAE5BZCuedA8UItZ56RGFXuNPenTLSwBoyHJSIhCKYYsUVXNHW\nSRTlhtMNFyT1trECckEkGxHQSoGtx4rBj8KSCqYEdJZEwacjDvugGTVGGImtlSSK\nwBasMFwKETzcvURAOpz8fwqY+kZDsFJ4uJZ3tXypvy8ffF8uUy0pu3IRPnYlSWX+\npUpihNDlxKSjxKSY68gZMDEC22+N5tQZcDBjYDiR51Iip2G+kSPUc2IpmIMJLhhs\nkXHC/kfLNVmZQmLiYkqgYvqLZeMLDcOx1nrYCgqN4gF5p3QUtNRaGRDeYgM7Yoyn\nLAQdZIgEOAdUE0JzAnUdvmgYehQJryTiUsPagYWJ1MHUgdncaGwooxE4l+OBSxIE\nLDfTSOFgrIsQge3VHMbIEqBRT80nI9HwoFEPAb68h4T1lbgaCnEp661QKBO8Ug3r\nstcBzNSCvxoVMWaZ90Ex0Bc2XroQOIKlFhqQaBKY14JoqqkCtxWRjbSzF6lS8LzE\ndEJOYFwySV815OqPN88Y19Tc0nTV+CvSp66pZcKkf9+/61uu/ru2fMVd204eeVvs\nPvDD2R1zD+35uFm1zuo4tfFEx5mnfvwgnvvgc9f0/TV5fUdl+6tLDj19z9Iv+LZd\n0zYN3jo5vwtt2TjwS88r5IHe66ZPfKJ3+ye/rrtx2eC5oT3nL/w29dCfx77cPO3o\nqkJx8I1ziw+f+L19zrxNr/Nxj+3/due5C4Wm0zeXbmi/5cz9Xz97bCH6jmh7aubW\n4ys3zbr76KSvntlx76x5u09ufe/s0c376sXX0NCC968Y2LvqQ81O/HTm+JqD912L\njsw9vqQ+fnbnC+2ti5a+/OQMsvSjfUemsimHb993x8F3vpl52xTTPv3dN3cODb41\nsODOFw/nh+acHvj0j3rxkc8WdUz4vvOl57fctHwv/Qc=\n=r2Pi\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.7.7\"},\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"BF805B6051655371\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"merkle_root\":{\"ctime\":1428026721,\"hash\":\"f738bcab8901f89d0dd74032088b99b0167d9557d61ee3e48a09e15c8d684d7f548a8e516e5c72d4c714a8355f2129ad85e4509e99bd2707d12a758f390250f5\",\"seqno\":190016},\"track\":{\"basics\":{\"id_version\":11,\"last_id_change\":1421988227,\"username\":\"jinyang\"},\"ctime\":1428026738,\"id\":\"9403ede05906b942fd7361f40a679500\",\"key\":{\"key_fingerprint\":\"bc6ec1a94d4685c033f15c22dbab32427d7a0894\",\"kid\":\"0101b71108a15c28127b49178831ee27aa8bab875b9a74cbf7d88bcd168e598910a70a\"},\"remote_proofs\":[{\"ctime\":1420748180,\"curr\":\"c05142b0c5f300f4928ae67c640604e363a27d824a43b924234c3f5513eb91bc\",\"etime\":1578428180,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"jinyangli\"},\"proof_type\":3,\"state\":1},\"seqno\":2,\"sig_id\":\"22c94e32ab376cdcc75b23bd3d039945c9dca7abf841e86f6821318582ab9bc70f\",\"sig_type\":2},{\"ctime\":1420756274,\"curr\":\"b761bd184848360564867a02fed29ad9e1943aa6a71bb72ff07bb6a5766ed761\",\"etime\":1578436274,\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"news.cs.nyu.edu\",\"protocol\":\"http:\"},\"proof_type\":1000,\"state\":1},\"seqno\":3,\"sig_id\":\"3159fca3c42d0dba953ca4bfaa7b40d5770c96850c23d52b3320aecea1b0ac6b0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"bbbda0833995e0dc89fe97998a420b1a010aad34ee9e7ef6e99aec6695220ac1\",\"seqno\":9,\"sig_id\":\"d870579b329d06740385a91a343fee3c5e572e636149081eabcf021a995ab3b20f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1428026738,\"expire_in\":157680000,\"prev\":\"07dd7b38ba8f44b4dde84a959177cee50e5969eab92e4d962939389556fbf9cb\",\"seqno\":31,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1428026738,
            "etime": 1585706738,
            "rtime": null,
            "sig_status": 0,
            "prev": "07dd7b38ba8f44b4dde84a959177cee50e5969eab92e4d962939389556fbf9cb",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 32,
            "payload_hash": "edc19df0e628808413d1c956595f9f22d8e60413770efb89ba34125a766fcbed",
            "sig_id": "d86695d924d4724c90500939a54c6264954d4968cef0eedcfb70bc74ed8336970f",
            "sig_id_short": "2GaV2STUckyQUAk5pUxiZJVNSWjO8O7c-3C8",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n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\nOvbgvvL421948eJ33bVj9r8=\n=G6jD\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.7.7\"},\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"BF805B6051655371\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"merkle_root\":{\"ctime\":1428026739,\"hash\":\"4b43148abdda2ac81cedf97ad7525b9574786de1482e2caf9ce1fffe255294342757cae534fd6ec76b607801f8e2aa51860d966c739bd752156dde2477858deb\",\"seqno\":190018},\"track\":{\"basics\":{\"id_version\":64,\"last_id_change\":1427173495,\"username\":\"patrick\"},\"ctime\":1428026754,\"id\":\"1563ec26dc20fd162a4f783551141200\",\"key\":{\"key_fingerprint\":\"246201479f89c599c08f29b3fa706aef3d045f20\",\"kid\":\"01017c13682b0965e718feaead9f04e468086c3864ba8294072c97594a5957a99f6f0a\"},\"remote_proofs\":[{\"ctime\":1392412788,\"curr\":\"c20d2386bb1714908f0de84f76d9aabc7c2911a1bc6ee451f98bb8833a528fac\",\"etime\":1550092788,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"patrickxb\"},\"proof_type\":2,\"state\":1},\"seqno\":1,\"sig_id\":\"fff3da1be82b15fa7a5633f85d2ae19e4f128986e6e80e64a4859dedb56f47080f\",\"sig_type\":2},{\"ctime\":1392412864,\"curr\":\"e4e726145a1e148ed4bb2c8481d6a6f399d0e65d04af5e590be98c2ca2f23705\",\"etime\":1550092864,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"patrickxb\"},\"proof_type\":3,\"state\":1},\"seqno\":2,\"sig_id\":\"81a9003a098ec52cd98a9f4983ad7edea81e9beae23064ceb97518138e3cd0f50f\",\"sig_type\":2},{\"ctime\":1420864565,\"curr\":\"171a9d6e5b305d8eaf72f78e75a4bc814bbc2c5e5a429724640f0727212c0cca\",\"etime\":1578544565,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"hackernews\",\"username\":\"patrickxb\"},\"proof_type\":6,\"state\":1},\"seqno\":23,\"sig_id\":\"9127676da00562183585508aaf5b5fff4e936f208fc238e388d37a40eb99b3fa0f\",\"sig_type\":2},{\"ctime\":1395889480,\"curr\":\"7a1d453f123e38a125f0e872c6ec4ef0ba80119008996742abfad2675a16110e\",\"etime\":1553569480,\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"patrickcrosby.com\",\"protocol\":\"http:\"},\"proof_type\":1000,\"state\":1},\"seqno\":11,\"sig_id\":\"8e0043c22d467a8b966c528c17882118c23bec1b7803fd4b5430c500e04352e20f\",\"sig_type\":2},{\"ctime\":1420864719,\"curr\":\"db4612f774b4e4689b5b8d9ab9c99447d30bce0a969a131d65fa2810b352cc44\",\"etime\":1578544719,\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"stathat.com\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":24,\"sig_id\":\"250861cbd50c104723977c779c4a7ab7919d664dabf0228a61fc65d24dd4001d0f\",\"sig_type\":2},{\"ctime\":1420864917,\"curr\":\"6c1866e5442084ff8b6892ca7a8cf2c5555e5c6281553331593e3d8efb911ff7\",\"etime\":1578544917,\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"numerotron.com\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":25,\"sig_id\":\"3c666d4cf57d5f1b11a556def8d0f78e89ec92b994867a4ce0245f5bd187355b0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"0c2030fc412bd7e93cc7f4e4530ac8f3d5335bc5b15cece3b9eaafef423c09e4\",\"seqno\":33,\"sig_id\":\"9dd68ac987691394ab4c8aed90937024a7510433f1c1e5892ae5e560b9f219600f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1428026754,\"expire_in\":157680000,\"prev\":\"85aed10aa65e11763a402b7cadcff428144eb4e9f23fb1b361cf68dcbbec687e\",\"seqno\":32,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1428026754,
            "etime": 1585706754,
            "rtime": null,
            "sig_status": 0,
            "prev": "85aed10aa65e11763a402b7cadcff428144eb4e9f23fb1b361cf68dcbbec687e",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 33,
            "payload_hash": "e6bfc5876ff42db8a4f2c61c4ceaeed90ac38b35a2933f5bd26f504f62c2e50b",
            "sig_id": "c700a11886fc4c6db4627e1cd33dcb1579c2af39cb0b473be4cb726f4c3beade0f",
            "sig_id_short": "xwChGIb8TG20Yn4c0z3LFXnCrznLC0c75Mty",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n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\n23d428aDZ75y7vyx53+qbvj2h/989P2nv3Xzc/c++V8=\n=Tadj\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.7.7\"},\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"BF805B6051655371\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"merkle_root\":{\"ctime\":1428026755,\"hash\":\"5fd4bcc28542f0319ba74132de7b5f84e8da6570afba1cdf426b5a2bcdb7c201bf250f124a64fb4b9c3c07bebc0d731078ad79db38584f8a42db878efe418c66\",\"seqno\":190020},\"track\":{\"basics\":{\"id_version\":107,\"last_id_change\":1425174365,\"username\":\"max\"},\"ctime\":1428026790,\"id\":\"dbb165b7879fe7b1174df73bed0b9500\",\"key\":{\"key_fingerprint\":\"8efbe2e4dd56b35273634e8f6052b2ad31a6631c\",\"kid\":\"01013ef90b4c4e62121d12a51d18569b57996002c8bdccc9b2740935c9e4a07d20b40a\"},\"remote_proofs\":[{\"ctime\":1392252388,\"curr\":\"c610bab1245d05ac6fa95eda4d008d67c3732bd441dcc5c74f48fc8f8ba44c31\",\"etime\":1549932388,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"maxtaco\"},\"proof_type\":2,\"state\":1},\"seqno\":4,\"sig_id\":\"667062cc7300f11292079f7c4da0ed8e53e52ca484bb55ae693e7d0aa614322a0f\",\"sig_type\":2},{\"ctime\":1391720984,\"curr\":\"35782a3561e40a5e7217500bd67b5aadac5d7f6f9594dcd5d15cd06879da6e19\",\"etime\":1549400984,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"maxtaco\"},\"proof_type\":3,\"state\":1},\"seqno\":3,\"sig_id\":\"9f731fa7c75b64e1c9d70300b4383196a8fb432294a4308f8d7a379376a0b1900f\",\"sig_type\":2},{\"ctime\":1407166100,\"curr\":\"acc5ffcf17cd722b5844b24564e2f644d265e010047bfdd378919be8f0603ba3\",\"etime\":1564846100,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"reddit\",\"username\":\"maxtaco\"},\"proof_type\":4,\"state\":1},\"seqno\":55,\"sig_id\":\"095d87d153e6590c2d6c032752d3c43485a1f64e61b99d58f989f6da28b2a8320f\",\"sig_type\":2},{\"ctime\":1407167610,\"curr\":\"8d906bd4d0bb2d667f5c8191d6e7616f1de1e69e950f1f2e6ea181316d8d6dbd\",\"etime\":1564847610,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"coinbase\",\"username\":\"maxtaco\"},\"proof_type\":5,\"state\":1},\"seqno\":56,\"sig_id\":\"d0f9aaff73dc67f40fb6ebb6fdf473e11de43f11ae64a5fbffb00218914e19af0f\",\"sig_type\":2},{\"ctime\":1407186953,\"curr\":\"c9d4ca19adac4680957b2ce37d38280a1a32d2a7dfe606ae7236a9c982e2c75e\",\"etime\":1564866953,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"hackernews\",\"username\":\"maxtaco\"},\"proof_type\":6,\"state\":1},\"seqno\":57,\"sig_id\":\"a9fb80f43bed684a40e48de1e365e7ace765a3309819f9f79cdb27e3dd9802f50f\",\"sig_type\":2},{\"ctime\":1395867108,\"curr\":\"7d72a9f98ca4739e3939c89339d3f67966a7160da9df6215acb6a27c4a1ea292\",\"etime\":1553547108,\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"keybase.io\",\"protocol\":\"https:\"},\"proof_type\":1000,\"state\":1},\"seqno\":24,\"sig_id\":\"3517ddd1a5c9d708f21634d126814647129a64086a967daa5017e7966b89c9820f\",\"sig_type\":2},{\"ctime\":1395868002,\"curr\":\"ea61172238d18212d53a71316dc09e8e48f2e654490262872018d09854ab26e0\",\"etime\":1553548002,\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"oneshallpass.com\",\"protocol\":\"https:\"},\"proof_type\":1000,\"state\":1},\"seqno\":25,\"sig_id\":\"93734bd81d3b50d25401d7451a136e171757990d35110b585b729dbca2019d050f\",\"sig_type\":2},{\"ctime\":1397071049,\"curr\":\"eec8f9d3a754e793301b108ddb548dc46bbefaaaec189306b169a3ef71f22abd\",\"etime\":1554751049,\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"oneshallpass.com\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":34,\"sig_id\":\"0e577a1475085a07ad10663400de1cd7c321d2349cf2446de112e2f2f51a928b0f\",\"sig_type\":2},{\"ctime\":1397071368,\"curr\":\"ad3efd2c8ff323f96841b482bc3cfe92195bc72fe318ce0b95cf66ee3160aa55\",\"etime\":1554751368,\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"maxk.org\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":35,\"sig_id\":\"baacd979d2ebe325482d2921ea72df72dbf9a31aeaa69f2b14b1eccfed31c72e0f\",\"sig_type\":2},{\"ctime\":1401977613,\"curr\":\"467a473af5b7d548f1e1a1d199398f7fe3a9ea40e4c615d261e6b44a56b96642\",\"etime\":1559657613,\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"nutflex.com\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":49,\"sig_id\":\"4b4c67c83b4995aae5d4a5dde916fb35762551e67e35d4d4f1be99e2309238600f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"90ae8c481d2f81d42be0a27dce82574ad5a7dc3c573834e13df239b803cb1f8f\",\"seqno\":89,\"sig_id\":\"9818bb85927d78591eb30187f21f2cae2bb62d60b318a5e42bdd9e7d36753d4b0f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1428026790,\"expire_in\":157680000,\"prev\":\"edc19df0e628808413d1c956595f9f22d8e60413770efb89ba34125a766fcbed\",\"seqno\":33,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1428026790,
            "etime": 1585706790,
            "rtime": null,
            "sig_status": 0,
            "prev": "edc19df0e628808413d1c956595f9f22d8e60413770efb89ba34125a766fcbed",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 34,
            "payload_hash": "8c5e68c5a8f0f6e68d610fe1d8c086ad5870ebdecef02dd8f778a34ac300a18c",
            "sig_id": "39a4d445e1a5a2cc1aa1605cf3b3afadb8458f85fdb5c43e6bf85b65159d3e420f",
            "sig_id_short": "OaTUReGloswaoWBc87OvrbhFj4X9tcQ-a_hb",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nowGVlmuIVVUUx8fSTMWKqOhhkIeg1GnY78dYUUaGBElEWVhc92Ptuce5c+/MvWcq\nk4mJoCzoBX2qoCI/ZFYQaqFEfYig1KgoKIIiSntJllCBH9LWvjNO4yMq7od7uHed\ntdf6rf/6n/P4/JN75s54e3z1mhvhppEZu/b5npvTsmvWF74V1xX964vQKKFZ5aum\nG4KivxiEdd51oK9sXdRsRehb27loMqa3uBPanbLVxCjSp/t0Mdabw/PNqWwOQHu4\nXeZcBdE+Uc2DcI4RJbwxhhmHF8kQ6RWRVEnJNcWU9VanOurUopuzVkb8ddlyjF92\nVPxo9w/wHmi0VlDivCaUcaZYSCamYLQlJAd2oD3Zkht0jbKZqx2C9mADau1Wq9ty\nqMocQAVnWgnLJNbjOvV8i6dGeAjSMxojc0YbA4EoL13A46RJxERHmGHJKOpAERuF\nU1YQz7yJEoJKzEYSuVfcKuuidVonrlMi0qXAUwRnRQhSS+ITUw5b4tzR5Dhn3mID\nHRhptop+xgkXCmuv2i4M5qoRVBk6+aqMtamRMNVbNFynQnK1UHc4jonGBKFC0qN4\ndOrQiK1mVc9IjmEgbG8xgTh6ZgxCToppy1WiASkHJrS12Bgppmafx3Xs/EMkWqmQ\nEIz0EG2MxBIWKNXWCaGTAO2Fyjm6hxFKKDDQjCUVQmSGBAEShLCZhreGc8mjAVBW\nB2a1t5RRhXhFMkkwKqVUxOV22jDUqqA2jCNOyGj1tCFTQiSx1GDPo+02niq9FR5b\nkj6RwBVXTgcUAXWUBLBaOGARRyONlzjSEIFxDgGAE4/TM7l8mEwuldZ8IvlkBZlK\nt4qu0OoQBmvRVa62tpOnNbVu1V1lVUG7+OcBdZPUqnXD+A9DWVSuyieOTSkEh9sp\nByY2JsrI8aOT9By4tEhJSG4SWGkpESxYISFSljiY6KMCXFRjk8Zw5AOapGIi2+R5\nY73HATRqCqDCVUtCU1RO0DwlSjRDWVPUuEwpWQKggzFeW8qT8YFFq3VkTHOXcGH0\n8QBz8v8HcKCs6qP+P/LjJ+LHpvEL2jjBFcEvicVIbURMLgGzgjkUpxGEEM8tEFS0\nTkpLxoNFY1DapRjhX/kxS+UUP84VIwR1zBlHp4zJSEdtEMHLfIJTMSUFREj0CGUc\nuicqLEpcS4t3AD2W30Ty/8gvO+8REbbaod4oB+pV3xS6vtAaKrr0qlZoNTCqXlXD\n/ccSxZ7IiaCKaVC59SYbINGSUhEYAyo8bnNAKapAGPaPtfPIosKOo+CWa3TzrC4V\nrCbHQb2je0ytcmUjdzLs1jVaLtYmvVtymjeWhK5fG2dTTJ7TkF3HGSmBWgbgOLoT\nR2MxMqCpWxa4CVRGPc17zfTNwr0XVgaUNzoR904lqSQo6XTkgUgRveUJaMImEnji\nDJdME3w4WUONzsoYyy7ebWHSzKc9UemJrRjuHi7bUCtzhNTKkC7u4TbcmS1a+RSk\n0SqhB0ZkLBILiqJ+wAGaLnHYk+fSMbRRdITIVFoiiUBHDwxQU393ynFelRvIu1MO\nNF012oZi7KEZC2b2zJjbc8qsk/JLQ8/cOWcceZPY/tXsw0Prly7avnn/wtkbf77q\no9m3v3Do5X27539wfeO7A3H5lW/csGbesws//+yXH8/dueDgnBfHt7+5663X7/vC\nnvbc2fqpnX9euunTpQuuu3/N4idWyM9h26+vPHDo8o1XHh5v7dzyy4pf9XOH4rNs\nf39n7sxNZz5y2oatF57yx721paeuPXD7k9+NhHkXvjTLf7t7y8pFfbeN/LRx6VlP\nb7h6xz0raxd/fM6jjfXfvvbe/sHGT7s2jo9suOzrLTf1XfH71oWj2x7b+urqx7/f\nLNzbe07/ePnwZwufP+mZd295+Lxbr7YPbhga/OZa/1tx/qp3Plx1/yWLluy9p/cW\n+fSe0R1kd+2HvYu//GTW1hvmzb/gwI4VZ77/wPjFdx0c2PXeXw==\n=NiIu\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.7.7\"},\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"BF805B6051655371\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"merkle_root\":{\"ctime\":1432764925,\"hash\":\"ab184bec5b21dd2a8788ec06b5ac01258f08da0282f861ae609d4a6940b2b8d5ec6f29d0d3b63969ad9a77f37ff05afc3fdea94cc5750bf26aebb33a1fa332b9\",\"seqno\":230346},\"track\":{\"basics\":{\"id_version\":26,\"last_id_change\":1432401451,\"username\":\"sheldonth\"},\"ctime\":1432764949,\"id\":\"edb2881d9f627936f1c232c247999d00\",\"key\":{\"key_fingerprint\":\"07cd0766cfe605bed9dd0902c1179a447f4e7b46\",\"kid\":\"0101e2e722f6ccd280c4e5e449332bb983353d8ee697c297b91216d0d4f8f42155560a\"},\"remote_proofs\":[{\"ctime\":1410050918,\"curr\":\"5b94b2c25bf0c3636a7c9251a10ce974ae2df2658b58d5cde233ecee30bdea86\",\"etime\":1567730918,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"sheldonth\"},\"proof_type\":2,\"state\":1},\"seqno\":1,\"sig_id\":\"d5d3d3d7f5b3e3592164538fe9591042c945ed12f3e8dbd6ebf189f75b3410e70f\",\"sig_type\":2},{\"ctime\":1410050986,\"curr\":\"6900f471276c73ff1072ea91f055fff90ee7c88b7913f8bc2d977d2273af09d7\",\"etime\":1567730986,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"sheldonth\"},\"proof_type\":3,\"state\":1},\"seqno\":2,\"sig_id\":\"c78a436078a59865784dfafe2942a2f684000b39e0dd07f67523c9c0667afdde0f\",\"sig_type\":2},{\"ctime\":1410052915,\"curr\":\"3362006d0323a20df85a19c4cb59e0da6dff6e04550b68af80918d58819323e1\",\"etime\":1567732915,\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"torchlight.sheldonth.com\",\"protocol\":\"http:\"},\"proof_type\":1000,\"state\":1},\"seqno\":4,\"sig_id\":\"39b8ad9a075114c22e14bd8ec5ed6c02d037323d2d66dfd439372cf69006c9700f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"5318b580cc06b58a9fdfb31c7f4ea855e192eea3cfe383385c5ac92c38c15d79\",\"seqno\":8,\"sig_id\":\"dee3495ca913d83ba6f565e65a7d3c054db93fe1fd66feb0a83527070198187e0f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1432764949,\"expire_in\":157680000,\"prev\":\"e6bfc5876ff42db8a4f2c61c4ceaeed90ac38b35a2933f5bd26f504f62c2e50b\",\"seqno\":34,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1432764949,
            "etime": 1590444949,
            "rtime": null,
            "sig_status": 0,
            "prev": "e6bfc5876ff42db8a4f2c61c4ceaeed90ac38b35a2933f5bd26f504f62c2e50b",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        },
        {
            "seqno": 35,
            "payload_hash": "1d5e3bc7f6c8643ea05fd72103ca008139c92514ab43167dd53aa8ff45cf1157",
            "sig_id": "52d347437a9b39d8bb9963ed2395391bd3976f2e67bc3f84646db8e5de3d66930f",
            "sig_id_short": "UtNHQ3qbOdi7mWPtI5U5G9OXby5nvD-EZG24",
            "kid": "0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG/MacGPG2 v2\nComment: GPGTools - https://gpgtools.org\n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\ny+t/U7dedu72BeXFN/85f1HvXXO2NP4C\n=OKuu\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"f41a7d6fbb57b25bbfc781a4496dce572f77e86e\"},\"key\":{\"fingerprint\":\"07bf173c4aa2064b88828a64bf805b6051655371\",\"host\":\"keybase.io\",\"key_id\":\"bf805b6051655371\",\"kid\":\"0101d9d3cab2c3bda4b695eedae218f3f2958b57a7be639cd032a671603c7e9df80f0a\",\"uid\":\"ebbe1d99410ab70123262cf8dfc87900\",\"username\":\"akalin\"},\"track\":{\"basics\":{\"id_version\":19,\"last_id_change\":1435054373,\"username\":\"adnissen\"},\"id\":\"b204a8bfbcf1554cb9ef527a6cc13800\",\"key\":{\"key_fingerprint\":\"6e5a5edfb38be22a8b6b63c5771913ec3429d935\",\"kid\":\"0101a4d5131a79b7f22e72557381de2c77cae2d99c80f8404e3ac4992a6dd426ed180a\"},\"remote_proofs\":[{\"ctime\":1411006844,\"curr\":\"fb86764142c0457fd165e8bad4c6e56d0e2b7177e83d00047264ebada156da25\",\"etime\":1568686844,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"ambushsabre\"},\"proof_type\":2,\"state\":1},\"seqno\":1,\"sig_id\":\"61be4ae24221b97f6f36cd41d4d73ef63197b41cfd8916475123299e0bd902100f\",\"sig_type\":2},{\"ctime\":1411006881,\"curr\":\"c4da006efbea852fae2c6087a6c0e929b824838dc6ceed5ac3a7e7c23ea47ac1\",\"etime\":1568686881,\"prev\":\"fb86764142c0457fd165e8bad4c6e56d0e2b7177e83d00047264ebada156da25\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"adnissen\"},\"proof_type\":3,\"state\":1},\"seqno\":2,\"sig_id\":\"010b9721173864f25519d33348b9d4d0336dc35884b402f56be47c963b2d8e730f\",\"sig_type\":2},{\"ctime\":1411006933,\"curr\":\"e6a8826e158c5233ab4e3765c8db771ad6fef9c34a38eadf89c75754dc3ff3c2\",\"etime\":1568686933,\"prev\":\"c4da006efbea852fae2c6087a6c0e929b824838dc6ceed5ac3a7e7c23ea47ac1\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"adnissen.com\",\"protocol\":\"http:\"},\"proof_type\":1000,\"state\":1},\"seqno\":3,\"sig_id\":\"7610462783daff41ad748bb03b50a8cb12d57475fde7b9637cc824fd677629f30f\",\"sig_type\":2},{\"ctime\":1411007048,\"curr\":\"0de696506a50d763794ca5609b117e416f2ee58248104d11e191cfa3f54bf5a9\",\"etime\":1568687048,\"prev\":\"e6a8826e158c5233ab4e3765c8db771ad6fef9c34a38eadf89c75754dc3ff3c2\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"hackernews\",\"username\":\"adn\"},\"proof_type\":6,\"state\":1},\"seqno\":4,\"sig_id\":\"f962c7a94a6d079e9a711ef90c8397bdd5a54d4050e1c621298ffbed1d0cf1fc0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"2d7d3071220ae03900798d5982798aa25e19bf0e38c338171a7db8f0ae899b3c\",\"seqno\":13,\"sig_id\":\"38859711c2609118672e504848e15098ac49cd271f37b084449df235c5415ff20f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1435098735,\"expire_in\":157680000,\"prev\":\"8c5e68c5a8f0f6e68d610fe1d8c086ad5870ebdecef02dd8f778a34ac300a18c\",\"seqno\":35,\"tag\":\"signature\"}\n",
            "sig_type": 3,
            "ctime": 1435098735,
            "etime": 1592778735,
            "rtime": null,
            "sig_status": 0,
            "prev": "8c5e68c5a8f0f6e68d610fe1d8c086ad5870ebdecef02dd8f778a34ac300a18c",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "07bf173c4aa2064b88828a64bf805b6051655371"
        }
    ],
    "keys": [
        "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nmQENBFQC4E4BCADQgPeE4CNqg5b+eQ4rEcx8afoLw9hGKM3FvWvsuvruIyuzbuza\nlF+fawDahaFhS8CnarmNiHe525rv3xiTHcyAQ2Gmz+6UDvOP8NJOXIxtYXWdsXZr\nL4+b/dXg/OG1MKWKbcgFgemegSTID4Zgxsc8lGG9IuMx5Vbt2ewArj3QK6BS7AVo\nelE5N+KAznyJRuhohEEFqQt5DDO67pB2ySHs6YbwOeJ9h2VX+s9oCXUQZ3ItNrgX\n2fJR/qYAGAfImt+rDriZdPqLLcf4WQLdG14T+aeIXbCs52hbTeYTZMWzPDyrcDBx\nX4R8RLIC0DY6cZD5BRkmG3OH4hSMBnEOQgPhABEBAAG0I0ZyZWRlcmljayBBa2Fs\naW4gPGFrYWxpbkBnbWFpbC5jb20+iQE9BBMBCgAnBQJUAuBOAhsDBQkHhh+ABQsJ\nCAcDBRUKCQgLBRYCAwEAAh4BAheAAAoJEL+AW2BRZVNxrTgIAJWmdsU4g4McTzhC\nsJMHtlhc+DmjPsPozrQdt+qfVPVB4FAkVtadUch4jZ1O+U395g3vHCSBlxcqM5sj\nT4Ix0T94gD1XYktuOy0Qgsit5Wsq66KPEwvSZxexQlW5YB52H6mPfowJJr5WUZB+\nAjPpWJesuX13JzUcUJ6zqxi/Vq9f1rU8cSztmD+KGruYTIeu9vUeIIEs1RAV5uDl\nbmf8r7WSXkqgH9CitKz2sYJPFO/cbKJ0ixfYCF5pp+lDauMZ24vOojhbNSER28//\nQm7FUQGmlGsfkyY4e6hiyl/UL5lItaIejDAwo5qCvCvaVprdsQVxvPsbx9e/KxVX\nzn356N+5AQ0EVALgTgEIAL1ZrRiW7DFSFDRPkuiCuP5KzUAArqG6jdfL3mMBzMsR\npsjKv/3+h4D6fuiswc2RZlGXwZ0RawGsYwAH0XDcSjZ7Bc5YcK+twjJdaVx6N8P1\nF1bHATxQYh2a2/VUkY0amYzsI5x275htQPkta3lZyzbGSnTOwEA+kRRaKlunu1O0\n2gm69VEOHthRV8plwvmhdcFYwIR8tRwKX/tP8QCc6zAiRKPZ7SouekPNwJZBkyNZ\nzHOilvWLdbH6ZTjQoYgG+fD9WbQo7ofArWYft6zZIoKqgWNtCrgA+Uk5ZwpmcL6e\nM6Rra6gwdhh9Zu94LLllTCe95P8GtY618C1nS0UBdqkAEQEAAYkBJQQYAQoADwUC\nVALgTgIbDAUJB4YfgAAKCRC/gFtgUWVTcbv9B/9VNmDQ9FpUx8hnWBj7JA+VyKS3\natPlWhBsJk4G433mJuxWExLN2FKEc6Dowax0d5KDM3OO0j/cCCwI+I4WUFZCh247\nuXmF7sofcZJdFD1o3MNsxMa04ojmFeaJriWngUnJ1TjLSpkkBYI4j9TwGG1z5wZO\n5xTUoU0jFBYAb1dBQGB8YaAfmhqaWNivxNmHhfemTjQ4siMpnq3+7rOJQM+Qotg+\nD8UGKZpm6rTnK80m0jlYI/BzaB6Nm2svjY196p7dI5ZmnZWy1E2D98G5mAbZPBp6\nXb8VEE6fCSrGuJMZ4dwHE+Hi+DRzpjpJ9w6djEUmudeDjmRFUmGQg527JqIw\n=2oDo\n-----END PGP PUBLIC KEY BLOCK-----",
        "0120bad63d4a9b7852b1c092dea088bef3000efa1e14c8d72bf077025f7ad0550b050a",
        "012118781edeaae9c613d53026a4ef920c6c8382a1c4fd40c3992a9c7970a70158310a",
        "01214d73b7d5b4e2004db395fdf67b8818f3ff4324dff1bc301689f806157b8eda5a0a",
        "0120aa39aa0458019ae603412421f7136806ce8f1efc13ba1d0ccec5121df647df490a",
        "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: GnuPG/MacGPG2 v2.0.22 (Darwin)\nComment: GPGTools - https://gpgtools.org\n\nmQENBFQC4E4BCADQgPeE4CNqg5b+eQ4rEcx8afoLw9hGKM3FvWvsuvruIyuzbuza\nlF+fawDahaFhS8CnarmNiHe525rv3xiTHcyAQ2Gmz+6UDvOP8NJOXIxtYXWdsXZr\nL4+b/dXg/OG1MKWKbcgFgemegSTID4Zgxsc8lGG9IuMx5Vbt2ewArj3QK6BS7AVo\nelE5N+KAznyJRuhohEEFqQt5DDO67pB2ySHs6YbwOeJ9h2VX+s9oCXUQZ3ItNrgX\n2fJR/qYAGAfImt+rDriZdPqLLcf4WQLdG14T+aeIXbCs52hbTeYTZMWzPDyrcDBx\nX4R8RLIC0DY6cZD5BRkmG3OH4hSMBnEOQgPhABEBAAG0I0ZyZWRlcmljayBBa2Fs\naW4gPGFrYWxpbkBnbWFpbC5jb20+iQE9BBMBCgAnBQJUAuBOAhsDBQkHhh+ABQsJ\nCAcDBRUKCQgLBRYCAwEAAh4BAheAAAoJEL+AW2BRZVNxrTgIAJWmdsU4g4McTzhC\nsJMHtlhc+DmjPsPozrQdt+qfVPVB4FAkVtadUch4jZ1O+U395g3vHCSBlxcqM5sj\nT4Ix0T94gD1XYktuOy0Qgsit5Wsq66KPEwvSZxexQlW5YB52H6mPfowJJr5WUZB+\nAjPpWJesuX13JzUcUJ6zqxi/Vq9f1rU8cSztmD+KGruYTIeu9vUeIIEs1RAV5uDl\nbmf8r7WSXkqgH9CitKz2sYJPFO/cbKJ0ixfYCF5pp+lDauMZ24vOojhbNSER28//\nQm7FUQGmlGsfkyY4e6hiyl/UL5lItaIejDAwo5qCvCvaVprdsQVxvPsbx9e/KxVX\nzn356N+5AQ0EVALgTgEIAL1ZrRiW7DFSFDRPkuiCuP5KzUAArqG6jdfL3mMBzMsR\npsjKv/3+h4D6fuiswc2RZlGXwZ0RawGsYwAH0XDcSjZ7Bc5YcK+twjJdaVx6N8P1\nF1bHATxQYh2a2/VUkY0amYzsI5x275htQPkta3lZyzbGSnTOwEA+kRRaKlunu1O0\n2gm69VEOHthRV8plwvmhdcFYwIR8tRwKX/tP8QCc6zAiRKPZ7SouekPNwJZBkyNZ\nzHOilvWLdbH6ZTjQoYgG+fD9WbQo7ofArWYft6zZIoKqgWNtCrgA+Uk5ZwpmcL6e\nM6Rra6gwdhh9Zu94LLllTCe95P8GtY618C1nS0UBdqkAEQEAAYkBJQQYAQoADwUC\nVALgTgIbDAUJB4YfgAAKCRC/gFtgUWVTcbv9B/9VNmDQ9FpUx8hnWBj7JA+VyKS3\natPlWhBsJk4G433mJuxWExLN2FKEc6Dowax0d5KDM3OO0j/cCCwI+I4WUFZCh247\nuXmF7sofcZJdFD1o3MNsxMa04ojmFeaJriWngUnJ1TjLSpkkBYI4j9TwGG1z5wZO\n5xTUoU0jFBYAb1dBQGB8YaAfmhqaWNivxNmHhfemTjQ4siMpnq3+7rOJQM+Qotg+\nD8UGKZpm6rTnK80m0jlYI/BzaB6Nm2svjY196p7dI5ZmnZWy1E2D98G5mAbZPBp6\nXb8VEE6fCSrGuJMZ4dwHE+Hi+DRzpjpJ9w6djEUmudeDjmRFUmGQg527JqIw\n=2oDo\n-----END PGP PUBLIC KEY BLOCK-----"
    ]
}
`,
	"bad_full_hash.json": `{
    "chain": [
        {
            "seqno": 1,
            "prev": null,
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.44\nComment: https://keybase.io/crypto\n\nyMHlAnicbZFdSBRRGIbXskxjo4Qs88L2hES0yPzPzupNISQRCJJppa5nZs6ugzo7\nOzMriq1EWUFSaWq5ZBn4lyWSEf0QiWteuRj+hBYUohC2ZYFYUFHS2Y3uOjfnfO/7\nfs8H5xuzrrckxeVdsw7w7JGmuPDIkt9SEFnLqgOiV64FzjpQgWKXW1E9SNd0RTWB\nE7CQp0URsYQgEpJAiTIlQRHSLMeLDp6kHZCnIEtzDLADt7+y0lUOjXLc5vZ6Rahj\nsdxrRDGYLUIDZSperOHCpchY/Q+kImYQJEHyNJQhhY9EQcSSPMFJjOAWSZJ384KI\nCJpHLJQExs2LlOTgBMbBUxSDOAIKIhZJAmKcP4ZzCDJDy6KbFSBHS4JIQ5JzUCyL\nEOTdLClEgwbSVViFcFrzaC6/JkMTuUxkmEgHATswa7WohyplLOF8NdINxasCJ4lN\nyVSinSTDkLTgoGnSDlCNpujIpeAEjXWBIwjCDjQdVQOnij/KDgzkc/2lkrFC9cZe\nJvTgOYbiUaHp1xEIjI4Ux1vikiwbN6yL7suSlLj13xZThhN+J+96/HxL5O6lve/1\nuUehkX5PxteHt2pmX84Fe3yR8e3XCbCzs2BifPBHSfHPhtbQvsBkbtnhC6GB1QOd\nyeXhHO3K4HzHu9fWuvb46ZnktuVIWyl579WNm5k9yyVpQdvVvqppX0bg7AqVveR7\nuvdzo407OvRsreVB6OSMlLrkD9frXXRaWkJTd2//wcSpFwuLjR+maj/2NXe0n4N7\nPpWhyRZC7aU6LxLbSrNP5x/P/1VovT25eKer7HtIuG/aihoKx77szlvLH229XJHj\nSjnTdWxT0LZgbi4+cehtpv4muDp8Kj1smaW7pcT6UO4Embpj/8qQFkhvLvp2fng+\n60nKH6ETE6k=\n=8ZaC\n-----END PGP MESSAGE-----\n",
            "payload_hash": "9d6cea32a5cc0a0d9cfb9a58d60f0c1a9720e748dcd83901afb32e9de9c8b03c",
            "sig_id": "3530b961e15781b6e833f8e9bae1e857a9b52b3aeb1a3c34e83a0e4e91b5d29f0f",
            "payload_json": "{\"body\":{\"key\":{\"fingerprint\":\"5a73bbe509b0c92bd2caba3567b87138a72a5364\",\"full_hash\":\"foobar\",\"host\":\"keybase.io\",\"key_id\":\"67b87138a72a5364\",\"kid\":\"010173ada2222c2ae51706c49fb117f79be037e5ac94f7b2c869487224e60a9b4f710a\",\"uid\":\"89d43dbf59a63c9b3a168255eea7f519\",\"username\":\"pgp_update_tester\"},\"type\":\"eldest\",\"version\":1},\"ctime\":1441398331,\"expire_in\":314496000,\"prev\":null,\"seq_type\":1,\"seqno\":1,\"tag\":\"signature\"}",
            "kid": "010173ada2222c2ae51706c49fb117f79be037e5ac94f7b2c869487224e60a9b4f710a",
            "ctime": 1441398331
        }
    ],
    "keys": [
        "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: GnuPG v1\n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\nnQ==\n=olf1\n-----END PGP PUBLIC KEY BLOCK-----"
    ],
    "uid": "89d43dbf59a63c9b3a168255eea7f519",
    "username": "pgp_update_tester",
    "label_kids": {
        "start": "010173ada2222c2ae51706c49fb117f79be037e5ac94f7b2c869487224e60a9b4f710a"
    },
    "label_sigs": {
        "undefined": "3530b961e15781b6e833f8e9bae1e857a9b52b3aeb1a3c34e83a0e4e91b5d29f0f"
    }
}
`,
	"bad_per_user_key_reverse_sig.json": `{
   "chain" : [
      {
         "payload_json" : "{\"body\":{\"key\":{\"host\":\"keybase.io\",\"kid\":\"012078890f133767e09a0b15b7fc90f8d5abf2a8a146ff27e84854a052ad81d3f17d0a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"type\":\"eldest\",\"version\":1},\"ctime\":1500387191,\"expire_in\":314496000,\"prev\":null,\"seq_type\":1,\"seqno\":1,\"tag\":\"signature\"}",
         "ctime" : 1500387191,
         "prev" : null,
         "kid" : "012078890f133767e09a0b15b7fc90f8d5abf2a8a146ff27e84854a052ad81d3f17d0a",
         "seqno" : 1,
         "payload_hash" : "39d1d7c527a4105e2a537426e70b59d9c1ef80fcc6017b60adef17f83e2d0f7e",
         "sig_id" : "d8d3fe88c65c5e2eb1c307dd49ce6f56914eb4358e2c98c1ca5bb2f0eaeced9c0f",
         "sig_version" : 1,
         "sig" : "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgeIkPEzdn4JoLFbf8kPjVq/KooUb/J+hIVKBSrYHT8X0Kp3BheWxvYWTFASt7ImJvZHkiOnsia2V5Ijp7Imhvc3QiOiJrZXliYXNlLmlvIiwia2lkIjoiMDEyMDc4ODkwZjEzMzc2N2UwOWEwYjE1YjdmYzkwZjhkNWFiZjJhOGExNDZmZjI3ZTg0ODU0YTA1MmFkODFkM2YxN2QwYSIsInVpZCI6Ijc0YzM4Y2Y3Y2ViOTQ3ZjU2MzIwNDVkOGNhNWQ0ODE5IiwidXNlcm5hbWUiOiJtYXgzMiJ9LCJ0eXBlIjoiZWxkZXN0IiwidmVyc2lvbiI6MX0sImN0aW1lIjoxNTAwMzg3MTkxLCJleHBpcmVfaW4iOjMxNDQ5NjAwMCwicHJldiI6bnVsbCwic2VxX3R5cGUiOjEsInNlcW5vIjoxLCJ0YWciOiJzaWduYXR1cmUifaNzaWfEQAZ7YloYQtCO/PFA5Y/nWDR/NRljIvEHD9gp4s/Wv+m6vSqOYlAND/eYJ1Vh6uiAe1NyoTthQgfO8628YbpIXAyoc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ=="
      },
      {
         "ctime" : 1500387291,
         "payload_json" : "{\"body\":{\"key\":{\"host\":\"keybase.io\",\"kid\":\"012078890f133767e09a0b15b7fc90f8d5abf2a8a146ff27e84854a052ad81d3f17d0a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"per_user_key\":{\"encryption_kid\":\"01214f4fef89b8b06efc20c30500eb24fa3882246117150b9644ec0e581d9198a45d0a\",\"generation\":1,\"reverse_sig\":\"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\",\"signing_kid\":\"0120328d658ba4dbc2f2e590e3fa5a1050a2d80ac9d075307330778bbd796cf155fc0a\"},\"type\":\"per_user_key\",\"version\":1},\"ctime\":1500387291,\"expire_in\":314496000,\"prev\":\"39d1d7c527a4105e2a537426e70b59d9c1ef80fcc6017b60adef17f83e2d0f7e\",\"seq_type\":1,\"seqno\":2,\"tag\":\"signature\"}",
         "kid" : "012078890f133767e09a0b15b7fc90f8d5abf2a8a146ff27e84854a052ad81d3f17d0a",
         "prev" : "39d1d7c527a4105e2a537426e70b59d9c1ef80fcc6017b60adef17f83e2d0f7e",
         "payload_hash" : "c710c1eeec26b4b03a0fb30cb61ffbe1408a6571d21c07c6192255d0f6533b4b",
         "seqno" : 2,
         "sig" : "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgeIkPEzdn4JoLFbf8kPjVq/KooUb/J+hIVKBSrYHT8X0Kp3BheWxvYWTFC8V7ImJvZHkiOnsia2V5Ijp7Imhvc3QiOiJrZXliYXNlLmlvIiwia2lkIjoiMDEyMDc4ODkwZjEzMzc2N2UwOWEwYjE1YjdmYzkwZjhkNWFiZjJhOGExNDZmZjI3ZTg0ODU0YTA1MmFkODFkM2YxN2QwYSIsInVpZCI6Ijc0YzM4Y2Y3Y2ViOTQ3ZjU2MzIwNDVkOGNhNWQ0ODE5IiwidXNlcm5hbWUiOiJtYXgzMiJ9LCJwZXJfdXNlcl9rZXkiOnsiZW5jcnlwdGlvbl9raWQiOiIwMTIxNGY0ZmVmODliOGIwNmVmYzIwYzMwNTAwZWIyNGZhMzg4MjI0NjExNzE1MGI5NjQ0ZWMwZTU4MWQ5MTk4YTQ1ZDBhIiwiZ2VuZXJhdGlvbiI6MSwicmV2ZXJzZV9zaWciOiJnNlJpYjJSNWhxaGtaWFJoWTJobFpNT3BhR0Z6YUY5MGVYQmxDcU5yWlhuRUl3RWdNbzFsaTZUYnd2TGxrT1A2V2hCUW90Z0t5ZEIxTUhNd2Q0dTllV3p4VmZ3S3AzQmhlV3h2WVdURkJtVjdJbUp2WkhraU9uc2lhMlY1SWpwN0ltaHZjM1FpT2lKclpYbGlZWE5sTG1sdklpd2lhMmxrSWpvaU1ERXlNRGM0T0Rrd1pqRXpNemMyTjJVd09XRXdZakUxWWpkbVl6a3daamhrTldGaVpqSmhPR0V4TkRabVpqSTNaVGcwT0RVMFlUQTFNbUZrT0RGa00yWXhOMlF3WVNJc0luVnBaQ0k2SWpjMFl6TTRZMlkzWTJWaU9UUTNaalUyTXpJd05EVmtPR05oTldRME9ERTVJaXdpZFhObGNtNWhiV1VpT2lKdFlYZ3pNaUo5TENKd1pYSmZkWE5sY2w5clpYa2lPbnNpWlc1amNubHdkR2x2Ymw5cmFXUWlPaUl3TVRJeE5HWTBabVZtT0RsaU9HSXdObVZtWXpJd1l6TXdOVEF3WldJeU5HWmhNemc0TWpJME5qRXhOekUxTUdJNU5qUTBaV013WlRVNE1XUTVNVGs0WVRRMVpEQmhJaXdpWjJWdVpYSmhkR2x2YmlJNk1Td2ljbVYyWlhKelpWOXphV2NpT2lKbk5sSnBZakpTTldoeGFHdGFXRkpvV1RKb2JGcE5UM0JoUjBaNllVWTVNR1ZZUW14RGNVNXlXbGh1UlVsM1JXZE5iekZzYVRaVVluZDJUR3hyVDFBMlYyaENVVzkwWjB0NVpFSXhUVWhOZDJRMGRUbGxWM3A0Vm1aM1MzQXpRbWhsVjNoMldWZFVSa0ZzVGpkSmJVcDJXa2hyYVU5dWMybGhNbFkxU1dwd04wbHRhSFpqTTFGcFQybEtjbHBZYkdsWldFNXNURzFzZGtscGQybGhNbXhyU1dwdmFVMUVSWGxOUkdNMFQwUnJkMXBxUlhwTmVtTXlUakpWZDA5WFJYZFpha1V4V1dwa2JWbDZhM2RhYW1oclRsZEdhVnBxU21oUFIwVjRUa1JhYlZwcVNUTmFWR2N3VDBSVk1GbFVRVEZOYlVaclQwUkdhMDB5V1hoT01sRjNXVk5KYzBsdVZuQmFRMGsyU1dwak1GbDZUVFJaTWxreldUSldhVTlVVVROYWFsVXlUWHBKZDA1RVZtdFBSMDVvVGxkUk1FOUVSVFZKYVhkcFpGaE9iR050TldoaVYxVnBUMmxLZEZsWVozcE5hVW81VEVOS2QxcFlTbVprV0U1c1kydzVjbHBZYTJsUGJuTnBXbGMxYW1OdWJIZGtSMngyWW13NWNtRlhVV2xQYVVsM1RWUkplRTVIV1RCYWJWWnRUMFJzYVU5SFNYZE9iVlp0V1hwSmQxbDZUWGRPVkVGM1dsZEplVTVIV21oTmVtYzBUV3BKTUU1cVJYaE9la1V4VFVkSk5VNXFVVEJhVjAxM1dsUlZORTFYVVRWTlZHczBXVlJSTVZwRVFtaEphWGRwV2pKV2RWcFlTbWhrUjJ4MlltbEpOazFUZDJsamJWWXlXbGhLZWxwV09YcGhWMk5wVDIwMU1XSkhkM05KYms1d1dqSTFjR0p0WkdaaE1teHJTV3B2YVUxRVJYbE5SRTE1VDBkUk1rNVVhR2xaVkZKcldXMU5lVnBxU214T1ZHdDNXbFJPYlZsVVZtaE5WRUV4VFVkRmVWcEVaM2RaVjAwMVdrUkJNMDVVVFhkT2VrMTZUVVJqTTA5SFNtbGFSR00xVG0xT2JVMVVWVEZhYlUxM1dWTktPVXhEU2pCbFdFSnNTV3B2YVdOSFZubFlNMVo2V2xoS1ptRXlWalZKYVhkcFpHMVdlV015YkhaaWFVazJUVmd3YzBsdFRqQmhWekZzU1dwdmVFNVVRWGROZW1jelRXcHJlRXhEU214bFNFSndZMjFXWm1GWE5HbFBhazE0VGtSUk5VNXFRWGROUTNkcFkwaEtiR1JwU1RaSmFrMDFXa1JHYTA0eVRURk5hbVJvVGtSRmQwNVhWWGxaVkZWNlRucFJlVTV0VlROTlIwa3hUMWRSTlZsNlJteGFhbWQzV20xT2FrNXFRWGhPTWtreVRVZEdhMXBYV1hoT01sazBUVEpWZVZwRVFtMU9NbFZwVEVOS2VscFlSbVprU0d4M1dsTkpOazFUZDJsak1sWjRZbTA0YVU5cVNYTkpibEpvV25sSk5rbHVUbkJhTWpWb1pFaFdlVnBUU2psdk0wNXdXamhTUVNzME4xTnVZVk5EWW1aQ1oycENNQ3Q2YVdkdk5YWk9TVWhSV2pOU2VrZHpjRFl2WWpSVlVFRmhTMVJJYm5GM1p6UlBTV1JYUWtOR1IzcFJXRmRwTVRkaFJuVnFlWGh4VjB4TlVHZHFjR3M1ZHk5cWVFTnhhSHBoVjJSbVpFaHNkMXBUUTJwa1IwWnVlbEZKUTNBeldteGpiazV3WWpJMFFpSXNJbk5wWjI1cGJtZGZhMmxrSWpvaU1ERXlNRE15T0dRMk5UaGlZVFJrWW1NeVpqSmxOVGt3WlRObVlUVmhNVEExTUdFeVpEZ3dZV001WkRBM05UTXdOek16TURjM09HSmlaRGM1Tm1ObU1UVTFabU13WVNKOUxDSjBlWEJsSWpvaWNHVnlYM1Z6WlhKZmEyVjVJaXdpZG1WeWMybHZiaUk2TVgwc0ltTjBhVzFsSWpveE5UQXdNemczTWpreExDSmxlSEJwY21WZmFXNGlPak14TkRRNU5qQXdNQ3dpYUdWc2JHOGlPaUozYjNKc1pDSXNJbkJ5WlhZaU9pSXpPV1F4WkRkak5USTNZVFF4TURWbE1tRTFNemMwTWpabE56QmlOVGxrT1dNeFpXWTRNR1pqWXpZd01UZGlOakJoWkdWbU1UZG1PRE5sTW1Rd1pqZGxJaXdpYzJWeFgzUjVjR1VpT2pFc0luTmxjVzV2SWpveUxDSjBZV2NpT2lKemFXZHVZWFIxY21VaWZhTnphV2ZFUURnMHNpOEkzUzVRclZZZWROTStFU3ptVGs3VjZ5ckpSQTl4R2VUZHhxUjdDMmRQMk5KSFlaZFg3NUFUS1hBZ2tEcUZDUmtLUTd1eDJLeGQ3SjNLclFhb2MybG5YM1I1Y0dVZ28zUmhaODBDQXFkMlpYSnphVzl1QVE9PSIsInNpZ25pbmdfa2lkIjoiMDEyMDMyOGQ2NThiYTRkYmMyZjJlNTkwZTNmYTVhMTA1MGEyZDgwYWM5ZDA3NTMwNzMzMDc3OGJiZDc5NmNmMTU1ZmMwYSJ9LCJ0eXBlIjoicGVyX3VzZXJfa2V5IiwidmVyc2lvbiI6MX0sImN0aW1lIjoxNTAwMzg3MjkxLCJleHBpcmVfaW4iOjMxNDQ5NjAwMCwicHJldiI6IjM5ZDFkN2M1MjdhNDEwNWUyYTUzNzQyNmU3MGI1OWQ5YzFlZjgwZmNjNjAxN2I2MGFkZWYxN2Y4M2UyZDBmN2UiLCJzZXFfdHlwZSI6MSwic2Vxbm8iOjIsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RANfmK5Q39SuZV+bBrGmHGeYTBlbtYM9iypWMErhiGpXVuYDWWYl5XDhzWVrYmGyQHwgp/g4GmXz/4qF05hFcoCKhzaWdfdHlwZSCjdGFnzQICp3ZlcnNpb24B",
         "sig_version" : 1,
         "sig_id" : "ff0d4fbd8689894ffbe89bed2805ed2f203fe3327850a020ef676453bbb2fbfe0f"
      }
   ],
   "label_kids" : {
      "e" : "012078890f133767e09a0b15b7fc90f8d5abf2a8a146ff27e84854a052ad81d3f17d0a",
      "puk1" : "01214f4fef89b8b06efc20c30500eb24fa3882246117150b9644ec0e581d9198a45d0a"
   },
   "username" : "max32",
   "keys" : [
      "012078890f133767e09a0b15b7fc90f8d5abf2a8a146ff27e84854a052ad81d3f17d0a",
      "0120328d658ba4dbc2f2e590e3fa5a1050a2d80ac9d075307330778bbd796cf155fc0a",
      "01214f4fef89b8b06efc20c30500eb24fa3882246117150b9644ec0e581d9198a45d0a"
   ],
   "label_sigs" : {
      "e" : "d8d3fe88c65c5e2eb1c307dd49ce6f56914eb4358e2c98c1ca5bb2f0eaeced9c0f",
      "puk1" : "ff0d4fbd8689894ffbe89bed2805ed2f203fe3327850a020ef676453bbb2fbfe0f"
   },
   "uid" : "74c38cf7ceb947f5632045d8ca5d4819"
}
`,
	"bad_prev_chain.json": `{
    "chain": [
        {
            "seqno": 1,
            "prev": null,
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgtAfp8c2vIsq6o/a4fzGays16KagtWSwxbObC7mU2UngKp3BheWxvYWTFASp7ImJvZHkiOnsia2V5Ijp7Imhvc3QiOiJrZXliYXNlLmlvIiwia2lkIjoiMDEyMGI0MDdlOWYxY2RhZjIyY2FiYWEzZjZiODdmMzE5YWNhY2Q3YTI5YTgyZDU5MmMzMTZjZTZjMmVlNjUzNjUyNzgwYSIsInVpZCI6Ijc0YzM4Y2Y3Y2ViOTQ3ZjU2MzIwNDVkOGNhNWQ0ODE5IiwidXNlcm5hbWUiOiJtYXgzMiJ9LCJ0eXBlIjoiZWxkZXN0IiwidmVyc2lvbiI6MX0sImN0aW1lIjoxNDMyMjI1OTA0LCJleHBpcmVfaW4iOjEwMDAwMDAwLCJwcmV2IjpudWxsLCJzZXFfdHlwZSI6MSwic2Vxbm8iOjEsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RALK1iab8sGnnb4waKvKzJAO2kpRvTyaCy7u6W8ZI2+OALN0SmeZZ4ZZk9A+qDWNWFaw7LRDg7/wafrn4sVjwlDKhzaWdfdHlwZSCjdGFnzQICp3ZlcnNpb24B",
            "payload_hash": "df5d9f31e681f921d15e1dc98dd875f3f299dcebdf5f4267827416a2117cb22d",
            "sig_id": "3dc0ed1000573ccde355fd4004274c2aa45aed986df5fcbc187ce1ceffdfcb560f",
            "payload_json": "{\"body\":{\"key\":{\"host\":\"keybase.io\",\"kid\":\"0120b407e9f1cdaf22cabaa3f6b87f319acacd7a29a82d592c316ce6c2ee653652780a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"type\":\"eldest\",\"version\":1},\"ctime\":1432225904,\"expire_in\":10000000,\"prev\":null,\"seq_type\":1,\"seqno\":1,\"tag\":\"signature\"}",
            "kid": "0120b407e9f1cdaf22cabaa3f6b87f319acacd7a29a82d592c316ce6c2ee653652780a",
            "ctime": 1432225904
        },
        {
            "seqno": 2,
            "prev": "deadbeef00000000000000000000000000000000000000000000000000000000",
            "sig": "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",
            "payload_hash": "bca91e71714a849c255c531c60bc6e4a69c33782edeba8afcab9fc22b286171e",
            "sig_id": "10a9ec422404d3b0b062e56c809db16e3301bd01deb4d54ee77e5895c29a79660f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120b407e9f1cdaf22cabaa3f6b87f319acacd7a29a82d592c316ce6c2ee653652780a\",\"host\":\"keybase.io\",\"kid\":\"0120b407e9f1cdaf22cabaa3f6b87f319acacd7a29a82d592c316ce6c2ee653652780a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"sibkey\":{\"kid\":\"0120126e4cb40127102247b460ac6e4872c35b53075af0598ccb04034b961cf740cd0a\",\"reverse_sig\":\"g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgEm5MtAEnECJHtGCsbkhyw1tTB1rwWYzLBANLlhz3QM0Kp3BheWxvYWTFAit7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwYjQwN2U5ZjFjZGFmMjJjYWJhYTNmNmI4N2YzMTlhY2FjZDdhMjlhODJkNTkyYzMxNmNlNmMyZWU2NTM2NTI3ODBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwYjQwN2U5ZjFjZGFmMjJjYWJhYTNmNmI4N2YzMTlhY2FjZDdhMjlhODJkNTkyYzMxNmNlNmMyZWU2NTM2NTI3ODBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInNpYmtleSI6eyJraWQiOiIwMTIwMTI2ZTRjYjQwMTI3MTAyMjQ3YjQ2MGFjNmU0ODcyYzM1YjUzMDc1YWYwNTk4Y2NiMDQwMzRiOTYxY2Y3NDBjZDBhIiwicmV2ZXJzZV9zaWciOm51bGx9LCJ0eXBlIjoic2lia2V5IiwidmVyc2lvbiI6MX0sImN0aW1lIjoxNDMyMjI2MDA0LCJleHBpcmVfaW4iOjEwMDAwMDAwLCJwcmV2IjoiZGVhZGJlZWYwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwMCIsInNlcV90eXBlIjoxLCJzZXFubyI6MiwidGFnIjoic2lnbmF0dXJlIn2jc2lnxEB7MkCVRfz1rQw9QyfguqbncdwBve1YGTcopsxfibwjRj98Io1n4kAo8i5pBP4Wk7HiTSsr7DJNF6GRtUiJLMkCqHNpZ190eXBlIKN0YWfNAgKndmVyc2lvbgE=\"},\"type\":\"sibkey\",\"version\":1},\"ctime\":1432226004,\"expire_in\":10000000,\"prev\":\"deadbeef00000000000000000000000000000000000000000000000000000000\",\"seq_type\":1,\"seqno\":2,\"tag\":\"signature\"}",
            "kid": "0120b407e9f1cdaf22cabaa3f6b87f319acacd7a29a82d592c316ce6c2ee653652780a",
            "ctime": 1432226004
        }
    ],
    "keys": [
        "0120b407e9f1cdaf22cabaa3f6b87f319acacd7a29a82d592c316ce6c2ee653652780a",
        "0120126e4cb40127102247b460ac6e4872c35b53075af0598ccb04034b961cf740cd0a"
    ],
    "uid": "74c38cf7ceb947f5632045d8ca5d4819",
    "username": "max32"
}
`,
	"bad_reverse_signature_chain.json": `{
    "chain": [
        {
            "seqno": 1,
            "prev": null,
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgd1JXIa21+kSY41/Nx2jLADJcFxGLZ0Rgxug6xaFH67EKp3BheWxvYWTFASp7ImJvZHkiOnsia2V5Ijp7Imhvc3QiOiJrZXliYXNlLmlvIiwia2lkIjoiMDEyMDc3NTI1NzIxYWRiNWZhNDQ5OGUzNWZjZGM3NjhjYjAwMzI1YzE3MTE4YjY3NDQ2MGM2ZTgzYWM1YTE0N2ViYjEwYSIsInVpZCI6Ijc0YzM4Y2Y3Y2ViOTQ3ZjU2MzIwNDVkOGNhNWQ0ODE5IiwidXNlcm5hbWUiOiJtYXgzMiJ9LCJ0eXBlIjoiZWxkZXN0IiwidmVyc2lvbiI6MX0sImN0aW1lIjoxNDMyMTQ0MjQ1LCJleHBpcmVfaW4iOjEwMDAwMDAwLCJwcmV2IjpudWxsLCJzZXFfdHlwZSI6MSwic2Vxbm8iOjEsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RAz1C4EVTGxElMcIHOJqRFTmqA0XeFS8oAweX4Bih+6yfzi6sqiOefKBg5AvG3gXeWnHdsSCzJIun8Anwuhf+zAqhzaWdfdHlwZSCjdGFnzQICp3ZlcnNpb24B",
            "payload_hash": "2eddd3fa83f7f34808526b28bb887a95c09dc81f833c88b234a4be2311dd89f4",
            "sig_id": "7c7ef1f35cf56e63b62f42d1afee7682022bed0c58981f9ca4f8601929cb923c0f",
            "payload_json": "{\"body\":{\"key\":{\"host\":\"keybase.io\",\"kid\":\"012077525721adb5fa4498e35fcdc768cb00325c17118b674460c6e83ac5a147ebb10a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"type\":\"eldest\",\"version\":1},\"ctime\":1432144245,\"expire_in\":10000000,\"prev\":null,\"seq_type\":1,\"seqno\":1,\"tag\":\"signature\"}",
            "kid": "012077525721adb5fa4498e35fcdc768cb00325c17118b674460c6e83ac5a147ebb10a",
            "ctime": 1432144245
        },
        {
            "seqno": 2,
            "prev": "2eddd3fa83f7f34808526b28bb887a95c09dc81f833c88b234a4be2311dd89f4",
            "sig": "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",
            "payload_hash": "4ccc4185a501dd16f2e2226182d3fadf8558aa59b593b5e4a239aea35581102d",
            "sig_id": "2d8dcd8fe4e7ceb9c67a76c18bbee71524d0ba0aa7c2e10298ce5688cf3cd2c80f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"012077525721adb5fa4498e35fcdc768cb00325c17118b674460c6e83ac5a147ebb10a\",\"host\":\"keybase.io\",\"kid\":\"012077525721adb5fa4498e35fcdc768cb00325c17118b674460c6e83ac5a147ebb10a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"sibkey\":{\"kid\":\"012057257eeb8feed5d331d63ba382b0d5cffba17c95d32a83c7a58b5527bad873250a\",\"reverse_sig\":\"g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgVyV+64/u1dMx1jujgrDVz/uhfJXTKoPHpYtVJ7rYcyUKp3BheWxvYWTFAit7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwNzc1MjU3MjFhZGI1ZmE0NDk4ZTM1ZmNkYzc2OGNiMDAzMjVjMTcxMThiNjc0NDYwYzZlODNhYzVhMTQ3ZWJiMTBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwNzc1MjU3MjFhZGI1ZmE0NDk4ZTM1ZmNkYzc2OGNiMDAzMjVjMTcxMThiNjc0NDYwYzZlODNhYzVhMTQ3ZWJiMTBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInNpYmtleSI6eyJraWQiOiIwMTIwNTcyNTdlZWI4ZmVlZDVkMzMxZDYzYmEzODJiMGQ1Y2ZmYmExN2M5NWQzMmE4M2M3YTU4YjU1MjdiYWQ4NzMyNTBhIiwicmV2ZXJzZV9zaWciOm51bGx9LCJ0eXBlIjoic2lia2V5IiwidmVyc2lvbiI6MX0sImN0aW1lIjoxNDMyMTQ0MzQ1LCJleHBpcmVfaW4iOjEwMDAwMDAwLCJwcmV2IjoiMmVkZGQzZmE4M2Y3ZjM0ODA4NTI2YjI4YmI4ODdhOTVjMDlkYzgxZjgzM2M4OGIyMzRhNGJlMjMxMWRkODlmNCIsInNlcV90eXBlIjoxLCJzZXFubyI6MiwidGFnIjoic2lnbmF0dXJlIn2jc2lnxEAAAAAAQcgOiJ9NVPw9R/PVasiuow1AKdCTncJcmqJTYcG+0U2CSVUsKbT6iop7qEuJn7kD5NNoz3kRE31SJqsJqHNpZ190eXBlIKN0YWfNAgKndmVyc2lvbgE=\"},\"type\":\"sibkey\",\"version\":1},\"ctime\":1432144345,\"expire_in\":10000000,\"prev\":\"2eddd3fa83f7f34808526b28bb887a95c09dc81f833c88b234a4be2311dd89f4\",\"seq_type\":1,\"seqno\":2,\"tag\":\"signature\"}",
            "kid": "012077525721adb5fa4498e35fcdc768cb00325c17118b674460c6e83ac5a147ebb10a",
            "ctime": 1432144345
        }
    ],
    "keys": [
        "012077525721adb5fa4498e35fcdc768cb00325c17118b674460c6e83ac5a147ebb10a",
        "012057257eeb8feed5d331d63ba382b0d5cffba17c95d32a83c7a58b5527bad873250a"
    ],
    "uid": "74c38cf7ceb947f5632045d8ca5d4819",
    "username": "max32"
}
`,
	"bad_seqno_chain.json": `{
    "chain": [
        {
            "seqno": 1,
            "prev": null,
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgsDiGCNcoqXQu5wYLlcIpCJv1Vlj7OJ3mmHN1pC8If7cKp3BheWxvYWTFASp7ImJvZHkiOnsia2V5Ijp7Imhvc3QiOiJrZXliYXNlLmlvIiwia2lkIjoiMDEyMGIwMzg4NjA4ZDcyOGE5NzQyZWU3MDYwYjk1YzIyOTA4OWJmNTU2NThmYjM4OWRlNjk4NzM3NWE0MmYwODdmYjcwYSIsInVpZCI6Ijc0YzM4Y2Y3Y2ViOTQ3ZjU2MzIwNDVkOGNhNWQ0ODE5IiwidXNlcm5hbWUiOiJtYXgzMiJ9LCJ0eXBlIjoiZWxkZXN0IiwidmVyc2lvbiI6MX0sImN0aW1lIjoxNDMyMTQ0MzQzLCJleHBpcmVfaW4iOjEwMDAwMDAwLCJwcmV2IjpudWxsLCJzZXFfdHlwZSI6MSwic2Vxbm8iOjEsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RAzscTjEcAa9aF8Ze5MOUpDaR2BRSFObI7CtWlFLSAySxG0kKv99hqh53WjTyUL9G4CZcg14o1WSJcS9P05XI2DahzaWdfdHlwZSCjdGFnzQICp3ZlcnNpb24B",
            "payload_hash": "fafd9bbd0b7e905b96cb4b6c8022f7ee712098ef78ae048acb2fa296220d760f",
            "sig_id": "8dcc6e4d4de09e9bf0a1bf543e9bafbef589a2dce7797bd45296414a7b7cbff50f",
            "payload_json": "{\"body\":{\"key\":{\"host\":\"keybase.io\",\"kid\":\"0120b0388608d728a9742ee7060b95c229089bf55658fb389de6987375a42f087fb70a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"type\":\"eldest\",\"version\":1},\"ctime\":1432144343,\"expire_in\":10000000,\"prev\":null,\"seq_type\":1,\"seqno\":1,\"tag\":\"signature\"}",
            "kid": "0120b0388608d728a9742ee7060b95c229089bf55658fb389de6987375a42f087fb70a",
            "ctime": 1432144343
        },
        {
            "seqno": 3,
            "prev": "fafd9bbd0b7e905b96cb4b6c8022f7ee712098ef78ae048acb2fa296220d760f",
            "sig": "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",
            "payload_hash": "db4a190291265dceb1d047089ca40c21cea4258e2eb7c353fd7aa19729e7b0c1",
            "sig_id": "9a0eb84b985fa57a38a850189206cd0cd257db3d9c6c1f6169373ba1b5b1e4be0f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120b0388608d728a9742ee7060b95c229089bf55658fb389de6987375a42f087fb70a\",\"host\":\"keybase.io\",\"kid\":\"0120b0388608d728a9742ee7060b95c229089bf55658fb389de6987375a42f087fb70a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"sibkey\":{\"kid\":\"012015d09e9f3d13242c0b531228825b2762163fc4a72eca3439f91867734da305e20a\",\"reverse_sig\":\"g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgFdCenz0TJCwLUxIoglsnYhY/xKcuyjQ5+Rhnc02jBeIKp3BheWxvYWTFAit7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwYjAzODg2MDhkNzI4YTk3NDJlZTcwNjBiOTVjMjI5MDg5YmY1NTY1OGZiMzg5ZGU2OTg3Mzc1YTQyZjA4N2ZiNzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwYjAzODg2MDhkNzI4YTk3NDJlZTcwNjBiOTVjMjI5MDg5YmY1NTY1OGZiMzg5ZGU2OTg3Mzc1YTQyZjA4N2ZiNzBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInNpYmtleSI6eyJraWQiOiIwMTIwMTVkMDllOWYzZDEzMjQyYzBiNTMxMjI4ODI1YjI3NjIxNjNmYzRhNzJlY2EzNDM5ZjkxODY3NzM0ZGEzMDVlMjBhIiwicmV2ZXJzZV9zaWciOm51bGx9LCJ0eXBlIjoic2lia2V5IiwidmVyc2lvbiI6MX0sImN0aW1lIjoxNDMyMTQ0NDQzLCJleHBpcmVfaW4iOjEwMDAwMDAwLCJwcmV2IjoiZmFmZDliYmQwYjdlOTA1Yjk2Y2I0YjZjODAyMmY3ZWU3MTIwOThlZjc4YWUwNDhhY2IyZmEyOTYyMjBkNzYwZiIsInNlcV90eXBlIjoxLCJzZXFubyI6MywidGFnIjoic2lnbmF0dXJlIn2jc2lnxEBZPnISDDH3rPasRuEiblt2HHP587d05UMxqNZwloGuVmIR2+SZMGfRGWCuNz5ZXoa47+GSErv6QD7sVwB4h9wPqHNpZ190eXBlIKN0YWfNAgKndmVyc2lvbgE=\"},\"type\":\"sibkey\",\"version\":1},\"ctime\":1432144443,\"expire_in\":10000000,\"prev\":\"fafd9bbd0b7e905b96cb4b6c8022f7ee712098ef78ae048acb2fa296220d760f\",\"seq_type\":1,\"seqno\":3,\"tag\":\"signature\"}",
            "kid": "0120b0388608d728a9742ee7060b95c229089bf55658fb389de6987375a42f087fb70a",
            "ctime": 1432144443
        }
    ],
    "keys": [
        "0120b0388608d728a9742ee7060b95c229089bf55658fb389de6987375a42f087fb70a",
        "012015d09e9f3d13242c0b531228825b2762163fc4a72eca3439f91867734da305e20a"
    ],
    "uid": "74c38cf7ceb947f5632045d8ca5d4819",
    "username": "max32"
}
`,
	"bad_signature_chain.json": `{
    "chain": [
        {
            "seqno": 1,
            "prev": null,
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.10\nComment: https://keybase.io/crypto\n\nyMDyAnicbZG9axVBFMU3fkEeCMGIYBWcLriE+d6ZBwYFRaysLCQxj9mZO+uSl93N\n7r4kL/EVgtj4B2ir2FinEMRCRCvFYGchigSxsLKxC7q7wc5pZs65v3suw31z8mjQ\nm03O725/+bTJpt69ejsKbgwuzeygOHdj1N9Bq9BdPs0SKIsyzWrURwriSHBLY08l\njihgLZTwAFQLACV47LGVlhMUott51XY0MbGpYCHNG68Rg9Q17n/41a6ACWHaSSo4\ncAXESkUki7yWxmPNsQTBBaWWRwBaSUUxBmdjCk4bxb3jkYwdNk3cqIuLuGXK+shC\nrHnkhWQUc+GUNcJxRXQLVlBmZg0aes1sMYomIarHRath6KD5RIg2oKzSPEN90hRt\nnbY04YwyLDQTIYKtIi1hkLYEPjwhKkrYQP1sNByGqIL1wWEo6USWd6/aJM2YKk0y\nU49KQJPXK8eC2V5w4viRdhlBb3rm34o2f08Fd1/+ef7h5t6vp/OTa+P3V65eeHbv\n89yD/cUnZ/ZXitPLt4KDU0s/zj76ee5r/uLO/MWH673h3lJSP853P377foCu37/8\nF4rxoso=\n=Zv9D\n-----END PGP MESSAGE-----\n",
            "payload_hash": "eea2476670616f48f64531bcad5ed592e053671dae9aa8ce35550af03151e77a",
            "sig_id": "dfc1b98e0a9094f3f68b74f698b7a084006bb88cf26ca592c84eac49663e7fd40f",
            "payload_json": "{\"body\":{\"key\":{\"fingerprint\":\"8eb754c2bf26072e09585fee295ee854bf0c6c41\",\"host\":\"keybase.io\",\"key_id\":\"295ee854bf0c6c41\",\"kid\":\"01139d6254e48e1c681637f96af09406e54522c47ee9868200edcb2ed9a84fd476bd0a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"type\":\"eldest\",\"version\":1},\"ctime\":1432305935,\"expire_in\":10000000,\"prev\":null,\"seq_type\":1,\"seqno\":1,\"tag\":\"signature\"}",
            "kid": "01139d6254e48e1c681637f96af09406e54522c47ee9868200edcb2ed9a84fd476bd0a",
            "ctime": 1432305935
        },
        {
            "seqno": 2,
            "prev": "eea2476670616f48f64531bcad5ed592e053671dae9aa8ce35550af03151e77a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.10\nComment: https://keybase.io/crypto\n\nyMQhAnicrVU7jONEAM0dHBIrIR06CShXaRCXZeOxPf5EOgknsZPdxInzcbIJ0S1j\ne5xMsh5/87GPEy2cREWBkKBBSNAjIdEhPhUFHaK6ElEfBS1O9tA1VzLNaGbePL15\n1nv++ZUXCkd35qVvs8d/bLkbv/7CrQvmpXL7QdHynbRYeVBc4cOErxwcJ5cr4hQr\nRQYATnYEFvKYlzCwBQkInOjKAnIZmWcEDHnIsjYvYixLgsQyDHZsi8WOjCTedXhR\nsBwGFU+KLqFzHAURoUlOK2FLhLzNWi4rMCKLGRlK0MWYlSHGEuQtl7EFmwf5xYUf\n72/k4iwU41Pi53v54vIg7zn4/1n3+kAn8jYn2a5oY0vmRRcKHMvw0JFsBB1eAvIe\nGOOIIg/naA/tOLb48KQYE+upqc9UuVBE2OWB6LqAs5BrQwe7ruxYEmsBKOXM2AEy\ny7D5ieQyyIKOCCwRW5bA5Tj2oCrCGxzF+DIm85z27f2oqo2zzrHRMI51dTBQGuph\nd0ZHOZD4tHLcunbwuBtguodt2FPmFDAzWvM9D9OkcrxIkiCulMvPvC7bURok/ozO\naKprE4USO5qeiQj0q5rubFxVG899zZ+Mm4tJbdzOjPUuqDbKczVQY0MRlpkRdaZr\n0eM2m5zBR71d3G7UYb0xWqnxTrXUga/76nxORkG7kW76rLaO7KE9bDCe19BLTrm8\ndpue22FFWh/P6NghUuxiUxRFRpeTDeezGSpTrXRxsVp4aid3sa+vtpuGMvA9LoW1\nFmnqtaZeVVdaVRCqM4pb3vm05iM00NmmGvcutHq47p1tx17Y0nvDTUvpLzuTUrVV\nWtWSkLQnV03W7wejvrNajbegm7uFhYCAPun3VMs3TdVXJwrSNWenQWF41V6I46Yz\nIj1+yq55U8mG422Uthy+LBulKc5SMKOmvM1EZKS8srswzUVD7ztVnMY1IA2Y7SIM\nmRAqZlie1FDH9/lNuF3kjx3II00PalAdajM6OCcZM4kVfRrNO82RKnR7+rxPWsiO\nfNJihr1+g27r9Hzlw244CPkaMDt4xDEmkuz6FgW5k5bOoiB25W5KRleqPpB7yxb2\nLH9lLqnWaJyPESGskrZIdEYAqsqMdLGcnmf6vLHt6ECZ0RZJz5farpn2zGWWxPMz\nhQyN1XYbmmQ7aWWjZjDvKaHb18nZWatjEhDVkJh2+2ikdM2J5OWvSNZubzNts/w2\n6lUTq8PLqxrkYraj1cXh1KMOhUa2EUStzYYydUpiHfBJVo+wZCQrWZrR9jBKXSqk\n67KRCAHVciHZRbldHRqTJeY4bLgl2DE2YakbBoFtRrQEE7oEcsk1/FLIcTO6TLFB\nkwzW+5NEWDP3ZvSeCna5QYf0qJ36cxK1T3aSBvukPw34SXFznbJiBeSHdkL2PQB4\njuUYgeHgSRHvAhLhS7JHMNfjpBjkMc5JMEZsXjiCyAhAcHnJFXjIActGDsQOlPN2\nhJwgAgdhGeUfEHMQQibvMQ5AgEVx3wcxDi+vJYHDgvrFCpurRPODyDlFyTrCxYc/\n3n+xcOeo8NKtm/vqLxy9fPu/H8JPb94ofCxnDz599/PLQj6+Pvrqw2/eWP3wW124\n+/j9t+7yf43jG4Xvgn/okz/fe/L9l+88un/zM+x9cvX6R68+qvx967XTD34nX/wL\nbL0JeQ==\n=965F\n-----END PGP MESSAGE-----\n",
            "payload_hash": "3eafef1d80fc9e3c2d559bbc4a7b57522caaefc48d3a399cf9c2d7213284b0a7",
            "sig_id": "7b0b5f4fb66663c2290d78cddb1898236d6e065eb783e9aa85c8c67ac941e1a40f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01139d6254e48e1c681637f96af09406e54522c47ee9868200edcb2ed9a84fd476bd0a\",\"fingerprint\":\"8eb754c2bf26072e09585fee295ee854bf0c6c41\",\"host\":\"keybase.io\",\"key_id\":\"295ee854bf0c6c41\",\"kid\":\"01139d6254e48e1c681637f96af09406e54522c47ee9868200edcb2ed9a84fd476bd0a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"sibkey\":{\"kid\":\"0113f57aef417ff13bafc5deff9db82b158632ed19202fc58f0ab5d71b7ebb63ff920a\",\"reverse_sig\":\"-----BEGIN PGP MESSAGE-----\\nVersion: Keybase OpenPGP v2.0.10\\nComment: https://keybase.io/crypto\\n\\nyMFYAnicrZI7a1RBFMdvfEFWgoFoYWHhYCWLzPuxpBG/gEpEsPA6jzPrNZu7m3vv\\nyoaQxsLGD5DGVkEsxEbESoMoEggiVpLGyvR2FurcTcTG0mmGM+d//ufHmfN27nDW\\nWdi8sfeU7770M9tv3o2za/nF+XXkhmEN9dbRMkwvGASom3y5CKiHMCHMBEkFB66B\\neKmJZCoaaSM2HEsQXFDquQIwWmqKMQTvKARjNY+BK+kCtqiLYlH2oRpVRdkkWw1O\\nCe6pi1RiRQEboUUEoEYAaMFdxF56TlLh7WHdViQ4Z2u4UAzTWwryKd4/9P+Zezy1\\nU9wz7aPy4AxXUUhGMRdBeysC18S0whqq0q5AUq/YCaNoo4vqwh0M9S9VFMpC5ETF\\nSJiz0YsAMZrgNHVE6OQMgRiKacroiK0TQRGnwDnJko5OqSq4C1UNeV30Ua8cDwap\\nWbM2apsf9OyiVlEMS9QjKembokUjnFGGJWaii2AyKirIi1aB908XjZJzMgGwNM1A\\nKiyJjFxHyQUjztsgIAiTPkwwqUiwYKzVHpgQAqfRMiIIKNUi1rCa7yORaVAOUY8m\\nStufQvZL24wrQBtbN49kC53s2NFD7TZmndn5Pzv67FL2q9nd+7D14tzDre8Ptk98\\nLTryfn6yu/Pt6pnFjztzX/LBTPYje33ePf+5NPvq+OqppcUrn+5tnj19+fPo+q33\\njyePntz5DRYt6u0=\\n=E1xp\\n-----END PGP MESSAGE-----\\n\"},\"type\":\"sibkey\",\"version\":1},\"ctime\":1432306035,\"expire_in\":10000000,\"prev\":\"eea2476670616f48f64531bcad5ed592e053671dae9aa8ce35550af03151e77a\",\"seq_type\":1,\"seqno\":2,\"tag\":\"signature\"}",
            "kid": "01139d6254e48e1c681637f96af09406e54522c47ee9868200edcb2ed9a84fd476bd0a",
            "ctime": 1432306035
        }
    ],
    "keys": [
        "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: Keybase OpenPGP v2.0.10\nComment: https://keybase.io/crypto\n\nxm8EVV9BDxMFK4EEACIDAwRSzcsRfEMpwQ7RvwWHPxXf97lwjf8mqqCeTZXkntaK\nLAYBBi4ZH2dPpVL2Nk6Bh0K7Zc9II0ksc0BL+z0fJ/3hIDhq9NgfusjqjiX8NYZ7\ndbeT+gyDPp5gzXaxPqrF0vXNBW1heDMywpIEExMKABoFAlVfQQ8CGwEDCwkHAxUK\nCAIeAQIXgAIZAQAKCRApXuhUvwxsQUGxAX9Y1nv+tbCxZGE4P22kbVQbi4BTyIkL\nYgNKqeoOlNxuWeDdh0xaDsimsTpEKWtSagMBgK179Xs4gsE9jPBtHGuzYpbHbROS\nhVCq9ssZjvId45D2UHFSjyAm8spEeFJzLARqEs5WBFVfQQ8SCCqGSM49AwEHAgME\nntY21MrNuVBiA28QJRFx/+nw8O6URDuF7P+a1Ou+c/mzeH8bH9NB+fozm+wt0+kU\nsAQ1rrmAdK9oXfbxFHw+VgMBCgnChwQYEwoADwUCVV9BDwUJDwmcAAIbDAAKCRAp\nXuhUvwxsQYFyAYCuu59mWoB68DUaSFY1YoIjXt10oKkJyqeaF/MDKCs4RZvnLkcM\nVfm4ANHRx8P4jTwBgKF4bBRJYKbGCh9Sdve3ivaihIjKueXbIkIwzlHKnDhH0ryF\nZy7AuYErrAMqZiEyG85SBFVfQQ8TCCqGSM49AwEHAgME2hMvKEfCqKxsX7+B70Lq\nfOOQg3mAP5vNEE9fP/O4CHS3nG7DRv4Di1vZlHE7u2OHAfHSAFq3ir935z4C0d7B\ncsLAJwQYEwoADwUCVV9BDwUJDwmcAAIbIgBqCRApXuhUvwxsQV8gBBkTCgAGBQJV\nX0EPAAoJEGcrs3re2HczDekBAJOT5B0avXmVYnPGkTwVZ3oeoXNwQEpYOm/kdPEx\nBFVhAQCKGjWboi7vVftw1cU/IFp1uh7lYWEzDlB9yvUCyd7/INoFAYC9wV1R++yE\nroIfq9hkzttJnvmuobSYaJHLMIQXnIADOBLHbHxrKInDtYqIToUoYsoBf2CMDj0r\nVwx1bGn7KprTMnu89hv/rIjnAcDYorsmqfECoXXZNBrVHS/DBpDROPPpag==\n=wdBi\n-----END PGP PUBLIC KEY BLOCK-----\n",
        "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: Keybase OpenPGP v2.0.10\nComment: https://keybase.io/crypto\n\nxm8EVV9BEBMFK4EEACIDAwS9o76Gzhjx/aRI+8EdNO7F011t/m6clC6VHpKASpc/\nmnygwK6qvxou750c6WgrBlFtnskjjpJcY2zqkgSKi+MKtpt6PJwWfQ1CR6aJBqNe\nSJyJVWS0yLuvf94v8u+e2c/NBW1heDMywpIEExMKABoFAlVfQRACGwEDCwkHAxUK\nCAIeAQIXgAIZAQAKCRCRTHSn/50zJ11MAYDI+TPMhrTb73FWiUxTkRvehyfWrLzk\nSFQS0RZpujJpzaaI2tE8Q/g5KU2oLEZpvdkBgPubVOmKL0LSWaaQzGaPrv7pR5+D\naGjOMf1VlUYSprgDiHrPnft1EiAlUsWsfX8yKs5WBFVfQRASCCqGSM49AwEHAgME\n77H3BHjofaqD5KYjvEkq9SAxc9i8VBuSrWplGelgeG/Ctr9UIvindI7uoRhHqAVa\nbjTFylO+CZy88cKdsRcj3wMBCgnChwQYEwoADwUCVV9BEAUJDwmcAAIbDAAKCRCR\nTHSn/50zJ+KDAX9nCQ75y87n0onZqDR3N+CC8QNvj6z+oaxvEh9w8AoF5auU9AQn\ny+/L5h5TNuQ/UkwBgLhredcA4cJ1+iBUOIQKJituuVYwvZ5wRFzRw6O7bs9XrbQZ\nZhki4WY7K64PuLvSQs5SBFVfQRATCCqGSM49AwEHAgMErT7N/XUzhm0YrnBF5uCk\nl9UIbLoiN4Zv8Vfj+KRJ8cWxQZVGKUd1lbTSQpE7js9adGEGqloqCEaxPaAYWtor\nQcLAJwQYEwoADwUCVV9BEAUJDwmcAAIbIgBqCRCRTHSn/50zJ18gBBkTCgAGBQJV\nX0EQAAoJEJda6Ks03LhjKjcBAOc8qE3v+yh8buW/vgeHbxTZRQLKaJoGSHC+TvgZ\nCe9/APwPIo89pm4CLE5vukuv0rSV3kQUF/LLkAtG2kwcvIUGO/2kAX9+aHMa/KlU\nLR4NJzhprqbXG9hWqR/R0DJLCm3HrP/J7v2glQ/yRWrAqdaYvxhASZoBfiTH1x2U\nL1C2MzFN8F/M8MQiobo5Kp4O/jLvj7leBQk/mXgaDAZrnAulV0cYn9xgfA==\n=PeLw\n-----END PGP PUBLIC KEY BLOCK-----\n"
    ],
    "uid": "74c38cf7ceb947f5632045d8ca5d4819",
    "username": "max32"
}
`,
	"bad_uid_chain.json": `{
    "chain": [
        {
            "seqno": 1,
            "prev": null,
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgj8Et28Wy41nCNQ2UVwC2dpyeKou5x1Z51AtO17SyjGsKp3BheWxvYWTFASp7ImJvZHkiOnsia2V5Ijp7Imhvc3QiOiJrZXliYXNlLmlvIiwia2lkIjoiMDEyMDhmYzEyZGRiYzViMmUzNTljMjM1MGQ5NDU3MDBiNjc2OWM5ZTJhOGJiOWM3NTY3OWQ0MGI0ZWQ3YjRiMjhjNmIwYSIsInVpZCI6ImRlYWRiZWVmMDAwMDAwMDAwMDAwMDAwMDAwMDAwMDAwIiwidXNlcm5hbWUiOiJtYXgzMiJ9LCJ0eXBlIjoiZWxkZXN0IiwidmVyc2lvbiI6MX0sImN0aW1lIjoxNDMyMTQ0NTk1LCJleHBpcmVfaW4iOjEwMDAwMDAwLCJwcmV2IjpudWxsLCJzZXFfdHlwZSI6MSwic2Vxbm8iOjEsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RAc4WxvRU1roBEB1HStIiE2d48q4cOLgHlYSu9sGawOYOc4nvDvrzPKq72jrTgjSJFOuwa9t0bQDpXMTD1/s/XA6hzaWdfdHlwZSCjdGFnzQICp3ZlcnNpb24B",
            "payload_hash": "4ffdc8466a0ffa380da8c4ef6477cc80e3a80734eda97d148f659bc0410c7344",
            "sig_id": "84083329f6ebe6fc7157cb7be43a6ea7526db482416949d1f54eb5159eb8c4890f",
            "payload_json": "{\"body\":{\"key\":{\"host\":\"keybase.io\",\"kid\":\"01208fc12ddbc5b2e359c2350d945700b6769c9e2a8bb9c75679d40b4ed7b4b28c6b0a\",\"uid\":\"deadbeef000000000000000000000000\",\"username\":\"max32\"},\"type\":\"eldest\",\"version\":1},\"ctime\":1432144595,\"expire_in\":10000000,\"prev\":null,\"seq_type\":1,\"seqno\":1,\"tag\":\"signature\"}",
            "kid": "01208fc12ddbc5b2e359c2350d945700b6769c9e2a8bb9c75679d40b4ed7b4b28c6b0a",
            "ctime": 1432144595
        }
    ],
    "keys": [
        "01208fc12ddbc5b2e359c2350d945700b6769c9e2a8bb9c75679d40b4ed7b4b28c6b0a"
    ],
    "uid": "74c38cf7ceb947f5632045d8ca5d4819",
    "username": "max32"
}
`,
	"bad_username_chain.json": `{
    "chain": [
        {
            "seqno": 1,
            "prev": null,
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEguI0gTYv5jJi0OvFbTgAsh7xXonZn2UXGOIZlClWJN/oKp3BheWxvYWTFATF7ImJvZHkiOnsia2V5Ijp7Imhvc3QiOiJrZXliYXNlLmlvIiwia2lkIjoiMDEyMGI4OGQyMDRkOGJmOThjOThiNDNhZjE1YjRlMDAyYzg3YmM1N2EyNzY2N2Q5NDVjNjM4ODY2NTBhNTU4OTM3ZmEwYSIsInVpZCI6Ijc0YzM4Y2Y3Y2ViOTQ3ZjU2MzIwNDVkOGNhNWQ0ODE5IiwidXNlcm5hbWUiOiJnYXJiYWdlX3VzZXIifSwidHlwZSI6ImVsZGVzdCIsInZlcnNpb24iOjF9LCJjdGltZSI6MTQzMjE0NDY0MSwiZXhwaXJlX2luIjoxMDAwMDAwMCwicHJldiI6bnVsbCwic2VxX3R5cGUiOjEsInNlcW5vIjoxLCJ0YWciOiJzaWduYXR1cmUifaNzaWfEQG0IAj1rTzCBW2H64Hh66oer8vzOAmiL70khO1/49o8hL/QjXRUm8/8aabGrHmESu7FC+TU6oyvO1sH++bK2LgSoc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
            "payload_hash": "a867b0e83da675b6076e3ba7e5f34e2020ea96836fba50e6e1a4092c9b653aac",
            "sig_id": "a9d87f33ed10c1295bedb5b7369b9af11bbb69fc38872b8a7a85cb660049dc480f",
            "payload_json": "{\"body\":{\"key\":{\"host\":\"keybase.io\",\"kid\":\"0120b88d204d8bf98c98b43af15b4e002c87bc57a27667d945c63886650a558937fa0a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"garbage_user\"},\"type\":\"eldest\",\"version\":1},\"ctime\":1432144641,\"expire_in\":10000000,\"prev\":null,\"seq_type\":1,\"seqno\":1,\"tag\":\"signature\"}",
            "kid": "0120b88d204d8bf98c98b43af15b4e002c87bc57a27667d945c63886650a558937fa0a",
            "ctime": 1432144641
        }
    ],
    "keys": [
        "0120b88d204d8bf98c98b43af15b4e002c87bc57a27667d945c63886650a558937fa0a"
    ],
    "uid": "74c38cf7ceb947f5632045d8ca5d4819",
    "username": "max32"
}
`,
	"bad_v2_delegate_stubbed.json": `{
   "label_kids" : {
      "sub1" : "0121a460af9d60e548128afce00ab99097eb4884c44aa5e0c221e3b7a2622ffa9e570a",
      "sib1" : "012033f3c773fdf54105857562b78684e6ca814e276ace25ac768cdb65eb1a61b3810a",
      "e" : "0120ed4fcdcb4ed303bd1e4ea34f3406c7b2888c5b3da002dc3baad3afd222a119cb0a",
      "s" : "0121ca3899b58e809c6a338441c88a8d37cd2f8293c776c498646e9066152168ae450a"
   },
   "label_sigs" : {
      "sib1" : "985d24474d62729168f998b92b18ec9c2cc3218e6c9d7ed0efa90a8a2bfd8fa90f",
      "sub1" : "93c71047fa768248b69debba883fb2c291f86e02bc067691ba7cdc545dea4d740f",
      "e" : "b448021e826ce42b5af482c910eb085257794339b44c3f151257cd17606da6c60f",
      "s" : "dc2349451b3c1d2660525c37c7ad40d774268ef25c29d1a529d32e552799d9f70f"
   },
   "uid" : "74c38cf7ceb947f5632045d8ca5d4819",
   "username" : "max32",
   "keys" : [
      "0120ed4fcdcb4ed303bd1e4ea34f3406c7b2888c5b3da002dc3baad3afd222a119cb0a",
      "0121ca3899b58e809c6a338441c88a8d37cd2f8293c776c498646e9066152168ae450a",
      "012033f3c773fdf54105857562b78684e6ca814e276ace25ac768cdb65eb1a61b3810a",
      "0121a460af9d60e548128afce00ab99097eb4884c44aa5e0c221e3b7a2622ffa9e570a"
   ],
   "chain" : [
      {
         "payload_json" : "{\"body\":{\"key\":{\"host\":\"keybase.io\",\"kid\":\"0120ed4fcdcb4ed303bd1e4ea34f3406c7b2888c5b3da002dc3baad3afd222a119cb0a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"type\":\"eldest\",\"version\":2},\"ctime\":1493669692,\"expire_in\":314496000,\"prev\":null,\"seq_type\":1,\"seqno\":1,\"tag\":\"signature\"}",
         "sig_id" : "b448021e826ce42b5af482c910eb085257794339b44c3f151257cd17606da6c60f",
         "sig" : "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEg7U/Ny07TA70eTqNPNAbHsoiMWz2gAtw7qtOv0iKhGcsKp3BheWxvYWTEJ5UCAcDEIGgVpivS+FAPyo9jqv+loWxVyKEiQPUW4tiamDSWZQhCAaNzaWfEQAy0X4yYPDZ4fG9nQubIi3Gkg/nIIuBm+x7RJUHcYHu967yLZpPUeqD9nW8ALSFL8NzIE3Q50+yGWZ+jhsz5Dgyoc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
         "kid" : "0120ed4fcdcb4ed303bd1e4ea34f3406c7b2888c5b3da002dc3baad3afd222a119cb0a",
         "payload_hash" : "c7f89231c9291d04f63223ed5e99ef07b7095d3391fb384a15ed40d4cd487a28",
         "ctime" : 1493669692,
         "sig_version" : 2,
         "prev" : null,
         "seqno" : 1
      },
      {
         "kid" : "0120ed4fcdcb4ed303bd1e4ea34f3406c7b2888c5b3da002dc3baad3afd222a119cb0a",
         "payload_hash" : "3813619f2fc29ff1855fe1010fd72615db3fceb374ab3eaf78c328714589e61f",
         "sig_version" : 2,
         "ctime" : 1493669692,
         "prev" : "c7f89231c9291d04f63223ed5e99ef07b7095d3391fb384a15ed40d4cd487a28",
         "seqno" : 2,
         "sig_id" : "dc2349451b3c1d2660525c37c7ad40d774268ef25c29d1a529d32e552799d9f70f",
         "sig" : "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEg7U/Ny07TA70eTqNPNAbHsoiMWz2gAtw7qtOv0iKhGcsKp3BheWxvYWTESJUCAsQgx/iSMckpHQT2MiPtXpnvB7cJXTOR+zhKFe1A1M1IeijEIAs8D4xngmqA58/z3ZxP2FbJr7jqER86PGwbMRxfG3q8DKNzaWfEQK7Wu3qhdg3K/1Fawp60t0p/pj4tfqIkU7OwhLpycVsxxJTzvkEkWRVQBT0Ycsx2aKfXDabmiqsCUhwDZpZUgAeoc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
         "payload_json" : "{\"body\":{\"key\":{\"eldest_kid\":\"0120ed4fcdcb4ed303bd1e4ea34f3406c7b2888c5b3da002dc3baad3afd222a119cb0a\",\"host\":\"keybase.io\",\"kid\":\"0120ed4fcdcb4ed303bd1e4ea34f3406c7b2888c5b3da002dc3baad3afd222a119cb0a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"subkey\":{\"kid\":\"0121ca3899b58e809c6a338441c88a8d37cd2f8293c776c498646e9066152168ae450a\",\"parent_kid\":\"0120ed4fcdcb4ed303bd1e4ea34f3406c7b2888c5b3da002dc3baad3afd222a119cb0a\",\"reverse_sig\":null},\"type\":\"subkey\",\"version\":2},\"ctime\":1493669692,\"expire_in\":314496000,\"prev\":\"c7f89231c9291d04f63223ed5e99ef07b7095d3391fb384a15ed40d4cd487a28\",\"seq_type\":1,\"seqno\":2,\"tag\":\"signature\"}"
      },
      {
         "s2" : "lQIDxCA4E2GfL8Kf8YVf4QEP1yYV2z/Os3SrPq94wyhxRYnmH8QgNBjIyirea6AqC33zv8Yz0ewdj0TY30Kd4sh4piP/KEIL"
      },
      {
         "seqno" : 4,
         "prev" : "387f7e0542174618a75a3e236e2f5717a7350a6cf6eb4a143fff3df2b24497cb",
         "ctime" : 1493669792,
         "sig_version" : 2,
         "kid" : "012033f3c773fdf54105857562b78684e6ca814e276ace25ac768cdb65eb1a61b3810a",
         "payload_hash" : "9987c36c699cbe48285cdd7fcea0c794af908a04835809afa22ddc310d041b67",
         "payload_json" : "{\"body\":{\"key\":{\"eldest_kid\":\"0120ed4fcdcb4ed303bd1e4ea34f3406c7b2888c5b3da002dc3baad3afd222a119cb0a\",\"host\":\"keybase.io\",\"kid\":\"012033f3c773fdf54105857562b78684e6ca814e276ace25ac768cdb65eb1a61b3810a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"subkey\":{\"kid\":\"0121a460af9d60e548128afce00ab99097eb4884c44aa5e0c221e3b7a2622ffa9e570a\",\"parent_kid\":\"012033f3c773fdf54105857562b78684e6ca814e276ace25ac768cdb65eb1a61b3810a\",\"reverse_sig\":null},\"type\":\"subkey\",\"version\":2},\"ctime\":1493669792,\"expire_in\":314496000,\"prev\":\"387f7e0542174618a75a3e236e2f5717a7350a6cf6eb4a143fff3df2b24497cb\",\"seq_type\":1,\"seqno\":4,\"tag\":\"signature\"}",
         "sig" : "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgM/PHc/31QQWFdWK3hoTmyoFOJ2rOJax2jNtl6xphs4EKp3BheWxvYWTESJUCBMQgOH9+BUIXRhinWj4jbi9XF6c1Cmz260oUP/898rJEl8vEIEkLoFlS0XxJoA67Z/0NcZ/vyUaZWDwch/uTGXx6Ih4UDKNzaWfEQI9WIJKmbpvFH4Yq1a2O9uY1++6O3LqdCajhnZAy8n51vp/BcUlHhbX71VF6OhCOiI8Jf+4wvplicEnmCiQIog6oc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
         "sig_id" : "93c71047fa768248b69debba883fb2c291f86e02bc067691ba7cdc545dea4d740f"
      }
   ]
}
`,
	"bad_v2_first_stubbed.json": `{
   "username" : "max32",
   "label_kids" : {
      "s" : "01211b9374de4f4380cd5b3af432e96f87b8f67852765705e2be4432e7c6fbadd0670a",
      "sib1" : "0120bafd7607838fea8bd6b46ebc6e75178c4f6feee67cbd947dc5698eb79de969820a",
      "e" : "01206f8cf0ed707e47cc4e727697b4e81d542ca77644cf5d40cc640ba414d44abfa30a",
      "sub1" : "0121ba05726b34fb062ce0de96480a79f93b89497d5c12232c6e389683f3c526c61a0a"
   },
   "keys" : [
      "01206f8cf0ed707e47cc4e727697b4e81d542ca77644cf5d40cc640ba414d44abfa30a",
      "01211b9374de4f4380cd5b3af432e96f87b8f67852765705e2be4432e7c6fbadd0670a",
      "0120bafd7607838fea8bd6b46ebc6e75178c4f6feee67cbd947dc5698eb79de969820a",
      "0121ba05726b34fb062ce0de96480a79f93b89497d5c12232c6e389683f3c526c61a0a"
   ],
   "uid" : "74c38cf7ceb947f5632045d8ca5d4819",
   "label_sigs" : {
      "s" : "61e2e7fc7e2803734794e0c2e25e7906dca897c1c4cc1624a795226f6086e0ea0f",
      "sib1" : "a9305b8ec3d46b1fa2a18bdce374ada7b1767f9dcc2704d30cc0614c97fdccd80f",
      "e" : "fc83ca570861f4492b8fd65d069346e64610961a8b3c66b84bd49d1f8fb1607f0f",
      "sub1" : "256045878dd1938dd76680403630335a1b99891062906496f6650b5b3fb15f680f"
   },
   "chain" : [
      {
         "s2" : "lQIBwMQgIkNRU86wpNSh8fNJOjaq+KpqqpXQhZFwLqzk9UU4SvcB"
      },
      {
         "ctime" : 1493674260,
         "payload_json" : "{\"body\":{\"key\":{\"eldest_kid\":\"01206f8cf0ed707e47cc4e727697b4e81d542ca77644cf5d40cc640ba414d44abfa30a\",\"host\":\"keybase.io\",\"kid\":\"01206f8cf0ed707e47cc4e727697b4e81d542ca77644cf5d40cc640ba414d44abfa30a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"subkey\":{\"kid\":\"01211b9374de4f4380cd5b3af432e96f87b8f67852765705e2be4432e7c6fbadd0670a\",\"parent_kid\":\"01206f8cf0ed707e47cc4e727697b4e81d542ca77644cf5d40cc640ba414d44abfa30a\",\"reverse_sig\":null},\"type\":\"subkey\",\"version\":2},\"ctime\":1493674260,\"expire_in\":314496000,\"prev\":\"63f98572c1a8de89c9cfbf4dfff46344ca7b7b753b343de1358470eed7a047c4\",\"seq_type\":1,\"seqno\":2,\"tag\":\"signature\"}",
         "prev" : "63f98572c1a8de89c9cfbf4dfff46344ca7b7b753b343de1358470eed7a047c4",
         "seqno" : 2,
         "payload_hash" : "1678fe82cbbfe28404fda4ff2755d6d2d753d6c091d8b5a1c3c324fd1370a97f",
         "sig_id" : "61e2e7fc7e2803734794e0c2e25e7906dca897c1c4cc1624a795226f6086e0ea0f",
         "sig" : "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgb4zw7XB+R8xOcnaXtOgdVCyndkTPXUDMZAukFNRKv6MKp3BheWxvYWTESJUCAsQgY/mFcsGo3onJz79N//RjRMp7e3U7ND3hNYRw7tegR8TEICrqgc48NedEHZENvCOukjA26COJ4WOsQHdbOEmjlfv1DKNzaWfEQD3xhmFACAFb6TPnYe7h/2hzSJ5eSpY2OtySb9Tst3o2lgsQtbXL91iYJj817kk8yVF7pmLOfvGXjDSxdm5ufgyoc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
         "kid" : "01206f8cf0ed707e47cc4e727697b4e81d542ca77644cf5d40cc640ba414d44abfa30a",
         "sig_version" : 2
      },
      {
         "ctime" : 1493674360,
         "payload_json" : "{\"body\":{\"key\":{\"eldest_kid\":\"01206f8cf0ed707e47cc4e727697b4e81d542ca77644cf5d40cc640ba414d44abfa30a\",\"host\":\"keybase.io\",\"kid\":\"01206f8cf0ed707e47cc4e727697b4e81d542ca77644cf5d40cc640ba414d44abfa30a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"sibkey\":{\"kid\":\"0120bafd7607838fea8bd6b46ebc6e75178c4f6feee67cbd947dc5698eb79de969820a\",\"reverse_sig\":\"g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEguv12B4OP6ovWtG68bnUXjE9v7uZ8vZR9xWmOt53paYIKp3BheWxvYWTFAix7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwNmY4Y2YwZWQ3MDdlNDdjYzRlNzI3Njk3YjRlODFkNTQyY2E3NzY0NGNmNWQ0MGNjNjQwYmE0MTRkNDRhYmZhMzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwNmY4Y2YwZWQ3MDdlNDdjYzRlNzI3Njk3YjRlODFkNTQyY2E3NzY0NGNmNWQ0MGNjNjQwYmE0MTRkNDRhYmZhMzBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInNpYmtleSI6eyJraWQiOiIwMTIwYmFmZDc2MDc4MzhmZWE4YmQ2YjQ2ZWJjNmU3NTE3OGM0ZjZmZWVlNjdjYmQ5NDdkYzU2OThlYjc5ZGU5Njk4MjBhIiwicmV2ZXJzZV9zaWciOm51bGx9LCJ0eXBlIjoic2lia2V5IiwidmVyc2lvbiI6Mn0sImN0aW1lIjoxNDkzNjc0MzYwLCJleHBpcmVfaW4iOjMxNDQ5NjAwMCwicHJldiI6IjE2NzhmZTgyY2JiZmUyODQwNGZkYTRmZjI3NTVkNmQyZDc1M2Q2YzA5MWQ4YjVhMWMzYzMyNGZkMTM3MGE5N2YiLCJzZXFfdHlwZSI6MSwic2Vxbm8iOjMsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RAg4gr5IVy/O3GzLTbh5+zKIrA5VDAzhQHugOCt3d/3k2Y6+EIiVDxUI5guYIdsRu/35i9EDE/nwWehgVvG1qjA6hzaWdfdHlwZSCjdGFnzQICp3ZlcnNpb24B\"},\"type\":\"sibkey\",\"version\":2},\"ctime\":1493674360,\"expire_in\":314496000,\"prev\":\"1678fe82cbbfe28404fda4ff2755d6d2d753d6c091d8b5a1c3c324fd1370a97f\",\"seq_type\":1,\"seqno\":3,\"tag\":\"signature\"}",
         "prev" : "1678fe82cbbfe28404fda4ff2755d6d2d753d6c091d8b5a1c3c324fd1370a97f",
         "seqno" : 3,
         "payload_hash" : "83f93d26da88266fe885c252db605a82e5bab7b4d200191980c30f89416039a7",
         "sig" : "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgb4zw7XB+R8xOcnaXtOgdVCyndkTPXUDMZAukFNRKv6MKp3BheWxvYWTESJUCA8QgFnj+gsu/4oQE/aT/J1XW0tdT1sCR2LWhw8Mk/RNwqX/EIJUBhGHSGIkejYEunZ5iQ8MgtwsDXSlJStPKlHJqnTheC6NzaWfEQJYZqbhk110p3ZaI0GwI0U0/ioY339ZC0sYlGXqy4DSurRaaCAVknVKtc3JO95laFW+5dmm9aYYYDC5OQd0agwWoc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
         "sig_id" : "a9305b8ec3d46b1fa2a18bdce374ada7b1767f9dcc2704d30cc0614c97fdccd80f",
         "kid" : "01206f8cf0ed707e47cc4e727697b4e81d542ca77644cf5d40cc640ba414d44abfa30a",
         "sig_version" : 2
      },
      {
         "sig_version" : 2,
         "sig_id" : "256045878dd1938dd76680403630335a1b99891062906496f6650b5b3fb15f680f",
         "sig" : "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEguv12B4OP6ovWtG68bnUXjE9v7uZ8vZR9xWmOt53paYIKp3BheWxvYWTESJUCBMQgg/k9JtqIJm/ohcJS22BaguW6t7TSABkZgMMPiUFgOafEIF8nubQSx8nd3GJpkKMN45R+3azJF52gczzs5Fl0uZmZDKNzaWfEQCCQYBVNmM1Ll9aAWfw45Y5uGt0x4QvnFyVtH45034Cr3i3Hq4uQfduMPDqDwWVi/sxnrD8MilJKJhJCiv0tfgaoc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
         "kid" : "0120bafd7607838fea8bd6b46ebc6e75178c4f6feee67cbd947dc5698eb79de969820a",
         "prev" : "83f93d26da88266fe885c252db605a82e5bab7b4d200191980c30f89416039a7",
         "seqno" : 4,
         "payload_hash" : "41dc3fad37587f53b306feff14b0a5aa7667b2d2c8ebb48375882d39bc7584e8",
         "ctime" : 1493674360,
         "payload_json" : "{\"body\":{\"key\":{\"eldest_kid\":\"01206f8cf0ed707e47cc4e727697b4e81d542ca77644cf5d40cc640ba414d44abfa30a\",\"host\":\"keybase.io\",\"kid\":\"0120bafd7607838fea8bd6b46ebc6e75178c4f6feee67cbd947dc5698eb79de969820a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"subkey\":{\"kid\":\"0121ba05726b34fb062ce0de96480a79f93b89497d5c12232c6e389683f3c526c61a0a\",\"parent_kid\":\"0120bafd7607838fea8bd6b46ebc6e75178c4f6feee67cbd947dc5698eb79de969820a\",\"reverse_sig\":null},\"type\":\"subkey\",\"version\":2},\"ctime\":1493674360,\"expire_in\":314496000,\"prev\":\"83f93d26da88266fe885c252db605a82e5bab7b4d200191980c30f89416039a7\",\"seq_type\":1,\"seqno\":4,\"tag\":\"signature\"}"
      }
   ]
}
`,
	"bad_v2_hash_mismatch.json": `{
   "keys" : [
      "0120f1e65e8006957c53567a5b0f2136405a58eb3bc29f3ca2693493779a4c28ac5f0a",
      "012003e0021116f9aaacd913d806506dc33fd558c5122abe6719761fc5b3238218db0a"
   ],
   "chain" : [
      {
         "sig_id" : "865ee40499f7021c413cda0e1b59bd5bf1addb40a250a3b778271e5afafd66990f",
         "seqno" : 1,
         "payload_json" : "{\"body\":{\"key\":{\"host\":\"keybase.io\",\"kid\":\"0120f1e65e8006957c53567a5b0f2136405a58eb3bc29f3ca2693493779a4c28ac5f0a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"type\":\"eldest\",\"version\":2},\"ctime\":1493674326,\"expire_in\":314496000,\"prev\":null,\"seq_type\":1,\"seqno\":1,\"tag\":\"signature\"}",
         "ctime" : 1493674326,
         "prev" : null,
         "sig_version" : 2,
         "kid" : "0120f1e65e8006957c53567a5b0f2136405a58eb3bc29f3ca2693493779a4c28ac5f0a",
         "payload_hash" : "831e96475dbbbaf017290340815c8c2ceb5e2ec0d6210048d6fe06a13e2a253b",
         "sig" : "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEg8eZegAaVfFNWelsPITZAWljrO8KfPKJpNJN3mkworF8Kp3BheWxvYWTEJ5UCAcDEIDOIBBS93jv9V7cur2cZnNAOkD8qMkCuHBVEbxZOdh6vAaNzaWfEQCLzgJcN7nhpXVHAzd/AXYQQdipWNktSh0RQ/45iC0/XYU/+FKKQFZYhRW9bCqM2IdPh5pfvhPVzJUng+0RO8gKoc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ=="
      },
      {
         "prev" : "831e96475dbbbaf017290340815c8c2ceb5e2ec0d6210048d6fe06a13e2a253b",
         "ctime" : 1493674426,
         "seqno" : 2,
         "payload_json" : "{\"body\":{\"key\":{\"eldest_kid\":\"0120f1e65e8006957c53567a5b0f2136405a58eb3bc29f3ca2693493779a4c28ac5f0a\",\"host\":\"keybase.io\",\"kid\":\"0120f1e65e8006957c53567a5b0f2136405a58eb3bc29f3ca2693493779a4c28ac5f0a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"sibkey\":{\"kid\":\"012003e0021116f9aaacd913d806506dc33fd558c5122abe6719761fc5b3238218db0a\",\"reverse_sig\":\"g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgA+ACERb5qqzZE9gGUG3DP9VYxRIqvmcZdh/FsyOCGNsKp3BheWxvYWTFAix7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwZjFlNjVlODAwNjk1N2M1MzU2N2E1YjBmMjEzNjQwNWE1OGViM2JjMjlmM2NhMjY5MzQ5Mzc3OWE0YzI4YWM1ZjBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwZjFlNjVlODAwNjk1N2M1MzU2N2E1YjBmMjEzNjQwNWE1OGViM2JjMjlmM2NhMjY5MzQ5Mzc3OWE0YzI4YWM1ZjBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInNpYmtleSI6eyJraWQiOiIwMTIwMDNlMDAyMTExNmY5YWFhY2Q5MTNkODA2NTA2ZGMzM2ZkNTU4YzUxMjJhYmU2NzE5NzYxZmM1YjMyMzgyMThkYjBhIiwicmV2ZXJzZV9zaWciOm51bGx9LCJ0eXBlIjoic2lia2V5IiwidmVyc2lvbiI6MX0sImN0aW1lIjoxNDkzNjc0NDI2LCJleHBpcmVfaW4iOjMxNDQ5NjAwMCwicHJldiI6IjgzMWU5NjQ3NWRiYmJhZjAxNzI5MDM0MDgxNWM4YzJjZWI1ZTJlYzBkNjIxMDA0OGQ2ZmUwNmExM2UyYTI1M2IiLCJzZXFfdHlwZSI6MSwic2Vxbm8iOjIsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RAq+JJOmoCDPSpj6Rb4uRQEZIJEfNSQp80PTNyXKgoOa/5fN6AQIcTWxHEsFqyrKJo2tPTMPO1bzef9hItEXeoA6hzaWdfdHlwZSCjdGFnzQICp3ZlcnNpb24B\"},\"type\":\"sibkey\",\"version\":2},\"ctime\":1493674426,\"expire_in\":314496000,\"prev\":\"831e96475dbbbaf017290340815c8c2ceb5e2ec0d6210048d6fe06a13e2a253b\",\"seq_type\":1,\"seqno\":2,\"tag\":\"signature\"}",
         "sig_id" : "184dd796f2fc0ccca4db64a41e72c822e50c20a6c13032a22baa84a065513ff00f",
         "sig" : "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEg8eZegAaVfFNWelsPITZAWljrO8KfPKJpNJN3mkworF8Kp3BheWxvYWTESJUCAsQggx6WR127uvAXKQNAgVyMLOteLsDWIQBI1v4GoT4qJTvEIMJZNJl6FbjovMlv9gKkhy5rZjxuFvqoV4A98J28UP8IC6NzaWfEQCbJg+0j961sIxxoZ0XhmeXsqE8xMpnqQNur/EuDXFO4TBuTvkbA+OAcJ+TRH8X8xFmoAOh56lYv3TGQAgWW3Aeoc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
         "payload_hash" : "08ff798749c0c0a7cd7018b644ac38d1fa5f95ff1901f5131d2cffab074fa08b",
         "kid" : "0120f1e65e8006957c53567a5b0f2136405a58eb3bc29f3ca2693493779a4c28ac5f0a",
         "sig_version" : 2
      }
   ],
   "label_kids" : {
      "e" : "0120f1e65e8006957c53567a5b0f2136405a58eb3bc29f3ca2693493779a4c28ac5f0a",
      "sib1" : "012003e0021116f9aaacd913d806506dc33fd558c5122abe6719761fc5b3238218db0a"
   },
   "uid" : "74c38cf7ceb947f5632045d8ca5d4819",
   "label_sigs" : {
      "e" : "865ee40499f7021c413cda0e1b59bd5bf1addb40a250a3b778271e5afafd66990f",
      "sib1" : "184dd796f2fc0ccca4db64a41e72c822e50c20a6c13032a22baa84a065513ff00f"
   },
   "username" : "max32"
}
`,
	"bad_v2_prev_mismatch.json": `{
   "label_kids" : {
      "e" : "01200b41c968a2e60e1189125cc5e5bde5658582440a13178dd2e68064104a6d51d30a",
      "sib1" : "0120a2d598da09ad9bdfccf040fc54b03a2024aee8fc01d725032088514ba6f373100a"
   },
   "chain" : [
      {
         "sig" : "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgC0HJaKLmDhGJElzF5b3lZYWCRAoTF43S5oBkEEptUdMKp3BheWxvYWTEJ5UCAcDEIIFvIOF0uaPu/F/iJsN8eTA+9J0Byr9d/YBrtElGroKcAaNzaWfEQJ1GmcYzVjlTfVJZRkFrYjI8nCfa0Q58Wn38bVtnJjIjVw6UK4NoNVfnleyGQd4Kn6nOUvEms7Gf074D6MQTPgKoc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
         "sig_version" : 2,
         "payload_json" : "{\"body\":{\"key\":{\"host\":\"keybase.io\",\"kid\":\"01200b41c968a2e60e1189125cc5e5bde5658582440a13178dd2e68064104a6d51d30a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"type\":\"eldest\",\"version\":2},\"ctime\":1493674567,\"expire_in\":314496000,\"prev\":null,\"seq_type\":1,\"seqno\":1,\"tag\":\"signature\"}",
         "ctime" : 1493674567,
         "payload_hash" : "9252db61bfa88414ee46856205dec69f31e1837b2e43d139823ab095df155cd3",
         "kid" : "01200b41c968a2e60e1189125cc5e5bde5658582440a13178dd2e68064104a6d51d30a",
         "prev" : null,
         "sig_id" : "4af3db216150c1706cd1cff2dea557e12b9f44ac791ee73674f23e397b1f7e0e0f",
         "seqno" : 1
      },
      {
         "sig" : "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgC0HJaKLmDhGJElzF5b3lZYWCRAoTF43S5oBkEEptUdMKp3BheWxvYWTESJUCAsQgk1LbYb+ohBTuRoViBd7GnzHhg3suQ9E5gjqwld8VXNPEICpN/GOcdrmoeBvTwlFOoop6QNjwpsZojCbSy9d/U5GPC6NzaWfEQKRXA+VvZ7aHzYkUiVmwxRPwaca5XCrZmSQy+SbikfO6Yucy5ZBCQRGNBBsenmjEiMO+EtIiwPvXqW9SsKreKQGoc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
         "sig_version" : 2,
         "payload_json" : "{\"body\":{\"key\":{\"eldest_kid\":\"01200b41c968a2e60e1189125cc5e5bde5658582440a13178dd2e68064104a6d51d30a\",\"host\":\"keybase.io\",\"kid\":\"01200b41c968a2e60e1189125cc5e5bde5658582440a13178dd2e68064104a6d51d30a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"sibkey\":{\"kid\":\"0120a2d598da09ad9bdfccf040fc54b03a2024aee8fc01d725032088514ba6f373100a\",\"reverse_sig\":\"g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgotWY2gmtm9/M8ED8VLA6ICSu6PwB1yUDIIhRS6bzcxAKp3BheWxvYWTFAix7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwMGI0MWM5NjhhMmU2MGUxMTg5MTI1Y2M1ZTViZGU1NjU4NTgyNDQwYTEzMTc4ZGQyZTY4MDY0MTA0YTZkNTFkMzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwMGI0MWM5NjhhMmU2MGUxMTg5MTI1Y2M1ZTViZGU1NjU4NTgyNDQwYTEzMTc4ZGQyZTY4MDY0MTA0YTZkNTFkMzBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInNpYmtleSI6eyJraWQiOiIwMTIwYTJkNTk4ZGEwOWFkOWJkZmNjZjA0MGZjNTRiMDNhMjAyNGFlZThmYzAxZDcyNTAzMjA4ODUxNGJhNmYzNzMxMDBhIiwicmV2ZXJzZV9zaWciOm51bGx9LCJ0eXBlIjoic2lia2V5IiwidmVyc2lvbiI6MX0sImN0aW1lIjoxNDkzNjc0NjY3LCJleHBpcmVfaW4iOjMxNDQ5NjAwMCwicHJldiI6IjkyNTJkYjYxYmZhODg0MTRlZTQ2ODU2MjA1ZGVjNjlmMzFlMTgzN2IyZTQzZDEzOTgyM2FiMDk1ZGYxNTVjZDMiLCJzZXFfdHlwZSI6MSwic2Vxbm8iOjIsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RAB846H6JXvwGNL8rE9ZS7HaniU490xfiX7GrvX1ieYQBao2/xjblMEoRaVD00Ggu5FCukkmYywiWfL1ATAsEwDKhzaWdfdHlwZSCjdGFnzQICp3ZlcnNpb24B\"},\"type\":\"sibkey\",\"version\":2},\"ctime\":1493674667,\"expire_in\":314496000,\"prev\":\"9252db61bfa88414ee46856205dec69f31e1837b2e43d139823ab095df155cd3\",\"seq_type\":1,\"seqno\":2,\"tag\":\"signature\"}",
         "ctime" : 1493674667,
         "kid" : "01200b41c968a2e60e1189125cc5e5bde5658582440a13178dd2e68064104a6d51d30a",
         "payload_hash" : "1a3e4b4e449628c01d6dadaefe612ce3d4c0b11466e859d34d213b6b4882a1b9",
         "prev" : "9252db61bfa88414ee46856205dec69f31e1837b2e43d139823ab095df155cd3",
         "sig_id" : "53f271c3a2bf6033be90db57853691ff6c6b68b7c237e1a13bf93864b3392acd0f",
         "seqno" : 2
      }
   ],
   "uid" : "74c38cf7ceb947f5632045d8ca5d4819",
   "username" : "max32",
   "keys" : [
      "01200b41c968a2e60e1189125cc5e5bde5658582440a13178dd2e68064104a6d51d30a",
      "0120a2d598da09ad9bdfccf040fc54b03a2024aee8fc01d725032088514ba6f373100a"
   ],
   "label_sigs" : {
      "sib1" : "53f271c3a2bf6033be90db57853691ff6c6b68b7c237e1a13bf93864b3392acd0f",
      "e" : "4af3db216150c1706cd1cff2dea557e12b9f44ac791ee73674f23e397b1f7e0e0f"
   }
}
`,
	"bad_v2_revoke_type_mismatch.json": `{
   "chain" : [
      {
         "seqno" : 1,
         "sig" : "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgLj3jD8PnDo8qPQ8AoFv0P2TuviTxxiEyaPX79PSXAf4Kp3BheWxvYWTEJ5UCAcDEIIXbK96RxDltW94QUHMeP5TnGG9Qc3g/AZ8ENp7hhuXCAaNzaWfEQPQMWjtJ4nvaDX5kh0WZDvl1rBakfxx8bWH2E6Z6ue4/b+M4fPX0L5iXGdz2VlceG5koELYwd/65irUzaTarnQioc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
         "sig_id" : "5bcd53117eff56b200476ec1f00189096116d45829acc82bdaab7c26416724e10f",
         "payload_json" : "{\"body\":{\"key\":{\"host\":\"keybase.io\",\"kid\":\"01202e3de30fc3e70e8f2a3d0f00a05bf43f64eebe24f1c6213268f5fbf4f49701fe0a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"type\":\"eldest\",\"version\":2},\"ctime\":1493675071,\"expire_in\":314496000,\"prev\":null,\"seq_type\":1,\"seqno\":1,\"tag\":\"signature\"}",
         "prev" : null,
         "kid" : "01202e3de30fc3e70e8f2a3d0f00a05bf43f64eebe24f1c6213268f5fbf4f49701fe0a",
         "sig_version" : 2,
         "ctime" : 1493675071,
         "payload_hash" : "6257c5a09075fca36ea42875c9dbba4003f92ecb9799eb33ba216ee387480f82"
      },
      {
         "sig" : "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgLj3jD8PnDo8qPQ8AoFv0P2TuviTxxiEyaPX79PSXAf4Kp3BheWxvYWTESJUCAsQgYlfFoJB1/KNupCh1ydu6QAP5LsuXmeszuiFu44dID4LEIG0Tbanvh/nwxNUgFhxojOkM4OFQRxok3IEy5hG/vMlmC6NzaWfEQIJQMqQ3zyZgpCfFytEU8ASNf+q4MZExZthdkpus6z8+V663VMBsgMBnUOh8N3J+m64DxvA4d9wdOryDle2FBQGoc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
         "sig_id" : "24e165f92ca1f0e40e77c8fa8e7522b5a7dc27802ea8c661cab8410dfbb3aa2d0f",
         "seqno" : 2,
         "payload_json" : "{\"body\":{\"key\":{\"eldest_kid\":\"01202e3de30fc3e70e8f2a3d0f00a05bf43f64eebe24f1c6213268f5fbf4f49701fe0a\",\"host\":\"keybase.io\",\"kid\":\"01202e3de30fc3e70e8f2a3d0f00a05bf43f64eebe24f1c6213268f5fbf4f49701fe0a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"sibkey\":{\"kid\":\"012060a6eca6632bf87dc35bb6a3150e6c4dfc7872f450e4b3ced46a51d7559c00730a\",\"reverse_sig\":\"g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgYKbspmMr+H3DW7ajFQ5sTfx4cvRQ5LPO1GpR11WcAHMKp3BheWxvYWTFAix7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwMmUzZGUzMGZjM2U3MGU4ZjJhM2QwZjAwYTA1YmY0M2Y2NGVlYmUyNGYxYzYyMTMyNjhmNWZiZjRmNDk3MDFmZTBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwMmUzZGUzMGZjM2U3MGU4ZjJhM2QwZjAwYTA1YmY0M2Y2NGVlYmUyNGYxYzYyMTMyNjhmNWZiZjRmNDk3MDFmZTBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInNpYmtleSI6eyJraWQiOiIwMTIwNjBhNmVjYTY2MzJiZjg3ZGMzNWJiNmEzMTUwZTZjNGRmYzc4NzJmNDUwZTRiM2NlZDQ2YTUxZDc1NTljMDA3MzBhIiwicmV2ZXJzZV9zaWciOm51bGx9LCJ0eXBlIjoic2lia2V5IiwidmVyc2lvbiI6Mn0sImN0aW1lIjoxNDkzNjc1MTcxLCJleHBpcmVfaW4iOjMxNDQ5NjAwMCwicHJldiI6IjYyNTdjNWEwOTA3NWZjYTM2ZWE0Mjg3NWM5ZGJiYTQwMDNmOTJlY2I5Nzk5ZWIzM2JhMjE2ZWUzODc0ODBmODIiLCJzZXFfdHlwZSI6MSwic2Vxbm8iOjIsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RAfbj1+BDaTxom89qmpO4xDC6v0bF+IbgMoQysGKgJgJsCr2EmY6CnzXAg0evt8B2U6eb0HKqSvhl8Xs+IbhlTCahzaWdfdHlwZSCjdGFnzQICp3ZlcnNpb24B\"},\"type\":\"sibkey\",\"version\":2},\"ctime\":1493675171,\"expire_in\":314496000,\"prev\":\"6257c5a09075fca36ea42875c9dbba4003f92ecb9799eb33ba216ee387480f82\",\"seq_type\":1,\"seqno\":2,\"tag\":\"signature\"}",
         "ctime" : 1493675171,
         "sig_version" : 2,
         "prev" : "6257c5a09075fca36ea42875c9dbba4003f92ecb9799eb33ba216ee387480f82",
         "kid" : "01202e3de30fc3e70e8f2a3d0f00a05bf43f64eebe24f1c6213268f5fbf4f49701fe0a",
         "payload_hash" : "41915cf032bef0aac37d20f261d1e3884eae607a1d99f70678937d39dbed4e87"
      },
      {
         "payload_json" : "{\"body\":{\"cryptocurrency\":{\"address\":\"1F9oF2u7mQXuSBqXp7SqmMHMuMZ5ZJWWsu\",\"type\":\"bitcoin\"},\"key\":{\"eldest_kid\":\"01202e3de30fc3e70e8f2a3d0f00a05bf43f64eebe24f1c6213268f5fbf4f49701fe0a\",\"host\":\"keybase.io\",\"kid\":\"012060a6eca6632bf87dc35bb6a3150e6c4dfc7872f450e4b3ced46a51d7559c00730a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"type\":\"cryptocurrency\",\"version\":2},\"ctime\":1493675271,\"expire_in\":314496000,\"prev\":\"41915cf032bef0aac37d20f261d1e3884eae607a1d99f70678937d39dbed4e87\",\"seq_type\":1,\"seqno\":3,\"tag\":\"signature\"}",
         "seqno" : 3,
         "sig" : "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgYKbspmMr+H3DW7ajFQ5sTfx4cvRQ5LPO1GpR11WcAHMKp3BheWxvYWTESJUCA8QgQZFc8DK+8KrDfSDyYdHjiE6uYHodmfcGeJN9OdvtTofEIJ/hgdIbZArwgJLKAF2fA4nEAej/7NUGFl5/lneSsEueBqNzaWfEQA3bqmZRA6tRWDKlgYNohAcBUCvdf4bLCYQsefLpGzuXLUAwo+DiOESwdzVDdCEbMw73uP5B6D/q07MpyrmMEweoc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
         "sig_id" : "056a55603ec1d809d755e63e36c74569de130bdd9694537e327ed3e1bf3788810f",
         "payload_hash" : "501fdb904f9f5869f903b19bd7037f888f03d56f38e92cf2b51330d417c5eec4",
         "prev" : "41915cf032bef0aac37d20f261d1e3884eae607a1d99f70678937d39dbed4e87",
         "kid" : "012060a6eca6632bf87dc35bb6a3150e6c4dfc7872f450e4b3ced46a51d7559c00730a",
         "ctime" : 1493675271,
         "sig_version" : 2
      },
      {
         "kid" : "012060a6eca6632bf87dc35bb6a3150e6c4dfc7872f450e4b3ced46a51d7559c00730a",
         "prev" : "501fdb904f9f5869f903b19bd7037f888f03d56f38e92cf2b51330d417c5eec4",
         "sig_version" : 2,
         "ctime" : 1493675371,
         "payload_hash" : "9dcf93385c1bb511715cc433faaaff21d6bc204d64b7ddda767e6cf9900183f5",
         "seqno" : 4,
         "sig_id" : "d399755408ecdf0c95274ba7db970c9d23e66ef494d1ac44b938a9e1a5534de20f",
         "sig" : "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgYKbspmMr+H3DW7ajFQ5sTfx4cvRQ5LPO1GpR11WcAHMKp3BheWxvYWTESJUCBMQgUB/bkE+fWGn5A7Gb1wN/iI8D1W846SzytRMw1BfF7sTEIPYLbTMRsRO0N5lQgOmNmkaEHeRJF7QAybknl70wLw35BqNzaWfEQLwzipHOiliIgoDrXtCt95vE2gWKedi2FMOGYD8JRtoGUKhnQ7AjBNEx5axTnhFkk/fMePLLDMi3VPCuW3RCGgSoc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
         "payload_json" : "{\"body\":{\"cryptocurrency\":{\"address\":\"1ABNbAJW5C9ovspCgetx34zvm7uKziB8AF\",\"type\":\"bitcoin\"},\"key\":{\"eldest_kid\":\"01202e3de30fc3e70e8f2a3d0f00a05bf43f64eebe24f1c6213268f5fbf4f49701fe0a\",\"host\":\"keybase.io\",\"kid\":\"012060a6eca6632bf87dc35bb6a3150e6c4dfc7872f450e4b3ced46a51d7559c00730a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"revoke\":{\"sig_ids\":[\"056a55603ec1d809d755e63e36c74569de130bdd9694537e327ed3e1bf3788810f\"]},\"type\":\"cryptocurrency\",\"version\":2},\"ctime\":1493675371,\"expire_in\":314496000,\"prev\":\"501fdb904f9f5869f903b19bd7037f888f03d56f38e92cf2b51330d417c5eec4\",\"seq_type\":1,\"seqno\":4,\"tag\":\"signature\"}"
      }
   ],
   "label_kids" : {
      "e" : "01202e3de30fc3e70e8f2a3d0f00a05bf43f64eebe24f1c6213268f5fbf4f49701fe0a",
      "sib1" : "012060a6eca6632bf87dc35bb6a3150e6c4dfc7872f450e4b3ced46a51d7559c00730a"
   },
   "label_sigs" : {
      "btc1" : "056a55603ec1d809d755e63e36c74569de130bdd9694537e327ed3e1bf3788810f",
      "btc2" : "d399755408ecdf0c95274ba7db970c9d23e66ef494d1ac44b938a9e1a5534de20f",
      "e" : "5bcd53117eff56b200476ec1f00189096116d45829acc82bdaab7c26416724e10f",
      "sib1" : "24e165f92ca1f0e40e77c8fa8e7522b5a7dc27802ea8c661cab8410dfbb3aa2d0f"
   },
   "username" : "max32",
   "keys" : [
      "01202e3de30fc3e70e8f2a3d0f00a05bf43f64eebe24f1c6213268f5fbf4f49701fe0a",
      "012060a6eca6632bf87dc35bb6a3150e6c4dfc7872f450e4b3ced46a51d7559c00730a"
   ],
   "uid" : "74c38cf7ceb947f5632045d8ca5d4819"
}
`,
	"bad_v2_seqno_mismatch.json": `{
   "username" : "max32",
   "chain" : [
      {
         "kid" : "0120775ee61abbc325add5aef96b181d5e7801d6e26dcc2537271901752c5af344f50a",
         "ctime" : 1493675114,
         "payload_hash" : "28fa979fbea7b175a05178c70f75b5759df6216ed2ad9b21266d55294f38960f",
         "prev" : null,
         "payload_json" : "{\"body\":{\"key\":{\"host\":\"keybase.io\",\"kid\":\"0120775ee61abbc325add5aef96b181d5e7801d6e26dcc2537271901752c5af344f50a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"type\":\"eldest\",\"version\":2},\"ctime\":1493675114,\"expire_in\":314496000,\"prev\":null,\"seq_type\":1,\"seqno\":1,\"tag\":\"signature\"}",
         "sig" : "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgd17mGrvDJa3VrvlrGB1eeAHW4m3MJTcnGQF1LFrzRPUKp3BheWxvYWTEJ5UCAcDEIC6EEwIBcK/NSOXbs7IhYl2oyeFlbKJlPOCe5eB7Z/qnAaNzaWfEQH6xd+JxbyrfoVWLv74ysl6J3EdXDQQBKfEMScI4YZS0GC2yFkz2wetbUti6hq4vOvRed420hQfg7MpA2/mP/AGoc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
         "sig_version" : 2,
         "sig_id" : "d6ba9584c8d37ae8f14acf419d5a612db8696b3d94f434b007b36a0355d0efd90f",
         "seqno" : 1
      },
      {
         "payload_hash" : "00f90aabdfba086b4af0c6c7e77093954a7149cd553b82b2344edd4d7dc110fe",
         "prev" : "28fa979fbea7b175a05178c70f75b5759df6216ed2ad9b21266d55294f38960f",
         "payload_json" : "{\"body\":{\"key\":{\"eldest_kid\":\"0120775ee61abbc325add5aef96b181d5e7801d6e26dcc2537271901752c5af344f50a\",\"host\":\"keybase.io\",\"kid\":\"0120775ee61abbc325add5aef96b181d5e7801d6e26dcc2537271901752c5af344f50a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"sibkey\":{\"kid\":\"0120fb6a112a220f8464ac5f26887db09bae94c5b805379485f2facb43fba02ae1b50a\",\"reverse_sig\":\"g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEg+2oRKiIPhGSsXyaIfbCbrpTFuAU3lIXy+stD+6Aq4bUKp3BheWxvYWTFAix7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwNzc1ZWU2MWFiYmMzMjVhZGQ1YWVmOTZiMTgxZDVlNzgwMWQ2ZTI2ZGNjMjUzNzI3MTkwMTc1MmM1YWYzNDRmNTBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwNzc1ZWU2MWFiYmMzMjVhZGQ1YWVmOTZiMTgxZDVlNzgwMWQ2ZTI2ZGNjMjUzNzI3MTkwMTc1MmM1YWYzNDRmNTBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInNpYmtleSI6eyJraWQiOiIwMTIwZmI2YTExMmEyMjBmODQ2NGFjNWYyNjg4N2RiMDliYWU5NGM1YjgwNTM3OTQ4NWYyZmFjYjQzZmJhMDJhZTFiNTBhIiwicmV2ZXJzZV9zaWciOm51bGx9LCJ0eXBlIjoic2lia2V5IiwidmVyc2lvbiI6MX0sImN0aW1lIjoxNDkzNjc1MjE0LCJleHBpcmVfaW4iOjMxNDQ5NjAwMCwicHJldiI6IjI4ZmE5NzlmYmVhN2IxNzVhMDUxNzhjNzBmNzViNTc1OWRmNjIxNmVkMmFkOWIyMTI2NmQ1NTI5NGYzODk2MGYiLCJzZXFfdHlwZSI6MSwic2Vxbm8iOjIsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RAPbg5CplANz1xtuxguccFp1lt4PwBRanCKAZ+k86A1e9WkGOr/aEGweA9SndOCOvrr0O6V9eHtBXqppa2JsMnA6hzaWdfdHlwZSCjdGFnzQICp3ZlcnNpb24B\"},\"type\":\"sibkey\",\"version\":2},\"ctime\":1493675214,\"expire_in\":314496000,\"prev\":\"28fa979fbea7b175a05178c70f75b5759df6216ed2ad9b21266d55294f38960f\",\"seq_type\":1,\"seqno\":2,\"tag\":\"signature\"}",
         "kid" : "0120775ee61abbc325add5aef96b181d5e7801d6e26dcc2537271901752c5af344f50a",
         "ctime" : 1493675214,
         "seqno" : 2,
         "sig_id" : "07c934e9da43992c8239358576d99eed8234be95c1aca50caa24e3c8263f89910f",
         "sig" : "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgd17mGrvDJa3VrvlrGB1eeAHW4m3MJTcnGQF1LFrzRPUKp3BheWxvYWTESJUCA8QgKPqXn76nsXWgUXjHD3W1dZ32IW7SrZshJm1VKU84lg/EIAtigrLCLEzVJ05lWwQBazAOjzMs28ZSAKE2Q7+BO4ILC6NzaWfEQMC2L9/oN6SZ64rQzFFDnVyMWOdtllFGRB17DS1QO68qsnYgZbQ7G2aXoJXsRROzqs6bD32ARyQe/fzDf9+vBQqoc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
         "sig_version" : 2
      }
   ],
   "uid" : "74c38cf7ceb947f5632045d8ca5d4819",
   "label_sigs" : {
      "sib1" : "07c934e9da43992c8239358576d99eed8234be95c1aca50caa24e3c8263f89910f",
      "e" : "d6ba9584c8d37ae8f14acf419d5a612db8696b3d94f434b007b36a0355d0efd90f"
   },
   "keys" : [
      "0120775ee61abbc325add5aef96b181d5e7801d6e26dcc2537271901752c5af344f50a",
      "0120fb6a112a220f8464ac5f26887db09bae94c5b805379485f2facb43fba02ae1b50a"
   ],
   "label_kids" : {
      "e" : "0120775ee61abbc325add5aef96b181d5e7801d6e26dcc2537271901752c5af344f50a",
      "sib1" : "0120fb6a112a220f8464ac5f26887db09bae94c5b805379485f2facb43fba02ae1b50a"
   }
}
`,
	"bad_v2_stubbed_revoke_sig.json": `{
   "chain" : [
      {
         "prev" : null,
         "sig_id" : "294b432fd19c68b9cfd00b5195e89345c94ae839a7d8b54c0d6212f733bde1a80f",
         "sig_version" : 2,
         "ctime" : 1493675145,
         "payload_hash" : "0ff1c29eebacc0e3577ef15f9cbebe699da2eb58a15098be4fb175e0a5393405",
         "payload_json" : "{\"body\":{\"key\":{\"host\":\"keybase.io\",\"kid\":\"01207311769ecf281c4f7b7df4039b758896f07641c698d694e93367da29aacbe9530a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"type\":\"eldest\",\"version\":2},\"ctime\":1493675145,\"expire_in\":314496000,\"prev\":null,\"seq_type\":1,\"seqno\":1,\"tag\":\"signature\"}",
         "sig" : "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgcxF2ns8oHE97ffQDm3WIlvB2QcaY1pTpM2faKarL6VMKp3BheWxvYWTEJ5UCAcDEIE2T0bF7cia1vBgZ+j3QX50r8c5VfPhlihjJiaYxEb3sAaNzaWfEQPYawUiOYNUW0rInO8/+tkqclRz+rLInCE0/khPp+ahBrTFOckN+ea2nXoTdMkyhWBfPESg3uz/Prl2lbx/NIwioc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
         "seqno" : 1,
         "kid" : "01207311769ecf281c4f7b7df4039b758896f07641c698d694e93367da29aacbe9530a"
      },
      {
         "payload_hash" : "62c976017698d3fb41957ace017a81e16737e7cd3b9f7cd85ee9c6586aa07d8b",
         "sig" : "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgcxF2ns8oHE97ffQDm3WIlvB2QcaY1pTpM2faKarL6VMKp3BheWxvYWTESJUCAsQgD/HCnuuswONXfvFfnL6+aZ2i61ihUJi+T7F14KU5NAXEIPudbNmyHTYWaw8WgLF2FEuPrfnWz4k3C4CnX3RoZcHjC6NzaWfEQM6CQgFv5iEF7WN9uIM9cWuCed8Ypqpiz4n3Pdnz4cSb+2R+wTKMUig18bSIV/I0Q1ybjgShtAeLCPKsPnUN/Aeoc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
         "seqno" : 2,
         "kid" : "01207311769ecf281c4f7b7df4039b758896f07641c698d694e93367da29aacbe9530a",
         "payload_json" : "{\"body\":{\"key\":{\"eldest_kid\":\"01207311769ecf281c4f7b7df4039b758896f07641c698d694e93367da29aacbe9530a\",\"host\":\"keybase.io\",\"kid\":\"01207311769ecf281c4f7b7df4039b758896f07641c698d694e93367da29aacbe9530a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"sibkey\":{\"kid\":\"01201e601b886fb764c32b85a8810e78e6c3e150a58bfe9ab06db1484ff87fbb95630a\",\"reverse_sig\":\"g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgHmAbiG+3ZMMrhaiBDnjmw+FQpYv+mrBtsUhP+H+7lWMKp3BheWxvYWTFAix7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwNzMxMTc2OWVjZjI4MWM0ZjdiN2RmNDAzOWI3NTg4OTZmMDc2NDFjNjk4ZDY5NGU5MzM2N2RhMjlhYWNiZTk1MzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwNzMxMTc2OWVjZjI4MWM0ZjdiN2RmNDAzOWI3NTg4OTZmMDc2NDFjNjk4ZDY5NGU5MzM2N2RhMjlhYWNiZTk1MzBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInNpYmtleSI6eyJraWQiOiIwMTIwMWU2MDFiODg2ZmI3NjRjMzJiODVhODgxMGU3OGU2YzNlMTUwYTU4YmZlOWFiMDZkYjE0ODRmZjg3ZmJiOTU2MzBhIiwicmV2ZXJzZV9zaWciOm51bGx9LCJ0eXBlIjoic2lia2V5IiwidmVyc2lvbiI6Mn0sImN0aW1lIjoxNDkzNjc1MjQ1LCJleHBpcmVfaW4iOjMxNDQ5NjAwMCwicHJldiI6IjBmZjFjMjllZWJhY2MwZTM1NzdlZjE1ZjljYmViZTY5OWRhMmViNThhMTUwOThiZTRmYjE3NWUwYTUzOTM0MDUiLCJzZXFfdHlwZSI6MSwic2Vxbm8iOjIsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RA7X0UPdqBD3wl4G9oq4BGql4+MAae3rdY+JxoU20FBfqYf35qP9tfL8hk51Kkw3D/AnIFvIn/vroaXk3D2T2TA6hzaWdfdHlwZSCjdGFnzQICp3ZlcnNpb24B\"},\"type\":\"sibkey\",\"version\":2},\"ctime\":1493675245,\"expire_in\":314496000,\"prev\":\"0ff1c29eebacc0e3577ef15f9cbebe699da2eb58a15098be4fb175e0a5393405\",\"seq_type\":1,\"seqno\":2,\"tag\":\"signature\"}",
         "prev" : "0ff1c29eebacc0e3577ef15f9cbebe699da2eb58a15098be4fb175e0a5393405",
         "sig_id" : "119fa31d58fb8586444c48b9cee03014b2a29b4124713d9a098cfee3e8cbc3c10f",
         "ctime" : 1493675245,
         "sig_version" : 2
      },
      {
         "ctime" : 1493675345,
         "sig_version" : 2,
         "prev" : "62c976017698d3fb41957ace017a81e16737e7cd3b9f7cd85ee9c6586aa07d8b",
         "sig_id" : "4ed7c9165fb7ef1ee7e9944634ce6a40c57b62f0a8dde78e53c82eae0847cce10f",
         "seqno" : 3,
         "sig" : "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgHmAbiG+3ZMMrhaiBDnjmw+FQpYv+mrBtsUhP+H+7lWMKp3BheWxvYWTESJUCA8QgYsl2AXaY0/tBlXrOAXqB4Wc35807n3zYXunGWGqgfYvEIL0UZT6rMZ4ghJrHhP+NnWstylFVnmi9DoKdnDGxfAG5BqNzaWfEQHBnPMRqfPXRr3RRe5zKyXevBJe0j9yweedpinSIcj5FYT5Feji9p2zSn3HTsQ2w0dku9xoRmJ2crTCckWsEEQmoc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
         "kid" : "01201e601b886fb764c32b85a8810e78e6c3e150a58bfe9ab06db1484ff87fbb95630a",
         "payload_json" : "{\"body\":{\"cryptocurrency\":{\"address\":\"1N1mCVEmeS7dzxNFeny8zRVEvGN3vpycyv\",\"type\":\"bitcoin\"},\"key\":{\"eldest_kid\":\"01207311769ecf281c4f7b7df4039b758896f07641c698d694e93367da29aacbe9530a\",\"host\":\"keybase.io\",\"kid\":\"01201e601b886fb764c32b85a8810e78e6c3e150a58bfe9ab06db1484ff87fbb95630a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"type\":\"cryptocurrency\",\"version\":2},\"ctime\":1493675345,\"expire_in\":314496000,\"prev\":\"62c976017698d3fb41957ace017a81e16737e7cd3b9f7cd85ee9c6586aa07d8b\",\"seq_type\":1,\"seqno\":3,\"tag\":\"signature\"}",
         "payload_hash" : "4d690640be57d8c93bb9eb8a8e58d1da4363055b35dfa1703c63618d9fe1dc8a"
      },
      {
         "payload_hash" : "599120588eec891e34a40ee07c78a3c6dd864d8ce934fef67025e0c218844d38",
         "kid" : "01201e601b886fb764c32b85a8810e78e6c3e150a58bfe9ab06db1484ff87fbb95630a",
         "seqno" : 4,
         "sig" : "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgHmAbiG+3ZMMrhaiBDnjmw+FQpYv+mrBtsUhP+H+7lWMKp3BheWxvYWTESJUCBMQgTWkGQL5X2Mk7ueuKjljR2kNjBVs136FwPGNhjZ/h3IrEIO5han1qkM5/8LrF52/EpkeXOHjNYaI3MVLln2pCJD48CqNzaWfEQMRLZH9MKL7mj3swDnOs4KtEc8HYKnpNy0iMXGe3EsK3Md0PmpNWrp8AKEBcJeemSOKOxuETOsA0eRwrHPqq4gCoc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
         "payload_json" : "{\"body\":{\"cryptocurrency\":{\"address\":\"1E4sFXukPk1GBrERqn3Ny13QHpVs9fiv8G\",\"type\":\"bitcoin\"},\"key\":{\"eldest_kid\":\"01207311769ecf281c4f7b7df4039b758896f07641c698d694e93367da29aacbe9530a\",\"host\":\"keybase.io\",\"kid\":\"01201e601b886fb764c32b85a8810e78e6c3e150a58bfe9ab06db1484ff87fbb95630a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"revoke\":{\"sig_ids\":[\"4ed7c9165fb7ef1ee7e9944634ce6a40c57b62f0a8dde78e53c82eae0847cce10f\"]},\"type\":\"cryptocurrency\",\"version\":2},\"ctime\":1493675445,\"expire_in\":314496000,\"prev\":\"4d690640be57d8c93bb9eb8a8e58d1da4363055b35dfa1703c63618d9fe1dc8a\",\"seq_type\":1,\"seqno\":4,\"tag\":\"signature\"}",
         "sig_id" : "ff901c32c229c7e72fa1eec0a73a8315c44bf48ebda124ec27f8f01b414c4af80f",
         "prev" : "4d690640be57d8c93bb9eb8a8e58d1da4363055b35dfa1703c63618d9fe1dc8a",
         "ctime" : 1493675445,
         "sig_version" : 2
      },
      {
         "s2" : "lQIFxCBZkSBYjuyJHjSkDuB8eKPG3YZNjOk0/vZwJeDCGIRNOMQgkqDLvAyAwYk9akVZJQkQACMFusRJLh/ivyVi4+VleIUK"
      },
      {
         "s2" : "lQIGxCCIGu5dtpLD2WaSZDjWefk4A79pAg1SNnctgvBSHuTuXsQg36kalHkGqahSnE4Wq7e0tWx/kUinVI0fDMabJM7S4y0G"
      },
      {
         "sig_version" : 2,
         "ctime" : 1493675745,
         "sig_id" : "77e8d2dca4eca0bd4d4800a73109bbd50fecb1e1f818890edf0795b54ee563320f",
         "prev" : "7d1be65097c13b32b73148871b398503fe8a8e3dc183c76ccf83c08be510a22a",
         "payload_json" : "{\"body\":{\"cryptocurrency\":{\"address\":\"1KmDoUzkNVRgMTJNyH9ZZPS4Lbab7ZVAex\",\"type\":\"bitcoin\"},\"key\":{\"eldest_kid\":\"01207311769ecf281c4f7b7df4039b758896f07641c698d694e93367da29aacbe9530a\",\"host\":\"keybase.io\",\"kid\":\"01201e601b886fb764c32b85a8810e78e6c3e150a58bfe9ab06db1484ff87fbb95630a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"type\":\"cryptocurrency\",\"version\":2},\"ctime\":1493675745,\"expire_in\":314496000,\"prev\":\"7d1be65097c13b32b73148871b398503fe8a8e3dc183c76ccf83c08be510a22a\",\"seq_type\":1,\"seqno\":7,\"tag\":\"signature\"}",
         "kid" : "01201e601b886fb764c32b85a8810e78e6c3e150a58bfe9ab06db1484ff87fbb95630a",
         "seqno" : 7,
         "sig" : "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgHmAbiG+3ZMMrhaiBDnjmw+FQpYv+mrBtsUhP+H+7lWMKp3BheWxvYWTESJUCB8QgfRvmUJfBOzK3MUiHGzmFA/6Kjj3Bg8dsz4PAi+UQoirEIMwIx4haB6i8sTKlNWnnM3uGi3lvkxK1GIbKLbx/tZdcBqNzaWfEQF3tU1Hf1rZDVcVRU4WnWVDsJoQ+P1Tn6qHfQ1gG4WvlaBsofV+M3x35cGzELKrEn9ZTpih4mlgk+2y10dEERgaoc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
         "payload_hash" : "d61e45332f0f0b0d607983dffaef47dd2ea26e7b09dfb2db1b27a263892fd0d2"
      }
   ],
   "username" : "max32",
   "keys" : [
      "01207311769ecf281c4f7b7df4039b758896f07641c698d694e93367da29aacbe9530a",
      "01201e601b886fb764c32b85a8810e78e6c3e150a58bfe9ab06db1484ff87fbb95630a"
   ],
   "label_sigs" : {
      "btc2" : "ff901c32c229c7e72fa1eec0a73a8315c44bf48ebda124ec27f8f01b414c4af80f",
      "sib1" : "119fa31d58fb8586444c48b9cee03014b2a29b4124713d9a098cfee3e8cbc3c10f",
      "btc4" : "3ed06eceefa9d75ee6a994a8211057d14e8f61b3af86bc8b46cb3c6290f5c9b90f",
      "btc1" : "4ed7c9165fb7ef1ee7e9944634ce6a40c57b62f0a8dde78e53c82eae0847cce10f",
      "btc3" : "afcb9eda20f07825c6a586739e353604fa2623a3a2552fe3f7a19a3468a793170f",
      "btc5" : "77e8d2dca4eca0bd4d4800a73109bbd50fecb1e1f818890edf0795b54ee563320f",
      "e" : "294b432fd19c68b9cfd00b5195e89345c94ae839a7d8b54c0d6212f733bde1a80f"
   },
   "label_kids" : {
      "sib1" : "01201e601b886fb764c32b85a8810e78e6c3e150a58bfe9ab06db1484ff87fbb95630a",
      "e" : "01207311769ecf281c4f7b7df4039b758896f07641c698d694e93367da29aacbe9530a"
   },
   "uid" : "74c38cf7ceb947f5632045d8ca5d4819"
}
`,
	"bad_v2_stubbed_revoke_sigs.json": `{
   "username" : "max32",
   "chain" : [
      {
         "kid" : "012022e7c67d1d9e2d7b653f776a44f0839987062a19902bd0a63fd395b76f75ed940a",
         "seqno" : 1,
         "sig_version" : 2,
         "ctime" : 1493675183,
         "payload_hash" : "bb3441bd8b08c96e80b1ef17ea452afd63a84672df3b8afd6452ebc25c30f7f3",
         "sig" : "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgIufGfR2eLXtlP3dqRPCDmYcGKhmQK9CmP9OVt2917ZQKp3BheWxvYWTEJ5UCAcDEIP6iACmz1LVgRH4ZfiSe1xiCOSpB4qFhPdletb+xukR6AaNzaWfEQEiilCEnwI3jycFUzb4e0/+Iab0x5JAVHeLBkj1iaK7J9TdHa3Bl4g9UrgAXVT6szT8AONUyplGZ9wEr1Ycc/wKoc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
         "sig_id" : "7d1e012725f42e9643ca4208fc7626d3cf28daf62163c5ce5214296280d4f76f0f",
         "prev" : null,
         "payload_json" : "{\"body\":{\"key\":{\"host\":\"keybase.io\",\"kid\":\"012022e7c67d1d9e2d7b653f776a44f0839987062a19902bd0a63fd395b76f75ed940a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"type\":\"eldest\",\"version\":2},\"ctime\":1493675183,\"expire_in\":314496000,\"prev\":null,\"seq_type\":1,\"seqno\":1,\"tag\":\"signature\"}"
      },
      {
         "sig" : "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgIufGfR2eLXtlP3dqRPCDmYcGKhmQK9CmP9OVt2917ZQKp3BheWxvYWTESJUCAsQguzRBvYsIyW6Ase8X6kUq/WOoRnLfO4r9ZFLrwlww9/PEIHcvzzdK7KJb3xyXMrCxfkAqiLHn5nb7dSVZe5OonFm6C6NzaWfEQFB1hQOtSc2fbkAHdC2vCVPv/GIj2fcre9zYcP3ZNDAKCvCdAK1kPEnoQT2Tjbwwjppn+aTvKafX690Lc94brwWoc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
         "payload_hash" : "21111e4a17e5ede6694bd49a27bc79662b727c9706c89ed066b6eaa2d99d15fc",
         "ctime" : 1493675283,
         "payload_json" : "{\"body\":{\"key\":{\"eldest_kid\":\"012022e7c67d1d9e2d7b653f776a44f0839987062a19902bd0a63fd395b76f75ed940a\",\"host\":\"keybase.io\",\"kid\":\"012022e7c67d1d9e2d7b653f776a44f0839987062a19902bd0a63fd395b76f75ed940a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"sibkey\":{\"kid\":\"012070ea0a0a67da7b92f47240d04b476a6aa7696a9d3d43c72553c2dc137f32c4bc0a\",\"reverse_sig\":\"g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgcOoKCmfae5L0ckDQS0dqaqdpap09Q8clU8LcE38yxLwKp3BheWxvYWTFAix7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwMjJlN2M2N2QxZDllMmQ3YjY1M2Y3NzZhNDRmMDgzOTk4NzA2MmExOTkwMmJkMGE2M2ZkMzk1Yjc2Zjc1ZWQ5NDBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwMjJlN2M2N2QxZDllMmQ3YjY1M2Y3NzZhNDRmMDgzOTk4NzA2MmExOTkwMmJkMGE2M2ZkMzk1Yjc2Zjc1ZWQ5NDBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInNpYmtleSI6eyJraWQiOiIwMTIwNzBlYTBhMGE2N2RhN2I5MmY0NzI0MGQwNGI0NzZhNmFhNzY5NmE5ZDNkNDNjNzI1NTNjMmRjMTM3ZjMyYzRiYzBhIiwicmV2ZXJzZV9zaWciOm51bGx9LCJ0eXBlIjoic2lia2V5IiwidmVyc2lvbiI6Mn0sImN0aW1lIjoxNDkzNjc1MjgzLCJleHBpcmVfaW4iOjMxNDQ5NjAwMCwicHJldiI6ImJiMzQ0MWJkOGIwOGM5NmU4MGIxZWYxN2VhNDUyYWZkNjNhODQ2NzJkZjNiOGFmZDY0NTJlYmMyNWMzMGY3ZjMiLCJzZXFfdHlwZSI6MSwic2Vxbm8iOjIsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RAVfZfcDqN/D74W7f+G2xFeaEtkhxcQoN7QYJzb8PQhteUkZ4H2snC6hlscPPot0+m8lZP71oyzIEJVV83fyODDKhzaWdfdHlwZSCjdGFnzQICp3ZlcnNpb24B\"},\"type\":\"sibkey\",\"version\":2},\"ctime\":1493675283,\"expire_in\":314496000,\"prev\":\"bb3441bd8b08c96e80b1ef17ea452afd63a84672df3b8afd6452ebc25c30f7f3\",\"seq_type\":1,\"seqno\":2,\"tag\":\"signature\"}",
         "prev" : "bb3441bd8b08c96e80b1ef17ea452afd63a84672df3b8afd6452ebc25c30f7f3",
         "sig_id" : "6c3d68cacab47cab4154e2bb0fc4b6db2f65f2965ad19e6eff6f409bdb6820b00f",
         "sig_version" : 2,
         "seqno" : 2,
         "kid" : "012022e7c67d1d9e2d7b653f776a44f0839987062a19902bd0a63fd395b76f75ed940a"
      },
      {
         "sig_version" : 2,
         "seqno" : 3,
         "kid" : "012070ea0a0a67da7b92f47240d04b476a6aa7696a9d3d43c72553c2dc137f32c4bc0a",
         "payload_json" : "{\"body\":{\"cryptocurrency\":{\"address\":\"126vxLmZQaVYhgiHsK8GpiDGa9PRERwG9s\",\"type\":\"bitcoin\"},\"key\":{\"eldest_kid\":\"012022e7c67d1d9e2d7b653f776a44f0839987062a19902bd0a63fd395b76f75ed940a\",\"host\":\"keybase.io\",\"kid\":\"012070ea0a0a67da7b92f47240d04b476a6aa7696a9d3d43c72553c2dc137f32c4bc0a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"type\":\"cryptocurrency\",\"version\":2},\"ctime\":1493675383,\"expire_in\":314496000,\"prev\":\"21111e4a17e5ede6694bd49a27bc79662b727c9706c89ed066b6eaa2d99d15fc\",\"seq_type\":1,\"seqno\":3,\"tag\":\"signature\"}",
         "prev" : "21111e4a17e5ede6694bd49a27bc79662b727c9706c89ed066b6eaa2d99d15fc",
         "sig_id" : "801190f327fa34b167e927d60452bd4ebc9a27589f185cb31e72ab89ee4aea260f",
         "sig" : "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgcOoKCmfae5L0ckDQS0dqaqdpap09Q8clU8LcE38yxLwKp3BheWxvYWTESJUCA8QgIREeShfl7eZpS9SaJ7x5ZityfJcGyJ7QZrbqotmdFfzEIFi18RwEXH7PME6n5WdxPyfdtIGAsTSA9gs7NZq4F8hrBqNzaWfEQP03Dcrudb33dXMDGg38yb9R2obeNALxjlqb9YW0snTNi4fLJD3+w/0IArTdFvDoM3IB/jtWnqn1/5b6Eb1rNQmoc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
         "payload_hash" : "bd3a985a5d0e11ff714488b80cd9d34124b43116a505d3b0bd95ee721f466f0e",
         "ctime" : 1493675383
      },
      {
         "s2" : "lQIExCC9OphaXQ4R/3FEiLgM2dNBJLQxFqUF07C9le5yH0ZvDsQg6+RKL+GXEyR/P9SPAYwnzDwFEHGD8HUGhaYgkiWB3H4K"
      },
      {
         "sig_version" : 2,
         "kid" : "012070ea0a0a67da7b92f47240d04b476a6aa7696a9d3d43c72553c2dc137f32c4bc0a",
         "seqno" : 5,
         "ctime" : 1493675583,
         "payload_hash" : "4818c31aeb5c8816576903c26d6ddd0d500bc4321eb52fdaf9b9cb29a88422e5",
         "sig" : "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgcOoKCmfae5L0ckDQS0dqaqdpap09Q8clU8LcE38yxLwKp3BheWxvYWTESJUCBcQgJ+z/UW9AGo9303TYjdU6Fo6rAyGEoAxj0idsR7Da3CPEIFlaqT/Mm6EakOOu+A8H5ffNIVd7/89PE1PLXnTCNdojCqNzaWfEQJ8irRbyWZSkBfWKoTnMz4c4HoUmjcOXeEg7AQVdMOSADa150SgtCTRTpIAtQonqcXfC0dpL4dvxRBJVwTlKJQ6oc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
         "payload_json" : "{\"body\":{\"cryptocurrency\":{\"address\":\"1AVfvjjuCCeZ3ASz2UMGQU8vHXVJ8TKZnw\",\"type\":\"bitcoin\"},\"key\":{\"eldest_kid\":\"012022e7c67d1d9e2d7b653f776a44f0839987062a19902bd0a63fd395b76f75ed940a\",\"host\":\"keybase.io\",\"kid\":\"012070ea0a0a67da7b92f47240d04b476a6aa7696a9d3d43c72553c2dc137f32c4bc0a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"revoke\":{\"sig_id\":\"d9a47ca4777143989c0a1f698c2cda90ee596563aed0587b966eabfdc8afd4040f\"},\"type\":\"cryptocurrency\",\"version\":2},\"ctime\":1493675583,\"expire_in\":314496000,\"prev\":\"27ecff516f401a8f77d374d88dd53a168eab032184a00c63d2276c47b0dadc23\",\"seq_type\":1,\"seqno\":5,\"tag\":\"signature\"}",
         "sig_id" : "f174626e8061380c81f25ebabda3337ef6ed5df1379a340ad7a15ab8b726713b0f",
         "prev" : "27ecff516f401a8f77d374d88dd53a168eab032184a00c63d2276c47b0dadc23"
      },
      {
         "s2" : "lQIGxCBIGMMa61yIFldpA8Jtbd0NUAvEMh61L9r5ucspqIQi5cQgLvFTZZLk5ifEf1njJQFy4mL0WsXuWjasLnai2hddhKkG"
      },
      {
         "sig" : "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgcOoKCmfae5L0ckDQS0dqaqdpap09Q8clU8LcE38yxLwKp3BheWxvYWTESJUCB8QgpsNtSHXU8fH1Ca2sWuXGWJVQzwiMe1wxdv3do38KSLnEINin3FHiPMqsV7ix+nELWAYmVKQwB33xQ96BXm5aV4w3BqNzaWfEQCzqQhYvzab/cHDYgP+RereZunQxEtdjDubC66EfCPzHn16g9ZNEH3zhkm7gpAzUSk6164krt/GwKb69GhwD/Aioc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
         "payload_hash" : "f83f38a24d83ff8698a79d5b72ce4b50f55620b37a4c4f9ca2b9194b8e20dbf8",
         "ctime" : 1493675783,
         "prev" : "a6c36d4875d4f1f1f509adac5ae5c6589550cf088c7b5c3176fddda37f0a48b9",
         "sig_id" : "110722a532eccc525cb472059066738b4c11bcbd355f0f9d44e31e0d6d479f730f",
         "payload_json" : "{\"body\":{\"cryptocurrency\":{\"address\":\"1Nc7PMZmbGXJYL6mNiLRuBXhzb9xeR4fp7\",\"type\":\"bitcoin\"},\"key\":{\"eldest_kid\":\"012022e7c67d1d9e2d7b653f776a44f0839987062a19902bd0a63fd395b76f75ed940a\",\"host\":\"keybase.io\",\"kid\":\"012070ea0a0a67da7b92f47240d04b476a6aa7696a9d3d43c72553c2dc137f32c4bc0a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"type\":\"cryptocurrency\",\"version\":2},\"ctime\":1493675783,\"expire_in\":314496000,\"prev\":\"a6c36d4875d4f1f1f509adac5ae5c6589550cf088c7b5c3176fddda37f0a48b9\",\"seq_type\":1,\"seqno\":7,\"tag\":\"signature\"}",
         "seqno" : 7,
         "kid" : "012070ea0a0a67da7b92f47240d04b476a6aa7696a9d3d43c72553c2dc137f32c4bc0a",
         "sig_version" : 2
      }
   ],
   "keys" : [
      "012022e7c67d1d9e2d7b653f776a44f0839987062a19902bd0a63fd395b76f75ed940a",
      "012070ea0a0a67da7b92f47240d04b476a6aa7696a9d3d43c72553c2dc137f32c4bc0a"
   ],
   "uid" : "74c38cf7ceb947f5632045d8ca5d4819",
   "label_sigs" : {
      "btc5" : "110722a532eccc525cb472059066738b4c11bcbd355f0f9d44e31e0d6d479f730f",
      "sib1" : "6c3d68cacab47cab4154e2bb0fc4b6db2f65f2965ad19e6eff6f409bdb6820b00f",
      "btc1" : "801190f327fa34b167e927d60452bd4ebc9a27589f185cb31e72ab89ee4aea260f",
      "btc4" : "0b5dc1869c5d82622056f6ac1d6809ac68e4982417e0245a6e83cc3e24fc1d100f",
      "btc2" : "d9a47ca4777143989c0a1f698c2cda90ee596563aed0587b966eabfdc8afd4040f",
      "e" : "7d1e012725f42e9643ca4208fc7626d3cf28daf62163c5ce5214296280d4f76f0f",
      "btc3" : "f174626e8061380c81f25ebabda3337ef6ed5df1379a340ad7a15ab8b726713b0f"
   },
   "label_kids" : {
      "sib1" : "012070ea0a0a67da7b92f47240d04b476a6aa7696a9d3d43c72553c2dc137f32c4bc0a",
      "e" : "012022e7c67d1d9e2d7b653f776a44f0839987062a19902bd0a63fd395b76f75ed940a"
   }
}
`,
	"bad_v2_type_mismatch.json": `{
   "keys" : [
      "01206a5dfef82fe91be01903186863046885941ba97b315a2b0f306d6acf77512b850a",
      "0120d469cbe5e890fe71969d3dea3978227dbaa5a925ea4f1718f2204674d8b33deb0a"
   ],
   "username" : "max32",
   "label_sigs" : {
      "e" : "e78f21d4da055e7835aa8cbd9fff49d2dea437358e17a1fedf9ba54e9f67cd330f",
      "sib1" : "818e1c458ccb45bc3b5155e2fd6d2c2d59040a4fecc97892c37006795c7c0b630f"
   },
   "label_kids" : {
      "e" : "01206a5dfef82fe91be01903186863046885941ba97b315a2b0f306d6acf77512b850a",
      "sib1" : "0120d469cbe5e890fe71969d3dea3978227dbaa5a925ea4f1718f2204674d8b33deb0a"
   },
   "chain" : [
      {
         "ctime" : 1493675228,
         "prev" : null,
         "payload_hash" : "60a7aae72b4a338b8db4e0af86421d7ad5952bfe8e8e50a30d6062fc7757ca9c",
         "seqno" : 1,
         "sig_version" : 2,
         "payload_json" : "{\"body\":{\"key\":{\"host\":\"keybase.io\",\"kid\":\"01206a5dfef82fe91be01903186863046885941ba97b315a2b0f306d6acf77512b850a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"type\":\"eldest\",\"version\":2},\"ctime\":1493675228,\"expire_in\":314496000,\"prev\":null,\"seq_type\":1,\"seqno\":1,\"tag\":\"signature\"}",
         "kid" : "01206a5dfef82fe91be01903186863046885941ba97b315a2b0f306d6acf77512b850a",
         "sig" : "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgal3++C/pG+AZAxhoYwRohZQbqXsxWisPMG1qz3dRK4UKp3BheWxvYWTEJ5UCAcDEIB+IaMCjpQ3BLZz+4t10Us59JmNYsqrO8Ufu0Fz4+lG1AaNzaWfEQPTupWCbU0Kob+HsX2dg0J8DJderjbncv+1p9IJ0a9wsne6GHo/oFv9bfHqf/5UFsxxTXcQ9O83Cli/pUpQwqwCoc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
         "sig_id" : "e78f21d4da055e7835aa8cbd9fff49d2dea437358e17a1fedf9ba54e9f67cd330f"
      },
      {
         "sig_id" : "818e1c458ccb45bc3b5155e2fd6d2c2d59040a4fecc97892c37006795c7c0b630f",
         "payload_json" : "{\"body\":{\"key\":{\"eldest_kid\":\"01206a5dfef82fe91be01903186863046885941ba97b315a2b0f306d6acf77512b850a\",\"host\":\"keybase.io\",\"kid\":\"01206a5dfef82fe91be01903186863046885941ba97b315a2b0f306d6acf77512b850a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"sibkey\":{\"kid\":\"0120d469cbe5e890fe71969d3dea3978227dbaa5a925ea4f1718f2204674d8b33deb0a\",\"reverse_sig\":\"g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEg1GnL5eiQ/nGWnT3qOXgifbqlqSXqTxcY8iBGdNizPesKp3BheWxvYWTFAix7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwNmE1ZGZlZjgyZmU5MWJlMDE5MDMxODY4NjMwNDY4ODU5NDFiYTk3YjMxNWEyYjBmMzA2ZDZhY2Y3NzUxMmI4NTBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwNmE1ZGZlZjgyZmU5MWJlMDE5MDMxODY4NjMwNDY4ODU5NDFiYTk3YjMxNWEyYjBmMzA2ZDZhY2Y3NzUxMmI4NTBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInNpYmtleSI6eyJraWQiOiIwMTIwZDQ2OWNiZTVlODkwZmU3MTk2OWQzZGVhMzk3ODIyN2RiYWE1YTkyNWVhNGYxNzE4ZjIyMDQ2NzRkOGIzM2RlYjBhIiwicmV2ZXJzZV9zaWciOm51bGx9LCJ0eXBlIjoic2lia2V5IiwidmVyc2lvbiI6MX0sImN0aW1lIjoxNDkzNjc1MzI4LCJleHBpcmVfaW4iOjMxNDQ5NjAwMCwicHJldiI6IjYwYTdhYWU3MmI0YTMzOGI4ZGI0ZTBhZjg2NDIxZDdhZDU5NTJiZmU4ZThlNTBhMzBkNjA2MmZjNzc1N2NhOWMiLCJzZXFfdHlwZSI6MSwic2Vxbm8iOjIsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RAUfh5XLSTEvg03RxvFUQnInsMWg+UIm3diDDrKsohNWsH8/GVTZ5hqwjPazb6AwYOiMjs+ZC08QyykKnTYBN1CahzaWdfdHlwZSCjdGFnzQICp3ZlcnNpb24B\"},\"type\":\"sibkey\",\"version\":2},\"ctime\":1493675328,\"expire_in\":314496000,\"prev\":\"60a7aae72b4a338b8db4e0af86421d7ad5952bfe8e8e50a30d6062fc7757ca9c\",\"seq_type\":1,\"seqno\":2,\"tag\":\"signature\"}",
         "sig" : "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgal3++C/pG+AZAxhoYwRohZQbqXsxWisPMG1qz3dRK4UKp3BheWxvYWTESJUCAsQgYKeq5ytKM4uNtOCvhkIdetWVK/6OjlCjDWBi/HdXypzEIBd4cxkZ30vfDI1ryrGeXs1Mqi1alA7NCLfvLbguRFqxDKNzaWfEQDgJYbYcwbUEXsww7rfCyPcKl/2Yft95yJ9y3n7nNwlaF2Nc7owKIeRP6pMZqYKN91DeFcj57yjyIg5zAJ2TeQCoc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
         "kid" : "01206a5dfef82fe91be01903186863046885941ba97b315a2b0f306d6acf77512b850a",
         "sig_version" : 2,
         "ctime" : 1493675328,
         "prev" : "60a7aae72b4a338b8db4e0af86421d7ad5952bfe8e8e50a30d6062fc7757ca9c",
         "payload_hash" : "fca882e0cfb47336d8232369f872c6fcc9ca3792de5d8248b8f0620a146521ab",
         "seqno" : 2
      }
   ],
   "uid" : "74c38cf7ceb947f5632045d8ca5d4819"
}
`,
	"bad_v2_wrong_payload_hash.json": `{
    "chain": [
        {
            "seqno": 1,
            "prev": null,
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgGZ4p8tHU0MFjgGYEjX4znHOTYzBCO1aibwgBnMRilnMKp3BheWxvYWTEJ5UCAcDEIP6LWvyC7XA2H2L960wn7/XBK5SMLoH3grNDL2iJ3Xx/AaNzaWfEQJ1DeYXbvoyduJcmZ+SFI+hvqiXOxDXwyCLEIuUt48TA1Al22XldtKAOiWkf7D1Eswm6JJCd0+rkIt19BfHXGQGoc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
            "payload_hash": "362b32b821134b306f8df67526573cfd3388e55e2cd1285fcd79f06b770801f3",
            "sig_id": "f91210ba18f32f22805c14ee08f1893179638430af61cd668122730670df00a40f",
            "payload_json": "{\"body\":{\"key\":{\"host\":\"keybase.io\",\"kid\":\"0120199e29f2d1d4d0c1638066048d7e339c73936330423b56a26f08019cc46296730a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"type\":\"eldest\",\"version\":2},\"ctime\":1493691200,\"expire_in\":314496000,\"prev\":null,\"seq_type\":1,\"seqno\":1,\"tag\":\"signature\"}",
            "kid": "0120199e29f2d1d4d0c1638066048d7e339c73936330423b56a26f08019cc46296730a",
            "ctime": 1493691200,
            "sig_version": 2
        },
        {
            "seqno": 2,
            "prev": "362b32b821134b306f8df67526573cfd3388e55e2cd1285fcd79f06b770801f3",
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgGZ4p8tHU0MFjgGYEjX4znHOTYzBCO1aibwgBnMRilnMKp3BheWxvYWTESJUCAsQgNisyuCETSzBvjfZ1Jlc8/TOI5V4s0ShfzXnwa3cIAfPEIAHL0boTUSDsdDEjYo9DACTqt/U++cKFYBjj3ILYs24EC6NzaWfEQOtHktRINiLP02t3vLtC4mmjny89Zt1zT+BcM6g+uErDMmdS+ioLvBIkSbH7XTqVwIoyaGKH6VS1naYvIlfhgQuoc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
            "payload_hash": "8492c482f16a5fec83c9379fbe67247eaf0be30fa99fe091945381bc658896af",
            "sig_id": "37715e053fd362577ec3a5baf2b067fc43911f73de7be41bc0550205cf1c967e0f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"0120199e29f2d1d4d0c1638066048d7e339c73936330423b56a26f08019cc46296730a\",\"host\":\"keybase.io\",\"kid\":\"0120199e29f2d1d4d0c1638066048d7e339c73936330423b56a26f08019cc46296730a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"sibkey\":{\"kid\":\"012065e9413101be385fb27e5a6a74d4168fe0b54f90ba5ba3811be734279dad45920a\",\"reverse_sig\":\"g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgZelBMQG+OF+yflpqdNQWj+C1T5C6W6OBG+c0J52tRZIKp3BheWxvYWTFAix7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwMTk5ZTI5ZjJkMWQ0ZDBjMTYzODA2NjA0OGQ3ZTMzOWM3MzkzNjMzMDQyM2I1NmEyNmYwODAxOWNjNDYyOTY3MzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwMTk5ZTI5ZjJkMWQ0ZDBjMTYzODA2NjA0OGQ3ZTMzOWM3MzkzNjMzMDQyM2I1NmEyNmYwODAxOWNjNDYyOTY3MzBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInNpYmtleSI6eyJraWQiOiIwMTIwNjVlOTQxMzEwMWJlMzg1ZmIyN2U1YTZhNzRkNDE2OGZlMGI1NGY5MGJhNWJhMzgxMWJlNzM0Mjc5ZGFkNDU5MjBhIiwicmV2ZXJzZV9zaWciOm51bGx9LCJ0eXBlIjoic2lia2V5IiwidmVyc2lvbiI6Mn0sImN0aW1lIjoxNDkzNjkxMzAwLCJleHBpcmVfaW4iOjMxNDQ5NjAwMCwicHJldiI6IjM2MmIzMmI4MjExMzRiMzA2ZjhkZjY3NTI2NTczY2ZkMzM4OGU1NWUyY2QxMjg1ZmNkNzlmMDZiNzcwODAxZjMiLCJzZXFfdHlwZSI6MSwic2Vxbm8iOjIsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RAE/I1fyQ14A9p+BE1iH8/l5pGkn+GqJaipWjIvhDvATOvSIKIbH1vLaFxF0exf98M59DzRlqTvQgxk/mh7Kp8AKhzaWdfdHlwZSCjdGFnzQICp3ZlcnNpb24B\"},\"type\":\"sibkey\",\"version\":2},\"ctime\":1493691300,\"expire_in\":314496000,\"prev\":\"362b32b821134b306f8df67526573cfd3388e55e2cd1285fcd79f06b770801f3\",\"seq_type\":1,\"seqno\":2,\"tag\":\"signature\"}",
            "kid": "0120199e29f2d1d4d0c1638066048d7e339c73936330423b56a26f08019cc46296730a",
            "ctime": 1493691300,
            "sig_version": 2
        }
    ],
    "keys": [
        "0120199e29f2d1d4d0c1638066048d7e339c73936330423b56a26f08019cc46296730a",
        "012065e9413101be385fb27e5a6a74d4168fe0b54f90ba5ba3811be734279dad45920a"
    ],
    "uid": "74c38cf7ceb947f5632045d8ca5d4819",
    "username": "max32",
    "label_kids": {
        "e": "0120199e29f2d1d4d0c1638066048d7e339c73936330423b56a26f08019cc46296730a",
        "sib1": "012065e9413101be385fb27e5a6a74d4168fe0b54f90ba5ba3811be734279dad45920a"
    },
    "label_sigs": {
        "e": "f91210ba18f32f22805c14ee08f1893179638430af61cd668122730670df00a40f",
        "sib1": "37715e053fd362577ec3a5baf2b067fc43911f73de7be41bc0550205cf1c967e0f"
    }
}`,
	"empty_chain.json": `{
    "_comment": "The 'ralph' key has ralph@keybase.io signed into it. The 'steve' key does not. This is for testing key ownership when the sigchain in empty.",
    "username": "ralph",
    "uid": "bf65266d0d8df3ad5d1b367f578e6819",
    "chain": [],
    "keys": [
        "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: GnuPG v2\n\nmI0EVXnWkQEEALmSERRK4+fbTg5C28GOXxpiI/IfQZDFmJfWzNKNW6CrX1uj4o8G\nu3292bU5meCRCJstd09LFaELOASbknUOiXFQcnuzpY7J9Ou8uqxKcOYxi4xWjXih\nK8KQfL4khBIpB+7B0tp16qalqbd1+V2wIAJzmNdkY+cFGeXZE45gr+i3ABEBAAG0\nGHJhbHBoIDxyYWxwaEBrZXliYXNlLmlvPoi5BBMBCAAjBQJVedaRAhsDBwsJCAcD\nAgEGFQgCCQoLBBYCAwECHgECF4AACgkQUGs4lh8OKGN+TgQAhTSvOZV+8me6UuIc\ntBJTsZzAkzWsP3QpdSZ6zUyZ40yA5WMZMpnQb+6kd3xzo/tkm9imTK2xLtoCj52d\n3QZRN3BOqweetaiWng6sgIkMZc8EaQp68bPeVhtcAYU0Y/I9Jcim6d2nSXQcwuph\na3RWghsf37A9CfXlbZl36PvQ2ko=\n=qGhY\n-----END PGP PUBLIC KEY BLOCK-----\n",
        "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: GnuPG v2\n\nmI0EVXnWyQEEAMUSBo5Q/X3et/ExeX6UHvsH9Wv5IoaEmMqdT7zSxaphcFS+vo5E\n63QQrokfIUKZeMDmWjI65SNVNhpWng5mhrkz/bRql0pdikhwXnALLVnwT+0J/Spl\nyWiQKVfZ6pqz2anto56/rlhLui7oTLnG7q2CttIaAp1aaMzaHwMH4iW1ABEBAAG0\nBXN0ZXZliLkEEwEIACMFAlV51skCGwMHCwkIBwMCAQYVCAIJCgsEFgIDAQIeAQIX\ngAAKCRA0feCGaTASHtOGA/4u7MAY2z3UzVKzH4XxYknep4hD+75HH6TY1fCUHBcS\nAfVptSna59alV37JJtsBQhmNowyIqvNeUSl3VcN2oOBvJwxGyPhoh6Roxj+4phhO\nPGH9dzzaZVQVSYMBMU8Wuf9OsEM/2k7Wo2oes9XLiNVBqYNwvk6Q2BupT77FyzDI\ntg==\n=zpuJ\n-----END PGP PUBLIC KEY BLOCK-----\n",
        "0120a4dabfb39e2bc635eadc861b72d9f9512dfe4261562822433d261718235d5b6f0a"
    ],
    "label_kids": {
        "ralph": "0101de638c5bee00e1b473b5f3fa7065e55cde33e88d230a2cd253da5d78a1c5a8910a",
        "steve": "0101d17f13d98d9a21e3fcc2ba30ef38fe1ca0e805950839f7ecfb8b77ad5d128e4f0a",
        "nonexistent": "0101deadbeef0000000000000000000000000000000000000000000000000000000000",
        "naclkey": "0120a4dabfb39e2bc635eadc861b72d9f9512dfe4261562822433d261718235d5b6f0a"
    }
}
`,
	"example_revokes_chain.json": `{
    "chain": [
        {
            "seqno": 1,
            "prev": null,
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgKvldOpi5fFXv0twJsMnGBv0LXqp+NbWaV14zbexQTUkKp3BheWxvYWTFASp7ImJvZHkiOnsia2V5Ijp7Imhvc3QiOiJrZXliYXNlLmlvIiwia2lkIjoiMDEyMDJhZjk1ZDNhOThiOTdjNTVlZmQyZGMwOWIwYzljNjA2ZmQwYjVlYWE3ZTM1YjU5YTU3NWUzMzZkZWM1MDRkNDkwYSIsInVpZCI6Ijc0YzM4Y2Y3Y2ViOTQ3ZjU2MzIwNDVkOGNhNWQ0ODE5IiwidXNlcm5hbWUiOiJtYXgzMiJ9LCJ0eXBlIjoiZWxkZXN0IiwidmVyc2lvbiI6MX0sImN0aW1lIjoxNDMyMTQ0NzAwLCJleHBpcmVfaW4iOjEwMDAwMDAwLCJwcmV2IjpudWxsLCJzZXFfdHlwZSI6MSwic2Vxbm8iOjEsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RAelPb6xpVpK5Kdk57aOoiA438uOVw8Ha6eHBGydFnWYLw12EWT02pKXQYnIvCVSQNOJIMeXXn99hPYLSS3cv2DKhzaWdfdHlwZSCjdGFnzQICp3ZlcnNpb24B",
            "payload_hash": "69a9b17aca289091b869bddb00385857147515fb4c335d157fd0be86f0bb5e9c",
            "sig_id": "71e44e4c690d0d7ac6bcb244d9f72cf173bfae15e3fa7d4cb0ee2f24c3d6de9f0f",
            "payload_json": "{\"body\":{\"key\":{\"host\":\"keybase.io\",\"kid\":\"01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"type\":\"eldest\",\"version\":1},\"ctime\":1432144700,\"expire_in\":10000000,\"prev\":null,\"seq_type\":1,\"seqno\":1,\"tag\":\"signature\"}",
            "kid": "01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a",
            "ctime": 1432144700
        },
        {
            "seqno": 2,
            "prev": "69a9b17aca289091b869bddb00385857147515fb4c335d157fd0be86f0bb5e9c",
            "sig": "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",
            "payload_hash": "29ef238a6283a937300a13b3e5542a92e3b9014380c0798979a9b27133d6c8ee",
            "sig_id": "77a2a70d5c8af0f1201027076504065e80ddd6d5ff2a622eb57e149dc8fa00470f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a\",\"host\":\"keybase.io\",\"kid\":\"01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"sibkey\":{\"kid\":\"0120d20fe982e4e7ba1a08dd60dd527a027bbe0225d4fe89eef979020c5b9efedbfb0a\",\"reverse_sig\":\"g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEg0g/pguTnuhoI3WDdUnoCe74CJdT+ie75eQIMW57+2/sKp3BheWxvYWTFAit7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwMmFmOTVkM2E5OGI5N2M1NWVmZDJkYzA5YjBjOWM2MDZmZDBiNWVhYTdlMzViNTlhNTc1ZTMzNmRlYzUwNGQ0OTBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwMmFmOTVkM2E5OGI5N2M1NWVmZDJkYzA5YjBjOWM2MDZmZDBiNWVhYTdlMzViNTlhNTc1ZTMzNmRlYzUwNGQ0OTBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInNpYmtleSI6eyJraWQiOiIwMTIwZDIwZmU5ODJlNGU3YmExYTA4ZGQ2MGRkNTI3YTAyN2JiZTAyMjVkNGZlODllZWY5NzkwMjBjNWI5ZWZlZGJmYjBhIiwicmV2ZXJzZV9zaWciOm51bGx9LCJ0eXBlIjoic2lia2V5IiwidmVyc2lvbiI6MX0sImN0aW1lIjoxNDMyMTQ0ODAwLCJleHBpcmVfaW4iOjEwMDAwMDAwLCJwcmV2IjoiNjlhOWIxN2FjYTI4OTA5MWI4NjliZGRiMDAzODU4NTcxNDc1MTVmYjRjMzM1ZDE1N2ZkMGJlODZmMGJiNWU5YyIsInNlcV90eXBlIjoxLCJzZXFubyI6MiwidGFnIjoic2lnbmF0dXJlIn2jc2lnxEBjqhTAjgisDTZInxciSQyXFdMBVRal98GzY1ZuVLpUOU5gyr7Hk8xrpRH7fPtXpWGRp+uZsigkVVmU/1mrLcAOqHNpZ190eXBlIKN0YWfNAgKndmVyc2lvbgE=\"},\"type\":\"sibkey\",\"version\":1},\"ctime\":1432144800,\"expire_in\":10000000,\"prev\":\"69a9b17aca289091b869bddb00385857147515fb4c335d157fd0be86f0bb5e9c\",\"seq_type\":1,\"seqno\":2,\"tag\":\"signature\"}",
            "kid": "01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a",
            "ctime": 1432144800
        },
        {
            "seqno": 3,
            "prev": "29ef238a6283a937300a13b3e5542a92e3b9014380c0798979a9b27133d6c8ee",
            "sig": "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",
            "payload_hash": "c3deb587280bf27e7b7554c0304c5b58c4183d888306c65386d7a57287ba0a8d",
            "sig_id": "e65e67f0e1748ccfb4c1b05c84c87eb8311771a5d6bb908093edfb68a4e691180f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a\",\"host\":\"keybase.io\",\"kid\":\"0120d20fe982e4e7ba1a08dd60dd527a027bbe0225d4fe89eef979020c5b9efedbfb0a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"sibkey\":{\"kid\":\"012031fe75cae16e364949f99b6fc4fe79fab0b07d1234dd50b9eafb0d731571bf6c0a\",\"reverse_sig\":\"g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgMf51yuFuNklJ+ZtvxP55+rCwfRI03VC56vsNcxVxv2wKp3BheWxvYWTFAit7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwMmFmOTVkM2E5OGI5N2M1NWVmZDJkYzA5YjBjOWM2MDZmZDBiNWVhYTdlMzViNTlhNTc1ZTMzNmRlYzUwNGQ0OTBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwZDIwZmU5ODJlNGU3YmExYTA4ZGQ2MGRkNTI3YTAyN2JiZTAyMjVkNGZlODllZWY5NzkwMjBjNWI5ZWZlZGJmYjBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInNpYmtleSI6eyJraWQiOiIwMTIwMzFmZTc1Y2FlMTZlMzY0OTQ5Zjk5YjZmYzRmZTc5ZmFiMGIwN2QxMjM0ZGQ1MGI5ZWFmYjBkNzMxNTcxYmY2YzBhIiwicmV2ZXJzZV9zaWciOm51bGx9LCJ0eXBlIjoic2lia2V5IiwidmVyc2lvbiI6MX0sImN0aW1lIjoxNDMyMTQ0OTAwLCJleHBpcmVfaW4iOjEwMDAwMDAwLCJwcmV2IjoiMjllZjIzOGE2MjgzYTkzNzMwMGExM2IzZTU1NDJhOTJlM2I5MDE0MzgwYzA3OTg5NzlhOWIyNzEzM2Q2YzhlZSIsInNlcV90eXBlIjoxLCJzZXFubyI6MywidGFnIjoic2lnbmF0dXJlIn2jc2lnxEAcpFUiEVRD2yvqIa0t93Vynjdf8lSrpVXevhsHt06LeKsR+SDXa98EJo6Ym/fjpWxJj8ejsnKBLagqEtQfEjAMqHNpZ190eXBlIKN0YWfNAgKndmVyc2lvbgE=\"},\"type\":\"sibkey\",\"version\":1},\"ctime\":1432144900,\"expire_in\":10000000,\"prev\":\"29ef238a6283a937300a13b3e5542a92e3b9014380c0798979a9b27133d6c8ee\",\"seq_type\":1,\"seqno\":3,\"tag\":\"signature\"}",
            "kid": "0120d20fe982e4e7ba1a08dd60dd527a027bbe0225d4fe89eef979020c5b9efedbfb0a",
            "ctime": 1432144900
        },
        {
            "seqno": 4,
            "prev": "c3deb587280bf27e7b7554c0304c5b58c4183d888306c65386d7a57287ba0a8d",
            "sig": "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",
            "payload_hash": "e89c2dc5a08333fe90d9af7dcb5ec251feb72f87daf44b281ffc9645bb49f025",
            "sig_id": "506bf48c368a84c9e9ab6fc22438359241ee12784c72f2e68c2afa1d3dd71aae0f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a\",\"host\":\"keybase.io\",\"kid\":\"012031fe75cae16e364949f99b6fc4fe79fab0b07d1234dd50b9eafb0d731571bf6c0a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"sibkey\":{\"kid\":\"0120ff5aff3b56026d50f6225ff5c53fed9bb9c27e5230abe636c5e03be1c841dd760a\",\"reverse_sig\":\"g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEg/1r/O1YCbVD2Il/1xT/tm7nCflIwq+Y2xeA74chB3XYKp3BheWxvYWTFAit7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwMmFmOTVkM2E5OGI5N2M1NWVmZDJkYzA5YjBjOWM2MDZmZDBiNWVhYTdlMzViNTlhNTc1ZTMzNmRlYzUwNGQ0OTBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwMzFmZTc1Y2FlMTZlMzY0OTQ5Zjk5YjZmYzRmZTc5ZmFiMGIwN2QxMjM0ZGQ1MGI5ZWFmYjBkNzMxNTcxYmY2YzBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInNpYmtleSI6eyJraWQiOiIwMTIwZmY1YWZmM2I1NjAyNmQ1MGY2MjI1ZmY1YzUzZmVkOWJiOWMyN2U1MjMwYWJlNjM2YzVlMDNiZTFjODQxZGQ3NjBhIiwicmV2ZXJzZV9zaWciOm51bGx9LCJ0eXBlIjoic2lia2V5IiwidmVyc2lvbiI6MX0sImN0aW1lIjoxNDMyMTQ1MDAwLCJleHBpcmVfaW4iOjEwMDAwMDAwLCJwcmV2IjoiYzNkZWI1ODcyODBiZjI3ZTdiNzU1NGMwMzA0YzViNThjNDE4M2Q4ODgzMDZjNjUzODZkN2E1NzI4N2JhMGE4ZCIsInNlcV90eXBlIjoxLCJzZXFubyI6NCwidGFnIjoic2lnbmF0dXJlIn2jc2lnxECsK0hSuzQTJU/RN3M762G2t7ckJMCDkUK7jA6rg3pFroy0XEZwLhPAiM87mzUVTBQHe9TUHmrEof9aehU864IDqHNpZ190eXBlIKN0YWfNAgKndmVyc2lvbgE=\"},\"type\":\"sibkey\",\"version\":1},\"ctime\":1432145000,\"expire_in\":10000000,\"prev\":\"c3deb587280bf27e7b7554c0304c5b58c4183d888306c65386d7a57287ba0a8d\",\"seq_type\":1,\"seqno\":4,\"tag\":\"signature\"}",
            "kid": "012031fe75cae16e364949f99b6fc4fe79fab0b07d1234dd50b9eafb0d731571bf6c0a",
            "ctime": 1432145000
        },
        {
            "seqno": 5,
            "prev": "e89c2dc5a08333fe90d9af7dcb5ec251feb72f87daf44b281ffc9645bb49f025",
            "sig": "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",
            "payload_hash": "33b022b996f1faaff00ad6eafdb9f5745ec0c0baa08588cab74017edeaf0f233",
            "sig_id": "b0d3a36fc061b317a260514eff8f5ac1b8ab1cfc60c54731e79e979f9e638b320f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a\",\"host\":\"keybase.io\",\"kid\":\"0120ff5aff3b56026d50f6225ff5c53fed9bb9c27e5230abe636c5e03be1c841dd760a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"sibkey\":{\"kid\":\"0120818b638074d0b15e766af7a61c08fb5fd2314cf0e2d0d5a104d8347d21e6f5ed0a\",\"reverse_sig\":\"g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEggYtjgHTQsV52avemHAj7X9IxTPDi0NWhBNg0fSHm9e0Kp3BheWxvYWTFAit7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwMmFmOTVkM2E5OGI5N2M1NWVmZDJkYzA5YjBjOWM2MDZmZDBiNWVhYTdlMzViNTlhNTc1ZTMzNmRlYzUwNGQ0OTBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwZmY1YWZmM2I1NjAyNmQ1MGY2MjI1ZmY1YzUzZmVkOWJiOWMyN2U1MjMwYWJlNjM2YzVlMDNiZTFjODQxZGQ3NjBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInNpYmtleSI6eyJraWQiOiIwMTIwODE4YjYzODA3NGQwYjE1ZTc2NmFmN2E2MWMwOGZiNWZkMjMxNGNmMGUyZDBkNWExMDRkODM0N2QyMWU2ZjVlZDBhIiwicmV2ZXJzZV9zaWciOm51bGx9LCJ0eXBlIjoic2lia2V5IiwidmVyc2lvbiI6MX0sImN0aW1lIjoxNDMyMTQ1MTAwLCJleHBpcmVfaW4iOjEwMDAwMDAwLCJwcmV2IjoiZTg5YzJkYzVhMDgzMzNmZTkwZDlhZjdkY2I1ZWMyNTFmZWI3MmY4N2RhZjQ0YjI4MWZmYzk2NDViYjQ5ZjAyNSIsInNlcV90eXBlIjoxLCJzZXFubyI6NSwidGFnIjoic2lnbmF0dXJlIn2jc2lnxEBN4dDSklpCQVTgSSbfFMPEGFO+9ZF0smlN0nHRbEMP1HsOgnuTgHaqtBtAI0fO3OhZIz7mKVAeAJDI24tDrR0MqHNpZ190eXBlIKN0YWfNAgKndmVyc2lvbgE=\"},\"type\":\"sibkey\",\"version\":1},\"ctime\":1432145100,\"expire_in\":10000000,\"prev\":\"e89c2dc5a08333fe90d9af7dcb5ec251feb72f87daf44b281ffc9645bb49f025\",\"seq_type\":1,\"seqno\":5,\"tag\":\"signature\"}",
            "kid": "0120ff5aff3b56026d50f6225ff5c53fed9bb9c27e5230abe636c5e03be1c841dd760a",
            "ctime": 1432145100
        },
        {
            "seqno": 6,
            "prev": "33b022b996f1faaff00ad6eafdb9f5745ec0c0baa08588cab74017edeaf0f233",
            "sig": "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",
            "payload_hash": "81356413d54fb759fdb6f3bdbcf020c7208fba6f3635a81469f9581be2be2340",
            "sig_id": "f52a55c59023f5e188e85cf0933e0301b1143b462c7f361abcdc7989492b1bf00f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a\",\"host\":\"keybase.io\",\"kid\":\"0120818b638074d0b15e766af7a61c08fb5fd2314cf0e2d0d5a104d8347d21e6f5ed0a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"sibkey\":{\"kid\":\"0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a\",\"reverse_sig\":\"g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEg1Whs0Da7MlPMAmd1kDZIgKu5lFdleYysUKKFus2jYmAKp3BheWxvYWTFAit7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwMmFmOTVkM2E5OGI5N2M1NWVmZDJkYzA5YjBjOWM2MDZmZDBiNWVhYTdlMzViNTlhNTc1ZTMzNmRlYzUwNGQ0OTBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwODE4YjYzODA3NGQwYjE1ZTc2NmFmN2E2MWMwOGZiNWZkMjMxNGNmMGUyZDBkNWExMDRkODM0N2QyMWU2ZjVlZDBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInNpYmtleSI6eyJraWQiOiIwMTIwZDU2ODZjZDAzNmJiMzI1M2NjMDI2Nzc1OTAzNjQ4ODBhYmI5OTQ1NzY1Nzk4Y2FjNTBhMjg1YmFjZGEzNjI2MDBhIiwicmV2ZXJzZV9zaWciOm51bGx9LCJ0eXBlIjoic2lia2V5IiwidmVyc2lvbiI6MX0sImN0aW1lIjoxNDMyMTQ1MjAwLCJleHBpcmVfaW4iOjEwMDAwMDAwLCJwcmV2IjoiMzNiMDIyYjk5NmYxZmFhZmYwMGFkNmVhZmRiOWY1NzQ1ZWMwYzBiYWEwODU4OGNhYjc0MDE3ZWRlYWYwZjIzMyIsInNlcV90eXBlIjoxLCJzZXFubyI6NiwidGFnIjoic2lnbmF0dXJlIn2jc2lnxEC79ZsL2owjcjaW+hKyjeTX01rEbAeCJV1NhjPJI6PQBOgKLpQFbWsrxle82B+zfCy6BFhTlfmrELidnVWD+G4AqHNpZ190eXBlIKN0YWfNAgKndmVyc2lvbgE=\"},\"type\":\"sibkey\",\"version\":1},\"ctime\":1432145200,\"expire_in\":10000000,\"prev\":\"33b022b996f1faaff00ad6eafdb9f5745ec0c0baa08588cab74017edeaf0f233\",\"seq_type\":1,\"seqno\":6,\"tag\":\"signature\"}",
            "kid": "0120818b638074d0b15e766af7a61c08fb5fd2314cf0e2d0d5a104d8347d21e6f5ed0a",
            "ctime": 1432145200
        },
        {
            "seqno": 7,
            "prev": "81356413d54fb759fdb6f3bdbcf020c7208fba6f3635a81469f9581be2be2340",
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEg1Whs0Da7MlPMAmd1kDZIgKu5lFdleYysUKKFus2jYmAKp3BheWxvYWTFAoF7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwMmFmOTVkM2E5OGI5N2M1NWVmZDJkYzA5YjBjOWM2MDZmZDBiNWVhYTdlMzViNTlhNTc1ZTMzNmRlYzUwNGQ0OTBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwZDU2ODZjZDAzNmJiMzI1M2NjMDI2Nzc1OTAzNjQ4ODBhYmI5OTQ1NzY1Nzk4Y2FjNTBhMjg1YmFjZGEzNjI2MDBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInN1YmtleSI6eyJraWQiOiIwMTIxMzBiZjU3Y2MwMTY4ZjJhZGQzYTYxZGI1ZmNiOTNkMTlmN2MwZTQyM2EyNjZmZDVkNWE5OWQ1NmQ4MWMzNWM1NTBhIiwicGFyZW50X2tpZCI6IjAxMjBkNTY4NmNkMDM2YmIzMjUzY2MwMjY3NzU5MDM2NDg4MGFiYjk5NDU3NjU3OThjYWM1MGEyODViYWNkYTM2MjYwMGEiLCJyZXZlcnNlX3NpZyI6bnVsbH0sInR5cGUiOiJzdWJrZXkiLCJ2ZXJzaW9uIjoxfSwiY3RpbWUiOjE0MzIxNDUzMDAsImV4cGlyZV9pbiI6MTAwMDAwMDAsInByZXYiOiI4MTM1NjQxM2Q1NGZiNzU5ZmRiNmYzYmRiY2YwMjBjNzIwOGZiYTZmMzYzNWE4MTQ2OWY5NTgxYmUyYmUyMzQwIiwic2VxX3R5cGUiOjEsInNlcW5vIjo3LCJ0YWciOiJzaWduYXR1cmUifaNzaWfEQOhVHMGds7KBnZCG0IkbF7E8nEpHYhoFI2s54a69Cl8LikH6JdcCkAkFrHM7Oeu2gOFIHk5/w3XU7z51n+DKBw6oc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
            "payload_hash": "ba371fdc2fc138718b7183204c92a2ae0eb1f7b3717f457a68ac7add0ff34bcb",
            "sig_id": "81824a207a2a3ecbc958074167346990f0f5e61fd4b0ebd22b7c7b1b0987b7920f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a\",\"host\":\"keybase.io\",\"kid\":\"0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"subkey\":{\"kid\":\"012130bf57cc0168f2add3a61db5fcb93d19f7c0e423a266fd5d5a99d56d81c35c550a\",\"parent_kid\":\"0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a\",\"reverse_sig\":null},\"type\":\"subkey\",\"version\":1},\"ctime\":1432145300,\"expire_in\":10000000,\"prev\":\"81356413d54fb759fdb6f3bdbcf020c7208fba6f3635a81469f9581be2be2340\",\"seq_type\":1,\"seqno\":7,\"tag\":\"signature\"}",
            "kid": "0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a",
            "ctime": 1432145300
        },
        {
            "seqno": 8,
            "prev": "ba371fdc2fc138718b7183204c92a2ae0eb1f7b3717f457a68ac7add0ff34bcb",
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEg1Whs0Da7MlPMAmd1kDZIgKu5lFdleYysUKKFus2jYmAKp3BheWxvYWTFAoF7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwMmFmOTVkM2E5OGI5N2M1NWVmZDJkYzA5YjBjOWM2MDZmZDBiNWVhYTdlMzViNTlhNTc1ZTMzNmRlYzUwNGQ0OTBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwZDU2ODZjZDAzNmJiMzI1M2NjMDI2Nzc1OTAzNjQ4ODBhYmI5OTQ1NzY1Nzk4Y2FjNTBhMjg1YmFjZGEzNjI2MDBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInN1YmtleSI6eyJraWQiOiIwMTIxNTYwMWM0YzRiNDZlMTlmMzU2NTRhMWRlYmM4ZGMwZTNmNjgwMTNjMzUxN2ViODA5ZTVmMzY1NDE3MjZmZDMzNjBhIiwicGFyZW50X2tpZCI6IjAxMjBkNTY4NmNkMDM2YmIzMjUzY2MwMjY3NzU5MDM2NDg4MGFiYjk5NDU3NjU3OThjYWM1MGEyODViYWNkYTM2MjYwMGEiLCJyZXZlcnNlX3NpZyI6bnVsbH0sInR5cGUiOiJzdWJrZXkiLCJ2ZXJzaW9uIjoxfSwiY3RpbWUiOjE0MzIxNDU0MDAsImV4cGlyZV9pbiI6MTAwMDAwMDAsInByZXYiOiJiYTM3MWZkYzJmYzEzODcxOGI3MTgzMjA0YzkyYTJhZTBlYjFmN2IzNzE3ZjQ1N2E2OGFjN2FkZDBmZjM0YmNiIiwic2VxX3R5cGUiOjEsInNlcW5vIjo4LCJ0YWciOiJzaWduYXR1cmUifaNzaWfEQMUM7CpuUEq39bXVQE8Idxp7Ezolki56TrLsua40gmrzdj7bTFmj0YeEqRegn50dxqgAzoi/eqECouYHAdxb7Amoc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
            "payload_hash": "f28bad7570323729cbfbb3cd945e03b4ab9957450006fc339e297f3788add3c0",
            "sig_id": "a535a3972951c2077647d567949c73209672afd5c586fe7a0f4407d5894099000f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a\",\"host\":\"keybase.io\",\"kid\":\"0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"subkey\":{\"kid\":\"01215601c4c4b46e19f35654a1debc8dc0e3f68013c3517eb809e5f36541726fd3360a\",\"parent_kid\":\"0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a\",\"reverse_sig\":null},\"type\":\"subkey\",\"version\":1},\"ctime\":1432145400,\"expire_in\":10000000,\"prev\":\"ba371fdc2fc138718b7183204c92a2ae0eb1f7b3717f457a68ac7add0ff34bcb\",\"seq_type\":1,\"seqno\":8,\"tag\":\"signature\"}",
            "kid": "0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a",
            "ctime": 1432145400
        },
        {
            "seqno": 9,
            "prev": "f28bad7570323729cbfbb3cd945e03b4ab9957450006fc339e297f3788add3c0",
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEg1Whs0Da7MlPMAmd1kDZIgKu5lFdleYysUKKFus2jYmAKp3BheWxvYWTFAoF7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwMmFmOTVkM2E5OGI5N2M1NWVmZDJkYzA5YjBjOWM2MDZmZDBiNWVhYTdlMzViNTlhNTc1ZTMzNmRlYzUwNGQ0OTBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwZDU2ODZjZDAzNmJiMzI1M2NjMDI2Nzc1OTAzNjQ4ODBhYmI5OTQ1NzY1Nzk4Y2FjNTBhMjg1YmFjZGEzNjI2MDBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInN1YmtleSI6eyJraWQiOiIwMTIxNWNiOTAyZTRiNmEwNzgzZjQ1NThlNGI0ODNmMDkxODUyZWI1ODVhZWNkZWMwMTkzNTZlYjgxM2FjNTA0MGQwMjBhIiwicGFyZW50X2tpZCI6IjAxMjBkNTY4NmNkMDM2YmIzMjUzY2MwMjY3NzU5MDM2NDg4MGFiYjk5NDU3NjU3OThjYWM1MGEyODViYWNkYTM2MjYwMGEiLCJyZXZlcnNlX3NpZyI6bnVsbH0sInR5cGUiOiJzdWJrZXkiLCJ2ZXJzaW9uIjoxfSwiY3RpbWUiOjE0MzIxNDU1MDAsImV4cGlyZV9pbiI6MTAwMDAwMDAsInByZXYiOiJmMjhiYWQ3NTcwMzIzNzI5Y2JmYmIzY2Q5NDVlMDNiNGFiOTk1NzQ1MDAwNmZjMzM5ZTI5N2YzNzg4YWRkM2MwIiwic2VxX3R5cGUiOjEsInNlcW5vIjo5LCJ0YWciOiJzaWduYXR1cmUifaNzaWfEQJEPySgBxHJeapOmXGKSEDP7H2EL4pB78+oM6j22Qa5CaEIzC1NPE/nGlpb3a1/iVxORoMWLohakYoHEgJVYOAaoc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
            "payload_hash": "d56391bc51ce01b9c6851edd75d7b5e8d8b3c7622c5d7600b71437075adaa232",
            "sig_id": "d5f5e75c809a6c1207d35791dd8361c68b5a7a0de9853d9ee647ba3f1620b9e00f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a\",\"host\":\"keybase.io\",\"kid\":\"0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"subkey\":{\"kid\":\"01215cb902e4b6a0783f4558e4b483f091852eb585aecdec019356eb813ac5040d020a\",\"parent_kid\":\"0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a\",\"reverse_sig\":null},\"type\":\"subkey\",\"version\":1},\"ctime\":1432145500,\"expire_in\":10000000,\"prev\":\"f28bad7570323729cbfbb3cd945e03b4ab9957450006fc339e297f3788add3c0\",\"seq_type\":1,\"seqno\":9,\"tag\":\"signature\"}",
            "kid": "0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a",
            "ctime": 1432145500
        },
        {
            "seqno": 10,
            "prev": "d56391bc51ce01b9c6851edd75d7b5e8d8b3c7622c5d7600b71437075adaa232",
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEg1Whs0Da7MlPMAmd1kDZIgKu5lFdleYysUKKFus2jYmAKp3BheWxvYWTFAhl7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwMmFmOTVkM2E5OGI5N2M1NWVmZDJkYzA5YjBjOWM2MDZmZDBiNWVhYTdlMzViNTlhNTc1ZTMzNmRlYzUwNGQ0OTBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwZDU2ODZjZDAzNmJiMzI1M2NjMDI2Nzc1OTAzNjQ4ODBhYmI5OTQ1NzY1Nzk4Y2FjNTBhMjg1YmFjZGEzNjI2MDBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInJldm9rZSI6eyJraWQiOiIwMTIwZDIwZmU5ODJlNGU3YmExYTA4ZGQ2MGRkNTI3YTAyN2JiZTAyMjVkNGZlODllZWY5NzkwMjBjNWI5ZWZlZGJmYjBhIn0sInR5cGUiOiJyZXZva2UiLCJ2ZXJzaW9uIjoxfSwiY3RpbWUiOjE0MzIxNDU2MDAsImV4cGlyZV9pbiI6MTAwMDAwMDAsInByZXYiOiJkNTYzOTFiYzUxY2UwMWI5YzY4NTFlZGQ3NWQ3YjVlOGQ4YjNjNzYyMmM1ZDc2MDBiNzE0MzcwNzVhZGFhMjMyIiwic2VxX3R5cGUiOjEsInNlcW5vIjoxMCwidGFnIjoic2lnbmF0dXJlIn2jc2lnxECvZgjAO0Re95cU0cwiw2IMaomncIU6ir6lbQup6QM789tiDoed42Xf9+vKXB+lyHmNQwPOt6Qd/cYHafREYR8HqHNpZ190eXBlIKN0YWfNAgKndmVyc2lvbgE=",
            "payload_hash": "601b3462742a14955c12017a7bbf5615cb3cad8bea04734d00dd61d492ad1952",
            "sig_id": "4d4a63b6914fc2750ff9e03f7f5f54ea62cfaf91c17d4311b0991fbedefbff760f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a\",\"host\":\"keybase.io\",\"kid\":\"0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"revoke\":{\"kid\":\"0120d20fe982e4e7ba1a08dd60dd527a027bbe0225d4fe89eef979020c5b9efedbfb0a\"},\"type\":\"revoke\",\"version\":1},\"ctime\":1432145600,\"expire_in\":10000000,\"prev\":\"d56391bc51ce01b9c6851edd75d7b5e8d8b3c7622c5d7600b71437075adaa232\",\"seq_type\":1,\"seqno\":10,\"tag\":\"signature\"}",
            "kid": "0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a",
            "ctime": 1432145600
        },
        {
            "seqno": 11,
            "prev": "601b3462742a14955c12017a7bbf5615cb3cad8bea04734d00dd61d492ad1952",
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEg1Whs0Da7MlPMAmd1kDZIgKu5lFdleYysUKKFus2jYmAKp3BheWxvYWTFAhl7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwMmFmOTVkM2E5OGI5N2M1NWVmZDJkYzA5YjBjOWM2MDZmZDBiNWVhYTdlMzViNTlhNTc1ZTMzNmRlYzUwNGQ0OTBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwZDU2ODZjZDAzNmJiMzI1M2NjMDI2Nzc1OTAzNjQ4ODBhYmI5OTQ1NzY1Nzk4Y2FjNTBhMjg1YmFjZGEzNjI2MDBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInJldm9rZSI6eyJraWQiOiIwMTIwZDIwZmU5ODJlNGU3YmExYTA4ZGQ2MGRkNTI3YTAyN2JiZTAyMjVkNGZlODllZWY5NzkwMjBjNWI5ZWZlZGJmYjBhIn0sInR5cGUiOiJyZXZva2UiLCJ2ZXJzaW9uIjoxfSwiY3RpbWUiOjE0MzIxNDU3MDAsImV4cGlyZV9pbiI6MTAwMDAwMDAsInByZXYiOiI2MDFiMzQ2Mjc0MmExNDk1NWMxMjAxN2E3YmJmNTYxNWNiM2NhZDhiZWEwNDczNGQwMGRkNjFkNDkyYWQxOTUyIiwic2VxX3R5cGUiOjEsInNlcW5vIjoxMSwidGFnIjoic2lnbmF0dXJlIn2jc2lnxEALFHOFpo85gpD2rMddo8E1U6ZxP3wRFOyj9eoxDGFxdtwwNbWCNb8bO0U1MsaiMFJCd/WOyg2NEHEcSR+5stsKqHNpZ190eXBlIKN0YWfNAgKndmVyc2lvbgE=",
            "payload_hash": "dce3f0a140b4549038a21b43aec620d5362c1ee99b462cc2c8d7c8efbf4cf233",
            "sig_id": "be6da4ea06918c7f5f6255834c9e41da9b94c75745edbe64659fe885c3c432a10f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a\",\"host\":\"keybase.io\",\"kid\":\"0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"revoke\":{\"kid\":\"0120d20fe982e4e7ba1a08dd60dd527a027bbe0225d4fe89eef979020c5b9efedbfb0a\"},\"type\":\"revoke\",\"version\":1},\"ctime\":1432145700,\"expire_in\":10000000,\"prev\":\"601b3462742a14955c12017a7bbf5615cb3cad8bea04734d00dd61d492ad1952\",\"seq_type\":1,\"seqno\":11,\"tag\":\"signature\"}",
            "kid": "0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a",
            "ctime": 1432145700
        },
        {
            "seqno": 12,
            "prev": "dce3f0a140b4549038a21b43aec620d5362c1ee99b462cc2c8d7c8efbf4cf233",
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEg1Whs0Da7MlPMAmd1kDZIgKu5lFdleYysUKKFus2jYmAKp3BheWxvYWTFAhx7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwMmFmOTVkM2E5OGI5N2M1NWVmZDJkYzA5YjBjOWM2MDZmZDBiNWVhYTdlMzViNTlhNTc1ZTMzNmRlYzUwNGQ0OTBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwZDU2ODZjZDAzNmJiMzI1M2NjMDI2Nzc1OTAzNjQ4ODBhYmI5OTQ1NzY1Nzk4Y2FjNTBhMjg1YmFjZGEzNjI2MDBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInJldm9rZSI6eyJraWRzIjpbIjAxMjAzMWZlNzVjYWUxNmUzNjQ5NDlmOTliNmZjNGZlNzlmYWIwYjA3ZDEyMzRkZDUwYjllYWZiMGQ3MzE1NzFiZjZjMGEiXX0sInR5cGUiOiJyZXZva2UiLCJ2ZXJzaW9uIjoxfSwiY3RpbWUiOjE0MzIxNDU4MDAsImV4cGlyZV9pbiI6MTAwMDAwMDAsInByZXYiOiJkY2UzZjBhMTQwYjQ1NDkwMzhhMjFiNDNhZWM2MjBkNTM2MmMxZWU5OWI0NjJjYzJjOGQ3YzhlZmJmNGNmMjMzIiwic2VxX3R5cGUiOjEsInNlcW5vIjoxMiwidGFnIjoic2lnbmF0dXJlIn2jc2lnxEBw0vZNoLLGZ+77z68uKIPVtTY9L+LB/Le7Qy+8ViETnpz1SMN8TFzNVtoTZvtCp8lUVzYqhCCGCczNkT6BfJkJqHNpZ190eXBlIKN0YWfNAgKndmVyc2lvbgE=",
            "payload_hash": "8689d6aa08e1a0818f955de82520fa736f61ced7eb640070c45660e127d63bda",
            "sig_id": "cb6682dcddf0e407635260320b21957c2df306f81813a5a7c8507936b296c5150f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a\",\"host\":\"keybase.io\",\"kid\":\"0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"revoke\":{\"kids\":[\"012031fe75cae16e364949f99b6fc4fe79fab0b07d1234dd50b9eafb0d731571bf6c0a\"]},\"type\":\"revoke\",\"version\":1},\"ctime\":1432145800,\"expire_in\":10000000,\"prev\":\"dce3f0a140b4549038a21b43aec620d5362c1ee99b462cc2c8d7c8efbf4cf233\",\"seq_type\":1,\"seqno\":12,\"tag\":\"signature\"}",
            "kid": "0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a",
            "ctime": 1432145800
        },
        {
            "seqno": 13,
            "prev": "8689d6aa08e1a0818f955de82520fa736f61ced7eb640070c45660e127d63bda",
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEg1Whs0Da7MlPMAmd1kDZIgKu5lFdleYysUKKFus2jYmAKp3BheWxvYWTFAhh7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwMmFmOTVkM2E5OGI5N2M1NWVmZDJkYzA5YjBjOWM2MDZmZDBiNWVhYTdlMzViNTlhNTc1ZTMzNmRlYzUwNGQ0OTBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwZDU2ODZjZDAzNmJiMzI1M2NjMDI2Nzc1OTAzNjQ4ODBhYmI5OTQ1NzY1Nzk4Y2FjNTBhMjg1YmFjZGEzNjI2MDBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInJldm9rZSI6eyJzaWdfaWQiOiI1MDZiZjQ4YzM2OGE4NGM5ZTlhYjZmYzIyNDM4MzU5MjQxZWUxMjc4NGM3MmYyZTY4YzJhZmExZDNkZDcxYWFlMGYifSwidHlwZSI6InJldm9rZSIsInZlcnNpb24iOjF9LCJjdGltZSI6MTQzMjE0NTkwMCwiZXhwaXJlX2luIjoxMDAwMDAwMCwicHJldiI6Ijg2ODlkNmFhMDhlMWEwODE4Zjk1NWRlODI1MjBmYTczNmY2MWNlZDdlYjY0MDA3MGM0NTY2MGUxMjdkNjNiZGEiLCJzZXFfdHlwZSI6MSwic2Vxbm8iOjEzLCJ0YWciOiJzaWduYXR1cmUifaNzaWfEQAVTx2OScgQI1tbC07Cg1VfRW4Zow+p7PFX3g4HCck5ZBjf/2wsj1JYihQZgmDnm8IFVdmUEXNui5F2MVxvYSweoc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
            "payload_hash": "2a8200e9f3f59f2db54f404a51a727784571586282b7786755a17196ec2f722e",
            "sig_id": "33214539afe47c134f16b520728da821a8f0f0259e954805d11a0ea285309f700f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a\",\"host\":\"keybase.io\",\"kid\":\"0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"revoke\":{\"sig_id\":\"506bf48c368a84c9e9ab6fc22438359241ee12784c72f2e68c2afa1d3dd71aae0f\"},\"type\":\"revoke\",\"version\":1},\"ctime\":1432145900,\"expire_in\":10000000,\"prev\":\"8689d6aa08e1a0818f955de82520fa736f61ced7eb640070c45660e127d63bda\",\"seq_type\":1,\"seqno\":13,\"tag\":\"signature\"}",
            "kid": "0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a",
            "ctime": 1432145900
        },
        {
            "seqno": 14,
            "prev": "2a8200e9f3f59f2db54f404a51a727784571586282b7786755a17196ec2f722e",
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEg1Whs0Da7MlPMAmd1kDZIgKu5lFdleYysUKKFus2jYmAKp3BheWxvYWTFAhh7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwMmFmOTVkM2E5OGI5N2M1NWVmZDJkYzA5YjBjOWM2MDZmZDBiNWVhYTdlMzViNTlhNTc1ZTMzNmRlYzUwNGQ0OTBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwZDU2ODZjZDAzNmJiMzI1M2NjMDI2Nzc1OTAzNjQ4ODBhYmI5OTQ1NzY1Nzk4Y2FjNTBhMjg1YmFjZGEzNjI2MDBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInJldm9rZSI6eyJzaWdfaWQiOiI1MDZiZjQ4YzM2OGE4NGM5ZTlhYjZmYzIyNDM4MzU5MjQxZWUxMjc4NGM3MmYyZTY4YzJhZmExZDNkZDcxYWFlMGYifSwidHlwZSI6InJldm9rZSIsInZlcnNpb24iOjF9LCJjdGltZSI6MTQzMjE0NjAwMCwiZXhwaXJlX2luIjoxMDAwMDAwMCwicHJldiI6IjJhODIwMGU5ZjNmNTlmMmRiNTRmNDA0YTUxYTcyNzc4NDU3MTU4NjI4MmI3Nzg2NzU1YTE3MTk2ZWMyZjcyMmUiLCJzZXFfdHlwZSI6MSwic2Vxbm8iOjE0LCJ0YWciOiJzaWduYXR1cmUifaNzaWfEQK2ipVbmehRpRKz8yoBDxNlxipjLFEfSFZpbz3HBB7pOCSQKcEQP9jDn4Das48z08CS6Bb59ZUD+hlYjI6joWQ+oc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
            "payload_hash": "d0ff71f5026ff1a92b8baec3b06336df8d46f961077f39442fb019553b2b0a18",
            "sig_id": "67d4abca611b70d909cbfa3fcf3311320791a57309f54799a6e2ae49db858a9e0f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a\",\"host\":\"keybase.io\",\"kid\":\"0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"revoke\":{\"sig_id\":\"506bf48c368a84c9e9ab6fc22438359241ee12784c72f2e68c2afa1d3dd71aae0f\"},\"type\":\"revoke\",\"version\":1},\"ctime\":1432146000,\"expire_in\":10000000,\"prev\":\"2a8200e9f3f59f2db54f404a51a727784571586282b7786755a17196ec2f722e\",\"seq_type\":1,\"seqno\":14,\"tag\":\"signature\"}",
            "kid": "0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a",
            "ctime": 1432146000
        },
        {
            "seqno": 15,
            "prev": "d0ff71f5026ff1a92b8baec3b06336df8d46f961077f39442fb019553b2b0a18",
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEg1Whs0Da7MlPMAmd1kDZIgKu5lFdleYysUKKFus2jYmAKp3BheWxvYWTFAmB7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwMmFmOTVkM2E5OGI5N2M1NWVmZDJkYzA5YjBjOWM2MDZmZDBiNWVhYTdlMzViNTlhNTc1ZTMzNmRlYzUwNGQ0OTBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwZDU2ODZjZDAzNmJiMzI1M2NjMDI2Nzc1OTAzNjQ4ODBhYmI5OTQ1NzY1Nzk4Y2FjNTBhMjg1YmFjZGEzNjI2MDBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInJldm9rZSI6eyJzaWdfaWRzIjpbIjc3YTJhNzBkNWM4YWYwZjEyMDEwMjcwNzY1MDQwNjVlODBkZGQ2ZDVmZjJhNjIyZWI1N2UxNDlkYzhmYTAwNDcwZiIsImIwZDNhMzZmYzA2MWIzMTdhMjYwNTE0ZWZmOGY1YWMxYjhhYjFjZmM2MGM1NDczMWU3OWU5NzlmOWU2MzhiMzIwZiJdfSwidHlwZSI6InJldm9rZSIsInZlcnNpb24iOjF9LCJjdGltZSI6MTQzMjE0NjEwMCwiZXhwaXJlX2luIjoxMDAwMDAwMCwicHJldiI6ImQwZmY3MWY1MDI2ZmYxYTkyYjhiYWVjM2IwNjMzNmRmOGQ0NmY5NjEwNzdmMzk0NDJmYjAxOTU1M2IyYjBhMTgiLCJzZXFfdHlwZSI6MSwic2Vxbm8iOjE1LCJ0YWciOiJzaWduYXR1cmUifaNzaWfEQH9HYHx8/DWAwcC5rRKcNQ+Bwk//XSc196Nr89E6dS1XrKYGurIL0+wqWvfRn1NxLT/zSvcwAXT/OoEu5PtHDguoc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
            "payload_hash": "159fd5164720001518c6a6d194c755ed6bca0eeedce5ac3f409f67fe629e655d",
            "sig_id": "577d4cc5654ced3ed4740b35eac9312701bb46177a68249bd44fec4a274669680f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a\",\"host\":\"keybase.io\",\"kid\":\"0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"revoke\":{\"sig_ids\":[\"77a2a70d5c8af0f1201027076504065e80ddd6d5ff2a622eb57e149dc8fa00470f\",\"b0d3a36fc061b317a260514eff8f5ac1b8ab1cfc60c54731e79e979f9e638b320f\"]},\"type\":\"revoke\",\"version\":1},\"ctime\":1432146100,\"expire_in\":10000000,\"prev\":\"d0ff71f5026ff1a92b8baec3b06336df8d46f961077f39442fb019553b2b0a18\",\"seq_type\":1,\"seqno\":15,\"tag\":\"signature\"}",
            "kid": "0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a",
            "ctime": 1432146100
        },
        {
            "seqno": 16,
            "prev": "159fd5164720001518c6a6d194c755ed6bca0eeedce5ac3f409f67fe629e655d",
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgKvldOpi5fFXv0twJsMnGBv0LXqp+NbWaV14zbexQTUkKp3BheWxvYWTFAhl7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwMmFmOTVkM2E5OGI5N2M1NWVmZDJkYzA5YjBjOWM2MDZmZDBiNWVhYTdlMzViNTlhNTc1ZTMzNmRlYzUwNGQ0OTBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwMmFmOTVkM2E5OGI5N2M1NWVmZDJkYzA5YjBjOWM2MDZmZDBiNWVhYTdlMzViNTlhNTc1ZTMzNmRlYzUwNGQ0OTBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInJldm9rZSI6eyJraWQiOiIwMTIxMzBiZjU3Y2MwMTY4ZjJhZGQzYTYxZGI1ZmNiOTNkMTlmN2MwZTQyM2EyNjZmZDVkNWE5OWQ1NmQ4MWMzNWM1NTBhIn0sInR5cGUiOiJyZXZva2UiLCJ2ZXJzaW9uIjoxfSwiY3RpbWUiOjE0MzIxNDYyMDAsImV4cGlyZV9pbiI6MTAwMDAwMDAsInByZXYiOiIxNTlmZDUxNjQ3MjAwMDE1MThjNmE2ZDE5NGM3NTVlZDZiY2EwZWVlZGNlNWFjM2Y0MDlmNjdmZTYyOWU2NTVkIiwic2VxX3R5cGUiOjEsInNlcW5vIjoxNiwidGFnIjoic2lnbmF0dXJlIn2jc2lnxEBRTqKeefnwZwlcVwjun1IsWKFhkP3xNdSliTIq/YVuuBOS4Iw66xtvShzdfwrqUR569G4mcU0AkLV320hWVVQGqHNpZ190eXBlIKN0YWfNAgKndmVyc2lvbgE=",
            "payload_hash": "ba47575973c4e7f3faf6c6c3ee346a8f5d7a1c37f5c03bc0df7e15c5603774fd",
            "sig_id": "00ce703cbc1ec4576392064e378aaf5fd78983a934e4cfae97af3d6d8231743e0f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a\",\"host\":\"keybase.io\",\"kid\":\"01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"revoke\":{\"kid\":\"012130bf57cc0168f2add3a61db5fcb93d19f7c0e423a266fd5d5a99d56d81c35c550a\"},\"type\":\"revoke\",\"version\":1},\"ctime\":1432146200,\"expire_in\":10000000,\"prev\":\"159fd5164720001518c6a6d194c755ed6bca0eeedce5ac3f409f67fe629e655d\",\"seq_type\":1,\"seqno\":16,\"tag\":\"signature\"}",
            "kid": "01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a",
            "ctime": 1432146200
        },
        {
            "seqno": 17,
            "prev": "ba47575973c4e7f3faf6c6c3ee346a8f5d7a1c37f5c03bc0df7e15c5603774fd",
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgKvldOpi5fFXv0twJsMnGBv0LXqp+NbWaV14zbexQTUkKp3BheWxvYWTFAhh7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwMmFmOTVkM2E5OGI5N2M1NWVmZDJkYzA5YjBjOWM2MDZmZDBiNWVhYTdlMzViNTlhNTc1ZTMzNmRlYzUwNGQ0OTBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwMmFmOTVkM2E5OGI5N2M1NWVmZDJkYzA5YjBjOWM2MDZmZDBiNWVhYTdlMzViNTlhNTc1ZTMzNmRlYzUwNGQ0OTBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInJldm9rZSI6eyJzaWdfaWQiOiJhNTM1YTM5NzI5NTFjMjA3NzY0N2Q1Njc5NDljNzMyMDk2NzJhZmQ1YzU4NmZlN2EwZjQ0MDdkNTg5NDA5OTAwMGYifSwidHlwZSI6InJldm9rZSIsInZlcnNpb24iOjF9LCJjdGltZSI6MTQzMjE0NjMwMCwiZXhwaXJlX2luIjoxMDAwMDAwMCwicHJldiI6ImJhNDc1NzU5NzNjNGU3ZjNmYWY2YzZjM2VlMzQ2YThmNWQ3YTFjMzdmNWMwM2JjMGRmN2UxNWM1NjAzNzc0ZmQiLCJzZXFfdHlwZSI6MSwic2Vxbm8iOjE3LCJ0YWciOiJzaWduYXR1cmUifaNzaWfEQKXqUkdDSab896qNZKjMQ9sF3n3SbDS66JvGc9vFZ7/sBXNjTlt2SWRy40ukK3VIGE10D6QT8DBiEeSz5ffAmwaoc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==",
            "payload_hash": "ae4ed6b838a8395e9f44f7e1e5fbad442134de72691dcfc5dc04eb9b47da00f2",
            "sig_id": "ee77d80410f106b29f3b6294ae5de84308166188ba01e64d7ce3e1628044ce380f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a\",\"host\":\"keybase.io\",\"kid\":\"01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"revoke\":{\"sig_id\":\"a535a3972951c2077647d567949c73209672afd5c586fe7a0f4407d5894099000f\"},\"type\":\"revoke\",\"version\":1},\"ctime\":1432146300,\"expire_in\":10000000,\"prev\":\"ba47575973c4e7f3faf6c6c3ee346a8f5d7a1c37f5c03bc0df7e15c5603774fd\",\"seq_type\":1,\"seqno\":17,\"tag\":\"signature\"}",
            "kid": "01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a",
            "ctime": 1432146300
        }
    ],
    "keys": [
        "01202af95d3a98b97c55efd2dc09b0c9c606fd0b5eaa7e35b59a575e336dec504d490a",
        "0120d20fe982e4e7ba1a08dd60dd527a027bbe0225d4fe89eef979020c5b9efedbfb0a",
        "012031fe75cae16e364949f99b6fc4fe79fab0b07d1234dd50b9eafb0d731571bf6c0a",
        "0120ff5aff3b56026d50f6225ff5c53fed9bb9c27e5230abe636c5e03be1c841dd760a",
        "0120818b638074d0b15e766af7a61c08fb5fd2314cf0e2d0d5a104d8347d21e6f5ed0a",
        "0120d5686cd036bb3253cc02677590364880abb9945765798cac50a285bacda362600a",
        "012130bf57cc0168f2add3a61db5fcb93d19f7c0e423a266fd5d5a99d56d81c35c550a",
        "01215601c4c4b46e19f35654a1debc8dc0e3f68013c3517eb809e5f36541726fd3360a",
        "01215cb902e4b6a0783f4558e4b483f091852eb585aecdec019356eb813ac5040d020a"
    ],
    "uid": "74c38cf7ceb947f5632045d8ca5d4819",
    "username": "max32"
}
`,
	"expired_key_chain.json": `{
    "chain": [
        {
            "seqno": 1,
            "prev": null,
            "sig": "g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgDTTfqXtH50CdFdtB6s5oGZiTHBiSB8+PU6MnAafmgr4Kp3BheWxvYWTFASN7ImJvZHkiOnsia2V5Ijp7Imhvc3QiOiJrZXliYXNlLmlvIiwia2lkIjoiMDEyMDBkMzRkZmE5N2I0N2U3NDA5ZDE1ZGI0MWVhY2U2ODE5OTg5MzFjMTg5MjA3Y2Y4ZjUzYTMyNzAxYTdlNjgyYmUwYSIsInVpZCI6Ijc0YzM4Y2Y3Y2ViOTQ3ZjU2MzIwNDVkOGNhNWQ0ODE5IiwidXNlcm5hbWUiOiJtYXgzMiJ9LCJ0eXBlIjoiZWxkZXN0IiwidmVyc2lvbiI6MX0sImN0aW1lIjoxNDMyMTQ0NzUwLCJleHBpcmVfaW4iOjEsInByZXYiOm51bGwsInNlcV90eXBlIjoxLCJzZXFubyI6MSwidGFnIjoic2lnbmF0dXJlIn2jc2lnxEAi+mx5hSdvAHor5WhocvcOzK0M/JjQcB6/6BNzIm7DHXopkvVGK6qXrg6QvydIoZ0OW8lD23eWrKdIHaEITJUOqHNpZ190eXBlIKN0YWfNAgKndmVyc2lvbgE=",
            "payload_hash": "cc2f840119fc2734c852c41c3274b69e53fd6771f15a31b2c06bb57c6a9f1752",
            "sig_id": "bdc3c02698b224ba0e27ed3c9782106ecc83ffa5c5a436e438b3de7dbc42554e0f",
            "payload_json": "{\"body\":{\"key\":{\"host\":\"keybase.io\",\"kid\":\"01200d34dfa97b47e7409d15db41eace681998931c189207cf8f53a32701a7e682be0a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"type\":\"eldest\",\"version\":1},\"ctime\":1432144750,\"expire_in\":1,\"prev\":null,\"seq_type\":1,\"seqno\":1,\"tag\":\"signature\"}",
            "kid": "01200d34dfa97b47e7409d15db41eace681998931c189207cf8f53a32701a7e682be0a",
            "ctime": 1432144750
        },
        {
            "seqno": 2,
            "prev": "cc2f840119fc2734c852c41c3274b69e53fd6771f15a31b2c06bb57c6a9f1752",
            "sig": "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",
            "payload_hash": "d945735df7741a001ceeee778c79fc4d3dfd55d2fa5d6b672b7bd198395ab746",
            "sig_id": "79579a28083e801d565a330e3c8a7d0cd76dda85f07603976ccb016f259cfadf0f",
            "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01200d34dfa97b47e7409d15db41eace681998931c189207cf8f53a32701a7e682be0a\",\"host\":\"keybase.io\",\"kid\":\"01200d34dfa97b47e7409d15db41eace681998931c189207cf8f53a32701a7e682be0a\",\"uid\":\"74c38cf7ceb947f5632045d8ca5d4819\",\"username\":\"max32\"},\"sibkey\":{\"kid\":\"012063557949acce3245ce74063632e9dd2a61ccc0350e9be1000107a8cee0d213180a\",\"reverse_sig\":\"g6Rib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgY1V5SazOMkXOdAY2MundKmHMwDUOm+EAAQeozuDSExgKp3BheWxvYWTFAiR7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwMGQzNGRmYTk3YjQ3ZTc0MDlkMTVkYjQxZWFjZTY4MTk5ODkzMWMxODkyMDdjZjhmNTNhMzI3MDFhN2U2ODJiZTBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwMGQzNGRmYTk3YjQ3ZTc0MDlkMTVkYjQxZWFjZTY4MTk5ODkzMWMxODkyMDdjZjhmNTNhMzI3MDFhN2U2ODJiZTBhIiwidWlkIjoiNzRjMzhjZjdjZWI5NDdmNTYzMjA0NWQ4Y2E1ZDQ4MTkiLCJ1c2VybmFtZSI6Im1heDMyIn0sInNpYmtleSI6eyJraWQiOiIwMTIwNjM1NTc5NDlhY2NlMzI0NWNlNzQwNjM2MzJlOWRkMmE2MWNjYzAzNTBlOWJlMTAwMDEwN2E4Y2VlMGQyMTMxODBhIiwicmV2ZXJzZV9zaWciOm51bGx9LCJ0eXBlIjoic2lia2V5IiwidmVyc2lvbiI6MX0sImN0aW1lIjoxNDMyMTQ0ODUwLCJleHBpcmVfaW4iOjEsInByZXYiOiJjYzJmODQwMTE5ZmMyNzM0Yzg1MmM0MWMzMjc0YjY5ZTUzZmQ2NzcxZjE1YTMxYjJjMDZiYjU3YzZhOWYxNzUyIiwic2VxX3R5cGUiOjEsInNlcW5vIjoyLCJ0YWciOiJzaWduYXR1cmUifaNzaWfEQH7iM4vtfx9PQgmxX3eXfBeP4BhtA+CGudKoGMY0fEdj4LFPn98DONwV/Me+QNzXYfkMqZt8Rp/ES2Hzxr7qew2oc2lnX3R5cGUgo3RhZ80CAqd2ZXJzaW9uAQ==\"},\"type\":\"sibkey\",\"version\":1},\"ctime\":1432144850,\"expire_in\":1,\"prev\":\"cc2f840119fc2734c852c41c3274b69e53fd6771f15a31b2c06bb57c6a9f1752\",\"seq_type\":1,\"seqno\":2,\"tag\":\"signature\"}",
            "kid": "01200d34dfa97b47e7409d15db41eace681998931c189207cf8f53a32701a7e682be0a",
            "ctime": 1432144850
        }
    ],
    "keys": [
        "01200d34dfa97b47e7409d15db41eace681998931c189207cf8f53a32701a7e682be0a",
        "012063557949acce3245ce74063632e9dd2a61ccc0350e9be1000107a8cee0d213180a"
    ],
    "uid": "74c38cf7ceb947f5632045d8ca5d4819",
    "username": "max32"
}
`,
	"goozbach.json": `{
  "uid": "eb80388edfb7189e512410d8b2a5f419",
  "username": "goozbach",
  "chain": [
    {
      "seqno": 1,
      "payload_hash": "adbf252be1480592c97abac314f94b4f2b4c550384177c52a7c86d966a56b7c2",
      "sig_id": "8d3589326dbd156f4aeae47bab40e1cfda3f71160ba39b825472289b8edd585c0f",
      "sig_id_short": "jTWJMm29FW9K6uR7q0Dhz9o_cRYLo5uCVHIo",
      "kid": "01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a",
      "sig": "hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgPdwSxzimfzXSBPTf9J5JMhoukVAnZ4nGy8xiKUJDoPcKp3BheWxvYWTFAsV7ImJvZHkiOnsiZGV2aWNlIjp7ImlkIjoiYmIzZDE5MDBjODc0Yzg0N2ZkNjA5OWYxNzJlYjI0MTgiLCJraWQiOiIwMTIwM2RkYzEyYzczOGE2N2YzNWQyMDRmNGRmZjQ5ZTQ5MzIxYTJlOTE1MDI3Njc4OWM2Y2JjYzYyMjk0MjQzYTBmNzBhIiwibmFtZSI6ImhvdGgiLCJzdGF0dXMiOjEsInR5cGUiOiJkZXNrdG9wIn0sImtleSI6eyJob3N0Ijoia2V5YmFzZS5pbyIsImtpZCI6IjAxMjAzZGRjMTJjNzM4YTY3ZjM1ZDIwNGY0ZGZmNDllNDkzMjFhMmU5MTUwMjc2Nzg5YzZjYmNjNjIyOTQyNDNhMGY3MGEiLCJ1aWQiOiJlYjgwMzg4ZWRmYjcxODllNTEyNDEwZDhiMmE1ZjQxOSIsInVzZXJuYW1lIjoiZ29vemJhY2gifSwidHlwZSI6ImVsZGVzdCIsInZlcnNpb24iOjF9LCJjbGllbnQiOnsibmFtZSI6ImtleWJhc2UuaW8gZ28gY2xpZW50IiwidmVyc2lvbiI6IjEuMC4xNyJ9LCJjdGltZSI6MTQ3MzE5MTUwNiwiZXhwaXJlX2luIjo1MDQ1NzYwMDAsIm1lcmtsZV9yb290Ijp7ImN0aW1lIjoxNDczMTkxNDM2LCJoYXNoIjoiOTdlOWFlOGVmNzE1YzAyMzY1YThiNjMxNDE1ZGU4N2Q0N2VlNzViMTQzZmZiMzFhMDVhODg1NDlhNGFmMGVhMDk4YTZhOWE4OTQxYWYwZmRmMjBhZWE0OTRiMThkZjQyMDU0NGM5ZDRhZGFlNzBmZWQ1NWIyYzVhMzNkOWFjOGQiLCJzZXFubyI6NjIxNjE4fSwicHJldiI6bnVsbCwic2Vxbm8iOjEsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RAT7dM4IN68EDR4Ek/P5alRMJZArCQ6HJ4QIss+zV6E5TA7m4jfHewYHZ82o46jIei9kOqQqX+IixjeYTlQ6+2DqhzaWdfdHlwZSCkaGFzaIKkdHlwZQildmFsdWXEICUay0YgUaUUtnKOqz9RUW4WwW3EuNmit2KUa5Z8WPkpo3RhZ80CAqd2ZXJzaW9uAQ==",
      "payload_json": "{\"body\":{\"device\":{\"id\":\"bb3d1900c874c847fd6099f172eb2418\",\"kid\":\"01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a\",\"name\":\"hoth\",\"status\":1,\"type\":\"desktop\"},\"key\":{\"host\":\"keybase.io\",\"kid\":\"01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a\",\"uid\":\"eb80388edfb7189e512410d8b2a5f419\",\"username\":\"goozbach\"},\"type\":\"eldest\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.17\"},\"ctime\":1473191506,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1473191436,\"hash\":\"97e9ae8ef715c02365a8b631415de87d47ee75b143ffb31a05a88549a4af0ea098a6a9a8941af0fdf20aea494b18df420544c9d4adae70fed55b2c5a33d9ac8d\",\"seqno\":621618},\"prev\":null,\"seqno\":1,\"tag\":\"signature\"}",
      "sig_type": 1,
      "sig_version": 1,
      "ctime": 1473195106,
      "etime": 1977771106,
      "rtime": null,
      "sig_status": 0,
      "prev": null,
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 2,
      "payload_hash": "c85b6f785fa4ec3c725540cff52e678bd0394b86b4656e13f03d59d9c3face2c",
      "sig_id": "e3ba9b202ac3d61855e57df20b953656c2048a3b802af8b1a4b541b0f577af370f",
      "sig_id_short": "47qbICrD1hhV5X3yC5U2VsIEijuAKvixpLVB",
      "kid": "01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a",
      "sig": "hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgPdwSxzimfzXSBPTf9J5JMhoukVAnZ4nGy8xiKUJDoPcKp3BheWxvYWTFA+p7ImJvZHkiOnsiZGV2aWNlIjp7ImlkIjoiYmIzZDE5MDBjODc0Yzg0N2ZkNjA5OWYxNzJlYjI0MTgiLCJraWQiOiIwMTIxYmM4NmJlMTE4MzQ1MmMzM2NmMjc0MDg1NWRmYjk5YTY5MGE0YmIyODM5ZDJhNmI0YzY1ZTA3NDUwYjc2YjIxZjBhIiwic3RhdHVzIjoxfSwia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwM2RkYzEyYzczOGE2N2YzNWQyMDRmNGRmZjQ5ZTQ5MzIxYTJlOTE1MDI3Njc4OWM2Y2JjYzYyMjk0MjQzYTBmNzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwM2RkYzEyYzczOGE2N2YzNWQyMDRmNGRmZjQ5ZTQ5MzIxYTJlOTE1MDI3Njc4OWM2Y2JjYzYyMjk0MjQzYTBmNzBhIiwidWlkIjoiZWI4MDM4OGVkZmI3MTg5ZTUxMjQxMGQ4YjJhNWY0MTkiLCJ1c2VybmFtZSI6Imdvb3piYWNoIn0sInN1YmtleSI6eyJraWQiOiIwMTIxYmM4NmJlMTE4MzQ1MmMzM2NmMjc0MDg1NWRmYjk5YTY5MGE0YmIyODM5ZDJhNmI0YzY1ZTA3NDUwYjc2YjIxZjBhIiwicGFyZW50X2tpZCI6IjAxMjAzZGRjMTJjNzM4YTY3ZjM1ZDIwNGY0ZGZmNDllNDkzMjFhMmU5MTUwMjc2Nzg5YzZjYmNjNjIyOTQyNDNhMGY3MGEifSwidHlwZSI6InN1YmtleSIsInZlcnNpb24iOjF9LCJjbGllbnQiOnsibmFtZSI6ImtleWJhc2UuaW8gZ28gY2xpZW50IiwidmVyc2lvbiI6IjEuMC4xNyJ9LCJjdGltZSI6MTQ3MzE5MTUwNiwiZXhwaXJlX2luIjo1MDQ1NzYwMDAsIm1lcmtsZV9yb290Ijp7ImN0aW1lIjoxNDczMTkxNDM2LCJoYXNoIjoiOTdlOWFlOGVmNzE1YzAyMzY1YThiNjMxNDE1ZGU4N2Q0N2VlNzViMTQzZmZiMzFhMDVhODg1NDlhNGFmMGVhMDk4YTZhOWE4OTQxYWYwZmRmMjBhZWE0OTRiMThkZjQyMDU0NGM5ZDRhZGFlNzBmZWQ1NWIyYzVhMzNkOWFjOGQiLCJzZXFubyI6NjIxNjE4fSwicHJldiI6ImFkYmYyNTJiZTE0ODA1OTJjOTdhYmFjMzE0Zjk0YjRmMmI0YzU1MDM4NDE3N2M1MmE3Yzg2ZDk2NmE1NmI3YzIiLCJzZXFubyI6MiwidGFnIjoic2lnbmF0dXJlIn2jc2lnxED8vVidviAjtMHEWhmRGdKtrvKRKEiqKU/069hUvi5ZFa+/zBXaHVbUBbitO+pRGQxZ7NbEKPIFjhKmiLjbEmsMqHNpZ190eXBlIKRoYXNogqR0eXBlCKV2YWx1ZcQgHpIrLWKlOEqpKqYFZpvMjUUnstGamF0O6vr9vK2N7gqjdGFnzQICp3ZlcnNpb24B",
      "payload_json": "{\"body\":{\"device\":{\"id\":\"bb3d1900c874c847fd6099f172eb2418\",\"kid\":\"0121bc86be1183452c33cf2740855dfb99a690a4bb2839d2a6b4c65e07450b76b21f0a\",\"status\":1},\"key\":{\"eldest_kid\":\"01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a\",\"host\":\"keybase.io\",\"kid\":\"01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a\",\"uid\":\"eb80388edfb7189e512410d8b2a5f419\",\"username\":\"goozbach\"},\"subkey\":{\"kid\":\"0121bc86be1183452c33cf2740855dfb99a690a4bb2839d2a6b4c65e07450b76b21f0a\",\"parent_kid\":\"01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a\"},\"type\":\"subkey\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.17\"},\"ctime\":1473191506,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1473191436,\"hash\":\"97e9ae8ef715c02365a8b631415de87d47ee75b143ffb31a05a88549a4af0ea098a6a9a8941af0fdf20aea494b18df420544c9d4adae70fed55b2c5a33d9ac8d\",\"seqno\":621618},\"prev\":\"adbf252be1480592c97abac314f94b4f2b4c550384177c52a7c86d966a56b7c2\",\"seqno\":2,\"tag\":\"signature\"}",
      "sig_type": 1,
      "sig_version": 1,
      "ctime": 1473195106,
      "etime": 1977771106,
      "rtime": null,
      "sig_status": 0,
      "prev": "adbf252be1480592c97abac314f94b4f2b4c550384177c52a7c86d966a56b7c2",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 3,
      "payload_hash": "7868923366c3d0dfb08d6b2e366ceb9efc15a815cd5925ba7db42bda7b53852f",
      "sig_id": "448fe98847f191fee12326cd79229cd0a4d77742ab66d468bea25ebd2f2051a70f",
      "sig_id_short": "RI_piEfxkf7hIybNeSKc0KTXd0KrZtRovqJe",
      "kid": "01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a",
      "sig": "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",
      "payload_json": "{\"body\":{\"device\":{\"id\":\"1f394f2192a9f1ff71ce08bdc9372f18\",\"kid\":\"0120bc9e1022ac7f0463e910601642d6ac3a7d8fd3056f177de35e89aa602f6da6a20a\",\"name\":\"autumn curious\",\"status\":1,\"type\":\"backup\"},\"key\":{\"eldest_kid\":\"01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a\",\"host\":\"keybase.io\",\"kid\":\"01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a\",\"uid\":\"eb80388edfb7189e512410d8b2a5f419\",\"username\":\"goozbach\"},\"sibkey\":{\"kid\":\"0120bc9e1022ac7f0463e910601642d6ac3a7d8fd3056f177de35e89aa602f6da6a20a\",\"reverse_sig\":\"hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgvJ4QIqx/BGPpEGAWQtasOn2P0wVvF33jXomqYC9tpqIKp3BheWxvYWTFA897ImJvZHkiOnsiZGV2aWNlIjp7ImlkIjoiMWYzOTRmMjE5MmE5ZjFmZjcxY2UwOGJkYzkzNzJmMTgiLCJraWQiOiIwMTIwYmM5ZTEwMjJhYzdmMDQ2M2U5MTA2MDE2NDJkNmFjM2E3ZDhmZDMwNTZmMTc3ZGUzNWU4OWFhNjAyZjZkYTZhMjBhIiwibmFtZSI6ImF1dHVtbiBjdXJpb3VzIiwic3RhdHVzIjoxLCJ0eXBlIjoiYmFja3VwIn0sImtleSI6eyJlbGRlc3Rfa2lkIjoiMDEyMDNkZGMxMmM3MzhhNjdmMzVkMjA0ZjRkZmY0OWU0OTMyMWEyZTkxNTAyNzY3ODljNmNiY2M2MjI5NDI0M2EwZjcwYSIsImhvc3QiOiJrZXliYXNlLmlvIiwia2lkIjoiMDEyMDNkZGMxMmM3MzhhNjdmMzVkMjA0ZjRkZmY0OWU0OTMyMWEyZTkxNTAyNzY3ODljNmNiY2M2MjI5NDI0M2EwZjcwYSIsInVpZCI6ImViODAzODhlZGZiNzE4OWU1MTI0MTBkOGIyYTVmNDE5IiwidXNlcm5hbWUiOiJnb296YmFjaCJ9LCJzaWJrZXkiOnsia2lkIjoiMDEyMGJjOWUxMDIyYWM3ZjA0NjNlOTEwNjAxNjQyZDZhYzNhN2Q4ZmQzMDU2ZjE3N2RlMzVlODlhYTYwMmY2ZGE2YTIwYSIsInJldmVyc2Vfc2lnIjpudWxsfSwidHlwZSI6InNpYmtleSIsInZlcnNpb24iOjF9LCJjbGllbnQiOnsibmFtZSI6ImtleWJhc2UuaW8gZ28gY2xpZW50IiwidmVyc2lvbiI6IjEuMC4xNyJ9LCJjdGltZSI6MTQ3MzE5MTUwNywiZXhwaXJlX2luIjo1MDQ1NzYwMDAsIm1lcmtsZV9yb290Ijp7ImN0aW1lIjoxNDczMTkxNDM2LCJoYXNoIjoiOTdlOWFlOGVmNzE1YzAyMzY1YThiNjMxNDE1ZGU4N2Q0N2VlNzViMTQzZmZiMzFhMDVhODg1NDlhNGFmMGVhMDk4YTZhOWE4OTQxYWYwZmRmMjBhZWE0OTRiMThkZjQyMDU0NGM5ZDRhZGFlNzBmZWQ1NWIyYzVhMzNkOWFjOGQiLCJzZXFubyI6NjIxNjE4fSwicHJldiI6ImM4NWI2Zjc4NWZhNGVjM2M3MjU1NDBjZmY1MmU2NzhiZDAzOTRiODZiNDY1NmUxM2YwM2Q1OWQ5YzNmYWNlMmMiLCJzZXFubyI6MywidGFnIjoic2lnbmF0dXJlIn2jc2lnxEAqX0UvzkklxXlkhMYf74CDEpjO//PDzj2EreXcW+x823Y0yhhLcf1Pzs7GYG3jirJw+Pyj1yfkQOeBFqLC3noEqHNpZ190eXBlIKRoYXNogqR0eXBlCKV2YWx1ZcQgSOrU/g6NzY6V1hmbHVQDrObNn041FpUsgB5WOqbUJt6jdGFnzQICp3ZlcnNpb24B\"},\"type\":\"sibkey\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.17\"},\"ctime\":1473191507,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1473191436,\"hash\":\"97e9ae8ef715c02365a8b631415de87d47ee75b143ffb31a05a88549a4af0ea098a6a9a8941af0fdf20aea494b18df420544c9d4adae70fed55b2c5a33d9ac8d\",\"seqno\":621618},\"prev\":\"c85b6f785fa4ec3c725540cff52e678bd0394b86b4656e13f03d59d9c3face2c\",\"seqno\":3,\"tag\":\"signature\"}",
      "sig_type": 1,
      "sig_version": 1,
      "ctime": 1473195107,
      "etime": 1977771107,
      "rtime": null,
      "sig_status": 0,
      "prev": "c85b6f785fa4ec3c725540cff52e678bd0394b86b4656e13f03d59d9c3face2c",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 4,
      "payload_hash": "83a798c450a49be2399d1fb7b2a81b7e661cee2d0d4ec54f6697e3adee48da4a",
      "sig_id": "948ca54d36bbde7054328de6a9f1ee585778bd18046cd141fd704062204bdb5d0f",
      "sig_id_short": "lIylTTa73nBUMo3mqfHuWFd4vRgEbNFB_XBA",
      "kid": "0120bc9e1022ac7f0463e910601642d6ac3a7d8fd3056f177de35e89aa602f6da6a20a",
      "sig": "hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgvJ4QIqx/BGPpEGAWQtasOn2P0wVvF33jXomqYC9tpqIKp3BheWxvYWTFA+p7ImJvZHkiOnsiZGV2aWNlIjp7ImlkIjoiMWYzOTRmMjE5MmE5ZjFmZjcxY2UwOGJkYzkzNzJmMTgiLCJraWQiOiIwMTIxZThiNWI0Nzg3MTI2YjdjODc4NzQ5MTJhMGE3Y2RmNWFhNGQwYTZmMTgzN2JhNDJlOWQxYTYyZGYzN2ZhNTM0MDBhIiwic3RhdHVzIjoxfSwia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwM2RkYzEyYzczOGE2N2YzNWQyMDRmNGRmZjQ5ZTQ5MzIxYTJlOTE1MDI3Njc4OWM2Y2JjYzYyMjk0MjQzYTBmNzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwYmM5ZTEwMjJhYzdmMDQ2M2U5MTA2MDE2NDJkNmFjM2E3ZDhmZDMwNTZmMTc3ZGUzNWU4OWFhNjAyZjZkYTZhMjBhIiwidWlkIjoiZWI4MDM4OGVkZmI3MTg5ZTUxMjQxMGQ4YjJhNWY0MTkiLCJ1c2VybmFtZSI6Imdvb3piYWNoIn0sInN1YmtleSI6eyJraWQiOiIwMTIxZThiNWI0Nzg3MTI2YjdjODc4NzQ5MTJhMGE3Y2RmNWFhNGQwYTZmMTgzN2JhNDJlOWQxYTYyZGYzN2ZhNTM0MDBhIiwicGFyZW50X2tpZCI6IjAxMjBiYzllMTAyMmFjN2YwNDYzZTkxMDYwMTY0MmQ2YWMzYTdkOGZkMzA1NmYxNzdkZTM1ZTg5YWE2MDJmNmRhNmEyMGEifSwidHlwZSI6InN1YmtleSIsInZlcnNpb24iOjF9LCJjbGllbnQiOnsibmFtZSI6ImtleWJhc2UuaW8gZ28gY2xpZW50IiwidmVyc2lvbiI6IjEuMC4xNyJ9LCJjdGltZSI6MTQ3MzE5MTUwNywiZXhwaXJlX2luIjo1MDQ1NzYwMDAsIm1lcmtsZV9yb290Ijp7ImN0aW1lIjoxNDczMTkxNDM2LCJoYXNoIjoiOTdlOWFlOGVmNzE1YzAyMzY1YThiNjMxNDE1ZGU4N2Q0N2VlNzViMTQzZmZiMzFhMDVhODg1NDlhNGFmMGVhMDk4YTZhOWE4OTQxYWYwZmRmMjBhZWE0OTRiMThkZjQyMDU0NGM5ZDRhZGFlNzBmZWQ1NWIyYzVhMzNkOWFjOGQiLCJzZXFubyI6NjIxNjE4fSwicHJldiI6Ijc4Njg5MjMzNjZjM2QwZGZiMDhkNmIyZTM2NmNlYjllZmMxNWE4MTVjZDU5MjViYTdkYjQyYmRhN2I1Mzg1MmYiLCJzZXFubyI6NCwidGFnIjoic2lnbmF0dXJlIn2jc2lnxECC2v0UWztVxYgg41Q2OUnNIzkJPRS3fX0XEpRKfxR0v845zcLRNf9hYXYqe2aTlFDDhwOIm10ShCWAebatDTQIqHNpZ190eXBlIKRoYXNogqR0eXBlCKV2YWx1ZcQgW5cSH/L9b1rF9mGGJNBuUfJq8bbJDmCz+qoa+LMow/SjdGFnzQICp3ZlcnNpb24B",
      "payload_json": "{\"body\":{\"device\":{\"id\":\"1f394f2192a9f1ff71ce08bdc9372f18\",\"kid\":\"0121e8b5b4787126b7c87874912a0a7cdf5aa4d0a6f1837ba42e9d1a62df37fa53400a\",\"status\":1},\"key\":{\"eldest_kid\":\"01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a\",\"host\":\"keybase.io\",\"kid\":\"0120bc9e1022ac7f0463e910601642d6ac3a7d8fd3056f177de35e89aa602f6da6a20a\",\"uid\":\"eb80388edfb7189e512410d8b2a5f419\",\"username\":\"goozbach\"},\"subkey\":{\"kid\":\"0121e8b5b4787126b7c87874912a0a7cdf5aa4d0a6f1837ba42e9d1a62df37fa53400a\",\"parent_kid\":\"0120bc9e1022ac7f0463e910601642d6ac3a7d8fd3056f177de35e89aa602f6da6a20a\"},\"type\":\"subkey\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.17\"},\"ctime\":1473191507,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1473191436,\"hash\":\"97e9ae8ef715c02365a8b631415de87d47ee75b143ffb31a05a88549a4af0ea098a6a9a8941af0fdf20aea494b18df420544c9d4adae70fed55b2c5a33d9ac8d\",\"seqno\":621618},\"prev\":\"7868923366c3d0dfb08d6b2e366ceb9efc15a815cd5925ba7db42bda7b53852f\",\"seqno\":4,\"tag\":\"signature\"}",
      "sig_type": 1,
      "sig_version": 1,
      "ctime": 1473195107,
      "etime": 1977771107,
      "rtime": null,
      "sig_status": 0,
      "prev": "7868923366c3d0dfb08d6b2e366ceb9efc15a815cd5925ba7db42bda7b53852f",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 5,
      "payload_hash": "52182a5f38435f2d1fc8d9c15c3f63e069c47f16eca306046319db91c2597410",
      "sig_id": "89515cbdec34e50f763f2be7ecc354662d4a4288e98f76ef176d558f0759bfd90f",
      "sig_id_short": "iVFcvew05Q92Pyvn7MNUZi1KQojpj3bvF21V",
      "kid": "01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a",
      "sig": "hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgPdwSxzimfzXSBPTf9J5JMhoukVAnZ4nGy8xiKUJDoPcKp3BheWxvYWTFAu17ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwM2RkYzEyYzczOGE2N2YzNWQyMDRmNGRmZjQ5ZTQ5MzIxYTJlOTE1MDI3Njc4OWM2Y2JjYzYyMjk0MjQzYTBmNzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwM2RkYzEyYzczOGE2N2YzNWQyMDRmNGRmZjQ5ZTQ5MzIxYTJlOTE1MDI3Njc4OWM2Y2JjYzYyMjk0MjQzYTBmNzBhIiwidWlkIjoiZWI4MDM4OGVkZmI3MTg5ZTUxMjQxMGQ4YjJhNWY0MTkiLCJ1c2VybmFtZSI6Imdvb3piYWNoIn0sInNlcnZpY2UiOnsibmFtZSI6InR3aXR0ZXIiLCJ1c2VybmFtZSI6Imdvb3piYWNoIn0sInR5cGUiOiJ3ZWJfc2VydmljZV9iaW5kaW5nIiwidmVyc2lvbiI6MX0sImNsaWVudCI6eyJuYW1lIjoia2V5YmFzZS5pbyBnbyBjbGllbnQiLCJ2ZXJzaW9uIjoiMS4wLjE3In0sImN0aW1lIjoxNDczMTkxNzY2LCJleHBpcmVfaW4iOjUwNDU3NjAwMCwibWVya2xlX3Jvb3QiOnsiY3RpbWUiOjE0NzMxOTE2MTYsImhhc2giOiI3Mzg3ZWI3ZmU3ZGFhOGViNDViNzIxNmI2NmQyZWY1M2M3ZGNhZDFmYzAxMzliZDk4OWQ1YzgxOTdjYjRkZjdmZmNmZTVlNjQ3ZDEzOTY1NjRmMzhiODg0Yjg0YTkzZGU4N2Q4NDcxNzhjMzNkM2QxODFlNTgxOWRmYWQ5ODc1YSIsInNlcW5vIjo2MjE2MjN9LCJwcmV2IjoiODNhNzk4YzQ1MGE0OWJlMjM5OWQxZmI3YjJhODFiN2U2NjFjZWUyZDBkNGVjNTRmNjY5N2UzYWRlZTQ4ZGE0YSIsInNlcW5vIjo1LCJ0YWciOiJzaWduYXR1cmUifaNzaWfEQCUynogIVit+dgrt5r3ZciOGcctm3/2cAe+PhswM1sA4tfkW+Hsj7w3JotyK+IPcFhd1ltS7tREESgxm+Z2nFQGoc2lnX3R5cGUgpGhhc2iCpHR5cGUIpXZhbHVlxCBTrzIW9Ao2YLcHWCWzsa0uRuKDlJN1IzuWyxy9dBgFhKN0YWfNAgKndmVyc2lvbgE=",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a\",\"host\":\"keybase.io\",\"kid\":\"01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a\",\"uid\":\"eb80388edfb7189e512410d8b2a5f419\",\"username\":\"goozbach\"},\"service\":{\"name\":\"twitter\",\"username\":\"goozbach\"},\"type\":\"web_service_binding\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.17\"},\"ctime\":1473191766,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1473191616,\"hash\":\"7387eb7fe7daa8eb45b7216b66d2ef53c7dcad1fc0139bd989d5c8197cb4df7ffcfe5e647d1396564f38b884b84a93de87d847178c33d3d181e5819dfad9875a\",\"seqno\":621623},\"prev\":\"83a798c450a49be2399d1fb7b2a81b7e661cee2d0d4ec54f6697e3adee48da4a\",\"seqno\":5,\"tag\":\"signature\"}",
      "sig_type": 2,
      "sig_version": 1,
      "ctime": 1473195366,
      "etime": 1977771366,
      "rtime": null,
      "sig_status": 0,
      "prev": "83a798c450a49be2399d1fb7b2a81b7e661cee2d0d4ec54f6697e3adee48da4a",
      "proof_id": "dc2f9254537f6db8396b5d10",
      "proof_type": 2,
      "proof_text_check": "Verifying myself: I am goozbach on Keybase.io. iVFcvew05Q92Pyvn7MNUZi1KQojpj3bvF21V /",
      "proof_text_full": "Verifying myself: I am goozbach on Keybase.io. iVFcvew05Q92Pyvn7MNUZi1KQojpj3bvF21V / https://keybase.io/goozbach/sigs/iVFcvew05Q92Pyvn7MNUZi1KQojpj3bvF21V\n",
      "check_data_json": "{\"name\":\"twitter\",\"username\":\"goozbach\"}",
      "remote_id": "773248519754752000",
      "api_url": "https://twitter.com/goozbach/status/773248519754752000",
      "human_url": "https://twitter.com/goozbach/status/773248519754752000",
      "proof_state": 1,
      "proof_status": 1,
      "retry_count": 182,
      "hard_fail_count": 0,
      "last_check": 1491747331,
      "last_success": 1491747331,
      "version": 1,
      "fingerprint": ""
    },
    {
      "seqno": 6,
      "payload_hash": "67bad17649e31ccfe9cc6215285996f8c748eaafefee884d4c1a694fc709cda6",
      "sig_id": "900031c08400ed08c1dfc4d6837e1a064444044ea1e39c6e8c0d3ccc07c544c40f",
      "sig_id_short": "kAAxwIQA7QjB38TWg34aBkREBE6h45xujA08",
      "kid": "01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a",
      "sig": "hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgPdwSxzimfzXSBPTf9J5JMhoukVAnZ4nGy8xiKUJDoPcKp3BheWxvYWTFAux7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwM2RkYzEyYzczOGE2N2YzNWQyMDRmNGRmZjQ5ZTQ5MzIxYTJlOTE1MDI3Njc4OWM2Y2JjYzYyMjk0MjQzYTBmNzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwM2RkYzEyYzczOGE2N2YzNWQyMDRmNGRmZjQ5ZTQ5MzIxYTJlOTE1MDI3Njc4OWM2Y2JjYzYyMjk0MjQzYTBmNzBhIiwidWlkIjoiZWI4MDM4OGVkZmI3MTg5ZTUxMjQxMGQ4YjJhNWY0MTkiLCJ1c2VybmFtZSI6Imdvb3piYWNoIn0sInNlcnZpY2UiOnsibmFtZSI6ImdpdGh1YiIsInVzZXJuYW1lIjoiZ29vemJhY2gifSwidHlwZSI6IndlYl9zZXJ2aWNlX2JpbmRpbmciLCJ2ZXJzaW9uIjoxfSwiY2xpZW50Ijp7Im5hbWUiOiJrZXliYXNlLmlvIGdvIGNsaWVudCIsInZlcnNpb24iOiIxLjAuMTcifSwiY3RpbWUiOjE0NzMxOTE4NjIsImV4cGlyZV9pbiI6NTA0NTc2MDAwLCJtZXJrbGVfcm9vdCI6eyJjdGltZSI6MTQ3MzE5MTg1NCwiaGFzaCI6ImM5OGI4MjVkNTBiOGUzYzZkODhlYzgyODczMGQxODMyYmI3M2E0NjhkNjk3MzQ0ZDBhYTAwNGU4ZGRmYWVjNjk1ODRhYTRhM2M3ZTdhZDQ5MWU3OWIyZWVhZDRjOTE3MzBjMTIwZTliZDIyNWZjMzlhNDI3OTQwY2E1MzViOWIwIiwic2Vxbm8iOjYyMTYyOH0sInByZXYiOiI1MjE4MmE1ZjM4NDM1ZjJkMWZjOGQ5YzE1YzNmNjNlMDY5YzQ3ZjE2ZWNhMzA2MDQ2MzE5ZGI5MWMyNTk3NDEwIiwic2Vxbm8iOjYsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RAJmfRPbDI+FK0T7f3Z/XTaYx4spHkZ2KO1Ato1kE8yqulFCg7Hx0l2dwDDhR1v+iFm82rcfEblHHnZ0ZUX7clDqhzaWdfdHlwZSCkaGFzaIKkdHlwZQildmFsdWXEIP+GZFQIhSq54hTfj9Mpm6759gU++K4Lf9esWm7JA1K2o3RhZ80CAqd2ZXJzaW9uAQ==",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a\",\"host\":\"keybase.io\",\"kid\":\"01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a\",\"uid\":\"eb80388edfb7189e512410d8b2a5f419\",\"username\":\"goozbach\"},\"service\":{\"name\":\"github\",\"username\":\"goozbach\"},\"type\":\"web_service_binding\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.17\"},\"ctime\":1473191862,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1473191854,\"hash\":\"c98b825d50b8e3c6d88ec828730d1832bb73a468d697344d0aa004e8ddfaec69584aa4a3c7e7ad491e79b2eead4c91730c120e9bd225fc39a427940ca535b9b0\",\"seqno\":621628},\"prev\":\"52182a5f38435f2d1fc8d9c15c3f63e069c47f16eca306046319db91c2597410\",\"seqno\":6,\"tag\":\"signature\"}",
      "sig_type": 2,
      "sig_version": 1,
      "ctime": 1473195462,
      "etime": 1977771462,
      "rtime": null,
      "sig_status": 0,
      "prev": "52182a5f38435f2d1fc8d9c15c3f63e069c47f16eca306046319db91c2597410",
      "proof_id": "d2f06bf9449114d045b4d110",
      "proof_type": 3,
      "proof_text_check": "hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgPdwSxzimfzXSBPTf9J5JMhoukVAnZ4nGy8xiKUJDoPcKp3BheWxvYWTFAux7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwM2RkYzEyYzczOGE2N2YzNWQyMDRmNGRmZjQ5ZTQ5MzIxYTJlOTE1MDI3Njc4OWM2Y2JjYzYyMjk0MjQzYTBmNzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwM2RkYzEyYzczOGE2N2YzNWQyMDRmNGRmZjQ5ZTQ5MzIxYTJlOTE1MDI3Njc4OWM2Y2JjYzYyMjk0MjQzYTBmNzBhIiwidWlkIjoiZWI4MDM4OGVkZmI3MTg5ZTUxMjQxMGQ4YjJhNWY0MTkiLCJ1c2VybmFtZSI6Imdvb3piYWNoIn0sInNlcnZpY2UiOnsibmFtZSI6ImdpdGh1YiIsInVzZXJuYW1lIjoiZ29vemJhY2gifSwidHlwZSI6IndlYl9zZXJ2aWNlX2JpbmRpbmciLCJ2ZXJzaW9uIjoxfSwiY2xpZW50Ijp7Im5hbWUiOiJrZXliYXNlLmlvIGdvIGNsaWVudCIsInZlcnNpb24iOiIxLjAuMTcifSwiY3RpbWUiOjE0NzMxOTE4NjIsImV4cGlyZV9pbiI6NTA0NTc2MDAwLCJtZXJrbGVfcm9vdCI6eyJjdGltZSI6MTQ3MzE5MTg1NCwiaGFzaCI6ImM5OGI4MjVkNTBiOGUzYzZkODhlYzgyODczMGQxODMyYmI3M2E0NjhkNjk3MzQ0ZDBhYTAwNGU4ZGRmYWVjNjk1ODRhYTRhM2M3ZTdhZDQ5MWU3OWIyZWVhZDRjOTE3MzBjMTIwZTliZDIyNWZjMzlhNDI3OTQwY2E1MzViOWIwIiwic2Vxbm8iOjYyMTYyOH0sInByZXYiOiI1MjE4MmE1ZjM4NDM1ZjJkMWZjOGQ5YzE1YzNmNjNlMDY5YzQ3ZjE2ZWNhMzA2MDQ2MzE5ZGI5MWMyNTk3NDEwIiwic2Vxbm8iOjYsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RAJmfRPbDI+FK0T7f3Z/XTaYx4spHkZ2KO1Ato1kE8yqulFCg7Hx0l2dwDDhR1v+iFm82rcfEblHHnZ0ZUX7clDqhzaWdfdHlwZSCkaGFzaIKkdHlwZQildmFsdWXEIP+GZFQIhSq54hTfj9Mpm6759gU++K4Lf9esWm7JA1K2o3RhZ80CAqd2ZXJzaW9uAQ==",
      "proof_text_full": "### Keybase proof\n\nI hereby claim:\n\n  * I am goozbach on github.\n  * I am goozbach (https://keybase.io/goozbach) on keybase.\n  * I have a public key ASA93BLHOKZ_NdIE9N_0nkkyGi6RUCdnicbLzGIpQkOg9wo\n\nTo claim this, I am signing this object:\n\n`+"`"+``+"`"+``+"`"+`json\n{\n    \"body\": {\n        \"key\": {\n            \"eldest_kid\": \"01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a\",\n            \"host\": \"keybase.io\",\n            \"kid\": \"01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a\",\n            \"uid\": \"eb80388edfb7189e512410d8b2a5f419\",\n            \"username\": \"goozbach\"\n        },\n        \"service\": {\n            \"name\": \"github\",\n            \"username\": \"goozbach\"\n        },\n        \"type\": \"web_service_binding\",\n        \"version\": 1\n    },\n    \"client\": {\n        \"name\": \"keybase.io go client\",\n        \"version\": \"1.0.17\"\n    },\n    \"ctime\": 1473191862,\n    \"expire_in\": 504576000,\n    \"merkle_root\": {\n        \"ctime\": 1473191854,\n        \"hash\": \"c98b825d50b8e3c6d88ec828730d1832bb73a468d697344d0aa004e8ddfaec69584aa4a3c7e7ad491e79b2eead4c91730c120e9bd225fc39a427940ca535b9b0\",\n        \"seqno\": 621628\n    },\n    \"prev\": \"52182a5f38435f2d1fc8d9c15c3f63e069c47f16eca306046319db91c2597410\",\n    \"seqno\": 6,\n    \"tag\": \"signature\"\n}\n`+"`"+``+"`"+``+"`"+`\n\nwith the key [ASA93BLHOKZ_NdIE9N_0nkkyGi6RUCdnicbLzGIpQkOg9wo](https://keybase.io/goozbach), yielding the signature:\n\n`+"`"+``+"`"+``+"`"+`\nhKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgPdwSxzimfzXSBPTf9J5JMhoukVAnZ4nGy8xiKUJDoPcKp3BheWxvYWTFAux7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwM2RkYzEyYzczOGE2N2YzNWQyMDRmNGRmZjQ5ZTQ5MzIxYTJlOTE1MDI3Njc4OWM2Y2JjYzYyMjk0MjQzYTBmNzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwM2RkYzEyYzczOGE2N2YzNWQyMDRmNGRmZjQ5ZTQ5MzIxYTJlOTE1MDI3Njc4OWM2Y2JjYzYyMjk0MjQzYTBmNzBhIiwidWlkIjoiZWI4MDM4OGVkZmI3MTg5ZTUxMjQxMGQ4YjJhNWY0MTkiLCJ1c2VybmFtZSI6Imdvb3piYWNoIn0sInNlcnZpY2UiOnsibmFtZSI6ImdpdGh1YiIsInVzZXJuYW1lIjoiZ29vemJhY2gifSwidHlwZSI6IndlYl9zZXJ2aWNlX2JpbmRpbmciLCJ2ZXJzaW9uIjoxfSwiY2xpZW50Ijp7Im5hbWUiOiJrZXliYXNlLmlvIGdvIGNsaWVudCIsInZlcnNpb24iOiIxLjAuMTcifSwiY3RpbWUiOjE0NzMxOTE4NjIsImV4cGlyZV9pbiI6NTA0NTc2MDAwLCJtZXJrbGVfcm9vdCI6eyJjdGltZSI6MTQ3MzE5MTg1NCwiaGFzaCI6ImM5OGI4MjVkNTBiOGUzYzZkODhlYzgyODczMGQxODMyYmI3M2E0NjhkNjk3MzQ0ZDBhYTAwNGU4ZGRmYWVjNjk1ODRhYTRhM2M3ZTdhZDQ5MWU3OWIyZWVhZDRjOTE3MzBjMTIwZTliZDIyNWZjMzlhNDI3OTQwY2E1MzViOWIwIiwic2Vxbm8iOjYyMTYyOH0sInByZXYiOiI1MjE4MmE1ZjM4NDM1ZjJkMWZjOGQ5YzE1YzNmNjNlMDY5YzQ3ZjE2ZWNhMzA2MDQ2MzE5ZGI5MWMyNTk3NDEwIiwic2Vxbm8iOjYsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RAJmfRPbDI+FK0T7f3Z/XTaYx4spHkZ2KO1Ato1kE8yqulFCg7Hx0l2dwDDhR1v+iFm82rcfEblHHnZ0ZUX7clDqhzaWdfdHlwZSCkaGFzaIKkdHlwZQildmFsdWXEIP+GZFQIhSq54hTfj9Mpm6759gU++K4Lf9esWm7JA1K2o3RhZ80CAqd2ZXJzaW9uAQ==\n\n`+"`"+``+"`"+``+"`"+`\n\nAnd finally, I am proving ownership of the github account by posting this as a gist.\n\n### My publicly-auditable identity:\n\nhttps://keybase.io/goozbach\n\n### From the command line:\n\nConsider the [keybase command line program](https://keybase.io/download).\n\n`+"`"+``+"`"+``+"`"+`bash\n# look me up\nkeybase id goozbach\n`+"`"+``+"`"+``+"`"+`\n",
      "check_data_json": "{\"name\":\"github\",\"username\":\"goozbach\"}",
      "remote_id": "1fe30ad50c821521fc44e9ab271237ea",
      "api_url": "https://gist.githubusercontent.com/goozbach/1fe30ad50c821521fc44e9ab271237ea/raw/1816086cfe8866b6fb85adfae5958b0b61439ef3/keybase.md",
      "human_url": "https://gist.github.com/1fe30ad50c821521fc44e9ab271237ea",
      "proof_state": 1,
      "proof_status": 1,
      "retry_count": 167,
      "hard_fail_count": 0,
      "last_check": 1491840614,
      "last_success": 1491840614,
      "version": 1,
      "fingerprint": ""
    },
    {
      "seqno": 7,
      "payload_hash": "b9dd57162cfb0054eb2fe7d02d33c08a6c28bc0feed9fa2589b7b78446601702",
      "sig_id": "dfe301f1c4af593c4a1a1e7c1af4960fff945991712e3d7df2899eff88248c640f",
      "sig_id_short": "3-MB8cSvWTxKGh58GvSWD_-UWZFxLj198ome",
      "kid": "01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a",
      "sig": "hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgPdwSxzimfzXSBPTf9J5JMhoukVAnZ4nGy8xiKUJDoPcKp3BheWxvYWTFAvR7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwM2RkYzEyYzczOGE2N2YzNWQyMDRmNGRmZjQ5ZTQ5MzIxYTJlOTE1MDI3Njc4OWM2Y2JjYzYyMjk0MjQzYTBmNzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwM2RkYzEyYzczOGE2N2YzNWQyMDRmNGRmZjQ5ZTQ5MzIxYTJlOTE1MDI3Njc4OWM2Y2JjYzYyMjk0MjQzYTBmNzBhIiwidWlkIjoiZWI4MDM4OGVkZmI3MTg5ZTUxMjQxMGQ4YjJhNWY0MTkiLCJ1c2VybmFtZSI6Imdvb3piYWNoIn0sInNlcnZpY2UiOnsiaG9zdG5hbWUiOiJnb296YmFjaC5jb20iLCJwcm90b2NvbCI6Imh0dHBzOiJ9LCJ0eXBlIjoid2ViX3NlcnZpY2VfYmluZGluZyIsInZlcnNpb24iOjF9LCJjbGllbnQiOnsibmFtZSI6ImtleWJhc2UuaW8gZ28gY2xpZW50IiwidmVyc2lvbiI6IjEuMC4xNyJ9LCJjdGltZSI6MTQ3MzE5MTk4MiwiZXhwaXJlX2luIjo1MDQ1NzYwMDAsIm1lcmtsZV9yb290Ijp7ImN0aW1lIjoxNDczMTkxOTQwLCJoYXNoIjoiZTEyNzA3Nzk0MzA4ZjViMGRmODlkMzI4ZTk0N2FiYjIzZjAwN2UwNmZhZmRjMTQwNjFhZjA3MmViM2Y3N2FkMjc3YThjN2ZhN2ZjZjAzYTBiNDk4MjIzZmEwYzk0Y2Y2YWNmYzExNTZjZTA5MTQ2N2ZmMDZlOWM5YzI2ODlmZTAiLCJzZXFubyI6NjIxNjMxfSwicHJldiI6IjY3YmFkMTc2NDllMzFjY2ZlOWNjNjIxNTI4NTk5NmY4Yzc0OGVhYWZlZmVlODg0ZDRjMWE2OTRmYzcwOWNkYTYiLCJzZXFubyI6NywidGFnIjoic2lnbmF0dXJlIn2jc2lnxEBfxw42b1sIy4wYZ8p66/MEltauKOTLNHYkRLKxeadxvzdQnOJ65RTBN2FSgkc0mAfdsWAtapj1lM5tsWnx+n4DqHNpZ190eXBlIKRoYXNogqR0eXBlCKV2YWx1ZcQgQDTT+siqhvD19HWL6RTOWXhsXXN4L/te75uM1zh24dmjdGFnzQICp3ZlcnNpb24B",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a\",\"host\":\"keybase.io\",\"kid\":\"01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a\",\"uid\":\"eb80388edfb7189e512410d8b2a5f419\",\"username\":\"goozbach\"},\"service\":{\"hostname\":\"goozbach.com\",\"protocol\":\"https:\"},\"type\":\"web_service_binding\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.17\"},\"ctime\":1473191982,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1473191940,\"hash\":\"e12707794308f5b0df89d328e947abb23f007e06fafdc14061af072eb3f77ad277a8c7fa7fcf03a0b498223fa0c94cf6acfc1156ce091467ff06e9c9c2689fe0\",\"seqno\":621631},\"prev\":\"67bad17649e31ccfe9cc6215285996f8c748eaafefee884d4c1a694fc709cda6\",\"seqno\":7,\"tag\":\"signature\"}",
      "sig_type": 2,
      "sig_version": 1,
      "ctime": 1473195582,
      "etime": 1977771582,
      "rtime": null,
      "sig_status": 0,
      "prev": "67bad17649e31ccfe9cc6215285996f8c748eaafefee884d4c1a694fc709cda6",
      "proof_id": "62fed17b524b7ecc077b0310",
      "proof_type": 1000,
      "proof_text_check": "hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgPdwSxzimfzXSBPTf9J5JMhoukVAnZ4nGy8xiKUJDoPcKp3BheWxvYWTFAvR7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwM2RkYzEyYzczOGE2N2YzNWQyMDRmNGRmZjQ5ZTQ5MzIxYTJlOTE1MDI3Njc4OWM2Y2JjYzYyMjk0MjQzYTBmNzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwM2RkYzEyYzczOGE2N2YzNWQyMDRmNGRmZjQ5ZTQ5MzIxYTJlOTE1MDI3Njc4OWM2Y2JjYzYyMjk0MjQzYTBmNzBhIiwidWlkIjoiZWI4MDM4OGVkZmI3MTg5ZTUxMjQxMGQ4YjJhNWY0MTkiLCJ1c2VybmFtZSI6Imdvb3piYWNoIn0sInNlcnZpY2UiOnsiaG9zdG5hbWUiOiJnb296YmFjaC5jb20iLCJwcm90b2NvbCI6Imh0dHBzOiJ9LCJ0eXBlIjoid2ViX3NlcnZpY2VfYmluZGluZyIsInZlcnNpb24iOjF9LCJjbGllbnQiOnsibmFtZSI6ImtleWJhc2UuaW8gZ28gY2xpZW50IiwidmVyc2lvbiI6IjEuMC4xNyJ9LCJjdGltZSI6MTQ3MzE5MTk4MiwiZXhwaXJlX2luIjo1MDQ1NzYwMDAsIm1lcmtsZV9yb290Ijp7ImN0aW1lIjoxNDczMTkxOTQwLCJoYXNoIjoiZTEyNzA3Nzk0MzA4ZjViMGRmODlkMzI4ZTk0N2FiYjIzZjAwN2UwNmZhZmRjMTQwNjFhZjA3MmViM2Y3N2FkMjc3YThjN2ZhN2ZjZjAzYTBiNDk4MjIzZmEwYzk0Y2Y2YWNmYzExNTZjZTA5MTQ2N2ZmMDZlOWM5YzI2ODlmZTAiLCJzZXFubyI6NjIxNjMxfSwicHJldiI6IjY3YmFkMTc2NDllMzFjY2ZlOWNjNjIxNTI4NTk5NmY4Yzc0OGVhYWZlZmVlODg0ZDRjMWE2OTRmYzcwOWNkYTYiLCJzZXFubyI6NywidGFnIjoic2lnbmF0dXJlIn2jc2lnxEBfxw42b1sIy4wYZ8p66/MEltauKOTLNHYkRLKxeadxvzdQnOJ65RTBN2FSgkc0mAfdsWAtapj1lM5tsWnx+n4DqHNpZ190eXBlIKRoYXNogqR0eXBlCKV2YWx1ZcQgQDTT+siqhvD19HWL6RTOWXhsXXN4L/te75uM1zh24dmjdGFnzQICp3ZlcnNpb24B",
      "proof_text_full": "==================================================================\nhttps://keybase.io/goozbach\n--------------------------------------------------------------------\n\nI hereby claim:\n\n  * I am an admin of https://goozbach.com\n  * I am goozbach (https://keybase.io/goozbach) on keybase.\n  * I have a public key ASA93BLHOKZ_NdIE9N_0nkkyGi6RUCdnicbLzGIpQkOg9wo\n\nTo do so, I am signing this object:\n\n{\n    \"body\": {\n        \"key\": {\n            \"eldest_kid\": \"01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a\",\n            \"host\": \"keybase.io\",\n            \"kid\": \"01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a\",\n            \"uid\": \"eb80388edfb7189e512410d8b2a5f419\",\n            \"username\": \"goozbach\"\n        },\n        \"service\": {\n            \"hostname\": \"goozbach.com\",\n            \"protocol\": \"https:\"\n        },\n        \"type\": \"web_service_binding\",\n        \"version\": 1\n    },\n    \"client\": {\n        \"name\": \"keybase.io go client\",\n        \"version\": \"1.0.17\"\n    },\n    \"ctime\": 1473191982,\n    \"expire_in\": 504576000,\n    \"merkle_root\": {\n        \"ctime\": 1473191940,\n        \"hash\": \"e12707794308f5b0df89d328e947abb23f007e06fafdc14061af072eb3f77ad277a8c7fa7fcf03a0b498223fa0c94cf6acfc1156ce091467ff06e9c9c2689fe0\",\n        \"seqno\": 621631\n    },\n    \"prev\": \"67bad17649e31ccfe9cc6215285996f8c748eaafefee884d4c1a694fc709cda6\",\n    \"seqno\": 7,\n    \"tag\": \"signature\"\n}\n\nwhich yields the signature:\n\nhKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgPdwSxzimfzXSBPTf9J5JMhoukVAnZ4nGy8xiKUJDoPcKp3BheWxvYWTFAvR7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwM2RkYzEyYzczOGE2N2YzNWQyMDRmNGRmZjQ5ZTQ5MzIxYTJlOTE1MDI3Njc4OWM2Y2JjYzYyMjk0MjQzYTBmNzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwM2RkYzEyYzczOGE2N2YzNWQyMDRmNGRmZjQ5ZTQ5MzIxYTJlOTE1MDI3Njc4OWM2Y2JjYzYyMjk0MjQzYTBmNzBhIiwidWlkIjoiZWI4MDM4OGVkZmI3MTg5ZTUxMjQxMGQ4YjJhNWY0MTkiLCJ1c2VybmFtZSI6Imdvb3piYWNoIn0sInNlcnZpY2UiOnsiaG9zdG5hbWUiOiJnb296YmFjaC5jb20iLCJwcm90b2NvbCI6Imh0dHBzOiJ9LCJ0eXBlIjoid2ViX3NlcnZpY2VfYmluZGluZyIsInZlcnNpb24iOjF9LCJjbGllbnQiOnsibmFtZSI6ImtleWJhc2UuaW8gZ28gY2xpZW50IiwidmVyc2lvbiI6IjEuMC4xNyJ9LCJjdGltZSI6MTQ3MzE5MTk4MiwiZXhwaXJlX2luIjo1MDQ1NzYwMDAsIm1lcmtsZV9yb290Ijp7ImN0aW1lIjoxNDczMTkxOTQwLCJoYXNoIjoiZTEyNzA3Nzk0MzA4ZjViMGRmODlkMzI4ZTk0N2FiYjIzZjAwN2UwNmZhZmRjMTQwNjFhZjA3MmViM2Y3N2FkMjc3YThjN2ZhN2ZjZjAzYTBiNDk4MjIzZmEwYzk0Y2Y2YWNmYzExNTZjZTA5MTQ2N2ZmMDZlOWM5YzI2ODlmZTAiLCJzZXFubyI6NjIxNjMxfSwicHJldiI6IjY3YmFkMTc2NDllMzFjY2ZlOWNjNjIxNTI4NTk5NmY4Yzc0OGVhYWZlZmVlODg0ZDRjMWE2OTRmYzcwOWNkYTYiLCJzZXFubyI6NywidGFnIjoic2lnbmF0dXJlIn2jc2lnxEBfxw42b1sIy4wYZ8p66/MEltauKOTLNHYkRLKxeadxvzdQnOJ65RTBN2FSgkc0mAfdsWAtapj1lM5tsWnx+n4DqHNpZ190eXBlIKRoYXNogqR0eXBlCKV2YWx1ZcQgQDTT+siqhvD19HWL6RTOWXhsXXN4L/te75uM1zh24dmjdGFnzQICp3ZlcnNpb24B\n\nAnd finally, I am proving ownership of this host by posting or\nappending to this document.\n\nView my publicly-auditable identity here: https://keybase.io/goozbach\n\n==================================================================\n",
      "check_data_json": "{\"hostname\":\"goozbach.com\",\"protocol\":\"https:\"}",
      "remote_id": "https://goozbach.com/.well-known/keybase.txt",
      "api_url": "https://goozbach.com/.well-known/keybase.txt",
      "human_url": "https://goozbach.com/.well-known/keybase.txt",
      "proof_state": 3,
      "proof_status": 101,
      "retry_count": 2,
      "hard_fail_count": 501,
      "last_check": 1491592579,
      "last_success": 1491163680,
      "version": 7,
      "fingerprint": ""
    },
    {
      "seqno": 8,
      "payload_hash": "aa20a1446d5d5edb60241c1168634f8ad4797a1cdfece05dd9274894f470944e",
      "sig_id": "d1d75c54cad8b8578bd0aefec05a3c3c0bd4c576a37eab18f7a0edb4d82601ed0f",
      "sig_id_short": "0ddcVMrYuFeL0K7-wFo8PAvUxXajfqsY96Dt",
      "kid": "01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a",
      "sig": "hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgPdwSxzimfzXSBPTf9J5JMhoukVAnZ4nGy8xiKUJDoPcKp3BheWxvYWTFAux7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwM2RkYzEyYzczOGE2N2YzNWQyMDRmNGRmZjQ5ZTQ5MzIxYTJlOTE1MDI3Njc4OWM2Y2JjYzYyMjk0MjQzYTBmNzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwM2RkYzEyYzczOGE2N2YzNWQyMDRmNGRmZjQ5ZTQ5MzIxYTJlOTE1MDI3Njc4OWM2Y2JjYzYyMjk0MjQzYTBmNzBhIiwidWlkIjoiZWI4MDM4OGVkZmI3MTg5ZTUxMjQxMGQ4YjJhNWY0MTkiLCJ1c2VybmFtZSI6Imdvb3piYWNoIn0sInNlcnZpY2UiOnsibmFtZSI6InJlZGRpdCIsInVzZXJuYW1lIjoiZ29vemJhY2gifSwidHlwZSI6IndlYl9zZXJ2aWNlX2JpbmRpbmciLCJ2ZXJzaW9uIjoxfSwiY2xpZW50Ijp7Im5hbWUiOiJrZXliYXNlLmlvIGdvIGNsaWVudCIsInZlcnNpb24iOiIxLjAuMTcifSwiY3RpbWUiOjE0NzMxOTIyMTksImV4cGlyZV9pbiI6NTA0NTc2MDAwLCJtZXJrbGVfcm9vdCI6eyJjdGltZSI6MTQ3MzE5MjE4NSwiaGFzaCI6IjU5ZDMzNzZkMzM0ZDM3OTNlMTQ3NTA2N2UzNDIyYzIyOWY2MmM4Mzc1NzhjOGM2YmI3ZTE5ZjZiMzY4YzM1ZjUyNjhmYmFjNTU5YTZiMzY2YTk3ODZlMTA4YzIxMGM0YjY4NDBjMjU0ZDlkM2RmZjBhOWEwYzgwMTI1MzI4MDI2Iiwic2Vxbm8iOjYyMTY0MX0sInByZXYiOiJiOWRkNTcxNjJjZmIwMDU0ZWIyZmU3ZDAyZDMzYzA4YTZjMjhiYzBmZWVkOWZhMjU4OWI3Yjc4NDQ2NjAxNzAyIiwic2Vxbm8iOjgsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RA7lDgX2PHxmdSgBep5KUN9d2Z5DB6S6ssRvOFllu6unO3wlY7Skd9jU4/BPgBXjnnGxXgFJmaqwVm8jBM1PKiDqhzaWdfdHlwZSCkaGFzaIKkdHlwZQildmFsdWXEIDbzdW/ChIoN10GFN2ehuTLQ1fC8ilWaSeb8aejiDGu7o3RhZ80CAqd2ZXJzaW9uAQ==",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a\",\"host\":\"keybase.io\",\"kid\":\"01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a\",\"uid\":\"eb80388edfb7189e512410d8b2a5f419\",\"username\":\"goozbach\"},\"service\":{\"name\":\"reddit\",\"username\":\"goozbach\"},\"type\":\"web_service_binding\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.17\"},\"ctime\":1473192219,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1473192185,\"hash\":\"59d3376d334d3793e1475067e3422c229f62c837578c8c6bb7e19f6b368c35f5268fbac559a6b366a9786e108c210c4b6840c254d9d3dff0a9a0c80125328026\",\"seqno\":621641},\"prev\":\"b9dd57162cfb0054eb2fe7d02d33c08a6c28bc0feed9fa2589b7b78446601702\",\"seqno\":8,\"tag\":\"signature\"}",
      "sig_type": 2,
      "sig_version": 1,
      "ctime": 1473195819,
      "etime": 1977771819,
      "rtime": null,
      "sig_status": 0,
      "prev": "b9dd57162cfb0054eb2fe7d02d33c08a6c28bc0feed9fa2589b7b78446601702",
      "proof_id": "1c6b9e0e6a370a9f0efefb10",
      "proof_type": 4,
      "proof_text_check": "hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgPdwSxzimfzXSBPTf9J5JMhoukVAnZ4nGy8xiKUJDoPcKp3BheWxvYWTFAux7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwM2RkYzEyYzczOGE2N2YzNWQyMDRmNGRmZjQ5ZTQ5MzIxYTJlOTE1MDI3Njc4OWM2Y2JjYzYyMjk0MjQzYTBmNzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwM2RkYzEyYzczOGE2N2YzNWQyMDRmNGRmZjQ5ZTQ5MzIxYTJlOTE1MDI3Njc4OWM2Y2JjYzYyMjk0MjQzYTBmNzBhIiwidWlkIjoiZWI4MDM4OGVkZmI3MTg5ZTUxMjQxMGQ4YjJhNWY0MTkiLCJ1c2VybmFtZSI6Imdvb3piYWNoIn0sInNlcnZpY2UiOnsibmFtZSI6InJlZGRpdCIsInVzZXJuYW1lIjoiZ29vemJhY2gifSwidHlwZSI6IndlYl9zZXJ2aWNlX2JpbmRpbmciLCJ2ZXJzaW9uIjoxfSwiY2xpZW50Ijp7Im5hbWUiOiJrZXliYXNlLmlvIGdvIGNsaWVudCIsInZlcnNpb24iOiIxLjAuMTcifSwiY3RpbWUiOjE0NzMxOTIyMTksImV4cGlyZV9pbiI6NTA0NTc2MDAwLCJtZXJrbGVfcm9vdCI6eyJjdGltZSI6MTQ3MzE5MjE4NSwiaGFzaCI6IjU5ZDMzNzZkMzM0ZDM3OTNlMTQ3NTA2N2UzNDIyYzIyOWY2MmM4Mzc1NzhjOGM2YmI3ZTE5ZjZiMzY4YzM1ZjUyNjhmYmFjNTU5YTZiMzY2YTk3ODZlMTA4YzIxMGM0YjY4NDBjMjU0ZDlkM2RmZjBhOWEwYzgwMTI1MzI4MDI2Iiwic2Vxbm8iOjYyMTY0MX0sInByZXYiOiJiOWRkNTcxNjJjZmIwMDU0ZWIyZmU3ZDAyZDMzYzA4YTZjMjhiYzBmZWVkOWZhMjU4OWI3Yjc4NDQ2NjAxNzAyIiwic2Vxbm8iOjgsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RA7lDgX2PHxmdSgBep5KUN9d2Z5DB6S6ssRvOFllu6unO3wlY7Skd9jU4/BPgBXjnnGxXgFJmaqwVm8jBM1PKiDqhzaWdfdHlwZSCkaGFzaIKkdHlwZQildmFsdWXEIDbzdW/ChIoN10GFN2ehuTLQ1fC8ilWaSeb8aejiDGu7o3RhZ80CAqd2ZXJzaW9uAQ==",
      "proof_text_full": "### Keybase proof\n\nI hereby claim:\n\n  * I am [goozbach](https://www.reddit.com/user/goozbach) on reddit.\n  * I am [goozbach](https://keybase.io/goozbach) on keybase.\n  * I have a public key ASA93BLHOKZ_NdIE9N_0nkkyGi6RUCdnicbLzGIpQkOg9wo\n\nTo claim this, I am signing this object:\n\n\n    {\n        \"body\": {\n            \"key\": {\n                \"eldest_kid\": \"01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a\",\n                \"host\": \"keybase.io\",\n                \"kid\": \"01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a\",\n                \"uid\": \"eb80388edfb7189e512410d8b2a5f419\",\n                \"username\": \"goozbach\"\n            },\n            \"service\": {\n                \"name\": \"reddit\",\n                \"username\": \"goozbach\"\n            },\n            \"type\": \"web_service_binding\",\n            \"version\": 1\n        },\n        \"client\": {\n            \"name\": \"keybase.io go client\",\n            \"version\": \"1.0.17\"\n        },\n        \"ctime\": 1473192219,\n        \"expire_in\": 504576000,\n        \"merkle_root\": {\n            \"ctime\": 1473192185,\n            \"hash\": \"59d3376d334d3793e1475067e3422c229f62c837578c8c6bb7e19f6b368c35f5268fbac559a6b366a9786e108c210c4b6840c254d9d3dff0a9a0c80125328026\",\n            \"seqno\": 621641\n        },\n        \"prev\": \"b9dd57162cfb0054eb2fe7d02d33c08a6c28bc0feed9fa2589b7b78446601702\",\n        \"seqno\": 8,\n        \"tag\": \"signature\"\n    }\n\nwith the key from above, yielding:\n\n\n    hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgPdwSxzimfzXSBPTf9J5JMhoukVAnZ4nGy8xiKUJDoPcKp3BheWxvYWTFAux7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwM2RkYzEyYzczOGE2N2YzNWQyMDRmNGRmZjQ5ZTQ5MzIxYTJlOTE1MDI3Njc4OWM2Y2JjYzYyMjk0MjQzYTBmNzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwM2RkYzEyYzczOGE2N2YzNWQyMDRmNGRmZjQ5ZTQ5MzIxYTJlOTE1MDI3Njc4OWM2Y2JjYzYyMjk0MjQzYTBmNzBhIiwidWlkIjoiZWI4MDM4OGVkZmI3MTg5ZTUxMjQxMGQ4YjJhNWY0MTkiLCJ1c2VybmFtZSI6Imdvb3piYWNoIn0sInNlcnZpY2UiOnsibmFtZSI6InJlZGRpdCIsInVzZXJuYW1lIjoiZ29vemJhY2gifSwidHlwZSI6IndlYl9zZXJ2aWNlX2JpbmRpbmciLCJ2ZXJzaW9uIjoxfSwiY2xpZW50Ijp7Im5hbWUiOiJrZXliYXNlLmlvIGdvIGNsaWVudCIsInZlcnNpb24iOiIxLjAuMTcifSwiY3RpbWUiOjE0NzMxOTIyMTksImV4cGlyZV9pbiI6NTA0NTc2MDAwLCJtZXJrbGVfcm9vdCI6eyJjdGltZSI6MTQ3MzE5MjE4NSwiaGFzaCI6IjU5ZDMzNzZkMzM0ZDM3OTNlMTQ3NTA2N2UzNDIyYzIyOWY2MmM4Mzc1NzhjOGM2YmI3ZTE5ZjZiMzY4YzM1ZjUyNjhmYmFjNTU5YTZiMzY2YTk3ODZlMTA4YzIxMGM0YjY4NDBjMjU0ZDlkM2RmZjBhOWEwYzgwMTI1MzI4MDI2Iiwic2Vxbm8iOjYyMTY0MX0sInByZXYiOiJiOWRkNTcxNjJjZmIwMDU0ZWIyZmU3ZDAyZDMzYzA4YTZjMjhiYzBmZWVkOWZhMjU4OWI3Yjc4NDQ2NjAxNzAyIiwic2Vxbm8iOjgsInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RA7lDgX2PHxmdSgBep5KUN9d2Z5DB6S6ssRvOFllu6unO3wlY7Skd9jU4/BPgBXjnnGxXgFJmaqwVm8jBM1PKiDqhzaWdfdHlwZSCkaGFzaIKkdHlwZQildmFsdWXEIDbzdW/ChIoN10GFN2ehuTLQ1fC8ilWaSeb8aejiDGu7o3RhZ80CAqd2ZXJzaW9uAQ==\n\nFinally, I am proving my reddit account by posting it in /r/KeybaseProofs\n",
      "check_data_json": "{\"name\":\"reddit\",\"username\":\"goozbach\"}",
      "remote_id": "t3_51h4pk",
      "api_url": "https://www.reddit.com/r/KeybaseProofs/comments/51h4pk/my_keybase_proof_redditgoozbach_keybasegoozbach/.json",
      "human_url": "https://www.reddit.com/r/KeybaseProofs/comments/51h4pk/my_keybase_proof_redditgoozbach_keybasegoozbach/",
      "proof_state": 1,
      "proof_status": 1,
      "retry_count": 183,
      "hard_fail_count": 0,
      "last_check": 1491817975,
      "last_success": 1491817975,
      "version": 5,
      "fingerprint": ""
    },
    {
      "seqno": 9,
      "payload_hash": "96a3c57fe1a37b70cbf824eb511b7a218fbb9ceaba70ebbd97a5abe26b2baeb1",
      "sig_id": "0f8e3d091a20ea43b948017ccf8cd915a2adbf05b5bf5ba9bcc4cb6721e9087c0f",
      "sig_id_short": "D449CRog6kO5SAF8z4zZFaKtvwW1v1upvMTL",
      "kid": "01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a",
      "sig": "hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgPdwSxzimfzXSBPTf9J5JMhoukVAnZ4nGy8xiKUJDoPcKp3BheWxvYWTFAu97ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwM2RkYzEyYzczOGE2N2YzNWQyMDRmNGRmZjQ5ZTQ5MzIxYTJlOTE1MDI3Njc4OWM2Y2JjYzYyMjk0MjQzYTBmNzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwM2RkYzEyYzczOGE2N2YzNWQyMDRmNGRmZjQ5ZTQ5MzIxYTJlOTE1MDI3Njc4OWM2Y2JjYzYyMjk0MjQzYTBmNzBhIiwidWlkIjoiZWI4MDM4OGVkZmI3MTg5ZTUxMjQxMGQ4YjJhNWY0MTkiLCJ1c2VybmFtZSI6Imdvb3piYWNoIn0sInNlcnZpY2UiOnsiZG9tYWluIjoiZ29vemJhY2guY29tIiwicHJvdG9jb2wiOiJkbnMifSwidHlwZSI6IndlYl9zZXJ2aWNlX2JpbmRpbmciLCJ2ZXJzaW9uIjoxfSwiY2xpZW50Ijp7Im5hbWUiOiJrZXliYXNlLmlvIGdvIGNsaWVudCIsInZlcnNpb24iOiIxLjAuMTcifSwiY3RpbWUiOjE0NzMxOTIzMDksImV4cGlyZV9pbiI6NTA0NTc2MDAwLCJtZXJrbGVfcm9vdCI6eyJjdGltZSI6MTQ3MzE5MjI3OSwiaGFzaCI6IjkxYTJmOWQyOGNlZWRkZWUwYTEzYTlmMTVmNmYxZmU0ZGRkYTdhNDU0MDYwNmE2YTVlZTA3YjQxZjVjZjRhNTgyNDQ4MjNiZTZmZDJmZGU5MzM3ZWE2MzNlNDIwN2M5NGZlMzQyYWQ1YzZkNTZlNTY0M2I5MGNmNDczMTJhOGQyIiwic2Vxbm8iOjYyMTY0NX0sInByZXYiOiJhYTIwYTE0NDZkNWQ1ZWRiNjAyNDFjMTE2ODYzNGY4YWQ0Nzk3YTFjZGZlY2UwNWRkOTI3NDg5NGY0NzA5NDRlIiwic2Vxbm8iOjksInRhZyI6InNpZ25hdHVyZSJ9o3NpZ8RANJfA0qznlHyzcRkamaHY4twTMXFx/zqd65rP8kBgMSdeqYVqwjfNv8faJx9DVNc5k9/0gz+y+mpKB/fj3GNZCqhzaWdfdHlwZSCkaGFzaIKkdHlwZQildmFsdWXEILHiGVVmc2rbo+K1YofYmoAQVErxOGAelk4M63EekNGTo3RhZ80CAqd2ZXJzaW9uAQ==",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a\",\"host\":\"keybase.io\",\"kid\":\"01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a\",\"uid\":\"eb80388edfb7189e512410d8b2a5f419\",\"username\":\"goozbach\"},\"service\":{\"domain\":\"goozbach.com\",\"protocol\":\"dns\"},\"type\":\"web_service_binding\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.17\"},\"ctime\":1473192309,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1473192279,\"hash\":\"91a2f9d28ceeddee0a13a9f15f6f1fe4ddda7a4540606a6a5ee07b41f5cf4a58244823be6fd2fde9337ea633e4207c94fe342ad5c6d56e5643b90cf47312a8d2\",\"seqno\":621645},\"prev\":\"aa20a1446d5d5edb60241c1168634f8ad4797a1cdfece05dd9274894f470944e\",\"seqno\":9,\"tag\":\"signature\"}",
      "sig_type": 2,
      "sig_version": 1,
      "ctime": 1473195909,
      "etime": 1977771909,
      "rtime": null,
      "sig_status": 0,
      "prev": "aa20a1446d5d5edb60241c1168634f8ad4797a1cdfece05dd9274894f470944e",
      "proof_id": "49903932de9142e409c76110",
      "proof_type": 1001,
      "proof_text_check": "keybase-site-verification=D449CRog6kO5SAF8z4zZFaKtvwW1v1upvMTLZyHpCHw",
      "proof_text_full": "keybase-site-verification=D449CRog6kO5SAF8z4zZFaKtvwW1v1upvMTLZyHpCHw",
      "check_data_json": "{\"domain\":\"goozbach.com\",\"protocol\":\"dns\"}",
      "remote_id": "dns://goozbach.com",
      "api_url": "dns://goozbach.com",
      "human_url": "dns://goozbach.com",
      "proof_state": 1,
      "proof_status": 1,
      "retry_count": 471,
      "hard_fail_count": 0,
      "last_check": 1491835251,
      "last_success": 1491835251,
      "version": 2,
      "fingerprint": ""
    },
    {
      "seqno": 10,
      "payload_hash": "54bed32c4d5148c30dc1eca5c051b945e4933ca93a1268fcc8cddccc251229af",
      "sig_id": "397a92e20350e83ac161aacb7149e538025c6f59d5245dd94c73e621747733e50f",
      "sig_id_short": "OXqS4gNQ6DrBYarLcUnlOAJcb1nVJF3ZTHPm",
      "kid": "01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a",
      "sig": "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",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a\",\"host\":\"keybase.io\",\"kid\":\"01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a\",\"uid\":\"eb80388edfb7189e512410d8b2a5f419\",\"username\":\"goozbach\"},\"sibkey\":{\"fingerprint\":\"ad7da0ea51833fd25b786e0a2da8ce94a6ff8833\",\"full_hash\":\"fc3fde077451afc477ad47b0dcc1312a895cc2a9228a5aba1ec684d9a7b2b14a\",\"key_id\":\"2DA8CE94A6FF8833\",\"kid\":\"010179b130c5708e454055b2f738b93084f945620156185c9c6ab5519869009806790a\",\"reverse_sig\":\"-----BEGIN PGP MESSAGE-----\\nVersion: Keybase Go 1.0.18 (darwin)\\nComment: https://keybase.io/download\\n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\\n=D7u9\\n-----END PGP MESSAGE-----\\n\"},\"type\":\"sibkey\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.18\"},\"ctime\":1478817642,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1478817538,\"hash\":\"5bc5d5ac6a80e648bd0ed563d4941db3cbc502de501124bfb632faec3657be03729c7220c49b66988c92da0e44348a5637b8e4f1d454521282b9a69dc76fd8da\",\"seqno\":707879},\"prev\":\"96a3c57fe1a37b70cbf824eb511b7a218fbb9ceaba70ebbd97a5abe26b2baeb1\",\"seqno\":10,\"tag\":\"signature\"}",
      "sig_type": 1,
      "sig_version": 1,
      "ctime": 1478817642,
      "etime": 1983397242,
      "rtime": null,
      "sig_status": 0,
      "prev": "96a3c57fe1a37b70cbf824eb511b7a218fbb9ceaba70ebbd97a5abe26b2baeb1",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 11,
      "payload_hash": "659c9e79d340f75122db70e866b6b757a989d17b31283ba78cdc3cd39089ded2",
      "sig_id": "882f404342802970abe0baa97c84d7fa54e38a164f169eea428847e1bc740db30f",
      "sig_id_short": "iC9AQ0KAKXCr4LqpfITX-lTjihZPFp7qQohH",
      "kid": "01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a",
      "sig": "hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgPdwSxzimfzXSBPTf9J5JMhoukVAnZ4nGy8xiKUJDoPcKp3BheWxvYWTFAwt7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwM2RkYzEyYzczOGE2N2YzNWQyMDRmNGRmZjQ5ZTQ5MzIxYTJlOTE1MDI3Njc4OWM2Y2JjYzYyMjk0MjQzYTBmNzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwM2RkYzEyYzczOGE2N2YzNWQyMDRmNGRmZjQ5ZTQ5MzIxYTJlOTE1MDI3Njc4OWM2Y2JjYzYyMjk0MjQzYTBmNzBhIiwidWlkIjoiZWI4MDM4OGVkZmI3MTg5ZTUxMjQxMGQ4YjJhNWY0MTkiLCJ1c2VybmFtZSI6Imdvb3piYWNoIn0sInJldm9rZSI6eyJraWRzIjpbIjAxMDE3OWIxMzBjNTcwOGU0NTQwNTViMmY3MzhiOTMwODRmOTQ1NjIwMTU2MTg1YzljNmFiNTUxOTg2OTAwOTgwNjc5MGEiXX0sInR5cGUiOiJyZXZva2UiLCJ2ZXJzaW9uIjoxfSwiY2xpZW50Ijp7Im5hbWUiOiJrZXliYXNlLmlvIGdvIGNsaWVudCIsInZlcnNpb24iOiIxLjAuMTgifSwiY3RpbWUiOjE0Nzg4MzQ1NjIsImV4cGlyZV9pbiI6NTA0NTc2MDAwLCJtZXJrbGVfcm9vdCI6eyJjdGltZSI6MTQ3ODgzNDUzMiwiaGFzaCI6ImQxNTNiMDNjYzMwM2FmMzE2YTJmMWE3OWZhN2QyYTJmZDliOTRiMzdlMDNhMmNjNDhjM2EyYTY1M2VkNmNmNDA2YjgyYjg1NTY5MjM2NWJlNWE4NzgyNDI4YjBhZTVkNDdmOTEwYmY0YzI3ZDhlZWI1ODA4YjFlNTNkNDVhNTJmIiwic2Vxbm8iOjcwODMwNX0sInByZXYiOiI1NGJlZDMyYzRkNTE0OGMzMGRjMWVjYTVjMDUxYjk0NWU0OTMzY2E5M2ExMjY4ZmNjOGNkZGNjYzI1MTIyOWFmIiwic2Vxbm8iOjExLCJ0YWciOiJzaWduYXR1cmUifaNzaWfEQOQLaLdIbfx9TI/XN2N1xNyXG9CJb5VSWQ7KEdSCF63o57rxbgKM0XvlWhgo5T+4a48yRXXrGZEWo1i0Qr+mOQKoc2lnX3R5cGUgpGhhc2iCpHR5cGUIpXZhbHVlxCDdRFBo8iSakUfkdetet03Vw+H2citJU4GmuV+6DN8yU6N0YWfNAgKndmVyc2lvbgE=",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a\",\"host\":\"keybase.io\",\"kid\":\"01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a\",\"uid\":\"eb80388edfb7189e512410d8b2a5f419\",\"username\":\"goozbach\"},\"revoke\":{\"kids\":[\"010179b130c5708e454055b2f738b93084f945620156185c9c6ab5519869009806790a\"]},\"type\":\"revoke\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.18\"},\"ctime\":1478834562,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1478834532,\"hash\":\"d153b03cc303af316a2f1a79fa7d2a2fd9b94b37e03a2cc48c3a2a653ed6cf406b82b855692365be5a8782428b0ae5d47f910bf4c27d8eeb5808b1e53d45a52f\",\"seqno\":708305},\"prev\":\"54bed32c4d5148c30dc1eca5c051b945e4933ca93a1268fcc8cddccc251229af\",\"seqno\":11,\"tag\":\"signature\"}",
      "sig_type": 5,
      "sig_version": 1,
      "ctime": 1478834562,
      "etime": 1983414162,
      "rtime": null,
      "sig_status": 0,
      "prev": "54bed32c4d5148c30dc1eca5c051b945e4933ca93a1268fcc8cddccc251229af",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 12,
      "payload_hash": "8fcc339ab07900182fe2db086e081007cd442d34a2b52690eae8b46756b6036e",
      "sig_id": "c88b98e98c80e01ef83c7d028af74b45c685c4d5df28ae85437857aef011d7e80f",
      "sig_id_short": "yIuY6YyA4B74PH0CivdLRcaFxNXfKK6FQ3hX",
      "kid": "01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a",
      "sig": "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",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a\",\"host\":\"keybase.io\",\"kid\":\"01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a\",\"uid\":\"eb80388edfb7189e512410d8b2a5f419\",\"username\":\"goozbach\"},\"sibkey\":{\"fingerprint\":\"ad7da0ea51833fd25b786e0a2da8ce94a6ff8833\",\"full_hash\":\"9e9fcf955ae6743732ba1aa8bb1ab4fe068e100831b99b183cc3475be80660b8\",\"key_id\":\"2DA8CE94A6FF8833\",\"kid\":\"010179b130c5708e454055b2f738b93084f945620156185c9c6ab5519869009806790a\",\"reverse_sig\":\"-----BEGIN PGP MESSAGE-----\\nVersion: Keybase Go 1.0.18 (darwin)\\nComment: https://keybase.io/download\\n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\\n=ieYD\\n-----END PGP MESSAGE-----\\n\"},\"type\":\"sibkey\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.18\"},\"ctime\":1478835561,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1478835526,\"hash\":\"788af2c92c677bf23719cedd3d8f720d31b69e855f42510a0e2951397017b8a55c65e20afed83883b8a8afd68dae05dac565ffeaab7709dabed0eeffd5b5a183\",\"seqno\":708325},\"prev\":\"659c9e79d340f75122db70e866b6b757a989d17b31283ba78cdc3cd39089ded2\",\"seqno\":12,\"tag\":\"signature\"}",
      "sig_type": 1,
      "sig_version": 1,
      "ctime": 1478835561,
      "etime": 1983415161,
      "rtime": null,
      "sig_status": 0,
      "prev": "659c9e79d340f75122db70e866b6b757a989d17b31283ba78cdc3cd39089ded2",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 13,
      "payload_hash": "007baca03135fdd901ca2ff50275ee9b051547bb9f5a50aaaba779d43d8d19be",
      "sig_id": "9fa914dfd17052be426085383b4aa8d8aabfed07f1d992b16afc6ea3cd8c69be0f",
      "sig_id_short": "n6kU39FwUr5CYIU4O0qo2Kq_7Qfx2ZKxavxu",
      "kid": "01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a",
      "sig": "hKRib2R5hqhkZXRhY2hlZMOpaGFzaF90eXBlCqNrZXnEIwEgPdwSxzimfzXSBPTf9J5JMhoukVAnZ4nGy8xiKUJDoPcKp3BheWxvYWTFAwt7ImJvZHkiOnsia2V5Ijp7ImVsZGVzdF9raWQiOiIwMTIwM2RkYzEyYzczOGE2N2YzNWQyMDRmNGRmZjQ5ZTQ5MzIxYTJlOTE1MDI3Njc4OWM2Y2JjYzYyMjk0MjQzYTBmNzBhIiwiaG9zdCI6ImtleWJhc2UuaW8iLCJraWQiOiIwMTIwM2RkYzEyYzczOGE2N2YzNWQyMDRmNGRmZjQ5ZTQ5MzIxYTJlOTE1MDI3Njc4OWM2Y2JjYzYyMjk0MjQzYTBmNzBhIiwidWlkIjoiZWI4MDM4OGVkZmI3MTg5ZTUxMjQxMGQ4YjJhNWY0MTkiLCJ1c2VybmFtZSI6Imdvb3piYWNoIn0sInJldm9rZSI6eyJraWRzIjpbIjAxMDE3OWIxMzBjNTcwOGU0NTQwNTViMmY3MzhiOTMwODRmOTQ1NjIwMTU2MTg1YzljNmFiNTUxOTg2OTAwOTgwNjc5MGEiXX0sInR5cGUiOiJyZXZva2UiLCJ2ZXJzaW9uIjoxfSwiY2xpZW50Ijp7Im5hbWUiOiJrZXliYXNlLmlvIGdvIGNsaWVudCIsInZlcnNpb24iOiIxLjAuMTgifSwiY3RpbWUiOjE0Nzg4MzU2NjEsImV4cGlyZV9pbiI6NTA0NTc2MDAwLCJtZXJrbGVfcm9vdCI6eyJjdGltZSI6MTQ3ODgzNTYwMiwiaGFzaCI6IjQxMTIwYWM4ZTU5NjU5YjEzNTE2MTZjNDdlNWFhMTMxNmMxMmUwMjQwMWViZTYzN2Q4Y2M0YTBmNTE0OTY2ZmExOTg0MDA1MGU1ODczM2IwN2M3MDY3YTNmNmJmZWNkYTE4MDJjNzUzYmYxOThmNDc4NjY1Y2JhN2NkNTU2ODk4Iiwic2Vxbm8iOjcwODMyN30sInByZXYiOiI4ZmNjMzM5YWIwNzkwMDE4MmZlMmRiMDg2ZTA4MTAwN2NkNDQyZDM0YTJiNTI2OTBlYWU4YjQ2NzU2YjYwMzZlIiwic2Vxbm8iOjEzLCJ0YWciOiJzaWduYXR1cmUifaNzaWfEQIfwl1Cq0wYqB3vlgt8CEJGZzr1dcp73kQHxsyY6G3crqfEWYWEwNrLSlPRm62BASsawrfdNyoiuTDyTwUx5wg+oc2lnX3R5cGUgpGhhc2iCpHR5cGUIpXZhbHVlxCA/EN4QzLISLCwJmJIdXgZZRk3tBJFdr3Yfu9GMExi5MKN0YWfNAgKndmVyc2lvbgE=",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a\",\"host\":\"keybase.io\",\"kid\":\"01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a\",\"uid\":\"eb80388edfb7189e512410d8b2a5f419\",\"username\":\"goozbach\"},\"revoke\":{\"kids\":[\"010179b130c5708e454055b2f738b93084f945620156185c9c6ab5519869009806790a\"]},\"type\":\"revoke\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.18\"},\"ctime\":1478835661,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1478835602,\"hash\":\"41120ac8e59659b1351616c47e5aa1316c12e02401ebe637d8cc4a0f514966fa19840050e58733b07c7067a3f6bfecda1802c753bf198f478665cba7cd556898\",\"seqno\":708327},\"prev\":\"8fcc339ab07900182fe2db086e081007cd442d34a2b52690eae8b46756b6036e\",\"seqno\":13,\"tag\":\"signature\"}",
      "sig_type": 5,
      "sig_version": 1,
      "ctime": 1478835661,
      "etime": 1983415261,
      "rtime": null,
      "sig_status": 0,
      "prev": "8fcc339ab07900182fe2db086e081007cd442d34a2b52690eae8b46756b6036e",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    },
    {
      "seqno": 14,
      "payload_hash": "7d40c306bb893944d81fefa16c353ab034df505ec65d18e6c882e3abdc0d6da5",
      "sig_id": "73e7596c3aa50955b49f202ad2cb09613450447be156623b02b611c3bfd312290f",
      "sig_id_short": "c-dZbDqlCVW0nyAq0ssJYTRQRHvhVmI7ArYR",
      "kid": "01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a",
      "sig": "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",
      "payload_json": "{\"body\":{\"key\":{\"eldest_kid\":\"01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a\",\"host\":\"keybase.io\",\"kid\":\"01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a\",\"uid\":\"eb80388edfb7189e512410d8b2a5f419\",\"username\":\"goozbach\"},\"sibkey\":{\"fingerprint\":\"ad7da0ea51833fd25b786e0a2da8ce94a6ff8833\",\"full_hash\":\"58767c3e79c7c7c29dc3bfc570f3c60d394049060f725fda377b384c427c66fc\",\"key_id\":\"2DA8CE94A6FF8833\",\"kid\":\"010179b130c5708e454055b2f738b93084f945620156185c9c6ab5519869009806790a\",\"reverse_sig\":\"-----BEGIN PGP MESSAGE-----\\nComment: https://keybase.io/download\\nVersion: Keybase Go 1.0.18 (darwin)\\n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\\n=Ilbw\\n-----END PGP MESSAGE-----\\n\"},\"type\":\"sibkey\",\"version\":1},\"client\":{\"name\":\"keybase.io go client\",\"version\":\"1.0.18\"},\"ctime\":1478837293,\"expire_in\":504576000,\"merkle_root\":{\"ctime\":1478837243,\"hash\":\"cb6019679f0082764ec390438460329b64f9cb263070834f4bb8eecae997943249fa75451dfe79ba6b1b3f596a5769603a79be3890da08efdda5efcc9d0a9ce5\",\"seqno\":708356},\"prev\":\"007baca03135fdd901ca2ff50275ee9b051547bb9f5a50aaaba779d43d8d19be\",\"seqno\":14,\"tag\":\"signature\"}",
      "sig_type": 1,
      "sig_version": 1,
      "ctime": 1478837293,
      "etime": 1983416893,
      "rtime": null,
      "sig_status": 0,
      "prev": "007baca03135fdd901ca2ff50275ee9b051547bb9f5a50aaaba779d43d8d19be",
      "proof_id": null,
      "proof_type": null,
      "proof_text_check": null,
      "proof_text_full": null,
      "check_data_json": null,
      "remote_id": null,
      "api_url": null,
      "human_url": null,
      "proof_state": null,
      "proof_status": null,
      "retry_count": null,
      "hard_fail_count": null,
      "last_check": null,
      "last_success": null,
      "version": null,
      "fingerprint": ""
    }
  ],
  "keys": [
    "01203ddc12c738a67f35d204f4dff49e49321a2e9150276789c6cbcc62294243a0f70a",
    "0121bc86be1183452c33cf2740855dfb99a690a4bb2839d2a6b4c65e07450b76b21f0a",
    "0121e8b5b4787126b7c87874912a0a7cdf5aa4d0a6f1837ba42e9d1a62df37fa53400a",
    "0120bc9e1022ac7f0463e910601642d6ac3a7d8fd3056f177de35e89aa602f6da6a20a",
    "-----BEGIN PGP PUBLIC KEY BLOCK-----\nComment: https://keybase.io/download\nVersion: Keybase Go 1.0.18 (darwin)\n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\nuq/0Pt7yImCBUfdze4sOB21w/VVAiQ9cGdUbypyJsv9JvgA6j+UgZg==\n=ARIH\n-----END PGP PUBLIC KEY BLOCK-----",
    "-----BEGIN PGP PUBLIC KEY BLOCK-----\nComment: https://keybase.io/download\nVersion: Keybase Go 1.0.18 (darwin)\n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\nuq/0Pt7yImCBUfdze4sOB21w/VVAiQ9cGdUbypyJsv9JvgA6j+UgZg==\n=HUCj\n-----END PGP PUBLIC KEY BLOCK-----",
    "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: Keybase Go 1.0.18 (darwin)\nComment: https://keybase.io/download\n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\nLvW91QxcJA==\n=hMEa\n-----END PGP PUBLIC KEY BLOCK-----"
  ]
}`,
	"hardcoded_reset.json": `{
    "chain": [
        {
            "seqno": 1,
            "payload_hash": "45aa7f220a5f1a54e0d2a7f7ec16fff99c90849842f9eb06c51fab5282c4c997",
            "sig_id": "ab1dc65843751ca45a730b7ee4f191e40df133a4f7f2b1bb9c23cdd739a523b10f",
            "sig_id_short": "qx3GWEN1HKRacwt-5PGR5A3xM6T38rG7nCPN",
            "kid": "0101550c284019ad5249e04ccbcf0a6176eddd371e5ba17c034b27b8d59d5cce84300a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n\nowGtUmtIFFEU3vXdkpWJ9CMtG6F+uK53Xjuz5g8rjKKCHqBsisvMvXd1dJtdZ2bV\n9QEVBIVSUWmRiJVhRFBkVlhpVBSKb62tkAoziTXECoqIimbMogj61f117znfd853\nv3MOx4abYsx7jg8NmBPyC8y9IdGUE3V9cRUhelGAyKgioEfCsmbcZGEXJjKIEhwQ\nBRXbJG+y7EXYVqwmz2GsRBlWVMkr6yhg420UIGqsBt5gYw/CquYqkZCRJQHJsgBS\nPANIh4BYinFgwEAoQjcQ7CRnxwghmiMxKwokBwHNiBQn8oh1IBZCzDM0AILe0C3J\nhVjxKZIhkSDddo6nMUQMYLH+gA5EUQwmAYUdkEF2gYd2TAOsE4u8qvbHZ4hZpa4f\n8v7G/2fd/tlyNEdBEoqIozDj5llS4BhE0RTD8QJH0aTDAKpYmTMeCWUS8gmKEjBs\n1eNlEsS/DUYrlzQNK/8iaQGfESzHomuO7xIlGekm/hperj49UodCTTIKkAxDUzyw\ns8BK4AqfpGCXZCBYzs4D/VgJn4LLiAzZ7/EYokplr57VOwmFeiNVKpQFza9gouaA\nOTHCZI4xRUWGGQtmssxb9HPrChtiTIc6q9ZAwI86iQpbLXU1ZUfditrMjumLXS7f\nbnWmZaw7dapzui27ftOJ2++am86+uLutryl+b99N37qCDVvFxuFzoeev3y/LP+gB\n9U+dlo2VFpvz5a3onk/70hJbxk+y0elvF0aMV13pusA4r5W0NzTIqRnHUJ8zu/1S\n0pvHQ/csHVnSnbSE2LyjfdalkzMB08fcQN6T6TPssxs5cEEwFPyWPqVOPPw6v/b8\naN1EsG6w2sG/qmzdcj8RBUfi+6tTVofxSV1trQPx+5PA6Qdx+cM714c/6u+pjij9\nkFk0Wdq8OW151ph/sDg00btqZElR4+XGL93EZ3vk9iOn1q6M46a/Aw==\n=KYmi\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.8.20\"},\"key\":{\"eldest_kid\":\"0101550c284019ad5249e04ccbcf0a6176eddd371e5ba17c034b27b8d59d5cce84300a\",\"fingerprint\":\"1f6783ecd405e1f6c9d224e102e9c4d6a8c6e30e\",\"host\":\"keybase.io\",\"key_id\":\"02e9c4d6a8c6e30e\",\"kid\":\"0101550c284019ad5249e04ccbcf0a6176eddd371e5ba17c034b27b8d59d5cce84300a\",\"uid\":\"372c1cbd72e4f851a74d232478a72319\",\"username\":\"davidparry\"},\"service\":{\"name\":\"twitter\",\"username\":\"davidparry\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1443280650,\"expire_in\":157680000,\"prev\":null,\"seqno\":1,\"tag\":\"signature\"}",
            "sig_type": 2,
            "ctime": 1443280650,
            "etime": 1600960650,
            "rtime": null,
            "sig_status": 2,
            "prev": null,
            "proof_id": "7cd19fa85fc05075a670dd10",
            "proof_type": 2,
            "proof_text_check": "Verifying myself: I am davidparry on Keybase.io. qx3GWEN1HKRacwt-5PGR5A3xM6T38rG7nCPN /",
            "proof_text_full": "Verifying myself: I am davidparry on Keybase.io. qx3GWEN1HKRacwt-5PGR5A3xM6T38rG7nCPN / https://keybase.io/davidparry/sigs/qx3GWEN1HKRacwt-5PGR5A3xM6T38rG7nCPN\n",
            "check_data_json": "{\"name\":\"twitter\",\"username\":\"davidparry\"}",
            "remote_id": "647792257505054720",
            "api_url": "https://twitter.com/davidparry/status/647792257505054720",
            "human_url": "https://twitter.com/davidparry/status/647792257505054720",
            "proof_state": 5,
            "proof_status": 1,
            "retry_count": 1,
            "hard_fail_count": 0,
            "last_check": 1443280691,
            "last_success": 1443280691,
            "version": 1,
            "fingerprint": "1f6783ecd405e1f6c9d224e102e9c4d6a8c6e30e"
        },
        {
            "seqno": 2,
            "payload_hash": "bc898feeb7a2717e23dc1f457dc18902fb9157bf86374b2a0b1aaba4f2831bee",
            "sig_id": "32eab86aa31796db3200f42f2553d330b8a68931544bbb98452a80ad2b0003d30f",
            "sig_id_short": "Muq4aqMXltsyAPQvJVPTMLimiTFUS7uYRSqA",
            "kid": "0101550c284019ad5249e04ccbcf0a6176eddd371e5ba17c034b27b8d59d5cce84300a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n\nowGtUllsTUEYvtfakiZiaeIGaQ4i0utmzpw5W1se2gfLA61YSug1Z+Y/10Hvvc49\nbbU3RAlNJbYgqqFB0ojGGtRSiS14EEtSS4gttrRKqhFBQzinSojEk3ma+ef7vvn+\n759NaT19Kf7K7bdu+IfMK/JfazV8s/pcSk8KRoyXC1lJgS2xIOp4uygtBiFLWAzl\nBk1AyIplRGMcQosSGd2YoFAKdsKKRV0UCmkhjITlQQ/vsWEJh4QTXmxx71ZEoiwj\nhjWCRJ1yGRMdEGHMYCaiiqgqwDmXVBFkg4oqQxIxsGpoXNa5zBhoREKIug+aVjQC\ndty2PIuCaCqqJgHjBMngHpjOMSYgIgw6I1yhGlNAQuASF8YSzh/NCF1Owz/s/Y3/\nz75LuuQkFTORGVzFQExNFqlKOJYwUTWqYknUPWAC7O7gOS21eJzadrkXq1svtRj8\nNhinzHIcsP9FcsrjXrEMjHA3P2xYUe6G+Gt429zpiS6UOZYnIBIiYc3tSAkKsCxu\n2RC2PISsKhpyV1CI21DqahKZUtXEGFHZFKlMAHHsFlRgomKapq4zHWlE1wg2dTCQ\nwmTRpIaMNcwI03VV8FpaGo0JWdj1SSOuZMKKRKlTYoOwvNo/rJfPn+Lr07uH9z19\n/VIH/PyzK6r6fmP6p7M1Z5wPT5OFg8z8ptZnn5Xg/iktJUJzfUvvtXV525sCKLCr\n4GF2YmdqRf7EewuuXqoQLuaO0KQJK89ffrOv/uCcQy2Nd47W1RbY6fxCTv7oJzNa\nUw4lq/ZW5aZ1VB+YbgYK4NXFpsebC0Nvp57syFl9vQHqHmWMjd/fndPeeOX6+v6n\nZ4+b7IYTeeGPT3p9O/nl3IOhY09rw+nxkfkzw+Et7060Bs5+OzWwaMeawfU1aRue\nj0/v+FB4bS6af3h01c2szOzL9ccONmze37mx0maBPVvff53ZEWtYlXtr3deXtcV1\nzdMa7xY54zPVvE61R2R4+GP74DE5zZnVbW3O7iOjsr8D\n=wPnA\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.8.20\"},\"key\":{\"eldest_kid\":\"0101550c284019ad5249e04ccbcf0a6176eddd371e5ba17c034b27b8d59d5cce84300a\",\"fingerprint\":\"1f6783ecd405e1f6c9d224e102e9c4d6a8c6e30e\",\"host\":\"keybase.io\",\"key_id\":\"02e9c4d6a8c6e30e\",\"kid\":\"0101550c284019ad5249e04ccbcf0a6176eddd371e5ba17c034b27b8d59d5cce84300a\",\"uid\":\"372c1cbd72e4f851a74d232478a72319\",\"username\":\"davidparry\"},\"service\":{\"name\":\"twitter\",\"username\":\"davidparry\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1443283716,\"expire_in\":157680000,\"prev\":\"45aa7f220a5f1a54e0d2a7f7ec16fff99c90849842f9eb06c51fab5282c4c997\",\"seqno\":2,\"tag\":\"signature\"}",
            "sig_type": 2,
            "ctime": 1443283716,
            "etime": 1600963716,
            "rtime": null,
            "sig_status": 0,
            "prev": "45aa7f220a5f1a54e0d2a7f7ec16fff99c90849842f9eb06c51fab5282c4c997",
            "proof_id": "215636b046e902ceacacde10",
            "proof_type": 2,
            "proof_text_check": "Verifying myself: I am davidparry on Keybase.io. Muq4aqMXltsyAPQvJVPTMLimiTFUS7uYRSqA /",
            "proof_text_full": "Verifying myself: I am davidparry on Keybase.io. Muq4aqMXltsyAPQvJVPTMLimiTFUS7uYRSqA / https://keybase.io/davidparry/sigs/Muq4aqMXltsyAPQvJVPTMLimiTFUS7uYRSqA\n",
            "check_data_json": "{\"name\":\"twitter\",\"username\":\"davidparry\"}",
            "remote_id": "647805236594278404",
            "api_url": "https://twitter.com/davidparry/status/647805236594278404",
            "human_url": "https://twitter.com/davidparry/status/647805236594278404",
            "proof_state": 1,
            "proof_status": 1,
            "retry_count": 34,
            "hard_fail_count": 0,
            "last_check": 1445242202,
            "last_success": 1445242202,
            "version": 2,
            "fingerprint": "1f6783ecd405e1f6c9d224e102e9c4d6a8c6e30e"
        }
    ],
    "keys": [
        "-----BEGIN PGP PUBLIC KEY BLOCK-----\nVersion: Keybase OpenPGP v2.0.46\nComment: https://keybase.io/crypto\n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\nS3sAQg==\n=KcM3\n-----END PGP PUBLIC KEY BLOCK-----"
    ],
    "uid": "372c1cbd72e4f851a74d232478a72319",
    "username": "davidparry"
}
`,
	"jack_chain.json": `{
    "username": "oconnor663",
    "uid": "41b1f75fb55046d370608425a3208100",
    "chain": [
        {
            "seqno": 1,
            "payload_hash": "b421b35b23fc5da8a69fab3978ff34689fcaaf0048fc8af822aa32b5121c8193",
            "sig_id": "e43b8971ff4bdc1beb57fecc15c1b3455997d1c6ea5596e14f25bdf4d5ad47580f",
            "sig_id_short": "5DuJcf9L3BvrV_7MFcGzRVmX0cbqVZbhTyW9",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n\nowGbwMvMwMFY2J6dsdZD14jx9IGPSQwh0yOXVisl5adUKllVKyXnZKbmlYBYeYm5\nqUpWStmplUmJxal6mfkKefkpqXpZxQpQNTpKZalFxZn5eUBVBnrmesZKtTog5SDN\naZl56alFBUWZILOUTC0SjUxMU4xTLJKMTJMNTdPMklON09LSzA0tzM2SzCwSU0ws\njFKMjYBGZuQXl6DYqgQ2Mz4zBSgKVu9kZuHoAlTvAlZfCpYwMUwyTDM3TUsyNTUw\nMUsxNjcwM7AwMTJNNDYysDA0MAApLE4tgnopPzk/Ly+/yMwM7GKgeFlmciqSl0vK\nM0tKUovwaSqpLAAJlqcmxUP1xydl5qUAfY0cLIZAlcklmSD9hiaGlkYGpgZGljpK\nqRUFmUWp8ZkgFabmZhYGQKCjVFCUWqZklVeakwNyU2FePlAWaFFiOtCe4sz0vMSS\n0qJUpdpORhkWBkYOBjZWJlDMMXBxCsDic9kVDobOWXPTXnulBP/5+WXyw94r/dXi\nayb80JuTNUfjzJXohNsN04zSOku9Xv77VyD46tHWzPqi+xe25ESJmS+d6JHZvOh+\n/KHTETHKb/1FKlborXgV9cO6bL3k3mXef9a93/Djg76cYFmxF2P5gcmXxK+IvjjF\nu7IhjvdNvMTbMP5V+qqKF9KdFjo093V/6VGeUKP8TvnFpv6sgIoXVv3HcvzermQ2\n+d/0dhrDj86Y3o8FB2ae5uu5dNv0zifxzjdX0yMzjy51Of36cnF8pMwj3zQrQyPP\nqPmlvcFzJrZlXeiZbda+sEV4Z6pFD9dKg8IPFT18nSE7TjPyde1rfbHydFjQPL3p\n736G/OYocJgxqw8A\n=bgPh\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.7.3\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876B68AD482D32\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"service\":{\"name\":\"twitter\",\"username\":\"oconnor663\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1419205029,\"expire_in\":157680000,\"prev\":null,\"seqno\":1,\"tag\":\"signature\"}",
            "sig_type": 2,
            "ctime": 1419205029,
            "etime": 1576885029,
            "rtime": null,
            "sig_status": 0,
            "prev": null,
            "proof_id": "756a07b444d2f1044a162210",
            "proof_type": 2,
            "proof_text_check": "Verifying myself: I am oconnor663 on Keybase.io. 5DuJcf9L3BvrV_7MFcGzRVmX0cbqVZbhTyW9 /",
            "proof_text_full": "Verifying myself: I am oconnor663 on Keybase.io. 5DuJcf9L3BvrV_7MFcGzRVmX0cbqVZbhTyW9 / https://keybase.io/oconnor663/sigs/5DuJcf9L3BvrV_7MFcGzRVmX0cbqVZbhTyW9\n",
            "check_data_json": "{\"name\":\"twitter\",\"username\":\"oconnor663\"}",
            "remote_id": "546811878073061377",
            "api_url": "https://twitter.com/oconnor663/status/546811878073061377",
            "human_url": "https://twitter.com/oconnor663/status/546811878073061377",
            "proof_state": 1,
            "proof_status": 1,
            "retry_count": 337,
            "hard_fail_count": 0,
            "last_check": 1434439958,
            "last_success": 1434439958,
            "version": 2,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 2,
            "payload_hash": "382e1d68b4b81c6643483a74715ac0964e291fb2b52c4e9fa6c73d304ab9108f",
            "sig_id": "9d5b60c0d486f34089a922bcc5e77fb643b9781cbcc894af7e123fad50645d6a0f",
            "sig_id_short": "nVtgwNSG80CJqSK8xed_tkO5eBy8yJSvfhI_",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n\nowF9kltoFFcYx3et1VWrWBQJsaCdFq3tusy5zhn1QYOUVh9MglQq1vWcM+dkJ5eZ\ndWaTmNqIWmlTrDRtpaLSghdaIqYS6614wbsWry/6oEVoi6AvkgRFVHA9s1iqL52n\nme/8vv/3/893uke/ksqkl3/ZVOj9YDpMX7grUgs3LR610hKh12HNWGnJZl8FpeQt\n4C3KmmE1qQ7BY5Xzw8lB6KlcYzz5OZO12lQU+2FgKDvn5JDVmU3wpFn7QYOKipGf\naFmEcYiJhzwmIJGAaCoV0lo7gDlUUMY9zKCHoJEshHHppalWRTPve6Za4WsomzPX\n8HMrfGvlAAMBtEO0IMTG1EOOTW2GIeEI2gzYdgLGKnoeKZRhEIQRpRXHLSpqalb5\nKAwrsWXJTyCAgQttYiPHeOJxwbR51HGl4EBShRHChAosJQEMONz2kKnbLuFcJY1A\nEQKAhK459Lh0lJBcciopcIiHkbax4Awg1xFMCcQEEwQJ6EmICMXCVo4rHMk5ocYC\n0tyMMTNNiFgtD0JjzrjDbmfyHbX5Ur2wrga/VGgV/5e31FFMiu1K5A2yK+nPCz/w\nzMZeXCkw6Mv/AiCQtdSKoh+pvJ8QxKHMNk/WKkaqzWgKDIFARECkJfE449TVXJiY\nTGuEKXO1CaVtGzMtGdcMQm42JAiAQDLgov8SQuOTNxjJ2G8IeKk1UlbnV+k3hqbS\nmdSwV4ckdzY1csTYfy9yqjeTOji8+KB672c9PasmiqPhVFS+N38n+e2cqvqkdikt\nrC4e7e//R7w57/j8R69lahr3/Tx4/mI0ru/bqku98Cy41f/37acXa8+sv5a7scb9\n/tjA/m/WvfPk5M3amikb3qvecvvWYOav4VtP73n9u3dnLdzx9YGldX0f7uiCi7qf\n/vrWD79kV/fdf7v2fPHwPae+ceKPB2bWb/+U7qsfv/Exm3R/wF3c/WTtwZ5j8wZ3\nnZnQdm7KF3WNh6ovjKH4Sv+GzPvTRi+b+cd6sK5c9/Dy6SHtH338aFthSbpqQupU\nze/l9s/317XUXyuvmj30z40DW6ubx95Zc6hq77jC7gXl+PD18XPOnuw6Me2nI11X\nN496Bg==\n=diQc\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.7.3\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876B68AD482D32\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"merkle_root\":{\"ctime\":1419205037,\"hash\":\"d679cba1c6e433456b4cc51817a0d3a1c095aae41921e5511c29817dac7ebcaca6c6175d43f04ba81397b8eb38b8b53b2dc23564b0e79b7caa562053facc56e4\",\"seqno\":114149},\"service\":{\"name\":\"github\",\"username\":\"oconnor663\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1419205131,\"expire_in\":157680000,\"prev\":\"b421b35b23fc5da8a69fab3978ff34689fcaaf0048fc8af822aa32b5121c8193\",\"seqno\":2,\"tag\":\"signature\"}",
            "sig_type": 2,
            "ctime": 1419205131,
            "etime": 1576885131,
            "rtime": null,
            "sig_status": 0,
            "prev": "b421b35b23fc5da8a69fab3978ff34689fcaaf0048fc8af822aa32b5121c8193",
            "proof_id": "034487589bf1339ef37e6210",
            "proof_type": 3,
            "proof_text_check": "\n\nowF9kltoFFcYx3et1VWrWBQJsaCdFq3tusy5zhn1QYOUVh9MglQq1vWcM+dkJ5eZ\ndWaTmNqIWmlTrDRtpaLSghdaIqYS6614wbsWry/6oEVoi6AvkgRFVHA9s1iqL52n\nme/8vv/3/893uke/ksqkl3/ZVOj9YDpMX7grUgs3LR610hKh12HNWGnJZl8FpeQt\n4C3KmmE1qQ7BY5Xzw8lB6KlcYzz5OZO12lQU+2FgKDvn5JDVmU3wpFn7QYOKipGf\naFmEcYiJhzwmIJGAaCoV0lo7gDlUUMY9zKCHoJEshHHppalWRTPve6Za4WsomzPX\n8HMrfGvlAAMBtEO0IMTG1EOOTW2GIeEI2gzYdgLGKnoeKZRhEIQRpRXHLSpqalb5\nKAwrsWXJTyCAgQttYiPHeOJxwbR51HGl4EBShRHChAosJQEMONz2kKnbLuFcJY1A\nEQKAhK459Lh0lJBcciopcIiHkbax4Awg1xFMCcQEEwQJ6EmICMXCVo4rHMk5ocYC\n0tyMMTNNiFgtD0JjzrjDbmfyHbX5Ur2wrga/VGgV/5e31FFMiu1K5A2yK+nPCz/w\nzMZeXCkw6Mv/AiCQtdSKoh+pvJ8QxKHMNk/WKkaqzWgKDIFARECkJfE449TVXJiY\nTGuEKXO1CaVtGzMtGdcMQm42JAiAQDLgov8SQuOTNxjJ2G8IeKk1UlbnV+k3hqbS\nmdSwV4ckdzY1csTYfy9yqjeTOji8+KB672c9PasmiqPhVFS+N38n+e2cqvqkdikt\nrC4e7e//R7w57/j8R69lahr3/Tx4/mI0ru/bqku98Cy41f/37acXa8+sv5a7scb9\n/tjA/m/WvfPk5M3amikb3qvecvvWYOav4VtP73n9u3dnLdzx9YGldX0f7uiCi7qf\n/vrWD79kV/fdf7v2fPHwPae+ceKPB2bWb/+U7qsfv/Exm3R/wF3c/WTtwZ5j8wZ3\nnZnQdm7KF3WNh6ovjKH4Sv+GzPvTRi+b+cd6sK5c9/Dy6SHtH338aFthSbpqQupU\nze/l9s/317XUXyuvmj30z40DW6ubx95Zc6hq77jC7gXl+PD18XPOnuw6Me2nI11X\nN496Bg==\n",
            "proof_text_full": "### Keybase proof\n\nI hereby claim:\n\n  * I am oconnor663 on github.\n  * I am oconnor663 (https://keybase.io/oconnor663) on keybase.\n  * I have a public key whose fingerprint is 58A2 45D3 D8B2 5C15 F6CE  3FFF 7187 6B68 AD48 2D32\n\nTo claim this, I am signing this object:\n\n`+"`"+``+"`"+``+"`"+`json\n{\n    \"body\": {\n        \"client\": {\n            \"name\": \"keybase.io node.js client\",\n            \"version\": \"0.7.3\"\n        },\n        \"key\": {\n            \"fingerprint\": \"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\n            \"host\": \"keybase.io\",\n            \"key_id\": \"71876B68AD482D32\",\n            \"uid\": \"41b1f75fb55046d370608425a3208100\",\n            \"username\": \"oconnor663\"\n        },\n        \"merkle_root\": {\n            \"ctime\": 1419205037,\n            \"hash\": \"d679cba1c6e433456b4cc51817a0d3a1c095aae41921e5511c29817dac7ebcaca6c6175d43f04ba81397b8eb38b8b53b2dc23564b0e79b7caa562053facc56e4\",\n            \"seqno\": 114149\n        },\n        \"service\": {\n            \"name\": \"github\",\n            \"username\": \"oconnor663\"\n        },\n        \"type\": \"web_service_binding\",\n        \"version\": 1\n    },\n    \"ctime\": 1419205131,\n    \"expire_in\": 157680000,\n    \"prev\": \"b421b35b23fc5da8a69fab3978ff34689fcaaf0048fc8af822aa32b5121c8193\",\n    \"seqno\": 2,\n    \"tag\": \"signature\"\n}\n`+"`"+``+"`"+``+"`"+`\n\nwith the PGP key whose fingerprint is\n[58A2 45D3 D8B2 5C15 F6CE  3FFF 7187 6B68 AD48 2D32](https://keybase.io/oconnor663)\n(captured above as `+"`"+`body.key.fingerprint`+"`"+`), yielding the PGP signature:\n\n`+"`"+``+"`"+``+"`"+`\n-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n\nowF9kltoFFcYx3et1VWrWBQJsaCdFq3tusy5zhn1QYOUVh9MglQq1vWcM+dkJ5eZ\ndWaTmNqIWmlTrDRtpaLSghdaIqYS6614wbsWry/6oEVoi6AvkgRFVHA9s1iqL52n\nme/8vv/3/893uke/ksqkl3/ZVOj9YDpMX7grUgs3LR610hKh12HNWGnJZl8FpeQt\n4C3KmmE1qQ7BY5Xzw8lB6KlcYzz5OZO12lQU+2FgKDvn5JDVmU3wpFn7QYOKipGf\naFmEcYiJhzwmIJGAaCoV0lo7gDlUUMY9zKCHoJEshHHppalWRTPve6Za4WsomzPX\n8HMrfGvlAAMBtEO0IMTG1EOOTW2GIeEI2gzYdgLGKnoeKZRhEIQRpRXHLSpqalb5\nKAwrsWXJTyCAgQttYiPHeOJxwbR51HGl4EBShRHChAosJQEMONz2kKnbLuFcJY1A\nEQKAhK459Lh0lJBcciopcIiHkbax4Awg1xFMCcQEEwQJ6EmICMXCVo4rHMk5ocYC\n0tyMMTNNiFgtD0JjzrjDbmfyHbX5Ur2wrga/VGgV/5e31FFMiu1K5A2yK+nPCz/w\nzMZeXCkw6Mv/AiCQtdSKoh+pvJ8QxKHMNk/WKkaqzWgKDIFARECkJfE449TVXJiY\nTGuEKXO1CaVtGzMtGdcMQm42JAiAQDLgov8SQuOTNxjJ2G8IeKk1UlbnV+k3hqbS\nmdSwV4ckdzY1csTYfy9yqjeTOji8+KB672c9PasmiqPhVFS+N38n+e2cqvqkdikt\nrC4e7e//R7w57/j8R69lahr3/Tx4/mI0ru/bqku98Cy41f/37acXa8+sv5a7scb9\n/tjA/m/WvfPk5M3amikb3qvecvvWYOav4VtP73n9u3dnLdzx9YGldX0f7uiCi7qf\n/vrWD79kV/fdf7v2fPHwPae+ceKPB2bWb/+U7qsfv/Exm3R/wF3c/WTtwZ5j8wZ3\nnZnQdm7KF3WNh6ovjKH4Sv+GzPvTRi+b+cd6sK5c9/Dy6SHtH338aFthSbpqQupU\nze/l9s/317XUXyuvmj30z40DW6ubx95Zc6hq77jC7gXl+PD18XPOnuw6Me2nI11X\nN496Bg==\n=diQc\n-----END PGP MESSAGE-----\n\n`+"`"+``+"`"+``+"`"+`\n\nAnd finally, I am proving ownership of the github account by posting this as a gist.\n\n### My publicly-auditable identity:\n\nhttps://keybase.io/oconnor663\n\n### From the command line:\n\nConsider the [keybase command line program](https://keybase.io/docs/command_line).\n\n`+"`"+``+"`"+``+"`"+`bash\n# look me up\nkeybase id oconnor663\n\n# encrypt a message to me\nkeybase encrypt oconnor663 -m 'a secret message...'\n\n# ...and more...\n`+"`"+``+"`"+``+"`"+`\n",
            "check_data_json": "{\"name\":\"github\",\"username\":\"oconnor663\"}",
            "remote_id": "df527ad8438a424efcec",
            "api_url": "https://gist.githubusercontent.com/oconnor663/df527ad8438a424efcec/raw/0c24241710275124f07db83ab5b6f97fc6952165/keybase.md",
            "human_url": "https://gist.github.com/df527ad8438a424efcec",
            "proof_state": 1,
            "proof_status": 1,
            "retry_count": 142,
            "hard_fail_count": 0,
            "last_check": 1434467728,
            "last_success": 1434467728,
            "version": 3,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 3,
            "payload_hash": "44a567581a31b596f31745c4633feced0806f427bac69e720aecfda02279b9bb",
            "sig_id": "a5039b256eb381b395e2fe9018e931c7dbe052dc1f1cf213b9e8e930ef0feabf0f",
            "sig_id_short": "pQObJW6zgbOV4v6QGOkxx9vgUtwfHPITuejp",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n\nowF9kltoHUUcxk9sm9ZqSIsXfIgStwUpTU7mPrMnDdUawWiIgkIxUk/mtidrkt2T\n3ZOYGKMFi02EqiCaFvom7dFqHqrRqOhDQdBYifqgra3QSERaxQuitUorzh4qti/O\n01x+/2++b/7zXMOK3Jq64d0D/bN3tqK6o2dU7v6Z3hcnPBWbca8w4enB0EaVbBbJ\nIesVvAE7rmRq82HcHMXG5h9Omy8yLd6oTdIwjhwF8jyPvcmWDM+KgzAq2aSchJmW\nR4VEhBpshEJUQxowbXEQBBwKzhQT0hCBDEZOsj9OK5fd6tU0i6FxuzV+GxO3dTq+\ns8aP1A4IVDDgNFCUAsIM5oABQRCVGAEBAcjA1CYXI8U6jqI4YazmeMgmA4O2mMRx\nLbauhBkECfQRoJAR50mm/a5MCetTbQgEkkAmsG8UVr6EPsYKSYWg70NEEUOMBZZw\nHViojUvGCIIKcQmUIpJBjQx3qSH2XQjOKIQ+0BhA3z2L5BAjoxhWGLoCLgjnwAgI\nEcbEhUjtcBQ7c84dRZPZOhkNtb2kXYk1Jqz8X97KeDnbfMSqokMOZfVFFUbGdezS\nlkKHXv4WiIMWz46Vw8QWw4ygnAngRotXTuyo08QCWWiYUEQJqBkjmAgsOeGQSg18\nRizyYaCQokgT6weSaY4NBkQqHwIR/JcQO5+y5CTTsBTJykhivcnpuqaVubo1ufpV\nV2R/Nrf2ynX/fuSNJ1b/vaF1+be0Mdf1Tld0w9lVi3ubCu/KRXnhlpOf98z1jd13\nCOyZOn3q+lj90Lc4N/P2L4WP83eLXYdv36zr35w+cu1rbe9tuKP86plkHiwffKvh\ncPT6tvOn/Zd3/rV759LWA6a1c/aJ6vvb0dahPY+dO9n42UfV/cvtv3+3aWFkxYmk\ntGVmqffPVx7atfR0Kej+dKHvp59fWMnEV6ee2etddWx9YcfqWx+/50hPd1fV7mi/\naazj7Lpvp+YXrut444NfG/ddaKebJo43w3T25gce7H6pybtmcbrx+z+eXf/j5vPD\nc0+Ob+FTg4/e9c3XN66tHudH91ePbU8+vHqk3FD/VFvH/L2fHKx/vld/ea4t/uIf\n=i2ny\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.7.3\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876B68AD482D32\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"merkle_root\":{\"ctime\":1419205164,\"hash\":\"b8e95cd410a416839db3b9a1933b2ab219912526266fe47cfe1cdd486421b27a0bb4a61c2d776b1394827651190c30195f6a7132db63b317a0784770d8112334\",\"seqno\":114152},\"service\":{\"name\":\"reddit\",\"username\":\"oconnor663\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1419205270,\"expire_in\":157680000,\"prev\":\"382e1d68b4b81c6643483a74715ac0964e291fb2b52c4e9fa6c73d304ab9108f\",\"seqno\":3,\"tag\":\"signature\"}",
            "sig_type": 2,
            "ctime": 1419205270,
            "etime": 1576885270,
            "rtime": null,
            "sig_status": 0,
            "prev": "382e1d68b4b81c6643483a74715ac0964e291fb2b52c4e9fa6c73d304ab9108f",
            "proof_id": "7374183938ae0741e08d3c10",
            "proof_type": 4,
            "proof_text_check": "\n\nowF9kltoHUUcxk9sm9ZqSIsXfIgStwUpTU7mPrMnDdUawWiIgkIxUk/mtidrkt2T\n3ZOYGKMFi02EqiCaFvom7dFqHqrRqOhDQdBYifqgra3QSERaxQuitUorzh4qti/O\n01x+/2++b/7zXMOK3Jq64d0D/bN3tqK6o2dU7v6Z3hcnPBWbca8w4enB0EaVbBbJ\nIesVvAE7rmRq82HcHMXG5h9Omy8yLd6oTdIwjhwF8jyPvcmWDM+KgzAq2aSchJmW\nR4VEhBpshEJUQxowbXEQBBwKzhQT0hCBDEZOsj9OK5fd6tU0i6FxuzV+GxO3dTq+\ns8aP1A4IVDDgNFCUAsIM5oABQRCVGAEBAcjA1CYXI8U6jqI4YazmeMgmA4O2mMRx\nLbauhBkECfQRoJAR50mm/a5MCetTbQgEkkAmsG8UVr6EPsYKSYWg70NEEUOMBZZw\nHViojUvGCIIKcQmUIpJBjQx3qSH2XQjOKIQ+0BhA3z2L5BAjoxhWGLoCLgjnwAgI\nEcbEhUjtcBQ7c84dRZPZOhkNtb2kXYk1Jqz8X97KeDnbfMSqokMOZfVFFUbGdezS\nlkKHXv4WiIMWz46Vw8QWw4ygnAngRotXTuyo08QCWWiYUEQJqBkjmAgsOeGQSg18\nRizyYaCQokgT6weSaY4NBkQqHwIR/JcQO5+y5CTTsBTJykhivcnpuqaVubo1ufpV\nV2R/Nrf2ynX/fuSNJ1b/vaF1+be0Mdf1Tld0w9lVi3ubCu/KRXnhlpOf98z1jd13\nCOyZOn3q+lj90Lc4N/P2L4WP83eLXYdv36zr35w+cu1rbe9tuKP86plkHiwffKvh\ncPT6tvOn/Zd3/rV759LWA6a1c/aJ6vvb0dahPY+dO9n42UfV/cvtv3+3aWFkxYmk\ntGVmqffPVx7atfR0Kej+dKHvp59fWMnEV6ee2etddWx9YcfqWx+/50hPd1fV7mi/\naazj7Lpvp+YXrut444NfG/ddaKebJo43w3T25gce7H6pybtmcbrx+z+eXf/j5vPD\nc0+Ob+FTg4/e9c3XN66tHudH91ePbU8+vHqk3FD/VFvH/L2fHKx/vld/ea4t/uIf\n",
            "proof_text_full": "### Keybase proof\n\nI hereby claim:\n\n  * I am [oconnor663](https://www.reddit.com/user/oconnor663) on reddit.\n  * I am [oconnor663](https://keybase.io/oconnor663) on keybase.\n  * I have a public key whose fingerprint is 58A2 45D3 D8B2 5C15 F6CE  3FFF 7187 6B68 AD48 2D32\n\nTo claim this, I am signing this object:\n\n\n    {\n        \"body\": {\n            \"client\": {\n                \"name\": \"keybase.io node.js client\",\n                \"version\": \"0.7.3\"\n            },\n            \"key\": {\n                \"fingerprint\": \"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\n                \"host\": \"keybase.io\",\n                \"key_id\": \"71876B68AD482D32\",\n                \"uid\": \"41b1f75fb55046d370608425a3208100\",\n                \"username\": \"oconnor663\"\n            },\n            \"merkle_root\": {\n                \"ctime\": 1419205164,\n                \"hash\": \"b8e95cd410a416839db3b9a1933b2ab219912526266fe47cfe1cdd486421b27a0bb4a61c2d776b1394827651190c30195f6a7132db63b317a0784770d8112334\",\n                \"seqno\": 114152\n            },\n            \"service\": {\n                \"name\": \"reddit\",\n                \"username\": \"oconnor663\"\n            },\n            \"type\": \"web_service_binding\",\n            \"version\": 1\n        },\n        \"ctime\": 1419205270,\n        \"expire_in\": 157680000,\n        \"prev\": \"382e1d68b4b81c6643483a74715ac0964e291fb2b52c4e9fa6c73d304ab9108f\",\n        \"seqno\": 3,\n        \"tag\": \"signature\"\n    }\n\nwith the PGP key referenced above, yielding the PGP signature:\n\n\n    -----BEGIN PGP MESSAGE-----\n    Version: GnuPG v2\n    \n    owF9kltoHUUcxk9sm9ZqSIsXfIgStwUpTU7mPrMnDdUawWiIgkIxUk/mtidrkt2T\n    3ZOYGKMFi02EqiCaFvom7dFqHqrRqOhDQdBYifqgra3QSERaxQuitUorzh4qti/O\n    01x+/2++b/7zXMOK3Jq64d0D/bN3tqK6o2dU7v6Z3hcnPBWbca8w4enB0EaVbBbJ\n    IesVvAE7rmRq82HcHMXG5h9Omy8yLd6oTdIwjhwF8jyPvcmWDM+KgzAq2aSchJmW\n    R4VEhBpshEJUQxowbXEQBBwKzhQT0hCBDEZOsj9OK5fd6tU0i6FxuzV+GxO3dTq+\n    s8aP1A4IVDDgNFCUAsIM5oABQRCVGAEBAcjA1CYXI8U6jqI4YazmeMgmA4O2mMRx\n    LbauhBkECfQRoJAR50mm/a5MCetTbQgEkkAmsG8UVr6EPsYKSYWg70NEEUOMBZZw\n    HViojUvGCIIKcQmUIpJBjQx3qSH2XQjOKIQ+0BhA3z2L5BAjoxhWGLoCLgjnwAgI\n    EcbEhUjtcBQ7c84dRZPZOhkNtb2kXYk1Jqz8X97KeDnbfMSqokMOZfVFFUbGdezS\n    lkKHXv4WiIMWz46Vw8QWw4ygnAngRotXTuyo08QCWWiYUEQJqBkjmAgsOeGQSg18\n    RizyYaCQokgT6weSaY4NBkQqHwIR/JcQO5+y5CTTsBTJykhivcnpuqaVubo1ufpV\n    V2R/Nrf2ynX/fuSNJ1b/vaF1+be0Mdf1Tld0w9lVi3ubCu/KRXnhlpOf98z1jd13\n    COyZOn3q+lj90Lc4N/P2L4WP83eLXYdv36zr35w+cu1rbe9tuKP86plkHiwffKvh\n    cPT6tvOn/Zd3/rV759LWA6a1c/aJ6vvb0dahPY+dO9n42UfV/cvtv3+3aWFkxYmk\n    tGVmqffPVx7atfR0Kej+dKHvp59fWMnEV6ee2etddWx9YcfqWx+/50hPd1fV7mi/\n    aazj7Lpvp+YXrut444NfG/ddaKebJo43w3T25gce7H6pybtmcbrx+z+eXf/j5vPD\n    c0+Ob+FTg4/e9c3XN66tHudH91ePbU8+vHqk3FD/VFvH/L2fHKx/vld/ea4t/uIf\n    =i2ny\n    -----END PGP MESSAGE-----\n\nFinally, I am proving my reddit account by posting it in [KeybaseProofs](https://www.reddit.com/r/KeybaseProofs).\n",
            "check_data_json": "{\"name\":\"reddit\",\"username\":\"oconnor663\"}",
            "remote_id": "t3_2q0kk1",
            "api_url": "https://www.reddit.com/r/KeybaseProofs/comments/2q0kk1/my_keybase_proof_redditoconnor663/.json",
            "human_url": "https://www.reddit.com/r/KeybaseProofs/comments/2q0kk1/my_keybase_proof_redditoconnor663/",
            "proof_state": 1,
            "proof_status": 1,
            "retry_count": 4,
            "hard_fail_count": 0,
            "last_check": 1434485079,
            "last_success": 1434485079,
            "version": 9,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 4,
            "payload_hash": "00279b0704b01ac6171beef1a5718a8de52d5f0fd0ab63dc947dd6f111bb673f",
            "sig_id": "89e70d05df9c659d1ad8b0e2ead5790cc500d29345979af2e5807edefab5666f0f",
            "sig_id_short": "iecNBd-cZZ0a2LDi6tV5DMUA0pNFl5ry5YB-",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n\nowF9kk1sFVUYhu+9SlsKJARRiGJoJkIa017P/zmDC6J2URMEF9UoKNfzN+3YMnOZ\nub20aa/SVetCRRaUSMPKGCJlgQqYaGpK1DRFY4yp1aj8JBgEqQJBgwsTz1wxwsbJ\nLM6ceb73vO/3nT3L7sg15XeO9vYc6WxH+VMXVa5rfOuVIU/FZtDbMOTpvtBGlWwV\nyR3W2+D12kElU1sM45YoNrb4Ytpyk2nzqjZJwzhyFCjyIvZqbRmeFQdh1G2TchJm\nWh4VEhFqsBEKUQ1pwLTFQRBwKDhTTEhDBDIYOcmeOK3cdqpX1yyFxu3W+UeZeKTD\n8R11vr/+g0AFA04DRSkgzGAOGBAEUYkREBCADExtcjNSrOMoihPG6o532KS3z5aS\nOK7H1pUwgyCBPgIUceg8ybTHlVkCiK+sltkJDHJiqDBCIy10AJUvlQCWc0JsYAJo\nKEQGYUZ96rtXQGYA0pJqQYEwhHLgK2PdkktnllJOhAOkJpZQoaRUriAwWAPMfCNc\nBfZdiNTujGJnzrmjuJZ9J9VQ21vGpeMwyjr3f4krg+Vsc5dVpXTyH4WSCiPjZnbr\nUKFDb+8GZqDNswPlMLGlMCMoZwK4p80rJ7aaDYJIyjgVUGKoqM8C7PpENWEYB1Zb\nAwRgAUFcSc18yxGQVgdGAoS4r3yl/stInE/Z7STTsDuSlf7EerVX8mvuzOWbcg2L\nCtmtzTUvXv7vVd5Wbsq9e7JhazT+wIGZqOdc4zNdy/5sPDDxzfaBLYdHPi8Uvrs4\nvef15hfkj61zuSvrTo7WVnVVv7w8NNCyce10n/71s2eX/3D/iYYnL7//06FhOLl/\nYWV/x0O/zc9utBfODy+91vpS68d7fz/2xFV4+vRrl8pjbMnKa4eO8qOjm1bPXT0y\nvmK2c8Ru3l+899hSVug4c8+3xS/u27Tm/NjM3ZfOPpVfe2JbemPFxKmpwvW3v1qY\n2vfgDTC1+K/Hdg3Mr9+3W8y9sb755erh7Wc/fPiD4teLDrafWfLR7PePH9zCnnt1\n83vz7+xddVfjW+uuT/5SrM0Nf7L7XLWl1E7109Njx9+80PfpBH7+55k/VneOHPcW\n/gY=\n=2nAo\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.7.3\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876B68AD482D32\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"merkle_root\":{\"ctime\":1419205271,\"hash\":\"e4049beca46d36174d58d8c2c8cf1b9ab80e7744efdf1d512d2365959959816d02ca5c8508d45709bde08d7a70655748d02ac4e458baab16dfd3c0369d88d439\",\"seqno\":114153},\"service\":{\"name\":\"coinbase\",\"username\":\"oconnor663\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1419205360,\"expire_in\":157680000,\"prev\":\"44a567581a31b596f31745c4633feced0806f427bac69e720aecfda02279b9bb\",\"seqno\":4,\"tag\":\"signature\"}",
            "sig_type": 2,
            "ctime": 1419205360,
            "etime": 1576885360,
            "rtime": null,
            "sig_status": 0,
            "prev": "44a567581a31b596f31745c4633feced0806f427bac69e720aecfda02279b9bb",
            "proof_id": "de92300787097c3f7ae36910",
            "proof_type": 5,
            "proof_text_check": "\n\nowF9kk1sFVUYhu+9SlsKJARRiGJoJkIa017P/zmDC6J2URMEF9UoKNfzN+3YMnOZ\nub20aa/SVetCRRaUSMPKGCJlgQqYaGpK1DRFY4yp1aj8JBgEqQJBgwsTz1wxwsbJ\nLM6ceb73vO/3nT3L7sg15XeO9vYc6WxH+VMXVa5rfOuVIU/FZtDbMOTpvtBGlWwV\nyR3W2+D12kElU1sM45YoNrb4Ytpyk2nzqjZJwzhyFCjyIvZqbRmeFQdh1G2TchJm\nWh4VEhFqsBEKUQ1pwLTFQRBwKDhTTEhDBDIYOcmeOK3cdqpX1yyFxu3W+UeZeKTD\n8R11vr/+g0AFA04DRSkgzGAOGBAEUYkREBCADExtcjNSrOMoihPG6o532KS3z5aS\nOK7H1pUwgyCBPgIUceg8ybTHlVkCiK+sltkJDHJiqDBCIy10AJUvlQCWc0JsYAJo\nKEQGYUZ96rtXQGYA0pJqQYEwhHLgK2PdkktnllJOhAOkJpZQoaRUriAwWAPMfCNc\nBfZdiNTujGJnzrmjuJZ9J9VQ21vGpeMwyjr3f4krg+Vsc5dVpXTyH4WSCiPjZnbr\nUKFDb+8GZqDNswPlMLGlMCMoZwK4p80rJ7aaDYJIyjgVUGKoqM8C7PpENWEYB1Zb\nAwRgAUFcSc18yxGQVgdGAoS4r3yl/stInE/Z7STTsDuSlf7EerVX8mvuzOWbcg2L\nCtmtzTUvXv7vVd5Wbsq9e7JhazT+wIGZqOdc4zNdy/5sPDDxzfaBLYdHPi8Uvrs4\nvef15hfkj61zuSvrTo7WVnVVv7w8NNCyce10n/71s2eX/3D/iYYnL7//06FhOLl/\nYWV/x0O/zc9utBfODy+91vpS68d7fz/2xFV4+vRrl8pjbMnKa4eO8qOjm1bPXT0y\nvmK2c8Ru3l+899hSVug4c8+3xS/u27Tm/NjM3ZfOPpVfe2JbemPFxKmpwvW3v1qY\n2vfgDTC1+K/Hdg3Mr9+3W8y9sb755erh7Wc/fPiD4teLDrafWfLR7PePH9zCnnt1\n83vz7+xddVfjW+uuT/5SrM0Nf7L7XLWl1E7109Njx9+80PfpBH7+55k/VneOHPcW\n/gY=\n",
            "proof_text_full": "### Keybase proof\n\nI hereby claim:\n\n  * I am oconnor663 on coinbase.\n  * I am oconnor663 on keybase.\n  * I have a public key whose fingerprint is 58A2 45D3 D8B2 5C15 F6CE  3FFF 7187 6B68 AD48 2D32\n\nTo claim this, I am signing this object:\n\n`+"`"+``+"`"+``+"`"+`json\n{\n    \"body\": {\n        \"client\": {\n            \"name\": \"keybase.io node.js client\",\n            \"version\": \"0.7.3\"\n        },\n        \"key\": {\n            \"fingerprint\": \"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\n            \"host\": \"keybase.io\",\n            \"key_id\": \"71876B68AD482D32\",\n            \"uid\": \"41b1f75fb55046d370608425a3208100\",\n            \"username\": \"oconnor663\"\n        },\n        \"merkle_root\": {\n            \"ctime\": 1419205271,\n            \"hash\": \"e4049beca46d36174d58d8c2c8cf1b9ab80e7744efdf1d512d2365959959816d02ca5c8508d45709bde08d7a70655748d02ac4e458baab16dfd3c0369d88d439\",\n            \"seqno\": 114153\n        },\n        \"service\": {\n            \"name\": \"coinbase\",\n            \"username\": \"oconnor663\"\n        },\n        \"type\": \"web_service_binding\",\n        \"version\": 1\n    },\n    \"ctime\": 1419205360,\n    \"expire_in\": 157680000,\n    \"prev\": \"44a567581a31b596f31745c4633feced0806f427bac69e720aecfda02279b9bb\",\n    \"seqno\": 4,\n    \"tag\": \"signature\"\n}\n`+"`"+``+"`"+``+"`"+`\n\nwith the PGP key whose fingerprint is 58A2 45D3 D8B2 5C15 F6CE  3FFF 7187 6B68 AD48 2D32 (captured above as `+"`"+`body.key.fingerprint`+"`"+`), yielding the PGP signature:\n\n`+"`"+``+"`"+``+"`"+`\n-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n\nowF9kk1sFVUYhu+9SlsKJARRiGJoJkIa017P/zmDC6J2URMEF9UoKNfzN+3YMnOZ\nub20aa/SVetCRRaUSMPKGCJlgQqYaGpK1DRFY4yp1aj8JBgEqQJBgwsTz1wxwsbJ\nLM6ceb73vO/3nT3L7sg15XeO9vYc6WxH+VMXVa5rfOuVIU/FZtDbMOTpvtBGlWwV\nyR3W2+D12kElU1sM45YoNrb4Ytpyk2nzqjZJwzhyFCjyIvZqbRmeFQdh1G2TchJm\nWh4VEhFqsBEKUQ1pwLTFQRBwKDhTTEhDBDIYOcmeOK3cdqpX1yyFxu3W+UeZeKTD\n8R11vr/+g0AFA04DRSkgzGAOGBAEUYkREBCADExtcjNSrOMoihPG6o532KS3z5aS\nOK7H1pUwgyCBPgIUceg8ybTHlVkCiK+sltkJDHJiqDBCIy10AJUvlQCWc0JsYAJo\nKEQGYUZ96rtXQGYA0pJqQYEwhHLgK2PdkktnllJOhAOkJpZQoaRUriAwWAPMfCNc\nBfZdiNTujGJnzrmjuJZ9J9VQ21vGpeMwyjr3f4krg+Vsc5dVpXTyH4WSCiPjZnbr\nUKFDb+8GZqDNswPlMLGlMCMoZwK4p80rJ7aaDYJIyjgVUGKoqM8C7PpENWEYB1Zb\nAwRgAUFcSc18yxGQVgdGAoS4r3yl/stInE/Z7STTsDuSlf7EerVX8mvuzOWbcg2L\nCtmtzTUvXv7vVd5Wbsq9e7JhazT+wIGZqOdc4zNdy/5sPDDxzfaBLYdHPi8Uvrs4\nvef15hfkj61zuSvrTo7WVnVVv7w8NNCyce10n/71s2eX/3D/iYYnL7//06FhOLl/\nYWV/x0O/zc9utBfODy+91vpS68d7fz/2xFV4+vRrl8pjbMnKa4eO8qOjm1bPXT0y\nvmK2c8Ru3l+899hSVug4c8+3xS/u27Tm/NjM3ZfOPpVfe2JbemPFxKmpwvW3v1qY\n2vfgDTC1+K/Hdg3Mr9+3W8y9sb755erh7Wc/fPiD4teLDrafWfLR7PePH9zCnnt1\n83vz7+xddVfjW+uuT/5SrM0Nf7L7XLWl1E7109Njx9+80PfpBH7+55k/VneOHPcW\n/gY=\n=2nAo\n-----END PGP MESSAGE-----\n\n`+"`"+``+"`"+``+"`"+`\n\nAnd finally, I am proving ownership of the coinbase account by posting this on coinbase.com\n\n### My publicly-auditable identity:\n\nhttps://keybase.io/oconnor663\n\n`+"`"+``+"`"+``+"`"+`\n",
            "check_data_json": "{\"name\":\"coinbase\",\"username\":\"oconnor663\"}",
            "remote_id": "oconnor663",
            "api_url": "https://coinbase.com/oconnor663/public-key",
            "human_url": "https://coinbase.com/oconnor663/public-key",
            "proof_state": 1,
            "proof_status": 1,
            "retry_count": 192,
            "hard_fail_count": 0,
            "last_check": 1434481477,
            "last_success": 1434481477,
            "version": 3,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 5,
            "payload_hash": "d64c8d89811ef70c83ab58cf6350d4e847f1750cff3c468d7e8556c65073e09a",
            "sig_id": "8b4bc6a736a2a651763300384d54e72daa0ee3f9464ae98fd2df4f06f93d59f60f",
            "sig_id_short": "i0vGpzaiplF2MwA4TVTnLaoO4_lGSumP0t9P",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n\nowF9km2IVFUYx2e23ZwEJRctBYPda2DlOJxz7nm5o/lBd3uxkMi2/ZDYdF5nb7ve\nO957d7dRBmwNRC3ayFIye1nKEsoUMTIUjcoWAwk0tk39YJCUFpZIBqtu5w4b6ZfO\np/Pye/7n+T/PMzjllkwuu3pDd9fHD89H2W9/FZmOrStqax0RqqqzYK0je3wdJOku\n4Ku0s8Dp1lXBY13ww5YgVLrwbNwyweSdPh3FfhhYChRYwXVq+RRPg40flHVUifxU\nyyEeR5goV3kCEQmJoVK7xhgGPUYF9bjCHlIuspJdYZzc9KtT1yz5yt7W+SXUW9xu\n+fY631t/wFBAw4gRhABMlcsABR5GhLsIeBCAFIx1NGEplGEQhBGl9YxX6ai7R5ei\nMKzblomfQhDDIgIEQ2Jz4nGXDdPY0xoY7hFBsGcwABxZI0QVFRESY44MAspwyj1F\nDVSEYcklUYBpVwpPmSIVwGhoFJWQy6Isaq05QBwSz37kcokos89WESlsMFZS2HjK\nCGMQedyaiPXqILTJ2ewIqaXnqM+X+oZ2dXHZbY3q/vj/PCfVSnrZr8Xu0oRGSfiB\nsl27sa3QojfXgwCYd/RzFT/SJT8lCKMesCvvVCLdl04CQKwoAANYAOuSQgaF1gZy\nYrtnK6MJUsQAowAX1FWyiJlKywWhEJS55j+XtvIJL1vJ2C8HPOmNtFPbmJ3dmMnm\nMrc2NaRzm5l82+3/DnPSlstsPvyGOXpg194r5dnN2zrfGg7K58Ym/Z1cCzq+mffj\nH51fPrX7euflphkHK18cGbq/9Obcx7KtT4+NrF/y1V1t4wfId4s+jB4d/awpM6V5\nXUNruP3M4uL++Pqex38aOPLzlgv4oca2X4ZWztq7+fjyma8fL7RefHfr9E/m9b49\nkKBTi1ZmT48NLl1zbOelXYeH57y4b+bUu0/uG3llzed//p7cM+3c90eP+cuevPf9\n4Q9O/fbIlcYTGShySX9nw9yhQs+lOwfveH75xa9HWs4/8d4zHevzB18YLu+c9FHu\ntWrrnJ6XPt30wH3b/2qunR24Ot5iTrw6Om1F+zuj29pPL1s469CDS3dUR9tGftgS\nnnx5HHf8Aw==\n=B0oP\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.7.3\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876B68AD482D32\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"merkle_root\":{\"ctime\":1419205415,\"hash\":\"e48ee0fa85b548f400a26ce5d9d5bc44a2f20dfa6a8d6f1d574cac5d07e3cb8df96b0fe1fd6c1ac9c9eeea02a1585413ac267e1fe5d2d4f44dcbac567577128a\",\"seqno\":114155},\"service\":{\"name\":\"hackernews\",\"username\":\"oconnor663\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1419205501,\"expire_in\":157680000,\"prev\":\"00279b0704b01ac6171beef1a5718a8de52d5f0fd0ab63dc947dd6f111bb673f\",\"seqno\":5,\"tag\":\"signature\"}",
            "sig_type": 2,
            "ctime": 1419205501,
            "etime": 1576885501,
            "rtime": null,
            "sig_status": 0,
            "prev": "00279b0704b01ac6171beef1a5718a8de52d5f0fd0ab63dc947dd6f111bb673f",
            "proof_id": "ff25378bd64cc2b0d2942f10",
            "proof_type": 6,
            "proof_text_check": "i0vGpzaiplF2MwA4TVTnLaoO4_lGSumP0t9PBvk9WfY",
            "proof_text_full": "[ my public key: https://keybase.io/oconnor663; my proof: https://keybase.io/oconnor663/sigs/i0vGpzaiplF2MwA4TVTnLaoO4_lGSumP0t9PBvk9WfY ]\n",
            "check_data_json": "{\"name\":\"hackernews\",\"username\":\"oconnor663\"}",
            "remote_id": "oconnor663",
            "api_url": "https://hacker-news.firebaseio.com/v0/user/oconnor663/about.json",
            "human_url": "https://news.ycombinator.com/user?id=oconnor663",
            "proof_state": 1,
            "proof_status": 1,
            "retry_count": 210,
            "hard_fail_count": 0,
            "last_check": 1434471730,
            "last_success": 1434471730,
            "version": 8,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 6,
            "payload_hash": "37ad41cf064989dfd849a0af178aa066dd396b3452433a9883a69e88e35e75b8",
            "sig_id": "5941324c9d6c5c830c832f44494cbc73af0cfaff6e3c9e69ae88bbda45e72c3e0f",
            "sig_id_short": "WUEyTJ1sXIMMgy9ESUy8c68M-v9uPJ5proi7",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v1.1.6\nComment: https://keybase.io/crypto\n\nyMHMAnicO8LLzMDFWNienbHWQ9eI8fSBXaUMIdNj86qVkvJTKpWsqpWyU8FUWmZe\nempRQVFmXomSlZKpRaKRiWmKcYpFkpFpsqFpmllyqnFaWpq5oYW5WZKZRWKKiYVR\nirGRko5SRn4xSAfQmKTE4lS9zHygGJATn5kCFMWivhQsYWKYZJhmbpqWZGpqYGKW\nYmxuYGZgYWJkmmhsZGBhaGAAUlicWpSXmJsKVJ2fnJ+Xl19kZmasVKujVFJZABIs\nT02KByopy0xOjU/KzEsBegCoqyy1qDgzP0/JyhCoMrkkE6Tf0MTQ0sjA1NLSQkcp\ntaIgsyg1PhOkwtTczMIACHSUCopSy4BGppiZJFukWFhaGBqmppkbJFsYJyaZWiSn\nmRmbGqSYpFqYmKcZmpsaJKelGSebmFmkmKdamJqaJZuZGpgbpxpYJgLtL04tzMtX\nsjIDOjMxHWhkcWZ6XmJJaVGqUu2hAzEsDIxcDGysTKAoYODiFIBFjLMKB8OJ2qe/\nZr3gLy6SvvjN507UXm6uyy+2Jr2S2Jmconh4rcmchIcyYbsTGpuXb+XbvvHYnyK+\n6Qr1z9Vku4vzZgazFKw7pivUk3NdcXWEterdmca6ynKPhS78UWX5l2veJ7FycfPu\nSJfvTz3uTQzgft/3eM+CH29FHyfOy57Z1hf5/c1qt8XH93Umxs0/tca7+0n5oj/t\ni953tFf9d/bvu3HXTeGx5vIL6zdL+3qzfSu4nx0p9tZG2MhAte2tfMaFD6sZOP+r\nRzwPDotbUiO+3PQGSzGf99aK4Iebbk4XfF4WK3TycN/VWaX+W2LSCgzXbXlwiOu0\nFM+WnNRVXZHdM2cHnpxq+eXelcO9MplbLTUA7nMASQ==\n=dQNg\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1419205998,\"expire_in\":157680000,\"prev\":\"d64c8d89811ef70c83ab58cf6350d4e847f1750cff3c468d7e8556c65073e09a\",\"seqno\":6,\"tag\":\"signature\"}",
            "sig_type": 1,
            "ctime": 1419205998,
            "etime": 1576885998,
            "rtime": null,
            "sig_status": 0,
            "prev": "d64c8d89811ef70c83ab58cf6350d4e847f1750cff3c468d7e8556c65073e09a",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 7,
            "payload_hash": "2d8dfe977e6d359a03430c5cf22993d16438b1e10546b1b247055abc9cad3edf",
            "sig_id": "bd1445bcc75e2d6ed5031c245306863fbaddf8673e674903fe5ac7c1414136a90f",
            "sig_id_short": "vRRFvMdeLW7VAxwkUwaGP7rd-Gc-Z0kD_lrH",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n\nowFdkl1oHFUUxzdNTFI1WKEIQcU4FTF1E++dO/djAlYiGwzBwLakpU0iy/1MJh8z\n25lNNMZVMMYiRoiGFlpt6IfUNqUP2qJQqAj1QVofLKgPpU0fpAShopJIo1XqnSWK\ndV5mOPPjd//3nDPbUJ2pr9q9Z3jwVGeLW3XxR5Hp2b94e9IRkZpw2iYdORLosJR+\nhXxUO23OsJ4QPNGtQdQURkq3DiVNa0zWGddxEkShpUArbUVOOevIeKJYiuRYHOtQ\nVoxcqVgniYVgV0deIt0ebMW93dtCd+jFwZ5Sx3bk+uM+kjkPdVhnaaKYHiuCkoyC\nMFXaBKnHBOGAjotxkMZzMOOuhxVSTLhYQmyI1MgYQyGjRBDGlcdchVxrHIyS0h0X\ncSrOQqBstcI/S1h7zvK5Cj9W+eFBAQ3FRmAMPKIQBQQwz8UcuYBBAFIw0fFalyIZ\nhWEUE1JpwqiOh0d0IY6iSidlKUgh6EHfauzLZuLJYNo2ZrRA1FMeMEYyV1BglJBA\nu8rFyBYo1QBITJXxPGIMI4hSSY3CwIeSA0WQwD4hRkHGMZGA23zWRSQiHqXGYE8Y\nbL22IJFluCKaad9XkGBNtZP9osZJ9O4wsvEgxswv/zuB/03yP9OG6ZzvuJOL/Kyj\nXyoGsS4EKYEpYcA+WacY63FrQ9ROBEoDiOczXxnFPJ8DbiC1oQAhSiGfCORh10OI\n+4whTnzNmEY2JxbMnr+Wk9qEfMAqk2Ag5KWxWDvlt6oeqslU1Wdq71qXrnPm7vUb\n/tnxh6vrbtWWy6XtP8MDr7xTx/uvwtX7Z3a0Ty18dG3nwqGVB/sy1+tOrLuc25J9\n/M39BdEc7+t9ufYFlv9w58hfPcstS/7Sei7nZ16DF9798/zlngfsNu+6ZzrbfOP0\n9PkF58xXb8//seWDba9ePF68WX6jb/rCvb8ttXS/3/jknrlre59o/P7o4aN9lz7d\nHG98rHDwyqaVTb9c/Xbfro+f75KP/PB07uu5hmcefS65cXOme1asXL/vm8bFs8dX\nDx+Yqu89NtwwuVw32bn597nmM+VCvvG96qm9Q8nW0defaspnv5v7ZMftYLHfPXf6\ny18b8h1G9M/PfvbTQVF/5MqxkVvLhc7PazZeOrnQdgisgq6/AQ==\n=5wxk\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.7.3\"},\"cryptocurrency\":{\"address\":\"1JEPc3eAiQ5ZMRn2jwhTtEU329v93cD43E\",\"type\":\"bitcoin\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876B68AD482D32\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"merkle_root\":{\"ctime\":1419370141,\"hash\":\"08feb374d40ffc82b70fdbc0e2d253c8277e00c57df446ff86377c7fd5091ca0d63b5966fd18a56c0a5a340f6c36477ff54bf582b6c3c318aad6e8e99d165e7e\",\"seqno\":115589},\"type\":\"cryptocurrency\",\"version\":1},\"ctime\":1419370239,\"expire_in\":157680000,\"prev\":\"37ad41cf064989dfd849a0af178aa066dd396b3452433a9883a69e88e35e75b8\",\"seqno\":7,\"tag\":\"signature\"}",
            "sig_type": 6,
            "ctime": 1419370239,
            "etime": 1577050239,
            "rtime": null,
            "sig_status": 0,
            "prev": "37ad41cf064989dfd849a0af178aa066dd396b3452433a9883a69e88e35e75b8",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 8,
            "payload_hash": "57c95524e2fdb546201af092292ae9d27a055a8934d81a8f8488eec777441d1e",
            "sig_id": "f6689d2fa3ff3b2fc130fcadb73e7de28ca55b3c59b493de45f9c61d980b5d220f",
            "sig_id_short": "9midL6P_Oy_BMPyttz594oylWzxZtJPeRfnG",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.1\nComment: https://keybase.io/crypto\n\nyMPVAnicdVRriFVVGJ2Z7OGIYpJQoBKHULAx9/uhRFmBIUExzWiadvn2a+5xZu4Z\n7znXZpCh6EcpRknmg8wyEhmFgdQ/EQSTjJhFREHSixBMS4gIS1Aiap87kzpQ59e+\nZ5+71vrW961vfOZNLe2tm17qrY4+uoS0fjre32jp2n9wfEtiMjeULNuS2L7U14ry\nVIN+nyxLev2Qgdzfl2Z3P+tN0pFs9vU8zWrxylmNLGfeSaqUos5ggICYFByhIL30\nhlPALiTDHSVMCRrSWo+vD9TTkiPhCgjjjjplCLeYB2E9DSFIrKQwQoFjijhKIms1\ny4spapImZiV18e1/fN/bvEAYYeoRp4IIcIQKJbySJIDkRjDgRiGKCdcSGW5BgjTS\naWWpjj8UidcYCIII12jCMWxwkDwYzhETjkokkGKEAyVIYYTKD3Nfn3Qus1mtltWF\noKUBRR1sb2lB1J/avDylrnLNTNqR9EFexHoqtgrRpGQZZlgzoSPXFNiNvVme9w3V\nStCmquiiwEgSpCEo7oTVlhBttGIaO9ZUNel+adjUDoCLD3UoFi6Mk0R6BSGaxYAA\nJyYIDt5H46Y4qoXQ2DOJqXDeaBY4ozggIYLmIBQTgRJpo4GeMBdw7CoHA9hC/A/X\nzkdHo/a6788KXxmoZ1mIdjwdZ69I+6/XzTjvSGyjXo+sWHNPhMLEK6SARS7GvA4o\ncjMfOEGSUmcDcCuIZAp5Rpx3XALCJZaP8v0kOJcSswnwSQWlK00VzfmvettbcVBA\nZWNeNuZaEHrSotowyf/2oglRKYYGfLObeQFFyRdvcr+plsVjPKU9ExOLqaRWO2kt\nt8wBozErgLShAjtHLA9Ee8kRRVh5jJ3FKGCMGFIxUIYYikIygTbBR4Y3NGkqBaR9\npeYBGOrLwFWqkFcjnRAids0aqaWmJlCOZcyblwikhtJuMPFMtI4Z4LGHcSyAQqBg\nsHbO8ORaEeSGIhh45wXyQkpJNMVGeqGIDlRT6wL3DAclwAYvnQXr4pCCxiw48CTm\nisaakuEyGc0SJgNyw34prbthJASWcbXERg4OpHVfSWvNZgqF4lO67zdHEOKUC17L\nqMRRrgFRRuOOsiEmQlOHBaPKYI8RZ8JEJ5lEsV4TIwOOeheuF6qiMOiJkLHcGhSN\nuk+GT4ytn9bS2t5yy81t5dZsaZ8++99dutvd+vfqHSu2LX5415d3kjNzVp6ffezq\nocZrzyx4e+v+eR+8PPPcqSOd/o87BjeN/3Xpu8PvVc3H73e+sKHt3PjJ4qnuGc9n\ne3etubTnwY/2zOrad1z2v37xnauPLThxYIa7evL+Kztnbb/3q4Xzz46ceujyyZ2X\nD7W17lytty1ad3r7+cE3vnl8JF/Z/cSZhXNH0wvTalnl21+6nrxr2x74fu46Li7f\nvqrozBf/9EX3xcaVH7d2fP57/4YDoQuvfW70zZGtvz1ycN7RkWM7zp7+dfT4J0fC\n0uVn1/9wbN6fe98dGT+84nx7embtqle+vqAPHl236J4H5nfMectNH7tt6e4FY2PJ\n4L41n21esvziq9K92KY6Pyx+/ges9xbZ\n=2qFk\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"dc90c54ed738883db1aaf0476500f7e7eb53a1df\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":3,\"last_id_change\":1419469550,\"username\":\"jkosslyn\"},\"id\":\"58a6107209af85d6c9c229b98491d400\",\"key\":{\"key_fingerprint\":\"adddd3d07b76bd727e8af6264a2a52bf65aeed23\",\"kid\":\"010196691e47136deb94f5431f066f95a6846f327c782e24df1f6c5aba1cae4759de0a\"},\"remote_proofs\":[{\"ctime\":1419469455,\"curr\":\"195e26812e808a4f0644e9f094f4ef520733dcfa5c627480e42ded57a011946e\",\"etime\":1577149455,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"jkosslyn\"},\"proof_type\":3,\"state\":1},\"seqno\":1,\"sig_id\":\"1373c9d7cc5c4da43eb5a09b361dd2c5f29e7503018e11dc10f1104083a1b2b30f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"666196cb79793bf35175c1e70a79a4194ab70a299a7a55a6adda3af3ab19ddb5\",\"seqno\":2,\"sig_id\":\"4aede60e67772931b7e6829f393cdf5e41f86acfe7dcacd610a914fdae21a2310f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1419617476,\"expire_in\":157680000,\"prev\":\"2d8dfe977e6d359a03430c5cf22993d16438b1e10546b1b247055abc9cad3edf\",\"seqno\":8,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1419617476,
            "etime": 1577297476,
            "rtime": null,
            "sig_status": 0,
            "prev": "2d8dfe977e6d359a03430c5cf22993d16438b1e10546b1b247055abc9cad3edf",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 9,
            "payload_hash": "d70f647a155416bd94ad669254e5513aa667c03ed55f047f81cd6fc9bc3ed89c",
            "sig_id": "7ec6ea814573f2d7db7947262a624e4b0990ae5cc48feff1c8572b69f60b769e0f",
            "sig_id_short": "fsbqgUVz8tfbeUcmKmJOSwmQrlzEj-_xyFcr",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.1\nComment: https://keybase.io/crypto\n\nyMSiAnicrVVbiFVVGD6T3RyNmUqSLqZtUYxGW/fLeRgDMSytqCQIJ07rOrOdyznu\ns0edxklfNMtECroR3R5KIiQJKy2K0hgljXpworCUsIcghHkoRLLs38dxmiEfeuht\nrb3W/v7//77v/9cXV00pNTeteaK7a9eyhaTp8PDS/tLK3Sv2DSa26geS8mDietLQ\nlxerPtMbknLSHQasqYdFaXXOumCTtmRtyOpptQ+OCCYOReWFwcypwG1kHgtBuVOe\nWaGc5I4F5ZKhtgKmAI1pX2fIallaxEi4MoRxT72yhDvMo3CBxhglVlIAgPFMEU8J\nRO2q1vNJ2SQNzErq4etF7nc3DhBGmAbEqSDCeEKFEkFJEo3kVjDDrUIUE64lstwZ\naaSVXitHNWwUgWNsCDIA19+AY9jiKHm0nCMmPJVIIMUIN5QghREqLtZDNsZc1VX7\n+qoZEFIQkGfGdRcUQP6pqxer1FfGycSyLekx9RwKqrguAyzBN4Y101hRPAm3nmep\nLSAbOYVIAtNa4OC9MVIizSzjwRcyRNfIaYz7gq7J/MsYHFMMu4C9IMxJLEBEawwV\nmgmBuLMkeGcm8cmVlVJoHaWlhmjuGQtWaWohoCc8aiCYwf+KBsGIDJgbuAFkI0KU\nlhr4hNyz0FvNQ6WWVasRyFgFzsvT3kbRSHHBIZW2xPVnGUTVInoZoieE+Yg8kpTJ\naDhxyjjLDbJBw4ZFLpiGpRXUe4WZgoysB1dB+mEMnAuo8zx4LQtrAdxiHyiVETGG\nPC7K9TGGwIKWQAgEYIobISkKihZyO8miYlQSbzTHTCbjtRT8Nupp9FFXcN0Vb3JT\nWV0vBB5sWPiCNXq8q2Zhkav2JkUm1RzM0gMHXXleKxf8NIAq+UCtyBoh1Aaym7zY\nwGE9rOmrJmUOq7TzQgsIo0B7QaGnQHzDDZHYYWMDEgoJr0kAj2ojsNZgcWMl904G\nY72PoB8qWCrQzockQ22TFBFMCo7HFVFG6+A5U0Y55r2nnHAFvSuUpNQJb6F7qCch\nYm09olw6473TRmkvYmx01ARFKDkPPqbI/yD3f1RkTI3ONO/qt8nFm2yiEPRiKogJ\nKuDIHMwT6zTCzikwkpKehUAU9z5gQqR2gkQsEMxL0EVZ6D4rJOcSBgoT/1LhkUaY\nSm7SniLhmhnoqRpf6TL1rkIGbkA7G3GA/mPRIGUdjgp7bS3DVGHFsVYIBYcxtLfg\nIRiNYGRHwqTyxZwcKwKjCVUI6GwfWSGiRRRkhSFKvBIgeQzSUOMcgYkZJXS0DdYE\nypSMMDItPAXWFV4aKiZeo4axwTfh3Si4G7cWDE6GMAH1w/pamoVKWtzgEjzbcP2Y\nJ8BBmnPCAoneciYIwiYiTYgmJmhPpEGcm4IFsIJR0KFKheCklIxhj8M/lWpIzHQW\nCqedfSbvz0IytP+zjktLTc2lyy+7pHgNS81TWy+8kdPeveLcHV89fe9o86nHsk8O\nr/781NeHdz3XvX354jmbPt6zYNuts7a+cfWh0y2lqvtwSm3JsH372S1Tl3fcvm7D\nxv3zjy75+eDJcph/6pmZ59666Zf81yU/tR1o//73v+4cXf5N6+md82Zsapr96Ob3\nbmg5eeTmE9c//uLw8dbnj7llO6d/ef+ylU/evbXlxtKxo/v63xxomdYxPb3unhcO\nnX3/24fnvnZ88we37B356OVzZ+c9sG36gbL59GD3suGlZ3pqw90L4ow/XmnvGN14\n4rYNO3e80zMyJz515Mot5dWjuzpeXbG79aXy3M6WWav2nhzh4s8HL9+x59Dxa9fe\n1b69fYDNfH3qD/sPDK2/r7L4msEzv/3YMfuh0ncLR/4GukqcnQ==\n=TY+X\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"212c0f8d6a14c8e5bf4d16635c8d4b68c75c4e8c\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":17,\"last_id_change\":1419491831,\"username\":\"strib\"},\"id\":\"ef2e49961eddaa77094b45ed635cfc00\",\"key\":{\"key_fingerprint\":\"7fec4841ce1d624c7165bfbaa36946605cb2edca\",\"kid\":\"010158b77699f7b3a295d44eb893b35cd25f9872469483e6427e15a44e4a502289790a\"},\"remote_proofs\":[{\"ctime\":1408565716,\"curr\":\"96fd7efd224df0d07347fa52c8acb5a0be952c4f5649e95b63dd8148b35bd15f\",\"etime\":1566245716,\"prev\":\"b1de337f0440d12edcdffee4e974c7acb485a6730e830608c74f84372da95147\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"oldcore.com\",\"protocol\":\"http:\"},\"proof_type\":1000,\"state\":1},\"seqno\":5,\"sig_id\":\"716a8094638a2d63a5a271c1abe06806d92e0819a619998cab75dc7eabddfeb80f\",\"sig_type\":2},{\"ctime\":1408647651,\"curr\":\"8a99ed548a8c4ddd35258ce368733c6db7823d2ef19bd0357caddc9a89d6ff0a\",\"etime\":1566327651,\"prev\":\"96fd7efd224df0d07347fa52c8acb5a0be952c4f5649e95b63dd8148b35bd15f\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"strib\"},\"proof_type\":3,\"state\":1},\"seqno\":6,\"sig_id\":\"1f4c125bc901cc8ffe87d4ee285dde12279c62f160c8e0688b841b67557046460f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"85afebbf1e8934fa08bc1f81d9bb413818519800ec1124c65eea900f8f2478d2\",\"seqno\":10,\"sig_id\":\"67b3df4b782b03ce30132d869edfe7a3acc270bf7022bebae3487fca7bd6abc80f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1420840121,\"expire_in\":157680000,\"prev\":\"57c95524e2fdb546201af092292ae9d27a055a8934d81a8f8488eec777441d1e\",\"seqno\":9,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1420840121,
            "etime": 1578520121,
            "rtime": null,
            "sig_status": 0,
            "prev": "57c95524e2fdb546201af092292ae9d27a055a8934d81a8f8488eec777441d1e",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 10,
            "payload_hash": "251d1c8b830c3c4b131a182d27188712bf85a7026755bd0ad16ec2f58c880a07",
            "sig_id": "2c449b4270eeb6094bd48557866343e89dde1983a1694a5625b8076ed793fbc30f",
            "sig_id_short": "LESbQnDutglL1IVXhmND6J3eGYOhaUpWJbgH",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.1\nComment: https://keybase.io/crypto\n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\n/fOhx5fP2ffg1w6ed+015rc3fOvBp++87PDF/wN3bEWj\n=S3O8\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"212c0f8d6a14c8e5bf4d16635c8d4b68c75c4e8c\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":103,\"last_id_change\":1419928822,\"username\":\"max\"},\"id\":\"dbb165b7879fe7b1174df73bed0b9500\",\"key\":{\"key_fingerprint\":\"8efbe2e4dd56b35273634e8f6052b2ad31a6631c\",\"kid\":\"01013ef90b4c4e62121d12a51d18569b57996002c8bdccc9b2740935c9e4a07d20b40a\"},\"remote_proofs\":[{\"ctime\":1391720984,\"curr\":\"35782a3561e40a5e7217500bd67b5aadac5d7f6f9594dcd5d15cd06879da6e19\",\"etime\":1549400984,\"prev\":\"0146be7c3669a5f5da89e9908e8bb5f1fdb9fcfd6d2d66a51664f1075a7637b1\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"maxtaco\"},\"proof_type\":3,\"state\":1},\"seqno\":3,\"sig_id\":\"9f731fa7c75b64e1c9d70300b4383196a8fb432294a4308f8d7a379376a0b1900f\",\"sig_type\":2},{\"ctime\":1392252388,\"curr\":\"c610bab1245d05ac6fa95eda4d008d67c3732bd441dcc5c74f48fc8f8ba44c31\",\"etime\":1549932388,\"prev\":\"35782a3561e40a5e7217500bd67b5aadac5d7f6f9594dcd5d15cd06879da6e19\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"maxtaco\"},\"proof_type\":2,\"state\":1},\"seqno\":4,\"sig_id\":\"667062cc7300f11292079f7c4da0ed8e53e52ca484bb55ae693e7d0aa614322a0f\",\"sig_type\":2},{\"ctime\":1395867108,\"curr\":\"7d72a9f98ca4739e3939c89339d3f67966a7160da9df6215acb6a27c4a1ea292\",\"etime\":1553547108,\"prev\":\"2df5de847a546d662d3dda502aa283328f477430bfc24ce0c681ca68c4d4a200\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"keybase.io\",\"protocol\":\"https:\"},\"proof_type\":1000,\"state\":1},\"seqno\":24,\"sig_id\":\"3517ddd1a5c9d708f21634d126814647129a64086a967daa5017e7966b89c9820f\",\"sig_type\":2},{\"ctime\":1395868002,\"curr\":\"ea61172238d18212d53a71316dc09e8e48f2e654490262872018d09854ab26e0\",\"etime\":1553548002,\"prev\":\"7d72a9f98ca4739e3939c89339d3f67966a7160da9df6215acb6a27c4a1ea292\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"oneshallpass.com\",\"protocol\":\"https:\"},\"proof_type\":1000,\"state\":1},\"seqno\":25,\"sig_id\":\"93734bd81d3b50d25401d7451a136e171757990d35110b585b729dbca2019d050f\",\"sig_type\":2},{\"ctime\":1397071049,\"curr\":\"eec8f9d3a754e793301b108ddb548dc46bbefaaaec189306b169a3ef71f22abd\",\"etime\":1554751049,\"prev\":\"b6168f3a49898b4aba5c5fb162c2b5b6df87407bca7649feba5364751e693c13\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"oneshallpass.com\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":34,\"sig_id\":\"0e577a1475085a07ad10663400de1cd7c321d2349cf2446de112e2f2f51a928b0f\",\"sig_type\":2},{\"ctime\":1397071368,\"curr\":\"ad3efd2c8ff323f96841b482bc3cfe92195bc72fe318ce0b95cf66ee3160aa55\",\"etime\":1554751368,\"prev\":\"eec8f9d3a754e793301b108ddb548dc46bbefaaaec189306b169a3ef71f22abd\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"maxk.org\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":35,\"sig_id\":\"baacd979d2ebe325482d2921ea72df72dbf9a31aeaa69f2b14b1eccfed31c72e0f\",\"sig_type\":2},{\"ctime\":1401977613,\"curr\":\"467a473af5b7d548f1e1a1d199398f7fe3a9ea40e4c615d261e6b44a56b96642\",\"etime\":1559657613,\"prev\":\"4111d9457743da3ad15826242bac5db8bb5ebf77d6de71a41a23ecfc4ed4bcf9\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"nutflex.com\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":49,\"sig_id\":\"4b4c67c83b4995aae5d4a5dde916fb35762551e67e35d4d4f1be99e2309238600f\",\"sig_type\":2},{\"ctime\":1407166100,\"curr\":\"acc5ffcf17cd722b5844b24564e2f644d265e010047bfdd378919be8f0603ba3\",\"etime\":1564846100,\"prev\":\"b328cdb4904ebcc196a49c67355f3ee9239bd735d7246e90cf20c7e7307e706b\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"reddit\",\"username\":\"maxtaco\"},\"proof_type\":4,\"state\":1},\"seqno\":55,\"sig_id\":\"095d87d153e6590c2d6c032752d3c43485a1f64e61b99d58f989f6da28b2a8320f\",\"sig_type\":2},{\"ctime\":1407167610,\"curr\":\"8d906bd4d0bb2d667f5c8191d6e7616f1de1e69e950f1f2e6ea181316d8d6dbd\",\"etime\":1564847610,\"prev\":\"acc5ffcf17cd722b5844b24564e2f644d265e010047bfdd378919be8f0603ba3\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"coinbase\",\"username\":\"maxtaco\"},\"proof_type\":5,\"state\":1},\"seqno\":56,\"sig_id\":\"d0f9aaff73dc67f40fb6ebb6fdf473e11de43f11ae64a5fbffb00218914e19af0f\",\"sig_type\":2},{\"ctime\":1407186953,\"curr\":\"c9d4ca19adac4680957b2ce37d38280a1a32d2a7dfe606ae7236a9c982e2c75e\",\"etime\":1564866953,\"prev\":\"8d906bd4d0bb2d667f5c8191d6e7616f1de1e69e950f1f2e6ea181316d8d6dbd\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"hackernews\",\"username\":\"maxtaco\"},\"proof_type\":6,\"state\":1},\"seqno\":57,\"sig_id\":\"a9fb80f43bed684a40e48de1e365e7ace765a3309819f9f79cdb27e3dd9802f50f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"040933ff11e2a44476ef0395989349a917a0aad723e41585d1087330c4365c3e\",\"seqno\":86,\"sig_id\":\"46ec4cb327088e808d068ab8236f39dd38a6cd840e229d5f8a5d65b4c7d6a58d0f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1420840192,\"expire_in\":157680000,\"prev\":\"d70f647a155416bd94ad669254e5513aa667c03ed55f047f81cd6fc9bc3ed89c\",\"seqno\":10,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1420840192,
            "etime": 1578520192,
            "rtime": null,
            "sig_status": 0,
            "prev": "d70f647a155416bd94ad669254e5513aa667c03ed55f047f81cd6fc9bc3ed89c",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 11,
            "payload_hash": "a6808c59df1a6406a961e81b7038eb132c9ab04eb827ee844663ef5a7e651e90",
            "sig_id": "cf90e4629d273b677662b511269fff5756714d2889278c4297aebe9d723ac8240f",
            "sig_id_short": "z5DkYp0nO2d2YrURJp__V1ZxTSiJJ4xCl66-",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.1\nComment: https://keybase.io/crypto\n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\nxl07Gran/Y2t75XvfPM3px744zLn55bn/gsCjHb2\n=ofws\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"212c0f8d6a14c8e5bf4d16635c8d4b68c75c4e8c\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":167,\"last_id_change\":1418935207,\"username\":\"chris\"},\"id\":\"23260c2ce19420f97b58d7d95b68ca00\",\"key\":{\"key_fingerprint\":\"94aa3a5bdbd40ea549cabaf9fbc07d6a97016cb3\",\"kid\":\"0101d49265e97c636dbffa62070f29e07bdee44aca90b0eaa1645cc29fe5320f8a6f0a\"},\"remote_proofs\":[{\"ctime\":1391738324,\"curr\":\"461262ff766cf5da5eb40e5db29128a453ffb939840a23c0ffcb95073c888302\",\"etime\":1549418324,\"prev\":\"34152df8ec04aa33ad1bdbfd4927ed4d64d1be2ed3c4060043f313041a3722cf\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"malgorithms\"},\"proof_type\":3,\"state\":1},\"seqno\":4,\"sig_id\":\"3e1e20a1a11f4f0ee4ea4470b694fc115f928ce0b1f9aa2951a65081da8340820f\",\"sig_type\":2},{\"ctime\":1392240892,\"curr\":\"4396ec2c10ba4c4e1d931af42ed4d65ecea9f57a858fd44e86f2950071ef871f\",\"etime\":1549920892,\"prev\":\"d2284c9ab38c28650ccf4d1feb7af9f4ee46ee760ca45711a5e3afe008e518c3\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"malgorithms\"},\"proof_type\":2,\"state\":1},\"seqno\":7,\"sig_id\":\"0d9f6b71c043f2ef80b6dea44338471edc3ef76e2cee236dd397412ad9631aa90f\",\"sig_type\":2},{\"ctime\":1395868048,\"curr\":\"45c6893a87e18f27502266c08f0d0118f5ecd3fd0bd52432b18cddbd3f6520a8\",\"etime\":1553548048,\"prev\":\"26662070e4cd5345a302af7d513bc8d08a9a21cae783c6ad18f0113b039211f5\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"keybase.io\",\"protocol\":\"https:\"},\"proof_type\":1000,\"state\":1},\"seqno\":82,\"sig_id\":\"58ff6251889edccbb54495ea3f7d33f57c2d18e114dda887aeab6e06552aed780f\",\"sig_type\":2},{\"ctime\":1407170371,\"curr\":\"9fe1cf15276a8c34ebe03eae24e90f4ea44e6c6afd69a96bd6a85bf1ec025780\",\"etime\":1564850371,\"prev\":\"fe90114a30fcf13d0b9af77cb4291c1d550462fa567b0ecffdf436840f0fd8da\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"coinbase\",\"username\":\"bitcoyne\"},\"proof_type\":5,\"state\":1},\"seqno\":202,\"sig_id\":\"c6d2c827fd7292ae36a359fef3ad04736ff373901b3db037af773ce13db7fab90f\",\"sig_type\":2},{\"ctime\":1407767238,\"curr\":\"510fb8ebf6ea77aa220f0a4cd7d0be212f9b294161f0cb2b43b6be024aa9ec67\",\"etime\":1565447238,\"prev\":\"044e1a2cc9d956c4780d4eb7fcaae9dfdf560eb9d5095ad653d8b37f63172c7f\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"hackernews\",\"username\":\"malgorithms\"},\"proof_type\":6,\"state\":1},\"seqno\":209,\"sig_id\":\"c60cae4ab2b165fd15960433b0af48c66d37f3940e56a8d3b4955991947cb3f00f\",\"sig_type\":2},{\"ctime\":1408375965,\"curr\":\"57a78d3f45c749eab6a6b2796db103c6887ccbd7827a27e3f33cf3f0950cad8a\",\"etime\":1566055965,\"prev\":\"94e9a25ab3e551d43ff700e94be3cc446785ed4f44d83d93d00fb3ddc601070e\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"chriscoyne.com\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":221,\"sig_id\":\"dbf530c689d2f3af60c5b11341d5eb2ad229995d6ef39dfd1661962d088a75ec0f\",\"sig_type\":2},{\"ctime\":1408376008,\"curr\":\"ba697b9daa2617a80f53d94a8204a6db5aaf2215faf1feb4a43d2a09a95c6a7c\",\"etime\":1566056008,\"prev\":\"57a78d3f45c749eab6a6b2796db103c6887ccbd7827a27e3f33cf3f0950cad8a\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"chriscoyne.com\",\"protocol\":\"https:\"},\"proof_type\":1000,\"state\":1},\"seqno\":222,\"sig_id\":\"7dbc723fc7c0abfe761af6729fb27603ef9be58718153329ed2d4ddd8d6ddb910f\",\"sig_type\":2},{\"ctime\":1414608934,\"curr\":\"2f7e338d5d1915975611c4d99f225986f7978c2b0a89f7efed973cabc92b6e0b\",\"etime\":1572288934,\"prev\":\"b7036840be4000a1f5a006bc8b164c535300507c103651f389290e1ac42885e9\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"reddit\",\"username\":\"malgorithms\"},\"proof_type\":4,\"state\":1},\"seqno\":255,\"sig_id\":\"1a8fa75aa365dd96c95fd229722668dfb0026973d89e489b256d87ee06a4bd670f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"d24c186ca79e522e96c4d10f407646b9c1b5a2beb35fbaa03f2d070638af65b2\",\"seqno\":299,\"sig_id\":\"677463e4cd27e7cc4382f20aa7f21882f694feebdd4ddee623786b1cd21083940f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1421363548,\"expire_in\":157680000,\"prev\":\"251d1c8b830c3c4b131a182d27188712bf85a7026755bd0ad16ec2f58c880a07\",\"seqno\":11,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1421363548,
            "etime": 1579043548,
            "rtime": null,
            "sig_status": 0,
            "prev": "251d1c8b830c3c4b131a182d27188712bf85a7026755bd0ad16ec2f58c880a07",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 12,
            "payload_hash": "ed4c85c1ebc1765450203ad9f9c092dff4f1580367add60cf5db8eb894088521",
            "sig_id": "18c9a91230de00b8ce9f42845ffb24d2b690a2b2f9ee18efd66d0f14d000fb070f",
            "sig_id_short": "GMmpEjDeALjOn0KEX_sk0raQorL57hjv1m0P",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.1\nComment: https://keybase.io/crypto\n\nyMQNAnicbVRriFVVGJ3xlQ5IjSVaIsYho3LM/X6MlKEgUf0wMxNKrvs59zjjPddz\n76ijTRGWk1GBhSU+wqSQyNESggzULCGsFJMIC0F60FCBgT8sMrPvXKfJoTh/9tnf\nZu211re+fXT8yKaW5pV9neX+e2eS5s+OHu5uWnTgwgPrEpv5nqR9XeK60lCpF6uK\nWRGS9qQz9FhTC3em2c2rg03aklUhr6VZBUoEE4ei8sJg5lTgNjKPhaDcKc+sUE5y\nx4JySW9bAVOAxrTSEfJqnhZ3JFwZwrinXlnCHeZRuEBjjBIrKQDAeKaIpwRuLWe1\n+jA2SQOzlHrY/Z/znY0CwgjTgDgVRBhPqFAiKEmikdwKZrhViGLCtUSWOyONtNJr\n5aiGH0WgjA1BBuC6G3AMWxwlj5ZzxISnEgmkGOGGEqQwQsXBWsgHnctcVqlkORhS\nGFDPjessLAD+qasVq9SXhszEqC3pMrU6CCq5sgGXYI8RzLggFA3H7apllVowlc4C\nt0FMIS85UdxEJjUKxunIvfaSYKUlbxAbbEDh2fAmeBXBO8KRjdpbKaJHDBTiaKP0\ngtiIGdXYxWGmCqSpsxhBo6kmRjAbHAFnDDOUIbA7WIqFoyoGb63knFEhFDFGMasU\nVWAqcM/DiqweStU8yyI48ijEr56uGFLONeNtievO80IilZZSEayWQXBZtFkor4kF\nVEcExdIaLb2jAQvLtSdQw4ITUhCwTgL9MAjOpcb0Cng1D6sAXAuFEGGRUaQ1Z9Bk\nHbSAD8QLo4ziBKxByBENumW0IFkyxRwcNkAgGdJS+NvQ0ximcnCdJW/qprS8VnR5\naKo60nq52yb/7evMorEzG51twJTqPVUo0rakVjf1gj1UamFlJUvaGazSjisjYDnw\n4AgyCzyhhzo4DgRjIB5YSqqI4IGSECNG3BARVVBBSMwhvTwYjIr+FmhX7iO9SxvX\nlOom7Sp4V01PV2Z8qWxq5WL0iXNKBQiIFox6KbT2RjtGYNq0ZhBHH7CVVEMKuVba\n2kgFtwyuciGawrBBEfwqEQDHkDUM8RiopjIYqYkLoEgbYrzmgB5AHtcGIg6zjrj1\nHsPL4axTBh6jpLcYtYaEwYm76sEqrLs6YJJSLiAWa6ppHkpppRGNIgkIDSXDwL9y\nkKeIIeVIGC1wUCANURUspsRpYxELEEMZgmIMxj1EboqQ4qDRv0IxAWamAzBBb8XU\nu/OQ9H704WOjmppbmsaMHlG8w00t467753WeMHDNX7Om7WS/Xryp0rf43EPnp/75\nzdlfJsya/vE7m+tPrxs5ctOR9u3L75l3d/WFeX29Z15a+/OaU9f/3rpp0qQtO5bM\n/WT2xFf7zISBC/nst4/dsD9/ZePlhYcWXRpx7PT63Y/v2Xdp9Jtr9xx85qmpn645\n9psvTb3cv3dz85NfSXx2y6hbph9dMHb+qcq+Px6ptE75YODc/p0UzT9x0ew6+f72\nbG/rhq65L792/r3WGUd6x97+Xf8dPy7b83zPxN39Gt217IuB0wvXL3i29fUHmw7v\n37b4vpMHv2Rzfrp/ylvd/mz7mJYnug/Yvq27ftg658xz045/Xl7y/dfjb0tvXTru\n+IY3Di3aMXnX6rXbN1778Il3F9747eQZq178G3IcKcE=\n=dEMP\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"212c0f8d6a14c8e5bf4d16635c8d4b68c75c4e8c\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":10,\"last_id_change\":1421456230,\"username\":\"olsonseank\"},\"id\":\"80d75285af4790eac9f5d9d721897500\",\"key\":{\"key_fingerprint\":\"d8f6b6250bf9db76fd04fb51fbf7d62bf14391cf\",\"kid\":\"01016093cb108d4392a64bec275fa4a340362eb316c38fedbb755436682aa84b88380a\"},\"remote_proofs\":[{\"ctime\":1421455945,\"curr\":\"837b336eb97e6572d3268d92b82ac26317ba97dc3e16b59d2d9216522edbbbc7\",\"etime\":1579135945,\"prev\":\"9680024f43099541f79e96969f146a8a852db700c2993c7fb2a67484c954a82a\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"olson-sean-k\"},\"proof_type\":3,\"state\":1},\"seqno\":4,\"sig_id\":\"b5a825025bb706fd9ec5484fe2d4c9738265e32eff105a26f8e8e671525a5ea10f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"22cc88e7d69643d7699da9c423e09940d7de1b739d9d5989bbf365b45eacefaa\",\"seqno\":5,\"sig_id\":\"7d640ba405fe3937ea792ceb709a2ad953e0e70659aac948205bdd1c15cbc8ac0f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1421473356,\"expire_in\":157680000,\"prev\":\"a6808c59df1a6406a961e81b7038eb132c9ab04eb827ee844663ef5a7e651e90\",\"seqno\":12,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1421473356,
            "etime": 1579153356,
            "rtime": null,
            "sig_status": 0,
            "prev": "a6808c59df1a6406a961e81b7038eb132c9ab04eb827ee844663ef5a7e651e90",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 13,
            "payload_hash": "1421995af7068ad3bf3982028330e56b89e187a1c94aa1f5426cf61696e1478c",
            "sig_id": "19da9eba50589b026df038f1ca162d5b008a101fa102abc728d3b4a44a4747e00f",
            "sig_id_short": "GdqeulBYmwJt8DjxyhYtWwCKEB-hAqvHKNO0",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.1\nComment: https://keybase.io/crypto\n\nyMSlAnicrVVbiFVVGJ7RdNLMMhKxyHRDBDaN634Zgx4ydKywSEspOazrnO2ZOWfm\nnD2j0zBBRkkvGhRZWqMMpd00sx5EfPASUkghJRaFJAQ9lFY+KNZQ9O8z0+iYDz3E\nedlnr7W/9X+X/1+fXD+xYWpj94ZScdeSu0njsaPLehqWH/50sD+xFd+XtPYnriMN\n5Sx/KpvOkLQmpdBnTS20pJW5a4NNmpPeUK2llTIsEUwcisoLg5lTgdvIPBaCcqc8\ns0I5yR0LyiUDzTlMDhrTcnuodlXT/IyEK0MY99QrS7jDPAoXaIxRYiUFABjPFPGU\nwKnFSi0bV01SxyykHt5eZX+pvoAwwjQgTgURxhMqlAhKkmgkt4IZbhWimHAtkeXO\nSCOt9Fo5quGPIrCMDUEG4HrqcAxbHCWPlnPEhKcSCaQY4YYSpDBC+cZaqI4qV3GV\ncrlSBUFyAbKqcaVcAqg/dbX8KfWFMTExbk46TC0DQgVXNKASvGMEa6UIkeNw16Tl\nPtiQg9ar0gzR4IGlRsJqRqKXVODIkBFS83pVo+rngo13wDoRHDaaeSYUd4jSiLkj\nxFtjKWFEemmQ0mycolZijJTJNypMpGUaS6UoDoFIYxR8qkBfbSRzNkqvlHWQCwiI\nVhojI0FRqL0aOitZKHRVK5UIcjwB2cvSzhHaSDKFFWpOXE+1Cqc6xOGtRY5HilBk\nmigThHSCgQMsUEENlKoIM4xaDXVT5mjkHMy3GlsH5YdRcC7BsRHwrmrozV0VlDjt\nIQKEY084E5FFUMAH4iOREXRhNhDtBRce5LXSQEQD0xqEM1olY1xyfet86p1UDK5U\n8CYzhTW13OKxlmpPs2KPTa5makeaS1PHKGR9XbBCm5NaZrK8dFiphe5yJWkl8JS2\nj4SfQPEgAQHDpHDeQddZQi10FaJaMw7U8mjbqBgOSkShCKZYcQVfaOskiskI2sh5\nZKB5vBNcEMnGnLBSYOuxYvCjAnHBlICIkBg80cbrgDWjxggjsbXQaRFJa4XhUojg\n4dsrnKAj4KNO/A82/0cn8nEyqnw5rK21uFpLua+nJfiepC5/Bs3bAYvFLOtqvdIS\n6HR0NVfoZa5QzHV0hjpGPIJu0pw6w2w0MGMY8lxK5DS0HHKEek4shRFiggsGW2Sc\nsP9yZXX9mEJm0o6cQJfp66gYXyiaWhGO81Z4ZDDMAOMB0UXtGJPEMhO1IpRAW+d+\nGaOFh6MxBw6CEYE8i4i7XLlREvIyEhyqZjxoC32NIPAseoYFV9EFL+AkyJkJ0gjM\nuVKcAV3LYVzkbhAkBFiYDOSjr05hdAJedoHk0l1KGsnNVwLysa4rrYZCWq5nRChU\nl3s0IsHDVQN3RbAOS8EZRwRRyB0QRhr6FVoXc5jqEErvBXKRe6uChRGGoESCLxHF\nYFdm2gET+JZN1lMNycDhg09e09A4tWHypAn5vdgwdcqN/9yW33zd9OfsZSc+Wnhu\n4dJXhtKh6+YcX/Lgw3tX7Hx/i+ufuXtz/7Y9Nw33tr45f92JhyY2vds29yVz1+M7\nu+2SzXu2bl/wQvx5/X3PT/9t5oIflv7YO2vlycFFpWNixSPL27ZtL71z9tfu2wcX\nvPz0yROHLhydsvoD7p57asZrG5uWvd0y+7sPC4e+Onhmy+vTVr03Ye/p9Yv3DQ+u\nmnPg91vWzMOLvmi+Z96FO7cfmLNpx5ns0dPz93dM3r3p4ucTe88uXnnvbZNmvHHt\n/lk7OndNb/t2cHjGefP9x037yS/PnNpzvrWzePzmixMee5XuaywsPzVh2sY/yl3k\nyE/tf20YttNunf/ZA0ND8v5nV5Mv5dbSDefaVh2Z9NYdL/4NNTaZww==\n=nhwa\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"212c0f8d6a14c8e5bf4d16635c8d4b68c75c4e8c\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":11,\"last_id_change\":1421988227,\"username\":\"jinyang\"},\"id\":\"9403ede05906b942fd7361f40a679500\",\"key\":{\"key_fingerprint\":\"bc6ec1a94d4685c033f15c22dbab32427d7a0894\",\"kid\":\"0101b71108a15c28127b49178831ee27aa8bab875b9a74cbf7d88bcd168e598910a70a\"},\"remote_proofs\":[{\"ctime\":1420748180,\"curr\":\"c05142b0c5f300f4928ae67c640604e363a27d824a43b924234c3f5513eb91bc\",\"etime\":1578428180,\"prev\":\"4632c9d0b5251d2546f4f2dbde2df27f85c4be29d656d42fb7aff7e4991a9a98\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"jinyangli\"},\"proof_type\":3,\"state\":1},\"seqno\":2,\"sig_id\":\"22c94e32ab376cdcc75b23bd3d039945c9dca7abf841e86f6821318582ab9bc70f\",\"sig_type\":2},{\"ctime\":1420756274,\"curr\":\"b761bd184848360564867a02fed29ad9e1943aa6a71bb72ff07bb6a5766ed761\",\"etime\":1578436274,\"prev\":\"c05142b0c5f300f4928ae67c640604e363a27d824a43b924234c3f5513eb91bc\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"news.cs.nyu.edu\",\"protocol\":\"http:\"},\"proof_type\":1000,\"state\":1},\"seqno\":3,\"sig_id\":\"3159fca3c42d0dba953ca4bfaa7b40d5770c96850c23d52b3320aecea1b0ac6b0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"db6d0a1059adc23cf9c4472b4af982323f14836aa96d7701510064260d4f05cc\",\"seqno\":7,\"sig_id\":\"595345e9b27a0e494fd41658fced69adb23ae7a61558854ca3b5a083eb9206600f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1422051486,\"expire_in\":157680000,\"prev\":\"ed4c85c1ebc1765450203ad9f9c092dff4f1580367add60cf5db8eb894088521\",\"seqno\":13,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1422051486,
            "etime": 1579731486,
            "rtime": null,
            "sig_status": 0,
            "prev": "ed4c85c1ebc1765450203ad9f9c092dff4f1580367add60cf5db8eb894088521",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 14,
            "payload_hash": "1d21eed55eeb085102bf28673e316156187760fc5836ebc0af9aa579ba5e9b7f",
            "sig_id": "c7b288e398279c6d0de3d9accfcbd93dc601a0b51460b92bd965523a7fc246390f",
            "sig_id_short": "x7KI45gnnG0N49msz8vZPcYBoLUUYLkr2WVS",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.1\nComment: https://keybase.io/crypto\n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\nS3d98rr6+daBpx5901rrkf8B5q5j+A==\n=997G\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"212c0f8d6a14c8e5bf4d16635c8d4b68c75c4e8c\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":54,\"last_id_change\":1421371756,\"username\":\"patrick\"},\"id\":\"1563ec26dc20fd162a4f783551141200\",\"key\":{\"key_fingerprint\":\"246201479f89c599c08f29b3fa706aef3d045f20\",\"kid\":\"01017c13682b0965e718feaead9f04e468086c3864ba8294072c97594a5957a99f6f0a\"},\"remote_proofs\":[{\"ctime\":1392412788,\"curr\":\"c20d2386bb1714908f0de84f76d9aabc7c2911a1bc6ee451f98bb8833a528fac\",\"etime\":1550092788,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"patrickxb\"},\"proof_type\":2,\"state\":1},\"seqno\":1,\"sig_id\":\"fff3da1be82b15fa7a5633f85d2ae19e4f128986e6e80e64a4859dedb56f47080f\",\"sig_type\":2},{\"ctime\":1392412864,\"curr\":\"e4e726145a1e148ed4bb2c8481d6a6f399d0e65d04af5e590be98c2ca2f23705\",\"etime\":1550092864,\"prev\":\"c20d2386bb1714908f0de84f76d9aabc7c2911a1bc6ee451f98bb8833a528fac\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"patrickxb\"},\"proof_type\":3,\"state\":1},\"seqno\":2,\"sig_id\":\"81a9003a098ec52cd98a9f4983ad7edea81e9beae23064ceb97518138e3cd0f50f\",\"sig_type\":2},{\"ctime\":1395889480,\"curr\":\"7a1d453f123e38a125f0e872c6ec4ef0ba80119008996742abfad2675a16110e\",\"etime\":1553569480,\"prev\":\"bff52a25614ccd76fafabd54adbede2a2832899a0c0fe324a433de6100845331\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"patrickcrosby.com\",\"protocol\":\"http:\"},\"proof_type\":1000,\"state\":1},\"seqno\":11,\"sig_id\":\"8e0043c22d467a8b966c528c17882118c23bec1b7803fd4b5430c500e04352e20f\",\"sig_type\":2},{\"ctime\":1420864565,\"curr\":\"171a9d6e5b305d8eaf72f78e75a4bc814bbc2c5e5a429724640f0727212c0cca\",\"etime\":1578544565,\"prev\":\"eae03042fbc56ad02ffd97e0d07b43a5d9c4f4959a03bdc3a5d1659d0f28797d\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"hackernews\",\"username\":\"patrickxb\"},\"proof_type\":6,\"state\":1},\"seqno\":23,\"sig_id\":\"9127676da00562183585508aaf5b5fff4e936f208fc238e388d37a40eb99b3fa0f\",\"sig_type\":2},{\"ctime\":1420864719,\"curr\":\"db4612f774b4e4689b5b8d9ab9c99447d30bce0a969a131d65fa2810b352cc44\",\"etime\":1578544719,\"prev\":\"171a9d6e5b305d8eaf72f78e75a4bc814bbc2c5e5a429724640f0727212c0cca\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"stathat.com\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":24,\"sig_id\":\"250861cbd50c104723977c779c4a7ab7919d664dabf0228a61fc65d24dd4001d0f\",\"sig_type\":2},{\"ctime\":1420864917,\"curr\":\"6c1866e5442084ff8b6892ca7a8cf2c5555e5c6281553331593e3d8efb911ff7\",\"etime\":1578544917,\"prev\":\"db4612f774b4e4689b5b8d9ab9c99447d30bce0a969a131d65fa2810b352cc44\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"numerotron.com\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":25,\"sig_id\":\"3c666d4cf57d5f1b11a556def8d0f78e89ec92b994867a4ce0245f5bd187355b0f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"bfdae4f98d9910608c2760cf87fb6e3e1ea3527fa57ef2221e4d6e5cafd25cb3\",\"seqno\":31,\"sig_id\":\"baeb664889dc7f1e203bffe3cb364249a55661f45f9c84e3a470bbde36e34f360f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1422051540,\"expire_in\":157680000,\"prev\":\"1421995af7068ad3bf3982028330e56b89e187a1c94aa1f5426cf61696e1478c\",\"seqno\":14,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1422051540,
            "etime": 1579731540,
            "rtime": null,
            "sig_status": 0,
            "prev": "1421995af7068ad3bf3982028330e56b89e187a1c94aa1f5426cf61696e1478c",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 15,
            "payload_hash": "185e2496a0dc012deea1c883759a74e0f7c33e38e11a40530aac76cd1916c7f5",
            "sig_id": "de2c80dde617e34b0715fd082b593a2b7eff7d4915c2095e4f11e92871d92d9f0f",
            "sig_id_short": "3iyA3eYX40sHFf0IK1k6K37_fUkVwgleTxHp",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.1\nComment: https://keybase.io/crypto\n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\nv9Ii8F4=\n=Az9U\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"212c0f8d6a14c8e5bf4d16635c8d4b68c75c4e8c\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":45,\"last_id_change\":1421805379,\"username\":\"gabrielh\"},\"id\":\"b7c2eaddcced7727bcb229751d91e800\",\"key\":{\"key_fingerprint\":\"3f6e504945ae44e6a53d49845431890fe1eebcde\",\"kid\":\"0101230aa5a191471449cc05ee95ce3637ebb16e409bb3b92ac0a8cc28eef92c0b500a\"},\"remote_proofs\":[{\"ctime\":1413405722,\"curr\":\"55d1e2013c91f4ef410538cb03cadfcd0b40e86dfa8a0594a42036eedfd61fff\",\"etime\":1571085722,\"prev\":\"461f0051ad08190465b5c8f4cab2fb8c5f04290c3f3cbd53794e7dbe5b89d448\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"gabrlh\"},\"proof_type\":2,\"state\":1},\"seqno\":19,\"sig_id\":\"dbbb07105a43c18ab1a9d01b564c591a6d4d57e35dd4ba3f1a0ed8e33f8517eb0f\",\"sig_type\":2},{\"ctime\":1413405744,\"curr\":\"5381de2b714095bcf65f41c1b235c11ab6fb87cea8b2c9a3bcb95095ab26e191\",\"etime\":1571085744,\"prev\":\"55d1e2013c91f4ef410538cb03cadfcd0b40e86dfa8a0594a42036eedfd61fff\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"gabriel\"},\"proof_type\":3,\"state\":1},\"seqno\":20,\"sig_id\":\"916bccc5e1fd7e784340cca2795d1ae352922bd498ec196bbff6c00b7a304d2d0f\",\"sig_type\":2},{\"ctime\":1413406032,\"curr\":\"eba1b91ed0048983d7e32efeee7fecce16547998ff3e05b67045a751ef99be1c\",\"etime\":1571086032,\"prev\":\"5381de2b714095bcf65f41c1b235c11ab6fb87cea8b2c9a3bcb95095ab26e191\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"coinbase\",\"username\":\"gabrielh\"},\"proof_type\":5,\"state\":1},\"seqno\":21,\"sig_id\":\"2c9e345b5acedbcfe3d5d9b2d6115b59e9d6ad31ad262bb07266ccc6db0b95940f\",\"sig_type\":2},{\"ctime\":1421558613,\"curr\":\"fdd474158530c9816ca9707c3cda809aaa24a0f69b5bcc06f5776a54cea54fff\",\"etime\":1579238613,\"prev\":\"3f0c4b94e2c2d6435aa583012793a77222c3ec7b2ab48ff95359df23fa5f422c\",\"remote_key_proof\":{\"check_data_json\":{\"hostname\":\"rel.me\",\"protocol\":\"https:\"},\"proof_type\":1000,\"state\":1},\"seqno\":25,\"sig_id\":\"fdee6f84bdd854d055fe51be696b175e5f2d2d6b143bc09a52f7bad57dfe05160f\",\"sig_type\":2},{\"ctime\":1421558741,\"curr\":\"158668859ea086ccd02ce16724f2b35a6a71834f98867429cd1210a075346443\",\"etime\":1579238741,\"prev\":\"fdd474158530c9816ca9707c3cda809aaa24a0f69b5bcc06f5776a54cea54fff\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"rel.me\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":26,\"sig_id\":\"9c8edb92c68188a43266c7037e1148c9320a969b248b97059aeca65ddfc39e100f\",\"sig_type\":2},{\"ctime\":1421805307,\"curr\":\"a0a74b05db02a6c66c9cf2ffd3754449ff680cd4fb61f80cba2038a23e7c0bfb\",\"etime\":1579485307,\"prev\":\"a71816d134d54dae20073688eea5753fd756903173b0461213aa3956a3c66de5\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"hackernews\",\"username\":\"gabrielhn\"},\"proof_type\":6,\"state\":1},\"seqno\":28,\"sig_id\":\"2787406a0b57a2eb7de822c5ecea6bdcd28a3ed96191a27d46459d747a6b292c0f\",\"sig_type\":2},{\"ctime\":1421805358,\"curr\":\"b8e2fb00585d67ee5144b37483a94e251444144c9dcd5949ca30bf275bafe59f\",\"etime\":1579485358,\"prev\":\"a0a74b05db02a6c66c9cf2ffd3754449ff680cd4fb61f80cba2038a23e7c0bfb\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"reddit\",\"username\":\"gabrlh\"},\"proof_type\":4,\"state\":1},\"seqno\":29,\"sig_id\":\"a3a73a75b6941ff0aeabdededc9640d27a45cff49f2a9b57538b4608a45ba6590f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"b8e2fb00585d67ee5144b37483a94e251444144c9dcd5949ca30bf275bafe59f\",\"seqno\":29,\"sig_id\":\"a3a73a75b6941ff0aeabdededc9640d27a45cff49f2a9b57538b4608a45ba6590f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1422051554,\"expire_in\":157680000,\"prev\":\"1d21eed55eeb085102bf28673e316156187760fc5836ebc0af9aa579ba5e9b7f\",\"seqno\":15,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1422051554,
            "etime": 1579731554,
            "rtime": null,
            "sig_status": 0,
            "prev": "1d21eed55eeb085102bf28673e316156187760fc5836ebc0af9aa579ba5e9b7f",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 16,
            "payload_hash": "e11940eb12b36e89eadfd73af91558819262952825faca0af1ccc744fb896a6a",
            "sig_id": "08fa6ab7f3d57852666fc4cc2ccf946557970be7bccd64cf6894d2940d90e53d0f",
            "sig_id_short": "CPpqt_PVeFJmb8TMLM-UZVeXC-e8zWTPaJTS",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: GnuPG v2\n\nowGtVVuIVWUYPaNpGgk+GNgN7ASRzaX/fjHL64NERmWQ3Tz91zl7LmcfzzkzOdpA\nRhEVSpRPhmZoD0USYoYXBi1ItJsGFURaFEl5gwxJQm369p5pmEEfEjpPm/N/e/3r\nW2t93351yvjCpKblL3aWty5qJU2fnbCFhz/+Vq8q2tT3FWetKrquJFQa2VPFdIfi\nrGJn6LOmHtqSdEYl9aGtoz5juKal2Btq9SStQBVqk2202N+SlWcvx6TSHmrVWpJh\nFbkyhHFPvbKEO8yjcIHGGCVWUlihjGeKeEoAspzWG2NuLeaYpcTDv3n9fKHmLYT6\nhXl9T37AsMVR8mg5R0x4KpFAihFuKEEKI5QV1kNtuKXUpZVKWhMiZ9wdap1doVRL\n07xt10iyIswIQZwpqoCTqZfhNaKkshS56IVhRivNGUPBWapJDJFQzyUmkQaCcAhO\nMsGww5ZF5yNVxJIorceYRUkwtoRGbKK22vgIgDoI77VDkjpjNGXKGCWR9FEzAhiC\nY8mhiXpYXkmBHNWSU+DeqBnXmbEGsRJXz54SXxqxBdOWYpepN0C9kisbsCRvDGst\nqFBjNemqp5V6MJXOTJOxImihW4q5zgp5yYniJjKpUTBOR+61h44UMMp1Hg5A5tnY\nEHgVwWvCkY3aWymiRwwMw9FG6QWxETOqsYsZRn4ZwggLpKmzGCkPh8QIZoMjYDQY\nQBmiggRLsXBUxeCtlZwzKoQioB6zCsxDJmunFrrTRihVweMIIj0+2mXMONeMQ889\ntVrWIpWWUhGslkFwmcVSKK+JBVRHBMXSGi29owELy7UncIYFJyQjYJ0E+mEYnEuN\n6RD4MINMlZxFnrRycJ0lbxqm1FHP7BqZufakUe6xxUsNas0cas0tymFKjb4qHILN\n9YZpZHf2j4SEwVPSPjQ4lhsFyhMOGiFQXgcH2WYxkOOeOS0hn4IHSkKMGHFDRFRB\nBSExhxniwWCU2ZLBDV1I+lvGaAiBwpyNaBjBHIYIgeDAjFNhA/c+S4xGMJEOvJTK\nw1vKWi24l1oI4pFmAgMZo8RYDYUcAr8yDV2aVLIVchkVR2I+WkN+OQ3lKA2ZBnuR\ncdwKaTkxNID9lsJCo9FBLK3SzED4FCw3LYITmFJtYNBBQMihuETCJ/NrSg2TdGW8\nq6avKzW+NLxt/gcN/1MT+2ZccRv92ebJmxheQKO+BPjy2yOsqCa1UEoquaVCIfhl\n+odeAMGKBwJ1BnmHMPEBAudgeiXXRrKAonSUBqoCxoYhTpExTgrnsYbRl3H0YhTA\nzLQDJnRcMY2eWij2v9R001WFpkmFiRPGZd+6wjWTp/77Afz0+asHO9ae+vHm/o5f\n1qzdqH9vffDNu2bPWaR2pJM3L9ozezF+qof2Du7a/uEN+sC2R1f/dvrkpu+++WPl\n4Y3b1DN41Xn7xvHTT6y/eOvEo+u+Otf79bKlZ+bsf2ja0bffXT3ur7OquG/dETew\nTC4pP3fgpw33Frae+fWRO5dunDD/gw175u2+3x2qbe/e23awOvDYwMy/F7i3Zurm\nu6dM9ys+Gb9307Xd6nxHYUeTOtn85cs/XEi/WP50877Wc5/fNnDdO/rG9Rc2d0zf\neeiBbfsX306X7D92ik21r63ZufT6uR+tPjhhy9xnX5g6f8HhOy7esnLa+68vXtb8\n8xl5T/m9P1/ZW921+/sjxw6fRbPvG2g61zI4GE78Aw==\n=mZ4m\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io node.js client\",\"version\":\"0.7.3\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876B68AD482D32\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"merkle_root\":{\"ctime\":1422054838,\"hash\":\"2878b30cfd6a4a9895440ecb392fef23d5712f3e201eec74641c1b4fcdf382b2f7bd114f7211b23f1af9b9adffd69e6dd9c073caa9348aa8707df9421b465175\",\"seqno\":139753},\"track\":{\"basics\":{\"id_version\":13,\"last_id_change\":1421996368,\"username\":\"olsonseank\"},\"ctime\":1422054969,\"id\":\"80d75285af4790eac9f5d9d721897500\",\"key\":{\"key_fingerprint\":\"d8f6b6250bf9db76fd04fb51fbf7d62bf14391cf\",\"kid\":\"01016093cb108d4392a64bec275fa4a340362eb316c38fedbb755436682aa84b88380a\"},\"remote_proofs\":[{\"ctime\":1421455945,\"curr\":\"837b336eb97e6572d3268d92b82ac26317ba97dc3e16b59d2d9216522edbbbc7\",\"etime\":1579135945,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"olson-sean-k\"},\"proof_type\":3,\"state\":1},\"seqno\":4,\"sig_id\":\"b5a825025bb706fd9ec5484fe2d4c9738265e32eff105a26f8e8e671525a5ea10f\",\"sig_type\":2},{\"ctime\":1421996154,\"curr\":\"fa344022969f7136be5dd479090a32cd4378d2198bb965d79662d094614c9a86\",\"etime\":1579676154,\"remote_key_proof\":{\"check_data_json\":{\"name\":\"coinbase\",\"username\":\"olsonseank\"},\"proof_type\":5,\"state\":1},\"seqno\":7,\"sig_id\":\"49bbc0ac5b67b52a3eb59b325c3fcbecb894a66886ce96ec61339ab23a5e80a60f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"fa344022969f7136be5dd479090a32cd4378d2198bb965d79662d094614c9a86\",\"seqno\":7,\"sig_id\":\"49bbc0ac5b67b52a3eb59b325c3fcbecb894a66886ce96ec61339ab23a5e80a60f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1422054969,\"expire_in\":157680000,\"prev\":\"185e2496a0dc012deea1c883759a74e0f7c33e38e11a40530aac76cd1916c7f5\",\"seqno\":16,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1422054969,
            "etime": 1579734969,
            "rtime": null,
            "sig_status": 0,
            "prev": "185e2496a0dc012deea1c883759a74e0f7c33e38e11a40530aac76cd1916c7f5",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 17,
            "payload_hash": "c49c8c4bef4862d13da267f19416f3edc650159b18f25abfe8a0f3c5417159e2",
            "sig_id": "8afe41e09f1bb4b99765f8fc97ad11aad3d2de26c110d5b858628db443decf600f",
            "sig_id_short": "iv5B4J8btLmXZfj8l60RqtPS3ibBENW4WGKN",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.1\nComment: https://keybase.io/crypto\n\nyMIXAnicbVJbSBVBGD5aiimRRVeDiK1AymJmdueyp16OZUYiBdaDkMns7qxtJ3eP\nZ4+WliFU9FJhHeqlwiAzMizRStColKLIIqXLQy8ZEVqYdBGrhy6zVm/Ny/B///d9\n/8zHf3f6lFBmStXh6I62DStRSn/v6urQlnvv/b2K4Vm1SnivEhWTl+24FSIeiztu\nQgkrmHGkYUu1mIGwCbFNTKHatk0ho8QgjFsaQ5aKlDxlh+cHCmljcF+scjyJyaLc\nsST6H350sgEggKoAWCWIcAuphBHBKLI5xQbRODYYUCHCOgUGNjnl1KCWzkxVlwVD\nsg05AlzaVU/aadCANsW2gTHQiKVSQADTEOYqAgwCEBB9EXd5pZBsz/Rc14sToir1\neYrEaxxTBCFYXiV3XMnYyc2o9+cvsbiXkIJdErVcPxAkamOBy25hlP/VlhuOa8kA\nJb1GxH3Hkx5QMs2EEwyEGkIIE4pxniL2xJy4KA+mQEwJA/IEQ0SNtBQQ6hoQBkSG\nKvPQBbdsi6rc1iHGjEEdEaRjJAOwuckBt6FpmlTTbIPphJMgD19UuZ70pvKdvEJ6\n+k6FyxPVcaHU993ZNjWUkhlKT0sNdiCUOS3732b0nMsIXfuU3ZpfNrBWtQqalTWf\nmlakHr31bLyl43gSJAc/tnWlNXe2v1pwe+BUyZcZpb/yu78tSj/biJu/Nik/IpXn\nu3+ODNWNL9566XWscHvR5YWbI8cKk7MbNl4fpjlzJ2Yv35/VfyK1r7X+frL/wb2b\ng1UTnaONXQfG2ExcMPYhq+GKvQsDOON138uHkRv2nPCbh/6+OYUt4UfhK+klRWrv\nuo+Rk6Uds0JyDzpOd3w4WDwUKsjyvnubiusOPW1PvDiybMqlRPLz456RocS03uGM\ncO5YfmRe1ysr7dvohXd33hqbcrau/1wy+vZ52Zkl+5bWLTPrL169OL+2ZSw3M+sJ\nPbOxu/E3EGAmLQ==\n=QKaN\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"service\":{\"domain\":\"jacko.io\",\"protocol\":\"dns\"},\"type\":\"web_service_binding\",\"version\":1},\"ctime\":1422256755,\"expire_in\":157680000,\"prev\":\"e11940eb12b36e89eadfd73af91558819262952825faca0af1ccc744fb896a6a\",\"seqno\":17,\"tag\":\"signature\"}",
            "sig_type": 2,
            "ctime": 1422256755,
            "etime": 1579936755,
            "rtime": null,
            "sig_status": 0,
            "prev": "e11940eb12b36e89eadfd73af91558819262952825faca0af1ccc744fb896a6a",
            "proof_id": "c8f453c788deabc4839f5210",
            "proof_type": 1001,
            "proof_text_check": "keybase-site-verification=iv5B4J8btLmXZfj8l60RqtPS3ibBENW4WGKNtEPez2A",
            "proof_text_full": "keybase-site-verification=iv5B4J8btLmXZfj8l60RqtPS3ibBENW4WGKNtEPez2A",
            "check_data_json": "{\"domain\":\"jacko.io\",\"protocol\":\"dns\"}",
            "remote_id": "dns://jacko.io",
            "api_url": "dns://jacko.io",
            "human_url": "dns://jacko.io",
            "proof_state": 1,
            "proof_status": 1,
            "retry_count": 737,
            "hard_fail_count": 0,
            "last_check": 1434480198,
            "last_success": 1434480198,
            "version": 2,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 18,
            "payload_hash": "88d873d0340dfc5aa391dcfbe4a07cce979d8805c910b535af043c8e17579119",
            "sig_id": "e8ea962435235ba18daa87ad739bb59415706dbf82be6663e18e02529b9c38320f",
            "sig_id_short": "6OqWJDUjW6GNqoetc5u1lBVwbb-CvmZj4Y4C",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.1\nComment: https://keybase.io/crypto\n\nyMSGAnicrVVbiFVVGB5TsyZHjHroJZQtGugU634ZTYquWJpCxNRkp3Wdsztzzhn3\n2aNO42BB2oASWtSTD1oYlWVgvQRJWd5ActCiUEyIkB4iqSDNkuzfZ6ZpxiQIett7\nrbW/9X+X/9/72ya3tE5a9Xyl/M79t5JJRw4u72t5+FjbxYHE1n1/0jGQuJ401PLi\nqWaqIelIKqHfmka4La3PXhNs0p6sDlkjrddgi2DiUFReGMycCtxG5rEQlDvlmRXK\nSe5YUC4ZbC9gCtCY1rpD1pulxR0JV4Yw7qlXlnCHeRQu0BijxEoKADCeKeIpgVvL\n9UY+oZqkiVlKPaxe4XyluYEwwjQgTgURxhMqlAhKkmgkt4IZbhWimHAtkeXOSCOt\n9Fo5quFFEdjGhiADcH1NOIYtjpJHyzliwlOJBFKMcEMJUhih4mAjZKPK1V29Vqtn\nIEghQJ4ZVykkgPpT1yieUl8aE1O3Jz2mkQOfkisbECnpwIwQyRTjZAJsqKY9/VVX\nNWX4DoCblRGjsNTNQiKRhIaoIgIfLDwh16xs1IFCtIku4KCRidSzGD0HLSxjyggQ\ngoK9HvgxFzS1cYKqBgfnqbYCa+YwHLZBgnpOI4kQskxJrpgNkXjEhYeYYEOtoFpT\nLKIArUxRexaq9TyUerN6PYIkXZC/PK2OcZeCAHfXl2UFb6YQCp5RjgJjjEvhMHXS\nMGKRM1wyjh2zPiIiJOImaIJRDMEwZbXwQkD5YRScK8TICHhvFlYXCdKcaM9YIBRZ\n4RGxkGAonGhjpFZaS6kk5FsIpjEOAVFOLLM0CsEpNUXiRrkU+jb5NLupHFyl5E1u\nSk81CpvH2srV01qR5ORfvW0ClfL+Xtjk7UkjN3lRP+w0wqpaPekABo20e6QLgCi1\n0HyQTM4sFi56Y7jXUVqMLHRp4I5CzwrGI2IRXI1eQHBi0JFYA2olI2gj95HB9svs\ngCioMTskioyHKAkzAWSKnCsP7nvMYE1bFZxWEsQzxMAIEBQ+D1YRjQiWRih+uR0j\n4KN2/A9e/zc78jVpnodsohujRoQsbVQvd4NcyQ06zg1jHfcMhEBgBg1cR+vAA2IQ\nRQraQWBho7EghcEKpAlEcGBDLcTewOo/3FjZvKaUm7SnKLzX9PfUjS+VTaMM18UI\nXYkCIoFia4JBXEpOLcQZpJMwlwkSIB6m0mEEiS5OUB2cgGxjrVkyRoKNIyECZUVP\ne3AQZqsTmgbKsQ0UWlkSL7jH0cqoZGSE+oCDD5GzAPdYa5gBEoPF7GtSGB2B4/4g\nhXTjEqaVgJRALtb2plkopcUJsB6igNBYNBzTTrlitDAliMfUG5AtwhiCwUKDd4Ij\nzLXFKsJgtjEoqII6zrCE5UD+Jorhqtx0AybwrZm8LwvJ4CcfPz6lZVJry9VTryp+\njC2t187863f5bcu0S+b1Z5P193zwyE+L2ru/XndJLX5r09YHbpcPzjr24sUNF15Y\nXjlZP/9Etvrsh8cXrliwaO8Bu7N1/V07liy+6dG2qUuOz1k594ste7c/tHMb2br0\n3uV37pr/9onrrvlteM8Nv+7b3bXrUH3h/i/XHejccfjJb2auXzbn3elH/tgx15+v\nVLoGZs/Smzsv/Hz9d3vmLxv6bJve1XnLPLdh3UuCts0+03Hu6CubLvxee/q+T6cf\n+vFUfvqjjafDmf43Zmz39R+G1v7y2InDN++7cXjDgnlL+cCJrs5NSW1O9c1Tw++n\nKw4ffa805diWu4fPPfcav3jH5OTzkxvZtDPlZw7OODhkNie9L3916tWS3P79wNmz\nuxcP/Qm5OaOL\n=HJ4a\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"212c0f8d6a14c8e5bf4d16635c8d4b68c75c4e8c\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":9,\"last_id_change\":1422748452,\"username\":\"emilymcmahon\"},\"id\":\"2a81795a320f2723ef8f05c8b3ef0c00\",\"key\":{\"key_fingerprint\":\"1e90af3d4ffd5597b448a67d932c0d8424ce93bf\",\"kid\":\"0101a1ecd39b6194c17d9be77b7c907000b487584bef2d056dc0f1a3b6399316f6100a\"},\"remote_proofs\":[{\"ctime\":1422747622,\"curr\":\"e4800ed4350e444576c13c7a42b0ca57451c4bdf026705ae9210feea48b96d66\",\"etime\":1580427622,\"prev\":\"79529d44e230b6d02bd4b56d29aa798997787bf4664911ee0352b4b3f66533a2\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"coinbase\",\"username\":\"emilymcmahon\"},\"proof_type\":5,\"state\":1},\"seqno\":2,\"sig_id\":\"48b3b35c46d54b16cfdaa5d9f7b10bd6ae5c3212645f04f4cefd62a8fe9f2ba10f\",\"sig_type\":2},{\"ctime\":1422748428,\"curr\":\"70f45ef724aed29f558dc90d14f729b8ec987aa7a2ae8c63748eb8290217a685\",\"etime\":1580428428,\"prev\":\"e4800ed4350e444576c13c7a42b0ca57451c4bdf026705ae9210feea48b96d66\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"mcmahonerism\"},\"proof_type\":2,\"state\":1},\"seqno\":3,\"sig_id\":\"abc5d49b8054b3e59fbcd6a2a03080f1616bfaba68a188ebe2652673b747aa680f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"ff0100e02e31baea057753b2306c174d12062b0137c10798ea0539ec68991994\",\"seqno\":4,\"sig_id\":\"6e347c90d8c6d48c693e351be363972d65d1fb7f87f423de1edef54e013bba4a0f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1422986748,\"expire_in\":157680000,\"prev\":\"c49c8c4bef4862d13da267f19416f3edc650159b18f25abfe8a0f3c5417159e2\",\"seqno\":18,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1422986748,
            "etime": 1580666748,
            "rtime": null,
            "sig_status": 0,
            "prev": "c49c8c4bef4862d13da267f19416f3edc650159b18f25abfe8a0f3c5417159e2",
            "proof_id": null,
            "proof_type": null,
            "proof_text_check": null,
            "proof_text_full": null,
            "check_data_json": null,
            "remote_id": null,
            "api_url": null,
            "human_url": null,
            "proof_state": null,
            "proof_status": null,
            "retry_count": null,
            "hard_fail_count": null,
            "last_check": null,
            "last_success": null,
            "version": null,
            "fingerprint": "58a245d3d8b25c15f6ce3fff71876b68ad482d32"
        },
        {
            "seqno": 19,
            "payload_hash": "abf4894d0e34671e497ab1f2982dcb24c8e8974d10a00576eeb81ef236d75da2",
            "sig_id": "e7ee174db64a5f37d2ce5765367b8bd073aa81627356b9aec84af7b0136ff6380f",
            "sig_id_short": "5-4XTbZKXzfSzldlNnuL0HOqgWJzVrmuyEr3",
            "kid": "01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a",
            "sig": "-----BEGIN PGP MESSAGE-----\nVersion: Keybase OpenPGP v2.0.4\nComment: https://keybase.io/crypto\n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\nk5E=\n=vfBK\n-----END PGP MESSAGE-----",
            "payload_json": "{\"body\":{\"client\":{\"name\":\"keybase.io web\",\"version\":\"939665abec79c7598695e34a8599905b29fd8ac6\"},\"key\":{\"fingerprint\":\"58a245d3d8b25c15f6ce3fff71876b68ad482d32\",\"host\":\"keybase.io\",\"key_id\":\"71876b68ad482d32\",\"kid\":\"01013e053626ad23686e872fa75b64a5b803125970b5ca7a7b7d98c39ca7825b81a20a\",\"uid\":\"41b1f75fb55046d370608425a3208100\",\"username\":\"oconnor663\"},\"track\":{\"basics\":{\"id_version\":19,\"last_id_change\":1419475392,\"username\":\"willwilson\"},\"id\":\"dc1944ea9df3dcddecbf994031a89e00\",\"key\":{\"key_fingerprint\":\"dd613e6f6261ad1ea6a72c3d32c8cec143040cb1\",\"kid\":\"010180b3d73a72827897132d4cc422ec07e5786db6212b6a9ddacad48b767a1c88610a\"},\"remote_proofs\":[{\"ctime\":1419468116,\"curr\":\"c4b068dc645df3ac1352e1bb5647f37633fecd0024593e716f98cb2d4832fcdf\",\"etime\":1577148116,\"prev\":\"b8da9a936fa9489b58943c64c170e095209f58a750da769bfa821b24e42a8422\",\"remote_key_proof\":{\"check_data_json\":{\"domain\":\"williamaugustwilson.com\",\"protocol\":\"dns\"},\"proof_type\":1001,\"state\":1},\"seqno\":5,\"sig_id\":\"5f576a0731d37e3e025f4170d84ef0cb8f54709d2d3882bb247cbfc8db446cf90f\",\"sig_type\":2},{\"ctime\":1419468293,\"curr\":\"b7c1e5f80170b827f93cdea9831bcd6612b9b4c1997310d7b7cbe0ab3f6b154c\",\"etime\":1577148293,\"prev\":\"c4b068dc645df3ac1352e1bb5647f37633fecd0024593e716f98cb2d4832fcdf\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"github\",\"username\":\"wwilson\"},\"proof_type\":3,\"state\":1},\"seqno\":6,\"sig_id\":\"41a916d2bc13a407b22d71271e4b85d624d0ca61110198eae7f9afd8694731e10f\",\"sig_type\":2},{\"ctime\":1419468468,\"curr\":\"00ed23721676c981446cbfe1d8fecb5e08af42c562c27b6502e7dc0686a9f738\",\"etime\":1577148468,\"prev\":\"b7c1e5f80170b827f93cdea9831bcd6612b9b4c1997310d7b7cbe0ab3f6b154c\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"twitter\",\"username\":\"willwilsonFDB\"},\"proof_type\":2,\"state\":1},\"seqno\":7,\"sig_id\":\"bafbd853dd34d2398538d9015f3ea2d596dfbd191e7618cb1a5ad35535200d450f\",\"sig_type\":2},{\"ctime\":1419469445,\"curr\":\"8a1a8927a662c0f0ecd438b7c4c5de1ef91ad48c93382bda5467c8c30af328fe\",\"etime\":1577149445,\"prev\":\"6304bb04a2777bd09fbdd8556e0434d6b4c77c0d1247c5b017278547fd886f12\",\"remote_key_proof\":{\"check_data_json\":{\"name\":\"coinbase\",\"username\":\"willwilson\"},\"proof_type\":5,\"state\":1},\"seqno\":10,\"sig_id\":\"ded5deb64539634e097a17d666d1a7e637e26470b7861c482a3273c63728bfb10f\",\"sig_type\":2}],\"seq_tail\":{\"payload_hash\":\"8a1a8927a662c0f0ecd438b7c4c5de1ef91ad48c93382bda5467c8c30af328fe\",\"seqno\":10,\"sig_id\":\"ded5deb64539634e097a17d666d1a7e637e26470b7861c482a3273c63728bfb10f\"}},\"type\":\"track\",\"version\":1},\"ctime\":1424279037,\"expire_in\":157680000,\"prev\":\"88d873d0340dfc5aa391dcfbe4a07cce979d8805c910b535af043c8e17579119\",\"seqno\":19,\"tag\":\"signature\"}",
            "sig_type": 3,
            "ctime": 1424279037,
            "etime": 1581959037,
            "rtime": null,
            "sig_status": 0,
            "prev": "88d873d0340dfc5aa391dcfbe4a07cce979d8805c910b535af043c8e17579119",
            "proof_id": null,
            "proof_type": null,
            "proof