package libkb

import (
	"errors"
	"fmt"
	"time"

	lru "github.com/hashicorp/golang-lru"
	"github.com/keybase/client/go/protocol/keybase1"
	"golang.org/x/net/context"
)

// UPAK Loader is a loader for UserPlusKeysV2AllIncarnations. It's a thin user object that is
// almost as good for many purposes, but can be safely copied and serialized.
type UPAKLoader interface {
	ClearMemory()
	Load(arg LoadUserArg) (ret *keybase1.UserPlusAllKeys, user *User, err error)
	LoadV2(arg LoadUserArg) (ret *keybase1.UserPlusKeysV2AllIncarnations, user *User, err error)
	CheckKIDForUID(ctx context.Context, uid keybase1.UID, kid keybase1.KID) (found bool, revokedAt *keybase1.KeybaseTime, deleted bool, err error)
	LoadUserPlusKeys(ctx context.Context, uid keybase1.UID, pollForKID keybase1.KID) (keybase1.UserPlusKeys, error)
	LoadKeyV2(ctx context.Context, uid keybase1.UID, kid keybase1.KID) (*keybase1.UserPlusKeysV2, *keybase1.PublicKeyV2NaCl, map[keybase1.Seqno]keybase1.LinkID, error)
	Invalidate(ctx context.Context, uid keybase1.UID)
	LoadDeviceKey(ctx context.Context, uid keybase1.UID, deviceID keybase1.DeviceID) (upak *keybase1.UserPlusAllKeys, deviceKey *keybase1.PublicKey, revoked *keybase1.RevokedKey, err error)
	LookupUsername(ctx context.Context, uid keybase1.UID) (NormalizedUsername, error)
	LookupUID(ctx context.Context, un NormalizedUsername) (keybase1.UID, error)
	LookupUsernameAndDevice(ctx context.Context, uid keybase1.UID, did keybase1.DeviceID) (username NormalizedUsername, deviceName string, deviceType string, err error)
	ListFollowedUIDs(uid keybase1.UID) ([]keybase1.UID, error)
	PutUserToCache(ctx context.Context, user *User) error
}

// CachedUPAKLoader is a UPAKLoader implementation that can cache results both
// in memory and on disk.
type CachedUPAKLoader struct {
	Contextified
	cache          *lru.Cache
	locktab        LockTable
	Freshness      time.Duration
	noCache        bool
	TestDeadlocker func()
}

// NewCachedUPAKLoader constructs a new CachedUPAKLoader
func NewCachedUPAKLoader(g *GlobalContext, f time.Duration) *CachedUPAKLoader {
	c, err := lru.New(g.Env.GetUPAKCacheSize())
	if err != nil {
		panic(fmt.Sprintf("could not create lru cache (size = %d)", g.Env.GetUPAKCacheSize()))
	}
	return &CachedUPAKLoader{
		Contextified: NewContextified(g),
		Freshness:    f,
		cache:        c,
		noCache:      false,
	}
}

// NewUncachedUPAKLoader creates a UPAK loader that doesn't do any caching.
// It uses the implementation of CachedUPAKLoader but disables all caching.
func NewUncachedUPAKLoader(g *GlobalContext) UPAKLoader {
	return &CachedUPAKLoader{
		Contextified: NewContextified(g),
		Freshness:    time.Duration(0),
		noCache:      true,
	}
}

func culDBKeyV1(uid keybase1.UID) DbKey {
	return DbKeyUID(DBUserPlusAllKeysV1, uid)
}

func culDBKeyVersioned(version int, uid keybase1.UID) DbKey {
	return DbKey{
		Typ: DBUserPlusKeysVersioned,
		Key: fmt.Sprintf("%d:%s", version, uid.String()),
	}
}

func culDBKeyV2(uid keybase1.UID) DbKey {
	return culDBKeyVersioned(2, uid)
}

func (u *CachedUPAKLoader) ClearMemory() {
	if u.noCache {
		return
	}
	u.purgeMemCache()
}

const UPK2MinorVersionCurrent = keybase1.UPK2MinorVersion_V5

func (u *CachedUPAKLoader) getCachedUPAK(ctx context.Context, uid keybase1.UID, info *CachedUserLoadInfo) (*keybase1.UserPlusKeysV2AllIncarnations, bool) {

	if u.Freshness == time.Duration(0) || u.noCache {
		u.G().VDL.CLogf(ctx, VLog0, "| cache miss since cache disabled")
		return nil, false
	}

	upak := u.getMemCache(ctx, uid)

	// Try loading from persistent storage if we missed memory cache.
	if upak != nil {
		// Note that below we check the minor version and then discard the cached object if it's
		// stale. But no need in memory, since we'll never have the old version in memory.
		u.G().VDL.CLogf(ctx, VLog0, "| hit memory cache")
		if info != nil {
			info.InCache = true
		}
	} else {
		var tmp keybase1.UserPlusKeysV2AllIncarnations
		found, err := u.G().LocalDb.GetInto(&tmp, culDBKeyV2(uid))
		if err != nil {
			u.G().Log.CWarningf(ctx, "trouble accessing UserPlusKeysV2AllIncarnations cache: %s", err)
		} else if !found {
			u.G().VDL.CLogf(ctx, VLog0, "| missed disk cache")
		} else if tmp.MinorVersion != UPK2MinorVersionCurrent {
			u.G().VDL.CLogf(ctx, VLog0, "| found old minor version %d, but wanted %d; will overwrite with fresh UPAK", tmp.MinorVersion, UPK2MinorVersionCurrent)
		} else {
			u.G().VDL.CLogf(ctx, VLog0, "| hit disk cache")
			upak = &tmp
			if info != nil {
				info.InDiskCache = true
			}
			// Insert disk object into memory.
			u.putMemCache(ctx, uid, tmp)
		}
	}

	if upak == nil {
		u.G().VDL.CLogf(ctx, VLog0, "| missed cache")
		return nil, true
	}
	diff := u.G().Clock().Now().Sub(keybase1.FromTime(upak.Uvv.CachedAt))
	fresh := (diff <= u.Freshness)
	if fresh {
		u.G().VDL.CLogf(ctx, VLog0, "| cache hit was fresh (cached %s ago)", diff)
	} else {
		u.G().VDL.CLogf(ctx, VLog0, "| cache hit was stale (by %s)", u.Freshness-diff)
	}
	return upak, fresh
}

type CachedUserLoadInfo struct {
	InCache      bool
	InDiskCache  bool
	TimedOut     bool
	StaleVersion bool
	LoadedLeaf   bool
	LoadedUser   bool
}

func (u *CachedUPAKLoader) Disable() {
	u.Freshness = time.Duration(0)
}

func culDebug(u keybase1.UID) string {
	return fmt.Sprintf("CachedUPAKLoader#Load(%s)", u)
}

func (u *CachedUPAKLoader) extractDeviceKey(upak keybase1.UserPlusAllKeys, deviceID keybase1.DeviceID) (deviceKey *keybase1.PublicKey, revoked *keybase1.RevokedKey, err error) {
	for i := range upak.Base.RevokedDeviceKeys {
		r := &upak.Base.RevokedDeviceKeys[i]
		pk := &r.Key
		if pk.DeviceID == deviceID {
			deviceKey = pk
			revoked = r
		}
	}
	for i := range upak.Base.DeviceKeys {
		pk := &upak.Base.DeviceKeys[i]
		if pk.DeviceID == deviceID {
			deviceKey = pk
			revoked = nil
		}
	}

	if deviceKey == nil {
		dkey := fmt.Sprintf("%s:%s", upak.Base.Uid, deviceID)
		return nil, nil, fmt.Errorf("device not found for %s", dkey)
	}

	return deviceKey, revoked, nil
}

func (u *CachedUPAKLoader) putUPAKToCache(ctx context.Context, obj *keybase1.UserPlusKeysV2AllIncarnations) error {

	if u.noCache {
		u.G().VDL.CLogf(ctx, VLog0, "| no cache enabled, so not putting UPAK")
		return nil
	}

	uid := obj.Current.Uid
	u.G().VDL.CLogf(ctx, VLog0, "| Caching UPAK for %s", uid)

	stale := false
	existing := u.getMemCache(ctx, uid)
	if existing != nil {
		if obj.IsOlderThan(*existing) {
			stale = true
		} else {
			u.putMemCache(ctx, uid, *obj)
		}
	} else {
		u.putMemCache(ctx, uid, *obj)
	}

	if stale {
		u.G().VDL.CLogf(ctx, VLog0, "| CachedUpakLoader#putUPAKToCache: Refusing to overwrite with stale object")
		return errors.New("stale object rejected")
	}

	err := u.G().LocalDb.PutObj(culDBKeyV2(uid), nil, *obj)
	if err != nil {
		u.G().Log.CWarningf(ctx, "Error in writing UPAK for %s: %s", uid, err)
	}
	u.deleteV1UPAK(uid)
	return err
}

func (u *CachedUPAKLoader) PutUserToCache(ctx context.Context, user *User) error {

	lock := u.locktab.AcquireOnName(ctx, u.G(), user.GetUID().String())
	defer lock.Release(ctx)
	upak, err := user.ExportToUPKV2AllIncarnations()
	if err != nil {
		return err
	}
	upak.Uvv.CachedAt = keybase1.ToTime(u.G().Clock().Now())
	err = u.putUPAKToCache(ctx, upak)
	return err
}

// loadWithInfo loads a user from the CachedUPAKLoader object. The 'info'
// object contains information about how the request was handled, but otherwise,
// this method behaves like (and implements) the public CachedUPAKLoader#Load
// method below. If `accessor` is nil, then a deep copy of the UPAK is returned.
// In some cases, that deep copy can be expensive, so as for users who have lots of
// followees. So if you provide accessor, the UPAK won't be deep-copied, but you'll
// be able to access it from inside the accessor with exclusion.
func (u *CachedUPAKLoader) loadWithInfo(arg LoadUserArg, info *CachedUserLoadInfo, accessor func(k *keybase1.UserPlusKeysV2AllIncarnations) error, shouldReturnFullUser bool) (ret *keybase1.UserPlusKeysV2AllIncarnations, user *User, err error) {

	// Shorthand
	g := u.G()

	// Add a LU= tax to this context, for all subsequent debugging
	ctx := arg.WithLogTag()

	defer g.CVTrace(ctx, VLog0, culDebug(arg.UID), func() error { return err })()

	if arg.UID.IsNil() {
		if len(arg.Name) == 0 {
			return nil, nil, errors.New("need a UID or username to load UPAK from loader")
		}
		// Modifies the load arg, setting a UID
		arg.UID, err = u.LookupUID(ctx, NewNormalizedUsername(arg.Name))
		if err != nil {
			return nil, nil, err
		}
	}

	lock := u.locktab.AcquireOnName(ctx, g, arg.UID.String())

	defer func() {
		lock.Release(ctx)

		if !shouldReturnFullUser {
			user = nil
		}
		if user != nil && err == nil {
			// Update the full-self cacher after the lock is released, to avoid
			// any circular locking.
			if fs := u.G().GetFullSelfer(); fs != nil {
				fs.Update(ctx, user)
			}
		}
	}()

	returnUPAK := func(upak *keybase1.UserPlusKeysV2AllIncarnations, needCopy bool) (*keybase1.UserPlusKeysV2AllIncarnations, *User, error) {
		if accessor != nil {
			err := accessor(upak)
			if err != nil {
				return nil, nil, err
			}
			return nil, user, err
		}
		if needCopy {
			tmp := upak.DeepCopy()
			upak = &tmp
		}
		return upak, user, nil
	}

	var upak *keybase1.UserPlusKeysV2AllIncarnations
	var fresh bool

	if !arg.ForceReload {
		upak, fresh = u.getCachedUPAK(ctx, arg.UID, info)
	}
	if arg.ForcePoll {
		g.VDL.CLogf(ctx, VLog0, "%s: force-poll required us to repoll (fresh=%v)", culDebug(arg.UID), fresh)
		fresh = false
	}

	if upak != nil {
		g.VDL.CLogf(ctx, VLog0, "%s: cache-hit; fresh=%v", culDebug(arg.UID), fresh)
		if fresh || arg.StaleOK {
			return returnUPAK(upak, true)
		}
		if info != nil {
			info.TimedOut = true
		}

		var sigHints *SigHints
		var leaf *MerkleUserLeaf

		sigHints, leaf, err = lookupSigHintsAndMerkleLeaf(ctx, u.G(), arg.UID, true)
		if err != nil {
			return nil, nil, err
		}

		if info != nil {
			info.LoadedLeaf = true
		}

		if leaf.eldest == "" {
			g.VDL.CLogf(ctx, VLog0, "%s: cache-hit; but user is nuked, evicting", culDebug(arg.UID))

			// Our cached user turned out to be in reset state (without
			// current sigchain), remove from cache, and then fall
			// through. LoadUser shall return an error, which we will
			// return to the caller.
			u.removeMemCache(ctx, arg.UID)

			err := u.G().LocalDb.Delete(culDBKeyV2(arg.UID))
			if err != nil {
				u.G().Log.Warning("Failed to remove %s from disk cache: %s", arg.UID, err)
			}
			u.deleteV1UPAK(arg.UID)
		} else if leaf.public != nil && leaf.public.Seqno == keybase1.Seqno(upak.Uvv.SigChain) {
			g.VDL.CLogf(ctx, VLog0, "%s: cache-hit; fresh after poll", culDebug(arg.UID))

			upak.Uvv.CachedAt = keybase1.ToTime(g.Clock().Now())
			// This is only necessary to update the levelDB representation,
			// since the previous line updates the in-memory cache satisfactorially.
			u.putUPAKToCache(ctx, upak)

			return returnUPAK(upak, true)
		}

		if info != nil {
			info.StaleVersion = true
		}
		arg.SigHints = sigHints
		arg.MerkleLeaf = leaf
	} else if arg.CachedOnly {
		return nil, nil, UserNotFoundError{UID: arg.UID, Msg: "no cached user found"}
	}

	g.VDL.CLogf(ctx, VLog0, "%s: LoadUser", culDebug(arg.UID))
	user, err = LoadUser(arg)
	if info != nil {
		info.LoadedUser = true
	}

	if user != nil {
		// The `err` value might be non-nil above! Don't overwrite it.
		var exportErr error
		ret, exportErr = user.ExportToUPKV2AllIncarnations()
		if exportErr != nil {
			return nil, nil, exportErr
		}
		ret.Uvv.CachedAt = keybase1.ToTime(g.Clock().Now())
	}

	// In some cases, it's OK to have a user object and an error. This comes up in
	// Identify2 when identifying users who don't have a sigchain. Note that we'll never
	// hit the cache in this case (for now...)
	if err != nil {
		return ret, user, err
	}

	if user == nil {
		return nil, nil, UserNotFoundError{UID: arg.UID, Msg: "LoadUser failed"}
	}

	err = u.putUPAKToCache(ctx, ret)

	if u.TestDeadlocker != nil {
		u.TestDeadlocker()
	}

	return returnUPAK(ret, false)
}

// Load a UserPlusKeysV2AllIncarnations from the local cache, falls back to
// LoadUser, and cache the user. Can only perform lookups by UID. Will return a
// non-nil UserPlusKeysV2AllIncarnations, or a non-nil error, but never both
// non-nil, nor never both nil. If we had to do a full LoadUser as part of the
// request, it's returned too. Convert to UserPlusAllKeys on the way out, for
// backwards compatibility.
func (u *CachedUPAKLoader) Load(arg LoadUserArg) (*keybase1.UserPlusAllKeys, *User, error) {
	ret, user, err := u.loadWithInfo(arg, nil, nil, true)

	// NOTE -- it's OK to return an error and a user, since certain code paths
	// want both (see note in loadWithInfo).
	var converted *keybase1.UserPlusAllKeys
	if ret != nil {
		tmp := keybase1.UPAKFromUPKV2AI(*ret)
		converted = &tmp
	}

	return converted, user, err
}

// Load a UserPlusKeysV2AllIncarnations from the local cache, falls back to
// LoadUser, and cache the user. Can only perform lookups by UID. Will return a
// non-nil UserPlusKeysV2AllIncarnations, or a non-nil error, but never both
// non-nil, nor never both nil. If we had to do a full LoadUser as part of the
// request, it's returned too.
func (u *CachedUPAKLoader) LoadV2(arg LoadUserArg) (*keybase1.UserPlusKeysV2AllIncarnations, *User, error) {
	return u.loadWithInfo(arg, nil, nil, true)
}

func (u *CachedUPAKLoader) CheckKIDForUID(ctx context.Context, uid keybase1.UID, kid keybase1.KID) (found bool, revokedAt *keybase1.KeybaseTime, deleted bool, err error) {

	var info CachedUserLoadInfo
	larg := NewLoadUserByUIDArg(ctx, u.G(), uid)
	larg.PublicKeyOptional = true
	upak, _, err := u.loadWithInfo(larg, &info, nil, false)

	if err != nil {
		return false, nil, false, err
	}
	found, revokedAt, deleted = CheckKID(upak, kid)
	if found || info.LoadedLeaf || info.LoadedUser {
		return found, revokedAt, deleted, nil
	}
	larg.ForceReload = true
	upak, _, err = u.loadWithInfo(larg, nil, nil, false)
	if err != nil {
		return false, nil, false, err
	}
	found, revokedAt, deleted = CheckKID(upak, kid)
	return found, revokedAt, deleted, nil
}

func (u *CachedUPAKLoader) LoadUserPlusKeys(ctx context.Context, uid keybase1.UID, pollForKID keybase1.KID) (keybase1.UserPlusKeys, error) {
	var up keybase1.UserPlusKeys
	if uid.IsNil() {
		return up, NoUIDError{}
	}

	arg := NewLoadUserArg(u.G())
	arg.UID = uid
	arg.PublicKeyOptional = true
	arg.NetContext = ctx

	forcePollValues := []bool{false, true}

	for _, fp := range forcePollValues {

		arg.ForcePoll = fp

		upak, _, err := u.Load(arg)
		if err != nil {
			return up, err
		}
		if upak == nil {
			return up, fmt.Errorf("Nil user, nil error from LoadUser")
		}
		up = upak.Base
		if pollForKID.IsNil() || up.FindKID(pollForKID) != nil {
			break
		}

	}
	return up, nil
}

// LoadKeyV2 looks through all incarnations for the user and returns the incarnation with the given
// KID, as well as the Key data associated with that KID. It picks the latest such
// incarnation if there are multiple.
func (u *CachedUPAKLoader) LoadKeyV2(ctx context.Context, uid keybase1.UID, kid keybase1.KID) (ret *keybase1.UserPlusKeysV2, key *keybase1.PublicKeyV2NaCl, linkMap map[keybase1.Seqno]keybase1.LinkID, err error) {
	defer u.G().CVTrace(ctx, VLog0, fmt.Sprintf("uid:%s,kid:%s", uid, kid), func() error { return err })()
	if uid.IsNil() {
		return nil, nil, nil, NoUIDError{}
	}

	arg := NewLoadUserArg(u.G())
	arg.UID = uid
	arg.PublicKeyOptional = true
	arg.NetContext = ctx

	forcePollValues := []bool{false, true}

	for _, fp := range forcePollValues {

		arg.ForcePoll = fp

		upak, _, err := u.LoadV2(arg)
		if err != nil {
			return nil, nil, nil, err
		}
		if upak == nil {
			return nil, nil, nil, fmt.Errorf("Nil user, nil error from LoadUser")
		}

		linkMap = upak.SeqnoLinkIDs
		ret, key = upak.FindKID(kid)
		if key != nil {
			u.G().VDL.CLogf(ctx, VLog0, "- found kid in UPAK: %v", *ret)
			break
		}
		ret = nil
	}
	return ret, key, linkMap, nil
}

func (u *CachedUPAKLoader) Invalidate(ctx context.Context, uid keybase1.UID) {

	u.G().VDL.CLogf(ctx, VLog0, "| CachedUPAKLoader#Invalidate(%s)", uid)

	if u.noCache {
		return
	}

	lock := u.locktab.AcquireOnName(ctx, u.G(), uid.String())
	defer lock.Release(ctx)

	u.removeMemCache(ctx, uid)

	err := u.G().LocalDb.Delete(culDBKeyV2(uid))
	if err != nil {
		u.G().Log.CWarningf(ctx, "Failed to remove %s from disk cache: %s", uid, err)
	}
	u.deleteV1UPAK(uid)
}

// Load the PublicKey for a user's device from the local cache, falling back to LoadUser, and cache the user.
// If the user exists but the device doesn't, will force a load in case the device is very new.
func (u *CachedUPAKLoader) LoadDeviceKey(ctx context.Context, uid keybase1.UID, deviceID keybase1.DeviceID) (upakv1 *keybase1.UserPlusAllKeys, deviceKey *keybase1.PublicKey, revoked *keybase1.RevokedKey, err error) {
	var info CachedUserLoadInfo
	larg := NewLoadUserByUIDArg(ctx, u.G(), uid)
	upakV2, _, err := u.loadWithInfo(larg, &info, nil, false)
	if err != nil {
		return nil, nil, nil, err
	}
	upakV1 := keybase1.UPAKFromUPKV2AI(*upakV2)

	deviceKey, revoked, err = u.extractDeviceKey(upakV1, deviceID)
	if err == nil {
		// Early success, return
		return &upakV1, deviceKey, revoked, err
	}

	// Try again with a forced load in case the device is very new.
	larg.ForcePoll = true
	upakV2, _, err = u.loadWithInfo(larg, nil, nil, false)
	if err != nil {
		return nil, nil, nil, err
	}
	upakV1 = keybase1.UPAKFromUPKV2AI(*upakV2)

	deviceKey, revoked, err = u.extractDeviceKey(upakV1, deviceID)
	return &upakV1, deviceKey, revoked, err
}

func (u *CachedUPAKLoader) LookupUsername(ctx context.Context, uid keybase1.UID) (NormalizedUsername, error) {
	var info CachedUserLoadInfo
	arg := NewLoadUserByUIDArg(ctx, u.G(), uid)
	arg.StaleOK = true
	var ret NormalizedUsername
	_, _, err := u.loadWithInfo(arg, &info, func(upak *keybase1.UserPlusKeysV2AllIncarnations) error {
		if upak == nil {
			return UserNotFoundError{UID: uid, Msg: "in CachedUPAKLoader"}
		}
		ret = NewNormalizedUsername(upak.Current.Username)
		return nil
	}, false)
	return ret, err
}

// LookupUID is a verified map of username -> UID. IT calls into the resolver, which gives un untrusted
// UID, but verifies with the UPAK loader that the mapping UID -> username is correct.
func (u *CachedUPAKLoader) LookupUID(ctx context.Context, un NormalizedUsername) (keybase1.UID, error) {
	rres := u.G().Resolver.Resolve(un.String())
	if err := rres.GetError(); err != nil {
		return keybase1.UID(""), err
	}
	un2, err := u.LookupUsername(ctx, rres.GetUID())
	if err != nil {
		return keybase1.UID(""), err
	}
	if !un.Eq(un2) {
		u.G().Log.CWarningf(ctx, "Unexpected mismatched usernames (uid=%s): %s != %s", rres.GetUID(), un.String(), un2.String())
		return keybase1.UID(""), NewBadUsernameError(un.String())
	}
	return rres.GetUID(), nil
}

func (u *CachedUPAKLoader) lookupUsernameAndDeviceWithInfo(ctx context.Context, uid keybase1.UID, did keybase1.DeviceID, info *CachedUserLoadInfo) (username NormalizedUsername, deviceName string, deviceType string, err error) {
	arg := NewLoadUserByUIDArg(ctx, u.G(), uid)

	// First iteration through, say it's OK to load a stale user. Note that the
	// mappings of UID to Username and DeviceID to DeviceName are immutable, so this
	// data can never be stale. However, our user might be out of date and lack the
	// mappings, so the second time through, we request a fresh object.
	staleOK := []bool{true, false}
	for _, b := range staleOK {
		arg.StaleOK = b
		found := false
		u.loadWithInfo(arg, info, func(upak *keybase1.UserPlusKeysV2AllIncarnations) error {
			if upak == nil {
				return nil
			}
			if pk := upak.FindDevice(did); pk != nil {
				username = NewNormalizedUsername(upak.Current.Username)
				deviceName = pk.DeviceDescription
				deviceType = pk.DeviceType
				found = true
			}
			return nil
		}, false)
		if found {
			return username, deviceName, deviceType, nil
		}
	}
	if err == nil {
		err = NotFoundError{fmt.Sprintf("UID/Device pair %s/%s not found", uid, did)}
	}
	return NormalizedUsername(""), "", "", err
}

func (u *CachedUPAKLoader) LookupUsernameAndDevice(ctx context.Context, uid keybase1.UID, did keybase1.DeviceID) (username NormalizedUsername, deviceName string, deviceType string, err error) {
	return u.lookupUsernameAndDeviceWithInfo(ctx, uid, did, nil)
}

func (u *CachedUPAKLoader) ListFollowedUIDs(uid keybase1.UID) ([]keybase1.UID, error) {
	arg := NewLoadUserByUIDArg(nil, u.G(), uid)
	upak, _, err := u.Load(arg)
	if err != nil {
		return nil, err
	}
	var ret []keybase1.UID
	for _, t := range upak.RemoteTracks {
		ret = append(ret, t.Uid)
	}
	return ret, nil
}

// v1 UPAKs are all legacy and need to be gradually cleaned from cache.
func (u *CachedUPAKLoader) deleteV1UPAK(uid keybase1.UID) {
	err := u.G().LocalDb.Delete(culDBKeyV1(uid))
	if err != nil {
		u.G().Log.Warning("Failed to remove %s v1 object from disk cache: %s", uid, err)
	}
}

func (u *CachedUPAKLoader) getMemCache(ctx context.Context, uid keybase1.UID) *keybase1.UserPlusKeysV2AllIncarnations {
	val, ok := u.cache.Get(uid)
	if !ok {
		return nil
	}

	upak, ok := val.(keybase1.UserPlusKeysV2AllIncarnations)
	if !ok {
		u.G().Log.CWarningf(ctx, "invalid type in upak cache: %T", val)
		return nil
	}

	return &upak
}

func (u *CachedUPAKLoader) putMemCache(ctx context.Context, uid keybase1.UID, upak keybase1.UserPlusKeysV2AllIncarnations) {
	u.cache.Add(uid, upak)
}

func (u *CachedUPAKLoader) removeMemCache(ctx context.Context, uid keybase1.UID) {
	u.cache.Remove(uid)
}

func (u *CachedUPAKLoader) purgeMemCache() {
	u.cache.Purge()
}
