/*
 * Copyright (c) 2003 Regents of The University of Michigan.
 * All Rights Reserved.  See COPYRIGHT.
 */

#include <CoreFoundation/CoreFoundation.h>
#include <Security/Authorization.h>

#include <sys/types.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <signal.h>
#include <stdio.h>
#include <unistd.h>

#include "selfrepair.h"

#define	AUTHMODE (S_ISUID|S_IRUSR|S_IXUSR | S_IRGRP|S_IXGRP | S_IROTH|S_IXOTH)

extern int	errno;
extern char	**environ;

    static int
exec_cmd( int argc, char *argv[] )
{
    int		status;

    argv++;

    switch ( fork()) {
    case 0:
	execve( argv[ 0 ], argv, ( char ** )environ );
	fprintf( stderr, "execve %s: %s\n", argv[ 0 ], strerror( errno ));
	fflush( stderr );
	_exit( 2 );

    case -1:
	fprintf( stderr, "fork failed: %s\n", strerror( errno ));
	exit( 2 );  

    default:
	break;
    }

    wait( &status );
    return( WEXITSTATUS( status ));
}

    static int
kill_pg( pid_t pgid )
{
    if ( killpg( pgid, SIGTERM ) < 0 ) {
	fprintf( stderr, "killpg %d: %s\n", pgid, strerror( errno ));
	return( errno );
    }
    return( 0 );
}

    int
main( int argc, char *argv[] )
{
    int				rc;
    const char	    		*rightname = "edu.umich.ihook.selfrepair";
    struct stat			st;
    mode_t			mode;
    AuthorizationRef		authref;
    AuthorizationExternalForm	extauth;
    AuthorizationItem   	right = { rightname, 0, NULL, 0 };
    AuthorizationRights 	rights = { 1, &right };
    AuthorizationFlags  	flags = kAuthorizationFlagDefaults |
				    kAuthorizationFlagInteractionAllowed |
				    kAuthorizationFlagExtendRights;

    if ( argc < 2 ) {
        fprintf( stderr, "Usage: %s args\n", argv[ 0 ] );
        exit( 1 );
    }

    if (( rc = read( 0, &extauth, sizeof( extauth ))) != sizeof( extauth )) {
	perror( "read AuthorizationExternalForm" );
	exit( 2 );
    }

    if ( AuthorizationCreateFromExternalForm( &extauth, &authref ) != 0 ) {
	exit( 2 );
    }

    if ( geteuid() != 0 ) {
	exit( selfrepair( argc, argv, rightname, authref, extauth ));
    }
    if ( setuid( 0 ) < 0 ) {
	fprintf( stderr, "setuid 0: %s\n", strerror( errno ));
	exit( errno );
    }

    if ( stat( argv[ 0 ], &st ) < 0 ) {
	fprintf( stderr, "stat %s: %s\n", argv[ 0 ], strerror( errno ));
	exit( errno );
    }
    mode = st.st_mode;
    if (( mode & S_IWUSR ) || ( mode & S_IWGRP ) || ( mode & S_IWOTH )) {
	if ( chmod( argv[ 0 ], AUTHMODE ) < 0 ) {
	    fprintf( stderr, "chmod 4555 %s: %s\n", argv[ 0 ],
			strerror( errno ));
	}
    }

    if ( AuthorizationCopyRights( authref, &rights,
	    kAuthorizationEmptyEnvironment,
	    flags, NULL ) != 0 ) {
	fprintf( stderr, "AuthCopyRights failed.\n" );
	exit( 1 );
    }

    if ( strcmp( argv[ 1 ], "--kill" ) == 0 && argc == 3 ) {
	pid_t 		pgid = strtol( argv[ 2 ], NULL, 10 );

	if ( pgid <= 1 ) {
	    fprintf( stderr, "Will not kill process 0 or 1\n" );
	    exit( 1 );
	}
	rc = kill_pg( pgid );
    } else {
	rc = exec_cmd( argc, argv );
    }

    return( rc );
}
