-- #hide
module HsShellScript.ProcErr
   (
   -- * Processes, Pipes
   -- ** Ways of Calling External Programs
     call, spawn, run
   , exec, execp, exece, execpe
   , echo
   , system_throw
   -- ** Redirecting Input and Output
   , (->-), (->>-), (=>-), (=>>-), (-<-)
   , (-&>-), (-&>>-)
   , err_to_out, out_to_err
   -- ** Building Pipes
   , (-|-), (=|-), (-|=), (=|=)
   , pipe_to, h_pipe_to
   , pipe_from, lazy_pipe_from, h_pipe_from
   , pipe_from2, lazy_pipe_from2, h_pipe_from2
   , pipes
   -- * Error Handling
   , errno
   , strerror
   , perror'
   , perror
   , abort
   , HsShellScript.ProcErr.failIO
   , exitcode
   , module Foreign.C.Error
   , throwErrno'
   , show_ioerror
   ) where

-- empty lines in order to work around c2hs bug
#include <string.h>

#include <limits.h>

#include <stdlib.h>

#include <unistd.h>

#include <stdio.h>

import Data.Dynamic
import Control.Exception
import Foreign
import Foreign.C
import Foreign.C.Error
import HsShellScript.Args
import HsShellScript.Shell
import IO
import IOExts
import Int
import Maybe
import Posix   -- from hslibs, _not_ System.Posix
import PosixUtil
import System
import qualified Control.Exception
import GHC.IOBase

infixr 2 -|-    -- left handed, stdout
infixr 2 =|-    -- left handed, stderr
infixl 2 -|=    -- right handed, stdout
infixl 2 =|=    -- right handed, stderr
infixl 3 ->-    -- write stdout to file
infixl 3 =>-    -- write stderr to file
infixl 3 ->>-   -- append stdout to file
infixl 3 =>>-   -- append stderr to file
infixl 3 -<-    -- read stdin from file or string
infixl 3 -&>-   -- write stdout and stderr to file
infixl 3 -&>>-  -- append stdout and stderr to file



-- The GHC library function doesn't fill in the file name.
executeFile' prog a b c =
   executeFile prog a b c
      `IO.catch` (\ioe -> ioError (ioe { ioe_filename = Just prog }))

-- |
-- Execute an IO action as a separate process, and wait for it to finish.
-- Report errors as exceptions.
--
-- The program forks a child process and performs the specified action.
-- Then it waits for the child process to finish. If it exits in any way
-- which indicates an error, the @ProcessStatus@ is thrown as a dynamic
-- exception. It can then be catched with the @catchDyn@ function from the
-- GHC @Exception@ library.
--
-- The @waitpid@ call is done blockingly. It waits if the program has been
-- stopped.
call :: IO ()  -- ^ action to execute as a child process
     -> IO ()
call io = do
    pid <- spawn io
    (Just ps) <- getProcessStatus True False pid
    if ps == Exited ExitSuccess
        then return ()
        else Control.Exception.throwDyn ps


-- |
-- Execute an IO action as a separate process, and continue without waiting
-- for it to finish.
--
-- The program forks a child process and performs the specified action.
-- The process ID is returned. It can be used in conjunction with
-- @getProcessStatus@.
spawn :: IO ()          -- ^ Action to execute as a child process.
      -> IO ProcessID   -- ^ Process ID of the new process.
spawn io = do
    pid <- fmap fromIntegral {#call fork#}
    if pid == 0 then io >> exitWith ExitSuccess
                else return pid

-- |
-- Run an external program. This starts an external program as a child
-- process, and waits for it to finish. The executable is searched via the
-- @PATH@.
--
-- If the program exits in a way which indicates an error, the
-- @ProcessStatus@ is thrown as a dynamic exception, which can be catched
-- with the @catchDyn@ function from the GHC @Exception@ library.
--
-- This is a frontend to 'call'.
--
-- See 'call'.
run :: FilePath                    -- name of the executable to run
    -> [String]                    -- command line arguments
    -> IO ()
run prog par =
   call (executeFile' prog True par Nothing)


-- | Print an action as a shell command, then perform it.
--
-- This is used with actions such as 'run', 'exec' or 'call'. For instance,
-- @echo run prog args@ is a variant of @run prog args@, which prints what
-- is being run, before running it.
--
-- See 'run', 'call', 'exec'.
echo :: ( FilePath -> [String] -> IO () )       -- ^ Action to perform
     -> FilePath                                -- ^ Name or path of the executable to run
     -> [String]                                -- ^ Command line arguments
     -> IO ()
echo action path args = do
   putStrLn (shell_command path args)
   hFlush stdout
   action path args


-- | Execute an external program. This replaces the running process. The
-- path is not searched, and the environment is not changed.
--
-- This is a shorthand for @executeFile' ... False ... Nothing@.
--
-- See man page @exec(3)@.
exec :: String          -- ^ Full path to the executable
     -> [String]        -- ^ Command line arguments
     -> IO ()
exec path args =
   executeFile' path False args Nothing


-- | Execute an external program. This replaces the running process. The
-- @PATH@ is searched, the environment is not changed.
--
-- This is a shorthand for @executeFile' ... True ... Nothing@.
--
-- See man page @exec(3)@.
execp :: String        -- ^ Name or path of the executable
      -> [String]      -- ^ Command line arguments
      -> IO ()
execp path args =
   executeFile' path True args Nothing


-- | Execute an external program. This replaces the running process. The
-- path is not searched. The environment is set to the specified value.
--
-- This is a shorthand for @executeFile' ... False ... (Just ...)@.
--
-- See man page @exec(3)@.
exece :: String                 -- ^ Full path to the executable
      -> [String]               -- ^ Command line arguments
      -> [(String,String)]      -- ^ Environment
      -> IO ()
exece path args env =
   executeFile' path False args (Just env)


-- | Execute an external program. This replaces the running process. The
-- @PATH@ is searched. The environment is set to the specified value.
--
-- This is a shorthand for @executeFile' ... True ... (Just ...)@.
--
-- See man page @exec(3)@.
execpe :: String                -- ^ Name or path of the executable
       -> [String]              -- ^ Command line arguments
       -> [(String,String)]     -- ^ Environment
       -> IO ()
execpe path args env =
   executeFile' path True args (Just env)


{-
data ProcessStatus = Exited ExitCode
                   | Terminated Signal
                   | Stopped Signal
		   deriving (Eq, Ord, Show)
-}
instance Typeable ProcessStatus where
   typeOf = const tyCon_ProcessStatus

tyCon_ProcessStatus = mkAppTy (mkTyCon "Posix.ProcessStatus") []



-- | Build left handed pipe of stdout.
--
-- \"@p -|- q@\" builds an IO action from the two IO actions @p@ and @q@.
-- @q@ is executed in an external process. The standard output of @p@ is sent
-- to the standard input of @q@ through a pipe. The result action consists
-- of forking off @q@ (connected with a pipe), and @p@.
--
-- The result action does /not/ run @p@ in a separate process. So, the pipe
-- itself can be seen as a modified action @p@, forking a connected @q@.
-- Normally, the pipe itself will be forked, too. The pipe is called \"left
-- handed\", because @p@ has this property, and not @q@.
--
-- /The exit code of q is silently ignored./ The process ID of the forked
-- copy of @q@ isn't returned to the caller, so it's lost.
-- 
-- Example: @call (exec \"\/usr\/bin\/foo\" [] -|- exec \"\/usr\/bin\/bar\" [])@
--
-- See 'call', '(=|-)', '(-|=)'.
(-|-) :: IO a   -- ^ Action which won't be forked
      -> IO b   -- ^ Action which will be forked and connected with a pipe
      -> IO a   -- ^ Result action
p -|- q = do
   (Just fd, _, _, _) <- pipe_fork_dup q True False False
   dupTo fd (intToFd 1)
   fdClose fd
   p


-- | Build left handed pipe of stderr.
--
-- \"@p =|- q@\" builds an IO action from the two IO actions @p@ and @q@.
-- @q@ is executed in an external process. The standard error output of @p@ is sent
-- to the standard input of @q@ through a pipe. The result action consists
-- of forking off @q@ (connected with a pipe), and @p@.
--
-- The result action does /not/ run @p@ in a separate process. So, the pipe
-- itself can be seen as a modified action @p@, forking a connected @q@.
-- Normally, the pipe itself will be forked, too. The pipe is called \"left
-- handed\", because @p@ has this property, and not @q@.
--
-- /The exit code of q is silently ignored./ The process ID of the forked
-- copy of @q@ isn't returned to the caller, so it's lost.
--
-- Example: @call (exec \"\/usr\/bin\/foo\" [] =|- exec \"\/usr\/bin\/bar\" [])@
--
-- See 'call', '(-|-)', '(-|=)'.
(=|-) :: IO a    -- ^ Action which won't be forked
      -> IO b    -- ^ Action which will be forked and connected with a pipe
      -> IO a    -- ^ Result action
p =|- q = do
   (Just fd, _, _, _) <- pipe_fork_dup q True False False
   dupTo fd (intToFd 2)
   fdClose fd
   p


-- | Build right handed pipe of stdout.
--
-- \"@p -|= q@\" builds an IO action from the two IO actions @p@ and @q@.
-- @p@ is executed in an external process. The standard output of @p@ is sent
-- to the standard input of @q@ through a pipe. The result action consists
-- of forking off @p@ (connected with a pipe), and @q@.
--
-- The result action does /not/ run @q@ in a separate process. So, the pipe
-- itself can be seen as a modified action @q@, forking a connected @p@.
-- Normally, the pipe itself will be forked, too. The pipe is called \"right
-- handed\", because @q@ has this property, and not @p@.
--
-- /The exit code of p is silently ignored./ The process ID of the forked
-- copy of @q@ isn't returned to the caller, so it's lost.
--
-- Example: @call (exec \"\/usr\/bin\/foo\" [] -|= exec \"\/usr\/bin\/bar\" [])@
--
-- See 'call', '(=|-)', '(=|=)'.
(-|=) :: IO a     -- ^ Action which will be forked and connected with a pipe
      -> IO b     -- ^ Action which won't be forked
      -> IO b     -- ^ Result action
p -|= q = do
   (_, Just fd, _, _) <- pipe_fork_dup p False True False
   dupTo fd (intToFd 0)
   fdClose fd
   q

-- | Build right handed pipe of stderr.
--
-- \"@p =|= q@\" builds an IO action from the two IO actions @p@ and @q@.
-- @p@ is executed in an external process. The standard error output of @p@ is sent
-- to the standard input of @q@ through a pipe. The result action consists
-- of forking off @p@ (connected with a pipe), and @q@.
--
-- The result action does /not/ run @q@ in a separate process. So, the pipe
-- itself can be seen as a modified action @q@, forking a connected @p@.
-- Normally, the pipe itself will be forked, too. The pipe is called \"right
-- handed\", because @q@ has this property, and not @p@.
--
-- /The exit code of p is silently ignored./ The process ID of the forked
-- copy of @q@ isn't returned to the caller, so it's lost.
--
-- Example: @call (exec \"\/usr\/bin\/foo\" [] =|= exec \"\/usr\/bin\/bar\" [])@
--
-- See 'call', '(=|-)', '(-|=)'.
(=|=) :: IO a     -- ^ Action which will be forked and connected with a pipe
      -> IO b     -- ^ Action which won't be forked
      -> IO b     -- ^ Result action
p =|= q = do
   (_, _, Just fd, _) <- pipe_fork_dup p False False True
   dupTo fd (intToFd 0)
   fdClose fd
   q


-- | Redirect stdout to a file. This modifies the specified action, such
-- that the standard output is redirected to a file. This will also affect
-- all subsequently executed actions, if the action isn't called in a
-- separate process. The file will be overwritten, if it already exists.
--
-- Example: @call (exec \"\/path\/to\/foo\" [] -\>- \"bar\")@
--
-- See 'call', 'run', '->>-', '=>-'.
(->-) :: IO a           -- ^ Action, whose output will be redirected
      -> FilePath       -- ^ File to redirect the output to
      -> IO a           -- ^ Result action
(->-) = write_redirect 1 False


-- | Redirect stdout to a file. This modifies the specified action, such
-- that the standard output is redirected to a file. This will also affect
-- all subsequently executed actions, if the action isn't called in a
-- separate process. If the file already exists, the output will be
-- appended.
--
-- Example: @call (exec \"\/path\/to\/foo\" [] -\>\>- \"bar\")@
--
-- See 'call', 'run', '(->-)', '(=>>-)'.
(->>-) :: IO a          -- ^ Action, whose output will be redirected
       -> FilePath      -- ^ File to redirect the output to
       -> IO a          -- ^ Result action
(->>-) = write_redirect 1 True


-- | Redirect stderr to a file. This modifies the specified action, such
-- that the standard error output is redirected to a file. This will also affect
-- all subsequently executed actions, if the action isn't called in a
-- separate process. The file will be overwritten, if it already exists.
--
-- Example: @call (exec \"\/path\/to\/foo\" [] =\>- \"\/dev\/null\")@
--
-- See 'call', 'run', '(->-)', '(=>>-)'.
(=>-) :: IO a           -- ^ Action, whose error output will be redirected
      -> FilePath       -- ^ File to redirect the error output to
      -> IO a           -- ^ Result action
(=>-) = write_redirect 2 False

-- | Redirect stderr to a file. This modifies the specified action, such
-- that the standard error output is redirected to a file. This will also affect
-- all subsequently executed actions, if the action isn't called in a
-- separate process. If the file already exists, the output will be
-- appended.
--
-- Example: @call (exec \"\/path\/to\/foo\" [] =\>\>- \"log\")@
--
-- See 'call', 'run', '(->>-)', '(=>-)'.
(=>>-) :: IO a          -- ^ Action, whose error output will be redirected
       -> FilePath      -- ^ File to redirect the error output to
       -> IO a          -- ^ Result action
(=>>-) = write_redirect 2 True


{- | Redirect both stdout and stderr to a file. This is equivalent to the shell's @&>@ operator. If the file already exists, it will be overwritten.
   This will also affect all subsequently executed actions, if the action isn't called in a separate process. 

   Example: @call (exec \"\/path\/to\/foo\" [] -&\>- \"log\")@

>(-&>-) io path = (err_to_out >> io) (->-) path 
-}
(-&>-) :: IO a          -- ^ Action, whose output and error output will be redirected
       -> FilePath      -- ^ File to redirect to
       -> IO a          -- ^ Result action
(-&>-) io path =
   (err_to_out >>  io) ->- path


{- | Redirect both stdout and stderr to a file. If the file already exists, the output will be appended. This will also affect
   all subsequently executed actions, if the action isn't called in a separate process. 

   Example: @call (exec \"\/path\/to\/foo\" [] -&\>\>- \"log\")@

>(-&>>-) io path = (err_to_out >> io) (->>-) path 
-}
(-&>>-) :: IO a         -- ^ Action, whose output and error output will be redirected
       -> FilePath      -- ^ File to redirect to
       -> IO a          -- ^ Result action
(-&>>-) io path =
   (err_to_out >> io) ->>- path


-- | Redirect stdin from a file. This modifies the specified action, such
-- that the standard input is read from a file. This will also affect
-- all subsequently executed actions, if the action isn't called in a
-- separate process.
--
-- Example: @call (exec \"\/path\/to\/foo\" [] -\<- \"bar\")@
--
-- See 'call', 'run', '(->-)', '(=>-)'.
(-<-) :: IO a
      -> FilePath
      -> IO a
io -<- path = do
   fd <- openFd path ReadOnly (Just stdFileMode) (OpenFileFlags False False False False False)
   dupTo fd (intToFd 0)
   io



-- Helper for ->-, =>-, ->>-, =>>-
write_redirect :: Int           -- file descriptor to redirect (1 or 2)
               -> Bool          -- whether to append to the file
               -> IO a          -- IO action
               -> FilePath      -- file to redirect to
               -> IO a
write_redirect fdnr append io path = do
   fd <- openFd path WriteOnly (Just stdFileMode) (OpenFileFlags append False False False (not append))
   dupTo fd (intToFd fdnr)
   io


{- | Send the error output of the current process to its standard output. This will affect all subsequent IO actions.

>err_to_out = dupTo (intToFd 1) (intToFd 2) >> return ()
-}
err_to_out :: IO ()
err_to_out = dupTo (intToFd 1) (intToFd 2) >> return ()


{- | Send the standard output of the current process to its error output. This will affect all subsequent IO actions.

>out_to_err = dupTo (intToFd 2) (intToFd 1) >> return ()
-}
out_to_err :: IO ()
out_to_err = dupTo (intToFd 2) (intToFd 1) >> return ()


-- Run an IO action as a new process, and optionally connect its
-- stdin, stdout and stderr via pipes.
pipe_fork_dup :: IO a                   -- Action to run in a new process.
              -> Bool                   -- make stdin pipe?
              -> Bool                   -- make stdout pipe?
              -> Bool                   -- make stderr pipe?
              -> IO ( Maybe Fd          -- Handle to the new process's stdin, if applicable.
                    , Maybe Fd          -- Handle to the new process's stdout, if applicable.
                    , Maybe Fd          -- Handle to the new process's stderr, if applicable.
                    , ProcessID
                    )
pipe_fork_dup io fd0 fd1 fd2 = do
    pipe0 <- pipe_if fd0
    pipe1 <- pipe_if fd1
    pipe2 <- pipe_if fd2
    pid <- fmap fromIntegral {#call fork#}
    if pid == 0 then do
           -- child
           dup_close pipe0 0 True
           dup_close pipe1 1 False
           dup_close pipe2 2 False
           io
           exitWith ExitSuccess
       else do
           -- parent
           p0 <- finish_pipe pipe0 True
           p1 <- finish_pipe pipe1 False
           p2 <- finish_pipe pipe2 False
           return (p0, p1, p2, pid)
  where
     -- Make a pipe, if applicable.
     pipe_if False = return Nothing
     pipe_if True  = fmap Just $ createPipe  -- Just (read,write)

     -- Child work after fork: connect a fd of the new process to the pipe.
     dup_close :: Maybe (Fd, Fd)        -- maybe the pipe
               -> Int                   -- which file descriptor to connect to the pipe
               -> Bool                  -- whether the fd is for reading
               -> IO ()
     dup_close Nothing _ _ =
         return ()
     dup_close m@(Just (readend,writeend)) dest read =
         do dupTo (if read then readend else writeend) (intToFd dest)
            fdClose readend
            fdClose writeend

     -- Parent work after fork: close surplus end of the pipe and make a handle from the other end.
     finish_pipe :: Maybe (Fd, Fd)      -- maybe the pipe
                      -> Bool           -- whether the fd is for reading
                      -> IO (Maybe Fd)
     finish_pipe Nothing _ =
         return Nothing
     finish_pipe (Just (readend,writeend)) read =
         do fdClose (if read then readend else writeend)
            return (Just (if read then writeend else readend))



-- | Run an IO action as a separate process, and pipe some text to its
-- standard input. The pipe is closed afterwards.
--
-- Example: @pipe_to \"blah\" $ exec \"\/usr\/bin\/foo\" [\"bar\"]@
--
-- See 'call', 'run', '-<-', 'h_pipe_to'.
pipe_to :: String       -- ^ Text to pipe
        -> IO a         -- ^ Action to run as a separate process, and to pipe to
        -> IO ()
pipe_to str io = do
   h <- h_pipe_to io
   hPutStr h str
   hClose h


-- | Run an IO action as a separate process, and connect to its standard input
-- with a pipe.
--
-- Example: @h \<- h_pipe_to $ exec \"\/usr\/bin\/foo\" [\"bar\"]@
--
-- See 'call', 'run', '-<-', 'pipe_to', 'pipe_from', 'pipe_from2'.
h_pipe_to :: IO a         -- ^ Action to run as a separate process, and to pipe to
          -> IO Handle
h_pipe_to io = do
   (Just fd, _, _, _) <- pipe_fork_dup io True False False
   h <- fdToHandle fd
   return h



-- | Run an IO action as a separate process, and connect to its standard output
-- with a pipe.
--
-- Example: @h \<- h_pipe_from $ exec \"\/usr\/bin\/foo\" [\"bar\"]@
--
-- See 'call', 'run', '-<-', 'pipe_to', 'pipe_from2'
h_pipe_from :: IO a         -- ^ Action to run as a separate process
            -> IO Handle
h_pipe_from io = do
   (_, Just fd, _, _) <- pipe_fork_dup io False True False
   h <- fdToHandle fd
   return h



-- | Run an IO action as a separate process, and lazily read its standard
-- output. This is like the backquote feature of shells. The output is read
-- lazily, as the returned string is evaluated.
--
-- The result is the process' output and a close action, which must be
-- called after the caller has finished reading. It closes the pipe
-- and waits for the child process to finish. If not all of the program's
-- output has been read yet, the operating system will send it a
-- @SIGPIPE@ signal, which causes the child to terminate, unless it
-- catches this signal. The close action returns the child process'
-- process status, which is either @Exited exitCode@ or
-- @Terminated signal@.
--
-- Example:
--
-- >do (txt,close) <- lazy_pipe_from $ exec "/usr/bin/foo" ["bar"]
-- >   ...
-- >   close
--
-- See 'call', 'run', '-<-', 'pipe_to', 'pipe_from2'
lazy_pipe_from :: IO a                          -- ^ Action to run as a separate process
               -> IO (String, IO ProcessStatus) -- ^ The action's lazy output and the close action
lazy_pipe_from io = do
   (_, Just fd, _, pid) <- pipe_fork_dup io False True False
   lazy_pipe_from0 io fd pid



-- | Run an IO action as a separate process, and lazily read its standard
-- error output. The output is read lazily, as the returned string is
-- evaluated.
--
-- The result is the process' error output and a close action, which must
-- be called after the caller has finished reading. It closes the pipe and
-- waits for the child process to finish. If not all of the program's
-- error output has been read yet, the operating system will send it a @SIGPIPE@
-- signal, which causes the child to terminate, unless it catches this
-- signal. The close action returns the child process' process status,
-- which is either @Exited exitCode@ or @Terminated signal@.
--
-- Example:
--
-- >do (err,close) \<- lazy_pipe_from2 $ exec "/usr/bin/foo" ["bar"]
-- >   ...
-- >   close
--
-- See 'call', 'run', '-<-', 'pipe_to', 'pipe_from'
lazy_pipe_from2 :: IO a                          -- ^ Action to run as a separate process
                -> IO (String, IO ProcessStatus) -- ^ The action's lazy error output and the close action
lazy_pipe_from2 io = do
   (_, _, Just fd, pid) <- pipe_fork_dup io False False True
   lazy_pipe_from0 io fd pid


-- Helper function for lazy_pipe_from and lazy_pipe_from2
lazy_pipe_from0 io fd pid = do
   h <- fdToHandle fd
   txt <- hGetContents h
   return ( txt
          , do hClose h
               (Just ps) <- getProcessStatus True True pid
               return ps
          )


-- | Run an IO action as a separate process, and read its standard
-- output immediately. This is like the backquote feature of shells.
--
-- If the process exits in any way which indicates an error, the
-- @ProcessStatus@ is thrown as a dynamic exception.
--
-- Example: @txt \<- pipe_from $ exec \"\/usr\/bin\/foo\" [\"bar\"]@
--
-- See 'call', 'run', '-<-', 'pipe_to', 'pipe_from2'
pipe_from :: IO a                          -- ^ Action to run as a separate process
          -> IO String          -- ^ The called program's standard output.
pipe_from io =
    pipe_from0 io lazy_pipe_from


-- | Run an IO action as a separate process, and read its standard
-- error output immediately.
--
-- If the process exits in any way which indicates an error, the
-- @ProcessStatus@ is thrown as a dynamic exception.
--
-- Example: @err \<- pipe_from $ exec \"\/usr\/bin\/foo\" [\"bar\"]@
--
-- See 'call', 'run', '-<-', 'pipe_to', 'pipe_from'
pipe_from2 :: IO a                          -- ^ Action to run as a separate process
           -> IO String                     -- ^ The called program's standard error output.
pipe_from2 io =
    pipe_from0 io lazy_pipe_from2


-- Helper function for pipe_from and pipe_from0.
pipe_from0 io lazy = do
    (txt, close) <- lazy io
    ps <- seq (length txt) close
    if ps == Exited ExitSuccess
       then return txt
       else Control.Exception.throwDyn ps



-- | Run an IO action as a separate process, and connect to its standard
-- error output with a pipe.
--
-- Example: @h \<- h_pipe_from2 $ exec \"\/usr\/bin\/foo\" [\"bar\"]@
--
-- See 'call', 'run', '-<-', 'pipe_to', 'pipe_from'.
h_pipe_from2 :: IO a         -- ^ Action to run as a separate process, and to connect with a pipe
             -> IO Handle
h_pipe_from2 io = do
   (_, _, Just fd, _) <- pipe_fork_dup io False False True
   h <- fdToHandle fd
   return h


-- | Run an IO action as a separate process, and optionally connect to its
-- standard input, its standard output and its standard error output with
-- pipes.
--
-- See 'pipe_from', 'pipe_from2', 'pipe_to'.
pipes :: IO a                   -- ^ Action to run in a new process
      -> Bool                   -- ^ Whether to make stdin pipe
      -> Bool                   -- ^ Whether to make stdout pipe
      -> Bool                   -- ^ Whether to make stderr pipe
      -> IO ( Maybe Handle
            , Maybe Handle
            , Maybe Handle
            , ProcessID
            )                   -- ^ Pipes to the new process's stdin, stdout and stderr, if applicable; and its process id.
pipes io p0 p1 p2 = do
   (mfd0, mfd1, mfd2, pid) <- pipe_fork_dup io p0 p1 p2
   h0 <- maybe_handle mfd0
   h1 <- maybe_handle mfd1
   h2 <- maybe_handle mfd2
   return (h0, h1, h2, pid)
   where maybe_handle Nothing = return Nothing
         maybe_handle (Just fd) = fmap Just $ fdToHandle fd



{- | Execute the supplied action. In case of an error, print a message and
exit.

An error is a dynamic exception, thrown using @throwDyn@ as a type which is
instance of @Typeable@. The type err is supposed to be a specific type used
for specific errors. The program is terminated with @exitFailure@.
-}
abort :: Typeable err
      => (err -> String)        -- ^ Error message generation function
      -> IO a                   -- ^ IO action to monitor
      -> IO a                   -- ^ Same action, but abort with error message in case of user exception
abort msgf io =
   catchDyn io
            (\errval -> hPutStrLn stderr (msgf errval) >> exitFailure)


-- | Generate an error message from an @errno@ value. This is the POSIX
-- @strerror@ system library function.
--
-- See the man page @strerror(3)@.
strerror :: Errno       -- ^ @errno@ value
         -> IO String   -- ^ Corresponding error message
strerror (Errno errno) = do
    peekCString ({#call pure strerror as foreign_strerror#} errno)


-- | Read the global system error number. This is the POSIX @errno@ value. This function is redundant. Use @Foreign.C.Error.getErrno@ instead.
errno :: IO Errno       -- ^ @errno@ value
errno = getErrno


-- | Print error message corresponding to the specified @errno@ error
-- number. This is like to the POSIX system library function @perror@.
--
-- See the man page @perror(3)@.
perror' :: Errno        -- ^ @errno@ error number
        -> String       -- ^ Text to precede the message, separated by \"@: @\"
        -> IO ()
perror' errno txt = do
   str <- strerror errno
   hPutStrLn stderr ((if txt == "" then "" else txt ++ ": ") ++ str)


-- | Print error message corresponding to the global @errno@ error
-- number. This is the same as the POSIX system library function @perror@.
--
-- See the man page @perror(3)@.
perror :: String        -- ^ Text to precede the message, separated by \"@: @\"
       -> IO ()
perror txt = do
   eno <- getErrno
   perror' eno txt


-- | Print a message to @stderr@ and exit with an exit code
-- indicating an error.
--
-- >failIO msg = hPutStrLn stderr msg >> exitFailure
failIO :: String -> IO a
failIO meld =
   hPutStrLn stderr meld >> exitFailure


instance Show Foreign.C.Error.Errno where
   show (Errno e) = show e


-- |Return the exit code, instead of throwing it as a dynamic exception.
--
-- This is used to modify the error reporting behaviour of an IO action. It
-- is used in conjunction with 'run' or 'call'. Normally, they throw a
-- process status, which indicates any error as a dynamic exception. After
-- @exitcode@ has been applied, only termination by a signal causes an exception
-- to be thrown. An exit code which indicates failure, instead is returned.
--
-- Example: @ec \<- exitcode $ run \"foo\" [\"bar\"]@
--
-- See 'run', 'call'.
exitcode :: IO ()             -- ^ Action to modify
         -> IO ExitCode       -- ^ Modified action
exitcode io =
   do io
      return ExitSuccess
   `Control.Exception.catchDyn`
      (\processstatus ->
          case processstatus of
             (Exited ec) -> return ec
             ps          -> Control.Exception.throwDyn ps)


-- |Create and throw an @IOError@ from the current @errno@ value, an optional handle and an optional file name.
--
-- This is an extended version of the @Foreign.C.Error.throwErrno@ function
-- from the GHC libraries, which additionally allows to specify a handle and a file
-- name to include in the @IOError@ thrown. 
--
-- See @Foreign.C.Error.throwErrno@, @Foreign.C.Error.errnoToIOError@.
throwErrno' :: String           -- ^ Description of the location where the error occurs in the program
            -> Maybe Handle     -- ^ Optional handle
            -> Maybe FilePath   -- ^ Optional file name (for failing operations on files)
            -> IO a
throwErrno' loc maybe_handle maybe_filename =
  do
    errno <- getErrno
    ioError (errnoToIOError loc errno maybe_handle maybe_filename)


-- |Convert an @IOError@ to a string.
--
-- There is an instance declaration of @IOError@ in @Show@ in the @GHC.IOBase@ library, but @show_ioerror@ produces a more readable, and more
-- complete, message.
show_ioerror :: IOError -> String 
show_ioerror ioe = 
   "IO-Error\n\
   \   Error type:   " ++ show (ioe_type ioe) ++ "\n\
   \   Error string: " ++ ioeGetErrorString ioe ++ "\n\
   \   Location:     " ++ ioe_location ioe ++ "\n\
   \   " ++ fn (ioeGetFileName ioe)
   where fn (Just n) = "File name:    " ++ shell_quote n
         fn Nothing  = "(no file name)"


{- |
   Call the shell to execute a command. In case of an error, throw the @ProcessStatus@ (such as @(Exited (ExitFailure ec))@) as a dynamic exception.
   This is like the Haskell standard library function @system@, except that error handling is brought in accordance with HsShellScript's scheme.

   @exitcode . system_throw@ is the same as the @system@ function, except that when the called shell is terminated or stopped by a signal, this still
   leads to the @ProcessStatus@ thrown as a dynamic exception. The Haskell library report says nothing about what happens in this case, when using the
   @system@ function.

>system_throw cmd = run "/bin/sh" ["-c", "--", cmd]
-}
-- This function should go to HsShellScript.Shell, but this would introduce a circular dependency.
system_throw :: String -> IO ()
system_throw cmd =
   run "/bin/sh" ["-c", "--", cmd]
