//////////////////////////////////////////////////////////////////////////////
//    Copyright 2004-2019, SenseGraphics AB
//
//    This file is part of H3D API.
//
//    H3D API is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    H3D API is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with H3D API; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//    A commercial license is also available. Please contact us at
//    www.sensegraphics.com for more information.
//
//
/// \file ComposedShader.cpp
/// \brief CPP file for ComposedShader.
///
//
//
//////////////////////////////////////////////////////////////////////////////

#include <H3D/ComposedShader.h>
#include <H3D/X3DTexture2DNode.h>
#include <H3D/X3DTexture3DNode.h>
#include <H3D/ShaderAtomicCounter.h>
#include <H3D/GlobalSettings.h>
#include <H3D/GraphicsHardwareInfo.h>

#include<fstream>
#include<sstream>
#include<algorithm>

using namespace H3D;
using namespace std;

bool ComposedShader::tessellation_support_checked= false;

map<string, GLhandleARB> ComposedShader::phandles_map;
map<GLhandleARB, int> ComposedShader::phandle_counts;

// Add this node to the H3DNodeDatabase system.
H3DNodeDatabase ComposedShader::database(
                                   "ComposedShader",
                                   &(newInstance<ComposedShader>),
                                   typeid( ComposedShader ),
                                   &X3DShaderNode::database );

namespace ComposedShaderInternals {
  FIELDDB_ELEMENT( ComposedShader, parts, INPUT_OUTPUT )
  FIELDDB_ELEMENT( ComposedShader, suppressUniformWarnings, INPUT_OUTPUT )
  FIELDDB_ELEMENT( ComposedShader, geometryInputType, INPUT_OUTPUT )
  FIELDDB_ELEMENT( ComposedShader, geometryOutputType, INPUT_OUTPUT )
  FIELDDB_ELEMENT( ComposedShader, geometryVerticesOut, INPUT_OUTPUT )
  FIELDDB_ELEMENT( ComposedShader, transparencyDetectMode, INPUT_OUTPUT )
  FIELDDB_ELEMENT( ComposedShader, transformFeedbackVaryings, INPUT_OUTPUT )
  FIELDDB_ELEMENT( ComposedShader, printShaderWarnings, INPUT_OUTPUT )
  FIELDDB_ELEMENT( ComposedShader, shaderConstants, INPUT_OUTPUT )
#ifdef EXPORT_SHADER
  FIELDDB_ELEMENT( ComposedShader, saveShadersToUrl, INPUT_OUTPUT )
#endif
}

ComposedShader::ComposedShader( Inst< DisplayList  > _displayList,
                                Inst< SFNode       > _metadata,
                                Inst< SFBool       > _isSelected ,
                                Inst< SFBool       > _isValid,
                                Inst< SFBool       > _activate,
                                Inst< SFString     > _language,
                                Inst< MFShaderPart > _parts,
                                Inst< SFBool       > _suppressUniformWarnings,
                                Inst< SFString     > _geometryInputType,
                                Inst< SFString     > _geometryOutputType,
                                Inst< SFInt32      > _geometryVerticesOut,
                                Inst< SFString     > _transparencyDetectMode,
                                Inst< MFString     > _transformFeedbackVaryings,
                                Inst< SFBool       > _printShaderWarnings,
                                Inst< SFShaderConstants > _shaderConstants
#ifdef EXPORT_SHADER
                                ,
                                Inst< UpdateSaveShadersToUrl > _saveShadersToUrl
#endif
                                ) :
  X3DShaderNode( _displayList, _metadata, _isSelected,
                 _isValid, _activate, _language),
  X3DProgrammableShaderObject( &database ),
  parts( _parts ),
  suppressUniformWarnings( _suppressUniformWarnings ),
  geometryInputType( _geometryInputType ),
  geometryOutputType( _geometryOutputType ),
  geometryVerticesOut( _geometryVerticesOut ),
  transparencyDetectMode( _transparencyDetectMode ),
  transformFeedbackVaryings ( _transformFeedbackVaryings ),
  shaderConstants(_shaderConstants),
  printShaderWarnings ( _printShaderWarnings ),
#ifdef EXPORT_SHADER
  saveShadersToUrl( _saveShadersToUrl ),
#endif
  program_handle( 0 ),
  setupDynamicRoutes( new SetupDynamicRoutes ),
  updateUniforms ( new UpdateUniforms ),
  debug_options_previous( NULL ),
  updateCache( new UpdateCache ),
  max_texture_in_shader( 0 ),
  max_image_in_shader( 0 ) {
  type_name = "ComposedShader";
  database.initFields( this );

  suppressUniformWarnings->setValue( false );
  printShaderWarnings->setValue( false );

  geometryInputType->addValidValue( "POINTS");
  geometryInputType->addValidValue( "LINES");
  geometryInputType->addValidValue( "TRIANGLES");
  geometryInputType->addValidValue( "TRIANGLES_ADJACENCY");
  geometryInputType->addValidValue( "LINES_ADJACENCY");
  geometryInputType->setValue( "TRIANGLES" );

  geometryOutputType->addValidValue( "POINTS" );
  geometryOutputType->addValidValue( "LINE_STRIP" );
  geometryOutputType->addValidValue( "TRIANGLE_STRIP" );

  geometryOutputType->setValue( "TRIANGLE_STRIP" );

  geometryVerticesOut->setValue( 64 );

  transparencyDetectMode->addValidValue( "AS_MATERIAL" );
  transparencyDetectMode->addValidValue( "TRANSPARENT" );
  transparencyDetectMode->addValidValue( "OPAQUE" );
  transparencyDetectMode->setValue( "AS_MATERIAL" );

  setupDynamicRoutes->setName( "setupDynamicRoutes" );
  setupDynamicRoutes->setOwner( this );

  updateUniforms->setName ( "updateUniforms" );
  updateUniforms->setOwner ( this );

  updateCache->setName( "updateCache" );
  updateCache->setOwner( this );

  activate->route( displayList, id );
  parts->route( displayList, id );
  setupDynamicRoutes->route( displayList );
  transformFeedbackVaryings->route ( displayList, id );
  shaderConstants->route( displayList, id );

  // need to update uniform values if shader is re-linked
  // displayList->route ( updateUniforms );
  activate->route( updateUniforms, id );
}

ComposedShader::~ComposedShader() {
  
  // Decrease the count for the program handle
  if( phandle_counts.find( program_handle ) != phandle_counts.end() ) {
    --(phandle_counts[program_handle]);
    
    // Delete it when no instance is using it after detaching and deleting the shaders parts
    if( phandle_counts[program_handle] == 0 ) {
      
      for( vector< GLhandleARB >::iterator i = current_shaders.begin();
           i != current_shaders.end(); ++i ) {
        glDetachObjectARB( program_handle, *i );
        glDeleteObjectARB( *i );
      }
      current_shaders.clear();
      glDeleteObjectARB( program_handle );

      // Clean up both the handles and the count maps
      phandle_counts.erase( program_handle );
      for( std::map<string, GLhandleARB>::iterator it = phandles_map.begin(); it != phandles_map.end(); ++it ) {
        if( it->second == program_handle ) {
          phandles_map.erase( it );
          break;
        }
      }
    }
  } else if( program_handle ) {
    // If no instance is using the program handle, make sure it's deleted anyway
    glDeleteObjectARB( program_handle );
  }
}

bool ComposedShader::shader_support_checked = false;

bool ComposedShader::isTransparent( X3DMaterialNode *material ) {
  const string &mode = transparencyDetectMode->getValue();
  if( mode == "TRANSPARENT" ) {
    return true;
  } else if( mode == "OPAQUE" ) {
    return false;
  } else {
    if( mode != "AS_MATERIAL") {
      Console(LogLevel::Error) << "Warning: Invalid transparencyDetectMode \"" << mode << "\" in ComposedShader."
     << " Must be one of \"AS_MATERIAL\", \"TRANSPARENT\" or \"OPAQUE\"." << endl;
    }
    if( material ) return material->isTransparent();
  }
  return false;
}

bool ComposedShader::addField( const string &_name,
                               const Field::AccessType &access, Field *field ){
  // composed shader added field can be node field such as texture
  // or just field such as SFFloat, SFBool
  // as field name should be the same as they are in shaders, name will be used
  // as the id for field.
  // For shader program, the name in dynamic field will match the field in shader

  if( !field || uniformFields.find( _name )!=uniformFields.end() ) {
    // different shader part may add the same field with same name
    // as uniformFields is a global dataset for the shader program
      return false;
    }
  // the field being added have a unique name never being used before in this shader
  bool success = X3DProgrammableShaderObject::addField( _name, access, field  );
  if( success ) {
    // insert a new entry for the newly added field

    H3D::Shaders::UniformInfo ui = { field, 0 };
    uniformFields.insert( std::pair< string, H3D::Shaders::UniformInfo >( _name , ui ) );
    // if inserted node is a SFNode, or MFNode, need to handle the actul node inside
    // the field specially
    SFNode * sf_node_field = dynamic_cast< SFNode * >( field );
    MFNode * mf_node_field = dynamic_cast< MFNode * >( field );
    if( sf_node_field || mf_node_field ) {
      field->route( setupDynamicRoutes, id );
    }
    // no need to route field to displayList all the time, the routing will only be
    // handle when caching is on
    field->route( updateUniforms );
  }else{
    // can not add same field twice,even though it has different name.
    // This is the limit of h3d dynamic field object.
    Console(LogLevel::Error)<<"Warning: Failed to add field: "<<_name<<" to shader "<< getName()
      <<", either current node is invalid or the added field is already in the node database!"
      <<endl;
  }
  return success;
}

bool ComposedShader::removeField ( const string& _name ) {
  Field* f= getField ( _name );
  if ( !f ) return false;

  // Remove from uniformFields map
  for( UniformFieldMap::iterator i = uniformFields.begin(); i != uniformFields.end(); ++i  ) {
    const string &name_tmp = i->first;
    if( name_tmp == _name ) {
      uniformFields.erase ( i );
      break;
    }
  }

  // Fields containing nodes require special handling
  SFNode* sf_node_field = dynamic_cast< SFNode* >( f );
  MFNode* mf_node_field = dynamic_cast< MFNode* >( f );
  if( sf_node_field || mf_node_field ) {
    f->unroute( setupDynamicRoutes );

    // Unroute the displayList of all nodes contained from our displayList
    map< Field*, NodeVector >::iterator in_map =
      setupDynamicRoutes->fields_to_nodes.find( f );

    if ( in_map != setupDynamicRoutes->fields_to_nodes.end() ) {
      const NodeVector& node_vector= (*in_map).second;
      for( unsigned int i = 0; i < node_vector.size(); ++i ) {
        H3DDisplayListObject* hdln =
          dynamic_cast< H3DDisplayListObject * >( node_vector[i] );
        if( hdln )
          hdln->displayList->unroute( displayList );
      }
      setupDynamicRoutes->fields_to_nodes.erase( in_map );
    }
    H3DSingleTextureNode *tex = dynamic_cast< H3DSingleTextureNode* >(f);
    if( tex ) {
      shader_textures.remove(tex);
    }
  } else {
    f->unroute( displayList );
  }
  f->unroute( updateUniforms );

  return X3DProgrammableShaderObject::removeField ( _name );
}

GLbitfield ComposedShader::getAffectedGLAttribs() {
  GLbitfield res = X3DShaderNode::getAffectedGLAttribs();
  if( GLEW_ARB_shader_objects ) {
    res = res | Shaders::getAffectedGLAttribs( this );
  }
  res = res | GL_COLOR_BUFFER_BIT;
  return res;
}

void ComposedShader::preRender() {
  if( GLEW_ARB_shader_objects ) {
    glUseProgramObjectARB( program_handle );
    Shaders::preRenderTextures( this, &max_texture_in_shader );
    X3DShaderNode::preRender();
  }

  glEnable( GL_BLEND );
  glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
}

void ComposedShader::postRender() {
  if( GLEW_ARB_shader_objects ) {
    glUseProgramObjectARB( 0 );
    Shaders::postRenderTextures( this, &max_texture_in_shader );
    X3DShaderNode::postRender();
  }
}

void ComposedShader::traverseSG ( TraverseInfo& ti ) {
  X3DShaderNode::traverseSG ( ti );
  Node* n;
  for( H3DDynamicFieldsObject::field_iterator f = this->firstField();f != this->endField(); ++f )
  {
    X3DTypes::X3DType x3d_type = (*f)->getX3DType();
    if(x3d_type==X3DTypes::SFNODE){
      n = static_cast<SFNode*>(*f)->getValue();
      if( ShaderAtomicCounter* sac = dynamic_cast<ShaderAtomicCounter*>(n) ) {
        sac->traverseSG(ti);
      }
    }
  }

  // Will be set to true if tessellation shader is present
  static bool require_patches;

  // Look for tessellation shader
  require_patches = false;
  for( MFShaderPart::const_iterator i = parts->begin();
        i != parts->end(); ++i ) {
    ShaderPart* shader_part= static_cast< ShaderPart * >(*i);
    if ( shader_part->type->getValue() == "TESS_CONTROL" || shader_part->type->getValue() == "TESS_EVALUATION" ) {
      require_patches= true;
      break;
    }
  }

  // Let the geometry know  if it should render patches
  if( require_patches ) {
    if ( GLEW_ARB_tessellation_shader ) {
      ti.setUserData( "shaderRequiresPatches", &require_patches );
    } else if ( !tessellation_support_checked ) {
      Console(LogLevel::Error) << "Your graphic card driver does not support tessellation shaders."
                 << " The ComposedShader node " << getName() << " will not be rendered correctly!" << endl;
      tessellation_support_checked = true;
    }
  }

  if( updateCache->getRoutesIn().size()==0 ) {
    // if initially udpateCache is not routed in from useCaching, need to check
    // if graphic options is dynamically added, if added route in its useCaching
    GraphicsOptions *options = NULL;
    GlobalSettings *default_settings = GlobalSettings::getActive();
    if( default_settings&&default_settings->optionNodesUpdated() ) {
      default_settings->getOptionNode( options );
      if( options ) {
        options->useCaching->route( updateCache );
      }
    }
  }
  updateCache->upToDate();
}

// The ComposedShader is modified to use 1 instance of different program_handlers
// that share the same set of shader parts. However, a new program_handle object
// will be created when the Shader is re-activated (regardsless)


void ComposedShader::render() {
  if( !GLEW_ARB_shader_objects ) {
    if( !shader_support_checked ) {
      Console(LogLevel::Error) << "Your graphic card driver does not support shader objects( ARB_shader_objects) so you cannot"
                 << " use the ComposedShader node. Shader will be disabled" << endl;
      shader_support_checked = true;
    }
    if( isValid->getValue() ) isValid->setValue( false, id );
  } else {
    bool all_parts_valid = true;

    // compile all shader parts
    bool re_link= false;
    for( MFShaderPart::const_iterator i = parts->begin();
         i != parts->end(); ++i ) {
      ShaderPart* s= static_cast< ShaderPart * >(*i);
      re_link|= !s->isCompiled();
      if( s->compileShader() == 0 ) {
        all_parts_valid = false;
      }
    }

    if ( all_parts_valid && re_link ) {
      activate->setValue ( true );
    }

    if( isValid->getValue() != all_parts_valid )
      isValid->setValue( all_parts_valid, id );

    if( all_parts_valid )
    {
      // if the first time we run the shader and there is some part attached
      if ( !program_handle && parts->size() ) {
        std::string key = ComposedShader::genKeyFromShader( this );

        if (phandles_map.find(key) != phandles_map.end()) {
          // if a handle found, use that!
          program_handle = phandles_map[key];
          ++(phandle_counts[program_handle]);
          //std::cout<< getName() << " use program handle " << program_handle
          // << std::endl;
          glUseProgramObjectARB( program_handle );
        } else {
          // if not, create one, link to shaderparts
          GLhandleARB h = createHandle( this );
          if (h != 0) {
            // use that handle
            program_handle = h;
            glUseProgramObjectARB( h );
            phandle_counts[h] = 1;
            phandles_map[key] = h;
            // register shader objects
            for ( MFShaderPart::const_iterator i = parts->begin();
                  i != parts->end(); ++i ) {
              current_shaders.push_back(
                static_cast< ShaderPart * >(*i)->getShaderHandle() );
            }
          }
        }
      }

      // if a TRUE event has been sent to the activate field we
      // relink the program (without looking up)
      else if( activateMonitor->hasCausedEvent(activate)&&activate->getValue(id) ) {
        // deallocate old instance if not used anywhere
        if (phandle_counts.find(program_handle) != phandle_counts.end()) {
          --(phandle_counts[program_handle]);
          if (phandle_counts[program_handle] == 0) {
            // detach the old shaders from the program
            for( vector< GLhandleARB >::iterator i = current_shaders.begin();
              i != current_shaders.end(); ++i ) {
              glDetachObjectARB( program_handle, *i );
            }
            current_shaders.clear();
            // delete object
            //std::cout<< this->getName() << " remove phandle "
            //         << program_handle << std::endl;
            glDeleteObjectARB( program_handle );

            // Remove deleted handle from cache
            phandle_counts.erase(program_handle);
            for (std::map<string, GLhandleARB>::iterator it = phandles_map.begin(); it != phandles_map.end(); ++it) {
              if (it->second == program_handle) {
                phandles_map.erase(it);
                break;
              }
            }
          }
        } else {
          // if not, this is a floating program handle. delete it anyway
          //std::cout<< this->getName() << " remove phandle " << program_handle
          //         << std::endl;
          glDeleteObjectARB( program_handle );
        }

        // we can't use the old instance (because that forces other
        // shaders to re-link)

        GLhandleARB h = createHandle(this);
        if (h != 0) {
          program_handle = h;
          glUseProgramObjectARB( h );
          // register shader objects
          for ( MFShaderPart::const_iterator i = parts->begin();
                i != parts->end(); ++i ) {
            current_shaders.push_back(
              static_cast< ShaderPart * >(*i)->getShaderHandle() );
          }
        }
      }
    }

    if( program_handle ) {
      Shaders::renderTextures( this, &max_texture_in_shader, &max_image_in_shader );

      Shaders::renderShaderResources( this, program_handle );

      // Lazily update uniform values, i.e., only those that have changed

      updateUniforms->upToDate();

    }
  }
  activateMonitor->upToDate();
}

// check if any existing program handle using the same set of ShaderParts
// so that we can reuse the program handle
// Return the handle if found, zero if not found
std::string ComposedShader::genKeyFromShader(ComposedShader* shader)
{
  // build the key as string. we use simple key building method
  // by making a string of all the shaderParts' handles combined.
  vector<GLhandleARB> keys;
  keys.reserve( shader->parts->size() );
  for( MFShaderPart::const_iterator i = shader->parts->begin();
       i != shader->parts->end(); ++i ) {
    GLhandleARB handle = static_cast< ShaderPart * >(*i)->getShaderHandle();
    keys.push_back( handle );
  }
  sort( keys.begin(), keys.end() );
  stringstream ss;
  for( unsigned int i = 0; i < keys.size(); ++i ) ss << keys[i] << "_";
  return ss.str();
}


// create handle and link to shaderparts. return 0 if failed.
// Preclusion: parts > 0
GLhandleARB ComposedShader::createHandle(ComposedShader* shader) {
  if ( !shader->parts->size() )
    return 0;

  GLhandleARB program_handle = glCreateProgramObjectARB();

  // add the shaders to program
  for ( MFShaderPart::const_iterator i = shader->parts->begin();
        i != shader->parts->end(); ++i ) {
    GLhandleARB handle = static_cast< ShaderPart * >(*i)->getShaderHandle();
    glAttachObjectARB( program_handle, handle );
  }

  // set geometry shader values
  shader->setGeometryShaderParameters( program_handle );

  // Add transform feedback varyings before program is linked
  if ( !shader->transformFeedbackVaryings->empty() ) {
    std::vector<const GLchar*> varyings ( shader->transformFeedbackVaryings->size() );
    for ( size_t i= 0; i < shader->transformFeedbackVaryings->size(); ++i ) {
      varyings[i]= shader->transformFeedbackVaryings->getValueByIndex ( i ).c_str();
    }
    glTransformFeedbackVaryings(program_handle, static_cast<GLsizei>(varyings.size()), &varyings[0], GL_INTERLEAVED_ATTRIBS);
  }

  // link shader program
  glLinkProgramARB( program_handle );
  GLint link_success;
  glGetObjectParameterivARB( program_handle, GL_OBJECT_LINK_STATUS_ARB,
                             &link_success );
  int print_error = 0;
  if( link_success == GL_FALSE ) print_error = 1;
  else if( shader->printShaderLog() ) print_error = 2;
  if( print_error != 0 ) {
    // linking failed, print error message
    GLint nr_characters;
    glGetObjectParameterivARB( program_handle, GL_OBJECT_INFO_LOG_LENGTH_ARB,
                               &nr_characters );
    if( nr_characters > 1 ) {
      GLcharARB *log = new GLcharARB[nr_characters];
      glGetInfoLogARB( program_handle, nr_characters, NULL, log );

	 if(print_error == 2) {
        Console(LogLevel::Error) << "Warning: Error while linking shader parts in \""
          << const_cast<ComposedShader&>(*shader).getName() << "\" node. "
          << endl << log << endl;
      } else {
        Console(LogLevel::Warning) << "Warning: While linking shader parts in \""
          << const_cast<ComposedShader&>(*shader).getName() << "\" node. "
          << endl << log << endl;
      }

      if( print_error == 1 ) {
        glDeleteObjectARB( program_handle );
        program_handle = 0;
      }
      delete [] log;
    }
  } else {
#if defined( H3D_WINDOWS )
    if (char *buffer = getenv("H3D_EXTRACT_COMPILED_SHADER_STR")) {
      if( strcmp(buffer, "TRUE" )==0 ) {
        // extract shader
        // create a folder to gather all the shaders
        if( CreateDirectory("extractedShaders",NULL)||ERROR_ALREADY_EXISTS == GetLastError() ) {
          stringstream shader_strings_stream;
          for( NodeVector::const_iterator it = shader->parts->getValue().begin();
            it != shader->parts->getValue().end(); ++it ) {
            ShaderPart* sp = dynamic_cast<ShaderPart*>(*it);
            if( sp ) {
              string ss = sp->shaderString->getValue();
              shader_strings_stream << "H3D_" << sp->type->getValue() << "_TAG " << ss.length() << "\n" << ss<<"\n";
            }
          }
          // save shader string to a file
          stringstream file_name;
          hash<string> str_hash;
          file_name << "./extractedShaders/" << str_hash(string( shader_strings_stream.str() ));
          ifstream f( file_name.str() );
          if( !f.good() ) {
            // only recreate extracted shader file when original file either does not exist
            // or the file somehow cannot be opened
            // use binary mode to avoid newline get converted as different type of newlines 
            // will have different length and to maintain length is critical because length is stored 
            // in the file and will be used to extract different shader part strings
            f.close();
            ofstream outFile( file_name.str(), ofstream::binary );
            outFile << shader_strings_stream.str();
            outFile.close();
          }
        } else {
          Console( LogLevel::Error ) << "Error: cannot extract compiled shader"<<
            " since the extractedShaders folder does not exist and cannot be created!" << std::endl;
        }
      }
    }
#endif
  }

  //std::cout<< const_cast<ComposedShader&>(*shader).getName()
  //  << " created program handle " << program_handle << std::endl;


  return program_handle;
}


void ComposedShader::setGeometryShaderParameters( GLenum _program_handle ) {
  if( GLEW_EXT_geometry_shader4 ) {
    // Setting input type.
    const string &input_type = geometryInputType->getValue();
    if( input_type == "POINTS" ) {
      glProgramParameteriEXT(_program_handle,
                             GL_GEOMETRY_INPUT_TYPE_EXT, GL_POINTS);
    } else if( input_type == "LINES" ) {
      glProgramParameteriEXT(_program_handle,
                             GL_GEOMETRY_INPUT_TYPE_EXT,GL_LINES);
    } else if( input_type == "TRIANGLES" ) {
      glProgramParameteriEXT(_program_handle,
                             GL_GEOMETRY_INPUT_TYPE_EXT, GL_TRIANGLES);
    } else if( input_type == "LINES_ADJACENCY" ) {
      glProgramParameteriEXT(_program_handle,
                             GL_GEOMETRY_INPUT_TYPE_EXT,
                             GL_LINES_ADJACENCY_EXT );
    } else if( input_type == "TRIANGLES_ADJACENCY" ) {
      glProgramParameteriEXT(_program_handle,
                             GL_GEOMETRY_INPUT_TYPE_EXT,
                             GL_TRIANGLES_ADJACENCY_EXT );
    } else {
      Console(LogLevel::Error) << "Invalid geometryInputType \"" << input_type
                 << "\" in ComposedShader. Using \"TRIANGLES\" instead." << endl;
      glProgramParameteriEXT(_program_handle,
                             GL_GEOMETRY_INPUT_TYPE_EXT, 4/*GL_TRIANGLES*/);
    }

    // Setting output type.
    const string &output_type = geometryOutputType->getValue();
    if( output_type == "POINTS" ) {
      glProgramParameteriEXT(_program_handle,
                             GL_GEOMETRY_OUTPUT_TYPE_EXT, GL_POINTS);
    } else if( output_type == "LINE_STRIP" ) {
      glProgramParameteriEXT(_program_handle,
                             GL_GEOMETRY_OUTPUT_TYPE_EXT, GL_LINE_STRIP );
    } else if( output_type == "TRIANGLE_STRIP" ) {
      glProgramParameteriEXT(_program_handle,
                             GL_GEOMETRY_OUTPUT_TYPE_EXT,
                             GL_TRIANGLE_STRIP );
    } else {
      Console(LogLevel::Error) << "Invalid geometryOutputType \"" << output_type
                 << "\" in ComposedShader. Using \"TRIANGLE_STRIP\" instead."
                 << endl;
      glProgramParameteriEXT(_program_handle,
                             GL_GEOMETRY_OUTPUT_TYPE_EXT, 4/*GL_TRIANGLES*/);
    }

    int max_output_vertices;
     glGetIntegerv(GL_MAX_GEOMETRY_OUTPUT_VERTICES_EXT, &max_output_vertices);

    int nr_output_vertices = geometryVerticesOut->getValue();

    if( nr_output_vertices > max_output_vertices ) {
      Console(LogLevel::Error) << "Invalid geomtryVerticesOut value " << nr_output_vertices
                   << " in ComposedShader. Hardware supports a maximum of "
                   << max_output_vertices << "." << endl;
      nr_output_vertices = max_output_vertices;
    }

    // number of output vertices for geometry shader.
    glProgramParameteriEXT(_program_handle,
                           GL_GEOMETRY_VERTICES_OUT_EXT,
                           nr_output_vertices );

  }
}

void ComposedShader::SetupDynamicRoutes::update() {
  AutoUpdate< Field >::update();
  ComposedShader * cs = static_cast< ComposedShader * >( getOwner() );
  SFNode * sf_node_field = dynamic_cast< SFNode * >( event.ptr );
  MFNode * mf_node_field = dynamic_cast< MFNode * >( event.ptr );

  map< Field *, NodeVector >::iterator in_map =
    fields_to_nodes.find( event.ptr );

  // Start by removing the entry in fields_to_nodes map. Since it might be
  // that the node added is not the same as the previous node, and it might
  // also be so that the new node does not inherit from H3DDisplayListObject.
  if( in_map != fields_to_nodes.end() ) {
    const NodeVector &node_vector = (*in_map).second;
    for( unsigned int i = 0; i < node_vector.size(); ++i ) {
      H3DDisplayListObject *hdln =
        dynamic_cast< H3DDisplayListObject * >( node_vector[i] );
      if( hdln ){
        hdln->displayList->unroute( cs->displayList );
      }


      H3DSingleTextureNode *tex = dynamic_cast< H3DSingleTextureNode* >(node_vector[i]);
      if( tex ) {
        cs->shader_textures.remove(tex);
      }
    }
    fields_to_nodes.erase( in_map );
  }

  if( sf_node_field ) {
    // Setup route for the node contained in sf_node_field.
    // Add entry to map to remove later.
    Node * n = sf_node_field->getValue();
    H3DDisplayListObject *hdln =
      dynamic_cast< H3DDisplayListObject * >( n );
    if( hdln ) {
      hdln->displayList->route( cs->displayList, cs->id );
      cs->activate->setValue( true );
      NodeVector tmp_node_vector;
      tmp_node_vector.push_back( n );
      fields_to_nodes[ event.ptr ] = tmp_node_vector;
      H3DSingleTextureNode *tex = dynamic_cast< H3DSingleTextureNode* >(n);
      if( tex ) {
        cs->shader_textures.push_back(tex);
      }
    }
  } else if( mf_node_field ) {
    // Setup routes for all nodes contained in mf_node_field.
    // Add entry to map to remove later.
    const NodeVector &node_vector = mf_node_field->getValue();
    NodeVector tmp_node_vector;
    for( unsigned int i = 0; i < node_vector.size(); ++i ) {
      H3DDisplayListObject *hdln =
        dynamic_cast< H3DDisplayListObject * >( node_vector[i] );
      if( hdln ) {
        hdln->displayList->route( cs->displayList, cs->id );
        cs->activate->setValue( true );
        tmp_node_vector.push_back( node_vector[i] );
        H3DSingleTextureNode *tex = dynamic_cast< H3DSingleTextureNode* >(node_vector[i]);
        if( tex ) {
          cs->shader_textures.push_back(tex);
        }
      }
    }
    if( !tmp_node_vector.empty() )
      fields_to_nodes[ event.ptr ] = tmp_node_vector;
  }
}

void ComposedShader::UpdateUniforms::update() {
  ComposedShader* node= static_cast<ComposedShader*>(getOwner());
  bool update_all= hasCausedEvent ( node->activate );
  if( update_all ) { // program re-linked, need to update all uniform
    // update the uniform location information in unifromFields
    UniformFieldMap::iterator it;
    for( it = node->uniformFields.begin(); it!= node->uniformFields.end(); ++it  ) {
      const string &_name = it->first;
      GLint location = glGetUniformLocationARB( node->program_handle,
        _name.c_str() );
      it->second.location = location;
      if( !Shaders::setGLSLUniformVariableValue( node->program_handle, it->second.field, &(it->second), true /* force update */ )
        && !node->suppressUniformWarnings->getValue() ) {
        Console(LogLevel::Warning) << "Warning: Uniform variable \"" << it->first
          << "\" not defined in shader source or field is of unsupported field type of the ShaderPart nodes "
          << "in the node \"" << node->getName() << "\"" << endl;
      }
    }
    // all fields are updated, clear the event to finish the update and return.
    EventCollectingField < Field >::update();
    return;
  }

  // no need to update all, check field one by one to update the one needs to be updated
  UniformFieldMap::iterator it;
  for( it = node->uniformFields.begin(); it!= node->uniformFields.end(); ++it ) {
    Field* current_field = it->second.field;
    if( hasCausedEvent( current_field ) ) {// current_field update since last time
      //if( current_field->getTypeName()=="SFUniform" ) {
      //  // this is a SFUniform value, check if its value actually changed
      current_field->upToDate();
      // within setGLSLUniformVariableValue, check if the updated value
      // is the same as before to decide whether to reload uniform value to GPU
      if( !Shaders::setGLSLUniformVariableValue( node->program_handle,
        it->second.field, &it->second ) &&
        !node->suppressUniformWarnings->getValue() ) {
          Console(LogLevel::Warning) << "Warning: Uniform variable \"" << it->first
            << "\" not defined in shader source or field is of unsupported field type of the ShaderPart nodes "
            << "in the node \"" << node->getName() << "\"" << endl;
      }
    }
    // It is not the current_field caused the update event, skip uniform update
  }
  EventCollectingField < Field >::update();
}


void ComposedShader::MFShaderPart::onAdd(Node *n) {
  ComposedShader::MFShaderPartBase::onAdd(n);
  ComposedShader* owner = static_cast<ComposedShader*>(getOwner());
  ShaderPart* part = dynamic_cast<ShaderPart*>(n);
  if(part) {
    part->setParentComposedShader(owner);
  }
}


void ComposedShader::MFShaderPart::onRemove(Node *n) {
  ShaderPart* part = dynamic_cast<ShaderPart*>(n);
  if(part) {
    part->setParentComposedShader(NULL);
  }
  ComposedShader::MFShaderPartBase::onRemove(n);
}

void ComposedShader::SFShaderConstants::onAdd( Node *n ) {
  ComposedShader::SFShaderConstantsBase::onAdd( n );
  ShaderConstants* scs = dynamic_cast<ShaderConstants*>(n);
  StereoInfo* si = StereoInfo::getActive();
  if( scs&&si ) {
    for ( H3DDynamicFieldsObject::field_iterator it = scs->firstField(); it!=scs->endField(); ++it ) {
      Field* constant_field = (*it);
      if ( "matrixProjShift"==constant_field->getName() ) {
        si->matrixProjShift->route( constant_field );
      }
      if( "matrixViewShift"==constant_field->getName() ) {
        si->matrixViewShift->route( constant_field );
      }
    }
  }
}

#ifdef EXPORT_SHADER

void ComposedShader::UpdateSaveShadersToUrl::onNewValue( const std::string &v ){

  // when url string changed, re export the shaders
  if ( v.empty ( ) ) return;
  ComposedShader* cs = static_cast< ComposedShader* >( getOwner() );
  GLenum error = glGetError();
  vector< GLhandleARB > handlers = cs->current_shaders;
  GLchar* shader_content;
  for( vector< GLhandleARB >::iterator it = handlers.begin(); it != handlers.end(); it++ ) {
    GLint shader_type;
    GLsizei shader_length;
    glGetShaderiv ( *it, GL_SHADER_SOURCE_LENGTH , &shader_length);


    glGetShaderiv( *it, GL_SHADER_TYPE, &shader_type );
    shader_content = new GLchar [shader_length+1];
    glGetShaderSource ( *it, shader_length+1, NULL, shader_content );
    // if
    error = glGetError();
    if( error!=GL_NO_ERROR ) {
      Console(LogLevel::Error)<<" Warning: extract shader information error: "<<gluErrorString(error)<<endl;
      continue;
    }
    //glewGetString( shader_type );
    if( shader_type==GL_VERTEX_SHADER_ARB ) {
      ofstream   outFile( v+"_vertex_shader.txt", ofstream::binary  );
      outFile<< shader_content <<endl;
      outFile.close();
    }else if( shader_type==GL_GEOMETRY_SHADER_ARB ) {
      ofstream   outFile( v+"_geometry_shader.txt", ofstream::binary );
      outFile<< shader_content <<endl;
      outFile.close();
    }else if( shader_type==GL_TESS_EVALUATION_SHADER ) {
      ofstream   outFile( v+"_tessEva_shader.txt", ofstream::binary );
      outFile<< shader_content <<endl;
      outFile.close();
    }else if( shader_type==GL_TESS_CONTROL_SHADER ) {
      ofstream   outFile( v+"_tessControl_shader.txt", ofstream::binary );
      outFile<< shader_content <<endl;
      outFile.close();
    }else if( shader_type==GL_FRAGMENT_SHADER_ARB ) {
      ofstream   outFile( v+"_fragment_shader.txt", ofstream::binary );
      outFile<< shader_content <<endl;
      outFile.close();
    }else{
      Console(LogLevel::Error)<<"shader type unsupported yet"<<endl;
    }
  }
  //Console(LogLevel::Error)<<"will output shader uniform"<<endl;
  ofstream   outFile( v+"_shader_uniform.txt", ofstream::binary  );
  //Console(LogLevel::Error)<<"shader uniform txt opened."<<endl;
  int total = -1;
  GLhandleARB program_id = cs->getProgramHandle();

  glGetProgramiv( program_id , GL_ACTIVE_UNIFORMS, &total );
  outFile<<"current active uniform values:"<<endl;
  for(int i=0; i<total; ++i)  {
    int name_len=-1, num=-1;
    GLenum type = GL_ZERO;
    char name[256];
    glGetActiveUniform( program_id, GLuint(i), sizeof(name)-1,
      &name_len, &num, &type, name );
    name[name_len] = 0;
    GLuint location = glGetUniformLocation( program_id, name );
    float uniform_value[50];
    glGetUniformfvARB( program_id, location, uniform_value  );
    outFile<< name <<" : " ;
    switch ( type )
    {
    case GL_FLOAT:
    case GL_INT:
    case GL_BOOL:
    case GL_DOUBLE:
      outFile<<uniform_value[0] <<endl;
      break;
    case GL_FLOAT_VEC3:
    case GL_BOOL_VEC3:
    case GL_INT_VEC3:
    case GL_DOUBLE_VEC3_EXT:
      outFile<<uniform_value[0] <<" , " << uniform_value[1] <<
      " , " << uniform_value[2]<< endl;
      break;
    default:
      outFile<<"unsupported uniform type!"<<endl;
    }
  }
  outFile.close();
}
#endif

bool ComposedShader::printShaderLog() {
  if( printShaderWarnings->getValue() ) {
    return true;
  }

  DebugOptions *debug_options = NULL;
  GlobalSettings *default_settings = GlobalSettings::getActive();
  if( default_settings&&default_settings->optionNodesUpdated() ) {
    default_settings->getOptionNode( debug_options );
    if( debug_options ) {// update debug options
      debug_options_previous = debug_options;
      return debug_options->printShaderWarnings->getValue();
    }else{
      // global setting change in last frame, but no debug options in it now
      debug_options_previous = NULL;
      return false;
    }
  }
  else if( default_settings ) {
    // global setting option node exist but not updated
    if( debug_options_previous!=NULL ) {
      return debug_options_previous->printShaderWarnings->getValue();
    }else{
      return false;
    }
  }else{
    // no global settings at all now
    debug_options_previous = NULL;
    return false;
  }
}

void ComposedShader::initialize() {
  X3DShaderNode::initialize();
  if( GraphicsHardwareInfo::infoIsInitialized() ) {
    max_texture_in_shader = GraphicsHardwareInfo::getInfo().max_combined_texture_image_units;
    max_image_in_shader = GraphicsHardwareInfo::getInfo().max_image_units;
  }else{
    glGetIntegerv( GL_MAX_COMBINED_TEXTURE_IMAGE_UNITS_ARB, &max_texture_in_shader );
#ifdef GLEW_ARB_shader_image_load_store
    if ( GLEW_ARB_shader_image_load_store ) {
      glGetIntegerv( GL_MAX_IMAGE_UNITS, &max_image_in_shader );
    }
#endif // GLEW_ARB_shader_image_load_store
  }
  GlobalSettings *default_settings = GlobalSettings::getActive();
  GraphicsOptions *graphic_options = NULL;
  if( default_settings )
  {
      default_settings->getOptionNode( debug_options_previous );
      default_settings->getOptionNode( graphic_options );
  }
  if( graphic_options!=NULL ) {
    // if initially graphic option exist, route its useCaching to updateCache
    graphic_options->useCaching->route( updateCache );
  }
}

void ComposedShader::UpdateCache::onValueChange( const bool &new_value ){
  ComposedShader* cs = static_cast<ComposedShader*>( getOwner() );
  ComposedShader::UniformFieldMap::const_iterator it;
  it = cs->uniformFields.begin();
  if( new_value==true ) {
    //caching on
    for( ; it!=cs->uniformFields.end(); ++it ) {
      Field* f = it->second.field;
      SFNode * sf_node_field = dynamic_cast< SFNode * >( f );
      MFNode * mf_node_field = dynamic_cast< MFNode * >( f );
      if( sf_node_field==NULL&&mf_node_field==NULL ) {
        f->route(cs->displayList);
      }
    }
  }else{
    for( ; it!=cs->uniformFields.end(); ++it ) {
      Field* f = it->second.field;
      SFNode * sf_node_field = dynamic_cast< SFNode * >( f );
      MFNode * mf_node_field = dynamic_cast< MFNode * >( f );
      if( sf_node_field==NULL&&mf_node_field==NULL ){
        f->unroute(cs->displayList);
      }

    }

  }

}