package org.springframework.uaa.client.internal;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.UUID;
import java.util.prefs.BackingStoreException;
import java.util.prefs.Preferences;

import org.springframework.uaa.client.UaaService;
import org.springframework.uaa.client.VersionHelper;
import org.springframework.uaa.client.protobuf.UaaClient;
import org.springframework.uaa.client.protobuf.UaaClient.FeatureUse;
import org.springframework.uaa.client.protobuf.UaaClient.Privacy;
import org.springframework.uaa.client.protobuf.UaaClient.Product;
import org.springframework.uaa.client.protobuf.UaaClient.ProductUse;
import org.springframework.uaa.client.protobuf.UaaClient.Project;
import org.springframework.uaa.client.protobuf.UaaClient.UaaEnvelope;
import org.springframework.uaa.client.protobuf.UaaClient.UserAgent;
import org.springframework.uaa.client.protobuf.UaaClient.Privacy.PrivacyLevel;
import org.springframework.uaa.client.util.Assert;
import org.springframework.uaa.client.util.Base64;
import org.springframework.uaa.client.util.PreferencesUtils;
import org.springframework.uaa.client.util.StringUtils;

import com.google.protobuf.ByteString;
import com.google.protobuf.InvalidProtocolBufferException;

/**
 * Basic implementation of {@link UaaService} that uses the Java Preferences API
 * to record usage details.
 * 
 * <p>
 * The preferences API is used on a per-user basis (not per-system). This
 * reduces potential security issues if the user is not an administrator/root
 * user. It also improves data quality by ensuring products and features used by
 * individual users are separately represented.
 * 
 * <p>
 * The Google Protocol Buffers binary format is used for storing data in the
 * Preferences API. This extends the efficiency and long-term versioning
 * advantages of Protocol Buffers to data that is stored via the Preferences
 * API.
 * 
 * <p>
 * By default this implementation will only present usage data up to 180 days
 * old. Data older than this will be removed from the Preferences API storage
 * and not presented in HTTP User-Agent headers.
 * 
 * <p>
 * The HTTP User-Agent headers produced by this implementation will adopt the
 * form <code>
 * uaa/1 (base64_encoded_value_of_a_protocol_buffers_user_agent_message)
 * </code>. This is therefore compatible with the HTTP specification. While the
 * "/1" suffix implies versioning, this is reserved for very substantive changes
 * to the UAA model. For normal modification and retirement of fields, it should
 * be sufficient to modify the protocol buffers specification file and
 * regenerate the {@link UaaClient}.
 * 
 * @author Ben Alex
 * @author Christian Dupuis
 * 
 */
public class UaaServiceImpl implements UaaService {

	static final Preferences P = PreferencesUtils.getPreferencesFor(UaaServiceImpl.class);
	private static final byte[] EMPTY = {};
	private static final String EMPTY_STRING= "";
	private static final String PRODUCT_PREFIX_VERSION_1 = "_UAA_";
	// since UAA 1.0.0 expects the byte[] to not be Base64 encoded when it comes back from the 
	// Preferences API we need this version identifier; also UAA 1.0.0 wouldn't know how to handle
	// chunked data.
	private static final String PRODUCT_PREFIX_VERSION_2 = "_UAA/2_";  
	private static final String PRODUCT_CHUNK_PREFIX = "_CHUNK_";
	private static final String UIID_KEY = "uiid"; 	// this below is not a typo. UIID stands for universal installation identifier
	private static final String PRIVACY_LEVEL_KEY = "privacy_level";
	private static final String UNCHANGED_SINCE_LAST_USER_AGENT_REQUEST = "unchanged_since_last_user_agent_request";
	private static final String MAXIMUM_DATA_AGE_KEY = "maximum_data_age";
	private static final String COMMUNICATION_RESTRICTED_DOMAINS = "communication_restricted_domains";
	static long MAXIMUM_DATA_AGE_DEFAULT = 180L * 24L * 60L * 60L * 1000L; // 180 days (can be changed by tests)
	private static final char SEPARATOR_CHAR = '_';
	private static final String SEPARATOR = Character.toString(SEPARATOR_CHAR);
	private static final int UUID_LENGTH = 36;
	private static final int UUID_AND_SEPARATOR_LENGTH = UUID_LENGTH + SEPARATOR.length();
	private static final int MAX_VALUE_LENGTH = Preferences.MAX_VALUE_LENGTH - UUID_AND_SEPARATOR_LENGTH; // keep 37 letters for a uuid and ? with total length of 37

	public UaaServiceImpl() {
		// Force initialization
		UaaConfigurationProcessor.updateConfiguration(
				UaaConfigurationProcessor.class.getResourceAsStream("/org/springframework/uaa/client/uaa-client.xml"),
				UaaConfigurationProcessor.class.getResourceAsStream("/org/springframework/uaa/client/uaa-client.xml.asc"));
	}
	
	public boolean isCommunicationRestricted(URL url) {
		if (url == null) throw new IllegalStateException("URL required");
		
		String data = P.get(COMMUNICATION_RESTRICTED_DOMAINS, EMPTY_STRING);
		if (EMPTY_STRING.equals(data)) {
			// Domain information is unexpectedly unavailable, so assume that communications are restricted (play it safe)
			return true;
		}
		
		for (String domain : data.split(",")) {
			if (url.getHost().endsWith(domain)) {
				return true;
			}
		}
		return false;
	}

	public boolean isUaaTermsOfUseAccepted() {
		PrivacyLevel privacyLevel = getPrivacyLevel();
		return !(privacyLevel == PrivacyLevel.DECLINE_TOU || privacyLevel == PrivacyLevel.UNDECIDED_TOU); 
	}

	public void registerProductUsage(Product product) {
		registerProductUsage(product, null, null, false, false);
	}

	public void registerProductUsage(Product product, String projectId) {
		registerProductUsage(product, null, projectId, false, false);
	}

	public void registerProductUsage(Product product, byte[] productData) {
		registerProductUsage(product, productData, null, true, false);
	}

	public void registerProductUsage(Product product, byte[] productData, String projectId) {
		registerProductUsage(product, productData, projectId, true, false);
	}

	public void registerFeatureUsage(Product product, FeatureUse feature) {
		registerFeatureUsage(product, feature, null, false);
	}

	public void registerFeatureUsage(Product product, FeatureUse feature, byte[] featureData) {
		registerFeatureUsage(product, feature, featureData, true);
	}

	public String getReadablePayload() {
		return StringUtils.toString(createUaaEnvelope()); 
	}
	
	Privacy getPrivacy() {
		byte[] data = P.getByteArray(PRIVACY_LEVEL_KEY, EMPTY);

		if (EMPTY != data) {
			try {
				return Privacy.parseFrom(data);
			}
			catch (InvalidProtocolBufferException recreateItBelow) {
			}
		}

		// To get this far we need to replace the privacy value or create it for the first time
		return addDefaultPrivacyConfiguration();
	}

	public PrivacyLevel getPrivacyLevel() {
		return getPrivacy().getPrivacyLevel();
	}

	public Date getPrivacyLevelLastChanged() {
		return new Date(getPrivacy().getDateLastChanged());
	}

	private Privacy addDefaultPrivacyConfiguration() {
		Privacy.Builder pb = Privacy.newBuilder();
		pb.setPrivacyLevel(PrivacyLevel.UNDECIDED_TOU);
		pb.setDateLastChanged(new Date().getTime());
		Privacy privacy = pb.build();
		P.putByteArray(PRIVACY_LEVEL_KEY, privacy.toByteArray());
		recordChangesToStorage();
		return privacy;
	}

	public void setPrivacyLevel(PrivacyLevel privacyLevel) {
		Privacy.Builder pb = Privacy.newBuilder();
		pb.setPrivacyLevel(privacyLevel);
		pb.setDateLastChanged(new Date().getTime());
		P.putByteArray(PRIVACY_LEVEL_KEY, pb.build().toByteArray());
		recordChangesToStorage();
	}
	
	/**
	 * Create a {@link UaaEnvelope.Builder} instance which is populated with the unique install id
	 * and {@link Privacy} settings. If UR
	 */
	protected UaaEnvelope createUaaEnvelope() {
		UaaEnvelope.Builder builder = UaaEnvelope.newBuilder();
		UUID installUuid = getUiid();
		builder.setInstallationIdentifierMostSignificantBits(installUuid.getMostSignificantBits());
		builder.setInstallationIdentifierLeastSignificantBits(installUuid.getLeastSignificantBits());
		builder.setPrivacy(getPrivacy());
		
		if (isUaaTermsOfUseAccepted()) {
			builder.setUserAgent(rebuildPersistedDetails(false));
		}
		
		recordNoChangesToStorage();
		
		return builder.build();
	}


	/**
	 * Produces a new {@link UserAgent} to represent the non-expired information
	 * stored in the Preferences API. This will include removing from the
	 * returned object and Preferences API any products or features which have
	 * exceeded {@link #MAXIMUM_DATA_AGE_DEFAULT} since their last use.
	 * 
	 * @param deleteAll indicates to delete all usage up to the current date and
	 * time
	 * @return a representation of non-expired usage information (never null)
	 */
	 private UserAgent rebuildPersistedDetails(boolean deleteAll) {
		// Decide the cut-off date for deleting old preferences
		long time = System.currentTimeMillis();
		long deleteEntriesOlderThan = time - P.getLong(MAXIMUM_DATA_AGE_KEY, MAXIMUM_DATA_AGE_DEFAULT);

		if (deleteAll) {
			deleteEntriesOlderThan = System.currentTimeMillis();
		}

		// Create our user agent builder
		UserAgent.Builder builder = UserAgent.newBuilder();

		// Retrieve the privacy level
		PrivacyLevel privacyLevel = getPrivacyLevel();

		// Retrieve the UUID (or assign it if necessary)
		getUiid();

		// Record the locale and country
		if (privacyLevel.equals(PrivacyLevel.ENABLE_UAA) || privacyLevel.equals(PrivacyLevel.LIMITED_DATA)) {
			builder.setUserCountry(System.getProperty("user.country"));
			builder.setUserLanguage(System.getProperty("user.language"));
		}

		// Register infrastructure products
		registerInfrastructureProducts();

		// Get the keys
		String[] keys;
		try {
			// Make sure we sync changes to the backend store back into this JVM
			P.sync();
			keys = P.keys();
		}
		catch (BackingStoreException bse) {
			throw new IllegalStateException("Backing store failure for UAA", bse);
		}

		// Add all product usage, pruning old product usage and projects where required
		for (String key : keys) {
			// Only load keys that start with _UAA_ or _UAA/2_ and aren't product chunks identified by _CHUNK_
			if ((!key.startsWith(PRODUCT_PREFIX_VERSION_1) && !key.startsWith(PRODUCT_PREFIX_VERSION_2)) 
					|| (key.startsWith(PRODUCT_PREFIX_VERSION_2) && key.contains(PRODUCT_CHUNK_PREFIX))) {
				continue;
			}
			String data = P.get(key, EMPTY_STRING);
			if (EMPTY_STRING.equals(data)) {
				continue;
			}
			ProductUse pu;
			try {
				pu = loadProductUse(key);
			}
			catch (Exception e) {
				throw new IllegalStateException("Could not decode '" + key + "' from preferences", e);
			}

			// We have a product use; decide whether to retain it or not based on age of product use registration.
			if (pu.getDateLastUsed() < deleteEntriesOlderThan) {
				P.remove(key);
				continue;
			}

			ProductUse.Builder replacementProductUse = ProductUse.newBuilder(pu);

			// Remove features no longer used
			List<FeatureUse> newFeatures = new ArrayList<FeatureUse>();
			for (FeatureUse existing : replacementProductUse.getFeatureUseList()) {
				if (existing.getDateLastUsed() >= deleteEntriesOlderThan) {
					newFeatures.add(existing);
				}
			}
			replacementProductUse.clearFeatureUse();
			replacementProductUse.addAllFeatureUse(newFeatures);

			// Remove projects no longer used
			List<Project> newProjects = new ArrayList<Project>();
			for (Project existing : replacementProductUse.getProjectsUsingList()) {
				if (existing.getDateLastUsed() >= deleteEntriesOlderThan) {
					newProjects.add(existing);
				}
			}
			replacementProductUse.clearProjectsUsing();
			replacementProductUse.addAllProjectsUsing(newProjects);

			// Store the potentially-modified product use back in preferences for next time
			pu = saveProductUse(replacementProductUse);

			// Add this product use to our UserAgent, subject to privacy settings
			if (privacyLevel.equals(PrivacyLevel.ENABLE_UAA)) {
				builder.addProductUse(pu);
			}
			else if (privacyLevel.equals(PrivacyLevel.LIMITED_DATA)) {
				// Create a replacement ProductUse, discarding dates and binary data
				ProductUse.Builder reducedProductUse = ProductUse.newBuilder(pu);
				reducedProductUse.clearProductData(); // not copied to destination
				reducedProductUse.clearDateLastUsed(); // not copied to destination
				reducedProductUse.clearProjectsUsing(); // not copied to destination
				reducedProductUse.clearFeatureUse(); // copied to destination
				// Copy back over the features for this ProductUse, but clear private data
				List<FeatureUse> features = pu.getFeatureUseList();
				for (FeatureUse feature : features) {
					FeatureUse.Builder fub = FeatureUse.newBuilder(feature);
					fub.clearDateLastUsed();
					fub.clearFeatureData();
					reducedProductUse.addFeatureUse(fub.build());
				}

				// Now add this new product use
				builder.addProductUse(reducedProductUse.build());
			}
		}

		return builder.build();
	}

	/**
	 * Obtains the existing {@link UUID} that represents this installation from
	 * the Preferences API. If there is no known UUID, a new one will be
	 * allocated and stored in the Preferences API.
	 * 
	 * @return the UUID associated with this user's installation (never returns
	 * null)
	 */
	UUID getUiid() {
		String uuidString = P.get(UIID_KEY, EMPTY_STRING);
		if (!EMPTY_STRING.equals(uuidString)) {
			try {
				return UUID.fromString(uuidString);
			}
			catch (RuntimeException ignoreAndRecreateBelow) {
			}
		}

		// To get this far we need to create a new UUID or replace the current one
		UUID uuid = UUID.randomUUID();
		P.put(UIID_KEY, uuid.toString());
		recordChangesToStorage();
		return uuid;
	}

	/**
	 * Produces a Preferences API key for the presented product. The product
	 * need not have been previously stored. The keys are unique to a product's
	 * name and version details down to and including the release qualified.
	 * However, the {@link Product#getSourceControlIdentifier()} is not used in
	 * the produced key. This means an identical product version but built with
	 * a different source control code will result in the same product key. This
	 * is acceptable for the intended purpose of the user agent analysis module.
	 * 
	 * @param product to return a Preferences API key for (required)
	 * @return a new key (never null)
	 */
	String getKey(Product product) {
		Assert.notNull(product, "Product required");
		Assert.hasLength(product.getName(), "Product name missing");
		Assert.notNull(product.getReleaseQualifier(), "Product release qualifier null for '" + product.toString() + "'");
		String ver = product.getMajorVersion() + "." + product.getMinorVersion() + "." + product.getPatchVersion()
				+ "." + "." + product.getReleaseQualifier();
		String key = product.getName() + " " + ver;
		try {
			MessageDigest sha1 = MessageDigest.getInstance("SHA-1");
			byte[] digest = sha1.digest(key.getBytes("UTF-8"));
			return PRODUCT_PREFIX_VERSION_2 + Base64.encodeBytes(digest);
		}
		catch (NoSuchAlgorithmException e) {
			// This can't happen as we know that there is an SHA-1 algorithm
		}
		catch (UnsupportedEncodingException e) {
			// This can't happen as we know that there is an UTF-8 encoding
		}
		return key;
	}

	/**
	 * Obtains from the Preferences API a {@link ProductUse.Builder} for the
	 * presented {@link Product}. This method returns a builder for any existing
	 * usage of the {@link Product}, or alternately creates a new
	 * {@link ProductUse.Builder} ready for subsequent storage via
	 * {@link #saveProductUse(ProductUse)}.
	 * 
	 * @param product
	 * @return
	 */
	private ProductUse.Builder loadProductUse(Product product) {
		// Get the existing product use from preferences, if available
		String key = getKey(product);
		ProductUse pu = loadProductUse(key);
		if (pu != null) {
			return pu.toBuilder();
		}
		ProductUse.Builder result = ProductUse.newBuilder();
		result.setProduct(product);
		return result;
	}

	/**
	 * Obtains from the Preferences API a {@link ProductUse.Builder} for the
	 * presented product key.
	 * 
	 * @param key the product key to load
	 * @return a {@link ProductUse} entry or <code>null</code> if reading fails
	 */
	private ProductUse loadProductUse(String key) {
		String data = P.get(key, EMPTY_STRING);
		if (!EMPTY_STRING.equals(data)) {
			try {
				StringBuilder builder = new StringBuilder();
				// Assemble chunks into builder
				while (!EMPTY_STRING.equals(data)) {
					if (data.length() > UUID_AND_SEPARATOR_LENGTH && data.charAt(UUID_LENGTH) == SEPARATOR_CHAR) {
						String uuid = data.substring(0, UUID_LENGTH);
						builder.append(data.substring(UUID_AND_SEPARATOR_LENGTH));
						data = P.get(key + PRODUCT_CHUNK_PREFIX + uuid, EMPTY_STRING);
					}
					else {
						builder.append(data);
						data = EMPTY_STRING;
					}
				}
				return ProductUse.parseFrom(Base64.decode(builder.toString()));
			}
			catch (InvalidProtocolBufferException ipe) {
				// It is ok here to ignore and return null
			}
			catch (IOException e) {
				// It is ok here to ignore and return null
			}
		}
		return null;
	}

	/**
	 * Stores the presented {@link ProductUse} in the Preferences API.
	 * 
	 * @param productUse to store (required)
	 */
	private ProductUse saveProductUse(ProductUse.Builder productUse) {
		ProductUse pu = productUse.build();
		String key = getKey(pu.getProduct());
		try {
			// Encode the productUse message to base64; use GZIP as additional compression
			byte[] bytes = pu.toByteArray();
			String compressed = Base64.encodeBytes(bytes, Base64.GZIP);
			String uncompressed = Base64.encodeBytes(bytes, 0);

			String productUseBase64 = null;
			if (compressed.length() < uncompressed.length()) {
				productUseBase64 = compressed;
			}
			else {
				productUseBase64 = uncompressed;
			}
			
			String uuid = null;
			// Create chunks
			for (int i = 0; i < productUseBase64.length(); i += MAX_VALUE_LENGTH) {
				String chunk = productUseBase64.substring(i, Math.min(productUseBase64.length(), i + MAX_VALUE_LENGTH));
				String chunkKey = (uuid != null ? key + PRODUCT_CHUNK_PREFIX + uuid : key);
				
				// If there are more chunks to come create the uuid of the next entry
				if (i + MAX_VALUE_LENGTH < productUseBase64.length()) {
					uuid = UUID.randomUUID().toString();
					P.put(chunkKey, uuid + SEPARATOR + chunk);
				}
				else {
					// No more chunks to come
					uuid = null;
					P.put(chunkKey, chunk);
				}
			}
		}
		catch (IOException e) {
			throw new IllegalStateException("Error zipping base64 encoded byte[]", e);
		}
		// no need to use recordChangesToStorage() as this method is only called from saveProductUse(..) which has
		// already done it
		return pu;
	}

	/**
	 * Registers the infrastructure products that are always present in a UAA
	 * {@link UserAgent}.
	 */
	private void registerInfrastructureProducts() {
		registerProductUsage(VersionHelper.getJvm(), null, null, false, true);
		registerProductUsage(VersionHelper.getOs(), null, null, false, true);
		registerProductUsage(VersionHelper.getUaa(), null, null, false, true);
	}

	private void registerProductUsage(Product product, byte[] productData, String projectId, boolean replace,
			boolean infrastructure) {
		Assert.notNull(product, "Product required");
		if (projectId == null)
			projectId = EMPTY_STRING;

		// Store changes
		recordChangesToStorage();

		// Ensure a UUID and privacy details are allocated
		getUiid();
		PrivacyLevel privacyLevel = getPrivacyLevel();

		// Never record any information unless the user has accepted UAA terms of use
		if (privacyLevel.equals(PrivacyLevel.UNDECIDED_TOU) || privacyLevel.equals(PrivacyLevel.DECLINE_TOU)) {
			return;
		}

		// Ensure Java and OS details are stored
		if (!infrastructure) {
			registerInfrastructureProducts();
		}

		// Get the relevant ProductUse record (creating if necessary)
		ProductUse.Builder builder = loadProductUse(product);

		// Store that it was just used
		long time = System.currentTimeMillis();
		builder.setDateLastUsed(time);

		if (!EMPTY_STRING.equals(projectId)) {
			// Create a representation of the Project use for later use
			Project.Builder projectBuilder = Project.newBuilder();
			projectBuilder.setDateLastUsed(time);

			// Update the builder with the hash code of the project (we have no desire to know the user's actual project
			// name, we just want to see if a team edits the same project across multiple members, which a hash can indicate
			// for us)
			try {
				MessageDigest sha256 = MessageDigest.getInstance("SHA-256");
				byte[] digest = sha256.digest((projectId).getBytes());
				projectBuilder.setProjectSha256Hash(ByteString.copyFrom(digest));
			}
			catch (Throwable hashingFailure) {
				projectBuilder.setProjectSha256Hash(ByteString.copyFrom("0000000000000000000".getBytes()));
			}

			// Try to locate an existing record and update it
			boolean replaced = false;
			for (int i = 0; i < builder.getProjectsUsingCount(); i++) {
				Project existing = builder.getProjectsUsing(i);
				if (existing.getProjectSha256Hash().equals(projectBuilder.getProjectSha256Hash())) {
					// Located
					builder.setProjectsUsing(i, projectBuilder);
					replaced = true;
					break;
				}
			}

			// Create a new record, given this is the first usage
			if (!replaced) {
				builder.addProjectsUsing(projectBuilder);
			}

		}

		// Store the presented product data
		if (replace && productData != null) {
			builder.setProductData(ByteString.copyFrom(productData));
		}

		// Store it back
		saveProductUse(builder);
	}

	private void registerFeatureUsage(Product product, FeatureUse feature, byte[] featureData, boolean replace) {
		Assert.notNull(product, "Product required");
		Assert.notNull(feature, "Feature required for product '" + product.getName() + "'");
		Assert.isTrue(!UaaService.PRODUCT_UAA_KEYWORD.equals(feature.getName()), "Feature name is reserved for UAA internal use");

		recordChangesToStorage();

		// Ensure a UUID and privacy details are allocated
		getUiid();
		PrivacyLevel privacyLevel = getPrivacyLevel();

		// Never record any information unless the user has accepted UAA terms
		// of use
		if (privacyLevel.equals(PrivacyLevel.UNDECIDED_TOU) || privacyLevel.equals(PrivacyLevel.DECLINE_TOU)) {
			return;
		}

		// Ensure Java and OS details are stored
		registerInfrastructureProducts();

		// Get the relevant ProductUse record (creating if necessary)
		ProductUse.Builder builder = loadProductUse(product);

		// Store that it was just used
		long time = System.currentTimeMillis();
		builder.setDateLastUsed(time);

		// Create a representation of the feature use
		String featureName = feature.getName();
		FeatureUse.Builder fu = FeatureUse.newBuilder(feature);
		fu.setDateLastUsed(time);
		
		if (replace && featureData != null) {
			fu.setFeatureData(ByteString.copyFrom(featureData));
		}

		// Attempt to locate this existing feature
		boolean replaced = false;
		for (int i = 0; i < builder.getFeatureUseCount(); i++) {
			FeatureUse existing = builder.getFeatureUse(i);
			if (existing.getName().equals(featureName)) {
				// Located
				if (!replace) {
					// Store any existing feature_data into the new record, as we don't want to lose the current_feature data
					fu.setFeatureData(existing.getFeatureData());
				}
				builder.setFeatureUse(i, fu);
				replaced = true;
				break;
			}
		}

		// Add the feature if it's new
		if (!replaced) {
			builder.addFeatureUse(fu);
		}

		// Store it back
		saveProductUse(builder);
	}

	/**
	 * Records there have been changed made to the preferences since the last
	 * {@link UaaEnvelope} object was acquired.
	 */
	private void recordChangesToStorage() {
		P.putBoolean(UNCHANGED_SINCE_LAST_USER_AGENT_REQUEST, false);
		try {
			P.flush();
		}
		catch (BackingStoreException ignore) {
		}
	}
	
	/**
	 * Records there are <b>no</b> changes made to the preferences sine the last {@link UaaEnvelope} 
	 * object was acquired.
	 */
	private void recordNoChangesToStorage() {
		P.putBoolean(UNCHANGED_SINCE_LAST_USER_AGENT_REQUEST, true);
		try {
			P.flush();
		}
		catch (BackingStoreException ignore) {
		}
	}
	
	/**
	 * Removes all usage information from the database assuming no changes have taken place since the last call to
	 * {@link #toHttpUserAgentHeaderValue()}. This is strongly recommended after a successful invocation of that method
	 * and the header has been presented in a HTTP request that resulted in a 200 (OK) response. This prevents the header
	 * from growing too large. 
	 * 
	 * <p>
	 * This method preserves key information such as the user's privacy level and installation identifier.
	 * 
	 * <p>
	 * The database will not be cleared if there have been changes since the last {@link #toHttpUserAgentHeaderValue()} request.
	 * 
	 * @return the new {@link UserAgent} the storage now represents (never null)
	 */
	UserAgent clearIfPossible() {
		if (P.getBoolean(UNCHANGED_SINCE_LAST_USER_AGENT_REQUEST, false)) {
			// It is safe to truncate, but make sure we don't do this again
			recordChangesToStorage();

			// Truncate it
			return rebuildPersistedDetails(true);
		}
		// It is not safe to truncate
		return rebuildPersistedDetails(false);
	}

	/**
	 * Used for testing purposes only.
	 */
	void clearPreferences() {
		try {
			P.clear();
		}
		catch (BackingStoreException e) {
			throw new IllegalStateException(e);
		}
	}

	/**
	 * Used for testing purposes only.
	 * 
	 * @return the total number of keys in the preferences node (includes both
	 * infrastructure and product usage keys)
	 */
	int keyCount() {
		try {
			return P.keys().length;
		}
		catch (BackingStoreException e) {
			throw new IllegalStateException(e);
		}
	}

	boolean isDatabaseClean() {
		return P.getBoolean(UNCHANGED_SINCE_LAST_USER_AGENT_REQUEST, false);
	}
}
