package org.springframework.uaa.client;

import java.net.URL;
import java.util.Date;

import org.springframework.uaa.client.protobuf.UaaClient.FeatureUse;
import org.springframework.uaa.client.protobuf.UaaClient.Privacy.PrivacyLevel;
import org.springframework.uaa.client.protobuf.UaaClient.Product;

/**
 * User Agent service interface.
 * 
 * <p>
 * This is the main interface to be used by tools wishing to use the UAA feature.
 * 
 * <p>
 * The most common usage is to invoke the various "register" methods and rely on the
 * implementation to periodically send the usage data to the server.
 * 
 * <p>
 * A great deal of efficiency and future compatibility has been designed into this
 * architecture. Google Protocol Buffers has been used to encode the information in
 * an efficient binary form that supports future version changes.
 * 
 * @author Ben Alex
 * @since 1.0.0
 *
 */
public interface UaaService {

	/** Reserved keyword that cannot be used as a feature name **/
	public static final String PRODUCT_UAA_KEYWORD = "__PRODUCT__";

	/**
	 * Registers a particular product has been used, preserving any existing binary product data.
	 * 
	 * <p>
	 * Binary product data is optional and has no UAA-defined meaning. Products that wish to use
	 * this capability are permitted to format the binary product data in their own preferred manner.
	 * 
	 * @param product to register (required)
	 */
	void registerProductUsage(Product product);

	/**
	 * Registers a particular product has been used by a particular project, preserving any existing binary product data.
	 * 
	 * @param product to register (required)
	 * @param projectId the ID of the project as known by the tool (required; stored only as an SHA1 hash internally)
	 */
	void registerProductUsage(Product product, String projectId);
	
	/**
	 * Registers a particular product has been used, replacing the binary product data with that specified.
	 * 
	 * <p>
	 * Binary product data is optional and has no UAA-defined meaning. Products that wish to use
	 * this capability are permitted to format the binary product data in their own preferred manner.
	 * 
	 * @param product to register (required)
	 * @param productData the replacement product data
	 */
	void registerProductUsage(Product product, byte[] productData);

	/**
	 * Registers a particular product has been used by a particular project, replacing the binary product data with that specified.
	 * 
	 * @param product to register (required)
	 * @param productData the replacement product data
	 * @param projectId the ID of the project as known by the tool (required; stored only as an SHA1 hash internally)
	 */
	void registerProductUsage(Product product, byte[] productData, String projectId);

	/**
	 * Registers a particular product feature has been used, preserving any existing binary data for both
	 * the product and the feature. Will ensure the product usage time is also updated.
	 * 
	 * @param product to register (required)
	 * @param feature to register (required)
	 */
	void registerFeatureUsage(Product product, FeatureUse feature);

	/**
	 * Registers a particular product feature has been used, replacing the existing binary data for that
	 * feature. Any binary data for the product itself will be retained. Will ensure the product usage time is
	 * also updated.
	 * 
	 * <p>
	 * Binary feature data is optional and has no UAA-defined meaning. Features that wish to use
	 * this capability are permitted to format the binary feature data in their own preferred manner.
	 * 
	 * @param product to register (required)
	 * @param feature to register (required)
	 * @param featureData the replacement feature data
	 */
	void registerFeatureUsage(Product product, FeatureUse feature, byte[] featureData);

	/**
	 * Presents the current content suitable for user display.
	 * 
	 * @return a string (never null; will be an empty string if the input was null)
	 */
	String getReadablePayload();
	
	/**
	 * Obtains the current user privacy level. Defaults to {@link PrivacyLevel#UNDECIDED_TOU}, set on the current date.
	 * UAA-enabled clients should use {@link #isUaaTermsOfUseAccepted()} as a preference to this method if the invocation
	 * is intended to reveal whether the UAA Terms of Use has been accepted.
	 * 
	 * @return the level (never null)
	 */
	PrivacyLevel getPrivacyLevel();
	
	/**
	 * Obtains the date the user last changed the privacy level. Defaults to {@link PrivacyLevel#UNDECIDED_TOU}, set on the current date.
	 * 
	 * @return the date the level was last changed (never null)
	 */
	Date getPrivacyLevelLastChanged();
	
	/**
	 * Sets the current user privacy level. Note that full data is still collected if the user has accepted the terms of use (as
	 * indicated by their privacy level). Invoking this method simply controls the amount of data included in user agent values.
	 * 
	 * @param privacyLevel the new privacy level (required)
	 */
	void setPrivacyLevel(PrivacyLevel privacyLevel);
	
	/**
	 * Indicates whether the presented URL is for a communication-restricted URL. A communication-restricted URL should not be
	 * contacted by a UAA-enabled client unless {@link #isUaaTermsOfUseAccepted()}. A small number of special exceptions are
	 * permitted as expressly defined in the UAA Terms of Use (mainly dealing with automatic UAA functions).
	 * 
	 * @param url any URL (required)
	 * @return true if the presented URL should have communications restricted if the terms of use have not been accepted
	 */
	boolean isCommunicationRestricted(URL url);
	
	/**
	 * Indicates whether the user has accepted the UAA Terms of Use. While this could be determined by looking at the
	 * {@link #getPrivacyLevel()} result, this method is intended to abstract future changes to the privacy levels from
	 * needing to be mirrored in every UAA-enabled client. UAA-enabled clients should NOT resolve privacy levels directly.
	 * 
	 * @return true if UAA Terms of Use have been accepted, otherwise false
	 */
	boolean isUaaTermsOfUseAccepted();
}