/* -*- c++ -*- */
/* 
 * Copyright 2014 Ron Economos.
 * 
 * This is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this software; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street,
 * Boston, MA 02110-1301, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <gnuradio/io_signature.h>
#include "modulator_bc_impl.h"

namespace gr {
  namespace dvbs2 {

    modulator_bc::sptr
    modulator_bc::make(dvbs2_constellation_t constellation, dvbs2_code_rate_t rate, dvbs2_framesize_t framesize)
    {
      return gnuradio::get_initial_sptr
        (new modulator_bc_impl(constellation, rate, framesize));
    }

    /*
     * The private constructor
     */
    modulator_bc_impl::modulator_bc_impl(dvbs2_constellation_t constellation, dvbs2_code_rate_t rate, dvbs2_framesize_t framesize)
      : gr::block("modulator_bc",
              gr::io_signature::make(1, 1, sizeof(unsigned char)),
              gr::io_signature::make(1, 1, sizeof(gr_complex)))
    {
        double r1, r2, r3, r4, r5, r6, r7, r8;
        double m = 1.0;
        r1 = m;
        switch (constellation)
        {
            case gr::dvbs2::MOD_QPSK:
                m_qpsk[0].real() = (r1 * cos(M_PI / 4.0));
                m_qpsk[0].imag() = (r1 * sin(M_PI / 4.0));
                m_qpsk[1].real() = (r1 * cos(7 * M_PI / 4.0));
                m_qpsk[1].imag() = (r1 * sin(7 * M_PI / 4.0));
                m_qpsk[2].real() = (r1 * cos(3 * M_PI / 4.0));
                m_qpsk[2].imag() = (r1 * sin(3 * M_PI / 4.0));
                m_qpsk[3].real() = (r1 * cos(5 * M_PI / 4.0));
                m_qpsk[3].imag() = (r1 * sin(5 * M_PI / 4.0));
                break;
            case gr::dvbs2::MOD_8PSK:
                m_8psk[0].real() = (r1 * cos(M_PI / 4.0));
                m_8psk[0].imag() = (r1 * sin(M_PI / 4.0));
                m_8psk[1].real() = (r1 * cos(0.0));
                m_8psk[1].imag() = (r1 * sin(0.0));
                m_8psk[2].real() = (r1 * cos(4 * M_PI / 4.0));
                m_8psk[2].imag() = (r1 * sin(4 * M_PI / 4.0));
                m_8psk[3].real() = (r1 * cos(5 * M_PI / 4.0));
                m_8psk[3].imag() = (r1 * sin(5 * M_PI / 4.0));
                m_8psk[4].real() = (r1 * cos(2 * M_PI / 4.0));
                m_8psk[4].imag() = (r1 * sin(2 * M_PI / 4.0));
                m_8psk[5].real() = (r1 * cos(7 * M_PI / 4.0));
                m_8psk[5].imag() = (r1 * sin(7 * M_PI / 4.0));
                m_8psk[6].real() = (r1 * cos(3 * M_PI / 4.0));
                m_8psk[6].imag() = (r1 * sin(3 * M_PI / 4.0));
                m_8psk[7].real() = (r1 * cos(6 * M_PI / 4.0));
                m_8psk[7].imag() = (r1 * sin(6 * M_PI / 4.0));
                break;
            case gr::dvbs2::MOD_8APSK:
                r3 = m;
                switch(rate)
                {
                    case gr::dvbs2::C100_180:
                        r1 = r3 / 6.8;
                        r2 = r1 * 5.32;
                        break;
                    case gr::dvbs2::C104_180:
                        r1 = r3 / 8.0;
                        r2 = r1 * 6.39;
                        break;
                    default:
                        r1 = 0;
                        r2 = 0;
                        break;
                }
                m_8psk[0].real() = (r1 * cos(M_PI));
                m_8psk[0].imag() = (r1 * sin(M_PI));
                m_8psk[1].real() = (r2 * cos(M_PI * 1.352));
                m_8psk[1].imag() = (r2 * sin(M_PI * 1.352));
                m_8psk[2].real() = (r2 * cos(M_PI * -1.352));
                m_8psk[2].imag() = (r2 * sin(M_PI * -1.352));
                m_8psk[3].real() = (r3 * cos(M_PI));
                m_8psk[3].imag() = (r3 * sin(M_PI));
                m_8psk[4].real() = (r1 * cos(0.0));
                m_8psk[4].imag() = (r1 * sin(0.0));
                m_8psk[5].real() = (r2 * cos(M_PI * -0.352));
                m_8psk[5].imag() = (r2 * sin(M_PI * -0.352));
                m_8psk[6].real() = (r2 * cos(M_PI * 0.352));
                m_8psk[6].imag() = (r2 * sin(M_PI * 0.352));
                m_8psk[7].real() = (r3 * cos(0.0));
                m_8psk[7].imag() = (r3 * sin(0.0));
                break;
            case gr::dvbs2::MOD_16APSK:
                r2 = m;
                if (framesize == gr::dvbs2::FECFRAME_NORMAL)
                {
                    switch(rate)
                    {
                        case gr::dvbs2::C2_3:
                            r1 = r2 / 3.15;
                            break;
                        case gr::dvbs2::C3_4:
                            r1 = r2 / 2.85;
                            break;
                        case gr::dvbs2::C4_5:
                            r1 = r2 / 2.75;
                            break;
                        case gr::dvbs2::C5_6:
                            r1 = r2 / 2.70;
                            break;
                        case gr::dvbs2::C8_9:
                            r1 = r2 / 2.60;
                            break;
                        case gr::dvbs2::C9_10:
                            r1 = r2 / 2.57;
                            break;
                        case gr::dvbs2::C26_45:
                        case gr::dvbs2::C3_5:
                            r1 = r2 / 3.70;
                            break;
                        case gr::dvbs2::C28_45:
                            r1 = r2 / 3.50;
                            break;
                        case gr::dvbs2::C23_36:
                        case gr::dvbs2::C25_36:
                            r1 = r2 / 3.10;
                            break;
                        case gr::dvbs2::C13_18:
                            r1 = r2 / 2.85;
                            break;
                        case gr::dvbs2::C140_180:
                            r1 = r2 / 3.60;
                            break;
                        case gr::dvbs2::C154_180:
                            r1 = r2 / 3.20;
                            break;
                        default:
                            r1 = 0;
                            break;
                    }
                }
                else
                {
                    switch(rate)
                    {
                        case gr::dvbs2::C2_3:
                            r1 = r2 / 3.15;
                            break;
                        case gr::dvbs2::C3_4:
                            r1 = r2 / 2.85;
                            break;
                        case gr::dvbs2::C4_5:
                            r1 = r2 / 2.75;
                            break;
                        case gr::dvbs2::C5_6:
                            r1 = r2 / 2.70;
                            break;
                        case gr::dvbs2::C8_9:
                            r1 = r2 / 2.60;
                            break;
                        case gr::dvbs2::C7_15:
                            r1 = r2 / 3.32;
                            break;
                        case gr::dvbs2::C8_15:
                            r1 = r2 / 3.50;
                            break;
                        case gr::dvbs2::C26_45:
                        case gr::dvbs2::C3_5:
                            r1 = r2 / 3.70;
                            break;
                        case gr::dvbs2::C32_45:
                            r1 = r2 / 2.85;
                            break;
                        default:
                            r1 = 0;
                            break;
                    }
                }
                m_16apsk[0].real()  = (r2 * cos(M_PI / 4.0));
                m_16apsk[0].imag()  = (r2 * sin(M_PI / 4.0));
                m_16apsk[1].real()  = (r2 * cos(-M_PI / 4.0));
                m_16apsk[1].imag()  = (r2 * sin(-M_PI / 4.0));
                m_16apsk[2].real()  = (r2 * cos(3 * M_PI / 4.0));
                m_16apsk[2].imag()  = (r2 * sin(3 * M_PI / 4.0));
                m_16apsk[3].real()  = (r2 * cos(-3 * M_PI / 4.0));
                m_16apsk[3].imag()  = (r2 * sin(-3 * M_PI / 4.0));
                m_16apsk[4].real()  = (r2 * cos(M_PI / 12.0));
                m_16apsk[4].imag()  = (r2 * sin(M_PI / 12.0));
                m_16apsk[5].real()  = (r2 * cos(-M_PI / 12.0));
                m_16apsk[5].imag()  = (r2 * sin(-M_PI / 12.0));
                m_16apsk[6].real()  = (r2 * cos(11 * M_PI / 12.0));
                m_16apsk[6].imag()  = (r2 * sin(11 * M_PI / 12.0));
                m_16apsk[7].real()  = (r2 * cos(-11 * M_PI / 12.0));
                m_16apsk[7].imag()  = (r2 * sin(-11 * M_PI / 12.0));
                m_16apsk[8].real()  = (r2 * cos(5 * M_PI / 12.0));
                m_16apsk[8].imag()  = (r2 * sin(5 * M_PI / 12.0));
                m_16apsk[9].real()  = (r2 * cos(-5 * M_PI / 12.0));
                m_16apsk[9].imag()  = (r2 * sin(-5 * M_PI / 12.0));
                m_16apsk[10].real() = (r2 * cos(7 * M_PI / 12.0));
                m_16apsk[10].imag() = (r2 * sin(7 * M_PI / 12.0));
                m_16apsk[11].real() = (r2 * cos(-7 * M_PI / 12.0));
                m_16apsk[11].imag() = (r2 * sin(-7 * M_PI / 12.0));
                m_16apsk[12].real() = (r1 * cos(M_PI / 4.0));
                m_16apsk[12].imag() = (r1 * sin(M_PI / 4.0));
                m_16apsk[13].real() = (r1 * cos(-M_PI / 4.0));
                m_16apsk[13].imag() = (r1 * sin(-M_PI / 4.0));
                m_16apsk[14].real() = (r1 * cos(3 * M_PI / 4.0));
                m_16apsk[14].imag() = (r1 * sin(3 * M_PI / 4.0));
                m_16apsk[15].real() = (r1 * cos(-3 * M_PI / 4.0));
                m_16apsk[15].imag() = (r1 * sin(-3 * M_PI / 4.0));
                break;
            case gr::dvbs2::MOD_8_8APSK:
                if (rate == gr::dvbs2::C18_30)
                {
                    m_16apsk[0].real()  = 0.4718;
                    m_16apsk[0].imag()  = 0.2606;
                    m_16apsk[1].real()  = 0.2606;
                    m_16apsk[1].imag()  = 0.4718;
                    m_16apsk[2].real()  = -0.4718;
                    m_16apsk[2].imag()  = 0.2606;
                    m_16apsk[3].real()  = -0.2606;
                    m_16apsk[3].imag()  = 0.4718;
                    m_16apsk[4].real()  = 0.4718;
                    m_16apsk[4].imag()  = -0.2606;
                    m_16apsk[5].real()  = 0.2606;
                    m_16apsk[5].imag()  = -0.4718;
                    m_16apsk[6].real()  = -0.4718;
                    m_16apsk[6].imag()  = -0.2606;
                    m_16apsk[7].real()  = -0.2606;
                    m_16apsk[7].imag()  = -0.4718;
                    m_16apsk[8].real()  = 1.2088;
                    m_16apsk[8].imag()  = 0.4984;
                    m_16apsk[9].real()  = 0.4984;
                    m_16apsk[9].imag()  = 1.2088;
                    m_16apsk[10].real() = -1.2088;
                    m_16apsk[10].imag() = 0.4984;
                    m_16apsk[11].real() = -0.4984;
                    m_16apsk[11].imag() = 1.2088;
                    m_16apsk[12].real() = 1.2088;
                    m_16apsk[12].imag() = -0.4984;
                    m_16apsk[13].real() = 0.4984;
                    m_16apsk[13].imag() = -1.2088;
                    m_16apsk[14].real() = -1.2088;
                    m_16apsk[14].imag() = -0.4984;
                    m_16apsk[15].real() = -0.4984;
                    m_16apsk[15].imag() = -1.2088;
                    for (int i = 0; i < 16; i++)
                    {
                        m_16apsk[i].real() /= 1.2088;
                        m_16apsk[i].imag() /= 1.2088;
                    }
                }
                else if (rate == gr::dvbs2::C20_30)
                {
                    m_16apsk[0].real()  = 0.5061;
                    m_16apsk[0].imag()  = 0.2474;
                    m_16apsk[1].real()  = 0.2474;
                    m_16apsk[1].imag()  = 0.5061;
                    m_16apsk[2].real()  = -0.5061;
                    m_16apsk[2].imag()  = 0.2474;
                    m_16apsk[3].real()  = -0.2474;
                    m_16apsk[3].imag()  = 0.5061;
                    m_16apsk[4].real()  = 0.5061;
                    m_16apsk[4].imag()  = -0.2474;
                    m_16apsk[5].real()  = 0.2474;
                    m_16apsk[5].imag()  = -0.5061;
                    m_16apsk[6].real()  = -0.5061;
                    m_16apsk[6].imag()  = -0.2474;
                    m_16apsk[7].real()  = -0.2474;
                    m_16apsk[7].imag()  = -0.5061;
                    m_16apsk[8].real()  = 1.2007;
                    m_16apsk[8].imag()  = 0.4909;
                    m_16apsk[9].real()  = 0.4909;
                    m_16apsk[9].imag()  = 1.2007;
                    m_16apsk[10].real() = -1.2007;
                    m_16apsk[10].imag() = 0.4909;
                    m_16apsk[11].real() = -0.4909;
                    m_16apsk[11].imag() = 1.2007;
                    m_16apsk[12].real() = 1.2007;
                    m_16apsk[12].imag() = -0.4909;
                    m_16apsk[13].real() = 0.4909;
                    m_16apsk[13].imag() = -1.2007;
                    m_16apsk[14].real() = -1.2007;
                    m_16apsk[14].imag() = -0.4909;
                    m_16apsk[15].real() = -0.4909;
                    m_16apsk[15].imag() = -1.2007;
                    for (int i = 0; i < 16; i++)
                    {
                        m_16apsk[i].real() /= 1.2007;
                        m_16apsk[i].imag() /= 1.2007;
                    }
                }
                else
                {
                    r2 = m;
                    switch(rate)
                    {
                        case gr::dvbs2::C90_180:
                        case gr::dvbs2::C96_180:
                        case gr::dvbs2::C100_180:
                            r1 = r2 / 2.19;
                            break;
                        default:
                            r1 = 0;
                            break;
                    }
                    m_16apsk[0].real()  = (r1 * cos(M_PI / 8.0));
                    m_16apsk[0].imag()  = (r1 * sin(M_PI / 8.0));
                    m_16apsk[1].real()  = (r1 * cos(3 * M_PI / 8.0));
                    m_16apsk[1].imag()  = (r1 * sin(3 * M_PI / 8.0));
                    m_16apsk[2].real()  = (r1 * cos(7 * M_PI / 8.0));
                    m_16apsk[2].imag()  = (r1 * sin(7 * M_PI / 8.0));
                    m_16apsk[3].real()  = (r1 * cos(5 * M_PI / 8.0));
                    m_16apsk[3].imag()  = (r1 * sin(5 * M_PI / 8.0));
                    m_16apsk[4].real()  = (r1 * cos(15 * M_PI / 8.0));
                    m_16apsk[4].imag()  = (r1 * sin(15 * M_PI / 8.0));
                    m_16apsk[5].real()  = (r1 * cos(13 * M_PI / 8.0));
                    m_16apsk[5].imag()  = (r1 * sin(13 * M_PI / 8.0));
                    m_16apsk[6].real()  = (r1 * cos(9 * M_PI / 8.0));
                    m_16apsk[6].imag()  = (r1 * sin(9 * M_PI / 8.0));
                    m_16apsk[7].real()  = (r1 * cos(11 * M_PI / 8.0));
                    m_16apsk[7].imag()  = (r1 * sin(11 * M_PI / 8.0));
                    m_16apsk[8].real()  = (r2 * cos(M_PI / 8.0));
                    m_16apsk[8].imag()  = (r2 * sin(M_PI / 8.0));
                    m_16apsk[9].real()  = (r2 * cos(3 * M_PI / 8.0));
                    m_16apsk[9].imag()  = (r2 * sin(3 * M_PI / 8.0));
                    m_16apsk[10].real() = (r2 * cos(7 * M_PI / 8.0));
                    m_16apsk[10].imag() = (r2 * sin(7 * M_PI / 8.0));
                    m_16apsk[11].real() = (r2 * cos(5 * M_PI / 8.0));
                    m_16apsk[11].imag() = (r2 * sin(5 * M_PI / 8.0));
                    m_16apsk[12].real() = (r2 * cos(15 * M_PI / 8.0));
                    m_16apsk[12].imag() = (r2 * sin(15 * M_PI / 8.0));
                    m_16apsk[13].real() = (r2 * cos(13 * M_PI / 8.0));
                    m_16apsk[13].imag() = (r2 * sin(13 * M_PI / 8.0));
                    m_16apsk[14].real() = (r2 * cos(9 * M_PI / 8.0));
                    m_16apsk[14].imag() = (r2 * sin(9 * M_PI / 8.0));
                    m_16apsk[15].real() = (r2 * cos(11 * M_PI / 8.0));
                    m_16apsk[15].imag() = (r2 * sin(11 * M_PI / 8.0));
                }
                break;
            case gr::dvbs2::MOD_32APSK:
                r3 = m;
                switch(rate)
                {
                    case gr::dvbs2::C3_4:
                        r1 = r3 / 5.27;
                        r2 = r1 * 2.84;
                        break;
                    case gr::dvbs2::C4_5:
                        r1 = r3 / 4.87;
                        r2 = r1 * 2.72;
                        break;
                    case gr::dvbs2::C5_6:
                        r1 = r3 / 4.64;
                        r2 = r1 * 2.64;
                        break;
                    case gr::dvbs2::C8_9:
                        r1 = r3 / 4.33;
                        r2 = r1 * 2.54;
                        break;
                    case gr::dvbs2::C9_10:
                        r1 = r3 / 4.30;
                        r2 = r1 * 2.53;
                        break;
                    default:
                        r1 = 0;
                        r2 = 0;
                        break;
                }
                m_32apsk[0].real()  = (r2 * cos(M_PI / 4.0));
                m_32apsk[0].imag()  = (r2 * sin(M_PI / 4.0));
                m_32apsk[1].real()  = (r2 * cos(5 * M_PI / 12.0));
                m_32apsk[1].imag()  = (r2 * sin(5 * M_PI / 12.0));
                m_32apsk[2].real()  = (r2 * cos(-M_PI / 4.0));
                m_32apsk[2].imag()  = (r2 * sin(-M_PI / 4.0));
                m_32apsk[3].real()  = (r2 * cos(-5 * M_PI / 12.0));
                m_32apsk[3].imag()  = (r2 * sin(-5 * M_PI / 12.0));
                m_32apsk[4].real()  = (r2 * cos(3 * M_PI / 4.0));
                m_32apsk[4].imag()  = (r2 * sin(3 * M_PI / 4.0));
                m_32apsk[5].real()  = (r2 * cos(7 * M_PI / 12.0));
                m_32apsk[5].imag()  = (r2 * sin(7 * M_PI / 12.0));
                m_32apsk[6].real()  = (r2 * cos(-3 * M_PI / 4.0));
                m_32apsk[6].imag()  = (r2 * sin(-3 * M_PI / 4.0));
                m_32apsk[7].real()  = (r2 * cos(-7 * M_PI / 12.0));
                m_32apsk[7].imag()  = (r2 * sin(-7 * M_PI / 12.0));
                m_32apsk[8].real()  = (r3 * cos(M_PI / 8.0));
                m_32apsk[8].imag()  = (r3 * sin(M_PI / 8.0));
                m_32apsk[9].real()  = (r3 * cos(3 * M_PI / 8.0));
                m_32apsk[9].imag()  = (r3 * sin(3 * M_PI / 8.0));
                m_32apsk[10].real() = (r3 * cos(-M_PI / 4.0));
                m_32apsk[10].imag() = (r3 * sin(-M_PI / 4.0));
                m_32apsk[11].real() = (r3 * cos(-M_PI / 2.0));
                m_32apsk[11].imag() = (r3 * sin(-M_PI / 2.0));
                m_32apsk[12].real() = (r3 * cos(3 * M_PI / 4.0));
                m_32apsk[12].imag() = (r3 * sin(3 * M_PI / 4.0));
                m_32apsk[13].real() = (r3 * cos(M_PI / 2.0));
                m_32apsk[13].imag() = (r3 * sin(M_PI / 2.0));
                m_32apsk[14].real() = (r3 * cos(-7 * M_PI / 8.0));
                m_32apsk[14].imag() = (r3 * sin(-7 * M_PI / 8.0));
                m_32apsk[15].real() = (r3 * cos(-5 * M_PI / 8.0));
                m_32apsk[15].imag() = (r3 * sin(-5 * M_PI / 8.0));
                m_32apsk[16].real() = (r2 * cos(M_PI / 12.0));
                m_32apsk[16].imag() = (r2 * sin(M_PI / 12.0));
                m_32apsk[17].real() = (r1 * cos(M_PI / 4.0));
                m_32apsk[17].imag() = (r1 * sin(M_PI / 4.0));
                m_32apsk[18].real() = (r2 * cos(-M_PI / 12.0));
                m_32apsk[18].imag() = (r2 * sin(-M_PI / 12.0));
                m_32apsk[19].real() = (r1 * cos(-M_PI / 4.0));
                m_32apsk[19].imag() = (r1 * sin(-M_PI / 4.0));
                m_32apsk[20].real() = (r2 * cos(11 * M_PI / 12.0));
                m_32apsk[20].imag() = (r2 * sin(11 * M_PI / 12.0));
                m_32apsk[21].real() = (r1 * cos(3 * M_PI / 4.0));
                m_32apsk[21].imag() = (r1 * sin(3 * M_PI / 4.0));
                m_32apsk[22].real() = (r2 * cos(-11 * M_PI / 12.0));
                m_32apsk[22].imag() = (r2 * sin(-11 * M_PI / 12.0));
                m_32apsk[23].real() = (r1 * cos(-3 * M_PI / 4.0));
                m_32apsk[23].imag() = (r1 * sin(-3 * M_PI / 4.0));
                m_32apsk[24].real() = (r3 * cos(0.0));
                m_32apsk[24].imag() = (r3 * sin(0.0));
                m_32apsk[25].real() = (r3 * cos(M_PI / 4.0));
                m_32apsk[25].imag() = (r3 * sin(M_PI / 4.0));
                m_32apsk[26].real() = (r3 * cos(-M_PI / 8.0));
                m_32apsk[26].imag() = (r3 * sin(-M_PI / 8.0));
                m_32apsk[27].real() = (r3 * cos(-3 * M_PI / 8.0));
                m_32apsk[27].imag() = (r3 * sin(-3 * M_PI / 8.0));
                m_32apsk[28].real() = (r3 * cos(7 * M_PI / 8.0));
                m_32apsk[28].imag() = (r3 * sin(7 * M_PI / 8.0));
                m_32apsk[29].real() = (r3 * cos(5 * M_PI / 8.0));
                m_32apsk[29].imag() = (r3 * sin(5 * M_PI / 8.0));
                m_32apsk[30].real() = (r3 * cos(M_PI));
                m_32apsk[30].imag() = (r3 * sin(M_PI));
                m_32apsk[31].real() = (r3 * cos(-3 * M_PI / 4.0));
                m_32apsk[31].imag() = (r3 * sin(-3 * M_PI / 4.0));
                break;
            case gr::dvbs2::MOD_4_12_16APSK:
                r3 = m;
                if (framesize == gr::dvbs2::FECFRAME_NORMAL)
                {
                    switch(rate)
                    {
                        case gr::dvbs2::C2_3:
                            r1 = r3 / 5.55;
                            r2 = r1 * 2.85;
                            break;
                        default:
                            r1 = 0;
                            r2 = 0;
                            break;
                    }
                }
                else
                {
                    switch(rate)
                    {
                        case gr::dvbs2::C2_3:
                            r1 = r3 / 5.54;
                            r2 = r1 * 2.84;
                            break;
                        case gr::dvbs2::C32_45:
                            r1 = r3 / 5.26;
                            r2 = r1 * 2.84;
                            break;
                        default:
                            r1 = 0;
                            r2 = 0;
                            break;
                    }
                }
                m_32apsk[0].real()  = (r3 * cos(11 * M_PI / 16.0));
                m_32apsk[0].imag()  = (r3 * sin(11 * M_PI / 16.0));
                m_32apsk[1].real()  = (r3 * cos(9 * M_PI / 16.0));
                m_32apsk[1].imag()  = (r3 * sin(9 * M_PI / 16.0));
                m_32apsk[2].real()  = (r3 * cos(5 * M_PI / 16.0));
                m_32apsk[2].imag()  = (r3 * sin(5 * M_PI / 16.0));
                m_32apsk[3].real()  = (r3 * cos(7 * M_PI / 16.0));
                m_32apsk[3].imag()  = (r3 * sin(7 * M_PI / 16.0));
                m_32apsk[4].real()  = (r2 * cos(3 * M_PI / 4.0));
                m_32apsk[4].imag()  = (r2 * sin(3 * M_PI / 4.0));
                m_32apsk[5].real()  = (r2 * cos(7 * M_PI / 12.0));
                m_32apsk[5].imag()  = (r2 * sin(7 * M_PI / 12.0));
                m_32apsk[6].real()  = (r2 * cos(M_PI / 4.0));
                m_32apsk[6].imag()  = (r2 * sin(M_PI / 4.0));
                m_32apsk[7].real()  = (r2 * cos(5 * M_PI / 12.0));
                m_32apsk[7].imag()  = (r2 * sin(5 * M_PI / 12.0));
                m_32apsk[8].real()  = (r3 * cos(13 * M_PI / 16.0));
                m_32apsk[8].imag()  = (r3 * sin(13 * M_PI / 16.0));
                m_32apsk[9].real()  = (r3 * cos(15 * M_PI / 16.0));
                m_32apsk[9].imag()  = (r3 * sin(15 * M_PI / 16.0));
                m_32apsk[10].real() = (r3 * cos(3 * M_PI / 16.0));
                m_32apsk[10].imag() = (r3 * sin(3 * M_PI / 16.0));
                m_32apsk[11].real() = (r3 * cos(M_PI / 16.0));
                m_32apsk[11].imag() = (r3 * sin(M_PI / 16.0));
                m_32apsk[12].real() = (r2 * cos(11 * M_PI / 12.0));
                m_32apsk[12].imag() = (r2 * sin(11 * M_PI / 12.0));
                m_32apsk[13].real() = (r1 * cos(3 * M_PI / 4.0));
                m_32apsk[13].imag() = (r1 * sin(3 * M_PI / 4.0));
                m_32apsk[14].real() = (r2 * cos(M_PI / 12.0));
                m_32apsk[14].imag() = (r2 * sin(M_PI / 12.0));
                m_32apsk[15].real() = (r1 * cos(M_PI / 4.0));
                m_32apsk[15].imag() = (r1 * sin(M_PI / 4.0));
                m_32apsk[16].real() = (r3 * cos(21 * M_PI / 16.0));
                m_32apsk[16].imag() = (r3 * sin(21 * M_PI / 16.0));
                m_32apsk[17].real() = (r3 * cos(23 * M_PI / 16.0));
                m_32apsk[17].imag() = (r3 * sin(23 * M_PI / 16.0));
                m_32apsk[18].real() = (r3 * cos(27 * M_PI / 16.0));
                m_32apsk[18].imag() = (r3 * sin(27 * M_PI / 16.0));
                m_32apsk[19].real() = (r3 * cos(25 * M_PI / 16.0));
                m_32apsk[19].imag() = (r3 * sin(25 * M_PI / 16.0));
                m_32apsk[20].real() = (r2 * cos(5 * M_PI / 4.0));
                m_32apsk[20].imag() = (r2 * sin(5 * M_PI / 4.0));
                m_32apsk[21].real() = (r2 * cos(17 * M_PI / 12.0));
                m_32apsk[21].imag() = (r2 * sin(17 * M_PI / 12.0));
                m_32apsk[22].real() = (r2 * cos(7 * M_PI / 4.0));
                m_32apsk[22].imag() = (r2 * sin(7 * M_PI / 4.0));
                m_32apsk[23].real() = (r2 * cos(19 * M_PI / 12.0));
                m_32apsk[23].imag() = (r2 * sin(19 * M_PI / 12.0));
                m_32apsk[24].real() = (r3 * cos(19 * M_PI / 16.0));
                m_32apsk[24].imag() = (r3 * sin(19 * M_PI / 16.0));
                m_32apsk[25].real() = (r3 * cos(17 * M_PI / 16.0));
                m_32apsk[25].imag() = (r3 * sin(17 * M_PI / 16.0));
                m_32apsk[26].real() = (r3 * cos(29 * M_PI / 16.0));
                m_32apsk[26].imag() = (r3 * sin(29 * M_PI / 16.0));
                m_32apsk[27].real() = (r3 * cos(31 * M_PI / 16.0));
                m_32apsk[27].imag() = (r3 * sin(31 * M_PI / 16.0));
                m_32apsk[28].real() = (r2 * cos(13 * M_PI / 12.0));
                m_32apsk[28].imag() = (r2 * sin(13 * M_PI / 12.0));
                m_32apsk[29].real() = (r1 * cos(5 * M_PI / 4.0));
                m_32apsk[29].imag() = (r1 * sin(5 * M_PI / 4.0));
                m_32apsk[30].real() = (r2 * cos(23 * M_PI / 12.0));
                m_32apsk[30].imag() = (r2 * sin(23 * M_PI / 12.0));
                m_32apsk[31].real() = (r1 * cos(7 * M_PI / 4.0));
                m_32apsk[31].imag() = (r1 * sin(7 * M_PI / 4.0));
                break;
            case gr::dvbs2::MOD_4_8_4_16APSK:
                r4 = m;
                switch(rate)
                {
                    case gr::dvbs2::C128_180:
                        r1 = r4 / 5.6;
                        r3 = r1 * 2.99;
                        r2 = r1 * 2.6;
                        break;
                    case gr::dvbs2::C132_180:
                        r1 = r4 / 5.6;
                        r3 = r1 * 2.86;
                        r2 = r1 * 2.6;
                        break;
                    case gr::dvbs2::C140_180:
                        r1 = r4 / 5.6;
                        r3 = r1 * 3.08;
                        r2 = r1 * 2.6;
                        break;
                    default:
                        r1 = 0;
                        r2 = 0;
                        r3 = 0;
                        break;
                }
                m_32apsk[0].real()  = (r1 * cos(M_PI / 4.0));
                m_32apsk[0].imag()  = (r1 * sin(M_PI / 4.0));
                m_32apsk[1].real()  = (r4 * cos(7 * M_PI / 16.0));
                m_32apsk[1].imag()  = (r4 * sin(7 * M_PI / 16.0));
                m_32apsk[2].real()  = (r1 * cos(7 * M_PI / 4.0));
                m_32apsk[2].imag()  = (r1 * sin(7 * M_PI / 4.0));
                m_32apsk[3].real()  = (r4 * cos(25 * M_PI / 16.0));
                m_32apsk[3].imag()  = (r4 * sin(25 * M_PI / 16.0));
                m_32apsk[4].real()  = (r1 * cos(3 * M_PI / 4.0));
                m_32apsk[4].imag()  = (r1 * sin(3 * M_PI / 4.0));
                m_32apsk[5].real()  = (r4 * cos(9 * M_PI / 16.0));
                m_32apsk[5].imag()  = (r4 * sin(9 * M_PI / 16.0));
                m_32apsk[6].real()  = (r1 * cos(5 * M_PI / 4.0));
                m_32apsk[6].imag()  = (r1 * sin(5 * M_PI / 4.0));
                m_32apsk[7].real()  = (r4 * cos(23 * M_PI / 16.0));
                m_32apsk[7].imag()  = (r4 * sin(23 * M_PI / 16.0));
                m_32apsk[8].real()  = (r2 * cos(M_PI / 12.0));
                m_32apsk[8].imag()  = (r2 * sin(M_PI / 12.0));
                m_32apsk[9].real()  = (r4 * cos(M_PI / 16.0));
                m_32apsk[9].imag()  = (r4 * sin(M_PI / 16.0));
                m_32apsk[10].real() = (r2 * cos(23 * M_PI / 12.0));
                m_32apsk[10].imag() = (r2 * sin(23 * M_PI / 12.0));
                m_32apsk[11].real() = (r4 * cos(31 * M_PI / 16.0));
                m_32apsk[11].imag() = (r4 * sin(31 * M_PI / 16.0));
                m_32apsk[12].real() = (r2 * cos(11 * M_PI / 12.0));
                m_32apsk[12].imag() = (r2 * sin(11 * M_PI / 12.0));
                m_32apsk[13].real() = (r4 * cos(15 * M_PI / 16.0));
                m_32apsk[13].imag() = (r4 * sin(15 * M_PI / 16.0));
                m_32apsk[14].real() = (r2 * cos(13 * M_PI / 12.0));
                m_32apsk[14].imag() = (r2 * sin(13 * M_PI / 12.0));
                m_32apsk[15].real() = (r4 * cos(17 * M_PI / 16.0));
                m_32apsk[15].imag() = (r4 * sin(17 * M_PI / 16.0));
                m_32apsk[16].real() = (r2 * cos(5 * M_PI / 12.0));
                m_32apsk[16].imag() = (r2 * sin(5 * M_PI / 12.0));
                m_32apsk[17].real() = (r4 * cos(5 * M_PI / 16.0));
                m_32apsk[17].imag() = (r4 * sin(5 * M_PI / 16.0));
                m_32apsk[18].real() = (r2 * cos(19 * M_PI / 12.0));
                m_32apsk[18].imag() = (r2 * sin(19 * M_PI / 12.0));
                m_32apsk[19].real() = (r4 * cos(27 * M_PI / 16.0));
                m_32apsk[19].imag() = (r4 * sin(27 * M_PI / 16.0));
                m_32apsk[20].real() = (r2 * cos(7 * M_PI / 12.0));
                m_32apsk[20].imag() = (r2 * sin(7 * M_PI / 12.0));
                m_32apsk[21].real() = (r4 * cos(11 * M_PI / 16.0));
                m_32apsk[21].imag() = (r4 * sin(11 * M_PI / 16.0));
                m_32apsk[22].real() = (r2 * cos(17 * M_PI / 12.0));
                m_32apsk[22].imag() = (r2 * sin(17 * M_PI / 12.0));
                m_32apsk[23].real() = (r4 * cos(21 * M_PI / 16.0));
                m_32apsk[23].imag() = (r4 * sin(21 * M_PI / 16.0));
                m_32apsk[24].real() = (r3 * cos(M_PI / 4.0));
                m_32apsk[24].imag() = (r3 * sin(M_PI / 4.0));
                m_32apsk[25].real() = (r4 * cos(3 * M_PI / 16.0));
                m_32apsk[25].imag() = (r4 * sin(3 * M_PI / 16.0));
                m_32apsk[26].real() = (r3 * cos(7 * M_PI / 4.0));
                m_32apsk[26].imag() = (r3 * sin(7 * M_PI / 4.0));
                m_32apsk[27].real() = (r4 * cos(29 * M_PI / 16.0));
                m_32apsk[27].imag() = (r4 * sin(29 * M_PI / 16.0));
                m_32apsk[28].real() = (r3 * cos(3 * M_PI / 4.0));
                m_32apsk[28].imag() = (r3 * sin(3 * M_PI / 4.0));
                m_32apsk[29].real() = (r4 * cos(13 * M_PI / 16.0));
                m_32apsk[29].imag() = (r4 * sin(13 * M_PI / 16.0));
                m_32apsk[30].real() = (r3 * cos(5 * M_PI / 4.0));
                m_32apsk[30].imag() = (r3 * sin(5 * M_PI / 4.0));
                m_32apsk[31].real() = (r4 * cos(19 * M_PI / 16.0));
                m_32apsk[31].imag() = (r4 * sin(19 * M_PI / 16.0));
                break;
            case gr::dvbs2::MOD_64APSK:
                r4 = m;
                switch(rate)
                {
                    case gr::dvbs2::C128_180:
                        r1 = r4 / 3.95;
                        r3 = r1 * 2.72;
                        r2 = r1 * 1.88;
                        break;
                    default:
                        r1 = 0;
                        r2 = 0;
                        r3 = 0;
                        break;
                }
                m_64apsk[0].real()  = (r1 * cos(M_PI / 16.0));
                m_64apsk[0].imag()  = (r1 * sin(M_PI / 16.0));
                m_64apsk[1].real()  = (r1 * cos(3 * M_PI / 16.0));
                m_64apsk[1].imag()  = (r1 * sin(3 * M_PI / 16.0));
                m_64apsk[2].real()  = (r1 * cos(7 * M_PI / 16.0));
                m_64apsk[2].imag()  = (r1 * sin(7 * M_PI / 16.0));
                m_64apsk[3].real()  = (r1 * cos(5 * M_PI / 16.0));
                m_64apsk[3].imag()  = (r1 * sin(5 * M_PI / 16.0));
                m_64apsk[4].real()  = (r1 * cos(31 * M_PI / 16.0));
                m_64apsk[4].imag()  = (r1 * sin(31 * M_PI / 16.0));
                m_64apsk[5].real()  = (r1 * cos(29 * M_PI / 16.0));
                m_64apsk[5].imag()  = (r1 * sin(29 * M_PI / 16.0));
                m_64apsk[6].real()  = (r1 * cos(25 * M_PI / 16.0));
                m_64apsk[6].imag()  = (r1 * sin(25 * M_PI / 16.0));
                m_64apsk[7].real()  = (r1 * cos(27 * M_PI / 16.0));
                m_64apsk[7].imag()  = (r1 * sin(27 * M_PI / 16.0));
                m_64apsk[8].real()  = (r1 * cos(15 * M_PI / 16.0));
                m_64apsk[8].imag()  = (r1 * sin(15 * M_PI / 16.0));
                m_64apsk[9].real()  = (r1 * cos(13 * M_PI / 16.0));
                m_64apsk[9].imag()  = (r1 * sin(13 * M_PI / 16.0));
                m_64apsk[10].real() = (r1 * cos(9 * M_PI / 16.0));
                m_64apsk[10].imag() = (r1 * sin(9 * M_PI / 16.0));
                m_64apsk[11].real() = (r1 * cos(11 * M_PI / 16.0));
                m_64apsk[11].imag() = (r1 * sin(11 * M_PI / 16.0));
                m_64apsk[12].real() = (r1 * cos(17 * M_PI / 16.0));
                m_64apsk[12].imag() = (r1 * sin(17 * M_PI / 16.0));
                m_64apsk[13].real() = (r1 * cos(19 * M_PI / 16.0));
                m_64apsk[13].imag() = (r1 * sin(19 * M_PI / 16.0));
                m_64apsk[14].real() = (r1 * cos(23 * M_PI / 16.0));
                m_64apsk[14].imag() = (r1 * sin(23 * M_PI / 16.0));
                m_64apsk[15].real() = (r1 * cos(21 * M_PI / 16.0));
                m_64apsk[15].imag() = (r1 * sin(21 * M_PI / 16.0));
                m_64apsk[16].real() = (r2 * cos(M_PI / 16.0));
                m_64apsk[16].imag() = (r2 * sin(M_PI / 16.0));
                m_64apsk[17].real() = (r2 * cos(3 * M_PI / 16.0));
                m_64apsk[17].imag() = (r2 * sin(3 * M_PI / 16.0));
                m_64apsk[18].real() = (r2 * cos(7 * M_PI / 16.0));
                m_64apsk[18].imag() = (r2 * sin(7 * M_PI / 16.0));
                m_64apsk[19].real() = (r2 * cos(5 * M_PI / 16.0));
                m_64apsk[19].imag() = (r2 * sin(5 * M_PI / 16.0));
                m_64apsk[20].real() = (r2 * cos(31 * M_PI / 16.0));
                m_64apsk[20].imag() = (r2 * sin(31* M_PI / 16.0));
                m_64apsk[21].real() = (r2 * cos(29 * M_PI / 16.0));
                m_64apsk[21].imag() = (r2 * sin(29 * M_PI / 16.0));
                m_64apsk[22].real() = (r2 * cos(25 * M_PI / 16.0));
                m_64apsk[22].imag() = (r2 * sin(25 * M_PI / 16.0));
                m_64apsk[23].real() = (r2 * cos(27 * M_PI / 16.0));
                m_64apsk[23].imag() = (r2 * sin(27 * M_PI / 16.0));
                m_64apsk[24].real() = (r2 * cos(15 * M_PI / 16.0));
                m_64apsk[24].imag() = (r2 * sin(15 * M_PI / 16.0));
                m_64apsk[25].real() = (r2 * cos(13 * M_PI / 16.0));
                m_64apsk[25].imag() = (r2 * sin(13 * M_PI / 16.0));
                m_64apsk[26].real() = (r2 * cos(9 * M_PI / 16.0));
                m_64apsk[26].imag() = (r2 * sin(9 * M_PI / 16.0));
                m_64apsk[27].real() = (r2 * cos(11 * M_PI / 16.0));
                m_64apsk[27].imag() = (r2 * sin(11 * M_PI / 16.0));
                m_64apsk[28].real() = (r2 * cos(17 * M_PI / 16.0));
                m_64apsk[28].imag() = (r2 * sin(17 * M_PI / 16.0));
                m_64apsk[29].real() = (r2 * cos(19 * M_PI / 16.0));
                m_64apsk[29].imag() = (r2 * sin(19 * M_PI / 16.0));
                m_64apsk[30].real() = (r2 * cos(23 * M_PI / 16.0));
                m_64apsk[30].imag() = (r2 * sin(23 * M_PI / 16.0));
                m_64apsk[31].real() = (r2 * cos(21 * M_PI / 16.0));
                m_64apsk[31].imag() = (r2 * sin(21 * M_PI / 16.0));
                m_64apsk[32].real() = (r4 * cos(M_PI / 16.0));
                m_64apsk[32].imag() = (r4 * sin(M_PI / 16.0));
                m_64apsk[33].real() = (r4 * cos(3 * M_PI / 16.0));
                m_64apsk[33].imag() = (r4 * sin(3 * M_PI / 16.0));
                m_64apsk[34].real() = (r4 * cos(7 * M_PI / 16.0));
                m_64apsk[34].imag() = (r4 * sin(7 * M_PI / 16.0));
                m_64apsk[35].real() = (r4 * cos(5 * M_PI / 16.0));
                m_64apsk[35].imag() = (r4 * sin(5 * M_PI / 16.0));
                m_64apsk[36].real() = (r4 * cos(31 * M_PI / 16.0));
                m_64apsk[36].imag() = (r4 * sin(31 * M_PI / 16.0));
                m_64apsk[37].real() = (r4 * cos(29 * M_PI / 16.0));
                m_64apsk[37].imag() = (r4 * sin(29 * M_PI / 16.0));
                m_64apsk[38].real() = (r4 * cos(25 * M_PI / 16.0));
                m_64apsk[38].imag() = (r4 * sin(25 * M_PI / 16.0));
                m_64apsk[39].real() = (r4 * cos(27 * M_PI / 16.0));
                m_64apsk[39].imag() = (r4 * sin(27 * M_PI / 16.0));
                m_64apsk[40].real() = (r4 * cos(15 * M_PI / 16.0));
                m_64apsk[40].imag() = (r4 * sin(15 * M_PI / 16.0));
                m_64apsk[41].real() = (r4 * cos(13 * M_PI / 16.0));
                m_64apsk[41].imag() = (r4 * sin(13 * M_PI / 16.0));
                m_64apsk[42].real() = (r4 * cos(9 * M_PI / 16.0));
                m_64apsk[42].imag() = (r4 * sin(9 * M_PI / 16.0));
                m_64apsk[43].real() = (r4 * cos(11 * M_PI / 16.0));
                m_64apsk[43].imag() = (r4 * sin(11 * M_PI / 16.0));
                m_64apsk[44].real() = (r4 * cos(17 * M_PI / 16.0));
                m_64apsk[44].imag() = (r4 * sin(17 * M_PI / 16.0));
                m_64apsk[45].real() = (r4 * cos(19 * M_PI / 16.0));
                m_64apsk[45].imag() = (r4 * sin(19 * M_PI / 16.0));
                m_64apsk[46].real() = (r4 * cos(23 * M_PI / 16.0));
                m_64apsk[46].imag() = (r4 * sin(23 * M_PI / 16.0));
                m_64apsk[47].real() = (r4 * cos(21 * M_PI / 16.0));
                m_64apsk[47].imag() = (r4 * sin(21 * M_PI / 16.0));
                m_64apsk[48].real() = (r3 * cos(M_PI / 16.0));
                m_64apsk[48].imag() = (r3 * sin(M_PI / 16.0));
                m_64apsk[49].real() = (r3 * cos(3 * M_PI / 16.0));
                m_64apsk[49].imag() = (r3 * sin(3 * M_PI / 16.0));
                m_64apsk[50].real() = (r3 * cos(7 * M_PI / 16.0));
                m_64apsk[50].imag() = (r3 * sin(7 * M_PI / 16.0));
                m_64apsk[51].real() = (r3 * cos(5 * M_PI / 16.0));
                m_64apsk[51].imag() = (r3 * sin(5 * M_PI / 16.0));
                m_64apsk[52].real() = (r3 * cos(31 * M_PI / 16.0));
                m_64apsk[52].imag() = (r3 * sin(31 * M_PI / 16.0));
                m_64apsk[53].real() = (r3 * cos(29 * M_PI / 16.0));
                m_64apsk[53].imag() = (r3 * sin(29 * M_PI / 16.0));
                m_64apsk[54].real() = (r3 * cos(25 * M_PI / 16.0));
                m_64apsk[54].imag() = (r3 * sin(25 * M_PI / 16.0));
                m_64apsk[55].real() = (r3 * cos(27 * M_PI / 16.0));
                m_64apsk[55].imag() = (r3 * sin(27 * M_PI / 16.0));
                m_64apsk[56].real() = (r3 * cos(15 * M_PI / 16.0));
                m_64apsk[56].imag() = (r3 * sin(15 * M_PI / 16.0));
                m_64apsk[57].real() = (r3 * cos(13 * M_PI / 16.0));
                m_64apsk[57].imag() = (r3 * sin(13 * M_PI / 16.0));
                m_64apsk[58].real() = (r3 * cos(9 * M_PI / 16.0));
                m_64apsk[58].imag() = (r3 * sin(9 * M_PI / 16.0));
                m_64apsk[59].real() = (r3 * cos(11 * M_PI / 16.0));
                m_64apsk[59].imag() = (r3 * sin(11 * M_PI / 16.0));
                m_64apsk[60].real() = (r3 * cos(17 * M_PI / 16.0));
                m_64apsk[60].imag() = (r3 * sin(17 * M_PI / 16.0));
                m_64apsk[61].real() = (r3 * cos(19 * M_PI / 16.0));
                m_64apsk[61].imag() = (r3 * sin(19 * M_PI / 16.0));
                m_64apsk[62].real() = (r3 * cos(23 * M_PI / 16.0));
                m_64apsk[62].imag() = (r3 * sin(23 * M_PI / 16.0));
                m_64apsk[63].real() = (r3 * cos(21 * M_PI / 16.0));
                m_64apsk[63].imag() = (r3 * sin(21 * M_PI / 16.0));
                break;
            case gr::dvbs2::MOD_8_16_20_20APSK:
                r4 = m;
                switch(rate)
                {
                    case gr::dvbs2::C7_9:
                    case gr::dvbs2::C4_5:
                        r1 = r4 / 5.2;
                        r3 = r1 * 3.6;
                        r2 = r1 * 2.2;
                        break;
                    case gr::dvbs2::C5_6:
                        r1 = r4 / 5.0;
                        r3 = r1 * 3.5;
                        r2 = r1 * 2.2;
                        break;
                    default:
                        r1 = 0;
                        r2 = 0;
                        r3 = 0;
                        break;
                }
                m_64apsk[0].real()   = (r2 * cos(25 * M_PI / 16.0));
                m_64apsk[0].imag()   = (r2 * sin(25 * M_PI / 16.0));
                m_64apsk[1].real()   = (r4 * cos(7 * M_PI / 4.0));
                m_64apsk[1].imag()   = (r4 * sin(7 * M_PI / 4.0));
                m_64apsk[2].real()   = (r2 * cos(27 * M_PI / 16.0));
                m_64apsk[2].imag()   = (r2 * sin(27 * M_PI / 16.0));
                m_64apsk[3].real()   = (r3 * cos(7 * M_PI / 4.0));
                m_64apsk[3].imag()   = (r3 * sin(7 * M_PI / 4.0));
                m_64apsk[4].real()   = (r4 * cos(31 * M_PI / 20.0));
                m_64apsk[4].imag()   = (r4 * sin(31 * M_PI / 20.0));
                m_64apsk[5].real()   = (r4 * cos(33 * M_PI / 20.0));
                m_64apsk[5].imag()   = (r4 * sin(33 * M_PI / 20.0));
                m_64apsk[6].real()   = (r3 * cos(31 * M_PI / 20.0));
                m_64apsk[6].imag()   = (r3 * sin(31 * M_PI / 20.0));
                m_64apsk[7].real()   = (r3 * cos(33 * M_PI / 20.0));
                m_64apsk[7].imag()   = (r3 * sin(33 * M_PI / 20.0));
                m_64apsk[8].real()   = (r2 * cos(23 * M_PI / 16.0));
                m_64apsk[8].imag()   = (r2 * sin(23 * M_PI / 16.0));
                m_64apsk[9].real()   = (r4 * cos(5 * M_PI / 4.0));
                m_64apsk[9].imag()   = (r4 * sin(5 * M_PI / 4.0));
                m_64apsk[10].real()   = (r2 * cos(21 * M_PI / 16.0));
                m_64apsk[10].imag()   = (r2 * sin(21 * M_PI / 16.0));
                m_64apsk[11].real()   = (r3 * cos(5 * M_PI / 4.0));
                m_64apsk[11].imag()   = (r3 * sin(5 * M_PI / 4.0));
                m_64apsk[12].real()   = (r4 * cos(29 * M_PI / 20.0));
                m_64apsk[12].imag()   = (r4 * sin(29 * M_PI / 20.0));
                m_64apsk[13].real()   = (r4 * cos(27 * M_PI / 20.0));
                m_64apsk[13].imag()   = (r4 * sin(27 * M_PI / 20.0));
                m_64apsk[14].real()   = (r3 * cos(29 * M_PI / 20.0));
                m_64apsk[14].imag()   = (r3 * sin(29 * M_PI / 20.0));
                m_64apsk[15].real()   = (r3 * cos(27 * M_PI / 20.0));
                m_64apsk[15].imag()   = (r3 * sin(27 * M_PI / 20.0));
                m_64apsk[16].real()   = (r1 * cos(13 * M_PI / 8.0));
                m_64apsk[16].imag()   = (r1 * sin(13 * M_PI / 8.0));
                m_64apsk[17].real()   = (r4 * cos(37 * M_PI / 20.0));
                m_64apsk[17].imag()   = (r4 * sin(37 * M_PI / 20.0));
                m_64apsk[18].real()   = (r2 * cos(29 * M_PI / 16.0));
                m_64apsk[18].imag()   = (r2 * sin(29 * M_PI / 16.0));
                m_64apsk[19].real()   = (r3 * cos(37 * M_PI / 20.0));
                m_64apsk[19].imag()   = (r3 * sin(37 * M_PI / 20.0));
                m_64apsk[20].real()   = (r1 * cos(15 * M_PI / 8.0));
                m_64apsk[20].imag()   = (r1 * sin(15 * M_PI / 8.0));
                m_64apsk[21].real()   = (r4 * cos(39 * M_PI / 20.0));
                m_64apsk[21].imag()   = (r4 * sin(39 * M_PI / 20.0));
                m_64apsk[22].real()   = (r2 * cos(31 * M_PI / 16.0));
                m_64apsk[22].imag()   = (r2 * sin(31 * M_PI / 16.0));
                m_64apsk[23].real()   = (r3 * cos(39 * M_PI / 20.0));
                m_64apsk[23].imag()   = (r3 * sin(39 * M_PI / 20.0));
                m_64apsk[24].real()   = (r1 * cos(11 * M_PI / 8.0));
                m_64apsk[24].imag()   = (r1 * sin(11 * M_PI / 8.0));
                m_64apsk[25].real()   = (r4 * cos(23 * M_PI / 20.0));
                m_64apsk[25].imag()   = (r4 * sin(23 * M_PI / 20.0));
                m_64apsk[26].real()   = (r2 * cos(19 * M_PI / 16.0));
                m_64apsk[26].imag()   = (r2 * sin(19 * M_PI / 16.0));
                m_64apsk[27].real()   = (r3 * cos(23 * M_PI / 20.0));
                m_64apsk[27].imag()   = (r3 * sin(23 * M_PI / 20.0));
                m_64apsk[28].real()   = (r1 * cos(9 * M_PI / 8.0));
                m_64apsk[28].imag()   = (r1 * sin(9 * M_PI / 8.0));
                m_64apsk[29].real()   = (r4 * cos(21 * M_PI / 20.0));
                m_64apsk[29].imag()   = (r4 * sin(21 * M_PI / 20.0));
                m_64apsk[30].real()   = (r2 * cos(17 * M_PI / 16.0));
                m_64apsk[30].imag()   = (r2 * sin(17 * M_PI / 16.0));
                m_64apsk[31].real()   = (r3 * cos(21 * M_PI / 20.0));
                m_64apsk[31].imag()   = (r3 * sin(21 * M_PI / 20.0));
                m_64apsk[32].real()   = (r2 * cos(7 * M_PI / 16.0));
                m_64apsk[32].imag()   = (r2 * sin(7 * M_PI / 16.0));
                m_64apsk[33].real()   = (r4 * cos(M_PI / 4.0));
                m_64apsk[33].imag()   = (r4 * sin(M_PI / 4.0));
                m_64apsk[34].real()   = (r2 * cos(5 * M_PI / 16.0));
                m_64apsk[34].imag()   = (r2 * sin(5 * M_PI / 16.0));
                m_64apsk[35].real()   = (r3 * cos(M_PI / 4.0));
                m_64apsk[35].imag()   = (r3 * sin(M_PI / 4.0));
                m_64apsk[36].real()   = (r4 * cos(9 * M_PI / 20.0));
                m_64apsk[36].imag()   = (r4 * sin(9 * M_PI / 20.0));
                m_64apsk[37].real()   = (r4 * cos(7 * M_PI / 20.0));
                m_64apsk[37].imag()   = (r4 * sin(7 * M_PI / 20.0));
                m_64apsk[38].real()   = (r3 * cos(9 * M_PI / 20.0));
                m_64apsk[38].imag()   = (r3 * sin(9 * M_PI / 20.0));
                m_64apsk[39].real()   = (r3 * cos(7 * M_PI / 20.0));
                m_64apsk[39].imag()   = (r3 * sin(7 * M_PI / 20.0));
                m_64apsk[40].real()   = (r2 * cos(9 * M_PI / 16.0));
                m_64apsk[40].imag()   = (r2 * sin(9 * M_PI / 16.0));
                m_64apsk[41].real()   = (r4 * cos(3 * M_PI / 4.0));
                m_64apsk[41].imag()   = (r4 * sin(3 * M_PI / 4.0));
                m_64apsk[42].real()   = (r2 * cos(11 * M_PI / 16.0));
                m_64apsk[42].imag()   = (r2 * sin(11 * M_PI / 16.0));
                m_64apsk[43].real()   = (r3 * cos(3 * M_PI / 4.0));
                m_64apsk[43].imag()   = (r3 * sin(3 * M_PI / 4.0));
                m_64apsk[44].real()   = (r4 * cos(11 * M_PI / 20.0));
                m_64apsk[44].imag()   = (r4 * sin(11 * M_PI / 20.0));
                m_64apsk[45].real()   = (r4 * cos(13 * M_PI / 20.0));
                m_64apsk[45].imag()   = (r4 * sin(13 * M_PI / 20.0));
                m_64apsk[46].real()   = (r3 * cos(11 * M_PI / 20.0));
                m_64apsk[46].imag()   = (r3 * sin(11 * M_PI / 20.0));
                m_64apsk[47].real()   = (r3 * cos(13 * M_PI / 20.0));
                m_64apsk[47].imag()   = (r3 * sin(13 * M_PI / 20.0));
                m_64apsk[48].real()   = (r1 * cos(3 * M_PI / 8.0));
                m_64apsk[48].imag()   = (r1 * sin(3 * M_PI / 8.0));
                m_64apsk[49].real()   = (r4 * cos(3 * M_PI / 20.0));
                m_64apsk[49].imag()   = (r4 * sin(3 * M_PI / 20.0));
                m_64apsk[50].real()   = (r2 * cos(3 * M_PI / 16.0));
                m_64apsk[50].imag()   = (r2 * sin(3 * M_PI / 16.0));
                m_64apsk[51].real()   = (r3 * cos(3 * M_PI / 20.0));
                m_64apsk[51].imag()   = (r3 * sin(3 * M_PI / 20.0));
                m_64apsk[52].real()   = (r1 * cos(M_PI / 8.0));
                m_64apsk[52].imag()   = (r1 * sin(M_PI / 8.0));
                m_64apsk[53].real()   = (r4 * cos(M_PI / 20.0));
                m_64apsk[53].imag()   = (r4 * sin(M_PI / 20.0));
                m_64apsk[54].real()   = (r2 * cos(M_PI / 16.0));
                m_64apsk[54].imag()   = (r2 * sin(M_PI / 16.0));
                m_64apsk[55].real()   = (r3 * cos(M_PI / 20.0));
                m_64apsk[55].imag()   = (r3 * sin(M_PI / 20.0));
                m_64apsk[56].real()   = (r1 * cos(5 * M_PI / 8.0));
                m_64apsk[56].imag()   = (r1 * sin(5 * M_PI / 8.0));
                m_64apsk[57].real()   = (r4 * cos(17 * M_PI / 20.0));
                m_64apsk[57].imag()   = (r4 * sin(17 * M_PI / 20.0));
                m_64apsk[58].real()   = (r2 * cos(13 * M_PI / 16.0));
                m_64apsk[58].imag()   = (r2 * sin(13 * M_PI / 16.0));
                m_64apsk[59].real()   = (r3 * cos(17 * M_PI / 20.0));
                m_64apsk[59].imag()   = (r3 * sin(17 * M_PI / 20.0));
                m_64apsk[60].real()   = (r1 * cos(7 * M_PI / 8.0));
                m_64apsk[60].imag()   = (r1 * sin(7 * M_PI / 8.0));
                m_64apsk[61].real()   = (r4 * cos(19 * M_PI / 20.0));
                m_64apsk[61].imag()   = (r4 * sin(19 * M_PI / 20.0));
                m_64apsk[62].real()   = (r2 * cos(15 * M_PI / 16.0));
                m_64apsk[62].imag()   = (r2 * sin(15 * M_PI / 16.0));
                m_64apsk[63].real()   = (r3 * cos(19 * M_PI / 20.0));
                m_64apsk[63].imag()   = (r3 * sin(19 * M_PI / 20.0));
                break;
            case gr::dvbs2::MOD_4_12_20_28APSK:
                r4 = m;
                switch(rate)
                {
                    case gr::dvbs2::C132_180:
                        r1 = r4 / 7.0;
                        r3 = r1 * 4.3;
                        r2 = r1 * 2.4;
                        break;
                    default:
                        r1 = 0;
                        r2 = 0;
                        r3 = 0;
                        break;
                }
                m_64apsk[0].real()   = (r4 * cos(M_PI / 4.0));
                m_64apsk[0].imag()   = (r4 * sin(M_PI / 4.0));
                m_64apsk[1].real()   = (r4 * cos(7 * M_PI / 4.0));
                m_64apsk[1].imag()   = (r4 * sin(7 * M_PI / 4.0));
                m_64apsk[2].real()   = (r4 * cos(3 * M_PI / 4.0));
                m_64apsk[2].imag()   = (r4 * sin(3 * M_PI / 4.0));
                m_64apsk[3].real()   = (r4 * cos(5 * M_PI / 4.0));
                m_64apsk[3].imag()   = (r4 * sin(5 * M_PI / 4.0));
                m_64apsk[4].real()   = (r4 * cos(13 * M_PI / 28.0));
                m_64apsk[4].imag()   = (r4 * sin(13 * M_PI / 28.0));
                m_64apsk[5].real()   = (r4 * cos(43 * M_PI / 28.0));
                m_64apsk[5].imag()   = (r4 * sin(43 * M_PI / 28.0));
                m_64apsk[6].real()   = (r4 * cos(15 * M_PI / 28.0));
                m_64apsk[6].imag()   = (r4 * sin(15 * M_PI / 28.0));
                m_64apsk[7].real()   = (r4 * cos(41 * M_PI / 28.0));
                m_64apsk[7].imag()   = (r4 * sin(41 * M_PI / 28.0));
                m_64apsk[8].real()   = (r4 * cos(M_PI / 28.0));
                m_64apsk[8].imag()   = (r4 * sin(M_PI / 28.0));
                m_64apsk[9].real()   = (r4 * cos(55 * M_PI / 28.0));
                m_64apsk[9].imag()   = (r4 * sin(55 * M_PI / 28.0));
                m_64apsk[10].real()   = (r4 * cos(27 * M_PI / 28.0));
                m_64apsk[10].imag()   = (r4 * sin(27 * M_PI / 28.0));
                m_64apsk[11].real()   = (r4 * cos(29 * M_PI / 28.0));
                m_64apsk[11].imag()   = (r4 * sin(29 * M_PI / 28.0));
                m_64apsk[12].real()   = (r1 * cos(M_PI / 4.0));
                m_64apsk[12].imag()   = (r1 * sin(M_PI / 4.0));
                m_64apsk[13].real()   = (r1 * cos(7 * M_PI / 4.0));
                m_64apsk[13].imag()   = (r1 * sin(7 * M_PI / 4.0));
                m_64apsk[14].real()   = (r1 * cos(3 * M_PI / 4.0));
                m_64apsk[14].imag()   = (r1 * sin(3 * M_PI / 4.0));
                m_64apsk[15].real()   = (r1 * cos(5 * M_PI / 4.0));
                m_64apsk[15].imag()   = (r1 * sin(5 * M_PI / 4.0));
                m_64apsk[16].real()   = (r4 * cos(9 * M_PI / 28.0));
                m_64apsk[16].imag()   = (r4 * sin(9 * M_PI / 28.0));
                m_64apsk[17].real()   = (r4 * cos(47 * M_PI / 28.0));
                m_64apsk[17].imag()   = (r4 * sin(47 * M_PI / 28.0));
                m_64apsk[18].real()   = (r4 * cos(19 * M_PI / 28.0));
                m_64apsk[18].imag()   = (r4 * sin(19 * M_PI / 28.0));
                m_64apsk[19].real()   = (r4 * cos(37 * M_PI / 28.0));
                m_64apsk[19].imag()   = (r4 * sin(37 * M_PI / 28.0));
                m_64apsk[20].real()   = (r4 * cos(11 * M_PI / 28.0));
                m_64apsk[20].imag()   = (r4 * sin(11 * M_PI / 28.0));
                m_64apsk[21].real()   = (r4 * cos(45 * M_PI / 28.0));
                m_64apsk[21].imag()   = (r4 * sin(45 * M_PI / 28.0));
                m_64apsk[22].real()   = (r4 * cos(17 * M_PI / 28.0));
                m_64apsk[22].imag()   = (r4 * sin(17 * M_PI / 28.0));
                m_64apsk[23].real()   = (r4 * cos(39 * M_PI / 28.0));
                m_64apsk[23].imag()   = (r4 * sin(39 * M_PI / 28.0));
                m_64apsk[24].real()   = (r3 * cos(M_PI / 20.0));
                m_64apsk[24].imag()   = (r3 * sin(M_PI / 20.0));
                m_64apsk[25].real()   = (r3 * cos(39 * M_PI / 20.0));
                m_64apsk[25].imag()   = (r3 * sin(39 * M_PI / 20.0));
                m_64apsk[26].real()   = (r3 * cos(19 * M_PI / 20.0));
                m_64apsk[26].imag()   = (r3 * sin(19 * M_PI / 20.0));
                m_64apsk[27].real()   = (r3 * cos(21 * M_PI / 20.0));
                m_64apsk[27].imag()   = (r3 * sin(21 * M_PI / 20.0));
                m_64apsk[28].real()   = (r2 * cos(M_PI / 12.0));
                m_64apsk[28].imag()   = (r2 * sin(M_PI / 12.0));
                m_64apsk[29].real()   = (r2 * cos(23 * M_PI / 12.0));
                m_64apsk[29].imag()   = (r2 * sin(23 * M_PI / 12.0));
                m_64apsk[30].real()   = (r2 * cos(11 * M_PI / 12.0));
                m_64apsk[30].imag()   = (r2 * sin(11 * M_PI / 12.0));
                m_64apsk[31].real()   = (r2 * cos(13 * M_PI / 12.0));
                m_64apsk[31].imag()   = (r2 * sin(13 * M_PI / 12.0));
                m_64apsk[32].real()   = (r4 * cos(5 * M_PI / 28.0));
                m_64apsk[32].imag()   = (r4 * sin(5 * M_PI / 28.0));
                m_64apsk[33].real()   = (r4 * cos(51 * M_PI / 28.0));
                m_64apsk[33].imag()   = (r4 * sin(51 * M_PI / 28.0));
                m_64apsk[34].real()   = (r4 * cos(23 * M_PI / 28.0));
                m_64apsk[34].imag()   = (r4 * sin(23 * M_PI / 28.0));
                m_64apsk[35].real()   = (r4 * cos(33 * M_PI / 28.0));
                m_64apsk[35].imag()   = (r4 * sin(33 * M_PI / 28.0));
                m_64apsk[36].real()   = (r3 * cos(9 * M_PI / 20.0));
                m_64apsk[36].imag()   = (r3 * sin(9 * M_PI / 20.0));
                m_64apsk[37].real()   = (r3 * cos(31 * M_PI / 20.0));
                m_64apsk[37].imag()   = (r3 * sin(31 * M_PI / 20.0));
                m_64apsk[38].real()   = (r3 * cos(11 * M_PI / 20.0));
                m_64apsk[38].imag()   = (r3 * sin(11 * M_PI / 20.0));
                m_64apsk[39].real()   = (r3 * cos(29 * M_PI / 20.0));
                m_64apsk[39].imag()   = (r3 * sin(29 * M_PI / 20.0));
                m_64apsk[40].real()   = (r4 * cos(3 * M_PI / 28.0));
                m_64apsk[40].imag()   = (r4 * sin(3 * M_PI / 28.0));
                m_64apsk[41].real()   = (r4 * cos(53 * M_PI / 28.0));
                m_64apsk[41].imag()   = (r4 * sin(53 * M_PI / 28.0));
                m_64apsk[42].real()   = (r4 * cos(25 * M_PI / 28.0));
                m_64apsk[42].imag()   = (r4 * sin(25 * M_PI / 28.0));
                m_64apsk[43].real()   = (r4 * cos(31 * M_PI / 28.0));
                m_64apsk[43].imag()   = (r4 * sin(31 * M_PI / 28.0));
                m_64apsk[44].real()   = (r2 * cos(5 * M_PI / 12.0));
                m_64apsk[44].imag()   = (r2 * sin(5 * M_PI / 12.0));
                m_64apsk[45].real()   = (r2 * cos(19 * M_PI / 12.0));
                m_64apsk[45].imag()   = (r2 * sin(19 * M_PI / 12.0));
                m_64apsk[46].real()   = (r2 * cos(7 * M_PI / 12.0));
                m_64apsk[46].imag()   = (r2 * sin(7 * M_PI / 12.0));
                m_64apsk[47].real()   = (r2 * cos(17 * M_PI / 12.0));
                m_64apsk[47].imag()   = (r2 * sin(17 * M_PI / 12.0));
                m_64apsk[48].real()   = (r3 * cos(M_PI / 4.0));
                m_64apsk[48].imag()   = (r3 * sin(M_PI / 4.0));
                m_64apsk[49].real()   = (r3 * cos(7 * M_PI / 4.0));
                m_64apsk[49].imag()   = (r3 * sin(7 * M_PI / 4.0));
                m_64apsk[50].real()   = (r3 * cos(3 * M_PI / 4.0));
                m_64apsk[50].imag()   = (r3 * sin(3 * M_PI / 4.0));
                m_64apsk[51].real()   = (r3 * cos(5 * M_PI / 4.0));
                m_64apsk[51].imag()   = (r3 * sin(5 * M_PI / 4.0));
                m_64apsk[52].real()   = (r3 * cos(7 * M_PI / 20.0));
                m_64apsk[52].imag()   = (r3 * sin(7 * M_PI / 20.0));
                m_64apsk[53].real()   = (r3 * cos(33 * M_PI / 20.0));
                m_64apsk[53].imag()   = (r3 * sin(33 * M_PI / 20.0));
                m_64apsk[54].real()   = (r3 * cos(13 * M_PI / 20.0));
                m_64apsk[54].imag()   = (r3 * sin(13 * M_PI / 20.0));
                m_64apsk[55].real()   = (r3 * cos(27 * M_PI / 20.0));
                m_64apsk[55].imag()   = (r3 * sin(27 * M_PI / 20.0));
                m_64apsk[56].real()   = (r3 * cos(3 * M_PI / 20.0));
                m_64apsk[56].imag()   = (r3 * sin(3 * M_PI / 20.0));
                m_64apsk[57].real()   = (r3 * cos(37 * M_PI / 20.0));
                m_64apsk[57].imag()   = (r3 * sin(37 * M_PI / 20.0));
                m_64apsk[58].real()   = (r3 * cos(17 * M_PI / 20.0));
                m_64apsk[58].imag()   = (r3 * sin(17 * M_PI / 20.0));
                m_64apsk[59].real()   = (r3 * cos(23 * M_PI / 20.0));
                m_64apsk[59].imag()   = (r3 * sin(23 * M_PI / 20.0));
                m_64apsk[60].real()   = (r2 * cos(M_PI / 4.0));
                m_64apsk[60].imag()   = (r2 * sin(M_PI / 4.0));
                m_64apsk[61].real()   = (r2 * cos(7 * M_PI / 4.0));
                m_64apsk[61].imag()   = (r2 * sin(7 * M_PI / 4.0));
                m_64apsk[62].real()   = (r2 * cos(3 * M_PI / 4.0));
                m_64apsk[62].imag()   = (r2 * sin(3 * M_PI / 4.0));
                m_64apsk[63].real()   = (r2 * cos(5 * M_PI / 4.0));
                m_64apsk[63].imag()   = (r2 * sin(5 * M_PI / 4.0));
                break;
            case gr::dvbs2::MOD_128APSK:
                r6 = m;
                switch(rate)
                {
                    case gr::dvbs2::C135_180:
                        r1 = r6 / 3.819;
                        r5 = r1 * 2.75;
                        r4 = r1 * 2.681;
                        r3 = r1 * 2.118;
                        r2 = r1 * 1.715;
                        break;
                    case gr::dvbs2::C140_180:
                        r1 = r6 / 3.733;
                        r5 = r1 * 2.75;
                        r4 = r1 * 2.681;
                        r3 = r1 * 2.118;
                        r2 = r1 * 1.715;
                        break;
                    default:
                        r1 = 0;
                        r2 = 0;
                        r3 = 0;
                        r4 = 0;
                        r5 = 0;
                        break;
                }
                m_128apsk[0].real()   = (r1 * cos(83 * M_PI / 1260.0));
                m_128apsk[0].imag()   = (r1 * sin(83 * M_PI / 1260.0));
                m_128apsk[1].real()   = (r6 * cos(11 * M_PI / 105.0));
                m_128apsk[1].imag()   = (r6 * sin(11 * M_PI / 105.0));
                m_128apsk[2].real()   = (r6 * cos(37 * M_PI / 1680.0));
                m_128apsk[2].imag()   = (r6 * sin(37 * M_PI / 1680.0));
                m_128apsk[3].real()   = (r6 * cos(11 * M_PI / 168.0));
                m_128apsk[3].imag()   = (r6 * sin(11 * M_PI / 168.0));
                m_128apsk[4].real()   = (r2 * cos(121 * M_PI / 2520.0));
                m_128apsk[4].imag()   = (r2 * sin(121 * M_PI / 2520.0));
                m_128apsk[5].real()   = (r3 * cos(23 * M_PI / 280.0));
                m_128apsk[5].imag()   = (r3 * sin(23 * M_PI / 280.0));
                m_128apsk[6].real()   = (r5 * cos(19 * M_PI / 720.0));
                m_128apsk[6].imag()   = (r5 * sin(19 * M_PI / 720.0));
                m_128apsk[7].real()   = (r4 * cos(61 * M_PI / 720.0));
                m_128apsk[7].imag()   = (r4 * sin(61 * M_PI / 720.0));
                m_128apsk[8].real()   = (r1 * cos(103 * M_PI / 560.0));
                m_128apsk[8].imag()   = (r1 * sin(103 * M_PI / 560.0));
                m_128apsk[9].real()   = (r6 * cos(61 * M_PI / 420.0));
                m_128apsk[9].imag()   = (r6 * sin(61 * M_PI / 420.0));
                m_128apsk[10].real()   = (r6 * cos(383 * M_PI / 1680.0));
                m_128apsk[10].imag()   = (r6 * sin(383 * M_PI / 1680.0));
                m_128apsk[11].real()   = (r6 * cos(929 * M_PI / 5040.0));
                m_128apsk[11].imag()   = (r6 * sin(929 * M_PI / 5040.0));
                m_128apsk[12].real()   = (r2 * cos(113 * M_PI / 560.0));
                m_128apsk[12].imag()   = (r2 * sin(113 * M_PI / 560.0));
                m_128apsk[13].real()   = (r3 * cos(169 * M_PI / 1008.0));
                m_128apsk[13].imag()   = (r3 * sin(169 * M_PI / 1008.0));
                m_128apsk[14].real()   = (r5 * cos(563 * M_PI / 2520.0));
                m_128apsk[14].imag()   = (r5 * sin(563 * M_PI / 2520.0));
                m_128apsk[15].real()   = (r4 * cos(139 * M_PI / 840.0));
                m_128apsk[15].imag()   = (r4 * sin(139 * M_PI / 840.0));
                m_128apsk[16].real()   = (r1 * cos(243 * M_PI / 560.0));
                m_128apsk[16].imag()   = (r1 * sin(243 * M_PI / 560.0));
                m_128apsk[17].real()   = (r6 * cos(1993 * M_PI / 5040.0));
                m_128apsk[17].imag()   = (r6 * sin(1993 * M_PI / 5040.0));
                m_128apsk[18].real()   = (r6 * cos(43 * M_PI / 90.0));
                m_128apsk[18].imag()   = (r6 * sin(43 * M_PI / 90.0));
                m_128apsk[19].real()   = (r6 * cos(73 * M_PI / 168.0));
                m_128apsk[19].imag()   = (r6 * sin(73 * M_PI / 168.0));
                m_128apsk[20].real()   = (r2 * cos(1139 * M_PI / 2520.0));
                m_128apsk[20].imag()   = (r2 * sin(1139 * M_PI / 2520.0));
                m_128apsk[21].real()   = (r3 * cos(117 * M_PI / 280.0));
                m_128apsk[21].imag()   = (r3 * sin(117 * M_PI / 280.0));
                m_128apsk[22].real()   = (r5 * cos(341 * M_PI / 720.0));
                m_128apsk[22].imag()   = (r5 * sin(341 * M_PI / 720.0));
                m_128apsk[23].real()   = (r4 * cos(349 * M_PI / 840.0));
                m_128apsk[23].imag()   = (r4 * sin(349 * M_PI / 840.0));
                m_128apsk[24].real()   = (r1 * cos(177 * M_PI / 560.0));
                m_128apsk[24].imag()   = (r1 * sin(177 * M_PI / 560.0));
                m_128apsk[25].real()   = (r6 * cos(1789 * M_PI / 5040.0));
                m_128apsk[25].imag()   = (r6 * sin(1789 * M_PI / 5040.0));
                m_128apsk[26].real()   = (r6 * cos(49 * M_PI / 180.0));
                m_128apsk[26].imag()   = (r6 * sin(49 * M_PI / 180.0));
                m_128apsk[27].real()   = (r6 * cos(53 * M_PI / 168.0));
                m_128apsk[27].imag()   = (r6 * sin(53 * M_PI / 168.0));
                m_128apsk[28].real()   = (r2 * cos(167 * M_PI / 560.0));
                m_128apsk[28].imag()   = (r2 * sin(167 * M_PI / 560.0));
                m_128apsk[29].real()   = (r3 * cos(239 * M_PI / 720.0));
                m_128apsk[29].imag()   = (r3 * sin(239 * M_PI / 720.0));
                m_128apsk[30].real()   = (r5 * cos(199 * M_PI / 720.0));
                m_128apsk[30].imag()   = (r5 * sin(199 * M_PI / 720.0));
                m_128apsk[31].real()   = (r4 * cos(281 * M_PI / 840.0));
                m_128apsk[31].imag()   = (r4 * sin(281 * M_PI / 840.0));
                m_128apsk[32].real()   = (r1 * cos(1177 * M_PI / 1260.0));
                m_128apsk[32].imag()   = (r1 * sin(1177 * M_PI / 1260.0));
                m_128apsk[33].real()   = (r6 * cos(94 * M_PI / 105.0));
                m_128apsk[33].imag()   = (r6 * sin(94 * M_PI / 105.0));
                m_128apsk[34].real()   = (r6 * cos(1643 * M_PI / 1680.0));
                m_128apsk[34].imag()   = (r6 * sin(1643 * M_PI / 1680.0));
                m_128apsk[35].real()   = (r6 * cos(157 * M_PI / 168.0));
                m_128apsk[35].imag()   = (r6 * sin(157 * M_PI / 168.0));
                m_128apsk[36].real()   = (r2 * cos(2399 * M_PI / 2520.0));
                m_128apsk[36].imag()   = (r2 * sin(2399 * M_PI / 2520.0));
                m_128apsk[37].real()   = (r3 * cos(257 * M_PI / 280.0));
                m_128apsk[37].imag()   = (r3 * sin(257 * M_PI / 280.0));
                m_128apsk[38].real()   = (r5 * cos(701 * M_PI / 720.0));
                m_128apsk[38].imag()   = (r5 * sin(701 * M_PI / 720.0));
                m_128apsk[39].real()   = (r4 * cos(659 * M_PI / 720.0));
                m_128apsk[39].imag()   = (r4 * sin(659 * M_PI / 720.0));
                m_128apsk[40].real()   = (r1 * cos(457 * M_PI / 560.0));
                m_128apsk[40].imag()   = (r1 * sin(457 * M_PI / 560.0));
                m_128apsk[41].real()   = (r6 * cos(359 * M_PI / 420.0));
                m_128apsk[41].imag()   = (r6 * sin(359 * M_PI / 420.0));
                m_128apsk[42].real()   = (r6 * cos(1297 * M_PI / 1680.0));
                m_128apsk[42].imag()   = (r6 * sin(1297 * M_PI / 1680.0));
                m_128apsk[43].real()   = (r6 * cos(4111 * M_PI / 5040.0));
                m_128apsk[43].imag()   = (r6 * sin(4111 * M_PI / 5040.0));
                m_128apsk[44].real()   = (r2 * cos(447 * M_PI / 560.0));
                m_128apsk[44].imag()   = (r2 * sin(447 * M_PI / 560.0));
                m_128apsk[45].real()   = (r3 * cos(839 * M_PI / 1008.0));
                m_128apsk[45].imag()   = (r3 * sin(839 * M_PI / 1008.0));
                m_128apsk[46].real()   = (r5 * cos(1957 * M_PI / 2520.0));
                m_128apsk[46].imag()   = (r5 * sin(1957 * M_PI / 2520.0));
                m_128apsk[47].real()   = (r4 * cos(701 * M_PI / 840.0));
                m_128apsk[47].imag()   = (r4 * sin(701 * M_PI / 840.0));
                m_128apsk[48].real()   = (r1 * cos(317 * M_PI / 560.0));
                m_128apsk[48].imag()   = (r1 * sin(317 * M_PI / 560.0));
                m_128apsk[49].real()   = (r6 * cos(3047 * M_PI / 5040.0));
                m_128apsk[49].imag()   = (r6 * sin(3047 * M_PI / 5040.0));
                m_128apsk[50].real()   = (r6 * cos(47 * M_PI / 90.0));
                m_128apsk[50].imag()   = (r6 * sin(47 * M_PI / 90.0));
                m_128apsk[51].real()   = (r6 * cos(95 * M_PI / 168.0));
                m_128apsk[51].imag()   = (r6 * sin(95 * M_PI / 168.0));
                m_128apsk[52].real()   = (r2 * cos(1381 * M_PI / 2520.0));
                m_128apsk[52].imag()   = (r2 * sin(1381 * M_PI / 2520.0));
                m_128apsk[53].real()   = (r3 * cos(163 * M_PI / 280.0));
                m_128apsk[53].imag()   = (r3 * sin(163 * M_PI / 280.0));
                m_128apsk[54].real()   = (r5 * cos(379 * M_PI / 720.0));
                m_128apsk[54].imag()   = (r5 * sin(379 * M_PI / 720.0));
                m_128apsk[55].real()   = (r4 * cos(491 * M_PI / 840.0));
                m_128apsk[55].imag()   = (r4 * sin(491 * M_PI / 840.0));
                m_128apsk[56].real()   = (r1 * cos(383 * M_PI / 560.0));
                m_128apsk[56].imag()   = (r1 * sin(383 * M_PI / 560.0));
                m_128apsk[57].real()   = (r6 * cos(3251 * M_PI / 5040.0));
                m_128apsk[57].imag()   = (r6 * sin(3251 * M_PI / 5040.0));
                m_128apsk[58].real()   = (r6 * cos(131 * M_PI / 180.0));
                m_128apsk[58].imag()   = (r6 * sin(131 * M_PI / 180.0));
                m_128apsk[59].real()   = (r6 * cos(115 * M_PI / 168.0));
                m_128apsk[59].imag()   = (r6 * sin(115 * M_PI / 168.0));
                m_128apsk[60].real()   = (r2 * cos(393 * M_PI / 560.0));
                m_128apsk[60].imag()   = (r2 * sin(393 * M_PI / 560.0));
                m_128apsk[61].real()   = (r3 * cos(481 * M_PI / 720.0));
                m_128apsk[61].imag()   = (r3 * sin(481 * M_PI / 720.0));
                m_128apsk[62].real()   = (r5 * cos(521 * M_PI / 720.0));
                m_128apsk[62].imag()   = (r5 * sin(521 * M_PI / 720.0));
                m_128apsk[63].real()   = (r4 * cos(559 * M_PI / 840.0));
                m_128apsk[63].imag()   = (r4 * sin(559 * M_PI / 840.0));
                m_128apsk[64].real()   = (r1 * cos(2437 * M_PI / 1260.0));
                m_128apsk[64].imag()   = (r1 * sin(2437 * M_PI / 1260.0));
                m_128apsk[65].real()   = (r6 * cos(199 * M_PI / 105.0));
                m_128apsk[65].imag()   = (r6 * sin(199 * M_PI / 105.0));
                m_128apsk[66].real()   = (r6 * cos(3323 * M_PI / 1680.0));
                m_128apsk[66].imag()   = (r6 * sin(3323 * M_PI / 1680.0));
                m_128apsk[67].real()   = (r6 * cos(325 * M_PI / 168.0));
                m_128apsk[67].imag()   = (r6 * sin(325 * M_PI / 168.0));
                m_128apsk[68].real()   = (r2 * cos(4919 * M_PI / 2520.0));
                m_128apsk[68].imag()   = (r2 * sin(4919 * M_PI / 2520.0));
                m_128apsk[69].real()   = (r3 * cos(537 * M_PI / 280.0));
                m_128apsk[69].imag()   = (r3 * sin(537 * M_PI / 280.0));
                m_128apsk[70].real()   = (r5 * cos(1421 * M_PI / 720.0));
                m_128apsk[70].imag()   = (r5 * sin(1421 * M_PI / 720.0));
                m_128apsk[71].real()   = (r4 * cos(1379 * M_PI / 720.0));
                m_128apsk[71].imag()   = (r4 * sin(1379 * M_PI / 720.0));
                m_128apsk[72].real()   = (r1 * cos(1017 * M_PI / 560.0));
                m_128apsk[72].imag()   = (r1 * sin(1017 * M_PI / 560.0));
                m_128apsk[73].real()   = (r6 * cos(779 * M_PI / 420.0));
                m_128apsk[73].imag()   = (r6 * sin(779 * M_PI / 420.0));
                m_128apsk[74].real()   = (r6 * cos(2977 * M_PI / 1680.0));
                m_128apsk[74].imag()   = (r6 * sin(2977 * M_PI / 1680.0));
                m_128apsk[75].real()   = (r6 * cos(9151 * M_PI / 5040.0));
                m_128apsk[75].imag()   = (r6 * sin(9151 * M_PI / 5040.0));
                m_128apsk[76].real()   = (r2 * cos(1007 * M_PI / 560.0));
                m_128apsk[76].imag()   = (r2 * sin(1007 * M_PI / 560.0));
                m_128apsk[77].real()   = (r3 * cos(1847 * M_PI / 1008.0));
                m_128apsk[77].imag()   = (r3 * sin(1847 * M_PI / 1008.0));
                m_128apsk[78].real()   = (r5 * cos(4477 * M_PI / 2520.0));
                m_128apsk[78].imag()   = (r5 * sin(4477 * M_PI / 2520.0));
                m_128apsk[79].real()   = (r4 * cos(1541 * M_PI / 840.0));
                m_128apsk[79].imag()   = (r4 * sin(1541 * M_PI / 840.0));
                m_128apsk[80].real()   = (r1 * cos(877 * M_PI / 560.0));
                m_128apsk[80].imag()   = (r1 * sin(877 * M_PI / 560.0));
                m_128apsk[81].real()   = (r6 * cos(8087 * M_PI / 5040.0));
                m_128apsk[81].imag()   = (r6 * sin(8087 * M_PI / 5040.0));
                m_128apsk[82].real()   = (r6 * cos(137 * M_PI / 90.0));
                m_128apsk[82].imag()   = (r6 * sin(137 * M_PI / 90.0));
                m_128apsk[83].real()   = (r6 * cos(263 * M_PI / 168.0));
                m_128apsk[83].imag()   = (r6 * sin(263 * M_PI / 168.0));
                m_128apsk[84].real()   = (r2 * cos(3901 * M_PI / 2520.0));
                m_128apsk[84].imag()   = (r2 * sin(3901 * M_PI / 2520.0));
                m_128apsk[85].real()   = (r3 * cos(443 * M_PI / 280.0));
                m_128apsk[85].imag()   = (r3 * sin(443 * M_PI / 280.0));
                m_128apsk[86].real()   = (r5 * cos(1099 * M_PI / 720.0));
                m_128apsk[86].imag()   = (r5 * sin(1099 * M_PI / 720.0));
                m_128apsk[87].real()   = (r4 * cos(1331 * M_PI / 840.0));
                m_128apsk[87].imag()   = (r4 * sin(1331 * M_PI / 840.0));
                m_128apsk[88].real()   = (r1 * cos(943 * M_PI / 560.0));
                m_128apsk[88].imag()   = (r1 * sin(943 * M_PI / 560.0));
                m_128apsk[89].real()   = (r6 * cos(8291 * M_PI / 5040.0));
                m_128apsk[89].imag()   = (r6 * sin(8291 * M_PI / 5040.0));
                m_128apsk[90].real()   = (r6 * cos(311 * M_PI / 180.0));
                m_128apsk[90].imag()   = (r6 * sin(311 * M_PI / 180.0));
                m_128apsk[91].real()   = (r6 * cos(283 * M_PI / 168.0));
                m_128apsk[91].imag()   = (r6 * sin(283 * M_PI / 168.0));
                m_128apsk[92].real()   = (r2 * cos(953 * M_PI / 560.0));
                m_128apsk[92].imag()   = (r2 * sin(953 * M_PI / 560.0));
                m_128apsk[93].real()   = (r3 * cos(1201 * M_PI / 720.0));
                m_128apsk[93].imag()   = (r3 * sin(1201 * M_PI / 720.0));
                m_128apsk[94].real()   = (r5 * cos(1241 * M_PI / 720.0));
                m_128apsk[94].imag()   = (r5 * sin(1241 * M_PI / 720.0));
                m_128apsk[95].real()   = (r4 * cos(1399 * M_PI / 840.0));
                m_128apsk[95].imag()   = (r4 * sin(1399 * M_PI / 840.0));
                m_128apsk[96].real()   = (r1 * cos(1343 * M_PI / 1260.0));
                m_128apsk[96].imag()   = (r1 * sin(1343 * M_PI / 1260.0));
                m_128apsk[97].real()   = (r6 * cos(116 * M_PI / 105.0));
                m_128apsk[97].imag()   = (r6 * sin(116 * M_PI / 105.0));
                m_128apsk[98].real()   = (r6 * cos(1717 * M_PI / 1680.0));
                m_128apsk[98].imag()   = (r6 * sin(1717 * M_PI / 1680.0));
                m_128apsk[99].real()   = (r6 * cos(179 * M_PI / 168.0));
                m_128apsk[99].imag()   = (r6 * sin(179 * M_PI / 168.0));
                m_128apsk[100].real()   = (r2 * cos(2641 * M_PI / 2520.0));
                m_128apsk[100].imag()   = (r2 * sin(2641 * M_PI / 2520.0));
                m_128apsk[101].real()   = (r3 * cos(303 * M_PI / 280.0));
                m_128apsk[101].imag()   = (r3 * sin(303 * M_PI / 280.0));
                m_128apsk[102].real()   = (r5 * cos(739 * M_PI / 720.0));
                m_128apsk[102].imag()   = (r5 * sin(739 * M_PI / 720.0));
                m_128apsk[103].real()   = (r4 * cos(781 * M_PI / 720.0));
                m_128apsk[103].imag()   = (r4 * sin(781 * M_PI / 720.0));
                m_128apsk[104].real()   = (r1 * cos(663 * M_PI / 560.0));
                m_128apsk[104].imag()   = (r1 * sin(663 * M_PI / 560.0));
                m_128apsk[105].real()   = (r6 * cos(481 * M_PI / 420.0));
                m_128apsk[105].imag()   = (r6 * sin(481 * M_PI / 420.0));
                m_128apsk[106].real()   = (r6 * cos(2063 * M_PI / 1680.0));
                m_128apsk[106].imag()   = (r6 * sin(2063 * M_PI / 1680.0));
                m_128apsk[107].real()   = (r6 * cos(5969 * M_PI / 5040.0));
                m_128apsk[107].imag()   = (r6 * sin(5969 * M_PI / 5040.0));
                m_128apsk[108].real()   = (r2 * cos(673 * M_PI / 560.0));
                m_128apsk[108].imag()   = (r2 * sin(673 * M_PI / 560.0));
                m_128apsk[109].real()   = (r3 * cos(1177 * M_PI / 1008.0));
                m_128apsk[109].imag()   = (r3 * sin(1177 * M_PI / 1008.0));
                m_128apsk[110].real()   = (r5 * cos(3083 * M_PI / 2520.0));
                m_128apsk[110].imag()   = (r5 * sin(3083 * M_PI / 2520.0));
                m_128apsk[111].real()   = (r4 * cos(979 * M_PI / 840.0));
                m_128apsk[111].imag()   = (r4 * sin(979 * M_PI / 840.0));
                m_128apsk[112].real()   = (r1 * cos(803 * M_PI / 560.0));
                m_128apsk[112].imag()   = (r1 * sin(803 * M_PI / 560.0));
                m_128apsk[113].real()   = (r6 * cos(7033 * M_PI / 5040.0));
                m_128apsk[113].imag()   = (r6 * sin(7033 * M_PI / 5040.0));
                m_128apsk[114].real()   = (r6 * cos(133 * M_PI / 90.0));
                m_128apsk[114].imag()   = (r6 * sin(133 * M_PI / 90.0));
                m_128apsk[115].real()   = (r6 * cos(241 * M_PI / 168.0));
                m_128apsk[115].imag()   = (r6 * sin(241 * M_PI / 168.0));
                m_128apsk[116].real()   = (r2 * cos(3659 * M_PI / 2520.0));
                m_128apsk[116].imag()   = (r2 * sin(3659 * M_PI / 2520.0));
                m_128apsk[117].real()   = (r3 * cos(397 * M_PI / 280.0));
                m_128apsk[117].imag()   = (r3 * sin(397 * M_PI / 280.0));
                m_128apsk[118].real()   = (r5 * cos(1061 * M_PI / 720.0));
                m_128apsk[118].imag()   = (r5 * sin(1061 * M_PI / 720.0));
                m_128apsk[119].real()   = (r4 * cos(1189 * M_PI / 840.0));
                m_128apsk[119].imag()   = (r4 * sin(1189 * M_PI / 840.0));
                m_128apsk[120].real()   = (r1 * cos(737 * M_PI / 560.0));
                m_128apsk[120].imag()   = (r1 * sin(737 * M_PI / 560.0));
                m_128apsk[121].real()   = (r6 * cos(6829 * M_PI / 5040.0));
                m_128apsk[121].imag()   = (r6 * sin(6829 * M_PI / 5040.0));
                m_128apsk[122].real()   = (r6 * cos(229 * M_PI / 180.0));
                m_128apsk[122].imag()   = (r6 * sin(229 * M_PI / 180.0));
                m_128apsk[123].real()   = (r6 * cos(221 * M_PI / 168.0));
                m_128apsk[123].imag()   = (r6 * sin(221 * M_PI / 168.0));
                m_128apsk[124].real()   = (r2 * cos(727 * M_PI / 560.0));
                m_128apsk[124].imag()   = (r2 * sin(727 * M_PI / 560.0));
                m_128apsk[125].real()   = (r3 * cos(959 * M_PI / 720.0));
                m_128apsk[125].imag()   = (r3 * sin(959 * M_PI / 720.0));
                m_128apsk[126].real()   = (r5 * cos(919 * M_PI / 720.0));
                m_128apsk[126].imag()   = (r5 * sin(919 * M_PI / 720.0));
                m_128apsk[127].real()   = (r4 * cos(1121 * M_PI / 840.0));
                m_128apsk[127].imag()   = (r4 * sin(1121 * M_PI / 840.0));
                break;
            case gr::dvbs2::MOD_256APSK:
                if (rate == gr::dvbs2::C20_30)
                {
                    m_256apsk[0].real()   = 1.6350;
                    m_256apsk[0].imag()   = 0.1593;
                    m_256apsk[1].real()   = 1.5776;
                    m_256apsk[1].imag()   = 0.4735;
                    m_256apsk[2].real()   = 0.9430;
                    m_256apsk[2].imag()   = 0.1100;
                    m_256apsk[3].real()   = 0.9069;
                    m_256apsk[3].imag()   = 0.2829;
                    m_256apsk[4].real()   = 0.3237;
                    m_256apsk[4].imag()   = 0.0849;
                    m_256apsk[5].real()   = 0.3228;
                    m_256apsk[5].imag()   = 0.0867;
                    m_256apsk[6].real()   = 0.7502;
                    m_256apsk[6].imag()   = 0.1138;
                    m_256apsk[7].real()   = 0.7325;
                    m_256apsk[7].imag()   = 0.2088;
                    m_256apsk[8].real()   = 0.1658;
                    m_256apsk[8].imag()   = 1.6747;
                    m_256apsk[9].real()   = 0.4907;
                    m_256apsk[9].imag()   = 1.6084;
                    m_256apsk[10].real()  = 0.1088;
                    m_256apsk[10].imag()  = 0.9530;
                    m_256apsk[11].real()  = 0.2464;
                    m_256apsk[11].imag()  = 0.9270;
                    m_256apsk[12].real()  = 0.0872;
                    m_256apsk[12].imag()  = 0.1390;
                    m_256apsk[13].real()  = 0.0871;
                    m_256apsk[13].imag()  = 0.1392;
                    m_256apsk[14].real()  = 0.1091;
                    m_256apsk[14].imag()  = 0.7656;
                    m_256apsk[15].real()  = 0.1699;
                    m_256apsk[15].imag()  = 0.7537;
                    m_256apsk[16].real()  = -1.6350;
                    m_256apsk[16].imag()  = 0.1593;
                    m_256apsk[17].real()  = -1.5776;
                    m_256apsk[17].imag()  = 0.4735;
                    m_256apsk[18].real()  = -0.9430;
                    m_256apsk[18].imag()  = 0.1100;
                    m_256apsk[19].real()  = -0.9069;
                    m_256apsk[19].imag()  = 0.2829;
                    m_256apsk[20].real()  = -0.3237;
                    m_256apsk[20].imag()  = 0.0849;
                    m_256apsk[21].real()  = -0.3228;
                    m_256apsk[21].imag()  = 0.0867;
                    m_256apsk[22].real()  = -0.7502;
                    m_256apsk[22].imag()  = 0.1138;
                    m_256apsk[23].real()  = -0.7325;
                    m_256apsk[23].imag()  = 0.2088;
                    m_256apsk[24].real()  = -0.1658;
                    m_256apsk[24].imag()  = 1.6747;
                    m_256apsk[25].real()  = -0.4907;
                    m_256apsk[25].imag()  = 1.6084;
                    m_256apsk[26].real()  = -0.1088;
                    m_256apsk[26].imag()  = 0.9530;
                    m_256apsk[27].real()  = -0.2464;
                    m_256apsk[27].imag()  = 0.9270;
                    m_256apsk[28].real()  = -0.0872;
                    m_256apsk[28].imag()  = 0.1390;
                    m_256apsk[29].real()  = -0.0871;
                    m_256apsk[29].imag()  = 0.1392;
                    m_256apsk[30].real()  = -0.1091;
                    m_256apsk[30].imag()  = 0.7656;
                    m_256apsk[31].real()  = -0.1699;
                    m_256apsk[31].imag()  = 0.7537;
                    m_256apsk[32].real()  = 1.3225;
                    m_256apsk[32].imag()  = 0.1320;
                    m_256apsk[33].real()  = 1.2742;
                    m_256apsk[33].imag()  = 0.3922;
                    m_256apsk[34].real()  = 1.0854;
                    m_256apsk[34].imag()  = 0.1139;
                    m_256apsk[35].real()  = 1.0441;
                    m_256apsk[35].imag()  = 0.3296;
                    m_256apsk[36].real()  = 0.4582;
                    m_256apsk[36].imag()  = 0.1123;
                    m_256apsk[37].real()  = 0.4545;
                    m_256apsk[37].imag()  = 0.1251;
                    m_256apsk[38].real()  = 0.6473;
                    m_256apsk[38].imag()  = 0.1138;
                    m_256apsk[39].real()  = 0.6339;
                    m_256apsk[39].imag()  = 0.1702;
                    m_256apsk[40].real()  = 0.1322;
                    m_256apsk[40].imag()  = 1.3631;
                    m_256apsk[41].real()  = 0.3929;
                    m_256apsk[41].imag()  = 1.3102;
                    m_256apsk[42].real()  = 0.1124;
                    m_256apsk[42].imag()  = 1.1327;
                    m_256apsk[43].real()  = 0.3160;
                    m_256apsk[43].imag()  = 1.0913;
                    m_256apsk[44].real()  = 0.0928;
                    m_256apsk[44].imag()  = 0.3970;
                    m_256apsk[45].real()  = 0.0937;
                    m_256apsk[45].imag()  = 0.3973;
                    m_256apsk[46].real()  = 0.1054;
                    m_256apsk[46].imag()  = 0.5979;
                    m_256apsk[47].real()  = 0.1230;
                    m_256apsk[47].imag()  = 0.5949;
                    m_256apsk[48].real()  = -1.3225;
                    m_256apsk[48].imag()  = 0.1320;
                    m_256apsk[49].real()  = -1.2742;
                    m_256apsk[49].imag()  = 0.3922;
                    m_256apsk[50].real()  = -1.0854;
                    m_256apsk[50].imag()  = 0.1139;
                    m_256apsk[51].real()  = -1.0441;
                    m_256apsk[51].imag()  = 0.3296;
                    m_256apsk[52].real()  = -0.4582;
                    m_256apsk[52].imag()  = 0.1123;
                    m_256apsk[53].real()  = -0.4545;
                    m_256apsk[53].imag()  = 0.1251;
                    m_256apsk[54].real()  = -0.6473;
                    m_256apsk[54].imag()  = 0.1138;
                    m_256apsk[55].real()  = -0.6339;
                    m_256apsk[55].imag()  = 0.1702;
                    m_256apsk[56].real()  = -0.1322;
                    m_256apsk[56].imag()  = 1.3631;
                    m_256apsk[57].real()  = -0.3929;
                    m_256apsk[57].imag()  = 1.3102;
                    m_256apsk[58].real()  = -0.1124;
                    m_256apsk[58].imag()  = 1.1327;
                    m_256apsk[59].real()  = -0.3160;
                    m_256apsk[59].imag()  = 1.0913;
                    m_256apsk[60].real()  = -0.0928;
                    m_256apsk[60].imag()  = 0.3970;
                    m_256apsk[61].real()  = -0.0937;
                    m_256apsk[61].imag()  = 0.3973;
                    m_256apsk[62].real()  = -0.1054;
                    m_256apsk[62].imag()  = 0.5979;
                    m_256apsk[63].real()  = -0.1230;
                    m_256apsk[63].imag()  = 0.5949;
                    m_256apsk[64].real()  = 1.6350;
                    m_256apsk[64].imag()  = -0.1593;
                    m_256apsk[65].real()  = 1.5776;
                    m_256apsk[65].imag()  = -0.4735;
                    m_256apsk[66].real()  = 0.9430;
                    m_256apsk[66].imag()  = -0.1100;
                    m_256apsk[67].real()  = 0.9069;
                    m_256apsk[67].imag()  = -0.2829;
                    m_256apsk[68].real()  = 0.3237;
                    m_256apsk[68].imag()  = -0.0849;
                    m_256apsk[69].real()  = 0.3228;
                    m_256apsk[69].imag()  = -0.0867;
                    m_256apsk[70].real()  = 0.7502;
                    m_256apsk[70].imag()  = -0.1138;
                    m_256apsk[71].real()  = 0.7325;
                    m_256apsk[71].imag()  = -0.2088;
                    m_256apsk[72].real()  = 0.1658;
                    m_256apsk[72].imag()  = -1.6747;
                    m_256apsk[73].real()  = 0.4907;
                    m_256apsk[73].imag()  = -1.6084;
                    m_256apsk[74].real()  = 0.1088;
                    m_256apsk[74].imag()  = -0.9530;
                    m_256apsk[75].real()  = 0.2464;
                    m_256apsk[75].imag()  = -0.9270;
                    m_256apsk[76].real()  = 0.0872;
                    m_256apsk[76].imag()  = -0.1390;
                    m_256apsk[77].real()  = 0.0871;
                    m_256apsk[77].imag()  = -0.1392;
                    m_256apsk[78].real()  = 0.1091;
                    m_256apsk[78].imag()  = -0.7656;
                    m_256apsk[79].real()  = 0.1699;
                    m_256apsk[79].imag()  = -0.7537;
                    m_256apsk[80].real()  = -1.6350;
                    m_256apsk[80].imag()  = -0.1593;
                    m_256apsk[81].real()  = -1.5776;
                    m_256apsk[81].imag()  = -0.4735;
                    m_256apsk[82].real()  = -0.9430;
                    m_256apsk[82].imag()  = -0.1100;
                    m_256apsk[83].real()  = -0.9069;
                    m_256apsk[83].imag()  = -0.2829;
                    m_256apsk[84].real()  = -0.3237;
                    m_256apsk[84].imag()  = -0.0849;
                    m_256apsk[85].real()  = -0.3228;
                    m_256apsk[85].imag()  = -0.0867;
                    m_256apsk[86].real()  = -0.7502;
                    m_256apsk[86].imag()  = -0.1138;
                    m_256apsk[87].real()  = -0.7325;
                    m_256apsk[87].imag()  = -0.2088;
                    m_256apsk[88].real()  = -0.1658;
                    m_256apsk[88].imag()  = -1.6747;
                    m_256apsk[89].real()  = -0.4907;
                    m_256apsk[89].imag()  = -1.6084;
                    m_256apsk[90].real()  = -0.1088;
                    m_256apsk[90].imag()  = -0.9530;
                    m_256apsk[91].real()  = -0.2464;
                    m_256apsk[91].imag()  = -0.9270;
                    m_256apsk[92].real()  = -0.0872;
                    m_256apsk[92].imag()  = -0.1390;
                    m_256apsk[93].real()  = -0.0871;
                    m_256apsk[93].imag()  = -0.1392;
                    m_256apsk[94].real()  = -0.1091;
                    m_256apsk[94].imag()  = -0.7656;
                    m_256apsk[95].real()  = -0.1699;
                    m_256apsk[95].imag()  = -0.7537;
                    m_256apsk[96].real()  = 1.3225;
                    m_256apsk[96].imag()  = -0.1320;
                    m_256apsk[97].real()  = 1.2742;
                    m_256apsk[97].imag()  = -0.3922;
                    m_256apsk[98].real()  = 1.0854;
                    m_256apsk[98].imag()  = -0.1139;
                    m_256apsk[99].real()  = 1.0441;
                    m_256apsk[99].imag()  = -0.3296;
                    m_256apsk[100].real() = 0.4582;
                    m_256apsk[100].imag() = -0.1123;
                    m_256apsk[101].real() = 0.4545;
                    m_256apsk[101].imag() = -0.1251;
                    m_256apsk[102].real() = 0.6473;
                    m_256apsk[102].imag() = -0.1138;
                    m_256apsk[103].real() = 0.6339;
                    m_256apsk[103].imag() = -0.1702;
                    m_256apsk[104].real() = 0.1322;
                    m_256apsk[104].imag() = -1.3631;
                    m_256apsk[105].real() = 0.3929;
                    m_256apsk[105].imag() = -1.3102;
                    m_256apsk[106].real() = 0.1124;
                    m_256apsk[106].imag() = -1.1327;
                    m_256apsk[107].real() = 0.3160;
                    m_256apsk[107].imag() = -1.0913;
                    m_256apsk[108].real() = 0.0928;
                    m_256apsk[108].imag() = -0.3970;
                    m_256apsk[109].real() = 0.0937;
                    m_256apsk[109].imag() = -0.3973;
                    m_256apsk[110].real() = 0.1054;
                    m_256apsk[110].imag() = -0.5979;
                    m_256apsk[111].real() = 0.1230;
                    m_256apsk[111].imag() = -0.5949;
                    m_256apsk[112].real() = -1.3225;
                    m_256apsk[112].imag() = -0.1320;
                    m_256apsk[113].real() = -1.2742;
                    m_256apsk[113].imag() = -0.3922;
                    m_256apsk[114].real() = -1.0854;
                    m_256apsk[114].imag() = -0.1139;
                    m_256apsk[115].real() = -1.0441;
                    m_256apsk[115].imag() = -0.3296;
                    m_256apsk[116].real() = -0.4582;
                    m_256apsk[116].imag() = -0.1123;
                    m_256apsk[117].real() = -0.4545;
                    m_256apsk[117].imag() = -0.1251;
                    m_256apsk[118].real() = -0.6473;
                    m_256apsk[118].imag() = -0.1138;
                    m_256apsk[119].real() = -0.6339;
                    m_256apsk[119].imag() = -0.1702;
                    m_256apsk[120].real() = -0.1322;
                    m_256apsk[120].imag() = -1.3631;
                    m_256apsk[121].real() = -0.3929;
                    m_256apsk[121].imag() = -1.3102;
                    m_256apsk[122].real() = -0.1124;
                    m_256apsk[122].imag() = -1.1327;
                    m_256apsk[123].real() = -0.3160;
                    m_256apsk[123].imag() = -1.0913;
                    m_256apsk[124].real() = -0.0928;
                    m_256apsk[124].imag() = -0.3970;
                    m_256apsk[125].real() = -0.0937;
                    m_256apsk[125].imag() = -0.3973;
                    m_256apsk[126].real() = -0.1054;
                    m_256apsk[126].imag() = -0.5979;
                    m_256apsk[127].real() = -0.1230;
                    m_256apsk[127].imag() = -0.5949;
                    m_256apsk[128].real() = 1.2901;
                    m_256apsk[128].imag() = 1.0495;
                    m_256apsk[129].real() = 1.4625;
                    m_256apsk[129].imag() = 0.7740;
                    m_256apsk[130].real() = 0.7273;
                    m_256apsk[130].imag() = 0.6160;
                    m_256apsk[131].real() = 0.8177;
                    m_256apsk[131].imag() = 0.4841;
                    m_256apsk[132].real() = 0.2844;
                    m_256apsk[132].imag() = 0.1296;
                    m_256apsk[133].real() = 0.2853;
                    m_256apsk[133].imag() = 0.1309;
                    m_256apsk[134].real() = 0.5902;
                    m_256apsk[134].imag() = 0.4857;
                    m_256apsk[135].real() = 0.6355;
                    m_256apsk[135].imag() = 0.4185;
                    m_256apsk[136].real() = 1.0646;
                    m_256apsk[136].imag() = 1.2876;
                    m_256apsk[137].real() = 0.7949;
                    m_256apsk[137].imag() = 1.4772;
                    m_256apsk[138].real() = 0.5707;
                    m_256apsk[138].imag() = 0.7662;
                    m_256apsk[139].real() = 0.4490;
                    m_256apsk[139].imag() = 0.8461;
                    m_256apsk[140].real() = 0.1053;
                    m_256apsk[140].imag() = 0.1494;
                    m_256apsk[141].real() = 0.1052;
                    m_256apsk[141].imag() = 0.1495;
                    m_256apsk[142].real() = 0.4294;
                    m_256apsk[142].imag() = 0.6363;
                    m_256apsk[143].real() = 0.3744;
                    m_256apsk[143].imag() = 0.6744;
                    m_256apsk[144].real() = -1.2901;
                    m_256apsk[144].imag() = 1.0495;
                    m_256apsk[145].real() = -1.4625;
                    m_256apsk[145].imag() = 0.7740;
                    m_256apsk[146].real() = -0.7273;
                    m_256apsk[146].imag() = 0.6160;
                    m_256apsk[147].real() = -0.8177;
                    m_256apsk[147].imag() = 0.4841;
                    m_256apsk[148].real() = -0.2844;
                    m_256apsk[148].imag() = 0.1296;
                    m_256apsk[149].real() = -0.2853;
                    m_256apsk[149].imag() = 0.1309;
                    m_256apsk[150].real() = -0.5902;
                    m_256apsk[150].imag() = 0.4857;
                    m_256apsk[151].real() = -0.6355;
                    m_256apsk[151].imag() = 0.4185;
                    m_256apsk[152].real() = -1.0646;
                    m_256apsk[152].imag() = 1.2876;
                    m_256apsk[153].real() = -0.7949;
                    m_256apsk[153].imag() = 1.4772;
                    m_256apsk[154].real() = -0.5707;
                    m_256apsk[154].imag() = 0.7662;
                    m_256apsk[155].real() = -0.4490;
                    m_256apsk[155].imag() = 0.8461;
                    m_256apsk[156].real() = -0.1053;
                    m_256apsk[156].imag() = 0.1494;
                    m_256apsk[157].real() = -0.1052;
                    m_256apsk[157].imag() = 0.1495;
                    m_256apsk[158].real() = -0.4294;
                    m_256apsk[158].imag() = 0.6363;
                    m_256apsk[159].real() = -0.3744;
                    m_256apsk[159].imag() = 0.6744;
                    m_256apsk[160].real() = 1.0382;
                    m_256apsk[160].imag() = 0.8623;
                    m_256apsk[161].real() = 1.1794;
                    m_256apsk[161].imag() = 0.6376;
                    m_256apsk[162].real() = 0.8504;
                    m_256apsk[162].imag() = 0.7217;
                    m_256apsk[163].real() = 0.9638;
                    m_256apsk[163].imag() = 0.5407;
                    m_256apsk[164].real() = 0.3734;
                    m_256apsk[164].imag() = 0.2560;
                    m_256apsk[165].real() = 0.3799;
                    m_256apsk[165].imag() = 0.2517;
                    m_256apsk[166].real() = 0.4968;
                    m_256apsk[166].imag() = 0.3947;
                    m_256apsk[167].real() = 0.5231;
                    m_256apsk[167].imag() = 0.3644;
                    m_256apsk[168].real() = 0.8555;
                    m_256apsk[168].imag() = 1.0542;
                    m_256apsk[169].real() = 0.6363;
                    m_256apsk[169].imag() = 1.2064;
                    m_256apsk[170].real() = 0.6961;
                    m_256apsk[170].imag() = 0.8850;
                    m_256apsk[171].real() = 0.5229;
                    m_256apsk[171].imag() = 1.0037;
                    m_256apsk[172].real() = 0.1938;
                    m_256apsk[172].imag() = 0.3621;
                    m_256apsk[173].real() = 0.1909;
                    m_256apsk[173].imag() = 0.3627;
                    m_256apsk[174].real() = 0.3224;
                    m_256apsk[174].imag() = 0.5236;
                    m_256apsk[175].real() = 0.3016;
                    m_256apsk[175].imag() = 0.5347;
                    m_256apsk[176].real() = -1.0382;
                    m_256apsk[176].imag() = 0.8623;
                    m_256apsk[177].real() = -1.1794;
                    m_256apsk[177].imag() = 0.6376;
                    m_256apsk[178].real() = -0.8504;
                    m_256apsk[178].imag() = 0.7217;
                    m_256apsk[179].real() = -0.9638;
                    m_256apsk[179].imag() = 0.5407;
                    m_256apsk[180].real() = -0.3734;
                    m_256apsk[180].imag() = 0.2560;
                    m_256apsk[181].real() = -0.3799;
                    m_256apsk[181].imag() = 0.2517;
                    m_256apsk[182].real() = -0.4968;
                    m_256apsk[182].imag() = 0.3947;
                    m_256apsk[183].real() = -0.5231;
                    m_256apsk[183].imag() = 0.3644;
                    m_256apsk[184].real() = -0.8555;
                    m_256apsk[184].imag() = 1.0542;
                    m_256apsk[185].real() = -0.6363;
                    m_256apsk[185].imag() = 1.2064;
                    m_256apsk[186].real() = -0.6961;
                    m_256apsk[186].imag() = 0.8850;
                    m_256apsk[187].real() = -0.5229;
                    m_256apsk[187].imag() = 1.0037;
                    m_256apsk[188].real() = -0.1938;
                    m_256apsk[188].imag() = 0.3621;
                    m_256apsk[189].real() = -0.1909;
                    m_256apsk[189].imag() = 0.3627;
                    m_256apsk[190].real() = -0.3224;
                    m_256apsk[190].imag() = 0.5236;
                    m_256apsk[191].real() = -0.3016;
                    m_256apsk[191].imag() = 0.5347;
                    m_256apsk[192].real() = 1.2901;
                    m_256apsk[192].imag() = -1.0495;
                    m_256apsk[193].real() = 1.4625;
                    m_256apsk[193].imag() = -0.7740;
                    m_256apsk[194].real() = 0.7273;
                    m_256apsk[194].imag() = -0.6160;
                    m_256apsk[195].real() = 0.8177;
                    m_256apsk[195].imag() = -0.4841;
                    m_256apsk[196].real() = 0.2844;
                    m_256apsk[196].imag() = -0.1296;
                    m_256apsk[197].real() = 0.2853;
                    m_256apsk[197].imag() = -0.1309;
                    m_256apsk[198].real() = 0.5902;
                    m_256apsk[198].imag() = -0.4857;
                    m_256apsk[199].real() = 0.6355;
                    m_256apsk[199].imag() = -0.4185;
                    m_256apsk[200].real() = 1.0646;
                    m_256apsk[200].imag() = -1.2876;
                    m_256apsk[201].real() = 0.7949;
                    m_256apsk[201].imag() = -1.4772;
                    m_256apsk[202].real() = 0.5707;
                    m_256apsk[202].imag() = -0.7662;
                    m_256apsk[203].real() = 0.4490;
                    m_256apsk[203].imag() = -0.8461;
                    m_256apsk[204].real() = 0.1053;
                    m_256apsk[204].imag() = -0.1494;
                    m_256apsk[205].real() = 0.1052;
                    m_256apsk[205].imag() = -0.1495;
                    m_256apsk[206].real() = 0.4294;
                    m_256apsk[206].imag() = -0.6363;
                    m_256apsk[207].real() = 0.3744;
                    m_256apsk[207].imag() = -0.6744;
                    m_256apsk[208].real() = -1.2901;
                    m_256apsk[208].imag() = -1.0495;
                    m_256apsk[209].real() = -1.4625;
                    m_256apsk[209].imag() = -0.7740;
                    m_256apsk[210].real() = -0.7273;
                    m_256apsk[210].imag() = -0.6160;
                    m_256apsk[211].real() = -0.8177;
                    m_256apsk[211].imag() = -0.4841;
                    m_256apsk[212].real() = -0.2844;
                    m_256apsk[212].imag() = -0.1296;
                    m_256apsk[213].real() = -0.2853;
                    m_256apsk[213].imag() = -0.1309;
                    m_256apsk[214].real() = -0.5902;
                    m_256apsk[214].imag() = -0.4857;
                    m_256apsk[215].real() = -0.6355;
                    m_256apsk[215].imag() = -0.4185;
                    m_256apsk[216].real() = -1.0646;
                    m_256apsk[216].imag() = -1.2876;
                    m_256apsk[217].real() = -0.7949;
                    m_256apsk[217].imag() = -1.4772;
                    m_256apsk[218].real() = -0.5707;
                    m_256apsk[218].imag() = -0.7662;
                    m_256apsk[219].real() = -0.4490;
                    m_256apsk[219].imag() = -0.8461;
                    m_256apsk[220].real() = -0.1053;
                    m_256apsk[220].imag() = -0.1494;
                    m_256apsk[221].real() = -0.1052;
                    m_256apsk[221].imag() = -0.1495;
                    m_256apsk[222].real() = -0.4294;
                    m_256apsk[222].imag() = -0.6363;
                    m_256apsk[223].real() = -0.3744;
                    m_256apsk[223].imag() = -0.6744;
                    m_256apsk[224].real() = 1.0382;
                    m_256apsk[224].imag() = -0.8623;
                    m_256apsk[225].real() = 1.1794;
                    m_256apsk[225].imag() = -0.6376;
                    m_256apsk[226].real() = 0.8504;
                    m_256apsk[226].imag() = -0.7217;
                    m_256apsk[227].real() = 0.9638;
                    m_256apsk[227].imag() = -0.5407;
                    m_256apsk[228].real() = 0.3734;
                    m_256apsk[228].imag() = -0.2560;
                    m_256apsk[229].real() = 0.3799;
                    m_256apsk[229].imag() = -0.2517;
                    m_256apsk[230].real() = 0.4968;
                    m_256apsk[230].imag() = -0.3947;
                    m_256apsk[231].real() = 0.5231;
                    m_256apsk[231].imag() = -0.3644;
                    m_256apsk[232].real() = 0.8555;
                    m_256apsk[232].imag() = -1.0542;
                    m_256apsk[233].real() = 0.6363;
                    m_256apsk[233].imag() = -1.2064;
                    m_256apsk[234].real() = 0.6961;
                    m_256apsk[234].imag() = -0.8850;
                    m_256apsk[235].real() = 0.5229;
                    m_256apsk[235].imag() = -1.0037;
                    m_256apsk[236].real() = 0.1938;
                    m_256apsk[236].imag() = -0.3621;
                    m_256apsk[237].real() = 0.1909;
                    m_256apsk[237].imag() = -0.3627;
                    m_256apsk[238].real() = 0.3224;
                    m_256apsk[238].imag() = -0.5236;
                    m_256apsk[239].real() = 0.3016;
                    m_256apsk[239].imag() = -0.5347;
                    m_256apsk[240].real() = -1.0382;
                    m_256apsk[240].imag() = -0.8623;
                    m_256apsk[241].real() = -1.1794;
                    m_256apsk[241].imag() = -0.6376;
                    m_256apsk[242].real() = -0.8504;
                    m_256apsk[242].imag() = -0.7217;
                    m_256apsk[243].real() = -0.9638;
                    m_256apsk[243].imag() = -0.5407;
                    m_256apsk[244].real() = -0.3734;
                    m_256apsk[244].imag() = -0.2560;
                    m_256apsk[245].real() = -0.3799;
                    m_256apsk[245].imag() = -0.2517;
                    m_256apsk[246].real() = -0.4968;
                    m_256apsk[246].imag() = -0.3947;
                    m_256apsk[247].real() = -0.5231;
                    m_256apsk[247].imag() = -0.3644;
                    m_256apsk[248].real() = -0.8555;
                    m_256apsk[248].imag() = -1.0542;
                    m_256apsk[249].real() = -0.6363;
                    m_256apsk[249].imag() = -1.2064;
                    m_256apsk[250].real() = -0.6961;
                    m_256apsk[250].imag() = -0.8850;
                    m_256apsk[251].real() = -0.5229;
                    m_256apsk[251].imag() = -1.0037;
                    m_256apsk[252].real() = -0.1938;
                    m_256apsk[252].imag() = -0.3621;
                    m_256apsk[253].real() = -0.1909;
                    m_256apsk[253].imag() = -0.3627;
                    m_256apsk[254].real() = -0.3224;
                    m_256apsk[254].imag() = -0.5236;
                    m_256apsk[255].real() = -0.3016;
                    m_256apsk[255].imag() = -0.5347;
                    for (int i = 0; i < 256; i++)
                    {
                        m_256apsk[i].real() /= 1.6747;
                        m_256apsk[i].imag() /= 1.6747;
                    }
                }
                else if (rate == gr::dvbs2::C22_30)
                {
                    m_256apsk[0].real()   = 1.5977;
                    m_256apsk[0].imag()   = 0.1526;
                    m_256apsk[1].real()   = 1.3187;
                    m_256apsk[1].imag()   = 0.1269;
                    m_256apsk[2].real()   = -1.5977;
                    m_256apsk[2].imag()   = 0.1526;
                    m_256apsk[3].real()   = -1.3187;
                    m_256apsk[3].imag()   = 0.1269;
                    m_256apsk[4].real()   = 0.2574;
                    m_256apsk[4].imag()   = 0.0733;
                    m_256apsk[5].real()   = 0.4496;
                    m_256apsk[5].imag()   = 0.0807;
                    m_256apsk[6].real()   = -0.2574;
                    m_256apsk[6].imag()   = 0.0733;
                    m_256apsk[7].real()   = -0.4496;
                    m_256apsk[7].imag()   = 0.0807;
                    m_256apsk[8].real()   = 1.5977;
                    m_256apsk[8].imag()   = -0.1526;
                    m_256apsk[9].real()   = 1.3187;
                    m_256apsk[9].imag()   = -0.1269;
                    m_256apsk[10].real()  = -1.5977;
                    m_256apsk[10].imag()  = -0.1526;
                    m_256apsk[11].real()  = -1.3187;
                    m_256apsk[11].imag()  = -0.1269;
                    m_256apsk[12].real()  = 0.2574;
                    m_256apsk[12].imag()  = -0.0733;
                    m_256apsk[13].real()  = 0.4496;
                    m_256apsk[13].imag()  = -0.0807;
                    m_256apsk[14].real()  = -0.2574;
                    m_256apsk[14].imag()  = -0.0733;
                    m_256apsk[15].real()  = -0.4496;
                    m_256apsk[15].imag()  = -0.0807;
                    m_256apsk[16].real()  = 0.9269;
                    m_256apsk[16].imag()  = 0.0943;
                    m_256apsk[17].real()  = 1.1024;
                    m_256apsk[17].imag()  = 0.1086;
                    m_256apsk[18].real()  = -0.9269;
                    m_256apsk[18].imag()  = 0.0943;
                    m_256apsk[19].real()  = -1.1024;
                    m_256apsk[19].imag()  = 0.1086;
                    m_256apsk[20].real()  = 0.7663;
                    m_256apsk[20].imag()  = 0.0867;
                    m_256apsk[21].real()  = 0.6115;
                    m_256apsk[21].imag()  = 0.0871;
                    m_256apsk[22].real()  = -0.7663;
                    m_256apsk[22].imag()  = 0.0867;
                    m_256apsk[23].real()  = -0.6115;
                    m_256apsk[23].imag()  = 0.0871;
                    m_256apsk[24].real()  = 0.9269;
                    m_256apsk[24].imag()  = -0.0943;
                    m_256apsk[25].real()  = 1.1024;
                    m_256apsk[25].imag()  = -0.1086;
                    m_256apsk[26].real()  = -0.9269;
                    m_256apsk[26].imag()  = -0.0943;
                    m_256apsk[27].real()  = -1.1024;
                    m_256apsk[27].imag()  = -0.1086;
                    m_256apsk[28].real()  = 0.7663;
                    m_256apsk[28].imag()  = -0.0867;
                    m_256apsk[29].real()  = 0.6115;
                    m_256apsk[29].imag()  = -0.0871;
                    m_256apsk[30].real()  = -0.7663;
                    m_256apsk[30].imag()  = -0.0867;
                    m_256apsk[31].real()  = -0.6115;
                    m_256apsk[31].imag()  = -0.0871;
                    m_256apsk[32].real()  = 1.2701;
                    m_256apsk[32].imag()  = 1.0139;
                    m_256apsk[33].real()  = 1.0525;
                    m_256apsk[33].imag()  = 0.8406;
                    m_256apsk[34].real()  = -1.2701;
                    m_256apsk[34].imag()  = 1.0139;
                    m_256apsk[35].real()  = -1.0525;
                    m_256apsk[35].imag()  = 0.8406;
                    m_256apsk[36].real()  = 0.2487;
                    m_256apsk[36].imag()  = 0.1978;
                    m_256apsk[37].real()  = 0.3523;
                    m_256apsk[37].imag()  = 0.2915;
                    m_256apsk[38].real()  = -0.2487;
                    m_256apsk[38].imag()  = 0.1978;
                    m_256apsk[39].real()  = -0.3523;
                    m_256apsk[39].imag()  = 0.2915;
                    m_256apsk[40].real()  = 1.2701;
                    m_256apsk[40].imag()  = -1.0139;
                    m_256apsk[41].real()  = 1.0525;
                    m_256apsk[41].imag()  = -0.8406;
                    m_256apsk[42].real()  = -1.2701;
                    m_256apsk[42].imag()  = -1.0139;
                    m_256apsk[43].real()  = -1.0525;
                    m_256apsk[43].imag()  = -0.8406;
                    m_256apsk[44].real()  = 0.2487;
                    m_256apsk[44].imag()  = -0.1978;
                    m_256apsk[45].real()  = 0.3523;
                    m_256apsk[45].imag()  = -0.2915;
                    m_256apsk[46].real()  = -0.2487;
                    m_256apsk[46].imag()  = -0.1978;
                    m_256apsk[47].real()  = -0.3523;
                    m_256apsk[47].imag()  = -0.2915;
                    m_256apsk[48].real()  = 0.7359;
                    m_256apsk[48].imag()  = 0.6043;
                    m_256apsk[49].real()  = 0.8807;
                    m_256apsk[49].imag()  = 0.7105;
                    m_256apsk[50].real()  = -0.7359;
                    m_256apsk[50].imag()  = 0.6043;
                    m_256apsk[51].real()  = -0.8807;
                    m_256apsk[51].imag()  = 0.7105;
                    m_256apsk[52].real()  = 0.6017;
                    m_256apsk[52].imag()  = 0.5019;
                    m_256apsk[53].real()  = 0.4747;
                    m_256apsk[53].imag()  = 0.3996;
                    m_256apsk[54].real()  = -0.6017;
                    m_256apsk[54].imag()  = 0.5019;
                    m_256apsk[55].real()  = -0.4747;
                    m_256apsk[55].imag()  = 0.3996;
                    m_256apsk[56].real()  = 0.7359;
                    m_256apsk[56].imag()  = -0.6043;
                    m_256apsk[57].real()  = 0.8807;
                    m_256apsk[57].imag()  = -0.7105;
                    m_256apsk[58].real()  = -0.7359;
                    m_256apsk[58].imag()  = -0.6043;
                    m_256apsk[59].real()  = -0.8807;
                    m_256apsk[59].imag()  = -0.7105;
                    m_256apsk[60].real()  = 0.6017;
                    m_256apsk[60].imag()  = -0.5019;
                    m_256apsk[61].real()  = 0.4747;
                    m_256apsk[61].imag()  = -0.3996;
                    m_256apsk[62].real()  = -0.6017;
                    m_256apsk[62].imag()  = -0.5019;
                    m_256apsk[63].real()  = -0.4747;
                    m_256apsk[63].imag()  = -0.3996;
                    m_256apsk[64].real()  = 1.5441;
                    m_256apsk[64].imag()  = 0.4545;
                    m_256apsk[65].real()  = 1.2750;
                    m_256apsk[65].imag()  = 0.3775;
                    m_256apsk[66].real()  = -1.5441;
                    m_256apsk[66].imag()  = 0.4545;
                    m_256apsk[67].real()  = -1.2750;
                    m_256apsk[67].imag()  = 0.3775;
                    m_256apsk[68].real()  = 0.2586;
                    m_256apsk[68].imag()  = 0.0752;
                    m_256apsk[69].real()  = 0.4435;
                    m_256apsk[69].imag()  = 0.1065;
                    m_256apsk[70].real()  = -0.2586;
                    m_256apsk[70].imag()  = 0.0752;
                    m_256apsk[71].real()  = -0.4435;
                    m_256apsk[71].imag()  = 0.1065;
                    m_256apsk[72].real()  = 1.5441;
                    m_256apsk[72].imag()  = -0.4545;
                    m_256apsk[73].real()  = 1.2750;
                    m_256apsk[73].imag()  = -0.3775;
                    m_256apsk[74].real()  = -1.5441;
                    m_256apsk[74].imag()  = -0.4545;
                    m_256apsk[75].real()  = -1.2750;
                    m_256apsk[75].imag()  = -0.3775;
                    m_256apsk[76].real()  = 0.2586;
                    m_256apsk[76].imag()  = -0.0752;
                    m_256apsk[77].real()  = 0.4435;
                    m_256apsk[77].imag()  = -0.1065;
                    m_256apsk[78].real()  = -0.2586;
                    m_256apsk[78].imag()  = -0.0752;
                    m_256apsk[79].real()  = -0.4435;
                    m_256apsk[79].imag()  = -0.1065;
                    m_256apsk[80].real()  = 0.8925;
                    m_256apsk[80].imag()  = 0.2771;
                    m_256apsk[81].real()  = 1.0649;
                    m_256apsk[81].imag()  = 0.3219;
                    m_256apsk[82].real()  = -0.8925;
                    m_256apsk[82].imag()  = 0.2771;
                    m_256apsk[83].real()  = -1.0649;
                    m_256apsk[83].imag()  = 0.3219;
                    m_256apsk[84].real()  = 0.7362;
                    m_256apsk[84].imag()  = 0.2279;
                    m_256apsk[85].real()  = 0.5936;
                    m_256apsk[85].imag()  = 0.1699;
                    m_256apsk[86].real()  = -0.7362;
                    m_256apsk[86].imag()  = 0.2279;
                    m_256apsk[87].real()  = -0.5936;
                    m_256apsk[87].imag()  = 0.1699;
                    m_256apsk[88].real()  = 0.8925;
                    m_256apsk[88].imag()  = -0.2771;
                    m_256apsk[89].real()  = 1.0649;
                    m_256apsk[89].imag()  = -0.3219;
                    m_256apsk[90].real()  = -0.8925;
                    m_256apsk[90].imag()  = -0.2771;
                    m_256apsk[91].real()  = -1.0649;
                    m_256apsk[91].imag()  = -0.3219;
                    m_256apsk[92].real()  = 0.7362;
                    m_256apsk[92].imag()  = -0.2279;
                    m_256apsk[93].real()  = 0.5936;
                    m_256apsk[93].imag()  = -0.1699;
                    m_256apsk[94].real()  = -0.7362;
                    m_256apsk[94].imag()  = -0.2279;
                    m_256apsk[95].real()  = -0.5936;
                    m_256apsk[95].imag()  = -0.1699;
                    m_256apsk[96].real()  = 1.4352;
                    m_256apsk[96].imag()  = 0.7452;
                    m_256apsk[97].real()  = 1.1866;
                    m_256apsk[97].imag()  = 0.6182;
                    m_256apsk[98].real()  = -1.4352;
                    m_256apsk[98].imag()  = 0.7452;
                    m_256apsk[99].real()  = -1.1866;
                    m_256apsk[99].imag()  = 0.6182;
                    m_256apsk[100].real() = 0.2523;
                    m_256apsk[100].imag() = 0.1944;
                    m_256apsk[101].real() = 0.3695;
                    m_256apsk[101].imag() = 0.2695;
                    m_256apsk[102].real() = -0.2523;
                    m_256apsk[102].imag() = 0.1944;
                    m_256apsk[103].real() = -0.3695;
                    m_256apsk[103].imag() = 0.2695;
                    m_256apsk[104].real() = 1.4352;
                    m_256apsk[104].imag() = -0.7452;
                    m_256apsk[105].real() = 1.1866;
                    m_256apsk[105].imag() = -0.6182;
                    m_256apsk[106].real() = -1.4352;
                    m_256apsk[106].imag() = -0.7452;
                    m_256apsk[107].real() = -1.1866;
                    m_256apsk[107].imag() = -0.6182;
                    m_256apsk[108].real() = 0.2523;
                    m_256apsk[108].imag() = -0.1944;
                    m_256apsk[109].real() = 0.3695;
                    m_256apsk[109].imag() = -0.2695;
                    m_256apsk[110].real() = -0.2523;
                    m_256apsk[110].imag() = -0.1944;
                    m_256apsk[111].real() = -0.3695;
                    m_256apsk[111].imag() = -0.2695;
                    m_256apsk[112].real() = 0.8273;
                    m_256apsk[112].imag() = 0.4493;
                    m_256apsk[113].real() = 0.9911;
                    m_256apsk[113].imag() = 0.5243;
                    m_256apsk[114].real() = -0.8273;
                    m_256apsk[114].imag() = 0.4493;
                    m_256apsk[115].real() = -0.9911;
                    m_256apsk[115].imag() = 0.5243;
                    m_256apsk[116].real() = 0.6708;
                    m_256apsk[116].imag() = 0.3859;
                    m_256apsk[117].real() = 0.5197;
                    m_256apsk[117].imag() = 0.3331;
                    m_256apsk[118].real() = -0.6708;
                    m_256apsk[118].imag() = 0.3859;
                    m_256apsk[119].real() = -0.5197;
                    m_256apsk[119].imag() = 0.3331;
                    m_256apsk[120].real() = 0.8273;
                    m_256apsk[120].imag() = -0.4493;
                    m_256apsk[121].real() = 0.9911;
                    m_256apsk[121].imag() = -0.5243;
                    m_256apsk[122].real() = -0.8273;
                    m_256apsk[122].imag() = -0.4493;
                    m_256apsk[123].real() = -0.9911;
                    m_256apsk[123].imag() = -0.5243;
                    m_256apsk[124].real() = 0.6708;
                    m_256apsk[124].imag() = -0.3859;
                    m_256apsk[125].real() = 0.5197;
                    m_256apsk[125].imag() = -0.3331;
                    m_256apsk[126].real() = -0.6708;
                    m_256apsk[126].imag() = -0.3859;
                    m_256apsk[127].real() = -0.5197;
                    m_256apsk[127].imag() = -0.3331;
                    m_256apsk[128].real() = 0.1646;
                    m_256apsk[128].imag() = 1.6329;
                    m_256apsk[129].real() = 0.1379;
                    m_256apsk[129].imag() = 1.3595;
                    m_256apsk[130].real() = -0.1646;
                    m_256apsk[130].imag() = 1.6329;
                    m_256apsk[131].real() = -0.1379;
                    m_256apsk[131].imag() = 1.3595;
                    m_256apsk[132].real() = 0.0736;
                    m_256apsk[132].imag() = 0.0898;
                    m_256apsk[133].real() = 0.0742;
                    m_256apsk[133].imag() = 0.5054;
                    m_256apsk[134].real() = -0.0736;
                    m_256apsk[134].imag() = 0.0898;
                    m_256apsk[135].real() = -0.0742;
                    m_256apsk[135].imag() = 0.5054;
                    m_256apsk[136].real() = 0.1646;
                    m_256apsk[136].imag() = -1.6329;
                    m_256apsk[137].real() = 0.1379;
                    m_256apsk[137].imag() = -1.3595;
                    m_256apsk[138].real() = -0.1646;
                    m_256apsk[138].imag() = -1.6329;
                    m_256apsk[139].real() = -0.1379;
                    m_256apsk[139].imag() = -1.3595;
                    m_256apsk[140].real() = 0.0736;
                    m_256apsk[140].imag() = -0.0898;
                    m_256apsk[141].real() = 0.0742;
                    m_256apsk[141].imag() = -0.5054;
                    m_256apsk[142].real() = -0.0736;
                    m_256apsk[142].imag() = -0.0898;
                    m_256apsk[143].real() = -0.0742;
                    m_256apsk[143].imag() = -0.5054;
                    m_256apsk[144].real() = 0.0992;
                    m_256apsk[144].imag() = 0.9847;
                    m_256apsk[145].real() = 0.1170;
                    m_256apsk[145].imag() = 1.1517;
                    m_256apsk[146].real() = -0.0992;
                    m_256apsk[146].imag() = 0.9847;
                    m_256apsk[147].real() = -0.1170;
                    m_256apsk[147].imag() = 1.1517;
                    m_256apsk[148].real() = 0.0894;
                    m_256apsk[148].imag() = 0.8287;
                    m_256apsk[149].real() = 0.0889;
                    m_256apsk[149].imag() = 0.6739;
                    m_256apsk[150].real() = -0.0894;
                    m_256apsk[150].imag() = 0.8287;
                    m_256apsk[151].real() = -0.0889;
                    m_256apsk[151].imag() = 0.6739;
                    m_256apsk[152].real() = 0.0992;
                    m_256apsk[152].imag() = -0.9847;
                    m_256apsk[153].real() = 0.1170;
                    m_256apsk[153].imag() = -1.1517;
                    m_256apsk[154].real() = -0.0992;
                    m_256apsk[154].imag() = -0.9847;
                    m_256apsk[155].real() = -0.1170;
                    m_256apsk[155].imag() = -1.1517;
                    m_256apsk[156].real() = 0.0894;
                    m_256apsk[156].imag() = -0.8287;
                    m_256apsk[157].real() = 0.0889;
                    m_256apsk[157].imag() = -0.6739;
                    m_256apsk[158].real() = -0.0894;
                    m_256apsk[158].imag() = -0.8287;
                    m_256apsk[159].real() = -0.0889;
                    m_256apsk[159].imag() = -0.6739;
                    m_256apsk[160].real() = 1.0516;
                    m_256apsk[160].imag() = 1.2481;
                    m_256apsk[161].real() = 0.8742;
                    m_256apsk[161].imag() = 1.0355;
                    m_256apsk[162].real() = -1.0516;
                    m_256apsk[162].imag() = 1.2481;
                    m_256apsk[163].real() = -0.8742;
                    m_256apsk[163].imag() = 1.0355;
                    m_256apsk[164].real() = 0.0970;
                    m_256apsk[164].imag() = 0.2450;
                    m_256apsk[165].real() = 0.1959;
                    m_256apsk[165].imag() = 0.4045;
                    m_256apsk[166].real() = -0.0970;
                    m_256apsk[166].imag() = 0.2450;
                    m_256apsk[167].real() = -0.1959;
                    m_256apsk[167].imag() = 0.4045;
                    m_256apsk[168].real() = 1.0516;
                    m_256apsk[168].imag() = -1.2481;
                    m_256apsk[169].real() = 0.8742;
                    m_256apsk[169].imag() = -1.0355;
                    m_256apsk[170].real() = -1.0516;
                    m_256apsk[170].imag() = -1.2481;
                    m_256apsk[171].real() = -0.8742;
                    m_256apsk[171].imag() = -1.0355;
                    m_256apsk[172].real() = 0.0970;
                    m_256apsk[172].imag() = -0.2450;
                    m_256apsk[173].real() = 0.1959;
                    m_256apsk[173].imag() = -0.4045;
                    m_256apsk[174].real() = -0.0970;
                    m_256apsk[174].imag() = -0.2450;
                    m_256apsk[175].real() = -0.1959;
                    m_256apsk[175].imag() = -0.4045;
                    m_256apsk[176].real() = 0.6150;
                    m_256apsk[176].imag() = 0.7441;
                    m_256apsk[177].real() = 0.7345;
                    m_256apsk[177].imag() = 0.8743;
                    m_256apsk[178].real() = -0.6150;
                    m_256apsk[178].imag() = 0.7441;
                    m_256apsk[179].real() = -0.7345;
                    m_256apsk[179].imag() = 0.8743;
                    m_256apsk[180].real() = 0.4932;
                    m_256apsk[180].imag() = 0.6301;
                    m_256apsk[181].real() = 0.3620;
                    m_256apsk[181].imag() = 0.5258;
                    m_256apsk[182].real() = -0.4932;
                    m_256apsk[182].imag() = 0.6301;
                    m_256apsk[183].real() = -0.3620;
                    m_256apsk[183].imag() = 0.5258;
                    m_256apsk[184].real() = 0.6150;
                    m_256apsk[184].imag() = -0.7441;
                    m_256apsk[185].real() = 0.7345;
                    m_256apsk[185].imag() = -0.8743;
                    m_256apsk[186].real() = -0.6150;
                    m_256apsk[186].imag() = -0.7441;
                    m_256apsk[187].real() = -0.7345;
                    m_256apsk[187].imag() = -0.8743;
                    m_256apsk[188].real() = 0.4932;
                    m_256apsk[188].imag() = -0.6301;
                    m_256apsk[189].real() = 0.3620;
                    m_256apsk[189].imag() = -0.5258;
                    m_256apsk[190].real() = -0.4932;
                    m_256apsk[190].imag() = -0.6301;
                    m_256apsk[191].real() = -0.3620;
                    m_256apsk[191].imag() = -0.5258;
                    m_256apsk[192].real() = 0.4866;
                    m_256apsk[192].imag() = 1.5660;
                    m_256apsk[193].real() = 0.4068;
                    m_256apsk[193].imag() = 1.3027;
                    m_256apsk[194].real() = -0.4866;
                    m_256apsk[194].imag() = 1.5660;
                    m_256apsk[195].real() = -0.4068;
                    m_256apsk[195].imag() = 1.3027;
                    m_256apsk[196].real() = 0.0732;
                    m_256apsk[196].imag() = 0.0899;
                    m_256apsk[197].real() = 0.0877;
                    m_256apsk[197].imag() = 0.4997;
                    m_256apsk[198].real() = -0.0732;
                    m_256apsk[198].imag() = 0.0899;
                    m_256apsk[199].real() = -0.0877;
                    m_256apsk[199].imag() = 0.4997;
                    m_256apsk[200].real() = 0.4866;
                    m_256apsk[200].imag() = -1.5660;
                    m_256apsk[201].real() = 0.4068;
                    m_256apsk[201].imag() = -1.3027;
                    m_256apsk[202].real() = -0.4866;
                    m_256apsk[202].imag() = -1.5660;
                    m_256apsk[203].real() = -0.4068;
                    m_256apsk[203].imag() = -1.3027;
                    m_256apsk[204].real() = 0.0732;
                    m_256apsk[204].imag() = -0.0899;
                    m_256apsk[205].real() = 0.0877;
                    m_256apsk[205].imag() = -0.4997;
                    m_256apsk[206].real() = -0.0732;
                    m_256apsk[206].imag() = -0.0899;
                    m_256apsk[207].real() = -0.0877;
                    m_256apsk[207].imag() = -0.4997;
                    m_256apsk[208].real() = 0.2927;
                    m_256apsk[208].imag() = 0.9409;
                    m_256apsk[209].real() = 0.3446;
                    m_256apsk[209].imag() = 1.1023;
                    m_256apsk[210].real() = -0.2927;
                    m_256apsk[210].imag() = 0.9409;
                    m_256apsk[211].real() = -0.3446;
                    m_256apsk[211].imag() = 1.1023;
                    m_256apsk[212].real() = 0.2350;
                    m_256apsk[212].imag() = 0.7945;
                    m_256apsk[213].real() = 0.1670;
                    m_256apsk[213].imag() = 0.6529;
                    m_256apsk[214].real() = -0.2350;
                    m_256apsk[214].imag() = 0.7945;
                    m_256apsk[215].real() = -0.1670;
                    m_256apsk[215].imag() = 0.6529;
                    m_256apsk[216].real() = 0.2927;
                    m_256apsk[216].imag() = -0.9409;
                    m_256apsk[217].real() = 0.3446;
                    m_256apsk[217].imag() = -1.1023;
                    m_256apsk[218].real() = -0.2927;
                    m_256apsk[218].imag() = -0.9409;
                    m_256apsk[219].real() = -0.3446;
                    m_256apsk[219].imag() = -1.1023;
                    m_256apsk[220].real() = 0.2350;
                    m_256apsk[220].imag() = -0.7945;
                    m_256apsk[221].real() = 0.1670;
                    m_256apsk[221].imag() = -0.6529;
                    m_256apsk[222].real() = -0.2350;
                    m_256apsk[222].imag() = -0.7945;
                    m_256apsk[223].real() = -0.1670;
                    m_256apsk[223].imag() = -0.6529;
                    m_256apsk[224].real() = 0.7867;
                    m_256apsk[224].imag() = 1.4356;
                    m_256apsk[225].real() = 0.6561;
                    m_256apsk[225].imag() = 1.1927;
                    m_256apsk[226].real() = -0.7867;
                    m_256apsk[226].imag() = 1.4356;
                    m_256apsk[227].real() = -0.6561;
                    m_256apsk[227].imag() = 1.1927;
                    m_256apsk[228].real() = 0.0947;
                    m_256apsk[228].imag() = 0.2451;
                    m_256apsk[229].real() = 0.1865;
                    m_256apsk[229].imag() = 0.4121;
                    m_256apsk[230].real() = -0.0947;
                    m_256apsk[230].imag() = 0.2451;
                    m_256apsk[231].real() = -0.1865;
                    m_256apsk[231].imag() = 0.4121;
                    m_256apsk[232].real() = 0.7867;
                    m_256apsk[232].imag() = -1.4356;
                    m_256apsk[233].real() = 0.6561;
                    m_256apsk[233].imag() = -1.1927;
                    m_256apsk[234].real() = -0.7867;
                    m_256apsk[234].imag() = -1.4356;
                    m_256apsk[235].real() = -0.6561;
                    m_256apsk[235].imag() = -1.1927;
                    m_256apsk[236].real() = 0.0947;
                    m_256apsk[236].imag() = -0.2451;
                    m_256apsk[237].real() = 0.1865;
                    m_256apsk[237].imag() = -0.4121;
                    m_256apsk[238].real() = -0.0947;
                    m_256apsk[238].imag() = -0.2451;
                    m_256apsk[239].real() = -0.1865;
                    m_256apsk[239].imag() = -0.4121;
                    m_256apsk[240].real() = 0.4677;
                    m_256apsk[240].imag() = 0.8579;
                    m_256apsk[241].real() = 0.5537;
                    m_256apsk[241].imag() = 1.0081;
                    m_256apsk[242].real() = -0.4677;
                    m_256apsk[242].imag() = 0.8579;
                    m_256apsk[243].real() = -0.5537;
                    m_256apsk[243].imag() = 1.0081;
                    m_256apsk[244].real() = 0.3893;
                    m_256apsk[244].imag() = 0.7143;
                    m_256apsk[245].real() = 0.3110;
                    m_256apsk[245].imag() = 0.5686;
                    m_256apsk[246].real() = -0.3893;
                    m_256apsk[246].imag() = 0.7143;
                    m_256apsk[247].real() = -0.3110;
                    m_256apsk[247].imag() = 0.5686;
                    m_256apsk[248].real() = 0.4677;
                    m_256apsk[248].imag() = -0.8579;
                    m_256apsk[249].real() = 0.5537;
                    m_256apsk[249].imag() = -1.0081;
                    m_256apsk[250].real() = -0.4677;
                    m_256apsk[250].imag() = -0.8579;
                    m_256apsk[251].real() = -0.5537;
                    m_256apsk[251].imag() = -1.0081;
                    m_256apsk[252].real() = 0.3893;
                    m_256apsk[252].imag() = -0.7143;
                    m_256apsk[253].real() = 0.3110;
                    m_256apsk[253].imag() = -0.5686;
                    m_256apsk[254].real() = -0.3893;
                    m_256apsk[254].imag() = -0.7143;
                    m_256apsk[255].real() = -0.3110;
                    m_256apsk[255].imag() = -0.5686;
                    for (int i = 0; i < 256; i++)
                    {
                        m_256apsk[i].real() /= 1.6329;
                        m_256apsk[i].imag() /= 1.6329;
                    }
                }
                else
                {
                    r8 = m;
                    switch(rate)
                    {
                        case gr::dvbs2::C116_180:
                        case gr::dvbs2::C124_180:
                            r1 = r8 / 6.536;
                            r7 = r1 * 5.078;
                            r6 = r1 * 4.235;
                            r5 = r1 * 3.569;
                            r4 = r1 * 2.98;
                            r3 = r1 * 2.405;
                            r2 = r1 * 1.791;
                            break;
                        case gr::dvbs2::C128_180:
                            r1 = r8 / 5.4;
                            r7 = r1 * 4.6;
                            r6 = r1 * 4.045;
                            r5 = r1 * 3.579;
                            r4 = r1 * 2.986;
                            r3 = r1 * 2.409;
                            r2 = r1 * 1.794;
                            break;
                        case gr::dvbs2::C135_180:
                            r1 = r8 / 5.2;
                            r7 = r1 * 4.5;
                            r6 = r1 * 4.045;
                            r5 = r1 * 3.579;
                            r4 = r1 * 2.986;
                            r3 = r1 * 2.409;
                            r2 = r1 * 1.794;
                            break;
                        default:
                            r1 = 0;
                            r2 = 0;
                            r3 = 0;
                            r4 = 0;
                            r5 = 0;
                            r6 = 0;
                            r7 = 0;
                            break;
                    }
                    m_256apsk[0].real()   = (r1 * cos(M_PI / 32.0));
                    m_256apsk[0].imag()   = (r1 * sin(M_PI / 32.0));
                    m_256apsk[1].real()   = (r1 * cos(3 * M_PI / 32.0));
                    m_256apsk[1].imag()   = (r1 * sin(3 * M_PI / 32.0));
                    m_256apsk[2].real()   = (r1 * cos(7 * M_PI / 32.0));
                    m_256apsk[2].imag()   = (r1 * sin(7 * M_PI / 32.0));
                    m_256apsk[3].real()   = (r1 * cos(5 * M_PI / 32.0));
                    m_256apsk[3].imag()   = (r1 * sin(5 * M_PI / 32.0));
                    m_256apsk[4].real()   = (r1 * cos(15 * M_PI / 32.0));
                    m_256apsk[4].imag()   = (r1 * sin(15 * M_PI / 32.0));
                    m_256apsk[5].real()   = (r1 * cos(13 * M_PI / 32.0));
                    m_256apsk[5].imag()   = (r1 * sin(13 * M_PI / 32.0));
                    m_256apsk[6].real()   = (r1 * cos(9 * M_PI / 32.0));
                    m_256apsk[6].imag()   = (r1 * sin(9 * M_PI / 32.0));
                    m_256apsk[7].real()   = (r1 * cos(11 * M_PI / 32.0));
                    m_256apsk[7].imag()   = (r1 * sin(11 * M_PI / 32.0));
                    m_256apsk[8].real()   = (r1 * cos(31 * M_PI / 32.0));
                    m_256apsk[8].imag()   = (r1 * sin(31 * M_PI / 32.0));
                    m_256apsk[9].real()   = (r1 * cos(29 * M_PI / 32.0));
                    m_256apsk[9].imag()   = (r1 * sin(29 * M_PI / 32.0));
                    m_256apsk[10].real()   = (r1 * cos(25 * M_PI / 32.0));
                    m_256apsk[10].imag()   = (r1 * sin(25 * M_PI / 32.0));
                    m_256apsk[11].real()   = (r1 * cos(27 * M_PI / 32.0));
                    m_256apsk[11].imag()   = (r1 * sin(27 * M_PI / 32.0));
                    m_256apsk[12].real()   = (r1 * cos(17 * M_PI / 32.0));
                    m_256apsk[12].imag()   = (r1 * sin(17 * M_PI / 32.0));
                    m_256apsk[13].real()   = (r1 * cos(19 * M_PI / 32.0));
                    m_256apsk[13].imag()   = (r1 * sin(19 * M_PI / 32.0));
                    m_256apsk[14].real()   = (r1 * cos(23 * M_PI / 32.0));
                    m_256apsk[14].imag()   = (r1 * sin(23 * M_PI / 32.0));
                    m_256apsk[15].real()   = (r1 * cos(21 * M_PI / 32.0));
                    m_256apsk[15].imag()   = (r1 * sin(21 * M_PI / 32.0));
                    m_256apsk[16].real()   = (r1 * cos(-1 * M_PI / 32.0));
                    m_256apsk[16].imag()   = (r1 * sin(-1 * M_PI / 32.0));
                    m_256apsk[17].real()   = (r1 * cos(-3 * M_PI / 32.0));
                    m_256apsk[17].imag()   = (r1 * sin(-3 * M_PI / 32.0));
                    m_256apsk[18].real()   = (r1 * cos(-7 * M_PI / 32.0));
                    m_256apsk[18].imag()   = (r1 * sin(-7 * M_PI / 32.0));
                    m_256apsk[19].real()   = (r1 * cos(-5 * M_PI / 32.0));
                    m_256apsk[19].imag()   = (r1 * sin(-5 * M_PI / 32.0));
                    m_256apsk[20].real()   = (r1 * cos(-15 * M_PI / 32.0));
                    m_256apsk[20].imag()   = (r1 * sin(-15 * M_PI / 32.0));
                    m_256apsk[21].real()   = (r1 * cos(-13 * M_PI / 32.0));
                    m_256apsk[21].imag()   = (r1 * sin(-13 * M_PI / 32.0));
                    m_256apsk[22].real()   = (r1 * cos(-9 * M_PI / 32.0));
                    m_256apsk[22].imag()   = (r1 * sin(-9 * M_PI / 32.0));
                    m_256apsk[23].real()   = (r1 * cos(-11 * M_PI / 32.0));
                    m_256apsk[23].imag()   = (r1 * sin(-11 * M_PI / 32.0));
                    m_256apsk[24].real()   = (r1 * cos(33 * M_PI / 32.0));
                    m_256apsk[24].imag()   = (r1 * sin(33 * M_PI / 32.0));
                    m_256apsk[25].real()   = (r1 * cos(35 * M_PI / 32.0));
                    m_256apsk[25].imag()   = (r1 * sin(35 * M_PI / 32.0));
                    m_256apsk[26].real()   = (r1 * cos(39 * M_PI / 32.0));
                    m_256apsk[26].imag()   = (r1 * sin(39 * M_PI / 32.0));
                    m_256apsk[27].real()   = (r1 * cos(37 * M_PI / 32.0));
                    m_256apsk[27].imag()   = (r1 * sin(37 * M_PI / 32.0));
                    m_256apsk[28].real()   = (r1 * cos(47 * M_PI / 32.0));
                    m_256apsk[28].imag()   = (r1 * sin(47 * M_PI / 32.0));
                    m_256apsk[29].real()   = (r1 * cos(45 * M_PI / 32.0));
                    m_256apsk[29].imag()   = (r1 * sin(45 * M_PI / 32.0));
                    m_256apsk[30].real()   = (r1 * cos(41 * M_PI / 32.0));
                    m_256apsk[30].imag()   = (r1 * sin(41 * M_PI / 32.0));
                    m_256apsk[31].real()   = (r1 * cos(43 * M_PI / 32.0));
                    m_256apsk[31].imag()   = (r1 * sin(43 * M_PI / 32.0));
                    m_256apsk[32].real()   = (r2 * cos(M_PI / 32.0));
                    m_256apsk[32].imag()   = (r2 * sin(M_PI / 32.0));
                    m_256apsk[33].real()   = (r2 * cos(3 * M_PI / 32.0));
                    m_256apsk[33].imag()   = (r2 * sin(3 * M_PI / 32.0));
                    m_256apsk[34].real()   = (r2 * cos(7 * M_PI / 32.0));
                    m_256apsk[34].imag()   = (r2 * sin(7 * M_PI / 32.0));
                    m_256apsk[35].real()   = (r2 * cos(5 * M_PI / 32.0));
                    m_256apsk[35].imag()   = (r2 * sin(5 * M_PI / 32.0));
                    m_256apsk[36].real()   = (r2 * cos(15 * M_PI / 32.0));
                    m_256apsk[36].imag()   = (r2 * sin(15 * M_PI / 32.0));
                    m_256apsk[37].real()   = (r2 * cos(13 * M_PI / 32.0));
                    m_256apsk[37].imag()   = (r2 * sin(13 * M_PI / 32.0));
                    m_256apsk[38].real()   = (r2 * cos(9 * M_PI / 32.0));
                    m_256apsk[38].imag()   = (r2 * sin(9 * M_PI / 32.0));
                    m_256apsk[39].real()   = (r2 * cos(11 * M_PI / 32.0));
                    m_256apsk[39].imag()   = (r2 * sin(11 * M_PI / 32.0));
                    m_256apsk[40].real()   = (r2 * cos(31 * M_PI / 32.0));
                    m_256apsk[40].imag()   = (r2 * sin(31 * M_PI / 32.0));
                    m_256apsk[41].real()   = (r2 * cos(29 * M_PI / 32.0));
                    m_256apsk[41].imag()   = (r2 * sin(29 * M_PI / 32.0));
                    m_256apsk[42].real()   = (r2 * cos(25 * M_PI / 32.0));
                    m_256apsk[42].imag()   = (r2 * sin(25 * M_PI / 32.0));
                    m_256apsk[43].real()   = (r2 * cos(27 * M_PI / 32.0));
                    m_256apsk[43].imag()   = (r2 * sin(27 * M_PI / 32.0));
                    m_256apsk[44].real()   = (r2 * cos(17 * M_PI / 32.0));
                    m_256apsk[44].imag()   = (r2 * sin(17 * M_PI / 32.0));
                    m_256apsk[45].real()   = (r2 * cos(19 * M_PI / 32.0));
                    m_256apsk[45].imag()   = (r2 * sin(19 * M_PI / 32.0));
                    m_256apsk[46].real()   = (r2 * cos(23 * M_PI / 32.0));
                    m_256apsk[46].imag()   = (r2 * sin(23 * M_PI / 32.0));
                    m_256apsk[47].real()   = (r2 * cos(21 * M_PI / 32.0));
                    m_256apsk[47].imag()   = (r2 * sin(21 * M_PI / 32.0));
                    m_256apsk[48].real()   = (r2 * cos(-1 * M_PI / 32.0));
                    m_256apsk[48].imag()   = (r2 * sin(-1 * M_PI / 32.0));
                    m_256apsk[49].real()   = (r2 * cos(-3 * M_PI / 32.0));
                    m_256apsk[49].imag()   = (r2 * sin(-3 * M_PI / 32.0));
                    m_256apsk[50].real()   = (r2 * cos(-7 * M_PI / 32.0));
                    m_256apsk[50].imag()   = (r2 * sin(-7 * M_PI / 32.0));
                    m_256apsk[51].real()   = (r2 * cos(-5 * M_PI / 32.0));
                    m_256apsk[51].imag()   = (r2 * sin(-5 * M_PI / 32.0));
                    m_256apsk[52].real()   = (r2 * cos(-15 * M_PI / 32.0));
                    m_256apsk[52].imag()   = (r2 * sin(-15 * M_PI / 32.0));
                    m_256apsk[53].real()   = (r2 * cos(-13 * M_PI / 32.0));
                    m_256apsk[53].imag()   = (r2 * sin(-13 * M_PI / 32.0));
                    m_256apsk[54].real()   = (r2 * cos(-9 * M_PI / 32.0));
                    m_256apsk[54].imag()   = (r2 * sin(-9 * M_PI / 32.0));
                    m_256apsk[55].real()   = (r2 * cos(-11 * M_PI / 32.0));
                    m_256apsk[55].imag()   = (r2 * sin(-11 * M_PI / 32.0));
                    m_256apsk[56].real()   = (r2 * cos(33 * M_PI / 32.0));
                    m_256apsk[56].imag()   = (r2 * sin(33 * M_PI / 32.0));
                    m_256apsk[57].real()   = (r2 * cos(35 * M_PI / 32.0));
                    m_256apsk[57].imag()   = (r2 * sin(35 * M_PI / 32.0));
                    m_256apsk[58].real()   = (r2 * cos(39 * M_PI / 32.0));
                    m_256apsk[58].imag()   = (r2 * sin(39 * M_PI / 32.0));
                    m_256apsk[59].real()   = (r2 * cos(37 * M_PI / 32.0));
                    m_256apsk[59].imag()   = (r2 * sin(37 * M_PI / 32.0));
                    m_256apsk[60].real()   = (r2 * cos(47 * M_PI / 32.0));
                    m_256apsk[60].imag()   = (r2 * sin(47 * M_PI / 32.0));
                    m_256apsk[61].real()   = (r2 * cos(45 * M_PI / 32.0));
                    m_256apsk[61].imag()   = (r2 * sin(45 * M_PI / 32.0));
                    m_256apsk[62].real()   = (r2 * cos(41 * M_PI / 32.0));
                    m_256apsk[62].imag()   = (r2 * sin(41 * M_PI / 32.0));
                    m_256apsk[63].real()   = (r2 * cos(43 * M_PI / 32.0));
                    m_256apsk[63].imag()   = (r2 * sin(43 * M_PI / 32.0));
                    m_256apsk[64].real()   = (r4 * cos(M_PI / 32.0));
                    m_256apsk[64].imag()   = (r4 * sin(M_PI / 32.0));
                    m_256apsk[65].real()   = (r4 * cos(3 * M_PI / 32.0));
                    m_256apsk[65].imag()   = (r4 * sin(3 * M_PI / 32.0));
                    m_256apsk[66].real()   = (r4 * cos(7 * M_PI / 32.0));
                    m_256apsk[66].imag()   = (r4 * sin(7 * M_PI / 32.0));
                    m_256apsk[67].real()   = (r4 * cos(5 * M_PI / 32.0));
                    m_256apsk[67].imag()   = (r4 * sin(5 * M_PI / 32.0));
                    m_256apsk[68].real()   = (r4 * cos(15 * M_PI / 32.0));
                    m_256apsk[68].imag()   = (r4 * sin(15 * M_PI / 32.0));
                    m_256apsk[69].real()   = (r4 * cos(13 * M_PI / 32.0));
                    m_256apsk[69].imag()   = (r4 * sin(13 * M_PI / 32.0));
                    m_256apsk[70].real()   = (r4 * cos(9 * M_PI / 32.0));
                    m_256apsk[70].imag()   = (r4 * sin(9 * M_PI / 32.0));
                    m_256apsk[71].real()   = (r4 * cos(11 * M_PI / 32.0));
                    m_256apsk[71].imag()   = (r4 * sin(11 * M_PI / 32.0));
                    m_256apsk[72].real()   = (r4 * cos(31 * M_PI / 32.0));
                    m_256apsk[72].imag()   = (r4 * sin(31 * M_PI / 32.0));
                    m_256apsk[73].real()   = (r4 * cos(29 * M_PI / 32.0));
                    m_256apsk[73].imag()   = (r4 * sin(29 * M_PI / 32.0));
                    m_256apsk[74].real()   = (r4 * cos(25 * M_PI / 32.0));
                    m_256apsk[74].imag()   = (r4 * sin(25 * M_PI / 32.0));
                    m_256apsk[75].real()   = (r4 * cos(27 * M_PI / 32.0));
                    m_256apsk[75].imag()   = (r4 * sin(27 * M_PI / 32.0));
                    m_256apsk[76].real()   = (r4 * cos(17 * M_PI / 32.0));
                    m_256apsk[76].imag()   = (r4 * sin(17 * M_PI / 32.0));
                    m_256apsk[77].real()   = (r4 * cos(19 * M_PI / 32.0));
                    m_256apsk[77].imag()   = (r4 * sin(19 * M_PI / 32.0));
                    m_256apsk[78].real()   = (r4 * cos(23 * M_PI / 32.0));
                    m_256apsk[78].imag()   = (r4 * sin(23 * M_PI / 32.0));
                    m_256apsk[79].real()   = (r4 * cos(21 * M_PI / 32.0));
                    m_256apsk[79].imag()   = (r4 * sin(21 * M_PI / 32.0));
                    m_256apsk[80].real()   = (r4 * cos(-1 * M_PI / 32.0));
                    m_256apsk[80].imag()   = (r4 * sin(-1 * M_PI / 32.0));
                    m_256apsk[81].real()   = (r4 * cos(-3 * M_PI / 32.0));
                    m_256apsk[81].imag()   = (r4 * sin(-3 * M_PI / 32.0));
                    m_256apsk[82].real()   = (r4 * cos(-7 * M_PI / 32.0));
                    m_256apsk[82].imag()   = (r4 * sin(-7 * M_PI / 32.0));
                    m_256apsk[83].real()   = (r4 * cos(-5 * M_PI / 32.0));
                    m_256apsk[83].imag()   = (r4 * sin(-5 * M_PI / 32.0));
                    m_256apsk[84].real()   = (r4 * cos(-15 * M_PI / 32.0));
                    m_256apsk[84].imag()   = (r4 * sin(-15 * M_PI / 32.0));
                    m_256apsk[85].real()   = (r4 * cos(-13 * M_PI / 32.0));
                    m_256apsk[85].imag()   = (r4 * sin(-13 * M_PI / 32.0));
                    m_256apsk[86].real()   = (r4 * cos(-9 * M_PI / 32.0));
                    m_256apsk[86].imag()   = (r4 * sin(-9 * M_PI / 32.0));
                    m_256apsk[87].real()   = (r4 * cos(-11 * M_PI / 32.0));
                    m_256apsk[87].imag()   = (r4 * sin(-11 * M_PI / 32.0));
                    m_256apsk[88].real()   = (r4 * cos(33 * M_PI / 32.0));
                    m_256apsk[88].imag()   = (r4 * sin(33 * M_PI / 32.0));
                    m_256apsk[89].real()   = (r4 * cos(35 * M_PI / 32.0));
                    m_256apsk[89].imag()   = (r4 * sin(35 * M_PI / 32.0));
                    m_256apsk[90].real()   = (r4 * cos(39 * M_PI / 32.0));
                    m_256apsk[90].imag()   = (r4 * sin(39 * M_PI / 32.0));
                    m_256apsk[91].real()   = (r4 * cos(37 * M_PI / 32.0));
                    m_256apsk[91].imag()   = (r4 * sin(37 * M_PI / 32.0));
                    m_256apsk[92].real()   = (r4 * cos(47 * M_PI / 32.0));
                    m_256apsk[92].imag()   = (r4 * sin(47 * M_PI / 32.0));
                    m_256apsk[93].real()   = (r4 * cos(45 * M_PI / 32.0));
                    m_256apsk[93].imag()   = (r4 * sin(45 * M_PI / 32.0));
                    m_256apsk[94].real()   = (r4 * cos(41 * M_PI / 32.0));
                    m_256apsk[94].imag()   = (r4 * sin(41 * M_PI / 32.0));
                    m_256apsk[95].real()   = (r4 * cos(43 * M_PI / 32.0));
                    m_256apsk[95].imag()   = (r4 * sin(43 * M_PI / 32.0));
                    m_256apsk[96].real()   = (r3 * cos(M_PI / 32.0));
                    m_256apsk[96].imag()   = (r3 * sin(M_PI / 32.0));
                    m_256apsk[97].real()   = (r3 * cos(3 * M_PI / 32.0));
                    m_256apsk[97].imag()   = (r3 * sin(3 * M_PI / 32.0));
                    m_256apsk[98].real()   = (r3 * cos(7 * M_PI / 32.0));
                    m_256apsk[98].imag()   = (r3 * sin(7 * M_PI / 32.0));
                    m_256apsk[99].real()   = (r3 * cos(5 * M_PI / 32.0));
                    m_256apsk[99].imag()   = (r3 * sin(5 * M_PI / 32.0));
                    m_256apsk[100].real()   = (r3 * cos(15 * M_PI / 32.0));
                    m_256apsk[100].imag()   = (r3 * sin(15 * M_PI / 32.0));
                    m_256apsk[101].real()   = (r3 * cos(13 * M_PI / 32.0));
                    m_256apsk[101].imag()   = (r3 * sin(13 * M_PI / 32.0));
                    m_256apsk[102].real()   = (r3 * cos(9 * M_PI / 32.0));
                    m_256apsk[102].imag()   = (r3 * sin(9 * M_PI / 32.0));
                    m_256apsk[103].real()   = (r3 * cos(11 * M_PI / 32.0));
                    m_256apsk[103].imag()   = (r3 * sin(11 * M_PI / 32.0));
                    m_256apsk[104].real()   = (r3 * cos(31 * M_PI / 32.0));
                    m_256apsk[104].imag()   = (r3 * sin(31 * M_PI / 32.0));
                    m_256apsk[105].real()   = (r3 * cos(29 * M_PI / 32.0));
                    m_256apsk[105].imag()   = (r3 * sin(29 * M_PI / 32.0));
                    m_256apsk[106].real()   = (r3 * cos(25 * M_PI / 32.0));
                    m_256apsk[106].imag()   = (r3 * sin(25 * M_PI / 32.0));
                    m_256apsk[107].real()   = (r3 * cos(27 * M_PI / 32.0));
                    m_256apsk[107].imag()   = (r3 * sin(27 * M_PI / 32.0));
                    m_256apsk[108].real()   = (r3 * cos(17 * M_PI / 32.0));
                    m_256apsk[108].imag()   = (r3 * sin(17 * M_PI / 32.0));
                    m_256apsk[109].real()   = (r3 * cos(19 * M_PI / 32.0));
                    m_256apsk[109].imag()   = (r3 * sin(19 * M_PI / 32.0));
                    m_256apsk[110].real()   = (r3 * cos(23 * M_PI / 32.0));
                    m_256apsk[110].imag()   = (r3 * sin(23 * M_PI / 32.0));
                    m_256apsk[111].real()   = (r3 * cos(21 * M_PI / 32.0));
                    m_256apsk[111].imag()   = (r3 * sin(21 * M_PI / 32.0));
                    m_256apsk[112].real()   = (r3 * cos(-1 * M_PI / 32.0));
                    m_256apsk[112].imag()   = (r3 * sin(-1 * M_PI / 32.0));
                    m_256apsk[113].real()   = (r3 * cos(-3 * M_PI / 32.0));
                    m_256apsk[113].imag()   = (r3 * sin(-3 * M_PI / 32.0));
                    m_256apsk[114].real()   = (r3 * cos(-7 * M_PI / 32.0));
                    m_256apsk[114].imag()   = (r3 * sin(-7 * M_PI / 32.0));
                    m_256apsk[115].real()   = (r3 * cos(-5 * M_PI / 32.0));
                    m_256apsk[115].imag()   = (r3 * sin(-5 * M_PI / 32.0));
                    m_256apsk[116].real()   = (r3 * cos(-15 * M_PI / 32.0));
                    m_256apsk[116].imag()   = (r3 * sin(-15 * M_PI / 32.0));
                    m_256apsk[117].real()   = (r3 * cos(-13 * M_PI / 32.0));
                    m_256apsk[117].imag()   = (r3 * sin(-13 * M_PI / 32.0));
                    m_256apsk[118].real()   = (r3 * cos(-9 * M_PI / 32.0));
                    m_256apsk[118].imag()   = (r3 * sin(-9 * M_PI / 32.0));
                    m_256apsk[119].real()   = (r3 * cos(-11 * M_PI / 32.0));
                    m_256apsk[119].imag()   = (r3 * sin(-11 * M_PI / 32.0));
                    m_256apsk[120].real()   = (r3 * cos(33 * M_PI / 32.0));
                    m_256apsk[120].imag()   = (r3 * sin(33 * M_PI / 32.0));
                    m_256apsk[121].real()   = (r3 * cos(35 * M_PI / 32.0));
                    m_256apsk[121].imag()   = (r3 * sin(35 * M_PI / 32.0));
                    m_256apsk[122].real()   = (r3 * cos(39 * M_PI / 32.0));
                    m_256apsk[122].imag()   = (r3 * sin(39 * M_PI / 32.0));
                    m_256apsk[123].real()   = (r3 * cos(37 * M_PI / 32.0));
                    m_256apsk[123].imag()   = (r3 * sin(37 * M_PI / 32.0));
                    m_256apsk[124].real()   = (r3 * cos(47 * M_PI / 32.0));
                    m_256apsk[124].imag()   = (r3 * sin(47 * M_PI / 32.0));
                    m_256apsk[125].real()   = (r3 * cos(45 * M_PI / 32.0));
                    m_256apsk[125].imag()   = (r3 * sin(45 * M_PI / 32.0));
                    m_256apsk[126].real()   = (r3 * cos(41 * M_PI / 32.0));
                    m_256apsk[126].imag()   = (r3 * sin(41 * M_PI / 32.0));
                    m_256apsk[127].real()   = (r3 * cos(43 * M_PI / 32.0));
                    m_256apsk[127].imag()   = (r3 * sin(43 * M_PI / 32.0));
                    m_256apsk[128].real()   = (r8 * cos(M_PI / 32.0));
                    m_256apsk[128].imag()   = (r8 * sin(M_PI / 32.0));
                    m_256apsk[129].real()   = (r8 * cos(3 * M_PI / 32.0));
                    m_256apsk[129].imag()   = (r8 * sin(3 * M_PI / 32.0));
                    m_256apsk[130].real()   = (r8 * cos(7 * M_PI / 32.0));
                    m_256apsk[130].imag()   = (r8 * sin(7 * M_PI / 32.0));
                    m_256apsk[131].real()   = (r8 * cos(5 * M_PI / 32.0));
                    m_256apsk[131].imag()   = (r8 * sin(5 * M_PI / 32.0));
                    m_256apsk[132].real()   = (r8 * cos(15 * M_PI / 32.0));
                    m_256apsk[132].imag()   = (r8 * sin(15 * M_PI / 32.0));
                    m_256apsk[133].real()   = (r8 * cos(13 * M_PI / 32.0));
                    m_256apsk[133].imag()   = (r8 * sin(13 * M_PI / 32.0));
                    m_256apsk[134].real()   = (r8 * cos(9 * M_PI / 32.0));
                    m_256apsk[134].imag()   = (r8 * sin(9 * M_PI / 32.0));
                    m_256apsk[135].real()   = (r8 * cos(11 * M_PI / 32.0));
                    m_256apsk[135].imag()   = (r8 * sin(11 * M_PI / 32.0));
                    m_256apsk[136].real()   = (r8 * cos(31 * M_PI / 32.0));
                    m_256apsk[136].imag()   = (r8 * sin(31 * M_PI / 32.0));
                    m_256apsk[137].real()   = (r8 * cos(29 * M_PI / 32.0));
                    m_256apsk[137].imag()   = (r8 * sin(29 * M_PI / 32.0));
                    m_256apsk[138].real()   = (r8 * cos(25 * M_PI / 32.0));
                    m_256apsk[138].imag()   = (r8 * sin(25 * M_PI / 32.0));
                    m_256apsk[139].real()   = (r8 * cos(27 * M_PI / 32.0));
                    m_256apsk[139].imag()   = (r8 * sin(27 * M_PI / 32.0));
                    m_256apsk[140].real()   = (r8 * cos(17 * M_PI / 32.0));
                    m_256apsk[140].imag()   = (r8 * sin(17 * M_PI / 32.0));
                    m_256apsk[141].real()   = (r8 * cos(19 * M_PI / 32.0));
                    m_256apsk[141].imag()   = (r8 * sin(19 * M_PI / 32.0));
                    m_256apsk[142].real()   = (r8 * cos(23 * M_PI / 32.0));
                    m_256apsk[142].imag()   = (r8 * sin(23 * M_PI / 32.0));
                    m_256apsk[143].real()   = (r8 * cos(21 * M_PI / 32.0));
                    m_256apsk[143].imag()   = (r8 * sin(21 * M_PI / 32.0));
                    m_256apsk[144].real()   = (r8 * cos(-1 * M_PI / 32.0));
                    m_256apsk[144].imag()   = (r8 * sin(-1 * M_PI / 32.0));
                    m_256apsk[145].real()   = (r8 * cos(-3 * M_PI / 32.0));
                    m_256apsk[145].imag()   = (r8 * sin(-3 * M_PI / 32.0));
                    m_256apsk[146].real()   = (r8 * cos(-7 * M_PI / 32.0));
                    m_256apsk[146].imag()   = (r8 * sin(-7 * M_PI / 32.0));
                    m_256apsk[147].real()   = (r8 * cos(-5 * M_PI / 32.0));
                    m_256apsk[147].imag()   = (r8 * sin(-5 * M_PI / 32.0));
                    m_256apsk[148].real()   = (r8 * cos(-15 * M_PI / 32.0));
                    m_256apsk[148].imag()   = (r8 * sin(-15 * M_PI / 32.0));
                    m_256apsk[149].real()   = (r8 * cos(-13 * M_PI / 32.0));
                    m_256apsk[149].imag()   = (r8 * sin(-13 * M_PI / 32.0));
                    m_256apsk[150].real()   = (r8 * cos(-9 * M_PI / 32.0));
                    m_256apsk[150].imag()   = (r8 * sin(-9 * M_PI / 32.0));
                    m_256apsk[151].real()   = (r8 * cos(-11 * M_PI / 32.0));
                    m_256apsk[151].imag()   = (r8 * sin(-11 * M_PI / 32.0));
                    m_256apsk[152].real()   = (r8 * cos(33 * M_PI / 32.0));
                    m_256apsk[152].imag()   = (r8 * sin(33 * M_PI / 32.0));
                    m_256apsk[153].real()   = (r8 * cos(35 * M_PI / 32.0));
                    m_256apsk[153].imag()   = (r8 * sin(35 * M_PI / 32.0));
                    m_256apsk[154].real()   = (r8 * cos(39 * M_PI / 32.0));
                    m_256apsk[154].imag()   = (r8 * sin(39 * M_PI / 32.0));
                    m_256apsk[155].real()   = (r8 * cos(37 * M_PI / 32.0));
                    m_256apsk[155].imag()   = (r8 * sin(37 * M_PI / 32.0));
                    m_256apsk[156].real()   = (r8 * cos(47 * M_PI / 32.0));
                    m_256apsk[156].imag()   = (r8 * sin(47 * M_PI / 32.0));
                    m_256apsk[157].real()   = (r8 * cos(45 * M_PI / 32.0));
                    m_256apsk[157].imag()   = (r8 * sin(45 * M_PI / 32.0));
                    m_256apsk[158].real()   = (r8 * cos(41 * M_PI / 32.0));
                    m_256apsk[158].imag()   = (r8 * sin(41 * M_PI / 32.0));
                    m_256apsk[159].real()   = (r8 * cos(43 * M_PI / 32.0));
                    m_256apsk[159].imag()   = (r8 * sin(43 * M_PI / 32.0));
                    m_256apsk[160].real()   = (r7 * cos(M_PI / 32.0));
                    m_256apsk[160].imag()   = (r7 * sin(M_PI / 32.0));
                    m_256apsk[161].real()   = (r7 * cos(3 * M_PI / 32.0));
                    m_256apsk[161].imag()   = (r7 * sin(3 * M_PI / 32.0));
                    m_256apsk[162].real()   = (r7 * cos(7 * M_PI / 32.0));
                    m_256apsk[162].imag()   = (r7 * sin(7 * M_PI / 32.0));
                    m_256apsk[163].real()   = (r7 * cos(5 * M_PI / 32.0));
                    m_256apsk[163].imag()   = (r7 * sin(5 * M_PI / 32.0));
                    m_256apsk[164].real()   = (r7 * cos(15 * M_PI / 32.0));
                    m_256apsk[164].imag()   = (r7 * sin(15 * M_PI / 32.0));
                    m_256apsk[165].real()   = (r7 * cos(13 * M_PI / 32.0));
                    m_256apsk[165].imag()   = (r7 * sin(13 * M_PI / 32.0));
                    m_256apsk[166].real()   = (r7 * cos(9 * M_PI / 32.0));
                    m_256apsk[166].imag()   = (r7 * sin(9 * M_PI / 32.0));
                    m_256apsk[167].real()   = (r7 * cos(11 * M_PI / 32.0));
                    m_256apsk[167].imag()   = (r7 * sin(11 * M_PI / 32.0));
                    m_256apsk[168].real()   = (r7 * cos(31 * M_PI / 32.0));
                    m_256apsk[168].imag()   = (r7 * sin(31 * M_PI / 32.0));
                    m_256apsk[169].real()   = (r7 * cos(29 * M_PI / 32.0));
                    m_256apsk[169].imag()   = (r7 * sin(29 * M_PI / 32.0));
                    m_256apsk[170].real()   = (r7 * cos(25 * M_PI / 32.0));
                    m_256apsk[170].imag()   = (r7 * sin(25 * M_PI / 32.0));
                    m_256apsk[171].real()   = (r7 * cos(27 * M_PI / 32.0));
                    m_256apsk[171].imag()   = (r7 * sin(27 * M_PI / 32.0));
                    m_256apsk[172].real()   = (r7 * cos(17 * M_PI / 32.0));
                    m_256apsk[172].imag()   = (r7 * sin(17 * M_PI / 32.0));
                    m_256apsk[173].real()   = (r7 * cos(19 * M_PI / 32.0));
                    m_256apsk[173].imag()   = (r7 * sin(19 * M_PI / 32.0));
                    m_256apsk[174].real()   = (r7 * cos(23 * M_PI / 32.0));
                    m_256apsk[174].imag()   = (r7 * sin(23 * M_PI / 32.0));
                    m_256apsk[175].real()   = (r7 * cos(21 * M_PI / 32.0));
                    m_256apsk[175].imag()   = (r7 * sin(21 * M_PI / 32.0));
                    m_256apsk[176].real()   = (r7 * cos(-1 * M_PI / 32.0));
                    m_256apsk[176].imag()   = (r7 * sin(-1 * M_PI / 32.0));
                    m_256apsk[177].real()   = (r7 * cos(-3 * M_PI / 32.0));
                    m_256apsk[177].imag()   = (r7 * sin(-3 * M_PI / 32.0));
                    m_256apsk[178].real()   = (r7 * cos(-7 * M_PI / 32.0));
                    m_256apsk[178].imag()   = (r7 * sin(-7 * M_PI / 32.0));
                    m_256apsk[179].real()   = (r7 * cos(-5 * M_PI / 32.0));
                    m_256apsk[179].imag()   = (r7 * sin(-5 * M_PI / 32.0));
                    m_256apsk[180].real()   = (r7 * cos(-15 * M_PI / 32.0));
                    m_256apsk[180].imag()   = (r7 * sin(-15 * M_PI / 32.0));
                    m_256apsk[181].real()   = (r7 * cos(-13 * M_PI / 32.0));
                    m_256apsk[181].imag()   = (r7 * sin(-13 * M_PI / 32.0));
                    m_256apsk[182].real()   = (r7 * cos(-9 * M_PI / 32.0));
                    m_256apsk[182].imag()   = (r7 * sin(-9 * M_PI / 32.0));
                    m_256apsk[183].real()   = (r7 * cos(-11 * M_PI / 32.0));
                    m_256apsk[183].imag()   = (r7 * sin(-11 * M_PI / 32.0));
                    m_256apsk[184].real()   = (r7 * cos(33 * M_PI / 32.0));
                    m_256apsk[184].imag()   = (r7 * sin(33 * M_PI / 32.0));
                    m_256apsk[185].real()   = (r7 * cos(35 * M_PI / 32.0));
                    m_256apsk[185].imag()   = (r7 * sin(35 * M_PI / 32.0));
                    m_256apsk[186].real()   = (r7 * cos(39 * M_PI / 32.0));
                    m_256apsk[186].imag()   = (r7 * sin(39 * M_PI / 32.0));
                    m_256apsk[187].real()   = (r7 * cos(37 * M_PI / 32.0));
                    m_256apsk[187].imag()   = (r7 * sin(37 * M_PI / 32.0));
                    m_256apsk[188].real()   = (r7 * cos(47 * M_PI / 32.0));
                    m_256apsk[188].imag()   = (r7 * sin(47 * M_PI / 32.0));
                    m_256apsk[189].real()   = (r7 * cos(45 * M_PI / 32.0));
                    m_256apsk[189].imag()   = (r7 * sin(45 * M_PI / 32.0));
                    m_256apsk[190].real()   = (r7 * cos(41 * M_PI / 32.0));
                    m_256apsk[190].imag()   = (r7 * sin(41 * M_PI / 32.0));
                    m_256apsk[191].real()   = (r7 * cos(43 * M_PI / 32.0));
                    m_256apsk[191].imag()   = (r7 * sin(43 * M_PI / 32.0));
                    m_256apsk[192].real()   = (r5 * cos(M_PI / 32.0));
                    m_256apsk[192].imag()   = (r5 * sin(M_PI / 32.0));
                    m_256apsk[193].real()   = (r5 * cos(3 * M_PI / 32.0));
                    m_256apsk[193].imag()   = (r5 * sin(3 * M_PI / 32.0));
                    m_256apsk[194].real()   = (r5 * cos(7 * M_PI / 32.0));
                    m_256apsk[194].imag()   = (r5 * sin(7 * M_PI / 32.0));
                    m_256apsk[195].real()   = (r5 * cos(5 * M_PI / 32.0));
                    m_256apsk[195].imag()   = (r5 * sin(5 * M_PI / 32.0));
                    m_256apsk[196].real()   = (r5 * cos(15 * M_PI / 32.0));
                    m_256apsk[196].imag()   = (r5 * sin(15 * M_PI / 32.0));
                    m_256apsk[197].real()   = (r5 * cos(13 * M_PI / 32.0));
                    m_256apsk[197].imag()   = (r5 * sin(13 * M_PI / 32.0));
                    m_256apsk[198].real()   = (r5 * cos(9 * M_PI / 32.0));
                    m_256apsk[198].imag()   = (r5 * sin(9 * M_PI / 32.0));
                    m_256apsk[199].real()   = (r5 * cos(11 * M_PI / 32.0));
                    m_256apsk[199].imag()   = (r5 * sin(11 * M_PI / 32.0));
                    m_256apsk[200].real()   = (r5 * cos(31 * M_PI / 32.0));
                    m_256apsk[200].imag()   = (r5 * sin(31 * M_PI / 32.0));
                    m_256apsk[201].real()   = (r5 * cos(29 * M_PI / 32.0));
                    m_256apsk[201].imag()   = (r5 * sin(29 * M_PI / 32.0));
                    m_256apsk[202].real()   = (r5 * cos(25 * M_PI / 32.0));
                    m_256apsk[202].imag()   = (r5 * sin(25 * M_PI / 32.0));
                    m_256apsk[203].real()   = (r5 * cos(27 * M_PI / 32.0));
                    m_256apsk[203].imag()   = (r5 * sin(27 * M_PI / 32.0));
                    m_256apsk[204].real()   = (r5 * cos(17 * M_PI / 32.0));
                    m_256apsk[204].imag()   = (r5 * sin(17 * M_PI / 32.0));
                    m_256apsk[205].real()   = (r5 * cos(19 * M_PI / 32.0));
                    m_256apsk[205].imag()   = (r5 * sin(19 * M_PI / 32.0));
                    m_256apsk[206].real()   = (r5 * cos(23 * M_PI / 32.0));
                    m_256apsk[206].imag()   = (r5 * sin(23 * M_PI / 32.0));
                    m_256apsk[207].real()   = (r5 * cos(21 * M_PI / 32.0));
                    m_256apsk[207].imag()   = (r5 * sin(21 * M_PI / 32.0));
                    m_256apsk[208].real()   = (r5 * cos(-1 * M_PI / 32.0));
                    m_256apsk[208].imag()   = (r5 * sin(-1 * M_PI / 32.0));
                    m_256apsk[209].real()   = (r5 * cos(-3 * M_PI / 32.0));
                    m_256apsk[209].imag()   = (r5 * sin(-3 * M_PI / 32.0));
                    m_256apsk[210].real()   = (r5 * cos(-7 * M_PI / 32.0));
                    m_256apsk[210].imag()   = (r5 * sin(-7 * M_PI / 32.0));
                    m_256apsk[211].real()   = (r5 * cos(-5 * M_PI / 32.0));
                    m_256apsk[211].imag()   = (r5 * sin(-5 * M_PI / 32.0));
                    m_256apsk[212].real()   = (r5 * cos(-15 * M_PI / 32.0));
                    m_256apsk[212].imag()   = (r5 * sin(-15 * M_PI / 32.0));
                    m_256apsk[213].real()   = (r5 * cos(-13 * M_PI / 32.0));
                    m_256apsk[213].imag()   = (r5 * sin(-13 * M_PI / 32.0));
                    m_256apsk[214].real()   = (r5 * cos(-9 * M_PI / 32.0));
                    m_256apsk[214].imag()   = (r5 * sin(-9 * M_PI / 32.0));
                    m_256apsk[215].real()   = (r5 * cos(-11 * M_PI / 32.0));
                    m_256apsk[215].imag()   = (r5 * sin(-11 * M_PI / 32.0));
                    m_256apsk[216].real()   = (r5 * cos(33 * M_PI / 32.0));
                    m_256apsk[216].imag()   = (r5 * sin(33 * M_PI / 32.0));
                    m_256apsk[217].real()   = (r5 * cos(35 * M_PI / 32.0));
                    m_256apsk[217].imag()   = (r5 * sin(35 * M_PI / 32.0));
                    m_256apsk[218].real()   = (r5 * cos(39 * M_PI / 32.0));
                    m_256apsk[218].imag()   = (r5 * sin(39 * M_PI / 32.0));
                    m_256apsk[219].real()   = (r5 * cos(37 * M_PI / 32.0));
                    m_256apsk[219].imag()   = (r5 * sin(37 * M_PI / 32.0));
                    m_256apsk[220].real()   = (r5 * cos(47 * M_PI / 32.0));
                    m_256apsk[220].imag()   = (r5 * sin(47 * M_PI / 32.0));
                    m_256apsk[221].real()   = (r5 * cos(45 * M_PI / 32.0));
                    m_256apsk[221].imag()   = (r5 * sin(45 * M_PI / 32.0));
                    m_256apsk[222].real()   = (r5 * cos(41 * M_PI / 32.0));
                    m_256apsk[222].imag()   = (r5 * sin(41 * M_PI / 32.0));
                    m_256apsk[223].real()   = (r5 * cos(43 * M_PI / 32.0));
                    m_256apsk[223].imag()   = (r5 * sin(43 * M_PI / 32.0));
                    m_256apsk[224].real()   = (r6 * cos(M_PI / 32.0));
                    m_256apsk[224].imag()   = (r6 * sin(M_PI / 32.0));
                    m_256apsk[225].real()   = (r6 * cos(3 * M_PI / 32.0));
                    m_256apsk[225].imag()   = (r6 * sin(3 * M_PI / 32.0));
                    m_256apsk[226].real()   = (r6 * cos(7 * M_PI / 32.0));
                    m_256apsk[226].imag()   = (r6 * sin(7 * M_PI / 32.0));
                    m_256apsk[227].real()   = (r6 * cos(5 * M_PI / 32.0));
                    m_256apsk[227].imag()   = (r6 * sin(5 * M_PI / 32.0));
                    m_256apsk[228].real()   = (r6 * cos(15 * M_PI / 32.0));
                    m_256apsk[228].imag()   = (r6 * sin(15 * M_PI / 32.0));
                    m_256apsk[229].real()   = (r6 * cos(13 * M_PI / 32.0));
                    m_256apsk[229].imag()   = (r6 * sin(13 * M_PI / 32.0));
                    m_256apsk[230].real()   = (r6 * cos(9 * M_PI / 32.0));
                    m_256apsk[230].imag()   = (r6 * sin(9 * M_PI / 32.0));
                    m_256apsk[231].real()   = (r6 * cos(11 * M_PI / 32.0));
                    m_256apsk[231].imag()   = (r6 * sin(11 * M_PI / 32.0));
                    m_256apsk[232].real()   = (r6 * cos(31 * M_PI / 32.0));
                    m_256apsk[232].imag()   = (r6 * sin(31 * M_PI / 32.0));
                    m_256apsk[233].real()   = (r6 * cos(29 * M_PI / 32.0));
                    m_256apsk[233].imag()   = (r6 * sin(29 * M_PI / 32.0));
                    m_256apsk[234].real()   = (r6 * cos(25 * M_PI / 32.0));
                    m_256apsk[234].imag()   = (r6 * sin(25 * M_PI / 32.0));
                    m_256apsk[235].real()   = (r6 * cos(27 * M_PI / 32.0));
                    m_256apsk[235].imag()   = (r6 * sin(27 * M_PI / 32.0));
                    m_256apsk[236].real()   = (r6 * cos(17 * M_PI / 32.0));
                    m_256apsk[236].imag()   = (r6 * sin(17 * M_PI / 32.0));
                    m_256apsk[237].real()   = (r6 * cos(19 * M_PI / 32.0));
                    m_256apsk[237].imag()   = (r6 * sin(19 * M_PI / 32.0));
                    m_256apsk[238].real()   = (r6 * cos(23 * M_PI / 32.0));
                    m_256apsk[238].imag()   = (r6 * sin(23 * M_PI / 32.0));
                    m_256apsk[239].real()   = (r6 * cos(21 * M_PI / 32.0));
                    m_256apsk[239].imag()   = (r6 * sin(21 * M_PI / 32.0));
                    m_256apsk[240].real()   = (r6 * cos(-1 * M_PI / 32.0));
                    m_256apsk[240].imag()   = (r6 * sin(-1 * M_PI / 32.0));
                    m_256apsk[241].real()   = (r6 * cos(-3 * M_PI / 32.0));
                    m_256apsk[241].imag()   = (r6 * sin(-3 * M_PI / 32.0));
                    m_256apsk[242].real()   = (r6 * cos(-7 * M_PI / 32.0));
                    m_256apsk[242].imag()   = (r6 * sin(-7 * M_PI / 32.0));
                    m_256apsk[243].real()   = (r6 * cos(-5 * M_PI / 32.0));
                    m_256apsk[243].imag()   = (r6 * sin(-5 * M_PI / 32.0));
                    m_256apsk[244].real()   = (r6 * cos(-15 * M_PI / 32.0));
                    m_256apsk[244].imag()   = (r6 * sin(-15 * M_PI / 32.0));
                    m_256apsk[245].real()   = (r6 * cos(-13 * M_PI / 32.0));
                    m_256apsk[245].imag()   = (r6 * sin(-13 * M_PI / 32.0));
                    m_256apsk[246].real()   = (r6 * cos(-9 * M_PI / 32.0));
                    m_256apsk[246].imag()   = (r6 * sin(-9 * M_PI / 32.0));
                    m_256apsk[247].real()   = (r6 * cos(-11 * M_PI / 32.0));
                    m_256apsk[247].imag()   = (r6 * sin(-11 * M_PI / 32.0));
                    m_256apsk[248].real()   = (r6 * cos(33 * M_PI / 32.0));
                    m_256apsk[248].imag()   = (r6 * sin(33 * M_PI / 32.0));
                    m_256apsk[249].real()   = (r6 * cos(35 * M_PI / 32.0));
                    m_256apsk[249].imag()   = (r6 * sin(35 * M_PI / 32.0));
                    m_256apsk[250].real()   = (r6 * cos(39 * M_PI / 32.0));
                    m_256apsk[250].imag()   = (r6 * sin(39 * M_PI / 32.0));
                    m_256apsk[251].real()   = (r6 * cos(37 * M_PI / 32.0));
                    m_256apsk[251].imag()   = (r6 * sin(37 * M_PI / 32.0));
                    m_256apsk[252].real()   = (r6 * cos(47 * M_PI / 32.0));
                    m_256apsk[252].imag()   = (r6 * sin(47 * M_PI / 32.0));
                    m_256apsk[253].real()   = (r6 * cos(45 * M_PI / 32.0));
                    m_256apsk[253].imag()   = (r6 * sin(45 * M_PI / 32.0));
                    m_256apsk[254].real()   = (r6 * cos(41 * M_PI / 32.0));
                    m_256apsk[254].imag()   = (r6 * sin(41 * M_PI / 32.0));
                    m_256apsk[255].real()   = (r6 * cos(43 * M_PI / 32.0));
                    m_256apsk[255].imag()   = (r6 * sin(43 * M_PI / 32.0));
                }
                break;
            default:
                m_qpsk[0].real() = (r1 * cos(M_PI / 4.0));
                m_qpsk[0].imag() = (r1 * sin(M_PI / 4.0));
                m_qpsk[1].real() = (r1 * cos(7 * M_PI / 4.0));
                m_qpsk[1].imag() = (r1 * sin(7 * M_PI / 4.0));
                m_qpsk[2].real() = (r1 * cos(3 * M_PI / 4.0));
                m_qpsk[2].imag() = (r1 * sin(3 * M_PI / 4.0));
                m_qpsk[3].real() = (r1 * cos(5 * M_PI / 4.0));
                m_qpsk[3].imag() = (r1 * sin(5 * M_PI / 4.0));
                break;
        }
        signal_constellation = constellation;
        set_output_multiple(2);
    }

    /*
     * Our virtual destructor.
     */
    modulator_bc_impl::~modulator_bc_impl()
    {
    }

    void
    modulator_bc_impl::forecast (int noutput_items, gr_vector_int &ninput_items_required)
    {
        ninput_items_required[0] = noutput_items;
    }

    int
    modulator_bc_impl::general_work (int noutput_items,
                       gr_vector_int &ninput_items,
                       gr_vector_const_void_star &input_items,
                       gr_vector_void_star &output_items)
    {
        const unsigned char *in = (const unsigned char *) input_items[0];
        gr_complex *out = (gr_complex *) output_items[0];
        int index;

        switch (signal_constellation)
        {
            case gr::dvbs2::MOD_QPSK:
                for (int i = 0; i < noutput_items; i++)
                {
                    index = *in++;
                    *out++ = m_qpsk[index & 0x3];
                }
                break;
            case gr::dvbs2::MOD_8PSK:
            case gr::dvbs2::MOD_8APSK:
                for (int i = 0; i < noutput_items; i++)
                {
                    index = *in++;
                    *out++ = m_8psk[index & 0x7];
                }
                break;
            case gr::dvbs2::MOD_16APSK:
            case gr::dvbs2::MOD_8_8APSK:
                for (int i = 0; i < noutput_items; i++)
                {
                    index = *in++;
                    *out++ = m_16apsk[index & 0xf];
                }
                break;
            case gr::dvbs2::MOD_32APSK:
            case gr::dvbs2::MOD_4_12_16APSK:
            case gr::dvbs2::MOD_4_8_4_16APSK:
                for (int i = 0; i < noutput_items; i++)
                {
                    index = *in++;
                    *out++ = m_32apsk[index & 0x1f];
                }
                break;
            case gr::dvbs2::MOD_64APSK:
            case gr::dvbs2::MOD_8_16_20_20APSK:
            case gr::dvbs2::MOD_4_12_20_28APSK:
                for (int i = 0; i < noutput_items; i++)
                {
                    index = *in++;
                    *out++ = m_64apsk[index & 0x3f];
                }
                break;
            case gr::dvbs2::MOD_128APSK:
                for (int i = 0; i < noutput_items; i++)
                {
                    index = *in++;
                    *out++ = m_128apsk[index & 0x7f];
                }
                break;
            case gr::dvbs2::MOD_256APSK:
                for (int i = 0; i < noutput_items; i++)
                {
                    index = *in++;
                    *out++ = m_256apsk[index & 0xff];
                }
                break;
            default:
                for (int i = 0; i < noutput_items; i++)
                {
                    index = *in++;
                    *out++ = m_qpsk[index & 0x3];
                }
                break;
        }

        // Tell runtime system how many input items we consumed on
        // each input stream.
        consume_each (noutput_items);

        // Tell runtime system how many output items we produced.
        return noutput_items;
    }

  } /* namespace dvbs2 */
} /* namespace gr */

