program uv_stat
  use gildas_def
  use gkernel_interfaces
  !---------------------------------------------------------------------
  ! TASK  Analyse a UV data set to define approximate beam size,
  !       field of view, expected "best beam", etc...
  ! Input :
  !     a precessed UV table
  ! Output :
  !     informations and optionally a beam image ?
  !---------------------------------------------------------------------
  !
  character(len=filename_length) :: uv_table
  character(len=16) :: mode
  real :: uv_taper(3),uniform(2)
  integer :: wcol
  logical :: error, do_beam
  !
  call gildas_open
  call gildas_char('UV_TABLE$',uv_table)
  call gildas_char('UV_MODE$',mode)
  call gildas_real('UV_TAPER$',uv_taper,3)
  call gildas_real('UV_CELL$',uniform,2)
  call gildas_inte('WCOL$',wcol,1)
  call gildas_logi('DO_BEAMS$',do_beam,1)
  call gildas_close
  call sic_upper(mode)
  !
  call sub_uv_stat (uv_table,mode,uv_taper,uniform,wcol,do_beam,error)
  if (error) call sysexi(fatale)
  !
  contains
!  
subroutine sub_uv_stat (uv_table,mode,uv_taper,uniform,wcol,do_beam,error)
  use gkernel_interfaces
  use mapping_interfaces
  use image_def
  use gbl_message
  !---------------------------------------------------------------------
  ! TASK  Analyse a UV data set to define approximate beam size,
  !       field of view, expected "best beam", etc...
  ! Input :
  !     a precessed UV table
  ! Output :
  !     a precessed, rotated, shifted UV table, sorted in V,
  !     ordered in (U,V,W,D,T,iant,jant,nchan(real,imag,weig))
  !     a beam image ?
  ! Work space :
  !	by GET_VM
  !---------------------------------------------------------------------
  character(len=*), intent(in) :: uv_table
  character(len=*), intent(inout) :: mode
  real, intent(inout) :: uv_taper(3),uniform(2)
  integer, intent(inout) :: wcol
  logical, intent(in) :: do_beam
  logical, intent(out) :: error
  ! Global
  real(8), parameter :: pi=3.14159265358979323846d0
  real(8), parameter :: f_to_k=2.d0*pi/299792458.d-6
  !
  type(gildas) :: huv, hall
  real, allocatable :: duv(:,:), sduv(:,:)
  real, allocatable :: beams(:,:), fft(:)
  !
  real :: map_cell(2),uvmax,uvmin
  real :: start,step,taper,result(10,8)
  integer :: map_size(2),mcol(2), n, kuv, nc(2)
  logical :: shift, more
  real(8) :: freq
  real :: uvm, uvma, uvmi
  integer :: muv,nuv,ier
  real(8) :: xy(2), cs(2)
  logical :: sort_data
  real :: cpu1, cpu0
  character(len=*), parameter :: rname='UV_STAT'
  character(len=80) :: chain
  !
  shift = .false.
  call gildas_null(hall,type='UVT')
  call gdf_read_gildas (hall, uv_table, '.uvt', error, data=.false.)
  if (error) then
    call map_message(seve%e,rname,'Cannot read input table')
    return
  endif
  !
  if (wcol.eq.0) then
    wcol = max(1,hall%gil%nchan/3)
  else if (wcol.lt.0 .or. wcol.gt.hall%gil%nchan) then
    call map_message(seve%e,rname,'WCOL out of range ')
    error = .true.
    return
  endif
  !
  ! Read only the WCOL channel whole data
  nc = [wcol,wcol]
  call gildas_null(huv,type='UVT')
  call gdf_copy_header(hall, huv, error)
  huv%gil%nchan = nc(2)-nc(1)+1
  huv%gil%ref(1) = hall%gil%ref(1)-nc(1)+1
  huv%gil%dim(1) = huv%gil%nlead + huv%gil%natom*huv%gil%nchan + huv%gil%ntrail
  !
  ! Read the subset
  muv = huv%gil%dim(1)
  nuv = huv%gil%dim(2)
  huv%blc = 0  
  huv%trc = 0  
  allocate (duv(muv,nuv),sduv(muv,nuv),stat=huv%status)
  if (gildas_error(huv,rname,error)) return
  allocate (huv%r2d(huv%gil%dim(1), huv%gil%dim(2)), stat=ier)
  if (ier.ne.0) then
    call map_message(seve%e,rname,'WCOL out of range ')
    error = .true.
    return
  endif 
  call gdf_read_uvdataset(hall,huv,nc,duv,error)
  !
  !! Print *,'Done reading UV'
  !
  ! Apply phase shift and copy to output visibilities
  xy = 0.
  cs = (/1.,0./)
  sort_data = .false.
  call t_uvsort (muv,nuv,duv,sduv,xy,cs,uvmax,uvmin,sort_data,error)
  deallocate (duv)
  !
  freq = huv%gil%val(1)+huv%gil%fres*(huv%gil%val(1)/huv%gil%freq)*   &
     &    (huv%gil%nchan+1)*0.5
  !
  ! Print header information
  uvma = uvmax
  uvmi = uvmin
  uvmin = uvmi*(freq*f_to_k)
  uvmax = uvma*(freq*f_to_k)
  write(6,100) huv%gil%dim(2),huv%gil%nchan 
  write(6,101) uvmi,uvma,' meters'
  write(6,101) uvmin*1e-3,uvmax*1e-3,' kiloWavelength'
  map_cell(1) = 0.02*nint(180.0*3600.0*50.0/uvmax/4.0)
  if (map_cell(1).le.0.02) then
    map_cell(1) = 0.002*nint(180.0*3600.0*500.0/uvmax/4.0)
    if (map_cell(1).le.0.002) then
      map_cell(1) = 0.0002*nint(180.0*3600.0*5000.0/uvmax/4.0)
    endif
  endif
  map_cell(2) = map_cell(1)    ! In ", rounded to 0.1"
  map_size(2) = nint(8.0*uvmax/uvmin)
  map_size(1) = 16
  more = .true.
  do while (more)
    if (map_size(1).lt.map_size(2)) then
      map_size(1) = 2*map_size(1)
    else
      more = .false.
    endif
  enddo
  map_size(2) = map_size(1)
  write(6,103) map_size
  write(6,104) map_cell
  write(6,105) map_cell(1)*float(map_size(1)),   &
     &    map_cell(2)*float(map_size(2))
  mcol = 0
  !
  ! Header mode: stop
  uvm = uvmax/(freq*f_to_k)
  if (mode.eq.'HEADER') then
    return
  elseif  (mode.eq.'TAPER') then
    step = sqrt(2.0)
    start = 10*nint(uvm/160.0)
    taper = 16*start
  elseif  (mode.eq.'WEIGHT') then
    step = 0.0
    start = 0.0
    taper = sqrt(uv_taper(1)*uv_taper(2))
    if (taper.ne.0) then
      taper = min(taper,uvm)
    else
      taper = uvm
    endif
  endif
  call map_message(seve%i,rname,'Computing beams')
  write(6,102) uvm,uvmax*1e-3
  !
  ! Define MAP_CELL and MAP_SIZE
  taper = uvm*freq*f_to_k
  map_cell(1) = 0.02*nint(180.0*3600.0*50.0/taper/4.0)
  if (map_cell(1).le.0.02) then
    map_cell(1) = 0.002*nint(180.0*3600.0*500.0/taper/4.0)
  endif
  map_cell(2) = map_cell(1)    ! In ", rounded to 0.01"
  if (mode.eq.'TAPER') then
    map_size(1) = 256
    map_size(2) = 256
  else
    map_size(1) = 64
    map_size(2) = 64
  endif
  write(6,103) map_size
  write(6,104) map_cell
  !
  ! Redefine some parameters
  map_cell(1) = map_cell(1)*pi/180.0/3600.0   ! In radians
  map_cell(2) = map_cell(2)*pi/180.0/3600.0   ! In radians
  ! Process sorted UV Table according to the type of beam produced
  n = 2*max(map_size(1),map_size(2))
  allocate (beams(map_size(1),map_size(2)),fft(n),stat=ier)
  if (ier.ne.0) then
    call map_message(seve%e,rname,'Memory allocation failure')
    error = .true.
    return
  endif
  !
  call uniform_beam (' ',uv_taper,   &
     &    map_size,map_cell,uniform,wcol,mcol,fft,   &
     &    error,mode,beams,1,start,step,uvm,result,huv,sduv)
  deallocate (beams,fft)
  !
  call map_message(seve%i,rname,'Successful completion')
  return
  !
  100   format('I-UV_STAT,  Found ',i12,' Visibilities, ',i4,' channels')
  101   format('I-UV_STAT,  Baselines ',f9.1,' - ',f9.1,a)
  102   format('I-UV_STAT,  Maximum baseline is   ',f8.1,' m,  ',   &
     &    f8.1,' kWavelength')
  103   format('I-UV_STAT,  Map size is   ',i4,' by ',i4)
  104   format('I-UV_STAT,  Pixel size is ',f7.3,' by ',f7.3,'"')
  105   format('I-UV_STAT,  Image size is ',f8.2,' by ',f8.2,'"')
end subroutine sub_uv_stat
!
include 'lib/util_task.f90'
!
end program uv_stat
