subroutine atm_init(error)
  use gbl_message
  use gkernel_interfaces
  use atm_params
  use ast_astro
  !---------------------------------------------------------------------
  ! @ private
  ! ATM
  !
  ! One time instantiation of the ASTRO variables which are used (as
  ! input and output) by the command ASTRO\ATM. These variables and
  ! their startup values are not needed if the programmer does want
  ! to make use of the command ASTRO\ATM.
  !---------------------------------------------------------------------
  logical, intent(inout) :: error   ! Logical error flag
  ! Local
  character(len=*), parameter :: rname='ATM_INIT'
  !
  if (atm_initdone) return
  !
  call atm_sicvariables(error)
  if (error) then
    call astro_message(seve%e,rname,'Error creating ATM% variables')
    return
  endif
  !
  ! Default values of input parameters
  t0 = 273.0
  p0 = 1013.0
  water = 1.0
  airmass = 1.414
  feff = 0.95
  freqs = 110.0
  freqi = 113.0
  gim = 1.0
  trec = 60
  !
  ! Input parameters
  call sic_def_real ('TEMPERATURE',t0,0,1,.false.,error)
  if (error) return
  call sic_def_real ('TRUE_PRESSURE',p1,0,1,.true.,error)
  if (error) return
  call sic_def_real ('ZERO_PRESSURE',p0,0,1,.false.,error)
  if (error) return
  call sic_def_dble ('ALTITUDE',altitude,0,1,.true.,error)
  if (error) return
  call sic_def_real ('WATER',water,0,1,.false.,error)
  !! if (error) return
  call sic_def_real ('AIRMASS',airmass,0,1,.false.,error)
  if (error) return
  call sic_def_real ('FORWARD_EFF',feff,0,1,.false.,error)
  !! if (error) return
  call sic_def_real ('FREQ_SIG',freqs,0,1,.false.,error)
  if (error) return
  call sic_def_real ('FREQ_IMA',freqi,0,1,.false.,error)
  if (error) return
  call sic_def_real ('GAIN_IMAGE',gim,0,1,.false.,error)
  !! if (error) return
  call sic_def_real ('TREC',trec,0,1,.false.,error)
  if (error) return
  !
  ! Output parameters
  call sic_def_real ('TSYS',tsys,0,1,.true.,error)
  !! if (error) return
  call sic_def_real ('TAU_O2',tauox,0,1,.true.,error)
  if (error) return
  call sic_def_real ('TAU_H2O',tauw,0,1,.true.,error)
  if (error) return
  call sic_def_real ('TAU_TOT',taut,0,1,.true.,error)
  if (error) return
  call sic_def_real ('EMIS_SIG',temis,0,1,.true.,error)
  if (error) return
  call sic_def_real ('EMIS_IMA',temii,0,1,.true.,error)
  if (error) return
  call sic_def_real ('PATH_SIG',paths,0,1,.true.,error)
  if (error) return
  call sic_def_real ('PATH_IMA',pathi,0,1,.true.,error)
  if (error) return
  call sic_def_real ('ATM_SIG',tatms,0,1,.true.,error)
  if (error) return
  call sic_def_real ('ATM_IMA',tatmi,0,1,.true.,error)
  if (error) return
  call sic_def_real ('TANT',tant,0,1,.true.,error)
  if (error) return
  !
  call astro_message(seve%d,rname,'ATM-ASTRO interface initialized')
  atm_initdone = .true.
  !
end subroutine atm_init
!
subroutine astro_atm(line,error)
  use gildas_def
  use gkernel_interfaces
  use atm_params
  use ast_astro
  !---------------------------------------------------------------------
  ! @ private
  ! ATM
  !
  ! Purpose: compute atmospheric properties
  !     from a full set of SIC variables
  !---------------------------------------------------------------------
  character(len=*), intent(in)    :: line   ! Input command line
  logical,          intent(inout) :: error  ! Logical error flag
  ! Local
  integer(kind=4) :: ier,nfile,nf
  real(kind=4) :: frequ,fmin,fmax
  character(len=filename_length) :: file
  integer(kind=4), parameter :: mact=3
  integer(kind=4) :: nact,nname
  character(len=16) :: actions(mact),action,name
  logical, save :: first
  ! Data
  data actions/'INTERPOLATE','MAKE','SAVE'/
  data first /.true./
  !
  ! Optionally print ATM status before the actions below
  if (sic_present(1,0))  call astro_atm_print(error)
  !
  h0 = altitude
  if (.not.sic_present(0,1)) then
    !
    p1 = p0*2.0**(-h0/5.5)     ! Pressure at altitude H0
    call atm_atmosp(t0,p1,h0)
    !
    frequ = freqi
    call atm_transm(water,airmass,frequ,temii,tatmi,tauox,tauw,taut,ier)
    call atm_path(water,airmass,frequ,pathi,ier)
    pathi = pathi*10           ! -> mm
    !
    frequ = freqs
    call atm_transm(water,airmass,frequ,temis,tatms,tauox,tauw,taut,ier)
    call atm_path(water,airmass,frequ,paths,ier)
    paths = paths*10           ! -> mm
    !
    tant = (gim * (feff*temii+(1.0-feff)*t0 + trec) +  &
           (feff*temis+(1.0-feff)*t0 + trec) ) / (1.0+gim)
    tsys = exp(taut*airmass) * (gim * (feff*temii+(1.0-feff)*t0 + trec) +  &
           (feff*temis+(1.0-feff)*t0 + trec) ) / feff
    !
  else
    ! Read binary file once.
    if (first) then
      call atm_i(error)
      first = .false.
    endif
    ! Retrieve and analyze first argument
    call sic_ke(line,0,1,name,nname,.true.,error)
    if (error) return
    call sic_ambigs('ATMOSPHERE',name(1:nname),action,nact,actions,mact,error)
    if (error) return
    !
    if (action.eq.'INTERPOLATE') then
      ! Interpolate from file:
      call atm_atmosp_i(t0,p1,h0)
      !
      frequ = freqi
      call atm_transm_i(water,airmass,frequ,temii,tatmi,tauox,tauw,taut,ier)
      call atm_path_i(water,airmass,frequ,pathi,ier)
      pathi = pathi*10         ! -> mm
      !
      frequ = freqs
      call atm_transm_i(water,airmass,frequ,temis,tatms,tauox,tauw,taut,ier)
      call atm_path_i(water,airmass,frequ,paths,ier)
      paths = paths*10         ! -> mm
      !
      tant = (gim * (feff*temii+(1.0-feff)*t0 + trec) +  &
             (feff*temis+(1.0-feff)*t0 + trec) ) / (1.0+gim)
      tsys = exp(taut*airmass) * (gim * (feff*temii+(1.0-feff)*t0 + trec) +  &
             (feff*temis+(1.0-feff)*t0 + trec) ) / feff
      !
    elseif (action.eq.'MAKE') then
      ! Make a file: ATM Make filename nf fmin fmax
      file = ' '
      call sic_ch(line,0,2,file,nfile,.true.,error)
      if (error) return
      nf = 200
      fmin = 75.
      fmax = 300.
      call sic_i4(line,0,3,nf,.false.,error)
      if (error) return
      call sic_r4(line,0,4,fmin,.false.,error)
      if (error) return
      call sic_r4(line,0,5,fmax,.false.,error)
      if (error) return
      call atmos_i_table(.true.,file,nfile,nf,fmin,fmax,h0,error)
      !
    elseif (action.eq.'SAVE') then
      ! Save on file previously loaded table: ATM Save filename
      ! This one may be used to save with the current machine byte order a
      ! table previously loaded whose byte order was different.
      file = ' '
      call sic_ch(line,0,2,file,nfile,.true.,error)
      if (error) return
      nf = 200
      fmin = 75.
      fmax = 300.
      call atmos_i_table(.false.,file,nfile,nf,fmin,fmax,h0,error)
      !
    endif
  endif
  !
end subroutine astro_atm
!
subroutine astro_atm_print(error)
  use gbl_message
  use atm_params
  use ast_astro
  !---------------------------------------------------------------------
  ! @ private
  ! ATM
  !
  ! Support routine for command:
  !   ASTRO\ATM /PRINT
  !---------------------------------------------------------------------
  logical, intent(inout) :: error   ! Logical error flag
  ! Local
  character(len=*), parameter :: rname='ATM_PRINT'
  character(len=message_length) :: mess
  !
  call astro_message(seve%r,rname,'Current status of ATM is')
  !
  ! Print ATM internal parameters
  call atm_print(error)
  !
  ! Print ASTRO_ATM internal parameters
  write(mess,101) 'TEMPERATURE',t0,'[K]','Ground temperature'
  call astro_message(seve%r,rname,mess)
  !
  write(mess,101) 'ZERO_PRESSURE',p0,'[hPa]','Pressure at sea level'
  call astro_message(seve%r,rname,mess)
  !
  write(mess,101) 'ALTITUDE',altitude,'[km]','Altitude of the observatory'
  call astro_message(seve%r,rname,mess)
  !
  write(mess,101) 'WATER',water,'[mm]','Precipitable water vapor'
  call astro_message(seve%r,rname,mess)
  !
  write(mess,101) 'AIRMASS',airmass,'[]','Number of airmasses'
  call astro_message(seve%r,rname,mess)
  !
  write(mess,101) 'FORWARD_EFF',feff,'[]','Forward efficiency'
  call astro_message(seve%r,rname,mess)
  !
  write(mess,101) 'FREQ_SIG',freqs,'[GHz]','Signal frequency'
  call astro_message(seve%r,rname,mess)
  !
  write(mess,101) 'FREQ_IMA',freqi,'[GHz]','Image frequency'
  call astro_message(seve%r,rname,mess)
  !
  write(mess,101) 'GAIN_IMAGE',gim,'[]','Gain image'
  call astro_message(seve%r,rname,mess)
  !
  write(mess,101) 'TREC',trec,'[K]','Receiver temperature'
  call astro_message(seve%r,rname,mess)
  !
101 format(A,T23,F12.3,T36,A6,T43,A)
  !
end subroutine astro_atm_print
!
subroutine noema_tsys(line,error)
  use gbl_message
  use gkernel_interfaces
  use astro_interfaces, except_this=>noema_tsys
  !---------------------------------------------------------------------
  ! @ private
  ! Support routine for command
  !   NOEMA\TSYS TableName
  ! Compute the Tsys + Opacity table
  !---------------------------------------------------------------------
  character(len=*), intent(in)    :: line
  logical,          intent(inout) :: error
  ! Local
  character(len=*), parameter :: rname='TSYS'
  character(len=filename_length) :: tsysfile
  integer(kind=4) :: nc
  !
  call sic_ch(line,0,1,tsysfile,nc,.true.,error)
  if (error)  return
  !
  if (gag_inquire(tsysfile,nc).eq.0) then
    call astro_message(seve%e,rname,  &
      'File '//tsysfile(1:nc)//' already exists. Remove it first.')
    error = .true.
    if (error)  return
  endif
  !
  call noema_tsys_table(tsysfile,error)
  if (error)  return
  !
end subroutine noema_tsys
!
subroutine noema_tsys_table(file,error)
  use gbl_message
  use gkernel_interfaces
  use gkernel_types
  use astro_interfaces, except_this=>noema_tsys_table
  use astro_types
  !---------------------------------------------------------------------
  ! @ private
  ! Compute the Tsys + Opacity table
  !---------------------------------------------------------------------
  character(len=*), intent(in)    :: file
  logical,          intent(inout) :: error
  ! Local
  character(len=*), parameter :: rname='TSYS'
  character(len=message_length) :: mess
  integer(kind=4) :: nf,nw,if,iw,it,ier,ib
  real(kind=4) :: p1,temis,temii,tatm,tauox,tauw,dummy
  real(kind=4), allocatable :: f(:),w(:),vtrec(:),vfeff(:)
  real(kind=4), allocatable :: tsys(:,:,:),taut(:,:,:)
  type(time_t) :: time
  type(receiver_t) :: rec
  !
  ! Parameters
  real(kind=4), parameter :: h0=2.560    ! [km] NOEMA altitude
  real(kind=4), parameter :: p0=1013.0   ! [HPa] At sea level
  real(kind=4), parameter :: airmass=1.  ! [Neper]
  real(kind=4), parameter :: gim=5.e-3   ! [   ] Image gain ratio
  real(kind=4), parameter :: intfreq=0.  ! [GHz] Intermediate frequency
  ! IF (used to compute the image frequency) is set to 0 because:
  ! 1) typical values show a Tsys difference near 0% at the center of
  !    the bands (where Tsys is flat), and less than 2% at the band
  !    sides (where Tsys varies faster in one sideband than in the
  !    other).
  ! 2) it is quite difficult/impossible to produce a generic Tsys table
  !    which would know in advance what is the image frequency distance
  !    and side (lower or upper). In other words we can not know what is
  !    the image frequency associated to a given signal frequency.
  !
  ! Frequencies
  integer(kind=4), parameter :: nb=3     ! Number of bands
  real(kind=4),    parameter :: f_step=1.  ! [GHz]
  real(kind=4)               :: f_min(nb)  ! [GHz] Read from 'rec_define_noema'
  real(kind=4)               :: f_max(nb)  ! [GHz] Read from 'rec_define_noema'
  ! PWV
  real(kind=4),    parameter :: w_step=0.5  ! [mm]
  real(kind=4),    parameter :: w_min=2.    ! [mm]
  real(kind=4),    parameter :: w_max=7.    ! [mm]
  ! Temperatures
  integer(kind=4), parameter :: nt=2     ! Number of atmosphere temperatures
  real(kind=4),    parameter :: t(nt) = (/ 273.,283. /)  ! [K]
  ! Trec, Feff
  real(kind=4),    parameter :: trec(nb) = (/ 30.,40.,45. /)  ! [K] Trec value
  real(kind=4),    parameter :: feff(nb) = (/ 0.95, 0.95, 0.90 /)  ! Forward efficiency
  !
  p1 = p0*2.0**(-h0/5.5)  ! [HPa] Pressure at altitude h0
  !
  ! Get receiver parameters
  call rec_define_noema(rec,error)
  if (error)  return
  !
  if (rec%desc%n_rbands.ne.nb) then
    ! If error, adapt Trec and Feff arrays
    call astro_message(seve%e,rname,'Number of bands not supported')
    error = .true.
    return
  endif
  !
  ! Frequency ranges
  do ib=1,rec%desc%n_rbands
    ! Compute our frequency range. Be conservative (floor/ceiling) + align
    ! to the nearest GHz (purely cosmetic)
    f_min(ib) = floor(rec%desc%rflim(1,ib)/1000.)
    f_max(ib) = ceiling(rec%desc%rflim(2,ib)/1000.)
  enddo
  ! Number of frequencies: the whole range is sampled with no gap between
  ! bands. Beware this feature is used by PMS.
  nf = (f_max(rec%desc%n_rbands)-f_min(1))/f_step+1  ! Assume that bands are ordered increasingly
  ! Number of pwv: easy
  nw = (w_max-w_min)/w_step+1
  !
  ! Define the vectors
  allocate(f(nf),w(nw),vtrec(nf),vfeff(nf),stat=ier)
  if (failed_allocate(rname,'dimension buffers',ier,error))  return
  ! Frequencies: regular sampling. Beware this feature is used by PMS.
  do if=1,nf
    f(if) = f_min(1)+(if-1)*f_step
  enddo
  ! PWV
  do iw=1,nw
    w(iw) = w_min+(iw-1)*w_step
  enddo
  ! Vectorized Trec and Feff:
  vtrec(:) = 1e4   ! Blank value: very bad Trec (but valid in computations)
  vfeff(:) = 1e-4  ! Blank value: very bad Feff (but valid in computations)
  do ib=1,rec%desc%n_rbands
    where (f.ge.f_min(ib) .and. f.le.f_max(ib))
      vtrec = trec(ib)
      vfeff = feff(ib)
    end where
  enddo
  !
  call astro_message(seve%i,rname,'Computing Tsys table with')
  write (mess,10)  nf,' frequencies  from ',f(1),' to ',f(nf),' x '
  call astro_message(seve%i,rname,mess)
  write (mess,10)  nw,' pwv          from ',w(1),' to ',w(nw),' x '
  call astro_message(seve%i,rname,mess)
  write (mess,10)  nt,' temperatures from ',t(1),' to ',t(nt)
  call astro_message(seve%i,rname,mess)
10 format(I5,A,F6.2,A,F6.2,A)
  !
  ! Now compute Tsys
  allocate(tsys(nf,nw,nt),taut(nf,nw,nt),stat=ier)
  if (failed_allocate(rname,'tsys buffers',ier,error))  return
  !
  call gtime_init(time,nt*nw*nf,error)
  if (error)  return
  !
  do it=1,nt
    call atm_atmosp(t(it),p1,h0)
    do iw=1,nw
      do if=1,nf
        ! Signal
        call atm_transm(w(iw),airmass,f(if),temis,tatm,tauox,tauw,taut(if,iw,it),ier)
        ! Image
        if (intfreq.eq.0.) then
          temii = temis
        else
          call atm_transm(w(iw),airmass,f(if)+2.*intfreq,temii,tatm,tauox,tauw,dummy,ier)
        endif
        ! Store Tsys at current airmass
        tsys(if,iw,it) = ( gim * (vfeff(if)*temii+(1.0-vfeff(if))*t(it) + vtrec(if)) +  &
                                 (vfeff(if)*temis+(1.0-vfeff(if))*t(it) + vtrec(if))    &
                          ) / vfeff(if) * exp(taut(if,iw,it)*airmass)
        !
        call gtime_current(time)
        if (sic_ctrlc()) then
          call astro_message(seve%e,rname,'Aborted')
          error = .true.
          return
        endif
        !
      enddo
    enddo
  enddo
  !
  call write_bintable(file,error)
  if (error)  return
  !
contains
  subroutine write_bintable(file,error)
    use gildas_def
    use gbl_format
    !-------------------------------------------------------------------
    !
    !-------------------------------------------------------------------
    character(len=*), intent(in)    :: file
    logical,          intent(inout) :: error
    ! Local
    integer(kind=4) :: ier,nfile,lun,currec,recpos
    character(len=message_length) :: mess
    character(len=4) :: tab_code
    integer(kind=4), parameter :: reclen=128  ! Words
    integer(kind=4) :: buffer(reclen)
    !
    ier = sic_getlun(lun)
    nfile = len_trim(file)
    open(unit=lun,file=file(1:nfile),status='NEW',access='DIRECT',   &
        form='UNFORMATTED',iostat=ier,recl=reclen*facunf)
    if (ier.ne.0) then
      call astro_message(seve%e,rname,'Filename: '//file)
      call putios('E-ATM, Open error: ',ier)
      error = .true.
      goto 99
    endif
    !
    call gdf_getcod(tab_code)
    call chtoby(tab_code,buffer(1),4)
    call r4tor4(p1,buffer(2),1)
    call r4tor4(gim,buffer(3),1)
    call r4tor4(intfreq,buffer(4),1)
    call i4toi4(nf,buffer(5),1)
    call i4toi4(nw,buffer(6),1)
    call i4toi4(nt,buffer(7),1)
    currec = 1
    recpos = 8
    !
    call write_my_array(lun,f,nf,buffer,currec,recpos,error)
    if (error)  goto 98
    call write_my_array(lun,w,nw,buffer,currec,recpos,error)
    if (error)  goto 98
    call write_my_array(lun,t,nt,buffer,currec,recpos,error)
    if (error)  goto 98
    call write_my_array(lun,vtrec,nf,buffer,currec,recpos,error)
    if (error)  goto 98
    call write_my_array(lun,vfeff,nf,buffer,currec,recpos,error)
    if (error)  goto 98
    !
    call write_my_array(lun,tsys,nf*nw*nt,buffer,currec,recpos,error)
    if (error)  goto 98
    call write_my_array(lun,taut,nf*nw*nt,buffer,currec,recpos,error)
    if (error)  goto 98
    !
    if (recpos.ne.1) then
      ! Flush last record
      write(lun,rec=currec,iostat=ier)  buffer
      if (ier.ne.0) then
        write(mess,'(A,I0)') 'Error writing record ',currec
        call astro_message(seve%e,rname,mess)
        error = .true.
        goto 98
      endif
    endif
    !
    write(mess,'(I0,A)') currec,' records written'
    call astro_message(seve%i,rname,mess)
    !
98  close(lun)
99  call sic_frelun(lun)
  end subroutine write_bintable
  !
  subroutine write_my_array(lun,array,n,buffer,currec,recpos,error)
    integer(kind=4), intent(in)    :: lun
    integer(kind=4), intent(in)    :: n
    real(kind=4),    intent(in)    :: array(n)
    integer(kind=4), intent(inout) :: buffer(:)
    integer(kind=4), intent(inout) :: currec
    integer(kind=4), intent(inout) :: recpos
    logical,         intent(inout) :: error
    ! Local
    integer(kind=4), parameter :: reclen=128
    integer(kind=4) :: nwritetot,nwritecur
    !
    nwritetot = 0
    do while (nwritetot.lt.n)
      !
      nwritecur = min(n-nwritetot,reclen-recpos+1)
      call r4tor4(array(nwritetot+1),buffer(recpos),nwritecur)
      nwritetot = nwritetot+nwritecur
      recpos = recpos+nwritecur
      !
      if (recpos.gt.reclen) then
        ! print *,"Writing record ",currec
        write(lun,rec=currec,iostat=ier)  buffer
        if (ier.ne.0) then
          write(mess,'(A,I0)') 'Error writing record ',currec
          call astro_message(seve%e,rname,mess)
          error = .true.
          return
        endif
        currec = currec+1
        recpos = 1
      endif
      !
    enddo
  end subroutine write_my_array
  !
end subroutine noema_tsys_table
