program uv_addsh
  !------------------------------------------------------------------------
  ! task  compute a short spacings uv table from a single-dish table
  !     by gridding, extending to 0, filtering in uv plane, multiplication
  !     by interferometer primary beam, and sampling in uv plane.
  !
  ! input :
  ! a single-dish table
  ! output :
  ! a uv table
  !------------------------------------------------------------------------
  !
  ! Variables
  ! ---------
  !
  use image_def
  use gbl_format
  use gkernel_interfaces
  !
  real(8), parameter :: pi=3.141592653589793d0
  !
  ! Images :
  ! sdt  : input table from Single Dish
  ! lmv  : output Single Dish image cube format lmv after resampling
  ! uvt  : output uv_table modeling short spacings from Single Dish values
  !
  type (gildas), save :: sdt, lmv, uvt
  real, dimension(:,:), allocatable :: sdt_data
  real, dimension(:,:,:), allocatable :: lmv_data
  real, dimension(:,:), allocatable :: uvt_data
  !
  ! sdw       : weights table
  ! rawcube   : work space for sdt gridded values
  ! gr_im_w   : work space for weights gridded values in image plane
  ! gr_uv_w   : work space for weights gridded values in uv plane
  ! xcoord, ycoord : lmv axes gridded coordinates
  ! sd_lobe   : SD primary beam
  ! int_lobe  : interferometer primary beam
  ! fftws     : work space for FFT
  ! int_lobe_comp : interferometer beam in complex plane
  ! ws_data   : intermediate single-dish uv table values computing
  !
  real, dimension(:), allocatable :: sdw
  real, dimension(:), allocatable :: rawcube
  real, dimension(:), allocatable :: gr_im_w
  complex, dimension(:), allocatable :: gr_uv_w
  real, dimension(:), allocatable :: xcoord
  real, dimension(:), allocatable :: ycoord
  real, dimension(:), allocatable :: sd_lobe
  real, dimension(:), allocatable :: int_lobe
  real, dimension(:), allocatable :: fftws
  complex, dimension(:,:,:), allocatable :: ws_data
  complex, dimension(:), allocatable :: int_lobe_comp
  !
  real ubias,vbias,ubuff(8192),vbuff(8192)
  common /tconv/ ubias,vbias,ubuff,vbuff
  !
  ! Dummy variables
  !
  integer nx,ny,nc,np,nd,n,m,nxmore,nymore,nn(2),ndim
  integer ctypx,ctypy
  integer ier
  integer xcol,ycol,mcol(2),ocol,lcol,wcol, blc(4), trc(4), i
  real(kind=8) xconv(3),yconv(3),new(2),old(2),tmp
  real xmin,xmax,ymin,ymax,tole,smooth
  real maxw,minw,xparm(10),yparm(10),support(2),cell(2)
  logical error
  character(len=filename_length) table, uv_table, map_name, name
  character(len=3) extension
  integer nvis, if, nf
  integer(kind=4), parameter :: mf=256
  real, dimension(:), allocatable, save :: raoff, deoff
  real sfactor
  !
  real sd_diam, sd_beam, ip_diam, ip_beam, sd_weight, sd_factor, uv_trunc
  logical do_single, do_primary, lmv_file
  character(len=2) weight_mode
  character(len=12) ctype(5)
  character(len=20) chra,chde
  data ctype /'box','exp','sinc','exp*sinc','spheroidal'/
  data blc/4*0/, trc/4*0/
  character(len=256) :: rname = 'UV_SHORT'
  !
  integer :: last
  logical :: positions = .false.
  integer :: short_mode=0
  !------------------------------------------------------------------------
  !
  ! Input parameters
  ! ----------------
  !
  call gildas_open
  call gildas_char('TABLE$',table)
  call gildas_char('UV_TABLE$', uv_table)
  call gildas_char('MAP$',map_name)
  !
  call gildas_inte('XCOL$',xcol,1)
  call gildas_inte('YCOL$',ycol,1)
  call gildas_inte('WCOL$',wcol,1)
  call gildas_inte('MCOL$',mcol,2)
  !
  call gildas_char('WEIGHT_MODE$',weight_mode)
  call gildas_real('TOLE$',tole,1)
  call gildas_real('MIN_WEIGHT$',minw,1)
  !
  call gildas_real('SD_BEAM$', sd_beam,1)         ! single dish beam (rad)
  call gildas_real('SD_DIAM$', sd_diam,1)         ! single dish diam. (m)
  call gildas_real('IP_BEAM$', ip_beam,1)         ! interf. primary beam (rad)
  call gildas_real('IP_DIAM$', ip_diam,1)         ! interf. diam. (m)
  call gildas_real('UV_TRUNC$',uv_trunc,1)        ! truncation radius (m)
  call gildas_logi('DO_SINGLE$', do_single, 1)
  call gildas_logi('DO_PRIMARY$', do_primary, 1)
  call gildas_real('SD_WEIGHT$', sd_weight,1)     ! single dish weight
  call gildas_real('SD_FACTOR$', sd_factor,1)     ! unit conversion factor
  !
  call gildas_inte('MODE$',short_mode,1)
  if (short_mode.ne.0) then
    write(6,*) 'F-UV_SHORT,  This historic version of UV_SHORT only supports MODE$ = 0'
    call sysexi(fatale)
  endif
  call gildas_char('MOSAIC_RA$',chra)
  call gildas_char('MOSAIC_DEC$',chde)
  !
  call gildas_inte('NF$',nf,1)
  if (nf.gt.mf) then
     write(6,*) 'F-UV_SHORT, Too many fields'
     call sysexi (fatale)
  endif
  nf = max(nf,1)
  allocate(raoff(nf),deoff(nf))
  call gildas_real('RA$',raoff,nf)
  call gildas_real('DEC$',deoff,nf)
  call gildas_close
  !
  print *,'I-UV_SHORT, weighting mode is '//weight_mode
  !
  !-----------------------------------------------------------------------
  !
  ! Various checks
  ! --------------
  !
  n = lenc(table)
  if (n.le.0) then
     write(6,*) 'F-UV_SHORT, Input table name empty'
     call sysexi (fatale)
  endif
  !
  if (sd_beam.eq.0) then
     write(6,*) 'F-UV_SHORT, SD_BEAM must be entered'
     call sysexi(fatale)
  endif
  !
  if (sd_diam.le.ip_diam) then
     write(6,*) 'F-UV_SHORT, SD diameter too small'
     call sysexi (fatale)
  endif
  !
  if (uv_trunc.gt.(sd_diam-ip_diam)) then
     write(6,*) 'W-UV_SHORT, Uncoherent input parameters'
     write(6,*) 'W-UV_SHORT, Setting UV_TRUNC to ',sd_diam-ip_diam
     uv_trunc = sd_diam-ip_diam
  endif
  !
  ! Initialize gildas headers
  !
  call gildas_null(sdt)
  call gildas_null(lmv)
  call gildas_null(uvt, type = 'UVT')
  !
  ! Check if it's a table or a cube lmv input file
  !
  n=n+1
  m=n-3
  extension = table(m:n)
  write(6,*) 'I-UV_SHORT, Input file extension is ',extension
  if (extension.eq.'tab') then
     lmv_file=.false.
  elseif (extension.eq.'lmv') then
     lmv_file=.true.
  else
     lmv_file = .false.  ! To avoid compiler warnings
     write(6,*) 'F-UV_SHORT, You should give input file name with his extension'
     call sysexi(fatale)
  endif
  !
  !-----------------------------------------------------------------------
  !
  ! Input file is a single-dish table
  ! ---------------------------------
  !
  if (.not.lmv_file) then
     !
     name = table(1:n)
     call sic_parsef(name,sdt%file,' ','.tab')
     !
     ! Read sdt header and check format
     !
     call gdf_read_header(sdt,error)
     if (gildas_error(sdt,rname,error)) then
        write(6,*) 'F-UV_SHORT, Cannot read header from Single Dish table'
        call sysexi(fatale)
     endif
     if (sdt%gil%form.ne.fmt_r4) then
        write(6,*) 'F-UV_SHORT, Only real format supported'
        call sysexi (fatale)
     endif
     nd = sdt%gil%dim(1)
     np = sdt%gil%dim(2)
     !
     ! Check xcol,ycol input parameters
     !
     if ((xcol.gt.nd).or. (ycol.gt.nd)) then
        write(6,*) 'F-UV_SHORT, X or Y column does not exist'
        call sysexi (fatale)
     endif
     !
     ! Allocate memory space and read data
     !
     allocate(sdt_data(nd,np), stat=sdt%status)
     if (gildas_error(sdt,rname,error)) then
        write(6,*) 'F-UV_SHORT, Cannot allocate memory for Single Dish table'
        call sysexi(fatale)
     endif
     !
     call gdf_read_data(sdt,sdt_data,error)
     if (gildas_error(sdt,rname,error)) then
        write(6,*) 'F-UV_SHORT, Cannot read data from Single Dish table'
        call sysexi(fatale)
     endif
     !
     ! Allocate memory for weight table
     !
     allocate(sdw(2*max(np,nd)),stat=ier)
     if (ier.ne.0) then
        write(6,*) 'F-UV_SHORT, Cannot allocate memory for weigths table'
        call sysexi(fatale)
     endif
     !
     !-----------------------------------------------------------------------
     !
     ! Arrange data
     ! ------------
     !
     ! Recompute offsets if reference position is to be modified
     !
     if (chra.eq.'0' .and. chde.eq.'0') then
        continue
     else
        call sic_decode(chra,new(1),24,error)
        if (error) call sysexi (fatale)
        call sic_decode(chde,new(2),360,error)
        if (error) call sysexi (fatale)
        old(1) = sdt%gil%a0
        old(2) = sdt%gil%d0
        !
        ! Recompute offsets
        !
        call dopoint (sdt_data,nd,np,xcol,ycol,old,new)
        sdt%gil%a0 = new(1)
        sdt%gil%d0 = new(2)
     endif
     !
     ! Set the sinus type projection
     !
     sdt%gil%ptyp = p_azimuthal
     !
     ! Order sdt_data with Y in increasing order
     ! sdw is used as work space only
     !
     call dosor (sdt_data,nd,np,sdw,ycol)
     !
     ! Read weights in sdw table
     !
     call dowei (sdt_data,nd,np,sdw,wcol)
     !
     !-----------------------------------------------------------------------
     !
     ! Find lmv image size
     ! -------------------
     !
     ! Find min,max offsets of the SD observations
     ! (subroutine working only on Y ordered array)
     !
     call finsiz (sdt_data,nd,np,xcol,ycol,sdw,xmin,xmax,ymin,ymax)
     !
     ! Find increment: 4 pixels per SD beam
     !
     xconv(3) = -sd_beam/4.0
     yconv(3) = sd_beam/4.0
     !
     ! ! Old code: used if sd_beam = 0
     ! ! pixel size = min distance between SD points
     ! xinc = xmax-xmin
     ! yinc = ymax-ymin
     ! call fininc (sdt_data,nd,np,xcol,ycol,sdw,xinc,yinc,tole)
     ! xconv(3) = -xinc
     ! yconv(3) = +yinc
     !
     ! Find size of output SD image
     !
     nx = 2 * max ( nint(abs(xmax/xconv(3))+1), nint(abs(xmin/xconv(3))+1) )
     ny = 2 * max ( nint(abs(ymax/yconv(3))+1), nint(abs(ymin/yconv(3))+1) )
     nxmore = nint(4*sd_beam/abs(xconv(3)))+1
     nymore = nint(4*sd_beam/abs(yconv(3)))+1
     nx = nx+2*nxmore
     ny = ny+2*nymore
     !
     ! Extend nx,ny to nearest power of two
     !
     i = 32
     do while(i.lt.nx)
        i = i*2
     enddo
     nx = i
     i = 32
     do while(i.lt.ny)
        i = i*2
     enddo
     ny = i
     !
     ! Reference position for lmv cube header
     !
     xconv(1) = nx/2+1
     xconv(2) = 0.0
     yconv(1) = ny/2+1
     yconv(2) = 0.0
     !
     tmp = 0.1*nint(yconv(3)*10*180*3600/pi)
     write(6,*) 'I-UV_SHORT, Creating a cube with ',nx,' by ',ny,' pixels'
     write(6,*) 'I-UV_SHORT, Pixel size: ',tmp,' arcsec'
     !
     ! Warn for big images
     !
     if (nx.gt.8192 .or. ny.gt.8192) then
        write(6,*) 'F-UV_SHORT, More than 8192 pixels in X or Y'
        write(6,*) 'W-UV_SHORT, Offset extrema are: ',xmin,xmax,ymin,ymax
        write(6,*) 'W-UV_SHORT, Pixel sizes are', xconv(3),yconv(3)
        call sysexi (fatale)
     elseif (nx.gt.512 .or. ny.gt.512) then
        write(6,*) 'W-UV_SHORT,  more than 512 pixels in X or Y'
        write(6,*) 'W-UV_SHORT, Offset extrema are: ',xmin,xmax,ymin,ymax
        write(6,*) 'W-UV_SHORT, Pixel sizes are', xconv(3),yconv(3)
     endif
     !
     ! Number of channels
     !
     if (mcol(2).eq.0) mcol(2) = nd
     mcol(1) = max(1,min(mcol(1),nd))
     mcol(2) = max(1,min(mcol(2),nd))
     ocol = min(mcol(1),mcol(2))       ! first channel to grid
     lcol = max(mcol(1),mcol(2))       ! last channel to grid
     nc = lcol-ocol+1
     write(6,*) 'I-UV_SHORT, Creating ',nc,' channels from ',mcol
     ocol = ocol-1
     !
     !-----------------------------------------------------------------------
     !
     ! Create lmv image
     ! ----------------
     !
     ! Copy header from input SD table
     !
     call gdf_copy_header(sdt,lmv,error)
     !
     ! Create image (in order l m v)
     !
     n=lenc(map_name)
     if (n.le.0) then
        write(6,*) 'F-UV_SHORT, Output image name empty'
        call sysexi (fatale)
     endif
     name = map_name(1:n)
     call sic_parsef(name,lmv%file,' ','.lmv')
     write(6,*) 'I-UV_SHORT, Creating map file '//lmv%file(1:lenc(lmv%file))
     !
     ! Fill in header
     !
     lmv%gil%ndim = 3
     lmv%gil%dim(1) = nx
     lmv%gil%dim(2) = ny
     lmv%gil%dim(3) = nc
     lmv%gil%dim(4) = 1
     lmv%loca%size = nx*ny*nc
     lmv%gil%ref(3) = sdt%gil%ref(1)-ocol
     lmv%gil%val(3) = lmv%gil%voff
     lmv%gil%inc(3) = lmv%gil%vres
     lmv%gil%convert(:,1) = xconv
     lmv%gil%convert(:,2) = yconv
     lmv%char%code(1) = sdt%char%code(2)
     lmv%char%code(2) = sdt%char%code(3)
     lmv%char%code(3) = sdt%char%code(1)
     lmv%gil%coor_words = 6*gdf_maxdims             ! not a table
     lmv%gil%extr_words = 0                   ! extrema not computed
     lmv%gil%xaxi = 1                         ! reset projected axis
     lmv%gil%yaxi = 2
     lmv%gil%faxi = 3
     lmv%gil%form = fmt_r4
     !
     ! Allocate memory
     !
     allocate(lmv_data(lmv%gil%dim(1),lmv%gil%dim(2),lmv%gil%dim(3)),&
          stat=lmv%status)
     if (gildas_error(lmv,rname,error)) then
        write(6,*) 'F-UV_SHORT, Cannot allocate memory for output lmv image'
        call sysexi(fatale)
     endif
     !
     !-----------------------------------------------------------------------
     !
     ! Memory allocations
     ! ------------------
     !
     ! Get the workspace to store data
     !
     allocate(rawcube(nc*nx*ny),stat=ier)
     if (ier.ne.0) then
        write(6,*) 'F-UV_SHORT, Cannot allocate memory for raw cube lmv'
        call sysexi(fatale)
     endif
     allocate(gr_im_w(2*nx*ny),stat=ier)
     if (ier.ne.0) then
        write(6,*) 'F-UV_SHORT, Cannot allocate memory for work space'
        call sysexi(fatale)
     endif
     allocate(xcoord(nx),stat=ier)
     if (ier.ne.0) then
        write(6,*) 'F-UV_SHORT, Cannot allocate memory for lmv axis X'
        call sysexi(fatale)
     endif
     allocate(ycoord(ny),stat=ier)
     if (ier.ne.0) then
        write(6,*) 'F-UV_SHORT, Cannot allocate memory for lmv axis Y'
        call sysexi(fatale)
     endif
     !
     !-----------------------------------------------------------------------
     !
     ! Gridding of the SD data
     ! -----------------------
     !
     ! Resampling in space of the original spectra on a regular grid,
     ! using a convolution kernel
     !
     ! Compute gridding function: a small (1/3 of SD beam) gaussian
     ! Note: since xconv and yconv depends on SD beam, all gridding
     ! parameters do depend on SD beam value...
     !
     smooth = sd_beam/3.0
     ctypx = 2
     ctypy = 2
     support(1) = 5*smooth              ! go far enough...
     support(2) = 5*smooth
     xparm(1) = support(1)/abs(xconv(3))
     yparm(1) = support(2)/abs(yconv(3))
     xparm(2) = smooth/(2*sqrt(log(2.0)))/abs(xconv(3))
     yparm(2) = smooth/(2*sqrt(log(2.0)))/abs(yconv(3))
     xparm(3) = 2
     yparm(3) = 2
     !
     ! useful only if ctypx and ctypx are set by the user
     ! call grdflt (ctypx, ctypy, xparm, yparm)
     !
     call convfn (ctypx, xparm, ubuff, ubias)
     call convfn (ctypy, yparm, vbuff, vbias)
     cell(1) = xconv(3)
     cell(2) = yconv(3)
     !
     ! Compute gridded coordinates
     !
     call docoor (nx,lmv%gil%ref(1),lmv%gil%val(1),lmv%gil%inc(1),xcoord)
     call docoor (ny,lmv%gil%ref(2),lmv%gil%val(2),lmv%gil%inc(2),ycoord)
     !
     ! Grid data: output = rawcube
     ! **WW** poids sdw --> poids gr_im_w = pour chaque pixel, somme des
     !        [poids des points pris en compte * gaussienne]
     !        (NON normalise)
     !        --> a normaliser par somme des gaussiennes ('results')?
     !
     gr_im_w = 0.
     call doconv (&
     nd,np,&                             ! number of input points
     sdt_data,&                          ! input values
     xcol,ycol,ocol,&                    ! pointers to special values
     sdw,&                               ! weights
     gr_im_w,&                           ! gridded weights
     nc,nx,ny,&                          ! cube size
     rawcube,&                           ! gridded data (output)
     xcoord,ycoord,&                     ! cube coordonates
     support,cell,maxw)
     !
     ! Min and max weights
     !
     minw = maxw*minw
     write(6,*) 'I-UV_SHORT, Done gridding'   ! ,minw,maxw
     !
     !
     !-----------------------------------------------------------------------
     !
     ! Extrapolation to zero outside the convex hull of the mapped region
     ! ------------------------------------------------------------------
     !
     ! Compute smoothing function = SD primary beam
     !
     ctypx = 2
     ctypy = 2
     support(1) = 3*sd_beam
     support(2) = 3*sd_beam
     xparm(1) = support(1)/abs(xconv(3))
     yparm(1) = support(2)/abs(yconv(3))
     xparm(2) = sd_beam/(2*sqrt(log(2.0)))/abs(xconv(3))
     yparm(2) = sd_beam/(2*sqrt(log(2.0)))/abs(yconv(3))
     xparm(3) = 2
     yparm(3) = 2
     !
     ! useful only if ctypx and ctypx are set by the user
     ! call grdflt (ctypx, ctypy, xparm, yparm)
     !
     call convfn (ctypx, xparm, ubuff, ubias)
     call convfn (ctypy, yparm, vbuff, vbias)
     cell(1) = xconv(3)
     cell(2) = yconv(3)
     !
     ! Smooth data by gaussian = SD beam
     !       input = rawcube
     !       output = lmv_data (used as work space)
     !
     ! **WW** Les poids sont passes a dosmo, mais ne sont pas
     !        utilises
     !
     call dosmoo (&
          rawcube,&                   ! raw gridded values
          gr_im_w,&                   ! gridded weights
          nc,nx,ny,&                  ! cube size
          lmv_data,&                  ! smoothed cube
          xcoord,ycoord,&             ! cube coordinates
          support,cell)
     write(6,*) 'I-UV_SHORT, Done smoothing'
     !
     ! Apodisation: - smooth image (lmv_data) is apodisated by gaussian
     !                at image edges
     !              - replace input image (rawcube) by smoothed-apodisated
     !                image at map edges
     !              - replace input image (rawcube) by smoothed image
     !                at points where weights < minw
     ! Output = rawcube
     !
     ! **WW** poids sont utilises par doapod, non modifies
     !
     call doapod (xmin,xmax,ymin,ymax,tole,sd_beam,&
          nc,nx,ny,&
          lmv_data,&              ! input smoothed cube
          rawcube,&               ! output after apodisation
          xcoord,ycoord,&         ! cube coordinates
          gr_im_w, minw)
     write(6,*) 'I-UV_SHORT, Done apodisation'
     !
     ! Transpose to the lmv order and put the result in lmv_data
     !
     call dotrans(rawcube,lmv_data,nc,nx*ny)
     write(6,*) 'I-UV_SHORT, Done transposing'
     !
     ! Free some memory
     !
     deallocate(sdt_data,sdw,rawcube,xcoord,ycoord)
     !
  !=====================================================================
  !
  ! Input file is a lmv cube
  ! ------------------------
  !
  else
     !
     ! Read file cube lmv
     !
     name = table(1:n)
     call sic_parsef(name,lmv%file,' ','.lmv')
     !
     call gdf_read_header(lmv,error)
     if (gildas_error(lmv,rname,error)) then
        write(6,*) 'F-UV_SHORT, Cannot read header from input cube lmv'
        call sysexi(fatale)
     endif
     !
     ! Allocate memory
     !
     allocate(lmv_data(lmv%gil%dim(1),lmv%gil%dim(2),lmv%gil%dim(3)),&
          stat=lmv%status)
     if (gildas_error(lmv,rname,error)) then
        write(6,*) 'F-UV_SHORT, Cannot allocate memory for output lmv image'
        call sysexi(fatale)
     endif
     !
     call gdf_read_data(lmv,lmv_data,error)
     if (gildas_error(lmv,rname,error)) then
        write(6,*) 'F-UV_SHORT, Cannot read data from input cube lmv'
        call sysexi(fatale)
     endif
     !
     ! Update image lmv variables
     !
     nx=lmv%gil%dim(1)
     ny=lmv%gil%dim(2)
     nc=lmv%gil%dim(3)
     if ((nx.eq.0).or.(ny.eq.0).or.(nc.eq.0)) then
        write(6,*) 'F-UV_SHORT, Inconsistent input lmv cube dimensions'
        call sysexi(fatale)
     endif
     !
     xconv(3)=lmv%gil%inc(1)
     yconv(3)=lmv%gil%inc(2)
     !
     write(6,*) 'I-UV_SHORT, Image cube lmv with ',nx,' by ',ny,' pixels'
     write(6,*) 'I-UV_SHORT, Pixel sizes are', xconv(3),yconv(3)
     !
     ! What will we do about weights....?
     !
     allocate(gr_im_w(2*nx*ny),stat=ier)
     if (ier.ne.0) then
        write(6,*) 'F-UV_SHORT, Cannot allocate memory for work space'
        call sysexi(fatale)
     endif
     gr_im_w=1.0
     !
  endif
  !
  !=====================================================================
  !
  ! Processing for all types .lmv or .tab of input file
  !
  !=====================================================================
  !
  ! Memory allocations
  !--------------------
  !
  allocate(ws_data(nx,ny,nc),stat=ier)
  if (ier.ne.0) then
     write(6,*) 'F-UV_SHORT, Cannot allocate memory for work space'
     call sysexi(fatale)
  endif
  allocate(sd_lobe(nx*ny),stat=ier)       ! for SD primary beam
  if (ier.ne.0) then
     write(6,*) 'F-UV_SHORT, Cannot allocate memory SD primary beam'
     call sysexi(fatale)
  endif
  allocate(int_lobe(nx*ny),stat=ier)      ! for interferometer beam
  if (ier.ne.0) then
     write(6,*) 'F-UV_SHORT, Cannot allocate memory for interferometer beam'
     call sysexi(fatale)
  endif
  allocate(gr_uv_w(nx*ny),stat=ier)     ! for gridded weights in uv plane
  if (ier.ne.0) then
     write(6,*) 'F-UV_SHORT, Cannot allocate memory for gridded weights'
     call sysexi(fatale)
  endif
  allocate(fftws(2*max(nx,ny)),stat=ier)         ! for fft
  if (ier.ne.0) then
     write(6,*) 'F-UV_SHORT, Cannot allocate memory for fft computation'
     call sysexi(fatale)
  endif
  allocate(int_lobe_comp(nx*ny),stat=ier)    ! for int. Primary Beam
  if (ier.ne.0) then
     write(6,*) 'F-UV_SHORT, Cannot allocate memory for primary beam'
     call sysexi(fatale)
  endif
  !
  !-----------------------------------------------------------------------
  !
  ! Visibilities extraction
  ! -----------------------
  !
  ! Update parameters for FFT computation
  !
  ndim = 2
  nn(1) = nx
  nn(2) = ny
  !
  ! Get inverse of FT of (primary beam + convolving function)
  ! CAUTION: output of dosdft function is a real, but with
  ! complex convention
  !
  sfactor = 1.0+1.0/9.0    ! take into account gridding function
  smooth = sd_beam*sqrt(sfactor)
  call dosdft(smooth,sd_diam,sd_lobe,nx,ny,sfactor,lmv)
  !
  ! Start loop on mosaic fields
  !
  do if=1,nf
     !
     ! 1) Compute interferometer primary beam
     ! --------------------------------------
     !
     ! 1.a) Int. Primary Beam = gaussian, computed in image plane
     !
     ! call doprim(ip_beam,int_lobe,nx,ny,raoff(if),deoff(if),lmv)
     !
     ! 1.b) Int. Primary Beam = truncated gaussian in the uv plane
     !      then FT
     !
     call dointft(ip_beam,ip_diam,int_lobe,nx,ny,1.0,lmv)
     int_lobe_comp(:) = cmplx(int_lobe)
     call shift(int_lobe_comp,nx,ny,raoff(if),deoff(if),lmv)
     call fourt(int_lobe_comp,nn,ndim,1,1,fftws)
     call cmtore(int_lobe_comp,int_lobe,nx,ny)
     !
     ! Normalize peak to 1
     !
     int_lobe(:) = int_lobe/maxval(int_lobe) ! ,1,.true.)
     !
     ! For checks
     ! call copy(lmv_data,int_lobe,nx,ny,nc,if)
     !
     ! 2) Compute pseudo-visibilities
     ! ------------------------------
     !
     write(6,*) 'I-UV_SHORT, Filtering  field ',if
     !
     ! Start loop on channels
     !
     do i = 1,nc
        !
        ! FT of SD image
        !
        call retocm (lmv_data(:,:,i),ws_data(:,:,i),nx,ny)
        call fourt  (ws_data(:,:,i),nn,ndim,1,1,fftws)
        !
        ! SD beam correction (in uv plane)
        !
        if (do_single) then
           call sdcorr (ws_data(:,:,i),sd_lobe,nx*ny)
        endif
        !
        ! Apply interferometer primary beam (in image plane)
        !
        if (do_primary) then
           call fourt  (ws_data(:,:,i),nn,ndim,-1,1,fftws)
           call prmult (ws_data(:,:,i),int_lobe,nx,ny)
           call fourt  (ws_data(:,:,i),nn,ndim,1,1,fftws)
           ws_data(:,:,i) = ws_data(:,:,i)/(nx*ny)
        endif
        !
        ! End loop on channels
        !
     enddo
     !
     ! 3) Compute weights
     ! ------------------
     !
     ! Weights: do something
     !
     if (weight_mode.eq.'UN') then
        gr_uv_w = 1.
     else
        call retocm (gr_im_w,gr_uv_w,nx,ny)
        call fourt  (gr_uv_w,nn,ndim,1,1,fftws)
        gr_uv_w(:) = gr_uv_w/(nx*ny)
!        call copy(lmv_data,gr_im_w,nx,ny,nc)
     endif
     if (do_single) then
        gr_uv_w = 1
        call wcorr(gr_uv_w,sd_lobe,nx*ny)
     endif
     !
     ! 4) Create uv table
     ! ------------------
     !
     ! Compute number of visibilities < uv_trunc
     !
     call uvcount(nx,ny,nvis,uv_trunc,lmv)
     !
     ! Output uv table
     !
     n = lenc(uv_table)
     if (n.eq.0) then
        write(6,*) 'F-UV_SHORT, Empty uv table name'
        call sysexi (fatale)
     endif
     name  = uv_table(1:n)
     if (nf.gt.1) then
        name(n+1:) = '-'
        n = n+2
        write(name(n:),'(i0)') if
     endif
     call sic_parsef(name,uvt%file,' ','.uvt')
     write(6,*) 'I-UV_SHORT, Creating uv table '//uvt%file(1:lenc(uvt%file))
     !
     ! Fill in header
     !
     call gdf_copy_header(lmv,uvt,error)
     uvt%char%code(2) = 'RANDOM'
     uvt%char%code(1) = 'UV-RAW'
     uvt%gil%coor_words = 6*gdf_maxdims
     uvt%gil%blan_words = 2
     uvt%gil%extr_words = 10
     uvt%gil%desc_words = 18
     uvt%gil%posi_words = 12
     uvt%gil%proj_words = 9
     uvt%gil%spec_words = 12
     uvt%gil%reso_words = 3
     uvt%gil%dim(2) = nvis
     uvt%gil%dim(1) = 3*nc+7        ! 7 daps + (real, imag, weight)*nchannels
     uvt%gil%convert = 0
     uvt%gil%ref(1) = lmv%gil%ref(3)
     uvt%gil%inc(1) = lmv%gil%fres
     uvt%gil%val(1) = lmv%gil%freq
     uvt%gil%inc(2) = 1.              ! needed to avoid funny crash in graphic...
     uvt%gil%ndim = 2
     uvt%gil%dim(3) = 1
     uvt%gil%dim(4) = 1
     !
     ! Here we could change the logic, keep A0,D0 and set the offsets
     ! in the UV table.  But this has consequences in the whole package.
     !
     uvt%gil%ra = uvt%gil%a0+raoff(if)/cos(uvt%gil%d0)
     uvt%gil%dec = uvt%gil%d0+deoff(if)
     uvt%char%type = 'GILDAS_UVFIL'
     uvt%char%unit = 'Jy'
     uvt%gil%nchan = nc
    !
    ! Here define the order in which you want the extra "columns"
     uvt%gil%column_pointer = 0
     uvt%gil%column_size = 0
     uvt%gil%column_pointer(code_uvt_u) = 1
     uvt%gil%column_pointer(code_uvt_v) = 2
     uvt%gil%column_pointer(code_uvt_w) = 3
     uvt%gil%column_pointer(code_uvt_date) = 4
     uvt%gil%column_pointer(code_uvt_time) = 5
     uvt%gil%column_pointer(code_uvt_anti) = 6
     uvt%gil%column_pointer(code_uvt_antj) = 7
     uvt%gil%natom = 3
     uvt%gil%nstokes = 1
     uvt%gil%fcol = 8
     last = uvt%gil%fcol + uvt%gil%natom * uvt%gil%nchan - 1
     !
     uvt%gil%form = fmt_r4
     !
     if (positions) then
       last = last+1
       uvt%gil%column_pointer(code_uvt_loff) = last
       last = last+1
       uvt%gil%column_pointer(code_uvt_moff) = last
     endif
     do i=1,code_uvt_last
       if (uvt%gil%column_pointer(i).ne.0) uvt%gil%column_size(i) = 1
     enddo
     !
     uvt%gil%nvisi = nvis
     uvt%gil%type_gdf = code_gdf_uvt
     call gdf_setuv (uvt,error)
     uvt%loca%size = uvt%gil%dim(1) * uvt%gil%dim(2)
    !!print *, 'Into newuvt_init ',xima%gil%nvisi, xima%gil%nchan, xima%gil%dim(1:2)
    !!print *, 'Into newuvt_init UVDA_WORDS', xima%gil%uvda_words
     !
     ! Allocate memory for uv table
     !
     allocate(uvt_data(uvt%gil%dim(1),uvt%gil%dim(2)),stat=ier)
     if (gildas_error(uvt,rname,error)) then
        write(6,*) 'F-UV_SHORT, Cannot allocate memory for uv table '
        call sysexi(fatale)
     endif
     !
     ! Fill in uv table
     !
     call uvtable(nx,ny,nc,ws_data,uvt_data,gr_uv_w,&
          nvis, uv_trunc, sd_weight, sd_factor,lmv)
     !
     ! Write uv table
     !
     call gdf_write_image(uvt,uvt_data,error)
     if (error) then
        write(6,*) 'F-UV_SHORT, Error writing image uv '
        call sysexi(fatale)
     endif
     write(6,*) 'I-UV_SHORT, ',nvis,' visibilities written'
     deallocate(uvt_data)
     !
     ! End loop on mosaic fields
     !
  enddo
  !
  ! Write output lmv image
  !
  call gdf_write_image(lmv,lmv_data,error)
  if (error) then
     write(6,*) 'F-UV_SHORT, Error writing image lmv '
     call sysexi(fatale)
  endif
  deallocate(lmv_data)
  !
  ! Delete scratch spaces
  !
  deallocate(ws_data,fftws,gr_im_w,gr_uv_w,sd_lobe,int_lobe)
  !
  ! End
  !
  stop 'I-UV_SHORT, Successful completion'
  !
end program uv_addsh
!
!=========================================================================
!
subroutine docoor (n,xref,xval,xinc,x)
  !
  ! Computes coordinate array from ref,val,inc
  !
  ! Dummy variables
  !
  integer n
  real x(n)
  real*8 xref,xval,xinc
  !
  ! Local variables
  !
  integer i
  !
  ! Code
  !
  x(1:n) = (/(real((dble(i)-xref)*xinc+xval),i=1,n)/)
end subroutine docoor
!
!-------------------------------------------------------------------------
!
subroutine doconv (nd,np,visi,jx,jy,jo,we,gwe,&
     nc,nx,ny,map,mapx,mapy,sup,cell,maxw)
  !
  ! Convolution of 'map' by function defined by vbuff & vbias
  ! (via subroutine convol).
  ! Used for gridding of SD data.
  !
  ! Variables
  !
  integer np                         ! number of values
  integer nd                         ! number of "visibilities"
  integer nc                         ! number of channels
  integer nx,ny                      ! map size
  integer jx,jy                      ! x coord, y coord location in visi
  real we(np)                        ! weights
  integer jo                         ! offset for data in visi
  real visi(nd,np)                   ! values
  real gwe(nx,ny)                    ! gridded weights
  real map(nc,nx,ny)                 ! gridded values
  real mapx(nx),mapy(ny)             ! coordinates of grid
  real sup(2)                        ! support of convolving function in user units
  real cell(2)                       ! cell size in user units
  real maxw                          ! maximum weight
  !
  ! Local variables
  !
  integer ifirs,ilast                ! range to be considered
  integer ix,iy,ic,i
  real result,weight
  real u,v,du,dv,um,up,vm,vp
  !
  ! Code
  !
  maxw = 0.0
  !
  ! Loop on Y
  !
  ifirs = 1
  do iy=1,ny
     v = mapy(iy)
     !
     ! sup is the support of the gridding function
     !
     vm = v-sup(2)
     vp = v+sup(2)
     !
     ! Find points to be considered.
     ! Optimized dichotomic search, taking into account the
     ! fact that mapy is an ordered array.
     !
     call findr (np,nd,jy,visi,vm,ifirs)
     ilast = ifirs
     call findr (np,nd,jy,visi,vp,ilast)
     ilast = ilast-1
     !
     ! Initialize x column
     !
     map(1:nc,1:nx,iy) = 0.0
     !
     if (ilast.ge.ifirs) then
        !
        ! Loop on x cells
        !
        do ix=1,nx
           u = mapx(ix)
           um = u-sup(1)
           up = u+sup(1)
           weight = 0.0
           !
           ! Loop on relevant data points
           !
           do i=ifirs,ilast
              !
              ! Test if X position is within the range to be
              ! considered
              !
              if (visi(jx,i).ge.um .and. visi(jx,i).le.up) then
                 !
                 ! Compute convolving factor
                 !
                 du = (u-visi(jx,i))/cell(1)
                 dv = (v-visi(jy,i))/cell(2)
                 call convol (du,dv,result)
                 if (result.ne.0.0) then
                    !
                    ! Do the convolution: map(pixel) = sum of
                    ! relevant values * convolving factor * weight
                    !
                    result = result*we(i)
                    weight = weight + result
                    map (1:nc,ix,iy) = map (1:nc,ix,iy) +&
                            visi((1+jo):(nc+jo),i)*result
                 endif
              endif
           enddo
           !
           ! gwe is the sum of the (convolving factor * weight) ie
           ! the sum of the weighting factors applied to the data
           !
           gwe(ix,iy) = weight
           maxw = max(maxw,weight)
           !
           ! Normalization (only in cells where some data exists)
           !
           if (weight.ne.0) then
              map (1:nc,ix,iy) = map(1:nc,ix,iy)/weight
           endif
        enddo
     endif
  enddo
end subroutine doconv
!
!---------------------------------------------------------------------------------------
!
subroutine dosmoo (raw,we,nc,nx,ny,map,mapx,mapy,sup,cell)
  !
  ! Smooth an input data cube raw in vlm along l and m by convolution
  ! by a gaussian function defined in vbuff & vbias via subroutine convol.
  !
  !
  ! Dummy variables
  !
  integer nc,nx,ny                   ! map size
  real we(nx,ny)                     ! weights
  real raw(nc,nx,ny)                 ! raw map
  real map(nc,nx,ny)                 ! smoothed map
  real mapx(nx),mapy(ny)             ! coordinates of grid
  real sup(2)                        ! support of convolving function in user units
  real cell(2)                       ! cell size in user units
  !
  ! Local variables
  !
  integer yfirs,ylast                ! range to be considered
  integer xfirs,xlast                ! range to be considered
  integer ix,iy
  integer jx,jy                      ! x coord, y coord location in raw
  real result,weight
  real u,v,du,dv,um,up,vm,vp,dx,dy
  !
  ! Code
  !
  dx = abs(mapx(2)-mapx(1))
  dy = abs(mapy(2)-mapy(1))
  !
  ! Loop on y rows
  !
  do iy=1,ny
     !
     ! Compute extrema positions on axe y
     ! of gaussian function centered on map(1:nc,ix,iy)
     !
     v = mapy(iy)
     vm = v-sup(2)
     vp = v+sup(2)
     !
     ! Compute extrema positions on axe y
     ! of relevant data points for map(1:nc,ix,iy) convolution
     !
     yfirs = max(1,nint((iy-sup(2)/dy)))
     ylast = min(ny,nint((iy+sup(2)/dy)))
     !
     ! Initialize x colum
     !
     map(1:nc,1:nx,iy) = 0.0
     !
     ! Loop on x cells
     !
     if (yfirs.le.ylast) then
        do ix=1,nx
           !
           ! Compute extrema positions on axe x, idem y
           !
           u = mapx(ix)
           um = u-sup(1)
           up = u+sup(1)
           weight = 0.0
           xfirs = max(1,nint(ix-sup(1)/dx))
           xlast = min(nx,nint(ix+sup(1)/dx))
           !
           ! Loop on relevant data points
           !
           if (xfirs.le.xlast) then
              do jy=yfirs,ylast
                 dv = (v-mapy(jy))/cell(2)
                 do jx=xfirs,xlast
                    du = (u-mapx(jx))/cell(1)
                    !
                    ! Compute convolving factor
                    !
                    call convol (du,dv,result)
                    if (result.ne.0.0) then
                       !
                       ! Do the convolution: map(pixel) = sum of
                       ! relevant values * convolving factor
                       !
                       weight = weight + result
                       map (1:nc,ix,iy) = map (1:nc,ix,iy) +&
                               raw(1:nc,jx,jy)*result
                    endif
                 enddo
              enddo
              !
              ! Normalize weight only in cells where some data exists...
              !
              if (weight.ne.0) then
                 map (1:nc,ix,iy) = map(1:nc,ix,iy)/weight
              endif
           endif
        enddo
     endif
  enddo
end subroutine dosmoo
!
!-----------------------------------------------------------------------
!
subroutine dowei (visi,nd,np,we,iw)
  !
  ! Fill in weights array from the input table
  !
  ! Dummy variable
  !
  integer nd,np,iw
  real visi(nd,np),we(np)
  !
  ! Local variables
  !
  integer i
  !
  ! Code
  !
  if (iw.le.0 .or. iw.gt.nd) then
     !
     ! Weight column does not exist...
     !
     we(1:np) = 1.0
  else
     !
     ! Weight colum do exist
     !
     we(1:np) = visi(iw,1:np)
  endif
end subroutine dowei
!
!-----------------------------------------------------------------------
!
subroutine findr (nv,nc,ic,xx,xlim,nlim)
  !
  ! gildas internal routine
  ! find nlim such as
  !   xx(ic,nlim-1) < xlim < xx(ic,nlim)
  ! for input data ordered, retrieved from memory
  ! assumes nlim already preset so that xx(ic,nlim-1) < xlim
  !
  !
  ! Dummy variables
  !
  integer nv,nc,ic,nlim
  real xx(nc,nv),xlim
  !
  ! Local variables
  !
  integer ninf,nsup,nmid
  !
  ! Code
  !
  if (nlim.gt.nv) return
  !
  ! Define limits of searching area in the table
  !
  if (xx(ic,nlim).gt.xlim) then
     return
  elseif (xx(ic,nv).lt.xlim) then
     nlim = nv+1
     return
  endif
  ninf = nlim
  nsup = nv
  !
  ! Loop while : dichotomic search for input data ordered
  !
  do while(nsup.gt.ninf+1)
     !
     ! Define middle of the searching area on the table
     !
     nmid = (nsup + ninf)/2
     !
     ! If it's not in the last part, it's in the first one...
     ! then defined new searching limits area
     !
     if (xx(ic,nmid).lt.xlim) then
        ninf = nmid
     else
        nsup = nmid
     endif
  enddo
  !
  ! Output
  !
  nlim = nsup
  !
end subroutine findr
!
!------------------------------------------------------------------------
!
subroutine convol (du,dv,resu)
  !
  ! Compute convolving factor resu
  ! resu is the result of the multiplication of the convolution functions,
  ! defined in ubuff & ubias and in vbuff & vbias, for u and v axes,
  ! at point (du,dv)
  !
  ! Global variables
  real ubias,vbias,ubuff(8192),vbuff(8192)
  common /tconv/ ubias,vbias,ubuff,vbuff
  !
  !
  ! Dummy variables
  !
  real resu,du,dv
  !
  ! Local variables
  !
  integer iu,iv
  !
  ! Code
  !
  ! convolving functions values are tabulated every 1/100 cell
  !
  iu = nint(100.0*du+ubias)
  iv = nint(100.0*dv+vbias)
  !
  ! Participation of u and v axes convolution functions
  !
  resu = ubuff(iu)*vbuff(iv)
  if (resu.lt.1e-20) resu = 0.0
end subroutine convol
!
!------------------------------------------------------------------------
!
subroutine convfn (cftype, parm, buffer, bias)
  use gkernel_interfaces
  use gildas_def
  !
  !   convfn computes the convolving functions and stores them in
  !     the supplied buffer. values are tabulated every 1/100 cell.
  ! arguments :
  ! cftype    i*4 convolving function type
  !   parm(10)        r*4   convolving function parameters.
  !       parm(1) = radius of support in cells
  ! buffer(8192)  r*4   work buffer.
  ! bias    r*4 center of convolution
  !
  !
  ! Dummy variables
  !
  integer cftype
  real*4  parm(10), buffer(8192), bias
  !
  ! Local variables
  !
  integer lim, i, im, ialf, ier, ibias
  real*4  pi, p1, p2, u, umax, absu, eta, psi
  data pi /3.1415926536/
  !
  ! Code
  !
  ! Compute number of rows
  !
  i = int( max (parm(1)+0.995 , 1.0) )
  i = i * 2 + 1
  lim = i * 100 + 1
  if (lim.gt.1.5*8192) then
     write(6,*) 'F-UV_SHORT, Work buffer insufficient ',lim
     call sysexi(fatale)
  elseif (lim.gt.8192) then
     lim = 8192
     bias = 4097
  else
     bias = 50.0 * i + 1.0
  endif
  umax = parm(1)
  !
  ! Type defaulted is 4 : function exp * sinc
  !
  if ((cftype.le.0).or.(cftype.gt.5)) then
     cftype = 4
     parm(1) = 3.0
     parm(2) = 1.55
     parm(3) = 2.52
     parm(4) = 2.00
   endif
  !
  ! Case of function type cftype
  !
  select case (cftype)
     !
  case(1)
     !
     ! Pill box function
     !
     do i = 1,lim
        u = (i-bias) * 0.01
        absu = abs (u)
        if (absu.lt.umax) then
           buffer(i) = 1.0
        elseif (absu.eq.umax) then
           buffer(i) = 0.5
        else
           buffer(i) = 0.0
        endif
     enddo
     !
  case(2)
     !
     ! Exponential function
     !
     p1 = 1.0 / parm(2)
     do i = 1,lim
        u = (i-bias) * 0.01
        absu = abs (u)
        if (absu.gt.umax) then
           buffer(i) = 0.0
        else
           buffer(i) = exp (-((p1*absu) ** parm(3)))
        endif
     enddo
     !
  case(3)
     !
     ! Function sinc
     !
     p1 = pi / parm(2)
     do i = 1,lim
        u = (i-bias)*0.01
        absu = abs (u)
        if (absu.gt.umax) then
           buffer(i) = 0.0
        elseif (absu.eq.0.0) then
           buffer(i) = 1.0
        else
           buffer(i) = sin (p1*absu) / (p1*absu)
        endif
     enddo
     !
  case(4)
     !
     ! Function exp * sinc
     !
     p1 = pi / parm(2)
     p2 = 1.0 / parm(3)
     do i = 1,lim
        u = (i-bias)*0.01
        absu = abs (u)
        if (absu.gt.umax) then
           buffer(i) = 0.0
        elseif (absu.lt.0.01) then
           buffer(i) = 1.0
        else
           buffer(i) = sin(u*p1) / (u*p1) *&
                exp (-((absu * p2) ** parm(4)))
        endif
     enddo
     !
  case(5)
     !
     ! Spheroidal function
     !
     do i = 1,lim
        buffer(i) = 0.0
     enddo
     ialf = 2.0 * parm(2) + 1.1
     im = 2.0 * parm(1) + 0.1
     ialf = max (1, min (5, ialf))
     im = max (4, min (8, im))
     lim = parm(1) * 100.0 + 0.1
     ibias = bias
     do i = 1,lim
        eta = float (i-1) / float (lim-1)
        call sphfn (ialf, im, 0, eta, psi, ier)
        buffer(ibias+i-1) = psi
     enddo
     lim = ibias-1
     do i = 1,lim
        buffer(ibias-i) = buffer(ibias+i)
     enddo
  !
  end select
  !
  return
end subroutine convfn
!
!-----------------------------------------------------------------------
!
subroutine sphfn (ialf, im, iflag, eta, psi, ier)
  !
  !     sphfn is a subroutine to evaluate rational approximations to se-
  !  lected zero-order spheroidal functions, psi(c,eta), which are, in a
  !  sense defined in vla scientific memorandum no. 132, optimal for
  !  gridding interferometer data.  the approximations are taken from
  !  vla computer memorandum no. 156.  the parameter c is related to the
  !  support width, m, of the convoluting function according to c=
  !  pi*m/2.  the parameter alpha determines a weight function in the
  !  definition of the criterion by which the function is optimal.
  !  sphfn incorporates approximations to 25 of the spheroidal func-
  !  tions, corresponding to 5 choices of m (4, 5, 6, 7, or 8 cells)
  !  and 5 choices of the weighting exponent (0, 1/2, 1, 3/2, or 2).
  !
  !  input:
  !    ialf    i*4   selects the weighting exponent, alpha.  ialf =
  !                  1, 2, 3, 4, and 5 correspond, respectively, to
  !                  alpha = 0, 1/2, 1, 3/2, and 2.
  !    im      i*4   selects the support width m, (=im) and, correspond-
  !                  ingly, the parameter c of the spheroidal function.
  !                  only the choices 4, 5, 6, 7, and 8 are allowed.
  !    iflag   i*4   chooses whether the spheroidal function itself, or
  !                  its fourier transform, is to be approximated.  the
  !                  latter is appropriate for gridding, and the former
  !                  for the u-v plane convolution.  the two differ on-
  !                  by a factor (1-eta**2)**alpha.  iflag less than or
  !                  equal to zero chooses the function appropriate for
  !                  gridding, and iflag positive chooses its f.t.
  !    eta     r*4   eta, as the argument of the spheroidal function, is
  !                  a variable which ranges from 0 at the center of the
  !                  convoluting function to 1 at its edge (also from 0
  !                  at the center of the gridding correction function
  !                  to unity at the edge of the map).
  !
  !  output:
  !    psi      r*4  the function value which, on entry to the subrou-
  !                  tine, was to have been computed.
  !    ier      i*4  an error flag whose meaning is as follows:
  !                     ier = 0  =>  no evident problem.
  !                           1  =>  ialf is outside the allowed range.
  !                           2  =>  im is outside of the allowed range.
  !                           3  =>  eta is larger than 1 in absolute
  !                                     value.
  !                          12  =>  ialf and im are out of bounds.
  !                          13  =>  ialf and eta are both illegal.
  !                          23  =>  im and eta are both illegal.
  !                         123  =>  ialf, im, and eta all are illegal.
  !
  !
  ! Dummy Variable
  !
  integer*4 ialf, im, iflag, ier
  !
  ! Local variables
  !
  integer*4 j
  real*4 alpha(5), eta, psi, eta2, x
  real*4 p4(5,5), q4(2,5), p5(7,5), q5(5), p6l(5,5), q6l(2,5),&
       p6u(5,5), q6u(2,5), p7l(5,5), q7l(2,5), p7u(5,5),&
       q7u(2,5), p8l(6,5), q8l(2,5), p8u(6,5), q8u(2,5)
  data alpha / 0., .5, 1., 1.5, 2. /
  data p4 /&
       1.584774e-2, -1.269612e-1, 2.333851e-1, -1.636744e-1,&
       5.014648e-2, 3.101855e-2, -1.641253e-1, 2.385500e-1,&
       -1.417069e-1, 3.773226e-2, 5.007900e-2, -1.971357e-1,&
       2.363775e-1, -1.215569e-1, 2.853104e-2, 7.201260e-2,&
       -2.251580e-1, 2.293715e-1, -1.038359e-1, 2.174211e-2,&
       9.585932e-2, -2.481381e-1, 2.194469e-1, -8.862132e-2,&
       1.672243e-2 /
  data q4 /&
       4.845581e-1, 7.457381e-2, 4.514531e-1, 6.458640e-2,&
       4.228767e-1, 5.655715e-2, 3.978515e-1, 4.997164e-2,&
       3.756999e-1, 4.448800e-2 /
  data p5 /&
       3.722238e-3, -4.991683e-2, 1.658905e-1, -2.387240e-1,&
       1.877469e-1, -8.159855e-2, 3.051959e-2, 8.182649e-3,&
       -7.325459e-2, 1.945697e-1, -2.396387e-1, 1.667832e-1,&
       -6.620786e-2, 2.224041e-2, 1.466325e-2, -9.858686e-2,&
       2.180684e-1, -2.347118e-1, 1.464354e-1, -5.350728e-2,&
       1.624782e-2, 2.314317e-2, -1.246383e-1, 2.362036e-1,&
       -2.257366e-1, 1.275895e-1, -4.317874e-2, 1.193168e-2,&
       3.346886e-2, -1.503778e-1, 2.492826e-1, -2.142055e-1,&
       1.106482e-1, -3.486024e-2, 8.821107e-3 /
  data q5 /&
       2.418820e-1, 2.291233e-1, 2.177793e-1, 2.075784e-1,&
       1.983358e-1 /
  data p6l /&
       5.613913e-2, -3.019847e-1, 6.256387e-1, -6.324887e-1,&
       3.303194e-1, 6.843713e-2, -3.342119e-1, 6.302307e-1,&
       -5.829747e-1, 2.765700e-1, 8.203343e-2, -3.644705e-1,&
       6.278660e-1, -5.335581e-1, 2.312756e-1, 9.675562e-2,&
       -3.922489e-1, 6.197133e-1, -4.857470e-1, 1.934013e-1,&
       1.124069e-1, -4.172349e-1, 6.069622e-1, -4.405326e-1,&
       1.618978e-1 /
  data q6l /&
       9.077644e-1, 2.535284e-1, 8.626056e-1, 2.291400e-1,&
       8.212018e-1, 2.078043e-1, 7.831755e-1, 1.890848e-1,&
       7.481828e-1, 1.726085e-1 /
  data p6u /&
       8.531865e-4, -1.616105e-2, 6.888533e-2, -1.109391e-1,&
       7.747182e-2, 2.060760e-3, -2.558954e-2, 8.595213e-2,&
       -1.170228e-1, 7.094106e-2, 4.028559e-3, -3.697768e-2,&
       1.021332e-1, -1.201436e-1, 6.412774e-2, 6.887946e-3,&
       -4.994202e-2, 1.168451e-1, -1.207733e-1, 5.744210e-2,&
       1.071895e-2, -6.404749e-2, 1.297386e-1, -1.194208e-1,&
       5.112822e-2 /
  data q6u /&
       1.101270e+0, 3.858544e-1, 1.025431e+0, 3.337648e-1,&
       9.599102e-1, 2.918724e-1, 9.025276e-1, 2.575336e-1,&
       8.517470e-1, 2.289667e-1 /
  data p7l /&
       2.460495e-2, -1.640964e-1, 4.340110e-1, -5.705516e-1,&
       4.418614e-1, 3.070261e-2, -1.879546e-1, 4.565902e-1,&
       -5.544891e-1, 3.892790e-1, 3.770526e-2, -2.121608e-1,&
       4.746423e-1, -5.338058e-1, 3.417026e-1, 4.559398e-2,&
       -2.362670e-1, 4.881998e-1, -5.098448e-1, 2.991635e-1,&
       5.432500e-2, -2.598752e-1, 4.974791e-1, -4.837861e-1,&
       2.614838e-1 /
  data q7l /&
       1.124957e+0, 3.784976e-1, 1.075420e+0, 3.466086e-1,&
       1.029374e+0, 3.181219e-1, 9.865496e-1, 2.926441e-1,&
       9.466891e-1, 2.698218e-1 /
  data p7u /&
       1.924318e-4, -5.044864e-3, 2.979803e-2, -6.660688e-2,&
       6.792268e-2, 5.030909e-4, -8.639332e-3, 4.018472e-2,&
       -7.595456e-2, 6.696215e-2, 1.059406e-3, -1.343605e-2,&
       5.135360e-2, -8.386588e-2, 6.484517e-2, 1.941904e-3,&
       -1.943727e-2, 6.288221e-2, -9.021607e-2, 6.193000e-2,&
       3.224785e-3, -2.657664e-2, 7.438627e-2, -9.500554e-2,&
       5.850884e-2 /
  data q7u /&
       1.450730e+0, 6.578685e-1, 1.353872e+0, 5.724332e-1,&
       1.269924e+0, 5.032139e-1, 1.196177e+0, 4.460948e-1,&
       1.130719e+0, 3.982785e-1 /
  data p8l /&
       1.378030e-2, -1.097846e-1, 3.625283e-1, -6.522477e-1,&
       6.684458e-1, -4.703556e-1, 1.721632e-2, -1.274981e-1,&
       3.917226e-1, -6.562264e-1, 6.305859e-1, -4.067119e-1,&
       2.121871e-2, -1.461891e-1, 4.185427e-1, -6.543539e-1,&
       5.904660e-1, -3.507098e-1, 2.580565e-2, -1.656048e-1,&
       4.426283e-1, -6.473472e-1, 5.494752e-1, -3.018936e-1,&
       3.098251e-2, -1.854823e-1, 4.637398e-1, -6.359482e-1,&
       5.086794e-1, -2.595588e-1 /
  data q8l /&
       1.076975e+0, 3.394154e-1, 1.036132e+0, 3.145673e-1,&
       9.978025e-1, 2.920529e-1, 9.617584e-1, 2.715949e-1,&
       9.278774e-1, 2.530051e-1 /
  data p8u /&
       4.290460e-5, -1.508077e-3, 1.233763e-2, -4.091270e-2,&
       6.547454e-2, -5.664203e-2, 1.201008e-4, -2.778372e-3,&
       1.797999e-2, -5.055048e-2, 7.125083e-2, -5.469912e-2,&
       2.698511e-4, -4.628815e-3, 2.470890e-2, -6.017759e-2,&
       7.566434e-2, -5.202678e-2, 5.259595e-4, -7.144198e-3,&
       3.238633e-2, -6.946769e-2, 7.873067e-2, -4.889490e-2,&
       9.255826e-4, -1.038126e-2, 4.083176e-2, -7.815954e-2,&
       8.054087e-2, -4.552077e-2 /
  data q8u /&
       1.379457e+0, 5.786953e-1, 1.300303e+0, 5.135748e-1,&
       1.230436e+0, 4.593779e-1, 1.168075e+0, 4.135871e-1,&
       1.111893e+0, 3.744076e-1 /
  !
  ! Code
  !
  ier = 0
  if (ialf.lt.1 .or. ialf.gt.5) ier = 1
  if (im.lt.4 .or. im.gt.8) ier = 2+10*ier
  if (abs(eta).gt.1.) ier = 3+10*ier
  if (ier.ne.0) then
     write(6,*) 'E-UV_SHORT, e-spheroidal, error ',ier
     return
  endif
  eta2 = eta**2
  j = ialf
  !
  ! Support width = 4 cells:
  !
  if (im.eq.4) then
     x = eta2-1.
     psi = (p4(1,j)+x*(p4(2,j)+x*(p4(3,j)+x*(p4(4,j)+x*p4(5,j)))))&
          / (1.+x*(q4(1,j)+x*q4(2,j)))
     !
     ! Support width = 5 cells:
     !
  elseif (im.eq.5) then
     x = eta2-1.
     psi = (p5(1,j)+x*(p5(2,j)+x*(p5(3,j)+x*(p5(4,j)+x*(p5(5,j)&
          +x*(p5(6,j)+x*p5(7,j)))))))&
          / (1.+x*q5(j))
     !
     ! support width = 6 cells:
  elseif (im.eq.6) then
     if (abs(eta).le..75) then
        x = eta2-.5625
        psi = (p6l(1,j)+x*(p6l(2,j)+x*(p6l(3,j)+x*(p6l(4,j)&
             +x*p6l(5,j))))) / (1.+x*(q6l(1,j)+x*q6l(2,j)))
     else
        x = eta2-1.
        psi = (p6u(1,j)+x*(p6u(2,j)+x*(p6u(3,j)+x*(p6u(4,j)&
             +x*p6u(5,j))))) / (1.+x*(q6u(1,j)+x*q6u(2,j)))
     endif
     !
     ! Support width = 7 cells:
     !
  elseif (im.eq.7) then
     if (abs(eta).le..775) then
        x = eta2-.600625
        psi = (p7l(1,j)+x*(p7l(2,j)+x*(p7l(3,j)+x*(p7l(4,j)&
             +x*p7l(5,j))))) / (1.+x*(q7l(1,j)+x*q7l(2,j)))
     else
        x = eta2-1.
        psi = (p7u(1,j)+x*(p7u(2,j)+x*(p7u(3,j)+x*(p7u(4,j)&
             +x*p7u(5,j))))) / (1.+x*(q7u(1,j)+x*q7u(2,j)))
     endif
     !
     ! Support width = 8 cells:
     !
  elseif (im.eq.8) then
     if (abs(eta).le..775) then
        x = eta2-.600625
        psi = (p8l(1,j)+x*(p8l(2,j)+x*(p8l(3,j)+x*(p8l(4,j)&
             +x*(p8l(5,j)+x*p8l(6,j)))))) / (1.+x*(q8l(1,j)+x*q8l(2,j)))
     else
        x = eta2-1.
        psi = (p8u(1,j)+x*(p8u(2,j)+x*(p8u(3,j)+x*(p8u(4,j)&
             +x*(p8u(5,j)+x*p8u(6,j)))))) / (1.+x*(q8u(1,j)+x*q8u(2,j)))
     endif
  endif
  !
  ! Normal return:
  !
  if (iflag.gt.0 .or. ialf.eq.1 .or. eta.eq.0.) return
  if (abs(eta).eq.1.) then
     psi = 0.0
  else
     psi = (1.-eta2)**alpha(ialf)*psi
  endif
  !
end subroutine sphfn
!
!------------------------------------------------------------------------
!
subroutine grdflt (ctypx, ctypy, xparm, yparm)
  !
  !     grdflt determines default parameters for the convolution functions
  !     if no convolving type is chosen, an spheroidal is picked.
  !     otherwise any unspecified values ( = 0.0) will be set to some
  !     value.
  ! arguments:
  !     ctypx,ctypy           i  convolution types for x and y direction
  !                                1 = pill box
  !                                2 = exponential
  !                                3 = sinc
  !                                4 = expontntial * sinc
  !                                5 = spheroidal function
  !     xparm(10),yparm(10)   r*4  parameters for the convolution fns.
  !                                (1) = support radius (cells)
  !
  !
  ! Dummy variables
  !
  integer ctypx,ctypy
  real*4    xparm(10), yparm(10)
  !
  ! Local variables
  !
  character*12 chtyps(5)
  integer numprm(5), i, k
  data numprm /1, 3, 2, 4, 2/
  data chtyps /'pillbox','exponential','sin(x)/(x)','exp*sinc','spheroidal'/
  !
  ! Code
  !
  ! Default type on x is 5 : spheroidal function
  !
  if ((ctypx.le.0) .or. (ctypx.gt.5)) ctypx = 5
  !
  ! Case of x function type
  !
  select case (ctypx)
     !
  case(1)
     !
     ! Pillbox
     !
     if (xparm(1).le.0.0) xparm(1) = 0.5
  case(2)
     !
     ! Exponential
     !
     if (xparm(1).le.0.0) xparm(1) = 3.0
     if (xparm(2).le.0.0) xparm(2) = 1.00
     if (xparm(3).le.0.0) xparm(3) = 2.00
  case(3)
     !
     ! sinc.
     !
     if (xparm(1).le.0.0) xparm(1) = 3.0
     if (xparm(2).le.0.0) xparm(2) = 1.14
  case(4)
     !
     ! exponential * sinc
     !
     if (xparm(1).le.0.0) xparm(1) = 3.0
     if (xparm(2).le.0.0) xparm(2) = 1.55
     if (xparm(3).le.0.0) xparm(3) = 2.52
     if (xparm(4).le.0.0) xparm(4) = 2.00
  case(5)
     !
     ! Spheroidal function
     !
     if (xparm(1).le.0.0) xparm(1) = 3.0
     if (xparm(2).le.0.0) xparm(2) = 1.0
     !
     !
  end select
  !
  ! Case of y function type
  !
  !
  ! Note that default type on y is same type than on x
  !
  select case (ctypy)
  case(1)
     !
     ! Pillbox
     !
     if (yparm(1).le.0.0) yparm(1) = 0.5
  case(2)
     !
     ! Exponential
     !
     if (yparm(1).le.0.0) yparm(1) = 3.0
     if (yparm(2).le.0.0) yparm(2) = 1.0
     if (yparm(3).le.0.0) yparm(3) = 2.0
  case(3)
     !
     ! sinc
     !
     if (yparm(1).le.0.0) yparm(1) = 3.0
     if (yparm(2).le.0.0) yparm(2) = 1.14
  case(4)
     !
     ! exponential * sinc
     !
     if (yparm(1).le.0.0) yparm(1) = 3.0
     if (yparm(2).le.0.0) yparm(2) = 1.55
     if (yparm(3).le.0.0) yparm(3) = 2.52
     if (yparm(4).le.0.0) yparm(4) = 2.00
  case(5)
     !
     ! Spheroidal function
     !
     if (yparm(1).le.0.0) yparm(1) = 3.0
     if (yparm(2).le.0.0) yparm(2) = 1.0
     !
  case default
     !
     ! Default :  use x values
     !
     ctypy = ctypx
     do i = 1,10
        yparm(i) = xparm(i)
     enddo
     !
  end select
  !
  ! Print parameters chosen.
  !
  write(6,*) 'I-UV_SHORT, Convolution'
  write(6,1001) 'x',chtyps(ctypx),(xparm(k),k=1,numprm(ctypx))
  write(6,1001) 'y',chtyps(ctypy),(yparm(k),k=1,numprm(ctypy))
1001 format(1x,a,' convolution ',a,' par.=',5f8.4)
!
end subroutine grdflt
!-------------------------------------------------------------------------
!
subroutine finsiz (x,nd,np,ix,iy,we,xmin,xmax,ymin,ymax)
  !
  ! Find extrema xmin, xmax in ix column values
  !          and ymin, ymax in iy column values,
  ! in table x(nd,np) for points where weight is not null
  ! taking in account that table x is ordered on iy column values.
  !
  ! Dummy variables
  !
  integer nd,np,ix,iy
  real x(nd,np),xmin,xmax,ymin,ymax,we(np)
  !
  ! Local variables
  !
  integer i,j
  !
  ! Code
  !
  i = 1
  !
  ! Loop to start after null weights measurements
  !
  do while (we(i).eq.0)
     i = i+1
  enddo
  !
  ! ymin is first y value with weight not null
  !
  ymin = x(iy,i)
  !
  ! initialize xmin and xmax for searching loop
  !
  xmin = x(ix,i)
  xmax = x(ix,i)
  i = i+1
  !
  ! Loop on table lines to find xmin and xmax
  !
  do j=i,np
     if (we(j).ne.0) then
        if (x(ix,j).lt.xmin) then
           xmin = x(ix,j)
        elseif (x(ix,j).gt.xmax) then
           xmax = x(ix,j)
        endif
     endif
  enddo
  i = np
  !
  ! Loop to find ymax = last y values with weight not null
  !
  do while (we(i).eq.0)
     i = i-1
  enddo
  ymax = x(iy,i)
  !
end subroutine finsiz
!
!-------------------------------------------------------------------------
!
subroutine finsiy (x,nd,np,iy,we,ymin,ymax)
  !
  ! Find extrema ymin and ymax of iy column values
  ! in iy ordered table x(nd,np),
  ! taking in acount only measurements where weights we are not null
  !
  ! Dummy variables
  !
  integer nd,np,iy
  real x(nd,np),ymin,ymax,we(np)
  !
  ! Local variables
  !
  integer i
  !
  ! Code
  !
  !
  ! Start searching ymin at the beginning of the table
  !
  i = 1
  do while (we(i).eq.0)
     i = i+1
  enddo
  ymin = x(iy,i)
  !
  ! Start searching ymax at the end of the table
  !
  i = np
  do while (we(i).eq.0)
     i = i-1
  enddo
  ymax = x(iy,i)
  !
end subroutine finsiy
!
!-------------------------------------------------------------------------
subroutine fininc (x,nd,np,ix,iy,we,xinc,yinc,tole)
  !
  ! Find increments xinc and yinc
  ! corresponding to smallest distance (greater than tole) on x and y axes
  ! between to points of measurements in x(nd,np) table with no null weights
  !
  ! Dummy variables
  !
  integer nd,np,ix,iy
  real x(nd,np),xinc,yinc,tole,we(np)
  !
  ! Local variables
  !
  integer i,j
  real dist
  !
  ! Code
  !
  do i=1,np-1
     if (we(i).ne.0) then
        do j=i+1,np
           if (we(j).ne.0) then
              dist = abs(x(ix,j)-x(ix,i))
              if (dist.gt.tole .and. dist.lt.xinc) xinc = dist
              dist = x(iy,j)-x(iy,i)
              if (dist.gt.tole .and. dist.lt.yinc) yinc = dist
           endif
        enddo
     endif
  enddo
  !
end subroutine fininc
!
!-------------------------------------------------------------------------
!
subroutine dosor (visi,nd,np,we,iy)
  use gkernel_interfaces
  !
  ! Output visi(nd,np) will contain ycol column values in increasing order
  ! Use procedure trione
  ! we is used as work space only
  !
  use gildas_def
  ! Dummy variables
  !
  integer nd,np,iy
  real visi(nd,np),we(nd)            ! corrected 22-mar-1995
  !
  ! Local variables
  !
  integer i,ier,trione
  !
  ! Code
  !
  do i=1,np-1
     if (visi(iy,i).gt.visi(iy,i+1)) then
        write(6,*) 'I-UV_SHORT, Sorting input table'
        ier = trione (visi,nd,np,iy,we)
        if (ier.ne.1) call sysexi (fatale)
        return
     endif
  enddo
  write(6,*) 'I-UV_SHORT, Input table is sorted'
  !
end subroutine dosor
!
!-------------------------------------------------------------------------
function trione (x,nd,n,ix,work)
  !
  !   sorting program that uses a quicksort algorithm.
  ! sort on one row
  ! x r*4(*)  unsorted array        input
  ! nd  i first dimension of x      input
  ! n i second dimension of x     input
  ! ix  i x(ix,*) is the key for sorting    input
  ! work  r*4(nd) work space for exchange     input
  !
  !
  ! Dummy variables
  !
  integer nd,n,ix
  real*4 x(nd,n), work(nd)
  !
  ! Local variables
  !
  integer trione, maxstack, nstop
  parameter (maxstack=1000,nstop=15)
  integer*4 i, j, k, l1, r1, l, r, m
  integer*4 lstack(maxstack), rstack(maxstack), sp
  real*4 key
  logical mgtl, lgtr, rgtm
  !
  ! Code
  !
  trione = 1
  if (n.le.nstop) goto 50
  sp = 0
  sp = sp + 1
  lstack(sp) = 1
  rstack(sp) = n
  !
  ! Sort a subrecord off the stack
  ! set key = median of x(l), x(m), x(r)
  ! no! this is not reasonable, as systematic very inequal partitioning will
  ! occur in some cases (especially for nearly already sorted files)
  ! to fix this problem, i found (but i cannot prove it) that it is best to
  ! select the estimation of the median value from intermediate records. p.v.
  !
1 l = lstack(sp)
  r = rstack(sp)
  sp = sp - 1
  m = (l + r) / 2
  l1=(2*l+r)/3
  r1=(l+2*r)/3
  !
  mgtl = x(ix,m) .gt. x(ix,l)
  rgtm = x(ix,r) .gt. x(ix,m)
  !
  ! Algorithm to select the median key. the original one from mongo
  ! was completely wrong. p. valiron, 24-jan-84 .
  !
  !       mgtl  rgtm  lgtr  mgtl.eqv.lgtr median_key
  !
  ! kl < km < kr  t t * *   km
  ! kl > km > kr  f f * *   km
  !
  ! kl < km > kr  t f f f   kr
  ! kl < km > kr  t f t t   kl
  !
  ! kl > km < kr  f t f t   kl
  ! kl > km < kr  f t t f   kr
  !
  if (mgtl .eqv. rgtm) then
     key = x(ix,m)
  else
     lgtr = x(ix,l) .gt. x(ix,r)
     if (mgtl .eqv. lgtr) then
        key = x(ix,l)
     else
        key = x(ix,r)
     endif
  endif
  i = l
  j = r
  !
  ! Find a big record on the left
  !
10 if (x(ix,i).ge.key) goto 11
  i = i + 1
  goto 10
11 continue
  !
  ! Find a small record on the right
  !
20 if (x(ix,j).le.key) goto 21
  j = j - 1
  goto 20
21 continue
  if (i.ge.j) goto 2
  !
  ! Exchange records
  !
  call r4tor4 (x(1,i),work,nd)
  call r4tor4 (x(1,j),x(1,i),nd)
  call r4tor4 (work,x(1,j),nd)
  i = i + 1
  j = j - 1
  goto 10
  !
  ! Subfile is partitioned into two halves, left .le. right
  ! push the two halves on the stack
  !
2 continue
  if (j-l+1 .gt. nstop) then
     sp = sp + 1
     if (sp.gt.maxstack) then
        write(6,*) 'E-UV_SHORT, Stack overflow ',sp
        trione = 0
        return
     endif
     lstack(sp) = l
     rstack(sp) = j
  endif
  if (r-j .gt. nstop) then
     sp = sp + 1
     if (sp.gt.maxstack) then
        write(6,*) 'E-UV_SHORT, Stack overflow ',sp
        trione = 0
        return
     endif
     lstack(sp) = j+1
     rstack(sp) = r
  endif
  !
  ! anything left to process?
  !
  if (sp.gt.0) goto 1
  !
50 continue
  !
  do 110 j = n-1,1,-1
     k = j
     do i = j+1,n
        if (x(ix,j).le.x(ix,i)) goto 121
        k = i
     enddo
121  continue
     if (k.eq.j) goto 110
     call r4tor4 (x(1,j),work,nd)
     do i = j+1,k
        call r4tor4 (x(1,i),x(1,i-1),nd)
     enddo
     call r4tor4 (work,x(1,k),nd)
110  continue
  !
end function trione
!
!-------------------------------------------------------------------------
!
subroutine doapod (xmin,xmax,ymin,ymax,tole,beam,&
     nc,nx,ny,map,raw,mapx,mapy, weight,wmin)
  !
  ! Replace map edges and bad quality values of raw input data cube
  ! with smoothed values contains in mapx data cube.
  ! Map edges corresponding to the part of the map between
  ! max SD observations locations xmin,xmax,ymin,ymax, and map size nx,ny
  ! Bad quality values corresponding to weights < wmin
  !
  ! Dummy variables
  !
  integer nc,nx,ny
  real mapx(nx),mapy(ny)
  real map(nc,nx,ny)
  real raw(nc,nx,ny)
  real beam,tole,xmin,xmax,ymin,ymax
  real weight(nx,ny),wmin
  !
  ! Local variables
  !
  integer ix,iy
  real lobe,apod,disty,distx
  !
  ! Code
  !
  ! Tests
  !
  ! pi = acos(-1.0)
  ! write(6,*) 'I-UV_SHORT, do apodisation :'
  ! write(6,*) 'min-max ',xmin,xmax,ymin,ymax
  ! write(6,*) 'beam et inc ',beam*180*3600/pi,tole*180*3600/pi&
  !      ,(mapx(1)-mapx(2))*180*3600/pi
  !
  ! Apodisation by a gaussian, twice as large than the SD beam
  !
  lobe = log(2.0)/beam**2
  !
  ! Loop on pixels
  !
  do iy=1,ny
     !
     ! Loop on Y
     ! Compute disty : distance between map size ny
     ! and ymin or ymax SD observations limits
     !
     if (mapy(iy).le.ymin-tole) then
        disty = ymin-mapy(iy)
     elseif (mapy(iy).ge.ymax+tole) then
        disty = mapy(iy)-ymax
     else
        disty = 0.0
     endif
     do ix=1,nx
        !
        ! Idem on X, compute distx
        !
        if (mapx(ix).le.xmin-tole) then
           distx = xmin-mapx(ix)
        elseif (mapx(ix).ge.xmax+tole) then
           distx = mapx(ix)-xmax
        else
           distx = 0.0
        endif
        !
        ! Apodisation factor
        !
        apod = (distx**2+disty**2)*lobe
        !
        ! 'raw' is replaced by something else only in two cases
        !
        if (apod.gt.80) then
           !
           ! Map edges
           !
           raw(1:nc,ix,iy) = 0.0
        elseif (apod.ne.0.0) then
           !
           ! Map edges
           !
           apod = exp(-apod)
           raw(1:nc,ix,iy) = map(1:nc,ix,iy)*apod
        elseif (weight(ix,iy).lt.wmin) then
           !
           ! Low weight point within the map
           !
           raw(1:nc,ix,iy) = map(1:nc,ix,iy)
        endif
     enddo
  enddo
  !
end subroutine doapod
!
!-------------------------------------------------------------------------
!
subroutine dosdft(beam,diam,f,nx,ny,fact,lmv)
  !
  ! computes inverse of ft of single-dish beam
  ! (uses a gaussian truncated at dish size)
  !
  !     beam     beam size in radian
  !     diam     diameter in meter
  !     f(nx,ny) multiplication factor
  !     nx,ny    size of problem
  !
  !
  ! Dummy variables
  !
  use image_def
  type (gildas) :: lmv
  !
  integer nx, ny
  real f(nx,ny), beam, diam,fact
  !
  ! Local variables
  !
  real(8), parameter :: pi=3.141592653589793d0
  real(8), parameter :: clight=299792458d-6    ! frequency in mhz
  real*8 lmvfreq, lmvinc1, lmvinc2, lmvdim1, lmvdim2
  integer i,j, ii, jj
  real a, b, xx, yy
  real*8 dx, dy
  !
  ! Code
  !
  ! Pick up usefull parameters :freq, inc and dim, in lmv image header
  !
  lmvfreq=lmv%gil%freq
  lmvinc1=lmv%gil%inc(1)
  lmvinc2=lmv%gil%inc(2)
  lmvdim1=lmv%gil%dim(1)
  lmvdim2=lmv%gil%dim(2)
  !
  dx = clight/lmvfreq/(lmvinc1*lmvdim1)
  dy = clight/lmvfreq/(lmvinc2*lmvdim2)
  b = (pi*beam/2/clight*lmvfreq)**2/alog(2.)
  !
  ! Equivalent beam area in square pixels ...
  !
  a = abs(4*alog(2.)/pi/beam**2*lmvinc2*lmvinc1)*fact
  !
  ! Loop on pixels
  !
  do j = 1, ny
     !
     ! Loop on Y, pixels locations on Fourier plane
     !
     jj = mod(j-1+ny/2,ny)-ny/2
     yy = ( jj*dy )**2
     do i = 1, nx
        !
        ! Loop on X, pixels locations on Fourier plane
        !
        ii = mod(i-1+nx/2,nx)-nx/2
        xx = ( ii*dx )**2
        !
        ! Truncation of the gaussian at diam
        !
        if (xx+yy.le.diam**2) then
           f(i,j) = exp(b*(xx+yy))*a
        else
           f(i,j) = 0.0
        endif
     enddo
  enddo
  !
end subroutine dosdft
!
!--------------------------------------------------------------------------
!
subroutine dointft(beam,diam,f,nx,ny,fact,lmv)
  !
  ! computes ft of single-dish beam
  ! (uses a gaussian truncated at dish size)
  !
  !     beam     beam size in radian
  !     diam     diameter in meter
  !     f(nx,ny) multiplication factor
  !     nx,ny    size of problem
  !
  ! Dummy variables
  !
  use image_def
  type (gildas) :: lmv
  integer nx, ny
  real f(nx,ny), beam, diam,fact
  !
  ! Local variables
  !
  real(8), parameter :: pi=3.141592653589793d0
  real(8), parameter :: clight=299792458d-6    ! frequency in mhz
  real*8 lmvfreq,lmvinc1,lmvinc2,lmvdim1,lmvdim2
  integer i,j, ii, jj
  real a, b, xx, yy
  real*8 dx, dy
  !
  ! Code
  !
  ! Pick up usefull parameters :freq, inc and dim, in lmv image header
  !
  lmvfreq=lmv%gil%freq
  lmvinc1=lmv%gil%inc(1)
  lmvinc2=lmv%gil%inc(2)
  lmvdim1=lmv%gil%dim(1)
  lmvdim2=lmv%gil%dim(2)
  !
  dx = clight/lmvfreq/(lmvinc1*lmvdim1)
  dy = clight/lmvfreq/(lmvinc2*lmvdim2)
  b = (pi*beam/2/clight*lmvfreq)**2/alog(2.)
  !
  ! Equivalent beam area in square pixels ...
  !
  a = abs(pi*beam**2/lmvinc2/lmvinc1)*fact/abs(4*alog(2.))
  !
  ! Loop on pixels
  !
  do j = 1, ny
     !
     ! Loop on Y, pixels locations on Fourier plane
     !
     jj = mod(j-1+ny/2,ny)-ny/2
     yy = ( jj*dy )**2
     do i = 1, nx
        !
        ! Loop on X, pixels locations on Fourier plane
        !
        ii = mod(i-1+nx/2,nx)-nx/2
        xx = ( ii*dx )**2
        !
        ! Truncation of the gaussian at diam
        !
        if (xx+yy.le.diam**2) then
           f(i,j) = exp(-b*(xx+yy))*a
        else
           f(i,j) = 0.0
        endif
     enddo
  enddo
!
end subroutine dointft
!
!-------------------------------------------------------------------------
!
subroutine shift(f,nx,ny,offra,offdec,lmv)
  !
  ! Centered interferometer beam f(nx,ny) of a mosaic field
  ! on is right position : offra and offdec shifted in uv plane
  !
  ! Global variables
  !
  use image_def
  !
  ! Dummy variables
  !
  type (gildas) :: lmv
  integer nx, ny
  complex :: f(nx,ny)
  real offra, offdec
  !
  real(8), parameter :: pi=3.141592653589793d0
  !
  ! Local variables
  !
  integer i, j, ii, jj
  real phi, sp, cp, xx, yy, re, im
  real*8 lmvfreq, lmvinc1, lmvinc2, lmvdim1, lmvdim2
  real*8 clight
  parameter (clight=299792458d-6)    ! frequency in mhz
  real*8 du, dv
  !
  ! Code
  !
  ! Pick up usefull parameters in lmv image header : freq, inc and dim
  !
  lmvfreq=lmv%gil%freq
  lmvinc1=lmv%gil%inc(1)
  lmvinc2=lmv%gil%inc(2)
  lmvdim1=lmv%gil%dim(1)
  lmvdim2=lmv%gil%dim(2)
  !
  du = 1.d0/(lmvinc1*lmvdim1)  ! clight/lmvfreq/(lmvinc1*lmvdim1)
  dv = 1.d0/(lmvinc2*lmvdim2)  ! clight/lmvfreq/(lmvinc2*lmvdim2)
  !
  ! Loop on pixels
  !
  do j = 1, ny
     !
     ! Loop on Y, pixels locations on Fourier plane
     !
     jj = mod(j-1+ny/2,ny)-ny/2
     yy = jj*dv
     do i = 1, nx
        !
        ! Loop on X, pixels locations on Fourier plane
        !
        ii = mod(i-1+nx/2,nx)-nx/2
        xx = ii*du
        !
        phi = -2*pi*(offra*xx + offdec*yy)
        cp = cos(phi)
        sp = sin(phi)
        !
        re = real(f(i,j))*cp - imag(f(i,j))*sp
        im = real(f(i,j))*sp + imag(f(i,j))*cp
        f(i,j) = cmplx(re,im)
     enddo
  enddo
  !
end subroutine shift
!
!-------------------------------------------------------------------------
!
subroutine dotrans (a,b,n,m)
  !
  ! Output table"b" is table "a" transposed in line/column order
  !
  !
  ! Dummy variables
  !
  integer n,m
  real a(n,m),b(m,n)
  !
  ! Local variables
  !
  integer i,j
  !
  ! Code
  !
  do i=1,m
     do j=1,n
        b(i,j) = a(j,i)
     enddo
  enddo
end subroutine dotrans
!
!-------------------------------------------------------------------------
!
subroutine sdcorr(z,f,nxy)
  !
  ! Update z(nxy) table with z*f, calculate in image plane
  ! Used to correct for single dish beam
  !
  ! Dummy variables
  !
  integer nxy
  complex z(*)
  real f(*)
  !
  ! Local variables
  !
  integer i
  !
  ! Code
  !
  do i = 1, nxy
     z(i) = z(i)*f(i)
  enddo
  !
end subroutine sdcorr
!
!-------------------------------------------------------------------------
!
subroutine retocm(r,z,nx,ny)
  !
  ! Real to complex
  ! input real r(nx,ny) will update real part of complex output z(nx,ny)
  !
  ! Dummy variables
  !
  integer nx, ny
  real r(nx,ny)
  complex z(nx,ny)
  !
  ! Local variables
  !
  integer i, j, ii, jj
  !
  ! Code
  !
  do i=1, nx
     ii = mod(i+nx/2-1,nx)+1
     do j=1, ny
        jj = mod(j+ny/2-1,ny)+1
        z(ii,jj) = r(i,j)
     enddo
  enddo
  !
end subroutine retocm
!
!-------------------------------------------------------------------------
!
subroutine cmtore(z,r,nx,ny)
  !
  ! Convert complex to real
  ! input complex table z(nx,ny) real part will update real r(nx,ny) output
  !
  ! Dummy variables
  !
  integer nx, ny
  real r(nx,ny)
  complex z(nx,ny)
  !
  ! Local variables
  !
  integer i, j, ii, jj
  !
  ! Code
  !
  do i=1, nx
     ii = mod(i+nx/2-1,nx)+1
     do j=1, ny
        jj = mod(j+ny/2-1,ny)+1
        r(ii,jj) = z(i,j)
     enddo
  enddo
  !
end subroutine cmtore
!
!-------------------------------------------------------------------------
!
subroutine mask(a,w,n,wm)
  !
  ! Variables
  !
  integer n
  real a(n),w(n),wm
  !
  ! Local variables
  !
  integer i
  !
  ! Code
  !
  do i=1,n
     if (w(i).lt.wm) a(i) = 0.0
  enddo
  !
end subroutine mask
!
!-------------------------------------------------------------------------
!
subroutine prmult(z,f,nx,ny)
  !
  ! Update z(nx,ny) table with z*f, calculate in uv plane
  ! Used to multiply by interferometer primary beam
  !
  ! Dummy variables
  !
  integer nx, ny
  complex z(nx, ny)
  real f(nx,ny)
  !
  ! Local variables
  !
  integer i, j, ii, jj
  !
  ! Code
  !
  do j = 1, ny
     jj = mod(j+ny/2-1,ny)+1
     do i = 1, nx
        ii = mod(i+nx/2-1,nx)+1
        z(ii,jj) = z(ii,jj) * f(i,j)
     enddo
  enddo
  !
end subroutine prmult
!
!-------------------------------------------------------------------------
!
subroutine uvcount(nx,ny,nvis,diam,lmv)
  !
  ! Compute number of visibilities nvis
  ! on a regular grid dx,dy sampled and inside the dish defined by diam
  ! nx and ny needed cause uvcount is working in Fourier plane
  !
  ! Dummy variables
  !
  use image_def
  type (gildas) :: lmv
  integer nx, ny, nvis
  real diam
  !
  ! Local variables
  !
  integer i, j, ii, jj
  real uu, vv
  real*8 clight, lmvfreq, lmvinc1, lmvinc2, lmvdim1, lmvdim2
  parameter (clight=299792458d-6)    ! frequency in mhz
  real*8 dx, dy
  !
  ! Code
  !
  !
  ! Pick up usefull parameters :freq, inc and dim, in lmv image header
  !
  lmvfreq=lmv%gil%freq
  lmvinc1=lmv%gil%inc(1)
  lmvinc2=lmv%gil%inc(2)
  lmvdim1=lmv%gil%dim(1)
  lmvdim2=lmv%gil%dim(2)
  !
  dx = clight/lmvfreq/(lmvinc1*lmvdim1)
  dy = clight/lmvfreq/(lmvinc2*lmvdim2)
  !
  nvis = 0
  do j = 1, ny
     jj = mod(j-1+ny/2,ny)-ny/2
     vv = jj*dy
     do i = 1, nx/2
        ii = mod(i-1+nx/2,nx)-nx/2
        uu = ii*dx
        if (uu**2+vv**2.le.diam**2) then
           nvis = nvis + 1
        endif
     enddo
  enddo
  !
end subroutine uvcount
!
!--------------------------------------------------------------------------
!
subroutine uvtable(nx,ny,nc,v,w,ww,nvis,diam,wfactor,factor,lmv)
  !
  ! Tabulate the visibilities
  !
  ! Global variables
  !
  use image_def
  !
  ! Dummy variables
  !
  type (gildas) :: lmv
  !
  integer nx, ny, nc, nvis
  complex :: v(nx,ny,nc), ww(nx,ny)
  real w(7+3*nc,nvis)
  real wfactor, diam, factor
  !
  ! Local variables
  !
  real(8), parameter :: pi=3.141592653589793d0
  real(8), parameter :: clight=299792458d-6    ! frequency in mhz
  integer i, j, k ,kk, ii, jj, kvis, k00
  real uu, vv, sw, we, duv, wfact
  real*8 lmvfreq, lmvinc1, lmvinc2, lmvdim1, lmvdim2
  real*8 dx, dy
  !
  !
  ! Code
  !
  !
  ! Pick up usefull parameters :freq, inc and dim, in lmv image header
  !
  lmvfreq=lmv%gil%freq
  lmvinc1=lmv%gil%inc(1)
  lmvinc2=lmv%gil%inc(2)
  lmvdim1=lmv%gil%dim(1)
  lmvdim2=lmv%gil%dim(2)
  !
  dx = clight/lmvfreq/(lmvinc1*lmvdim1)
  dy = clight/lmvfreq/(lmvinc2*lmvdim2)
  !
  wfact = wfactor/(factor**2)
  kvis = 0
  sw = 0
  !
  ! Loop on pixels of the visibility map
  !
  do j = 1, ny
     jj = mod(j-1+ny/2,ny)-ny/2
     vv = jj*dy
     do i = 1, nx/2
        ii = mod(i-1+nx/2,nx)-nx/2
        uu = ii*dx
        duv = uu**2+vv**2
        !
        ! Keep only points inside circle defined by diam
        !
        if (duv.le.diam**2) then
           kvis = kvis + 1
           w(1,kvis) = uu
           w(2,kvis) = vv
           w(3,kvis) = 0
           w(4,kvis) = 0
           w(5,kvis) = 0
           w(6,kvis) = 0
           w(7,kvis) = 0
           kk = 7
           !
           ! Weight
           !
           we = real(ww(i,j))
           if (i.eq.1 .and. j.ne.1) then
              we = we*0.5
           endif
           if (we.lt.0) we = -we
           !
           ! u=0 v=0 point
           !
           if (duv.eq.0) k00 = kvis
           !
           ! Extract visibilities
           ! - apply K-to-Jy conversion factor
           ! - wfact = wfactor/factor**2
           !
           do k=1, nc
              w(kk+1,kvis) = real(v(i,j,k))*factor
              w(kk+2,kvis) = imag(v(i,j,k))*factor
              w(kk+3,kvis) = we*wfact
              kk = kk + 3
           enddo
           sw = sw+we*wfact
        endif
     enddo
  enddo
  !
  ! Test number of visibilities
  !
  if (kvis.ne.nvis) then
     write (6,*) 'W-UV_SHORT, Inconsistent number of visibilities'
  endif

  ! weighting
  ! na 1 use natural weight
  ! un 2 use "uniform" weight
  ! au 3 recompute natural weight
  !
  ! normalize
  ! sw = truesw / sw / wfact
  !         sw = weight/sw/wfact   ! weight = new total weight
  !         sw = weight/wfact      ! weight = weight factor
  !if (weight_mode.eq.'au') then
  !
  ! 3 pass 3 sigma filtering to eliminate the lines (if any...)
  ! won't work with strong continuum... to be re-written
  !
  !  rms = 1e30
  ! do j =1,3
  !   nd = 0
  !  sw = 0
  ! do i=1,nc
  !   amp = w(5+3*i,k00)**2+w(6+3*i,k00)**2
  !  if (amp.le.9*rms) then
  !    sw = sw+amp
  !   nd = nd+1
  !         endif
  !     enddo
  !    rms = sw/nd
  !     enddo
  !   sw = 0.5e-6*(nd-1)/rms/w(10,k00)
  !endif
  !sw = weight * sw                   ! weight is now a scaling factor


  !
  ! Normalize the weights **WW**
  !
  if (sw.ne.0.) then
     sw = 1/sw
  endif
  do i=1, nvis
     do k=1, nc
        w(7+k*3,i) = w(7+k*3,i)*sw
     enddo
  enddo
!
end subroutine uvtable
!
!-------------------------------------------------------------------------
!
subroutine doprim(beam,f,nx,ny,dra,dde,lmv)
  !
  ! *** no longer used in uv_short.f90
  ! Ouptput f(nx,ny) will contain gaussian interferometer primary beam
  ! computed at distance dra,dde of the center of the mosaic
  !
  ! Global variables
  !
  use image_def
  type (gildas) ::lmv
  !
  ! Dummy variables
  !
  integer nx, ny
  real f(nx,ny), beam, dra, dde
  !
  ! Local variables:
  !
  real b, xx, yy, a, dd
  integer i, j
  real*8 lmvref1,lmvref2,lmvval1,lmvval2,lmvinc1,lmvinc2
  !
  ! Code
  !
  ! Pick up usefull parameters in lmv image header : freq, inc and dim
  !
  lmvref1=lmv%gil%ref(1)
  lmvval1=lmv%gil%val(1)
  lmvinc1=lmv%gil%inc(1)
  lmvref2=lmv%gil%ref(2)
  lmvval2=lmv%gil%val(2)
  lmvinc2=lmv%gil%inc(2)
  !
  b = 4.*alog(2.)/beam**2
  a = 1.0/(nx*ny)
  !
  ! Loop on pixels
  !
  do j = 1, ny
     yy = (lmvval2 + (j-lmvref2)*lmvinc2 - dde)**2
     do i = 1, nx
        xx = ( lmvval1 + (i-lmvref1)*lmvinc1 - dra)**2
        dd = b*(xx+yy)
        if (dd.gt.80.0) then
           f(i,j) = 0.0
        else
           f(i,j) = a*exp(-dd)
        endif
     enddo
  enddo
  !
end subroutine doprim
!
!-------------------------------------------------------------------------
!
subroutine dopoint(data,nd,np,xcol,ycol,old,new)
  !
  ! Recompute data(nd,np) xcol and ycol coordinates values
  ! in case of changing reference position a0 d0 from old to new
  !
  ! Dummy variables
  !
  integer nd,np,xcol,ycol
  real data(nd,np)
  real*8 old(2),new(2)
  !
  ! Local variables
  !
  real*8 dra,dde,ra,de,uncde,cde
  integer i
  !
  !
  ! Code
  !
  uncde = 1.d0/cos(old(2))
  cde = cos(new(2))
  !
  do i=1,np
     ra = old(1) + dble(data(xcol,i))*uncde
     de = old(2) + dble(data(ycol,i))
     dra = (ra - new(1)) * cde
     dde = de - new(2)
     data(xcol,i) = dra
     data(ycol,i) = dde
  enddo
!
end subroutine dopoint
!
!-------------------------------------------------------------------------
!
subroutine wcorr(z,f,n)
  !
  ! Update z(n) table with z/f**2 , calculate in image plane
  ! Use to correct weights values from single-dish lobe
  !
  ! Dummy variables
  !
  integer n
  complex z(n)
  real f(n)
  !
  ! Local variables
  !
  integer i
  !
  ! Code
  !
  do i = 1, n
     if (f(i).ne.0) then
        z(i) = z(i)/(f(i)*f(i))
     else
        z(i) = 0.0
     endif
  enddo
  !
end subroutine wcorr
!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!! The end !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
subroutine  copy(out, in, nx,ny,nc)
integer nx, ny, nc,i
real out(nx,ny,nc)
real in(nx,ny)
   do i = 1,nc
   out(1:nx,1:ny,i) = in(1:nx,1:ny)
enddo
end subroutine copy
