module mapping_interfaces_private
  interface
    subroutine mrc_clean (line,error)
      use clean_def
      use clean_arrays
      use gbl_message
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING Internal routine
      !   Implementation of Multi Resolution Clean
      !----------------------------------------------------------------------
      character(len=*), intent(inout) :: line
      logical, intent(out) :: error
    end subroutine mrc_clean
  end interface
  !
  interface
    subroutine multi_clean (line,error)
      use clean_def
      use clean_arrays
      use gbl_message
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING Internal routine
      !   Implementation of Multi Scale Clean
      !----------------------------------------------------------------------
      character(len=*), intent(inout) :: line
      logical, intent(out) :: error
    end subroutine multi_clean
  end interface
  !
  interface
    subroutine sdi_clean (line,error)
      use clean_def
      use clean_arrays
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING Internal routine
      !   Implementation of Steer Dewdney Ito Clean
      !----------------------------------------------------------------------
      character(len=*), intent(inout) :: line
      logical, intent(out) :: error
    end subroutine sdi_clean
  end interface
  !
  interface
    subroutine hogbom_clean (line,error)
      use clean_def
      use clean_arrays
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING Internal routine
      !   Implementation of Hogbom Clean
      !----------------------------------------------------------------------
      character(len=*), intent(inout) :: line
      logical, intent(out) :: error
    end subroutine hogbom_clean
  end interface
  !
  interface
    subroutine clark_clean (line,error)
      use clean_def
      use clean_arrays
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING Internal routine
      !   Implementation of Barry Clark Clean
      !----------------------------------------------------------------------
      character(len=*), intent(inout) :: line
      logical, intent(out) :: error
    end subroutine clark_clean
  end interface
  !
  interface
    subroutine sub_clean (line,error)
      use clean_def
      use clean_arrays
      use clean_types
      use gbl_message
      !$  use omp_lib
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING Internal routine
      !     Implementation of all standard CLEAN deconvolution algorithms,
      !----------------------------------------------------------------------
      character(len=*), intent(inout) :: line
      logical, intent(out) :: error
    end subroutine sub_clean
  end interface
  !
  interface
    subroutine clean_data(error)
      use clean_def
      use clean_arrays
      use gbl_message
      !--------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !    Prepare Clean parameters
      !--------------------------------------------------------
      logical, intent(out) :: error
    end subroutine clean_data
  end interface
  !
  interface
    subroutine write_image (line,error)
      use clean_def
      use clean_arrays
      use clean_types
      use gbl_format
      use gbl_message
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Support routine for command
      !   WRITE Type File [/RANGE Start End Kind] [/TRIM]
      !       [/APPEND] [/REPLACE]
      !----------------------------------------------------------------------
      character(len=*), intent(in) :: line
      logical, intent(out) :: error
    end subroutine write_image
  end interface
  !
  interface
    subroutine sub_write_image (name,ntype,nc,error)
      use clean_def
      use clean_arrays
      use clean_support
      use clean_types
      use gbl_format
      use gbl_message
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Support routine for command
      !   WRITE Type File [/RANGE Start End Kind] [/TRIM]
      !
      ! Dispatch the writing operation to specific subroutines depending
      ! on the Type of data to be written
      !----------------------------------------------------------------------
      character(len=*), intent(in) :: name
      integer, intent(in) :: nc(2)
      integer, intent(in) :: ntype
      logical, intent(out) :: error
    end subroutine sub_write_image
  end interface
  !
  interface
    subroutine sub_setmodif (file,opti,nc)
      use gkernel_types
      use clean_types
      !---------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Update the Read / Write optimization status
      !---------------------------------------------------------------------
      character(len=*), intent(in) :: file  ! Filename
      type(readop_t), intent(inout) :: opti ! Status of corresponding buffer
      integer, intent(in) :: nc(2)          ! Range to be read
    end subroutine sub_setmodif
  end interface
  !
  interface
    subroutine write_cct(file,error)
      use clean_def
      use clean_arrays
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Support routine for command
      !   WRITE CCT File [/RANGE Start End Kind]
      !----------------------------------------------------------------------
      !
      character(len=*), intent(in) :: file
      logical, intent(out) :: error
    end subroutine write_cct
  end interface
  !
  interface
    subroutine sub_write_uvdata(file,nc,uvin,duv,error, mv, ivis)
      use image_def
      use gbl_message
      !---------------------------------------------------------------------
      ! @ private-mandatory
      !
      ! MAPPING
      !   Write a subset of the loaded UV data
      !---------------------------------------------------------------------
      character(len=*), intent(in) :: file
      integer, intent(in) :: nc(2)
      type(gildas), intent(in) :: uvin
      real, intent(in) :: duv(:,:)
      logical, intent(out) :: error
      !
      integer, intent(in), optional :: mv
      integer, intent(in), optional :: ivis(:)
    end subroutine sub_write_uvdata
  end interface
  !
  interface
    subroutine sub_extract_block(out, dout, in, din, nvisi, nc)
      use image_def
      !---------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Extract a subset block of UV data
      !---------------------------------------------------------------------
      type(gildas), intent(in) :: out
      type(gildas), intent(in) :: in
      integer(kind=index_length) :: nvisi
      real, intent(in) :: din(in%gil%dim(1),nvisi)
      real, intent(out) :: dout(out%gil%dim(1),nvisi)
      integer, intent(in) :: nc(2)
    end subroutine sub_extract_block
  end interface
  !
  interface
    subroutine sub_replace_image (name,ntype,nc,error)
      use clean_def
      use clean_arrays
      use clean_support
      use clean_types
      use gbl_format
      use gbl_message
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Support routine for command
      !   WRITE Type File /REPLACE 
      !
      ! Dispatch the writing operation to specific subroutines depending
      ! on the Type of data to be written
      !----------------------------------------------------------------------
      character(len=*), intent(in) :: name
      integer, intent(in) :: nc(2)
      integer, intent(in) :: ntype
      logical, intent(out) :: error
    end subroutine sub_replace_image
  end interface
  !
  interface
    subroutine sub_replace(name,file,data,head,error)
      use image_def
      use gbl_message
      !-------------------------------------------------------
      ! @ private
      !   Support routine for WRITE /REPLACE
      !-------------------------------------------------------
      character(len=*), intent(in) :: name
      character(len=*), intent(in) :: file
      type(gildas), intent(inout) :: head
      real(kind=4), intent(in) :: data(:,:,:)
      logical, intent(inout) :: error
    end subroutine sub_replace
  end interface
  !
  interface
    subroutine alma_clean (line,error)
      use clean_def
      use clean_arrays
      use clean_types
      use gbl_message
      !----------------------------------------------------------------------
      ! @ private
      !
      ! GILDAS: MAPPING Internal routine
      !     Implementation of a Clean Method for Multi-Array
      !     Mosaic observations
      !----------------------------------------------------------------------
      character(len=*), intent(inout) :: line
      logical, intent(out) :: error
    end subroutine alma_clean
  end interface
  !
  interface
    subroutine clean_beam(line,error)
      use clean_def
      use clean_arrays
      use gbl_message
      !---------------------------------------------------------------------------
      ! @ private
      ! Support routine for command FIT [ifield]
      !---------------------------------------------------------------------------
      character(len=*), intent(in)    :: line
      logical,          intent(inout) :: error
    end subroutine clean_beam
  end interface
  !
  interface
    subroutine debug_all(line,error)
      use clean_def
      use clean_arrays
      use clean_types
      use clean_default
      use gbl_message
      !----------------------------------------------------------------------
      ! @  private
      !
      ! MAPPING     Print debug information
      !----------------------------------------------------------------------
      character(len=*), intent(inout) :: line
      logical, intent(out) :: error
    end subroutine debug_all
  end interface
  !
  interface
    subroutine mode_mosaic (line,error)
      !----------------------------------------------------------------------
      ! @  private
      !
      ! MAPPING     MOSAIC ON|OFF
      !             Activates or desactivates the mosaic mode
      !----------------------------------------------------------------------
      character(len=*), intent(inout) :: line
      logical, intent(out) :: error
    end subroutine mode_mosaic
  end interface
  !
  interface
    subroutine display_buffer(comm,line,error)
      use clean_arrays
      use clean_def
      use clean_types
      use gbl_message
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING support routine for command
      !     SHOW Name
      !----------------------------------------------------------------------
      character(len=*), intent(in)  :: comm  ! Command (SHOW or VIEW)
      character(len=*), intent(in)  :: line  ! Command line
      logical,          intent(out) :: error ! Logical error flag
    end subroutine display_buffer
  end interface
  !
  interface
    subroutine display_buffer_sub(comm,ntype,line,error)
      use clean_types
      use clean_arrays
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING support routine for command
      !     SHOW Name
      !
      !----------------------------------------------------------------------
      character(len=*), intent(in) :: comm
      integer, intent(in)  :: ntype ! Buffer type
      logical, intent(out) :: error ! Logical error flag
      character(len=*), intent(in) :: line
    end subroutine display_buffer_sub
  end interface
  !
  interface
    subroutine load_buffer(line,error)
      use clean_types
      use clean_default
      use clean_arrays
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING support routine for command
      !     LOAD Name
      !----------------------------------------------------------------------
      character(len=*), intent(in)  :: line  ! Command line
      logical,          intent(out) :: error ! Logical error flag
    end subroutine load_buffer
  end interface
  !
  interface
    subroutine buffer_copy(atype,namew,gw,error)
      use image_def
      use clean_def
      use gbl_message
      use clean_arrays
      use clean_types
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING support routine for commands
      !     LOAD and SHOW Name
      !----------------------------------------------------------------------
      character(len=*), intent(in) :: atype
      character(len=*), intent(in) :: namew
      type(gildas), intent(inout) :: gw
      logical, intent(out) :: error
    end subroutine buffer_copy
  end interface
  !
  interface
    subroutine create_fields(error)
      use clean_arrays
      use clean_types
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING internalroutine for command
      !     Create the FIELDS array (transposed version of the PRIMARY array)
      !----------------------------------------------------------------------
      logical, intent(out) :: error
    end subroutine create_fields
  end interface
  !
  interface
    subroutine dofft_quick_debug (np,nv,visi,jx,jy,jo   &
         &    ,nc,nx,ny,map,mapx,mapy,sup,cell,taper,we,vv,   &
         &    ubias,vbias,ubuff,vbuff)
      !$  use omp_lib
      !----------------------------------------------------------------------
      ! @ private
      !
      ! GILDAS  MAP_FAST
      !   Compute FFT of image by gridding UV data
      !   - Taper before gridding
      !   - Gridding with pre-computed support
      !   - Uses symmetry
      !----------------------------------------------------------------------
      integer, intent(in) :: nv                   ! number of values
      integer, intent(in) :: np                   ! Number of "visibilities", normally 7+3*nc
      real, intent(in) :: visi(np,nv)             ! values
      integer, intent(in) :: nc                   ! number of channels
      integer, intent(in) :: jx                   ! X coord location in VISI
      integer, intent(in) :: jy                   ! Y coord location in VISI
      integer, intent(in) :: jo                   ! first channel to map
      integer, intent(in) :: nx                   ! X map size
      integer, intent(in) :: ny                   ! Y map size
      real, intent(out) :: map(2*(nc+1),nx,ny)    ! gridded visibilities
      real, intent(in) :: mapx(nx)                ! X Coordinates of grid
      real, intent(in) :: mapy(ny)                ! Y Coordinates of grid
      real, intent(in) :: sup(2)                  ! Support of convolving function in Meters
      real, intent(in) :: cell(2)                 ! cell size in Meters
      real, intent(in) :: taper(4)                ! 1/e taper in Meters + Angle in Radians
      real, intent(in) :: we(nv)                  ! Weight array
      real, intent(in) :: vv(nv)                  ! V Values
      real, intent(in) :: ubias                   ! U gridding offset
      real, intent(in) :: vbias                   ! V gridding offset
      real, intent(in) :: ubuff(4096)             ! U gridding buffer
      real, intent(in) :: vbuff(4096)             ! V gridding buffer
    end subroutine dofft_quick_debug
  end interface
  !
  interface
    subroutine dofft_quick_para (np,nv,visi,jx,jy,jo   &
         &    ,nc,nx,ny,map,mapx,mapy,sup,cell,taper,we,vv,   &
         &    ubias,vbias,ubuff,vbuff)
      !$  use omp_lib
      !----------------------------------------------------------------------
      ! @ private
      !
      ! GILDAS  MAP_FAST
      !   Compute FFT of image by gridding UV data
      !   - Taper before gridding
      !   - Gridding with pre-computed support
      !   - Uses symmetry
      !----------------------------------------------------------------------
      integer, intent(in) :: nv                   ! number of values
      integer, intent(in) :: np                   ! Number of "visibilities", normally 7+3*nc
      real, intent(in) :: visi(np,nv)             ! values
      integer, intent(in) :: nc                   ! number of channels
      integer, intent(in) :: jx                   ! X coord location in VISI
      integer, intent(in) :: jy                   ! Y coord location in VISI
      integer, intent(in) :: jo                   ! first channel to map
      integer, intent(in) :: nx                   ! X map size
      integer, intent(in) :: ny                   ! Y map size
      real, intent(out) :: map(2*(nc+1),nx,ny)    ! gridded visibilities
      real, intent(in) :: mapx(nx)                ! X Coordinates of grid
      real, intent(in) :: mapy(ny)                ! Y Coordinates of grid
      real, intent(in) :: sup(2)                  ! Support of convolving function in Meters
      real, intent(in) :: cell(2)                 ! cell size in Meters
      real, intent(in) :: taper(4)                ! 1/e taper in Meters + Angle in Radians
      real, intent(in) :: we(nv)                  ! Weight array
      real, intent(in) :: vv(nv)                  ! V Values
      real, intent(in) :: ubias                   ! U gridding offset
      real, intent(in) :: vbias                   ! V gridding offset
      real, intent(in) :: ubuff(4096)             ! U gridding buffer
      real, intent(in) :: vbuff(4096)             ! V gridding buffer
    end subroutine dofft_quick_para
  end interface
  !
  interface
    subroutine dofft_parallel_v_pseudo_out (np,nv,visi,jx,jy,jo   &
         &    ,nc,nx,ny,map,mapx,mapy,sup,cell,taper,we,vv,   &
         &    ubias,vbias,ubuff,vbuff)
      !$  use omp_lib
      !----------------------------------------------------------------------
      ! @ private
      !
      ! GILDAS  MAP_FAST
      !   Compute FFT of image by gridding UV data
      !   - Taper before gridding
      !   - Gridding with pre-computed support
      !   - Uses symmetry
      !----------------------------------------------------------------------
      integer, intent(in) :: nv                   ! number of values
      integer, intent(in) :: np                   ! Number of "visibilities", normally 7+3*nc
      real, intent(in) :: visi(np,nv)             ! values
      integer, intent(in) :: nc                   ! number of channels
      integer, intent(in) :: jx                   ! X coord location in VISI
      integer, intent(in) :: jy                   ! Y coord location in VISI
      integer, intent(in) :: jo                   ! first channel to map
      integer, intent(in) :: nx                   ! X map size
      integer, intent(in) :: ny                   ! Y map size
      real, intent(out) :: map(2*(nc+1),nx,ny)    ! gridded visibilities
      real, intent(in) :: mapx(nx)                ! X Coordinates of grid
      real, intent(in) :: mapy(ny)                ! Y Coordinates of grid
      real, intent(in) :: sup(2)                  ! Support of convolving function in Meters
      real, intent(in) :: cell(2)                 ! cell size in Meters
      real, intent(in) :: taper(4)                ! 1/e taper in Meters + Angle in Radians
      real, intent(in) :: we(nv)                  ! Weight array
      real, intent(in) :: vv(nv)                  ! V Values
      real, intent(in) :: ubias                   ! U gridding offset
      real, intent(in) :: vbias                   ! V gridding offset
      real, intent(in) :: ubuff(4096)             ! U gridding buffer
      real, intent(in) :: vbuff(4096)             ! V gridding buffer
    end subroutine dofft_parallel_v_pseudo_out
  end interface
  !
  interface
    subroutine dofft_parallel_v_true_out (np,nv,visi,jx,jy,jo   &
         &    ,nc,nx,ny,map,mapx,mapy,sup,cell,taper,we,vv,   &
         &    ubias,vbias,ubuff,vbuff)
      !$  use omp_lib
      !----------------------------------------------------------------------
      ! @ private
      !
      ! GILDAS  MAP_FAST
      !   Compute FFT of image by gridding UV data
      !   - Taper before gridding
      !   - Gridding with pre-computed support
      !   - Uses symmetry
      !----------------------------------------------------------------------
      integer, intent(in) :: nv                   ! number of values
      integer, intent(in) :: np                   ! Number of "visibilities", normally 7+3*nc
      real, intent(in) :: visi(np,nv)             ! values
      integer, intent(in) :: nc                   ! number of channels
      integer, intent(in) :: jx                   ! X coord location in VISI
      integer, intent(in) :: jy                   ! Y coord location in VISI
      integer, intent(in) :: jo                   ! first channel to map
      integer, intent(in) :: nx                   ! X map size
      integer, intent(in) :: ny                   ! Y map size
      real, intent(out) :: map(2*(nc+1),nx,ny)    ! gridded visibilities
      real, intent(in) :: mapx(nx)                ! X Coordinates of grid
      real, intent(in) :: mapy(ny)                ! Y Coordinates of grid
      real, intent(in) :: sup(2)                  ! Support of convolving function in Meters
      real, intent(in) :: cell(2)                 ! cell size in Meters
      real, intent(in) :: taper(4)                ! 1/e taper in Meters + Angle in Radians
      real, intent(in) :: we(nv)                  ! Weight array
      real, intent(in) :: vv(nv)                  ! V Values
      real, intent(in) :: ubias                   ! U gridding offset
      real, intent(in) :: vbias                   ! V gridding offset
      real, intent(in) :: ubuff(4096)             ! U gridding buffer
      real, intent(in) :: vbuff(4096)             ! V gridding buffer
    end subroutine dofft_parallel_v_true_out
  end interface
  !
  interface
    subroutine dofft_parallel_v_pseudo (np,nv,visi,jx,jy,jo   &
         &    ,nc,nx,ny,map,mapx,mapy,sup,cell,taper,we,vv,   &
         &    ubias,vbias,ubuff,vbuff)
      !$  use omp_lib
      !----------------------------------------------------------------------
      ! @ private
      !
      ! GILDAS  MAP_FAST
      !   Compute FFT of image by gridding UV data
      !   - Taper before gridding
      !   - Gridding with pre-computed support
      !   - Uses symmetry
      !----------------------------------------------------------------------
      integer, intent(in) :: nv                   ! number of values
      integer, intent(in) :: np                   ! Number of "visibilities", normally 7+3*nc
      real, intent(in) :: visi(np,nv)             ! values
      integer, intent(in) :: nc                   ! number of channels
      integer, intent(in) :: jx                   ! X coord location in VISI
      integer, intent(in) :: jy                   ! Y coord location in VISI
      integer, intent(in) :: jo                   ! first channel to map
      integer, intent(in) :: nx                   ! X map size
      integer, intent(in) :: ny                   ! Y map size
      real, intent(out) :: map(2*(nc+1),nx,ny)    ! gridded visibilities
      real, intent(in) :: mapx(nx)                ! X Coordinates of grid
      real, intent(in) :: mapy(ny)                ! Y Coordinates of grid
      real, intent(in) :: sup(2)                  ! Support of convolving function in Meters
      real, intent(in) :: cell(2)                 ! cell size in Meters
      real, intent(in) :: taper(4)                ! 1/e taper in Meters + Angle in Radians
      real, intent(in) :: we(nv)                  ! Weight array
      real, intent(in) :: vv(nv)                  ! V Values
      real, intent(in) :: ubias                   ! U gridding offset
      real, intent(in) :: vbias                   ! V gridding offset
      real, intent(in) :: ubuff(4096)             ! U gridding buffer
      real, intent(in) :: vbuff(4096)             ! V gridding buffer
    end subroutine dofft_parallel_v_pseudo
  end interface
  !
  interface
    subroutine dofft_parallel_v_true (np,nv,visi,jx,jy,jo   &
         &    ,nc,nx,ny,map,mapx,mapy,sup,cell,taper,we,vv,   &
         &    ubias,vbias,ubuff,vbuff)
      !$  use omp_lib
      !----------------------------------------------------------------------
      ! @ private
      !
      ! GILDAS  MAP_FAST
      !   Compute FFT of image by gridding UV data
      !   - Taper before gridding
      !   - Gridding with pre-computed support
      !   - Uses symmetry
      !----------------------------------------------------------------------
      integer, intent(in) :: nv                   ! number of values
      integer, intent(in) :: np                   ! Number of "visibilities", normally 7+3*nc
      real, intent(in) :: visi(np,nv)             ! values
      integer, intent(in) :: nc                   ! number of channels
      integer, intent(in) :: jx                   ! X coord location in VISI
      integer, intent(in) :: jy                   ! Y coord location in VISI
      integer, intent(in) :: jo                   ! first channel to map
      integer, intent(in) :: nx                   ! X map size
      integer, intent(in) :: ny                   ! Y map size
      real, intent(out) :: map(2*(nc+1),nx,ny)    ! gridded visibilities
      real, intent(in) :: mapx(nx)                ! X Coordinates of grid
      real, intent(in) :: mapy(ny)                ! Y Coordinates of grid
      real, intent(in) :: sup(2)                  ! Support of convolving function in Meters
      real, intent(in) :: cell(2)                 ! cell size in Meters
      real, intent(in) :: taper(4)                ! 1/e taper in Meters + Angle in Radians
      real, intent(in) :: we(nv)                  ! Weight array
      real, intent(in) :: vv(nv)                  ! V Values
      real, intent(in) :: ubias                   ! U gridding offset
      real, intent(in) :: vbias                   ! V gridding offset
      real, intent(in) :: ubuff(4096)             ! U gridding buffer
      real, intent(in) :: vbuff(4096)             ! V gridding buffer
    end subroutine dofft_parallel_v_true
  end interface
  !
  interface
    subroutine dofft_parallel_v_true2 (np,nv,visi,jx,jy,jo   &
         &    ,nc,nx,ny,map,mapx,mapy,sup,cell,taper,we,vv,   &
         &    ubias,vbias,ubuff,vbuff)
      !$  use omp_lib
      !----------------------------------------------------------------------
      ! @ private
      !
      ! GILDAS  MAP_FAST
      !   Compute FFT of image by gridding UV data
      !   - Taper before gridding
      !   - Gridding with pre-computed support
      !   - Uses symmetry
      !----------------------------------------------------------------------
      integer, intent(in) :: nv                   ! number of values
      integer, intent(in) :: np                   ! Number of "visibilities", normally 7+3*nc
      real, intent(in) :: visi(np,nv)             ! values
      integer, intent(in) :: nc                   ! number of channels
      integer, intent(in) :: jx                   ! X coord location in VISI
      integer, intent(in) :: jy                   ! Y coord location in VISI
      integer, intent(in) :: jo                   ! first channel to map
      integer, intent(in) :: nx                   ! X map size
      integer, intent(in) :: ny                   ! Y map size
      real, intent(out) :: map(2*(nc+1),nx,ny)    ! gridded visibilities
      real, intent(in) :: mapx(nx)                ! X Coordinates of grid
      real, intent(in) :: mapy(ny)                ! Y Coordinates of grid
      real, intent(in) :: sup(2)                  ! Support of convolving function in Meters
      real, intent(in) :: cell(2)                 ! cell size in Meters
      real, intent(in) :: taper(4)                ! 1/e taper in Meters + Angle in Radians
      real, intent(in) :: we(nv)                  ! Weight array
      real, intent(in) :: vv(nv)                  ! V Values
      real, intent(in) :: ubias                   ! U gridding offset
      real, intent(in) :: vbias                   ! V gridding offset
      real, intent(in) :: ubuff(4096)             ! U gridding buffer
      real, intent(in) :: vbuff(4096)             ! V gridding buffer
    end subroutine dofft_parallel_v_true2
  end interface
  !
  interface
    subroutine dofft_parallel_y (np,nv,visi,jx,jy,jo   &
         &    ,nc,nx,ny,map,mapx,mapy,sup,cell,taper,we,vv,   &
         &    ubias,vbias,ubuff,vbuff)
      use gbl_message
      !$  use omp_lib
      !----------------------------------------------------------------------
      ! @ private
      !
      ! GILDAS  UVMAP
      !   Compute FFT of image by gridding UV data
      !   Taper after gridding
      !   Uses symmetry
      !----------------------------------------------------------------------
      integer, intent(in) :: nv                   ! number of values
      integer, intent(in) :: np                   ! Number of "visibilities", normally 7+3*nc
      real, intent(in) :: visi(np,nv)             ! values
      integer, intent(in) :: nc                   ! number of channels
      integer, intent(in) :: jx                   ! X coord location in VISI
      integer, intent(in) :: jy                   ! Y coord location in VISI
      integer, intent(in) :: jo                   ! first channel to map
      integer, intent(in) :: nx                   ! X map size
      integer, intent(in) :: ny                   ! Y map size
      real, intent(out) :: map(2*(nc+1),nx,ny)    ! gridded visibilities
      real, intent(in) :: mapx(nx)                ! X Coordinates of grid
      real, intent(in) :: mapy(ny)                ! Y Coordinates of grid
      real, intent(in) :: sup(2)                  ! Support of convolving function in Meters
      real, intent(in) :: cell(2)                 ! cell size in Meters
      real, intent(in) :: taper(4)                ! 1/e taper in Meters + Angle in Radians
      real, intent(in) :: we(nv)                  ! Weight array
      real, intent(in) :: vv(nv)                  ! V Values
      real, intent(in) :: ubias                   ! U gridding offset
      real, intent(in) :: vbias                   ! V gridding offset
      real, intent(in) :: ubuff(4096)             ! U gridding buffer
      real, intent(in) :: vbuff(4096)             ! V gridding buffer
    end subroutine dofft_parallel_y
  end interface
  !
  interface
    subroutine dofft_parallel_x (np,nv,visi,jx,jy,jo   &
         &    ,nc,nx,ny,map,mapx,mapy,sup,cell,taper,we,vv,   &
         &    ubias,vbias,ubuff,vbuff)
      use gbl_message
      !$  use omp_lib
      !----------------------------------------------------------------------
      ! @ private
      !
      ! GILDAS  UVMAP
      !   Compute FFT of image by gridding UV data
      !   Taper after gridding
      !   Uses symmetry
      !----------------------------------------------------------------------
      integer, intent(in) :: nv                   ! number of values
      integer, intent(in) :: np                   ! Number of "visibilities", normally 7+3*nc
      real, intent(in) :: visi(np,nv)             ! values
      integer, intent(in) :: nc                   ! number of channels
      integer, intent(in) :: jx                   ! X coord location in VISI
      integer, intent(in) :: jy                   ! Y coord location in VISI
      integer, intent(in) :: jo                   ! first channel to map
      integer, intent(in) :: nx                   ! X map size
      integer, intent(in) :: ny                   ! Y map size
      real, intent(out) :: map(2*(nc+1),nx,ny)    ! gridded visibilities
      real, intent(in) :: mapx(nx)                ! X Coordinates of grid
      real, intent(in) :: mapy(ny)                ! Y Coordinates of grid
      real, intent(in) :: sup(2)                  ! Support of convolving function in Meters
      real, intent(in) :: cell(2)                 ! cell size in Meters
      real, intent(in) :: taper(4)                ! 1/e taper in Meters + Angle in Radians
      real, intent(in) :: we(nv)                  ! Weight array
      real, intent(in) :: vv(nv)                  ! V Values
      real, intent(in) :: ubias                   ! U gridding offset
      real, intent(in) :: vbias                   ! V gridding offset
      real, intent(in) :: ubuff(4096)             ! U gridding buffer
      real, intent(in) :: vbuff(4096)             ! V gridding buffer
    end subroutine dofft_parallel_x
  end interface
  !
  interface
    subroutine major_cycle90 (rname,method,head,   &
         &    clean,beam,resid,nx,ny,tfbeam,fcomp,   &
         &    wcl,mcl,ixbeam,iybeam,ixpatch,iypatch,bgain,   &
         &    box, wfft, tcc, list, nl, np, primary, weight,       &
         &    major_plot, next_flux)
      use clean_def
      use image_def
      use gbl_message
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Major cycle loop according to B.Clark idea
      !----------------------------------------------------------------------
      external :: major_plot
      external :: next_flux
      !
      character(len=*), intent(in) :: rname
      type (clean_par), intent(inout) :: method
      type (gildas), intent(in)  :: head
      !
      integer, intent(in) :: np   ! Number of pointings
      integer, intent(in) :: nx   ! X size
      integer, intent(in) :: ny   ! Y size
      integer, intent(in) :: mcl  ! Maximum number of clean components
      real, intent(inout) :: clean(nx,ny)         ! Clean map
      real, intent(inout) :: resid(nx,ny)         ! Residual map
      real, intent(in) ::    beam(nx,ny,np)       ! Dirty beams (per pointing)
      real, intent(in) ::    tfbeam(nx,ny,np)     ! T.F. du beam
      complex, intent(inout) :: fcomp(nx,ny)      ! T.F. du vecteur modification
      real, intent(in) :: bgain                   ! Maximum sidelobe level
      integer, intent(in) :: ixbeam, iybeam       ! Beam maximum position
      integer, intent(in) :: ixpatch, iypatch     ! Beam patch radius
      integer, intent(in) :: box(4)               ! Cleaning box
      real, intent(inout) :: wfft(*)              ! Work space for FFT
      type(cct_par), intent(inout) :: tcc(method%m_iter) ! Clean components array
      type(cct_par), intent(inout) :: wcl(mcl)    ! Work space for Clean components
      integer, intent(inout) :: list(nx*ny)       ! list of searchable pixels
      integer, intent(inout) :: nl   ! List size
      !
      real, intent(in) :: primary(np,nx,ny)       ! Primary beams
      real, intent(in) :: weight (nx,ny)          ! Flat field response
    end subroutine major_cycle90
  end interface
  !
  interface
    subroutine minor_cycle90 (method, wcl, ncl,           &
         &    beam,nx,ny,ixbeam,iybeam,ixpatch,iypatch,   &
         &    clarkmin,limite,converge,   &
         &    tcc, np, primary, weight, wtrun, cum, pflux, next_flux )
      use clean_def
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING     Internal routine
      !   B.Clark minor cycles
      !   Deconvolve as in standard clean a list of NCL points
      !   selected in the map until the residuals is less than CLARKMIN
      !----------------------------------------------------------------------
      external :: next_flux
      type (clean_par), intent(inout) :: method
      integer, intent(in) :: np                   ! Number of fields in mosaic
      integer, intent(in) :: ncl                  ! nombre de points retenus
      integer, intent(in) :: nx,ny,ixbeam,iybeam  ! dimension et centre du beam
      integer, intent(in) :: ixpatch,iypatch      ! rayon utile du Beam
      real, intent(in) :: beam(nx,ny,np)          ! Dirty beam
      type(cct_par), intent(inout) :: wcl(*)      ! Clean components
      real, intent(in) :: clarkmin                ! borne d'arret de clean
      real, intent(in) :: limite                  ! borne d'arret de clean
      logical, intent(out) :: converge            ! Controle de la convergence
      type (cct_par), intent(out) :: tcc(*)       ! Effective clean components
      real, intent(in) :: primary(np,nx,ny)       ! Primary beams of mosaics
      real, intent(in) :: weight(nx,ny)           ! Effective weights on sky
      real, intent(in) :: wtrun                   ! Threshold of primary beam
      real, intent(inout) :: cum                     ! Cumulative flux
      logical, intent(in) :: pflux
      !
      real gain                    ! gain de clean
      logical goon
      integer kcl                  ! No de la composante courante
      integer nomax, nomin         ! No du max et min courant
      real rmax, rmin, sign        ! Max et Min courant
      real worry, xfac, spexp      ! Conservative and speedup factor
      integer kiter
      logical :: check             ! Controle de la convergence
      integer :: dimcum=10         ! par accumulation de flux positif
      real, allocatable :: oldcum(:)
      logical :: error
      real f, bmax
      integer n,ier,i
      logical abor
      character(len=20) comm
      !
      !call sic_get_inte('MAP_CONVERGE',dimcum,error)
    end subroutine minor_cycle90
  end interface
  !
  interface
    subroutine major_sdi90 (rname,method,head,clean,beam,resid,nx,ny,   &
         &    tfbeam,fcomp,wcl,mcl,ixbeam,iybeam,ixpatch,iypatch,bgain,   &
         &    box, wfft, comp, list, nl, np, primary, weight, &
         &    major_plot )
      use clean_def
      use image_def
      use gbl_message
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      ! Major cycle loop according to Steer Dewdney and Ito idea
      !----------------------------------------------------------------------
      external :: major_plot
      !
      character(len=*), intent(in) :: rname 
      type (clean_par), intent(inout) :: method
      type (gildas), intent(in)  :: head
      !
      integer, intent(in) :: np   ! Number of fields
      integer, intent(in) :: nx   ! X size
      integer, intent(in) :: ny   ! Y size
      integer, intent(in) :: mcl  ! Maximum number of clean components
      real, intent(inout) :: clean(nx,ny)         ! Clean map
      real, intent(inout) :: resid(nx,ny)         ! Residual map
      real, intent(in) ::    beam(nx,ny,np)       ! Dirty beams (per field)
      real, intent(in) ::    tfbeam(nx,ny,np)     ! T.F. du beam
      complex, intent(inout) :: fcomp(nx,ny)      ! T.F. du vecteur modification
      type(cct_par), intent(inout) :: wcl(mcl)    ! Work space for Clean components
      real, intent(in) :: bgain                   ! Maximum sidelobe level
      integer, intent(in) :: ixbeam, iybeam       ! Beam maximum position
      integer, intent(in) :: ixpatch, iypatch     ! Beam patch radius
      integer, intent(in) :: box(4)               ! Cleaning box
      real, intent(inout) :: wfft(*)              ! Work space for FFT
      integer, intent(inout) :: list(nx*ny)       ! list of searchable pixels
      integer, intent(inout) :: nl   ! List size
      real, intent(inout) :: comp(nx,ny)             ! Clean components array
      !
      real, intent(in) :: primary(np,nx,ny)       ! Primary beams
      real, intent(in) :: weight (nx,ny)
      !
      real    maxc,minc,maxabs     !max et min de la carte, absolu
      integer imax,jmax,imin,jmin  !leurs coordonnees
      real    borne                !fraction de la carte initiale
      real    limite               !intensite minimum des pts retenus
      real flux                    !Total clean flux density
      integer ncl                  !nb de pts reellement retenus
      logical fini                 !critere d'arret
      logical converge             ! Indique la conv par acc de flux
      logical error
      integer k
      integer ndim, nn(2)
      real factor                  ! Scaling factor
      character(len=message_length) :: chain
      !
      type (cct_par) :: tcc(1)     ! Dummy argument for Major_PLOT
      !
      ! Find maximum residual
    end subroutine major_sdi90
  end interface
  !
  interface
    subroutine normal (fcomp,tfbeam,nx,ny,   &
         &    wcl,ncl,wfft,factor)
      use clean_def
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING    Support routine for SDI
      ! Subtract last major cycle components from residual map.
      !----------------------------------------------------------------------
      integer, intent(in) :: nx    ! X size
      integer, intent(in) :: ny    ! Y size
      integer, intent(in) :: ncl   ! Number of clean components
      type(cct_par), intent(in) :: wcl(ncl)  ! Clean comp.
      real, intent(in) :: tfbeam(nx,ny)     ! Beam TF
      complex, intent(out) :: fcomp(nx,ny)  ! TF of clean comp.
      real, intent(inout) :: wfft(*)        ! Work array
      real, intent(out) :: factor           ! Max of clean
    end subroutine normal
  end interface
  !
  interface
    subroutine scalec(wcl,ncl,f,s,compon,nx,ny)
      use clean_def
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING    Support routine for SDI
      ! Subtract last major cycle components from residual map.
      !----------------------------------------------------------------------
      integer, intent(in) :: nx    ! X size
      integer, intent(in) :: ny    ! Y size
      integer, intent(in) :: ncl   ! Number of clean components
      type(cct_par), intent(inout) :: wcl(ncl)     ! Clean comp.
      real, intent(in) :: f           ! Gain factor
      real, intent(inout) :: s        ! Cumulative flux
      real, intent(inout) :: compon(nx,ny)  ! Cumulative clean component
    end subroutine scalec
  end interface
  !
  interface
    subroutine hogbom_cycle90 (rname,pflux, beam,mx,my,resid,nx,ny,   &
         &    ixbeam,iybeam, box, fracres, absres, miter, piter, niter,   &
         &    gainloop, keep, cct, msk, list, nl, np, primary, weight, wtrun, &
         &    cflux, jcode, next_flux)
      use clean_def
      use clean_default
      use gbl_message
      !$ use omp_lib
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING    Support routine for HOGBOM
      !     Deconvolve map into residual map and source list
      !-----------------------------------------------------------------------
      external :: next_flux
      character(len=*), intent(in) :: rname 
      logical, intent(in) :: pflux       ! Select only positive ?
      integer, intent(in) :: mx          ! X size of beam
      integer, intent(in) :: my          ! Y size of beam
      integer, intent(in) :: nx          ! X size of image
      integer, intent(in) :: ny          ! Y size of image
      integer, intent(in) :: np          ! Number of fields
      real, intent(in) :: beam(mx,my,np)          ! Primary beam(s)
      real, intent(inout) :: resid(nx,ny)         ! residual image
      real, intent(in) :: fracres                 ! Fractional residual
      real, intent(in) :: absres                  ! Absolute residual
      integer, intent(in) :: miter                ! Maximum number of clean components
      integer, intent(in) :: ixbeam, iybeam       ! Beam maximum position
      integer, intent(in) :: box(4)               ! Cleaning box
      real, intent(in) :: gainloop                ! Clean loop gain
      logical, intent(in) :: keep                 ! Keep cleaning after convergence
      integer, intent(out) :: niter               ! Iterations
      integer, intent(out) :: piter               ! Positive Iterations
      logical, intent(in) :: msk(nx,ny)           ! Mask for clean search
      integer, intent(in) :: nl                   ! Size of search list
      integer, intent(in) :: list(nl)             ! Search list
      real, intent(in) :: primary(np,nx,ny)       ! Primary beams
      real, intent(in) :: weight(nx,ny)           ! Weight function
      real, intent(in) :: wtrun                   ! Safety threshold on primary beams
      type (cct_par), intent(out) :: cct(miter)
      integer, intent(out) :: jcode               ! Stopping code
      real, intent(out) :: cflux                  ! Cleaned Flux
    end subroutine hogbom_cycle90
  end interface
  !
  interface
    subroutine many_beams_para (rname,map,huv,hbeam,hdirty,   &
         &    nx,ny,nu,nv,uvdata,   &
         &    r_weight, w_v, do_weig,    &
         &    wcol,mcol,sblock,cpu0,error,uvmax,jfield)
      use clean_def
      use image_def
      use gbl_message
      !$ use omp_lib
      !------------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Compute a map from a CLIC UV Sorted Table
      !   by Gridding and Fast Fourier Transform, with
      !   a different beam per channel.
      !
      ! Input :
      ! a precessed UV table, sorted in V, ordered in
      ! (U,V,W,D,T,iant,jant,nchan(real,imag,weig))
      ! Output :
      ! a beam image
      ! a VLM cube
      ! Work space :
      ! a  VLM complex Fourier cube (first V value is for beam)
      !------------------------------------------------------------------------
      character(len=*), intent(in) :: rname   ! Calling Task name
      type (uvmap_par), intent(inout) :: map  ! Mapping parameters
      type (gildas), intent(inout) :: huv     ! UV data set
      type (gildas), intent(inout) :: hbeam   ! Dirty beam data set
      type (gildas), intent(inout) :: hdirty  ! Dirty image data set
      integer, intent(in) :: nx   ! X size
      integer, intent(in) :: ny   ! Y size
      integer, intent(in) :: nu   ! Size of a visibilities
      integer, intent(in) :: nv   ! Number of visibilities
      real, intent(inout) :: uvdata(nu,nv)
      real, intent(inout), target :: r_weight(nv)    ! Weight of visibilities
      real, intent(inout) :: w_v(nv)         ! V values
      logical, intent(inout) :: do_weig
      integer, intent(in) :: jfield      ! Field number (for mosaic)
      !
      real, intent(inout) :: cpu0        ! CPU
      real, intent(inout) :: uvmax       ! Maximum baseline
      integer, intent(inout) :: sblock   ! Blocking factor
      integer, intent(inout) :: wcol     ! Weight channel
      integer, intent(inout) :: mcol(2)  ! First and last channel
      logical, intent(inout) :: error
    end subroutine many_beams_para
  end interface
  !
  interface
    subroutine map_resample_comm(line,comm,error)
      use clean_def
      use clean_types
      use clean_arrays
      use gbl_message
      use phys_const
      !---------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Resample in velocity the Images, or Compress them 
      !   Support for commands
      !     MAP_RESAMPLE WhichOne NC [Ref Val Inc]
      !     MAP_COMPRESS WhichOne NC
      !     MAP_INTEGRATE WhichOne Rmin Rmax RType
      !
      ! WhichOne    Name of map to be processed
      !             Allowed values are DIRTY, CLEAN, SKY
      !---------------------------------------------------------------------
      character(len=*), intent(in) :: line  ! Command line
      character(len=*), intent(in) :: comm  ! Calling command
      logical, intent(out) :: error         ! Error flag
    end subroutine map_resample_comm
  end interface
  !
  interface
    subroutine one_beam (rname,map,huv,hbeam,hdirty,   &
         &    nx,ny,nu,nv,uvdata,   &
         &    w_mapu, w_mapv, w_grid, w_weight, w_v, do_weig,    &
         &    wcol,mcol,wfft,sblock,cpu0,error,uvmax)
      use clean_def
      use image_def
      use gbl_message
      use clean_default
      !------------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Compute a map from a CLIC UV Sorted Table
      !   by Gridding and Fast Fourier Transform, with
      !   one single beam for all channels.
      !
      ! Input :
      ! a precessed UV table, sorted in V, ordered in
      ! (U,V,W,D,T,iant,jant,nchan(real,imag,weig))
      ! Output :
      ! a beam image
      ! a VLM cube
      ! Work space :
      ! a  VLM complex Fourier cube (first V value is for beam)
      !------------------------------------------------------------------------
      character(len=*), intent(in) :: rname   ! Calling Task name
      type (uvmap_par), intent(inout) :: map  ! Mapping parameters
      type (gildas), intent(inout) :: huv     ! UV data set
      type (gildas), intent(inout) :: hbeam   ! Dirty beam data set
      type (gildas), intent(inout) :: hdirty  ! Dirty image data set
      integer, intent(in) :: nx   ! X size
      integer, intent(in) :: ny   ! Y size
      integer, intent(in) :: nu   ! Size of a visibilities
      integer, intent(in) :: nv   ! Number of visibilities
      real, intent(inout) :: uvdata(nu,nv)
      real, intent(inout) :: w_mapu(nx)      ! U grid coordinates
      real, intent(inout) :: w_mapv(ny)      ! V grid coordinates
      real, intent(inout) :: w_grid(nx,ny)   ! Gridding space
      real, intent(inout) :: w_weight(nv)    ! Weight of visibilities
      real, intent(inout) :: w_v(nv)         ! V values
      logical, intent(inout) :: do_weig
      !
      real, intent(inout) :: wfft(*)     ! Work space
      real, intent(inout) :: cpu0        ! CPU
      real, intent(inout) :: uvmax       ! Maximum baseline
      integer, intent(inout) :: sblock   ! Blocking factor
      integer, intent(inout) :: wcol     ! Weight channel
      integer, intent(inout) :: mcol(2)  ! First and last channel
      logical, intent(inout) :: error
    end subroutine one_beam
  end interface
  !
  interface
    subroutine check_order(visi,np,nv,sorted)
      !----------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Check if visibilites are sorted.
      !   Chksuv does a similar job, but using V values and an index.
      !----------------------------------------------------------
      integer, intent(in) :: np       ! Size of a visibility
      integer, intent(in) :: nv       ! Number of visibilities
      real, intent(in) :: visi(np,nv) ! Visibilities
      logical, intent(out) :: sorted
    end subroutine check_order
  end interface
  !
  interface
    subroutine cct_prepare(line,nc,a_method,task,error)
      use clean_def
      use clean_arrays
      use clean_types
      use clean_default
      use gbl_message
      !---------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Prepare the Clean Component Table
      !---------------------------------------------------
      character(len=*), intent(in) :: line      ! Command line
      integer, intent(in) :: nc                 ! Number of channels
      type(clean_par), intent(in) :: a_method   ! Input method
      character(len=*), intent(in) :: task      ! Caller's name
      logical, intent(out) :: error  !
    end subroutine cct_prepare
  end interface
  !
  interface
    subroutine clean_mfs(line,error)
      use image_def
      use clean_def
      use clean_arrays
      use clean_types
      !
      !-----------------------------------------------
      ! @ private
      !
      !-----------------------------------------------
      character(len=*), intent(inout) :: line
      logical, intent(out) :: error
    end subroutine clean_mfs
  end interface
  !
  interface
    subroutine sub_clean_mfs(ixoff,iyoff,error)
      use image_def
      use clean_def
      use clean_arrays
      use gbl_message
      !-----------------------------------------------
      ! @ private
      !
      !-----------------------------------------------
      integer, intent(inout) :: ixoff, iyoff
      logical, intent(out) :: error
    end subroutine sub_clean_mfs
  end interface
  !
  interface
    subroutine com_modify(line,error)
      use phys_const
      use clean_arrays
      use clean_types
      use gbl_message
      !-----------------------------------------------------
      ! @ private
      !
      ! MAPPING  Support for command
      !   SPECIFY KeyWord Value
      !-----------------------------------------------------
      character(len=*), intent(inout) :: line
      logical, intent(out) :: error
    end subroutine com_modify
  end interface
  !
  interface
    subroutine map_copy_par(in,out)
      use clean_def
      !-----------------------------------------------------------------------
      !
      ! @ private
      !   Copy a MAP structure to another one, but avoid erasing
      !   the number of fields in it. The result must have an intent(inout)
      !   to avoid erasing the allocatable array in the derived type.
      !
      !-----------------------------------------------------------------------
      type(uvmap_par), intent(in) :: in
      type(uvmap_par), intent(inout) :: out
    end subroutine map_copy_par
  end interface
  !
  interface
    subroutine map_prepare(task,map,error)
      use clean_def
      use clean_default
      use gbl_message
      !
      ! @ private
      !
      character(len=*), intent(in) :: task
      type(uvmap_par), intent(inout) :: map
      logical, intent(out) :: error
    end subroutine map_prepare
  end interface
  !
  interface
    subroutine sub_mosaic(name,error)
      use clean_def
      use clean_arrays
      use gbl_message
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING     MOSAIC ON|OFF
      !             Activates or desactivates the mosaic mode
      !----------------------------------------------------------------------
      character(len=*), intent(in) :: name
      logical, intent(out) :: error
    end subroutine sub_mosaic
  end interface
  !
  interface
    subroutine get_bsize(h,rname,line,bsize,error,otrunc,btrunc)
      use image_def
      use gbl_message
      !---------------------------------------------------------------------
      ! @ private-mandatory
      !
      ! MAPPING  Support for command
      !   PRIMARY [BeamSize] [/TRUNCATE Percent]
      !   UV_MAP  [BeamSize] [/TRUNCATE Percent]
      !
      ! Return the primary beam size in radian
      !---------------------------------------------------------------------
      type(gildas), intent(in) :: h          ! UV data header
      character(len=*), intent(in) :: rname  ! Caller's name
      character(len=*), intent(in) :: line   ! Command line
      real(4), intent(out) :: bsize          ! Beam size in radian
      logical, intent(out) :: error          ! Error flag
      integer, optional, intent(in) ::  otrunc         ! Truncation option number
      real(4), optional, intent(out) :: btrunc         ! Truncation level [0,1]
    end subroutine get_bsize
  end interface
  !
  interface
    subroutine primary_comm(line,error)
      use clean_def
      use clean_arrays
      use clean_types
      use clean_default
      use gbl_message
      !------------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING  Support for command
      !   PRIMARY [BeamSize] [/TRUNCATE Percent]
      !
      ! Correct for primary beam
      !------------------------------------------------------------------------
      character(len=*), intent(in) :: line ! Command line
      logical, intent(out) :: error
    end subroutine primary_comm
  end interface
  !
  interface
    subroutine mosaic_uvmap(task,line,error)
      use clean_def
      use clean_arrays
      use clean_types
      use clean_default
      use gbl_message
      !------------------------------------------------------------------------
      ! @ private
      !
      ! TASK  Compute a Mosaic from a CLIC UV Table with
      ! pointing offset information.
      !
      ! Input :
      !     a UV table with pointing offset information
      !
      ! Ouput
      !   NX NY are the image sizes
      !   NC is the number of channels
      !   NF is the number of different frequencies
      !   NP is the number of pointing centers
      !
      ! 'NAME'.LMV  a 3-d cube containing the uniform noise
      !     combined mosaic, i.e. the sum of the product
      !     of the fields by the primary beam. (NX,NY,NC)
      ! 'NAME'.LOBE the primary beams pseudo-cube (NP,NX,NY,NB)
      ! 'NAME'.WEIGHT the sum of the square of the primary beams (NX,NY,NB)
      ! 'NAME'.BEAM a 4-d cube where each cube contains the synthesised
      !     beam for one field (NX,NY,NB,NP)
      !
      ! All images have the same X,Y sizes
      !------------------------------------------------------------------------
      character(len=*), intent(in) :: task ! Caller (MOSAIC)
      character(len=*), intent(in) :: line ! Command line
      logical, intent(out) :: error
    end subroutine mosaic_uvmap
  end interface
  !
  interface
    subroutine map_headers (rname,map,huv,hbeam,hdirty,hprim,nb,nf,mcol)
      use clean_def
      use image_def
      use gbl_message
      !------------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Define the image headers
      !------------------------------------------------------------------------
      character(len=*), intent(in) :: rname   ! Calling Task name
      type (uvmap_par), intent(inout) :: map  ! Mapping parameters
      type (gildas), intent(inout) :: huv     ! UV data set
      type (gildas), intent(inout) :: hbeam   ! Dirty beam data set
      type (gildas), intent(inout) :: hdirty  ! Dirty image data set
      type (gildas), intent(inout) :: hprim   ! Primary beam data set
      integer, intent(in) :: nb   ! Number of beams per field
      integer, intent(in) :: nf   ! Number of fields
      integer, intent(inout) :: mcol(2)  ! First and last channel
    end subroutine map_headers
  end interface
  !
  interface
    subroutine mosaic_sort (error,sorted,shift,new,uvmax,uvmin, &
      & ixoff,iyoff,nf,doff,voff)
      use clean_def
      use clean_arrays
      use gbl_message
      !----------------------------------------------------------------------
      ! @ private-mandatory
      !
      ! MAPPING
      !   Sort the input UV table
      !----------------------------------------------------------------------
      logical, intent(inout) :: sorted      ! Is table sorted ?
      logical, intent(inout) :: shift       ! Do we shift phase center ?
      logical, intent(out) :: error
      real(kind=8), intent(inout) :: new(3) ! New phase center and PA
      real, intent(out) :: uvmin            ! Min baseline
      real, intent(out) :: uvmax            ! Max baseline
      integer, intent(in) :: ixoff, iyoff   ! Offset pointers
      integer, intent(out) :: nf            ! Number of fields
      real, intent(out), allocatable :: doff(:,:)  ! Field offsets
      integer, allocatable :: voff(:)       ! Field visibility offsets
    end subroutine mosaic_sort
  end interface
  !
  interface
    subroutine check_order_mosaic(visi,np,nv,ixoff,iyoff,sorted)
      !----------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Check if visibilites are sorted.
      !   Chksuv does a similar job, but using V values and an index.
      !----------------------------------------------------------
      integer, intent(in) :: np       ! Size of a visibility
      integer, intent(in) :: nv       ! Number of visibilities
      real, intent(in) :: visi(np,nv) ! Visibilities
      integer, intent(in) :: ixoff    ! X pointing column
      integer, intent(in) :: iyoff    ! Y pointing column
      logical, intent(out) :: sorted
    end subroutine check_order_mosaic
  end interface
  !
  interface
    subroutine mosaic_restore(line,error)
      use clean_arrays
      use clean_types
      use gbl_message
      !---------------------------------------------------------------------
      ! @ private
      !
      ! Restore (like in UV_RESTORE) a mosaic image
      !
      ! Required input
      !   the CCT table
      !   the RESIDUAL image (obtained by imaging the UV table
      !     obtained by UV_RESIDUAL)
      !   the PRIMARY beams
      ! Output 
      !   the restored CLEAN image
      !---------------------------------------------------------------------
      character(len=*), intent(inout) :: line
      logical, intent(inout) :: error
    end subroutine mosaic_restore
  end interface
  !
  interface
    subroutine mx_clean (map,huv,uvdata,uvp,uvv,   &
         &    method,hdirty,hbeam,hclean,hresid,hprim,   &
         &    w_grid,w_mapu,w_mapv,p_cct, dcct, smask, slist,   &
         &    sblock, cpu0, uvmax)
      use clean_def
      use image_def
      use gbl_message
      !----------------------------------------------------------------------
      ! @ private
      ! GILDAS: CLEAN Internal routine
      ! Implementation of MX CLEAN deconvolution algorithm.
      !----------------------------------------------------------------------
      type (clean_par), intent(inout) :: method
      type (uvmap_par), intent(inout) :: map
      type (gildas), intent(inout) :: huv
      type (gildas), intent(inout) :: hdirty
      type (gildas), intent(inout) :: hbeam
      type (gildas), intent(inout) :: hclean
      type (gildas), intent(inout) :: hresid
      type (gildas), intent(inout) :: hprim
      real, intent(inout) :: dcct(3,hclean%gil%dim(3),*)
      logical, intent(inout) :: smask(:,:)
      integer, intent(in) :: slist(*)
      integer, intent(in) :: sblock
      real, intent(inout) :: uvdata(huv%gil%dim(1),huv%gil%dim(2))
      real, intent(in) :: uvp(*)
      real, intent(in) :: uvv(*)
      real, intent(inout) :: w_grid(*)
      real, intent(inout) :: w_mapu(*)
      real, intent(inout) :: w_mapv(*)
      real, intent(in) :: cpu0
      real, intent(inout) :: uvmax
      type (cct_par), intent(out) :: p_cct(method%m_iter)
    end subroutine mx_clean
  end interface
  !
  interface
    subroutine loadxy (method,huv,head,mapx,nx,mapy,ny)
      use clean_def
      use image_def
      !--------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Load x,y coordinates
      !--------------------------------------------------
      type (clean_par), intent(in) :: method
      type (gildas), intent(in) :: huv,head
      integer, intent(in) :: nx
      integer, intent(in) :: ny
      real, intent(out) :: mapx(nx)
      real, intent(out) :: mapy(ny)
    end subroutine loadxy
  end interface
  !
  interface
    subroutine mx_major_cycle90 (map,uvdata,nu,nv,   &
         &    w_weight,w_v,   &
         &    method,head,   &
         &    clean,beam,resid,nx,ny,nb,   &
         &    wcl,mcl,tcc,ncct,   &
         &    list, nl, nf, primary, weight,   &
         &    grid, mapu, mapv, mapx, mapy,   &
         &    wfft, cpu0, uvmax)
      use clean_def
      use image_def
      use gbl_message
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Support for MX
      !   Major cycle loop according to B.Clark idea
      !   Treat only one "plane/channel" at a time
      !   Plane is specified by method%iplane
      !----------------------------------------------------------------------
      type (clean_par), intent(inout) :: method
      type (uvmap_par), intent(in) :: map
      type (gildas), intent(inout) :: head
      !
      integer, intent(in) :: nu                 ! Visibility size
      integer, intent(in) :: nv                 ! Number of visibilities
      integer, intent(in) :: nb                 ! Number of beams
      real, intent(inout) :: uvdata(nu,nv)      ! Visibilities (iterated residuals)
      real, intent(in) ::  w_weight(nv)         ! Weight array
      real, intent(in) ::  w_v(nv)              ! V values
      real, intent(in) ::  uvmax                ! Max baseline
      !
      integer, intent(in) ::  nx,ny             ! X,Y size
      real, intent(in) ::  mapx(nx),mapy(ny)    ! X,Y coordinates
      real, intent(out) ::  mapu(*),mapv(*)      ! U,V coordinates
      real, intent(in) ::  grid(nx,ny)          ! Grid correction
      !
      real, intent(inout) ::  clean(nx,ny)      ! Clean image
      real, intent(inout) ::  resid(nx,ny)      ! Residuals
      real, intent(in) ::     beam(nx,ny,nb)    ! Synthesized beams
      integer, intent(in)  ::  nl               ! Search list size
      integer, intent(in)  ::  mcl              ! Maximum number of clean components
      integer, intent(out) ::  ncct             ! Number of clean components / plane
      real, intent(inout)  ::  wfft(*)          ! Work space for FFT
      type (cct_par), intent(out) :: tcc(method%m_iter) ! Clean components array
      integer, intent(in)  ::  list(nx*ny)      ! List of valid pixels
      !
      integer, intent(in) ::  nf                ! Number of fields
      real, intent(inout) ::  primary(nf,nx,ny) ! Primary beams
      real, intent(inout) ::  weight (nx,ny)    ! Flat field
      !
      type(cct_par), intent(inout) :: wcl(mcl)  ! Work array for Clean Components
      !
      real    maxc,minc,maxabs     !max et min de la carte, absolute
      integer imax,jmax,imin,jmin  !leurs coordonnees
      real    borne                !fraction de la carte initiale
      real    limite               !intensite minimum des pts retenus
      real    clarkl               !Clark worry limit
      real flux                    !Total clean flux density
      integer ncl                  !nb de pts reellement retenus
      logical fini                 !critere d'arret
      logical converge             ! Indique la conv par acc de flux
      logical error
      integer k, kcl
      character(len=message_length) :: chain
      !
      complex, allocatable :: tfgrid(:,:,:)
      complex, allocatable :: ftbeam(:,:)
      real, allocatable :: my_resid(:,:)
      !
      real cpu1,cpu0
    end subroutine mx_major_cycle90
  end interface
  !
  interface
    subroutine mx_uvsub90 (nx,ny,mapx,mapy,   &
         &    wcl,ncl,nu,nv,visi,ip)
      use clean_def
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Support for MX
      !   Subtract last major cycle components from UV table.
      !   Treats only one channel.
      !----------------------------------------------------------------------
      integer, intent(in) ::  nx,ny             ! X,Y size
      real, intent(in) ::  mapx(nx),mapy(ny)    ! X,Y coordinates
      integer, intent(in) :: nu                 ! Visibility size
      integer, intent(in) :: nv                 ! Number of visibilities
      integer, intent(in) :: ncl                ! Number of clean components
      real, intent(inout) :: visi(nu,nv)        ! Visibilities
      integer, intent(in) :: ip                 ! Current "plane/channel"
      type(cct_par), intent(inout) :: wcl(ncl)  ! Clean comp. value
    end subroutine mx_uvsub90
  end interface
  !
  interface
    subroutine one_beam_para (rname,map,huv,hbeam,hdirty,   &
         &    nx,ny,nu,nv,uvdata,   &
         &    w_mapu, w_mapv, w_grid, w_weight, w_v, do_weig,    &
         &    wcol,mcol,wfft,sblock,cpu0,error,uvmax)
      use clean_def
      use image_def
      use gbl_message
      !$ use omp_lib
      !------------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Compute a map from a CLIC UV Sorted Table
      !   by Gridding and Fast Fourier Transform, with
      !   one single beam for all channels.
      !
      ! Input :
      ! a precessed UV table, sorted in V, ordered in
      ! (U,V,W,D,T,iant,jant,nchan(real,imag,weig))
      ! Output :
      ! a beam image
      ! a VLM cube
      ! Work space :
      ! a  VLM complex Fourier cube (first V value is for beam)
      !------------------------------------------------------------------------
      character(len=*), intent(in) :: rname   ! Calling Task name
      type (uvmap_par), intent(inout) :: map  ! Mapping parameters
      type (gildas), intent(inout) :: huv     ! UV data set
      type (gildas), intent(inout) :: hbeam   ! Dirty beam data set
      type (gildas), intent(inout) :: hdirty  ! Dirty image data set
      integer, intent(in) :: nx   ! X size
      integer, intent(in) :: ny   ! Y size
      integer, intent(in) :: nu   ! Size of a visibilities
      integer, intent(in) :: nv   ! Number of visibilities
      real, intent(inout) :: uvdata(nu,nv)
      real, intent(inout) :: w_mapu(nx)      ! U grid coordinates
      real, intent(inout) :: w_mapv(ny)      ! V grid coordinates
      real, intent(inout) :: w_grid(nx,ny)   ! Gridding space
      real, intent(inout) :: w_weight(nv)    ! Weight of visibilities
      real, intent(inout) :: w_v(nv)         ! V values
      logical, intent(inout) :: do_weig
      !
      real, intent(inout) :: wfft(*)     ! Work space
      real, intent(inout) :: cpu0        ! CPU
      real, intent(inout) :: uvmax       ! Maximum baseline
      integer, intent(inout) :: sblock   ! Blocking factor
      integer, intent(inout) :: wcol     ! Weight channel
      integer, intent(inout) :: mcol(2)  ! First and last channel
      logical, intent(inout) :: error
    end subroutine one_beam_para
  end interface
  !
  interface
    subroutine one_beam_serial (rname,map,huv,hbeam,hdirty,   &
         &    nx,ny,nu,nv,uvdata,   &
         &    w_mapu, w_mapv, w_grid, w_weight, w_v, do_weig,    &
         &    wcol,mcol,wfft,sblock,cpu0,error,uvmax)
      use clean_def
      use image_def
      use gbl_message
      !------------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Compute a map from a CLIC UV Sorted Table
      !   by Gridding and Fast Fourier Transform, with
      !   one single beam for all channels.
      !
      ! Input :
      ! a precessed UV table, sorted in V, ordered in
      ! (U,V,W,D,T,iant,jant,nchan(real,imag,weig))
      ! Output :
      ! a beam image
      ! a VLM cube
      ! Work space :
      ! a  VLM complex Fourier cube (first V value is for beam)
      !------------------------------------------------------------------------
      character(len=*), intent(in) :: rname   ! Calling Task name
      type (uvmap_par), intent(inout) :: map  ! Mapping parameters
      type (gildas), intent(inout) :: huv     ! UV data set
      type (gildas), intent(inout) :: hbeam   ! Dirty beam data set
      type (gildas), intent(inout) :: hdirty  ! Dirty image data set
      integer, intent(in) :: nx   ! X size
      integer, intent(in) :: ny   ! Y size
      integer, intent(in) :: nu   ! Size of a visibilities
      integer, intent(in) :: nv   ! Number of visibilities
      real, intent(inout) :: uvdata(nu,nv)
      real, intent(inout) :: w_mapu(nx)      ! U grid coordinates
      real, intent(inout) :: w_mapv(ny)      ! V grid coordinates
      real, intent(inout) :: w_grid(nx,ny)   ! Gridding space
      real, intent(inout) :: w_weight(nv)    ! Weight of visibilities
      real, intent(inout) :: w_v(nv)         ! V values
      logical, intent(inout) :: do_weig
      !
      real, intent(inout) :: wfft(*)     ! Work space
      real, intent(inout) :: cpu0        ! CPU
      real, intent(inout) :: uvmax       ! Maximum baseline
      integer, intent(inout) :: sblock   ! Blocking factor
      integer, intent(inout) :: wcol     ! Weight channel
      integer, intent(inout) :: mcol(2)  ! First and last channel
      logical, intent(inout) :: error
    end subroutine one_beam_serial
  end interface
  !
  interface
    subroutine read_image (line,error)
      use gbl_message
      use clean_def
      use clean_arrays
      use clean_types
      !----------------------------------------------------------------------
      ! @ private
      ! Support routine for command
      !   READ Type File
      !     [/COMPACT]
      !     [/NOTRAIL]
      !     [/RANGE Start End Type]
      !     [/FREQUENCY RestFreqMHz]
      !     [/PLANE] (obsolete)
      !----------------------------------------------------------------------
      character(len=*), intent(in)    :: line
      logical,          intent(inout) :: error
    end subroutine read_image
  end interface
  !
  interface
    subroutine sub_read_image (name,ntype,dovrange,crange,compact,error, freq)
      use clean_def
      use clean_arrays
      use clean_types
      !---------------------------------------------------------------------
      ! @ private-mandatory
      !
      ! MAPPING
      !   Support routine for command
      !   READ Type File [/COMPACT] [/FREQUENCY RestFreqMHz] [/RANGE Start End Type]
      !   [/NOTRAIL] and obsolescent [/PLANE]
      !---------------------------------------------------------------------
      character(len=*), intent(in) :: name    ! Name of file
      integer, intent(in) :: ntype            ! Type of data
      real(8), intent(in) :: dovrange(2)        ! Range to be read
      character(len=*), intent(in) :: crange  ! Type of range
      logical, intent(in) :: compact          ! ACA buffer flag
      logical, intent(inout) :: error
      real(8), intent(in), optional :: freq   ! Desired rest frequency
    end subroutine sub_read_image
  end interface
  !
  interface
    subroutine sub_modified(atype,file,opti,nc,nochange)
      use gkernel_types
      use clean_types
      use clean_arrays
      use gbl_message
      !---------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Check if the file of type "atype" needs to be read again.
      !---------------------------------------------------------------------
      integer, intent(in) :: atype          ! Code of type of file
      character(len=*), intent(in) :: file  ! Filename
      type(readop_t), intent(inout) :: opti ! Status of corresponding buffer
      integer, intent(in) :: nc(2)          ! Range to be read
      logical, intent(out) :: nochange      ! Change status
    end subroutine sub_modified
  end interface
  !
  interface
    subroutine map_read (head,out,nc,compact,error)
      use clean_def
      use clean_arrays
      use clean_types
      use clean_default
      use code_names
      use gbl_message
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !  Read some type of input data.
      !------------------------------------------------------------------
      type (gildas), intent(inout) :: head ! Header of data
      character(len=*), intent(in) :: out  ! Desired data
      integer, intent(in) :: nc(2)         ! Channel range
      logical, intent(in) :: compact       ! Put in ACA space ?
      logical, intent(out) :: error
    end subroutine map_read
  end interface
  !
  interface
    function map_range (nc,hin,hou)
      use image_def
      !---------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Define channel range and set output Map header
      ! accordingly.
      !---------------------------------------------------
      logical :: map_range ! intent(out)
      integer, intent(in) :: nc(2)         ! Input channel range
      type (gildas), intent(inout) :: hin     ! Input header
      type (gildas), intent(inout) :: hou     ! Output header
    end function map_range
  end interface
  !
  interface
    subroutine out_range(rname,atype,dovrange,nc,head,error)
      use image_def
      use gbl_message
      !---------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Utility routine to define a channel range from a
      !   Velocity, Frequency or Channel range
      !---------------------------------------------------------------------
      character(len=*), intent(in)  :: rname        ! Caller name
      character(len=*), intent(in)  :: atype        ! Type of range
      real(kind=8),     intent(in)  :: dovrange(2)  ! Input range
      integer(kind=4),  intent(out) :: nc(2)        ! Ouput channel number
      type(gildas),     intent(in)  :: head         ! Reference GILDAS data frame
      logical,          intent(out) :: error        ! Error flag
    end subroutine out_range
  end interface
  !
  interface
    subroutine check_uvdata_type(huv,duv,map,error)
      use image_def
      use clean_def
      use gbl_message
      !-------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !     Internal routine
      ! Check number and types of fields
      !-------------------------------------------------------
      type(gildas), intent(inout) :: huv
      real, intent(inout) :: duv(:,:)
      type(uvmap_par), intent(inout) :: map
      logical, intent(inout) :: error
    end subroutine check_uvdata_type
  end interface
  !
  interface
    subroutine sg_map(task,line,error)
      use clean_def
      use clean_arrays
      use clean_types
      use clean_default
      use gbl_message
      !------------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !
      !     Support for commands
      !       UV_RESTORE
      !       UV_MAP
      ! and   MX (eventually, not coded yet)
      !
      ! Compute a map from a CLIC UV Sorted Table  by Gridding and Fast Fourier
      ! Transform, using adequate virtual memory space for optimisation. Also
      ! allows removal of Clean Components in the UV data and restoration of
      ! the clean image.
      !
      ! Input :
      !     a precessed UV table
      !     a list of Clean components
      ! Output :
      !     a precessed, rotated, shifted UV table, sorted in V,
      !     ordered in (U,V,W,D,T,iant,jant,nchan(real,imag,weig))
      !     a beam image or cube
      !     a LMV cube  (dirty or clean)
      !
      ! To be implemented
      !     Optionally, a Residual UV table
      !------------------------------------------------------------------------
      character(len=*), intent(in) :: task ! Caller (MX or UV_MAP or UV_RESTORE)
      character(len=*), intent(in) :: line ! Command line
      logical, intent(out) :: error
    end subroutine sg_map
  end interface
  !
  interface
    subroutine uvmap_and_restore (rname,map,huv,hbeam,hdirty,   &
         &    nx,ny,nu,nv,uvdata,r_weight, w_v, do_weig,    &
         &    rcol,wcol,mcol,sblock,cpu0,error,uvmax, &
         &    method, do_cct,hcct,hclean,do_slow,do_copy)
      use clean_def
      use clean_default
      use image_def
      use gbl_message
      !$ use omp_lib
      !------------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Compute a map from a CLIC UV Sorted Table
      !   by Gridding and Fast Fourier Transform, with
      !   a different beam per channel.
      !
      ! Input :
      ! a precessed UV table, sorted in V, ordered in
      ! (U,V,W,D,T,iant,jant,nchan(real,imag,weig))
      !
      ! (optionally) a list of Clean components
      !
      ! Output :
      ! a beam image
      ! a DIRTY or RESIDUAL and CLEAN cube
      !------------------------------------------------------------------------
      character(len=*), intent(in) :: rname   ! Calling Task name
      type (uvmap_par), intent(inout) :: map  ! Mapping parameters
      type (gildas), intent(inout) :: huv     ! UV data set
      type (gildas), intent(inout) :: hbeam   ! Dirty beam data set
      type (gildas), intent(inout) :: hdirty  ! Dirty image data set
      integer, intent(in) :: nx   ! X size
      integer, intent(in) :: ny   ! Y size
      integer, intent(in) :: nu   ! Size of a visibilities
      integer, intent(in) :: nv   ! Number of visibilities
      real, intent(inout) :: uvdata(nu,nv)
      real, intent(inout), target :: r_weight(nv)    ! Weight of visibilities
      real, intent(inout) :: w_v(nv)         ! V values
      logical, intent(inout) :: do_weig
      !
      real, intent(inout) :: cpu0        ! CPU
      real, intent(inout) :: uvmax       ! Maximum baseline
      integer, intent(inout) :: sblock   ! Blocking factor
      integer, intent(inout) :: rcol     ! Reference frequency channel
      integer, intent(inout) :: wcol     ! Weight channel
      integer, intent(inout) :: mcol(2)  ! First and last channel
      logical, intent(inout) :: error
      logical, intent(in) :: do_cct
      type (clean_par), intent(inout) :: method
      type (gildas), intent(inout) :: hcct
      type (gildas), intent(inout) :: hclean
      logical, intent(in) :: do_slow
      logical, intent(in) :: do_copy
    end subroutine uvmap_and_restore
  end interface
  !
  interface
    subroutine uv_removes_clean(huv,duv,ouv,nc,mic,dcct,freq,first,last, &
      & beams)
      use image_def
      use gbl_message
      !$ use omp_lib
      !-----------------------------------------------------------------
      ! @ private-mandatory
      !
      ! MAPPING   Support for UV_RESTORE
      !     Compute visibilities for a UV data set according to a
      !     set of Clean Components , and remove them from the original
      !     UV table.
      !       Semi-Slow version using interpolation from pre-tabulated
      !       Sin/Cos, which could still be optimized
      !-----------------------------------------------------------------
      type(gildas), intent(in) :: huv   ! header of UV data set
      real, intent(in) :: duv(:,:)      ! UV data set
      integer, intent(in) :: nc         ! Number of channels
      integer, intent(in) :: mic(:)     ! Number of Clean Components
      real, intent(out) :: ouv(:,:)     ! Extracted UV data set
      real, intent(in) :: dcct(:,:,:)   ! Clean component
      real(8), intent(in) :: freq       ! Apparent observing frequency
      integer, intent(in) :: first      ! First channel
      integer, intent(in) :: last       ! Last channel
      real, intent(in), optional :: beams(3) ! Mosaic primary beam (0 = single)
    end subroutine uv_removes_clean
  end interface
  !
  interface
    subroutine uv_squeeze_clean(nc,ccin,ccou, mic, first, last)
      use image_def
      !-----------------------------------------------------------------
      ! @ private-mandatory
      !
      ! MAPPING   Support for UV_RESTORE
      !   Compact the component list by summing up all values at the
      !   same position in a range of channels.
      !   The output list has a transposed order
      !-----------------------------------------------------------------
      integer, intent(in) :: nc         ! Number of channels
      real, intent(in) :: ccin(:,:,:)   ! Initial Clean Component List
      real, intent(out) :: ccou(:,:,:)  ! Resulting list
      integer, intent(inout) :: mic(:)  ! Number of Clean component per channel
      integer, intent(in) :: first      ! First channel
      integer, intent(in) :: last       ! Last channel
    end subroutine uv_squeeze_clean
  end interface
  !
  interface
    subroutine uv_removef_clean(hdirty,duv,ouv,nc,mic,dcct,freq,first,last, &
      & beams)
      use image_def
      use gbl_message
      !$ use omp_lib
      !-----------------------------------------------------------------
      ! @ private-mandatory
      !
      ! MAPPING   Support for UV_RESTORE
      !     Compute visibilities for a UV data set according to a
      !     set of Clean Components , and remove them from the original
      !     UV table
      !   This version is for tranpose CCT data (3,ncomp,nchannels)
      !   and uses an intermediate FFT for speed
      !-----------------------------------------------------------------
      type(gildas), intent(in) :: hdirty  ! header of Dirty image
      integer, intent(in) :: nc         ! Number of channels
      integer, intent(in) :: mic(:)     ! Number of Clean Components
      real, intent(in) :: duv(:,:)      ! Input visibilities
      real, intent(out) :: ouv(:,:)     ! Output visibilities
      real, intent(in) :: dcct(:,:,:)   ! Clean components
      real(8), intent(in) :: freq       ! Apparent observing frequency
      integer, intent(in) :: first      ! First
      integer, intent(in) :: last       ! and last channel
      real, intent(in), optional :: beams(3) ! Mosaic primary beam (in radians)
    end subroutine uv_removef_clean
  end interface
  !
  interface
    subroutine uv_extract_clean(duv,ouv,nc,first,last)
      !-----------------------------------------------------------------
      ! @ private-mandatory
      !
      ! MAPPING   Support for UV_MAP  &  UV_RESTORE
      !     Extract a subset of visibilities
      !-----------------------------------------------------------------
      integer, intent(in) :: nc         ! Number of channels
      real, intent(in) :: duv(:,:)      ! Input visbilities
      real, intent(out) :: ouv(:,:)     ! Output visibilities
      integer, intent(in) :: first      ! First channel
      integer, intent(in) :: last       ! Last channel
    end subroutine uv_extract_clean
  end interface
  !
  interface
    subroutine uv_clean_sizes(hcct,ccin, mic, first, last)
      use image_def
      !-----------------------------------------------------------------
      ! @ private-mandatory
      !
      ! MAPPING   Support for UV_RESTORE
      !   Compute the actual number of components
      !-----------------------------------------------------------------
      type(gildas), intent(in) :: hcct  ! header of CCT data set
      real, intent(in) :: ccin(:,:,:)   ! Clean component table
      integer, intent(out) :: mic(:)    ! Number of iterations per channel
      integer, intent(in) :: first      ! First
      integer, intent(in) :: last       ! and last channel
    end subroutine uv_clean_sizes
  end interface
  !
  interface
    subroutine do_smodel (visi,nc,nv,a,nx,ny,nf,freq,xinc,yinc,factor)
      !-----------------------------------------------------------------
      ! @ private-mandatory
      !
      ! MAPPING   Support for UV_RESTORE
      !
      !     Remove model visibilities from UV data set.
      !     Interpolate from a grid of values obtained by FFT before.
      !-----------------------------------------------------------------
      integer, intent(in) :: nc ! Visibility size
      integer, intent(in) :: nv ! Number of visibilities
      integer, intent(in) :: nx ! X size
      integer, intent(in) :: ny ! Y size
      integer, intent(in) :: nf ! Number of frequencies
      real, intent(inout) :: visi(:,:)    ! Computed visibilities
      real(8), intent(in) :: freq         ! Effective frequency
      real, intent(in)  :: factor         ! Flux factor
      complex, intent(in) ::  a(:,:,:)    ! FFT
      real(8), intent(in) :: xinc,yinc    ! Pixel sizes
    end subroutine do_smodel
  end interface
  !
  interface
    subroutine uv_shift_mosaic(line,error)
      use clean_def
      use clean_default
      use clean_arrays
      use phys_const
      use gkernel_types
      use gbl_message
      !-------------------------------------------------------------------
      ! @ private
      !  MAPPING
      !   Support routine for command
      !       UV_SHIFT  [Offx Offy [Angle]]
      !
      !   Phase shift a Mosaic or Single Dish UV Table to a new
      !   common phase center and orientation
      !     Offx OffY are offsets in ArcSec 
      !               (default 0,0)
      !     Angle     is the final position angle from North in Degree
      !               (default: no change)
      !
      !   Also called implicitely (with no command line arguments, i.e.
      !   no change of Phase Center unless the UV Table is with
      !   Phase Offsets )
      !   by command UV_MAP for Mosaics
      !-------------------------------------------------------------------
      character(len=*), intent(in)    :: line   ! Command line
      logical,          intent(inout) :: error  ! Logical error flag
    end subroutine uv_shift_mosaic
  end interface
  !
  interface
    subroutine uv_shift_center(line,abso,rela,angle,error)
      use phys_const
      use gbl_message
      use gkernel_types
      use clean_arrays
      !---------------------------------------------------------------------
      ! @ private
      ! Support routine for command UV_SHIFT
      ! Get relative and absolute positions from user inputs
      !---------------------------------------------------------------------
      character(len=*), intent(in)    :: line     !
      real(kind=8),     intent(out)   :: abso(2)  ! [rad] New absolute position
      real(kind=8),     intent(out)   :: rela(2)  ! [rad] New relative position from old absolute position
      real(kind=8),     intent(out)   :: angle    ! [rad] New angle
      logical,          intent(inout) :: error    !
    end subroutine uv_shift_center
  end interface
  !
  interface
    subroutine uv_sort (error,sorted,shift,new,uvmax,uvmin)
      use clean_def
      use clean_arrays
      use gbl_message
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Sort the input UV table
      !----------------------------------------------------------------------
      logical, intent(inout) :: sorted      ! Is table sorted ?
      logical, intent(inout) :: shift       ! Do we shift phase center ?
      logical, intent(out) :: error
      real(kind=8), intent(inout) :: new(3) ! New phase center and PA
      real, intent(out) :: uvmin            ! Min baseline
      real, intent(out) :: uvmax            ! Max baseline
    end subroutine uv_sort
  end interface
  !
  interface
    subroutine statistic (line,error)
      use clean_def
      use clean_arrays
      use clean_types
      use gbl_message
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !     Statistics on internal images
      !----------------------------------------------------------------------
      character(len=*), intent(inout) :: line
      logical, intent(out) :: error
    end subroutine statistic
  end interface
  !
  interface
    subroutine compute_stat(nx,ny,amap,box, &
         &    valmin,imin,jmin,valmax,imax,jmax,mean,rms,noise,np,nn, &
         &    support)
      !--------------------------------------------------------------
      ! @ private-mandatory
      !
      ! MAPPING
      !    Statistics on image
      !--------------------------------------------------------------
      integer, intent(out) :: np, nn
      integer, intent(in) :: nx,ny
      real, intent(in) :: amap(nx,ny)
      integer, intent(out) :: imin, jmin, imax, jmax
      integer, intent(in) ::  box(4)
      real, intent(out) :: valmin, valmax, mean, rms, noise
      logical, intent(in), optional :: support(nx,ny)
    end subroutine compute_stat
  end interface
  !
  interface
    subroutine do_stokes(line,comm,error)
      use gildas_def
      use gbl_message
      !
      ! @ private
      !
      ! STOKES Type FileIn [FileOut] 
      !
      character(len=*), intent(in) :: line
      character(len=*), intent(in) :: comm
      logical, intent(out) :: error
    end subroutine do_stokes
  end interface
  !
  interface
    subroutine sub_alma_quad (   &
         &    l_method,l_hdirty,l_hresid,l_hclean,   &
         &    l_hbeam,l_hprim,l_tfbeam, l_list,  &
         &    c_method,c_hdirty,c_hresid,c_hclean,   &
         &    c_hbeam,c_hprim,c_tfbeam, c_list,  &
         &    error,tcc)
      use clean_def
      use image_def
      use gbl_message
      !--------------------------------------------------------------
      ! @ private
      !
      ! MAPPING   Clean/Mosaic
      !     Perfom a CLEAN based on all CLEAN algorithms,
      !     except the Multi Resolution which requires a different tool
      !     Works for mosaic also, except for the Multi Scale clean
      !     (not yet implemented for this one, but feasible...)
      !--------------------------------------------------------------
      type (clean_par), intent(inout) :: l_method, c_method
      type (gildas), intent(inout)  :: l_hdirty, l_hbeam, l_hresid, l_hprim
      type (gildas), intent(inout)  :: l_hclean
      type (gildas), intent(inout)  :: c_hdirty, c_hbeam, c_hresid, c_hprim
      type (gildas), intent(inout)  :: c_hclean
      real, intent(inout) :: l_tfbeam(l_hbeam%gil%dim(2),l_hbeam%gil%dim(3),l_hbeam%gil%dim(1))
      real, intent(inout) :: c_tfbeam(c_hbeam%gil%dim(2),c_hbeam%gil%dim(3),c_hbeam%gil%dim(1))
      logical, intent(inout) :: error
      type (cct_par), intent(inout) :: tcc(l_method%m_iter)
      integer, intent(in) :: l_list(:)  ! List of searched pixels
      integer, intent(in) :: c_list(:)  ! List of searched pixels (compact array)
      !
      real, pointer :: c_dirty(:,:)    ! Dirty map
      real, pointer :: c_resid(:,:)    ! Iterated residual
      real, pointer :: c_clean(:,:)    ! Clean Map
      real, pointer :: c_dprim(:,:,:)  ! Primary beam
      real, pointer :: c_dbeam(:,:,:)  ! Dirty beam (per field)
      real, pointer :: c_weight(:,:)
      !
      real, pointer :: l_dirty(:,:)    ! Dirty map
      real, pointer :: l_resid(:,:)    ! Iterated residual
      real, pointer :: l_clean(:,:)    ! Clean Map
      real, pointer :: l_beam(:,:) ! Beam for fit
      real, pointer :: l_dprim(:,:,:)  ! Primary beam
      real, pointer :: l_dbeam(:,:,:)  ! Dirty beam (per field)
      real, pointer :: l_weight(:,:)
      !
      real, allocatable :: w_fft(:) ! TF work area
      complex, allocatable :: w_work(:,:)  ! Work area
      complex, allocatable :: c_work(:,:)  ! Expansion of residual
      real, allocatable :: r_work(:,:) ! Expansion of residual
      type(cct_par), allocatable :: w_comp(:)
      real, allocatable :: w_cctw(:,:), w_cctf(:,:)
      real, allocatable :: w_resid(:,:)
      type (cct_par), allocatable :: w_tcc(:)
      !
      integer icase, iext
      real last_flux, max_var, cpu0, cpu1
      integer iplane,ibeam
      integer nx,ny,np,mx,my,mp,nc
      integer ier, ix, iy, jx, jy, ncc
      real l_max, c_max
      logical do_fft, ok
      character(len=message_length) :: chain
      character(len=4) :: rname = 'ALMA'
      character(2) ans
      integer nker,mker
    end subroutine sub_alma_quad
  end interface
  !
  interface
    subroutine label_to_cct (image,nx,ny,c_method,label,mx,my,   &
         &    tcc,nv,ix,iy, icase)
      !
      use clean_def
      use gbl_message
      ! @ private
      type (clean_par), intent(in) :: c_method
      type (cct_par), intent(inout) :: tcc(*)
      integer, intent(in) :: nx,ny,mx,my,ix,iy
      integer, intent(out) :: nv
      real(4), intent(in) :: image(nx,ny)
      integer, intent(in) :: label(mx,my)
      integer, intent(in) :: icase
    end subroutine label_to_cct
  end interface
  !
  interface
    subroutine long_to_image (tcc,ncc,image,nx,ny)
      use clean_def
      ! @ private
      integer, intent(in) :: nx,ny,ncc
      type (cct_par), intent(in) :: tcc(ncc)
      real, intent(inout) :: image(nx,ny)
    end subroutine long_to_image
  end interface
  !
  interface
    subroutine image_to_long (image,nx,ny,tcc,mcc,ncc)
      use clean_def
      ! @ private
      integer, intent(in) :: nx,ny,mcc
      integer, intent(out) :: ncc
      type (cct_par), intent(out) :: tcc(mcc)
      real, intent(in) :: image(nx,ny)
    end subroutine image_to_long
  end interface
  !
  interface
    subroutine display_cct(method,tcc,ncc,ioff)
      use clean_def
      ! @ private
      integer, intent(in) :: ncc
      type(clean_par), intent(inout) :: method
      type(cct_par), intent(in) :: tcc(ncc)
      integer, intent(in) :: ioff
    end subroutine display_cct
  end interface
  !
  interface
    subroutine cct_normal (tcc,nc,fcomp,tfbeam,nx,ny,wfft,factor)
      !----------------------------------------------------------------------
      ! @ private
      !
      ! GILDAS: CLEAN   Internal routine
      !   Compute the SDI normalisation factor
      !----------------------------------------------------------------------
      use clean_def
      integer, intent(in) :: nx,ny,nc
      type (cct_par), intent(in) :: tcc(nc)
      real, intent(in) :: tfbeam(nx,ny)
      complex, intent(inout) :: fcomp(nx,ny)
      real, intent(inout) :: wfft(*)
      real factor
    end subroutine cct_normal
  end interface
  !
  interface
    subroutine choice_to_cct (c_method,tcc,nc,xcoord,ycoord,values)
      use clean_def
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING   Internal routine
      !   Put the selected components into the TCC structure
      !----------------------------------------------------------------------
      type (clean_par), intent(in) :: c_method
      integer, intent(in) :: nc
      type (cct_par), intent(inout) :: tcc(nc)
      integer, intent(in) :: xcoord(nc), ycoord(nc)
      real, intent(in) :: values(nc)
    end subroutine choice_to_cct
  end interface
  !
  interface
    subroutine compact_to_image (c_method,mx,my,tcc,ncc,image,nx,ny)
      use clean_def
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING   Internal routine
      !   Put the TCC values on the map grid
      !----------------------------------------------------------------------
      type (clean_par) :: c_method
      integer nx,ny,ncc,mx,my
      type (cct_par) :: tcc(ncc)
      real image(nx,ny)
    end subroutine compact_to_image
  end interface
  !
  interface
    subroutine sub_alma_bis (   &
         &    l_method,l_hdirty,l_hresid,l_hclean,   &
         &    l_hbeam,l_hprim,l_tfbeam,l_list,   &
         &    c_method,c_hdirty,c_hresid,c_hclean,   &
         &    c_hbeam,c_hprim,c_tfbeam,c_list,   &
         &    error,tcc)
      use clean_def
      use image_def
      !--------------------------------------------------------------
      ! @ private
      !
      ! MAPPING   Clean/Mosaic
      !     Perfom a CLEAN based on all CLEAN algorithms,
      !     except the Multi Resolution which requires a different tool
      !     Works for mosaic also, except for the Multi Scale clean
      !     (not yet implemented for this one, but feasible...)
      !--------------------------------------------------------------
      type (clean_par), intent(inout) :: l_method, c_method
      type (gildas), intent(inout)  :: l_hdirty, l_hbeam, l_hresid, l_hprim
      type (gildas), intent(inout)  :: l_hclean
      type (gildas), intent(inout)  :: c_hdirty, c_hbeam, c_hresid, c_hprim
      type (gildas), intent(inout)  :: c_hclean
      real, intent(inout) :: l_tfbeam(l_hbeam%gil%dim(2),l_hbeam%gil%dim(3),l_hbeam%gil%dim(1))
      real, intent(inout) :: c_tfbeam(c_hbeam%gil%dim(2),c_hbeam%gil%dim(3),c_hbeam%gil%dim(1))
      logical, intent(inout) :: error
      type (cct_par), intent(inout) :: tcc(l_method%m_iter)
      integer, intent(in) :: l_list(:)
      integer, intent(in) :: c_list(:)
      !
      real, pointer :: c_dirty(:,:)    ! Dirty map
      real, pointer :: c_resid(:,:)    ! Iterated residual
      real, pointer :: c_clean(:,:)    ! Clean Map
      real, pointer :: c_dprim(:,:,:)  ! Primary beam
      real, pointer :: c_dbeam(:,:,:)  ! Dirty beam (per field)
      real, pointer :: c_weight(:,:)
      !
      real, pointer :: l_dirty(:,:)    ! Dirty map
      real, pointer :: l_resid(:,:)    ! Iterated residual
      real, pointer :: l_clean(:,:)    ! Clean Map
      real, pointer :: l_beam(:,:) ! Beam for fit
      real, pointer :: l_dprim(:,:,:)  ! Primary beam
      real, pointer :: l_dbeam(:,:,:)  ! Dirty beam (per field)
      real, pointer :: l_weight(:,:)
      !
      real, allocatable :: w_fft(:) ! TF work area
      complex, allocatable :: w_work(:,:)  ! Work area
      complex, allocatable :: c_work(:,:)  ! Expansion of residual
      real, allocatable :: r_work(:,:) ! Expansion of residual
      type(cct_par), allocatable :: w_comp(:)
      real, allocatable :: w_cct(:,:)
      real, allocatable :: w_resid(:,:)
      !
      integer iplane,ibeam
      integer nx,ny,np,mx,my,mp,nc,kx,ky
      integer ip, ier, ix, iy
      real l_max, c_max
      logical do_fft, ok
      character(len=80) chain
      character(len=2) ans
      integer nker,mker
    end subroutine sub_alma_bis
  end interface
  !
  interface
    function imaxlst (method,list,resid,nx,ny,ix,jy)
      use clean_def
      !--------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Support for ALMA deconvolution
      !--------------------------------------------------------------
      real imaxlst  ! intent(out)
      type (clean_par), intent(in) :: method
      integer, intent(in) :: nx,ny
      integer, intent(out) :: ix,jy
      integer, intent(in) :: list(:)
      real, intent(in) :: resid(nx,ny)
    end function imaxlst
  end interface
  !
  interface
    subroutine remove_incompact(method,resid,nx,ny,tfbeam,wfft,   &
         &    np,primary,weight,tcc,first,last,mx,my)
      use clean_def
      !--------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Remove components in "compact" array
      !--------------------------------------------------------------
      type (clean_par), intent(inout) :: method
      !
      integer, intent(in) ::  np,nx,ny,mx,my,first,last
      real, intent(inout) ::  resid(nx,ny)
      real, intent(inout) ::  tfbeam(nx,ny,np)     ! T.F. du beam  Complex ?
      real, intent(in) :: primary(np,nx,ny)
      real, intent(in) :: weight(nx,ny)
      type(cct_par), intent(inout) :: tcc(last)
      real, intent(inout) :: wfft(*)
    end subroutine remove_incompact
  end interface
  !
  interface
    subroutine remove_inlong(method,resid,nx,ny,tfbeam,wfft,   &
         &    np,primary,weight,tcc,first,last)
      use clean_def
      !--------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Remove components in "long baseline" array
      !--------------------------------------------------------------
      type (clean_par), intent(inout) :: method
      !
      integer, intent(in) ::  np,nx,ny,first,last
      real, intent(inout) ::  resid(nx,ny)
      real, intent(inout) ::  tfbeam(nx,ny,np)     ! T.F. du beam  Complex ?
      real, intent(in) :: primary(np,nx,ny)
      real, intent(in) :: weight(nx,ny)
      type(cct_par), intent(inout) :: tcc(last)
      real, intent(inout) :: wfft(*)
    end subroutine remove_inlong
  end interface
  !
  interface
    subroutine expand_kernel(c_method,mx,my,ix,iy,c_max,   &
         &    tcc,first,last,nx,ny,kernel,nker)
      use clean_def
      ! @ private
      type (clean_par), intent(inout) :: c_method
      integer, intent(in) :: nx,ny
      integer, intent(in) :: mx,my
      integer, intent(in) :: ix,iy
      integer, intent(in) :: nker
      integer, intent(in) :: first
      integer, intent(inout) :: last
      real, intent(in) :: kernel(nker,nker)
      real, intent(in) :: c_max
      type (cct_par), intent(inout) :: tcc(*)
    end subroutine expand_kernel
  end interface
  !
  interface
    subroutine alma_make (method, l_hclean, tcc)
      use clean_def
      use image_def
      ! @ private
      type (clean_par), intent(inout) :: method
      type (gildas), intent(in) :: l_hclean
      type (cct_par), intent(in) :: tcc(method%n_iter)
    end subroutine alma_make
  end interface
  !
  interface
    subroutine sub_alma (   &
         &    l_method,l_hdirty,l_hresid,l_hclean,   &
         &    l_hbeam,l_hprim,l_tfbeam,l_list,   &
         &    c_method,c_hdirty,c_hresid,c_hclean,   &
         &    c_hbeam,c_hprim,c_tfbeam,c_list,   &
         &    error,tcc)
      use clean_def
      use image_def
      use gbl_message
      !--------------------------------------------------------------
      ! @ private
      !
      ! MAPPING   Clean/Mosaic
      !     Perfom a CLEAN based on all CLEAN algorithms,
      !     except the Multi Resolution which requires a different tool
      !     Works for mosaic also, except for the Multi Scale clean
      !     (not yet implemented for this one, but feasible...)
      !--------------------------------------------------------------
      type (clean_par), intent(inout) :: l_method, c_method
      type (gildas), intent(inout)  :: l_hdirty, l_hbeam, l_hresid, l_hprim
      type (gildas), intent(inout)  :: l_hclean
      type (gildas), intent(inout)  :: c_hdirty, c_hbeam, c_hresid, c_hprim
      type (gildas), intent(inout)  :: c_hclean
      real, intent(inout) :: l_tfbeam(l_hbeam%gil%dim(2),l_hbeam%gil%dim(3),l_hbeam%gil%dim(1))
      real, intent(inout) :: c_tfbeam(c_hbeam%gil%dim(2),c_hbeam%gil%dim(3),c_hbeam%gil%dim(1))
      logical, intent(inout) :: error
      type (cct_par), intent(inout) :: tcc(l_method%m_iter)
      integer, intent(in) :: l_list(:)
      integer, intent(in) :: c_list(:)
      !
      real, pointer :: c_dirty(:,:)    ! Dirty map
      real, pointer :: c_resid(:,:)    ! Iterated residual
      real, pointer :: c_clean(:,:)    ! Clean Map
      real, pointer :: c_dprim(:,:,:)  ! Primary beam
      real, pointer :: c_dbeam(:,:,:)  ! Dirty beam (per field)
      real, pointer :: c_weight(:,:)
      !
      real, pointer :: l_dirty(:,:)    ! Dirty map
      real, pointer :: l_resid(:,:)    ! Iterated residual
      real, pointer :: l_clean(:,:)    ! Clean Map
      real, pointer :: l_beam(:,:) ! Beam for fit
      real, pointer :: l_dprim(:,:,:)  ! Primary beam
      real, pointer :: l_dbeam(:,:,:)  ! Dirty beam (per field)
      real, pointer :: l_weight(:,:)
      !
      real, allocatable :: w_fft(:) ! TF work area
      complex, allocatable :: w_work(:,:)  ! Work area
      complex, allocatable :: c_work(:,:)  ! Expansion of residual
      real, allocatable :: r_work(:,:) ! Expansion of residual
      type(cct_par), allocatable :: w_comp(:)
      real, allocatable, target :: sc_beam(:,:,:)
      real, allocatable :: w_resid(:,:)
      !
      character(len=*), parameter :: rname='SUB_ALMA'
      !
      integer iplane,ibeam
      integer nx,ny,np,mx,my,mp,nc,kx,ky
      integer ip, ier
      real l_max, c_max
      logical do_fft, ok
      character(len=80) :: chain
      character(len=2) ans
      integer nker,mker
    end subroutine sub_alma
  end interface
  !
  interface
    function amaxlst (method,list,resid,nx,ny)
      use clean_def
      !-----------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !-----------------------------------------------------
      real amaxlst      ! intent(out)
      type (clean_par), intent(inout) :: method
      integer, intent(in) :: nx,ny
      integer, intent(in) :: list(:)
      real, intent(in) :: resid(nx,ny)
    end function amaxlst
  end interface
  !
  interface
    subroutine one_cycle90 (method,head,   &
         &    clean,beam,resid,nx,ny,tfbeam,fcomp,   &
         &    wcl,mcl,ixbeam,iybeam,ixpatch,iypatch,bgain,   &
         &    box, work, tcc, list, nl, nf, primary, weight, maxabs)
      use clean_def
      use image_def
      use gbl_message
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Major cycle loop according to B.Clark idea
      !----------------------------------------------------------------------
      type (clean_par), intent(inout) :: method
      type (gildas), intent(in) :: head
      !
      integer, intent(in) ::  nf,nx,ny,mcl,nl
      real, intent(inout) ::  clean(nx,ny)
      real, intent(inout) ::  resid(nx,ny)
      real, intent(in) ::     beam(nx,ny,nf)
      real, intent(inout) ::  tfbeam(nx,ny,nf)     ! T.F. du beam  Complex ?
      complex, intent(inout) :: fcomp(nx,ny)       ! T.F. du vecteur modification
      type(cct_par), intent(inout) ::  wcl(mcl)
      real, intent(in) ::  bgain                   ! Maximum sidelobe level
      integer, intent(in) ::  ixbeam, iybeam       ! Beam maximum position
      integer, intent(in) ::  ixpatch, iypatch     ! Beam patch radius
      integer, intent(in) ::  box(4)               ! Cleaning box
      real, intent(inout) ::  work(*)                 ! Work space for FFT
      type(cct_par), intent(inout) :: tcc(method%m_iter) ! Clean components array
      integer, intent(in) ::  list(nl)
      !
      real, intent(in) ::  primary(nf,nx,ny)       ! Primary beams
      real, intent(in) ::  weight (nx,ny)
      !
      real    maxabs               !max et min de la carte, absolute
    end subroutine one_cycle90
  end interface
  !
  interface
    subroutine remove_inother(method,resid,nx,ny,tfbeam,wfft,   &
         &    np,primary,weight,   &
         &    tcc,first,last,mx,my,kernel,nker)
      use clean_def
      !--------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Support for ALMA deconvolution
      !--------------------------------------------------------------
      type (clean_par), intent(inout) :: method
      !
      integer, intent(in) ::  np,nx,ny,mx,my,nker,first,last
      real, intent(inout) ::  resid(nx,ny)
      real, intent(inout) ::  tfbeam(nx,ny,np)     ! T.F. du beam  Complex ?
      real, intent(in) :: primary(np,nx,ny)
      real, intent(in) :: weight(nx,ny)
      real, intent(inout) :: kernel(nker,nker)
      real, intent(inout) :: wfft(*)
      type(cct_par), intent(inout) :: tcc(last)
    end subroutine remove_inother
  end interface
  !
  interface
    subroutine add_primker (clean,nx,ny,np,prim,value,kx,ky,kp,ker,nk)
      !-----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Smooth an array using a kernel, with Primary Beams
      !   For ALMA deconvolution
      !-----------------------------------------------------------------------
      integer, intent(in) :: nx,ny,np             ! Image size
      real, intent(in) :: value                   ! Input value
      real, intent(inout) :: clean(nx,ny)         ! Summed output array
      real, intent(in) :: prim(np,nx,ny)
      integer, intent(in) :: kp                   ! Field
      integer, intent(in) :: kx,ky                ! Center of value
      integer, intent(in) :: nk                   ! Kernel size
      real, intent(in) :: ker(nk,nk)
    end subroutine add_primker
  end interface
  !
  interface
    subroutine alma_make90 (method, l_hclean, c_hclean,   &
         &    tcc, kernel,nker)
      use clean_def
      use image_def
      !-------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Make clean map
      !-------------------------------------------------------------
      integer, intent(in) :: nker
      real, intent(in) :: kernel(nker,nker)
      type (clean_par), intent(inout) :: method
      type (gildas), intent(inout) :: l_hclean, c_hclean
      type (cct_par), intent(in) :: tcc(method%n_iter)
    end subroutine alma_make90
  end interface
  !
  interface
    subroutine sub_alma_ter (   &
         &    l_method,l_hdirty,l_hresid,l_hclean,   &
         &    l_hbeam,l_hprim,l_tfbeam,l_list,   &
         &    c_method,c_hdirty,c_hresid,c_hclean,   &
         &    c_hbeam,c_hprim,c_tfbeam,c_list,   &
         &    error,tcc)
      use clean_def
      use image_def
      use gbl_message
      !--------------------------------------------------------------
      ! @ private
      !
      ! Clean/Mosaic
      !     Perfom a CLEAN based on all CLEAN algorithms,
      !     except the Multi Resolution which requires a different tool
      !     Works for mosaic also, except for the Multi Scale clean
      !     (not yet implemented for this one, but feasible...)
      !--------------------------------------------------------------
      type (clean_par), intent(inout) :: l_method, c_method
      type (gildas), intent(inout) :: l_hdirty, l_hbeam, l_hresid, l_hprim
      type (gildas), intent(inout) :: l_hclean
      type (gildas), intent(inout) :: c_hdirty, c_hbeam, c_hresid, c_hprim
      type (gildas), intent(inout) :: c_hclean
      real, intent(inout) :: l_tfbeam(l_hbeam%gil%dim(2),l_hbeam%gil%dim(3),l_hbeam%gil%dim(1))
      real, intent(inout) :: c_tfbeam(c_hbeam%gil%dim(2),c_hbeam%gil%dim(3),c_hbeam%gil%dim(1))
      logical, intent(inout) :: error
      type (cct_par), intent(inout) :: tcc(l_method%m_iter)
      integer, intent(in) :: l_list(:)
      integer, intent(in) :: c_list(:)
      !
      real, pointer :: c_dirty(:,:)    ! Dirty map
      real, pointer :: c_resid(:,:)    ! Iterated residual
      real, pointer :: c_clean(:,:)    ! Clean Map
      real, pointer :: c_dprim(:,:,:)  ! Primary beam
      real, pointer :: c_dbeam(:,:,:)  ! Dirty beam (per field)
      real, pointer :: c_weight(:,:)
      !
      real, pointer :: l_dirty(:,:)    ! Dirty map
      real, pointer :: l_resid(:,:)    ! Iterated residual
      real, pointer :: l_clean(:,:)    ! Clean Map
      real, pointer :: l_beam(:,:) ! Beam for fit
      real, pointer :: l_dprim(:,:,:)  ! Primary beam
      real, pointer :: l_dbeam(:,:,:)  ! Dirty beam (per field)
      real, pointer :: l_weight(:,:)
      !
      real, allocatable :: w_fft(:) ! TF work area
      complex, allocatable :: w_work(:,:)  ! Work area
      complex, allocatable :: c_work(:,:)  ! Expansion of residual
      real, allocatable :: r_work(:,:) ! Expansion of residual
      type(cct_par), allocatable :: w_comp(:)
      real, allocatable :: w_resid(:,:)
      !
      integer iplane,ibeam
      integer nx,ny,np,mx,my,mp,nc
      integer ier, ix, iy, jx, jy
      real l_max, c_max
      logical do_fft, ok
      character(len=message_length) :: chain
      character(len=7) :: rname = 'COMPACT'
      character(len=2) ans
      integer nker,mker
    end subroutine sub_alma_ter
  end interface
  !
  interface
    subroutine load_cct (image,nx,ny,blc,trc,label,mx,my,   &
         &    nv,wcl,ix,iy,gain,tmax,ngoal)
      use clean_def
      ! @ private
      integer, intent(in) :: nx,ny,mx,my,blc(2),trc(2)
      integer, intent(out) :: nv
      integer, intent(in) :: ix,iy
      real(4), intent(in) :: image(nx,ny)
      integer, intent(inout) :: label(mx,my)
      type(cct_par), intent(inout) :: wcl(mx*my)
      integer, intent(in) :: ngoal
      real, intent(in) :: gain
      real, intent(out) :: tmax
    end subroutine load_cct
  end interface
  !
  interface
    subroutine init_convolve (i0,j0,nx,ny,beam,fbeam,area,work)
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING   FFT Utility routine
      !	  Compute the FT of a beam centered on pixel I0,J0
      !----------------------------------------------------------------------
      integer, intent(in) :: nx,ny          ! Problem size
      integer, intent(in) :: i0,j0          ! Position of peak
      complex, intent(out) :: fbeam(nx,ny)  ! TF of beam
      real, intent(in) :: beam(nx,ny)       ! Beam
      real, intent(inout) :: work(*)        ! Work space
      real, intent(out) :: area             ! Beam area
    end subroutine init_convolve
  end interface
  !
  interface
    subroutine sub_major(method,hdirty,hresid,hclean,   &
         &    hbeam,hprim,hmask,dcct,mask,list,error,        &
         &    major_plot, next_flux)
      use clean_def
      use image_def
      use gbl_message
      !--------------------------------------------------------------
      ! @ private
      !
      ! MAPPING Clean/Mosaic
      !     Perfom a CLEAN based on all CLEAN algorithms,
      !     except the MRC (Multi Resolution CLEAN)
      !     which requires a different tool
      !     Works for mosaic also, except for the Multi Scale clean
      !     (not yet implemented for this one, but feasible...)
      !--------------------------------------------------------------
      external :: major_plot
      external :: next_flux
      !
      type (clean_par), intent(inout) :: method
      type (gildas), intent(in) :: hdirty
      type (gildas), intent(inout) :: hbeam
      type (gildas), intent(inout) :: hclean
      type (gildas), intent(inout) :: hresid
      type (gildas), intent(in) :: hprim
      type (gildas), intent(in) :: hmask
      real, intent(inout) :: dcct(:,:,:) ! (3,hclean%gil%dim(3),*)
      logical, intent(in), target :: mask(:,:)
      integer, intent(in), target :: list(:)
      logical, intent(inout) ::  error
    end subroutine sub_major
  end interface
  !
  interface
    subroutine sub_major_omp(inout_method,hdirty,hresid,hclean,   &
         &    hbeam,hprim,hmask,dcct,mask,list,error,        &
         &    major_plot, next_flux)
      use clean_def
      use clean_default
      use clean_support
      use image_def
      use gbl_message
      !$ use omp_lib
      !--------------------------------------------------------------
      ! @ private
      !
      ! MAPPING Clean/Mosaic
      !     Perfom a CLEAN based on all CLEAN algorithms,
      !     except the MRC (Multi Resolution CLEAN)
      !     which requires a different tool
      !     Works for mosaic also, except for the Multi Scale clean
      !     (not yet implemented for this one, but feasible...)
      !--------------------------------------------------------------
      external :: major_plot
      external :: next_flux
      !
      type (clean_par), intent(inout) :: inout_method
      type (gildas), intent(in) :: hdirty
      type (gildas), intent(inout) :: hbeam
      type (gildas), intent(inout) :: hclean
      type (gildas), intent(inout) :: hresid
      type (gildas), intent(in) :: hprim
      type (gildas), intent(in) :: hmask
      real, intent(inout) :: dcct(:,:,:) ! (3,hclean%gil%dim(3),*)
      logical, intent(in), target :: mask(:,:)
      integer, intent(in), target :: list(:)
      logical, intent(inout) ::  error
    end subroutine sub_major_omp
  end interface
  !
  interface
    subroutine sub_major_lin(method,hdirty,hresid,hclean,   &
         &    hbeam,hprim,hmask,dcct,mask,list,error,        &
         &    major_plot, next_flux)
      use clean_def
      use image_def
      use gbl_message
      !--------------------------------------------------------------
      ! @ private
      !
      ! MAPPING Clean/Mosaic
      !     Perfom a CLEAN based on all CLEAN algorithms,
      !     except the MRC (Multi Resolution CLEAN)
      !     which requires a different tool
      !     Works for mosaic also, except for the Multi Scale clean
      !     (not yet implemented for this one, but feasible...)
      !--------------------------------------------------------------
      external :: major_plot
      external :: next_flux
      !
      type (clean_par), intent(inout) :: method
      type (gildas), intent(in) :: hdirty
      type (gildas), intent(inout) :: hbeam
      type (gildas), intent(inout) :: hclean
      type (gildas), intent(inout) :: hresid
      type (gildas), intent(in) :: hprim
      type (gildas), intent(in) :: hmask
      real, intent(inout) :: dcct(:,:,:) ! (3,hclean%gil%dim(3),*)
      logical, intent(in), target :: mask(:,:)
      integer, intent(in), target :: list(:)
      logical, intent(inout) ::  error
    end subroutine sub_major_lin
  end interface
  !
  interface
    subroutine get_beam(method,hbeam,hresid,hprim,  &
         &    tfbeam,w_work,w_fft,fhat,error, mask)
      use clean_def
      use image_def
      use gbl_message
      !-----------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !    Get beam related information
      !-----------------------------------------------------------------
      type (clean_par), intent(inout) :: method
      type (gildas), intent(in)  :: hbeam
      type (gildas), intent(in)  :: hresid
      type (gildas), intent(in)  :: hprim
      real, intent(inout) :: tfbeam(hbeam%gil%dim(1),hbeam%gil%dim(2),hbeam%gil%dim(3))
      complex, intent(inout) :: w_work(hbeam%gil%dim(1),hbeam%gil%dim(2))
      real, intent(inout) :: fhat,w_fft(:)
      logical, intent(inout) :: error
      logical, optional, intent(inout) :: mask(:,:)
    end subroutine get_beam
  end interface
  !
  interface
    subroutine sub_mrc(rname,method,hdirty,hresid,hclean,   &
         &    hbeam,hprim,mask,error, plot_routine)
      use clean_def
      use image_def
      use gbl_message
      !----------------------------------------------------------------------
      ! @ private
      !
      ! GILDAS  CLEAN Internal routine
      !   Implementation of Multi-Resolution CLEAN deconvolution algorithm.
      !   A smooth and a difference map are deconvolved using Clark's CLEAN
      !   with smooth and difference dirty beams.
      !----------------------------------------------------------------------
      character(len=*), intent(in) :: rname
      type (clean_par), intent(inout) :: method
      type (gildas), intent(in) :: hdirty, hbeam, hclean, hresid, hprim
      logical, target, intent(in) :: mask(:,:)
      logical, intent(out) :: error
      external :: plot_routine
    end subroutine sub_mrc
  end interface
  !
  interface
    subroutine mrc_maps (nx,ny,full,diff,wl,mx,my,smooth,ws,ratio,   &
         &    bmin,xinc,yinc,wfft)
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING CLEAN
      !   Multi-Resolution Clean
      !   Compute smooth and difference maps.
      !   Smoothing is done in UV plane, by Gauss convolution and
      !   Fourier truncation. Smooth map is smaller than original map
      !----------------------------------------------------------------------
      integer, intent(in) :: nx,ny,mx,my
      real, intent(in) :: full(nx,ny)
      real, intent(inout) :: diff(nx,ny),smooth(mx,my)
      real, intent(inout) :: wfft(*)
      real, intent(in) :: ratio,bmin,xinc,yinc
      complex, intent(inout) :: ws(mx,my),wl(nx,ny)
    end subroutine mrc_maps
  end interface
  !
  interface
    subroutine mrc_make_clean (nx,ny,beam,bmaj,bmin,pa,xinc,yinc)
      !--------------------------------------------------------------------
      ! @ private
      !
      ! MRC
      !     Make Clean Beam
      !--------------------------------------------------------------------
      integer, intent(in) :: nx,ny
      real, intent(out) :: beam(nx,ny)
      real, intent(in) :: bmaj,bmin,pa,xinc,yinc
    end subroutine mrc_make_clean
  end interface
  !
  interface
    subroutine mrc_setup (ratio,beam,bmin,xinc,yinc,   &
         &    nx,ny,dbeam,dbeam_fft,dwork,dmax,dxmax,dymax,   &
         &    mx,my,sbeam,sbeam_fft,swork,smax,sxmax,symax,   &
         &    type,work)
      use gbl_message
      !------------------------------------------------------------------
      ! @ private
      !
      ! MRC
      !     Compute Smooth and Difference BEAMS (dirty or clean) ie
      !        CALL MRC_MAPS + normalisations + TF
      !------------------------------------------------------------------
      character(len=*), intent(in) :: type
      integer, intent(in) :: nx,ny,mx,my
      real, intent(in) :: ratio,beam(nx,ny)
      real, intent(in) :: bmin,xinc,yinc
      real, intent(inout) :: dbeam(nx,ny),dmax
      integer, intent(inout) :: dxmax,dymax
      real, intent(inout) :: dbeam_fft(nx,ny)
      complex, intent(inout) :: dwork(nx,ny)
      real, intent(inout) :: sbeam(mx,my),smax
      integer, intent(inout) :: sxmax,symax
      real, intent(inout) :: sbeam_fft(mx,my)
      complex, intent(inout) :: swork(mx,my)
      real, intent(inout) :: work(*)
    end subroutine mrc_setup
  end interface
  !
  interface
    subroutine cmpmsk (s,mx,my,d,nx,ny,irat)
      !-----------------------------------------------------------------
      ! @ private
      !
      ! MRC
      !     Convert difference (ie initial) mask in smooth mask
      !-----------------------------------------------------------------
      integer, intent(in) ::  nx,ny,mx,my,irat
      logical, intent(out) :: s(mx,my)
      logical, intent(in) :: d(nx,ny)
    end subroutine cmpmsk
  end interface
  !
  interface
    subroutine restore_clean90 (method,clean,nx,ny,ft,   &
         &    tfbeam,scale,wfft,tcc,nc)
      !-----------------------------------------------------------------
      ! @ private
      !
      ! GILDAS: CLEAN   Internal routine
      ! Convolve source list into residual map using the Fourier method.
      ! The normalisation should be correct for flux or brightness maps,
      ! depending on value of SCALE.
      !-----------------------------------------------------------------
      use clean_def
      type (clean_par), intent(in) :: method
      integer, intent(in) :: nx, ny, nc
      real, intent(inout) :: clean(nx,ny),wfft(*)
      complex, intent(inout) :: ft(nx,ny)
      real, intent(in) :: tfbeam(nx,ny)
      real, intent(in) :: scale
      type (cct_par), intent(in) :: tcc(nc)
    end subroutine restore_clean90
  end interface
  !
  interface
    subroutine com_support(line,error)
      use gildas_def
      use clean_arrays
      use clean_support
      use gbl_message
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING Support routine for command
      !	  SUPPORT
      ! 1         [/PLOT]
      ! 2         [/CURSOR]
      ! 3         [/RESET]
      ! 4         [/MASK]
      ! 5         [/VARIABLE]
      ! 6         [/THRESHOLD  Raw Smooth [SmoothingLength]
      !----------------------------------------------------------------------
      character(len=*), intent(in)  :: line  ! Commandand line
      logical,          intent(out) :: error ! Logical error flag
    end subroutine com_support
  end interface
  !
  interface
    subroutine mask_clean (head,mask,data,raw,smo,length,margin,error) 
      use clean_def
      use image_def
      use gbl_message
      !----------------------------------------------------------------
      ! @ private-mandatory
      !
      ! MAPPING
      !   Build a Mask from a Threshold and Smoothing 
      !----------------------------------------------------------------
      type (gildas), intent(inout) :: head
      real, intent(out), target :: mask(head%gil%dim(1),head%gil%dim(2),head%gil%dim(3))
      real, intent(in) :: data (head%gil%dim(1),head%gil%dim(2),head%gil%dim(3))
      real, intent(in) :: raw
      real, intent(in) :: smo
      real, intent(in) :: length
      real, intent(in) :: margin
      logical, intent(out) :: error
    end subroutine mask_clean
  end interface
  !
  interface
    subroutine check_area(method,head,quiet)
      use clean_def
      use image_def
      use gbl_message
      !-----------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Check search area for Cleaning
      !-----------------------------------------------
      type (gildas), intent(in) :: head
      type (clean_par), intent(inout) :: method
      logical, intent(in) :: quiet
    end subroutine check_area
  end interface
  !
  interface
    subroutine beam_plane(method,hbeam,hdirty)
      use clean_def
      use image_def
      use gbl_message
      !----------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Define beam plane and check if fit is required
      !----------------------------------------------------------------
      type (clean_par), intent(inout) :: method
      type (gildas), intent(in) :: hbeam
      type (gildas), intent(in) :: hdirty ! Used to figure out which plane
    end subroutine beam_plane
  end interface
  !
  interface
    subroutine clean_make90(method,hclean,clean,tcc)
      use clean_def
      use image_def
      !----------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Build clean image from Component List
      !----------------------------------------------------------------
      type (clean_par), intent(inout) :: method
      type (gildas), intent(inout) :: hclean
      real, intent(inout) :: clean(hclean%gil%dim(1),hclean%gil%dim(2))
      type (cct_par), intent(in) :: tcc(method%n_iter)
    end subroutine clean_make90
  end interface
  !
  interface
    subroutine guess (nv,values,coords,par)
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Support for FIT of beam
      !   Setup starting values for beam fit
      !----------------------------------------------------------------------
      integer, intent(in)  :: nv         ! Number of data
      real, intent(in)  :: values(nv)    ! Data values
      real, intent(in)  :: coords(2,nv)  ! X and Y coordinates of data
      real, intent(out) :: par(6)        ! Guessed parameters
    end subroutine guess
  end interface
  !
  interface
    subroutine min2d (npar,g,f,var,iflag)
      use gildas_def
      use mod_fitbeam
      !------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Support for FIT of beam
      !   minimizing function
      !------------------------------------------------------------------
      integer, intent(in) :: npar        ! Number of parameters
      real(8), intent(out)  :: g(npar)   ! First derivatives
      real(8), intent(out)  :: f         ! Function value
      real(8), intent(in)  :: var(npar)  ! Parameter values
      integer, intent(in)  :: iflag      ! Operation code
    end subroutine min2d
  end interface
  !
  interface
    subroutine mn2d (npar,g,f,var,iflag,values,coords,nv,   &
         &    sigbas,sigrai)
      !---------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Support for FIT of beam
      !---------------------------------------------------------------------
      integer, intent(in) :: npar        ! Number of parameters
      real(8), intent(out)  :: g(npar)   ! First derivatives
      real(8), intent(out)  :: f         ! Function value
      real(8), intent(in)  :: var(npar)  ! Parameter values
      integer, intent(in)  :: iflag      ! Operation code
      integer, intent(in)  :: nv         ! Number of data
      real, intent(in)  :: values(nv)    ! Data values
      real, intent(in)  :: coords(2,nv)  ! X and Y coordinates of data
      real, intent(out)  :: sigbas       ! Baseline noise
      real, intent(out)  :: sigrai       ! On Source noise
    end subroutine mn2d
  end interface
  !
  interface
    subroutine fit2d (fcn,liter,error)
      use gildas_def
      use fit_minuit
      use mod_fitbeam
      !---------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Support for FIT of beam
      !   Setup and starts a GAUSS fit minimisation using MINUIT
      !---------------------------------------------------------------------
      external :: fcn                   !   Function to be minimized
      logical, intent(in) :: liter      !   Iterate ?
      logical, intent(out) :: error     !   Error flag
    end subroutine fit2d
  end interface
  !
  interface
    subroutine mid2d(fit,ifatal,liter)
      use gildas_def
      use mod_fitbeam
      use fit_minuit
      !---------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Support for FIT of beam
      !   Start a gaussian fit by building the PAR array and internal
      !   variable used by Minuit
      !---------------------------------------------------------------------
      type(fit_minuit_t), intent(inout) :: fit  ! Fitting variables
      integer, intent(out) :: ifatal            ! Error code
      logical, intent(in) :: liter              ! Iterate ?
    end subroutine mid2d
  end interface
  !
  interface
    subroutine loadv (image,nx,ny,blc,trc,label,mx,my,   &
         &    nv,values,coords)
    !---------------------------------------------------------
    ! @ private
    !
    ! MAPPING
    !   Support for FIT of beam
    !   Load the values from the region around the peak
    !   of the primary beam (assumed centered).
    !---------------------------------------------------------
      integer, intent(in) :: nx               ! X size
      integer, intent(in) :: ny               ! Y size
      real(4), intent(in) :: image(nx,ny)     ! Image
      integer, intent(in) :: blc(2)           ! BLC
      integer, intent(in) :: trc(2)           ! TRC
      integer, intent(in) :: mx               ! Output X size
      integer, intent(in) :: my               ! Output Y size
      integer, intent(in) :: label(mx,my)     ! Field identification
      integer, intent(out) :: nv              ! Number of values
      real(4), intent(out) :: values(mx*my)   ! Selected values
      real(4), intent(out) :: coords(2,mx*my) ! X and Y pixel numbers
    end subroutine loadv
  end interface
  !
  interface
    subroutine threshold (image,nx,ny,blc,trc,labelo,mx,my,   &
         &    nf,labeli,labelf,mf,thre,   &
         &    blank, eblank)
      !---------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Support for FIT of beam
      !   Divide an image in connex labelled areas above (or below)
      !   a given threshold
      !---------------------------------------------------------------------
      integer, intent(in) :: nx              ! X size
      integer, intent(in) :: ny              ! Y size
      real(4), intent(in) :: image(nx,ny)    ! Image
      integer, intent(in) :: blc(2)          ! BLC
      integer, intent(in) :: trc(2)          ! TRC
      integer, intent(in) :: mx              ! Output X size
      integer, intent(in) :: my              ! Output Y size
      integer, intent(out) :: labelo(mx,my)  ! Field identification
      integer, intent(out) :: nf             ! Number of fields
      integer, intent(in) :: mf              ! Maximum number of fields
      integer, intent(out) :: labeli(mf)     ! work area: intermediate numbers
      integer, intent(out) :: labelf(mf)     ! Final field number
      real(4), intent(in) :: thre            ! Threshold for field id
      real(4), intent(in) :: blank           ! Blanking value
      real(4), intent(in) :: eblank          ! Tolerance on blanking
    end subroutine threshold
  end interface
  !
  interface
    subroutine descen(label,labin,labout)
      !----------------------------------------------------
      ! @ private
      !   Identify connex areas and propagate label
      !----------------------------------------------------
      integer, intent(in) :: label(*)     ! Labels
      integer, intent(in) :: labin        ! Starting label
      integer, intent(out) :: labout      ! Ending label
    end subroutine descen
  end interface
  !
  interface
    subroutine dofft_test (np,nv,visi,jx,jy,jo   &
         &    ,nc,nx,ny,map,mapx,mapy,sup,cell,taper,we,vv   &
         &    ,ubias,vbias,ubuff,vbuff,ctype)
      !----------------------------------------------------------------------
      ! @  private
      !
      ! GILDAS  UV_MAP
      !   Compute FFT of image by gridding UV data
      !   Test version to compare speed of various methods
      !----------------------------------------------------------------------
      integer, intent(in) :: nv                   ! number of values
      integer, intent(in) :: np                   ! Number of "visibilities", normally 7+3*nc
      real, intent(in) :: visi(np,nv)             ! values
      integer, intent(in) :: nc                   ! number of channels
      integer, intent(in) :: jx                   ! X coord location in VISI
      integer, intent(in) :: jy                   ! Y coord location in VISI
      integer, intent(in) :: jo                   ! first channel to map
      integer, intent(in) :: nx                   ! X map size
      integer, intent(in) :: ny                   ! Y map size
      real, intent(out) :: map(2*(nc+1),nx,ny)    ! gridded visibilities
      real, intent(in) :: mapx(nx)                ! X Coordinates of grid
      real, intent(in) :: mapy(ny)                ! Y Coordinates of grid
      real, intent(in) :: sup(2)                  ! Support of convolving function in Meters
      real, intent(in) :: cell(2)                 ! cell size in Meters
      real, intent(in) :: taper(4)                ! 1/e taper in Meters + Angle in Radians
      real, intent(in) :: we(nv)                  ! Weight array
      real, intent(in) :: vv(nv)                  ! V Values
      real, intent(in) :: ubias                   ! U gridding offset
      real, intent(in) :: vbias                   ! V gridding offset
      real, intent(in) :: ubuff(4096)             ! U gridding buffer
      real, intent(in) :: vbuff(4096)             ! V gridding buffer
      integer, intent(in) :: ctype                ! type of gridding
    end subroutine dofft_test
  end interface
  !
  interface
    subroutine dofft_quick (np,nv,visi,jx,jy,jo   &
         &    ,nc,nx,ny,map,mapx,mapy,sup,cell,taper,we,vv,   &
         &    ubias,vbias,ubuff,vbuff)
      !----------------------------------------------------------------------
      ! @ private
      !
      ! GILDAS  MAP_FAST
      !   Compute FFT of image by gridding UV data
      !   - Taper before gridding
      !   - Gridding with pre-computed support
      !   - Uses symmetry
      !----------------------------------------------------------------------
      integer, intent(in) :: nv                   ! number of values
      integer, intent(in) :: np                   ! Number of "visibilities", normally 7+3*nc
      real, intent(in) :: visi(np,nv)             ! values
      integer, intent(in) :: nc                   ! number of channels
      integer, intent(in) :: jx                   ! X coord location in VISI
      integer, intent(in) :: jy                   ! Y coord location in VISI
      integer, intent(in) :: jo                   ! first channel to map
      integer, intent(in) :: nx                   ! X map size
      integer, intent(in) :: ny                   ! Y map size
      real, intent(out) :: map(2*(nc+1),nx,ny)    ! gridded visibilities
      real, intent(in) :: mapx(nx)                ! X Coordinates of grid
      real, intent(in) :: mapy(ny)                ! Y Coordinates of grid
      real, intent(in) :: sup(2)                  ! Support of convolving function in Meters
      real, intent(in) :: cell(2)                 ! cell size in Meters
      real, intent(in) :: taper(4)                ! 1/e taper in Meters + Angle in Radians
      real, intent(in) :: we(nv)                  ! Weight array
      real, intent(in) :: vv(nv)                  ! V Values
      real, intent(in) :: ubias                   ! U gridding offset
      real, intent(in) :: vbias                   ! V gridding offset
      real, intent(in) :: ubuff(4096)             ! U gridding buffer
      real, intent(in) :: vbuff(4096)             ! V gridding buffer
    end subroutine dofft_quick
  end interface
  !
  interface
    subroutine dofft_fast (np,nv,visi,jx,jy,jo   &
         &    ,nc,nx,ny,map,mapx,mapy,sup,cell,taper,we,vv)
      !----------------------------------------------------------------------
      ! @ private
      !
      ! GILDAS  MAP_FAST
      !   Compute FFT of image by gridding UV data
      !   Taper before gridding
      !   Only for "visibility in cell" gridding.
      !   Uses symmetry
      !----------------------------------------------------------------------
      integer, intent(in) :: nv                   ! number of values
      integer, intent(in) :: np                   ! Number of "visibilities", normally 7+3*nc
      real, intent(in) :: visi(np,nv)             ! values
      integer, intent(in) :: nc                   ! number of channels
      integer, intent(in) :: jx                   ! X coord location in VISI
      integer, intent(in) :: jy                   ! Y coord location in VISI
      integer, intent(in) :: jo                   ! first channel to map
      integer, intent(in) :: nx                   ! X map size
      integer, intent(in) :: ny                   ! Y map size
      real, intent(out) :: map(2*(nc+1),nx,ny)    ! gridded visibilities
      real, intent(in) :: mapx(nx)                ! X Coordinates of grid
      real, intent(in) :: mapy(ny)                ! Y Coordinates of grid
      real, intent(in) :: sup(2)                  ! Support of convolving function in Meters
      real, intent(in) :: cell(2)                 ! cell size in Meters
      real, intent(in) :: taper(4)                ! 1/e taper in Meters + Angle in Radians
      real, intent(in) :: we(nv)                  ! Weight array
      real, intent(in) :: vv(nv)                  ! V Values
    end subroutine dofft_fast
  end interface
  !
  interface
    subroutine dofft_slow (np,nv,visi,jx,jy,jo   &
         &    ,nc,nx,ny,map,mapx,mapy,sup,cell,taper,we,vv,   &
         &    ubias,vbias,ubuff,vbuff)
      !----------------------------------------------------------------------
      ! @ private
      !
      ! GILDAS  UVMAP
      !   Compute FFT of image by gridding UV data
      !   Taper after gridding
      !   Uses symmetry
      !----------------------------------------------------------------------
      integer, intent(in) :: nv                   ! number of values
      integer, intent(in) :: np                   ! Number of "visibilities", normally 7+3*nc
      real, intent(in) :: visi(np,nv)             ! values
      integer, intent(in) :: nc                   ! number of channels
      integer, intent(in) :: jx                   ! X coord location in VISI
      integer, intent(in) :: jy                   ! Y coord location in VISI
      integer, intent(in) :: jo                   ! first channel to map
      integer, intent(in) :: nx                   ! X map size
      integer, intent(in) :: ny                   ! Y map size
      real, intent(out) :: map(2*(nc+1),nx,ny)    ! gridded visibilities
      real, intent(in) :: mapx(nx)                ! X Coordinates of grid
      real, intent(in) :: mapy(ny)                ! Y Coordinates of grid
      real, intent(in) :: sup(2)                  ! Support of convolving function in Meters
      real, intent(in) :: cell(2)                 ! cell size in Meters
      real, intent(in) :: taper(4)                ! 1/e taper in Meters + Angle in Radians
      real, intent(in) :: we(nv)                  ! Weight array
      real, intent(in) :: vv(nv)                  ! V Values
      real, intent(in) :: ubias                   ! U gridding offset
      real, intent(in) :: vbias                   ! V gridding offset
      real, intent(in) :: ubuff(4096)             ! U gridding buffer
      real, intent(in) :: vbuff(4096)             ! V gridding buffer
    end subroutine dofft_slow
  end interface
  !
  interface
    subroutine dofft_quick1 (np,nv,visi,jx,jy,jo   &
         &    ,nc,nx,ny,map,mapx,mapy,sup,cell,taper,we,vv,   &
         &    ubias,vbias,ubuff,vbuff)
      !----------------------------------------------------------------------
      ! @ private
      !
      ! GILDAS  MAP_FAST
      !   Compute FFT of image by gridding UV data
      !   For any gridding support
      !   Taper before gridding
      !   Does not use symmetry
      !----------------------------------------------------------------------
      integer, intent(in) :: nv                   ! number of values
      integer, intent(in) :: np                   ! Number of "visibilities", normally 7+3*nc
      real, intent(in) :: visi(np,nv)             ! values
      integer, intent(in) :: nc                   ! number of channels
      integer, intent(in) :: jx                   ! X coord location in VISI
      integer, intent(in) :: jy                   ! Y coord location in VISI
      integer, intent(in) :: jo                   ! first channel to map
      integer, intent(in) :: nx                   ! X map size
      integer, intent(in) :: ny                   ! Y map size
      real, intent(out) :: map(2*(nc+1),nx,ny)    ! gridded visibilities
      real, intent(in) :: mapx(nx)                ! X Coordinates of grid
      real, intent(in) :: mapy(ny)                ! Y Coordinates of grid
      real, intent(in) :: sup(2)                  ! Support of convolving function in Meters
      real, intent(in) :: cell(2)                 ! cell size in Meters
      real, intent(in) :: taper(4)                ! 1/e taper in Meters + Angle in Radians
      real, intent(in) :: we(nv)                  ! Weight array
      real, intent(in) :: vv(nv)                  ! V Values
      real, intent(in) :: ubias                   ! U gridding offset
      real, intent(in) :: vbias                   ! V gridding offset
      real, intent(in) :: ubuff(4096)             ! U gridding buffer
      real, intent(in) :: vbuff(4096)             ! V gridding buffer
    end subroutine dofft_quick1
  end interface
  !
  interface
    subroutine dofft_fast1 (np,nv,visi,jx,jy,jo   &
         &    ,nc,nx,ny,map,mapx,mapy,sup,cell,taper,we,vv)
      !----------------------------------------------------------------------
      ! @ private
      !
      ! GILDAS  MAP_FAST
      !   Compute FFT of image by gridding UV data
      !   Only for "visibility in cell"
      !   Taper before gridding
      !   Do not use symmetry
      !----------------------------------------------------------------------
      integer, intent(in) :: nv                   ! number of values
      integer, intent(in) :: np                   ! Number of "visibilities", normally 7+3*nc
      real, intent(in) :: visi(np,nv)             ! values
      integer, intent(in) :: nc                   ! number of channels
      integer, intent(in) :: jx                   ! X coord location in VISI
      integer, intent(in) :: jy                   ! Y coord location in VISI
      integer, intent(in) :: jo                   ! first channel to map
      integer, intent(in) :: nx                   ! X map size
      integer, intent(in) :: ny                   ! Y map size
      real, intent(out) :: map(2*(nc+1),nx,ny)    ! gridded visibilities
      real, intent(in) :: mapx(nx)                ! X Coordinates of grid
      real, intent(in) :: mapy(ny)                ! Y Coordinates of grid
      real, intent(in) :: sup(2)                  ! Support of convolving function in Meters
      real, intent(in) :: cell(2)                 ! cell size in Meters
      real, intent(in) :: taper(4)                ! 1/e taper in Meters + Angle in Radians
      real, intent(in) :: we(nv)                  ! Weight array
      real, intent(in) :: vv(nv)                  ! V Values
    end subroutine dofft_fast1
  end interface
  !
  interface
    subroutine comshi (beam,nx,ny,ix,iy,shift)
      use gbl_message
      !-----------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Shift beam if needed
      !-----------------------------------------------------
      integer, intent(in) :: nx,ny     ! X,Y Size
      integer, intent(in) :: ix,iy     ! Position of maximum
      real, intent(in) :: beam(nx,ny)  ! Beam
      integer, intent(out) :: shift(3) ! Shift information
    end subroutine comshi
  end interface
  !
  interface
    subroutine no_check_mask(method,head)
      use clean_def
      use image_def
      !--------------------------------------------------------------
      ! @ private
      !
      ! MAPPING   Clean/Mosaic
      !     Check that the search mask is defined
      !--------------------------------------------------------------
      type(gildas),    intent(in)    :: head   !
      type(clean_par), intent(inout) :: method !
    end subroutine no_check_mask
  end interface
  !
  interface
    subroutine check_mask(amethod,head)
      use clean_def
      use image_def
      use gbl_message
      use gkernel_types
      use clean_support
      use clean_arrays
      !--------------------------------------------------------------
      ! @ private
      !
      ! MAPPING   Clean/Mosaic
      !     Check that the search mask is defined
      !--------------------------------------------------------------
      type(gildas),    intent(in)    :: head   !
      type(clean_par), intent(inout) :: amethod !
    end subroutine check_mask
  end interface
  !
  interface
    subroutine translate (in, nx, ny, trans, ix, iy)
      !-----------------------------------------------------------------------
      ! @ private
      !
      !     Translate the beam to new position IX, IY
      !-----------------------------------------------------------------------
      integer, intent(in) :: nx          ! X size
      integer, intent(in) :: ny          ! Y size
      real, intent(in) :: in(nx,ny)      ! Input beam
      real, intent(out) :: trans(nx,ny)  ! Translated beam
      integer, intent(in) :: ix          ! X shift
      integer, intent(in) :: iy          ! Y shift
    end subroutine translate
  end interface
  !
  interface
    subroutine get_maskplane(amethod,hmask,hdirty,mask,list)
      use clean_def
      use image_def
      use gbl_message
      use clean_support
      !----------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Define beam plane and check if fit is required
      !----------------------------------------------------------------
      type (clean_par), intent(inout) :: amethod
      type (gildas), intent(in) :: hmask
      type (gildas), intent(in) :: hdirty ! Used to figure out which plane
      logical, intent(inout) :: mask(:,:)
      integer, intent(inout) :: list(:)
    end subroutine get_maskplane
  end interface
  !
  interface
    subroutine init_flux90 (method,head,ylimn,ylimp,ipen)
      use clean_def
      use image_def
      !---------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Support for deconvolution
      !   Create or reuse the <FLUX window
      !---------------------------------------------------------------------
      type (clean_par), intent(in) :: method
      type (gildas), intent(in)    :: head
      real, intent(in)             :: ylimn, ylimp
      integer, intent(out)         :: ipen
    end subroutine init_flux90
  end interface
  !
  interface
    subroutine plot_multi90(niter,flux,is)
      use clean_def
      use image_def
      !---------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Support for deconvolution
      !   Plot in the <FLUX window
      !---------------------------------------------------------------------
      integer, intent(in) :: is
      integer, intent(in) :: niter
      real, intent(in) :: flux
    end subroutine plot_multi90
  end interface
  !
  interface
    subroutine next_flux90(niter,cum)
      !---------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Draw in the <FLUX window
      !---------------------------------------------------------------------
      integer, intent(in) :: niter
      real, intent(in) :: cum
    end subroutine next_flux90
  end interface
  !
  interface
    subroutine close_flux90(ipen,error)
      !---------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Close the <FLUX segments
      !---------------------------------------------------------------------
      integer, intent(in)    :: ipen   ! New pen to be used
      logical, intent(inout) :: error  ! Logical error flag
    end subroutine close_flux90
  end interface
  !
  interface
    subroutine re_mask(method,head,nl,error)
      use clean_def
      use image_def
      use clean_support
      !---------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Support for Clean
      !   Re-compute mask in Major Cycles
      !---------------------------------------------------------------------
      type (clean_par), intent(inout) :: method
      type (gildas), intent(in) :: head
      integer, intent(inout) :: nl
      logical, intent(inout) :: error
    end subroutine re_mask
  end interface
  !
  interface
    subroutine init_plot(method,head,pdata)
      use clean_def
      use image_def
      !---------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Create or reuse the <CLARK window, and draw in it. Then go back
      !   to <FLUX if needed.
      !---------------------------------------------------------------------
      type (clean_par), intent(in) :: method
      type (gildas), intent(in)    :: head
      real, intent(in)             :: pdata(head%gil%dim(1),head%gil%dim(2))
    end subroutine init_plot
  end interface
  !
  interface
    subroutine major_plot90 (method,head   &
         &    ,conv,niter,nx,ny,np,tcc  &
         &    ,clean,resid,poids)
      use clean_def
      use image_def
      use gbl_message
      !---------------------------------------------------------
      ! @ private
      !
      ! MAPPING 
      !   Plot result of Major Cycle
      !---------------------------------------------------------
      type (clean_par), intent(inout) :: method
      type (gildas), intent(in) :: head
      !
      logical, intent(inout) :: conv        ! Convergence status
      integer, intent(in) ::  niter         ! Number of iterations
      integer, intent(in) ::  nx            ! X size
      integer, intent(in) ::  ny            ! Y size
      integer, intent(in) ::  np            ! Number of planes
      real, intent(inout) :: clean(nx,ny)      ! Clean image
      real, intent(inout) :: resid(nx,ny)      ! Residuals
      real, intent(in) :: poids(nx,ny)         ! Weight image
      type (cct_par), intent(in) :: tcc(niter)
    end subroutine major_plot90
  end interface
  !
  interface
    subroutine plot_mrc(method,head,array,code)
      use image_def
      use clean_def
      !----------------------------------------------------------------
      ! 
      ! @ private
      !
      ! MAPPING
      !   Dispatch the various plotting actions in MRC
      !----------------------------------------------------------------
      type(clean_par), intent(in) :: method
      type(gildas), intent(in) :: head
      integer, intent(in) :: code
      real, intent(in) :: array(head%gil%dim(1),head%gil%dim(2))
    end subroutine  plot_mrc
  end interface
  !
  interface
    subroutine mrc_plot(image,nx,ny,type,name)
      use gildas_def
      !---------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Plot the smooth and difference final clean image in <MRC
      !   so as to keep them visible
      !---------------------------------------------------------------------
      integer, intent(in) :: nx,ny,type
      real, intent(in) :: image(nx*ny)
      character(len=*) name
    end subroutine mrc_plot
  end interface
  !
  interface
    subroutine uv_shift (new,ra,dec,ang,off,doit)
      use gkernel_types
      !-------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Compute shift parameters
      !-------------------------------------------------
      real(kind=8), intent(inout) :: new(3)   ! New phase center and angle
      real(kind=8), intent(in) :: ra,dec,ang  ! Old center and angle
      real(kind=8), intent(out) :: off(3)     ! Required Offsets and Angle
      logical, intent(inout) :: doit          ! Do shift
    end subroutine uv_shift
  end interface
  !
  interface
    subroutine shift_uvdata(hx,nu,nv,visi,cs,nc,xy)
      use image_def
      use gkernel_types
      use phys_const
      !$ use omp_lib
      !-------------------------------------------------------------------
      ! @ private
      !  MAPPING
      !   Support routine for commands UV_SHIFT and UV_MAP
      !   Shift phase center and apply U,V coordinates rotation if needed
      !-------------------------------------------------------------------
      type(gildas), intent(inout) :: hx   ! Input UV header
      integer, intent(in) :: nu           ! Size of a visibility
      integer, intent(in) :: nv           ! Number of visibilities
      real, intent(inout) :: visi(nu,nv)  ! Visibilities
      real, intent(in) :: cs(2)           ! Cos/Sin of Rotation
      integer, intent(in) :: nc           ! Number of Channels
      real(8), intent(in) :: xy(2,nc)     ! Position Shift per channel
    end subroutine shift_uvdata
  end interface
  !
  interface
    subroutine uv_listheader(huv,visi,mt,tf,nt,freq)
      use image_def
      use gbl_message
      !-----------------------------------------------------
      ! @ private
      ! MAPPING
      !
      ! Give a brief summary of the content of an UV Table
      !-----------------------------------------------------
      type(gildas), intent(in) :: huv
      integer, intent(in) :: mt         ! Maximum number of dates
      integer, intent(out) :: nt        ! Number of dates
      integer, intent(out) :: tf(mt)    ! Dates (in Gildas code)
      real(4), intent(in) :: visi(huv%gil%dim(1),huv%gil%dim(2)) ! Visbilities
      real(8), intent(in) :: freq       ! Observing frequency
    end subroutine uv_listheader
  end interface
  !
  interface
    subroutine do2weig (jc,nv,visi,jx,jy,iw,unif,we,wm,nw,taper,s,vv)
      use gildas_def
      use gbl_message
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Compute weights of the visibility points.
      !   Explore here the robust parameters.
      !----------------------------------------------------------------------
      integer, intent(in) :: nv  ! Size of a visibility
      integer, intent(in) :: jc  ! Number of visibilities
      integer, intent(in) :: jx  ! X coordinate pointer
      integer, intent(in) :: jy  ! Y coordinate pointer
      integer, intent(in) :: iw  ! Weight pointer
      integer, intent(in) :: nw  ! Number of weighting schemes
      real, intent(in) :: visi(jc,nv)  ! Visibilities
      real, intent(in) :: unif         ! Cell size in meters
      real, intent(out) :: we(nv,nw)   ! Weight arrays
      real, intent(inout) :: wm(nw)    ! on input: robust factors
      real, intent(in) :: taper(3)     ! Taper
      real, intent(out) :: s           ! Sum of weights (natural noise)
      real, intent(in) :: vv(nv)       ! V values
    end subroutine do2weig
  end interface
  !
  interface
    subroutine do3weig (jc,nv,visi,jx,jy,iw,cunif,we,cwm,nw,taper,s,vv)
      use gildas_def
      use gbl_message
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Compute weights of the visibility points.
      !   TAPER mode
      !----------------------------------------------------------------------
      integer, intent(in) :: nv  ! Size of a visibility
      integer, intent(in) :: jc  ! Number of visibilities
      integer, intent(in) :: jx  ! X coordinate pointer
      integer, intent(in) :: jy  ! Y coordinate pointer
      integer, intent(in) :: iw  ! Weight pointer
      integer, intent(in) :: nw  ! Number of weighting schemes
      real, intent(in) :: visi(jc,nv)  ! Visibilities
      real, intent(in) :: cunif(1)     ! Cell size in meters
      real, intent(out) :: we(nv,nw)   ! Weight arrays
      real, intent(in) :: cwm(1)       ! on input: robust factors
      real, intent(in) :: taper(nw)    ! Tapers
      real, intent(out) :: s           ! Sum of weights (natural noise)
      real, intent(in) :: vv(nv)       ! V values
    end subroutine do3weig
  end interface
  !
  interface
    subroutine dodate(nc,nv,visi,id,it)
      use gbl_message
      !-----------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   List the dates
      !-----------------------------------------------------
      integer, intent(in) :: id  ! Date pointer
      integer, intent(in) :: it  ! Time pointer
      integer, intent(in) :: nc  ! Size of a visibility
      integer, intent(in) :: nv  ! Number of visibilities
      real, intent(in) :: visi(nc,nv)
    end subroutine dodate
  end interface
  !
  interface
    subroutine do4weig (jc,nv,visi,jx,jy,iw,unif,we,wm,nw,taper,s,vv)
      use gbl_message
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Support for UV_STAT
      !   Compute weights of the visibility points.
      !   CELL mode
      !----------------------------------------------------------------------
      integer, intent(in) :: nv  ! Size of a visibility
      integer, intent(in) :: jc  ! Number of visibilities
      integer, intent(in) :: jx  ! X coordinate pointer
      integer, intent(in) :: jy  ! Y coordinate pointer
      integer, intent(in) :: iw  ! Weight pointer
      integer, intent(in) :: nw  ! Number of weighting schemes
      real, intent(in) :: visi(jc,nv)  ! Visibilities
      real, intent(in) :: unif(nw)     ! Cell size in meters
      real, intent(out) :: we(nv,nw)   ! Weight arrays
      real, intent(inout) :: wm(nw)    ! on input: robust factors
      real, intent(in) :: taper(3)     ! Tapers
      real, intent(out) :: s           ! Sum of weights (natural noise)
      real, intent(in) :: vv(nv)       ! Sorted V<0 values
    end subroutine do4weig
  end interface
  !
  interface
    subroutine sidelo (map,nx,ny,thre,majo,mino,pa,convert)
      !------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Compute sidelobes
      !------------------------------------------------------------
      integer, intent(in) :: nx,ny        ! X,Y sizes
      real, intent(in) :: map(nx,ny)      ! Beam map
      real, intent(in) :: majo            ! Major axis
      real, intent(in) :: mino            ! Minor axis
      real, intent(in) :: pa              ! PA
      real, intent(out) :: thre(2)        ! Pos and Neg sidelobe
      real(8), intent(in) :: convert(3,2) ! Conversion formula
    end subroutine sidelo
  end interface
  !
  interface
    subroutine do0weig (jc,nv,visi,jx,jy,jw,unif,we,vv,   &
         &    lv,umin,umax,vmin,vmax,nbcv)
      use gildas_def
      use gbl_message
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Support for UV_STAT
      !   Compute weights of the visibility points.
      !----------------------------------------------------------------------
      integer, intent(in) :: nv  ! Size of a visibility
      integer, intent(in) :: jc  ! Number of visibilities
      integer, intent(in) :: jx  ! X coordinate pointer
      integer, intent(in) :: jy  ! Y coordinate pointer
      integer, intent(in) :: jw  ! Weight pointer
      real, intent(in) :: visi(jc,nv)  ! Visibilities
      real, intent(in) :: unif         ! Cell size in meters
      real, intent(out) :: we(nv)      ! Weight arrays
      real, intent(in) :: vv(nv)       ! V values
      !
      integer, intent(in) :: lv        ! Unused
      real, intent(in) ::  umin,umax,vmin,vmax
      integer, intent(in) :: nbcv       ! Unused here
    end subroutine do0weig
  end interface
  !
  interface
    subroutine do2fft (np,nv,visi,jx,jy   &
         &    ,nx,ny,nw,map,mapx,mapy,we) 
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Support for UV_STAT
      !   Compute FFT of image by gridding UV data for several weights
      !   at once. Uses simple in-cell gridding
      !
      ! Input UV data must be sorted by increasing V < 0
      !----------------------------------------------------------------------
      ! Call
      integer, intent(in) :: nv                   ! number of values
      integer, intent(in) :: np                   ! Number of "visibilities", normally 7+3*nc
      integer, intent(in) :: nw                   ! number of weight schemes
      integer, intent(in) :: nx,ny                ! map size
      integer, intent(in) :: jx,jy                ! X coord, Y coord location in VISI
      real, intent(in) :: visi(np,nv)             ! values
      complex, intent(out) :: map(nw,nx,ny)       ! gridded visibilities
      real, intent(in) :: mapx(nx),mapy(ny)       ! Coordinates of grid
      real, intent(in) :: we(nv,nw)               ! Weight array
    end subroutine do2fft
  end interface
  !
  interface
    subroutine prnoise(prog,which,noise,rms)
      use gbl_message
      !--------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Printout the noise value
      !--------------------------------------------------
      character(len=*), intent(in) :: prog  ! Caller's name
      character(len=*), intent(in) :: which ! Type of image
      real, intent(in) :: noise             ! Noise value
      real, intent(out) :: rms              ! Unit of noise
    end subroutine prnoise
  end interface
  !
  interface
    subroutine doqfft (np,nv,visi,jx,jy,jw   &
         &    ,nx,ny,map,uvcell)
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Support for UV_STAT
      !   Compute FFT of beam by gridding UV data for Natural
      !   weighting only. Uses simple in-cell gridding
      !----------------------------------------------------------------------
      ! Call
      integer, intent(in) :: nv                   ! number of values
      integer, intent(in) :: np                   ! Number of "visibilities", normally 7+3*nc
      integer, intent(in) :: nx,ny                ! map size
      integer, intent(in) :: jx,jy,jw             ! X coord, Y coord & Weight location in VISI
      real, intent(in) :: visi(np,nv)             ! values
      complex, intent(out) :: map(nx,ny)          ! gridded visibilities
      real, intent(in) :: uvcell                  ! UV cell size 
    end subroutine doqfft
  end interface
  !
  interface
    subroutine sphfn (ialf, im, iflag, eta, psi, ier)
      use gbl_message
      !---------------------------------------------------------------------
      ! @ private
      !
      !     SPHFN is a subroutine to evaluate rational approximations to se-
      !  lected zero-order spheroidal functions, psi(c,eta), which are, in a
      !  sense defined in VLA Scientific Memorandum No. 132, optimal for
      !  gridding interferometer data.  The approximations are taken from
      !  VLA Computer Memorandum No. 156.  The parameter c is related to the
      !  support width, m, of the convoluting function according to c=
      !  pi*m/2.  The parameter alpha determines a weight function in the
      !  definition of the criterion by which the function is optimal.
      !  SPHFN incorporates approximations to 25 of the spheroidal func-
      !  tions, corresponding to 5 choices of m (4, 5, 6, 7, or 8 cells)
      !  and 5 choices of the weighting exponent (0, 1/2, 1, 3/2, or 2).
      !
      !  Input:
      !    IALF    I*4   Selects the weighting exponent, alpha.  IALF =
      !                  1, 2, 3, 4, and 5 correspond, respectively, to
      !                  alpha = 0, 1/2, 1, 3/2, and 2.
      !    IM      I*4   Selects the support width m, (=IM) and, correspond-
      !                  ingly, the parameter c of the spheroidal function.
      !                  Only the choices 4, 5, 6, 7, and 8 are allowed.
      !    IFLAG   I*4   Chooses whether the spheroidal function itself, or
      !                  its Fourier transform, is to be approximated.  The
      !                  latter is appropriate for gridding, and the former
      !                  for the u-v plane convolution.  The two differ on-
      !                  by a factor (1-eta**2)**alpha.  IFLAG less than or
      !                  equal to zero chooses the function appropriate for
      !                  gridding, and IFLAG positive chooses its F.T.
      !    ETA     R*4   Eta, as the argument of the spheroidal function, is
      !                  a variable which ranges from 0 at the center of the
      !                  convoluting function to 1 at its edge (also from 0
      !                  at the center of the gridding correction function
      !                  to unity at the edge of the map).
      !
      !  Output:
      !    PSI      R*4  The function value which, on entry to the subrou-
      !                  tine, was to have been computed.
      !    IER      I*4  An error flag whose meaning is as follows:
      !                     IER = 0  =>  No evident problem.
      !                           1  =>  IALF is outside the allowed range.
      !                           2  =>  IM is outside of the allowed range.
      !                           3  =>  ETA is larger than 1 in absolute
      !                                     value.
      !                          12  =>  IALF and IM are out of bounds.
      !                          13  =>  IALF and ETA are both illegal.
      !                          23  =>  IM and ETA are both illegal.
      !                         123  =>  IALF, IM, and ETA all are illegal.
      !
      !---------------------------------------------------------------------
      integer(4), intent(in)  :: ialf     ! Exponent
      integer(4), intent(in)  :: im       ! Width of support
      integer(4), intent(in)  :: iflag    ! Gridding function
      real(4), intent(in)     :: eta      ! For spheroidals only
      real(4), intent(out)    :: psi      ! Result
      integer(4), intent(out) :: ier      ! Error code
    end subroutine sphfn
  end interface
  !
  interface
    subroutine gridless_density (npts,sizecell,distmax,evex,evey,eveweight,&
      & evesumweight,xmin,xmax,ymin,ymax,error)
      use gildas_def
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Support routine for Robust weighting
      !
      ! Stephane Paulin   OASU CNRS / U.Bordeaux  2016
      !
      ! Algorithm to compute the weighted neighborhood of each point in a large
      ! catalog. This algorithm was described, among other sources, by Daniel Briggs in
      ! his thesis. The idea is to avoid npts**2 calculations of the distance between
      ! each pair of points (npts is the # points). For that, one defines a grid only used
      ! to speed-up computation (result is independent of the grid).
      ! Step 1: In a given cell of the grid, all events are linked and the total weight
      !   of the cell is computed and kept in memory.
      ! Step 2: a pattern is computed, that is the same all over the field, whose radius
      !   is the maximum distance. Grid boxes can have 3 states: entirely inside the
      !   maximum distance, entirely outside, or in between (boxes noted 'intermediate'
      !   or 'mixed').
      ! The idea is to compute the distances only for events in these intermediate boxes.
      ! For other events, or the total weight of the box is added (if the box is inside),
      ! or the box is ignored (if outside). Next, the main loop consists on a loop on every
      ! event. For a given event, all boxes in the pattern are checked and treated accordingly.
      ! With this algorithm, one can have CPU times prop. to factor*npts*log(npts) with
      ! factor~2 (ie. when npts is multiplied by 10, the CPU is multiplied by ~45-50)
      ! Remarks/notes:
      ! - this is a preliminary approach for testing, where symmetries are not
      !   taken into account !!! ==> a given couple of coordinates (evex,evey) corresponds
      !   to one event
      !
      !----------------------------------------------------------------------
      real, intent(in) :: sizecell                ! size of a cell.
      ! this is a control parameter that does not change the result but
      ! change the calculation time: to minimize the CPU, the larger the
      ! event density, the smaller sizecell. ex: sizecell~20 (10, 5 resp.)
      ! for a catalog of 10**5 (10**6, 3x10**7) events
      integer, intent(in) :: npts                 ! total # of events
      real, intent(in) :: distmax                 ! maximum distance
      real(kind=4), intent(in) :: evex(npts)      ! X coordinates
      real(kind=4), intent(in) :: evey(npts)      ! Y coordinates
      real(kind=4), intent(in) :: eveweight(npts) ! Weight of event
      real(kind=4), intent(out) :: evesumweight(npts) ! Sum of the weights of all
      ! the events closer nearer than the maximum distance from the current event.
      real(kind=4), intent(in) :: xmin,ymin,xmax,ymax ! Min Max
      logical, intent(out) :: error               ! Error flag
    end subroutine gridless_density
  end interface
  !
  interface
    function calcdistsq(x1,x2,y1,y2)
      ! @ private
      real(kind=4) calcdistsq ! intent(out)
      real(kind=4), intent(in) :: x1,y1,x2,y2
    end function calcdistsq
  end interface
  !
  interface
    subroutine linkidenticalbox(nb, ind_init, boxtotweight, evesumweight, nextevent)
      use gildas_def
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Support routine for Robust Weighting
      !
      !   Build the chain list of pixels in the same cell with
      !   the same status
      !
      real, intent(in) :: boxtotweight                    ! Box total weight
      integer(kind=4), intent(in) :: nb                   ! Size of chain
      integer(kind=4), intent(in) :: ind_init             ! Starting index in each box
      real(kind=4), intent(inout) :: evesumweight(:)      ! Sum of weight at each event
      integer(kind=4), intent(in) :: nextevent(:)         ! Chained pointers
    end subroutine linkidenticalbox
  end interface
  !
  interface
    subroutine linkgoodbox(nb1, nb2, ind1_init, ind2_init, &
      & boxtotweight1, boxtotweight2, evesumweight, nextevent)
      use gildas_def
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Support routine for Robust Weighting
      !
      !   Build the chain list of pixels in the same cell with
      !   the same status
      !
      !----------------------------------------------------------------------
      real, intent(in) :: boxtotweight1, boxtotweight2    ! Total weight of each box
      integer(kind=4), intent(in) :: nb1, nb2             ! Box sizes
      integer(kind=4), intent(in) :: ind1_init, ind2_init ! Starting index in each box
      real(kind=4), intent(inout) :: evesumweight(:)      ! Sum of weight at each event
      integer(kind=4), intent(in) :: nextevent(:)         ! Chained pointers
    end subroutine linkgoodbox
  end interface
  !
  interface
    subroutine linkmixedbox(nb1, nb2, ind1_init, ind2_init, evesumweight, &
      & eveweight, evex, evey, nextevent, distmax, distmaxsq)
      use gildas_def
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Support routine for Robust Weighting
      !
      !   Build chain list of different types
      !
      !----------------------------------------------------------------------
      integer(kind=4), intent(in) :: nb1, nb2             ! Box sizes
      integer(kind=4), intent(in) :: ind1_init, ind2_init ! Starting index in each box
      real, intent(in) :: distmax, distmaxsq              ! Averaging radius
      real(4), intent(in) :: eveweight(:)                 ! Initial weights
      real(4), intent(in) :: evex(:)                      ! Event X coordinates
      real(4), intent(in) :: evey(:)                      ! Event Y coordinates
      real(kind=4), intent(inout) :: evesumweight(:)      ! Sum of weight at each event
      integer(kind=4), intent(in) :: nextevent(:)         ! Chained pointers
    end subroutine linkmixedbox
  end interface
  !
  interface
    subroutine chkfft (a,nx,ny,error)
      !---------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Check if FFT is centered...
      !---------------------------------------------------------
      integer, intent(in)  ::  nx,ny   ! X,Y size
      logical, intent(out)  ::  error  ! Error flag
      real, intent(in)  ::  a(nx,ny)   ! Array
    end subroutine chkfft
  end interface
  !
  interface
    subroutine doweig_quick (jc,nv,visi,jx,jy,jw,unif,we,wm,vv,mv, &
         &     umin,umax,vmin,vmax,nbcv)
      use gildas_def
      use gbl_message
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !     Compute weights of the visibility points.
      !----------------------------------------------------------------------
      integer, intent(in) ::  nv          ! number of values
      integer, intent(in) ::  jc          ! Number of "visibilities"
      integer, intent(in) ::  jx          ! X coord location in VISI
      integer, intent(in) ::  jy          ! Y coord location in VISI
      integer, intent(in) ::  jw          ! Location of weights. If .LE.0, uniform weight
      integer, intent(in) ::  mv          ! Size of work arrays
      integer, intent(in) ::  nbcv        ! Buffering factor
      real, intent(in) ::  visi(jc,nv)    ! Visibilities
      real, intent(in) ::  unif           ! uniform cell size in Meters
      real, intent(inout) ::  we(nv)      ! Weight array
      real, intent(in) ::  wm             ! on input: % of uniformity
      real, intent(in) ::  vv(nv)         ! V values, pre-sorted
      real, intent(out) :: umin,umax,vmin,vmax
    end subroutine doweig_quick
  end interface
  !
  interface
    subroutine doweig_sub (nv,uu,vv,ww,we,unif)
      !$ use omp_lib
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !     Compute weights of the visibility points.
      !----------------------------------------------------------------------
      integer, intent(in) :: nv          ! number of values
      real, intent(in) ::  uu(nv)        ! U coordinates
      real, intent(in) ::  vv(nv)        ! V coordinates
      real, intent(in) ::  ww(nv)        ! Input Weights
      real, intent(out) ::  we(nv)       ! Output weights
      real, intent(in) ::  unif          ! Cell size
    end subroutine doweig_sub
  end interface
  !
  interface
    subroutine doweig_slow (jc,nv,visi,jx,jy,jw,unif,we,wm)
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !     Compute weights of the visibility points.
      !----------------------------------------------------------------------
      integer, intent(in) ::  nv          ! number of values
      integer, intent(in) ::  jc          ! Number of "visibilities"
      integer, intent(in) ::  jx          ! X coord location in VISI
      integer, intent(in) ::  jy          ! Y coord location in VISI
      integer, intent(in) ::  jw          ! Location of weights. If .LE.0, uniform weight
      real, intent(in) ::  visi(jc,nv)    ! Visibilities
      real, intent(in) ::  unif           ! uniform cell size in Meters
      real, intent(inout) ::  we(nv)      ! Weight array
      real, intent(in) ::  wm             ! on input: % of uniformity
    end subroutine doweig_slow
  end interface
  !
  interface
    subroutine uv_baseline_main(line,error)
      use gbl_message
      use gildas_def
      !----------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      ! Command line interaction for UV_BASELINE command
      !----------------------------------------------------------
      character(len=*), intent(inout) :: line  ! Command line
      logical, intent(out)            :: error ! Error flag
    end subroutine uv_baseline_main
  end interface
  !
  interface
    subroutine uv_baseline_sub(degree,channellist,nelem,error)
      use gildas_def
      use clean_arrays
      use gbl_message
      !----------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      ! Coordinates baseline fitting and subtraction
      !----------------------------------------------------------
      integer, intent(in)                   :: degree           ! Polynomial degree of baseline
      integer, allocatable,intent(in)       :: channellist(:,:) ! list of ranges of channels to exclude
      integer(kind=size_length),intent(in)  :: nelem            ! Number of excluded ranges
      logical, intent(out)                  :: error            ! Error flag
    end subroutine uv_baseline_sub
  end interface
  !
  interface
    subroutine uv_baseline_baseline(xdata,ydata,npoints,degree,acoeff,bcoeff,error)
      use gildas_def
      !----------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      ! Computes baseline from input arrays
      !----------------------------------------------------------
      integer(kind=index_length),allocatable,intent(in) :: xdata(:)
      real,allocatable,intent(in)                       :: ydata(:)
      integer,intent(in)                                :: npoints
      integer,intent(in)                                :: degree
      real,intent(out)                                  :: acoeff
      real,intent(out)                                  :: bcoeff
      logical, intent(out)                              :: error
    end subroutine uv_baseline_baseline
  end interface
  !
  interface
    subroutine uv_select_buffer(rname,nu,nv,error)
      use clean_def
      use clean_arrays
      use clean_types
      use gbl_message
      !---------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !
      !   Select the next available UV buffer (UVR or UVS) for
      ! commands using them.
      !---------------------------------------------------------------------
      character(len=*), intent(in) :: rname  ! Caller's name
      integer, intent(in) :: nu              ! Visibility size
      integer, intent(in) :: nv              ! Number of visibilities
      logical, intent(out) :: error          ! Error flag
    end subroutine uv_select_buffer
  end interface
  !
  interface
    subroutine uv_reset_buffer(rname)
      use clean_def
      use clean_arrays
      use clean_types
      use gbl_message
      !---------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Reset pointers to UV datasets (back to UVI). Deallocate
      ! UVR or UVS if needed, as well as the "transposed" buffer UVT
      !---------------------------------------------------------------------
      character(len=*) :: rname
    end subroutine uv_reset_buffer
  end interface
  !
  interface
    subroutine uv_dump_buffers(rname)
      use clean_def
      use clean_arrays
      use gbl_message
      !---------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Dump the allocation status of the UV buffers. (Debugging only)
      !---------------------------------------------------------------------
      character(len=*), intent(in) :: rname  ! Caller's name
    end subroutine uv_dump_buffers
  end interface
  !
  interface
    subroutine uv_find_buffers (rname,nu,nv,duv_previous, duv_next,error)
      use clean_arrays
      use gbl_message
      !---------------------------------------------------------------------
      ! @ private-mandatory
      !
      ! MAPPING
      !   Find the next available UV buffer (UVR or UVS).
      !---------------------------------------------------------------------
      character(len=*), intent(in) :: rname  ! Caller's name
      integer, intent(in) :: nu   ! Size of a visiblity
      integer, intent(in) :: nv   ! Number of visibilities
      real, pointer, intent(out) :: duv_previous(:,:)
      real, pointer, intent(out) :: duv_next(:,:)
      logical, intent(out) :: error
    end subroutine uv_find_buffers
  end interface
  !
  interface
    subroutine uv_clean_buffers(duv_previous,duv_next,error)
      use clean_arrays
      !---------------------------------------------------------------------
      ! @ private-mandatory
      !
      ! MAPPING
      !   Take care of freeing the unused UV buffer, and set UV to point to
      ! the new one.
      !---------------------------------------------------------------------
      real, pointer, intent(inout) :: duv_previous(:,:)
      real, pointer, intent(inout) :: duv_next(:,:)
      logical, intent(in) :: error
    end subroutine uv_clean_buffers
  end interface
  !
  interface
    subroutine uv_discard_buffers(duv_previous,duv_next,error)
      use clean_arrays
      !---------------------------------------------------------------------
      ! @ private-mandatory
      !
      ! MAPPING
      !   Take care of freeing the last used UV buffer, and reset 
      !   UV to point to the previous one.
      !---------------------------------------------------------------------
      real, pointer, intent(inout) :: duv_previous(:,:)
      real, pointer, intent(inout) :: duv_next(:,:)
      logical, intent(in) :: error
    end subroutine uv_discard_buffers
  end interface
  !
  interface
    subroutine uv_check_comm(line,error)
      use gbl_message
      use clean_types
      use clean_arrays
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING support routine for command
      !     UV_CHECK [Beams|Null]
      !----------------------------------------------------------------------
      logical, intent(out) :: error ! Logical error flag
      character(len=*), intent(in) :: line
    end subroutine uv_check_comm
  end interface
  !
  interface
    subroutine display_uv(chain,error)
      use clean_def
      use clean_arrays
      use gbl_message
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   SHOW UV
      !     TIME-BASE sort the UV table before displaying it
      !----------------------------------------------------------------------
      character(len=*), intent(in) :: chain
      logical, intent(out) :: error ! Logical error flag
    end subroutine display_uv
  end interface
  !
  interface
    subroutine uv_sort_comm(line,error)
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING Support routine for command
      !   UVSORT [TIME|BASE|UV]
      !----------------------------------------------------------------------
      character(len=*), intent(in)  :: line  ! Command line
      logical,          intent(out) :: error ! Logical error flag
    end subroutine uv_sort_comm
  end interface
  !
  interface
    subroutine uv_fsort(error)
      use clean_def
      use clean_arrays
      use gbl_message
      !-----------------------------------------------------------------
      ! @ private
      !
      ! MAPPING support routine for command UV_SORT
      ! Allows for sorting for fields using mosaic uv sorting routines
      !-----------------------------------------------------------------
      logical, intent(inout) :: error
    end subroutine uv_fsort
  end interface
  !
  interface
    subroutine uv_tri(code,error)
      use clean_def
      use clean_arrays
      use gbl_message
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING Support routine for command
      !   UVSORT [TIME|BASE|UV]
      !     BASE is code 1
      !     TIME is code 2
      !     UV is code 3
      !----------------------------------------------------------------------
      integer, intent(in)  :: code  ! Sorting code
      logical, intent(out) :: error ! Logical error flag
    end subroutine uv_tri
  end interface
  !
  interface
    subroutine uv_findtb(code,uv,nv,nd,order,it,ot,sorted)
      !---------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Support for SHOW UV
      !---------------------------------------------------------------
      integer, intent(in)       :: code      ! Code for sort operation
      integer, intent(in)       :: nd,nv     ! Size of visibility table
      real, intent(in)          :: uv(nd,nv) ! Input UV data
      real(kind=8), intent(out) :: order(nv) ! Sorting array
      integer, intent(out)      :: it(nv)    ! Sort index
      integer, intent(out)      :: ot(nv)    ! Reverse order
      logical, intent(out)      :: sorted    ! Is already sorted
    end subroutine uv_findtb
  end interface
  !
  interface
    subroutine triuv8(x,it,n,error)
      use gbl_message
      !------------------------------------------------------------------------
      ! @ private
      !
      !   Sorting program that uses a quicksort algorithm.
      ! Applies for an input array of real*8 values, which are left
      ! unchanged. Returns an array of indexes sorted for increasing
      ! order of X. Use GR8_SORT to reorder X.
      !------------------------------------------------------------------------
      integer,      intent(in)    :: n     ! Array length
      real(kind=8), intent(inout) :: x(n)  ! Unsorted array
      integer,      intent(out)   :: it(n) ! Sort index
      logical,      intent(out)   :: error ! Logical return flag
    end subroutine triuv8
  end interface
  !
  interface
    subroutine uv_flag_comm(line,error)
      use clean_def
      use clean_arrays
      use clean_support
      use clean_types
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING Support routine for command
      !     UV_FLAG [/ANTENNA] [/RESET]
      !----------------------------------------------------------------------
      character(len=*), intent(in)  :: line  ! Command line
      logical,          intent(out) :: error ! Logical error flag
    end subroutine uv_flag_comm
  end interface
  !
  interface
    subroutine get_uvflag_ant(uvs,nv,nd,iant)
      use gildas_def
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Set uvflag array from antenna number
      !----------------------------------------------------------------------
      integer(kind=index_length), intent(in) :: nv,nd        ! Size of UV data
      real, intent(inout) :: uvs(nv,nd)
      integer, intent(in) :: iant
    end subroutine get_uvflag_ant
  end interface
  !
  interface
    subroutine get_uvflag(uvs,nv,nd)
      use gildas_def
      use clean_support
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Get uvflag array from polygon (0 => Flag, 1 => Keep)
      !----------------------------------------------------------------------
      integer(kind=index_length), intent(in) :: nv,nd        ! Size of UV data
      real, intent(inout) :: uvs(nv,nd)
    end subroutine get_uvflag
  end interface
  !
  interface
    subroutine reset_uvflag(uvs,nv,nd)
      use gildas_def
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Reset uvflag array to 1
      !----------------------------------------------------------------------
      integer(kind=index_length), intent(in) :: nv,nd        ! Size of UV data
      real, intent(inout) :: uvs(nv,nd)   ! UV data set
    end subroutine reset_uvflag
  end interface
  !
  interface
    subroutine apply_uvflag(uvs,nv,nd,uv)
      use gildas_def
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Apply uvflag array by making negative the weight of the flagged data
      !----------------------------------------------------------------------
      integer(kind=index_length), intent(in) :: nv,nd        ! Size of UV data
      real, intent(inout) :: uvs(nv,nd)   ! UV data set
      real, intent(inout) :: uv(nd,nv)    ! Transposed version
    end subroutine apply_uvflag
  end interface
  !
  interface
    subroutine uv_extract_main(line, error)
      use gbl_message
      use clean_arrays
      !----------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      ! Command line interaction for UV_EXTRACT command
      !----------------------------------------------------------
      character(len=*), intent(inout) :: line  ! Command line
      logical, intent(out)            :: error ! Error flag
    end subroutine uv_extract_main
  end interface
  !
  interface
    subroutine uv_extract_sub(channels, error)
      use clean_def
      use clean_types
      use clean_arrays
      use gbl_message
      !----------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !    Support routine for command UV_extract
      !    Extract a range of channels from a UV table
      !----------------------------------------------------------
      integer, intent(in)         :: channels(2) ! Limits for the new UV table in channels
      logical, intent(inout)      :: error       ! Error flag
    end subroutine uv_extract_sub
  end interface
  !
  interface
    subroutine uv_filter_main(line,error)
      use gildas_def
      !----------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      ! Front end for UV_FILTER command
      !----------------------------------------------------------
      character(len=*), intent(inout) :: line  ! Command line
      logical, intent(out)            :: error ! Error flag
    end subroutine uv_filter_main
  end interface
  !
  interface
    subroutine uv_filter_parsing(line,channellist,nelem,dozero,rname,error)
      use gbl_message
      use clean_arrays
      use gkernel_types
      use gildas_def
      use gbl_message
      !----------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      ! Command line interaction for UV_FILTER and UV_BASELINE commands
      !----------------------------------------------------------
      character(len=*), intent(inout)       :: line             ! Command line
      integer, allocatable,intent(out)      :: channellist(:,:) ! Array containing the flagging ranges
      integer(kind=size_length),intent(out) :: nelem            ! Number of elements in descriptor
      logical,intent(out)                   :: dozero           ! Zero option is present
      character(len=*), intent(in)          :: rname            ! Who is calling?
      logical, intent(out)                  :: error            ! Error flag
    end subroutine uv_filter_parsing
  end interface
  !
  interface
    subroutine uv_filter_sub(channellist,nelem,dozero,error)
      use clean_arrays
      use gbl_message
      !--------------------------------------------------------
      ! @ private
      !
      ! Mapping
      ! Apply flagging to a list of channels given as input
      !--------------------------------------------------------
      integer(kind=4),allocatable,intent(in) :: channellist(:,:)
      integer(kind=size_length),intent(in)   :: nelem
      logical, intent(in)                    :: dozero
      logical, intent(inout)                 :: error
    end subroutine uv_filter_sub
  end interface
  !
  interface
    subroutine flux_dispatch (line,error)
      !---------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      ! Dispatching routine for command
      !  FLUX FIND DateInterval
      !  FLUX APPLY OutputVariable
      !  FLUX CALIBRATE
      !  FLUX LIST
      !---------------------------------------------------------------------
      character(len=*), intent(in) :: line
      logical, intent(inout) :: error
    end subroutine flux_dispatch
  end interface
  !
  interface
    subroutine flux_apply  (line,error)
      use gildas_def
      use clean_arrays
      !-----------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Support for command
      !   FLUX APPLY OutputVariable
      !-----------------------------------------------------------
      character(len=*), intent(in) :: line
      logical, intent(inout) :: error
    end subroutine flux_apply
  end interface
  !
  interface
    subroutine flux_calib (line,error)
      use gildas_def
      use clean_arrays
      !-----------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      ! Support for command
      !  FLUX CALIBRATE 
      !
      ! Apply the factors to the UV data set..
      !-----------------------------------------------------------
      character(len=*), intent(in) :: line
      logical, intent(inout) :: error
    end subroutine flux_calib
  end interface
  !
  interface
    subroutine flux_find (line,error)
      use gildas_def
      use clean_arrays
      !-----------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      ! Support routine for command
      !  FLUX FIND DateInterval
      !
      !  Scan the current UV table (obtained by READ UV) to determine
      !  how many independent dates exist
      !
      !  Compare it date by date with the model UV table (obtained by 
      !  READ MODEL) and compute, through linear regression, the best scaling
      !  factors to match the two tables
      !
      !  Return these flux factors as variables
      !  FLUX and D_FLUX
      !-----------------------------------------------------------
      character(len=*), intent(in) :: line
      logical, intent(inout) :: error
    end subroutine flux_find
  end interface
  !
  interface
    subroutine flux_list (line,error)
      !-----------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      ! Support routine for command
      !  FLUX LIST 
      !
      !  Printout the latest results from FLUX FIND
      !-----------------------------------------------------------
      character(len=*), intent(in) :: line
      logical, intent(inout) :: error
    end subroutine flux_list
  end interface
  !
  interface
    subroutine flux_factor (hduv,duv,hcuv,cuv,date_spacing)
      use image_def
      !-----------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !
      ! Find the flux scale factors for the Observations
      !
      ! The observations are defined by  Obs = Factor * Model + noise
      ! so the calibration factor to apply is the inverse of this solution
      !-----------------------------------------------------------
      integer, intent(in) :: date_spacing
      type (gildas), intent(in) :: hduv
      type (gildas), intent(in) :: hcuv
      real(kind=4), intent(in) :: duv(hduv%gil%dim(1),hduv%gil%dim(2))
      real(kind=4), intent(in) :: cuv(hcuv%gil%dim(1),hcuv%gil%dim(2))
    end subroutine flux_factor
  end interface
  !
  interface
    subroutine my_finddat(nc,nv,visi,itol,nt) 
      !-----------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Find how many dates
      !-----------------------------------------------------------
      integer, intent(in) :: nc  ! Visibility size 
      integer, intent(in) :: nv  ! Number of visibilities
      integer, intent(out) :: nt ! Number of time stamps
      integer, intent(in) :: itol  ! Tolerance to check dates 
      real, intent(in) :: visi(nc,nv)   ! Visibilities
    end subroutine my_finddat
  end interface
  !
  interface
    subroutine my_listdat(nc,nv,visi,nt,tf,itol,chain) 
      !-----------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   List the different dates 
      !-----------------------------------------------------------
      integer, intent(in) :: nc  ! Visibility size 
      integer, intent(in) :: nv  ! Number of visibilities
      integer, intent(in) :: nt  ! Number of time stamps
      integer, intent(out) :: tf(nt)              ! Time stamp values
      character(len=*), intent(out) :: chain(nt)  ! Associated message
      integer, intent(in) :: itol  ! Tolerance to check dates 
      real, intent(in) :: visi(nc,nv)   ! Visibilities
    end subroutine my_listdat
  end interface
  !
  interface
    subroutine uv_baseline(line,error)
      use gildas_def
      use gkernel_types
      use gbl_format
      use gbl_message
      use clean_types
      use clean_arrays
      !----------------------------------------------------------
      ! @ private
      !
      ! MAPPING Support for commands
      !   UV_BASELINE [Degree] /CHANNEL ListVariable [/ZERO]
      !   UV_BASELINE [Degree] /FREQUENCIES ListFreq /WIDTH Width
      !   UV_BASELINE [Degree] /VELOCITIES ListVelo /WIDTH Width
      !
      ! Subtract a continuum baseline, ignoring a list of channels
      ! in the current UV data set.
      !----------------------------------------------------------------
      character(len=*), intent(inout) :: line  ! Command line
      logical, intent(out) :: error            ! Error flag
    end subroutine uv_baseline
  end interface
  !
  interface
    subroutine uv_filter(line,error)
      use gildas_def
      use gkernel_types
      use gbl_format
      use gbl_message
      use clean_types
      use clean_arrays
      !----------------------------------------------------------------
      ! @ private
      !
      ! MAPPING  support for
      !   UV_FILTER  /CHANNEL ListVariable [/ZERO]
      !   UV_FILTER  /FREQUENCIES ListFreq /WIDTH Width [UNIT]
      !   UV_FILTER  /VELOCITY ListVelo /WIDTH Width [UNIT]
      !
      ! "Filter", i.e. flag, a list of channels in the current UV
      ! data set. Flagging is reversible, unless the /ZERO option is
      ! present. With /ZERO, the "filtered" visibilities are set to zero.
      !----------------------------------------------------------------
      character(len=*), intent(inout) :: line  ! Command line
      logical, intent(out) :: error            ! Error flag
    end subroutine uv_filter
  end interface
  !
  interface
    subroutine uv_line2cont(line,error)
      use clean_def
      use clean_default
      use clean_types
      use clean_arrays
      use gbl_message
      !----------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Support routine for command UV_CONT /INDEX Alpha [Frequency]
      !   Create a continuum UV table from a Line one
      !----------------------------------------------------------
      character(len=*), intent(inout) :: line  ! Command line
      logical, intent(out) :: error            ! Error flag
    end subroutine uv_line2cont
  end interface
  !
  interface
    subroutine uv_filter_base(line,error,rname,t_routine,zero)
      use gildas_def
      use gkernel_types
      use gbl_format
      use gbl_message
      use clean_types
      use clean_arrays
      !----------------------------------------------------------------
      ! @ private
      !
      ! MAPPING  support for
      !   UV_FILTER or UV_BASELINE  /CHANNEL ListVariable [/ZERO]
      !   UV_FILTER or UV_BASELINE  /FREQUENCIES ListFreq /WIDTH Width [UNIT]
      !   UV_FILTER or UV_BASELINE  /VELOCITY ListVelo /WIDTH Width [UNIT]
      !
      ! "Filter", i.e. flag, a list of channels in the current UV
      ! data set. Flagging is reversible, unless the /ZERO option is
      ! present. With /ZERO, the "filtered" visibilities are set to zero.
      !----------------------------------------------------------------
      character(len=*), intent(inout) :: line  ! Command line
      logical, intent(out) :: error            ! Error flag
      external :: t_routine
      character(len=*), intent(in) :: rname
      integer, intent(in) :: zero
    end subroutine uv_filter_base
  end interface
  !
  interface
    subroutine t_filter(mf,filter,zero,error)
      !----------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Support routine for UV_FILTER
      !   Filter / Flag a list of channels
      !----------------------------------------------------------
      use gildas_def
      use image_def
      use clean_arrays
      !
      integer, intent(in) :: mf          ! Number of values
      integer, intent(in) :: filter(mf)  ! Channel list
      integer, intent(in) :: zero        ! Zero or not
      logical, intent(out) :: error
    end subroutine t_filter
  end interface
  !
  interface
    subroutine t_baseline(mf,filter,degree,error)
      use gildas_def
      use image_def
      use clean_arrays
      !----------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !    Support routine for UV_BASELINE
      !    Subtract a baseline with a list of channels to be ignored
      !----------------------------------------------------------
      integer, intent(in) :: mf          ! Number of values
      integer, intent(in) :: filter(mf)  ! Channel list
      integer, intent(in) :: degree      ! Polynomial degree
      logical, intent(out) :: error
    end subroutine t_baseline
  end interface
  !
  interface
    subroutine uvmap(task,line,error)
      use clean_def
      use clean_arrays
      use clean_types
      use clean_default
      use gbl_message
      !------------------------------------------------------------------------
      ! @ private
      !
      ! TASK  Compute a map from a CLIC UV Sorted Table
      ! by Gridding and Fast Fourier Transform, using adequate
      ! scratch space for optimisation. Will work for
      ! up to 128x128x128 cube data size, may be more...
      !
      ! Input :
      !     a precessed UV table
      ! Output :
      !     a precessed, rotated, shifted UV table, sorted in V,
      !     ordered in (U,V,W,D,T,iant,jant,nchan(real,imag,weig))
      !     a beam image or cube
      !     a LMV cube
      !------------------------------------------------------------------------
      character(len=*), intent(in) :: task ! Caller (MX or UV_MAP)
      character(len=*), intent(in) :: line ! Command line
      logical, intent(out) :: error
    end subroutine uvmap
  end interface
  !
  interface
    subroutine map_get_radecang(rname,found,values,error)
      use phys_const
      use gbl_message
      !---------------------------------------------------------------------
      ! @ private
      ! Decode user inputs found in the Sic variables
      !   MAP_RA    [sexagesimal hour angle]
      !   MAP_DEC   [sexagesimal degrees]
      !   MAP_ANGLE [deg]
      ! Returned values are all radians. Also return a status indicating if
      ! the values were set or not.
      !---------------------------------------------------------------------
      character(len=*), intent(in)    :: rname      !
      logical,          intent(out)   :: found(3)   !
      real(kind=8),     intent(out)   :: values(3)  !
      logical,          intent(inout) :: error      !
    end subroutine map_get_radecang
  end interface
  !
  interface
    subroutine uv_reweight_comm(line,error)
      use clean_def
      use clean_arrays
      use gbl_message
      use phys_const
      !------------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Reweight the data
      !------------------------------------------------------------------------
      character(len=*), intent(in) :: line  ! Command line
      logical, intent(out) :: error         ! Error flag
    end subroutine uv_reweight_comm
  end interface
  !
  interface
    subroutine uv_resample_comm(line,comm,error)
      use clean_def
      use clean_arrays
      use clean_types
      use gbl_message
      use phys_const
      !---------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Resample in velocity the UV Table or compress it
      !   Support for commands
      !     UV_RESAMPLE NC [Ref Val Inc]
      !     UV_COMPRESS NC
      !---------------------------------------------------------------------
      character(len=*), intent(in) :: line  ! Command line
      character(len=*), intent(in) :: comm  ! Calling command
      logical, intent(out) :: error         ! Error flag
    end subroutine uv_resample_comm
  end interface
  !
  interface
    subroutine uv_residual(line,error)
      use clean_def
      use clean_arrays
      use clean_types
      use clean_default
      use gbl_message
      !$ use omp_lib
      !------------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING   Support for routine UV_RESIDUAL
      !     Remove all Clean Components from a UV Table
      !
      ! Input :
      !     a precessed UV table
      !     a list of Clean Components, in DCCT format
      !      i.e. (x,y,v)(iplane,icomponent)
      !     this organisation is not efficient, and one may need to switch to
      !           (x,y,v,)(icomponent,iplane)
      !     which is more easily transmitted
      ! Output :
      !     a precessed, rotated, shifted UV table, sorted in V,
      !     ordered in (U,V,W,D,T,iant,jant,nchan(real,imag,weig))
      !
      !------------------------------------------------------------------------
      character(len=*), intent(inout) :: line
      logical, intent(out) :: error
    end subroutine uv_residual
  end interface
  !
  interface
    subroutine sub_uv_residual(task,line,duv_previous,duv_next,do_clean,error)
      use clean_def
      use clean_arrays
      use clean_types
      use clean_default
      use gbl_message
      !$ use omp_lib
      !------------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING   Support for routine UV_RESIDUAL
      !     Remove all Clean Components from a UV Table
      !
      ! Input :
      !     a precessed UV table
      !     a list of Clean Components, in DCCT format
      !      i.e. (x,y,v)(iplane,icomponent)
      !     this organisation is not efficient, and one may need to switch to
      !           (x,y,v,)(icomponent,iplane)
      !     which is more easily transmitted
      ! Output :
      !     a precessed, rotated, shifted UV table, sorted in V,
      !     ordered in (U,V,W,D,T,iant,jant,nchan(real,imag,weig))
      !
      !------------------------------------------------------------------------
      character(len=*), intent(in) :: task
      character(len=*), intent(inout) :: line
      real, pointer, intent(out) :: duv_previous(:,:)
      real, pointer, intent(out) :: duv_next(:,:)
      logical, intent(in) :: do_clean
      logical, intent(out) :: error
    end subroutine sub_uv_residual
  end interface
  !
  interface
    subroutine uv_restore(line,error)
      use clean_def
      use clean_arrays
      use clean_types
      use clean_default
      use gbl_message
      !------------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING   Support for routine UV_RESTORE
      !     Restore a Clean map from a CLIC UV Table and a list of Clean
      !     Components, by Gridding and Fast Fourier Transform, using adequate
      !     scratch space for optimisation. Will work for
      !     up to 128x128x128 cube data size, may be more...
      !
      ! Input :
      !     a precessed UV table
      !     a list of Clean Components, in DCCT format
      !      i.e. (x,y,v)(iplane,icomponent)
      !     this organisation is not efficient, and one may need to switch to
      !           (x,y,v,)(icomponent,iplane)
      !     which is more easily transmitted
      ! Output :
      !     (optionally) a precessed, rotated, shifted UV table, sorted in V,
      !     ordered in (U,V,W,D,T,iant,jant,nchan(real,imag,weig))
      !     a beam image or cube
      !     a LMV-CLEAN cube
      !     a LMV-RES cube
      !
      ! To work, it would require
      !  1) to have the Pixel definition stored with the TCC Clean Component
      !     structure
      !  2) to re-use this pixel definition for the Images to be created
      !     Images may be different, but this would create some inconsistencies,
      !     which are perhaps not dramatic since the subtraction is in the UV
      !     plane.
      !
      ! Here, for the time being, I assume that the pixel of the TCC are
      ! the same as defined by the current imaging characteristics.
      !------------------------------------------------------------------------
      character(len=*), intent(inout) :: line
      logical, intent(out) :: error
    end subroutine uv_restore
  end interface
  !
  interface
    subroutine uv_subtract_clean(huv,duv,ouv,hcct,dcct,freq)
      use image_def
      use gbl_message
      !$ use omp_lib
      !-----------------------------------------------------------------
      ! @ private
      !
      ! MAPPING   Support for UV_RESTORE
      !     Compute visibilities for a UV data set according to a
      !     set of Clean Components , and remove them from the original
      !     UV table
      !   This version is for CCT tables from MAPPING
      !-----------------------------------------------------------------
      type(gildas), intent(in) :: huv   ! header of UV data set
      type(gildas), intent(in) :: hcct  ! header of CCT data set
      real, intent(in) :: duv(huv%gil%dim(1),huv%gil%dim(2))
      real, intent(out) :: ouv(huv%gil%dim(1),huv%gil%dim(2))
      real, intent(in) :: dcct(hcct%gil%dim(1),hcct%gil%dim(2),hcct%gil%dim(3))
      real(8), intent(in) :: freq       ! Apparent observing frequency
    end subroutine uv_subtract_clean
  end interface
  !
  interface
    subroutine uv_clean_size(hcct,ccin, mic)
      use image_def
      !-----------------------------------------------------------------
      ! @ private
      !
      ! MAPPING   Support for UV_RESTORE
      !   Compute the actual number of components
      !-----------------------------------------------------------------
      type(gildas), intent(in) :: hcct  ! header of CCT data set
      real, intent(in) :: ccin(hcct%gil%dim(1),hcct%gil%dim(2),hcct%gil%dim(3))
      integer, intent(out) :: mic
    end subroutine uv_clean_size
  end interface
  !
  interface
    subroutine uv_compact_clean(hcct,ccin,occt,ccou, mic)
      use image_def
      !-----------------------------------------------------------------
      ! @ private
      !
      ! MAPPING   Support for UV_RESTORE
      !   Compact the component list by summing up all values at the
      !   same position
      !-----------------------------------------------------------------
      type(gildas), intent(in) :: hcct  ! header of CCT data set
      type(gildas), intent(in) :: occt  ! header of CCT data set
      real, intent(in) :: ccin(hcct%gil%dim(1),hcct%gil%dim(2),hcct%gil%dim(3))
      real, intent(out) :: ccou(occt%gil%dim(1),occt%gil%dim(2),occt%gil%dim(3))
      integer, intent(inout) :: mic
    end subroutine uv_compact_clean
  end interface
  !
  interface
    subroutine uv_subtfast_clean(huv,duv,ouv,hcct,dcct,hdirty,freq,cpu0)
      use image_def
      use gbl_message
      !$ use omp_lib
      !-----------------------------------------------------------------
      ! @ private
      !
      ! MAPPING   Support for UV_RESTORE
      !     Compute visibilities for a UV data set according to a
      !     set of Clean Components , and remove them from the original
      !     UV table
      ! This version is for CCT tables from MAPPING
      !-----------------------------------------------------------------
      type(gildas), intent(in) :: huv   ! header of UV data set
      type(gildas), intent(in) :: hcct  ! header of CCT data set
      type(gildas), intent(in) :: hdirty ! header of DIRTY data set
      real, intent(in) :: duv(huv%gil%dim(1),huv%gil%dim(2))
      real, intent(out) :: ouv(huv%gil%dim(1),huv%gil%dim(2))
      real, intent(in) :: dcct(hcct%gil%dim(1),hcct%gil%dim(2),hcct%gil%dim(3))
      real(8), intent(in) :: freq       ! Observing frequency in MHz
      real, intent(in) :: cpu0
    end subroutine uv_subtfast_clean
  end interface
  !
  interface
    subroutine do_fmodel (visi,nc,nv,a,nx,ny,nf,   &
         &    freq,xinc,yinc,factor,elapsed)
      !$ use omp_lib
      !-----------------------------------------------------------------
      ! @ private
      !
      ! MAPPING   Support for UV_RESTORE
      !     Remove model visibilities from UV data set.
      !     "Fast" Model from an image: Use an intermediate FFT
      !-----------------------------------------------------------------
      integer, intent(in) :: nc ! Visibility size
      integer, intent(in) :: nv ! Number of visibilities
      integer, intent(in) :: nx ! X size
      integer, intent(in) :: ny ! Y size
      integer, intent(in) :: nf ! Number of frequencies
      real, intent(inout) :: visi(nc,nv)  ! Computed visibilities
      real(8), intent(in) :: freq           ! Effective frequency
      real, intent(in)  :: factor         ! Flux factor
      complex, intent(in) ::  a(nx,ny,nf) ! FFT
      real(8), intent(in) :: xinc,yinc    ! Pixel sizes
      real(8), intent(out) :: elapsed
    end subroutine do_fmodel
  end interface
  !
  interface
    subroutine doallffts (mx,my,comp,nx,ny,nf,fft,elapsed)
      !$ use omp_lib
      !------------------------------------------------------
      ! @ private
      !
      ! MAPPING   Support for UV_RESTORE and others
      !   Could be optimized with Fourt_Plan and intermediate
      !   array
      !------------------------------------------------------
      integer, intent(in) :: mx
      integer, intent(in) :: my
      real, intent(in) :: comp(mx,my,nf)
      integer, intent(in) :: nx ! X size
      integer, intent(in) :: ny ! Y size
      integer, intent(in) :: nf ! Number of frequencies
      complex, intent(inout) :: fft(nx,ny,nf)
      real(8), intent(out) :: elapsed
    end subroutine doallffts
  end interface
  !
  interface
    subroutine recent(nx,ny,z)
      !------------------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING   Support for UV_RESTORE and others
      !   Recenters the Fourier Transform, for easier display. The present version
      !   will only work for even dimensions.
      !------------------------------------------------------------------------------
      integer, intent(in) :: nx
      integer, intent(in) :: ny
      complex, intent(inout) :: z(nx,ny)
    end subroutine recent
  end interface
  !
  interface
    subroutine plunge_real (r,nx,ny,c,mx,my)
      !------------------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING   Support for UV_RESTORE and others
      !     Plunge a Real array into a Complex array
      !-----------------------------------------------------------------
      integer, intent(in)  :: nx,ny              ! Size of input array
      real, intent(in)     :: r(nx,ny)           ! Input real array
      integer, intent(in)  :: mx,my              ! Size of output array
      complex, intent(out) :: c(mx,my)           ! Output complex array
    end subroutine plunge_real
  end interface
  !
  interface
    subroutine generate_clean(hmap,first,last,ccou,mic)
      use clean_arrays
      !
      ! @ private-mandatory
      !
      type(gildas), intent(inout) :: hmap
      integer, intent(in) :: first
      integer, intent(in) :: last
      real, intent(in) :: ccou(:,:,:)  !  (x,y,Value),Component,Plane
      integer, intent(in) :: mic(:)    ! Niter per plane
    end subroutine generate_clean
  end interface
  !
  interface
    subroutine map_parameters(task,map,freq,uvmax,uvmin,error,print)
      use clean_def
      use clean_arrays
      use clean_types
      use clean_default
      use gbl_message
      !----------------------------------------------------------------------
      ! @ private-mandatory
      !
      ! Prepare the MAP parameters for UV_MAP, MX or UV_RESTORE
      ! Reads this from SIC variables
      !-----------------------------------------------------------------------
      character(len=*), intent(in)    :: task ! Input task name (UV_MAP or MX)
      type (uvmap_par), intent(inout) :: map  ! Map parameters
      real(8), intent(inout) :: freq          ! Observing frequency
      real(4), intent(in) :: uvmax, uvmin     ! Min & Max UV in m
      logical, intent(inout) :: error
      logical, optional :: print
    end subroutine map_parameters
  end interface
  !
  interface
    subroutine uv_stat_comm(line,error)
      use clean_def
      use clean_default
      use clean_arrays
      use code_names
      use gbl_message
      !------------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING   Support for command UV_STAT
      !       Analyse a UV data set to define approximate beam size,
      !       field of view, expected "best beam", etc...
      ! Input :
      !     a precessed UV table
      ! Output :
      !     a precessed, rotated, shifted UV table, sorted in V,
      !     ordered in (U,V,W,D,T,iant,jant,nchan(real,imag,weig))
      !     a beam image ?
      !------------------------------------------------------------------------
      character(len=*), intent(inout) :: line
      logical, intent(out) :: error
    end subroutine uv_stat_comm
  end interface
  !
  interface
    subroutine uv_printoffset(rname,map)
      use clean_def
      use gbl_message
      ! @ private
      character(len=*), intent(in) :: rname
      type(uvmap_par), intent(in) :: map
    end subroutine uv_printoffset
  end interface
  !
  interface
    subroutine uv_spectral_frequency_sel(frequency, width, unit, channels, error)
      use gbl_message
      use clean_arrays
      !--------------------------------------------------------
      ! @ private
      !
      ! Mapping
      ! returns spectral channels based on a frequency and width given as input
      !--------------------------------------------------------
      real(kind=8), intent(in)      :: frequency
      real(kind=4), intent(in)      :: width
      character(len=10), intent(in) :: unit
      integer(kind=4), intent(out)  :: channels(2)
      logical, intent(inout)        :: error
    end subroutine uv_spectral_frequency_sel
  end interface
  !
  interface
    subroutine uv_spectral_velocity_sel(velocity, width, unit, channels, error)
      use gbl_message
      use clean_arrays
      !--------------------------------------------------------
      ! @ private
      !
      ! Mapping
      ! returns spectral channels based on a velocity and width given as input
      !--------------------------------------------------------
      real(kind=8), intent(in)      :: velocity
      real(kind=4), intent(in)      :: width
      character(len=10), intent(in) :: unit
      integer(kind=4), intent(out)  :: channels(2)
      logical, intent(inout)        :: error
    end subroutine uv_spectral_velocity_sel
  end interface
  !
  interface
    subroutine uv_spectral_range_sel(range,unit,channels, error)
      use gbl_message
      use clean_arrays
      !--------------------------------------------------------
      ! @ private
      !
      ! Mapping
      ! returns spectral channels based on a range at a known unit
      !--------------------------------------------------------
      real(kind=4), intent(in)      :: range(2)
      character(len=10), intent(in) :: unit
      integer(kind=4), intent(out)  :: channels(2)
      logical, intent(inout)        :: error
    end subroutine uv_spectral_range_sel
  end interface
  !
  interface
    subroutine uv_spectral_getcols(channel,columns)
      use clean_arrays
      !--------------------------------------------------------
      ! @ private
      !
      ! Mapping
      ! returns real, imaginary and weight columns for a given channel
      ! Returns -1 for all columns if channel outside table
      !--------------------------------------------------------
      integer(kind=index_length),intent(in)  :: channel
      integer(kind=index_length),intent(out) :: columns(3)
    end subroutine uv_spectral_getcols
  end interface
  !
  interface
    subroutine uv_spectral_flag(channel,error)
      use clean_arrays
      !--------------------------------------------------------
      ! @ private
      !
      ! Mapping
      ! Flags a UV channel i.e. makes its weight negative.
      ! Returns error if channel is outside UV table range.
      !--------------------------------------------------------
      integer(kind=index_length),intent(in)  :: channel
      logical,intent(inout)                  :: error
    end subroutine uv_spectral_flag
  end interface
  !
  interface
    subroutine uv_spectral_zero(channel,error)
      use clean_arrays
      !--------------------------------------------------------
      ! @ private
      !
      ! Mapping
      ! zeros a UV channel
      ! Returns error if channel is outside UV table range.
      !--------------------------------------------------------
      integer(kind=index_length),intent(in)  :: channel
      logical,intent(inout)                  :: error
    end subroutine uv_spectral_zero
  end interface
  !
  interface
    subroutine uv_time_comm(line,error)
      use clean_arrays
      use clean_types
      use gbl_message
      !---------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !
      ! Time-Average the UV Table (usefull for ALMA data, although
      ! it is more efficient to do that on the original UV table
      ! before reading it).
      !
      ! UV_TIME Time /Weight Wcol
      !
      ! Works on the "current" UV data set: Uses UVS or UVR as needed.
      !
      !---------------------------------------------------------------------
      character(len=*), intent(in) :: line
      logical, intent(out) :: error
    end subroutine uv_time_comm
  end interface
  !
  interface
    subroutine uv_truncate_comm(line,error)
      use clean_arrays
      use clean_types
      use gbl_message
      !---------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !
      ! Truncate a UV Table, by removing baselines out of
      ! a given range (Min and Max)
      !
      ! UV_TRUNCATE Max [Min]
      !
      ! Works on the "current" UV data set: Uses UVS or UVR as needed.
      !
      !---------------------------------------------------------------------
      character(len=*), intent(in) :: line
      logical, intent(out) :: error
    end subroutine uv_truncate_comm
  end interface
  !
  interface
    subroutine get_weightmode(task,mode,error)
      use gbl_message
      !---------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Decode the weighting mode
      !---------------------------------------------------------------------
      character(len=*), intent(in) :: task    ! Caller's name
      character(len=*), intent(inout) :: mode ! Weighting mode
      logical, intent(out) :: error           ! Error flag
    end subroutine get_weightmode
  end interface
  !
  interface
    subroutine get_clean (method,hbeam,dbeam,error)
      use clean_def
      use image_def
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Get Clean beam parameters
      !----------------------------------------------------------------------
      type (gildas), intent(in) :: hbeam
      type (clean_par), intent(inout) :: method
      real, intent(in) :: dbeam(hbeam%gil%dim(1),hbeam%gil%dim(2))
      logical, intent(out) :: error
    end subroutine get_clean
  end interface
  !
  interface
    subroutine beam_unit_conversion(user_method)
      use clean_def
      use clean_default
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Major and minor SIC variables are in seconds but the method structure
      !   assumes that its major and minor fields are in radian. So we need
      !   a conversion...
      !----------------------------------------------------------------------
      type (clean_par) :: user_method
    end subroutine beam_unit_conversion
  end interface
  !
  interface
    subroutine copy_method(in,out)
      use clean_def
      !----------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Copy method
      !----------------------------------------------------------------------
      type (clean_par), intent(in) ::  in
      type (clean_par), intent(inout) ::  out
    end subroutine copy_method
  end interface
  !
  interface
    subroutine copy_param(in,out)
      !--------------------------------------------------------------------
      ! @ private
      !
      ! MAPPING
      !   Copy only the parameters of the method, not the pointers to the
      !   work arrays
      !   Do not copy the Loop Gain, which is Method specific
      !--------------------------------------------------------------------
      use clean_def
      type (clean_par), intent(in) ::  in
      type (clean_par), intent(inout) ::  out
    end subroutine copy_param
  end interface
  !
  interface
    subroutine mapping_print_debug(m)
      ! @ private
      use clean_def
      type (clean_par) :: m
    end subroutine mapping_print_debug
  end interface
  !
end module mapping_interfaces_private
