//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
//
//---------------------------------------------------------------------------
//
// GEANT4 Class file
//
// Description: Data on stopping power
//
// Author:      Anton Ivantchenko 07.08.2013
//
// Organisation:        G4AI Ltd, UK
// Customer:            ESA/ESTEC, NOORDWIJK
// Contract:            CSMAN-5288
//
// Modifications:
// 
//----------------------------------------------------------------------------
//

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

#include "G4ESTARStopping.hh" 
#include "G4SystemOfUnits.hh"
#include "G4Material.hh"

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

G4ESTARStopping::G4ESTARStopping(const G4String& datatype)
{
  currentMaterial = 0;
  matIndex = -1;

  name.resize(280,"");
  sdata.resize(280,0);

  type = 0;
  if("basic" == datatype)     { type = 1; }
  else if("long" == datatype) { type = 2; }

  Initialise();
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

G4ESTARStopping::~G4ESTARStopping()
{
  for(size_t i=1; i<280; ++i) { delete sdata[i]; }
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

G4int G4ESTARStopping:: GetIndex (const G4Material* mat)
{  
  if (mat != currentMaterial) {
    G4String matName = mat->GetName();
    for (G4int i=1; i<280; ++i){
      if (matName == name[i]){ 
	matIndex = i;
	currentMaterial = mat;
	break;
      }
    }
  }
  return matIndex;
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

G4double G4ESTARStopping::GetElectronicDEDX(G4int i, G4double energy)
{
  G4double res = 0.0;
  if (i<1 || i>279){
    G4cout << "### G4ESTARStopping WARNING: index " 
	   << i << " is out of range!" << G4endl; 
    return res;
  }
  if(energy < emin) { res = (*(sdata[i]))[0]*std::sqrt(energy/emin); } 
  else              { res = sdata[i]->Value(energy); }
  return res;
}
 
void G4ESTARStopping::Initialise()
{
  dirPath = std::getenv("G4LEDATA");
  if(!dirPath) {
    G4Exception("G4ESTARStopping()","em0006",FatalException,
		"Environment variable G4LEDATA not defined");
    return;
  }
 
static const G4double T0[25]={1.0E-03,3.0E-03,5.0E-03,9.0E-03,2.5E-02,4.5E-02,8.0E-02,.2,.4,.7,1.5,3.5,6.0,10.,30.,50.,90.,250.,450.,800.,2000.,4.0E+03,6.0E+03,8.0E+03,1.0E+04};
emin = T0[0]*MeV;

name[1]="G4_1,2-DICHLOROBENZENE";
static const G4double e1[25]={9.473E+01,4.632E+01,3.203E+01,2.063E+01,9.472,6.123,4.106,2.428,1.878,1.678,1.619,1.716,1.841,2.015,2.762,3.48,4.914,1.07E+01,1.799E+01,3.08E+01,7.487E+01,1.485E+02,2.221E+02,2.958E+02,3.695E+02};
AddData(T0,e1,1);
name[2]="G4_1,2-DICHLOROETHANE";
static const G4double e2[25]={9.312E+01,4.583E+01,3.175E+01,2.048E+01,9.424,6.098,4.092,2.423,1.874,1.677,1.623,1.733,1.873,2.07,2.936,3.778,5.465,1.229E+01,2.088E+01,3.598E+01,8.794E+01,1.747E+02,2.615E+02,3.484E+02,4.352E+02};
AddData(T0,e2,2);
name[3]="G4_A-150_TISSUE";
static const G4double e3[25]={1.246E+02,5.765E+01,3.923E+01,2.491E+01,1.124E+01,7.209,4.803,2.816,2.161,1.915,1.819,1.883,1.977,2.105,2.616,3.089,4.021,7.756,1.245E+01,2.071E+01,4.912E+01,9.659E+01,1.441E+02,1.916E+02,2.392E+02};
AddData(T0,e3,3);
name[4]="G4_ACETONE";
static const G4double e4[25]={1.256E+02,5.805E+01,3.949E+01,2.507E+01,1.13E+01,7.25,4.83,2.831,2.176,1.939,1.847,1.912,2.007,2.137,2.659,3.139,4.085,7.872,1.263E+01,2.1E+01,4.98E+01,9.79E+01,1.461E+02,1.942E+02,2.424E+02};
AddData(T0,e4,4);
name[5]="G4_ACETYLENE";
static const G4double e5[25]={1.268E+02,5.804E+01,3.938E+01,2.493E+01,1.121E+01,7.179,4.777,2.796,2.147,1.923,1.873,2.008,2.158,2.342,2.975,3.466,4.39,8.003,1.251E+01,2.04E+01,4.755E+01,9.291E+01,1.383E+02,1.837E+02,2.291E+02};
AddData(T0,e5,5);
name[6]="G4_ADENINE";
static const G4double e6[25]={1.138E+02,5.313E+01,3.625E+01,2.308E+01,1.044E+01,6.708,4.474,2.627,2.017,1.789,1.702,1.767,1.861,1.989,2.51,2.995,3.955,7.809,1.266E+01,2.118E+01,5.049E+01,9.946E+01,1.485E+02,1.975E+02,2.465E+02};
AddData(T0,e6,6);
name[7]="G4_ADIPOSE_TISSUE_ICRP";
static const G4double e7[25]={1.28E+02,5.907E+01,4.017E+01,2.549E+01,1.149E+01,7.366,4.906,2.875,2.209,1.962,1.865,1.928,2.023,2.151,2.668,3.143,4.079,7.824,1.253E+01,2.081E+01,4.93E+01,9.689E+01,1.445E+02,1.922E+02,2.399E+02};
AddData(T0,e7,7);
name[8]="G4_AIR";
static const G4double e8[25]={1.025E+02,4.879E+01,3.349E+01,2.142E+01,9.757,6.284,4.201,2.474,1.908,1.715,1.68,1.817,1.967,2.159,2.868,3.452,4.551,8.868,1.427E+01,2.372E+01,5.623E+01,1.105E+02,1.648E+02,2.191E+02,2.734E+02};
AddData(T0,e8,8);
name[9]="G4_ALANINE";
static const G4double e9[25]={1.18E+02,5.516E+01,3.765E+01,2.397E+01,1.085E+01,6.969,4.649,2.73,2.097,1.859,1.768,1.832,1.927,2.058,2.596,3.096,4.085,8.048,1.303E+01,2.179E+01,5.193E+01,1.023E+02,1.527E+02,2.031E+02,2.535E+02};
AddData(T0,e9,9);
name[10]="G4_ALUMINUM_OXIDE";
static const G4double e10[25]={8.033E+01,4.111E+01,2.878E+01,1.873E+01,8.715,5.664,3.815,2.269,1.753,1.57,1.519,1.611,1.725,1.887,2.606,3.299,4.683,1.027E+01,1.73E+01,2.966E+01,7.216E+01,1.431E+02,2.142E+02,2.852E+02,3.562E+02};
AddData(T0,e10,10);
name[11]="G4_AMBER";
static const G4double e11[25]={1.265E+02,5.837E+01,3.969E+01,2.518E+01,1.135E+01,7.277,4.847,2.84,2.179,1.93,1.832,1.894,1.987,2.112,2.612,3.072,3.978,7.609,1.217E+01,2.02E+01,4.781E+01,9.394E+01,1.401E+02,1.863E+02,2.325E+02};
AddData(T0,e11,11);
name[12]="G4_AMMONIA";
static const G4double e12[25]={1.421E+02,6.463E+01,4.376E+01,2.765E+01,1.24E+01,7.933,5.274,3.083,2.365,2.117,2.06,2.207,2.37,2.57,3.264,3.796,4.786,8.656,1.349E+01,2.196E+01,5.11E+01,9.978E+01,1.485E+02,1.973E+02,2.46E+02};
AddData(T0,e12,12);
name[13]="G4_ANILINE";
static const G4double e13[25]={1.211E+02,5.613E+01,3.822E+01,2.428E+01,1.096E+01,7.032,4.686,2.748,2.112,1.875,1.783,1.847,1.94,2.065,2.566,3.026,3.935,7.574,1.215E+01,2.02E+01,4.788E+01,9.414E+01,1.404E+02,1.868E+02,2.331E+02};
AddData(T0,e13,13);
name[14]="G4_ANTHRACENE";
static const G4double e14[25]={1.17E+02,5.446E+01,3.714E+01,2.362E+01,1.068E+01,6.857,4.572,2.683,2.059,1.826,1.736,1.80,1.892,2.016,2.507,2.961,3.857,7.45,1.197E+01,1.991E+01,4.724E+01,9.29E+01,1.386E+02,1.843E+02,2.301E+02};
AddData(T0,e14,14);
name[15]="G4_B-100_BONE";
static const G4double e15[25]={1.082E+02,5.152E+01,3.536E+01,2.262E+01,1.03E+01,6.638,4.438,2.614,2.013,1.791,1.714,1.793,1.902,2.054,2.698,3.307,4.519,9.398,1.554E+01,2.634E+01,6.348E+01,1.255E+02,1.876E+02,2.497E+02,3.119E+02};
AddData(T0,e15,15);
name[16]="G4_BAKELITE";
static const G4double e16[25]={1.154E+02,5.395E+01,3.683E+01,2.345E+01,1.062E+01,6.821,4.551,2.672,2.054,1.824,1.737,1.802,1.896,2.023,2.534,3.006,3.94,7.684,1.239E+01,2.067E+01,4.914E+01,9.671E+01,1.443E+02,1.919E+02,2.396E+02};
AddData(T0,e16,16);
name[17]="G4_BARIUM_FLUORIDE";
static const G4double e17[25]={3.755E+01,2.48E+01,1.839E+01,1.254E+01,6.162,4.098,2.814,1.722,1.362,1.25,1.269,1.473,1.725,2.111,4.009,5.934,9.844,2.576E+01,4.584E+01,8.111E+01,2.024E+02,4.049E+02,6.076E+02,8.102E+02,1.013E+03};
AddData(T0,e17,17);
name[18]="G4_BARIUM_SULFATE";
static const G4double e18[25]={4.926E+01,2.94E+01,2.135E+01,1.433E+01,6.913,4.563,3.114,1.888,1.48,1.347,1.347,1.527,1.748,2.084,3.712,5.355,8.685,2.223E+01,3.931E+01,6.932E+01,1.725E+02,3.448E+02,5.172E+02,6.896E+02,8.621E+02};
AddData(T0,e18,18);
name[19]="G4_BENZENE";
static const G4double e19[25]={1.231E+02,5.683E+01,3.865E+01,2.452E+01,1.105E+01,7.088,4.721,2.767,2.126,1.888,1.797,1.86,1.953,2.077,2.57,3.022,3.913,7.481,1.197E+01,1.986E+01,4.7E+01,9.235E+01,1.377E+02,1.832E+02,2.286E+02};
AddData(T0,e19,19);
name[20]="G4_BERYLLIUM_OXIDE";
static const G4double e20[25]={9.533E+01,4.583E+01,3.154E+01,2.023E+01,9.241,5.96,3.989,2.352,1.802,1.601,1.529,1.594,1.682,1.802,2.307,2.782,3.725,7.508,1.227E+01,2.063E+01,4.94E+01,9.744E+01,1.455E+02,1.936E+02,2.417E+02};
AddData(T0,e20,20);
name[21]="G4_BGO";
static const G4double e21[25]={2.579E+01,2.082E+01,1.597E+01,1.118E+01,5.647,3.798,2.633,1.634,1.31,1.216,1.258,1.502,1.802,2.265,4.583,6.951,1.178E+01,3.144E+01,5.621E+01,9.973E+01,2.494E+02,4.991E+02,7.49E+02,9.99E+02,1.249E+03};
AddData(T0,e21,21);
name[22]="G4_BLOOD_ICRP";
static const G4double e22[25]={1.186E+02,5.565E+01,3.803E+01,2.424E+01,1.099E+01,7.065,4.716,2.771,2.134,1.905,1.819,1.888,1.987,2.126,2.707,3.248,4.315,8.594,1.398E+01,2.343E+01,5.596E+01,1.103E+02,1.647E+02,2.191E+02,2.735E+02};
AddData(T0,e22,22);
name[23]="G4_BONE_COMPACT_ICRU";
static const G4double e23[25]={1.059E+02,5.084E+01,3.497E+01,2.242E+01,1.024E+01,6.601,4.418,2.605,2.004,1.782,1.705,1.786,1.898,2.055,2.729,3.369,4.645,9.784,1.626E+01,2.763E+01,6.675E+01,1.321E+02,1.975E+02,2.629E+02,3.283E+02};
AddData(T0,e23,23);
name[24]="G4_BONE_CORTICAL_ICRP";
static const G4double e24[25]={9.814E+01,4.798E+01,3.317E+01,2.136E+01,9.81,6.342,4.252,2.514,1.941,1.732,1.665,1.755,1.876,2.046,2.788,3.498,4.916,1.064E+01,1.784E+01,3.05E+01,7.405E+01,1.468E+02,2.196E+02,2.924E+02,3.652E+02};
AddData(T0,e24,24);
name[25]="G4_BORON_CARBIDE";
static const G4double e25[25]={9.703E+01,4.614E+01,3.165E+01,2.024E+01,9.214,5.933,3.966,2.334,1.785,1.584,1.511,1.573,1.655,1.761,2.182,2.574,3.349,6.456,1.037E+01,1.724E+01,4.088E+01,8.038E+01,1.199E+02,1.595E+02,1.99E+02};
AddData(T0,e25,25);
name[26]="G4_BORON_OXIDE";
static const G4double e26[25]={9.448E+01,4.58E+01,3.159E+01,2.03E+01,9.298,6.003,4.021,2.374,1.834,1.638,1.571,1.642,1.738,1.869,2.415,2.925,3.937,7.999,1.311E+01,2.209E+01,5.299E+01,1.046E+02,1.562E+02,2.079E+02,2.595E+02};
AddData(T0,e26,26);
name[27]="G4_BRAIN_ICRP";
static const G4double e27[25]={1.206E+02,5.646E+01,3.856E+01,2.456E+01,1.112E+01,7.148,4.769,2.801,2.156,1.925,1.837,1.905,2.004,2.143,2.721,3.259,4.321,8.577,1.393E+01,2.333E+01,5.569E+01,1.097E+02,1.638E+02,2.18E+02,2.721E+02};
AddData(T0,e27,27);
name[28]="G4_BUTANE";
static const G4double e28[25]={1.465E+02,6.602E+01,4.458E+01,2.81E+01,1.256E+01,8.025,5.33,3.11,2.383,2.131,2.071,2.214,2.371,2.563,3.153,3.619,4.51,7.999,1.235E+01,1.999E+01,4.629E+01,9.023E+01,1.342E+02,1.782E+02,2.223E+02};
AddData(T0,e28,28);
name[29]="G4_C-552";
static const G4double e29[25]={1.021E+02,4.867E+01,3.342E+01,2.139E+01,9.745,6.278,4.198,2.473,1.906,1.696,1.62,1.689,1.785,1.917,2.472,2.992,4.024,8.169,1.339E+01,2.255E+01,5.407E+01,1.067E+02,1.594E+02,2.121E+02,2.648E+02};
AddData(T0,e29,29);
name[30]="G4_CADMIUM_TELLURIDE";
static const G4double e30[25]={2.523E+01,2.051E+01,1.576E+01,1.103E+01,5.578,3.751,2.598,1.607,1.282,1.186,1.221,1.449,1.726,2.153,4.259,6.399,1.075E+01,2.844E+01,5.075E+01,8.995E+01,2.248E+02,4.498E+02,6.749E+02,9.0E+02,1.125E+03};
AddData(T0,e30,30);
name[31]="G4_CADMIUM_TUNGSTATE";
static const G4double e31[25]={3.063E+01,2.264E+01,1.712E+01,1.186E+01,5.926,3.968,2.74,1.69,1.343,1.238,1.267,1.491,1.771,2.204,4.352,6.54,1.099E+01,2.911E+01,5.194E+01,9.205E+01,2.3E+02,4.602E+02,6.905E+02,9.209E+02,1.151E+03};
AddData(T0,e31,31);
name[32]="G4_CALCIUM_CARBONATE";
static const G4double e32[25]={8.429E+01,4.271E+01,2.982E+01,1.936E+01,8.984,5.833,3.925,2.332,1.802,1.613,1.559,1.66,1.791,1.978,2.811,3.617,5.234,1.177E+01,2.001E+01,3.449E+01,8.43E+01,1.675E+02,2.507E+02,3.34E+02,4.172E+02};
AddData(T0,e32,32);
name[33]="G4_CALCIUM_FLUORIDE";
static const G4double e33[25]={7.46E+01,3.909E+01,2.753E+01,1.801E+01,8.432,5.495,3.71,2.213,1.719,1.547,1.505,1.615,1.755,1.959,2.881,3.78,5.587,1.29E+01,2.212E+01,3.832E+01,9.406E+01,1.871E+02,2.803E+02,3.734E+02,4.666E+02};
AddData(T0,e33,33);
name[34]="G4_CALCIUM_OXIDE";
static const G4double e34[25]={7.421E+01,3.932E+01,2.777E+01,1.821E+01,8.552,5.58,3.771,2.253,1.747,1.571,1.533,1.657,1.813,2.04,3.073,4.087,6.13,1.441E+01,2.485E+01,4.321E+01,1.064E+02,2.118E+02,3.173E+02,4.229E+02,5.284E+02};
AddData(T0,e34,34);
name[35]="G4_CALCIUM_SULFATE";
static const G4double e35[25]={7.994E+01,4.125E+01,2.894E+01,1.887E+01,8.798,5.724,3.858,2.298,1.779,1.595,1.548,1.656,1.793,1.989,2.865,3.717,5.426,1.234E+01,2.105E+01,3.636E+01,8.903E+01,1.77E+02,2.65E+02,3.53E+02,4.41E+02};
AddData(T0,e35,35);
name[36]="G4_CALCIUM_TUNGSTATE";
static const G4double e36[25]={3.722E+01,2.514E+01,1.872E+01,1.281E+01,6.316,4.207,2.893,1.775,1.408,1.295,1.316,1.53,1.798,2.211,4.254,6.33,1.055E+01,2.775E+01,4.942E+01,8.748E+01,2.184E+02,4.369E+02,6.555E+02,8.741E+02,1.093E+03};
AddData(T0,e36,36);
name[37]="G4_CARBON_DIOXIDE";
static const G4double e37[25]={1.029E+02,4.897E+01,3.36E+01,2.149E+01,9.784,6.301,4.212,2.48,1.913,1.719,1.684,1.822,1.973,2.165,2.872,3.451,4.557,8.914,1.436E+01,2.392E+01,5.674E+01,1.115E+02,1.664E+02,2.212E+02,2.761E+02};
AddData(T0,e37,37);
name[38]="G4_CARBON_TETRACHLORIDE";
static const G4double e38[25]={7.367E+01,3.861E+01,2.72E+01,1.78E+01,8.333,5.431,3.666,2.188,1.703,1.537,1.505,1.632,1.787,2.005,2.983,3.94,5.864,1.365E+01,2.347E+01,4.072E+01,1.001E+02,1.991E+02,2.983E+02,3.974E+02,4.966E+02};
AddData(T0,e38,38);
name[39]="G4_CELLULOSE_BUTYRATE";
static const G4double e39[25]={1.152E+02,5.403E+01,3.692E+01,2.353E+01,1.066E+01,6.853,4.573,2.687,2.069,1.84,1.754,1.82,1.916,2.047,2.583,3.079,4.059,7.99,1.293E+01,2.162E+01,5.151E+01,1.014E+02,1.514E+02,2.014E+02,2.514E+02};
AddData(T0,e39,39);
name[40]="G4_CELLULOSE_CELLOPHANE";
static const G4double e40[25]={1.13E+02,5.323E+01,3.642E+01,2.323E+01,1.054E+01,6.781,4.528,2.662,2.049,1.821,1.735,1.802,1.898,2.031,2.577,3.085,4.09,8.122,1.319E+01,2.211E+01,5.277E+01,1.04E+02,1.552E+02,2.065E+02,2.578E+02};
AddData(T0,e40,40);
name[41]="G4_CELLULOSE_NITRATE";
static const G4double e41[25]={1.05E+02,5.006E+01,3.437E+01,2.2E+01,1.002E+01,6.458,4.318,2.544,1.962,1.75,1.673,1.742,1.84,1.976,2.542,3.071,4.118,8.32,1.361E+01,2.29E+01,5.486E+01,1.082E+02,1.617E+02,2.151E+02,2.685E+02};
AddData(T0,e41,41);
name[42]="G4_CERIC_SULFATE";
static const G4double e42[25]={1.183E+02,5.565E+01,3.806E+01,2.427E+01,1.101E+01,7.08,4.727,2.779,2.14,1.914,1.83,1.90,2.001,2.143,2.741,3.30,4.402,8.824,1.438E+01,2.415E+01,5.777E+01,1.139E+02,1.701E+02,2.263E+02,2.825E+02};
AddData(T0,e42,42);
name[43]="G4_CESIUM_FLUORIDE";
static const G4double e43[25]={3.221E+01,2.299E+01,1.729E+01,1.192E+01,5.926,3.96,2.73,1.679,1.335,1.232,1.262,1.484,1.755,2.171,4.222,6.304,1.053E+01,2.776E+01,4.947E+01,8.763E+01,2.189E+02,4.379E+02,6.571E+02,8.763E+02,1.095E+03};
AddData(T0,e43,43);
name[44]="G4_CESIUM_IODIDE";
static const G4double e44[25]={2.435E+01,2.019E+01,1.555E+01,1.092E+01,5.531,3.723,2.581,1.60,1.281,1.189,1.232,1.474,1.767,2.218,4.451,6.723,1.134E+01,3.015E+01,5.386E+01,9.553E+01,2.388E+02,4.781E+02,7.174E+02,9.568E+02,1.196E+03};
AddData(T0,e44,44);
name[45]="G4_CHLOROBENZENE";
static const G4double e45[25]={1.042E+02,4.984E+01,3.425E+01,2.193E+01,1.0E+01,6.448,4.314,2.542,1.962,1.75,1.68,1.766,1.88,2.037,2.697,3.322,4.568,9.585,1.59E+01,2.701E+01,6.52E+01,1.29E+02,1.928E+02,2.567E+02,3.206E+02};
AddData(T0,e45,45);
name[46]="G4_CHLOROFORM";
static const G4double e46[25]={7.684E+01,3.981E+01,2.796E+01,1.825E+01,8.52,5.546,3.74,2.229,1.734,1.562,1.527,1.652,1.805,2.02,2.981,3.921,5.809,1.345E+01,2.308E+01,4.0E+01,9.82E+01,1.954E+02,2.926E+02,3.899E+02,4.872E+02};
AddData(T0,e46,46);
name[47]="G4_CONCRETE";
static const G4double e47[25]={8.517E+01,4.309E+01,3.008E+01,1.953E+01,9.056,5.878,3.955,2.349,1.82,1.632,1.58,1.678,1.801,1.973,2.733,3.463,4.922,1.081E+01,1.822E+01,3.124E+01,7.604E+01,1.508E+02,2.257E+02,3.006E+02,3.755E+02};
AddData(T0,e47,47);
name[48]="G4_CYCLOHEXANE";
static const G4double e48[25]={1.359E+02,6.204E+01,4.206E+01,2.661E+01,1.195E+01,7.65,5.089,2.977,2.284,2.026,1.922,1.983,2.075,2.198,2.685,3.13,4.003,7.495,1.188E+01,1.96E+01,4.615E+01,9.053E+01,1.35E+02,1.794E+02,2.239E+02};
AddData(T0,e48,48);
name[49]="G4_DICHLORODIETHYL_ETHER";
static const G4double e49[25]={9.862E+01,4.803E+01,3.317E+01,2.134E+01,9.79,6.325,4.24,2.506,1.938,1.732,1.67,1.767,1.894,2.072,2.837,3.571,5.037,1.095E+01,1.84E+01,3.149E+01,7.653E+01,1.518E+02,2.27E+02,3.023E+02,3.776E+02};
AddData(T0,e49,49);
name[50]="G4_DIETHYL_ETHER";
static const G4double e50[25]={1.322E+02,6.071E+01,4.122E+01,2.612E+01,1.175E+01,7.531,5.013,2.935,2.254,2.008,1.911,1.974,2.069,2.197,2.71,3.18,4.104,7.80,1.245E+01,2.061E+01,4.872E+01,9.567E+01,1.427E+02,1.897E+02,2.367E+02};
AddData(T0,e50,50);
name[51]="G4_DIMETHYL_SULFOXIDE";
static const G4double e51[25]={1.044E+02,5.055E+01,3.486E+01,2.239E+01,1.025E+01,6.619,4.434,2.618,2.023,1.81,1.743,1.837,1.96,2.131,2.858,3.55,4.929,1.048E+01,1.747E+01,2.976E+01,7.202E+01,1.426E+02,2.133E+02,2.839E+02,3.546E+02};
AddData(T0,e51,51);
name[52]="G4_ETHANE";
static const G4double e52[25]={1.528E+02,6.854E+01,4.621E+01,2.909E+01,1.298E+01,8.286,5.499,3.206,2.456,2.195,2.131,2.275,2.435,2.629,3.252,3.722,4.607,8.069,1.239E+01,1.995E+01,4.6E+01,8.953E+01,1.331E+02,1.767E+02,2.204E+02};
AddData(T0,e52,52);
name[53]="G4_ETHYL_ALCOHOL";
static const G4double e53[25]={1.296E+02,5.977E+01,4.064E+01,2.578E+01,1.162E+01,7.449,4.961,2.907,2.234,1.99,1.895,1.959,2.054,2.185,2.713,3.198,4.152,7.973,1.278E+01,2.122E+01,5.026E+01,9.879E+01,1.474E+02,1.96E+02,2.446E+02};
AddData(T0,e53,53);
name[54]="G4_ETHYL_CELLULOSE";
static const G4double e54[25]={1.208E+02,5.623E+01,3.833E+01,2.438E+01,1.102E+01,7.076,4.719,2.769,2.13,1.891,1.799,1.864,1.959,2.089,2.615,3.102,4.062,7.909,1.275E+01,2.125E+01,5.051E+01,9.939E+01,1.483E+02,1.973E+02,2.462E+02};
AddData(T0,e54,54);
name[55]="G4_ETHYLENE";
static const G4double e55[25]={1.406E+02,6.364E+01,4.302E+01,2.715E+01,1.215E+01,7.77,5.163,3.015,2.312,2.068,2.011,2.151,2.306,2.496,3.125,3.604,4.508,8.039,1.244E+01,2.016E+01,4.672E+01,9.11E+01,1.355E+02,1.8E+02,2.244E+02};
AddData(T0,e55,55);
name[56]="G4_EYE_LENS_ICRP";
static const G4double e56[25]={1.194E+02,5.59E+01,3.818E+01,2.432E+01,1.101E+01,7.077,4.722,2.774,2.135,1.903,1.815,1.882,1.98,2.116,2.682,3.207,4.245,8.403,1.363E+01,2.282E+01,5.444E+01,1.073E+02,1.601E+02,2.13E+02,2.659E+02};
AddData(T0,e56,56);
name[57]="G4_FERRIC_OXIDE";
static const G4double e57[25]={6.117E+01,3.427E+01,2.454E+01,1.628E+01,7.744,5.081,3.449,2.073,1.61,1.451,1.424,1.557,1.723,1.971,3.129,4.272,6.573,1.591E+01,2.769E+01,4.838E+01,1.195E+02,2.383E+02,3.572E+02,4.761E+02,5.95E+02};
AddData(T0,e57,57);
name[58]="G4_FERROBORIDE";
static const G4double e58[25]={5.472E+01,3.18E+01,2.296E+01,1.533E+01,7.352,4.839,3.293,1.98,1.539,1.39,1.372,1.513,1.688,1.949,3.182,4.404,6.872,1.69E+01,2.954E+01,5.175E+01,1.281E+02,2.557E+02,3.833E+02,5.109E+02,6.386E+02};
AddData(T0,e58,58);
name[59]="G4_FERROUS_OXIDE";
static const G4double e59[25]={5.472E+01,3.18E+01,2.296E+01,1.533E+01,7.352,4.839,3.293,1.98,1.539,1.39,1.372,1.513,1.688,1.949,3.182,4.404,6.872,1.69E+01,2.954E+01,5.175E+01,1.281E+02,2.557E+02,3.833E+02,5.109E+02,6.386E+02};
AddData(T0,e59,59);
name[60]="G4_FERROUS_SULFATE";
static const G4double e60[25]={1.186E+02,5.576E+01,3.813E+01,2.431E+01,1.103E+01,7.09,4.734,2.782,2.143,1.916,1.832,1.902,2.002,2.143,2.738,3.292,4.385,8.77,1.428E+01,2.397E+01,5.73E+01,1.13E+02,1.687E+02,2.244E+02,2.802E+02};
AddData(T0,e60,60);
name[61]="G4_FREON-12";
static const G4double e61[25]={7.915E+01,4.041E+01,2.827E+01,1.839E+01,8.551,5.557,3.742,2.226,1.729,1.563,1.527,1.638,1.775,1.968,2.817,3.636,5.275,1.19E+01,2.024E+01,3.49E+01,8.532E+01,1.695E+02,2.537E+02,3.38E+02,4.223E+02};
AddData(T0,e61,61);
name[62]="G4_FREON-12B2";
static const G4double e62[25]={4.963E+01,2.96E+01,2.149E+01,1.442E+01,6.952,4.587,3.128,1.893,1.489,1.363,1.369,1.538,1.739,2.036,3.447,4.859,7.709,1.925E+01,3.379E+01,5.933E+01,1.472E+02,2.938E+02,4.405E+02,5.872E+02,7.339E+02};
AddData(T0,e62,62);
name[63]="G4_FREON-13";
static const G4double e63[25]={8.357E+01,4.186E+01,2.914E+01,1.887E+01,8.728,5.659,3.804,2.257,1.749,1.58,1.543,1.644,1.766,1.939,2.689,3.407,4.836,1.06E+01,1.786E+01,3.061E+01,7.446E+01,1.477E+02,2.209E+02,2.942E+02,3.675E+02};
AddData(T0,e63,63);
name[64]="G4_FREON-13B1";
static const G4double e64[25]={6.146E+01,3.381E+01,2.41E+01,1.593E+01,7.549,4.946,3.353,2.013,1.575,1.434,1.424,1.567,1.739,1.99,3.165,4.331,6.675,1.616E+01,2.811E+01,4.909E+01,1.213E+02,2.418E+02,3.623E+02,4.829E+02,6.034E+02};
AddData(T0,e64,64);
name[65]="G4_FREON-13I1";
static const G4double e65[25]={4.76E+01,2.866E+01,2.085E+01,1.401E+01,6.771,4.472,3.054,1.853,1.462,1.341,1.351,1.537,1.762,2.104,3.756,5.415,8.775,2.243E+01,3.963E+01,6.986E+01,1.738E+02,3.473E+02,5.209E+02,6.946E+02,8.682E+02};
AddData(T0,e65,65);
name[66]="G4_GADOLINIUM_OXYSULFIDE";
static const G4double e66[25]={2.856E+01,2.18E+01,1.658E+01,1.153E+01,5.786,3.88,2.683,1.65,1.312,1.211,1.244,1.477,1.764,2.207,4.409,6.651,1.121E+01,2.979E+01,5.321E+01,9.436E+01,2.359E+02,4.721E+02,7.084E+02,9.448E+02,1.181E+03};
AddData(T0,e66,66);
name[67]="G4_GALLIUM_ARSENIDE";
static const G4double e67[25]={3.854E+01,2.572E+01,1.911E+01,1.305E+01,6.42,4.269,2.93,1.788,1.414,1.296,1.304,1.48,1.692,2.013,3.574,5.147,8.324,2.12E+01,3.743E+01,6.596E+01,1.64E+02,3.277E+02,4.915E+02,6.554E+02,8.192E+02};
AddData(T0,e67,67);
name[68]="G4_GEL_PHOTO_EMULSION";
static const G4double e68[25]={1.166E+02,5.469E+01,3.737E+01,2.382E+01,1.08E+01,6.939,4.631,2.721,2.095,1.861,1.773,1.839,1.936,2.069,2.614,3.119,4.119,8.125,1.317E+01,2.202E+01,5.249E+01,1.034E+02,1.543E+02,2.053E+02,2.563E+02};
AddData(T0,e68,68);
name[69]="G4_GLASS_LEAD";
 static const G4double e69[25]={2.63E+01,2.099E+01,1.608E+01,1.124E+01,5.673,3.814,2.643,1.641,1.316,1.222,1.266,1.514,1.818,2.287,4.632,7.029,1.191E+01,3.182E+01,5.69E+01,1.01E+02,2.525E+02,5.053E+02,7.583E+02,1.011E+03,1.264E+03};
AddData(T0,e69,69);
name[70]="G4_GLASS_PLATE";
static const G4double e70[25]={8.141E+01,4.168E+01,2.918E+01,1.899E+01,8.836,5.743,3.868,2.301,1.785,1.603,1.556,1.658,1.783,1.96,2.743,3.499,5.009,1.11E+01,1.878E+01,3.227E+01,7.867E+01,1.561E+02,2.337E+02,3.112E+02,3.888E+02};
AddData(T0,e70,70);
name[71]="G4_GLUCOSE";
static const G4double e71[25]={1.142E+02,5.376E+01,3.677E+01,2.345E+01,1.064E+01,6.844,4.569,2.686,2.066,1.834,1.746,1.812,1.909,2.042,2.595,3.11,4.129,8.215,1.336E+01,2.239E+01,5.346E+01,1.054E+02,1.573E+02,2.093E+02,2.613E+02};
AddData(T0,e71,71);
name[72]="G4_GLUTAMINE";
static const G4double e72[25]={1.161E+02,5.437E+01,3.713E+01,2.365E+01,1.071E+01,6.883,4.593,2.698,2.073,1.838,1.748,1.813,1.908,2.039,2.576,3.076,4.064,8.027,1.301E+01,2.177E+01,5.191E+01,1.023E+02,1.527E+02,2.031E+02,2.535E+02};
AddData(T0,e72,72);
name[73]="G4_GLYCEROL";
static const G4double e73[25]={1.185E+02,5.544E+01,3.785E+01,2.41E+01,1.091E+01,7.012,4.678,2.747,2.114,1.878,1.788,1.853,1.949,2.082,2.63,3.139,4.143,8.171,1.324E+01,2.214E+01,5.276E+01,1.039E+02,1.551E+02,2.063E+02,2.576E+02};
AddData(T0,e73,73);
name[74]="G4_GRAPHITE";
static const G4double e74[25]={1.062E+02,5.006E+01,3.425E+01,2.185E+01,9.916,6.375,4.252,2.489,1.901,1.687,1.609,1.677,1.771,1.896,2.396,2.857,3.768,7.426,1.203E+01,2.012E+01,4.795E+01,9.444E+01,1.41E+02,1.875E+02,2.341E+02};
AddData(T0,e74,74);
name[75]="G4_GUANINE";
static const G4double e75[25]={1.114E+02,5.226E+01,3.572E+01,2.276E+01,1.032E+01,6.633,4.427,2.601,1.997,1.77,1.685,1.75,1.844,1.974,2.503,2.997,3.975,7.904,1.285E+01,2.153E+01,5.142E+01,1.013E+02,1.513E+02,2.013E+02,2.513E+02};
AddData(T0,e75,75);
name[76]="G4_GYPSUM";
static const G4double e76[25]={8.826E+01,4.437E+01,3.092E+01,2.004E+01,9.279,6.019,4.047,2.402,1.859,1.663,1.607,1.707,1.837,2.021,2.838,3.626,5.204,1.158E+01,1.961E+01,3.371E+01,8.225E+01,1.633E+02,2.444E+02,3.255E+02,4.067E+02};
AddData(T0,e76,76);
name[77]="G4_KAPTON";
static const G4double e77[25]={1.082E+02,5.111E+01,3.499E+01,2.234E+01,1.015E+01,6.528,4.36,2.564,1.974,1.755,1.674,1.741,1.836,1.964,2.486,2.971,3.931,7.781,1.263E+01,2.114E+01,5.043E+01,9.935E+01,1.483E+02,1.973E+02,2.463E+02};
AddData(T0,e77,77);
name[78]="G4_LANTHANUM_OXYBROMIDE";
static const G4double e78[25]={3.263E+01,2.326E+01,1.749E+01,1.206E+01,5.993,4.003,2.758,1.695,1.344,1.237,1.263,1.476,1.735,2.132,4.095,6.088,1.014E+01,2.661E+01,4.739E+01,8.389E+01,2.095E+02,4.191E+02,6.287E+02,8.385E+02,1.048E+03};
AddData(T0,e78,78);
name[79]="G4_LANTHANUM_OXYSULFIDE";
static const G4double e79[25]={3.417E+01,2.381E+01,1.783E+01,1.225E+01,6.072,4.051,2.789,1.711,1.355,1.247,1.273,1.491,1.757,2.165,4.181,6.23,1.039E+01,2.736E+01,4.875E+01,8.633E+01,2.156E+02,4.314E+02,6.474E+02,8.633E+02,1.079E+03};
AddData(T0,e79,79);
name[80]="G4_LEAD_OXIDE";
 static const G4double e80[25]={1.325E+01,1.611E+01,1.299E+01,9.413,4.927,3.359,2.354,1.485,1.206,1.134,1.20,1.482,1.827,2.364,5.076,7.859,1.354E+01,3.671E+01,6.589E+01,1.172E+02,2.935E+02,5.877E+02,8.822E+02,1.177E+03,1.471E+03};
AddData(T0,e80,80);
name[81]="G4_LITHIUM_AMIDE";
static const G4double e81[25]={1.251E+02,5.707E+01,3.867E+01,2.446E+01,1.098E+01,7.026,4.673,2.733,2.085,1.845,1.75,1.804,1.887,1.999,2.454,2.874,3.702,7.012,1.117E+01,1.848E+01,4.364E+01,8.568E+01,1.278E+02,1.699E+02,2.12E+02};
AddData(T0,e81,81);
name[82]="G4_LITHIUM_CARBONATE";
static const G4double e82[25]={9.905E+01,4.73E+01,3.249E+01,2.08E+01,9.48,6.109,4.085,2.407,1.849,1.645,1.573,1.638,1.728,1.854,2.379,2.871,3.847,7.764,1.269E+01,2.135E+01,5.113E+01,1.009E+02,1.506E+02,2.004E+02,2.502E+02};
AddData(T0,e82,82);
name[83]="G4_LITHIUM_FLUORIDE";
static const G4double e83[25]={9.161E+01,4.409E+01,3.035E+01,1.947E+01,8.898,5.739,3.842,2.266,1.743,1.555,1.489,1.552,1.639,1.761,2.289,2.788,3.778,7.753,1.275E+01,2.154E+01,5.174E+01,1.022E+02,1.526E+02,2.031E+02,2.536E+02};
AddData(T0,e83,83);
name[84]="G4_LITHIUM_HYDRIDE";
static const G4double e84[25]={1.371E+02,6.052E+01,4.059E+01,2.544E+01,1.128E+01,7.181,4.755,2.759,2.093,1.847,1.747,1.784,1.841,1.917,2.209,2.469,2.974,4.963,7.454,1.183E+01,2.69E+01,5.211E+01,7.736E+01,1.026E+02,1.279E+02};
AddData(T0,e84,84);
name[85]="G4_LITHIUM_IODIDE";
static const G4double e85[25]={2.882E+01,2.177E+01,1.653E+01,1.148E+01,5.751,3.854,2.663,1.643,1.311,1.214,1.251,1.482,1.761,2.189,4.302,6.446,1.08E+01,2.854E+01,5.089E+01,9.018E+01,2.253E+02,4.508E+02,6.764E+02,9.021E+02,1.128E+03};
AddData(T0,e85,85);
name[86]="G4_LITHIUM_OXIDE";
static const G4double e86[25]={1.018E+02,4.768E+01,3.256E+01,2.074E+01,9.397,6.038,4.029,2.365,1.809,1.607,1.533,1.59,1.668,1.776,2.227,2.648,3.48,6.812,1.1E+01,1.836E+01,4.367E+01,8.595E+01,1.283E+02,1.706E+02,2.129E+02};
AddData(T0,e86,86);
name[87]="G4_LITHIUM_TETRABORATE";
static const G4double e87[25]={9.577E+01,4.613E+01,3.176E+01,2.038E+01,9.315,6.009,4.022,2.373,1.824,1.623,1.552,1.619,1.71,1.836,2.362,2.856,3.836,7.769,1.272E+01,2.141E+01,5.132E+01,1.013E+02,1.513E+02,2.013E+02,2.513E+02};
AddData(T0,e87,87);
name[88]="G4_lPROPANE";
static const G4double e88[25]={1.442E+02,6.54E+01,4.424E+01,2.794E+01,1.251E+01,8.003,5.319,3.107,2.383,2.129,2.029,2.091,2.184,2.31,2.799,3.24,4.104,7.551,1.188E+01,1.949E+01,4.57E+01,8.95E+01,1.334E+02,1.772E+02,2.211E+02};
AddData(T0,e88,88);
name[89]="G4_LUNG_ICRP";
static const G4double e89[25]={1.184E+02,5.56E+01,3.8E+01,2.422E+01,1.098E+01,7.06,4.712,2.769,2.132,1.904,1.819,1.888,1.987,2.126,2.707,3.248,4.317,8.599,1.399E+01,2.345E+01,5.601E+01,1.104E+02,1.648E+02,2.193E+02,2.737E+02};
AddData(T0,e89,89);
name[90]="G4_M3_WAX";
static const G4double e90[25]={1.241E+02,5.767E+01,3.929E+01,2.498E+01,1.128E+01,7.243,4.828,2.833,2.177,1.933,1.841,1.909,2.007,2.139,2.68,3.182,4.173,8.145,1.314E+01,2.192E+01,5.212E+01,1.026E+02,1.531E+02,2.036E+02,2.542E+02};
AddData(T0,e90,90);
name[91]="G4_MAGNESIUM_CARBONATE";
static const G4double e91[25]={8.973E+01,4.449E+01,3.088E+01,1.995E+01,9.20,5.957,4.00,2.369,1.828,1.632,1.57,1.65,1.756,1.904,2.547,3.16,4.379,9.288,1.547E+01,2.633E+01,6.367E+01,1.26E+02,1.884E+02,2.509E+02,3.133E+02};
AddData(T0,e91,91);
name[92]="G4_MAGNESIUM_FLUORIDE";
static const G4double e92[25]={8.183E+01,4.136E+01,2.886E+01,1.873E+01,8.684,5.636,3.791,2.251,1.745,1.566,1.516,1.601,1.709,1.865,2.553,3.214,4.531,9.839,1.652E+01,2.826E+01,6.863E+01,1.36E+02,2.035E+02,2.71E+02,3.384E+02};
AddData(T0,e92,92);
name[93]="G4_MAGNESIUM_OXIDE";
static const G4double e93[25]={8.166E+01,4.173E+01,2.92E+01,1.9E+01,8.835,5.741,3.866,2.298,1.779,1.595,1.545,1.636,1.75,1.911,2.629,3.321,4.699,1.026E+01,1.726E+01,2.956E+01,7.185E+01,1.425E+02,2.131E+02,2.838E+02,3.545E+02};
AddData(T0,e93,93);
name[94]="G4_MAGNESIUM_TETRABORATE";
static const G4double e94[25]={9.159E+01,4.488E+01,3.105E+01,2.001E+01,9.193,5.944,3.986,2.357,1.819,1.622,1.557,1.631,1.731,1.868,2.45,3.001,4.095,8.493,1.403E+01,2.375E+01,5.721E+01,1.131E+02,1.69E+02,2.249E+02,2.808E+02};
AddData(T0,e94,94);
name[95]="G4_MERCURIC_IODIDE";
 static const G4double e95[25]={1.711E+01,1.758E+01,1.393E+01,9.97,5.156,3.498,2.441,1.528,1.234,1.154,1.21,1.475,1.796,2.295,4.787,7.334,1.252E+01,3.366E+01,6.03E+01,1.071E+02,2.681E+02,5.367E+02,8.055E+02,1.074E+03,1.343E+03};
AddData(T0,e95,95);
name[96]="G4_METHANE";
static const G4double e96[25]={1.633E+02,7.277E+01,4.896E+01,3.077E+01,1.369E+01,8.732,5.79,3.372,2.58,2.305,2.235,2.383,2.547,2.745,3.394,3.868,4.744,8.15,1.239E+01,1.983E+01,4.544E+01,8.823E+01,1.311E+02,1.74E+02,2.168E+02};
AddData(T0,e96,96);
name[97]="G4_METHANOL";
static const G4double e97[25]={1.258E+02,5.842E+01,3.98E+01,2.53E+01,1.143E+01,7.334,4.889,2.868,2.206,1.971,1.881,1.948,2.045,2.178,2.724,3.227,4.218,8.182,1.317E+01,2.192E+01,5.206E+01,1.024E+02,1.528E+02,2.032E+02,2.536E+02};
AddData(T0,e97,97);
name[98]="G4_MIX_D_WAX";
static const G4double e98[25]={1.311E+02,6.029E+01,4.096E+01,2.596E+01,1.169E+01,7.49,4.987,2.92,2.24,1.985,1.884,1.947,2.042,2.169,2.681,3.153,4.082,7.803,1.248E+01,2.071E+01,4.901E+01,9.63E+01,1.436E+02,1.91E+02,2.384E+02};
AddData(T0,e98,98);
name[99]="G4_MS20_TISSUE";
static const G4double e99[25]={1.162E+02,5.455E+01,3.728E+01,2.376E+01,1.077E+01,6.924,4.621,2.715,2.091,1.864,1.782,1.854,1.953,2.088,2.642,3.157,4.174,8.252,1.338E+01,2.24E+01,5.342E+01,1.052E+02,1.571E+02,2.09E+02,2.609E+02};
AddData(T0,e99,99);
name[100]="G4_MUSCLE_SKELETAL_ICRP";
static const G4double e100[25]={1.184E+02,5.557E+01,3.798E+01,2.421E+01,1.098E+01,7.056,4.71,2.768,2.131,1.904,1.818,1.887,1.987,2.125,2.706,3.247,4.313,8.59,1.397E+01,2.342E+01,5.593E+01,1.102E+02,1.646E+02,2.19E+02,2.734E+02};
AddData(T0,e100,100);
name[101]="G4_MUSCLE_STRIATED_ICRU";
static const G4double e101[25]={1.189E+02,5.576E+01,3.81E+01,2.428E+01,1.1E+01,7.073,4.721,2.774,2.136,1.907,1.821,1.89,1.989,2.127,2.706,3.245,4.308,8.571,1.393E+01,2.335E+01,5.576E+01,1.099E+02,1.641E+02,2.183E+02,2.725E+02};
AddData(T0,e101,101);
name[102]="G4_MUSCLE_WITHOUT_SUCROSE";
static const G4double e102[25]={1.192E+02,5.586E+01,3.816E+01,2.432E+01,1.102E+01,7.082,4.726,2.776,2.138,1.908,1.821,1.888,1.987,2.124,2.699,3.233,4.288,8.515,1.383E+01,2.317E+01,5.531E+01,1.09E+02,1.627E+02,2.165E+02,2.702E+02};
AddData(T0,e102,102);
name[103]="G4_MUSCLE_WITH_SUCROSE";
static const G4double e103[25]={1.187E+02,5.566E+01,3.802E+01,2.423E+01,1.098E+01,7.056,4.709,2.767,2.13,1.90,1.812,1.879,1.978,2.115,2.686,3.217,4.265,8.467,1.375E+01,2.304E+01,5.499E+01,1.084E+02,1.618E+02,2.152E+02,2.687E+02};
AddData(T0,e103,103);
name[104]="G4_MYLAR";
static const G4double e104[25]={1.103E+02,5.203E+01,3.561E+01,2.273E+01,1.032E+01,6.638,4.433,2.607,2.007,1.784,1.701,1.768,1.863,1.993,2.524,3.017,3.991,7.902,1.282E+01,2.147E+01,5.121E+01,1.009E+02,1.506E+02,2.003E+02,2.501E+02};
AddData(T0,e104,104);
name[105]="G4_NAPHTHALENE";
static const G4double e105[25]={1.183E+02,5.501E+01,3.749E+01,2.384E+01,1.077E+01,6.914,4.61,2.704,2.077,1.843,1.754,1.817,1.91,2.034,2.526,2.979,3.874,7.459,1.197E+01,1.989E+01,4.717E+01,9.273E+01,1.383E+02,1.84E+02,2.296E+02};
AddData(T0,e105,105);
name[106]="G4_N-BUTYL_ALCOHOL";
static const G4double e106[25]={1.323E+02,6.074E+01,4.124E+01,2.613E+01,1.176E+01,7.533,5.015,2.936,2.255,2.004,1.904,1.966,2.06,2.188,2.70,3.169,4.093,7.789,1.244E+01,2.06E+01,4.87E+01,9.566E+01,1.427E+02,1.897E+02,2.367E+02};
AddData(T0,e106,106);
name[107]="G4_N-HEPTANE";
static const G4double e107[25]={1.395E+02,6.352E+01,4.301E+01,2.719E+01,1.22E+01,7.805,5.19,3.034,2.328,2.067,1.962,2.021,2.114,2.237,2.724,3.167,4.036,7.508,1.187E+01,1.954E+01,4.594E+01,9.007E+01,1.342E+02,1.784E+02,2.226E+02};
AddData(T0,e107,107);
name[108]="G4_N-HEXANE";
static const G4double e108[25]={1.402E+02,6.378E+01,4.319E+01,2.73E+01,1.224E+01,7.832,5.207,3.044,2.335,2.075,1.969,2.029,2.121,2.245,2.732,3.174,4.042,7.511,1.187E+01,1.953E+01,4.591E+01,8.999E+01,1.341E+02,1.783E+02,2.224E+02};
AddData(T0,e108,108);
name[109]="G4_NITROBENZENE";
static const G4double e109[25]={1.117E+02,5.25E+01,3.589E+01,2.288E+01,1.037E+01,6.67,4.452,2.616,2.015,1.792,1.71,1.777,1.872,2.002,2.532,3.024,3.995,7.893,1.28E+01,2.141E+01,5.106E+01,1.006E+02,1.501E+02,1.997E+02,2.493E+02};
AddData(T0,e109,109);
name[110]="G4_NITROUS_OXIDE";
static const G4double e110[25]={1.03E+02,4.898E+01,3.361E+01,2.149E+01,9.785,6.302,4.213,2.481,1.913,1.719,1.684,1.821,1.972,2.163,2.867,3.442,4.538,8.86,1.427E+01,2.374E+01,5.63E+01,1.107E+02,1.651E+02,2.195E+02,2.739E+02};
AddData(T0,e110,110);
name[111]="G4_N,N-DIMETHYL_FORMAMIDE";
static const G4double e111[25]={1.232E+02,5.713E+01,3.89E+01,2.472E+01,1.116E+01,7.161,4.773,2.799,2.152,1.915,1.823,1.887,1.982,2.112,2.638,3.123,4.08,7.912,1.273E+01,2.12E+01,5.035E+01,9.904E+01,1.478E+02,1.965E+02,2.453E+02};
AddData(T0,e111,111);
name[112]="G4_N-PENTANE";
static const G4double e112[25]={1.41E+02,6.41E+01,4.34E+01,2.742E+01,1.23E+01,7.867,5.23,3.057,2.345,2.085,1.979,2.039,2.131,2.256,2.742,3.184,4.052,7.516,1.187E+01,1.952E+01,4.586E+01,8.989E+01,1.34E+02,1.781E+02,2.222E+02};
AddData(T0,e112,112);
name[113]="G4_N-PROPYL_ALCOHOL";
static const G4double e113[25]={1.312E+02,6.035E+01,4.1E+01,2.599E+01,1.17E+01,7.499,4.993,2.924,2.246,1.998,1.90,1.963,2.058,2.186,2.705,3.18,4.116,7.859,1.257E+01,2.084E+01,4.93E+01,9.686E+01,1.445E+02,1.921E+02,2.397E+02};
AddData(T0,e113,113);
name[114]="G4_NYLON-11_RILSAN";
static const G4double e114[25]={1.287E+02,5.924E+01,4.025E+01,2.552E+01,1.149E+01,7.367,4.906,2.873,2.193,1.937,1.835,1.895,1.987,2.111,2.609,3.069,3.978,7.616,1.219E+01,2.023E+01,4.79E+01,9.414E+01,1.404E+02,1.867E+02,2.33E+02};
AddData(T0,e114,114);
name[115]="G4_NYLON-6-10";
static const G4double e115[25]={1.266E+02,5.843E+01,3.973E+01,2.521E+01,1.136E+01,7.285,4.852,2.843,2.181,1.931,1.833,1.894,1.987,2.113,2.619,3.085,4.005,7.687,1.232E+01,2.045E+01,4.846E+01,9.525E+01,1.421E+02,1.889E+02,2.358E+02};
AddData(T0,e115,115);
name[116]="G4_NYLON-6-6";
static const G4double e116[25]={1.251E+02,5.78E+01,3.931E+01,2.495E+01,1.125E+01,7.214,4.806,2.817,2.161,1.914,1.817,1.879,1.973,2.099,2.61,3.082,4.013,7.742,1.243E+01,2.067E+01,4.904E+01,9.643E+01,1.439E+02,1.913E+02,2.388E+02};
AddData(T0,e116,116);
name[117]="G4_NYLON-8062";
static const G4double e117[25]={1.255E+02,5.799E+01,3.945E+01,2.504E+01,1.129E+01,7.243,4.826,2.828,2.172,1.925,1.829,1.891,1.985,2.112,2.623,3.093,4.022,7.74,1.242E+01,2.063E+01,4.891E+01,9.616E+01,1.435E+02,1.908E+02,2.381E+02};
AddData(T0,e117,117);
name[118]="G4_OCTANE";
static const G4double e118[25]={1.39E+02,6.332E+01,4.289E+01,2.711E+01,1.217E+01,7.785,5.177,3.027,2.322,2.061,1.956,2.016,2.108,2.232,2.718,3.161,4.031,7.505,1.187E+01,1.955E+01,4.597E+01,9.012E+01,1.343E+02,1.786E+02,2.228E+02};
AddData(T0,e118,118);
name[119]="G4_PARAFFIN";
static const G4double e119[25]={1.368E+02,6.244E+01,4.231E+01,2.677E+01,1.202E+01,7.692,5.116,2.992,2.292,2.028,1.921,1.98,2.071,2.194,2.678,3.122,3.994,7.48,1.186E+01,1.956E+01,4.608E+01,9.038E+01,1.347E+02,1.791E+02,2.235E+02};
AddData(T0,e119,119);
name[120]="G4_PHOTO_EMULSION";
static const G4double e120[25]={4.498E+01,2.823E+01,2.072E+01,1.402E+01,6.826,4.522,3.095,1.883,1.484,1.354,1.36,1.547,1.777,2.123,3.80,5.488,8.906,2.278E+01,4.027E+01,7.099E+01,1.766E+02,3.53E+02,5.295E+02,7.06E+02,8.824E+02};
AddData(T0,e120,120);
name[121]="G4_PLASTIC_SC_VINYLTOLUENE";
static const G4double e121[25]={1.231E+02,5.693E+01,3.874E+01,2.459E+01,1.109E+01,7.115,4.741,2.779,2.134,1.893,1.799,1.862,1.954,2.078,2.569,3.02,3.909,7.468,1.195E+01,1.981E+01,4.689E+01,9.213E+01,1.374E+02,1.827E+02,2.28E+02};
AddData(T0,e121,121);
name[122]="G4_PLEXIGLASS";
static const G4double e122[25]={1.17E+02,5.481E+01,3.744E+01,2.385E+01,1.081E+01,6.946,4.635,2.723,2.096,1.864,1.776,1.842,1.937,2.067,2.595,3.082,4.044,7.897,1.274E+01,2.126E+01,5.057E+01,9.953E+01,1.485E+02,1.976E+02,2.466E+02};
AddData(T0,e122,122);
name[123]="G4_PLUTONIUM_DIOXIDE";
static const G4double e123[25]={1.419E+01,1.65E+01,1.325E+01,9.571,4.996,3.404,2.386,1.50,1.217,1.145,1.215,1.507,1.865,2.426,5.261,8.171,1.411E+01,3.84E+01,6.905E+01,1.229E+02,3.079E+02,6.168E+02,9.257E+02,1.235E+03,1.543E+03};
AddData(T0,e123,123);
name[124]="G4_POLYACRYLONITRILE";
static const G4double e124[25]={1.17E+02,5.447E+01,3.714E+01,2.363E+01,1.068E+01,6.859,4.574,2.684,2.063,1.831,1.742,1.807,1.90,2.026,2.534,3.002,3.929,7.642,1.231E+01,2.052E+01,4.877E+01,9.597E+01,1.432E+02,1.905E+02,2.377E+02};
AddData(T0,e124,124);
name[125]="G4_POLYCARBONATE";
static const G4double e125[25]={1.148E+02,5.372E+01,3.668E+01,2.336E+01,1.058E+01,6.799,4.536,2.664,2.05,1.821,1.735,1.801,1.895,2.022,2.537,3.012,3.951,7.715,1.245E+01,2.077E+01,4.941E+01,9.724E+01,1.451E+02,1.93E+02,2.409E+02};
AddData(T0,e125,125);
name[126]="G4_POLYCHLOROSTYRENE";
static const G4double e126[25]={1.098E+02,5.2E+01,3.563E+01,2.276E+01,1.035E+01,6.661,4.451,2.619,2.015,1.792,1.713,1.792,1.90,2.048,2.666,3.249,4.41,9.08,1.496E+01,2.529E+01,6.083E+01,1.202E+02,1.796E+02,2.391E+02,2.985E+02};
AddData(T0,e126,126);
name[127]="G4_POLYETHYLENE";
static const G4double e127[25]={1.351E+02,6.178E+01,4.19E+01,2.652E+01,1.192E+01,7.63,5.077,2.97,2.277,2.015,1.91,1.97,2.061,2.184,2.67,3.114,3.987,7.479,1.187E+01,1.958E+01,4.614E+01,9.051E+01,1.349E+02,1.794E+02,2.239E+02};
AddData(T0,e127,127);
name[128]="G4_POLYOXYMETHYLENE";
static const G4double e128[25]={1.137E+02,5.352E+01,3.661E+01,2.335E+01,1.06E+01,6.815,4.55,2.675,2.059,1.83,1.743,1.81,1.907,2.04,2.59,3.102,4.115,8.177,1.329E+01,2.226E+01,5.315E+01,1.047E+02,1.564E+02,2.08E+02,2.597E+02};
AddData(T0,e128,128);
name[129]="G4_POLYPROPYLENE";
static const G4double e129[25]={1.358E+02,6.201E+01,4.204E+01,2.66E+01,1.195E+01,7.648,5.088,2.976,2.281,2.019,1.914,1.973,2.065,2.188,2.673,3.117,3.991,7.483,1.187E+01,1.959E+01,4.614E+01,9.052E+01,1.349E+02,1.794E+02,2.239E+02};
AddData(T0,e129,129);
name[130]="G4_POLYSTYRENE";
static const G4double e130[25]={1.197E+02,5.569E+01,3.796E+01,2.414E+01,1.091E+01,7.003,4.669,2.739,2.106,1.871,1.781,1.845,1.938,2.062,2.554,3.006,3.897,7.466,1.195E+01,1.984E+01,4.699E+01,9.234E+01,1.377E+02,1.832E+02,2.286E+02};
AddData(T0,e130,130);
name[131]="G4_POLYTRIFLUOROCHLOROETHYLENE";
static const G4double e131[25]={8.575E+01,4.265E+01,2.963E+01,1.916E+01,8.844,5.729,3.848,2.281,1.767,1.582,1.526,1.616,1.732,1.896,2.612,3.30,4.675,1.022E+01,1.72E+01,2.947E+01,7.167E+01,1.421E+02,2.126E+02,2.832E+02,3.537E+02};
AddData(T0,e131,131);
name[132]="G4_POLYVINYL_ACETATE";
static const G4double e132[25]={1.16E+02,5.435E+01,3.713E+01,2.365E+01,1.071E+01,6.885,4.594,2.699,2.077,1.847,1.76,1.826,1.922,2.053,2.586,3.08,4.056,7.969,1.289E+01,2.154E+01,5.129E+01,1.01E+02,1.507E+02,2.005E+02,2.503E+02};
AddData(T0,e132,132);
name[133]="G4_POLYVINYL_ALCOHOL";
static const G4double e133[25]={1.207E+02,5.622E+01,3.834E+01,2.439E+01,1.103E+01,7.08,4.721,2.771,2.128,1.887,1.794,1.858,1.953,2.083,2.612,3.102,4.07,7.951,1.283E+01,2.141E+01,5.091E+01,1.002E+02,1.495E+02,1.989E+02,2.483E+02};
AddData(T0,e133,133);
name[134]="G4_POLYVINYL_BUTYRAL";
static const G4double e134[25]={1.224E+02,5.68E+01,3.869E+01,2.459E+01,1.11E+01,7.127,4.75,2.786,2.141,1.90,1.806,1.87,1.964,2.091,2.606,3.082,4.019,7.775,1.25E+01,2.08E+01,4.936E+01,9.707E+01,1.448E+02,1.926E+02,2.404E+02};
AddData(T0,e134,134);
name[135]="G4_POLYVINYL_CHLORIDE";
static const G4double e135[25]={9.572E+01,4.69E+01,3.245E+01,2.091E+01,9.607,6.212,4.167,2.465,1.906,1.704,1.645,1.746,1.877,2.06,2.849,3.61,5.132,1.127E+01,1.901E+01,3.261E+01,7.94E+01,1.575E+02,2.357E+02,3.139E+02,3.922E+02};
AddData(T0,e135,135);
name[136]="G4_POLYVINYLIDENE_CHLORIDE";
static const G4double e136[25]={8.415E+01,4.253E+01,2.967E+01,1.926E+01,8.931,5.797,3.901,2.317,1.796,1.61,1.563,1.676,1.818,2.015,2.89,3.743,5.451,1.236E+01,2.107E+01,3.637E+01,8.901E+01,1.769E+02,2.649E+02,3.529E+02,4.409E+02};
AddData(T0,e136,136);
name[137]="G4_POLYVINYLIDENE_FLUORIDE";
static const G4double e137[25]={1.012E+02,4.838E+01,3.324E+01,2.129E+01,9.707,6.256,4.184,2.466,1.902,1.695,1.621,1.69,1.786,1.921,2.489,3.024,4.083,8.34,1.37E+01,2.31E+01,5.547E+01,1.095E+02,1.636E+02,2.177E+02,2.718E+02};
AddData(T0,e137,137);
name[138]="G4_POLYVINYL_PYRROLIDONE";
static const G4double e138[25]={1.208E+02,5.612E+01,3.824E+01,2.43E+01,1.098E+01,7.047,4.697,2.755,2.115,1.875,1.781,1.844,1.938,2.065,2.578,3.052,3.988,7.739,1.246E+01,2.075E+01,4.929E+01,9.696E+01,1.447E+02,1.924E+02,2.402E+02};
AddData(T0,e138,138);
name[139]="G4_POTASSIUM_IODIDE";
static const G4double e139[25]={1.208E+02,5.612E+01,3.824E+01,2.43E+01,1.098E+01,7.047,4.697,2.755,2.115,1.875,1.781,1.844,1.938,2.065,2.578,3.052,3.988,7.739,1.246E+01,2.075E+01,4.929E+01,9.696E+01,1.447E+02,1.924E+02,2.402E+02};
AddData(T0,e139,139);
name[140]="G4_POTASSIUM_OXIDE";
static const G4double e140[25]={6.968E+01,3.749E+01,2.658E+01,1.748E+01,8.241,5.385,3.644,2.181,1.698,1.533,1.502,1.631,1.791,2.022,3.063,4.084,6.136,1.445E+01,2.494E+01,4.337E+01,1.068E+02,2.126E+02,3.185E+02,4.245E+02,5.305E+02};
AddData(T0,e140,140);
name[141]="G4_PROPANE";
static const G4double e141[25]={1.488E+02,6.694E+01,4.517E+01,2.846E+01,1.271E+01,8.118,5.39,3.144,2.409,2.154,2.092,2.235,2.393,2.586,3.19,3.657,4.546,8.027,1.237E+01,1.998E+01,4.619E+01,8.999E+01,1.338E+02,1.777E+02,2.216E+02};
AddData(T0,e141,141);
name[142]="G4_Pyrex_Glass";
static const G4double e142[25]={8.456E+01,4.273E+01,2.981E+01,1.934E+01,8.968,5.82,3.915,2.325,1.802,1.616,1.564,1.66,1.778,1.943,2.662,3.351,4.727,1.027E+01,1.725E+01,2.952E+01,7.172E+01,1.422E+02,2.127E+02,2.832E+02,3.538E+02};
AddData(T0,e142,142);
name[143]="G4_PYRIDINE";
static const G4double e143[25]={1.198E+02,5.551E+01,3.78E+01,2.401E+01,1.084E+01,6.954,4.635,2.718,2.089,1.855,1.766,1.83,1.923,2.049,2.552,3.015,3.93,7.596,1.221E+01,2.031E+01,4.82E+01,9.478E+01,1.414E+02,1.881E+02,2.347E+02};
AddData(T0,e143,143);
name[144]="G4_RUBBER_BUTYL";
static const G4double e144[25]={1.358E+02,6.201E+01,4.204E+01,2.66E+01,1.195E+01,7.648,5.088,2.976,2.28,2.018,1.913,1.972,2.063,2.186,2.672,3.116,3.989,7.481,1.187E+01,1.958E+01,4.614E+01,9.052E+01,1.349E+02,1.794E+02,2.239E+02};
AddData(T0,e144,144);
name[145]="G4_RUBBER_NATURAL";
static const G4double e145[25]={1.303E+02,5.982E+01,4.062E+01,2.573E+01,1.158E+01,7.418,4.938,2.891,2.218,1.966,1.866,1.927,2.019,2.143,2.631,3.078,3.958,7.479,1.191E+01,1.969E+01,4.647E+01,9.122E+01,1.36E+02,1.808E+02,2.257E+02};
AddData(T0,e145,145);
name[146]="G4_RUBBER_NEOPRENE";
static const G4double e146[25]={1.033E+02,4.966E+01,3.418E+01,2.192E+01,1.001E+01,6.458,4.323,2.55,1.968,1.754,1.685,1.775,1.895,2.06,2.762,3.432,4.768,1.015E+01,1.694E+01,2.886E+01,6.987E+01,1.384E+02,2.069E+02,2.755E+02,3.441E+02};
AddData(T0,e146,146);
name[147]="G4_SILICON_DIOXIDE";
static const G4double e147[25]={8.344E+01,4.242E+01,2.964E+01,1.926E+01,8.944,5.809,3.91,2.323,1.801,1.616,1.566,1.665,1.787,1.957,2.703,3.421,4.853,1.063E+01,1.79E+01,3.069E+01,7.466E+01,1.481E+02,2.215E+02,2.95E+02,3.685E+02};
AddData(T0,e147,147);
name[148]="G4_SILVER_BROMIDE";
static const G4double e148[25]={2.998E+01,2.268E+01,1.723E+01,1.197E+01,5.995,4.016,2.773,1.707,1.355,1.247,1.272,1.484,1.739,2.129,4.037,5.97,9.891,2.582E+01,4.59E+01,8.118E+01,2.025E+02,4.05E+02,6.077E+02,8.104E+02,1.013E+03};
AddData(T0,e148,148);
name[149]="G4_SILVER_CHLORIDE";
static const G4double e149[25]={3.768E+01,2.555E+01,1.904E+01,1.303E+01,6.43,4.282,2.942,1.80,1.419,1.299,1.315,1.518,1.766,2.142,3.968,5.814,9.559,2.478E+01,4.397E+01,7.768E+01,1.936E+02,3.871E+02,5.808E+02,7.745E+02,9.681E+02};
AddData(T0,e149,149);
name[150]="G4_SILVER_HALIDES";
static const G4double e150[25]={2.994E+01,2.267E+01,1.722E+01,1.196E+01,5.992,4.015,2.772,1.706,1.354,1.247,1.272,1.484,1.739,2.129,4.039,5.974,9.898,2.584E+01,4.594E+01,8.125E+01,2.027E+02,4.054E+02,6.082E+02,8.111E+02,1.014E+03};
AddData(T0,e150,150);
name[151]="G4_SILVER_IODIDE";
static const G4double e151[25]={2.553E+01,2.088E+01,1.605E+01,1.125E+01,5.691,3.828,2.652,1.641,1.308,1.21,1.247,1.481,1.765,2.202,4.358,6.549,1.1E+01,2.912E+01,5.195E+01,9.208E+01,2.301E+02,4.605E+02,6.909E+02,9.215E+02,1.152E+03};
AddData(T0,e151,151);
name[152]="G4_SKIN_ICRP";
static const G4double e152[25]={1.199E+02,5.608E+01,3.829E+01,2.438E+01,1.104E+01,7.093,4.733,2.779,2.139,1.906,1.817,1.884,1.982,2.118,2.682,3.206,4.241,8.39,1.361E+01,2.277E+01,5.432E+01,1.07E+02,1.597E+02,2.125E+02,2.653E+02};
AddData(T0,e152,152);
name[153]="G4_SODIUM_CARBONATE";
static const G4double e153[25]={8.615E+01,4.307E+01,2.996E+01,1.94E+01,8.966,5.811,3.905,2.315,1.793,1.607,1.552,1.633,1.738,1.887,2.532,3.145,4.365,9.275,1.545E+01,2.631E+01,6.364E+01,1.26E+02,1.884E+02,2.507E+02,3.132E+02};
AddData(T0,e153,153);
name[154]="G4_SODIUM_IODIDE";
static const G4double e154[25]={3.179E+01,2.301E+01,1.735E+01,1.198E+01,5.968,3.99,2.751,1.692,1.348,1.245,1.276,1.494,1.758,2.162,4.147,6.158,1.024E+01,2.684E+01,4.778E+01,8.456E+01,2.111E+02,4.222E+02,6.335E+02,8.448E+02,1.056E+03};
AddData(T0,e154,154);
name[155]="G4_SODIUM_MONOXIDE";
static const G4double e155[25]={7.835E+01,4.027E+01,2.822E+01,1.838E+01,8.562,5.567,3.751,2.231,1.733,1.562,1.524,1.618,1.729,1.886,2.581,3.248,4.572,9.904,1.662E+01,2.84E+01,6.894E+01,1.366E+02,2.044E+02,2.721E+02,3.399E+02};
AddData(T0,e155,155);
name[156]="G4_SODIUM_NITRATE";
static const G4double e156[25]={9.015E+01,4.451E+01,3.086E+01,1.992E+01,9.175,5.938,3.985,2.359,1.825,1.634,1.573,1.651,1.755,1.90,2.53,3.127,4.312,9.081,1.508E+01,2.563E+01,6.19E+01,1.225E+02,1.831E+02,2.437E+02,3.043E+02};
AddData(T0,e156,156);
name[157]="G4_STILBENE";
static const G4double e157[25]={1.192E+02,5.537E+01,3.773E+01,2.398E+01,1.083E+01,6.952,4.634,2.718,2.09,1.858,1.77,1.834,1.927,2.051,2.545,2.998,3.892,7.473,1.198E+01,1.989E+01,4.713E+01,9.263E+01,1.382E+02,1.837E+02,2.293E+02};
AddData(T0,e157,157);
name[158]="G4_SUCROSE";
static const G4double e158[25]={1.134E+02,5.338E+01,3.652E+01,2.33E+01,1.057E+01,6.799,4.539,2.669,2.052,1.821,1.734,1.80,1.896,2.028,2.576,3.086,4.095,8.142,1.323E+01,2.218E+01,5.296E+01,1.044E+02,1.558E+02,2.073E+02,2.588E+02};
AddData(T0,e158,158);
name[159]="G4_TEFLON";
static const G4double e159[25]={9.304E+01,4.507E+01,3.108E+01,1.997E+01,9.146,5.904,3.955,2.335,1.804,1.61,1.542,1.612,1.71,1.848,2.439,2.998,4.109,8.58,1.421E+01,2.409E+01,5.808E+01,1.148E+02,1.716E+02,2.284E+02,2.853E+02};
AddData(T0,e159,159);
name[160]="G4_TERPHENYL";
static const G4double e160[25]={1.144E+02,5.343E+01,3.646E+01,2.321E+01,1.05E+01,6.748,4.501,2.643,2.03,1.803,1.716,1.781,1.874,1.998,2.491,2.946,3.846,7.452,1.199E+01,1.996E+01,4.74E+01,9.324E+01,1.391E+02,1.85E+02,2.309E+02};
AddData(T0,e160,160);
name[161]="G4_TESTES_ICRP";
static const G4double e161[25]={1.189E+02,5.58E+01,3.814E+01,2.431E+01,1.102E+01,7.082,4.727,2.778,2.139,1.91,1.825,1.893,1.993,2.132,2.713,3.255,4.324,8.609,1.4E+01,2.347E+01,5.604E+01,1.105E+02,1.649E+02,2.194E+02,2.739E+02};
AddData(T0,e161,161);
name[162]="G4_TETRACHLOROETHYLENE";
static const G4double e162[25]={7.553E+01,3.927E+01,2.761E+01,1.803E+01,8.426,5.487,3.702,2.207,1.717,1.547,1.511,1.634,1.785,1.996,2.939,3.86,5.711,1.32E+01,2.264E+01,3.922E+01,9.628E+01,1.915E+02,2.869E+02,3.822E+02,4.776E+02};
AddData(T0,e162,162);
name[163]="G4_THALLIUM_CHLORIDE";
static const G4double e163[25]={1.681E+01,1.746E+01,1.385E+01,9.922,5.136,3.486,2.435,1.528,1.237,1.159,1.219,1.492,1.824,2.338,4.922,7.57,1.297E+01,3.498E+01,6.272E+01,1.114E+02,2.79E+02,5.586E+02,8.384E+02,1.118E+03,1.398E+03};
AddData(T0,e163,163);
name[164]="G4_TISSUE-METHANE";
static const G4double e164[25]={1.275E+02,5.864E+01,3.984E+01,2.526E+01,1.137E+01,7.288,4.853,2.842,2.184,1.957,1.908,2.049,2.205,2.40,3.085,3.622,4.634,8.599,1.355E+01,2.223E+01,5.208E+01,1.019E+02,1.518E+02,2.017E+02,2.517E+02};
AddData(T0,e164,164);
name[165]="G4_TISSUE-PROPANE";
static const G4double e165[25]={1.288E+02,5.908E+01,4.011E+01,2.541E+01,1.143E+01,7.323,4.874,2.854,2.192,1.964,1.913,2.054,2.209,2.401,3.053,3.568,4.55,8.408,1.323E+01,2.168E+01,5.073E+01,9.927E+01,1.479E+02,1.965E+02,2.451E+02};
AddData(T0,e165,165);
name[166]="G4_TISSUE_SOFT_ICRP";
static const G4double e166[25]={1.205E+02,5.635E+01,3.847E+01,2.449E+01,1.109E+01,7.124,4.753,2.791,2.148,1.917,1.829,1.896,1.995,2.131,2.697,3.222,4.259,8.412,1.364E+01,2.281E+01,5.439E+01,1.071E+02,1.599E+02,2.127E+02,2.656E+02};
AddData(T0,e166,166);
name[167]="G4_TISSUE_SOFT_ICRU-4";
static const G4double e167[25]={1.187E+02,5.569E+01,3.806E+01,2.425E+01,1.099E+01,7.067,4.717,2.771,2.134,1.907,1.822,1.891,1.99,2.128,2.705,3.241,4.30,8.541,1.388E+01,2.325E+01,5.549E+01,1.094E+02,1.633E+02,2.172E+02,2.711E+02};
AddData(T0,e167,167);
name[168]="G4_TITANIUM_DIOXIDE";
static const G4double e168[25]={6.999E+01,3.722E+01,2.631E+01,1.727E+01,8.117,5.299,3.581,2.141,1.657,1.488,1.45,1.563,1.707,1.918,2.88,3.822,5.716,1.34E+01,2.308E+01,4.009E+01,9.863E+01,1.964E+02,2.942E+02,3.92E+02,4.898E+02};
AddData(T0,e168,168);
name[169]="G4_TOLUENE";
static const G4double e169[25]={1.249E+02,5.756E+01,3.913E+01,2.482E+01,1.118E+01,7.169,4.774,2.797,2.149,1.908,1.815,1.878,1.971,2.095,2.587,3.038,3.926,7.483,1.196E+01,1.982E+01,4.687E+01,9.208E+01,1.373E+02,1.826E+02,2.279E+02};
AddData(T0,e169,169);
name[170]="G4_TRICHLOROETHYLENE";
static const G4double e170[25]={7.903E+01,4.058E+01,2.844E+01,1.852E+01,8.627,5.61,3.78,2.25,1.749,1.574,1.535,1.655,1.803,2.01,2.929,3.825,5.624,1.29E+01,2.207E+01,3.818E+01,9.361E+01,1.862E+02,2.788E+02,3.714E+02,4.641E+02};
AddData(T0,e170,170);
name[171]="G4_TRIETHYL_PHOSPHATE";
static const G4double e171[25]={1.127E+02,5.335E+01,3.655E+01,2.335E+01,1.061E+01,6.83,4.563,2.685,2.069,1.847,1.768,1.844,1.95,2.095,2.70,3.268,4.391,8.904,1.458E+01,2.456E+01,5.889E+01,1.162E+02,1.736E+02,2.31E+02,2.885E+02};
AddData(T0,e171,171);
name[172]="G4_TUNGSTEN_HEXAFLUORIDE";
static const G4double e172[25]={4.022E+01,2.592E+01,1.912E+01,1.299E+01,6.359,4.223,2.897,1.772,1.407,1.30,1.325,1.532,1.786,2.178,4.106,6.06,1.003E+01,2.617E+01,4.651E+01,8.222E+01,2.05E+02,4.1E+02,6.151E+02,8.202E+02,1.025E+03};
AddData(T0,e172,172);
name[173]="G4_URANIUM_DICARBIDE";
static const G4double e173[25]={1.365E+01,1.606E+01,1.291E+01,9.334,4.877,3.323,2.33,1.466,1.19,1.122,1.192,1.482,1.836,2.391,5.195,8.072,1.395E+01,3.798E+01,6.83E+01,1.215E+02,3.046E+02,6.103E+02,9.159E+02,1.221E+03,1.527E+03};
AddData(T0,e173,173);
name[174]="G4_URANIUM_MONOCARBIDE";
static const G4double e174[25]={9.271,1.445E+01,1.19E+01,8.74,4.637,3.178,2.238,1.417,1.157,1.096,1.173,1.473,1.84,2.414,5.328,8.321,1.444E+01,3.945E+01,7.1E+01,1.264E+02,3.169E+02,6.35E+02,9.531E+02,1.271E+03,1.589E+03};
AddData(T0,e174,174);
name[175]="G4_URANIUM_OXIDE";
static const G4double e175[25]={1.51E+01,1.663E+01,1.328E+01,9.559,4.971,3.382,2.367,1.487,1.205,1.133,1.20,1.484,1.832,2.377,5.125,7.945,1.37E+01,3.724E+01,6.693E+01,1.191E+02,2.984E+02,5.977E+02,8.971E+02,1.196E+03,1.495E+03};
AddData(T0,e175,175);
name[176]="G4_UREA";
static const G4double e176[25]={1.162E+02,5.438E+01,3.713E+01,2.364E+01,1.071E+01,6.879,4.59,2.696,2.074,1.841,1.752,1.817,1.913,2.045,2.592,3.101,4.107,8.142,1.322E+01,2.214E+01,5.283E+01,1.041E+02,1.554E+02,2.067E+02,2.581E+02};
AddData(T0,e176,176);
name[177]="G4_VALINE";
static const G4double e177[25]={1.223E+02,5.68E+01,3.87E+01,2.46E+01,1.111E+01,7.131,4.754,2.789,2.142,1.899,1.804,1.867,1.962,2.091,2.617,3.104,4.065,7.915,1.276E+01,2.127E+01,5.055E+01,9.947E+01,1.484E+02,1.974E+02,2.464E+02};
AddData(T0,e177,177);
name[178]="G4_VITON";
static const G4double e178[25]={9.438E+01,4.569E+01,3.151E+01,2.024E+01,9.267,5.982,4.007,2.365,1.827,1.635,1.569,1.641,1.739,1.876,2.462,3.015,4.111,8.518,1.406E+01,2.381E+01,5.731E+01,1.133E+02,1.692E+02,2.252E+02,2.812E+02};
AddData(T0,e178,178);
name[179]="G4_WATER";
static const G4double e179[25]={1.198E+02,5.621E+01,3.841E+01,2.448E+01,1.11E+01,7.134,4.761,2.798,2.154,1.926,1.841,1.91,2.01,2.149,2.738,3.286,4.366,8.698,1.414E+01,2.371E+01,5.664E+01,1.116E+02,1.667E+02,2.217E+02,2.768E+02};
AddData(T0,e179,179);
name[180]="G4_WATER_VAPOR";
static const G4double e180[25]={1.218E+02,5.689E+01,3.882E+01,2.471E+01,1.118E+01,7.185,4.793,2.814,2.166,1.944,1.899,2.047,2.209,2.413,3.155,3.764,4.884,9.258,1.473E+01,2.431E+01,5.725E+01,1.122E+02,1.673E+02,2.223E+02,2.774E+02};
AddData(T0,e180,180);
name[181]="G4_XYLENE";
static const G4double e181[25]={1.262E+02,5.811E+01,3.949E+01,2.504E+01,1.128E+01,7.229,4.814,2.82,2.166,1.923,1.828,1.891,1.983,2.107,2.598,3.048,3.935,7.483,1.195E+01,1.979E+01,4.678E+01,9.187E+01,1.37E+02,1.822E+02,2.273E+02};
AddData(T0,e181,181);
name[182]="G4_H";
static const G4double e182[25]={3.205E+02,1.361E+02,9.018E+01,5.585E+01,2.439E+01,1.542E+01,1.015E+01,5.852,4.447,3.949,3.796,3.997,4.222,4.479,5.212,5.686,6.39,8.913,1.203E+01,1.751E+01,3.644E+01,6.823E+01,1.001E+02,1.321E+02,1.64E+02};
AddData(T0,e182,182);
name[183]="G4_He";
static const G4double e183[25]={1.308E+02,5.83E+01,3.923E+01,2.465E+01,1.097E+01,6.997,4.639,2.701,2.066,1.845,1.787,1.898,2.018,2.157,2.58,2.886,3.412,5.173,7.30,1.102E+01,2.378E+01,4.508E+01,6.64E+01,8.772E+01,1.09E+02};
AddData(T0,e183,183);
name[184]="G4_Li";
static const G4double e184[25]={1.146E+02,5.093E+01,3.423E+01,2.149E+01,9.554,6.088,4.032,2.34,1.78,1.577,1.505,1.559,1.619,1.692,1.968,2.217,2.70,4.616,7.016,1.123E+01,2.574E+01,5.00E+01,7.428E+01,9.857E+01,1.229E+02};
AddData(T0,e184,184);
name[185]="G4_Be";
static const G4double e185[25]={1.015E+02,4.686E+01,3.187E+01,2.022E+01,9.115,5.843,3.889,2.269,1.729,1.532,1.46,1.516,1.584,1.669,2.002,2.308,2.911,5.314,8.332,1.364E+01,3.189E+01,6.239E+01,9.292E+01,1.235E+02,1.54E+02};
AddData(T0,e185,185);
name[186]="G4_B";
static const G4double e186[25]={9.932E+01,4.667E+01,3.191E+01,2.034E+01,9.223,5.927,3.952,2.31,1.761,1.56,1.486,1.547,1.628,1.731,2.134,2.508,3.245,6.20,9.916,1.645E+01,3.893E+01,7.649E+01,1.141E+02,1.517E+02,1.893E+02};
AddData(T0,e186,186);
name[187]="G4_C";
static const G4double e187[25]={1.062E+02,5.006E+01,3.425E+01,2.185E+01,9.916,6.375,4.252,2.489,1.901,1.687,1.609,1.677,1.771,1.896,2.396,2.857,3.768,7.426,1.203E+01,2.012E+01,4.795E+01,9.444E+01,1.41E+02,1.875E+02,2.341E+02};
AddData(T0,e187,187);
name[188]="G4_N";
static const G4double e188[25]={1.043E+02,4.943E+01,3.388E+01,2.164E+01,9.842,6.335,4.233,2.491,1.92,1.725,1.688,1.824,1.972,2.161,2.852,3.417,4.478,8.643,1.385E+01,2.297E+01,5.431E+01,1.066E+02,1.59E+02,2.114E+02,2.638E+02};
AddData(T0,e188,188);
name[189]="G4_O";
static const G4double e189[25]={9.86E+01,4.752E+01,3.272E+01,2.10E+01,9.60,6.193,4.146,2.446,1.889,1.70,1.667,1.808,1.962,2.16,2.903,3.526,4.697,9.303,1.507E+01,2.517E+01,5.986E+01,1.178E+02,1.757E+02,2.337E+02,2.917E+02};
AddData(T0,e189,189);
name[190]="G4_F";
static const G4double e190[25]={8.629E+01,4.263E+01,2.956E+01,1.908E+01,8.79,5.689,3.819,2.261,1.75,1.578,1.552,1.691,1.842,2.039,2.791,3.44,4.655,9.425,1.54E+01,2.586E+01,6.181E+01,1.218E+02,1.819E+02,2.419E+02,3.02E+02};
AddData(T0,e190,190);
name[191]="G4_Ne";
static const G4double e191[25]={8.344E+01,4.23E+01,2.954E+01,1.919E+01,8.903,5.78,3.89,2.31,1.792,1.619,1.598,1.748,1.912,2.126,2.961,3.701,5.105,1.052E+01,1.729E+01,2.915E+01,6.989E+01,1.379E+02,2.059E+02,2.739E+02,3.42E+02};
AddData(T0,e191,191);
name[192]="G4_Na";
static const G4double e192[25]={7.74E+01,3.979E+01,2.788E+01,1.817E+01,8.461,5.502,3.706,2.203,1.708,1.541,1.515,1.637,1.761,1.928,2.657,3.359,4.753,1.036E+01,1.741E+01,2.979E+01,7.236E+01,1.435E+02,2.146E+02,2.857E+02,3.569E+02};
AddData(T0,e192,192);
name[193]="G4_Mg";
static const G4double e193[25]={7.809E+01,4.046E+01,2.841E+01,1.854E+01,8.655,5.632,3.796,2.257,1.749,1.576,1.546,1.666,1.796,1.974,2.772,3.546,5.088,1.13E+01,1.913E+01,3.289E+01,8.019E+01,1.592E+02,2.382E+02,3.173E+02,3.964E+02};
AddData(T0,e193,193);
name[194]="G4_Al";
static const G4double e194[25]={7.385E+01,3.869E+01,2.725E+01,1.783E+01,8.345,5.437,3.668,2.183,1.691,1.522,1.491,1.609,1.739,1.921,2.746,3.55,5.158,1.165E+01,1.983E+01,3.421E+01,8.367E+01,1.662E+02,2.489E+02,3.315E+02,4.142E+02};
AddData(T0,e194,194);
name[195]="G4_Si";
static const G4double e195[25]={7.478E+01,3.949E+01,2.786E+01,1.826E+01,8.564,5.584,3.769,2.245,1.741,1.568,1.538,1.667,1.813,2.013,2.917,3.802,5.573,1.273E+01,2.176E+01,3.761E+01,9.216E+01,1.832E+02,2.744E+02,3.655E+02,4.567E+02};
AddData(T0,e195,195);
name[196]="G4_P";
static const G4double e196[25]={7.265E+01,3.836E+01,2.707E+01,1.774E+01,8.321,5.425,3.662,2.181,1.69,1.522,1.492,1.621,1.771,1.977,2.906,3.816,5.64,1.302E+01,2.232E+01,3.867E+01,9.49E+01,1.888E+02,2.827E+02,3.767E+02,4.707E+02};
AddData(T0,e196,196);
name[197]="G4_S";
static const G4double e197[25]={7.33E+01,3.90E+01,2.758E+01,1.81E+01,8.507,5.552,3.75,2.235,1.732,1.56,1.53,1.667,1.829,2.056,3.066,4.055,6.042,1.408E+01,2.421E+01,4.202E+01,1.033E+02,2.056E+02,3.079E+02,4.103E+02,5.127E+02};
AddData(T0,e197,197);
name[198]="G4_Cl";
static const G4double e198[25]={7.172E+01,3.791E+01,2.676E+01,1.754E+01,8.233,5.371,3.629,2.168,1.689,1.532,1.525,1.697,1.891,2.16,3.29,4.332,6.385,1.458E+01,2.486E+01,4.29E+01,1.049E+02,2.085E+02,3.12E+02,4.157E+02,5.193E+02};
AddData(T0,e198,198);
name[199]="G4_Ar";
static const G4double e199[25]={6.465E+01,3.471E+01,2.459E+01,1.617E+01,7.619,4.978,3.368,2.015,1.572,1.428,1.424,1.589,1.777,2.037,3.152,4.187,6.214,1.431E+01,2.446E+01,4.227E+01,1.035E+02,2.056E+02,3.078E+02,4.101E+02,5.123E+02};
AddData(T0,e199,199);
name[200]="G4_K";
static const G4double e200[25]={6.932E+01,3.73E+01,2.644E+01,1.74E+01,8.20,5.359,3.625,2.169,1.69,1.532,1.517,1.665,1.841,2.094,3.238,4.354,6.597,1.568E+01,2.714E+01,4.728E+01,1.166E+02,2.322E+02,3.48E+02,4.637E+02,5.795E+02};
AddData(T0,e200,200);
name[201]="G4_Ca";
static const G4double e201[25]={7.098E+01,3.823E+01,2.711E+01,1.784E+01,8.412,5.498,3.719,2.224,1.732,1.568,1.548,1.698,1.88,2.144,3.357,4.551,6.958,1.672E+01,2.904E+01,5.07E+01,1.252E+02,2.496E+02,3.741E+02,4.987E+02,6.232E+02};
AddData(T0,e201,201);
name[202]="G4_Sc";
static const G4double e202[25]={6.192E+01,3.427E+01,2.446E+01,1.619E+01,7.68,5.033,3.411,2.044,1.593,1.442,1.423,1.564,1.737,1.991,3.167,4.331,6.679,1.622E+01,2.825E+01,4.939E+01,1.221E+02,2.436E+02,3.652E+02,4.868E+02,6.084E+02};
AddData(T0,e202,202);
name[203]="G4_Ti";
static const G4double e203[25]={5.816E+01,3.279E+01,2.351E+01,1.561E+01,7.435,4.879,3.311,1.985,1.546,1.399,1.381,1.521,1.695,1.952,3.15,4.34,6.743,1.651E+01,2.882E+01,5.047E+01,1.249E+02,2.493E+02,3.737E+02,4.982E+02,6.226E+02};
AddData(T0,e203,203);
name[204]="G4_V";
static const G4double e204[25]={5.537E+01,3.162E+01,2.274E+01,1.513E+01,7.229,4.748,3.224,1.933,1.505,1.362,1.345,1.485,1.659,1.918,3.138,4.351,6.802,1.677E+01,2.933E+01,5.142E+01,1.274E+02,2.543E+02,3.813E+02,5.083E+02,6.352E+02};
AddData(T0,e204,204);
name[205]="G4_Cr";
static const G4double e205[25]={5.487E+01,3.174E+01,2.289E+01,1.527E+01,7.315,4.81,3.268,1.961,1.527,1.382,1.366,1.511,1.693,1.966,3.255,4.54,7.136,1.769E+01,3.099E+01,5.438E+01,1.348E+02,2.691E+02,4.035E+02,5.379E+02,6.723E+02};
AddData(T0,e205,205);
name[206]="G4_Mn";
static const G4double e206[25]={5.206E+01,3.061E+01,2.216E+01,1.483E+01,7.126,4.693,3.192,1.918,1.495,1.354,1.341,1.489,1.673,1.95,3.267,4.578,7.228,1.80E+01,3.159E+01,5.547E+01,1.376E+02,2.748E+02,4.12E+02,5.492E+02,6.865E+02};
AddData(T0,e206,206);
name[207]="G4_Fe";
static const G4double e207[25]={5.142E+01,3.07E+01,2.23E+01,1.496E+01,7.213,4.756,3.238,1.947,1.519,1.376,1.365,1.52,1.713,2.005,3.396,4.782,7.584,1.898E+01,3.334E+01,5.859E+01,1.454E+02,2.904E+02,4.354E+02,5.805E+02,7.256E+02};
AddData(T0,e207,207);
name[208]="G4_Co";
static const G4double e208[25]={4.924E+01,2.976E+01,2.167E+01,1.457E+01,7.04,4.646,3.165,1.905,1.486,1.347,1.338,1.493,1.688,1.982,3.393,4.80,7.646,1.922E+01,3.38E+01,5.943E+01,1.476E+02,2.947E+02,4.42E+02,5.893E+02,7.366E+02};
AddData(T0,e208,208);
name[209]="G4_Ni";
static const G4double e209[25]={4.955E+01,3.041E+01,2.221E+01,1.497E+01,7.257,4.795,3.27,1.97,1.537,1.393,1.385,1.552,1.76,2.074,3.588,5.099,8.153,2.057E+01,3.622E+01,6.373E+01,1.583E+02,3.162E+02,4.742E+02,6.322E+02,7.902E+02};
AddData(T0,e209,209);
name[210]="G4_Cu";
static const G4double e210[25]={4.615E+01,2.867E+01,2.10E+01,1.418E+01,6.892,4.56,3.114,1.881,1.469,1.333,1.327,1.49,1.693,2.001,3.491,4.978,7.983,2.02E+01,3.56E+01,6.265E+01,1.557E+02,3.109E+02,4.662E+02,6.216E+02,7.77E+02};
AddData(T0,e210,210);
name[211]="G4_Zn";
static const G4double e211[25]={4.551E+01,2.853E+01,2.093E+01,1.416E+01,6.89,4.561,3.117,1.886,1.476,1.341,1.337,1.507,1.717,2.036,3.577,5.117,8.227,2.087E+01,3.68E+01,6.478E+01,1.61E+02,3.216E+02,4.823E+02,6.43E+02,8.038E+02};
AddData(T0,e211,211);
name[212]="G4_Ga";
static const G4double e212[25]={4.369E+01,2.751E+01,2.02E+01,1.368E+01,6.661,4.412,3.017,1.83,1.438,1.311,1.311,1.479,1.686,2.001,3.529,5.066,8.172,2.076E+01,3.663E+01,6.45E+01,1.604E+02,3.204E+02,4.805E+02,6.406E+02,8.007E+02};
AddData(T0,e212,212);
name[213]="G4_Ge";
static const G4double e213[25]={4.168E+01,2.672E+01,1.97E+01,1.337E+01,6.534,4.334,2.967,1.804,1.421,1.30,1.307,1.479,1.69,2.008,3.562,5.126,8.289,2.111E+01,3.727E+01,6.566E+01,1.633E+02,3.263E+02,4.893E+02,6.525E+02,8.155E+02};
AddData(T0,e213,213);
name[214]="G4_As";
static const G4double e214[25]={4.195E+01,2.681E+01,1.975E+01,1.34E+01,6.544,4.339,2.97,1.805,1.422,1.30,1.309,1.485,1.699,2.024,3.612,5.215,8.457,2.161E+01,3.818E+01,6.73E+01,1.674E+02,3.346E+02,5.018E+02,6.691E+02,8.364E+02};
AddData(T0,e214,214);
name[215]="G4_Se";
static const G4double e215[25]={4.092E+01,2.617E+01,1.928E+01,1.309E+01,6.394,4.24,2.903,1.765,1.39,1.272,1.284,1.465,1.681,2.007,3.599,5.206,8.457,2.164E+01,3.826E+01,6.746E+01,1.679E+02,3.355E+02,5.032E+02,6.709E+02,8.386E+02};
AddData(T0,e215,215);
name[216]="G4_Br";
static const G4double e216[25]={4.212E+01,2.679E+01,1.972E+01,1.337E+01,6.527,4.328,2.963,1.804,1.426,1.311,1.337,1.555,1.811,2.189,3.941,5.656,9.086,2.289E+01,4.023E+01,7.069E+01,1.754E+02,3.502E+02,5.25E+02,6.999E+02,8.747E+02};
AddData(T0,e216,216);
name[217]="G4_Kr";
static const G4double e217[25]={4.044E+01,2.598E+01,1.916E+01,1.301E+01,6.364,4.223,2.893,1.763,1.394,1.283,1.31,1.527,1.782,2.159,3.921,5.647,9.098,2.297E+01,4.04E+01,7.10E+01,1.762E+02,3.517E+02,5.273E+02,7.03E+02,8.786E+02};
AddData(T0,e217,217);
name[218]="G4_Rb";
static const G4double e218[25]={3.97E+01,2.583E+01,1.91E+01,1.30E+01,6.368,4.229,2.90,1.768,1.398,1.285,1.307,1.51,1.751,2.11,3.833,5.571,9.085,2.334E+01,4.13E+01,7.286E+01,1.814E+02,3.625E+02,5.438E+02,7.251E+02,9.064E+02};
AddData(T0,e218,218);
name[219]="G4_Sr";
static const G4double e219[25]={3.949E+01,2.579E+01,1.907E+01,1.299E+01,6.368,4.23,2.901,1.768,1.398,1.284,1.305,1.508,1.75,2.112,3.871,5.648,9.247,2.386E+01,4.227E+01,7.463E+01,1.859E+02,3.717E+02,5.576E+02,7.435E+02,9.294E+02};
AddData(T0,e219,219);
name[220]="G4_Y";
static const G4double e220[25]={3.874E+01,2.568E+01,1.905E+01,1.30E+01,6.389,4.248,2.915,1.779,1.406,1.291,1.311,1.515,1.76,2.129,3.939,5.774,9.493,2.46E+01,4.365E+01,7.713E+01,1.923E+02,3.844E+02,5.767E+02,7.69E+02,9.614E+02};
AddData(T0,e220,220);
name[221]="G4_Zr";
static const G4double e221[25]={3.747E+01,2.525E+01,1.879E+01,1.285E+01,6.333,4.215,2.895,1.767,1.397,1.283,1.303,1.507,1.753,2.127,3.97,5.842,9.639,2.507E+01,4.453E+01,7.872E+01,1.963E+02,3.926E+02,5.891E+02,7.855E+02,9.82E+02};
AddData(T0,e221,221);
name[222]="G4_Nb";
static const G4double e222[25]={3.565E+01,2.472E+01,1.849E+01,1.27E+01,6.283,4.188,2.878,1.758,1.39,1.277,1.298,1.506,1.758,2.141,4.032,5.953,9.853,2.571E+01,4.57E+01,8.083E+01,2.017E+02,4.033E+02,6.051E+02,8.07E+02,1.009E+03};
AddData(T0,e222,222);
name[223]="G4_Mo";
static const G4double e223[25]={3.48E+01,2.433E+01,1.823E+01,1.253E+01,6.211,4.143,2.85,1.743,1.38,1.267,1.288,1.495,1.747,2.133,4.043,5.987,9.933,2.598E+01,4.622E+01,8.178E+01,2.041E+02,4.082E+02,6.125E+02,8.169E+02,1.021E+03};
AddData(T0,e223,223);
name[224]="G4_Tc";
static const G4double e224[25]={3.458E+01,2.429E+01,1.822E+01,1.253E+01,6.213,4.144,2.849,1.74,1.376,1.263,1.285,1.496,1.753,2.147,4.101,6.089,1.013E+01,2.656E+01,4.728E+01,8.369E+01,2.089E+02,4.179E+02,6.271E+02,8.363E+02,1.045E+03};
AddData(T0,e224,224);
name[225]="G4_Ru";
static const G4double e225[25]={3.325E+01,2.374E+01,1.785E+01,1.231E+01,6.115,4.082,2.808,1.716,1.357,1.247,1.269,1.481,1.741,2.139,4.111,6.12,1.02E+01,2.68E+01,4.773E+01,8.451E+01,2.11E+02,4.221E+02,6.334E+02,8.447E+02,1.056E+03};
AddData(T0,e225,225);
name[226]="G4_Rh";
static const G4double e226[25]={3.278E+01,2.364E+01,1.781E+01,1.229E+01,6.117,4.085,2.812,1.719,1.36,1.25,1.274,1.49,1.756,2.162,4.18,6.236,1.041E+01,2.74E+01,4.882E+01,8.646E+01,2.159E+02,4.32E+02,6.482E+02,8.644E+02,1.081E+03};
AddData(T0,e226,226);
name[227]="G4_Pd";
static const G4double e227[25]={3.085E+01,2.285E+01,1.729E+01,1.198E+01,5.981,4.001,2.756,1.688,1.337,1.23,1.256,1.473,1.741,2.151,4.183,6.251,1.045E+01,2.755E+01,4.911E+01,8.699E+01,2.172E+02,4.347E+02,6.522E+02,8.698E+02,1.087E+03};
AddData(T0,e227,227);
name[228]="G4_Ag";
static const G4double e228[25]={3.11E+01,2.303E+01,1.743E+01,1.207E+01,6.032,4.036,2.782,1.703,1.348,1.241,1.268,1.491,1.766,2.188,4.272,6.395,1.071E+01,2.825E+01,5.037E+01,8.924E+01,2.229E+02,4.46E+02,6.693E+02,8.925E+02,1.116E+03};
AddData(T0,e228,228);
name[229]="G4_Cd";
static const G4double e229[25]={3.055E+01,2.259E+01,1.709E+01,1.184E+01,5.914,3.958,2.729,1.674,1.327,1.221,1.249,1.472,1.746,2.167,4.247,6.364,1.067E+01,2.817E+01,5.023E+01,8.90E+01,2.223E+02,4.449E+02,6.675E+02,8.902E+02,1.113E+03};
AddData(T0,e229,229);
name[230]="G4_In";
static const G4double e230[25]={2.92E+01,2.213E+01,1.682E+01,1.168E+01,5.855,3.923,2.709,1.668,1.326,1.223,1.254,1.48,1.758,2.187,4.303,6.455,1.083E+01,2.862E+01,5.106E+01,9.048E+01,2.261E+02,4.524E+02,6.788E+02,9.052E+02,1.132E+03};
AddData(T0,e230,230);
name[231]="G4_Sn";
static const G4double e231[25]={2.882E+01,2.185E+01,1.66E+01,1.153E+01,5.78,3.873,2.675,1.648,1.311,1.211,1.243,1.469,1.747,2.176,4.297,6.457,1.085E+01,2.871E+01,5.124E+01,9.082E+01,2.269E+02,4.541E+02,6.815E+02,9.088E+02,1.136E+03};
AddData(T0,e231,231);
name[232]="G4_Sb";
static const G4double e232[25]={2.873E+01,2.175E+01,1.652E+01,1.148E+01,5.751,3.854,2.662,1.64,1.306,1.208,1.242,1.47,1.75,2.183,4.327,6.509,1.095E+01,2.901E+01,5.178E+01,9.18E+01,2.294E+02,4.591E+02,6.889E+02,9.188E+02,1.149E+03};
AddData(T0,e232,232);
name[233]="G4_Te";
static const G4double e233[25]={2.808E+01,2.12E+01,1.61E+01,1.118E+01,5.601,3.753,2.593,1.597,1.273,1.178,1.213,1.438,1.715,2.143,4.261,6.418,1.081E+01,2.867E+01,5.119E+01,9.076E+01,2.269E+02,4.54E+02,6.813E+02,9.087E+02,1.136E+03};
AddData(T0,e233,233);
name[234]="G4_I";
static const G4double e234[25]={2.837E+01,2.159E+01,1.642E+01,1.141E+01,5.724,3.838,2.653,1.638,1.308,1.211,1.25,1.486,1.774,2.219,4.425,6.67,1.124E+01,2.983E+01,5.328E+01,9.448E+01,2.362E+02,4.727E+02,7.093E+02,9.46E+02,1.183E+03};
AddData(T0,e234,234);
name[235]="G4_Xe";
static const G4double e235[25]={2.854E+01,2.147E+01,1.629E+01,1.131E+01,5.664,3.796,2.623,1.619,1.294,1.203,1.253,1.513,1.825,2.299,4.586,6.871,1.148E+01,3.015E+01,5.363E+01,9.487E+01,2.366E+02,4.732E+02,7.099E+02,9.467E+02,1.183E+03};
AddData(T0,e235,235);
name[236]="G4_Cs";
static const G4double e236[25]={2.832E+01,2.146E+01,1.631E+01,1.133E+01,5.681,3.809,2.633,1.626,1.299,1.206,1.25,1.498,1.797,2.256,4.52,6.818,1.149E+01,3.05E+01,5.447E+01,9.659E+01,2.414E+02,4.832E+02,7.251E+02,9.671E+02,1.209E+03};
AddData(T0,e236,236);
name[237]="G4_Ba";
static const G4double e237[25]={2.77E+01,2.109E+01,1.603E+01,1.114E+01,5.59,3.748,2.591,1.60,1.278,1.185,1.228,1.47,1.765,2.219,4.474,6.768,1.144E+01,3.045E+01,5.443E+01,9.658E+01,2.415E+02,4.834E+02,7.255E+02,9.676E+02,1.21E+03};
AddData(T0,e237,237);
name[238]="G4_La";
static const G4double e238[25]={2.723E+01,2.10E+01,1.60E+01,1.114E+01,5.597,3.755,2.597,1.604,1.28,1.187,1.228,1.471,1.768,2.228,4.521,6.861,1.162E+01,3.104E+01,5.554E+01,9.857E+01,2.466E+02,4.937E+02,7.409E+02,9.882E+02,1.236E+03};
AddData(T0,e238,238);
name[239]="G4_Ce";
static const G4double e239[25]={2.607E+01,2.071E+01,1.585E+01,1.108E+01,5.585,3.752,2.597,1.606,1.283,1.19,1.234,1.482,1.785,2.255,4.599,6.991,1.186E+01,3.173E+01,5.677E+01,1.008E+02,2.522E+02,5.049E+02,7.578E+02,1.011E+03,1.264E+03};
AddData(T0,e239,239);
name[240]="G4_Pr";
static const G4double e240[25]={2.562E+01,2.07E+01,1.589E+01,1.112E+01,5.617,3.776,2.615,1.618,1.293,1.20,1.245,1.499,1.809,2.291,4.693,7.145,1.214E+01,3.249E+01,5.816E+01,1.033E+02,2.584E+02,5.173E+02,7.764E+02,1.035E+03,1.295E+03};
AddData(T0,e240,240);
name[241]="G4_Nd";
static const G4double e241[25]={2.478E+01,2.034E+01,1.565E+01,1.097E+01,5.551,3.734,2.587,1.601,1.28,1.189,1.235,1.49,1.801,2.285,4.704,7.172,1.22E+01,3.269E+01,5.852E+01,1.039E+02,2.60E+02,5.207E+02,7.814E+02,1.042E+03,1.303E+03};
AddData(T0,e241,241);
name[242]="G4_Pm";
static const G4double e242[25]={2.424E+01,2.03E+01,1.566E+01,1.101E+01,5.582,3.758,2.605,1.613,1.29,1.199,1.246,1.506,1.825,2.321,4.799,7.329,1.248E+01,3.348E+01,5.997E+01,1.065E+02,2.665E+02,5.336E+02,8.008E+02,1.068E+03,1.336E+03};
AddData(T0,e242,242);
name[243]="G4_Sm";
static const G4double e243[25]={2.295E+01,1.962E+01,1.518E+01,1.069E+01,5.435,3.662,2.54,1.574,1.26,1.171,1.219,1.476,1.792,2.283,4.741,7.251,1.236E+01,3.32E+01,5.947E+01,1.056E+02,2.643E+02,5.293E+02,7.943E+02,1.059E+03,1.325E+03};
AddData(T0,e243,243);
name[244]="G4_Eu";
static const G4double e244[25]={2.273E+01,1.961E+01,1.52E+01,1.071E+01,5.45,3.675,2.551,1.582,1.268,1.179,1.229,1.492,1.815,2.317,4.823,7.379,1.258E+01,3.379E+01,6.053E+01,1.075E+02,2.69E+02,5.386E+02,8.084E+02,1.078E+03,1.348E+03};
AddData(T0,e244,244);
name[245]="G4_Gd";
static const G4double e245[25]={2.171E+01,1.905E+01,1.48E+01,1.044E+01,5.324,3.59,2.492,1.546,1.238,1.152,1.20,1.459,1.777,2.273,4.758,7.297,1.247E+01,3.355E+01,6.012E+01,1.068E+02,2.674E+02,5.354E+02,8.035E+02,1.072E+03,1.34E+03};
AddData(T0,e245,245);
name[246]="G4_Tb";
static const G4double e246[25]={2.059E+01,1.873E+01,1.462E+01,1.036E+01,5.297,3.578,2.486,1.544,1.238,1.152,1.203,1.465,1.788,2.292,4.818,7.398,1.265E+01,3.408E+01,6.108E+01,1.085E+02,2.717E+02,5.44E+02,8.165E+02,1.089E+03,1.361E+03};
AddData(T0,e246,246);
name[247]="G4_Dy";
static const G4double e247[25]={1.973E+01,1.836E+01,1.437E+01,1.02E+01,5.23,3.535,2.458,1.527,1.225,1.141,1.192,1.455,1.779,2.285,4.821,7.413,1.269E+01,3.421E+01,6.134E+01,1.09E+02,2.729E+02,5.464E+02,8.20E+02,1.094E+03,1.367E+03};
AddData(T0,e247,247);
name[248]="G4_Ho";
static const G4double e248[25]={1.863E+01,1.80E+01,1.415E+01,1.008E+01,5.184,3.508,2.442,1.519,1.22,1.137,1.189,1.455,1.783,2.294,4.859,7.482,1.282E+01,3.46E+01,6.204E+01,1.103E+02,2.761E+02,5.528E+02,8.297E+02,1.107E+03,1.384E+03};
AddData(T0,e248,248);
name[249]="G4_Er";
static const G4double e249[25]={1.825E+01,1.788E+01,1.408E+01,1.004E+01,5.172,3.502,2.438,1.518,1.219,1.137,1.19,1.458,1.788,2.305,4.899,7.552,1.296E+01,3.498E+01,6.274E+01,1.115E+02,2.792E+02,5.59E+02,8.39E+02,1.119E+03,1.399E+03};
AddData(T0,e249,249);
name[250]="G4_Tm";
static const G4double e250[25]={1.757E+01,1.77E+01,1.399E+01,1.00E+01,5.164,3.50,2.439,1.52,1.222,1.14,1.195,1.466,1.801,2.325,4.959,7.652,1.314E+01,3.55E+01,6.369E+01,1.132E+02,2.834E+02,5.676E+02,8.518E+02,1.136E+03,1.421E+03};
AddData(T0,e250,250);
name[251]="G4_Yb";
static const G4double e251[25]={1.693E+01,1.738E+01,1.377E+01,9.855,5.096,3.457,2.412,1.506,1.211,1.131,1.186,1.458,1.795,2.321,4.96,7.658,1.315E+01,3.554E+01,6.375E+01,1.133E+02,2.837E+02,5.68E+02,8.525E+02,1.137E+03,1.422E+03};
AddData(T0,e251,251);
name[252]="G4_Lu";
static const G4double e252[25]={1.652E+01,1.728E+01,1.372E+01,9.833,5.092,3.456,2.413,1.509,1.215,1.135,1.19,1.463,1.802,2.332,5.001,7.733,1.33E+01,3.599E+01,6.459E+01,1.148E+02,2.875E+02,5.758E+02,8.642E+02,1.153E+03,1.441E+03};
AddData(T0,e252,252);
name[253]="G4_Hf";
static const G4double e253[25]={1.592E+01,1.701E+01,1.353E+01,9.718,5.041,3.424,2.392,1.499,1.208,1.129,1.185,1.457,1.795,2.325,5.00,7.742,1.333E+01,3.613E+01,6.485E+01,1.153E+02,2.888E+02,5.784E+02,8.68E+02,1.158E+03,1.448E+03};
AddData(T0,e253,253);
name[254]="G4_Ta";
static const G4double e254[25]={1.534E+01,1.681E+01,1.342E+01,9.653,5.017,3.41,2.384,1.495,1.207,1.129,1.185,1.458,1.798,2.331,5.03,7.799,1.344E+01,3.648E+01,6.55E+01,1.165E+02,2.918E+02,5.844E+02,8.771E+02,1.17E+03,1.463E+03};
AddData(T0,e254,254);
name[255]="G4_W";
static const G4double e255[25]={1.491E+01,1.664E+01,1.331E+01,9.587,4.989,3.393,2.373,1.489,1.203,1.126,1.183,1.457,1.798,2.335,5.051,7.84,1.353E+01,3.674E+01,6.599E+01,1.174E+02,2.94E+02,5.889E+02,8.84E+02,1.179E+03,1.474E+03};
AddData(T0,e255,255);
name[256]="G4_Re";
static const G4double e256[25]={1.453E+01,1.652E+01,1.324E+01,9.548,4.975,3.384,2.367,1.486,1.201,1.125,1.184,1.46,1.804,2.345,5.089,7.906,1.365E+01,3.711E+01,6.667E+01,1.186E+02,2.971E+02,5.952E+02,8.933E+02,1.192E+03,1.49E+03};
AddData(T0,e256,256);
name[257]="G4_Os";
static const G4double e257[25]={1.399E+01,1.625E+01,1.305E+01,9.424,4.917,3.347,2.342,1.471,1.19,1.116,1.175,1.451,1.795,2.336,5.082,7.903,1.366E+01,3.716E+01,6.677E+01,1.188E+02,2.977E+02,5.962E+02,8.949E+02,1.194E+03,1.492E+03};
AddData(T0,e257,257);
name[258]="G4_Ir";
static const G4double e258[25]={1.357E+01,1.614E+01,1.299E+01,9.395,4.91,3.344,2.341,1.471,1.191,1.118,1.18,1.459,1.807,2.354,5.132,7.987,1.381E+01,3.76E+01,6.757E+01,1.202E+02,3.013E+02,6.035E+02,9.058E+02,1.208E+03,1.511E+03};
AddData(T0,e258,258);
name[259]="G4_Pt";
static const G4double e259[25]={1.22E+01,1.566E+01,1.269E+01,9.226,4.845,3.306,2.317,1.459,1.183,1.113,1.178,1.46,1.81,2.361,5.16,8.036,1.391E+01,3.787E+01,6.807E+01,1.211E+02,3.035E+02,6.079E+02,9.124E+02,1.217E+03,1.522E+03};
AddData(T0,e259,259);
name[260]="G4_Au";
static const G4double e260[25]={1.224E+01,1.571E+01,1.273E+01,9.256,4.861,3.318,2.327,1.466,1.189,1.119,1.186,1.472,1.827,2.385,5.217,8.129,1.407E+01,3.833E+01,6.891E+01,1.226E+02,3.072E+02,6.155E+02,9.237E+02,1.232E+03,1.541E+03};
AddData(T0,e260,260);
name[261]="G4_Hg";
static const G4double e261[25]={1.178E+01,1.549E+01,1.258E+01,9.159,4.817,3.29,2.309,1.458,1.184,1.115,1.186,1.477,1.834,2.395,5.239,8.162,1.413E+01,3.848E+01,6.916E+01,1.23E+02,3.084E+02,6.177E+02,9.27E+02,1.237E+03,1.546E+03};
AddData(T0,e261,261);
name[262]="G4_Tl";
static const G4double e262[25]={1.131E+01,1.526E+01,1.242E+01,9.058,4.771,3.26,2.289,1.448,1.178,1.111,1.183,1.477,1.835,2.398,5.249,8.18,1.416E+01,3.858E+01,6.935E+01,1.234E+02,3.092E+02,6.194E+02,9.297E+02,1.24E+03,1.55E+03};
AddData(T0,e262,262);
name[263]="G4_Pb";
static const G4double e263[25]={1.08E+01,1.507E+01,1.23E+01,8.988,4.743,3.243,2.279,1.442,1.175,1.11,1.183,1.48,1.841,2.407,5.279,8.231,1.426E+01,3.886E+01,6.986E+01,1.243E+02,3.115E+02,6.241E+02,9.368E+02,1.249E+03,1.562E+03};
AddData(T0,e263,263);
name[264]="G4_Bi";
static const G4double e264[25]={1.084E+01,1.512E+01,1.235E+01,9.02,4.76,3.255,2.287,1.448,1.181,1.117,1.192,1.493,1.859,2.433,5.339,8.329,1.443E+01,3.934E+01,7.073E+01,1.259E+02,3.154E+02,6.319E+02,9.485E+02,1.265E+03,1.582E+03};
AddData(T0,e264,264);
name[265]="G4_Po";
static const G4double e265[25]={1.07E+01,1.522E+01,1.244E+01,9.099,4.806,3.288,2.311,1.465,1.195,1.131,1.209,1.516,1.89,2.475,5.441,8.491,1.472E+01,4.015E+01,7.219E+01,1.285E+02,3.221E+02,6.452E+02,9.684E+02,1.292E+03,1.615E+03};
AddData(T0,e265,265);
name[266]="G4_At";
static const G4double e266[25]={1.097E+01,1.539E+01,1.257E+01,9.186,4.849,3.317,2.332,1.479,1.208,1.143,1.222,1.533,1.912,2.506,5.515,8.613,1.494E+01,4.077E+01,7.332E+01,1.305E+02,3.271E+02,6.553E+02,9.836E+02,1.312E+03,1.64E+03};
AddData(T0,e266,266);
name[267]="G4_Rn";
static const G4double e267[25]={1.167E+01,1.512E+01,1.227E+01,8.923,4.691,3.204,2.25,1.426,1.166,1.106,1.191,1.51,1.897,2.497,5.481,8.505,1.464E+01,3.962E+01,7.108E+01,1.263E+02,3.161E+02,6.329E+02,9.497E+02,1.266E+03,1.583E+03};
AddData(T0,e267,267);
name[268]="G4_Fr";
static const G4double e268[25]={1.05E+01,1.481E+01,1.21E+01,8.844,4.671,3.196,2.247,1.426,1.167,1.109,1.194,1.511,1.892,2.485,5.459,8.507,1.472E+01,4.006E+01,7.196E+01,1.28E+02,3.207E+02,6.424E+02,9.643E+02,1.286E+03,1.608E+03};
AddData(T0,e268,268);
name[269]="G4_Ra";
static const G4double e269[25]={1.052E+01,1.479E+01,1.208E+01,8.831,4.663,3.19,2.244,1.424,1.165,1.106,1.188,1.50,1.877,2.466,5.443,8.495,1.473E+01,4.019E+01,7.233E+01,1.288E+02,3.227E+02,6.466E+02,9.705E+02,1.294E+03,1.618E+03};
AddData(T0,e269,269);
name[270]="G4_Ac";
static const G4double e270[25]={1.003E+01,1.471E+01,1.205E+01,8.829,4.671,3.198,2.251,1.429,1.17,1.11,1.191,1.502,1.88,2.473,5.485,8.579,1.49E+01,4.075E+01,7.337E+01,1.307E+02,3.276E+02,6.564E+02,9.852E+02,1.314E+03,1.642E+03};
AddData(T0,e270,270);
name[271]="G4_Th";
static const G4double e271[25]={9.71,1.448E+01,1.188E+01,8.71,4.613,3.159,2.224,1.413,1.156,1.097,1.179,1.486,1.859,2.446,5.437,8.528,1.485E+01,4.065E+01,7.315E+01,1.303E+02,3.267E+02,6.546E+02,9.827E+02,1.311E+03,1.639E+03};
AddData(T0,e271,271);
name[272]="G4_Pa";
static const G4double e272[25]={8.744,1.433E+01,1.184E+01,8.717,4.635,3.179,2.239,1.424,1.166,1.107,1.19,1.503,1.886,2.489,5.556,8.711,1.516E+01,4.154E+01,7.483E+01,1.333E+02,3.343E+02,6.698E+02,1.005E+03,1.341E+03,1.676E+03};
AddData(T0,e272,272);
name[273]="G4_U";
static const G4double e273[25]={8.168,1.392E+01,1.153E+01,8.506,4.53,3.108,2.19,1.392,1.14,1.083,1.164,1.473,1.85,2.444,5.474,8.593,1.497E+01,4.105E+01,7.397E+01,1.318E+02,3.305E+02,6.623E+02,9.941E+02,1.326E+03,1.657E+03};
AddData(T0,e273,273);
name[274]="G4_Np";
static const G4double e274[25]={7.877,1.399E+01,1.162E+01,8.587,4.58,3.144,2.216,1.409,1.155,1.097,1.181,1.495,1.88,2.487,5.584,8.772,1.529E+01,4.197E+01,7.563E+01,1.347E+02,3.38E+02,6.773E+02,1.017E+03,1.356E+03,1.695E+03};
AddData(T0,e274,274);
name[275]="G4_Pu";
static const G4double e275[25]={7.25,1.381E+01,1.152E+01,8.533,4.563,3.134,2.21,1.407,1.154,1.097,1.182,1.499,1.888,2.501,5.627,8.846,1.543E+01,4.236E+01,7.634E+01,1.36E+02,3.412E+02,6.837E+02,1.026E+03,1.369E+03,1.711E+03};
AddData(T0,e275,275);
name[276]="G4_Am";
static const G4double e276[25]={6.776,1.358E+01,1.136E+01,8.434,4.518,3.106,2.192,1.398,1.148,1.093,1.18,1.50,1.89,2.506,5.64,8.863,1.545E+01,4.242E+01,7.644E+01,1.362E+02,3.416E+02,6.845E+02,1.027E+03,1.37E+03,1.713E+03};
AddData(T0,e276,276);
name[277]="G4_Cm";
static const G4double e277[25]={6.573,1.344E+01,1.126E+01,8.367,4.486,3.086,2.18,1.391,1.143,1.088,1.175,1.494,1.883,2.496,5.622,8.856,1.547E+01,4.249E+01,7.652E+01,1.363E+02,3.42E+02,6.854E+02,1.029E+03,1.372E+03,1.716E+03};
AddData(T0,e277,277);
name[278]="G4_Bk";
static const G4double e278[25]={6.217,1.344E+01,1.13E+01,8.406,4.515,3.108,2.196,1.403,1.153,1.098,1.187,1.51,1.906,2.529,5.709,8.998,1.573E+01,4.322E+01,7.785E+01,1.387E+02,3.48E+02,6.974E+02,1.047E+03,1.396E+03,1.746E+03};
AddData(T0,e278,278);
name[279]="G4_Cf";
static const G4double e279[25]={5.732,1.321E+01,1.114E+01,8.306,4.469,3.079,2.177,1.391,1.145,1.091,1.181,1.506,1.903,2.528,5.716,9.011,1.575E+01,4.327E+01,7.794E+01,1.389E+02,3.484E+02,6.981E+02,1.048E+03,1.398E+03,1.748E+03};
AddData(T0,e279,279);

} 

void G4ESTARStopping::AddData(const G4double* ekin, const G4double* stop, 
			      G4int idx)
{
  //G4cout << "G4ESTARStopping::AddData: idx= " << idx << G4endl;
  G4double x1, x2, x3, x4, x5, x6, x7;

  static const G4double fac = MeV*cm2/g;

  // hardtyped data
  if(0 == type) {
    sdata[idx] = new G4LPhysicsFreeVector(25, ekin[0]*MeV, ekin[24]*MeV);
    for(size_t i=0; i<25; ++i) { 
      sdata[idx]->PutValues(i, ekin[i]*MeV, stop[i]*fac); 
    }

    // basic data
  } else if(1 == type) {
    std::ostringstream ost;
    if(idx >= 182) {
      ost << dirPath << "/estar/estar_basic/elems/" << idx - 181 << ".dat";
    } else {
      size_t n = (name[idx]).size();
      ost << dirPath << "/estar/estar_basic/mater/" << (name[idx]).substr(3, n - 3);
    }
    std::ifstream fin(ost.str().c_str());
    if( !fin.is_open()) {
      G4ExceptionDescription ed;
      ed << "ESTAR data file <" << ost.str().c_str()
	 << "> is not retrieved!";
      G4Exception("G4ESTARStopping::AddData","em0003",FatalException,
                ed,"G4LEDATA version should be G4EMLOW6.34 or later.");
      return;
    }
    sdata[idx] = new G4LPhysicsFreeVector(81, 0.01*MeV, GeV);
    for(size_t i=0; i<81; ++i) { 
      fin >> x1 >> x2 >> x3 >> x4 >> x5 >> x6 >> x7;
      sdata[idx]->PutValues(i, x1*MeV, x4*fac); 
    }

    // long data
  } else {
    std::ostringstream ost;
    if(idx >= 182) {
      ost << dirPath << "/estar/estar_long/elems/" << idx - 181 << ".dat";
    } else {
      size_t n = (name[idx]).size();
      ost << dirPath << "/estar/estar_long/mater/" << (name[idx]).substr(3, n - 3);
    }
    std::ifstream fin(ost.str().c_str());
    if( !fin.is_open()) {
      G4ExceptionDescription ed;
      ed << "ESTAR data file <" << ost.str().c_str()
	 << "> is not retrieved!";
      G4Exception("G4ESTARStopping::AddData","em0003",FatalException,
                ed,"G4LEDATA version should be G4EMLOW6.34 or later.");
      return;
    }
    sdata[idx] = new G4LPhysicsFreeVector(97, 0.001*MeV, 10*GeV);
    for(size_t i=0; i<97; ++i) { 
      fin >> x1 >> x2 >> x3 >> x4 >> x5;
      sdata[idx]->PutValues(i, x1*MeV, x4*fac); 
    }
  }
  sdata[idx]->SetSpline(true);
  sdata[idx]->FillSecondDerivatives();
  //G4cout << "done " << G4endl;
}


