//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
// $Id: G4PSTARStopping.cc 95445 2016-02-11 08:18:10Z gcosmo $

//---------------------------------------------------------------------------
//
// GEANT4 Class file
//
// Description: Data on stopping power
//
// Author:      Anton Ivantchenko 18.04.2006
//
// Organisation:        QinetiQ Ltd, UK
// Customer:            ESA/ESTEC, NOORDWIJK
// Contract:            CSMAN-5288
//
// Modifications:  
// 25.04.2010 V.Ivanchenko introduce G4LPhysicsFreeVector
// 26.04.2010 A.Ivanchenko fixed data for G4_PROPANE and G4_Pt, 
//                         checked all data
// 
//----------------------------------------------------------------------------
//

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

#include "G4PSTARStopping.hh" 
#include "G4NISTStoppingData.hh" 

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......


G4PSTARStopping::G4PSTARStopping() : nvectors(0), emin(CLHEP::keV)
{
  // 1st initialisation 
  Initialise();
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

G4PSTARStopping::~G4PSTARStopping()
{
  if(0 < nvectors) {
    for(G4int i=0; i<nvectors; ++i) { delete sdata[i]; }
  }
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

void G4PSTARStopping::PrintWarning(G4int i) const
{
  G4ExceptionDescription ed;
  ed << "index of data " << i << " is <0 or >= " << nvectors 
     << " request ignored!";
  G4Exception("G4PSTARStopping::PrintWarning()","em0033",JustWarning,ed);
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......
 
void G4PSTARStopping::Initialise()
{
  // this method may be called several times during initialisation
  G4int nmat = G4Material::GetNumberOfMaterials();
  if(nmat == nvectors) { return; }

  // loop via material list to add extra data
  G4int j;
  for(G4int i=0; i<nmat; ++i) {
    const G4Material* mat = (*(G4Material::GetMaterialTable()))[i];

    G4bool isThere = false;  
    for(j=0; j<nvectors; ++j) {
      if(mat == materials[j]) {
	isThere = true;
	break;
      }
    }
    if(!isThere) {
      // check list of NIST materials
      G4String mname = mat->GetName();
      for(j=0; j<74; ++j) {
        if(mname == nameNIST[j]) {
          FindData(j, mat);
          isThere = true;
          break;
	}
      }
      if(!isThere) {
	G4String form = mat->GetChemicalFormula();
	for(j=0; j<numberOfMolecula; ++j) {
	  if(form == molecularName[j]) {
	    FindData(molecularIndex[j], mat);
	    break;
	  }
	}
      }
    }
  }
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

void G4PSTARStopping::FindData(G4int j, const G4Material* mat) 
{
  static const G4float e0[60] = { 176.8f, 216.6f, 250.1f, 279.6f, 306.3f, 351.1f, 390.9f, 426.8f, 459.8f, 489.9f, 517.8f, 544.f, 596.4f, 641.3f, 680.6f, 715.7f, 747.f, 775.f, 800.f, 822.6f, 861.7f, 894.2f, 920.7f, 941.5f, 957.3f, 968.6f, 976.2f, 980.5f, 982.f, 981.f, 978.f, 973.2f, 967.f, 959.5f, 910.4f, 853.6f, 797.9f, 746.9f, 701.5f, 661.5f, 626.4f, 595.6f, 544.8f, 504.4f, 471.1f, 442.8f, 416.4f, 392.f, 369.9f, 350.f, 332.2f, 316.4f, 302.2f, 289.5f, 278.1f, 267.9f, 228.9f, 200.8f, 179.5f, 162.6f }; 
  
  static const G4float e1[60] = { 185.5f, 227.2f, 262.3f, 293.2f, 321.2f, 364.8f, 404.f, 439.3f, 471.6f, 500.3f, 526.4f, 550.6f, 600.2f, 641.6f, 677.4f, 709.2f, 737.3f, 762.f, 783.7f, 803.1f, 837.2f, 866.8f, 891.7f, 911.6f, 926.6f, 937.4f, 944.7f, 949.f, 950.8f, 950.3f, 947.9f, 943.9f, 938.4f, 931.8f, 887.1f, 833.7f, 779.9f, 729.5f, 686.8f, 650.4f, 618.5f, 590.2f, 541.4f, 500.4f, 465.3f, 434.8f, 408.f, 384.6f, 364.4f, 346.5f, 330.5f, 316.1f, 303.2f, 291.3f, 280.4f, 270.4f, 230.7f, 202.2f, 180.5f, 163.5f }; 
  
  static const G4float e2[60] = { 195.9f, 239.9f, 277.f, 309.7f, 339.3f, 387.5f, 430.6f, 469.4f, 505.f, 537.2f, 566.9f, 594.6f, 649.9f, 696.7f, 737.4f, 773.3f, 805.2f, 833.4f, 858.3f, 880.5f, 918.9f, 951.f, 977.2f, 997.6f, 1013.f, 1023.f, 1030.f, 1034.f, 1034.f, 1033.f, 1029.f, 1023.f, 1016.f, 1008.f, 955.6f, 895.f, 835.1f, 779.6f, 729.7f, 685.6f, 646.6f, 612.3f, 555.f, 509.1f, 471.6f, 440.3f, 413.7f, 390.4f, 370.1f, 352.1f, 336.f, 321.4f, 308.3f, 296.3f, 285.3f, 275.2f, 234.9f, 206.f, 184.f, 166.7f }; 
  
  static const G4float e3[60] = { 29.73f, 36.41f, 42.05f, 47.01f, 51.5f, 59.46f, 66.48f, 72.83f, 78.66f, 84.09f, 89.2f, 94.02f, 103.7f, 112.1f, 119.7f, 126.6f, 132.8f, 138.5f, 143.7f, 148.5f, 157.1f, 164.3f, 170.5f, 175.7f, 180.1f, 183.7f, 186.7f, 189.1f, 191.1f, 192.5f, 193.6f, 194.4f, 194.8f, 195.f, 193.4f, 189.3f, 184.4f, 179.2f, 174.1f, 169.3f, 164.8f, 160.5f, 152.7f, 145.7f, 139.5f, 133.8f, 128.6f, 123.9f, 119.5f, 115.4f, 111.7f, 108.1f, 104.8f, 101.7f, 98.74f, 95.98f, 84.49f, 75.92f, 69.3f, 64.05f }; 
  
  static const G4float e4[60] = { 119.7f, 146.7f, 169.3f, 189.3f, 207.4f, 239.5f, 267.8f, 293.3f, 316.8f, 338.7f, 359.2f, 378.7f, 417.f, 450.4f, 480.1f, 506.7f, 530.7f, 552.6f, 572.4f, 590.5f, 622.1f, 648.3f, 670.f, 687.7f, 702.f, 713.2f, 721.7f, 727.8f, 731.9f, 734.1f, 734.8f, 734.f, 732.f, 729.f, 702.9f, 667.2f, 629.1f, 592.2f, 558.3f, 527.8f, 500.6f, 476.3f, 434.9f, 401.2f, 373.3f, 349.8f, 329.7f, 312.1f, 296.4f, 282.4f, 269.9f, 258.7f, 248.5f, 239.1f, 230.6f, 222.7f, 191.1f, 168.2f, 150.8f, 137.f }; 
  
  static const G4float e5[60] = { 92.38f, 113.1f, 130.6f, 146.1f, 160.f, 184.8f, 206.6f, 226.3f, 244.4f, 261.3f, 277.1f, 292.1f, 320.6f, 344.8f, 365.7f, 383.8f, 399.6f, 413.2f, 425.f, 435.1f, 451.f, 462.f, 469.2f, 473.4f, 475.2f, 475.1f, 473.7f, 471.2f, 468.f, 464.2f, 460.1f, 455.8f, 451.3f, 446.8f, 424.5f, 404.5f, 386.7f, 371.f, 356.8f, 344.f, 332.3f, 321.5f, 301.7f, 284.2f, 268.6f, 254.8f, 242.5f, 231.4f, 221.5f, 212.4f, 204.2f, 196.6f, 189.7f, 183.3f, 177.4f, 171.9f, 149.4f, 132.7f, 119.8f, 109.4f }; 
  
  static const G4float e6[60] = { 73.51f, 90.03f, 104.f, 116.2f, 127.3f, 147.f, 164.4f, 180.1f, 194.5f, 207.9f, 220.5f, 232.5f, 256.1f, 276.8f, 295.3f, 311.9f, 327.f, 340.8f, 353.4f, 364.9f, 385.3f, 402.5f, 417.f, 429.2f, 439.3f, 447.6f, 454.3f, 459.6f, 463.7f, 466.6f, 468.6f, 469.8f, 470.2f, 470.f, 461.9f, 447.2f, 430.1f, 412.7f, 396.5f, 381.4f, 367.5f, 354.5f, 331.1f, 310.6f, 292.6f, 276.5f, 262.2f, 249.4f, 238.1f, 227.9f, 218.7f, 210.2f, 202.6f, 195.7f, 189.2f, 183.2f, 158.9f, 140.9f, 127.f, 116.f }; 
  
  static const G4float e7[60] = { 86.08f, 105.4f, 121.7f, 136.1f, 149.1f, 172.2f, 192.5f, 210.9f, 227.7f, 243.5f, 258.2f, 272.2f, 299.7f, 323.5f, 344.5f, 363.3f, 380.2f, 395.3f, 409.f, 421.4f, 442.5f, 459.4f, 472.8f, 483.1f, 490.7f, 496.f, 499.2f, 500.7f, 500.8f, 499.5f, 497.2f, 494.f, 490.f, 485.5f, 457.4f, 426.7f, 397.7f, 371.9f, 349.5f, 330.1f, 313.2f, 298.5f, 274.2f, 254.9f, 239.f, 225.6f, 214.4f, 204.7f, 196.1f, 188.4f, 181.3f, 174.9f, 168.9f, 163.4f, 158.2f, 153.3f, 133.f, 118.2f, 106.8f, 97.72f }; 
  
  static const G4float e8[60] = { 14.76f, 18.07f, 20.87f, 23.33f, 25.56f, 29.51f, 33.f, 36.15f, 39.04f, 41.74f, 44.27f, 46.66f, 51.51f, 55.8f, 59.66f, 63.18f, 66.43f, 69.43f, 72.23f, 74.84f, 79.58f, 83.77f, 87.5f, 90.81f, 93.77f, 96.4f, 98.74f, 100.8f, 102.6f, 104.3f, 105.7f, 106.9f, 108.f, 108.9f, 111.5f, 112.f, 111.2f, 109.6f, 107.6f, 105.4f, 103.2f, 101.f, 96.33f, 91.8f, 87.7f, 84.05f, 80.83f, 78.f, 75.47f, 73.18f, 71.11f, 69.21f, 67.46f, 65.83f, 64.32f, 62.91f, 56.98f, 52.39f, 48.68f, 45.58f }; 
  
  static const G4float e9[60] = { 142.5f, 174.5f, 201.5f, 225.2f, 246.7f, 283.1f, 315.3f, 344.4f, 371.1f, 395.5f, 418.2f, 439.6f, 482.3f, 519.f, 551.3f, 580.1f, 606.f, 629.2f, 650.1f, 669.f, 701.9f, 729.5f, 752.2f, 770.4f, 784.5f, 795.1f, 802.6f, 807.4f, 810.f, 810.5f, 809.3f, 806.6f, 802.7f, 797.7f, 762.5f, 719.5f, 676.2f, 635.8f, 599.3f, 566.7f, 537.9f, 512.3f, 468.1f, 431.4f, 400.8f, 375.f, 353.f, 333.7f, 316.9f, 301.9f, 288.5f, 276.3f, 265.3f, 255.1f, 245.9f, 237.4f, 203.4f, 178.9f, 160.2f, 145.4f }; 
 
  static const G4float e10[60] = { 149.7f, 183.3f, 211.7f, 236.7f, 259.2f, 299.3f, 334.7f, 366.6f, 396.f, 423.3f, 449.f, 473.3f, 517.2f, 553.4f, 583.6f, 608.7f, 629.6f, 646.8f, 660.7f, 671.9f, 687.3f, 695.5f, 698.4f, 697.6f, 694.2f, 689.1f, 682.8f, 675.7f, 668.2f, 660.5f, 652.7f, 644.9f, 637.2f, 629.6f, 594.2f, 563.f, 535.4f, 510.7f, 488.4f, 468.2f, 449.8f, 432.8f, 404.4f, 380.1f, 358.1f, 337.6f, 318.8f, 302.2f, 287.4f, 274.2f, 262.3f, 251.5f, 241.7f, 232.7f, 224.4f, 216.8f, 186.f, 163.6f, 146.3f, 132.6f }; 
 
  static const G4float e11[60] = { 141.6f, 173.4f, 200.2f, 223.8f, 245.2f, 281.3f, 313.3f, 342.2f, 368.8f, 393.1f, 415.6f, 436.8f, 479.f, 515.2f, 547.f, 575.3f, 600.7f, 623.4f, 643.8f, 662.1f, 694.2f, 721.f, 743.2f, 760.9f, 774.8f, 785.2f, 792.6f, 797.6f, 800.3f, 801.1f, 800.3f, 797.9f, 794.4f, 789.9f, 756.4f, 714.5f, 671.4f, 630.5f, 593.2f, 559.7f, 530.f, 503.5f, 459.f, 423.f, 393.3f, 368.3f, 346.9f, 328.1f, 311.6f, 296.9f, 283.7f, 271.8f, 261.f, 251.1f, 242.f, 233.7f, 200.3f, 176.3f, 158.f, 143.5f }; 
 
  static const G4float e12[60] = { 138.4f, 169.5f, 195.7f, 218.8f, 239.7f, 270.2f, 297.8f, 322.7f, 345.4f, 365.1f, 382.6f, 398.7f, 433.5f, 462.4f, 487.4f, 509.9f, 530.f, 547.7f, 563.4f, 577.6f, 603.9f, 628.3f, 650.1f, 668.6f, 683.5f, 695.3f, 704.5f, 711.4f, 716.3f, 719.4f, 720.8f, 720.9f, 719.7f, 717.4f, 694.5f, 661.1f, 624.7f, 589.1f, 556.f, 526.f, 499.1f, 475.f, 435.8f, 404.4f, 377.9f, 354.6f, 334.f, 315.9f, 300.2f, 286.2f, 273.7f, 262.4f, 252.2f, 242.7f, 234.1f, 226.1f, 194.2f, 171.1f, 153.4f, 139.4f }; 
 
  static const G4float e13[60] = { 142.6f, 175.3f, 203.1f, 228.f, 250.5f, 283.4f, 313.3f, 340.4f, 365.1f, 386.7f, 405.9f, 423.5f, 462.f, 494.f, 521.9f, 546.9f, 569.3f, 589.2f, 606.9f, 622.9f, 652.4f, 679.9f, 704.5f, 725.3f, 742.2f, 755.6f, 766.1f, 774.f, 779.6f, 783.1f, 784.7f, 784.8f, 783.4f, 780.7f, 753.8f, 714.5f, 671.4f, 629.1f, 590.2f, 555.6f, 524.9f, 497.7f, 453.9f, 419.3f, 390.3f, 365.1f, 343.f, 323.7f, 307.1f, 292.3f, 279.1f, 267.3f, 256.6f, 246.8f, 237.8f, 229.5f, 196.7f, 173.f, 155.f, 140.8f }; 
 
  static const G4float e14[60] = { 225.8f, 276.5f, 319.3f, 357.f, 391.1f, 445.9f, 494.9f, 539.1f, 579.6f, 616.f, 649.5f, 680.6f, 742.9f, 795.1f, 840.1f, 879.9f, 914.8f, 945.5f, 972.4f, 996.1f, 1037.f, 1071.f, 1099.f, 1120.f, 1135.f, 1145.f, 1150.f, 1152.f, 1152.f, 1148.f, 1142.f, 1135.f, 1126.f, 1115.f, 1052.f, 981.5f, 912.8f, 849.7f, 795.6f, 749.f, 708.3f, 672.4f, 611.7f, 562.1f, 520.6f, 485.3f, 454.9f, 428.4f, 405.5f, 385.2f, 367.1f, 350.8f, 336.2f, 322.8f, 310.5f, 299.3f, 254.6f, 222.7f, 198.6f, 179.6f }; 
 
  static const G4float e15[60] = { 119.2f, 146.f, 168.6f, 188.5f, 206.5f, 235.1f, 260.7f, 283.8f, 305.f, 323.9f, 341.3f, 357.4f, 390.7f, 418.9f, 443.7f, 465.8f, 485.7f, 503.6f, 519.5f, 534.1f, 560.1f, 583.2f, 603.2f, 620.f, 633.7f, 644.7f, 653.4f, 660.1f, 665.f, 668.4f, 670.3f, 671.1f, 670.7f, 669.5f, 652.9f, 626.4f, 596.2f, 565.7f, 537.f, 510.7f, 486.8f, 465.1f, 427.5f, 396.1f, 369.6f, 347.f, 327.4f, 310.1f, 294.9f, 281.3f, 269.f, 258.f, 247.9f, 238.6f, 230.2f, 222.4f, 191.f, 168.2f, 150.8f, 137.f }; 
 
  static const G4float e16[60] = { 106.4f, 130.3f, 150.4f, 168.2f, 184.2f, 212.7f, 237.9f, 260.6f, 281.4f, 300.9f, 319.1f, 336.4f, 370.9f, 401.2f, 428.3f, 452.8f, 475.1f, 495.6f, 514.4f, 531.8f, 562.6f, 588.8f, 611.2f, 630.2f, 646.1f, 659.3f, 670.1f, 678.6f, 685.2f, 690.f, 693.2f, 694.9f, 695.4f, 694.8f, 679.f, 650.5f, 617.3f, 583.6f, 552.1f, 523.6f, 497.8f, 474.6f, 434.7f, 401.8f, 374.3f, 351.f, 330.9f, 313.2f, 297.6f, 283.7f, 271.2f, 259.9f, 249.6f, 240.3f, 231.7f, 223.7f, 191.9f, 168.9f, 151.4f, 137.5f }; 
 
  static const G4float e17[60] = { 74.48f, 91.21f, 105.3f, 117.8f, 129.f, 149.f, 166.5f, 182.4f, 197.f, 210.6f, 223.4f, 235.5f, 259.6f, 280.7f, 299.6f, 316.6f, 332.f, 346.2f, 359.1f, 371.1f, 392.2f, 410.1f, 425.2f, 438.f, 448.7f, 457.5f, 464.6f, 470.3f, 474.7f, 477.9f, 480.1f, 481.4f, 481.9f, 481.7f, 473.f, 456.7f, 437.6f, 417.8f, 398.6f, 380.6f, 364.f, 348.7f, 322.9f, 301.7f, 283.5f, 267.5f, 253.4f, 240.9f, 229.8f, 219.9f, 210.8f, 202.5f, 194.8f, 187.7f, 181.3f, 175.5f, 151.6f, 134.4f, 121.1f, 110.6f }; 
 
  static const G4float e18[60] = { 135.4f, 165.8f, 191.5f, 214.1f, 234.5f, 270.8f, 302.8f, 331.7f, 358.2f, 383.f, 406.2f, 428.2f, 471.7f, 509.8f, 543.6f, 574.f, 601.5f, 626.5f, 649.3f, 670.1f, 706.4f, 736.5f, 761.2f, 781.4f, 797.4f, 809.8f, 819.f, 825.3f, 829.2f, 830.8f, 830.6f, 828.7f, 825.4f, 820.9f, 785.9f, 741.4f, 696.f, 653.6f, 618.4f, 588.8f, 562.8f, 539.6f, 499.3f, 464.8f, 434.6f, 407.9f, 384.2f, 363.4f, 344.9f, 328.4f, 313.6f, 300.3f, 288.2f, 277.2f, 267.f, 257.7f, 220.4f, 193.5f, 173.f, 156.9f }; 
 
  static const G4float e19[60] = { 132.2f, 161.9f, 187.f, 209.f, 229.f, 262.6f, 292.5f, 319.4f, 344.2f, 366.8f, 387.8f, 407.5f, 447.f, 481.f, 510.8f, 537.6f, 561.6f, 583.2f, 602.7f, 620.4f, 651.5f, 677.9f, 700.f, 718.2f, 732.7f, 744.1f, 752.7f, 759.f, 763.2f, 765.5f, 766.2f, 765.5f, 763.5f, 760.6f, 734.4f, 698.2f, 659.3f, 621.4f, 586.f, 553.8f, 524.9f, 498.9f, 452.4f, 413.9f, 382.9f, 358.f, 337.7f, 319.7f, 303.9f, 289.7f, 277.1f, 265.6f, 255.2f, 245.6f, 236.9f, 228.8f, 196.4f, 173.f, 155.f, 140.8f }; 
 
  static const G4float e20[60] = { 123.5f, 151.3f, 174.7f, 195.3f, 214.f, 246.1f, 274.5f, 300.2f, 323.8f, 345.6f, 365.9f, 385.f, 423.f, 455.7f, 484.7f, 510.5f, 533.8f, 554.8f, 573.7f, 590.9f, 620.8f, 645.9f, 666.5f, 683.3f, 696.6f, 706.8f, 714.4f, 719.7f, 723.f, 724.5f, 724.5f, 723.1f, 720.6f, 717.1f, 689.6f, 653.5f, 615.8f, 579.8f, 546.6f, 516.7f, 490.f, 466.2f, 426.1f, 393.6f, 366.6f, 343.7f, 324.f, 306.7f, 291.5f, 278.f, 265.8f, 254.7f, 244.7f, 235.5f, 227.1f, 219.4f, 188.4f, 166.1f, 149.f, 135.4f }; 
 
  static const G4float e21[60] = { 34.67f, 42.46f, 49.03f, 54.82f, 60.05f, 69.34f, 77.52f, 84.92f, 91.73f, 98.06f, 104.f, 109.6f, 121.f, 130.9f, 139.9f, 148.f, 155.5f, 162.3f, 168.6f, 174.5f, 184.9f, 194.f, 201.8f, 208.5f, 214.2f, 219.f, 223.f, 226.3f, 228.9f, 231.f, 232.5f, 233.4f, 234.f, 234.1f, 230.4f, 221.9f, 211.4f, 200.6f, 190.1f, 180.6f, 171.9f, 164.1f, 151.1f, 140.6f, 132.1f, 125.1f, 119.1f, 114.f, 109.6f, 105.6f, 102.1f, 98.92f, 95.83f, 92.83f, 89.95f, 87.22f, 75.76f, 67.41f, 61.26f, 56.63f }; 
 
  static const G4float e22[60] = { 34.9f, 42.74f, 49.35f, 55.18f, 60.45f, 69.8f, 78.04f, 85.48f, 92.33f, 98.71f, 104.7f, 110.4f, 121.4f, 130.9f, 139.2f, 146.6f, 153.2f, 159.f, 164.3f, 168.9f, 176.9f, 183.2f, 188.4f, 192.5f, 195.9f, 198.7f, 201.f, 202.8f, 204.4f, 205.6f, 206.7f, 207.5f, 208.2f, 208.7f, 209.9f, 209.2f, 207.5f, 205.1f, 202.3f, 199.1f, 195.8f, 192.3f, 185.4f, 178.6f, 172.f, 165.8f, 159.6f, 153.6f, 148.f, 142.7f, 137.8f, 133.3f, 129.1f, 125.2f, 121.7f, 118.3f, 104.7f, 94.43f, 86.45f, 79.87f }; 
 
  static const G4float e23[60] = { 37.81f, 46.31f, 53.47f, 59.78f, 65.49f, 75.62f, 84.54f, 92.61f, 100.f, 106.9f, 113.4f, 119.6f, 132.f, 143.f, 153.f, 162.f, 170.4f, 178.2f, 185.4f, 192.2f, 204.6f, 215.6f, 225.4f, 234.2f, 242.1f, 249.2f, 255.5f, 261.2f, 266.2f, 270.7f, 274.6f, 278.1f, 281.f, 283.6f, 290.9f, 291.5f, 287.6f, 281.f, 272.8f, 263.9f, 254.9f, 246.f, 229.5f, 214.8f, 202.1f, 191.1f, 181.6f, 173.4f, 166.2f, 159.7f, 153.8f, 148.5f, 143.6f, 139.1f, 135.f, 131.1f, 115.3f, 103.5f, 94.02f, 86.52f }; 
 
  static const G4float e24[60] = { 135.1f, 165.5f, 191.1f, 213.6f, 234.f, 270.2f, 302.1f, 330.9f, 357.4f, 382.1f, 405.3f, 427.2f, 470.7f, 508.7f, 542.4f, 572.8f, 600.3f, 625.3f, 648.f, 668.8f, 705.f, 735.1f, 759.9f, 780.1f, 796.1f, 808.5f, 817.7f, 824.1f, 827.9f, 829.6f, 829.4f, 827.5f, 824.2f, 819.7f, 784.9f, 740.5f, 695.1f, 652.8f, 617.8f, 588.4f, 562.8f, 539.9f, 499.9f, 465.6f, 435.5f, 408.8f, 385.f, 364.1f, 345.6f, 329.1f, 314.3f, 300.9f, 288.8f, 277.7f, 267.6f, 258.3f, 220.8f, 193.9f, 173.4f, 157.2f }; 
 
  static const G4float e25[60] = { 25.71f, 31.49f, 36.37f, 40.66f, 44.54f, 51.43f, 57.5f, 62.99f, 68.03f, 72.73f, 77.14f, 81.32f, 89.78f, 97.24f, 104.f, 110.1f, 115.8f, 121.f, 125.8f, 130.4f, 138.5f, 145.7f, 152.1f, 157.7f, 162.6f, 166.9f, 170.7f, 173.9f, 176.7f, 179.f, 181.f, 182.5f, 183.8f, 184.7f, 185.6f, 182.1f, 176.3f, 169.4f, 162.2f, 155.2f, 148.7f, 142.6f, 132.f, 123.3f, 116.2f, 110.2f, 105.f, 100.6f, 96.67f, 93.2f, 90.08f, 87.27f, 84.72f, 82.38f, 80.24f, 78.26f, 70.14f, 63.87f, 58.87f, 54.76f }; 
 
  static const G4float e26[60] = { 45.91f, 56.23f, 64.93f, 72.59f, 79.52f, 91.82f, 102.7f, 112.5f, 121.5f, 129.9f, 137.7f, 145.2f, 159.7f, 172.3f, 183.3f, 193.1f, 201.7f, 209.4f, 216.3f, 222.3f, 232.3f, 240.f, 245.6f, 249.6f, 252.3f, 253.9f, 254.5f, 254.5f, 253.9f, 252.9f, 251.5f, 250.f, 248.2f, 246.4f, 236.5f, 227.2f, 218.9f, 211.6f, 205.1f, 199.3f, 193.9f, 189.f, 180.1f, 172.3f, 165.2f, 158.8f, 152.6f, 146.6f, 140.9f, 135.7f, 130.8f, 126.3f, 122.1f, 118.3f, 114.7f, 111.3f, 97.72f, 87.77f, 80.18f, 74.18f }; 
 
  static const G4float e27[60] = { 88.74f, 108.7f, 125.5f, 140.3f, 153.7f, 177.5f, 198.4f, 217.4f, 234.8f, 251.f, 266.2f, 280.6f, 308.9f, 333.6f, 355.6f, 375.3f, 393.1f, 409.2f, 423.9f, 437.3f, 460.8f, 480.4f, 496.7f, 510.2f, 521.2f, 530.1f, 537.1f, 542.5f, 546.4f, 549.1f, 550.6f, 551.2f, 551.f, 550.1f, 537.6f, 518.1f, 496.1f, 473.9f, 451.2f, 429.2f, 408.8f, 390.1f, 357.9f, 331.5f, 309.8f, 291.7f, 276.2f, 262.4f, 250.1f, 239.1f, 229.2f, 220.2f, 212.f, 204.6f, 197.7f, 191.4f, 165.6f, 146.7f, 132.1f, 120.4f }; 
 
  static const G4float e28[60] = { 746.5f, 914.3f, 1056.f, 1180.f, 1293.f, 1493.f, 1669.f, 1829.f, 1975.f, 2112.f, 2240.f, 2361.f, 2586.f, 2777.f, 2942.f, 3084.f, 3208.f, 3316.f, 3409.f, 3490.f, 3618.f, 3710.f, 3771.f, 3808.f, 3823.f, 3822.f, 3807.f, 3781.f, 3745.f, 3702.f, 3654.f, 3601.f, 3545.f, 3487.f, 3183.f, 2890.f, 2629.f, 2402.f, 2204.f, 2031.f, 1882.f, 1753.f, 1544.f, 1384.f, 1258.f, 1159.f, 1077.f, 1007.f, 947.f, 894.2f, 847.5f, 805.9f, 768.6f, 734.9f, 704.3f, 676.4f, 566.7f, 489.7f, 432.5f, 388.1f }; 
 
  static const G4float e29[60] = { 184.2f, 225.6f, 260.6f, 291.3f, 319.1f, 368.5f, 412.f, 451.3f, 487.4f, 521.1f, 552.7f, 582.6f, 641.3f, 691.9f, 736.7f, 776.6f, 812.5f, 844.9f, 874.2f, 900.8f, 946.6f, 984.1f, 1014.f, 1039.f, 1058.f, 1072.f, 1083.f, 1089.f, 1093.f, 1094.f, 1093.f, 1090.f, 1085.f, 1079.f, 1032.f, 973.f, 911.2f, 852.1f, 797.5f, 748.3f, 704.5f, 665.8f, 600.4f, 547.7f, 504.5f, 468.3f, 437.7f, 411.3f, 388.2f, 368.f, 349.9f, 333.8f, 319.3f, 306.2f, 294.2f, 283.2f, 239.6f, 208.6f, 185.2f, 167.f }; 
 
  static const G4float e30[60] = { 145.4f, 178.1f, 205.6f, 229.9f, 251.8f, 286.2f, 317.1f, 344.9f, 370.4f, 393.1f, 413.7f, 432.9f, 472.6f, 506.1f, 535.2f, 561.3f, 584.6f, 605.3f, 623.8f, 640.4f, 670.2f, 696.5f, 719.1f, 737.7f, 752.5f, 763.9f, 772.5f, 778.6f, 782.6f, 784.6f, 784.9f, 783.8f, 781.4f, 778.f, 749.2f, 710.6f, 669.6f, 630.f, 593.5f, 560.4f, 530.8f, 504.4f, 459.6f, 423.3f, 393.3f, 368.2f, 346.8f, 328.f, 311.6f, 296.9f, 283.8f, 272.f, 261.3f, 251.5f, 242.4f, 234.1f, 200.8f, 176.8f, 158.4f, 143.8f }; 
 
  static const G4float e31[60] = { 45.92f, 56.24f, 64.94f, 72.6f, 79.53f, 91.84f, 102.7f, 112.5f, 121.5f, 129.9f, 137.8f, 145.2f, 160.1f, 173.2f, 184.9f, 195.5f, 205.1f, 213.8f, 221.8f, 229.1f, 242.f, 252.7f, 261.6f, 268.8f, 274.6f, 279.f, 282.3f, 284.5f, 285.8f, 286.2f, 286.f, 285.1f, 283.8f, 281.9f, 268.6f, 252.2f, 236.2f, 221.7f, 209.f, 198.2f, 188.9f, 180.9f, 168.1f, 158.1f, 150.2f, 143.7f, 138.1f, 133.2f, 128.8f, 124.8f, 121.2f, 117.8f, 114.6f, 111.6f, 108.8f, 106.1f, 94.63f, 85.43f, 77.91f, 71.84f }; 
 
  static const G4float e32[60] = { 111.f, 135.9f, 157.f, 175.5f, 192.2f, 222.f, 248.2f, 271.9f, 293.6f, 313.9f, 333.f, 351.f, 386.5f, 417.4f, 444.8f, 469.4f, 491.6f, 511.7f, 530.1f, 546.7f, 575.8f, 599.9f, 619.8f, 636.f, 649.1f, 659.3f, 667.2f, 672.9f, 676.7f, 678.9f, 679.6f, 679.1f, 677.6f, 675.1f, 653.f, 622.4f, 589.5f, 557.3f, 525.5f, 495.6f, 468.4f, 444.1f, 403.1f, 370.6f, 344.6f, 323.5f, 305.9f, 290.3f, 276.4f, 263.9f, 252.7f, 242.4f, 233.2f, 224.7f, 216.9f, 209.6f, 180.4f, 159.2f, 142.8f, 129.9f }; 
 
  static const G4float e33[60] = { 80.87f, 99.04f, 114.4f, 127.9f, 140.1f, 161.7f, 180.8f, 198.1f, 214.f, 228.7f, 242.6f, 255.7f, 281.9f, 304.9f, 325.4f, 344.f, 360.9f, 376.5f, 390.8f, 404.f, 427.6f, 447.8f, 465.3f, 480.3f, 493.1f, 504.1f, 513.3f, 521.1f, 527.4f, 532.5f, 536.5f, 539.6f, 541.7f, 543.f, 540.3f, 527.4f, 509.2f, 488.8f, 467.4f, 446.5f, 426.9f, 408.7f, 376.9f, 350.4f, 328.f, 309.1f, 292.7f, 278.1f, 265.1f, 253.3f, 242.5f, 232.8f, 223.9f, 215.6f, 208.2f, 201.2f, 173.1f, 152.5f, 136.8f, 124.4f }; 
 
  static const G4float e34[60] = { 198.1f, 242.6f, 280.2f, 313.3f, 343.1f, 391.9f, 435.4f, 474.6f, 510.6f, 543.1f, 573.f, 601.f, 656.7f, 703.7f, 744.4f, 780.4f, 812.2f, 840.3f, 865.f, 887.f, 925.f, 956.5f, 982.1f, 1002.f, 1016.f, 1026.f, 1032.f, 1035.f, 1036.f, 1033.f, 1029.f, 1023.f, 1016.f, 1007.f, 953.3f, 891.9f, 831.3f, 775.4f, 725.1f, 680.6f, 641.3f, 606.6f, 546.6f, 497.8f, 458.8f, 427.4f, 401.7f, 379.2f, 359.6f, 342.2f, 326.7f, 312.7f, 300.1f, 288.5f, 277.8f, 268.1f, 229.1f, 201.f, 179.7f, 162.8f }; 
 
  static const G4float e35[60] = { 174.4f, 213.6f, 246.7f, 275.8f, 302.1f, 345.f, 383.2f, 417.7f, 449.4f, 477.9f, 504.3f, 528.8f, 578.1f, 619.8f, 656.f, 688.2f, 716.7f, 742.f, 764.4f, 784.4f, 819.3f, 848.7f, 872.8f, 892.f, 906.4f, 916.8f, 923.7f, 927.8f, 929.4f, 928.8f, 926.4f, 922.3f, 916.9f, 910.4f, 866.6f, 814.5f, 762.f, 712.8f, 668.3f, 628.6f, 593.3f, 562.1f, 507.7f, 463.3f, 427.6f, 398.9f, 375.3f, 354.7f, 336.7f, 320.6f, 306.2f, 293.3f, 281.6f, 270.9f, 261.f, 252.f, 215.7f, 189.5f, 169.6f, 153.9f }; 
 
  static const G4float e36[60] = { 272.8f, 334.1f, 385.8f, 431.4f, 472.5f, 545.6f, 610.f, 668.2f, 721.8f, 771.6f, 818.4f, 862.7f, 949.2f, 1024.f, 1090.f, 1148.f, 1200.f, 1246.f, 1288.f, 1325.f, 1387.f, 1436.f, 1473.f, 1500.f, 1518.f, 1528.f, 1532.f, 1530.f, 1524.f, 1514.f, 1501.f, 1485.f, 1467.f, 1447.f, 1338.f, 1227.f, 1127.f, 1041.f, 967.3f, 904.6f, 850.4f, 803.2f, 724.6f, 661.8f, 610.5f, 567.7f, 531.4f, 499.8f, 472.5f, 448.3f, 426.7f, 407.4f, 390.f, 374.1f, 359.6f, 346.4f, 293.7f, 256.2f, 227.9f, 205.7f }; 
 
  static const G4float e37[60] = { 40.18f, 49.21f, 56.82f, 63.53f, 69.59f, 80.35f, 89.84f, 98.41f, 106.3f, 113.6f, 120.5f, 127.f, 140.2f, 151.7f, 162.1f, 171.5f, 180.1f, 188.f, 195.4f, 202.1f, 214.3f, 224.8f, 233.8f, 241.6f, 248.3f, 253.9f, 258.7f, 262.6f, 265.7f, 268.2f, 270.f, 271.3f, 272.1f, 272.4f, 268.7f, 259.5f, 248.f, 235.8f, 224.f, 213.1f, 203.1f, 194.1f, 178.9f, 166.6f, 156.5f, 148.1f, 141.f, 134.9f, 129.5f, 124.7f, 120.4f, 116.4f, 112.6f, 109.1f, 105.8f, 102.6f, 89.8f, 80.34f, 73.15f, 67.51f }; 
 
  static const G4float e38[60] = { 149.4f, 183.f, 211.3f, 236.2f, 258.7f, 298.3f, 333.3f, 364.8f, 393.8f, 420.8f, 446.f, 469.8f, 516.6f, 557.2f, 593.1f, 625.2f, 654.f, 680.f, 703.5f, 724.8f, 761.7f, 792.f, 816.7f, 836.5f, 851.9f, 863.6f, 871.9f, 877.4f, 880.4f, 881.2f, 880.1f, 877.3f, 873.2f, 867.9f, 829.5f, 781.9f, 733.4f, 688.f, 647.9f, 612.8f, 581.8f, 554.2f, 507.1f, 468.4f, 435.9f, 408.3f, 384.4f, 363.5f, 344.9f, 328.4f, 313.7f, 300.3f, 288.2f, 277.2f, 267.f, 257.7f, 220.4f, 193.5f, 173.f, 156.9f }; 
 
  static const G4float e39[60] = { 146.6f, 179.5f, 207.3f, 231.8f, 253.9f, 292.9f, 327.3f, 358.4f, 386.9f, 413.5f, 438.4f, 461.9f, 508.2f, 548.4f, 584.f, 615.8f, 644.5f, 670.4f, 693.8f, 715.1f, 752.f, 782.3f, 807.1f, 827.f, 842.5f, 854.3f, 862.9f, 868.5f, 871.6f, 872.6f, 871.6f, 869.f, 865.f, 859.8f, 822.f, 774.9f, 726.9f, 682.f, 643.f, 609.2f, 579.4f, 552.9f, 507.4f, 469.6f, 437.7f, 410.1f, 386.1f, 365.1f, 346.5f, 329.9f, 315.f, 301.6f, 289.4f, 278.3f, 268.2f, 258.8f, 221.3f, 194.3f, 173.7f, 157.5f }; 
 
  static const G4float e40[60] = { 141.3f, 173.1f, 199.9f, 223.5f, 244.8f, 282.5f, 315.8f, 345.8f, 373.5f, 399.2f, 423.3f, 446.1f, 491.1f, 530.2f, 565.f, 596.1f, 624.2f, 649.7f, 672.8f, 693.8f, 730.4f, 760.6f, 785.3f, 805.3f, 821.1f, 833.1f, 841.9f, 847.9f, 851.4f, 852.6f, 852.f, 849.7f, 846.1f, 841.2f, 804.8f, 758.9f, 712.2f, 668.5f, 631.2f, 599.f, 570.8f, 545.6f, 502.2f, 465.8f, 434.6f, 407.5f, 383.7f, 362.8f, 344.4f, 327.9f, 313.1f, 299.8f, 287.7f, 276.7f, 266.6f, 257.3f, 220.f, 193.2f, 172.8f, 156.7f }; 
 
  static const G4float e41[60] = { 141.7f, 173.5f, 200.4f, 224.f, 245.4f, 283.2f, 316.5f, 346.6f, 374.3f, 400.f, 424.2f, 447.f, 492.1f, 531.3f, 566.1f, 597.3f, 625.4f, 650.9f, 674.f, 695.1f, 731.6f, 761.9f, 786.7f, 806.6f, 822.4f, 834.4f, 843.3f, 849.2f, 852.6f, 853.9f, 853.2f, 851.f, 847.3f, 842.4f, 805.8f, 759.8f, 713.f, 669.2f, 632.f, 600.1f, 572.1f, 547.1f, 504.f, 467.7f, 436.5f, 409.3f, 385.3f, 364.3f, 345.7f, 329.2f, 314.4f, 300.9f, 288.8f, 277.8f, 267.6f, 258.3f, 220.8f, 193.9f, 173.4f, 157.2f }; 
 
  static const G4float e42[60] = { 126.5f, 155.f, 179.f, 200.1f, 219.2f, 253.1f, 283.f, 310.f, 334.8f, 357.9f, 379.6f, 400.2f, 440.6f, 475.8f, 507.f, 535.f, 560.2f, 583.1f, 603.8f, 622.7f, 655.5f, 682.7f, 704.9f, 722.9f, 737.3f, 748.4f, 756.6f, 762.3f, 765.9f, 767.5f, 767.5f, 766.f, 763.3f, 759.4f, 729.3f, 689.8f, 648.6f, 609.1f, 572.9f, 540.4f, 511.6f, 486.2f, 443.4f, 408.8f, 380.1f, 356.f, 335.4f, 317.2f, 301.2f, 286.8f, 274.1f, 262.6f, 252.2f, 242.7f, 234.f, 225.9f, 193.8f, 170.5f, 152.8f, 138.8f }; 
 
  static const G4float e43[60] = { 41.1f, 50.34f, 58.12f, 64.98f, 71.19f, 82.2f, 91.9f, 100.7f, 108.7f, 116.2f, 123.3f, 130.f, 143.4f, 155.2f, 165.8f, 175.4f, 184.2f, 192.3f, 199.7f, 206.6f, 219.f, 229.6f, 238.9f, 246.8f, 253.5f, 259.2f, 264.f, 268.f, 271.1f, 273.6f, 275.4f, 276.7f, 277.4f, 277.7f, 273.7f, 264.2f, 252.3f, 239.7f, 227.6f, 216.2f, 205.9f, 196.6f, 180.7f, 167.8f, 157.3f, 148.5f, 141.1f, 134.7f, 129.1f, 124.1f, 119.7f, 115.7f, 111.9f, 108.4f, 105.1f, 101.9f, 88.79f, 78.99f, 71.48f, 65.54f }; 
 
  static const G4float e44[60] = { 58.01f, 71.05f, 82.04f, 91.73f, 100.5f, 116.f, 129.7f, 142.1f, 153.5f, 164.1f, 174.f, 183.5f, 202.5f, 219.3f, 234.5f, 248.3f, 261.f, 272.8f, 283.8f, 294.f, 312.7f, 329.2f, 343.8f, 356.9f, 368.6f, 379.f, 388.3f, 396.5f, 403.8f, 410.2f, 415.8f, 420.7f, 424.8f, 428.4f, 438.1f, 437.9f, 431.3f, 420.8f, 408.2f, 394.6f, 380.7f, 367.1f, 339.f, 313.8f, 293.3f, 277.2f, 264.3f, 252.6f, 241.9f, 232.2f, 223.2f, 215.f, 207.4f, 200.4f, 193.8f, 187.7f, 163.f, 144.6f, 130.3f, 118.9f }; 
 
  static const G4float e45[60] = { 191.f, 233.9f, 270.1f, 302.f, 330.8f, 377.8f, 419.7f, 457.5f, 492.1f, 523.4f, 552.3f, 579.2f, 633.f, 678.5f, 717.9f, 752.9f, 783.8f, 811.1f, 835.3f, 856.8f, 894.2f, 925.4f, 951.f, 971.f, 985.8f, 996.3f, 1003.f, 1007.f, 1008.f, 1006.f, 1003.f, 997.7f, 991.2f, 983.5f, 933.5f, 875.4f, 817.5f, 763.6f, 715.f, 671.7f, 633.5f, 599.6f, 542.9f, 497.4f, 460.4f, 429.7f, 403.7f, 381.f, 361.2f, 343.6f, 327.9f, 313.8f, 301.f, 289.3f, 278.6f, 268.8f, 229.5f, 201.3f, 179.8f, 162.9f }; 
 
  static const G4float e46[60] = { 99.46f, 121.8f, 140.7f, 157.3f, 172.3f, 198.9f, 222.4f, 243.6f, 263.1f, 281.3f, 298.4f, 314.5f, 346.6f, 374.7f, 399.8f, 422.4f, 443.f, 461.9f, 479.1f, 495.f, 523.1f, 547.f, 567.2f, 584.4f, 598.7f, 610.6f, 620.2f, 628.f, 633.9f, 638.3f, 641.3f, 643.f, 643.6f, 643.3f, 630.5f, 606.7f, 578.3f, 549.1f, 520.9f, 494.6f, 470.9f, 449.7f, 413.2f, 383.1f, 357.8f, 336.3f, 317.6f, 301.1f, 286.3f, 273.2f, 261.3f, 250.5f, 240.8f, 231.9f, 223.7f, 216.1f, 185.7f, 163.7f, 146.7f, 133.4f }; 
 
  static const G4float e47[60] = { 228.8f, 280.2f, 323.5f, 361.7f, 396.2f, 451.9f, 501.6f, 546.5f, 587.6f, 624.6f, 658.6f, 690.2f, 753.4f, 806.4f, 852.1f, 892.4f, 927.9f, 959.1f, 986.3f, 1010.f, 1052.f, 1086.f, 1114.f, 1135.f, 1150.f, 1160.f, 1166.f, 1168.f, 1166.f, 1163.f, 1157.f, 1149.f, 1140.f, 1129.f, 1064.f, 992.4f, 922.6f, 858.6f, 801.5f, 751.1f, 706.8f, 667.8f, 601.8f, 548.9f, 506.2f, 471.3f, 442.2f, 416.9f, 394.9f, 375.4f, 358.f, 342.3f, 328.2f, 315.3f, 303.4f, 292.6f, 249.3f, 218.3f, 194.8f, 176.3f }; 
 
  static const G4float e48[60] = { 15.4f, 18.86f, 21.78f, 24.35f, 26.68f, 30.81f, 34.44f, 37.73f, 40.75f, 43.57f, 46.21f, 48.71f, 53.82f, 58.34f, 62.45f, 66.21f, 69.7f, 72.94f, 75.98f, 78.85f, 84.11f, 88.84f, 93.13f, 97.02f, 100.6f, 103.8f, 106.8f, 109.5f, 111.9f, 114.1f, 116.2f, 118.f, 119.6f, 121.1f, 126.1f, 128.1f, 127.9f, 126.3f, 123.6f, 120.4f, 116.9f, 113.3f, 106.4f, 99.98f, 94.37f, 89.5f, 85.18f, 81.37f, 78.01f, 75.05f, 72.42f, 70.08f, 67.99f, 66.12f, 64.44f, 62.92f, 56.88f, 52.22f, 48.46f, 45.34f }; 
 
  static const G4float e49[60] = { 61.33f, 75.12f, 86.74f, 96.97f, 106.2f, 122.2f, 136.3f, 149.1f, 160.8f, 171.6f, 181.6f, 191.1f, 210.f, 226.3f, 240.6f, 253.5f, 265.f, 275.4f, 284.8f, 293.3f, 308.f, 320.3f, 330.3f, 338.4f, 344.7f, 349.4f, 352.8f, 355.1f, 356.4f, 356.9f, 356.6f, 355.7f, 354.3f, 352.4f, 338.9f, 322.3f, 305.5f, 289.9f, 275.7f, 263.f, 251.7f, 241.5f, 224.4f, 210.3f, 198.3f, 188.f, 179.f, 171.f, 163.9f, 157.5f, 151.6f, 146.3f, 141.3f, 136.6f, 132.1f, 128.f, 110.6f, 97.75f, 87.92f, 80.2f }; 
 
  static const G4float e50[60] = { 190.1f, 232.8f, 268.8f, 300.5f, 329.2f, 374.1f, 414.4f, 450.7f, 483.9f, 513.5f, 540.4f, 565.4f, 616.5f, 659.2f, 696.f, 728.7f, 757.6f, 783.f, 805.3f, 825.2f, 860.1f, 890.2f, 915.4f, 935.4f, 950.4f, 961.1f, 968.2f, 972.3f, 973.8f, 973.f, 970.2f, 965.7f, 959.9f, 952.8f, 906.f, 850.6f, 795.1f, 743.3f, 696.4f, 654.7f, 617.9f, 585.3f, 530.7f, 487.1f, 451.3f, 421.4f, 395.9f, 373.7f, 354.4f, 337.2f, 321.9f, 308.1f, 295.7f, 284.2f, 273.8f, 264.1f, 225.7f, 198.1f, 177.1f, 160.5f }; 
 
  static const G4float e51[60] = { 164.8f, 201.8f, 233.f, 260.5f, 285.4f, 324.6f, 359.7f, 391.4f, 420.3f, 446.2f, 469.7f, 491.7f, 536.5f, 574.3f, 607.1f, 636.2f, 662.2f, 685.1f, 705.4f, 723.6f, 755.9f, 783.9f, 807.7f, 826.9f, 841.7f, 852.8f, 860.8f, 866.f, 868.8f, 869.6f, 868.6f, 866.f, 862.1f, 857.1f, 820.5f, 774.5f, 727.2f, 682.2f, 641.1f, 604.1f, 571.2f, 542.f, 492.3f, 452.1f, 419.1f, 391.8f, 368.6f, 348.4f, 330.7f, 314.9f, 300.8f, 288.1f, 276.7f, 266.1f, 256.5f, 247.6f, 212.f, 186.3f, 166.8f, 151.3f }; 
 
  static const G4float e52[60] = { 184.9f, 226.5f, 261.5f, 292.4f, 320.3f, 369.9f, 413.5f, 453.f, 489.3f, 523.f, 554.8f, 584.8f, 643.9f, 695.3f, 740.8f, 781.4f, 817.9f, 850.9f, 880.6f, 907.5f, 953.7f, 990.9f, 1021.f, 1044.f, 1061.f, 1073.f, 1081.f, 1084.f, 1085.f, 1083.f, 1078.f, 1072.f, 1064.f, 1054.f, 994.1f, 927.8f, 865.f, 808.9f, 760.1f, 717.9f, 681.4f, 649.7f, 591.2f, 541.f, 499.2f, 464.9f, 436.3f, 411.4f, 389.8f, 370.6f, 353.4f, 338.1f, 324.2f, 311.4f, 299.8f, 289.1f, 246.4f, 215.9f, 192.7f, 174.4f }; 
 
  static const G4float e53[60] = { 150.9f, 184.9f, 213.5f, 238.7f, 261.4f, 297.8f, 330.2f, 359.6f, 386.4f, 410.5f, 432.5f, 453.f, 494.9f, 530.3f, 561.2f, 588.8f, 613.4f, 635.3f, 654.7f, 672.3f, 703.4f, 730.5f, 753.5f, 772.3f, 787.f, 798.3f, 806.6f, 812.4f, 816.f, 817.6f, 817.4f, 815.8f, 812.9f, 808.9f, 777.6f, 736.5f, 693.4f, 651.9f, 613.6f, 579.1f, 548.2f, 520.6f, 472.9f, 434.f, 402.2f, 376.1f, 354.3f, 335.f, 318.2f, 303.2f, 289.8f, 277.7f, 266.7f, 256.6f, 247.4f, 238.9f, 204.8f, 180.2f, 161.4f, 146.5f }; 
 
  static const G4float e54[60] = { 174.9f, 214.2f, 247.4f, 276.6f, 303.f, 345.9f, 384.1f, 418.7f, 450.4f, 479.f, 505.2f, 529.8f, 579.1f, 620.8f, 657.1f, 689.3f, 717.9f, 743.2f, 765.7f, 785.8f, 820.8f, 850.5f, 875.f, 894.5f, 909.3f, 920.1f, 927.4f, 931.9f, 933.9f, 933.7f, 931.6f, 928.f, 922.9f, 916.8f, 874.4f, 823.2f, 771.3f, 722.4f, 676.2f, 634.f, 596.2f, 562.8f, 507.2f, 463.4f, 428.5f, 400.2f, 376.7f, 356.1f, 338.f, 321.9f, 307.5f, 294.5f, 282.8f, 272.f, 262.1f, 253.f, 216.6f, 190.4f, 170.3f, 154.5f }; 
 
  static const G4float e55[60] = { 197.6f, 242.f, 279.4f, 312.4f, 342.2f, 395.1f, 441.8f, 483.9f, 522.7f, 558.8f, 592.7f, 624.8f, 687.3f, 741.3f, 789.f, 831.4f, 869.4f, 903.6f, 934.3f, 962.f, 1009.f, 1048.f, 1078.f, 1101.f, 1119.f, 1132.f, 1140.f, 1145.f, 1146.f, 1145.f, 1141.f, 1135.f, 1128.f, 1119.f, 1060.f, 991.7f, 924.3f, 862.f, 806.1f, 756.5f, 712.8f, 674.2f, 605.4f, 548.6f, 503.3f, 467.5f, 438.7f, 413.6f, 391.8f, 372.4f, 355.2f, 339.7f, 325.7f, 312.9f, 301.1f, 290.4f, 247.5f, 216.8f, 193.5f, 175.1f }; 
 
  static const G4float e56[60] = { 186.f, 227.8f, 263.1f, 294.1f, 322.2f, 372.1f, 416.f, 455.7f, 492.2f, 526.2f, 558.1f, 588.3f, 645.3f, 693.8f, 735.8f, 772.4f, 804.5f, 832.6f, 857.2f, 878.7f, 913.5f, 939.2f, 957.2f, 969.f, 975.6f, 978.f, 976.8f, 972.8f, 966.5f, 958.4f, 948.9f, 938.3f, 926.9f, 914.9f, 851.3f, 789.6f, 734.2f, 685.4f, 642.9f, 605.7f, 573.f, 543.9f, 499.5f, 465.2f, 435.8f, 409.1f, 384.6f, 363.2f, 344.6f, 328.1f, 313.3f, 300.f, 288.f, 277.f, 266.8f, 257.5f, 220.4f, 193.6f, 173.2f, 157.f }; 
 
  static const G4float e57[60] = { 83.27f, 102.f, 117.8f, 131.7f, 144.2f, 165.f, 183.4f, 200.1f, 215.3f, 229.2f, 242.f, 254.f, 278.7f, 300.f, 319.f, 336.2f, 351.9f, 366.1f, 379.2f, 391.2f, 413.1f, 432.7f, 450.f, 465.1f, 478.f, 489.1f, 498.5f, 506.4f, 513.f, 518.4f, 522.7f, 526.f, 528.4f, 530.1f, 528.6f, 516.7f, 499.4f, 479.6f, 460.4f, 442.3f, 425.4f, 409.6f, 381.2f, 356.4f, 334.6f, 315.4f, 298.3f, 283.1f, 269.6f, 257.4f, 246.5f, 236.5f, 227.4f, 219.1f, 211.5f, 204.4f, 175.9f, 155.2f, 139.3f, 126.7f }; 
 
  static const G4float e58[60] = { 140.2f, 171.7f, 198.3f, 221.7f, 242.8f, 277.9f, 309.f, 337.2f, 363.f, 386.4f, 408.1f, 428.4f, 468.9f, 503.4f, 533.6f, 560.3f, 584.1f, 605.3f, 624.1f, 640.9f, 670.1f, 694.5f, 714.3f, 729.8f, 741.4f, 749.6f, 755.f, 757.9f, 758.6f, 757.6f, 755.f, 751.f, 746.f, 740.f, 701.3f, 656.7f, 612.9f, 572.6f, 538.7f, 509.8f, 484.7f, 462.4f, 424.2f, 392.4f, 365.3f, 341.8f, 321.1f, 302.9f, 287.3f, 273.6f, 261.3f, 250.1f, 240.1f, 231.1f, 222.7f, 215.f, 184.5f, 162.6f, 145.9f, 132.6f }; 
 
  static const G4float e59[60] = { 2.183E+02f, 2.673E+02f, 3.087E+02f, 3.451E+02f, 3.780E+02f, 4.365E+02f, 4.880E+02f, 5.346E+02f, 5.775E+02f, 6.173E+02f, 6.548E+02f, 6.902E+02f, 7.595E+02f, 8.190E+02f, 8.714E+02f, 9.180E+02f, 9.597E+02f, 9.970E+02f, 1.031E+03f, 1.061E+03f, 1.112E+03f, 1.153E+03f, 1.186E+03f, 1.211E+03f, 1.229E+03f, 1.242E+03f, 1.250E+03f, 1.254E+03f, 1.254E+03f, 1.252E+03f, 1.247E+03f, 1.239E+03f, 1.230E+03f, 1.220E+03f, 1.153E+03f, 1.078E+03f, 1.005E+03f, 9.379E+02f, 8.769E+02f, 8.222E+02f, 7.736E+02f, 7.305E+02f, 6.583E+02f, 6.006E+02f, 5.537E+02f, 5.150E+02f, 4.825E+02f, 4.542E+02f, 4.297E+02f, 4.079E+02f, 3.886E+02f, 3.712E+02f, 3.556E+02f, 3.413E+02f, 3.282E+02f, 3.163E+02f, 2.687E+02f, 2.348E+02f, 2.092E+02f, 1.890E+02f };

  static const G4float e60[60] =  { 1.377E+01f, 1.687E+01f, 1.947E+01f, 2.177E+01f, 2.385E+01f, 2.754E+01f, 3.079E+01f, 3.373E+01f, 3.643E+01f, 3.895E+01f, 4.131E+01f, 4.355E+01f, 4.811E+01f, 5.216E+01f, 5.583E+01f, 5.920E+01f, 6.232E+01f, 6.523E+01f, 6.795E+01f, 7.051E+01f, 7.522E+01f, 7.946E+01f, 8.330E+01f, 8.679E+01f, 8.997E+01f, 9.287E+01f, 9.551E+01f, 9.792E+01f, 1.001E+02f, 1.021E+02f, 1.039E+02f, 1.055E+02f, 1.070E+02f, 1.083E+02f, 1.127E+02f, 1.145E+02f, 1.144E+02f, 1.130E+02f, 1.108E+02f, 1.081E+02f, 1.053E+02f, 1.024E+02f, 9.692E+01f, 9.212E+01f, 8.791E+01f, 8.422E+01f, 8.097E+01f, 7.811E+01f, 7.556E+01f, 7.326E+01f, 7.117E+01f, 6.926E+01f, 6.750E+01f, 6.587E+01f, 6.436E+01f, 6.294E+01f, 5.700E+01f, 5.239E+01f, 4.867E+01f, 4.557E+01f };

  static const G4float e61[60] = { 105.f, 128.6f, 148.5f, 166.f, 181.8f, 210.f, 234.8f, 257.2f, 277.8f, 296.9f, 315.f, 332.f, 364.6f, 392.5f, 416.7f, 437.8f, 456.3f, 472.5f, 486.6f, 498.9f, 518.4f, 532.3f, 541.6f, 547.f, 549.2f, 549.f, 546.8f, 543.f, 538.f, 532.2f, 525.7f, 518.8f, 511.7f, 504.4f, 468.7f, 437.3f, 410.7f, 388.4f, 369.4f, 353.f, 338.7f, 325.9f, 304.1f, 285.9f, 270.3f, 256.7f, 245.2f, 235.1f, 225.9f, 217.2f, 209.f, 201.1f, 193.6f, 187.f, 180.9f, 175.3f, 152.3f, 135.4f, 122.2f, 111.7f }; 
 
  static const G4float e62[60] = { 85.64f, 104.9f, 121.1f, 135.4f, 148.3f, 171.3f, 191.5f, 209.8f, 226.6f, 242.2f, 256.9f, 270.8f, 298.1f, 322.f, 343.1f, 362.1f, 379.3f, 394.9f, 409.1f, 422.1f, 444.8f, 463.8f, 479.7f, 492.8f, 503.6f, 512.3f, 519.2f, 524.5f, 528.5f, 531.2f, 532.9f, 533.7f, 533.6f, 532.9f, 521.9f, 503.9f, 483.5f, 462.8f, 442.9f, 424.2f, 406.7f, 390.6f, 358.9f, 330.8f, 307.6f, 288.9f, 273.6f, 259.9f, 247.8f, 236.9f, 227.1f, 218.2f, 210.1f, 202.7f, 195.9f, 189.7f, 164.2f, 145.5f, 131.1f, 119.6f }; 
 
  static const G4float e63[60] = { 179.2f, 219.5f, 253.4f, 283.3f, 310.4f, 352.2f, 389.8f, 423.8f, 454.7f, 482.3f, 507.2f, 530.3f, 577.9f, 617.7f, 652.1f, 682.6f, 709.7f, 733.5f, 754.4f, 773.f, 806.1f, 835.1f, 859.5f, 879.2f, 894.2f, 905.1f, 912.7f, 917.4f, 919.6f, 919.6f, 917.7f, 914.2f, 909.3f, 903.3f, 861.4f, 810.7f, 759.2f, 710.8f, 667.1f, 628.3f, 593.9f, 563.4f, 511.8f, 470.2f, 435.9f, 407.2f, 382.8f, 361.4f, 342.9f, 326.4f, 311.6f, 298.4f, 286.4f, 275.4f, 265.3f, 256.1f, 219.f, 192.4f, 172.1f, 156.1f }; 
 
  static const G4float e64[60] = { 60.88f, 74.56f, 86.1f, 96.26f, 105.4f, 121.8f, 136.1f, 149.1f, 161.1f, 172.2f, 182.6f, 192.5f, 212.3f, 229.6f, 245.1f, 259.1f, 271.8f, 283.5f, 294.2f, 304.1f, 321.6f, 336.5f, 349.1f, 359.8f, 368.7f, 376.f, 382.f, 386.7f, 390.2f, 392.7f, 394.4f, 395.2f, 395.3f, 394.8f, 385.4f, 369.3f, 351.f, 332.6f, 315.4f, 299.6f, 285.5f, 272.8f, 251.3f, 233.9f, 219.6f, 207.6f, 197.4f, 188.7f, 181.f, 174.1f, 167.9f, 162.2f, 157.f, 152.2f, 147.7f, 143.5f, 126.f, 112.5f, 101.7f, 93.07f }; 
 
  static const G4float e65[60] = { 32.4f, 39.68f, 45.82f, 51.23f, 56.12f, 64.8f, 72.45f, 79.36f, 85.72f, 91.64f, 97.2f, 102.5f, 113.1f, 122.4f, 130.8f, 138.5f, 145.5f, 151.9f, 157.9f, 163.4f, 173.4f, 182.f, 189.5f, 196.f, 201.6f, 206.4f, 210.4f, 213.8f, 216.5f, 218.7f, 220.4f, 221.6f, 222.4f, 222.9f, 220.7f, 213.8f, 204.9f, 195.4f, 186.f, 177.3f, 169.4f, 162.3f, 150.1f, 140.3f, 132.2f, 125.5f, 119.7f, 114.7f, 110.1f, 106.1f, 102.5f, 99.14f, 96.12f, 93.34f, 90.77f, 88.4f, 78.69f, 71.47f, 65.7f, 60.79f }; 
 
  static const G4float e66[60] = { 174.5f, 213.8f, 246.8f, 276.f, 302.3f, 349.1f, 390.3f, 427.6f, 461.8f, 493.7f, 523.6f, 552.f, 607.8f, 656.2f, 699.1f, 737.4f, 771.9f, 803.f, 831.1f, 856.5f, 900.2f, 935.6f, 963.8f, 985.9f, 1003.f, 1015.f, 1022.f, 1027.f, 1028.f, 1027.f, 1023.f, 1018.f, 1011.f, 1002.f, 947.8f, 885.5f, 824.9f, 769.7f, 720.9f, 678.2f, 641.f, 608.6f, 553.2f, 507.9f, 470.5f, 439.1f, 412.4f, 389.f, 368.6f, 350.5f, 334.4f, 319.8f, 306.6f, 294.6f, 283.6f, 273.5f, 233.2f, 204.4f, 182.5f, 165.2f }; 
 
  static const G4float e67[60] = { 169.9f, 208.1f, 240.3f, 268.6f, 294.3f, 339.8f, 379.9f, 416.2f, 449.5f, 480.6f, 509.7f, 537.3f, 591.5f, 638.4f, 679.9f, 716.9f, 750.2f, 780.2f, 807.4f, 832.f, 874.4f, 909.1f, 937.1f, 959.3f, 976.6f, 989.5f, 998.7f, 1005.f, 1008.f, 1008.f, 1006.f, 1003.f, 998.f, 991.7f, 946.9f, 891.9f, 836.f, 783.4f, 735.7f, 693.2f, 655.4f, 622.f, 563.5f, 515.3f, 475.8f, 443.4f, 416.2f, 392.6f, 371.9f, 353.6f, 337.3f, 322.5f, 309.3f, 297.1f, 286.f, 275.8f, 235.1f, 206.f, 183.9f, 166.5f }; 
 
  static const G4float e68[60] = { 191.6f, 234.7f, 271.f, 302.9f, 331.9f, 377.2f, 417.8f, 454.5f, 488.f, 517.9f, 545.1f, 570.4f, 621.9f, 665.f, 702.2f, 735.2f, 764.3f, 790.f, 812.4f, 832.5f, 867.7f, 897.9f, 923.2f, 943.3f, 958.3f, 969.f, 976.f, 980.f, 981.4f, 980.4f, 977.5f, 972.9f, 966.9f, 959.8f, 912.2f, 856.2f, 800.1f, 747.8f, 700.6f, 658.5f, 621.4f, 588.5f, 535.1f, 492.8f, 457.8f, 427.9f, 401.8f, 379.1f, 359.4f, 341.9f, 326.3f, 312.3f, 299.6f, 287.9f, 277.3f, 267.5f, 228.5f, 200.4f, 179.1f, 162.3f }; 
 
  static const G4float e69[60] = { 18.38f, 22.51f, 25.99f, 29.06f, 31.83f, 36.75f, 41.09f, 45.01f, 48.62f, 51.98f, 55.13f, 58.11f, 64.19f, 69.57f, 74.44f, 78.9f, 83.02f, 86.84f, 90.42f, 93.78f, 99.92f, 105.4f, 110.3f, 114.8f, 118.8f, 122.4f, 125.7f, 128.6f, 131.2f, 133.5f, 135.6f, 137.4f, 139.f, 140.4f, 144.4f, 144.8f, 142.7f, 139.1f, 134.5f, 129.6f, 124.2f, 118.8f, 108.7f, 100.f, 92.78f, 86.72f, 81.64f, 77.35f, 73.71f, 70.6f, 67.92f, 65.59f, 63.56f, 61.77f, 60.19f, 58.78f, 53.33f, 49.08f, 45.62f, 42.73f }; 
 
  static const G4float e70[60] = { 14.97f, 18.34f, 21.17f, 23.67f, 25.93f, 29.94f, 33.47f, 36.67f, 39.61f, 42.34f, 44.91f, 47.34f, 52.3f, 56.7f, 60.68f, 64.33f, 67.7f, 70.84f, 73.78f, 76.55f, 81.61f, 86.15f, 90.24f, 93.95f, 97.3f, 100.3f, 103.1f, 105.6f, 107.8f, 109.9f, 111.7f, 113.3f, 114.7f, 116.f, 120.f, 121.2f, 120.5f, 118.5f, 115.9f, 112.9f, 109.8f, 106.7f, 100.2f, 94.14f, 89.08f, 85.f, 81.68f, 78.78f, 76.2f, 73.88f, 71.78f, 69.86f, 68.09f, 66.45f, 64.93f, 63.5f, 57.52f, 52.9f, 49.15f, 46.04f }; 
 
  static const G4float e71[60] = { 133.7f, 163.8f, 189.1f, 211.4f, 231.6f, 267.5f, 299.f, 327.6f, 353.8f, 378.2f, 401.2f, 422.9f, 466.f, 503.6f, 537.2f, 567.3f, 594.6f, 619.5f, 642.1f, 662.8f, 698.9f, 729.f, 753.8f, 774.f, 790.1f, 802.6f, 811.9f, 818.3f, 822.3f, 824.1f, 823.9f, 822.2f, 819.f, 814.5f, 780.1f, 736.f, 695.9f, 660.4f, 628.6f, 599.9f, 573.7f, 549.7f, 507.5f, 471.4f, 440.1f, 412.8f, 388.8f, 367.6f, 348.9f, 332.2f, 317.2f, 303.7f, 291.4f, 280.3f, 270.f, 260.6f, 222.8f, 195.5f, 174.8f, 158.5f }; 
 
  static const G4float e72[60] = { 152.2f, 186.5f, 215.3f, 240.7f, 263.7f, 304.5f, 340.4f, 372.9f, 402.8f, 430.6f, 456.7f, 481.4f, 530.5f, 573.2f, 611.3f, 645.4f, 676.4f, 704.4f, 730.f, 753.3f, 793.8f, 827.4f, 855.f, 877.2f, 894.8f, 908.2f, 918.1f, 924.7f, 928.5f, 929.9f, 929.1f, 926.5f, 922.3f, 916.9f, 876.3f, 825.9f, 775.3f, 728.5f, 686.8f, 650.1f, 618.f, 589.9f, 536.1f, 489.2f, 450.8f, 420.2f, 395.5f, 373.8f, 354.6f, 337.5f, 322.2f, 308.3f, 295.8f, 284.4f, 273.9f, 264.3f, 225.7f, 198.f, 176.9f, 160.4f }; 
 
  static const G4float e73[60] = { 38.1f, 46.67f, 53.88f, 60.24f, 65.99f, 76.2f, 85.2f, 93.33f, 100.8f, 107.8f, 114.3f, 120.5f, 132.8f, 143.6f, 153.2f, 161.8f, 169.6f, 176.7f, 183.2f, 189.1f, 199.4f, 208.f, 215.f, 220.7f, 225.2f, 228.7f, 231.2f, 232.9f, 233.9f, 234.3f, 234.1f, 233.5f, 232.5f, 231.1f, 221.3f, 209.3f, 197.4f, 186.5f, 177.f, 168.6f, 161.3f, 155.f, 144.4f, 135.9f, 128.8f, 122.8f, 117.5f, 112.7f, 108.4f, 104.5f, 101.f, 97.75f, 94.8f, 92.09f, 89.6f, 87.29f, 77.8f, 70.52f, 64.84f, 60.13f }; 
 
  if(0 == j)      { AddData(e0, mat); }
  else if(1 == j) { AddData(e1, mat); }
  else if(2 == j) { AddData(e2, mat); }
  else if(3 == j) { AddData(e3, mat); }
  else if(4 == j) { AddData(e4, mat); }
  else if(5 == j) { AddData(e5, mat); }
  else if(6 == j) { AddData(e6, mat); }
  else if(7 == j) { AddData(e7, mat); }
  else if(8 == j) { AddData(e8, mat); }
  else if(9 == j) { AddData(e9, mat); }
  else if(10 == j) { AddData(e10, mat); }
  else if(11 == j) { AddData(e11, mat); }
  else if(12 == j) { AddData(e12, mat); }
  else if(13 == j) { AddData(e13, mat); }
  else if(14 == j) { AddData(e14, mat); }
  else if(15 == j) { AddData(e15, mat); }
  else if(16 == j) { AddData(e16, mat); }
  else if(17 == j) { AddData(e17, mat); }
  else if(18 == j) { AddData(e18, mat); }
  else if(19 == j) { AddData(e19, mat); }
  else if(20 == j) { AddData(e20, mat); }
  else if(21 == j) { AddData(e21, mat); }
  else if(22 == j) { AddData(e22, mat); }
  else if(23 == j) { AddData(e23, mat); }
  else if(24 == j) { AddData(e24, mat); }
  else if(25 == j) { AddData(e25, mat); }
  else if(26 == j) { AddData(e26, mat); }
  else if(27 == j) { AddData(e27, mat); }
  else if(28 == j) { AddData(e28, mat); }
  else if(29 == j) { AddData(e29, mat); }
  else if(30 == j) { AddData(e30, mat); }
  else if(31 == j) { AddData(e31, mat); }
  else if(32 == j) { AddData(e32, mat); }
  else if(33 == j) { AddData(e33, mat); }
  else if(34 == j) { AddData(e34, mat); }
  else if(35 == j) { AddData(e35, mat); }
  else if(36 == j) { AddData(e36, mat); }
  else if(37 == j) { AddData(e37, mat); }
  else if(38 == j) { AddData(e38, mat); }
  else if(39 == j) { AddData(e39, mat); }
  else if(40 == j) { AddData(e40, mat); }
  else if(41 == j) { AddData(e41, mat); }
  else if(42 == j) { AddData(e42, mat); }
  else if(43 == j) { AddData(e43, mat); }
  else if(44 == j) { AddData(e44, mat); }
  else if(45 == j) { AddData(e45, mat); }
  else if(46 == j) { AddData(e46, mat); }
  else if(47 == j) { AddData(e47, mat); }
  else if(48 == j) { AddData(e48, mat); }
  else if(49 == j) { AddData(e49, mat); }
  else if(50 == j) { AddData(e50, mat); }
  else if(51 == j) { AddData(e51, mat); }
  else if(52 == j) { AddData(e52, mat); }
  else if(53 == j) { AddData(e53, mat); }
  else if(54 == j) { AddData(e54, mat); }
  else if(55 == j) { AddData(e55, mat); }
  else if(56 == j) { AddData(e56, mat); }
  else if(57 == j) { AddData(e57, mat); }
  else if(58 == j) { AddData(e58, mat); }
  else if(59 == j) { AddData(e59, mat); }
  else if(60 == j) { AddData(e60, mat); }
  else if(61 == j) { AddData(e61, mat); }
  else if(62 == j) { AddData(e62, mat); }
  else if(63 == j) { AddData(e63, mat); }
  else if(64 == j) { AddData(e64, mat); }
  else if(65 == j) { AddData(e65, mat); }
  else if(66 == j) { AddData(e66, mat); }
  else if(67 == j) { AddData(e67, mat); }
  else if(68 == j) { AddData(e68, mat); }
  else if(69 == j) { AddData(e69, mat); }
  else if(70 == j) { AddData(e70, mat); }
  else if(71 == j) { AddData(e71, mat); }
  else if(72 == j) { AddData(e72, mat); }
  else if(73 == j) { AddData(e73, mat); }
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

void G4PSTARStopping::AddData(const G4float* stop, const G4Material* mat)
{
  G4LPhysicsFreeVector* v = 
    new G4LPhysicsFreeVector(60, T0[0], T0[59]);
  for(size_t i=0; i<60; ++i) { v->PutValues(i, T0[i], ((G4double)stop[i])*fac); }
  v->SetSpline(true);
  v->FillSecondDerivatives();
  materials.push_back(mat);
  sdata.push_back(v);
  ++nvectors;
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......
