/*
 * gdsreader - simple Calma parser/printer tool
 * Copyright (C) 1999 Serban-Mihai Popescu, serbanp@ix.netcom.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <strings.h>
#include <unistd.h>
#include <GDSstructs.h>
#include <GDSconsts.h>

char *
MALLOC(int size)
{
  return (char *)malloc(size);
}

void
FREE(void *ptr)
{
  free(ptr);
}

int
GDSreadInt2(unsigned char* record)
{
  int result;

  result = record[0];
  result <<= 8;
  result += record[1];
  if(result & 0x8000)
  {
    result &= 0x7fff;
    result ^= 0x7fff;
    result += 1;
    result = -result;
  }

  return result;
}

int
GDSreadInt4(unsigned char* record)
{
  int i;
  unsigned int result;

  for(i = 0, result = 0; i < 4; i++)
  {
    result <<= 8;
    result += record[i];
  }
  if(result & 0x80000000)
  {
    result &= 0x7fffffff;
    result ^= 0x7fffffff;
    result += 1;
    result = -result;
  }

  return result;
}

double
GDSreadReal8(unsigned char* record)
{
  int i, sign, exponent;
  unsigned long long mantissa_int;
  double mantissa_float, result;

  sign = record[0] & 0x80;
  exponent = (record[0] & 0x7f) - 64; 
  mantissa_int = 0;

  for(i = 1; i < 8; i++)
  {
    mantissa_int <<= 8;
    mantissa_int += record[i];
  }
  mantissa_float = (double)mantissa_int / pow(2, 56);
  result = mantissa_float * pow(16, (float)exponent);
  if(sign)
    result = -result;

  return result;
}

char *
GDSreadString(unsigned char* record, int len)
{
  char *result, string[1024];
  int i;

  if(len > 1024)
    len = 1024;
  for(i = 0; i < len; i++)
  {
    string[i] = record[i];
    if(record[i] == '\0')
      break;
  }
  string[len] = '\0';

  result = strdup(string);
  return result;
}

void
GDSaddDates(GDSlibrary* libptr, unsigned char* record)
{
  int my, mmo, md, mh, mmi, ms;
  int ay, amo, ad, ah, ami, as;
  char string[1024];

  my = GDSreadInt2(record + 2);
  mmo = GDSreadInt2(record + 4);
  md = GDSreadInt2(record + 6);
  mh = GDSreadInt2(record + 8);
  mmi = GDSreadInt2(record + 10);
  ms = GDSreadInt2(record + 12);

  ay = GDSreadInt2(record + 14);
  amo = GDSreadInt2(record + 16);
  ad = GDSreadInt2(record + 18);
  ah = GDSreadInt2(record + 20);
  ami = GDSreadInt2(record + 22);
  as = GDSreadInt2(record + 24);
  sprintf(string, "%d/%d/%d at %d:%d:%d", mmo, md, my, mh, mmi, ms);
  libptr->lastmod = strdup(string);
  sprintf(string, "%d/%d/%d at %d:%d:%d", amo, ad, ay, ah, ami, as);
  libptr->lastacc = strdup(string);
}

int
GDSrecordType(unsigned char record_type)
{
  int rtype;

  switch(record_type)
  {
    case 0x00:
      rtype = HEADER;
      break;
    case 0x01:
      rtype = BGNLIB;
      break;
    case 0x02:
      rtype = LIBNAME;
      break;
    case 0x03:
      rtype = UNITS;
      break;
    case 0x04:
      rtype = ENDLIB;
      break;
    case 0x05:
      rtype = BGNSTR;
      break;
    case 0x06:
      rtype = STRNAME;
      break;
    case 0x07:
      rtype = ENDSTR;
      break;
    case 0x08:
      rtype = BOUNDARY;
      break;
    case 0x09:
      rtype = PATH;
      break;
    case 0x0a:
      rtype = SREF;
      break;
    case 0x0b:
      rtype = AREF;
      break;
    case 0x0c:
      rtype = TEXT;
      break;
    case 0x0d:
      rtype = LAYER;
      break;
    case 0x0e:
      rtype = DATATYPE;
      break;
    case 0x0f:
      rtype = WIDTH;
      break;
    case 0x10:
      rtype = XY;
      break;
    case 0x11:
      rtype = ENDEL;
      break;
    case 0x12:
      rtype = SNAME;
      break;
    case 0x13:
      rtype = COLROW;
      break;
    case 0x14:
      rtype = TEXTNODE;
      break;
    case 0x15:
      rtype = NODE;
      break;
    case 0x16:
      rtype = TEXTTYPE;
      break;
    case 0x17:
      rtype = PRESENTATION;
      break;
    case 0x18:
      rtype = SPACING;
      break;
    case 0x19:
      rtype = STRING;
      break;
    case 0x1a:
      rtype = STRANS;
      break;
    case 0x1b:
      rtype = MAG;
      break;
    case 0x1c:
      rtype = ANGLE;
      break;
    case 0x1d:
      rtype = UINTEGER;
      break;
    case 0x1e:
      rtype = USTRING;
      break;
    case 0x1f:
      rtype = REFLIBS;
      break;
    case 0x20:
      rtype = FONTS;
      break;
    case 0x21:
      rtype = PATHTYPE;
      break;
    case 0x22:
      rtype = GENERATIONS;
      break;
    case 0x23:
      rtype = ATTRTABLE;
      break;
    case 0x24:
      rtype = STYPTABLE;
      break;
    case 0x25:
      rtype = STRTYPE;
      break;
    case 0x26:
      rtype = ELFLAGS;
      break;
    case 0x27:
      rtype = ELKEY;
      break;
    case 0x28:
      rtype = LINKTYPE;
      break;
    case 0x29:
      rtype = LINKKEYS;
      break;
    case 0x2a:
      rtype = NODETYPE;
      break;
    case 0x2b:
      rtype = PROPATTR;
      break;
    case 0x2c:
      rtype = PROPVALUE;
      break;
    case 0x2d:
      rtype = BOX;
      break;
    case 0x2e:
      rtype = BOXTYPE;
      break;
    case 0x2f:
      rtype = PLEX;
      break;
    case 0x30:
      rtype = BGNEXTN;
      break;
    case 0x31:
      rtype = ENDTEXTN;
      break;
    case 0x32:
      rtype = TAPENUM;
      break;
    case 0x33:
      rtype = TAPECODE;
      break;
    case 0x34:
      rtype = STRCLASS;
      break;
    case 0x35:
      rtype = RESERVED;
      break;
    case 0x36:
      rtype = FORMAT;
      break;
    case 0x37:
      rtype = MASK;
      break;
    case 0x38:
      rtype = ENDMASKS;
      break;
    case 0x39:
      rtype = LIBDIRSIZE;
      break;
    case 0x3a:
      rtype = SRFNAME;
      break;
    case 0x3b:
      rtype = LIBSECUR;
      break;
    default:
      rtype = UNKNOWN;
      break;
    }
  return rtype;
}

void
GDSunreadRecord(int gdsfildes, int nbytes)
{
  lseek(gdsfildes, -nbytes, SEEK_CUR);
  return;
}

int
GDSreadRecord(int gdsfildes, unsigned char **record, int *nbytes)
{
  int nread;
  unsigned char nbyte_array[2];

  if((nread = read(gdsfildes, nbyte_array, 2)) != 2)
  {
    if(nread == 1)
    {
      fprintf(stderr, "Error in reading the GDS file. Aborting\n");
      exit(1);
    }
    else
    {
      *record = NULL;
      return -1;
    }
  }
  *nbytes = nbyte_array[0] * 256 + nbyte_array[1];
  if(*nbytes != 0)
  {
    *record = (unsigned char *)MALLOC(*nbytes - 2);
    if((nread = read(gdsfildes, *record, *nbytes - 2)) != *nbytes - 2)
    {
      fprintf(stderr, "Error in reading the GDS file. Aborting\n");
      exit(1);
    }
  }
  return GDSrecordType((*record)[0]);
}

