/*----------------------
   Copyright (C): OpenGATE Collaboration

This software is distributed under the terms
of the GNU Lesser General  Public Licence (LGPL)
See GATE/LICENSE.txt for further details
----------------------*/

/*!
  \file   GateKermaFactorDatabase
  \brief  Database of neutron kerma factor
  \authors: Halima Elazhar (halima.elazhar@ihpc.cnrs.fr)
            Thomas Deschler (thomas.deschler@iphc.cnrs.fr)
*/

#ifndef GATEKERMAFACTORDATABASE_HH
#define GATEKERMAFACTORDATABASE_HH


#include <vector>

//-----------------------------------------------------------------------------
static const std::vector<double> energy_tableau = { // MeV
2.53e-08,
3.60e-08,
6.30e-08,
1.10e-07,
2.00e-07,
3.60e-07,
6.30e-07,
1.10e-06,
2.00e-06,
3.60e-06,
6.30e-06,
1.10e-05,
2.00e-05,
3.60e-05,
6.30e-05,
1.10e-04,
2.00e-04,
3.60e-04,
6.30e-04,
1.10e-03,
2.00e-03,
3.60e-03,
6.30e-03,
1.10e-02,
2.00e-02,
3.60e-02,
6.30e-02,
8.20e-02,
8.60e-02,
9.00e-02,
9.40e-02,
9.80e-02,
1.05e-01,
1.15e-01,
1.25e-01,
1.35e-01,
1.45e-01,
1.55e-01,
1.65e-01,
1.75e-01,
1.85e-01,
1.95e-01,
2.10e-01,
2.30e-01,
2.50e-01,
2.70e-01,
2.90e-01,
3.10e-01,
3.30e-01,
3.50e-01,
3.70e-01,
3.90e-01,
4.20e-01,
4.60e-01,
5.00e-01,
5.40e-01,
5.80e-01,
6.20e-01,
6.60e-01,
7.00e-01,
7.40e-01,
7.80e-01,
8.20e-01,
8.60e-01,
9.00e-01,
9.40e-01,
9.80e-01,
1.05e+00,
1.15e+00,
1.25e+00,
1.35e+00,
1.45e+00,
1.55e+00,
1.65e+00,
1.75e+00,
1.85e+00,
1.95e+00,
2.10e+00,
2.30e+00,
2.50e+00,
2.70e+00,
2.90e+00,
3.10e+00,
3.30e+00,
3.50e+00,
3.70e+00,
3.90e+00,
4.20e+00,
4.60e+00,
5.00e+00,
5.40e+00,
5.80e+00,
6.20e+00,
6.60e+00,
7.00e+00,
7.40e+00,
7.80e+00,
8.20e+00,
8.60e+00,
9.00e+00,
9.40e+00,
9.80e+00,
1.05e+01,
1.15e+01,
1.25e+01,
1.35e+01,
1.45e+01,
1.55e+01,
1.65e+01,
1.75e+01,
1.85e+01,
1.95e+01,
2.10e+01,
2.30e+01,
2.50e+01,
2.70e+01,
2.90e+01
};
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
static const std::vector<double> kerma_factor_table_hydrogen = {
4.20E-18,
3.55E-18,
2.69E-18,
2.04E-18,
1.52E-18,
1.16E-18,
9.09E-19,
7.49E-19,
6.72E-19,
7.07E-19,
8.85E-19,
1.28E-18,
2.11E-18,
3.64E-18,
6.25E-18,
1.08E-17,
1.96E-17,
3.52E-17,
6.14E-17,
1.07E-16,
1.93E-16,
3.45E-16,
5.93E-16,
1.00E-15,
1.74E-15,
2.87E-15,
4.44E-15,
5.35E-15,
5.53E-15,
5.70E-15,
5.87E-15,
6.03E-15,
6.31E-15,
6.69E-15,
7.05E-15,
7.39E-15,
7.72E-15,
8.03E-15,
8.32E-15,
8.61E-15,
8.67E-15,
9.13E-15,
9.51E-15,
9.99E-15,
1.04E-14,
1.09E-14,
1.13E-14,
1.17E-14,
1.20E-14,
1.24E-14,
1.27E-14,
1.31E-14,
1.35E-14,
1.41E-14,
1.47E-14,
1.53E-14,
1.58E-14,
1.63E-14,
1.68E-10,
1.73E-10,
1.77E-14,
1.82E-14,
1.86E-14,
1.90E-14,
1.94E-14,
1.98E-14,
2.02E-14,
2.09E-14,
2.18E-14,
2.27E-14,
2.35E-14,
2.43E-14,
2.50E-14,
2.57E-14,
2.64E-14,
2.71E-14,
2.77E-14,
2.86E-14,
2.97E-14,
3.07E-14,
3.17E-14,
3.26E-14,
3.35E-14,
3.43E-14,
3.50E-14,
3.57E-14,
3.64E-14,
3.73E-14,
3.84E-14,
3.94E-14,
4.02E-14,
4.10E-14,
4.17E-14,
4.23E-14,
4.28E-14,
4.33E-14,
4.38E-14,
4.42E-14,
4.45E-14,
4.48E-14,
4.52E-14,
4.54E-14,
4.59E-14,
4.63E-14,
4.67E-14,
4.69E-14,
4.71E-14,
4.71E-14,
4.72E-14,
4.71E-14,
4.70E-14,
4.69E-14,
4.67E-14,
4.63E-14,
4.58E-14,
4.53E-14,
4.47E-14
};
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
static const std::vector<double> kerma_factor_muscle_tableau = {
2.72e-17,
2.31e-17,
1.74e-17,
1.32e-17,
9.78e-18,
7.29e-18,
5.52e-18,
4.17e-18,
3.12e-18,
2.34e-18,
1.81e-18,
1.43e-18,
1.19e-18,
1.10e-18,
1.20e-18,
1.55e-18,
2.36e-18,
3.94e-18,
6.62e-18,
1.14e-17,
2.04e-17,
3.63e-17,
6.23e-17,
1.05e-16,
1.83e-16,
3.03e-16,
4.70e-16,
5.68e-16,
5.87e-16,
6.06e-16,
6.24e-16,
6.41e-16,
6.72e-16,
7.13e-16,
7.52e-16,
7.90e-16,
8.26e-16,
8.60e-16,
8.92e-16,
9.25e-16,
9.54e-16,
9.83e-16,
1.03e-15,
1.08e-15,
1.13e-15,
1.19e-15,
1.23e-15,
1.28e-15,
1.32e-15,
1.37e-15,
1.42e-15,
1.49e-15,
1.62e-15,
1.64e-15,
1.60e-15,
1.66e-15,
1.71e-15,
1.77e-15,
1.84e-15,
1.89e-15,
1.94e-15,
2.00e-15,
2.05e-15,
2.10e-15,
2.16e-15,
2.27e-15,
2.44e-15,
2.48e-15,
2.46e-15,
2.56e-15,
2.65e-15,
2.69e-15,
2.76e-15,
2.87e-15,
2.91e-15,
3.03e-15,
3.05e-15,
3.14e-15,
3.19e-15,
3.31e-15,
3.47e-15,
3.60e-15,
3.74e-15,
4.07e-15,
4.15e-15,
4.25e-15,
4.19e-15,
4.31e-15,
4.31e-15,
4.55e-15,
4.44e-15,
4.65e-15,
4.76e-15,
4.88e-15,
5.08e-15,
5.36e-15,
5.30e-15,
5.25e-15,
5.41e-15,
5.51e-15,
5.56e-15,
5.68e-15,
5.82e-15,
6.23e-15,
6.22e-15,
6.46e-15,
6.71e-15,
6.88e-15,
6.97e-15,
7.06e-15,
7.17e-15,
7.29e-15,
7.44e-15,
7.42e-15,
7.38e-15,
7.41e-15,
7.28e-15
};
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
static const std::vector<double> energy_table_DA = { // MeV
2.53E-8,
3.60E-8,
6.30E-8,
1.10E-7,
2.00E-7,
3.60E-7,
6.30E-7,
1.10E-6,
2.00E-6,
3.60E-6,
6.30E-6,
1.10E-5,
2.00E-5,
3.60E-5,
6.30E-5,
1.10E-4,
2.00E-4,
3.60E-4,
6.30E-4,
1.10E-3,
2.00E-3,
3.60E-3,
6.30E-3,
1.10E-2,
2.00E-2,
3.60E-2,
6.30E-2,
8.20E-2,
8.60E-2,
9.00E-2,
9.40E-2,
9.80E-2,
1.05E-1,
1.15E-1,
1.25E-1,
1.35E-1,
1.45E-1,
1.55E-1,
1.65E-1,
1.75E-1,
1.85E-1,
1.95E-1,
2.10E-1,
2.30E-1,
2.50E-1,
2.70E-1,
2.90E-1,
3.10E-1,
3.30E-1,
3.50E-1,
3.70E-1,
3.90E-1,
4.20E-1,
4.60E-1,
5.00E-1,
5.40E-1,
5.80E-1,
6.20E-1,
6.60E-1,
7.00E-1,
7.40E-1,
7.80E-1,
8.20E-1,
8.60E-1,
9.00E-1,
9.40E-1,
9.80E-1,
1.05E+0,
1.15E+0,
1.25E+0,
1.35E+0,
1.45E+0,
1.55E+0,
1.65E+0,
1.75E+0,
1.85E+0,
1.95E+0,
2.10E+0
};
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
static const std::vector<double> kerma_factor_table_muscle_DA = { // Gy.m²
2.7625E-17,
2.3423E-17,
1.7651E-17,
1.3564E-17,
1.0157E-17,
7.5897E-18,
5.9116E-18,
4.4149E-18,
3.2655E-18,
2.4698E-18,
1.8771E-18,
1.5026E-18,
1.2492E-18,
1.1665E-18,
1.2419E-18,
1.5257E-18,
2.3302E-18,
3.8173E-18,
6.4637E-18,
1.1044E-17,
1.9814E-17,
3.5308E-17,
6.0906E-17,
1.0379E-16,
1.8050E-16,
2.9831E-16,
4.6593E-16,
5.6550E-16,
5.8441E-16,
6.0215E-16,
6.1967E-16,
6.3594E-16,
6.6215E-16,
7.0706E-16,
7.4887E-16,
7.8888E-16,
8.2515E-16,
8.5881E-16,
8.8988E-16,
9.1827E-16,
9.4367E-16,
9.7488E-16,
1.0248E-15,
1.0849E-15,
1.1389E-15,
1.1858E-15,
1.2311E-15,
1.2706E-15,
1.3106E-15,
1.3659E-15,
1.4212E-15,
1.4874E-15,
1.6381E-15,
1.5818E-15,
1.5784E-15,
1.6476E-15,
1.7137E-15,
1.7788E-15,
1.8332E-15,
1.8867E-15,
1.9321E-15,
1.9710E-15,
2.0201E-15,
2.0786E-15,
2.1440E-15,
2.2543E-15,
2.4089E-15,
2.4206E-15,
2.4061E-15,
2.4888E-15,
2.5621E-15,
2.6244E-15,
2.7052E-15,
3.2676E-15,
2.8101E-15,
2.9181E-15,
2.9263E-15,
3.0326E-15
};
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
static const std::vector<double> kerma_factor_Cortical_Bone_tableau = {
3.34e-17,
2.86e-17,
2.16e-17,
1.63e-17,
1.22e-17,
9.12e-18,
6.98e-18,
5.32e-18,
4.03e-18,
3.07e-18,
2.38e-18,
1.88e-18,
1.51e-18,
1.26e-18,
1.14e-18,
1.14e-18,
1.33e-18,
1.80e-18,
2.69e-18,
4.31e-18,
7.40e-18,
1.29e-17,
2.20e-17,
3.69e-17,
6.41e-17,
1.06e-16,
1.65e-16,
2.00e-16,
2.08e-16,
2.14e-16,
2.20e-16,
2.26e-16,
2.37e-16,
2.52e-16,
2.66e-16,
2.82e-16,
2.94e-16,
3.07e-16,
3.18e-16,
3.29e-16,
3.41e-16,
3.51e-16,
3.69e-16,
3.88e-16,
4.10e-16,
4.29e-16,
4.47e-16,
4.64e-16,
4.83e-16,
5.04e-16,
5.25e-16,
5.53e-16,
6.23e-16,
6.21e-16,
5.83e-16,
5.98e-16,
6.22e-16,
6.49e-16,
6.73e-16,
6.90e-16,
7.07e-16,
7.30e-16,
7.49e-16,
7.73e-16,
8.04e-16,
8.55e-16,
9.52e-16,
9.56e-16,
9.19e-16,
9.58e-16,
9.94e-16,
1.00e-15,
1.03e-15,
1.07e-15,
1.08e-15,
1.14e-15,
1.13e-15,
1.18e-15,
1.18e-15,
1.24e-15,
1.33e-15,
1.42e-15,
1.49e-15,
1.71e-15,
1.77e-15,
1.82e-15,
1.78e-15,
1.85e-15,
1.81e-15,
1.93e-15,
1.85e-15,
1.97e-15,
2.05e-15,
2.09e-15,
2.21e-15,
2.41e-15,
2.40e-15,
2.35e-15,
2.44e-15,
2.53e-15,
2.59e-15,
2.65e-15,
2.74e-15,
3.04e-15,
3.08e-15,
3.25e-15,
3.44e-15,
3.59e-15,
3.65e-15,
3.70e-15,
3.77e-15,
3.87e-15,
4.03e-15,
4.13e-15,
4.23e-15,
4.36e-15,
4.41e-15
};
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
static const std::vector<double> kerma_factor_Lung_tableau = {
2.50e-17,
2.12e-17,
1.60e-17,
1.21e-17,
8.98e-18,
6.69e-18,
5.07e-18,
3.83e-18,
2.87e-18,
2.15e-18,
1.66e-18,
1.32e-18,
1.11e-18,
1.04e-18,
1.16e-18,
1.52e-18,
2.36e-18,
3.99e-18,
6.68e-18,
1.14e-17,
2.05e-17,
3.66e-17,
6.28e-17,
1.06e-16,
1.85e-16,
3.05e-16,
4.74e-16,
5.72e-16,
5.92e-16,
6.10e-16,
6.29e-16,
6.46e-16,
6.77e-16,
7.19e-16,
7.58e-16,
7.96e-16,
8.32e-16,
8.67e-16,
8.99e-16,
9.32e-16,
9.61e-16,
9.91e-16,
1.03e-15,
1.09e-15,
1.14e-15,
1.19e-15,
1.24e-15,
1.29e-15,
1.33e-15,
1.38e-15,
1.43e-15,
1.50e-15,
1.64e-15,
1.65e-15,
1.61e-15,
1.67e-15,
1.73e-15,
1.79e-15,
1.85e-15,
1.90e-15,
1.95e-15,
2.01e-15,
2.06e-15,
2.11e-15,
2.18e-15,
2.29e-15,
2.47e-15,
2.51e-15,
2.48e-15,
2.58e-15,
2.67e-15,
2.71e-15,
2.78e-15,
2.89e-15,
2.93e-15,
3.05e-15,
3.07e-15,
3.16e-15,
3.20e-15,
3.33e-15,
3.48e-15,
3.61e-15,
3.76e-15,
4.09e-15,
4.17e-15,
4.27e-15,
4.20e-15,
4.34e-15,
4.34e-15,
4.59e-15,
4.47e-15,
4.68e-15,
4.79e-15,
4.93e-15,
5.14e-15,
5.41e-15,
5.32e-15,
5.29e-15,
5.46e-15,
5.55e-15,
5.58e-15,
5.72e-15,
5.86e-15,
6.28e-15,
6.25e-15,
6.49e-15,
6.74e-15,
6.89e-15,
6.98e-15,
7.06e-15,
7.17e-15,
7.29e-15,
7.43e-15,
7.40e-15,
7.35e-15,
7.37e-15,
7.23e-15
};
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
static const std::vector<double> kerma_factor_Vacuum_tableau = {
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.,
0.
};
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
static const std::vector<double> kerma_factor_Lung_Griffith_tableau = {
3.33E-17,
2.83E-17,
2.13E-17,
1.61E-17,
1.20E-17,
8.94E-18,
6.78E-18,
5.12E-18,
3.83E-18,
2.87E-18,
2.20E-18,
1.73E-18,
1.39E-18,
1.21E-18,
1.21E-18,
1.44E-18,
2.05E-18,
3.27E-18,
5.41E-18,
9.21E-18,
1.64E-17,
2.92E-17,
5.02E-17,
8.47E-17,
1.48E-16,
2.44E-16,
3.80E-16,
4.60E-16,
4.76E-16,
4.91E-16,
5.06E-16,
5.20E-16,
5.45E-16,
5.79E-16,
6.11E-16,
6.42E-16,
6.72E-16,
7.00E-16,
7.26E-16,
7.53E-16,
7.77E-16,
8.01E-16,
8.36E-16,
8.81E-16,
9.20E-16,
9.66E-16,
1.00E-15,
1.04E-15,
1.07E-15,
1.11E-15,
1.15E-15,
1.19E-15,
1.26E-15,
1.30E-15,
1.32E-15,
1.37E-15,
1.42E-15,
1.47E-15,
1.52E-15,
1.56E-15,
1.60E-15,
1.65E-15,
1.68E-15,
1.72E-15,
1.77E-15,
1.82E-15,
1.91E-15,
1.96E-15,
2.00E-15,
2.08E-15,
2.16E-15,
2.21E-15,
2.27E-15,
2.34E-15,
2.40E-15,
2.47E-15,
2.51E-15,
2.63E-15,
2.68E-15,
2.80E-15,
2.96E-15,
3.18E-15,
3.16E-15,
3.49E-15,
3.64E-15,
3.68E-15,
3.64E-15,
3.64E-15,
3.58E-15,
3.68E-15,
3.68E-15,
3.80E-15,
4.00E-15,
3.88E-15,
3.98E-15,
4.29E-15,
4.59E-15,
4.36E-15,
4.34E-15,
4.64E-15,
4.85E-15,
4.76E-15,
4.82E-15,
5.12E-15,
5.35E-15,
5.56E-15,
5.87E-15,
6.25E-15,
6.47E-15,
6.55E-15,
6.69E-15,
6.80E-15,
7.06E-15,
7.18E-15,
7.29E-15,
7.37E-15,
7.44E-15
};
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
static const std::vector<double> kerma_factor_Air_ICRU_tableau = {
5.93e-16,
5.02e-16,
3.79e-16,
2.87e-16,
2.13e-16,
1.59e-16,
1.20e-16,
9.06e-17,
6.75e-17,
5.02e-17,
3.79e-17,
2.87e-17,
2.14e-17,
1.59e-17,
1.20e-17,
9.14e-18,
6.88e-18,
5.26e-18,
4.20e-18,
3.57e-18,
3.36e-18,
3.70e-18,
4.76e-18,
6.88e-18,
1.08e-17,
1.70e-17,
2.55e-17,
3.07e-17,
3.18e-17,
3.28e-17,
3.38e-17,
3.48e-17,
3.67e-17,
3.94e-17,
4.19e-17,
4.45e-17,
4.71e-17,
4.95e-17,
5.20e-17,
5.44e-17,
5.68e-17,
5.92e-17,
6.27e-17,
6.74e-17,
7.20e-17,
7.64e-17,
8.09e-17,
8.56e-17,
9.08e-17,
9.65e-17,
1.04e-16,
1.16e-16,
1.67e-16,
1.33e-16,
1.43e-16,
9.81e-17,
9.98e-17,
1.54e-16,
2.30e-16,
1.66e-16,
1.48e-16,
1.40e-16,
1.36e-16,
1.33e-16,
1.33e-16,
1.41e-16,
2.03e-16,
2.51e-16,
1.93e-16,
1.94e-16,
3.78e-16,
3.56e-16,
2.91e-16,
2.85e-16,
3.85e-16,
3.41e-16,
2.99e-16,
3.28e-16,
3.92e-16,
4.06e-16,
5.83e-16,
6.77e-16,
8.51e-16,
9.77e-16,
1.16e-15,
1.10e-15,
1.26e-15,
1.41e-15,
1.11e-15,
9.90e-16,
8.59e-16,
8.51e-16,
9.84e-16,
8.20e-16,
9.44e-16,
1.20e-15,
1.11e-15,
1.04e-15,
1.08e-15,
1.13e-15,
1.18e-15,
1.31e-15,
1.47e-15,
1.74e-15,
1.96e-15,
2.19e-15,
2.39e-15,
2.54e-15,
2.65e-15,
2.78e-15,
2.94e-15,
3.10e-15,
3.28e-15,
3.41e-15,
3.48e-15,
3.56e-15,
3.59e-15
};
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
static const std::vector<double> kerma_factor_Adipose_tissue_Adult_1_ICRU46_tableau = {
1.07E-17,
9.10E-18,
6.87E-18,
5.20E-18,
3.86E-18,
2.88E-18,
2.18E-18,
1.66E-18,
1.25E-18,
9.51E-19,
7.58E-19,
6.45E-19,
6.14E-19,
6.98E-19,
9.32E-19,
1.41E-18,
2.39E-18,
4.19E-18,
7.19E-18,
1.25E-17,
2.24E-17,
4.00E-17,
6.88E-17,
1.16E-16,
2.02E-16,
3.35E-16,
5.20E-16,
6.29E-16,
6.50E-16,
6.71E-16,
6.91E-16,
7.10E-16,
7.44E-16,
7.90E-16,
8.33E-16,
8.75E-16,
9.15E-16,
9.53E-16,
9.88E-16,
1.02E-15,
1.06E-15,
1.09E-15,
1.14E-15,
1.20E-15,
1.25E-15,
1.31E-15,
1.36E-15,
1.41E-15,
1.45E-15,
1.51E-15,
1.55E-15,
1.61E-15,
1.70E-15,
1.76E-15,
1.78E-15,
1.85E-15,
1.91E-15,
1.98E-15,
2.04E-15,
2.10E-15,
2.15E-15,
2.22E-15,
2.27E-15,
2.32E-15,
2.38E-15,
2.46E-15,
2.58E-15,
2.64E-15,
2.69E-15,
2.80E-15,
2.90E-15,
2.97E-15,
3.05E-15,
3.15E-15,
3.22E-15,
3.33E-15,
3.38E-15,
3.52E-15,
3.59E-15,
3.74E-15,
3.92E-15,
4.14E-15,
4.17E-15,
4.52E-15,
4.67E-15,
4.75E-15,
4.71E-15,
4.75E-15,
4.74E-15,
4.90E-15,
4.90E-15,
5.07E-15,
5.27E-15,
5.22E-15,
5.35E-15,
5.67E-15,
5.90E-15,
5.72E-15,
5.75E-15,
6.01E-15,
6.20E-15,
6.16E-15,
6.24E-15,
6.57E-15,
6.75E-15,
6.97E-15,
7.28E-15,
7.61E-15,
7.81E-15,
7.89E-15,
8.02E-15,
8.13E-15,
8.35E-15,
8.42E-15,
8.48E-15,
8.52E-15,
8.53E-15
};
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
static const std::vector<double> kerma_factor_Blood_Adult_ICRU46_tableau = {
2.65E-17,
2.25E-17,
1.70E-17,
1.29E-17,
9.54E-18,
7.11E-18,
5.38E-18,
4.07E-18,
3.04E-18,
2.28E-18,
1.76E-18,
1.40E-18,
1.16E-18,
1.08E-18,
1.18E-18,
1.53E-18,
2.38E-18,
3.97E-18,
6.63E-18,
1.13E-17,
2.03E-17,
3.63E-17,
6.22E-17,
1.05E-16,
1.83E-16,
3.02E-16,
4.69E-16,
5.67E-16,
5.86E-16,
6.05E-16,
6.23E-16,
6.40E-16,
6.71E-16,
7.12E-16,
7.51E-16,
7.89E-16,
8.25E-16,
8.59E-16,
8.91E-16,
9.24E-16,
9.53E-16,
9.82E-16,
1.02E-15,
1.08E-15,
1.13E-15,
1.18E-15,
1.23E-15,
1.28E-15,
1.32E-15,
1.37E-15,
1.42E-15,
1.49E-15,
1.62E-15,
1.64E-15,
1.60E-15,
1.65E-15,
1.71E-15,
1.77E-15,
1.83E-15,
1.89E-15,
1.93E-15,
1.99E-15,
2.04E-15,
2.10E-15,
2.16E-15,
2.27E-15,
2.45E-15,
2.48E-15,
2.46E-15,
2.56E-15,
2.65E-15,
2.69E-15,
2.76E-15,
2.87E-15,
2.91E-15,
3.03E-15,
3.04E-15,
3.13E-15,
3.18E-15,
3.30E-15,
3.45E-15,
3.58E-15,
3.73E-15,
4.06E-15,
4.14E-15,
4.24E-15,
4.17E-15,
4.30E-15,
4.31E-15,
4.55E-15,
4.43E-15,
4.64E-15,
4.75E-15,
4.89E-15,
5.10E-15,
5.37E-15,
5.28E-15,
5.25E-15,
5.42E-15,
5.51E-15,
5.54E-15,
5.68E-15,
5.82E-15,
6.24E-15,
6.21E-15,
6.45E-15,
6.70E-15,
6.85E-15,
6.94E-15,
7.03E-15,
7.13E-15,
7.26E-15,
7.40E-15,
7.37E-15,
7.32E-15,
7.34E-15,
7.21E-15
};
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
static const std::vector<double> kerma_factor_GI_track_intestine_Adult_ICRU46_tableau = {
1.79E-17,
1.51E-17,
1.14E-17,
8.64E-18,
6.42E-18,
4.78E-18,
3.62E-18,
2.74E-18,
2.05E-18,
1.55E-18,
1.21E-18,
9.80E-19,
8.55E-19,
8.62E-19,
1.03E-18,
1.44E-18,
2.33E-18,
4.01E-18,
6.80E-18,
1.17E-17,
2.11E-17,
3.76E-17,
6.46E-17,
1.09E-16,
1.90E-16,
3.14E-16,
4.87E-16,
5.88E-16,
6.09E-16,
6.28E-16,
6.47E-16,
6.65E-16,
6.96E-16,
7.39E-16,
7.80E-16,
8.18E-16,
8.56E-16,
8.91E-16,
9.25E-16,
9.58E-16,
9.88E-16,
1.02E-15,
1.06E-15,
1.12E-15,
1.17E-15,
1.23E-15,
1.28E-15,
1.33E-15,
1.37E-15,
1.42E-15,
1.47E-15,
1.54E-15,
1.68E-15,
1.70E-15,
1.66E-15,
1.71E-15,
1.77E-15,
1.84E-15,
1.90E-15,
1.96E-15,
2.01E-15,
2.07E-15,
2.12E-15,
2.17E-15,
2.24E-15,
2.35E-15,
2.54E-15,
2.57E-15,
2.54E-15,
2.65E-15,
2.74E-15,
2.78E-15,
2.86E-15,
2.97E-15,
3.01E-15,
3.14E-15,
3.15E-15,
3.25E-15,
3.29E-15,
3.42E-15,
3.57E-15,
3.71E-15,
3.85E-15,
4.19E-15,
4.27E-15,
4.37E-15,
4.30E-15,
4.44E-15,
4.45E-15,
4.70E-15,
4.58E-15,
4.80E-15,
4.91E-15,
5.05E-15,
5.26E-15,
5.54E-15,
5.45E-15,
5.42E-15,
5.59E-15,
5.68E-15,
5.72E-15,
5.85E-15,
6.00E-15,
6.42E-15,
6.39E-15,
6.63E-15,
6.88E-15,
7.03E-15,
7.12E-15,
7.21E-15,
7.31E-15,
7.44E-15,
7.58E-15,
7.55E-15,
7.49E-15,
7.52E-15,
7.38E-15
};
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
static const std::vector<double> kerma_factor_Lymph_Adult_ICRU46_ICRU46_tableau = {
9.35E-18,
7.92E-18,
5.98E-18,
4.53E-18,
3.36E-18,
2.51E-18,
1.90E-18,
1.44E-18,
1.09E-18,
8.28E-19,
6.62E-19,
5.68E-19,
5.50E-19,
6.38E-19,
8.68E-19,
1.33E-18,
2.27E-18,
4.04E-18,
6.87E-18,
1.19E-17,
2.14E-17,
3.82E-17,
6.56E-17,
1.11E-16,
1.93E-16,
3.19E-16,
4.95E-16,
5.97E-16,
6.18E-16,
6.37E-16,
6.56E-16,
6.75E-16,
7.07E-16,
7.50E-16,
7.91E-16,
8.31E-16,
8.69E-16,
9.05E-16,
9.38E-16,
9.72E-16,
1.00E-15,
1.03E-15,
1.08E-15,
1.14E-15,
1.19E-15,
1.25E-15,
1.30E-15,
1.35E-15,
1.39E-15,
1.44E-15,
1.49E-15,
1.57E-15,
1.71E-15,
1.73E-15,
1.68E-15,
1.74E-15,
1.80E-15,
1.86E-15,
1.92E-15,
1.98E-15,
2.03E-15,
2.09E-15,
2.15E-15,
2.20E-15,
2.28E-15,
2.39E-15,
2.59E-15,
2.62E-15,
2.58E-15,
2.69E-15,
2.78E-15,
2.82E-15,
2.90E-15,
3.02E-15,
3.05E-15,
3.18E-15,
3.19E-15,
3.28E-15,
3.33E-15,
3.46E-15,
3.60E-15,
3.72E-15,
3.89E-15,
4.22E-15,
4.29E-15,
4.40E-15,
4.33E-15,
4.48E-15,
4.50E-15,
4.78E-15,
4.64E-15,
4.87E-15,
4.96E-15,
5.14E-15,
5.38E-15,
5.64E-15,
5.50E-15,
5.49E-15,
5.69E-15,
5.75E-15,
5.76E-15,
5.93E-15,
6.09E-15,
6.52E-15,
6.45E-15,
6.69E-15,
6.93E-15,
7.06E-15,
7.13E-15,
7.21E-15,
7.31E-15,
7.44E-15,
7.56E-15,
7.51E-15,
7.43E-15,
7.45E-15,
7.28E-15
};
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
static const std::vector<double> kerma_factor_Skeleton_cartilage_Adult_ICRU46_tableau = {
1.79E-17,
1.52E-17,
1.15E-17,
8.67E-18,
6.44E-18,
4.80E-18,
3.64E-18,
2.75E-18,
2.07E-18,
1.56E-18,
1.21E-18,
9.80E-19,
8.47E-19,
8.38E-19,
9.79E-19,
1.34E-18,
2.14E-18,
3.69E-18,
6.20E-18,
1.07E-17,
1.91E-17,
3.42E-17,
5.86E-17,
9.89E-17,
1.72E-16,
2.85E-16,
4.42E-16,
5.34E-16,
5.53E-16,
5.70E-16,
5.87E-16,
6.04E-16,
6.32E-16,
6.71E-16,
7.08E-16,
7.43E-16,
7.78E-16,
8.10E-16,
8.40E-16,
8.71E-16,
8.98E-16,
9.26E-16,
9.66E-16,
1.02E-15,
1.06E-15,
1.12E-15,
1.16E-15,
1.21E-15,
1.25E-15,
1.30E-15,
1.34E-15,
1.41E-15,
1.54E-15,
1.55E-15,
1.51E-15,
1.56E-15,
1.61E-15,
1.67E-15,
1.73E-15,
1.78E-15,
1.82E-15,
1.88E-15,
1.93E-15,
1.98E-15,
2.04E-15,
2.15E-15,
2.33E-15,
2.35E-15,
2.32E-15,
2.42E-15,
2.50E-15,
2.54E-15,
2.60E-15,
2.71E-15,
2.74E-15,
2.86E-15,
2.87E-15,
2.95E-15,
2.99E-15,
3.11E-15,
3.25E-15,
3.37E-15,
3.51E-15,
3.83E-15,
3.91E-15,
4.01E-15,
3.93E-15,
4.06E-15,
4.06E-15,
4.30E-15,
4.18E-15,
4.39E-15,
4.49E-15,
4.63E-15,
4.84E-15,
5.10E-15,
5.00E-15,
4.97E-15,
5.14E-15,
5.23E-15,
5.26E-15,
5.39E-15,
5.53E-15,
5.94E-15,
5.91E-15,
6.14E-15,
6.38E-15,
6.53E-15,
6.61E-15,
6.70E-15,
6.80E-15,
6.93E-15,
7.06E-15,
7.04E-15,
6.98E-15,
7.01E-15,
6.88E-15
};
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
static const std::vector<double> kerma_factor_Skeleton_spongiosa_Adult_ICRU46_tableau = {
2.26E-17,
1.92E-17,
1.45E-17,
1.10E-17,
8.19E-18,
6.10E-18,
4.65E-18,
3.53E-18,
2.66E-18,
2.02E-18,
1.57E-18,
1.26E-18,
1.06E-18,
9.95E-19,
1.08E-18,
1.37E-18,
2.07E-18,
3.42E-18,
5.67E-18,
9.66E-18,
1.72E-17,
3.07E-17,
5.26E-17,
8.88E-17,
1.55E-16,
2.56E-16,
3.98E-16,
4.81E-16,
4.97E-16,
5.13E-16,
5.28E-16,
5.43E-16,
5.69E-16,
6.04E-16,
6.38E-16,
6.70E-16,
7.01E-16,
7.30E-16,
7.57E-16,
7.84E-16,
8.09E-16,
8.34E-16,
8.71E-16,
9.17E-16,
9.58E-16,
1.01E-15,
1.05E-15,
1.08E-15,
1.12E-15,
1.16E-15,
1.20E-15,
1.24E-15,
1.33E-15,
1.36E-15,
1.37E-15,
1.42E-15,
1.47E-15,
1.52E-15,
1.57E-15,
1.62E-15,
1.65E-15,
1.70E-15,
1.74E-15,
1.79E-15,
1.84E-15,
1.90E-15,
2.01E-15,
2.06E-15,
2.08E-15,
2.16E-15,
2.24E-15,
2.29E-15,
2.35E-15,
2.44E-15,
2.48E-15,
2.57E-15,
2.60E-15,
2.71E-15,
2.76E-15,
2.88E-15,
3.03E-15,
3.21E-15,
3.25E-15,
3.56E-15,
3.68E-15,
3.74E-15,
3.71E-15,
3.76E-15,
3.73E-15,
3.88E-15,
3.85E-15,
3.99E-15,
4.16E-15,
4.13E-15,
4.26E-15,
4.54E-15,
4.70E-15,
4.56E-15,
4.60E-15,
4.81E-15,
4.96E-15,
4.95E-15,
5.03E-15,
5.34E-15,
5.49E-15,
5.69E-15,
5.95E-15,
6.23E-15,
6.38E-15,
6.46E-15,
6.57E-15,
6.67E-15,
6.87E-15,
6.96E-15,
7.03E-15,
7.11E-15,
7.14E-15
};
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
static const std::vector<double> kerma_factor_Skin_Adult_ICRU46_tableau = {
3.36E-17,
2.85E-17,
2.15E-17,
1.63E-17,
1.21E-17,
8.99E-18,
6.81E-18,
5.15E-18,
3.85E-18,
2.88E-18,
2.21E-18,
1.74E-18,
1.41E-18,
1.27E-18,
1.31E-18,
1.62E-18,
2.40E-18,
3.97E-18,
6.57E-18,
1.12E-17,
2.00E-17,
3.57E-17,
6.13E-17,
1.03E-16,
1.80E-16,
2.98E-16,
4.62E-16,
5.58E-16,
5.78E-16,
5.96E-16,
6.14E-16,
6.31E-16,
6.61E-16,
7.02E-16,
7.40E-16,
7.77E-16,
8.13E-16,
8.47E-16,
8.78E-16,
9.10E-16,
9.39E-16,
9.68E-16,
1.01E-15,
1.06E-15,
1.11E-15,
1.17E-15,
1.21E-15,
1.26E-15,
1.30E-15,
1.35E-15,
1.39E-15,
1.46E-15,
1.58E-15,
1.60E-15,
1.58E-15,
1.63E-15,
1.69E-15,
1.75E-15,
1.81E-15,
1.86E-15,
1.91E-15,
1.97E-15,
2.02E-15,
2.07E-15,
2.13E-15,
2.22E-15,
2.39E-15,
2.43E-15,
2.42E-15,
2.52E-15,
2.61E-15,
2.65E-15,
2.72E-15,
2.83E-15,
2.87E-15,
2.98E-15,
3.00E-15,
3.10E-15,
3.15E-15,
3.28E-15,
3.43E-15,
3.58E-15,
3.69E-15,
4.03E-15,
4.12E-15,
4.21E-15,
4.16E-15,
4.27E-15,
4.26E-15,
4.47E-15,
4.38E-15,
4.57E-15,
4.70E-15,
4.79E-15,
4.98E-15,
5.26E-15,
5.25E-15,
5.17E-15,
5.31E-15,
5.44E-15,
5.51E-15,
5.60E-15,
5.73E-15,
6.12E-15,
6.15E-15,
6.38E-15,
6.64E-15,
6.84E-15,
6.95E-15,
7.04E-15,
7.15E-15,
7.28E-15,
7.44E-15,
7.44E-15,
7.41E-15,
7.45E-15,
7.35E-15
};
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
static const std::vector<double> kerma_factor_Testis_Adult_ICRU46_tableau = {
1.63E-17,
1.38E-17,
1.04E-17,
7.89E-18,
5.86E-18,
4.36E-18,
3.31E-18,
2.50E-18,
1.88E-18,
1.42E-18,
1.11E-18,
9.06E-19,
8.00E-19,
8.21E-19,
1.00E-18,
1.42E-18,
2.31E-18,
4.00E-18,
6.79E-18,
1.17E-17,
2.10E-17,
3.76E-17,
6.45E-17,
1.09E-16,
1.90E-16,
3.14E-16,
4.87E-16,
5.88E-16,
6.08E-16,
6.27E-16,
6.46E-16,
6.64E-16,
6.96E-16,
7.38E-16,
7.79E-16,
8.18E-16,
8.55E-16,
8.91E-16,
9.24E-16,
9.57E-16,
9.88E-16,
1.02E-15,
1.06E-15,
1.12E-15,
1.17E-15,
1.23E-15,
1.28E-15,
1.33E-15,
1.37E-15,
1.42E-15,
1.47E-15,
1.54E-15,
1.68E-15,
1.70E-15,
1.66E-15,
1.71E-15,
1.77E-15,
1.83E-15,
1.90E-15,
1.95E-15,
2.00E-15,
2.06E-15,
2.12E-15,
2.17E-15,
2.24E-15,
2.35E-15,
2.54E-15,
2.57E-15,
2.54E-15,
2.65E-15,
2.74E-15,
2.78E-15,
2.86E-15,
2.97E-15,
3.01E-15,
3.13E-15,
3.15E-15,
3.24E-15,
3.29E-15,
3.42E-15,
3.57E-15,
3.70E-15,
3.85E-15,
4.18E-15,
4.26E-15,
4.36E-15,
4.29E-15,
4.43E-15,
4.44E-15,
4.70E-15,
4.58E-15,
4.80E-15,
4.90E-15,
5.05E-15,
5.27E-15,
5.54E-15,
5.44E-15,
5.42E-15,
5.59E-15,
5.68E-15,
5.71E-15,
5.85E-15,
6.00E-15,
6.42E-15,
6.38E-15,
6.62E-15,
6.87E-15,
7.02E-15,
7.10E-15,
7.19E-15,
7.29E-15,
7.42E-15,
7.55E-15,
7.52E-15,
7.46E-15,
7.48E-15,
7.34E-15
};
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
static const std::vector<double> kerma_factor_Urinary_bladder_empty_Adult_ICRU46_tableau = {
2.11E-17,
1.78E-17,
1.35E-17,
1.02E-17,
7.57E-18,
5.64E-18,
4.28E-18,
3.23E-18,
2.42E-18,
1.82E-18,
1.41E-18,
1.13E-18,
9.69E-19,
9.45E-19,
1.09E-18,
1.48E-18,
2.35E-18,
4.02E-18,
6.77E-18,
1.16E-17,
2.09E-17,
3.73E-17,
6.40E-17,
1.08E-16,
1.88E-16,
3.11E-16,
4.82E-16,
5.83E-16,
6.03E-16,
6.21E-16,
6.40E-16,
6.58E-16,
6.89E-16,
7.32E-16,
7.72E-16,
8.10E-16,
8.47E-16,
8.83E-16,
9.16E-16,
9.49E-16,
9.79E-16,
1.01E-15,
1.05E-15,
1.11E-15,
1.16E-15,
1.22E-15,
1.26E-15,
1.31E-15,
1.35E-15,
1.41E-15,
1.46E-15,
1.53E-15,
1.66E-15,
1.68E-15,
1.64E-15,
1.70E-15,
1.76E-15,
1.82E-15,
1.88E-15,
1.94E-15,
1.99E-15,
2.05E-15,
2.10E-15,
2.15E-15,
2.22E-15,
2.33E-15,
2.52E-15,
2.55E-15,
2.52E-15,
2.63E-15,
2.71E-15,
2.76E-15,
2.83E-15,
2.94E-15,
2.98E-15,
3.11E-15,
3.12E-15,
3.21E-15,
3.26E-15,
3.39E-15,
3.54E-15,
3.67E-15,
3.82E-15,
4.15E-15,
4.23E-15,
4.33E-15,
4.26E-15,
4.40E-15,
4.41E-15,
4.66E-15,
4.54E-15,
4.76E-15,
4.87E-15,
5.01E-15,
5.23E-15,
5.50E-15,
5.40E-15,
5.37E-15,
5.55E-15,
5.63E-15,
5.66E-15,
5.81E-15,
5.95E-15,
6.37E-15,
6.34E-15,
6.58E-15,
6.82E-15,
6.97E-15,
7.05E-15,
7.14E-15,
7.24E-15,
7.37E-15,
7.50E-15,
7.47E-15,
7.41E-15,
7.43E-15,
7.29E-15
};
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
static const std::vector<double> kerma_factor_ICRU33_Soft_Tissue = { // Gy.m²
2.08E-17,
1.76E-17,
1.33E-17,
1.01E-17,
7.49E-18,
5.58E-18,
4.23E-18,
3.20E-18,
2.39E-18,
1.80E-18,
1.40E-18,
1.12E-18,
9.54E-19,
9.25E-19,
1.06E-18,
1.43E-18,
2.27E-18,
3.83E-18,
6.50E-18,
1.12E-17,
2.01E-17,
3.59E-17,
6.16E-17,
1.04E-16,
1.81E-16,
2.99E-16,
4.65E-16,
5.62E-16,
5.81E-16,
5.99E-16,
6.17E-16,
6.35E-16,
6.65E-16,
7.06E-16,
7.45E-16,
7.82E-16,
8.18E-16,
8.51E-16,
8.83E-16,
9.15E-16,
9.44E-16,
9.73E-16,
1.02E-15,
1.07E-15,
1.12E-15,
1.17E-15,
1.22E-15,
1.27E-15,
1.31E-15,
1.36E-15,
1.41E-15,
1.48E-15,
1.61E-15,
1.63E-15,
1.58E-15,
1.64E-15,
1.70E-15,
1.76E-15,
1.82E-15,
1.87E-15,
1.92E-15,
1.98E-15,
2.03E-15,
2.08E-15,
2.15E-15,
2.25E-15,
2.44E-15,
2.47E-15,
2.44E-15,
2.54E-15,
2.62E-15,
2.67E-15,
2.74E-15,
2.85E-15,
2.88E-15,
3.00E-15,
3.02E-15,
3.10E-15,
3.15E-15,
3.27E-15,
3.42E-15,
3.55E-15,
3.69E-15,
4.02E-15,
4.10E-15,
4.20E-15,
4.13E-15,
4.26E-15,
4.26E-15,
4.51E-15,
4.39E-15,
4.60E-15,
4.71E-15,
4.84E-15,
5.06E-15,
5.33E-15,
5.24E-15,
5.20E-15,
5.38E-15,
5.46E-15,
5.50E-15,
5.64E-15,
5.78E-15,
6.20E-15,
6.17E-15,
6.41E-15,
6.66E-15,
6.81E-15,
6.90E-15,
6.99E-15,
7.10E-15,
7.23E-15,
7.37E-15,
7.34E-15,
7.28E-15,
7.31E-15,
7.17E-15
};
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
static const std::vector<double> energy_table_KermaEquivalentFactor = { // MeV
0.1,
0.2,
0.3,
0.4,
0.5,
0.6,
0.7,
0.8,
0.9,
1.0,
1.1,
1.2,
1.3,
1.4,
1.5,
1.6,
1.7,
1.8,
1.9,
2.0,
2.1,
2.2,
2.3,
2.4,
2.5,
2.6,
2.7,
2.8,
2.9,
3.0,
3.1,
3.2,
3.3,
3.4,
3.5,
3.6,
3.7,
3.8,
3.9,
4.0,
4.1,
4.2,
4.3,
4.4,
4.5,
4.6,
4.7,
4.8,
4.9,
5.0,
5.1,
5.2,
5.3,
5.4,
5.5,
5.6,
5.7,
5.8,
5.9,
6.0,
6.1,
6.2,
6.3,
6.4,
6.5,
6.6,
6.7,
6.8,
6.9,
7.0,
7.1,
7.2,
7.3,
7.4,
7.5,
7.6,
7.7,
7.8,
7.9,
8.0,
8.1,
8.2,
8.3,
8.4,
8.5,
8.6,
8.7,
8.8,
8.9,
9.0,
9.1,
9.2,
9.3,
9.4,
9.5,
9.6,
9.7,
9.8,
9.9,
10.0,
10.1,
10.2,
10.3,
10.4,
10.5,
10.6,
10.7,
10.8,
10.9,
11.0,
11.1,
11.2,
11.3,
11.4,
11.5,
11.6,
11.7,
11.8,
11.9,
12.0,
12.1,
12.2,
12.3,
12.4,
12.5,
12.6,
12.7,
12.8,
12.9,
13.0,
13.1,
13.2,
13.3,
13.4,
13.5,
13.6,
13.7,
13.8,
13.9,
14.0,
14.1,
14.2,
14.3,
14.4,
14.5,
14.6,
14.7,
14.8,
14.9,
15.0,
15.1,
15.2,
15.3,
15.4,
15.5,
15.6,
15.7,
15.8,
15.9,
16.0,
16.1,
16.2,
16.3,
16.4,
16.5,
16.6,
16.7,
16.8,
16.9,
17.0,
17.1,
17.2,
17.3,
17.4,
17.5,
17.6,
17.7,
17.8,
17.9,
18.0,
18.1,
18.2,
18.3,
18.4,
18.5,
18.6,
18.7,
18.8,
18.9,
19.0,
19.1,
19.2,
19.3,
19.4,
19.5,
19.6,
19.7,
19.8,
19.9,
20.0
};
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
static const std::vector<double> kerma_equivalent_factor_ICRU33_Soft_Tissue = { //Sv.m²
1.12933E-14,
1.97155E-14,
2.50185E-14,
2.98331E-14,
2.99000E-14,
3.12769E-14,
3.27000E-14,
3.37660E-14,
3.50000E-14,
3.98388E-14,
3.72937E-14,
3.66484E-14,
3.71876E-14,
3.67636E-14,
3.62455E-14,
3.68167E-14,
3.66000E-14,
3.67081E-14,
3.67301E-14,
3.58388E-14,
3.58000E-14,
3.50297E-14,
3.41000E-14,
3.38817E-14,
3.41000E-14,
3.43058E-14,
3.45000E-14,
3.47324E-14,
3.48000E-14,
3.46769E-14,
3.53000E-14,
3.72975E-14,
3.91000E-14,
3.92332E-14,
3.88000E-14,
3.90072E-14,
3.89000E-14,
3.75131E-14,
3.60000E-14,
3.56239E-14,
3.59691E-14,
3.62000E-14,
3.57002E-14,
3.47303E-14,
3.37703E-14,
3.33000E-14,
3.36319E-14,
3.44090E-14,
3.51065E-14,
3.52000E-14,
3.43408E-14,
3.28838E-14,
3.13600E-14,
3.03000E-14,
3.00861E-14,
3.05057E-14,
3.11974E-14,
3.18000E-14,
3.20413E-14,
3.20059E-14,
3.18675E-14,
3.18000E-14,
3.19347E-14,
3.22333E-14,
3.26153E-14,
3.30000E-14,
3.33388E-14,
3.37110E-14,
3.42276E-14,
3.50000E-14,
3.60710E-14,
3.72103E-14,
3.81195E-14,
3.85000E-14,
3.81491E-14,
3.72477E-14,
3.60724E-14,
3.49000E-14,
3.39638E-14,
3.33239E-14,
3.29971E-14,
3.30000E-14,
3.33253E-14,
3.38691E-14,
3.45033E-14,
3.51000E-14,
3.55538E-14,
3.58498E-14,
3.59959E-14,
3.60000E-14,
3.58847E-14,
3.57318E-14,
3.56381E-14,
3.57000E-14,
3.59809E-14,
3.64104E-14,
3.68847E-14,
3.73000E-14,
3.75776E-14,
3.77384E-14,
3.78285E-14,
3.78937E-14,
3.79800E-14,
3.81335E-14,
3.84000E-14,
3.88122E-14,
3.93497E-14,
3.99786E-14,
4.06650E-14,
4.13753E-14,
4.20755E-14,
4.27319E-14,
4.33106E-14,
4.37780E-14,
4.41000E-14,
4.42536E-14,
4.42579E-14,
4.41429E-14,
4.39383E-14,
4.36739E-14,
4.33796E-14,
4.30853E-14,
4.28206E-14,
4.26156E-14,
4.25000E-14,
4.24964E-14,
4.25985E-14,
4.27929E-14,
4.30659E-14,
4.34042E-14,
4.37941E-14,
4.42221E-14,
4.46748E-14,
4.51386E-14,
4.56000E-14,
4.60478E-14,
4.64799E-14,
4.68966E-14,
4.72980E-14,
4.76845E-14,
4.80562E-14,
4.84133E-14,
4.87562E-14,
4.90850E-14,
4.94000E-14,
4.97010E-14,
4.99866E-14,
5.02550E-14,
5.05044E-14,
5.07329E-14,
5.09389E-14,
5.11203E-14,
5.12755E-14,
5.14027E-14,
5.15000E-14,
5.15667E-14,
5.16063E-14,
5.16235E-14,
5.16228E-14,
5.16087E-14,
5.15860E-14,
5.15592E-14,
5.15328E-14,
5.15116E-14,
5.15000E-14,
5.15018E-14,
5.15168E-14,
5.15442E-14,
5.15830E-14,
5.16321E-14,
5.16907E-14,
5.17577E-14,
5.18323E-14,
5.19134E-14,
5.20000E-14,
5.20913E-14,
5.21864E-14,
5.22847E-14,
5.23854E-14,
5.24878E-14,
5.25912E-14,
5.26949E-14,
5.27980E-14,
5.29000E-14,
5.30000E-14,
5.30975E-14,
5.31927E-14,
5.32858E-14,
5.33770E-14,
5.34666E-14,
5.35548E-14,
5.36421E-14,
5.37285E-14,
5.38144E-14,
5.39000E-14,
5.39856E-14,
5.40715E-14,
5.41579E-14,
5.42452E-14,
5.43334E-14
};
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
static const std::vector<double> energy_table_KermaEquivalentFactor_ICRP60 = { // MeV
1.00E-09,
1.58E-09,
2.51E-09,
3.98E-09,
6.32E-09,
1.00E-08,
2.53E-08,
3.60E-08,
6.30E-08,
1.10E-07,
2.00E-07,
3.60E-07,
6.30E-07,
1.10E-06,
2.00E-06,
3.60E-06,
6.30E-06,
1.10E-05,
2.00E-05,
3.60E-05,
6.30E-05,
1.10E-04,
2.00E-04,
3.60E-04,
6.30E-04,
1.10E-03,
2.00E-03,
3.60E-03,
6.30E-03,
1.10E-02,
2.00E-02,
3.60E-02,
6.30E-02,
8.20E-02,
8.60E-02,
9.00E-02,
9.40E-02,
9.80E-02,
1.05E-01,
1.15E-01,
1.25E-01,
1.35E-01,
1.45E-01,
1.55E-01,
1.65E-01,
1.75E-01,
1.85E-01,
1.95E-01,
2.10E-01,
2.30E-01,
2.50E-01,
2.70E-01,
2.90E-01,
3.10E-01,
3.30E-01,
3.50E-01,
3.70E-01,
3.90E-01,
4.20E-01,
4.60E-01,
5.00E-01,
5.40E-01,
5.80E-01,
6.20E-01,
6.60E-01,
7.00E-01,
7.40E-01,
7.80E-01,
8.20E-01,
8.60E-01,
9.00E-01,
9.40E-01,
9.80E-01,
1.05E+00,
1.15E+00,
1.25E+00,
1.35E+00,
1.45E+00,
1.55E+00,
1.65E+00,
1.75E+00,
1.85E+00,
1.95E+00,
2.10E+00,
2.30E+00,
2.50E+00,
2.70E+00,
2.90E+00,
3.10E+00,
3.30E+00,
3.50E+00,
3.70E+00,
3.90E+00,
4.20E+00,
4.60E+00,
5.00E+00,
5.40E+00,
5.80E+00,
6.20E+00,
6.60E+00,
7.00E+00,
7.40E+00,
7.80E+00,
8.20E+00,
8.60E+00,
9.00E+00,
9.40E+00,
9.80E+00,
1.05E+01,
1.15E+01,
1.25E+01,
1.35E+01,
1.45E+01,
1.55E+01,
1.65E+01,
1.75E+01,
1.85E+01,
1.95E+01
};
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
//static const std::vector<double> kerma_equivalent_factor_ICRU33_Soft_Tissue_ICRP60 = { //Sv.m²
//1.70E-15,
//1.35E-15,
//1.07E-15,
//8.50E-16,
//6.75E-16,
//5.36E-16,
//3.37E-16,
//2.86E-16,
//2.16E-16,
//1.63E-16,
//1.21E-16,
//9.05E-17,
//6.85E-17,
//5.17E-17,
//3.84E-17,
//2.87E-17,
//2.17E-17,
//1.65E-17,
//1.24E-17,
//9.47E-18,
//7.61E-18,
//6.73E-18,
//7.26E-18,
//1.03E-17,
//1.75E-17,
//3.30E-17,
//6.90E-17,
//1.50E-16,
//3.23E-16,
//7.01E-16,
//1.63E-15,
//3.54E-15,
//6.96E-15,
//9.29E-15,
//9.72E-15,
//1.02E-14,
//1.06E-14,
//1.11E-14,
//1.18E-14,
//1.29E-14,
//1.38E-14,
//1.48E-14,
//1.57E-14,
//1.65E-14,
//1.73E-14,
//1.80E-14,
//1.88E-14,
//1.94E-14,
//2.04E-14,
//2.16E-14,
//2.26E-14,
//2.37E-14,
//2.45E-14,
//2.55E-14,
//2.61E-14,
//2.71E-14,
//2.79E-14,
//2.90E-14,
//3.14E-14,
//3.13E-14,
//2.99E-14,
//3.05E-14,
//3.10E-14,
//3.16E-14,
//3.22E-14,
//3.27E-14,
//3.31E-14,
//3.36E-14,
//3.39E-14,
//3.43E-14,
//3.50E-14,
//3.64E-14,
//3.93E-14,
//3.87E-14,
//3.66E-14,
//3.70E-14,
//3.71E-14,
//3.64E-14,
//3.64E-14,
//3.70E-14,
//3.63E-14,
//3.71E-14,
//3.61E-14,
//3.58E-14,
//3.41E-14,
//3.41E-14,
//3.45E-14,
//3.48E-14,
//3.53E-14,
//3.91E-14,
//3.88E-14,
//3.89E-14,
//3.60E-14,
//3.62E-14,
//3.33E-14,
//3.52E-14,
//3.03E-14,
//3.18E-14,
//3.18E-14,
//3.30E-14,
//3.50E-14,
//3.85E-14,
//3.49E-14,
//3.30E-14,
//3.51E-14,
//3.60E-14,
//3.57E-14,
//3.73E-14,
//3.84E-14,
//4.41E-14,
//4.25E-14,
//4.56E-14,
//4.94E-14,
//5.15E-14,
//5.15E-14,
//5.20E-14,
//5.30E-14,
//5.39E-14
//};
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
static const std::vector<double> kerma_equivalent_factor_ICRU33_Soft_Tissue_ICRP60 = { //Sv.m²
2.01E-15,
1.59E-15,
1.27E-15,
1.00E-15,
7.97E-16,
6.33E-16,
3.99E-16,
3.38E-16,
2.55E-16,
1.93E-16,
1.43E-16,
1.07E-16,
8.10E-17,
6.12E-17,
4.58E-17,
3.43E-17,
2.63E-17,
2.03E-17,
1.55E-17,
1.20E-17,
9.73E-18,
8.82E-18,
9.87E-18,
1.41E-17,
2.37E-17,
4.45E-17,
9.36E-17,
2.06E-16,
4.53E-16,
9.91E-16,
2.30E-15,
4.93E-15,
9.50E-15,
1.25E-14,
1.31E-14,
1.37E-14,
1.42E-14,
1.48E-14,
1.57E-14,
1.70E-14,
1.82E-14,
1.93E-14,
2.04E-14,
2.14E-14,
2.24E-14,
2.33E-14,
2.41E-14,
2.49E-14,
2.61E-14,
2.75E-14,
2.86E-14,
2.99E-14,
3.09E-14,
3.20E-14,
3.27E-14,
3.37E-14,
3.45E-14,
3.57E-14,
3.82E-14,
3.81E-14,
3.66E-14,
3.74E-14,
3.79E-14,
3.85E-14,
3.90E-14,
3.95E-14,
3.99E-14,
4.03E-14,
4.06E-14,
4.09E-14,
4.15E-14,
4.29E-14,
4.58E-14,
4.52E-14,
4.30E-14,
4.33E-14,
4.34E-14,
4.25E-14,
4.23E-14,
4.28E-14,
4.20E-14,
4.28E-14,
4.17E-14,
4.13E-14,
3.95E-14,
3.94E-14,
3.96E-14,
3.98E-14,
4.01E-14,
4.39E-14,
4.35E-14,
4.34E-14,
4.04E-14,
4.03E-14,
3.73E-14,
3.89E-14,
3.41E-14,
3.54E-14,
3.52E-14,
3.57E-14,
3.63E-14,
3.85E-14,
3.63E-14,
3.47E-14,
3.57E-14,
3.66E-14,
3.70E-14,
3.79E-14,
3.91E-14,
4.32E-14,
4.24E-14,
4.55E-14,
4.92E-14,
5.12E-14,
5.14E-14,
5.18E-14,
5.29E-14,
5.37E-14
};
//-----------------------------------------------------------------------------

#endif
