(*
   Copyright 2008-2018 Microsoft Research

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
*)
module Examples

open FStar.Tactics
open FStar.SL.Heap
open Lang
open SL.Tactics

(*
 * Examples of programs and postconditions for the language in Lang
 * We use the generic tactic `solve` to solve these goals
 *)

#reset-options "--using_facts_from '* -FStar.Tactics -FStar.Reflection'"

(***** Examples *****)
open FStar.UInt
open FStar.UInt64

type t = UInt64.t

(*
 * Prove that if we write n to r, then sel h r = n
 * This gets completely solved using tactics
 * There are some SMT goals generated by type checker during unification, related to Prims.prop and Type0
 *)
let write_ok (h:heap) (r:addr) (n:t) =
  let c = (Write r n) in
  let p = fun _ h -> sel h r == n in
  let post = (lift_wpsep (wpsep_command c)) p h in
  assert (h `contains` r ==> post) by solve ()

(*
 * Read r into n, Write n + 1 to r, then sel h r = n + 1
 *)
let increment_ok (h:heap) (r:addr) (n:t) =
  let c = Bind (Read r) (fun n -> Write r (n +?^ 1uL)) in
  let p = fun _ h -> (sel h r == n +?^ 1uL) in
  let post = (lift_wpsep (wpsep_command c)) p h in
  assert (h `contains` r /\ sel h r == n ==> post) by solve ()

(*
 * Swapping two refs
 *)
let swap_ok (r1:addr) (r2:addr) (h:heap) (a:t) (b:t) =
  let c = Bind (Read r1) (fun n1 -> Bind (Read r2) (fun n2 -> Bind (Write r1 n2) (fun _ -> Write r2 n1))) in
  let p = fun _ h -> (sel h r1 == b /\ sel h r2 == a) in
  let post = (lift_wpsep (wpsep_command c)) p h in
  assert (h `contains` r1 /\  h `contains` r2 /\ addr_of r1 <> addr_of r2 /\ sel h r1 == a /\ sel h r2 == b ==> post) by solve ()

#reset-options "--using_facts_from '* -FStar.Tactics -FStar.Reflection'"
let double_increment_ok (r:addr) (h:heap) (n:t{size (v n + 2) FStar.UInt64.n}) =
  let c = Bind (Bind (Read r) (fun y -> Write r (y +?^ 1uL))) (fun _ -> (Bind (Read r) (fun y -> Write r (y +?^ 1uL))))  in
  let p = fun _ h -> sel h r == (n +?^ 2uL) in
  let t = (lift_wpsep (wpsep_command c)) p h in
  assert (h `contains` r /\ sel h r == n ==> t) by solve ()

(*
 * This example also goes through but takes a lot of time.
 * Because of many SMT goals that are generated as part of unification.
 *)
// let rotate_ok (r1:addr) (r2:addr) (r3:addr) (h:heap) (i:t) (j:t) (k:t) =
//   let c = Bind (Bind (Read r1) (fun n1 -> Bind (Read r2) (fun n2 -> Bind (Write r1 n2) (fun _ -> Write r2 n1)))) (fun _ -> Bind (Read r2) (fun n3 -> Bind (Read r3) (fun n4 -> Bind (Write r2 n4) (fun _ -> Write r3 n3)))) in
//   let p = fun _ h -> (sel h r1 == j /\ sel h r2 == k /\ sel h r3 == i) in
//   let t = (lift_wpsep (wpsep_command c)) p h in
//   assert (h `contains` r1 /\ h `contains` r2 /\ h `contains` r3 /\ 
//                     addr_of r1 <> addr_of r2 /\ addr_of r2 <> addr_of r3 /\ addr_of r1 <> addr_of r3 /\
// 		    sel h r1 == i /\ sel h r2 == j /\ sel h r3 == k ==> t) by solve ()

let init_ok (h:heap) =
  let c = Bind (Alloc) (fun (r1:addr) -> Bind (Write r1 7uL) (fun _ -> Return r1)) in
  let p = fun r h -> sel h r == 7uL in
  let t = (lift_wpsep (wpsep_command c)) p h in
  assert t by solve ()

let copy_ok (r1:addr) (r2:addr) (r3:addr) (h:heap) (i:t) (j:t) (k:t) =
  let c = Bind (Read r1) (fun n1 -> Write r2 (n1)) in
  let p = fun _ h -> (sel h r1 == i /\ sel h r2 == i /\ sel h r3 == k) in
  let post = (lift_wpsep (wpsep_command c)) p h in
  assert (h `contains` r1 /\ h `contains` r2 /\ h `contains` r3 /\ addr_of r1 <> addr_of r2 /\ addr_of r2 <> addr_of r3 /\ addr_of r1 <> addr_of r3 /\ sel h r1 == i /\ sel h r2 == j /\ sel h r3 == k ==> post) by solve ()
